// Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
//

/**
 @file
 @internalComponent
 @prototype
*/


#ifndef LBS_SUPL_PSY_ADAPTATION_H
#define LBS_SUPL_PSY_ADAPTATION_H

#include <e32std.h>
#include <e32base.h>
#include <lbs.h>
#include "lbsnetinternalapi.h"
#include "lbsnetinternalmsgtypes.h"


/**
The MLbsSuplPsyAdaptationObserver class is used in partnership with
CLbsSuplPsyAdaptation. The observer interface is used by LBS to provide 
responses to requests made by the SUPL PSY via CLbsSuplPsyAdaptation. 

Although all the interface methods in the MLbsSuplPsyAdaptationObserver
are synchronous they should return quickly. If required, the SUPL PSY should 
internally queue the responses and process them asynchronously. 

@see CLbsSuplPsyAdaptation
*/
class MLbsSuplPsyAdaptationObserver
	{
public:
	virtual TVersion Version() = 0;
	
	virtual void OnLocationUpdate(TUint aRequestId, TPositionInfoBase& aPosition, TInt aReason) = 0;

	/**
	LBS issues a single call to OnSessionComplete() to close a session and return the final 
	result code to the SUPL PSY. 

	The aReason parameter indicates the overall success or failure for the request. 

	The request ID parameter identifies the session that is being closed. This ID is allocated when
	a new request is created: 

	In normal situations, OnSessionComplete() will be invoked when the overall request has
	finished. OnSessionComplete() can also be used by LBS to terminate a request if a protocol 
	conflict has arisen. For example, the local SUPL subsystem or a remote SUPL Server is unable to 
	simultaneously perform two requests (for example a TerminalBased AGPS session is active).

	@param aRequestId The Id of the request being closed. 

	@param aReason Reason for the completion of the request. KErrNone if the request is successfully
	       completed, or one of a range of error codes otherwise. 
	*/
	virtual void OnSessionComplete(TUint aRequestId, TInt aReason) = 0;
	};


/**
The SUPL PSY adaptation interface is a point to point interface between the SUPL PSY and 
the Netowrk Gateway and allows the SUPL PSY to ask a remote SUPL Server for the position of 
the device. As the interface is point to point, only one instance is allowed and must be shared 
between one or more SUPL PSYs. It is the responsibility of the SUPL PSY(s) to combine multiple 
requests coming from multiple users.

The CLbsSuplPsyAdaptation class is used in partnership with MLbsSuplPsyAdaptationObserver. 
LBS responds to requests from the SUPL PSY via that observer class. 

Although all the interface methods in the class are synchronous they must return immediately.
When the SUPL PSY makes a request via CLbsSuplPsyAdaptation the LBS subsystem must
queue the request internally and return control to the caller immediately. Afterwards, when
the SUPL Server provides a response LBS uses the corresponding notification method in the
MLbsSuplPsyAdaptationObserver class. 

The interface is a temporary solution and should be removed (or incorporated into the SUPL PSY) 
once a common codeline is established.

The interface, even though declared as internalAll, MUST be used by the SUPL PSY EXCLUSIVELY.
It MUST NOT be used by the Generic Network PSY, or any other components.

@see MLbsSuplPsyAdaptationObserver 
*/
NONSHARABLE_CLASS(CLbsSuplPsyAdaptation) : public CActive, public MLbsNetChannelObserver
	{
public:
	static CLbsSuplPsyAdaptation * NewL(MLbsSuplPsyAdaptationObserver& aObserver);
	~ CLbsSuplPsyAdaptation ();
	
	/**
	RequestLocationUpdate() should be used by the SUPL PSY to request a SUPL Server to
	calculate the current location of the handset and return it to the terminal. 

	When the location information is received from the SUPL Server it is being forwarded to 
	the SUPl PSY via the method MLbsSuplPsyAdaptationObserver::OnLocationUpdate().

	The position information returned is network calculated and is normally
	expected to be an approximate location only. For example, related to the position
	of the current serving cell. It may also be a position calculated by the network with
	the assistance of the terminal (the terminal sends measurements to the network).

	This method automatically cancels any active location request and
	replaces it with the new request. The CancelLocationRequest() method is used when the
	SYPL PSY wants to a cancel request but does not immediately wish to replace it with
	another.

	The aRequestId parameter is generated by the SUPL PSY and is used to connect all
	corresponding responses and further requests. The same request ID must be used the 
	SUPL PSY if it wishes to cancel the request. Similarly, the same request ID must also
	be supplied by LBS when it responds to the SUPL PSY 
	via the MLbsSuplPsyAdaptationObserver::OnLocationUpdate() method.

	The request ID is also passed by the LBS subsystem when it closes a session via the
	method MLbsSuplPsyAdaptationObserver::OnSessionComplete().

	@param aRequestId The Id of the location request. This is generated by the SUPL PSY
	       and must be used by LBS when it returns the obtained position information. 

	@param aProtocolModuleId The Id of the protocol Module to be used.
	@param aProtocolModuleId The Id of the protocol Module to be used.
	@param aNewClient		 Indicates the first request from a new client.


	@see CLbsAdmin
	@see MLbsSuplPsyAdaptationObserver::OnSessionComplete()
	@see MLbsSuplPsyAdaptationObserver::OnLocationUpdate()
	@see CancelLocationRequest
	*/
	void RequestLocationUpdate(TUint aRequestId, TBool aNewClient, TUid aProtocolModule);

	/**
	This method attempts to cancel a request to obtain the location
	of the terminal. The previous request will have been initiated by calling
	RequestLocationUpdate()

	Note: If the SUPL PSY wishes to stop the location request and immediately issue a 
	new location request it should issue a subsequent call to RequestLocationUpdate().
	Submitting a new location request, automatically cancels any outstanding transfer.

	In some circumstances, LBS may still return the associated position to the SUPl PSY.
	This situation can occur when the cancel request is performed after a point where it 
	is no longer possible to stop the request being sent.

	@param aRequestId The Id of the network location request to be cancelled.
	@param aReason Indicates why the LBS subsystem wants the terminate the request. 

	@see RequestLocationUpdate() 
	@see MLbsSuplPsyAdaptationObserver::OnLocationUpdate()
	*/
	void CancelLocationRequest(TUint aRequestId, TInt aReason);

    virtual void ProcessNetChannelMessage(RLbsNetChannel::TLbsNetChannelId aChannelId, const TLbsNetInternalMsgBase& aMessage);

private:
        /**
        * From CActive
        */
        void RunL();

        /**
        * From CActive
        */
        void DoCancel();

        /**
        * From CActive
        */
        TInt RunError( TInt aError );

        TLbsNetInternalMsgBase* CreateTLbsCellLocationRequestMsgL(TUint aRequestId, TBool aNewClient, TUid aProtocolModule);

private:
	CLbsSuplPsyAdaptation (MLbsSuplPsyAdaptationObserver& aObserver);
	void ConstructL();
            /**
        * Construct HPositionGenericInfo and set requested fields
        */
        HPositionGenericInfo* ConstructGenericInfoL( 
            TInt aBufferSize = KPositionGenericInfoDefaultBufferSize ) const;

private:
            // Callback function for get location request complete event
        MLbsSuplPsyAdaptationObserver& iObserver;

        RLbsNetChannel iPsyChannel;

        TLbsNetInternalMsgBase* iSentMsg;
        
        // position info
        TPositionInfoBase* iPositionInfo;
        
        TUint iRequestid;    

        TBool iLocRequest;
	};

	
#endif // LBS_SUPL_PSY_ADAPTATION_H
