/*
* Copyright (c) 1998-2009 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description: 
* X509 certificate chain and the validation status implementations
*
*/


/**
 @file 
 @publishedAll
 @released
*/
 
#ifndef __X509CERTCHAIN_H__
#define __X509CERTCHAIN_H__

#include <e32std.h>
#include <x509cert.h>
#include <ct.h>

class TValidationStatus
/** The validation status.
* 
* Some errors cannot be blamed on any single certificate, in which case the 
* iCert value is meaningless. The same structure is used for errors and for 
* warnings. 
* 
* @since v6.0 */
	{
public:
	/** Creates a validation status object.	
	* 
	* @param aError	The error type that occurred when validating the certificate chain.
	* @param aCert	The index number identifying the certificate that gave rise to 
	* 				the error. */
	IMPORT_C TValidationStatus(const TValidationError aError, const TInt aCert);
	
	/** The reason for the error. */
	TValidationError iReason;
	
	/** The index number identifying the certificate that gave rise to the error. */
	TInt iCert;
	};

class CX509CertChain : public CBase
/** Abstract base class for X.509 certificate chain validation; 
* derive from this to suit your profile.
* 
* @since v6.0 */
	{
public:
	/** Gets the number of certificates in the chain.	
	* 
	* @return	The number of certificates in the chain. */
	IMPORT_C TInt Count() const;
	
	/** Gets the certificate identified by the specified index.
	* Note that Cert(Count()) corresponds to the root (if any)
	* whilst Cert(0) corresponds to the outmost certificate in the chain.
	*
	* @param aIndex	The ordinal number representing the position of the certificate 
	* 				within the chain.
	* @return		The X.509 certificate at the specified index. */
	IMPORT_C const CX509Certificate& Cert(TInt aIndex) const;
	
	/** Decodes the individual elements of the signed data to construct the certificates.
	* 
	* @param aBinaryData	The encoded binary representation.
	* @return				The certificate objects. */
	IMPORT_C CArrayPtrFlat<CX509Certificate>* DecodeCertsL(const TDesC8& aBinaryData);
	
	/** Destructor.
	* 
	* Frees all resources owned by the object, prior to its destruction. */
	IMPORT_C ~CX509CertChain();

	/** Tests whether the specified X.509 certificate chain is equal to this X.509 
	* certificate chain.
	* 
	* @param aOther	The X.509 certificate chain to be compared.
	* @return		ETrue, if the certificate chains are equal;EFalse, otherwise. */
	IMPORT_C TBool IsEqualL(const CX509CertChain& aOther) const;
protected:
	//certificate chain
	CArrayPtrFlat<CX509Certificate>* iChain;
private:
	static void CleanupCertArray(TAny* aArray);
	};
	
class CCertificateValidationWarnings : public CBase
	/** Encapsulates the critical extensions encountered and any warnings found
	* for a particular certificate in the chain during the process of validation.
	* 
	* @since v9.5 */
		{
	public:
		/** Creates an instance of CCertificateValidationWarnings.
		* 
		* @param aIndex	The index of aCert in the certificate chain.
		* @return	A pointer to the new CCertificateWarning object. */	
		IMPORT_C static CCertificateValidationWarnings* NewL(TInt aIndex);

		/** Creates an instance of CCertificateValidationWarnings.
		* 
		* @param aIndex	The index of aCert in the certificate chain.
		* @return	A pointer to the new CCertificateWarning object. */	
		IMPORT_C static CCertificateValidationWarnings* NewLC(TInt aIndex);		
		
		/** Gets a list of critical extension OIDs found in the certificate.
		* 
		* @return	An array of critical extensions found. */		
		IMPORT_C const RPointerArray<TDesC>& CriticalExtensionsFound() const;
		
		/** Gets a list of warnings generated by the certificate.
		* 
		* @return	An array of warnings generated. */	
		IMPORT_C const RArray<TValidationStatus>& Warnings() const;
		
		/** Gets the index of the certificate in the chain.
		* 
		* @return	The certificate index number. */		
		IMPORT_C TInt CertIndex() const;
		
		/** Externalises an object of this class to a write stream.
		* 
		* The presence of this function means that the standard templated operator<<() 
		* can be used to externalise objects of this class.
		* 
		* @param aStream    Stream to which the object should be externalised. */
		IMPORT_C void ExternalizeL(RWriteStream& aStream) const;
		
		/** Internalises an object of this class from a read stream.
		* 
		* The presence of this function means that the standard templated operator>>() 
		* can be used to internalise objects of this class.
		* 
		* Note that this function has assignment semantics: it replaces the old value 
		* of the object with a new value read from the read stream. 
		* 
		* @param aStream    Stream from which the object should be internalised. 
		* @return A pointer to the new CCertificateWarning object. */
		IMPORT_C static CCertificateValidationWarnings* InternalizeL(RReadStream& aStream);
		
		/** The destructor.
		* 
		* Frees all resources owned by the object. */
		IMPORT_C ~CCertificateValidationWarnings();
		
	public:
		/** Adds a warning.
		* 
	    */
		IMPORT_C void AppendWarningL(TValidationStatus aWarning);
		
		/** Adds a critical extension OID warning.
		* 
		*/
		IMPORT_C void AppendCriticalExtensionWarningL(TDesC& aCriticalExt);
		
	private:
		CCertificateValidationWarnings(TInt aIndex);
	
	private:
		TInt iCertIndex;
		RPointerArray<TDesC> iCriticalExtsFound;
		RArray<TValidationStatus> iWarnings;
		};

#endif


