#!/usr/bin/perl
# Copyright (c) 2010 Symbian Foundation Ltd
# This component and the accompanying materials are made available
# under the terms of the License "Eclipse Public License v1.0"
# which accompanies this distribution, and is available
# at the URL "http://www.eclipse.org/legal/epl-v10.html".
#
# Initial Contributors:
# Mike Kinghan, mikek@symbian.org for Symbian Foundation Ltd - initial contribution.

# Script to get a copy of upstream package at the baseline revision in
# ../baseline.txt and patch it to be the same as this package.

use strict;
use get_baseline;
use perl_run;
use places;
use get_hostplatform_dir;
use check_os;
use File::Spec;
use File::Temp 'tempfile';
use Cwd 'abs_path';
use Getopt::Long;


my $help =0;
my $baseline_dir = '';
my $patch_in = '';

sub usage(*);
sub usage_error($);
sub help();
sub get_old_baseline_dir($);
sub regen_patch_file($);

GetOptions("help=i" => \$help, "patch-file=s" => \$patch_in,
	"baseline-dir=s" => \$baseline_dir) or usage_error("Usage error");

if ($help) {
	help();
}
if (!$baseline_dir) {
	usage_error("Need -b, --baseline-dir");
}
print ">>> Testing for patch\n";
my $patch_test = "patch --version";
print ">>> Executing: $patch_test\n";
my $rc = system($patch_test);
die "*** Error: can't execute the patch tool ***", if ($rc);
$baseline_dir = abs_path($baseline_dir);
perl_run("get_upstream.pl $baseline_dir") and die $!;
my $epocroot = get_epocroot();
if (!$patch_in) {
	my $baseline_rev = get_baseline();
	$patch_in = File::Spec->catfile(get_pkg_dir(),"cross-plat-dev-utils",
		"patch-files","diffs","patch-$baseline_rev.patch");
}
die "*** Error: can't find patch file \"$patch_in\" ***", unless ( -f $patch_in);
$baseline_dir = File::Spec->catfile("$baseline_dir","build");
my $pnum = compute_pnum($patch_in);
my $patch_cmd = "patch -d $baseline_dir -p$pnum < $patch_in";
print ">>> Executing: $patch_cmd\n";
exit system($patch_cmd) >> 8;

sub usage(*)
{
	local *OUT = shift;
	print OUT "This script gets a copy of upstream package at the " .
		" baseline revision in ../baseline.txt and patches it to be the same ".
		"as this package\n" .
		"Options:\n" .
		"-h, --help:              Display this help and exit\n" .
		"-p, --patch-file FILE:   Apply the patch file FILE. " .
			"Default ./patch-files/diffs/patch-N, where N is the baseline revision\n" .
		"-b, --baseline-dir DIR:  The upstream baseline will be cloned into " .
			"DIR if this does not seem to have been done already. Otherwise " .
			"DIR/build will be udated to the baseline revision before patching\n" .
		"*** It is assumed that the command 'patch' will invoke the patch tool " .
		"***\n";
}

sub usage_error($)
{
	my $diag = shift;
	print STDERR "*** $diag ***\n";
	usage(*STDERR);
	exit 1;
}

sub help()
{
	usage(*STDOUT);
	exit 0;
}

sub get_old_baseline_dir($)
{
	my $patch_file = shift;
	open PATCH,"<$patch_file" or die $!;
	my $line = <PATCH>;
	if ($line !~ /^## diff generated by diff_upstream.pl/) {
		print "!!! Warning: patch file was not created by diff_upstream.pl. " .
			"proceeding at risk !!!\n";
	} else {
		$line = <PATCH>;
	} 
	close PATCH;
	die "*** Error: \"$patch_file\" is empty ***", unless ($line);
	my @words = split(/ /,$line);
	if (@words < 3) {
		die "*** Error: can't make sense of \"$patch_file\" as a patch file ***";
	}
	my $old_baseline_file = $words[-2];
	my $path_punct = os_is_windows() ? '\\' : '/';
	my @old_baseline_parts = split(/$path_punct/,$old_baseline_file);
	while($old_baseline_parts[-1] ne 'build') {
		pop @old_baseline_parts;
	}
	my $old_base_line_dir = join($path_punct,@old_baseline_parts);
	print ">>> Patch-file seems to have been diffed from \"$old_base_line_dir\"\n";
	return $old_base_line_dir;
}

sub compute_pnum($)
{
	my $patch_file = shift;
	my $old_base_line_dir = get_old_baseline_dir($patch_file);
	my $path_punct = os_is_windows() ? '\\' : '\/';
	my @old_base_line_dir_parts = split(/$path_punct/,$old_base_line_dir);
	return @old_base_line_dir_parts;
}

	
