/* Metrowerks x86 Runtime Support Library 
 * Copyright  1995-2003 Metrowerks Corporation.  All rights reserved.
 *
 * $Date: 2005/10/07 15:33:11 $
 * $Revision: 1.3 $
 */

#pragma ANSI_strict off
 
// Set up the argument count and argument vector that are passed to main

#include <TCHAR.H>
#include <stdlib.h>
#include <stddef.h>
#include <stdarg.h>

#define WIN32_LEAN_AND_MEAN
#include <windows.h>

#include <crtl.h>

//	When defined, we will look up wildcards for 
//	unquoted arguments with '*' or '?' in them.
#define _MSL_RUNTIME_EXPAND_WILDCARDS 	1

//
// Some comments about the command line parsing strategy.
//
// The Win32 command-line space is hideously polluted with
// problems using command lines, not limited to prominent
// spaces in key OS directories, using backslashes for directory
// separators when the backslash was being used as a C escape
// character, having a broken shell that doesn't expand wildcards,
// having a broken API that passes one command-line string to 
// the application rather than an array of strings, and not 
// stripping quote marks from command lines.
//
// Don't get me wrong, I like Windows.
//
// Anyway, with changes in Pro5 to the command-line argument
// parsing, the intent was to allow unbroken arguments like:
//
// 		myprogram "my string" "my other string"
//
// as well as:
//
// 		mwcc -c file.c -DSTRINGMACRO=\"string\"
//
// to be properly formed into arguments the programs would expect.
//
// Unfortunately, the latter case causes a problem with this
// not-so-special case:
//
//		myprogram "c:\here is a directory\"
//
// Here, the last character of the pathname is resolved to a quote
// mark, which is not quite expected.  It does not work merely
// to ignore an escaped quote inside a quoted string, since this
// case must work, too:
//
//		mwcc -c file.c -DSTRINGMACRO="\"string\""
//
// Although the behavior in the second-to-last example appears
// broken, we must keep it for compatibility with VC.
 
// The argument pointer and count variables
_TCHAR **__targv;
_TCHAR *_tcmdln;
int __argc, __argv1_offs;
static int __maxargc;

// Copy of command line, modified to fill argv[]
static _TCHAR *__command_line = 0;

#define QUOTE '\"'		//" stupid IDE syntax coloring

static void _FreeCommandLine(void)
{
	if (__command_line)
	{
		free(__command_line);
		__command_line = 0L;
	}
	
	// __argc == 0 => we ran out of memory and 
	// __targv is just a ptr to static memory
	if (__targv && __argc)
	{
		free(__targv);		
		__targv = 0L;
	}
}

static void _MemFailure(void)
{
	static _TCHAR *fakeargs[] = { NULL };
	__targv = fakeargs;
	__argc = 0;
}

/*
 *	Allocate more slots in argv.
 */
static int _MoreArgs(void)
{
	_TCHAR **newargs;
	
	if (__argc + 1 >= __maxargc)
	{
		__maxargc += 16;
		newargs = (_TCHAR **) realloc(__targv, __maxargc * sizeof(_TCHAR *));
		if (!newargs)
			return 0;

		__targv = newargs;
	}
	return 1;
}


#if _MSL_RUNTIME_EXPAND_WILDCARDS
/*
 *	Expand the previous argument (__targv[__argc-1]) into the files
 *	it matches, else leave it the same.
 */
static void _ExpandArgument(void)
{
	WIN32_FIND_DATA fd;
	HANDLE fh;
	_TCHAR *fname = __targv[__argc-1];
	_TCHAR *fnptr;
	
	fnptr = fname + _tcslen(fname);
	while (fnptr > fname && _tcschr(_T("/\\:"), *(fnptr - 1)) == 0L)
		fnptr--;

	fh = FindFirstFile(fname, &fd);
	if (fh != INVALID_HANDLE_VALUE)
	{
		do
		{
		   if (!(fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
		   {
				/*- EJS 031114: handle unconvertible names -*/
				/*- EJS 040109: do this right -*/
				_TCHAR *arg;

				/* make supposed final path using cFileName */
				arg = (_TCHAR *)malloc((fnptr - fname + _tcslen(fd.cFileName) + 1) * sizeof(_TCHAR));
				_tcsncpy(arg, fname, (fnptr - fname));
				_tcscpy(arg + (fnptr - fname), fd.cFileName);
				
#if !UNICODE
				/* see if it exists -- the cFileName may be a dummy value when ANSI version
					is unrepresentable */
				if (GetFileAttributes(arg) == 0xFFFFFFFF)
				{
					char *ptr;
					free(arg);

					/* use 8.3 name */
					arg = (_TCHAR *)malloc((fnptr - fname + _tcslen(fd.cAlternateFileName) + 1) * sizeof(_TCHAR));
					_tcsncpy(arg, fname, (fnptr - fname));
					
					/* lowercase it since uppercase extensions confuse everyone */
					for (ptr = fd.cAlternateFileName; *ptr; ptr++)
						*ptr = _totlower(*ptr);
					_tcscpy(arg + (fnptr - fname), fd.cAlternateFileName);
				}
#endif

				if (!_MoreArgs())
					break;
					
				// overwrite previous string
				__targv[__argc-1] = arg;
				__argc++;
		   }
		} while (FindNextFile(fh, &fd));

		__argc--;
		
		// note: if no matches are made, original wildcard is left in place
		
		FindClose(fh);
	}
	else
	{
		// no matches, ignore.
	}
}
#endif	// _MSL_RUNTIME_EXPAND_WILDCARDS

/*
 *	Setup C-style command line arguments from Windows' monolithic argument string.
 */
#if __MSL__ >= 0x9000
void _MSL_CDECL _tSetupArgs(void)
#else
void _MSL_CDECL _SetupArgs(void)
#endif
{
#if __dest_os== __win32_os

	#define WHITESPACE _T(" \t\r\n\v")
	
	_TCHAR	*clptr;		// initial incoming command line
	_TCHAR	*inptr;		// pointer into ASCII version of cmdline string 
	_TCHAR	*outptr;	// current outptr into copy of cmdline string and argv[x]

#if _MSL_RUNTIME_EXPAND_WILDCARDS
	int		wild;		// was the argument wildcardable?  ('*' or '?' not inside quotes)
#endif

	/*	This code forms the argv[] array from a copy of the GetCommandLine()
		string returned by Windows, and sets up argv[] with pointers to the
		data therein.  The string is modified by terminating each entry of
		argv[] with a NUL and shortening strings if quotes or escaped quotes 
		are encountered. */

	/*	EJS 040517: this is called once per DLL loaded using the shared runtime --
		only initialize once! */
		
	if (__targv) return;
	
	__maxargc = 1;
	__argc = 0;
	__targv = (_TCHAR **) malloc(sizeof(_TCHAR *) * __maxargc);
	__argv1_offs = 0;
	
	//  utter failure?!
	if (__targv == NULL) {
		_MemFailure();
		return;
	}
	
	_tcmdln = inptr = clptr = GetCommandLine();

	// 	make duplicate of command line to generate argv[]
	outptr = __command_line = (_TCHAR *)malloc((_tcslen(inptr)+1)*sizeof(_TCHAR));
	if (!outptr) {
		_MemFailure();
		return;
	}
	
	//	skip unthinkable leading spaces
	while (_tcschr(WHITESPACE, *inptr) != NULL)
		inptr++;
	
	while (*inptr)
	{
		int		quoting;	// inside quoted part?

		//  allocate more space for argv[]
		if (!_MoreArgs())
			// 	failed!  keep old args;
			//	fall through and zero last entry.
			break;

#if _MSL_RUNTIME_EXPAND_WILDCARDS
		wild = 0;
#endif
		// record offset of first argument
		if (__argc == 1)
			__argv1_offs = inptr - clptr;
			
		__targv[__argc] = outptr;
		__argc++;
	
		//  Parse one argument from inptr and copy into outptr.
		//  Rules:  a series of chars inside double quotes are not broken;
		//	and such double quotes are removed.
		//	Escaped quotes (\") are kept as one quote _TCHAR;
		//	other escapes are completely ignored.
	
		// 	EJS 990917:  remove newlines and such (encountered in Cygwin32
		//	tools from the bash shell with the `...` construct)
		
		quoting = 0;
		while (*inptr)
		{
			if (!quoting)
			{
				// argument breaker?
				if (_tcschr(WHITESPACE, *inptr) != NULL)
				{
					do	inptr++;	
					while (*inptr && _tcschr(WHITESPACE, *inptr) != NULL);
	
					*outptr++ = 0;	// break current argument
					break;			// on to next argument
				}
				else
				// one quote?
				if (*inptr == QUOTE)
				{
					// eat it and start quote mode
					inptr++;
					quoting = 1;
					continue;
				}
			}
			else	// quoting
			{
				// two quotes?
				if (*inptr == QUOTE && *(inptr+1) == QUOTE)
				{
					// morph into one quote
					*outptr++ = QUOTE;
					inptr += 2;
					continue;
				}
				else
				// one quote?
				if (*inptr == QUOTE)		
				{
					// eat quote and reset quote mode
					inptr++;
					quoting = 0;
					continue;
				}
			}
		
			// escapes?
			if (*inptr == '\\' && *(inptr+1) == '\\')
			{
				// for some bizarre reason, VC treats
				// escapes literally except when there is a
				// quote mark following.
				int count = 2;
				while (inptr[count] == '\\')
					count++;
					
				// followed by quote: halve the escapes
				if (inptr[count] == QUOTE)
				{
					// only deal with the even ones, and
					// pick up \" later
					count &= ~1;
					while (count)
					{
						*outptr++ = *inptr;
						inptr += 2;
						count -= 2;
					}
				}
				else 
				// nothing special, copy verbatim
				{
					while (count--)
						*outptr++ = *inptr++;
				}
			}
			else
			// escaped quote?
			if (*inptr == '\\' && *(inptr+1) == QUOTE)
			{
				*outptr++ = QUOTE;
				inptr += 2;
			}
			else
			// argument _TCHAR	
			{
#if _MSL_RUNTIME_EXPAND_WILDCARDS			
				if (!quoting && (*inptr == '*' || *inptr == '?'))
					wild = 1;
#endif					
				*outptr++ = *inptr++;
			}
		}
		
#if _MSL_RUNTIME_EXPAND_WILDCARDS
		if (wild)
		{
			*outptr = 0;	/*- EJS 040109 -*/
			_ExpandArgument();
		}
#endif

	}
	
	//	end last argument string
	*outptr = 0;
	
	//	terminate argv[] list
	if (__argc == 1)
		__argv1_offs = inptr - clptr;
	__targv[__argc] = NULL;
	
	//	free memory at exit
	atexit(_FreeCommandLine);
	
#else	// win32

	__argc = 0;
	__targv = (_TCHAR **)malloc(sizeof(_TCHAR *) * (__argc + 1));
	
	//  utter failure?
	if (__targv == NULL)
	{
		static _TCHAR *fakeargs[] = { NULL };
		__targv = fakeargs;
	}
	else
		__targv[__argc] = NULL;
	
#endif	// win32
}

/* Change record
 * mm  980730  Made sure that argv[argc] = NULL, C Std 5.1.2.2.1  MW08173
 * blc 980908  Fixed problem with no memory getting allocated for __targv on WinCE
 * ejs 990304  Rewrote to allow escaped quotes and to be more efficient
 * blc 990524  Fix for code stomping on global return value of GetCommandLine()
 * ejs 990917  Added more characters to whitespace stripping and clarified algorithm
 *             Also, support Unicode build.
 * ejs 000329  Added parsing of double quote marks, added commentary
 * ejs 001120  Added wildcard expansion
 * ejs 011015  Don't escape * or ? with \ (think about c:\dir\*.obj)
 * ejs 030721  Add UNICODE support
 * ejs 030805  Merge in wildcard expansion fixes from Carsten Hansen
 * ejs 031114  When a filename is untranslatable, use the alternate name instead
 * ejs 040109  Make previous fix work better ('?' doesn't necessarily indicate untranslatable)
 * ejs 040109  Fix missing trailing nul before wildcard expansion
 * ejs 040518  Do _tSetupArgs() work only once, and clean up more safely
 */
