/* Metrowerks Standard Library
 * Copyright  1995-2004 Metrowerks Corporation.  All rights reserved.
 *
 * $Date: 2004/01/26 21:32:33 $
 * $Revision: 1.7 $
 */

#ifndef _MSL_FENV_X87_H
#define _MSL_FENV_X87_H

#ifndef _MSL_FENV_H
#error This header may only be included from <fenv.h>
#endif

#if _MSL_USE_INLINE

#pragma only_std_keywords off 	/* allow asm */
#pragma volatile_asm off		/* optimize inline assembly */

/*- EJS 031217: add SSE support for exceptions and rounding modes.
	(Precision is not supported since SSE uses explicit data types.)
	When compiled into MSL, these routines support x87 as normal
	and only support SSE when the CPU supports it (based on cpuid).
	When this module is included in user code, the SSE or SSE-2 
	options must be enabled to include the SSE code, and no
	processor check occurs. */
	
/*	from crtl.h 	*/
#define _CPU_FEATURE_SSE		0x02000000
#define _CPU_FEATURE_SSE_2		0x04000000

/*******************************************************************************
*     The function "feclearexcept" clears the supported exceptions represented *
*     by its argument.                                                         *
*******************************************************************************/

_MSL_INLINE void _MSL_MATH_CDECL feclearexcept ( int flags )
{
	char env[28];
	asm
	{
			mov       edx, flags
			not       edx
			fnstenv   env
			and       word ptr [env+0x4],dx		/* modify status word */
			fldenv    env
#if _MSL_FENV_IMPL || __option(sse) || __option(sse2)
#if _MSL_FENV_IMPL			
			test      procflags, _CPU_FEATURE_SSE+_CPU_FEATURE_SSE_2
			jz        no_sse
#endif			
			stmxcsr	  [env]
			sal       edx, 7
			or        edx, 0x7f
			and       [env], edx
			ldmxcsr   [env]
	no_sse:
#endif	
	}
}

/* the most recent api in the C9X draft changed from feget/setexcept to 
   feget/setexceptflag to distinguish these two api's from the others in that
   these deal with the entire exception environment(i.e. they use/update the
   fexcept_t argument passed in).  Currently fexcept_t is just the status word. 
   This data structure may be extended in the future to contain other information
   such as a field containing the address of the code that first raised a particular
   exception.
   
   Apple hasn't updated their implementation as of the 3.0.1 
   Universal headers.  We reserve the old names on INTEL for portability. These
   might go away once apple updates their implementation.
   mf-  11/26/97
*/   

/* currently the variable fexcept_t is just the status word. For this reason fegetexceptflag
   is functionally equivalent to fetestexcept.  fetestexcept is slightly more efficient.
*/   
_MSL_INLINE void _MSL_MATH_CDECL fegetexceptflag ( fexcept_t *except, int flags)
{
	asm 
	{
			mov       ecx,dword ptr except
			fnstsw    ax
#if _MSL_FENV_IMPL || __option(sse) || __option(sse2)			
#if _MSL_FENV_IMPL			
			test      procflags, _CPU_FEATURE_SSE+_CPU_FEATURE_SSE_2
			jz        no_sse
#endif			
			sub       esp, 4
			stmxcsr	  [esp]
			mov       ecx, [esp]
			shr       ecx, 7
			or		  eax, ecx
			add       esp, 4
	no_sse:
#endif	
			and       eax,dword ptr flags
			and       eax,0x3d
			mov       dword ptr [ecx],eax
	}
}

/*******************************************************************************
*     The function "fesetexcept" or "fesetexceptflag"sets or clears the        *
*     exception flags indicated                                                *
*     by the int argument "excepts" according to the representation in the     *
*     object pointed to by the pointer argument "flagp".  The value of         *
*     "*flagp" must have been set by a previous call to "fegetexcept".         *
*     This function does not raise exceptions; it just sets the state of       *
*     the flags.                                                               *
*******************************************************************************/   

_MSL_INLINE void _MSL_MATH_CDECL fesetexceptflag ( const fexcept_t *except, int flags)
{
	char	env[28];
	asm
	{
			movzx     ax,word ptr flags
			mov       ecx,dword ptr except
			and       ax,0x3d
			and       ax,word ptr [ecx]
			fnstenv   env
			or        word ptr [env+0x4],ax
			fldenv    env
#if _MSL_FENV_IMPL || __option(sse) || __option(sse2)			
#if _MSL_FENV_IMPL			
			test      procflags, _CPU_FEATURE_SSE+_CPU_FEATURE_SSE_2
			jz        no_sse
#endif			
			sub       esp, 4
			stmxcsr	  [esp]
			sal       eax, 7
			or		  [esp], eax
			add       esp, 4
			ldmxcsr   [esp]
	no_sse:
#endif	
	}
}

_MSL_INLINE void _MSL_MATH_CDECL feraiseexcept (int flags)
{
	char 	env[28];
	asm
	{
			mov       ax,word ptr flags
			fnstenv   env
			and       ax,0x3d
			or        word ptr [env+0x4],ax
			fldenv    env
#if _MSL_FENV_IMPL || __option(sse) || __option(sse2)			
#if _MSL_FENV_IMPL			
			test      procflags, _CPU_FEATURE_SSE+_CPU_FEATURE_SSE_2
			jz        no_sse
#endif			
			sub       esp, 4
			stmxcsr	  [esp]
			sal       eax, 7
			or		  [esp], eax
			ldmxcsr   [esp]
			add       esp, 4
	no_sse:
#endif	
	}
}

/*******************************************************************************
*     The function "fetestexcept" determines which of the specified subset of  *
*     the exception flags are currently set.  The argument "excepts" specifies *
*     the exception flags to be queried as a bitwise OR of the exception       *
*     macros.  This function returns the bitwise OR of the exception macros    *
*     corresponding to the currently set exceptions included in "excepts".     *
*******************************************************************************/

_MSL_INLINE int _MSL_MATH_CDECL fetestexcept ( int flags )
{
	register int ret;
	asm
	{
			mov       dx,word ptr flags
			fnstsw    ax
#if _MSL_FENV_IMPL || __option(sse) || __option(sse2)			
#if _MSL_FENV_IMPL			
			test      procflags, _CPU_FEATURE_SSE+_CPU_FEATURE_SSE_2
			jz        no_sse
#endif			
			sub       esp, 4
			stmxcsr	  [esp]
			mov       ecx, [esp]
			sal       ecx, 7
			or		  ax, cx
			add       esp, 4
	no_sse:
#endif	
			and       ax,dx
			movzx     ret, al
	}
	return ret;
}

/*******************************************************************************
*     The following functions provide control of rounding direction modes.     *
*******************************************************************************/

/*******************************************************************************
*     The function "fegetround" returns the value of the rounding direction    *
*     macro which represents the current rounding direction.                   *
*******************************************************************************/
_MSL_INLINE int _MSL_MATH_CDECL fegetround ( void )
{
	short TEMP;
	register int ret;
	asm
	{
			fnstcw    TEMP
			and       word ptr TEMP,0xc00
			mov       ax,word ptr TEMP
			and       ax,0xc00
			movzx	  ret, ax
#if _MSL_FENV_IMPL || __option(sse) || __option(sse2)			
			/* the SSE mode should be the same as the x87 mode according to fesetround */
#endif	
			
	}
	return ret;
}

/*******************************************************************************
*     The function "fesetround" establishes the rounding direction represented *
*     by its argument.  It returns nonzero if and only if the argument matches *
*     a rounding direction macro.  If not, the rounding direction is not       *
*     changed.                                                                 *
*******************************************************************************/

_MSL_INLINE int _MSL_MATH_CDECL fesetround ( int flags )
{
	long TEMP;
	register int ret;
	asm
	{
			mov       edx,dword ptr flags
			mov       eax,edx
			test      eax,0xfffff3ff
			mov       ret,0x0
			jne       skip
			fstcw     word ptr TEMP
			and       word ptr TEMP,0xf3ff
			or        word ptr TEMP,dx
			fldcw     word ptr TEMP
#if _MSL_FENV_IMPL || __option(sse) || __option(sse2)			
#if _MSL_FENV_IMPL			
			test      procflags, _CPU_FEATURE_SSE+_CPU_FEATURE_SSE_2
			jz        no_sse
#endif			
			stmxcsr	  dword ptr TEMP
			and       dword ptr TEMP,~(0x6000)
			sal       edx, 3
			or        dword ptr TEMP,edx
			ldmxcsr   dword ptr TEMP
	no_sse:
#endif	
			mov       ret,0x1
		skip:
	}
	return ret;
}

/*******************************************************************************
*    The following functions manage the floating-point environment, exception  *
*    flags and dynamic modes, as one entity.                                   *
*******************************************************************************/

/*******************************************************************************
*     The function "fegetenv" stores the current floating-point environment    *
*     in the object pointed to by its pointer argument "envp".                 *
*******************************************************************************/

_MSL_INLINE void _MSL_MATH_CDECL fegetenv ( fenv_t *envp )
{
	char env[28];
	asm 
	{
			mov       edx,dword ptr envp
			fnstenv   env
			movzx     eax,word ptr [env+0x4]
			sal       eax,0x10
			mov       ax,word ptr [env]
			mov       dword ptr [edx],eax
	}
}

/*******************************************************************************
*     The function "feholdexcept" saves the current environment in the object  *
*     pointed to by its pointer argument "envp", clears the exception flags,   *
*     and clears floating-point exception enables.  This function supersedes   *
*     the SANE function "procentry", but it does not change the current        *
*     rounding direction mode.                                                 *
*******************************************************************************/

_MSL_INLINE int _MSL_MATH_CDECL feholdexcept ( fenv_t *envp )
{
	char	env[28];
	register int 	ret;
	asm
	{
			mov       edx,dword ptr envp
			fnstenv   [env]
			mov       ax,word ptr [env+0x4]
			sal       eax,0x10
			mov       ax,word ptr [env]
			mov       dword ptr [edx],eax
			and       word ptr [env+0x4],0xffffffc0
			or        word ptr [env],0x3f
			fldenv    [env]
			mov       ret,0x1
	}
	return ret;
}

/*******************************************************************************
*     The function "fesetenv" installs the floating-point environment          *
*     environment represented by the object pointed to by its argument         *
*     "envp".  The value of "*envp" must be set by a call to "fegetenv" or     *
*     "feholdexcept", by an implementation-defined macro of type "fenv_t",     *
*     or by the use of the pointer macro FE_DFL_ENV as the argument.           *
*******************************************************************************/

_MSL_INLINE void _MSL_MATH_CDECL fesetenv ( const fenv_t *envp )
{
	char	env[28];
	asm
	{
			mov       ecx,dword ptr envp
			mov       ecx,dword ptr [ecx]
			fnstenv   [env]
			mov       word ptr [env],cx
			shr       ecx,0x10
			and       cx,0x3f
			and       word ptr [env+0x4],0xffffffc0
			or        word ptr [env+0x4],cx
			fldenv    [env]
	}
}

/*******************************************************************************
*     The function "feupdateenv" saves the current exceptions into its         *
*     automatic storage, installs the environment represented through its      *
*     pointer argument "envp", and then re-raises the saved exceptions.        *
*     This function, which supersedes the SANE function "procexit", can be     *
*     used in conjunction with "feholdexcept" to write routines which hide     *
*     spurious exceptions from their callers.                                  *
*******************************************************************************/
      
_MSL_INLINE void _MSL_MATH_CDECL feupdateenv ( const fenv_t *envp  )
{
	char	env[28];
	asm
	{
			mov       edx,dword ptr envp
			fnstenv   [env]
			mov       ecx,dword ptr [edx]
			mov       word ptr [env],cx
			shr       ecx,0x10
			and       cx,0x3f
			or        word ptr [env+0x4],cx
			fldenv    [env]
	}
}

/*******************************************************************************
*     The following functions provide control of rounding precision.           *
*     Because the PowerPC does not provide this capability, these functions    *  
*     are available only for the 68K Macintosh and X86. Rounding precision     *
*     values are defined by the rounding precision macros.  These functions are*
*     equivalent to the SANE functions getprecision and setprecision.          *
*     These are an optional part of the C9X draft standard--m.f. 7/15/97       *
*******************************************************************************/

_MSL_INLINE int _MSL_MATH_CDECL fegetprec ( void )
{
	short 	TEMP;
	asm
	{
			fnstcw    TEMP
			and       word ptr TEMP,0x300
			movzx     eax,word ptr TEMP
			and       eax,0x300
#if _MSL_FENV_IMPL || __option(sse) || __option(sse2)			
			/* no control of SSE precision */
#endif	
	}
}

_MSL_INLINE int _MSL_MATH_CDECL fesetprec ( int flags )
{
	long 	TEMP;
	register int 	ret;
	asm
	{
			mov       edx,dword ptr flags
			mov       eax,edx
			test      eax,0xfffffcff
			mov       ret,0x0
			jne       skip
			fstcw     word ptr TEMP
			and       word ptr TEMP,0xfcff
			or        word ptr TEMP,dx
			fldcw     word ptr TEMP
#if _MSL_FENV_IMPL || __option(sse) || __option(sse2)			
			/* no control of SSE precision */
#endif	
			mov       ret,0x1
		skip:
	}
	return	ret;
}

#pragma volatile_asm reset		/* reset inline assembly optimization */
#pragma only_std_keywords reset /* reset keywords */

#endif	/* _MSL_USE_INLINE */

#endif	/* _MSL_FENV_X87_H */

/* Change record:
 * EJS 020124 Moved out of object file
 * JWW 020311 Changed _MSL_CDECL to _MSL_MATH_CDECL
 * EJS 031217 Add SSE support for exception and precision control
 * EJS 040109 Guard inlines with #if _MSL_USE_INLINE
 */