#include "make.h"

#include <assert.h>

#include "job.h"
#include "debug.h"
#include "filedef.h"
#include "commands.h"
#include "variable.h"
#include "dep.h"
#include "cvs.h"
#include "rcs.h"
#include "tsp.h"

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <regex.h>
#include <limits.h>

#include <time.h>

#ifndef FDOUT
#  define FDOUT 5  
#endif
#ifndef DEPS_DIR
#  define DEPS_DIR ".autodeps"
#endif
#ifndef DEPS_EXT
#  define DEPS_EXT ".P"
#endif

int file_audit = 0;
int autodepend = 0;

/*
  used_file management;
*/

enum state {
  s_init = 0,
  s_source,
  s_target,
  s_update,
  s_exe,
  s_intermediate,
  s_makefile,
  s_ffu,
  s_unknown1,
  s_unknown2
};

typedef struct _uf{
   struct _uf *next; /* daisy chain */
   char *name;	     /* filename */
   char *audit_method;     /* magic type (MD5,CVS,...)*/
   char magic[256];     /* magic value */
   enum state use;   /* state machine */
   char	set_audit:1; /* can generate audit on it */
   char	set_clean:1; /* can be cleaned */
} used_file;

typedef struct _df{
  struct _df *next;      /* daisy chain */
  int fd;                /* stream to get dependencies if local (not remote) */  
  used_file *first_file; /* list of used file */
  used_file *last_file;  /* tail of above list */
  struct file *file;     /* struct file reference */
  int remote_id;      	 /* remote id if remote is enabled */
}depend_file;

depend_file *depend_first=NULL; /* list of dependencies on going */


char *audit_pref_order = "CVS:MD5";
struct variable *var_audit=NULL;

/* 
   alloc a used_file element 
*/
used_file *alloc_file() {
  used_file *nouvo;
  nouvo = (used_file*) xmalloc(sizeof(used_file));
  return nouvo;
}
/*
  free a used_file list
*/
void free_file(used_file* list){
  used_file *next;
  for(;list;list=next){
    next = list->next;
    free (list->name);
    free (list);
  }
}
/*
  Find a file in the used_file list
  or add one if not found
*/

used_file *getfile(depend_file *df,char * name) {
  used_file *ptr;
  for (ptr = df->first_file;ptr;ptr = ptr->next) 
    if (!strcmp(ptr->name,name)) break;
    
  if (ptr == NULL) {
    ptr = alloc_file();
    
    ptr->next = NULL;
    if (df->first_file == NULL){
      df->first_file = df->last_file = ptr;
    } else {
      df->last_file->next = ptr;
      df->last_file = ptr;
    }
    ptr->name = strdup(name);
    ptr->use = s_init;
    
    ptr->audit_method = NULL;
    ptr->set_audit = 0;
    ptr->set_clean = 0;
  }
  return (ptr);
}


/*
  Play with the state machine 
*/
void manage(used_file *sfile,char *action){
int old_use = sfile->use;
  switch(sfile->use) {
  case s_init:
    if      (action[0] == 'w') sfile->use = s_target;  /* let say it's a target (even if w+ )*/
    else if (action[1] == '+') sfile->use = s_update;  /* it seems like a update */
    else if (action[0] == 'r') sfile->use = s_source;  /* let say it's a source */
    else if (action[0] == 'x') sfile->use = s_exe;     /* funny a exec */
    else if (action[0] == 'u') sfile->use = s_intermediate; /* Hum it can't be a source */
    else if (action[0] == 'm') sfile->use = s_makefile; /* Oh a Makefile */
    else                       sfile->use = s_ffu;	/* For futur use ? */
    break;
  case s_source:
    if      (action[1] == '+') sfile->use = s_update;  /* No it was a update */
    else if (action[0] == 'w') sfile->use = s_update;  /* read then write -> update */
    else if (action[0] == 'r') sfile->use = s_source;  /* it stills a source */
    else if (action[0] == 'x') sfile->use = s_exe;
    else if (action[0] == 'u') sfile->use = s_intermediate; /* we can't delete a source ! */
    else if (action[0] == 'm') sfile->use = s_makefile;
    else                       sfile->use = s_ffu;
    break;
  case s_target:
    if      (action[1] == '+') sfile->use = s_target;  /* stills a target  */
    else if (action[0] == 'w') sfile->use = s_target;  /* Oh same player shoot again ? */
    else if (action[0] == 'r') sfile->use = s_target;  /* If it's intermediate we will see a unlink later. Don'move yet */ 
    else if (action[0] == 'x') sfile->use = s_intermediate; /* why not */
    else if (action[0] == 'u') sfile->use = s_intermediate; /* Ok this one was a indermediate file */
    else if (action[0] == 'm') sfile->use = s_makefile;
    else                       sfile->use = s_ffu;
    break;
  case s_update:
    if      (action[1] == '+') sfile->use = s_update;  /* Ok go on */
    else if (action[0] == 'w') sfile->use = s_update;  /* Funny ! don't move */
    else if (action[0] == 'r') sfile->use = s_update;  /* d'ont move */
    else if (action[0] == 'x') sfile->use = s_intermediate;
    else if (action[0] == 'u') sfile->use = s_intermediate;
    else if (action[0] == 'm') sfile->use = s_makefile;
    else                       sfile->use = s_ffu;
    break;
  case s_exe:
    if      (action[1] == '+') sfile->use = s_intermediate;
    else if (action[0] == 'w') sfile->use = s_intermediate;
    else if (action[0] == 'r') sfile->use = s_exe;
    else if (action[0] == 'x') sfile->use = s_exe;
    else if (action[0] == 'u') sfile->use = s_intermediate;
    else if (action[0] == 'm') sfile->use = s_makefile;
    else                       sfile->use = s_ffu;
    break;
  case s_makefile:
    break;
  case s_intermediate:
    if      (action[1] == '+') sfile->use = s_update;
    else if (action[0] == 'w') sfile->use = s_target;
    else if (action[0] == 'r') sfile->use = s_intermediate; /* it c'ant be a source ! */
    else if (action[0] == 'x') sfile->use = s_intermediate;
    else if (action[0] == 'u') sfile->use = s_intermediate;
    else if (action[0] == 'm') sfile->use = s_makefile;
    else                       sfile->use = s_ffu;
    break;
  default:
    break;
  }

 /* printf (" >>%s %s %d -> %d\n", sfile->name,action,old_use,sfile->use);*/
}

/*
  mygets with a fd and not a FILE*
*/
static int mygets(char *buffer, int lg,int fd){
  int pos;
  char c;
  int ret;
  
  for(pos = 0;pos <lg;pos++) {
   ret = read(fd,&c,1);
   if ( ret == 1){
     if (c == '\n') { /* end of line */
       buffer[pos] = 0;
       return pos;
     } else if (c == '\r'){
       /* forget it*/
     } else {
        buffer[pos] = c;
     }
   } else { /* cant read -> eof or error */
     buffer[pos] = 0; 
     return ret;
   }
  }
  /* line to long */
  buffer[pos-1] = 0; 
  return pos;

}
/*
  get a dependency from the stream
  find the file
  play with its state machine
  
  if toeof replay until end of file !
*/
void catch_access(depend_file *df,int toeof)
{
    do {
      char buffer[256];
      char *action,*file;
    
      used_file *sfile;
    
      if (mygets(buffer, 256, df->fd) <= 0) {
        break;
      }
      
      action = buffer;
      
      if (action) {

        file = strchr(action,' ');
        if (file) {
          *file = 0;
          file++;
      
          sfile = getfile(df,file);
          if (sfile) manage(sfile,action);
        }
      }
    } while(toeof);
}

/*
  Regex management
*/
typedef struct reg{
   struct reg *next;
   char *expr;
   regex_t compiled;
} RegList;

RegList regs[] = {
{&regs[1],"-^[\\/\\.]*$",0}, /* do not trace  ../.. .. . etc */
{&regs[2],"-^.*\\.autodeps\\/.*\\.P$",0}, /* do not trace autodep files ! */
{&regs[3],"-^\\/var.*$",0},/* do not trace temp files ! */
{&regs[4],"-^\\/tmp.*$",0},/* do not trace temp files ! */
{&regs[5],"-.*include.*\\.h$",0},/* trace includes */
{&regs[6],"-^\\/.*$",0}, /* do not trace other not local files */
{0       ,"+^.*$",0}, /* trace others */
};

RegList *reg_first = regs;
/*
  compile regex and forget bad one
*/
void compile_regex(){
  RegList *ptr,*last_ok=NULL;
  for (ptr = reg_first;ptr;ptr = ptr->next) {
     if (regcomp(&ptr->compiled,&ptr->expr[1],REG_NOSUB)==0) {
       if (last_ok) last_ok->next = ptr;
       last_ok = ptr;
     } else {
        fatal(NILF,_("eg Error %s\n"),ptr->expr);
     }
       
  }
  if (last_ok) last_ok->next = 0;
}
/*
  try all regex
  if it a '-' one stop if it matches
*/
int exec_regex(char *string){
  RegList *ptr;
  for (ptr = reg_first;ptr;ptr = ptr->next){
    if (regexec(&ptr->compiled, string,0,0,0) == 0) break;
  }
  if (ptr == 0) return 0;
  if (ptr->expr[0] == '-') return 0;
  return 1;
}

/* 
  MD5 computation
*/
#include "md5.h"
int get_md5 (filename,hexkey)
char *filename;
char *hexkey;
{
  int file;
  MD5_CTX context;
  int len,i;
  unsigned char buffer[1024],digest[16];
  static char hex[]="0123456789ABCDEF";
  
  
  hexkey[0]=0;
  file = open (filename, O_RDONLY);
  if (file <= 0)
    return 0;

  MD5Init (&context);
  while ((len = read (file,buffer,  256))>0){
    MD5Update (&context, buffer, len);
  }
 
  MD5Final (digest, &context);

  close (file);
  
  for (i = 0; i < 16; i++){
    hexkey[i*2]=hex[(digest[i] & 0xF0) >> 4];
    hexkey[i*2+1]=hex[digest[i] & 0x0F];
  }
  
  hexkey[32] = 0; 
  return 1;
}

/*
  Depends maker
  
  for all target files 
    write source files 
  
  for all target 
    fill the TARGETS list
    
  for all target and source
    write the MD5 code
  
*/
void gen_deps(df,file)
 depend_file *df;
 struct file *file;
{
  used_file *tgt;
  used_file *tgt2;
  used_file *src;
  int line_end;
  int fdout=0;
  RegList *reg_p;
    
  for (tgt = df->first_file;tgt;tgt=tgt->next) { 
    if((tgt->use == s_target) || (tgt->use == s_update)) {                                
      /*
      for (reg_p = reg_first;reg_p;reg_p=reg_p->next) {
        if (regexec(&reg_p->compiled, tgt->name,0,0,0) == 0) break;
      }
      if (reg_p == 0) continue;
      if (reg_p->expr[0] == '-') continue;
	*/
      if (exec_regex (tgt->name) == 0) continue;

      if (fdout == 0) {
        char filename[1024];
	time_t today;
	char *d;
	
	sprintf(filename,"%s/%s%s",DEPS_DIR,file->name,DEPS_EXT);
	
	if (access(DEPS_DIR,0777))
          mkdir(DEPS_DIR,0777);
        
/*	if (tgt->use == s_update)
          fdout = open(filename,O_WRONLY|O_CREAT|O_APPEND,0666);
	else*/
          fdout = open(filename,O_WRONLY|O_CREAT|O_TRUNC,0666);
	if (fdout) {
	  write(fdout,"# Prerequisites generated by ",29);
	  write(fdout,program,strlen(program));
	  write(fdout,"\n",1);
	} else perror("open .P");
      }

      
      if (fdout > 0) {
        int i;
	write(fdout,tgt->name,strlen(tgt->name));
	/*
  	for (tgt2 = tgt->next;tgt2;tgt2=tgt2->next) { 
          if((tgt2->use == s_target) || (tgt2->use == s_update)) {                                
            for (reg_p = reg_first;reg_p;reg_p=reg_p->next) {
              if (regexec(&reg_p->compiled, tgt2->name,0,0,0) == 0) break;
            }
            if (reg_p == 0) continue;
            if (reg_p->expr[0] == '-') continue;
	    tgt = tgt2;
	    write(fdout," + ",3);
	    write(fdout,tgt->name,strlen(tgt->name));
	  }
	}
	*/
/*	if (tgt->use == s_update)
	  write(fdout,"::",2);
	else*/
	
	  write(fdout,":",1);
	
	line_end = 0;
	
	tgt->set_clean = 1;
	
	for (src = df->first_file;src;src=src->next)
	  if(src->use == s_source) {
	    
	    /*
	    for (reg_p = reg_first;reg_p;reg_p=reg_p->next) {
              if (regexec(&reg_p->compiled, src->name,0,0,0) == 0) break;
            }
	    if (reg_p == 0) continue;
            if (reg_p->expr[0] == '-') continue;
	    */
	    if (exec_regex (src->name) == 0) continue;

	    if (line_end) write(fdout," \\\n\t",4);
	    else write(fdout,"\t",1);
	    
            write(fdout,src->name,strlen(src->name));
	    line_end=1;
	    src->set_audit = 1;
	  }
	write(fdout,"\n",1);
      }
      
      if (fdout > 0) {
        for (src = df->first_file;src;src=src->next)
	  if(src->use == s_exe) {
	    write(fdout,"#\t",2);
	    write(fdout,src->name,strlen(src->name));
	    write(fdout,"\n",1);
	    
	  }
      }
      
    }
  } 

  if (fdout>0) {
      line_end = 0;
 
      write(fdout,"\n# Build Target list variable\n\n",31);
 
      for (tgt = df->first_file;tgt;tgt=tgt->next)
        if(tgt->set_clean){
  	  write(fdout,"TARGETS += ",11);
  	  write(fdout,tgt->name,strlen(tgt->name));
	  write(fdout,"\n",1);
    	}
      write(fdout,"\n\n",2);
  }
  
  if (file_audit && (fdout>0)) {
      line_end = 0;
 
      write(fdout,"\n# File audit rules\n\n.AUDIT:",28);
 
      for (src = df->first_file;src;src=src->next)
    	if(src->set_audit) {
	  char *audit_method= strdup (audit_pref_order);
	  char *audit_method_sav = audit_method;
	  for (audit_method = strtok(audit_method,": \t");audit_method;audit_method = strtok(NULL,": \t")) {
	  
	    if (!strcmp(audit_method,"CVS")) {
	      char cdate[50],wdate[50];
      	      if (cvs_get_cvsinfo(src->name,cdate,src->magic) && 
	        cvs_get_workdate(src->name,wdate) && 
		!strcmp(wdate,cdate)){
 	          src->audit_method = "CVS";
		  break;
	       }
	    }
	  
	    if (!strcmp(audit_method,"MD5")) {
	      src->audit_method = "MD5";
  	      get_md5(src->name,src->magic);
	      break;
	    }
	    
	    if (!strcmp(audit_method,"CC")) {
  	      if (cc_get_version(src->name,src->magic) == 0 ) {
	        src->audit_method = "CC";
	        break;
	      }
	    }
	    
	    if (!strcmp(audit_method,"TSP")) {
	      src->audit_method = "TSP";
  	      tsp_get_timestamp(src->name,src->magic);
	      break;
	    }
	    
	    if (!strcmp(audit_method,"RCS")) {
	      if (!rcs_get_version(src->name,src->magic) && 
	          !rcs_diff(src->name,src->magic)) {
	        src->audit_method = "RCS";
 	        break;
	      }
	    }
	  }
	  free ( audit_method_sav );
	  
	  if (src->audit_method) {
  	    if (line_end) write(fdout," \\\n\t",4);
  	    else write(fdout,"\t",1);
  	    write(fdout,src->audit_method,strlen(src->audit_method));
  	    write(fdout,"_",1);
  	    write(fdout,src->magic,strlen(src->magic));
    	    line_end = 1;
	  }
    	}
      write(fdout,"\n\n",2);
    
 
      for (src = df->first_file;src;src=src->next)
    	if(src->audit_method) {
  	  write(fdout,src->audit_method,strlen(src->audit_method));
 	  write(fdout,"_",1);
  	  write(fdout,src->magic,strlen(src->magic));
  	  write(fdout,": ",2);
  	  write(fdout,src->name,strlen(src->name));
  	  write(fdout,"\n",1);
	}
   
  }
  
  if (fdout>0) 
  	close(fdout);
}


/* 
  Make anchor
*/
#ifdef HAVE_WAITPID
# define WAIT_NOHANG(status)	waitpid (-1, (status), WNOHANG)
#else	/* Don't have waitpid.  */
# ifdef HAVE_WAIT3
#  ifndef wait3
extern int wait3 ();
#  endif
#  define WAIT_NOHANG(status)	wait3 ((status), WNOHANG, (struct rusage *) 0)
# endif /* Have wait3.  */
#endif /* Have waitpid.  */

pid_t dep_wait(int *stat_loc){
  pid_t pid;
  fd_set fd_inputs;
  depend_file *df;
  int max=0;

  if (depend_first) {
    FD_ZERO(&fd_inputs);
    for (df = depend_first;df;df=df->next) {
      if (max < df->fd) max = df->fd;
      FD_SET(df->fd,&fd_inputs);
    }

    for(;;){
      int selret;
      struct timeval timeout={1,0};
      if ((pid = WAIT_NOHANG (stat_loc))) return (pid);

      selret = select(max+1,&fd_inputs,NULL,NULL,&timeout);

      if (selret <0) break;
      for (df = depend_first;df;df=df->next) {
    	if (FD_ISSET(df->fd,&fd_inputs))
    	  catch_access(df,0);
      }
    }
  }

  pid=wait(stat_loc);

  return (pid);
}

char lib_env[256]="";
autodepend_init(read_makefiles)
struct dep *read_makefiles;
{
  struct variable *v;
  struct dep *makefile;

  v = define_variable ("AUDIT_METHOD", 12, audit_pref_order, o_default, 0);	 
  v->export = v_export;
  
  
  if (autodepend) {
     
     cvs_clean_cache();
     
     var_audit = lookup_variable ("AUDIT_METHOD", 12);
     if (var_audit) {
       audit_pref_order = var_audit->value;
     }

    if (lib_env[0] == 0) {
      sprintf(lib_env,"make_ad.so",program); 
    } 
    
    compile_regex();
    
    if (makelevel) {
      write(FDOUT,"m $(MAKE) -C ",13);
      write(FDOUT,starting_directory,strlen(starting_directory));
    
      for (makefile = read_makefiles;makefile;makefile = makefile->next)
       if (strncmp(makefile->file->name,DEPS_DIR,sizeof(DEPS_DIR)-1)){
         write(FDOUT," -f ",4);
         write(FDOUT,makefile->file->name,strlen(makefile->file->name));
       }
      write(FDOUT,"\n",1);
    }
    v = define_variable ("LD_PRELOAD", 10, lib_env, o_env, 0);
    v->export = v_export;
  }
}

void 
autodepend_start(file)
struct file *file;
{
  if (autodepend) {
    depend_file *df;
    df = (depend_file *) xmalloc(sizeof(depend_file));

    if (pipe(file->autodepend_fd)==0){
      df->next = depend_first;
      depend_first = df;
      df->fd = file->autodepend_fd[0];
      df->file = file;
      df->first_file = 0;
      df->last_file = 0;
    } else {
      pfatal_with_name("dependency pipe");
    }
  }
}

void 
autodepend_end(file)
 struct file *file;
{ 
  if (autodepend) {
    depend_file *df,*odf;
    
    for (df = depend_first,odf=0;df->file != file;odf=df,df=df->next) if (df == 0) printf("Oups");
    
    if (odf)
      odf->next=df->next;
    else
      depend_first=df->next;

    close(file->autodepend_fd[1]);
 
    catch_access(df,1);
    close(file->autodepend_fd[0]);
    
    file->autodepend_fd[0] = 0;
    file->autodepend_fd[1] = 0;
    
/* add explicite dependencies to audit only*/    
    {
      struct dep *d;
      struct file *f;
      f=df->file;
      if (f->double_colon) f=f->double_colon;
      for (;f;f=f->prev) {
         for (d=f->deps;d;d=d->next) {
          used_file *sfile;
          if (exec_regex (d->file->name) == 0) continue;
	  
          sfile = getfile(df,d->file->name);
          if (sfile) sfile->set_audit=1;
        }
      }
    }

    gen_deps(df,file);
 
    free_file(df->first_file);
    free (df);
  }
}

void
autodepend_execute_job (child,stdin_fd, stdout_fd, argv, envp)
     struct child *child;
     int stdin_fd, stdout_fd;
     char **argv, **envp;
{  
  if (stdin_fd != 0)
    (void) dup2 (stdin_fd, 0);
  if (stdout_fd != 1)
    (void) dup2 (stdout_fd, 1);
  if (stdin_fd != 0)
    (void) close (stdin_fd);
  if (stdout_fd != 1)
    (void) close (stdout_fd);
   
  if (autodepend) {
    close(child->file->autodepend_fd[0]);
    (void)dup2(child->file->autodepend_fd[1],FDOUT);
  }
  /* Run the command.  */
  exec_command (argv, envp);
}

int autodepend_start_remote_job (struct child *child, char **argv, char **envp, int stdin_fd,
				 int *is_remote, int *id_ptr, int *used_stdin)
{
  int                     ret;
  depend_file            *df;

  ret = start_remote_job (argv, envp, stdin_fd, is_remote, id_ptr, used_stdin);
  if (autodepend) {
    for (df = depend_first; df; df = df->next) {
      if (df->file == child->file) {
	df->remote_id = *id_ptr;
	break;
      }
    }
    if (df == NULL) {
      fatal (NILF, _("eg Can't find Child\n"));
    }
  }
  return ret;
}

void remote_catch_access (int id, char *line)
{
  if (autodepend) {
    depend_file            *df;
    for (df = depend_first; df; df = df->next) {
      if (df->remote_id == id) {
	char                   *action, *file;
	used_file              *sfile;

	action = line;
	file = strchr (action, ' ');
	if (file) {
	  *file = 0;
	  file++;

	  sfile = getfile (df, file);
	  if (sfile)
	    manage (sfile, action);
	}

	return;
      }
    }
    fatal (NILF, _("eg Can't find Child\n"));
  }
}


int check_autodepend(char *filename,char * magic) {

  if (strncmp("MD5_",magic,4) == 0) {
    char key[33];
    if (get_md5(filename,key) == 0) return -1;
    if (strcmp(key,&magic[4])) return 1;
    return 0;
  }
  
  else if (strncmp("CC_",magic,3) == 0) {
    char key[256];
    if (cc_get_version(filename,key)) return -1;
    if (strcmp(key,&magic[3])) return 1;
    return 0;
  }

  else if (strncmp("TSP_",magic,4) == 0) {
    char key[50];
    if (tsp_get_timestamp(filename,key)) return -1;
    if (strcmp(key,&magic[4])) return 1;
    return 0;
  }

  else if (strncmp("CVS_",magic,4) == 0) {
    char version[50],wdate[50],cdate[50];
  
    if (cvs_get_cvsinfo(filename,cdate,version) == 0) return -1;
    if (strcmp(version,&magic[4])) return 1;
    
    if (cvs_get_workdate(filename,wdate) == 0) return -1;
    if (strcmp(wdate,cdate)) return 1;
    return 0;
  }
  else if (strncmp("RCS_",magic,4) == 0) {
    return rcs_diff(filename,&magic[4]) ;
  }
  
  return -1;
}
