/* Metrowerks Standard Library
 * Copyright  1995-2004 Metrowerks Corporation.  All rights reserved.
 *
 * $Date: 2005/10/18 21:39:14 $
 * $Revision: 1.20.2.2 $
 */

/*
 * Move memory buffer to memory buffer.  This function guarantees that overlapping
 * operands are handled correctly.  This is done by moving backwards with the source
 * is lower in memory than the destination so the source is not clobbered by earlier
 * moves to the dest.
 */

#ifndef _MSL_MEM_X86_H
#define _MSL_MEM_X86_H

void * _MSL_CDECL memmove(void * dest, const void * src, size_t n)
{
    asm
    {
    	// get size in eax
    	//	   dest in edi
    	//	   src  in esi
    	    	
		mov		eax, n
		mov		esi, src
		mov		edi, dest
		xor		ecx, ecx

		// if dest address > source address use backward moves
		// so we dont overwrite characters we have yet to move
		
		cmp		edi, esi
		ja		move_backwards
		
		// check for short moves, do them simply
		
		cmp		eax, 16
		jl		short_one
		
		// move enough bytes to align the dest
		
		sub		ecx, edi
		and		ecx, 3
		je		skip1
		sub		eax, ecx
		rep		movsb
	skip1:
		mov		ecx, eax
		and		eax, 3
		shr		ecx, 2
		rep		movsd

		test	eax, eax
		je		done

	short_one:
		mov		ecx, eax
		rep		movsb
	skip3:
		jmp		done
		
		align
		
	move_backwards:

		// set backwards moves
		
		std
		
		// point to the last byte in the strings
		
		lea		esi, [esi+eax-1]
		lea		edi, [edi+eax-1]
		
		// check for a short move
		
		cmp		eax, 16
		jl		short_backwards

		// move enough bytes to align the dest
		
		lea		ecx, [edi-3]
		and		ecx, 3
		je		skip4
		sub		eax, ecx

		rep		movsb

	skip4:

		// adjust addresses for dword moves
		
		sub		esi, 3
		sub		edi, 3

		// get count of dwords
		
		mov		ecx, eax
		and		eax, 3
		shr		ecx, 2		

		rep		movsd

		test	eax, eax
		je		skip6
		
		add		esi, 3
		add		edi, 3			
		
	short_backwards:
		mov		ecx, eax
		rep		movsb
	skip6:	
		cld

	done:
    }

	return dest;
}


// memcpy is the same as memmove, but does not worry about overlapping
// source and destination.  You get what you deserve (or if you are clever,
// what you want) if they overlap.

void * _MSL_CDECL memcpy(void * _MSL_RESTRICT dest, const void * _MSL_RESTRICT src, size_t n)
{
    asm
    {
    	// get size in eax
    	//	   dest in edi
    	//	   src  in esi
    	    	
		mov		eax, n
		mov		esi, src
		mov		edi, dest
		xor		ecx, ecx

		// check for short moves, do them simply
		
		cmp		eax, 16
		jl		short_one
		
		// move enough bytes to align the dest
		
		sub		ecx, edi
		and		ecx, 3
		je		skip1
		sub		eax, ecx
		rep		movsb
	skip1:
		mov		ecx, eax
		and		eax, 3
		shr		ecx, 2
		rep		movsd

		test	eax, eax
		je		skip3

	short_one:
		mov		ecx, eax
		rep		movsb
	skip3:
    }

	return dest;
}


// Set each byte in a memory buffer to a specific character

void * _MSL_CDECL memset(void * dest, int c, size_t n)
{
    asm 
    {
		mov		eax, c				// get source char
		xor		ecx, ecx
		mov		edi, dest	 		// get dest pointer
		mov		edx, n
		and		eax, 0xff			// zero extend it		
		cmp		edx, 16
		jbe		skip2
		
		mov		ah, al				// duplicate low byte		
		mov		ecx, eax			// get a copy
		cmp		edx, 4
		jle		skip2
		shl		ecx, 16				// dup low word in upper
		or		eax, ecx			// and combine with eax to get 4
									// copies of the byte
		xor		ecx, ecx
		
		
		// move enough bytes to align
		
		sub		ecx, edi
		and		ecx, 3
		je		skip1
		sub		edx, ecx
		
		rep		stosb
	
	skip1:
	
		// move as many longs as we can
		
		mov		ecx, edx
		and		edx, 3
		shr		ecx, 2
		je		skip2

		rep		stosd
		
	skip2:
		mov		ecx, edx
		rep		stosb

	skip3:
    }
    
    return dest;
}


// search a memory buffer for a specific character.  If found, the address
// of character is returned, if not 0 is returned.
#ifndef _WIN32_WCE_EMULATION  

void * _MSL_CDECL memchr(const void *src , int c, size_t n)
{
	register void *loc = 0;
	
	if (src == NULL)				/*- mm 000609 -*/
		return(NULL);				/*- mm 000609 -*/

	asm
	{
		mov		edi, src
		mov		eax, c
		mov		ecx, n

		repne	scasb		
		je		found
		jmp		end
		
		align
	found:
		lea		loc, -1[edi]

	end:
    }
    
    return loc;
}

#endif

// search a memory buffer backwards for a specific character.  If found, the address
// of character is returned, if not 0 is returned.

void * _MSL_CDECL __memrchr(const void * src, int c, size_t n)
{
	register void *loc = 0;

	if (src == NULL)				/*- mm 000609 -*/
		return(NULL);				/*- mm 000609 -*/

    asm
    {
		mov		edi, src
		mov		eax, c
		mov		ecx, n
		lea		edi, -1[edi][ecx]
		
		std
		repne	scasb		
		cld
		je		found
		jmp		end
				
	found:
		lea		loc, 1[edi]
		
	end:
	}
	
	return loc;
}


// compare two strings of bytes and return
//
//		negative if src1 < src2
//			   0 if src1 == src2
//		positive if src1 > src2

int _MSL_CDECL memcmp(const void *s1, const void *s2, size_t n)
{
	int result = 0;

	asm {
		mov		ecx, n					// get size
		mov		esi, s1					// get first string
		mov		edi, s2					// get second string
		
		test	ecx, ecx				// check for size 0 => result is 0
		je		end				
		
		repe	cmpsb					// do the compare
		
		movzx	eax, BYTE PTR -1[esi]	// subtract the last chars compared to get result
		movzx	ecx, BYTE PTR -1[edi]
		sub		eax, ecx
		mov		result, eax
	end:
	}
	
	return result;
}

#endif /* _MSL_MEM_X86_H */


/* Change record:
 * mf  980620 change for wince 
 * mf  980620 change for wince 
 * blc 990203 fix comparison routine to be unsigned compare 
 * blc 990309 changed to not use declspec(naked) to allow for debug code
 * blc 990310 added aligns to improve code gen for branch targets 
 * mm  000609 Added tests to memchr and __memrchr for NULL passed as string ptr
 * ejs 010320 Add "register" keyword to build at opt 0
 * cc  011203 Added _MSL_CDECL for new name mangling 
 * JWW 020917 Added _MSL_RESTRICT to get the restrict type specifier for certain C99 functions
 */