/* Metrowerks Standard Library
 * Copyright  1995-2004 Metrowerks Corporation.  All rights reserved.
 *
 * $Date: 2004/01/26 21:32:49 $
 * $Revision: 1.8 $
 */

#include <dirent.h>
#include <errno.h>
#include <file_io.h>
#include <stdlib.h>

#define WIN32_LEAN_AND_MEAN
#include <windows.h>

DIR *opendir(const char *spec) _MSL_CANT_THROW
{
	char        wildcard[NAME_MAX + 4], *wptr;
	int			len;
	DIR			*ref;
	
	if (!spec)
	{
		errno = EINVAL;
		return NULL;
	}

	len = strlen(spec);
	if (len >= NAME_MAX)
	{
		errno = ENAMETOOLONG;
		return NULL;
	}

	ref = (DIR *)malloc(sizeof(DIR));
	if (ref == NULL)
	{
		errno = ENOMEM;
		return NULL;
	}

	ref->_d__dirent = (struct dirent *)malloc(sizeof(struct dirent));
	if (ref->_d__dirent == NULL)
	{
		free(ref);
		errno = ENOMEM;
		return NULL;
	}

	/* get data structure for directory walk */
	ref->_d__ffd = (void *) malloc(sizeof(WIN32_FIND_DATA));
	if (ref->_d__ffd == NULL)
	{
		free(ref->_d__dirent);
		free(ref);
		errno = ENOMEM;
		return NULL;
	}

	/* form wildcard */
	strcpy(wildcard, spec);
	wptr = wildcard + len;
	
	/* ensure trailing backslash */
	if (*(wptr-1) != '\\')
	{
		*wptr++ = '\\';
	}

	*wptr++ = '*';
	*wptr++ = 0;

	/* remember wildcard so we can form full path */
	ref->_d__wildcard = __msl_strdup(wildcard);				/*- cc 010725 -*/

	/* open directory */
	ref->_d__handle = FindFirstFile(ref->_d__wildcard, (WIN32_FIND_DATA *) ref->_d__ffd);
	if (ref->_d__handle == INVALID_HANDLE_VALUE)
	{
		__set_errno(GetLastError());

		free(ref->_d__dirent);
		free(ref->_d__ffd);
		free(ref->_d__wildcard);
		free(ref);
		return NULL;
	}
	else
	{
		errno = ENOERR;
		return ref;
	}
}

struct dirent *readdir(DIR *ref) _MSL_CANT_THROW
{
	WIN32_FIND_DATA *ffd = (WIN32_FIND_DATA *) ref->_d__ffd;
	struct dirent *ent = ref->_d__dirent;
	
	if (ref->_d__handle == INVALID_HANDLE_VALUE)
	{
		errno = ENOENT;
		return NULL;
	}
	else 
	{
		/*  we already have one entry cached */
		if (strlen(ffd->cFileName) < NAME_MAX)
			strncpy(ent->d_name, ffd->cFileName, NAME_MAX);
		else
			strncpy(ent->d_name, ffd->cAlternateFileName, NAME_MAX);
		ent->d_name[NAME_MAX - 1] = 0;

		/*  cache next value */
		if (FindNextFile(ref->_d__handle, ffd) == 0)
		{
			/* end of list, close and signal next iteration to fail */
			FindClose(ref->_d__handle);
			ref->_d__handle = INVALID_HANDLE_VALUE;
		}
	}
	
	return ent;
}

int readdir_r(DIR *ref, struct dirent *entry, struct dirent **result) _MSL_CANT_THROW
{
	WIN32_FIND_DATA *ffd = (WIN32_FIND_DATA *) ref->_d__ffd;
	
	if (ref->_d__handle == INVALID_HANDLE_VALUE)
	{
		*result = NULL;
		errno = ENOENT;
		return ENOENT;
	}
	else 
	{
		/*  we already have one entry cached */
		if (strlen(ffd->cFileName) < NAME_MAX)
			strncpy(entry->d_name, ffd->cFileName, NAME_MAX);
		else
			strncpy(entry->d_name, ffd->cAlternateFileName, NAME_MAX);
		entry->d_name[NAME_MAX - 1] = 0;

		/*  cache next value */
		if (FindNextFile(ref->_d__handle, ffd) == 0)
		{
			/* end of list, close and signal next iteration to fail */
			FindClose(ref->_d__handle);
			ref->_d__handle = INVALID_HANDLE_VALUE;
		}
	}
	
	*result = entry;
	return 0;
}

void rewinddir(DIR *ref) _MSL_CANT_THROW
{
	if (ref->_d__handle != INVALID_HANDLE_VALUE)
		FindClose(ref->_d__handle);
	ref->_d__handle = FindFirstFile(ref->_d__wildcard, (WIN32_FIND_DATA *) ref->_d__ffd);	
}

int closedir(DIR *ref) _MSL_CANT_THROW
{
	if (ref->_d__handle != INVALID_HANDLE_VALUE)
		FindClose(ref->_d__handle);
	free(ref->_d__dirent);
	free(ref->_d__ffd);
	free(ref->_d__wildcard);
	free(ref);
	errno = ENOERR;
	return 0;
}

/* Change record:
 * ejs 010417 Added dirent routines.
 * JWW 010927 Moved dirent routines out of extras.win32.c to its own dirent.win32.c file
 * JWW 031103 Added readdir_r
 */