// Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// OpenVG C API for Symbian DLL


#include "vgstate.h"



extern "C" {

/*
 Note: Comments at the start of each Open VG api are adapted from the Open VG 1.1 specification.
 The text has been chosen/adapted to give a helpful overview of the function, and the errors
 (other than VG_NO_CONTEXT_ERROR) that it may generate.  For more details and diagrams see the
 full Open VG specification.
 */

/*
 The vgGetParameter functions return the value of a parameter on a given VGHandlebased
 object.

 The original value passed to vgSetParameter (provided the call to
 vgSetParameter completed without error) should be returned by
 vgGetParameter (except where specifically noted), even if the implementation
 makes use of a truncated or quantized value internally.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if object is not a valid handle, or is not shared with the current context
   VG_ILLEGAL_ARGUMENT_ERROR
    if paramType is not a valid value from the appropriate enumeration
    if paramType refers to a vector parameter in vgGetParameterf or
     vgGetParameteri
 */
EXPORT_C VGint
	vgGetParameteri(VGHandle object,
		VGint paramType)
	{
	OPENVG_TRACE("vgGetParameteri object=0x%x, paramType=0x%x -->", object, paramType);

	VGint value = TGuestOpenVg::vgGetParameteri(object, paramType);
	OPENVG_TRACE("vgGetParameteri value=0x%x <--", value);
	return value;
	}


////////////////////////////////////////////////////////////////////////////////////////////
//Functions returning value
////////////////////////////////////////////////////////////////////////////////////////////
/*
 Returns the oldest error code provided by an API call on the current context since the
 previous  call to vgGetError on that context (or since the creation of the context). No
 error is indicated by a return value of 0 (VG_NO_ERROR). After the call, the error code is
 cleared to 0. The possible errors that may be generated by each OpenVG function (apart from
 VG_OUT_OF_MEMORY_ERROR) are shown below the definition of the function.
 If no context is current at the time vgGetError is called, the error code
 VG_NO_CONTEXT_ERROR is returned. Pending error codes on existing contexts are not affected
 by the call.
 */
EXPORT_C VGErrorCode
	vgGetError(void)
	{
	OPENVG_TRACE("vgGetError");

	MVgContext* vgContext = CVghwUtils::VgContext();
	VGErrorCode error = VG_NO_CONTEXT_ERROR;
	if (vgContext)
		{ // thread state already exists - get client or Host VG error
		error = vgContext->VgError();
		}
	return error;
	}

/*
 Returns the paint object currently set for the given paintMode, or VG_INVALID_HANDLE
 if an error occurs or if no paint object is set (i.e., the default paint is present) on
 the given context with the given paintMode.

 ERRORS
   VG_ILLEGAL_ARGUMENT_ERROR
    if paintMode is not a valid value from the VGPaintMode enumeration
 */
EXPORT_C VGPaint
	vgGetPaint(VGPaintMode paintMode)
	{
	OPENVG_TRACE("vgGetPaint paintMode=0x%x -->", paintMode);

	VGPaint paintHandle = TGuestOpenVg::vgGetPaint(paintMode);
	OPENVG_TRACE("vgGetPaint handle=0x%x <--", paintHandle);
	return paintHandle;
	}

/*
 Creates a new paint object that is initialized to a set of default values and returns
 a VGPaint handle to it. If insufficient memory is available to allocate a new object,
 VG_INVALID_HANDLE is returned.
 */
EXPORT_C VGPaint
	vgCreatePaint(void)
	{
	OPENVG_TRACE("vgCreatePaint -->");

	VGPaint paintHandle = TGuestOpenVg::vgCreatePaint();
	OPENVG_TRACE("vgCreatePaint handle=0x%x <--", paintHandle);
	return paintHandle;
	}

/*
 Appends a path, defined by interpolation (or extrapolation) between the paths
 startPath and endPath by the given amount, to the path dstPath. It returns
 VG_TRUE if interpolation was successful (i.e., the paths had compatible segment
 types after normalization), and VG_FALSE otherwise. If interpolation is
 unsuccessful, dstPath is left unchanged. It is legal for dstPath to be a handle
 to the same path object as either startPath or endPath or both, in which case
 the contents of the source path or paths referenced by dstPath will have the
 interpolated path appended. If dstPath is not the a handle to the same path
 object as either startPath or endPath, the contents of startPath and endPath
 will not be affected by the call.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if any of dstPath, startPath, or endPath is not a valid path handle, or is
     not shared with the current context
   VG_PATH_CAPABILITY_ERROR
    if VG_PATH_CAPABILITY_INTERPOLATE_TO is not enabled for dstPath
    if VG_PATH_CAPABILITY_INTERPOLATE_FROM is not enabled for startPath or endPath
 */
EXPORT_C VGboolean
	vgInterpolatePath(VGPath dstPath,
		VGPath startPath,
		VGPath endPath,
		VGfloat amount)
	{
	OPENVG_TRACE("vgInterpolatePath dstPath=0x%x, startPath=0x%x, endPath=0x%x, amount=%f",
			dstPath, startPath, endPath, amount);

	return TGuestOpenVg::vgInterpolatePath(dstPath, startPath, endPath, amount);
	}

/*
 Returns the length of a given portion of a path in the user coordinate system
 (that is, in the paths own coordinate system, disregarding any matrix
 settings). Only the subpath consisting of the numSegments path segments beginning
 with startSegment (where the initial path segment has index 0) is used. If an
 error occurs, -1.0f is returned.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if path is not a valid path handle, or is not shared with the current context
   VG_PATH_CAPABILITY_ERROR
    if VG_PATH_CAPABILITY_PATH_LENGTH is not enabled for path
   VG_ILLEGAL_ARGUMENT_ERROR
    if startSegment is less than 0 or greater than the index of the final path
     segment
    if numSegments is less than or equal to 0
    if (startSegment + numSegments  1) is greater than the index of the final
     path segment
 */
EXPORT_C VGfloat
	vgPathLength(VGPath path,
		VGint startSegment,
		VGint numSegments)
	{
	OPENVG_TRACE("vgPathLength path=0x%x, startSegment=%d, numSegments=%d", path, startSegment, numSegments);

	return TGuestOpenVg::vgPathLength(path, startSegment, numSegments);
	}

/*
 Returns the current capabilities of the path, as a bitwise OR of
 VGPathCapabilities constants. If an error occurs, 0 is returned.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if path is not a valid path handle, or is not shared with the current context
 */
EXPORT_C VGbitfield
	vgGetPathCapabilities(VGPath path)
	{
	OPENVG_TRACE("vgGetPathCapabilities path=0x%x", path);

	return TGuestOpenVg::vgGetPathCapabilities(path);
	}

/*
 Create a new path that is ready to accept segment data and return a VGPath handle
 to it. The path data will be formatted in the format given by pathFormat, typically
 VG_PATH_FORMAT_STANDARD. The datatype parameter contains a value from the
 VGPathDatatype enumeration indicating the datatype that will be used for coordinate
 data. The capabilities argument is a bitwise OR of the desired VGPathCapabilities
 values. Bits of capabilities that do not correspond to values from VGPathCapabilities
 have no effect.

 If an error occurs, VG_INVALID_HANDLE is returned.

 ERRORS
   VG_UNSUPPORTED_PATH_FORMAT_ERROR
    if pathFormat is not a supported format
   VG_ILLEGAL_ARGUMENT_ERROR
    if datatype is not a valid value from the VGPathDatatype enumeration
    if scale is equal to 0
 */
EXPORT_C VGPath
	vgCreatePath(VGint pathFormat,
		VGPathDatatype datatype,
		VGfloat scale, VGfloat bias,
		VGint segmentCapacityHint,
		VGint coordCapacityHint,
		VGbitfield capabilities)
	{
	OPENVG_TRACE("vgCreatePath pathFormat=%d, datatype=0x%x, scale=%f, bias=%f, segCapHint=%d, coordCapHint=%d, caps=0x%x -->",
			pathFormat, datatype, scale, bias, segmentCapacityHint, coordCapacityHint, capabilities);

	VGPath pathHandle = TGuestOpenVg::vgCreatePath(pathFormat, datatype, scale, bias, segmentCapacityHint, coordCapacityHint, capabilities);
	OPENVG_TRACE("vgCreatePath handle=0x%x <--", pathHandle);
	return pathHandle;
	}

/*
 The vgGet functions return the value of a parameter on the current context.
 The original value passed to vgSet (except as specifically noted, and provided the call to
 vgSet completed without error) is returned by vgGet, even if the implementation makes
 use of a truncated or quantized value internally. This rule ensures that OpenVG state may
 be saved and restored without degradation.

 If an error occurs during a call to vgGetf, vgGeti, or vgGetVectorSize, the return value
 is undefined.

 ERRORS
   VG_ILLEGAL_ARGUMENT_ERROR
    if paramType is not a valid value from the VGParamType enumeration
    if paramType refers to a vector parameter in vgGetf or vgGeti
 */
EXPORT_C VGfloat
	vgGetf(VGParamType type)
	{
	OPENVG_TRACE("vgGetf type=0x%x", type);

	return TGuestOpenVg::vgGetf(type);
	}

/*
 The vgGet functions return the value of a parameter on the current context.
 The original value passed to vgSet (except as specifically noted, and provided the call to
 vgSet completed without error) is returned by vgGet, even if the implementation makes
 use of a truncated or quantized value internally. This rule ensures that OpenVG state may
 be saved and restored without degradation.

 If an error occurs during a call to vgGetf, vgGeti, or vgGetVectorSize, the return value
 is undefined.

 ERRORS
   VG_ILLEGAL_ARGUMENT_ERROR
    if paramType is not a valid value from the VGParamType enumeration
    if paramType refers to a vector parameter in vgGetf or vgGeti
 */
EXPORT_C VGint
	vgGeti(VGParamType type)
	{
	OPENVG_TRACE("vgGeti type=0x%x", type);

	return TGuestOpenVg::vgGeti(type);
	}

/*
 The vgGetVectorSize function returns the maximum number of elements in the vector
 that will be retrieved by the vgGetiv or vgGetfv functions if called with the given
 paramType argument. For scalar values, 1 is returned. If vgGetiv or vgGetfv is
 called with a smaller value for count than that returned by vgGetVectorSize,
 only the first count elements of the vector are retrieved. Use of a greater value
 for count will result in an error.
 The original value passed to vgSet (except as specifically noted, and provided the call to
 vgSet completed without error) is returned by vgGet, even if the implementation makes
 use of a truncated or quantized value internally. This rule ensures that OpenVG state may
 be saved and restored without degradation.
 If an error occurs during a call to vgGetf, vgGeti, or vgGetVectorSize, the return value
 is undefined.

 ERRORS
   VG_ILLEGAL_ARGUMENT_ERROR
    if paramType is not a valid value from the VGParamType enumeration
 */
EXPORT_C VGint
	vgGetVectorSize(VGParamType type)
	{
	OPENVG_TRACE("vgGetVectorSize type=0x%x", type);

	return TGuestOpenVg::vgGetVectorSize(type);
	}

/*
 The vgGetParameter functions return the value of a parameter on a given VGHandlebased
 object.

 The original value passed to vgSetParameter (provided the call to
 vgSetParameter completed without error) should be returned by
 vgGetParameter (except where specifically noted), even if the implementation
 makes use of a truncated or quantized value internally.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if object is not a valid handle, or is not shared with the current context
   VG_ILLEGAL_ARGUMENT_ERROR
    if paramType is not a valid value from the appropriate enumeration
    if paramType refers to a vector parameter in vgGetParameterf or
     vgGetParameteri
 */
EXPORT_C VGfloat
	vgGetParameterf(VGHandle object,
		VGint paramType)
	{
	OPENVG_TRACE("vgGetParameterf object=0x%x, paramType=0x%x -->", object, paramType);

	VGfloat value = TGuestOpenVg::vgGetParameterf(object, paramType);
	OPENVG_TRACE("vgGetParameterf value=%f <--", value);
	return value;
	}

/*
 The vgGetParameterVectorSize function returns the number of elements in the vector
 that will be returned by the vgGetParameteriv or vgGetParameterfv functions if called
 with the given paramType argument. For scalar values, 1 is returned.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if object is not a valid handle, or is not shared with the current context
   VG_ILLEGAL_ARGUMENT_ERROR
    if paramType is not a valid value from the appropriate enumeration
 */
EXPORT_C VGint
	vgGetParameterVectorSize(VGHandle object,
		VGint paramType)
	{
	OPENVG_TRACE("vgGetParameterVectorSize object=0x%x, paramType=0x%x -->", object, paramType);

	VGint size = TGuestOpenVg::vgGetParameterVectorSize(object, paramType);
	OPENVG_TRACE("vgGetParameterVectorSize size=%i <--", size);
	return size;
	}

/*
 Creates an object capable of storing a mask layer with the given width and
 height and returns a VGMaskLayer handle to it. The mask layer is defined to
 be compatible with the format and multisampling properties of the current
 drawing surface. If there is no current drawing surface, no mask is
 configured for the current drawing surface, or an error occurs,
 VG_INVALID_HANDLE is returned. All mask layer values are initially set to one.

  ERRORS
   VG_ILLEGAL_ARGUMENT_ERROR
    if width or height are less than or equal to 0
    if width is greater than VG_MAX_IMAGE_WIDTH
    if height is greater than VG_MAX_IMAGE_HEIGHT
    if width*height is greater than VG_MAX_IMAGE_PIXELS
 */
EXPORT_C VGMaskLayer
	vgCreateMaskLayer(VGint width, VGint height)
	{
	OPENVG_TRACE("vgCreateMaskLayer width=%d, height=%d -->", width, height);

	VGMaskLayer maskHandle = TGuestOpenVg::vgCreateMaskLayer(width, height);
	OPENVG_TRACE("vgCreateMaskLayer handle=0x%x <--", maskHandle);
	return maskHandle;
	}

/*
 The current setting of the VG_PAINT_COLOR parameter on a given paint object may
 be queried as a 32-bit non-premultiplied sRGBA_8888 value. Each color channel or
 alpha value is clamped to the range [0, 1] , multiplied by 255, and rounded to obtain an
 8-bit integer; the resulting values are packed into a 32-bit value in the same format as for
 vgSetColor.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if paint is not a valid paint handle, or is not shared with the current context
 */
EXPORT_C VGuint
	vgGetColor(VGPaint paint)
	{
	OPENVG_TRACE("vgGetColor paint=0x%x", paint);

	return TGuestOpenVg::vgGetColor(paint);
	}

/*
 Creates an image with the given width, height, and pixel format and returns a
 VGImage handle to it. If an error occurs, VG_INVALID_HANDLE is returned. All
 color and alpha channel values are initially set to zero. The format parameter
 must contain a value from the VGImageFormat enumeration.
 The allowedQuality parameter is a bitwise OR of values from the
 VGImageQuality enumeration, indicating which levels of resampling quality may be
 used to draw the image. It is always possible to draw an image using the
 VG_IMAGE_QUALITY_NONANTIALIASED quality setting even if it is not explicitly
 specified.

 ERRORS
   VG_UNSUPPORTED_IMAGE_FORMAT_ERROR
    if format is not a valid value from the VGImageFormat enumeration
   VG_ILLEGAL_ARGUMENT_ERROR
    if width or height are less than or equal to 0
    if width is greater than VG_MAX_IMAGE_WIDTH
    if height is greater than VG_MAX_IMAGE_HEIGHT
    if width*height is greater than VG_MAX_IMAGE_PIXELS
    if width*height*(pixel size of format) is greater than
   VG_MAX_IMAGE_BYTES
    if allowedQuality is not a bitwise OR of values from the
     VGImageQuality enumeration
 */
EXPORT_C VGImage
	vgCreateImage(VGImageFormat format,
		VGint width, VGint height,
		VGbitfield allowedQuality)
	{
	OPENVG_TRACE("vgCreateImage format=0x%x, width=%d, height=%d, allowedQuality=0x%x -->",
			format, width, height, allowedQuality);

	VGImage imageHandle = TGuestOpenVg::vgCreateImage(format, width, height, allowedQuality);
	OPENVG_TRACE("vgCreateImage handle=0x%x <--", imageHandle);
	return imageHandle;
	}

/*
 Returns a new VGImage handle that refers to a portion of the parent image. The
 region is given by the intersection of the bounds of the parent image with the
 rectangle beginning at pixel (x, y) with dimensions width and height, which
 must define a positive region contained entirely within parent.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if parent is not a valid image handle, or is not shared with the current
     context
   VG_IMAGE_IN_USE_ERROR
    if parent is currently a rendering target
   VG_ILLEGAL_ARGUMENT_ERROR
    if x is less than 0 or greater than or equal to the parent width
    if y is less than 0 or greater than or equal to the parent height
    if width or height is less than or equal to 0
    if x + width is greater than the parent width
    if y + height is greater than the parent height
 */
EXPORT_C VGImage
	vgChildImage(VGImage parent,
		VGint x, VGint y,
		VGint width, VGint height)
	{
	OPENVG_TRACE("vgChildImage parent=oc%x, x=%d, y=%d, width=%d, height=%d -->",
			parent, x, y, width, height);

	VGImage imageHandle = TGuestOpenVg::vgChildImage(parent, x, y, width, height);
	OPENVG_TRACE("vgChildImage handle=0x%x <--", imageHandle);
	return imageHandle;
	}

/*
 Returns the closest valid ancestor (i.e., one that has not been the target of
 a vgDestroyImage call) of the given image. If image has no ancestors, image is
 returned.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if image is not a valid image handle, or is not shared with the current
     context
   VG_IMAGE_IN_USE_ERROR
    if image is currently a rendering target
 */
EXPORT_C VGImage vgGetParent(VGImage image)
	{
	OPENVG_TRACE("vgGetParent image=0x%x", image);

	return TGuestOpenVg::vgGetParent(image);
	}

/*
 Creates a new font object and returns a VGFont handle to it. The glyphCapacityHint
 argument provides a hint as to the capacity of a VGFont, i.e., the total number of
 glyphs that this VGFont object will be required to accept. A value of 0 indicates that
 the value is unknown. If an error occurs during execution, VG_INVALID_HANDLE is returned.

 ERRORS
   VG_ILLEGAL_ARGUMENT_ERROR
    if glyphCapacityHint is negative
 */
EXPORT_C VGFont vgCreateFont(VGint glyphCapacityHint)
	{
	OPENVG_TRACE("vgCreateFont glyphCapacityHint=%d -->", glyphCapacityHint);

	VGFont fontHandle = TGuestOpenVg::vgCreateFont(glyphCapacityHint);
	OPENVG_TRACE("vgCreateFont handle=0x%x <--", fontHandle);
	return fontHandle;
	}

/* Hardware Queries */

/*
 Returns a value indicating whether a given setting of a property of a type given
 by key is generally accelerated in hardware on the currently running OpenVG
 implementation.

 ERRORS
   VG_ILLEGAL_ARGUMENT_ERROR
    if key is not one of the values from the VGHardwareQueryType enumeration
    if setting is not one of the values from the enumeration associated with
     key
 */
EXPORT_C VGHardwareQueryResult
	vgHardwareQuery
		(VGHardwareQueryType key,
		VGint setting)
	{
	OPENVG_TRACE("vgHardwareQuery key=0x%x, setting=%d", key, setting);

	return TGuestOpenVg::vgHardwareQuery(key, setting);
	}

/*
 The vgGetString function returns information about the OpenVG implementation,
 including extension information. The values returned may vary according to
 the display (e.g., the EGLDisplay when using EGL) associated with the current
 context.

 Returns NULL if no context is current.
 */
EXPORT_C const VGubyte *
	vgGetString(VGStringID name)
	{
	OPENVG_TRACE("vgGetString name=0x%x", name);

	switch (name)
		{
		case VG_VENDOR:
			return (const VGubyte *)"Nokia";
		case VG_RENDERER:
			return (const VGubyte *)"Guest 1.0";
		case VG_VERSION:
			return (const VGubyte *)"1.1";
		case VG_EXTENSIONS: // supported VG extensions, (associated function addresses are fetched with eglGetProcAddress)
			return (const VGubyte *)"VG_KHR_EGL_image";
		}
	return (const VGubyte *)NULL;
	}

/*
 The vgGetParameter functions return the value of a parameter on a given VGHandlebased
 object.

 If vgGetParameteriv or vgGetParameterfv is called with a smaller value for
 count than that returned by vgGetParameterVectorSize, only the first count
 elements of the vector are retrieved. Use of a greater value for count will result
 in an error.

 The original value passed to vgSetParameter (provided the call to
 vgSetParameter completed without error) should be returned by
 vgGetParameter (except where specifically noted), even if the implementation
 makes use of a truncated or quantized value internally.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if object is not a valid handle, or is not shared with the current context
   VG_ILLEGAL_ARGUMENT_ERROR
    if paramType is not a valid value from the appropriate enumeration
    if values is NULL in vgGetParameterfv or vgGetParameteriv
    if values is not properly aligned in vgGetParameterfv or vgGetParameteriv
    if count is less than or equal to 0 in vgGetParameterfv or vgGetParameteriv
    if count is greater than the value returned by vgGetParameterVectorSize for
     the given parameter in vgGetParameterfv or vgGetParameteriv
 */
EXPORT_C void
	vgGetParameterfv(VGHandle object,
		VGint paramType,
		VGint count,
		VGfloat * values)
	{
	OPENVG_TRACE("vgGetParameterfv object=0x%x, paramType=0x%x, count=%d, values=0x%x",
			object, paramType, count, values);

	TGuestOpenVg::vgGetParameterfv(object, paramType, count, values);
	}

EXPORT_C void
	vgGetParameteriv(VGHandle object,
		VGint paramType,
		VGint count,
		VGint * values)
	{
	OPENVG_TRACE("vgGetParameteriv object=0x%x, paramType=0x%x, count=%d, values=0x%x",
			object, paramType, count, values);

	TGuestOpenVg::vgGetParameteriv(object, paramType, count, values);
	}

/* Renderer and Extension Information */

/*
The vgGetVectorSize function returns the maximum number of elements in the vector
that will be retrieved by the vgGetiv or vgGetfv functions if called with the given
paramType argument. For scalar values, 1 is returned. If vgGetiv or vgGetfv is
called with a smaller value for count than that returned by vgGetVectorSize,
only the first count elements of the vector are retrieved. Use of a greater value
for count will result in an error.
The original value passed to vgSet (except as specifically noted, and provided the call to
vgSet completed without error) is returned by vgGet, even if the implementation makes
use of a truncated or quantized value internally. This rule ensures that OpenVG state may
be saved and restored without degradation.
If an error occurs during a call to vgGetfv or vgGetiv, nothing is
written to values.

ERRORS
VG_ILLEGAL_ARGUMENT_ERROR
 if paramType is not a valid value from the VGParamType enumeration
 if values is NULL in vgGetfv or vgGetiv
 if values is not properly aligned in vgGetfv or vgGetiv
 if count is less than or equal to 0 in vgGetfv or vgGetiv
 if count is greater than the value returned by vgGetVectorSize for the
given parameter in vgGetfv or vgGetiv
 */
EXPORT_C void
	vgGetfv(VGParamType type, VGint count,
		VGfloat * values)
	{
	OPENVG_TRACE("vgGetfv type=0x%x, count=%d, values=0x%x", type, count, values);

	TGuestOpenVg::vgGetfv(type, count, values);
	}

EXPORT_C void
	vgGetiv(VGParamType type, VGint count,
		VGint * values)
	{
	OPENVG_TRACE("vgGetiv type=0x%x, count=%d, values=0x%x", type, count, values);

	TGuestOpenVg::vgGetiv(type, count, values);
	}

/*
 Retrieve the value of the current transformation.
 Nine values are written to m in the order:
 { sx, shy, w0, shx, sy, w1, tx, ty, w2 }
 For an affine matrix, w0 and w1 will always be 0 and w2 will always be 1.

 ERRORS
 VG_ILLEGAL_ARGUMENT_ERROR
  if m is NULL
  if m is not properly aligned
 */
EXPORT_C void
	vgGetMatrix(VGfloat * m)
	{
	OPENVG_TRACE("vgGetMatrix m=0x%x", m);

	TGuestOpenVg::vgGetMatrix(m);
	}

/*
 Read pixel values from a rectangular portion of an image, performs format conversion
 if necessary, and stores the resulting pixels into memory.

 Pixel values are written starting at the address given by the pointer data; adjacent
 scanlines are separated by dataStride bytes. Negative or zero values of
 dataStride are allowed. The region to be read is given by x, y, width, and
 height, which must define a positive region. Pixels that fall outside the bounds
 of the image are ignored.
 Pixel values in memory are formatted according to the dataFormat parameter, which
 must contain a value from the VGImageFormat enumeration. If dataFormat
 specifies a premultiplied format (VG_sRGBA_8888_PRE or VG_lRGBA_8888_PRE),
 color channel values of a pixel that are greater than their corresponding alpha value are
 clamped to the range [0, alpha]. The data pointer alignment and the pixel layout in
 memory are as described in the vgImageSubData section.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if image is not a valid image handle, or is not shared with the current context
   VG_IMAGE_IN_USE_ERROR
    if image is currently a rendering target
   VG_UNSUPPORTED_IMAGE_FORMAT_ERROR
    if dataFormat is not a valid value from the VGImageFormat enumeration
   VG_ILLEGAL_ARGUMENT_ERROR
    if width or height is less than or equal to 0
    if data is NULL
    if data is not properly aligned
 */
EXPORT_C void
	vgGetImageSubData(VGImage image,
		void * data,
		VGint dataStride,
		VGImageFormat dataFormat,
		VGint x, VGint y,
		VGint width, VGint height)
	{
	OPENVG_TRACE("vgGetImageSubData image=0x%x, data=0x%x, dataStride=%d, dataFormat=0x%x, x=%d, y=%d, width=%d, height=%d",
			image, data, dataStride, dataFormat, x, y, width, height);

	TGuestOpenVg::vgGetImageSubData(image, data, dataStride, dataFormat, x, y, width, height);
	}

/*
 Copy pixel data from the drawing surface without the creation of a VGImage object.
 Pixel values are written starting at the address given by the pointer data; adjacent
 scanlines are separated by dataStride bytes. Negative or zero values of
 dataStride are allowed. The region to be read is given by x, y, width, and
 height, which must define a positive region.

 Pixel values in memory are formatted according to the dataFormat parameter, which
 must contain a value from the VGImageFormat enumeration. The data pointer
 alignment and the pixel layout in memory is as described for vgImageSubData.

 Pixels whose source lies outside of the bounds of the drawing surface are
 ignored. Pixel format conversion is applied as needed. The scissoring region
 does not affect the reading of pixels.

 ERRORS
   VG_UNSUPPORTED_IMAGE_FORMAT_ERROR
    if dataFormat is not a valid value from the VGImageFormat enumeration
   VG_ILLEGAL_ARGUMENT_ERROR
    if width or height is less than or equal to 0
    if data is NULL
    if data is not properly aligned
 */
EXPORT_C void
	vgReadPixels(void * data, VGint dataStride,
		VGImageFormat dataFormat,
		VGint sx, VGint sy,
		VGint width, VGint height)
	{
	OPENVG_TRACE("vgReadPixels data=0x%x, dataStride=%d, dataFormat=0x%x, sx=%d, sy=%d, width=%d, height=%d",
			data, dataStride, dataFormat, sx, sy, width, height);

	TGuestOpenVg::vgReadPixels(data, dataStride, dataFormat, sx, sy, width, height);
	}

/*
 Returns an axis-aligned bounding box that tightly bounds the interior of the given
 path. Stroking parameters are ignored. If path is empty, minX and minY are set to 0
 and width and height are set to -1. If path contains a single point, minX and minY
 are set to the coordinates of the point and width and height are set to 0.

 The VG_PATH_CAPABILITY_PATH_BOUNDS capability must be enabled for path.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if path is not a valid path handle, or is not shared with the current context
   VG_ILLEGAL_ARGUMENT_ERROR
    if minX, minY, width, or height is NULL
    if minX, minY, width, or height is not properly aligned
   VG_PATH_CAPABILITY_ERROR
    if VG_PATH_CAPABILITY_PATH_BOUNDS is not enabled for path
 */
EXPORT_C void
	vgPathBounds(VGPath path,
		VGfloat * minX,
		VGfloat * minY,
		VGfloat * width,
		VGfloat * height)
	{
	OPENVG_TRACE("vgPathBounds path=0x%x, minX=0x%x, minY=0x%x, width=0x%x, height=0x%x",
			path, minX, minY, width, height);

	TGuestOpenVg::vgPathBounds(path, minX, minY, width, height);
	}

/*
 The vgPathTransformedBounds function returns an axis-aligned bounding box
 that is guaranteed to enclose the geometry of the given path following
 transformation by the current path-user-to-surface transform. The returned
 bounding box is not guaranteed to fit tightly around the path geometry. If path
 is empty, minX and minY are set to 0 and width and height are set to -1. If
 path contains a single point, minX and minY are set to the transformed
 coordinates of the point and width and height are set to 0.
 The VG_PATH_CAPABILITY_PATH_TRANSFORMED_BOUNDS capability must be
 enabled for path.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if path is not a valid path handle, or is not shared with the current context
   VG_ILLEGAL_ARGUMENT_ERROR
    if minX, minY, width, or height is NULL
    if minX, minY, width, or height is not properly aligned
   VG_PATH_CAPABILITY_ERROR
    if VG_PATH_CAPABILITY_PATH_TRANSFORMED_BOUNDS is not enabled
     for path
 */
EXPORT_C void
	vgPathTransformedBounds(VGPath path,
		VGfloat * minX,
		VGfloat * minY,
		VGfloat * width,
		VGfloat * height)
	{
	OPENVG_TRACE("vgPathTransformedBounds path=0x%x, minX=0x%x, minY=0x%x, width=0x%x, height=0x%x",
					path, minX, minY, width, height);

	TGuestOpenVg::vgPathTransformedBounds(path, minX, minY, width, height);
	}

////////////////////////////////////////////////////////////////////////////////////////////
//Syncing methods
////////////////////////////////////////////////////////////////////////////////////////////

/*
 Ensures that all outstanding requests on the current context will complete in finite time.
 vgFlush may return prior to the actual completion of all requests.
 */
EXPORT_C void
	vgFlush(void)
	{
	// ToDo comment
	OPENVG_TRACE("vgFlush");

	MVgContext* vgContext = CVghwUtils::VgContext();
	if (vgContext)
		{
		vgContext->ExecuteVgFlushCommand();
		}
	}

/*
 The vgFinish function forces all outstanding requests on the current context to
 complete, returning only when the last request has completed.
 */
EXPORT_C void
	vgFinish(void)
	{
	// ToDo comment
	OPENVG_TRACE("vgFinish");

	MVgContext* vgContext = CVghwUtils::VgContext();
	if (vgContext)
		{
		vgContext->ExecuteVgFinishCommand();
		}
	}

////////////////////////////////////////////////////////////////////////////////////////////
//Functions not returning value (possible to buffer)
////////////////////////////////////////////////////////////////////////////////////////////
/* Getters and Setters */

/*
 The vgSet functions set the value of a parameter on the current context.

 ERRORS
   VG_ILLEGAL_ARGUMENT_ERROR
    if paramType is not a valid value from the VGParamType enumeration
    if paramType refers to a vector parameter in vgSetf or vgSeti
    if value is not a legal enumerated value for the given parameter in vgSetf or
     vgSeti
 */
EXPORT_C void
   vgSetf (VGParamType type, VGfloat value)
	{
	OPENVG_TRACE("vgSetf type=0x%x, value=%f", type, value);

	TGuestOpenVg::vgSetf(type, value);
	}

EXPORT_C void
	vgSeti (VGParamType type, VGint value)
	{
	OPENVG_TRACE("vgSeti type=0x%x, value=%d", type, value);

	TGuestOpenVg::vgSeti(type, value);
	}

/*
 The vgSet functions set the value of a parameter on the current context.

 ERRORS
   VG_ILLEGAL_ARGUMENT_ERROR
    if paramType is not a valid value from the VGParamType enumeration
    if paramType refers to a scalar parameter in vgSetfv or vgSetiv and count is
     not equal to 1
    if values[i] is not a legal enumerated value for the given parameter in vgSetfv
     or vgSetiv for 0 <= i < count
    if values is NULL in vgSetfv or vgSetiv and count is greater than 0
    if values is not properly aligned in vgSetfv or vgSetiv
    if count is less than 0 in vgSetfv or vgSetiv
    if count is not a valid value for the given parameter
 */
EXPORT_C void
	vgSetfv(VGParamType type, VGint count,
		const VGfloat * values)
	{
	OPENVG_TRACE("vgSetfv type=0x%x, count=%d, values=0x%x", type, count, values);

	TGuestOpenVg::vgSetfv(type, count, values);
	}

EXPORT_C void
	vgSetiv(VGParamType type, VGint count,
		const VGint * values)
	{
	OPENVG_TRACE("vgSetiv type=0x%x, count=%d, values=0x%x", type, count, values);

	TGuestOpenVg::vgSetiv(type, count, values);
	}

/*
 For vgSetParameterf and vgSetParameteri.

 The vgSetParameter functions set the value of a parameter on a given VGHandlebased
 object.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if object is not a valid handle, or is not shared with the current context
   VG_ILLEGAL_ARGUMENT_ERROR
    if paramType is not a valid value from the appropriate enumeration
    if paramType refers to a vector parameter in vgSetParameterf or
     vgSetParameteri
    if value is not a legal enumerated value for the given parameter in
     vgSetParameterf or vgSetParameteri
 */
EXPORT_C void
	vgSetParameterf(VGHandle object,
		VGint paramType,
		VGfloat value)
	{
	OPENVG_TRACE("vgSetParameterf object=0x%x, paramType=0x%x, value=%f -->", object, paramType, value);

	TGuestOpenVg::vgSetParameterf(object, paramType, value);
	}

EXPORT_C void
	vgSetParameteri(VGHandle object,
		VGint paramType,
		VGint value)
	{
	OPENVG_TRACE("vgSetParameteri object=0x%x, paramType=0x%x, value=0x%x", object, paramType, value);

	TGuestOpenVg::vgSetParameteri(object, paramType, value);
	}

/*
 For vgSetParameterfv and vgSetParameteriv.

 The vgSetParameter functions set the value of a parameter on a given VGHandlebased
 object.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if object is not a valid handle, or is not shared with the current context
   VG_ILLEGAL_ARGUMENT_ERROR
    if paramType is not a valid value from the appropriate enumeration
    if paramType refers to a scalar parameter in vgSetParameterfv or
     vgSetParameteriv and count is not equal to 1
    if values[i] is not a legal enumerated value for the given parameter
     in vgSetParameterfv or vgSetParameteriv for 0 <= i < count
    if values is NULL in vgSetParameterfv or vgSetParameteriv and count is
     greater than 0
    if values is not properly aligned in vgSetParameterfv or vgSetParameteriv
    if count is less than 0 in vgSetParameterfv or vgSetParameteriv
    if count is not a valid value for the given parameter
 */
EXPORT_C void
	vgSetParameterfv(VGHandle object,
		VGint paramType,
		VGint count,
		const VGfloat * values)
	{
	OPENVG_TRACE("vgSetParameterfv object=0x%x, paramType=%d, count=%d, values=0x%x", object, paramType, count, values);

	TGuestOpenVg::vgSetParameterfv(object, paramType, count, values);
	}

EXPORT_C void
	vgSetParameteriv(VGHandle object,
		VGint paramType,
		VGint count,
		const VGint * values)
	{
	OPENVG_TRACE("vgSetParameteriv object=0x%x, paramType=%d, count=%d, values=0x%x", object, paramType, count, values);

	TGuestOpenVg::vgSetParameteriv(object, paramType, count, values);
	}

/* Matrix Manipulation */

/*
 Sets the current matrix M to the identity matrix:
 */
EXPORT_C void
	vgLoadIdentity(void)
	{
	OPENVG_TRACE("vgLoadIdentity");

	TGuestOpenVg::vgLoadIdentity();
	}

/*
 Loads an arbitrary set of matrix values into the current
 matrix. Nine matrix values are read from m, in the order:
   { sx, shy, w0, shx, sy, w1, tx, ty, w2 }
 However, if the targeted matrix is affine (i.e., the matrix mode is not
 VG_MATRIX_IMAGE_USER_TO_SURFACE), the values { w0, w1, w2 } are ignored and
 replaced by the values { 0, 0, 1 }.

 ERRORS
   VG_ILLEGAL_ARGUMENT_ERROR
    if m is NULL
    if m is not properly aligned
 */
EXPORT_C void
	vgLoadMatrix(const VGfloat * m)
	{
	OPENVG_TRACE("vgLoadMatrix m=0x%x", m);

	TGuestOpenVg::vgLoadMatrix(m);
	}

/*
 Right-multiplies the current matrix M by a given matrix:
   Nine matrix values are read from m in the order:
   { sx, shy, w0, shx, sy, w1, tx, ty, w2 }
 and the current matrix is multiplied by the resulting matrix. However, if the
 targeted matrix is affine (i.e., the matrix mode is not
 VG_MATRIX_IMAGE_USER_TO_SURFACE), the values { w0, w1, w2 } are ignored
 and replaced by the values { 0, 0, 1 } prior to multiplication.

 ERRORS
   VG_ILLEGAL_ARGUMENT_ERROR
    if m is NULL
    if m is not properly aligned
 */
EXPORT_C void
	vgMultMatrix(const VGfloat * m)
	{
	OPENVG_TRACE("vgMultMatrix m=0x%x", m);

	TGuestOpenVg::vgMultMatrix(m);
	}

/*
 Modifies the current transformation by appending a  translation. This is equivalent
 to right-multiplying the current matrix M by a translation matrix:
    [ 1  0  tx ]
    [ 0  1  ty ]
    [ 0  0  1  ]
 */
EXPORT_C void
	vgTranslate(VGfloat tx, VGfloat ty)
	{
	OPENVG_TRACE("vgTranslate tx=%f, ty=%f", tx, ty);

	TGuestOpenVg::vgTranslate(tx, ty);
	}

/*
 Modifies the current transformation by appending a scale. This is equivalent to
 right-multiplying the current matrix M by a scale matrix:
    [ sx  0   0 ]
    [ 0   sy  0 ]
    [ 0   0   1 ]
 */
EXPORT_C void
	vgScale(VGfloat sx, VGfloat sy)
	{
	OPENVG_TRACE("vgScale sx=%f, sy=%f", sx, sy);

	TGuestOpenVg::vgScale(sx, sy);
	}

/*
 Modifies the current transformation by appending a shear. This is equivalent to
 right-multiplying the current matrix M by a shear matrix:
   [  1  shx  0 ]
   [ shy  1   0 ]
   [  0   0   1 ]
 */
EXPORT_C void
	vgShear(VGfloat shx, VGfloat shy)
	{
	OPENVG_TRACE("vgShear shx=%f, shy=%f", shx, shy);

	TGuestOpenVg::vgShear(shx, shy);
	}

/*
 Modifies the current transformation by appending a counterclockwise rotation by a given
 angle (expressed in degrees) about the origin. This is equivalent to right-multiplying
 the current matrix M by the following matrix (using the symbol a to represent the value
 of the angle parameter):
   [ cos(a) -sin(a)  0 ]
   [ sin(a)  cos(a)  0 ]
   [  0       0      1 ]
 */
EXPORT_C void
	vgRotate(VGfloat angle)
	{
	OPENVG_TRACE("vgRotate angle=%f");

	TGuestOpenVg::vgRotate(angle);
	}

/*
 Modifies the drawing surface mask values according to a given operation, possibly
 using coverage values taken from a mask layer or bitmap image given by the mask
 parameter. If no mask is configured for the current drawing surface, vgMask has
 no effect.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if operation is not VG_CLEAR_MASK or VG_FILL_MASK, and mask is not a
     valid mask layer or image handle, or is not shared with the current context
   VG_IMAGE_IN_USE_ERROR
    if mask is a VGImage that is currently a rendering target
   VG_ILLEGAL_ARGUMENT_ERROR
    if operation is not a valid value from the VGMaskOperation
     enumeration
    if width or height is less than or equal to 0
    if mask is a VGMaskLayer and is not compatible with the current surface
     mask
 */
EXPORT_C void
	vgMask(VGHandle mask, VGMaskOperation operation,
		VGint x, VGint y,
		VGint width, VGint height)
	{
	OPENVG_TRACE("vgMask mask=0x%x, operation=0x%x, x=%d, y=%d, width=%d, height=%d",
			mask, operation, x, y, width, height);

	TGuestOpenVg::vgMask(mask, operation, x, y, width, height);
	}

/*
 Modifies the current surface mask by applying the given operation to the set of
 coverage values associated with the rendering of the given path. If paintModes
 contains VG_FILL_PATH, the path is filled; if it contains VG_STROKE_PATH, the path
 is stroked. If both are present, the mask operation is performed in two passes,
 first on the filled path geometry, then on the stroked path geometry.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if path is not a valid path handle
   VG_ILLEGAL_ARGUMENT_ERROR
    if paintModes is not a valid bitwise OR of values from the VGPaintMode
     enumeration
    if operation is not a valid value from the VGMaskOperation enumeration
 */
EXPORT_C void
	vgRenderToMask(VGPath path,
		VGbitfield paintModes,
		VGMaskOperation operation)
	{
	OPENVG_TRACE("vgRenderToMask path=0x%x, paintModes=0x%x, operation=0x%x", path, paintModes, operation);

	TGuestOpenVg::vgRenderToMask(path, paintModes, operation);
	}

/*
 Deallocate the resources associated with a mask layer. Following the call,
 the maskLayer handle is no longer valid in the current context.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if maskLayer is not a valid mask handle
 */
EXPORT_C void vgDestroyMaskLayer(VGMaskLayer maskLayer)
	{
	OPENVG_TRACE("vgDestroyMaskLayer maskLayer=0x%x", maskLayer);

	TGuestOpenVg::vgDestroyMaskLayer(maskLayer);
	}

/*
 Set the values of a given maskLayer within a given rectangular region to a given
 value. The floating-point value value must be between 0 and 1. The value is rounded
 to the closest available value supported by the mask layer. If two values are equally
 close, the larger value is used.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if maskLayer is not a valid mask layer handle, or is not shared with the
     current context
   VG_ILLEGAL_ARGUMENT_ERROR
    if value is less than 0 or greater than 1
    if width or height is less than or equal to 0
    if x or y is less than 0
    if x + width is greater than the width of the mask
    if y + height is greater than the height of the mask
 */
EXPORT_C void
	vgFillMaskLayer(VGMaskLayer maskLayer,
		VGint x, VGint y,
		VGint width, VGint height,
		VGfloat value)
	{
	OPENVG_TRACE("vgFillMaskLayer maskLayer=0x%x, x=%d, y=%d, width=%d, height=%d, value=%f",
			maskLayer, x, y, width, height, value);

	TGuestOpenVg::vgFillMaskLayer(maskLayer, x, y, width, height, value);
	}

/*
 Copies a portion of the current surface mask into a VGMaskLayer object. The source
 region starts at (sx, sy) in the surface mask, and the destination region starts
 at (dx, dy) in the destination maskLayer. The copied region is clipped to the given
 width and height and the bounds of the source and destination. If the current context
 does not contain a surface mask, vgCopyMask does nothing.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if maskLayer is not a valid mask layer handle
   VG_ILLEGAL_ARGUMENT_ERROR
    if width or height are less than or equal to 0
    if maskLayer is not compatible with the current surface mask
 */
EXPORT_C void
	vgCopyMask(VGMaskLayer maskLayer,
		VGint sx, VGint sy,
		VGint dx, VGint dy,
		VGint width, VGint height)
	{
	OPENVG_TRACE("vgCopyMask maskLayer=0x%x, sx=%d, sy=%d, dx=%d, dy=%d, width=%d, height=%d",
			maskLayer, sx, sy, dx, dy, width, height);

	TGuestOpenVg::vgCopyMask(maskLayer, sx, sy, dx, dy, width, height);
	}

/*
 Fills the portion of the drawing surface intersecting the rectangle extending from
 pixel (x, y) and having the given width and height with a constant color value, taken
 from the VG_CLEAR_COLOR parameter. The color value is expressed in non-premultiplied
 sRGBA (sRGB color plus alpha)format. Values outside the [0, 1] range are interpreted
 as the nearest endpoint of the range. The color is converted to the destination color
 space in the same manner as if a rectangular path were being filled. Clipping and
 scissoring take place in the usual fashion, but antialiasing, masking, and blending
 do not occur.

 ERRORS
   VG_ILLEGAL_ARGUMENT_ERROR
    if width or height is less than or equal to 0
 */
EXPORT_C void
	vgClear(VGint x, VGint y,
		VGint width, VGint height)
	{
	OPENVG_TRACE("vgClear x=%d, y=%d, width=%d, height=%d", x, y, width, height);

	TGuestOpenVg::vgClear(x, y, width, height);
	}


/* Paths */

/*
 Remove all segment command and coordinate data associated with a path. The handle
 continues to be valid for use in the future, and the path format and datatype retain
 their existing values. The capabilities argument is a bitwise OR of the desired
 VGPathCapabilities values. Bits of capabilities that do not correspond to values from
 VGPathCapabilities have no effect. Using vgClearPath may be more efficient than
 destroying and re-creating a path for short-lived paths.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if path is not a valid path handle, or is not shared with the current context
 */
EXPORT_C void
	vgClearPath(VGPath path, VGbitfield capabilities)
	{
	OPENVG_TRACE("vgClearPath path=0x%x, capabilities=0x%x", path, capabilities);

	TGuestOpenVg::vgClearPath(path, capabilities);
	}

/*
 Release any resources associated with path, and makes the handle invalid in all
 contexts that shared it.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if path is not a valid path handle, or is not shared with the current context
 */
EXPORT_C void vgDestroyPath(VGPath path)
	{
	OPENVG_TRACE("vgDestroyPath path=0x%x", path);

	TGuestOpenVg::vgDestroyPath(path);
	}

/*
 Requests the set of capabilities specified in the capabilities argument to
 be disabled for the given path. The capabilities argument is a bitwise OR of
 the VGPathCapabilities values whose removal is requested. Attempting to
 remove a capability that is already disabled has no effect. Bits of
 capabilities that do not correspond to values from VGPathCapabilities have
 no effect.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if path is not a valid path handle, or is not shared with the current context
 */
EXPORT_C void
	vgRemovePathCapabilities(VGPath path,
		VGbitfield capabilities)
	{
	OPENVG_TRACE("vgRemovePathCapabilities path=0x%x, capabilities=0x%x", path, capabilities);

	TGuestOpenVg::vgRemovePathCapabilities(path, capabilities);
	}

/*
 Appends a copy of all path segments from srcPath onto the end of the existing
 data in dstPath. It is legal for srcPath and dstPath to be handles to the same
 path object, in which case the contents of the path are duplicated. If srcPath
 and dstPath are handles to distinct path objects, the contents of srcPath will
 not be affected by the call.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if either dstPath or srcPath is not a valid path handle, or is not shared
     with the current context
   VG_PATH_CAPABILITY_ERROR
    if VG_PATH_CAPABILITY_APPEND_FROM is not enabled for srcPath
    if VG_PATH_CAPABILITY_APPEND_TO is not enabled for dstPath
 */
EXPORT_C void
	vgAppendPath(VGPath dstPath, VGPath srcPath)
	{
	OPENVG_TRACE("vgAppendPath dstPath=0x%x, srcPath=0x%x", dstPath, srcPath);

	TGuestOpenVg::vgAppendPath(dstPath, srcPath);
	}

/*
 Appends data taken from pathData to the given path dstPath. The data are formatted
 using the path format of dstPath (as returned by querying the paths VG_PATH_FORMAT
 parameter using vgGetParameteri). The numSegments parameter gives the total number
 of entries in the pathSegments array, and must be greater than 0. Legal values for
 the pathSegments array are the values from the VGPathCommand enumeration as well as
 VG_CLOSE_PATH and (VG_CLOSE_PATH | VG_RELATIVE) (which are synonymous).

 The pathData pointer must be aligned on a 1-, 2-, or 4-byte boundary (as defined in
 the Bytes column of Open VG spec Table 7) depending on the size of the coordinate
 datatype (as returned by querying the paths VG_PATH_DATATYPE parameter using
 vgGetParameteri). The VG_PATH_CAPABILITY_APPEND_TO capability must be enabled for
 path.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if dstPath is not a valid path handle, or is not shared with the current
     context
   VG_PATH_CAPABILITY_ERROR
    if VG_PATH_CAPABILITY_APPEND_TO is not enabled for dstPath
   VG_ILLEGAL_ARGUMENT_ERROR
    if pathSegments or pathData is NULL
    if pathData is not properly aligned
    if numSegments is less than or equal to 0
    if pathSegments contains an illegal command
 */
EXPORT_C void
	vgAppendPathData(VGPath dstPath,
		VGint numSegments,
		const VGubyte * pathSegments,
		const void * pathData)
	{
	OPENVG_TRACE("vgAppendPathData dstPath=0x%x, numSegments=%d, pathSegments=0x%x, pathData=0x%x",
			dstPath, numSegments, pathSegments, pathData);

	TGuestOpenVg::vgAppendPathData(dstPath, numSegments, pathSegments, pathData);
	}

/*
 Modifies the coordinate data for a contiguous range of segments of dstPath, starting
 at startIndex (where 0 is the index of the first path segment) and having length
 numSegments. The data in pathData must be formatted in exactly the same manner as the
 original coordinate data for the given segment range, unless the path has been
 transformed using vgTransformPath or interpolated using vgInterpolatePath. In these
 cases, the path will have been subject to the segment promotion rules specified in
 those functions.

 The pathData pointer must be aligned on a 1-, 2-, or 4-byte boundary
 depending on the size of the coordinate datatype (as returned by querying the
 paths VG_PATH_DATATYPE parameter using vgGetParameteri). The
 VG_PATH_CAPABILITY_MODIFY capability must be enabled for path.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if dstPath is not a valid path handle, or is not shared with the current
     context
   VG_PATH_CAPABILITY_ERROR
    if VG_PATH_CAPABILITY_MODIFY is not enabled for dstPath
   VG_ILLEGAL_ARGUMENT_ERROR
    if pathData is NULL
    if pathData is not properly aligned
    if startIndex is less than 0
    if numSegments is less than or equal to 0
    if startIndex + numSegments is greater than the number of segments in the path
 */
EXPORT_C void
	vgModifyPathCoords(VGPath dstPath,
		VGint startIndex,
		VGint numSegments,
		const void * pathData)
	{
	OPENVG_TRACE("vgModifyPathCoords dstPath=0x%x, startIndex=%d, numSegments=%d, pathData=0x%x",
			dstPath, startIndex, numSegments, pathData);

	TGuestOpenVg::vgModifyPathCoords(dstPath, startIndex, numSegments, pathData);
	}

/*
 Appends a transformed copy of srcPath to the current contents of dstPath.
 The appended path is equivalent to the results of applying the current
 pathuser-to-surface transformation (VG_MATRIX_PATH_USER_TO_SURFACE) to srcPath.

 It is legal for srcPath and dstPath to be handles to the same path object, in
 which case the transformed path will be appended to the existing path. If
 srcPath and dstPath are handles to distinct path objects, the contents of
 srcPath will not be affected by the call.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if either dstPath or srcPath is not a valid path handle, or is not shared with
     the current context
   VG_PATH_CAPABILITY_ERROR
    if VG_PATH_CAPABILITY_TRANSFORM_FROM is not enabled for srcPath
    if VG_PATH_CAPABILITY_TRANSFORM_TO is not enabled for dstPath
 */
EXPORT_C void
	vgTransformPath(VGPath dstPath, VGPath srcPath)
	{
	OPENVG_TRACE("vgTransformPath dstPath=0x%x, srcPath=0x%x", dstPath, srcPath);

	TGuestOpenVg::vgTransformPath(dstPath, srcPath);
	}

/*
 Returns the point lying a given distance along a given portion of a path and
 the unit-length tangent vector at that point. Only the subpath consisting of the
 numSegments path segments beginning with startSegment (where the initial path
 segment has index 0) is used. For the remainder of this section we refer only to
 this subpath when discussing paths.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if path is not a valid path handle, or is not shared with the current context
   VG_PATH_CAPABILITY_ERROR
    If x and y are both non-NULL, and the VG_PATH_CAPABILITY_POINT_ALONG_PATH is
     not enabled for path
    If tangentX and tangentY are both non-NULL, and the
     VG_PATH_CAPABILITY_TANGENT_ALONG_PATH capability is not enabled for path
   VG_ILLEGAL_ARGUMENT_ERROR
    if startSegment is less than 0 or greater than the index of the final path
     segment
    if numSegments is less than or equal to 0
    if (startSegment + numSegments  1) is less than 0 or greater than the index
     of the final path segment
    if x, y, tangentX or tangentY is not properly aligned
 */
EXPORT_C void
	vgPointAlongPath(VGPath path,
		VGint startSegment,
		VGint numSegments,
		VGfloat distance,
		VGfloat * x, VGfloat * y,
		VGfloat * tangentX,
		VGfloat * tangentY)
	{
	OPENVG_TRACE("vgPointAlongPath path=0x%x, startSeg=%d, numSegs=%d, distance=%f, x=0x%x, y=0x%x, tangentX=0x%x, tangentY=0x%x",
			path, startSegment, numSegments, distance, x, y, tangentX, tangentY);

	TGuestOpenVg::vgPointAlongPath(path, startSegment, numSegments, distance, x, y, tangentX, tangentY);
	}

/*
 Performs filling and stroking. The paintModes argument is a bitwise OR of
 values from the VGPaintMode enumeration, determining whether the path is to be
 filled (VG_FILL_PATH), stroked (VG_STROKE_PATH), or both (VG_FILL_PATH |
 VG_STROKE_PATH). If both filling and stroking are to be performed, the path is
 first filled, then stroked.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if path is not a valid path handle, or is not shared with the current context
   VG_ILLEGAL_ARGUMENT_ERROR
    if paintModes is not a valid bitwise OR of values from the VGPaintMode
     enumeration
 */
EXPORT_C void
	vgDrawPath(VGPath path, VGbitfield paintModes)
	{
	OPENVG_TRACE("vgDrawPath path=0x%x paintModes=0x%x", path, paintModes);

	TGuestOpenVg::vgDrawPath(path, paintModes);
	}

/* Paint */

/*
 Deallocates the resources associated with a paint object. Following the call, the
 paint handle is no longer valid in any of the contexts that shared it. If the paint
 object is currently active in a drawing context, the context continues to access it
 until it is replaced or the context is destroyed.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if paint is not a valid paint handle, or is not shared with the current context
 */
EXPORT_C void vgDestroyPaint(VGPaint paint)
	{
	OPENVG_TRACE("vgDestroyPaint paint=0x%x", paint);

	TGuestOpenVg::vgDestroyPaint(paint);
	}

/*
 Set paint definitions on the current context. The paintModes argument is a
 bitwise OR of values from the VGPaintMode enumeration, determining whether
 the paint object is to be used for filling (VG_FILL_PATH), stroking
 (VG_STROKE_PATH), or both (VG_FILL_PATH | VG_STROKE_PATH). The current paint
 replaces the previously set paint object, if any, for the given paint mode
 or modes. If paint is equal to VG_INVALID_HANDLE, the previously set paint
 object for the given mode (if present) is removed and the paint settings are
 restored to their default values.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if paint is neither a valid paint handle nor equal to VG_INVALID_HANDLE,
     or is not shared with the current context
   VG_ILLEGAL_ARGUMENT_ERROR
    if paintModes is not a valid bitwise OR of values from the VGPaintMode
     enumeration
 */
EXPORT_C void
	vgSetPaint(VGPaint paint, VGbitfield paintModes)
	{
	OPENVG_TRACE("vgSetPaint paint=0x%x, paintModes=0x%x", paint, paintModes);

	TGuestOpenVg::vgSetPaint(paint, paintModes);
	}

/*
 As a shorthand, the vgSetColor function allows the VG_PAINT_COLOR parameter of a
 given paint object to be set using a 32-bit non-premultiplied sRGBA_8888
 representation. The rgba parameter is a VGuint with 8 bits of red starting at the
 most significant bit, followed by 8 bits each of green, blue, and alpha. Each color
 or alpha channel value is conceptually divided by 255. So as to obtain a value
 between 0 and 1.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if paint is not a valid paint handle, or is not shared with the current context
 */
EXPORT_C void
	vgSetColor(VGPaint paint, VGuint rgba)
	{
	OPENVG_TRACE("vgSetColor paint=0x%x, rgba=0x%x", paint, rgba);

	TGuestOpenVg::vgSetColor(paint, rgba);
	}

/*
 Replaces any previous pattern image defined on the given paint object for the given
 set of paint modes with a new pattern image. A value of VG_INVALID_HANDLE for the
 pattern parameter removes the current pattern image from the paint object.
 If the current paint object has its VG_PAINT_TYPE parameter set to VG_PAINT_TYPE_PATTERN,
 but no pattern image is set, the paint object behaves as if VG_PAINT_TYPE were set to
 VG_PAINT_TYPE_COLOR. While an image is set as the paint pattern for any paint object, it
 may not be used as a rendering target. Conversely, an image that is currently a rendering
 target may not be set as a paint pattern.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if paint is not a valid paint handle, or is not shared with the current context
    if pattern is neither a valid image handle nor equal to VG_INVALID_HANDLE, or is not
     shared with the current context
   VG_IMAGE_IN_USE_ERROR
    if pattern is currently a rendering target
 */
EXPORT_C void
	vgPaintPattern(VGPaint paint, VGImage pattern)
	{
	OPENVG_TRACE("vgPaintPattern paint=0x%x, pattern=0x%x", paint, pattern);

	TGuestOpenVg::vgPaintPattern(paint, pattern);
	}

/* Images */

/*
 Deallocates the resources associated with an image. Following the call, the image
 handle is no longer valid in any context that shared it. If the image is currently
 in use as a rendering target, is the ancestor of another image (see vgChildImage),
 is set as a paint pattern image on a VGPaint object, or is set as a glyph an a
 VGFont object, its definition remains available to those consumers as long as they
 remain valid, but the handle may no longer be used. When those uses cease, the
 images resources will automatically be deallocated.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if image is not a valid image handle, or is not shared with the current
	 context
 */
EXPORT_C void vgDestroyImage(VGImage image)
	{
	OPENVG_TRACE("vgDestroyImage image=0x%x -->", image);

	TGuestOpenVg::vgDestroyImage(image);
	OPENVG_TRACE("vgDestroyImage <--");
	}

/*
 Fills a given rectangle of an image with the color specified by the VG_CLEAR_COLOR parameter.
 The rectangle to be cleared is given by x, y, width, and height, which must define a positive
 region. The rectangle is clipped to the bounds of the image.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if image is not a valid image handle, or is not shared with the current
     context
   VG_IMAGE_IN_USE_ERROR
    if image is currently a rendering target
   VG_ILLEGAL_ARGUMENT_ERROR
    if width or height is less than or equal to 0
 */
EXPORT_C void
	vgClearImage(VGImage image,
		VGint x, VGint y,
		VGint width, VGint height)
	{
	OPENVG_TRACE("vgClearImage image=0x%x, x=%d, y=%d, width=%d, height=%d",
			image, x, y, width, height);

	TGuestOpenVg::vgClearImage(image, x, y, width, height);
	}

/*
 Read pixel values from memory, perform format conversion if necessary, and store
 the resulting pixels into a rectangular portion of an image.

 Pixel values are read starting at the address given by the pointer data; adjacent
 scanlines are separated by dataStride bytes. Negative or zero values of
 dataStride are allowed. The region to be written is given by x, y, width, and
 height, which must define a positive region. Pixels that fall outside the bounds
 of the image are ignored.

 Pixel values in memory are formatted according to the dataFormat parameter, which
 must contain a value from the VGImageFormat enumeration. The data pointer must
 be aligned according to the number of bytes of the pixel format specified by
 dataFormat, unless dataFormat is equal to VG_BW_1, VG_A_1, or VG_A_4, in
 which case 1 byte alignment is sufficient. Each pixel is converted into the format of
 the destination image as it is written.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if image is not a valid image handle, or is not shared with the current
     context
   VG_IMAGE_IN_USE_ERROR
    if image is currently a rendering target
   VG_UNSUPPORTED_IMAGE_FORMAT_ERROR
    if dataFormat is not a valid value from the VGImageFormat enumeration
   VG_ILLEGAL_ARGUMENT_ERROR
    if width or height is less than or equal to 0
    if data is NULL
    if data is not properly aligned
 */
EXPORT_C void
	vgImageSubData(VGImage image,
		const void * data,
		VGint dataStride,
		VGImageFormat dataFormat,
		VGint x, VGint y,
		VGint width, VGint height)
	{
	OPENVG_TRACE("vgImageSubData image=0x%x, data=0x%x, dataStride=%d, dataFormat=0x%x, x=%d, y=%d, width=%d, height=%d",
			image, data, dataStride, dataFormat, x, y, width, height);

	TGuestOpenVg::vgImageSubData(image, data, dataStride, dataFormat, x, y, width, height);
	}

/*
 Copies pixels between images. The source image pixel (sx + i, sy + j) is copied
 to the destination image pixel (dx + i, dy + j), for 0 <= i < width and
 0 <= j < height. Pixels whose source or destination lie outside of the bounds
 of the respective image are ignored. Pixel format conversion is applied as needed.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if either dst or src is not a valid image handle, or is not shared with the
     current context
   VG_IMAGE_IN_USE_ERROR
    if either dst or src is currently a rendering target
   VG_ILLEGAL_ARGUMENT_ERROR
    if width or height is less than or equal to 0
 */
EXPORT_C void
	vgCopyImage(VGImage dst, VGint dx, VGint dy,
		VGImage src, VGint sx, VGint sy,
		VGint width, VGint height,
		VGboolean dither)
	{
	OPENVG_TRACE("vgCopyImage dst=0x%x, dx=%d, dy=%d, src=0x%x, sx=%d, sy=%d, width=%d, height=%d, dither=%d",
			dst, dx, dy, src, sx, sy, width, height, dither);

	TGuestOpenVg::vgCopyImage(dst, dx, dy, src, sx, sy, width, height, dither);
	}

/*
 Draw an image to the current drawing surface. The current image-user-to-surface
 transformation Ti is applied to the image, so that the image pixel centered at
 (px + 0.5, py + 0.5) is mapped to the point (Ti)(px + 0.5, py + 0.5). In practice,
 backwards mapping may be used. That is, a sample located at (x, y) in the surface
 coordinate system is colored according to an interpolated image pixel value at the
 point (Ti)-1(x, y) in the image coordinate system. If Ti is non-invertible (or nearly
 so, within the limits of numerical accuracy), no drawing occurs.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if image is not a valid image handle, or is not shared with the current
     context
   VG_IMAGE_IN_USE_ERROR
    if image is currently a rendering target
 */
EXPORT_C void
	vgDrawImage(VGImage image)
	{
	OPENVG_TRACE("vgDrawImage image=0x%x", image);

	TGuestOpenVg::vgDrawImage(image);
	}

/*
 Copies pixel data from the image src onto the drawing surface. The image pixel
 (sx + i, sy + j) is copied to the drawing surface pixel (dx + i, dy + j), for
 0 <= i < width and 0 <= j < height. Pixels whose source lies outside of the
 bounds of src or whose destination lies outside the bounds of the drawing surface
 are ignored. Pixel format conversion is applied as needed. Scissoring takes place
 normally. Transformations, masking, and blending are not applied.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if src is not a valid image handle, or is not shared with the current context
   VG_IMAGE_IN_USE_ERROR
    if src is currently a rendering target
   VG_ILLEGAL_ARGUMENT_ERROR
    if width or height is less than or equal to 0
 */
EXPORT_C void
	vgSetPixels(VGint dx, VGint dy,
		VGImage src, VGint sx, VGint sy,
		VGint width, VGint height)
	{
	OPENVG_TRACE("vgSetPixels dx=%d, dy=%d, src==0x%x, sx=%d, sy=%d, width=%d, height=%d",
			dx, dy, src, sx, sy, width, height);

	TGuestOpenVg::vgSetPixels(dx, dy, src, sx, sy, width, height);
	}

/*
 Copy pixel data to the drawing surface without the creation of a VGImage object. The pixel
 values to be drawn are taken from the data pointer at the time of the vgWritePixels call,
 so future changes to the data have no effect. The effects of changes to the data by another
 thread at the time of the call to vgWritePixels are undefined.

 ERRORS
   VG_UNSUPPORTED_IMAGE_FORMAT_ERROR
    if dataFormat is not a valid value from the VGImageFormat enumeration
   VG_ILLEGAL_ARGUMENT_ERROR
    if width or height is less than or equal to 0
    if data is NULL
    if data is not properly aligned
 */
EXPORT_C void
	vgWritePixels(const void * data, VGint dataStride,
		VGImageFormat dataFormat,
		VGint dx, VGint dy,
		VGint width, VGint height)
	{
	OPENVG_TRACE("vgWritePixels data=0x%x, dataStride=%d, dataFormat=0x%x, dx=%d, dy=%d, width=%d, height=%d",
			data, dataStride, dataFormat, dx, dy, width, height);

	TGuestOpenVg::vgWritePixels(data, dataStride, dataFormat, dx, dy, width, height);
	}

/*
 Retrieves pixel data from the drawing surface into the image dst. The drawing surface
 pixel (sx + i, sy + j) is copied to pixel (dx + i, dy + j) of the image dst, for
 0 <= i < width and 0 <= j < height. Pixels whose source lies outside of the bounds of
 the drawing surface or whose destination lies outside the bounds of dst are ignored.
 Pixel format conversion is applied as needed. The scissoring region does not affect
 the reading of pixels.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if dst is not a valid image handle, or is not shared with the current context
   VG_IMAGE_IN_USE_ERROR
    if dst is currently a rendering target
   VG_ILLEGAL_ARGUMENT_ERROR
    if width or height is less than or equal to 0
 */
EXPORT_C void
	vgGetPixels(VGImage dst, VGint dx, VGint dy,
		VGint sx, VGint sy,
		VGint width, VGint height)
	{
	OPENVG_TRACE("vgGetPixels dst=0x%x, dx=%d, dy=%d, sx=%d, sy=%d, width=%d, height=%d",
			dst, dx, dy, sx, sy, width, height);

	TGuestOpenVg::vgGetPixels(dst, dx, dy, sx, sy, width, height);
	}

/*
 Copy pixels from one region of the drawing surface to another. Copies between
 overlapping regions are allowed and always produce consistent results identical to
 copying the entire source region to a scratch buffer followed by copying the scratch
 buffer into the destination region. The drawing surface pixel (sx + i, sy + j) is
 copied to pixel (dx + i, dy + j) for 0 <= i < width and 0 <= j < height. Pixels whose
 source or destination lies outside of the bounds of the drawing surface are ignored.
 Transformations, masking, and blending are not applied. Scissoring is applied to the
 destination, but does not affect the reading of pixels.

 ERRORS
   VG_ILLEGAL_ARGUMENT_ERROR
    if width or height is less than or equal to 0
 */
EXPORT_C void
	vgCopyPixels(VGint dx, VGint dy,
		VGint sx, VGint sy,
		VGint width, VGint height)
	{
	OPENVG_TRACE("vgCopyPixels dx=%d, dy=%d, sx=%d, sy=%d, width=%d, height=%d",
			dx, dy, sx, sy, width, height);

	TGuestOpenVg::vgCopyPixels(dx, dy, sx, sy, width, height);
	}

/*
 Destroys the VGFont object pointed to by the font argument.
 Note that vgDestroyFont will not destroy underlying objects that were used to
 define glyphs in the font. It is the responsibility of an application to destroy all
 VGPath or VGImage objects that were used in a VGFont, if they are no longer in
 use.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if font is not a valid font handle, or is not shared with the current context
 */
EXPORT_C void
	vgDestroyFont(VGFont font)
	{
	OPENVG_TRACE("vgDestroyFont font=0x%x", font);

	TGuestOpenVg::vgDestroyFont(font);
	}

/*
 Creates a new glyph and assigns the given path to a glyph associated with the
 glyphIndex in a font object. The glyphOrigin argument defines the coordinates of
 the glyph origin within the path, and the escapement parameter determines the
 advance width for this glyph. Both glyphOrigin and escapement coordinates are
 defined in the same coordinate system as the path. For glyphs that have no
 visual representation (e.g., the <space> character), a value of VG_INVALID_HANDLE
 is used for path. The reference count for the path is incremented.

 The path object may define either an original glyph outline, or an outline that
 has been scaled and hinted to a particular size (in surface coordinate units); this
 is defined by the isHinted parameter, which can be used by implementation
 for text-specific optimizations (e.g., heuristic auto-hinting of unhinted outlines).
 When isHinted is equal to VG_TRUE, the implementation will never apply
 auto-hinting; otherwise, auto hinting will be applied at the implementation's
 discretion.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if font is not a valid font handle, or is not shared with the current context
    if path is not a valid font handle or VG_INVALID_HANDLE, or is not shared
     with the current context
   VG_ILLEGAL_ARGUMENT_ERROR
    if the pointer to glyphOrigin or escapement is NULL or is not properly
     aligned
 */
EXPORT_C void
	vgSetGlyphToPath(VGFont font,
		VGuint glyphIndex,
		VGPath path,
		VGboolean isHinted,
		const VGfloat glyphOrigin [2],
		const VGfloat escapement[2])
	{
	OPENVG_TRACE("vgSetGlyphToPath font=0x%x, glyphIndex=%u, path=0x%x, isHinted=%d, glyphOrigin=0x%x, escapement=0x%x",
			font, glyphIndex, path, isHinted, glyphOrigin, escapement);

	TGuestOpenVg::vgSetGlyphToPath(font, glyphIndex, path, isHinted, glyphOrigin, escapement);
	}

/*
 Creates a new glyph and assigns the given image into a glyph associated with
 the glyphIndex in a font object. The glyphOrigin argument defines the
 coordinates of the glyph origin within the image, and the escapement parameter
 determines the advance width for this glyph. Both glyphOrigin and escapement
 coordinates are defined in the image coordinate system. Applying transformations
 to an image (other than translations mapped to pixel grid in surface coordinate
 system) should be avoided as much as possible. For glyphs that have no visual
 representation (e.g., the <space> character), a value of VG_INVALID_HANDLE is
 used for image. The reference count for the image is incremented.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if font is not a valid font handle, or is not shared with the current context
    if image is not a valid image handle or VG_INVALID_HANDLE, or is not
     shared with the current context
   VG_ILLEGAL_ARGUMENT_ERROR
    if the pointer to glyphOrigin or escapement is NULL or is not properly
     aligned
   VG_IMAGE_IN_USE_ERROR
    if image is currently a rendering target
 */
EXPORT_C void
	vgSetGlyphToImage(VGFont font,
		VGuint glyphIndex,
		VGImage image,
		const VGfloat glyphOrigin [2],
		const VGfloat escapement[2])
	{
	OPENVG_TRACE("vgSetGlyphToImage font=0x%x, glyphIndex=%d, image=0x%x, glyphOrigin=0x%x, escapement=0x%x",
			font, glyphIndex, image, (void*)glyphOrigin, (void*)escapement);

	TGuestOpenVg::vgSetGlyphToImage(font, glyphIndex, image, glyphOrigin, escapement);
	}

/*
 Deletes the glyph defined by a glyphIndex parameter from a font. The reference count
 for the VGPath or VGImage object to which the glyph was previously set is decremented,
 and the object's resources are released if the count has fallen to 0.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if font is not a valid font handle, or is not shared with the current context
   VG_ILLEGAL_ARGUMENT_ERROR
    if glyphIndex is not defined for the font
 */
EXPORT_C void
	vgClearGlyph(VGFont font,
		VGuint glyphIndex)
	{
	OPENVG_TRACE("vgClearGlyph font=0x%x, glyphIndex=%u", font, glyphIndex);

	TGuestOpenVg::vgClearGlyph(font, glyphIndex);
	}

/*
 Renders a glyph defined by the glyphIndex using the given font object. The
 user space position of the glyph (the point where the glyph origin will be
 placed) is determined by value of VG_GLYPH_ORIGIN. vgDrawGlyph calculates the
 new text origin by translating the glyph origin by the escapement vector of
 the glyph defined by glyphIndex. Following the call, the VG_GLYPH_ORIGIN
 parameter will be updated with the new origin.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if font is not a valid font handle, or is not shared with the current context
   VG_ILLEGAL_ARGUMENT_ERROR
    if glyphIndex has not been defined for a given font object
    if paintModes is not a valid bitwise OR of values from the VGPaintMode
     enumeration, or 0
 */
EXPORT_C void
	vgDrawGlyph(VGFont font,
		VGuint glyphIndex,
		VGbitfield paintModes,
		VGboolean allowAutoHinting)
	{
	OPENVG_TRACE("vgDrawGlyph font=0x%x, glyphIndex=%u, paintModes=0x%x, allowAutoHinting=%d",
			font, glyphIndex, paintModes, allowAutoHinting);

	TGuestOpenVg::vgDrawGlyph(font, glyphIndex, paintModes, allowAutoHinting);
	}

/*
 Renders a sequence of glyphs defined by the array pointed to by glyphIndices
 using the given font object. The values in the adjustments_x and adjustments_y
 arrays define positional adjustment values for each pair of glyphs defined by
 the glyphIndices array. The glyphCount parameter defines the number of elements
 in the glyphIndices and adjustments_x and adjustments_y arrays. The adjustment
 values defined in these arrays may represent kerning or other positional adjustments
 required for each pair of glyphs. If no adjustments for glyph positioning in a
 particular axis are required (all horizontal and/or vertical adjustments are zero),
 NULL pointers may be passed for either or both of adjustment_x and
 adjustment_y. The adjustments values should be defined in the same
 coordinate system as the font glyphs; if the glyphs are defined by path objects
 with path data scaled (e.g., by a factor of 1/units-per-EM), the values in the
 adjustment_x and adjustment_y arrays are scaled using the same scale factor.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if font is not a valid font handle, or is not shared with the current context
   VG_ILLEGAL_ARGUMENT_ERROR
    if glyphCount is zero or a negative value
    if the pointer to the glyphIndices array is NULL or is not properly
     aligned
    if a pointer to either of the adjustments_x or adjustments_y arrays are
     non-NULL and are not properly aligned
    if any of the glyphIndices has not been defined in a given font object
    if paintModes is not a valid bitwise OR of values from the VGPaintMode
     enumeration, or 0
 */
EXPORT_C void
	vgDrawGlyphs(VGFont font,
		VGint glyphCount,
		const VGuint * glyphIndices,
		const VGfloat * adjustments_x,
		const VGfloat * adjustments_y,
		VGbitfield paintModes,
		VGboolean allowAutoHinting)
	{
	OPENVG_TRACE("vgDrawGlyphs font=0x%x, glyphCount=%d, glyphIndices=0x%x, adjustments_y=0x%x, "
			"adjustments_x=0x%x, paintModes=0x%x, allowAutoHinting=%d",
			font, glyphCount, (void*)glyphIndices, (void*)adjustments_x, (void*)adjustments_y, paintModes, allowAutoHinting);

	TGuestOpenVg::vgDrawGlyphs(font, glyphCount, glyphIndices, adjustments_x, adjustments_y, paintModes, allowAutoHinting);
	}

/* Image Filters */

/*
 Computes a linear combination of color and alpha values (Rsrc, Gsrc, Bsrc,
 ALPHAsrc) from the normalized source image src at each pixel.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if either dst or src is not a valid image handle, or is not shared with the
     current context
   VG_IMAGE_IN_USE_ERROR
    if either dst or src is currently a rendering target
   VG_ILLEGAL_ARGUMENT_ERROR
    if src and dst overlap
    if matrix is NULL
    if matrix is not properly aligned
 */
EXPORT_C void
	vgColorMatrix(VGImage dst, VGImage src,
		const VGfloat * matrix)
	{
	OPENVG_TRACE("vgColorMatrix dst=0x%x, src=0x%x, matrix=0x%x", dst, src, matrix);

	TGuestOpenVg::vgColorMatrix(dst, src, matrix);
	}

/*
 Applies a user-supplied convolution kernel to a normalized source image src. The
 dimensions of the kernel are given by kernelWidth and kernelHeight; the kernel values
 are specified as kernelWidth*kernelHeight VGshorts in column-major order. That is,
 the kernel entry (i, j) is located at position i*kernelHeight + j in the input sequence.
 The shiftX and shiftY parameters specify a translation between the source and
 destination images. The result of the convolution is multiplied by a scale factor, and
 a bias is added.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if either dst or src is not a valid image handle, or is not shared with the
     current context
   VG_IMAGE_IN_USE_ERROR
    if either dst or src is currently a rendering target
   VG_ILLEGAL_ARGUMENT_ERROR
    if src and dst overlap
    if kernelWidth or kernelHeight is less than or equal to 0 or greater than
     VG_MAX_KERNEL_SIZE
    if kernel is NULL
    if kernel is not properly aligned
    if tilingMode is not one of the values from the VGTilingMode enumeration
 */
EXPORT_C void
	vgConvolve(VGImage dst, VGImage src,
		VGint kernelWidth, VGint kernelHeight,
		VGint shiftX, VGint shiftY,
		const VGshort * kernel,
		VGfloat scale,
		VGfloat bias,
		VGTilingMode tilingMode)
	{
	OPENVG_TRACE("vgConvolve dst=0x%x, src=0x%x, kW=%d, kH=%d, shX=%d, shY=%d, kernel=0x%x, scale=%f, bias=%f, tiling=0x%x",
			dst, src, kernelWidth, kernelHeight, shiftX, shiftY, kernel, scale, bias, tilingMode);

	TGuestOpenVg::vgConvolve(dst, src, kernelWidth, kernelHeight, shiftX, shiftY, kernel, scale, bias, tilingMode);
	}

/*
 Applies a user-supplied separable convolution kernel to a normalized source image src.
 A separable kernel is a two-dimensional kernel in which each entry kij is equal to a
 product kxi * kyj of elements from two one dimensional kernels, one horizontal and one
 vertical.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if either dst or src is not a valid image handle, or is not shared with the
     current context
   VG_IMAGE_IN_USE_ERROR
    if either dst or src is currently a rendering target
   VG_ILLEGAL_ARGUMENT_ERROR
    if src and dst overlap
    if kernelWidth or kernelHeight is less than or equal to 0 or greater than
     VG_MAX_SEPARABLE_KERNEL_SIZE
    if kernelX or kernelY is NULL
    if kernelX or kernelY is not properly aligned
    if tilingMode is not one of the values from the VGTilingMode
     enumeration
 */
EXPORT_C void
	vgSeparableConvolve(VGImage dst, VGImage src,
		VGint kernelWidth,
		VGint kernelHeight,
		VGint shiftX, VGint shiftY,
		const VGshort * kernelX,
		const VGshort * kernelY,
		VGfloat scale,
		VGfloat bias,
		VGTilingMode tilingMode)
	{
	OPENVG_TRACE("vgSeparableConvolve dst=0x%x, src=0x%x, kW=%d, kH=%d, shX=%d, shY=%d, kX=0x%x, kY=0x%x, scale=%f, bias=%f, tiling=0x%x",
			dst, src, kernelWidth, kernelHeight, shiftX, shiftY, kernelX, kernelY, scale, bias, tilingMode);

	TGuestOpenVg::vgSeparableConvolve(dst, src, kernelWidth, kernelHeight, shiftX, shiftY, kernelX, kernelY, scale, bias, tilingMode);
	}

/*
 Computes the convolution of a normalized source image src with a separable kernel
 defined in each dimension by the Gaussian function G(x, s).

 ERRORS
   VG_BAD_HANDLE_ERROR
    if either dst or src is not a valid image handle, or is not shared with the
     current context
   VG_IMAGE_IN_USE_ERROR
    if either dst or src is currently a rendering target
   VG_ILLEGAL_ARGUMENT_ERROR
    if src and dst overlap
    if stdDeviationX or stdDeviationY is less than or equal to 0 or greater
     than VG_MAX_GAUSSIAN_STD_DEVIATION
    if tilingMode is not one of the values from the VGTilingMode
     enumeration
 */
EXPORT_C void
	vgGaussianBlur(VGImage dst, VGImage src,
		VGfloat stdDeviationX,
		VGfloat stdDeviationY,
		VGTilingMode tilingMode)
	{
	OPENVG_TRACE("vgGaussianBlur dst=0x%x, src=0x%x, stdDeviationX=%f, stdDeviationY=%f, tilingMode=%d",
			dst, src, stdDeviationX, stdDeviationY, tilingMode);

	TGuestOpenVg::vgGaussianBlur(dst, src, stdDeviationX, stdDeviationY, tilingMode);
	}

/*
 Passes each image channel of the normalized source image src through a separate lookup
 table.

 Each channel of the normalized source pixel is used as an index into the lookup table for
 that channel by multiplying the normalized value by 255 and rounding to obtain an 8-bit
 integral value. Each LUT parameter should contain 256 VGubyte entries. The
 outputs of the lookup tables are concatenated to form an RGBA_8888 pixel
 value, which is interpreted as lRGBA_8888, lRGBA_8888_PRE, sRGBA_8888,
 or sRGBA_8888_PRE, depending on the values of outputLinear and  outputPremultiplied.

 The resulting pixels are converted into the destination format using the normal
 pixel format conversion rules.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if either dst or src is not a valid image handle, or is not shared with the
     current context
   VG_IMAGE_IN_USE_ERROR
    if either dst or src is currently a rendering target
   VG_ILLEGAL_ARGUMENT_ERROR
    if src and dst overlap
    if any pointer parameter is NULL
 */
EXPORT_C void
	vgLookup(VGImage dst, VGImage src,
		const VGubyte * redLUT,
		const VGubyte * greenLUT,
		const VGubyte * blueLUT,
		const VGubyte * alphaLUT,
		VGboolean outputLinear,
		VGboolean outputPremultiplied)
	{
	OPENVG_TRACE("vgLookup dst=0x%x, src=0x%x, redLUT=0x%x, greenLUT=0x%x, blueLUT=0x%x, alphaLUT=0x%x, opLin=%d, opPremul=%d",
			dst, src, redLUT, greenLUT, blueLUT, alphaLUT, outputLinear, outputPremultiplied);

	TGuestOpenVg::vgLookup(dst, src, redLUT, greenLUT, blueLUT, alphaLUT,  outputLinear, outputPremultiplied);
	}

/*
 Passes a single image channel of the normalized source image src, selected by the
 sourceChannel parameter, through a combined lookup table that produces whole pixel
 values. Each normalized source channel value is multiplied by 255 and rounded to
 obtain an 8 bit integral value.

 ERRORS
   VG_BAD_HANDLE_ERROR
    if either dst or src is not a valid image handle, or is not shared with the
     current context
   VG_IMAGE_IN_USE_ERROR
    if either dst or src is currently a rendering target
   VG_ILLEGAL_ARGUMENT_ERROR
    if src and dst overlap
    if src is in an RGB pixel format and sourceChannel is not one of VG_RED,
     VG_GREEN, VG_BLUE or VG_ALPHA from the VGImageChannel enumeration
    if lookupTable is NULL
    if lookupTable is not properly aligned
 */
EXPORT_C void
	vgLookupSingle(VGImage dst, VGImage src,
		const VGuint * lookupTable,
		VGImageChannel sourceChannel,
		VGboolean outputLinear,
		VGboolean outputPremultiplied)
	{
	OPENVG_TRACE("vgLookupSingle dst=0x%x, src=0x%x, lookupTable=0x%x, sourceChannel=0x%x, opLin=%d, opPremul=%d",
			dst, src, lookupTable, sourceChannel, outputLinear, outputPremultiplied);

	TGuestOpenVg::vgLookupSingle(dst, src, lookupTable, sourceChannel, outputLinear, outputPremultiplied);
	}

// guest support for OpenVG extension #4, KHR_EGL_image - vgCreateEGLImageTargetKHR(VGeglImageKHR image)
/*
 Creates an EGLImage target VGImage object from the provided EGLImage
 <image>.  <image> should be of type EGLImageKHR, cast into the type
 VGeglImageKHR.  Assuming no errors are generated in this function,
 the resulting VGImage will be an EGLImage target of the specified
 EGLImage <image>.  As a side-effect of the referencing operation,
 all of the pixel data in the <buffer> used as the EGLImage source
 resource (i.e., the <buffer> parameter passed to the CreateImageKHR
 command that returned <image>) will become undefined.

 ERRORS
      VG_UNSUPPORTED_IMAGE_FORMAT_ERROR
      - if the OpenVG implementation is not able to create a VGImage
        compatible with the provided VGeglImageKHR for an implementation-
        dependent reason (this could be caused by, but not limited to,
        reasons such as unsupported pixel formats, anti-aliasing quality,
        etc.).

      VG_ILLEGAL_ARGUMENT_ERROR
      - if <image> is not a valid VGeglImageKHR.
 */
VGImage	vgCreateEGLImageTargetKHR(VGeglImageKHR image)
	{
	OPENVG_TRACE("vgCreateEGLImageTargetKHR image=0x%x -->", image);

	VGImage imageHandle = TGuestOpenVg::vgCreateEGLImageTargetKHR(image);
	OPENVG_TRACE("vgCreateEGLImageTargetKHR imageHandle=0x%x <--", imageHandle);
	return imageHandle;
	}


} /* extern "C" */

// end of file openvg.cpp
