/*
* Copyright (c) 2009-2010 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:  Container for analog clock.
*
*/

#include "snsranalogclockcontainer.h"

#include <QDebug>
#include <QTime>
#include <QGraphicsLinearLayout>

#include <hblabel.h>
#include <hbextendedlocale.h>

#include "snsranalogclockwidget.h"
#include "snsrindicatorwidget.h"
#include "snsrswipewidget.h"

/*!
    \class SnsrAnalogClockContainer
    \ingroup group_snsrbigclockscreensaverplugin
    \brief Container used for preparing layout for analog clock.
 */

const char *gAnalogLayoutDocml = ":/xml/snsrbigclockscreensaveranalog.docml";

const char *gPortraitSectionName = "portrait";
const char *gLandscapeSectionName = "landscape";

const char *gMainViewName = "view";
const char *gMainContainerName = "mainContainer";
const char *gClockContainerName = "clockContainer";

const char *gDateLabelName = "dateLabel";
const char *gAnalogClockWidgetName = "analogClockWidget";
const char *gIndicatorWidgetName = "indicatorWidget";

const char *gSwipeWidgetName = "swipeWidget";

const char *gDateFormatVerticalStr = r_qtn_date_usual_with_zero; //"%E%,% %*D%*N%/0%4%/1%5";
const char *gDateFormatHorizontalStr = r_qtn_date_usual_with_zero; //"%E%,% %*D%N%/0%4%/1%5";


/*!
    Constructs a new SnsrAnalogClockContainer.
 */
SnsrAnalogClockContainer::SnsrAnalogClockContainer() :
    SnsrBigClockContainer(),
    mDateLabel(0),
    mAnalogClockWidget(0)/*,
    mSwipeWidget(0)*/
    {
    SCREENSAVER_TEST_FUNC_ENTRY("SnsrAnalogClockContainer::SnsrAnalogClockContainer")

    bool ok(true);

    // load analog clock
    qDebug() << gAnalogLayoutDocml;
    mDocumentObjects = mDocumentLoader.load(gAnalogLayoutDocml, &ok);
    Q_ASSERT_X(ok, gAnalogLayoutDocml, "Invalid DocML file.");
    if (ok) {
        mMainView = mDocumentLoader.findWidget(gMainViewName);
        mDateLabel = qobject_cast<HbLabel *>(
            mDocumentLoader.findWidget(gDateLabelName));
        mAnalogClockWidget = qobject_cast<SnsrAnalogClockWidget *>(
            mDocumentLoader.findWidget(gAnalogClockWidgetName));
        mIndicatorWidget = qobject_cast<SnsrIndicatorWidget *>(
            mDocumentLoader.findWidget(gIndicatorWidgetName));
        //We don't implement swipe widget at this poin
        /*mSwipeWidget = qobject_cast<SnsrSwipeWidget *>(
            mDocumentLoader.findWidget(gSwipeWidgetName));*/
        
        Q_ASSERT_X(
                mMainView && mDateLabel && mAnalogClockWidget &&
                mIndicatorWidget /*&& mSwipeWidget*/,
                gAnalogLayoutDocml, "Objects not found in DocML file."
                );
        
        //connect( mSwipeWidget, SIGNAL(swipeDownDetected()), SIGNAL(unlockRequested()) );

        mIndicatorWidget->setLayoutType(SnsrIndicatorWidget::IndicatorsCentered);
        
        mBackgroundContainerLayout->addItem(mMainView);
 //       mSwipeWidget->start();
    }

    SCREENSAVER_TEST_FUNC_EXIT("SnsrAnalogClockContainer::SnsrAnalogClockContainer")
}

/*!
    Destructs the class.
 */
SnsrAnalogClockContainer::~SnsrAnalogClockContainer()
{
    //mDateLabel, mAnalogClockWidget - deleted by the parent
}

/*!
    Updates displayed time and date.
 */
void SnsrAnalogClockContainer::update()
{
    SCREENSAVER_TEST_FUNC_ENTRY("SnsrAnalogClockContainer::update")

    // time
    mAnalogClockWidget->tick();

    // date
    if (mCurrentOrientation == Qt::Vertical) {
        mDateLabel->setPlainText(
            HbExtendedLocale().format(QDate::currentDate(), gDateFormatVerticalStr)
        );
    } else {
        mDateLabel->setPlainText(
            HbExtendedLocale().format(QDate::currentDate(), gDateFormatHorizontalStr)
        );
    }
    
    SCREENSAVER_TEST_FUNC_EXIT("SnsrAnalogClockContainer::update")
}

/*!
    Changes screensaver layout basing on orientation changes.
    \param orientation Current orientation.
 */
void SnsrAnalogClockContainer::changeLayout(Qt::Orientation orientation)
{
    SCREENSAVER_TEST_FUNC_ENTRY("SnsrAnalogClockContainer::changeLayout")

    bool ok(false);
    if (mCurrentOrientation != orientation) {
        mCurrentOrientation = orientation;

        // hide controls to avoid screen flickering
        mMainView->hide();

        QString sectionToLoad("");
        if (mCurrentOrientation == Qt::Horizontal) {
            sectionToLoad = gLandscapeSectionName;
        }
        qDebug() << "loading: " << gAnalogLayoutDocml << ", section: " << sectionToLoad;
        mDocumentLoader.load(gAnalogLayoutDocml, sectionToLoad, &ok);
        // view is rebuilt and ready to show
        update();
        mMainView->show();
        Q_ASSERT_X(ok, gAnalogLayoutDocml, "Invalid section in DocML file.");
    }
    // update anyway - this is needed in situations when screensaver goes to
    // foreground but layout change did not occur
    if (!ok) {
        update();
    }

    SCREENSAVER_TEST_FUNC_EXIT("SnsrAnalogClockContainer::changeLayout")
}

int SnsrAnalogClockContainer::updateIntervalInMilliseconds()
{
    return 1000;
}

