/*
* Copyright (c) 2010 Ixonos Plc.
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - Initial contribution
*
* Contributors:
* Ixonos Plc
*
* Description: 
* Static utility class.
*
*/


#ifndef IMAGEEDITORUTILS_H
#define IMAGEEDITORUTILS_H

#include <e32base.h>
#include <coemain.h>

class CMGXFileManager;
class RFs;


/**
 *  Static utility class
 *
 *  @code
 *   ?good_class_usage_example(s)
 *  @endcode
 *
 *  @lib ImageEditorUtils.lib
 *  @since S60 v5.0
 */
class ImageEditorUtils
{
public:

    /**  Memory location */
    enum TMemoryLocation
	  {
		  EPhone,
		  EMmc,
		  EMixed
    };

    /**  Memory location selection options  */
    enum TMemorySelection
    {
      ESelectAutomatic,
      ESelectPhone,
      ESelectMmc
    };

    /**
     * Generates a new name for the loaded document at the startup.
     * If automatic memory selection is used, the generated file is located on the 
     * MMC dy default. If the MMC is full or not present, the file will be in phone 
     * memory. Optionally the file can be forced onto either drive.
     * Internally, ImageFitsToDriveL is used to check the space.
     *
     * Media gallery album id list is no longer supported due to Media Gallery removal. 
     * If an invalid album list is passed the behaviour is undefined.     
     *
     * @since S60 v5.0
     * @param aFsSession file server session handle
     * @param aSourceFileName source file name
     * @param aTargetFileName target file name
     * @param aMgAlbumIdList list of the albums the source image belongs to
     * @param aReservedFileNames  additional list of already reserved file names
     * @param aMemoryInUse  where the target file is created
     * @return KErrNone if successfully generated file name
     *         KSIEENotEnoughDiskSpace if no disk space to save the file
     *         (size of aSourceFileName used to check the space)
     *         KSIEEOpenFile if the source filename is invalid
     */
     IMPORT_C static TInt GenerateNewDocumentNameL (
        RFs& aFsSession, 
        const TDesC& aSourceFileName, 
        TDes& aTargetFileName,
        RArray<TInt>* aMgAlbumIdList = NULL,
        const CDesCArray* aReservedFileNames = NULL,
        ImageEditorUtils::TMemorySelection aMemoryInUse = ImageEditorUtils::ESelectAutomatic
        );
        
    /**
     * Generates a new name for the loaded document in filename query. A 
     * unique filename is generated by adding running number after source 
     * filename until unique name is found. e.g. if image name is "Image", 
     * first attempt for new filename is "Image-001". If that is already 
     * in use, "Image-002" is tried and so on. First unused filename is 
     * then proposed for the user.
     * ImageFitsToDriveL is used to check the space. 
     *
     * @since S60 v5.0
     * @param aFsSession file server session handle
     * @param aSourceFileName source file name
     * @param aTargetFileName target file name
     * @param aDrive the drive where the target file is to be created
     * @param aReservedFileNames  additional list of already reserved file names
     * @return KErrNone if successfully generated file name
     *         KSIEENotEnoughDiskSpace if no disk space to save the file
     *         (size of aSourceFileName used to check the space)
     *         KSIEEOpenFile if the source filename is invalid
     */    
    IMPORT_C static TInt GenerateNewFileNameL (
        RFs& aFsSession, 
        const TDesC& aSourceFileName, 
        TDes& aTargetFileName,
        TFileName aDrive,
        const CDesCArray* aReservedFileNames = NULL );
        
        
    /**	GenerateFilePath
	 *	    
     * Generates a path for an image file. The memory selection option is given
     * as a parameter.
     * ImageEditorUtilsImageFitsToDriveL is used to check the space.
     *
     * Media gallery album id list is no longer supported due to Media Gallery removal. 
     * If an invalid album list is passed the behaviour is undefined.     
     *
     * @since S60 v5.0
     * @param aFsSession file server session handle
     * @param aSourceFileName source file name
     * @param aTargetFileName target file name
     * @param aMgAlbumIdList list of the albums the source image belongs to
     * @param aReservedFileNames  additional list of already reserved file names
     * @param aMemoryInUse  where the target file is created
     * @return KErrNone if successfully generated file name
     *         KSIEENotEnoughDiskSpace if no disk space to save the file
     *         (size of aSourceFileName used to check the space)
     *         KSIEEOpenFile if the source filename is invalid
	*/
	IMPORT_C static TInt GenerateFilePathL (
	    RFs& aFsSession, 
    	const TDesC& aSourceFileName, 
    	TDes& aTargetFileName,
    	ImageEditorUtils::TMemorySelection aMemoryInUse );

    /**
     * Checks whether there is enough disk space to save
     * the given file.
     *
     * @since S60 v5.0
     * @param aFsSession file server session handle
     * @param aSourceFile the file that would be saved
     * @param aTargetPath the path to save (only drive part needed)
     * @return ETrue if there is enough disk space to save the file
     *         EFalse if there is not enough disk space
     */
     IMPORT_C static TBool ImageFitsToDriveL (
        RFs& aFsSession,
        const TDesC& aSourceFile,
        const TDesC& aTargetPath
        );

    /**
     * Checks whether there is enough disk space to save
     * the given list of files.     
     *
     * @since S60 v5.0
     * @param aFsSession file server session handle
     * @param aSourceFileList the files that would be saved
     * @param aTargetFileList the paths to save (only drive part needed)
     * @return ETrue if there is enough disk space to save the files
     *         EFalse if there is not enough disk space
     */
     IMPORT_C static TBool ImagesFitToDriveL (
        RFs& aFsSession,
        const CDesCArray& aSourceFileList,
        const CDesCArray& aTargetFileList
        );

    /**
     * Notifies the system that a new image has been saved,
     * This is depreciated now and not used after MediaGallery removal.
     *
     * @since S60 v5.0
     * @param aFsSession file server session handle
     * @param aFileName name of the new file
     * @return -
     */
     IMPORT_C static void NotifyNewMediaDocumentL (
        RFs& aFsSession,
        const TDesC& aFileName
        );

    /**
     * Notifies the system that new images have been saved,
     * making them visible in the Media Gallery.
     * This is depreciated now and not used after MediaGallery removal.
     *
     * @since S60 v5.0
     * @param aFsSession file server session handle
     * @param aFileNameArray new file names
     * @return -
     */
     IMPORT_C static void NotifyNewMediaDocumentL (
        RFs& aFsSession,
        const MDesCArray& aFileNameArray
        );

    /**
     * Finds out in which Media Gallery Album the file belongs to.
     * If compiled without album support, this function
     * returns an empty array.
     * Reset() is called for the array at the begining.
     *
     * @since S60 v5.0
     * @param aAlbumList on return, contains the ID:s of the 
     *                   Media Gallery albums the file belongs to
     * @param aFileName the image file name
     * @return -
     */
     IMPORT_C static void FindAlbumsForImageFileL ( 
        RArray<TInt>& aAlbumIdList,
        const TDesC& aFilename
        );

    /**
     * Adds the given image file to the given album (e.g. video album).
     * If compiled without album support, this function does nothing.
     *
     * @since S60 v5.0
     * @param aFileName file name to be added to the album
     * @param aAlbumId id of the album
     * @return -
     */
     IMPORT_C static void AddImageFileToAlbumL ( 
                                 const TDesC& aFileName,
                                 TInt aAlbumId
                                 );

    /**
     * Helper function to get the manufacturer and model of the device.
     * Uses Shared Data client to retrieve the information.
     * If compiled on a public SDKs, which do not contain shared data,
     * this returns empty descriptors.
     *
     * @since S60 v5.0
     * @param aMake manufacturer of the device
     * @param aModel model of the device
     * @return -
     */
     IMPORT_C static void GetMakeAndModelL( TDes8& aMake, TDes8& aModel );

    /**
     * Copies file without opening it, using RFs::ReadFileSection.
     * Synchronous.
     *
     * @since S60 v5.0
     * @param aFsSession file server session handle
     * @param aSourceFileName source file name
     * @param aDestinationFileName destination file name
     * @param aOverwriteTarget is the target overwritten if it exists
     * @return error code
     */
     IMPORT_C static TInt CopyFile (
		                          RFs& aFsSession,
                              const TDesC& aSourceFileName,
                              const TDesC& aDestinationFileName, 
                              TBool aOverwrite = ETrue
                              );

private:

    /**
     * Finds the offset of the edited image suffix of form "-NNN", 
     * where NNN is the edit sequence number.
     *
     * @since S60 v5.0
     * @param aName image name without extension
     * @return offset of the edited image suffix
     */
     static TInt FindSuffix ( 
                    const TDesC &   aName
                    );

    /**
     * Checks if a file with the specified name already exists in the system.
     *
     * @since S60 v5.0
     * @param aFs file server session handle
     * @param aFileName file name to be checked
     * @param aReservedFileNames explicitly recerved file names
     * @return TBool
     */
     static TBool FileAlreadyExistsL ( 
                       RFs& aFsSession, 
                       const TDesC& aFileName,
                       const CDesCArray* aReservedFileNames 
                       );

    /**
     * Internal implementation of CopyFile
     *
     * @since S60 v5.0
     * @param aFsSession file server session handle
     * @param aSourceFileName source file
     * @param aDestinationFileName destination file
     * @param aSourceFileSize source file size in bytes
     * @return -
     */
     static void DoCopyL (
                   RFs& aFsSession,
                   const TDesC& aSourceFileName,
                   const TDesC& aDestinationFileName,
                   TUint aSourceFileSize 
                   );
};


#endif // IMAGEEDITORUTILS_H


