/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Lynne Kues (IBM Corp) - modified to reflect eSWT API subset
 *     Nokia Corporation - S60 implementation
 *******************************************************************************/
package org.eclipse.swt.widgets;

import org.eclipse.swt.internal.symbian.*;
import org.eclipse.swt.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.events.*;

/**
 * Instances of this class are selectable user interface
 * objects that represent a range of positive, numeric values.
 * <p>
 * At any given moment, a given scroll bar will have a
 * single <em>selection</em> that is considered to be its
 * value, which is constrained to be within the range of
 * values the scroll bar represents (that is, between its
 * <em>minimum</em> and <em>maximum</em> values).
 * </p><p>
 * Typically, scroll bars will be made up of five areas:
 * <ol>
 * <li>an arrow button for decrementing the value</li>
 * <li>a page decrement area for decrementing the value by a larger amount</li>
 * <li>a <em>thumb</em> for modifying the value by mouse dragging</li>
 * <li>a page increment area for incrementing the value by a larger amount</li>
 * <li>an arrow button for incrementing the value</li>
 * </ol>
 * Based on their style, scroll bars are either <code>HORIZONTAL</code>
 * (which have a left facing button for decrementing the value and a
 * right facing button for incrementing it) or <code>VERTICAL</code>
 * (which have an upward facing button for decrementing the value
 * and a downward facing buttons for incrementing it).
 * </p><p>
 * On some platforms, the size of the scroll bar's thumb can be
 * varied relative to the magnitude of the range of values it
 * represents (that is, relative to the difference between its
 * maximum and minimum values). Typically, this is used to
 * indicate some proportional value such as the ratio of the
 * visible area of a document to the total amount of space that
 * it would take to display it. SWT supports setting the thumb
 * size even if the underlying platform does not, but in this
 * case the appearance of the scroll bar will not change.
 * </p><p>
 * Scroll bars are created by specifying either <code>H_SCROLL</code>,
 * <code>V_SCROLL</code> or both when creating a <code>Scrollable</code>.
 * They are accessed from the <code>Scrollable</code> using
 * <code>getHorizontalBar</code> and <code>getVerticalBar</code>.
 * </p><p>
 * Note: Scroll bars are not Controls.  On some platforms, scroll bars
 * that appear as part of some standard controls such as a text or list
 * have no operating system resources and are not children of the control.
 * For this reason, scroll bars are treated specially.  To create a control
 * that looks like a scroll bar but has operating system resources, use
 * <code>Slider</code>.
 * </p>
 * <dl>
 * <dt><b>Styles:</b></dt>
 * <dd>HORIZONTAL, VERTICAL</dd>
 * <dt><b>Events:</b></dt>
 * <dd>Selection</dd>
 * </dl>
 * <p>
 * Note: Only one of the styles HORIZONTAL and VERTICAL may be specified.
 * </p><p>
 * IMPORTANT: This class is <em>not</em> intended to be subclassed.
 * </p>
 *
 * @see Slider
 * @see Scrollable
 * @see Scrollable#getHorizontalBar
 * @see Scrollable#getVerticalBar
 */

public class ScrollBar extends Widget
{
    int handle;
    Scrollable parent;

    /**
     * Constructs a new instance of this class given its parent
     * and a style value describing its behavior and appearance.
     * <p>
     * The style value is either one of the style constants defined in
     * class <code>SWT</code> which is applicable to instances of this
     * class, or must be built by <em>bitwise OR</em>'ing together
     * (that is, using the <code>int</code> "|" operator) two or more
     * of those <code>SWT</code> style constants. The class description
     * lists the style constants that are applicable to the class.
     * Style bits are also inherited from superclasses.
     * </p>
     *
     * @param parent a composite control which will be the parent of the new instance (cannot be null)
     * @param style the style of control to construct
     *
     * @exception IllegalArgumentException <ul>
     *    <li>ERROR_NULL_ARGUMENT - if the parent is null</li>
     * </ul>
     * @exception SWTException <ul>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the parent</li>
     *    <li>ERROR_INVALID_SUBCLASS - if this class is not an allowed subclass</li>
     * </ul>
     *
     * @see SWT#HORIZONTAL
     * @see SWT#VERTICAL
     * @see Widget#checkSubclass
     * @see Widget#getStyle
     */
    ScrollBar(Scrollable parent, int style)
    {
        super(parent, checkStyle(style));
        this.parent = parent;
        internal_createHandle();
    }

    /**
     * Adds the listener to the collection of listeners who will
     * be notified when the receiver's value changes, by sending
     * it one of the messages defined in the <code>SelectionListener</code>
     * interface.
     * <p>
     * When <code>widgetSelected</code> is called, the event object detail field contains one of the following values:
     * <code>0</code> - for the end of a drag.
     * <code>SWT.DRAG</code>.
     * <code>SWT.HOME</code>.
     * <code>SWT.END</code>.
     * <code>SWT.ARROW_DOWN</code>.
     * <code>SWT.ARROW_UP</code>.
     * <code>SWT.PAGE_DOWN</code>.
     * <code>SWT.PAGE_UP</code>.
     * <code>widgetDefaultSelected</code> is not called.
     * </p>
     *
     * @param listener the listener which should be notified
     *
     * @exception IllegalArgumentException <ul>
     *    <li>ERROR_NULL_ARGUMENT - if the listener is null</li>
     * </ul>
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     *
     * @see SelectionListener
     * @see #removeSelectionListener
     * @see SelectionEvent
     */
    public void addSelectionListener(SelectionListener listener)
    {
        checkWidget();
        if (listener == null) internal_error(SWT.ERROR_NULL_ARGUMENT);
        TypedListener typedListener = new TypedListener(listener);
        addListener(SWT.Selection,typedListener);
        addListener(SWT.DefaultSelection,typedListener);
    }

    static int checkStyle(int style)
    {
        return internal_checkBits(style, SWT.HORIZONTAL, SWT.VERTICAL, 0, 0, 0, 0);
    }

    protected void internal_createHandle()
    {
        handle = parent.createScrollBarHandle(this, style);

        if (handle == 0)
        {
            return;
        }

        OS.ScrollBar_SetIncrement(handle, 1);
        OS.ScrollBar_SetPageIncrement(handle, 10);

        /*
        * Do not set the initial values of the maximum
        * or the thumb.  These values normally default
        * to 100 and 10 but may have been set already
        * by the widget that owns the scroll bar.  For
        * example, a scroll bar that is created for a
        * list widget, setting these defaults would
        * override the initial values provided by the
        * list widget.
        */
    }

    protected void internal_prepareForRelease()
    {
        if (parent.horizontalBar == this)
        {
            parent.horizontalBar = null;
            OS.Scrollable_SetVisibleScrollBar(parent.scrollableHandle, SWT.HORIZONTAL, false);
        }
        if (parent.verticalBar == this)
        {
            parent.verticalBar = null;
            OS.Scrollable_SetVisibleScrollBar(parent.scrollableHandle, SWT.VERTICAL, false);
        }
        super.internal_prepareForRelease();
    }

    protected void internal_releaseHandle()
    {
        if (handle != 0)
        {
            OS.ScrollBar_Dispose(handle);
            handle = 0;
        }
        super.internal_releaseHandle();
    }

    /**
     * Returns <code>true</code> if the receiver is enabled, and
     * <code>false</code> otherwise. A disabled control is typically
     * not selectable from the user interface and draws with an
     * inactive or "grayed" look.
     *
     * @return the receiver's enabled state
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     *
     * @see #isEnabled
     */
    public boolean getEnabled()
    {
        checkWidget();
        return (state & DISABLED) == 0;
    }

    /**
     * Returns the amount that the receiver's value will be
     * modified by when the up/down (or right/left) arrows
     * are pressed.
     *
     * @return the increment
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     */
    public int getIncrement()
    {
        checkWidget();
        return OS.ScrollBar_GetIncrement(handle);
    }

    /**
     * Returns the maximum value which the receiver will allow.
     *
     * @return the maximum
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     */
    public int getMaximum()
    {
        checkWidget();
        return OS.ScrollBar_GetMaximum(handle);
    }

    /**
     * Returns the minimum value which the receiver will allow.
     *
     * @return the minimum
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     */
    public int getMinimum()
    {
        checkWidget();
        return OS.ScrollBar_GetMinimum(handle);
    }

    /**
     * Returns the amount that the receiver's value will be
     * modified by when the page increment/decrement areas
     * are selected.
     *
     * @return the page increment
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     */
    public int getPageIncrement()
    {
        checkWidget();
        return OS.ScrollBar_GetPageIncrement(handle);
    }

    /**
     * Returns the receiver's parent, which must be scrollable.
     *
     * @return the receiver's parent
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     */
    public Scrollable getParent()
    {
        checkWidget();
        return parent;
    }

    /**
     * Returns the single <em>selection</em> that is the receiver's value.
     *
     * @return the selection
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     */
    public int getSelection()
    {
        checkWidget();
        return OS.ScrollBar_GetThumbPosition(handle);
    }

    /**
     * Returns a point describing the receiver's size. The
     * x coordinate of the result is the width of the receiver.
     * The y coordinate of the result is the height of the
     * receiver.
     *
     * @return the receiver's size
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     */
    public Point getSize()
    {
        checkWidget();
        return OS.ScrollBar_GetSize(handle);
    }

    /**
     * Answers the size of the receiver's thumb relative to the
     * difference between its maximum and minimum values.
     *
     * @return the thumb value
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     *
     * @see ScrollBar
     */
    public int getThumb()
    {
        checkWidget();
        return OS.ScrollBar_GetThumbSpan(handle);
    }

    /**
     * Returns <code>true</code> if the receiver is visible, and
     * <code>false</code> otherwise.
     * <p>
     * If one of the receiver's ancestors is not visible or some
     * other condition makes the receiver not visible, this method
     * may still indicate that it is considered visible even though
     * it may not actually be showing.
     * </p>
     *
     * @return the receiver's visibility state
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     */
    public boolean getVisible()
    {
        checkWidget();
        return (state & HIDDEN) == 0;
    }

    /**
     * Returns <code>true</code> if the receiver is enabled and all
     * of the receiver's ancestors are enabled, and <code>false</code>
     * otherwise. A disabled control is typically not selectable from the
     * user interface and draws with an inactive or "grayed" look.
     *
     * @return the receiver's enabled state
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     *
     * @see #getEnabled
     */
    public boolean isEnabled()
    {
        checkWidget();
        return getEnabled() && parent.isEnabled();
    }

    /**
     * Returns <code>true</code> if the receiver is visible and all
     * of the receiver's ancestors are visible and <code>false</code>
     * otherwise.
     *
     * @return the receiver's visibility state
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     *
     * @see #getVisible
     */
    public boolean isVisible()
    {
        checkWidget();
        return getVisible() && parent.isVisible();
    }

    /**
     * Removes the listener from the collection of listeners who will
     * be notified when the receiver's value changes.
     *
     * @param listener the listener which should no longer be notified
     *
     * @exception IllegalArgumentException <ul>
     *    <li>ERROR_NULL_ARGUMENT - if the listener is null</li>
     * </ul>
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     *
     * @see SelectionListener
     * @see #addSelectionListener
     */
    public void removeSelectionListener(SelectionListener listener)
    {
        checkWidget();
        if (listener == null) internal_error(SWT.ERROR_NULL_ARGUMENT);
        if (eventTable == null) return;
        eventTable.unhook(SWT.Selection, listener);
        eventTable.unhook(SWT.DefaultSelection,listener);
    }

    /**
     * Enables the receiver if the argument is <code>true</code>,
     * and disables it otherwise. A disabled control is typically
     * not selectable from the user interface and draws with an
     * inactive or "grayed" look.
     *
     * @param enabled the new enabled state
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     */
    public void setEnabled(boolean enabled)
    {
        checkWidget();
        if (isEnabled() == enabled) return;
        state = (enabled ? state & ~DISABLED : state | DISABLED);
        OS.ScrollBar_SetEnabled(handle, enabled);
    }

    /**
     * Sets the amount that the receiver's value will be
     * modified by when the up/down (or right/left) arrows
     * are pressed to the argument, which must be at least
     * one.
     *
     * @param value the new increment (must be greater than zero)
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     */
    public void setIncrement(int value)
    {
        checkWidget();
        if (value < 1) return;
        OS.ScrollBar_SetIncrement(handle, value);
    }

    /**
     * Sets the maximum. If this value is negative or less than or
     * equal to the minimum, the value is ignored. If necessary, first
     * the thumb and then the selection are adjusted to fit within the
     * new range.
     *
     * @param value the new maximum
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     */
    public void setMaximum(int value)
    {
        checkWidget();
        if (value < 0) return;
        OS.ScrollBar_SetMaximum(handle, value);
    }

    /**
     * Sets the minimum value. If this value is negative or greater
     * than or equal to the maximum, the value is ignored. If necessary,
     * first the thumb and then the selection are adjusted to fit within
     * the new range.
     *
     * @param value the new minimum
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     */
    public void setMinimum(int value)
    {
        checkWidget();
        if (value < 0) return;
        OS.ScrollBar_SetMinimum(handle, value);
    }

    /**
     * Sets the amount that the receiver's value will be
     * modified by when the page increment/decrement areas
     * are selected to the argument, which must be at least
     * one.
     *
     * @param value the page increment (must be greater than zero)
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     */
    public void setPageIncrement(int value)
    {
        checkWidget();
        if (value < 1) return;
        OS.ScrollBar_SetPageIncrement(handle, value);
    }

    /**
     * Sets the single <em>selection</em> that is the receiver's
     * value to the argument which must be greater than or equal
     * to zero.
     *
     * @param selection the new selection (must be zero or greater)
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     */
    public void setSelection(int selection)
    {
        checkWidget();
        OS.ScrollBar_SetThumbPosition(handle, selection);
    }

    /**
     * Sets the size of the receiver's thumb relative to the
     * difference between its maximum and minimum values.  This new
     * value will be ignored if it is less than one, and will be
     * clamped if it exceeds the receiver's current range.
     *
     * @param value the new thumb value, which must be at least one and not
     * larger than the size of the current range
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     */
    public void setThumb(int value)
    {
        checkWidget();
        if (value < 1) return;
        OS.ScrollBar_SetThumbSpan(handle, value);
    }

    /**
     * Sets the receiver's selection, minimum value, maximum
     * value, thumb, increment and page increment all at once.
     * <p>
     * Note: This is equivalent to setting the values individually
     * using the appropriate methods, but may be implemented in a
     * more efficient fashion on some platforms.
     * </p>
     *
     * @param selection the new selection value
     * @param minimum the new minimum value
     * @param maximum the new maximum value
     * @param thumb the new thumb value
     * @param increment the new increment value
     * @param pageIncrement the new pageIncrement value
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     */
    public void setValues(int selection, int minimum, int maximum,
                          int thumb, int increment, int pageIncrement)
    {
        checkWidget();
        if (maximum < 0) return;
        if (minimum < 0 || minimum >= maximum) return;
        if (thumb < 1 || thumb > (maximum-minimum)) return;
        if (increment < 1 || pageIncrement < 1) return;
        if (selection < minimum) selection = minimum;
        if (selection > maximum) selection = maximum;
        OS.ScrollBar_SetValues(handle, selection, minimum, maximum, thumb, increment, pageIncrement);
    }

    /**
     * Marks the receiver as visible if the argument is <code>true</code>,
     * and marks it invisible otherwise.
     * <p>
     * If one of the receiver's ancestors is not visible or some
     * other condition makes the receiver not visible, marking
     * it visible may not actually cause it to be displayed.
     * </p>
     *
     * @param visible the new visibility state
     *
     * @exception SWTException <ul>
     *    <li>ERROR_WIDGET_DISPOSED - if the receiver has been disposed</li>
     *    <li>ERROR_THREAD_INVALID_ACCESS - if not called from the thread that created the receiver</li>
     * </ul>
     */
    public void setVisible(boolean visible)
    {
        checkWidget();
        state = (visible ? (state & ~HIDDEN) : (state | HIDDEN));

        OS.Scrollable_SetVisibleScrollBar(parent.scrollableHandle, style & (SWT.HORIZONTAL | SWT.VERTICAL), visible);
    }

    private final void sendScrollEvent(int detail)
    {
        Event event = new Event();
        event.detail = detail;
        internal_sendEvent(SWT.Selection, event);
    }

    int internal_getHandle()
    {
        return handle;
    }

}
