/*
 * Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
 * All rights reserved.
 * This component and the accompanying materials are made available
 * under the terms of "Eclipse Public License v1.0"
 * which accompanies this distribution, and is available
 * at the URL "http://www.eclipse.org/legal/epl-v10.html".
 *
 * Initial Contributors:
 * Nokia Corporation - initial contribution.
 *
 * Contributors:
 *
 * Description: Helper class to monitor msg construction in unified editor
 *
 */

#ifndef MSG_UNIFIED_EDITOR_MONITOR_H
#define MSG_UNIFIED_EDITOR_MONITOR_H

#include <QObject>
#include "convergedmessage.h"

class MsgUnifiedEditorView;
class HbWidget;
class UniEditorGenUtils;

/**
 * Helper class to monitor message's contruction in unified editor.
 * This class is for only reading editor's content to run its logic.
 * This class is NOT a place for content editing inside editor.
 */
class MsgUnifiedEditorMonitor : public QObject
    {
    Q_OBJECT

public:
    /**
     * Constructor
     */
    MsgUnifiedEditorMonitor(QObject* parent=0);

    /**
     * Destructor
     */
    ~MsgUnifiedEditorMonitor();

    /**
     * Set to skip showing note for first time
     * @param skip true/false
     */
    inline void setSkipNote(bool skip = true)
    {
        mSkipNote = skip;
    }
    
    /**
     * Seeker method to access current message type
     */
    static inline ConvergedMessage::MessageType messageType()
    {
        return mMessageType;
    }

    /**
     * seeker function to get current msg size in bytes
     */
    static inline int messageSize()
    {
        return mBodyContentSize + mContainerSize + mSubjectSize;
    }

    /**
     * seeker function to get current msg's body size 
     * in bytes
     */
    static inline int bodySize()
    {
        return mBodyContentSize;
    }

    /**
     * seeker function to get current msg's attachment
     * container size in bytes
     */
    static inline int containerSize()
    {
        return mContainerSize;
    }

    /**
     * seeker function to get current msg's subject
     * size in bytes
     */
    static inline int subjectSize()
    {
        return mSubjectSize;
    }

    /**
     * seeker function to get max possible mms size
     * in bytes
     */
    static inline int maxMmsSize()
    {
        return mMaxMmsSize;
    }
    
    /**
     * seeker funtion to get max recipient count for sms
     */
    static inline int maxSmsRecipients()
    {
        return mMaxSmsRecipients;
    }

    /**
     * seeker funtion to get max recipient count for mms
     */
    static inline int maxMmsRecipients()
    {
        return mMaxMmsRecipients;
    }
    
    /**
     * get total address count in To, Cc & Bcc fields
     */
    static inline int msgAddressCount()
    {
        return  mToAddressCount + mCcAddressCount + mBccAddressCount;
    }
    
    /**
     * tells if message is ready for send.
     */
    static inline bool readyForSend()
    {
        return mReadyForSend;
    }

public slots:
    /**
     * slot to handle content change in any editor component
     */
    void handleContentsChanged(const QVariant& data);

signals:
    /**
     * Emitted when composed message can be sent.
     * @param enable true if send is possible else false.
     */
    void enableSend(bool enable);
    
    /**
     * Emitte when contents are changed, to set drafts flag.
     */
    void contentsChanged();

private:
    /**
     * setter method to initialize content
     */
    void init();

    /**
     * Determine the projected msg type due to change in editor content
     */
    ConvergedMessage::MessageType projectedMsgType();

    /**
     * show type change discreet note
     */
    void showPopup(const QString& text);
    
  
    /**
     * check editor body for MMS content
     * @return true if MMS content is present
     */
    bool bodyHasMMSContent();
    
    /**
     * check editor subject for MMS content
     * @return true if MMS content is present
     */
    bool subjectHasMMSContent();
    
    /**
     * check editor attachment container for MMS content
     * @return true if MMS content is present
     */
    bool containerHasMMSContent();
    
    /**
     * check for other MMS content criteria
     * @return true if MMS criteria is met
     */
    bool otherMMSCriteriaMet();

    /**
     * Checks if composed message can be sent and emits a signal.
     */
    void updateSend();
    
    /**
     * Updates body content size.
     */
    void updateBodyContentSize();
    
    /**
     * resets value of static member variables.
     */
    void reset();

private:

    /**
     * Flag to skip showing the type change popup.
     * Note need to be skipped when an mms is opened from draft.
     */
    bool mSkipNote;
    
    /**
     * Flag to keep information, if attachment container has mms content.
     */
    bool mContainerHasMmsContent;
    
    /**
     * Priority.
     */
    ConvergedMessage::Priority mPriority;
    
    /**
     * Email id is present in to field or not.
     */
    bool mEmailPresent;
    
    /**
     * Body text has uni code char(s).
     */
    bool mUniCode;
    
    /**
     * Flag to hold state of image resizing
     */
    bool mImageResizing;

    /**
     * Type of msg under composition in the editor 
     */
    static ConvergedMessage::MessageType mMessageType;
    
    /**
     * msg body size in bytes
     */
    static int mBodySize;
    
    /**
     * msg body text size in bytes
     */
    static int mBodyTextSize;
    
    /**
     * msg body content (mBodySize + mBodyTextSize + mime headers) size in bytes.
     */
    static int mBodyContentSize;
    
    /**
     * msg container size in bytes
     */
    static int mContainerSize;
    
    /**
     * msg subject size in bytes
     */
    static int mSubjectSize;

    /**
     * max mms size in bytes
     */
    static int mMaxMmsSize;
    
    /**
     * max recipient count in an sms
     */
    static int mMaxSmsRecipients;

    /**
     * max recipient count in an sms
     */
    static int mMaxMmsRecipients;
    
    /**
     * Address count in various fields.
     */
    static int mToAddressCount;
    static int mCcAddressCount;
    static int mBccAddressCount;
    
    /**
     * flag to check if message is ready for send.
     */
    static bool mReadyForSend;

    /**
     * UniEditorGenUtils object
     * Owned
     */
    UniEditorGenUtils* mUniEditorGenUtils;
#ifdef UNIEDITOR_UNIT_TEST
    /**
     * Unit Testing
     */
    friend class TestMsgUnifiedEditorView;
    friend class TestMsgUnifiedEditorMonitor;
#endif
    };

#endif //MSG_UNIFIED_EDITOR_MONITOR_H
