/*
 * Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
 * All rights reserved.
 * This component and the accompanying materials are made available
 * under the terms of "Eclipse Public License v1.0"
 * which accompanies this distribution, and is available
 * at the URL "http://www.eclipse.org/legal/epl-v10.html".
 *
 * Initial Contributors:
 * Nokia Corporation - initial contribution.
 *
 * Contributors:
 *
 * Description:uni editor view class.
 *
 */

#ifndef MSG_UNIFIED_EDITOR_VIEW_H
#define MSG_UNIFIED_EDITOR_VIEW_H

#ifdef UNIEDITOR_UNIT_TEST
 #define UNIFIEDEDITOR_EXPORT
#else
 #ifdef UNIFIEDEDITOR_DLL
  #define UNIFIEDEDITOR_EXPORT Q_DECL_EXPORT
 #else
  #define UNIFIEDEDITOR_EXPORT Q_DECL_IMPORT
 #endif
#endif

#include "msgbaseview.h"
#include "convergedmessage.h"
#include "convergedmessageid.h"
#include <HbStaticVkbHost>

class HbAction;
class QGraphicsLinearLayout;
class MsgUnifiedEditorMonitor;
class UniEditorPluginLoader;
class MsgUnifiedEditorBaseWidget;
class MsgAudioFetcherDialog;
class MsgUnifiedEditorScrollArea;
class MsgUnifiedEditorBodyEditor;

/**
 * options in menu.
 */
enum MENU_ACTIONS
{
    PHOTO = 0x00,
    SOUND = 0x01,
    VCARD = 0x02,
    SUBJECT = 0x03,
    CCBCC = 0x04
};

/**
 * Custom vkb host specific to unifiededitor.
 */
class MsgUniEditorVkbHost: public HbStaticVkbHost
{
    Q_OBJECT
    
public:
    MsgUniEditorVkbHost(HbWidget *target) : HbStaticVkbHost(target){}
    
signals:
    /**
     * emitted when focus changes.
     */
    void focusChanged();
    
protected:
    /**
     * Reimpleted from base class.
     */
    void refresh()
    {
        emit focusChanged();
        HbStaticVkbHost::refresh();
    }
};

/**
 * unified editor view class.
 */
class UNIFIEDEDITOR_EXPORT MsgUnifiedEditorView : public MsgBaseView
    {
    Q_OBJECT

public:

    /**
     * Constructor
     */
    MsgUnifiedEditorView(QGraphicsItem *parent = 0);

    /**
     * Destructor
     */
    ~MsgUnifiedEditorView();

    /**
     * Populate editor with passed information
     * @param editorData data for editor's usage
     */
    void populateContent(const QVariantList& editorData);
    
    /**
     * Opens the drafts message into editor
     * @param editorData data for editor's usage
     */
    void openDraftsMessage(const QVariantList& editorData);

    /**
     * Saves the content inside editor to save
     */
    int saveContentToDrafts();
    
    /**
     * Event handler
     * @param key Key code
     * @return true if key event handled else false.
     * @see MsgBaseView
     */
    bool handleKeyEvent(int key);
    

private:

    /**
     * helper method to add menu.
     */
    void addMenu();

    /**
     * Packs the content inside editor into converged message
     * @param [OUT]msg, converged message to hold editor data
     * @param isSave, flag to indicate that msg needs to be packed
     * for saving to draft or not
     * @return returns error if something fails
     */
    int packMessage(ConvergedMessage &msg, bool isSave=false);

    /**
     * Populate editor with prepopulated msg content
     * @param messageDetails message details
     * @param draftMessage boolean for specifying draft message
     */
    void populateContentIntoEditor(const ConvergedMessage& messageDetails, 
                                   bool draftMessage,
                                   bool checkForInline);
    
    /**
     * Populate the editor with the message's content
     * for various cases like forward, reply, replyAll
     * @param messageId, unique id of message in store
     * @param messageType, sms or mms
     * @param editorOperation, forward or reply or replyAll etc
     */
    void fetchMessageFromStore(ConvergedMessageId& messageId,
                               ConvergedMessage::MessageType messageType,
                               int editorOperation);
    
    /**
     * Create VCards for all the contacts available in the list
     * @param value list of fetched contacts
     * @param [OUT]filelist list of newly added vcard files
     * @return errId KErrNone if no error
     */
    int createVCards(const QVariant& value, QStringList& filelist);
    
    /**
     * Resolve name conflict when adding attachments to editor
     * @param suggestedName suggested file name to use
     * @return fully qualified path of the file in unieditor's temp folder
     */
    QString generateFileName(QString& suggestedName);
    

    /**
     * To hide/show chrome.
     */
    void hideChrome(bool hide);
    
    /**
     * To initialize view.
     */
    void initView();
    
    /**
     * Creates temp folder for editor.
     */
    bool createTempFolder();
    
    /**
     * Removes editors temp folder.
     */
    void removeTempFolder();
    
private slots:

    /**
     * slot to add subject field.
     */
    void addSubject(bool needFocus = false);

    /**
     * slot to add Cc,Bcc field.
     */
    void addCcBcc(bool needFocus = false);

    /**
     * slot to current delete message.
     */
    void deleteMessage();

    /**
     * slot to images fetched.
     */
    void imagesFetched(const QVariant& result );

    /**
     * slot to receive fetched contacts
     */
    void contactsFetched(const QVariant& value);

    /**
     * slot to send message.
     */
    void send();
	
    /**
     * add attachments to the editor
     * @param files list of file paths
     */
    void addAttachments(QStringList files);
    
    /**
     * Signal emitted when an error is generated.
     * @param errorCode Error code.
     * @param errorMessage Error description.
     */
    void serviceRequestError(int errorCode, const QString& errorMessage);
   
    /**
     * Activate Input Blocker
     */
    void activateInputBlocker();

    /**
     * Deactivate Input Blocker
     */
    void deactivateInputBlocker();
    
    /**
     * Resizes the view when VKB is opened.
     * This slot is triggered when vkb is opened.
     */
    void vkbOpened();

    /**
     * Resizes the view when VKB is closed.
     * This slot is triggered when VKB focus is lost.
     */
    void vkbClosed();
    
    /**
     * Slot to do delayed construction.
     */
    void doDelayedConstruction();
    
    /**
     * Sets focus to item.
     */
    void setFocus(MsgUnifiedEditorBaseWidget* item);
    
    /**
     * Listens to contentChanged signal for various editor fields.
     */
    void onContentChanged();

	/**
     * This slot is called when delete message dialog is launched.
     * @param action selected action (yes or no).
     */
    void onDialogDeleteMsg(HbAction* action);

	/**
     * This slot is called when define sms settings dialog is launched.
     * @param action selected action (yes or no).
     */
    void onDialogSmsSettings(HbAction* action);
    
    /**
     * This slot is called when define mms settings dialog is launched.
     * @param action selected action (yes or no).
     */
    void onDialogMmsSettings(HbAction* action);    
     
     /** 
      * This slot is called after sound clip is 
      * selected from audio fetcher dialog    
      */
     void onAudioSelected(QString& filePath);
	 
	 /**
      * Check if the reply-path is broken
      * @return bool, true if reply-path constraints are broken
      */
     bool isReplyPathBroken();
     
     /**
      * Fetch images 
      */
     void fetchImages();

     /**
      * Fetch contacts
      */
     void fetchContacts();

     /**
      * Fetch audio
      */
     void fetchAudio();
     
     /**
      * Enable/Disable/removes menu actions.
      * @param opt, submenu actions.
      * @param isEnabled, true/false
      */
     void enableMenuAction(int action, bool enable);
     
     /**
      * Resets layout when vkb is cosed.
      */
     void resetLayout();     
     
     /**
      * Perform layouting when vkb is opened.
      */
     void doLayout();
     
     /**
      * Ensures visibility of editor widget in to visible area.
      */
     void ensureVisibility();
     
     /**
      * Updates focus widget when editor gains focus.
      */
     void focusChanged();
     
    
private:
     
     /**
      * main layout of view.
      */
    QGraphicsLinearLayout* mMainLayout;
    MsgUnifiedEditorMonitor* mMsgMonitor;
    UniEditorPluginLoader* mPluginLoader;
    
    /**
     * Message id of current opened message.
     */
    ConvergedMessageId mOpenedMessageId;
    
    /**
     * flag to check if message can be saved to drafts.
     */
	bool mCanSaveToDrafts;
	
    /**
     * Instance of VKB host
     */
	MsgUniEditorVkbHost* mVkbHost;
	
	/**
	 * Instance of Audio Fetcher Dialog
	 * Need to show when attachment audio selected
	 */
	MsgAudioFetcherDialog* mDialog;

    /**
     * Originating SC, 
     * to be used only when reply-path is available
     */
    QString mOriginatingSC;

    /**
     * Originating SME,
     * to be used only when reply-path is available
     */
    QString mOriginatingSME;

    /**
     * Flag to tell if reply-path is available
     */
    bool mReplyPath;	
    
    /**
     * Scrollable conatiner.
     */
    MsgUnifiedEditorScrollArea* mScrollArea;
    
    /**
     * Body text editor.
     */
    MsgUnifiedEditorBodyEditor* mBodyEditor;
    
    /**
     * Various actions.
     */
    HbAction* mSubjectAction;
    HbAction* mCcBccAction;
    HbAction* mPhotoAction;
    HbAction* mSoundAction;
     
    /**
     * Widget currently focussed.
     */
    QGraphicsWidget* mFocusedWidget;
    
    /**
     * Min height of foucssed widget.
     */
    qreal mMinHeight;
    
    /**
     * Max height of focussed widget.
     */
    qreal mMaxHeight;
	
#ifdef UNIEDITOR_UNIT_TEST
    /**
     * Unit Testing
     */
    friend class TestMsgUnifiedEditorView;
#endif
	
    };

#endif //MSG_UNIFIED_EDITOR_VIEW_H
