/*
 * Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
 * All rights reserved.
 * This component and the accompanying materials are made available
 * under the terms of "Eclipse Public License v1.0"
 * which accompanies this distribution, and is available
 * at the URL "http://www.eclipse.org/legal/epl-v10.html".
 *
 * Initial Contributors:
 * Nokia Corporation - initial contribution.
 *
 * Contributors:
 *
 * Description: Helper class to monitor msg construction in unified editor
 *
 */

// INCLUDES
#include "debugtraces.h"
#include <HbNotificationDialog>

// USER INCLUDES
#include "msgunieditormonitor.h"
#include "UniEditorGenUtils.h"

// Constants

// Define static
ConvergedMessage::MessageType MsgUnifiedEditorMonitor::mMessageType = ConvergedMessage::Sms;
int MsgUnifiedEditorMonitor::mBodySize = 0;
int MsgUnifiedEditorMonitor::mBodyTextSize = 0;
int MsgUnifiedEditorMonitor::mContainerSize = 0;
int MsgUnifiedEditorMonitor::mBodyContentSize = 0;
int MsgUnifiedEditorMonitor::mSubjectSize = 0;
int MsgUnifiedEditorMonitor::mMaxMmsSize = 0;
int MsgUnifiedEditorMonitor::mMaxSmsRecipients = 0;
int MsgUnifiedEditorMonitor::mMaxMmsRecipients = 0;
int MsgUnifiedEditorMonitor::mToAddressCount = 0;
int MsgUnifiedEditorMonitor::mCcAddressCount = 0;
int MsgUnifiedEditorMonitor::mBccAddressCount = 0;
bool MsgUnifiedEditorMonitor::mReadyForSend = false;

//Localized strings
#define LOC_POP_MESSAGE_CHANGE_MUL hbTrId("txt_messaging_dpopinfo_message_type_changed_to_mul")
#define LOC_POP_MESSAGE_CHANGE_TEXT hbTrId("txt_messaging_dpopinfo_message_type_changed_to_tex")

//---------------------------------------------------------------
// MsgUnifiedEditorMonitor::MsgUnifiedEditorMonitor
// @see header file
//---------------------------------------------------------------
MsgUnifiedEditorMonitor::MsgUnifiedEditorMonitor(QObject* parent) :
QObject(parent),
mSkipNote(false),
mContainerHasMmsContent(false),
mPriority(ConvergedMessage::Normal),
mEmailPresent(false),
mUniCode(false),
mImageResizing(false)
{    
    mUniEditorGenUtils = q_check_ptr( new UniEditorGenUtils);
    init(); 
}

//---------------------------------------------------------------
// MsgUnifiedEditorMonitor::~MsgUnifiedEditorMonitor
// @see header file
//---------------------------------------------------------------
MsgUnifiedEditorMonitor::~MsgUnifiedEditorMonitor()
{
    reset();
    delete mUniEditorGenUtils;
}

//---------------------------------------------------------------
// MsgUnifiedEditorMonitor::reset
// @see header file
//---------------------------------------------------------------
void MsgUnifiedEditorMonitor::reset()
{
    mMessageType = ConvergedMessage::Sms;
    mBodySize = 0;
    mBodyTextSize = 0;
    mContainerSize = 0;
    mBodyContentSize = 0;
    mSubjectSize = 0;
    mMaxMmsSize = 0;
    mMaxSmsRecipients = 0;
    mMaxMmsRecipients = 0;
    mToAddressCount = 0;
    mCcAddressCount = 0;
    mBccAddressCount = 0;
    mReadyForSend = false;
}

//---------------------------------------------------------------
// MsgUnifiedEditorMonitor::init
// @see header file
//---------------------------------------------------------------
void MsgUnifiedEditorMonitor::init()
{
    mMaxMmsSize = KDefaultMaxSize;
   
    TRAP_IGNORE(mMaxMmsSize = mUniEditorGenUtils->MaxMmsMsgSizeL());

    mMaxSmsRecipients = KDefaultSmsRecipients;
    TRAP_IGNORE(mMaxSmsRecipients = mUniEditorGenUtils->MaxSmsRecipientsL());

    mMaxMmsRecipients = KDefaultMmsRecipients;
    TRAP_IGNORE(mMaxMmsRecipients = mUniEditorGenUtils->MaxMmsRecipientsL());
}

//---------------------------------------------------------------
// MsgUnifiedEditorMonitor::handleContentsChanged
// @see header file
//---------------------------------------------------------------
void MsgUnifiedEditorMonitor::handleContentsChanged(const QVariant& data)
{
    QString objectName = sender()->objectName();
    
    if(objectName == "attachmentContainer")
    {
        QVariantList dataList = data.toList();
        mContainerSize = dataList.at(0).toInt();
        mContainerHasMmsContent = dataList.at(1).toBool();        
    }

    if(objectName == "body")
    {
        if(data.type() == QVariant::Bool)
        {
          mImageResizing = data.toBool();
        }
        else
        {
        mBodySize = data.toInt();
        updateBodyContentSize();
        }
    }
    
    if(objectName == "bodyText")
    {
        QVariantList dataList = data.toList();
        mBodyTextSize = dataList.at(0).toInt();
        mUniCode = dataList.at(1).toBool(); 
        updateBodyContentSize();
        
    }
    
    if(objectName == "subject")
    {
        QVariantList dataList = data.toList();
        mSubjectSize = dataList.at(0).toInt();
        mPriority = ConvergedMessage::Priority(dataList.at(1).toInt());
        
    }
    
    if(objectName == "to")
    {
        QVariantList dataList = data.toList();
        mToAddressCount = dataList.at(0).toInt();
        mEmailPresent = dataList.at(1).toBool();        
    }
    
    if(objectName == "cc")
    {
        QVariantList dataList = data.toList();
        mCcAddressCount = dataList.at(0).toInt();
    }
    
    if(objectName == "bcc")
    {
        QVariantList dataList = data.toList();
        mBccAddressCount = dataList.at(0).toInt(); 
    }
    
    // get the projected message type & show the type change note
    ConvergedMessage::MessageType newMsgType = projectedMsgType();    
    if(mMessageType != newMsgType)
    {
        mMessageType = newMsgType;
        QString noteStr;
        if(newMsgType == ConvergedMessage::Sms)
        {
            noteStr = LOC_POP_MESSAGE_CHANGE_TEXT;
        }
        else
        {
            noteStr = LOC_POP_MESSAGE_CHANGE_MUL;
        }
        showPopup(noteStr);
    }
    
    updateSend();
    
    //emit when content is modified after opening a draft message.
    if(!mSkipNote)
    {
    emit contentsChanged();
    }
}

//---------------------------------------------------------------
// MsgUnifiedEditorMonitor::projectedMsgType
// @see header file
//---------------------------------------------------------------
ConvergedMessage::MessageType MsgUnifiedEditorMonitor::projectedMsgType()
{
    ConvergedMessage::MessageType newMsgType = ConvergedMessage::Sms;

    // check if MMS content is present in any of the editor component
    if( bodyHasMMSContent() ||
        subjectHasMMSContent() ||
        containerHasMMSContent() ||
        otherMMSCriteriaMet() )
    {
        newMsgType = ConvergedMessage::Mms;
    }
    return newMsgType;
}

//---------------------------------------------------------------
// MsgUnifiedEditorMonitor::showPopup
// @see header file
//---------------------------------------------------------------
void MsgUnifiedEditorMonitor::showPopup(const QString& text)
{
    if(!mSkipNote)
    {
        HbNotificationDialog* dlg = new HbNotificationDialog();
        dlg->setFocusPolicy(Qt::NoFocus);
        dlg->setDismissPolicy(HbPopup::TapAnywhere);
        dlg->setAttribute(Qt::WA_DeleteOnClose, true);
        dlg->setTitle(text);
        dlg->show();
    }
    // reset skip note flag
    mSkipNote = false;
}

//---------------------------------------------------------------
// MsgUnifiedEditorMonitor::bodyHasMMSContent
// @see header file
//---------------------------------------------------------------
bool MsgUnifiedEditorMonitor::bodyHasMMSContent()
{
    bool result = false;

    int maxSmsSize = 0;
    TRAP_IGNORE(maxSmsSize = mUniEditorGenUtils->MaxSmsMsgSizeL(mUniCode));

    // If body text size exceeds sms text-size limit
    if( mBodyTextSize > maxSmsSize || mBodySize)
    {
        result = true;
    }

    return result;
}

//---------------------------------------------------------------
// MsgUnifiedEditorMonitor::subjectHasMMSContent
// @see header file
//---------------------------------------------------------------
bool MsgUnifiedEditorMonitor::subjectHasMMSContent()
{
    bool result = false;
    if(mPriority != ConvergedMessage::Normal || mSubjectSize)
    {
        result =  true;
    }
    
    return result;
}

//---------------------------------------------------------------
// MsgUnifiedEditorMonitor::containerHasMMSContent
// @see header file
//---------------------------------------------------------------
bool MsgUnifiedEditorMonitor::containerHasMMSContent()
{
    bool result = false;
    
    if(mContainerHasMmsContent || mContainerSize && mBodyTextSize)
    {
        result = true;
    }
    
    return result;
}

//---------------------------------------------------------------
// MsgUnifiedEditorMonitor::otherMMSCriteriaMet
// @see header file
//---------------------------------------------------------------
bool MsgUnifiedEditorMonitor::otherMMSCriteriaMet()
{
    bool result = false;
    
    if(mCcAddressCount || mBccAddressCount || mEmailPresent ||
       mToAddressCount > mMaxSmsRecipients)
    {
        result = true;
    }
    return result;
}

//---------------------------------------------------------------
// MsgUnifiedEditorMonitor::updateSend
// @see header file
//---------------------------------------------------------------
void MsgUnifiedEditorMonitor::updateSend()
{

    if ( (mToAddressCount + mCcAddressCount + mBccAddressCount) > 0 &&
         (mSubjectSize + mBodyContentSize + mContainerSize) > 0 &&
         !mImageResizing )
    {
        emit enableSend(true);
        mReadyForSend = true;
    }
    else
    {
        emit enableSend(false);
        mReadyForSend = false;
    }

}

void MsgUnifiedEditorMonitor::updateBodyContentSize()
{
    if(mBodySize || mBodyTextSize)
    {
        mBodyContentSize = mBodySize + mBodyTextSize + KEstimatedMimeHeaderSize 
                           + KEstimatedMmsSmilHeaderSize;
    }
    else
    {
        mBodyContentSize = 0; 
    }
}

//EOF
