/*
 * Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
 * All rights reserved.
 * This component and the accompanying materials are made available
 * under the terms of "Eclipse Public License v1.0"
 * which accompanies this distribution, and is available
 * at the URL "http://www.eclipse.org/legal/epl-v10.html".
 *
 * Initial Contributors:
 * Nokia Corporation - initial contribution.
 *
 * Contributors:
 *
 * Description:
 *
 */

//SYSTEM INCLUDES
#include <ccsrequesthandler.h>
#include <ccsconversationentry.h>
#include <ccsclientconversation.h>
#include <s60qconversions.h>

//USER INCLUDES
#include "msghistoryprivate.h"
#include "msghistoryimpl.h"
#include <msgitem.h>

// CONSTANTS
_LIT(KUnixEpoch, "19700000:000000.000000");


//---------------------------------------------------------------
// MsgHistoryPrivate::MsgHistoryPrivate
// @see header
//---------------------------------------------------------------
MsgHistoryPrivate::MsgHistoryPrivate( MsgHistoryImpl* ptr )
:q_ptr(ptr)
    {    
    handler = CCSRequestHandler::NewL();
    handler->RequestResultsEventL(this);
    }

//---------------------------------------------------------------
// MsgHistoryPrivate::MsgHistoryPrivate
// @see header
//---------------------------------------------------------------
MsgHistoryPrivate::~MsgHistoryPrivate()
    {
    if ( handler )
        delete handler;
    }

//---------------------------------------------------------------
// MsgHistoryPrivate::GetMessagingHistory
// @see header
//---------------------------------------------------------------
TBool MsgHistoryPrivate::GetMessagingHistory( const TInt aContactId )
    {
    TRAPD(err, handler->GetMessagingHistoryL(aContactId));
    if ( err == KErrNone )
        return ETrue;

    return EFalse;
    }

//---------------------------------------------------------------
// MsgHistoryPrivate::ClearMessagingHistory
// @see header
//---------------------------------------------------------------
TBool MsgHistoryPrivate::ClearMessagingHistory( const TInt aContactId )
    {
    TRAPD(err, handler->ClearMessagingHistoryL(aContactId));
    if ( err == KErrNone )
        return ETrue;

    return EFalse;
    }

//---------------------------------------------------------------
// MsgHistoryPrivate::MarkReadMessagingHistory
// @see header
//---------------------------------------------------------------
TBool MsgHistoryPrivate::MarkReadMessagingHistory( const TInt aContactId )
    {
    TRAPD(err, handler->MarkMessagingHistoryReadL(aContactId));
    if ( err == KErrNone )
        return ETrue;

    return EFalse;
    }

//---------------------------------------------------------------
// MsgHistoryPrivate::Subscribe
// @see header
//---------------------------------------------------------------
TBool MsgHistoryPrivate::Subscribe( const TInt aContactId )
    {
    TInt conversationId = -1;
    TRAPD(err, 
            conversationId = handler->GetConversationIdL(aContactId));
    CCsClientConversation* clientConversation = CCsClientConversation::NewL();
    clientConversation->SetConversationEntryId(conversationId);
    // set the contact id, which will be used in case when the 
    // conversation entry id is -1.
    clientConversation->SetContactId(aContactId);
    //set dummy entry
    CCsConversationEntry *entry = CCsConversationEntry::NewL();
    clientConversation->SetConversationEntryL(entry);

    handler->RequestConversationChangeEventL(this, clientConversation);
    delete clientConversation;
    delete entry;        
    if ( err == KErrNone )
        return ETrue;

    return EFalse;
    }

//---------------------------------------------------------------
// MsgHistoryPrivate::UnSubscribe
// @see header
//---------------------------------------------------------------
TBool MsgHistoryPrivate::UnSubscribe( const TInt aContactId )
    {
    TInt conversationId = -1;
    TRAPD(err, 
            conversationId = handler->GetConversationIdL(aContactId));
    CCsClientConversation* clientConversation = CCsClientConversation::NewL();
    clientConversation->SetConversationEntryId(conversationId);
    // set the contact id, which will be used in case when the 
    // conversation entry id is -1.
    clientConversation->SetContactId(aContactId);
    //set dummy entry
    CCsConversationEntry *entry = CCsConversationEntry::NewL();
    clientConversation->SetConversationEntryL(entry);

    handler->RemoveConversationChangeEventL(this, clientConversation);
    delete clientConversation;
		
	delete entry;
    if ( err == KErrNone )
        return ETrue;

    return EFalse;
    }

//---------------------------------------------------------------
// MsgHistoryPrivate::ConversationList
// @see header
//---------------------------------------------------------------
void MsgHistoryPrivate::ConversationList
( RPointerArray<CCsClientConversation>& /*aClientConversationList*/ )
    {
    // No implementation required.
    }

//---------------------------------------------------------------
// MsgHistoryPrivate::Conversations
// @see header
//---------------------------------------------------------------
void MsgHistoryPrivate::Conversations
( RPointerArray<CCsConversationEntry>& aConversationEntryList )
    {
    QList<MsgItem> msgs;

    for(TInt i=aConversationEntryList.Count()-1; i >= 0; --i )
        {
        MsgItem item;
        PopulateMsgItem(item,*(aConversationEntryList[i]));
        msgs.append(item);
        }  
    //emit signal
    q_ptr->messagesReadyEvent(msgs);
    }

//---------------------------------------------------------------
// MsgHistoryPrivate::AddConversation
// @see header
//---------------------------------------------------------------
void MsgHistoryPrivate::AddConversation 
( const CCsConversationEntry& aConversationEntry )
    {
    MsgItem item;
    PopulateMsgItem(item, aConversationEntry);
    q_ptr->messageAddedEvent(item);
    }

//---------------------------------------------------------------
// MsgHistoryPrivate::ModifyConversation
// @see header
//---------------------------------------------------------------
void MsgHistoryPrivate::ModifyConversation
( const CCsConversationEntry& aConversationEntry )
    {
    MsgItem item;
    PopulateMsgItem(item, aConversationEntry);
    q_ptr->messageChangedEvent(item);
    }

//---------------------------------------------------------------
// MsgHistoryPrivate::DeleteConversation
// @see header
//---------------------------------------------------------------
void MsgHistoryPrivate::DeleteConversation
( const CCsConversationEntry& aConversationEntry )
    {
    MsgItem item;
    PopulateMsgItem(item, aConversationEntry);
    q_ptr->messageDeletedEvent(item);
    }

//---------------------------------------------------------------
// MsgHistoryPrivate::RefreshConversation
// @see header
//---------------------------------------------------------------
void MsgHistoryPrivate::RefreshConversation()
    {
    // No implementation required.
    }

//---------------------------------------------------------------
// MsgHistoryPrivate::populateMsgItem
// @see header
//---------------------------------------------------------------
void MsgHistoryPrivate::PopulateMsgItem
(MsgItem& item, const CCsConversationEntry& entry)
    {
    // id             
    item.setId(entry.EntryId());
    
    // message type.        
    item.setType((MsgItem::MsgType)entry.GetType());

    // description
    HBufC* description = entry.Description();    
    if( description && description->Length())
        {
        item.setBody(S60QConversions::s60DescToQString(*description)); 
        }

    // time stamp 
    TTime unixEpoch(KUnixEpoch);
    TTimeIntervalSeconds seconds;
    TTime timeStamp(entry.TimeStamp() );
    timeStamp.SecondsFrom(unixEpoch, seconds);    
    item.setTimeStamp(seconds.Int());

    //contact details
    HBufC* contact = entry.Contact();
    if( contact && contact->Length())
        {
        item.setPhoneNumber(S60QConversions::s60DescToQString(*contact));      
        }

    //sending state
    item.setState((MsgItem::MsgState)entry.GetSendState());    
    
    // direction
    item.setDirection((MsgItem::MsgDirection)entry.ConversationDir());  
    
    //attributes
    SetMsgAttributes(item, entry);    
    }

//---------------------------------------------------------------
// MsgHistoryPrivate::SetMsgAttributes
// @see header
//---------------------------------------------------------------
void MsgHistoryPrivate::SetMsgAttributes
(MsgItem& item, const CCsConversationEntry& entry)
    {
    //draft
    if(entry.IsAttributeSet(ECsAttributeDraft))
        {
        item.changeAttributes(MsgItem::MsgAttributeDraft); 
        }
    //sent
    if(entry.IsAttributeSet(ECsAttributeSent))
        {
        item.changeAttributes(MsgItem::MsgAttributeSent);  
        }
    //new
    if(entry.IsAttributeSet(ECsAttributeNew))
        {
        item.changeAttributes(MsgItem::MsgAttributeNew); 
        }
    //unread
    if(entry.IsAttributeSet(ECsAttributeUnread))
        {
        item.changeAttributes(MsgItem::MsgAttributeUnread); 
        }
    //High priority
    if(entry.IsAttributeSet(ECsAttributeHighPriority))
        {
        item.changeAttributes(MsgItem::MsgAttributeHighPriority); 
        }
    //low priority
    if(entry.IsAttributeSet(ECsAttributeLowPriority))
        {
        item.changeAttributes(MsgItem::MsgAttributeLowPriority); 
        }
    //attachment
    if(entry.IsAttributeSet(ECsAttributeAttachment))
        {
        item.changeAttributes(MsgItem::MsgAttributeAttachment); 
        }
    }

//EOF
