/*
* Copyright (c) 2007 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:  Implementation of MPX common container Texture Manager.
*
*/


#include <alf/alftexturemanager.h>
#include <fbs.h>
#include <mpxlog.h>
#include "mpxcommoncontainertexturemanager.h"

// CONSTANTS

// ----------------------------------------------------------------------------
// Constructor.
// ----------------------------------------------------------------------------
//
CMPXCommonContainerTextureManager::CMPXCommonContainerTextureManager(
    CAlfTextureManager& aTextureManager )
    : iTextureManager( aTextureManager )
    {
    // Set the TextureId range. Now we can set our own ID outside this Range
    iTextureManager.SetAutomaticTextureIdRange(
        KMPXFirstAlfThumbnailTextureId,
        KMPXLastAlfThumbnailTextureId);
    }

// ----------------------------------------------------------------------------
// Destructor
// ----------------------------------------------------------------------------
//
CMPXCommonContainerTextureManager::~CMPXCommonContainerTextureManager()
    {
    iTextureArray.Close();
    }

// ----------------------------------------------------------------------------
// Creates texture from the specified bitmap and mask
// ----------------------------------------------------------------------------
CAlfTexture& CMPXCommonContainerTextureManager::CreateTextureL(
    TInt aItemIndex,
    TMPXItemId aItemId,
    CFbsBitmap* aBitmap,
    CFbsBitmap* aMask )
    {
    MPX_DEBUG1("CMPXCommonContainerTextureManager::CreateTextureL(): Entering");

    // store the bitmaps and the item id for the ProvideBitmapL callback
    iTextureItem.iType = EMPXTextureImage;
    iTextureItem.iItemIndex = aItemIndex;
    iTextureItem.iItemId = aItemId;
    
    // store the bitmaps for the ProvideBitmapL callback
    iTextureItem.iBitmap = aBitmap;
    iTextureItem.iMask = aMask;

    /// @todo if CreateTextureL leaves, we may need to delete
    // iBitmap and iMask as we transfer their
    // ownership to Alf and if a leave occurs the ownership transfer
    // might have not yet happened
    CAlfTexture& texture = iTextureManager.CreateTextureL(
        KAlfAutoGeneratedTextureId,
        this,
        EAlfTextureFlagDefault );

    MPX_DEBUG1("CMPXCommonContainerTextureManager::CreateTextureL(): Exiting");
    return texture;
    }

// ----------------------------------------------------------------------------
// Creates icon texture from the specified bitmap and mask (not cached)
// ----------------------------------------------------------------------------
CAlfTexture& CMPXCommonContainerTextureManager::CreateIconTextureL(
    TInt aIconIndex,
    CFbsBitmap* aBitmap,
    CFbsBitmap* aMask )
    {
    MPX_DEBUG1("CMPXCommonContainerTextureManager::CreateTextureL(): Entering");

    // save the icon index
    iTextureItem.iType = EMPXTextureIcon;
    iTextureItem.iItemIndex = aIconIndex;
    
    // store the bitmaps for the ProvideBitmapL callback
    iTextureItem.iBitmap = aBitmap;
    iTextureItem.iMask = aMask;

    /// @todo if CreateTextureL leaves, we may need to delete
    // iBitmap and iMask as we transfer their
    // ownership to Alf and if a leave occurs the ownership transfer
    // might have not yet happened
    CAlfTexture& texture = iTextureManager.CreateTextureL(
        KAlfAutoGeneratedTextureId,
        this,
        EAlfTextureFlagDefault );

    MPX_DEBUG1("CMPXCommonContainerTextureManager::CreateTextureL(): Exiting");
    return texture;
    }

// ----------------------------------------------------------------------------
// Updates the specified media with the results from the search
// criteria
// ----------------------------------------------------------------------------
void CMPXCommonContainerTextureManager::ProvideBitmapL(
    TInt aId,
    CFbsBitmap*& aBitmap,
    CFbsBitmap*& aBitmapMask)
    {
    MPX_DEBUG1("CMPXCommonContainerTextureManager::ProvideBitmapL(): Entering");

    // store the ID
	iTextureItem.iTextureId = aId;
    // give the bitmaps to Alf, it takes ownership
    aBitmap = iTextureItem.iBitmap;
    aBitmapMask = iTextureItem.iMask;

    // cache the texture item
    if( iTextureItem.iType == EMPXTextureImage )
        {
        iTextureArray.AppendL( iTextureItem );
        }
    else // EMPXTextureIcon
        {
        iIconTextureArray.AppendL( iTextureItem );        
        }
    
    MPX_DEBUG1("CMPXCommonContainerTextureManager::ProvideBitmapL(): Exiting");
    }

// ----------------------------------------------------------------------------
// Returns the texture id for the specified item
// ----------------------------------------------------------------------------
TInt CMPXCommonContainerTextureManager::TextureId( TMPXItemId aItemId ) const
    {
    MPX_FUNC("CMPXCommonContainerTextureManager::TextureId()");

    TInt textureId( KErrNotFound );

    // Search for the texture id in the cache
    const TInt count( iTextureArray.Count() );
    for( TInt index = 0; index < count; index++ )
        {
        if( iTextureArray[index].iItemId == aItemId )
            {
            textureId = iTextureArray[index].iTextureId;
            break;
            }
        }
    
    return textureId;
    }

// ----------------------------------------------------------------------------
// Returns the texture id for the specified icon
// ----------------------------------------------------------------------------
TInt CMPXCommonContainerTextureManager::IconTextureId( TInt aIconIndex ) const
    {
    MPX_FUNC("CMPXCommonContainerTextureManager::IconTextureId()");
    
    TInt textureId( KErrNotFound );
    
    // Search for the texture id in the cache
    const TInt count( iIconTextureArray.Count() );
    for( TInt index = 0; index < count; index++ )
        {
        if( iIconTextureArray[index].iItemIndex == aIconIndex )
            {
            textureId = iIconTextureArray[index].iTextureId;
            break;
            }
        }
    
    return textureId;
    }

// ----------------------------------------------------------------------------
// Clears the texture cache array
// ----------------------------------------------------------------------------
void CMPXCommonContainerTextureManager::ClearCache( TMPXTextureType aType )
    {
    MPX_FUNC("CMPXCommonContainerTextureManager::ClearCache");
    switch ( aType )
        {
        case EMPXTextureImage:
            iTextureArray.Reset();
            break;
        case EMPXTextureIcon:
            iIconTextureArray.Reset();
            break;
        case EMPXTextureNone:
            iTextureArray.Reset();
            iIconTextureArray.Reset();
            break;
        }
    }

//End of file
