/*
* Copyright (c) 2006 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:    My Videos Client API header file.*
*/




#ifndef CIPTVMYVIDEOSCLIENT_H
#define CIPTVMYVIDEOSCLIENT_H

#include <e32base.h>
#include "CIptvUtil.h"
#include "MIptvMyVideos.h"
#include "RIptvClientSession.h"

class CIptvMyVideosVideoBriefDetails;
class CIptvMyVideosVideoFullDetails;
class CIptvMyVideosFolder;
class CIptvMyVideosGlobalFileId;
class CIptvMyVideosGlobalFolderId;
class MIptvMyVideosClientObserver;
class CIptvMyVideosContentUpdateObserver;
class CIptvMyVideosTotalVideoLengthObserver;
class CIptvMyVideosCopyOrMoveObserver;

/**
 *  My Videos Client API.
 *
 *  @lib IptvEngineClient.dll
 *  @since
 */
class CIptvMyVideosClient : public CActive,
                            public MIptvMyVideos
    {

public:

    /**
     * Two-phased constructor.
     *
     * @param aClientObserver Observer.
     * @param aMaintenance In MDS version only MDS maintenance handling client
     *        sets this parameter to true. In legacy DB version maintenance is
     *        performed by default and for example MobileTV client should set
     *        this to true to SKIP the maintenance.
     * @return New CIptvMyVideosClient object.
     */
    static IMPORT_C CIptvMyVideosClient* NewL(
        MIptvMyVideosClientObserver& aClientObserver,
        TBool aMaintenance = EFalse );

    /**
     * Two-phased constructor.
     *
     * @param aClientObserver Observer.
     * @param aMaintenance In MDS version only MDS maintenance handling client
     *        sets this parameter to true. In legacy DB version maintenance is
     *        performed by default and for example MobileTV client should set
     *        this to true to SKIP the maintenance.
     * @return New CIptvMyVideosClient object.
     */
    static IMPORT_C CIptvMyVideosClient* NewLC(
        MIptvMyVideosClientObserver& aClientObserver,
        TBool aMaintenance = EFalse );

    /**
     * Destructor.
     */
    virtual ~CIptvMyVideosClient();

// From MIptvMyVideos

    /**
     * Gets folder list in the alphabetical order.
     *
     * @param aParentFolderId Points to folder ID which sub-folders should be listed.
     *        Note that the root folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL().
     * @param aFrom The start index for listed folders.
     * @param aAmount Amount of folders to be listed. If zero, all are listed.
     * @param aTotalAmount Reference to a variable where total amount of folders is returned.
     * @param aFolderList Reference to a list of items returned.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt GetFolderListL(
        CIptvMyVideosGlobalFolderId& aParentFolderId,
        TUint32 aFrom,
        TUint32 aAmount,
        TUint32& aTotalAmount,
        RPointerArray<CIptvMyVideosFolder>& aFolderList );

    /**
     * Gets parent forlder ID identifier.
     * Note that at the beginning of the browse operation, the root folder ID is generated by
     * CIptvMyVideosGlobalFolderId::NewRootFolderL().
     *
     * @param aFolderId Points to a folder ID which parent folder is requested.
     * @param aParentFolder Reference to a storage where to parent folder ID is returned.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt GetParentFolderL(
        CIptvMyVideosGlobalFolderId& aFolderId,
        CIptvMyVideosGlobalFolderId& aParentFolder );

    /**
     * Gets folder name.
     *
     * @param aFolderId ID of the folder which name is requested.
     * @param aName On return, name of the folder.
     * @return KErrNone or one of the system wide error codes.
     */
    IMPORT_C TInt GetFolderNameL(
        CIptvMyVideosGlobalFolderId& aFolderId,
        TDes& aName);

    /**
     * Delete folder asynchronously.
     * Deletion includes all sub forlders and files at the tree.
     * MIptvMyVideosClientObserver::DeleteResponse is called when operation is finished.
     *
     * @param aFolderId Points to folder ID to be deleted.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt DeleteFolderReqL(
        CIptvMyVideosGlobalFolderId& aFolderId );

    /**
     * Delete folder.
     * Deletion includes all sub forlders and files at the tree.
     *
     * @param aFolderId Points to folder ID to be deleted.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt DeleteFolderL(
        CIptvMyVideosGlobalFolderId& aFolderId );

    /**
     * Gets a video list in a chronological order.
     *
     * @param aParentFolderId Points to folder ID which video files should be listed.
     *                        Note that the root folder ID is generated by
     *                        CIptvMyVideosGlobalFolderId::NewRootFolderL().
     * @param aFrom           The start index for listed folders.
     * @param aAmount         Amount of folders to be listed. If zero, all listed.
     * @param aTotalAmount    Reference to a variable where total amount of folders is returned.
     * @param aVideoList      Reference to a list of items returned.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt GetVideoListL(
        CIptvMyVideosGlobalFolderId& aParentFolderId,
        TUint32 aFrom,
        TUint32 aAmount,
        TUint32& aTotalAmount,
        RPointerArray<CIptvMyVideosVideoBriefDetails>& aVideoList );

    /**
     * Gets full video details.
     *
     * @param aFileId Identifier to a video file requested.
     * @param aFullDetails Reference to a list of items returned.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt GetVideoDetailsL(
        CIptvMyVideosGlobalFileId& aFileId,
        CIptvMyVideosVideoFullDetails& aFullDetails );

    /**
     * Gets full video details by file localpath
     *
     * @since R2.0
     * @param aLocalPath Path to video file, used as search key.
     * @param aFullDetails Reference to a item to be returned.
     * @return Error code, KErrNone if no errors, KErrNotFound if there is no such
     *         video. On unexpected error, one of the other system wide error codes.
     */
    IMPORT_C TInt GetVideoDetailsForPathL(
        const TDesC& aLocalPath,
        CIptvMyVideosVideoFullDetails& aFullDetails );

    /**
     * Requests total video lengt and total file size (sum of all videos).
     * The result is returned asynchronously trough observer callback.
     *
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt GetTotalVideoLengthRequestL();

    /**
     * Sets video details information.
     *
     * @param aVideoFullDetails New video details information.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt SetVideoDetailsL(
        CIptvMyVideosVideoFullDetails& aVideoFullDetails );

    /**
     * Deletes a video asynchronously.
     * Operation includes clean up that removes files from file system,
     * deletes data related to FileId and releases FileId identifier.
     * MIptvMyVideosClientObserver::DeleteResponse is called when operation is finished.
     *
     * @param aFileId File ID identifier for the video to be deleted.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt DeleteVideoReqL(
        CIptvMyVideosGlobalFileId& aFileId );

    /**
     * Deletes a video.
     * Operation includes clean up that removes files from file system,
     * deletes data related to FileId and releases FileId identifier.
     *
     * @param aFileId File ID identifier for the video to be deleted.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt DeleteVideoL(
        CIptvMyVideosGlobalFileId& aFileId );

    /**
     * Copy video file asynchronously to another My Videos Folder.
     * Target drive must be different that source drive.
     * MIptvMyVideosClientObserver::CopyOrMoveComplete is called when operation is finished.
     *
     * @param aSourceFileId Source video file ID to be copied.
     * @param aTargetFolderId My Videos target folder where to be copied. Note if target is a root,
     *        that root folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt CopyReqL(
        CIptvMyVideosGlobalFileId& aSourceFileId,
        CIptvMyVideosGlobalFolderId& aTargetFolderId );

    /**
     * Copy video file synchronously to another My Videos Folder.
     * Target drive must be different that source drive.
     *
     * @param aSourceFileId Source video file ID to be copied.
     * @param aTargetFolderId My Videos target folder where to be copied. Note if target is a root, that root
     *        folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt CopyL(
        CIptvMyVideosGlobalFileId& aSourceFileId,
        CIptvMyVideosGlobalFolderId& aTargetFolderId );

    /**
     * Copy My Videos folder tree and videos asynchronously.
     * Target drive must be different that source drive.
     * MIptvMyVideosClientObserver::CopyOrMoveComplete is called when operation is finished.
     *
     * @param aSourceFolderId Source folder ID to be copied.
     * @param aTargetFolderId My Videos target folder where to be copied. Note if target is a root, that
     *        root folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt CopyReqL(
        CIptvMyVideosGlobalFolderId& aSourceFolderId,
        CIptvMyVideosGlobalFolderId& aTargetFolderId );

    /**
     * Copy My Videos folder tree and videos synchronously.
     * Target drive must be different that source drive.
     *
     * @param aSourceFolderId Source folder ID to be copied.
     * @param aTargetFolderId My Videos target folder where to be copied. Note if target is a root, that
     *        root folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @param aFailed On return, count of items that could not be copied.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt CopyL(
        CIptvMyVideosGlobalFolderId& aSourceFolderId,
        CIptvMyVideosGlobalFolderId& aTargetFolderId,
        TUint32& aFailed );

    /**
     * Move video file to another My Videos Folder asynchronously.
     * If target is in an another physical file system drive, actual video contents is also moved.
     * MIptvMyVideosClientObserver::CopyOrMoveComplete is called when operation is finished.
     *
     * @param aSourceFileId Source video file ID to be moved.
     * @param aTargetFolderId My Videos target folder where to be moved. Note if target is a root, that
     *        root folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt MoveReqL(
        CIptvMyVideosGlobalFileId& aSourceFileId,
        CIptvMyVideosGlobalFolderId& aTargetFolderId );

    /**
     * Move video file to another My Videos Folder synchronously.
     * If target is in an another physical file system drive, actual video contents is also moved.
     *
     * @param aSourceFileId Source video file ID to be moved.
     * @param aTargetFolderId My Videos target folder where to be moved. Note if target is a root, that root
     *        folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt MoveL(
        CIptvMyVideosGlobalFileId& aSourceFileId,
        CIptvMyVideosGlobalFolderId& aTargetFolderId );

    /**
     * Move My Videos folder tree and videos asynchronously.
     * If target is in an another physical file system drive, actual video contents is also moved.
     * MIptvMyVideosClientObserver::CopyOrMoveComplete is called when operation is finished.
     *
     * @param aSourceFolderId Source folder ID to be moved.
     * @param aTargetFolderId My Videos target folder where to be moved. Note if target is a root, that
     *        root folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt MoveReqL(
        CIptvMyVideosGlobalFolderId& aSourceFolderId,
        CIptvMyVideosGlobalFolderId& aTargetFolderId );

    /**
     * Move My Videos folder tree and videos synchronously.
     * If target is in an another physical file system drive, actual video contents is also moved.
     *
     * @param aSourceFolderId Source folder ID to be moved.
     * @param aTargetFolderId My Videos target folder where to be moved. Note if target is a root, that root
     *        folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @param aFailed On return, count of items that could not be moved.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt MoveL(
        CIptvMyVideosGlobalFolderId& aSourceFolderId,
        CIptvMyVideosGlobalFolderId& aTargetFolderId,
        TUint32& aFailed );

    /**
     * Cancels any ongoing move or copy operation.
     * MIptvMyVideosClientObserver::CopyOrMoveComplete is called when cancel is actually done.
     *
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt CancelCopyOrMoveL();

    /**
     * Rename My Videos folder.
     *
     * @param aFolderId Folder to be renamed.
     * @param aName New name string.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt RenameDatabaseFolderL(
        CIptvMyVideosGlobalFolderId& aFolderId,
        TDesC& aName );

    /**
     * Create a new My Videos folder.
     *
     * @param aParentFolderId Parent folder ID where the new folder is created. Note if the parent folder is a root,
     *        that root folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @param aName Name fot the new folder.
     * @param aNewFolderId Reference to a storage where new folder ID is returned.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    IMPORT_C TInt NewDatabaseFolderL(
        CIptvMyVideosGlobalFolderId& aParentFolderId,
        TDesC& aName,
        CIptvMyVideosGlobalFolderId& aNewFolderId );

    /**
     * Adds a new video entry to the database.
     *
     * @since R2.0
     * @param aFullDetails             Full details structure of the video to be added to database.
     *                                 Note, this parameter is both input and output parameter.
     * @param aVideoType               Type of video, see enumeration TVideoType.
     * @param aSizeEstimateInKiloBytes Estimate of the size of the new video. Method will ensure
     *                                 that there is enough free space on disk for the video.
     * @return Error code, KErrNone if no errors, KErrNotSupported if video type to add is not supported,
     *         KErrArgument if arguments are invalid, KErrDiskFull if there is no space for new video,
     *         KErrNotReady if external resources are not available, and possibly other error in
     *         unexpected error  cases.
     */
    IMPORT_C TInt CreateVideoL(
        CIptvMyVideosVideoFullDetails& aFullDetails,
        TVideoType aVideoType,
        TUint32 aSizeEstimateInKiloBytes );

    /**
     * Ensures that there is enough of free space on disk for new video. This version of the
     * method cannot change the used drive, it just tries to free space on current drive.
     *
     * @since R2.0
     * @param aFileId                   Global File ID of the file that needs the space.
     * @param aRequiredSpaceInKiloBytes Required space (size of file) in kilobytes.
     * @return Error code, KErrNone if no errors, KErrArgument if the argument is invalid,
     *         KErrDiskFull if cannot free enough space for video, and possibly other error
     *         in unexpected error cases.
     */
    IMPORT_C TInt EnsureFreeSpaceL(
        CIptvMyVideosGlobalFileId& aFileId,
        TUint32 aRequiredSpaceInKiloBytes );

    /**
     * Ensures that there is enough of free space on disk for new video. This version of the
     * method might transfer the video to another drive if current drive cannot provide enough
     * of free space. If the video is transferred, the original video entry is removed.
     *
     * @since R2.0
     * @param aCurrentFileId            Global File ID of the file that needs the space.
     * @param aRequiredSpaceInKiloBytes Required space (size of file) in kilobytes.
     * @param aNewFileId                Possible new Global File ID of the file if file needs to
     *                                  be re-located on another drive. Matches original File ID
     *                                  if there is no need for re-location.
     * @return Error code, KErrNone if no errors, KErrArgument if the argument is invalid,
     *         KErrDiskFull if video needs to be re-located to another drive, KErrCancel if
     *         cannot free enough space for video, and possibly other error in unexpected
     *         error cases.
     */
    IMPORT_C TInt EnsureFreeSpaceL(
        CIptvMyVideosGlobalFileId& aCurrentFileId,
        TUint32 aRequiredSpaceInKiloBytes,
        CIptvMyVideosGlobalFileId& aNewFileId );

    /**
     * Count total video length and total file size (sum of all videos).
     *
     * @param aTotalPlayTime Total play time (not used).
     * @param aTotalFileSize Total file size in kilo bytes.
     * @return Error code,
     *         KErrNone if no errors,
     *         otherwise one of the system wide error codes.
     */
    IMPORT_C TInt GetTotalVideoLengthL(
        TIptvPlayTime& aTotalPlayTime,
        TIptvFileSize& aTotalFileSize );

    /**
     * Deletes all videos which are under downloading.
     *
     * @return Error code,
     *         KErrNone if no errors,
     *         otherwise one of the system wide error codes.
     */
    IMPORT_C TInt DeleteAllDownloadsL();

private:

    /**
     * From CActive, handles an active objects request completion event.
     */
    void RunL();

    /**
     * From CActive, implements cancellation of an outstanding request.
     * This function is called as part of the active objects Cancel().
     */
    void DoCancel();

    /**
     * From CActive, handles errors in RunL method.
     */
    TInt RunError( TInt aError );

    /**
     * C++ default constructor.
     *
     * @param aClientObserver Observer.
     */
    CIptvMyVideosClient( MIptvMyVideosClientObserver& aClientObserver );

    /**
     * Symbian 2nd phase constructor.
     *
     * @param aMaintenance In MDS version only MDS maintenance handling client
     *        sets this parameter to true. In legacy DB version maintenance is
     *        performed by default and for example MobileTV client should set
     *        this to true to SKIP the maintenance.
     */
    void ConstructL( TBool aMaintenance );

protected: // Sending of initial requests to server.

    /**
     * Send request to server.
     *
     * @param aMessage     Message to send.
     * @param aFileIdParam Message parameter, global file Id.
     * @return KErrNone or one of system wide error codes.
     */
    TInt SendRequestL(
        TInt aMessage, CIptvMyVideosGlobalFileId& aFileIdParam );

    /**
     * Send request to server.
     *
     * @param aMessage        Message to send.
     * @param aSourceFileId   Message parameter, global file Id.
     * @param aTargetFolderId Message parameter, global folder Id.
     * @return KErrNone or one of system wide error codes.
     */
    TInt SendRequestL(
        TInt aMessage,
        CIptvMyVideosGlobalFileId& aSourceFileId,
        CIptvMyVideosGlobalFolderId& aTargetFolderId );

    /**
     * Send request to server.
     *
     * @param aMessage       Message to send.
     * @param aFolderIdParam Message parameter, global folder Id.
     * @return KErrNone or one of system wide error codes.
     */
    TInt SendRequestL(
        TInt aMessage, CIptvMyVideosGlobalFolderId& aFolderIdParam );

    /**
     * Send request to server.
     *
     * @param aMessage  Message to send.
     * @param aFolderId Message parameter, global folder Id.
     * @param aName     Message parameter, name.
     * @return KErrNone or one of system wide error codes.
     */
    TInt SendRequestL(
        TInt aMessage,
        CIptvMyVideosGlobalFolderId& aFolderId,
        const TDesC& aName );

    /**
     * Send request to server.
     *
     * @param aMessage        Message to send.
     * @param aSourceFolderId Message parameter, global folder Id.
     * @param aTargetFolderId Message parameter, global folder Id.
     * @return KErrNone or one of system wide error codes.
     */
    TInt SendRequestL(
        TInt aMessage,
        CIptvMyVideosGlobalFolderId& aSourceFolderId,
        CIptvMyVideosGlobalFolderId& aTargetFolderId );

    /**
     * Send request to server.
     *
     * @param aMessage       Message to send.
     * @param aFolderIdParam Message parameter, global folder Id.
     * @param aUintParam1    Message parameter, TUint32.
     * @param aUintParam2    Message parameter, TUint32.
     * @return KErrNone or one of system wide error codes.
     */
    TInt SendRequestL(
        TInt aMessage,
        CIptvMyVideosGlobalFolderId& aFolderIdParam,
        TUint32 aUintParam1,
        TUint32 aUintParam2 );

    /**
     * Send request to server.
     *
     * @param aMessage Message to send.
     * @param aVideoFullDetails Message parameter, video full details.
     * @return KErrNone or one of system wide error codes.
     */
    TInt SendRequestL(
        TInt aMessage,
        CIptvMyVideosVideoFullDetails& aVideoFullDetails );

    /**
     * Send request to server.
     *
     * @param aMessage Message to send.
     * @param aVideoFullDetails Message parameter, video full details.
     * @param aUintParam1       Message parameter, TUint32.
     * @param aUintParam2       Message parameter, TUint32.
     * @return KErrNone or one of system wide error codes.
     */
    TInt SendRequestL(
        TInt aMessage,
        CIptvMyVideosVideoFullDetails& aVideoFullDetails,
        TUint32 aUintParam1,
        TUint32 aUintParam2 );

    /**
     * Send request to server.
     *
     * @param aMessage Message to send.
     * @param aFileIdParam Message parameter, global file Id.
     * @param aUintParam   Message parameter, TUint32.
     * @return KErrNone or one of system wide error codes.
     */
    TInt SendRequestL(
        TInt aMessage,
        CIptvMyVideosGlobalFileId& aFileIdParam,
        TUint32 aUintParam );

protected: // Handling servers replies to initial requests.

    /**
     * Handles received size reply and starts data request.
     *
     * @param aMessage Message.
     * @return KErrNone or one of system wide error codes.
     */
    TInt HandleMyVideosSizeReqL( TInt aMessage );

    /**
     * Generic method for finishing request with no response data.
     *
     * @return KErrNone or one of system wide error codes.
     */
    TInt HandleMyVideosReqL();

    /**
     * Handle get folder list data request.
     *
     * @param aTotalAmount On return, total amount of folders.
     * @param aFolderList  On return, folder list.
     * @return KErrNone or one of system wide error codes.
     */
    TInt HandleMyVideosGetFolderListDataReqL(
        TUint32& aTotalAmount,
        RPointerArray<CIptvMyVideosFolder>& aFolderList );

    /**
     * Handle get parent folder data request.
     *
     * @param aParentFolder On return, parent folder Id.
     * @return KErrNone or one of system wide error codes.
     */
    TInt HandleMyVideosGetParentFolderDataReqL(
        CIptvMyVideosGlobalFolderId& aParentFolder );

    /**
     * Handle get folder name data request.
     *
     * @param On return, name.
     * @return KErrNone or one of system wide error codes.
     */
    TInt HandleMyVideosGetFolderNameDataReqL( TDes& aName );

    /**
     * Handle get video list data request.
     *
     * @param On return, total amount of videos.
     * @param On return, list of videos.
     * @return KErrNone or one of system wide error codes.
     */
    TInt HandleMyVideosGetVideoListDataReqL(
        TUint32& aTotalAmount,
        RPointerArray<CIptvMyVideosVideoBriefDetails>& aVideoList );

    /**
     * Handle get video full details data request.
     *
     * @param On return, video full details.
     * @return KErrNone or one of system wide error codes.
     */
    TInt HandleMyVideosGetVideoDetailsDataReqL(
        CIptvMyVideosVideoFullDetails& aFullDetails );

    /**
     * Handle copy folder data request.
     *
     * @param aFailed On return, count of failed items.
     * @return KErrNone or one of system wide error codes.
     */
    TInt HandleMyVideosCopyFolderDataReqL(
        TUint32& aFailed );

    /**
     * Handle move folder data request.
     *
     * @param aFailed On return, count of failed items.
     * @return KErrNone or one of system wide error codes.
     */
    TInt HandleMyVideosMoveFolderDataReqL( TUint32& aFailed );

    /**
     * Handle new database folder data request.
     *
     * @param On return, Global folder Id of new folder.
     * @return KErrNone or one of system wide error codes.
     */
    TInt HandleMyVideosNewDatabaseFolderDataReqL(
        CIptvMyVideosGlobalFolderId& aNewFolderId );

    /**
     * Handle ensure free space data request.
     *
     * @param On return, Global File Id of new file.
     * @return KErrNone or one of system wide error codes.
     */
    TInt HandleMyVideosEnsureFreeSpaceDataReqL(
        CIptvMyVideosGlobalFileId& aNewFileId );

    /**
     * Checks if a string has any of the characters that are invalid
     * as a part of file name.
     *
     * @param aString String to check.
     * @return ETrue if string inludes invalid characters.
     */
    TBool HasInvalidFileNameChars( const TDesC& aString );

private: // Data

    /**
     * iClientObserver A referense to observer of the client which is making the requests.
     */
    MIptvMyVideosClientObserver& iClientObserver;

    /**
     * iSession, the server session.
     */
    RIptvClientSession iSession;

    /**
     * iMsg Heap object to restore memory for IPC messaging.
     */
    HBufC8* iMsg;

    /**
     * iMsgPtr Passed to RIptvClientSession object to store the sent and received data.
     *          Points to iData heap object.
     */
    TPtr8 iMsgPtr;

    /**
     * Observer for content update messages from server.
     */
    CIptvMyVideosContentUpdateObserver* iContentUpdateObserver;

    /**
     * Observer for total video length responses from server.
     */
    CIptvMyVideosTotalVideoLengthObserver* iTotalVideoLengthObserver;

    /**
     * Observer for copy or move completed responses from server.
     */
    CIptvMyVideosCopyOrMoveObserver* iCopyOrMoveObserver;

    /**
     * Active request of this object.
     */
    TInt iActiveRequest;
    };

#endif // CIPTVMYVIDEOSCLIENT_H
