/*
* Copyright (c) 2006 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:    My Videos Client API header file*
*/




#ifndef MIPTVMYVIDEOS_H
#define MIPTVMYVIDEOS_H

#include <e32base.h>
#include "CIptvUtil.h"

class CIptvMyVideosVideoBriefDetails;
class CIptvMyVideosVideoFullDetails;
class CIptvMyVideosFolder;
class CIptvMyVideosGlobalFileId;
class CIptvMyVideosGlobalFolderId;
class MIptvMyVideosClientObserver;

/**
 * My Videos Client API
 *
 * @lib IptvEngineClient.dll
 */
class MIptvMyVideos
    {

public: // Enumerations

    /**
     * TVideoType
     *
     * @since R2.0
     */
    enum TVideoType
        {
        EUndefined   = 0, // Type undefined.
        EVodDownload = 1, // Video originated from VoD/VodCast download.
        ERecording   = 2, // Video originated from Mobile TV recording.
        EMtpVideo    = 3, // Deprecated.
        EBrowser     = 4  // Video originated from external source (Browser).
        };
    
public: // Video Center API
    
    /**
     * Gets folder list in the alphabetical order.
     * 
     * @param aParentFolderId Points to folder ID which sub-folders should be listed. Note that
     *        the root folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL().
     * @param aFrom The start index for listed folders.
     * @param aAmount Amount of folders to be listed. If zero, all are listed.
     * @param aTotalAmount Reference to a variable where total amount of folders is returned.
     * @param aFolderList Reference to a list of items returned.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt GetFolderListL(
        CIptvMyVideosGlobalFolderId& aParentFolderId,
        TUint32 aFrom,
        TUint32 aAmount,
        TUint32& aTotalAmount,
        RPointerArray<CIptvMyVideosFolder>& aFolderList ) = 0;

    /**
     * Gets parent forlder ID identifier. 
     * Note that at the beginning of the browse operation, the root folder ID is generated by
     * CIptvMyVideosGlobalFolderId::NewRootFolderL().
     * 
     * @param aFolderId Points to a folder ID which parent folder is requested.
     * @param aParentFolder Reference to a storage where to parent folder ID is returned.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt GetParentFolderL(
        CIptvMyVideosGlobalFolderId& aFolderId, 
        CIptvMyVideosGlobalFolderId& aParentFolder ) = 0;

    /**
     * Gets folder name.
     *
     * @param aFolderId ID of the folder which name is requested.
     * @param aName On return, name of the folder.
     * @return KErrNone or one of the system wide error codes.
     */
    virtual TInt GetFolderNameL(
        CIptvMyVideosGlobalFolderId& aFolderId, 
        TDes& aName ) = 0;

    /**
     * Delete folder asynchronously.
     * Deletion includes all sub forlders and files at the tree.
     * MIptvMyVideosClientObserver::DeleteResponse is called when operation is finished.
     * 
     * @param aFolderId Points to folder ID to be deleted.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt DeleteFolderReqL(
        CIptvMyVideosGlobalFolderId& aFolderId ) = 0;

    /**
     * Delete folder.
     * Deletion includes all sub forlders and files at the tree.
     * 
     * @param aFolderId Points to folder ID to be deleted.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt DeleteFolderL(
        CIptvMyVideosGlobalFolderId& aFolderId ) = 0;
 
    /**
     * Gets a video list in a chronological order.
     * 
     * @param aParentFolderId Points to folder ID which video files should be listed.
     *                        Note that the root folder ID is generated by
     *                        CIptvMyVideosGlobalFolderId::NewRootFolderL().
     * @param aFrom           The start index for listed folders.
     * @param aAmount         Amount of folders to be listed. If zero, all listed.
     * @param aTotalAmount    Reference to a variable where total amount of folders is returned.
     * @param aVideoList      Reference to a list of items returned.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt GetVideoListL(
        CIptvMyVideosGlobalFolderId& aParentFolderId,
        TUint32 aFrom,
        TUint32 aAmount,
        TUint32& aTotalAmount,
        RPointerArray<CIptvMyVideosVideoBriefDetails>& aVideoList ) = 0;

    /**
     * Gets full video details.
     * 
     * @param aFileId Identifier to a video file requested.
     * @param aFullDetails Reference to a list of items returned.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt GetVideoDetailsL(
        CIptvMyVideosGlobalFileId& aFileId, 
        CIptvMyVideosVideoFullDetails& aFullDetails ) = 0;

    /**
     * Gets full video details by file localpath
     * 
     * @since R2.0
     * @param aLocalPath Path to video file, used as search key.
     * @param aFullDetails Reference to a item to be returned.
     * @return Error code, KErrNone if no errors, KErrNotFound if there is no such
     *         video. On unexpected error, one of the other system wide error codes.
     */
    virtual TInt GetVideoDetailsForPathL(
        const TDesC& aLocalPath, 
        CIptvMyVideosVideoFullDetails& aFullDetails ) = 0;

    /**
     * Requests total video lengt and total file size (sum of all videos). 
     * The result is returned asynchronously trough observer callback.
     * 
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt GetTotalVideoLengthRequestL() = 0;

    /**
     * Sets video details information.
     * 
     * @param aVideoFullDetails New video details information.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt SetVideoDetailsL(
        CIptvMyVideosVideoFullDetails& aVideoFullDetails ) = 0;

    /**
     * Deletes a video asynchronously.
     * Operation includes clean up that removes files from file system,
     * deletes data related to FileId and releases FileId identifier.
     * MIptvMyVideosClientObserver::DeleteResponse is called when operation is finished.
     *
     * @param aFileId File ID identifier for the video to be deleted.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt DeleteVideoReqL(
        CIptvMyVideosGlobalFileId& aFileId ) = 0;

    /**
     * Deletes a video.
     * Operation includes clean up that removes files from file system,
     * deletes data related to FileId and releases FileId identifier.
     * 
     * @param aFileId File ID identifier for the video to be deleted.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt DeleteVideoL(
        CIptvMyVideosGlobalFileId& aFileId ) = 0;

    /**
     * Copy video file asynchronously to another My Videos Folder.
     * Target drive must be different that source drive.
     * MIptvMyVideosClientObserver::CopyOrMoveComplete is called when operation is finished.
     *
     * @param aSourceFileId Source video file ID to be copied.
     * @param aTargetFolderId My Videos target folder where to be copied. Note if target is a root,
     *        that root folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt CopyReqL(
        CIptvMyVideosGlobalFileId& aSourceFileId, 
        CIptvMyVideosGlobalFolderId& aTargetFolderId ) = 0;

    /**
     * Copy video file synchronously to another My Videos Folder.
     * Target drive must be different that source drive.
     *
     * @param aSourceFileId Source video file ID to be copied.
     * @param aTargetFolderId My Videos target folder where to be copied. Note if target is a root,
     *        that root folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt CopyL(
        CIptvMyVideosGlobalFileId& aSourceFileId, 
        CIptvMyVideosGlobalFolderId& aTargetFolderId ) = 0;

    /**
     * Copy My Videos folder tree and videos asynchronously.
     * Target drive must be different that source drive.
     * MIptvMyVideosClientObserver::CopyOrMoveComplete is called when operation is finished.
     *
     * @param aSourceFolderId Source folder ID to be copied.
     * @param aTargetFolderId My Videos target folder where to be copied. Note if target is a root, 
     *        that root folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt CopyReqL(
        CIptvMyVideosGlobalFolderId& aSourceFolderId, 
        CIptvMyVideosGlobalFolderId& aTargetFolderId ) = 0;

    /**
     * Copy My Videos folder tree and videos synchronously.
     * Target drive must be different that source drive.
     *
     * @param aSourceFolderId Source folder ID to be copied.
     * @param aTargetFolderId My Videos target folder where to be copied. Note if target is a root,
     *        that root folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @param aFailed On return, count of items that could not be copied.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt CopyL(
        CIptvMyVideosGlobalFolderId& aSourceFolderId,
        CIptvMyVideosGlobalFolderId& aTargetFolderId,
        TUint32& aFailed ) = 0;

    /**
     * Move video file to another My Videos Folder asynchronously.
     * If target is in an another physical file system drive, actual video contents is also moved.
     * MIptvMyVideosClientObserver::CopyOrMoveComplete is called when operation is finished.
     *
     * @param aSourceFileId Source video file ID to be moved.
     * @param aTargetFolderId My Videos target folder where to be moved. Note if target is a root,
     *        that root folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt MoveReqL(
        CIptvMyVideosGlobalFileId& aSourceFileId, 
        CIptvMyVideosGlobalFolderId& aTargetFolderId ) = 0;

    /**
     * Move video file to another My Videos Folder synchronously.
     * If target is in an another physical file system drive, actual video contents is also moved.
     *
     * @param aSourceFileId Source video file ID to be moved.
     * @param aTargetFolderId My Videos target folder where to be moved. Note if target is a root,
     *        that root folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt MoveL(
        CIptvMyVideosGlobalFileId& aSourceFileId, 
        CIptvMyVideosGlobalFolderId& aTargetFolderId ) = 0;

    /**
     * Move My Videos folder tree and videos asynchronously.
     * If target is in an another physical file system drive, actual video contents is also moved.
     * MIptvMyVideosClientObserver::CopyOrMoveComplete is called when operation is finished.
     *
     * @param aSourceFolderId Source folder ID to be moved.
     * @param aTargetFolderId My Videos target folder where to be moved. Note if target is a root,
     *        that root folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt MoveReqL(
        CIptvMyVideosGlobalFolderId& aSourceFolderId, 
        CIptvMyVideosGlobalFolderId& aTargetFolderId ) = 0;

    /**
     * Move My Videos folder tree and videos synchronously.
     * If target is in an another physical file system drive, actual video contents is also moved.
     *
     * @param aSourceFolderId Source folder ID to be moved.
     * @param aTargetFolderId My Videos target folder where to be moved. Note if target is a root,
     *        that root folder ID is generated by CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @param aFailed On return, count of items that could not be moved.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt MoveL(
        CIptvMyVideosGlobalFolderId& aSourceFolderId,
        CIptvMyVideosGlobalFolderId& aTargetFolderId,
        TUint32& aFailed ) = 0;

    /**
     * Cancels any ongoing move or copy operation.
     * MIptvMyVideosClientObserver::CopyOrMoveComplete is called when cancel is actually done.
     *
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt CancelCopyOrMoveL() = 0;

    /**
     * Rename My Videos folder.
     *
     * @param aFolderId Folder to be renamed.
     * @param aName New name string.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt RenameDatabaseFolderL(
        CIptvMyVideosGlobalFolderId& aFolderId, 
        TDesC& aName ) = 0;

    /**
     * Create a new My Videos folder.
     *
     * @param aParentFolderId Parent folder ID where the new folder is created. Note if the parent
     *        folder is a root, that root folder ID is generated by 
     *        CIptvMyVideosGlobalFolderId::NewRootFolderL() with a drive letter.
     * @param aName Name fot the new folder.
     * @param aNewFolderId Reference to a storage where new folder ID is returned.
     * @return Error code, KErrNone if no errors, otherwise one of the system wide error codes.
     */
    virtual TInt NewDatabaseFolderL(
        CIptvMyVideosGlobalFolderId& aParentFolderId,
        TDesC& aName,
        CIptvMyVideosGlobalFolderId& aNewFolderId ) = 0;
    
    /**
     * Adds a new video entry to the database.
     * 
     * @since R2.0
     * @param aFullDetails             Full details structure of the video to be added to database.
     *                                 Note, this parameter is both input and output parameter.
     * @param aVideoType               Type of video, see enumeration TVideoType.
     * @param aSizeEstimateInKiloBytes Estimate of the size of the new video. Method will ensure 
     *                                 that there is enough free space on disk for the video.
     * @return Error code, KErrNone if no errors, KErrNotSupported if video type to add is not supported,
     *         KErrArgument if arguments are invalid, KErrDiskFull if there is no space for new video,
     *         KErrNotReady if external resources are not available, and possibly other error in 
     *         unexpected error  cases.
     */
    virtual TInt CreateVideoL(
        CIptvMyVideosVideoFullDetails& aFullDetails,
        TVideoType aVideoType,
        TUint32 aSizeEstimateInKiloBytes ) = 0;

    /**
     * Ensures that there is enough of free space on disk for new video. This version of the
     * method cannot change the used drive, it just tries to free space on current drive.
     * 
     * @since R2.0
     * @param aFileId                   Global File ID of the file that needs the space.
     * @param aRequiredSpaceInKiloBytes Required space (size of file) in kilobytes.
     * @return Error code, KErrNone if no errors, KErrArgument if the argument is invalid,
     *         KErrDiskFull if cannot free enough space for video, and possibly other error
     *         in unexpected error cases.
     */
    virtual TInt EnsureFreeSpaceL(
        CIptvMyVideosGlobalFileId& aFileId, 
        TUint32 aRequiredSpaceInKiloBytes ) = 0;

    /**
     * Ensures that there is enough of free space on disk for new video. This version of the
     * method might transfer the video to another drive if current drive cannot provide enough
     * of free space. If the video is transferred, the original video entry is removed.
     * 
     * @since R2.0
     * @param aCurrentFileId            Global File ID of the file that needs the space.
     * @param aRequiredSpaceInKiloBytes Required space (size of file) in kilobytes.
     * @param aNewFileId                Possible new Global File ID of the file if file needs to
     *                                  be re-located on another drive. Matches original File ID
     *                                  if there is no need for re-location.
     * @return Error code, KErrNone if no errors, KErrArgument if the argument is invalid,
     *         KErrDiskFull if video needs to be re-located to another drive, KErrCancel if 
     *         cannot free enough space for video, and possibly other error in unexpected 
     *         error cases.
     */
    virtual TInt EnsureFreeSpaceL(
        CIptvMyVideosGlobalFileId& aCurrentFileId,
        TUint32 aRequiredSpaceInKiloBytes,
        CIptvMyVideosGlobalFileId& aNewFileId ) = 0;

    /**
     * Count total video length and total file size (sum of all videos).
     *
     * @param aTotalPlayTime Total play time (not used).
     * @param aTotalFileSize Total file size in kilo bytes.
     * @return Error code,
     *         KErrNone if no errors,
     *         otherwise one of the system wide error codes.
     */
    virtual TInt GetTotalVideoLengthL(
        TIptvPlayTime& aTotalPlayTime,
        TIptvFileSize& aTotalFileSize )= 0;

    /**
     * Deletes all videos which are under downloading.
     *
     * @return Error code,
     *         KErrNone if no errors,
     *         otherwise one of the system wide error codes.
     */
    virtual TInt DeleteAllDownloadsL() = 0;
    };

#endif // MIPTVMYVIDEOS_H
