/**
 * Copyright (c) 2010 Sasken Communication Technologies Ltd.
 * All rights reserved.
 * This component and the accompanying materials are made available
 * under the terms of the "Eclipse Public License v1.0" 
 * which accompanies  this distribution, and is available
 * at the URL "http://www.eclipse.org/legal/epl-v10.html"
 *
 * Initial Contributors:
 * Chandradeep Gandhi, Sasken Communication Technologies Ltd - Initial contribution
 *
 * Contributors:
 * Manasij Roy, Nalina Hariharan
 * 
 * Description:
 * The SmfEvent class represents an event
 *
 */
#ifndef SMFPOSTPROVIDER_H
#define SMFPOSTPROVIDER_H

#include <QObject>
#include <qmobilityglobal.h>
#include <qgeopositioninfo.h>
#include <QTextDocument>

#include "smfglobal.h"
#include "smfprovider.h"
#include "smfcontact.h"
#include "smfplace.h"

class SmfProvider; //base-class for service provider
class SmfContact; //class for Contact in a social network



/**
 * class for information (text, image and url) contained in post in social network
 * This could be changed to contain html data
 * 
 * For those service provider which supports only string format, @see supportedFormats()
 */
class  SmfPost : public QTextDocument
	{
	Q_OBJECT
	
public:
	SmfPost ( QObject * parent = 0 );
	SmfPost ( const QString & text, QObject * parent = 0 );
	
private:
	QString*	id;
	SmfProvider* provider;
	};

/**
 *  Presence info of the user
 */
enum SmfAppearenceInfo
	{
	ENotSupported,
	EOnline,
	EOffline,
	EBusy,
	EDoNotDisturb,
	EAppearOffline,
	EOther
	};

//Post lists
typedef QList<SmfPost> SmfPostList;

/**
 * @ingroup smf_client_group 
 * Interface to post scrap/tweet like info.
 * Note that branding information for the particular service implementation
 * is available from getProvider() API. See also:
 * SmfProvider::serviceName(), SmfProvider::serviceIcon(), SmfProvider::description()
 *
 * Interface name for SmfPostProvider is org.symbian.smf.client.contact.posts
 */
class SMFCLIENT_EXPORT SmfPostProvider : public QObject
	{
	Q_OBJECT

public:
	/**
	 * Constructs SmfPostProvider.
	 * @param baseProvider The base provider info
	 * Seeing as this is a plug-in implementation, these will realistically
	 * be generated by SMF factory of some kind
	 */
	SmfPostProvider(SmfProvider* baseProvider);
	~SmfPostProvider();


public:
	/**
	 * returns maximum no of chars (unicode) that service provider can post without truncation
	 * negative value means no limit
	 */
	qint32 maxCharsInPost() const;
	/**
	 * returns maximum no of items that can be returned in a single query to getPosts
	 * negative value means feature not supported.
	 */
	qint32 maxItems() const;

	/**
	 * returns all the formatting of posts that this service provider supports.
	 * May return 0 items to mean only QString is supported.
	 */
	QVector<QTextFormat> supportedFormats () const;
	
	/**
	 * returns whether this SP supports Appearence @see SmfAppearenceInfo
	 */
	bool supportsAppearence () const;

public slots:
	/**
	 * Gets the posts asynchronously. The signal postsAvailable()with SmfPostList is emitted
	 * once the post lists are available
	 * @param user user's contact in this SP, omit for self contact
	 * @param pageNum Page number to download, SMF_FIRST_PAGE denotes fresh query.
     * @param perPage Item per page, default is SMF_ITEMS_PER_PAGE 
	 * @see postsAvailable()
	 */
	void posts(SmfContact* user = 0,int pageNum=SMF_FIRST_PAGE,int perPage=SMF_ITEMS_PER_PAGE);
	
	/**
	 * Updates a post to own area, the success of the post can be checked with signal
	 * updatePostFinished() signal
	 * @param postData data to be posted
	 * @param location location data
	 */
	void post(SmfPost& postData,SmfPlace& location) ; 
	/**
	 * Updates the last post to own area with new data, the success of the post can be checked with signal
	 * updatePostFinished() signal
	 * @param postData edited/new data to be posted
	 * @param location location data
	 */
	void updatePost(SmfPost& postData); 
	
	/**
	 * Updates a post to a particular Smf contact. the success of the post can be checked with signal
	 * updatePostFinished() signal.
	 * @param postData data to be posted
	 * @param contact contact to which the post is to be directed
	 * @param location location data
	 */
	void postDirected(SmfPost& postData,SmfContact& contact,SmfPlace* location=0);
	
	/**
	 * Posts appearance info of the user.e.g. appear offline, busy, do-not-disturb
	 * @param appearence user appearance
	 * @see SmfPresenceInfo
	 * @return False on Failure/Not supported 
	 */
	bool postAppearence(SmfAppearenceInfo appearence); 
	/**
	 * Share /a contact's post to user's friends and followers (e.g. retweet in twitter, share on facebook)
	 * emits updatePostFinished() signal when done.
	 * @param postData data to be posted
	 * @param contact contact to which the post belonged
	 * @param bool whether user changed items within the post
	 */
	void sharePost(SmfPost& postData,SmfContact& contact,bool edited);
	
	//APIs to get/set base provider info (SmfProvider)
	
	/**
	 * Gets the base provider info
	 */
	SmfProvider* getProvider() const;
	
signals:
	
	/**
	 * Emitted when a request to  getPosts() is finished
	 * Note if number of posts is large, then it can download the list page by page
	 * In that case this signal is emitted multiple times.
	 * @param list list of posts
	 * @param error error string
	 * @param resultPage Page number info
	 */
	void postsAvailable(SmfPostList* list, QString error, SmfResultPage resultPage);
	
	/**
	 * Emitted when update post is finished.
	 * @param success the success of the update
	 */
	void postFinished(bool success);
	
	private:
	SmfProvider* m_baseProvider;
	};

SMF_SERVICE_NAME(SmfPostProvider, "org.symbian.smf.client.contact.posts\0.2")
#endif // SMFPOSTPROVIDER_H
