/*
* Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
*
*/

#include <hbaction.h>
#include <hbdialog.h>
#include <hbinstance.h>
#include <hblabel.h>
#include <hblistwidget.h>
#include <hbmenu.h>
#include <hbscrollbar>
#include <QActionGroup>
#include <QGraphicsLinearLayout>
#include <QSettings>
#include "hgwidgettestview.h"
#include "hgtestview.h"
#include "hgwidgetoptionsview.h"
#include "hgselectiondialog.h"
#include "hgwidgettestdatamodel.h"
#include "hgflipwidget.h"
#include "hgitemsizedialog.h"
#include "hgcoverflowwidget.h"
#include "trace.h"
#include <hgwidgets/hggrid.h>
#include <hgwidgets/hgmediawall.h>


static const int GRIDBUFFERSIZE(400);
static const int COVERFLOWBUFFERSIZE(130);

HgWidgetTestView::HgWidgetTestView(QGraphicsItem *parent) :
    HbView(parent),
    mWidget(0),
    mModel(0),
    mWidgetType(HgWidgetNone),
    mListWidget(0),
    mToggleOrientation(false),
    mFlipWidget(0),
    mFrontItem(0),
    mOptionsView(0),
    mItemSizeDialog(0),
    mItemPosDialog(0)
{
    mModel = new HgWidgetTestDataModel(this);
    mSelectionModel = new QItemSelectionModel(mModel, this);

    createMenu();

    mLayout = new QGraphicsLinearLayout(Qt::Vertical);
    mLayout->setContentsMargins(0,0,0,0);
    setupWidget();
    setLayout(mLayout);
}

HgWidgetTestView::~HgWidgetTestView()
{
}

void HgWidgetTestView::createMenu()
{
    FUNC_LOG;

    menu()->addAction("Options", this, SLOT(showOptions()));
    menu()->addAction("Reset Options", this, SLOT(resetOptions()));
    menu()->addAction("Toggle scrolldirection", this, SLOT(toggleScrollDirection()));
    menu()->addAction("Simulate orientation switch", this, SLOT(orientationChanged()));
    menu()->addAction("Edit item size", this, SLOT(startItemSizeChange()));
    menu()->addAction("Edit item pos", this, SLOT(startItemPosChange()));
        
    HbMenu *modelChangeSubMenu = menu()->addMenu("Change model");
    modelChangeSubMenu->addAction("Remove items", this, SLOT(openDeleteItemsDialog()));
    modelChangeSubMenu->addAction("Move items", this, SLOT(openMoveItemsDialog()));
    modelChangeSubMenu->addAction("Add items", this, SLOT(openAddItemsDialog()));
    modelChangeSubMenu->addAction("Reset model", this, SLOT(resetModel()));
}

void HgWidgetTestView::toggleScrollDirection()
{
    FUNC_LOG;

    mToggleOrientation = !mToggleOrientation;
    initWidget(mWidgetType);
}

void HgWidgetTestView::initWidget(HgTestWidgetType type)
{
    FUNC_LOG;

    if (mWidgetType != type) {
        mWidgetType = type;

        // TODO, disconnecting signals required?

        if( mWidget )
            mLayout->removeItem(mWidget);
        if( mListWidget )
            mLayout->removeItem(mListWidget);

        delete mWidget;
        mWidget = NULL;

        delete mListWidget;
        mListWidget = NULL;

        mWidget = createWidget(type);
        mLayout->addItem(mWidget);

        switch (type) {
            case HgWidgetGrid:
            	setItemVisible(Hb::AllItems, true);
                mModel->enableLowResImages(false);
                // TODO, init grid different model,
                mModel->setThumbnailSize(ThumbnailManager::ThumbnailMedium);
                break;
            case HgWidgetCoverflow:
                mModel->setThumbnailSize(ThumbnailManager::ThumbnailLarge);
                setItemVisible(Hb::AllItems, orientation() != Qt::Horizontal);
                break;
            case HgWidgetTBone:
            	setItemVisible(Hb::AllItems, true);
                mModel->setThumbnailSize(ThumbnailManager::ThumbnailMedium);
                mListWidget = new HbListWidget;
                mLayout->addItem(mListWidget);
                mListWidget->addItem( "List item 1");
                mListWidget->addItem( "List item 2");
                mListWidget->addItem( "List item 3");
                break;
            default:
                break;
        }

        HANDLE_ERROR_NULL(mWidget);
        mWidget->setModel(mModel);
        connect(mWidget, SIGNAL(activated(QModelIndex)), SLOT(openDialog(QModelIndex)));
        connect(mWidget, SIGNAL(longPressed(QModelIndex, QPointF)), SLOT(openView(QModelIndex)));
        QList<HbMainWindow *> mainWindows = hbInstance->allMainWindows();
        if (mainWindows.count() > 0) {
            HbMainWindow *primaryWindow = mainWindows[0];
            connect(primaryWindow, SIGNAL(orientationChanged(Qt::Orientation)), mWidget, SLOT(orientationChanged(Qt::Orientation)));
            connect(primaryWindow, SIGNAL(orientationChanged(Qt::Orientation)), this, SLOT(orientationChanged(Qt::Orientation)));
        }
        setupWidgetOptions();
        setupWidgetSize();

        connect(mWidget, SIGNAL(scrollingStarted()), SLOT(onScrollingStarted()));
        connect(mWidget, SIGNAL(scrollingEnded()), SLOT(onScrollingEnded()));
    }
}

void HgWidgetTestView::changeScrollBarVisibility(HgWidget::ScrollBarPolicy policy)
{
    FUNC_LOG;

    if (mWidget->scrollBarPolicy() != policy) {
        mWidget->setScrollBarPolicy(policy);
    }
}

void HgWidgetTestView::changeScrollBarInteractivity(bool value)
{
    FUNC_LOG;

    if (mWidget->scrollBar()->isInteractive() != value) {
        mWidget->scrollBar()->setInteractive(value);

        if (mWidgetType == HgWidgetCoverflow || mWidgetType == HgWidgetTBone) {
            mWidget->setIndexFeedbackPolicy(HgWidget::IndexFeedbackSingleCharacter);
        }
    }
}

void HgWidgetTestView::changeModelImageType(HgTestImageType type)
{
    FUNC_LOG;

    if (mModel->imageDataType() != type) {
        mModel->setImageDataType(type);
    }
}

void HgWidgetTestView::changeLowResImageUse(bool value)
{
    FUNC_LOG;

    if (mWidgetType == HgWidgetCoverflow || mWidgetType == HgWidgetTBone) {
        mModel->enableLowResImages(value);
        initWidget(mWidgetType);
    }
}

void HgWidgetTestView::changeTitlePosition(HgMediawall::LabelPosition position)
{
    FUNC_LOG;

    HgMediawall *mediawall = qobject_cast<HgMediawall *>(mWidget);
    if (mediawall && mediawall->titlePosition() != position) {
        mediawall->setTitlePosition(position);
    }
}

void HgWidgetTestView::changeTitleFont(const HbFontSpec &fontSpec)
{
    FUNC_LOG;

    HgMediawall *mediawall = qobject_cast<HgMediawall *>(mWidget);
    if (mediawall && mediawall->titleFontSpec() != fontSpec) {
        mediawall->setTitleFontSpec(fontSpec);
    }
}

void HgWidgetTestView::changeDescriptionPosition(HgMediawall::LabelPosition position)
{
    FUNC_LOG;

    HgMediawall *mediawall = qobject_cast<HgMediawall *>(mWidget);
    if (mediawall && mediawall->descriptionPosition() != position) {
        mediawall->setDescriptionPosition(position);
    }
}

void HgWidgetTestView::changeDescriptionFont(const HbFontSpec &fontSpec)
{
    FUNC_LOG;

    HgMediawall *mediawall = qobject_cast<HgMediawall *>(mWidget);
    if (mediawall && mediawall->descriptionFontSpec() != fontSpec) {
        mediawall->setDescriptionFontSpec(fontSpec);
    }
}

void HgWidgetTestView::changeWidgetHeight(int value)
{
    FUNC_LOG;

    int height(rect().height());
    if (value < height) {
        qreal left(0), top(0), right(0), bottom(0);
        mLayout->getContentsMargins(&left, &top, &right, &bottom);

        if (mWidgetType == HgWidgetTBone) {
            mListWidget->setPreferredSize(mListWidget->preferredSize().width(), height-value);
            mLayout->setContentsMargins(left, 0, right, 0);
        }
        else {
            int margin(0);
            margin = (height-value)/2;
            mLayout->setContentsMargins(left, margin, right, margin);
        }
    }
}

void HgWidgetTestView::changeWidgetWidth(int value)
{
    FUNC_LOG;

    int width(rect().width());
    int margin(0);
    if (value < width) {
        margin = (width-value)/2;
    }

    qreal left(0), top(0), right(0), bottom(0);
    mLayout->getContentsMargins(&left, &top, &right, &bottom);
    mLayout->setContentsMargins(margin, top, margin, bottom);
}

void HgWidgetTestView::openDialog(const QModelIndex& index)
{
    FUNC_LOG;

    QVariant image = mModel->data(index, Qt::DecorationRole);
    QVariant texts = mModel->data(index, Qt::DisplayRole);

    if (mWidgetType == HgWidgetCoverflow || mWidgetType == HgWidgetTBone)
    {
        if (image.canConvert<QPixmap>() && texts.canConvert<QStringList>())
        {
            QStringList strList = texts.toStringList();
            if (strList.count() > 1)
            {
                if (mFrontItem)
                    mFrontItem->setVisible(false);

                if (mFlipWidget)
                    delete mFlipWidget;

                QPolygonF poly;
                if (!mWidget->getItemOutline(index, poly))
                    return;

                QRectF itemRect = poly.boundingRect();
                itemRect.moveTo(itemRect.topLeft() + mWidget->geometry().topLeft());

                mFlipWidget =
                    new HgFlipWidget(
                        strList.at(0),
                        strList.at(1),
                        image.value<QPixmap>(),
                        this);

                mFlipWidget->setPos(QPointF(
                    itemRect.center().x() - itemRect.width() / 2,
                    itemRect.center().y() - itemRect.height() / 2
                ));

                mFlipWidget->resize(itemRect.width(), itemRect.height());
                mFlipWidget->show();

                QObject::connect(mFlipWidget, SIGNAL(closed()), this, SLOT(flipClosed()));
                mModel->setData(index, false, Qt::UserRole+1);
                mFlippedIndex = index;
            }
        }

        return;
    }

    HbDialog dlg;
    dlg.setTimeout(HbPopup::NoTimeout);
    dlg.setDismissPolicy(HbPopup::TapInside);
    dlg.setPrimaryAction(new HbAction("Close"));
    if (texts.canConvert<QStringList>())
    {
        QStringList strList = texts.toStringList();
        if (strList.count() > 0)
        {
            dlg.setHeadingWidget(new HbLabel(strList.at(0)));
        }
    }
    if (image.canConvert<QImage>())
    {
        HbLabel *content = new HbLabel;
        QImage realImage(image.value<QImage>());
        QPixmap pixmap = QPixmap::fromImage( realImage  );
        content->setIcon(HbIcon(pixmap));
        dlg.setContentWidget(content);
    }
    dlg.exec();
}

void HgWidgetTestView::openView(const QModelIndex& index)
{
    FUNC_LOG;

    QVariant image = mModel->data(index, Qt::DecorationRole);
    QVariant texts = mModel->data(index, Qt::DisplayRole);

    if (image.canConvert<QImage>() && texts.canConvert<QStringList>())
    {
        QStringList strList = texts.toStringList();
        if (strList.count() > 1)
        {
            HgTestView* view =
                new HgTestView(
                    strList.at(0),
                    strList.at(1),
                    QPixmap::fromImage(image.value<QImage>()));
            connect(view, SIGNAL(closeRequested()), SLOT(closeCurrentView()));
            QList<HbMainWindow *> mainWindows = hbInstance->allMainWindows();
            if (mainWindows.count() > 0)
            {
                HbMainWindow *primaryWindow = mainWindows[0];
                primaryWindow->addView(view);
                primaryWindow->setCurrentView(view);
            }
        }
    }
}

void HgWidgetTestView::closeCurrentView()
{
    FUNC_LOG;

    QList<HbMainWindow *> mainWindows = hbInstance->allMainWindows();
    if (mainWindows.count() > 0) {
        HbMainWindow *primaryWindow = mainWindows[0];
        HbView *currentView = primaryWindow->currentView();
        primaryWindow->setCurrentView(this);
        primaryWindow->removeView(currentView);
    }
}

void HgWidgetTestView::openDeleteItemsDialog()
{
    FUNC_LOG;
    HANDLE_ERROR_NULL(mWidget);

    HgWidget *widget = copyWidget();
    HANDLE_ERROR_NULL(widget);

    HgSelectionDialog *dlg =
        new HgSelectionDialog("Remove items", "Remove", widget); // Takes ownership of widget
    HANDLE_ERROR_NULL(dlg);

    mWidget->hide();
    widget->setSelectionMode(HgWidget::MultiSelection);
    bool removeItems = (dlg->exec() == dlg->primaryAction());
    QItemSelection selection = mSelectionModel->selection();
    widget->setSelectionMode(HgWidget::NoSelection); // Clears the selection
    delete dlg;

    if (removeItems) {
        mModel->remove(selection);
    }

    mWidget->show();
}

void HgWidgetTestView::openMoveItemsDialog()
{
    FUNC_LOG;
    HANDLE_ERROR_NULL(mWidget);

    HgWidget *widget = copyWidget();
    HANDLE_ERROR_NULL(widget);
    HgSelectionDialog *dlg =
        new HgSelectionDialog("Select items to move", "Move to...", widget); // Takes ownership of widget
        HANDLE_ERROR_NULL(dlg);
    HANDLE_ERROR_NULL(dlg);

    mWidget->hide();
    widget->setSelectionMode(HgWidget::ContiguousSelection);
    bool moveItems = (dlg->exec() == dlg->primaryAction());
    QItemSelection selection = mSelectionModel->selection();
    widget->setSelectionMode(HgWidget::NoSelection); // Clears the selection
    delete dlg;
    dlg = 0;
    widget = 0;

    if (moveItems) {
        widget = copyWidget();
        HANDLE_ERROR_NULL(widget);
        widget->setPreferredSize(size().width(), 320);
        dlg = new HgSelectionDialog("Select target location", "Move", widget); // Takes ownership of widget
        HANDLE_ERROR_NULL(dlg);
        widget->setSelectionMode(HgWidget::SingleSelection);
        moveItems = (dlg->exec() == dlg->primaryAction());
        QItemSelection target = mSelectionModel->selection();
        widget->setSelectionMode(HgWidget::NoSelection); // Clears the selection
        delete dlg;

        if (moveItems) {
            mModel->move(selection, target.indexes().at(0));
        }
    }
    mWidget->show();
}

void HgWidgetTestView::openAddItemsDialog()
{
    FUNC_LOG;
    HANDLE_ERROR_NULL(mWidget);

    HgWidget *widget = copyWidget();
    HANDLE_ERROR_NULL(widget);
    HgSelectionDialog *dlg =
        new HgSelectionDialog("Select position to add items to", "Add", widget); // Takes ownership of widget
    HANDLE_ERROR_NULL(dlg);

    mWidget->hide();
    widget->setSelectionMode(HgWidget::SingleSelection);
    bool addItems = (dlg->exec() == dlg->primaryAction());
    QItemSelection target = mSelectionModel->selection();
    widget->setSelectionMode(HgWidget::NoSelection); // Clears the selection
    delete dlg;

    if (addItems) {
        mModel->add(target.indexes().at(0), 5);
    }

    mWidget->show();
}

void HgWidgetTestView::showOptions()
{
    FUNC_LOG;

    QList<HbMainWindow *> mainWindows = hbInstance->allMainWindows();
    if (mainWindows.count() > 0) {
        HbMainWindow *primaryWindow = mainWindows[0];
        if (!mOptionsView) {
            HgWidgetOptionsView* view = new HgWidgetOptionsView;
            connect(view, SIGNAL(optionsClosed()), SLOT(hideOptions()));
            connect(view, SIGNAL(widgetTypeChanged(HgTestWidgetType)),
                SLOT(initWidget(HgTestWidgetType)));
            connect(view, SIGNAL(scrollBarVisibilityChanged(HgWidget::ScrollBarPolicy)),
                SLOT(changeScrollBarVisibility(HgWidget::ScrollBarPolicy)));
            connect(view, SIGNAL(scrollBarInteractivityChanged(bool)),
                SLOT(changeScrollBarInteractivity(bool)));
            connect(view, SIGNAL(imageTypeChanged(HgTestImageType)),
                SLOT(changeModelImageType(HgTestImageType)));
            connect(view, SIGNAL(lowResImageUseChanged(bool)), SLOT(changeLowResImageUse(bool)));
            connect(view, SIGNAL(widgetHeightChanged(int)), SLOT(changeWidgetHeight(int)));
            connect(view, SIGNAL(widgetWidthChanged(int)), SLOT(changeWidgetWidth(int)));
            connect(view, SIGNAL(titlePositionChanged(HgMediawall::LabelPosition)),
                SLOT(changeTitlePosition(HgMediawall::LabelPosition)));
            connect(view, SIGNAL(titleFontChanged(HbFontSpec)),
                SLOT(changeTitleFont(HbFontSpec)));
            connect(view, SIGNAL(descriptionPositionChanged(HgMediawall::LabelPosition)),
                SLOT(changeDescriptionPosition(HgMediawall::LabelPosition)));
            connect(view, SIGNAL(descriptionFontChanged(HbFontSpec)),
                SLOT(changeDescriptionFont(HbFontSpec)));
            connect(view, SIGNAL(reflectionsEnabledChanged(bool)), 
                SLOT(changeReflectionsEnabled(bool)));

            mOptionsView = view;
            primaryWindow->addView(mOptionsView);
        }
        primaryWindow->setCurrentView(mOptionsView);
    }
}

void HgWidgetTestView::hideOptions()
{
    FUNC_LOG;

    QList<HbMainWindow *> mainWindows = hbInstance->allMainWindows();
    if (mainWindows.count() > 0) {
        HbMainWindow *primaryWindow = mainWindows[0];
        primaryWindow->setCurrentView(this);
    }
}

void HgWidgetTestView::setupWidget()
{
    FUNC_LOG;

    QSettings settings(SETT_ORGANIZATION, SETT_APPLICATION);

    QVariant value = settings.value(SETT_WIDGET_TYPE);
    if (value.isValid()) {
        initWidget(static_cast<HgTestWidgetType>(value.toInt()));
    }
    else {
        initWidget(HgWidgetGrid);
    }

    value = settings.value(SETT_LOW_RES_IMAGES);
    if (value.isValid()) {
        changeLowResImageUse(value.toBool());
    }
}

void HgWidgetTestView::setupWidgetOptions()
{
    FUNC_LOG;

    QSettings settings(SETT_ORGANIZATION, SETT_APPLICATION);

    QVariant value = settings.value(SETT_SCROLLBAR_VISIBILITY);
    if (value.isValid()) {
        changeScrollBarVisibility(static_cast<HgWidget::ScrollBarPolicy>(value.toInt()));
    }

    value = settings.value(SETT_SCROLLBAR_INTERACTIVITY);
    if (value.isValid()) {
        changeScrollBarInteractivity(value.toBool());
    }

    value = settings.value(SETT_MODEL_IMAGE_TYPE);
    if (value.isValid()) {
        changeModelImageType(static_cast<HgTestImageType>(value.toInt()));
    }
	else {
        changeModelImageType(ImageTypeQImage);
	}

    value = settings.value(SETT_TITLE_POSITION);
    if (value.isValid()) {
        changeTitlePosition(static_cast<HgMediawall::LabelPosition>(value.toInt()));
    }

    value = settings.value(SETT_TITLE_FONT);
    if (value.isValid()) {
        changeTitleFont(HbFontSpec(static_cast<HbFontSpec::Role>(value.toInt())));
    }

    value = settings.value(SETT_DESCRIPTION_POSITION);
    if (value.isValid()) {
        changeDescriptionPosition(static_cast<HgMediawall::LabelPosition>(value.toInt()));
    }

    value = settings.value(SETT_DESCRIPTION_FONT);
    if (value.isValid()) {
        changeDescriptionFont(HbFontSpec(static_cast<HbFontSpec::Role>(value.toInt())));
    }

    value = settings.value(SETT_REFLECTIONS_ENABLED);
    if (value.isValid()) {
        changeReflectionsEnabled(value.toBool());
    }

}

void HgWidgetTestView::setupWidgetSize()
{
    QSettings settings(SETT_ORGANIZATION, SETT_APPLICATION);

    QVariant value = settings.value(SETT_WIDGET_HEIGHT);
    if (value.isValid()) {
        changeWidgetHeight(value.toInt());
    }

    value = settings.value(SETT_WIDGET_WIDTH);
    if (value.isValid()) {
        changeWidgetWidth(value.toInt());
    }
}

HgWidget *HgWidgetTestView::createWidget(HgTestWidgetType type) const
{
    FUNC_LOG;
    HANDLE_ERROR_NULL(mModel);
    HANDLE_ERROR_NULL(mSelectionModel);

    Qt::Orientation scrollDirection = Qt::Vertical;
    QList<HbMainWindow *> mainWindows = hbInstance->allMainWindows();
    if (mainWindows.count() > 0)
    {
        HbMainWindow *primaryWindow = mainWindows[0];
        if (primaryWindow->orientation() == Qt::Horizontal) {
            scrollDirection = Qt::Horizontal;
        }
    }

    HgWidget* widget = 0;
    HgCoverflowWidget* temp = 0;
    
    switch (type) {
        case HgWidgetGrid:
            mModel->setThumbnailSize(ThumbnailManager::ThumbnailMedium);
            mModel->setBuffer(GRIDBUFFERSIZE, GRIDBUFFERSIZE/3);
            widget = new HgGrid(scrollDirection);
            break;
        case HgWidgetCoverflow:
            mModel->setThumbnailSize(ThumbnailManager::ThumbnailLarge);
            mModel->setBuffer(COVERFLOWBUFFERSIZE, COVERFLOWBUFFERSIZE/3);
            widget = new HgCoverflowWidget();
            temp = (HgCoverflowWidget*)widget;
            temp->setTitlePosition(HgMediawall::PositionNone);
            break;
        case HgWidgetTBone:
            mModel->setThumbnailSize(ThumbnailManager::ThumbnailLarge);
            mModel->setBuffer(COVERFLOWBUFFERSIZE, COVERFLOWBUFFERSIZE/3);
            widget = new HgCoverflowWidget();
            temp = (HgCoverflowWidget*)widget;
            temp->setTitlePosition(HgMediawall::PositionNone);
            break;
        default:
            break;
    }

    HANDLE_ERROR_NULL(widget);

    widget->setModel(mModel);
    widget->setSelectionModel(mSelectionModel);
    widget->setLongPressEnabled(true);
	widget->scrollTo(widget->currentIndex());

    return widget;
}

HgWidget *HgWidgetTestView::copyWidget() const
{
    FUNC_LOG;
    HANDLE_ERROR_NULL(mWidget);

    HgWidget* widget = createWidget(mWidgetType);
    widget->setPreferredSize(mWidget->size());

    HgMediawall *original = qobject_cast<HgMediawall *>(mWidget);
    HgMediawall *copy = qobject_cast<HgMediawall *>(widget);
    if (original && copy) {
        copy->setTitlePosition(original->titlePosition());
        copy->setDescriptionPosition(original->descriptionPosition());
        copy->setTitleFontSpec(original->titleFontSpec());
        copy->setDescriptionFontSpec(original->descriptionFontSpec());
    }

    return widget;
}

void HgWidgetTestView::flipClosed()
{
    delete mFlipWidget;
    mFlipWidget = 0;
    mModel->setData(mFlippedIndex, true, Qt::UserRole+1);

    if (mFrontItem) {
        mFrontItem->setVisible(true);
    }

}

void HgWidgetTestView::onScrollingStarted()
{
    FUNC_LOG;

    // scrolling started, need to hide
    // label displaying full resolution image
    if (mFrontItem)
        mFrontItem->setVisible(false);

}

void HgWidgetTestView::onScrollingEnded()
{
    FUNC_LOG;

    if (mModel->lowResImagesEnabled()) {

        if (!mWidget)
            return;

        // get index to current item
        QModelIndex index = mWidget->currentIndex();
        if (!index.isValid())
            return;

        // get outlines of the item so we know where to render
        QPolygonF poly;
        if (!mWidget->getItemOutline(index, poly))
            return;

        // fetch highresolution image from the model
        QVariant imgVariant = mModel->data(index, Qt::UserRole+2);
        if (imgVariant.isNull())
            return;

        QRectF itemRect = poly.boundingRect();
        itemRect.moveTo(itemRect.topLeft() + mWidget->geometry().topLeft());

        // show it using HbLabel
        QPixmap pixmap = imgVariant.value<QPixmap>().scaled(itemRect.width(), itemRect.height());

        if (!mFrontItem) {
            mFrontItem = new HbLabel(this);
        }


        mFrontItem->setVisible(false);
        mFrontItem->setIcon(HbIcon(pixmap));
        mFrontItem->setAlignment(Qt::AlignHCenter|Qt::AlignVCenter);
        mFrontItem->setPos(itemRect.center() - QPointF(itemRect.width()/2, itemRect.height()/2));
        mFrontItem->resize(itemRect.width(), itemRect.height());
        mFrontItem->setVisible(true);
    }
}

void HgWidgetTestView::orientationChanged(Qt::Orientation orientation)
{
    if (orientation == Qt::Horizontal && mWidgetType == HgWidgetCoverflow ) {
        setItemVisible(Hb::AllItems, false);
    }
    else if (orientation == Qt::Horizontal && mWidgetType == HgWidgetTBone) {
        initWidget(HgWidgetCoverflow);
//        HbEffect::add(mWidget,":/effect1.fxml", "end");
//        HbEffect::start(mWidget, "end");
        setItemVisible(Hb::AllItems, false);
    }            
    else if (orientation == Qt::Vertical && mWidgetType == HgWidgetCoverflow) {
        initWidget(HgWidgetTBone);
//        HbEffect::add(mWidget,":/effect1.fxml", "end");
//        HbEffect::start(mWidget, "end");
        setItemVisible(Hb::AllItems, true);
    }
    
    HgCoverflowWidget* wall = qobject_cast<HgCoverflowWidget*>(mWidget);
    if (wall)
        wall->updateTextPositions();
}
void HgWidgetTestView::resizeEvent(QGraphicsSceneResizeEvent *event)
{
    Q_UNUSED(event);
    FUNC_LOG;

    setupWidgetSize();

/*    if (mWidgetType == HgWidgetCoverflow ||
        mWidgetType == HgWidgetTBone)
    {
        HgMediawall* wall = qobject_cast<HgMediawall*>(mWidget);
        wall->setFrontItemPosition(QPointF(mWidget->size().width()/2, 
            mWidget->size().height()/2));
    }*/
}

Qt::Orientation HgWidgetTestView::orientation() const
{
    FUNC_LOG;

    if (mainWindow()) {
        return mainWindow()->orientation();
    }

    return Qt::Horizontal;
}

void HgWidgetTestView::startItemSizeChange()
{
    if (mItemSizeDialog)
        return;
    
    if (!mWidget)
        return;

    QSizeF itemSize = mWidget->itemSize();
    QSizeF itemSpacing = mWidget->itemSpacing();
    mItemSizeDialog = new HgItemSizeDialog(itemSize, itemSpacing, this);
    
    QObject::connect(mItemSizeDialog, SIGNAL(updateItemSizeAndSpacing()), this, SLOT(updateItemSizeAndSpacing()));
    QObject::connect(mItemSizeDialog, SIGNAL(closed()), this, SLOT(itemSizeDialogClosed()));
}

void HgWidgetTestView::startItemPosChange()
{
    if (mItemPosDialog)
        return;
    
    if (!mWidget)
        return;
    
    HgMediawall* wall = qobject_cast<HgMediawall*>(mWidget);
    if (!wall)
        return;
    
    QPointF itemPos = wall->frontItemPositionDelta();
    QSizeF s(itemPos.x(), itemPos.y());
    mItemPosDialog = new HgItemSizeDialog(s, s, this);
    
    mItemPosDialog->setSliderLimits(-500, 500);
    
    QObject::connect(mItemPosDialog, SIGNAL(updateItemSizeAndSpacing()), this, SLOT(updateItemPos()));
    QObject::connect(mItemPosDialog, SIGNAL(closed()), this, SLOT(itemPosDialogClosed()));
}



void HgWidgetTestView::updateItemSizeAndSpacing()
{
    if (!mItemSizeDialog)
        return;
    
    mWidget->setItemSize(mItemSizeDialog->itemSize());
    mWidget->setItemSpacing(mItemSizeDialog->itemSpacing());
    
    HgCoverflowWidget* wall = qobject_cast<HgCoverflowWidget*>(mWidget);
    if (wall)
    {
        wall->updateTextPositions();
    }

    mWidget->update();
    
}

void HgWidgetTestView::updateItemPos()
{
    if (!mItemPosDialog)
        return;
    
    HgCoverflowWidget* wall = qobject_cast<HgCoverflowWidget*>(mWidget);
    if (!wall)
        return;

    QSizeF s = mItemPosDialog->itemSize();
    wall->setFrontItemPositionDelta(QPointF(s.width(), s.height()));
    wall->updateTextPositions();
    mWidget->update();
}

void HgWidgetTestView::itemSizeDialogClosed()
{
    // dialog deletes it self at close
    mItemSizeDialog = NULL;
}

void HgWidgetTestView::itemPosDialogClosed()
{
    mItemPosDialog = NULL;
}

void HgWidgetTestView::resetOptions()
{
    QSettings settings(SETT_ORGANIZATION, SETT_APPLICATION);
    settings.clear();
    setupWidget();
}

void HgWidgetTestView::changeReflectionsEnabled(bool enabled)
{
    HgMediawall* wall = qobject_cast<HgMediawall*>(mWidget);
    if (wall)
        wall->enableReflections(enabled);
}

void HgWidgetTestView::resetModel()
{
    mModel->reset();
}
