/*
* Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
*
*/

#include "tstestmodel.h"

#include <QPainter>
#include <QDateTime>
#include <QDebug>
#include <QTimer>
#include "tstestpropertydefs.h"

TsTestModel::TsTestModel(QObject *parent) : QObject(parent), mWasValid(false) 
{
    verifyConnection();
}

TsTestModel::~TsTestModel()
{
    qDeleteAll(mBitmaps);

}

QList<QVariantHash> TsTestModel::taskList() const
{
    return mData;
}

QList<QVariantHash> TsTestModel::taskList(int limit) const
{
    qDebug()<<"task list";
    return mData.mid(0, limit);
}
    
bool TsTestModel::openTask(const QVariant &id)
{
    for (QList<QVariantHash>::iterator i = mData.begin(); i != mData.end(); ++i) {
        if (i->value("TaskId") == id) {
            if (!i->value("TaskIsRunning").toBool()) {
                i->insert("TaskIsRunning", true);
                i->insert("TaskCanBeClosed", true);
            }
            i->insert("TaskTimestamp", QDateTime::currentDateTime());
            emit dataChanged();
            return true;
        }
    }
    return false;
}

bool TsTestModel::closeTask(const QVariant &id)
{
    
    for(int i=0; i<mData.count(); i++) {
        QVariantHash vh = mData.at(i);
        if(vh.value("TaskId") == id) {
            if (vh.value("TaskCanBeClosed").toBool()) {
                mData.removeAt(i);
                if (mBitmaps.count()>i) {
                    delete mBitmaps.at(i);
                    mBitmaps.removeAt(i);
                }
            }
            emit dataChanged();
            return true;
        }        
    }        
    return false;
}

void TsTestModel::checkValue()
{
    bool ok;
    int value = mSubscriber.value().toInt(&ok);
    if (!ok) {
        return;
    }
    if(value == 0) {
        return;
    }
    if(value == -1) {
        removeAll();
        return;
    }

    int itemnum = byte(value, 0);
    if(itemnum == 0) {
        return;
    }
    bool running = byte(value, 1);
    bool closeable = byte(value, 2);    
    addItems(itemnum, running, closeable);
    
}

void TsTestModel::verifyConnection()
{
    mSubscriber.setPath(QString("%1/%2").arg(TsTestProperty::KTsTestPath).arg(TsTestProperty::KPluginPath));
    
    if (mSubscriber.value().isValid()) {
        if(mWasValid) {
            QTimer::singleShot(20000, this, SLOT(verifyConnection()));
            return;
        }
        mWasValid = true;
        connect(&mSubscriber, SIGNAL(contentsChanged()), this, SLOT(checkValue()));
        // might have missed first value change
        checkValue();
    } else {
        mWasValid = false;
        QTimer::singleShot(3000, this, SLOT(verifyConnection()));
    }
}

void TsTestModel::removeAll() 
{
    if(mData.count()>0) {
        mData.clear();
        qDeleteAll(mBitmaps);
        mBitmaps.clear();
        emit dataChanged();
    }
}

char TsTestModel::byte(int numtoget, int byte)
{
    char ret = 0;
    if(byte>3 || byte<0) {
        return ret;
    }
    numtoget = numtoget>>8*byte;
    ret = static_cast<char>(numtoget);
    
    return ret;
    
}

QColor TsTestModel::generateColor(int item)
{
    int r = item*230%255;
    int g = item*80%255;
    int b = item*150%255;
    
    QColor c(r, g, b);
    return c;
}

QPixmap TsTestModel::generatePixmap(int item)
{
    QColor color = generateColor(item);
    QPixmap pixmap(128, 128);
    pixmap.fill(Qt::transparent);
    {
        QPainter painter(&pixmap);
        painter.setBrush(Qt::white);
        painter.setPen(QPen(color, 2));
        QFont font;
        font.setPointSize(8);            
        painter.setFont(font);
        QString note;
        note = tr("Task %1").arg(item);
        painter.drawText(10, 80, note);
    }
    return pixmap;
}

void TsTestModel::addItem(bool running, bool closeable)
{
    QVariantHash entry;
    int num = mData.count() + 1;
    QPixmap pixmap = generatePixmap(num);
    CFbsBitmap *bitmap = pixmap.toSymbianCFbsBitmap();
    mBitmaps.append(bitmap);
    entry.insert("TaskScreenshot", bitmap->Handle());
    entry.insert("TaskIsRunning", running);
    entry.insert("TaskCanBeClosed", closeable);
    
    entry.insert("TaskId", num);
    QString name;
    name = tr("Task %1").arg(num);
    entry.insert("TaskName", name);
    
    entry.insert("TaskTimestamp", QDateTime::currentDateTime());
    entry.insert("TaskUid", QString("EB002E%1").arg(num, 2, 10, QChar('0')).toUInt(0, 16));

    mData.append(entry);
}

void TsTestModel::addItems(int num, bool running, bool closeable)
{
    for(int i=0; i<num; i++) {
        addItem(running, closeable);
    }
}
