/*
 * Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
 * All rights reserved.
 * This component and the accompanying materials are made available
 * under the terms of "Eclipse Public License v1.0"
 * which accompanies this distribution, and is available
 * at the URL "http://www.eclipse.org/legal/epl-v10.html".
 *
 * Initial Contributors:
 * Nokia Corporation - initial contribution.
 *
 * Contributors:
 *
 * Description: Telephony Multimedia Service
 *
 */

#ifndef TMS_GLOBAL_VOL_EFFECT_H
#define TMS_GLOBAL_VOL_EFFECT_H

#include <tms.h>
#include <tmseffect.h>

namespace TMS {

// FORWARD DECLARATIONS
class TMSGlobalVolEffectBody;
class TMSEffectObserver;

/**
 * TMSGlobalVolEffect class
 *
 * This class provides volume level control in the global TMS scope. Volume
 * level changes done by any client are adjusted accross all calls serviced
 * by the TMS server.
 *
 * @lib tmsapi.lib
 *
 */
class TMSGlobalVolEffect : public TMSEffect
    {
public:
    /**
     * Destructor
     */
    IMPORT_C virtual ~TMSGlobalVolEffect();

    /**
     * Add client as an observer to events generated by this effect object.
     *
     * This function can be called at any time. It is recommended adding the
     * observer before calling any functions on the Effect. Otherwise, the
     * observer may miss a callback.
     *
     * A reference to the obsrvr is stored in the effect. EffectsEvent callback
     * will be called when an effect event occurs. When effect is deleted, all
     * TMSEffectObserver references are automatically released. The ownership of
     * the obsrvr is still maintained by the client. Should the client need to
     * delete the obsrvr before deleting the effect, it should remove the obsrvr
     * reference from the effect by calling RemoveObserver() function.
     *
     * @param  obsrvr
     *      Listener to be added to the effect.
     *
     * @param  user_data
     *      Any user data passed to the function.
     *
     * @return
     *      TMS_RESULT_SUCCESS if the obsrvr is added to the list successfully.
     *      TMS_RESULT_ALREADY_EXIST if the obsrvr is already in the list.
     *
     */
    IMPORT_C gint AddObserver(TMSEffectObserver& obsrvr, gpointer user_data);

    /**
     * Remove client observer from the Effect.
     *
     * This function can be called at any time.
     *
     * @param  obsrvr
     *      Listener to be removed from the effect.
     *
     * @return
     *      TMS_RESULT_SUCCESS if the obsrvr is successfully removed.
     *      TMS_RESULT_DOES_NOT_EXIST if trying to remove the obsrvr that has
     *      not been added to the list.
     *
     */
    IMPORT_C gint RemoveObserver(TMSEffectObserver& obsrvr);

    /**
     * Set global audio volume level. The volume level should be between 0
     * and the max value returned by GetMaxLevel().
     *
     * @param  level
     *      the volume level to be set.
     * @return
     *      TMS_RESULT_SUCCESS if the operation was successful otherwise
     *      system error.
     */
    IMPORT_C gint SetLevel(const guint level);

    /**
     * Return current global audio volume level.
     *
     * @param  level
     *      Volume level to be set.
     *
     * @return
     *      TMS_RESULT_SUCCESS if the operation was successful otherwise
     *      system error.
     *
     */
    IMPORT_C gint GetLevel(guint& level);

    /**
     * Return max volume level supported by the device.
     *
     * @param  level
     *      Max volume level.
     *
     * @return
     *      TMS_RESULT_SUCCESS if the operation was successful otherwise
     *      system error.
     *
     */
    IMPORT_C gint GetMaxLevel(guint& level);

    /**
     * Return effect type.
     *
     * @param  effecttype
     *      Type of the effect object (TMS_EFFECT_GLOBAL_VOL).
     *
     * @return
     *      TMS_RESULT_SUCCESS if the operation was successful otherwise
     *      system error.
     *
     */
    IMPORT_C virtual gint GetType(TMSEffectType& effecttype);

protected:
    /**
     * Constructor
     */
    IMPORT_C TMSGlobalVolEffect();

protected:
    TMSGlobalVolEffectBody *iBody;
    };

} //namespace TMS

#endif //TMS_GLOBAL_VOL_EFFECT_H
