/*
* Copyright (c) 2005-2007 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:  Declares main application class.
*
*/


#ifndef C_COLLECTION_H
#define C_COLLECTION_H

#include <e32std.h>

class CField;


/**
 *  Collection of HID report descriptor
 *  A CCollection represents an individual collection within a HID
 *  report descriptor. Each collection may have a number of associated
 *  report fields (CField objects) and a number of child collection
 *  objects. The collections within a HID report descriptor form a tree
 *  structure, with a CReportRoot collection at the root.  The tree is
 *  generated by CParser.
 *
 *
 *  @lib generichid.lib
 *  @since S60 v5.0
 */
class CCollection : public CBase
    {

public:

   /**
    *  An enumerations giving the possible types of collection, as
    *  given in "USB Device Class Definition for Human Interface
    *  Devices (HID)", Firmware Specification, Version 1.11, USB
    *  Implementers' Forum, June 2001.
    *
    *  Note that a TUint32 is used for CCollection::iType, as it is
    *  possible to have a vendor defined collection type that isn't
    *  included in this list.
    */
    enum TType
        {
        EPhysical      = 0x00, //!< Physical (group of axes)
        EApplication   = 0x01, //!< Application (mouse,keyboard)
        ELogical       = 0x02, //!< Logical (interrelated data)
        EReport        = 0x03, //!< Report
        ENamedArray    = 0x04, //!< NamedArray
        EUsageSwitch   = 0x05, //!< UsageSwitch
        EUsageModifier = 0x06  //!< UsageModifier
        };

    static CCollection* NewL();
    static CCollection* NewLC();

    /**
    * Destructor.
    */
    virtual ~CCollection();

    /**
     * Get collection type
     *
     * @since S60 v5.0
     * @return The type for this collection item
     */
    IMPORT_C TUint32 Type() const;

    /**
     * Get usagepage
     *
     * @since S60 v5.0
     * @return The usage page for this collection
     */
    IMPORT_C TInt UsagePage() const;

    /**
     * Get usagepage
     *
     * @since S60 v5.0
     * @return The usage page for this collection
     */
    IMPORT_C TInt Usage() const;

    /**
     * Get number of collections
     *
     * @since S60 v5.0
     * @return The number of collections that have been created so far
     */
    IMPORT_C TInt CollectionCount() const;

    /**
     * Gets numbers of field stored in collection
     *
     * @since S60 v5.0
     * @return The number of fields stored in for the current collection
     */
    IMPORT_C TInt FieldCount() const;

    /**
     * Returns a pointer to a specific collection from the list of collections
     *
     * @since S60 v5.0
     * @param aIndex The index of the required collection
     * @return The number of fields stored in for the current collection.
     *         NULL if there are no collection object at the secified index
     */
    IMPORT_C const CCollection* CollectionByIndex(TInt aIndex) const;

    /**
     * Returns a pointer to the field object at the given index within the field
     * list
     *
     * @since S60 v5.0
     * @param aIndex The offset within the current field list
     * @return A pointer to the specified field object
     *         NULL if there is no field object at the specified index
     */
    IMPORT_C const CField* FieldByIndex(TInt aIndex) const;

    /**
     * Check if collection type is physical
     *
     * @since S60 v5.0
     * @return true if physical
     */
    IMPORT_C TBool IsPhysical() const;

    /**
     * Check if collection type is application
     *
     * @since S60 v5.0
     * @return true if application
     */
    IMPORT_C TBool IsApplication() const;

    /**
     * Check if collection type is logical
     *
     * @since S60 v5.0
     * @return true if logical
     */
    IMPORT_C TBool IsLogical() const;

    /**
     * Check if collection type is report
     *
     * @since S60 v5.0
     * @return true if report
     */
    IMPORT_C TBool IsReport() const;

    /**
     * Check if collection type is name array
     *
     * @since S60 v5.0
     * @return true if named array
     */
    IMPORT_C TBool IsNamedArray() const;

    /**
     * Check if collection type is usage switch
     *
     * @since S60 v5.0
     * @return true if is usage switch
     */
    IMPORT_C TBool IsUsageSwitch() const;

    /**
     * Check if collection type is usage modifier
     *
     * @since S60 v5.0
     * @return true if usage modifier
     */
    IMPORT_C TBool IsUsageModifier() const;

public:

    /**
     * Set Collection type
     *
     * @since S60 v5.0
     * @return None
     */
    void SetType(TUint32 aType);

    /**
     * Set usage page
     *
     * @since S60 v5.0
     * @return None
     */
    void SetUsagePage(TInt aUsagePage);

    /**
     * Set usage
     *
     * @since S60 v5.0
     * @return None
     */
    void SetUsage(TInt aUsage);

    /**
     * Called when a new collection object has been encountered in
     * the report descriptor.  This creates a new CCollection object
     * and adds it to the current list
     *
     * @since S60 v5.0
     * @return None
     */
    CCollection* AddCollectionL();

    /**
     * Called when a new field object has been encountered in the
     * report descriptor.  This creates a new field object and adds it
     * the the current list
     *
     * @since S60 v5.0
     * @return A pointer to the new field object
     */
    CField* AddFieldL();

protected:

    CCollection();
    void ConstructL();

private:

    /**
     * Collection type
     */
    TUint32 iType;

    /**
     * Usage page
     */
    TInt iUsagePage;

    /**
     * Usage
     */
    TInt iUsage;

    /**
     * List of collections for the report descriptor
     */
    RPointerArray<CCollection> iCollections;

    /**
     * List of fields in the current collection
     */
    RPointerArray<CField> iFields;

    };

#endif //  C_COLLECTION_H

