/*
* Copyright (c) 2007 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:  This is the header for the Diagnostics Logs Event Eraser.
*
*/


#ifndef DIAGLOGEVENTERASER_H
#define DIAGLOGEVENTERASER_H

// Include Files
#include <e32base.h>                    // CActive
#include <f32file.h>                    // RFs

// Forward Declarations
class MDiagLogEventEraserObserver;
class CActiveSchedulerWait;
class CLogClient;
class CLogViewEvent;
class CLogFilter;

/**
* Diagnostics Plugin Log Event Eraser
*
* This class is used to remove log events generated by diagnostics plugins.
* Events could be anything that is logged by the system log, such as call or
* data use records.
*
* @lib euser.lib
* @lib logcli.lib 
* @lib efsrv.lib
* @lib flogger.lib
*/
NONSHARABLE_CLASS( CDiagLogEventEraser ) : public CActive
    {

public: // New functions

    /**
    * Symbian OS two-phase constructor.
    *
    * @param aObserver A reference to the observer for this event eraser.
    * @param aFilter The filter to be used for log erasure.  Clients may pass
    *                NULL for no filtering.  Ownership is transferred.
    * @return An pointer to the new instance of CDiagLogEventEraser.
    */
    IMPORT_C static CDiagLogEventEraser* NewL(
        MDiagLogEventEraserObserver& aObserver,
        CLogFilter* aFilter = NULL );

    /**
    * Destructor.
    */
    IMPORT_C virtual ~CDiagLogEventEraser();

    /**
    * This gets the filter used for the log erasure.  Clients which want to
    * filter the logs to be erased must call this function and set the
    * appropriate filter before calling StartAsyncEraseLogsL or EraseLogsL.
    *
    * @return The filter to apply to the logs to check for erasure.
    */
    IMPORT_C CLogFilter& Filter();

    /**
    * This gets the log client used for the log erasure.
    *
    * @return The log client used for log erasure.
    */
    IMPORT_C CLogClient& LogClient();

    /**
    * This starts an asynchronous log erase operation.  The completion of this
    * request is notified by MDiagLogEventEraserObserver::CompleteEventEraseL.
    */
    IMPORT_C void StartAsyncEraseLogsL();

    /**
    * This performs a synchronous log erase operation.  The function will
    * leave if there is any error in log erasure.
    */
    IMPORT_C void EraseLogsL();


private: // New functions

    /**
    * C++ default constructor.
    *
    * @param aObserver A reference to the observer for this event eraser.
    * @param aFilter The filter to be used for log erasure.  If NULL is
    *                passed, a new filter will be created during the second
    *                phase of construction.  Ownership is transferred.
    */
    CDiagLogEventEraser( MDiagLogEventEraserObserver& aObserver,
                         CLogFilter* aFilter );
    
    /**
    * Symbian OS default constructor.
    */
    void ConstructL();

    /**
    * Handles the completion of log erasing and performs different operations
    * depending on whether the erasing was asynchronous or not.
    */
    void HandleCompletionL( TInt aError );


private: // From base class CActive

    /**
    * Handle active object completion event.
    */
    virtual void RunL();

    /**
    * Handle active object cancel event.
    */
    virtual void DoCancel();


private: // Data

    /** The state of the log erasing operation. */
    enum TDiagLogEventEraserState
        {
        EStateInitial = 0,
        EStateApplyingFilter,
        EStateReadingEntries,
        EStateDeletingEntry,
        EStateComplete
        };

    /** The observer for this instance of the log event eraser. */
    MDiagLogEventEraserObserver& iObserver;

    /** The active scheduler, used for synchronous log erasure.  Own. */
    CActiveSchedulerWait* iWaitScheduler;

    /** The Log client.  Own. */
    CLogClient* iLogClient;

    /** The Log view.  Own. */
    CLogViewEvent* iLogView;

    /** The Log filter.  Own. */
    CLogFilter* iLogFilter;

    /** A file server session used to delete the logs. */
    RFs iFsSession;

    /** The internal state of the log event eraser. */
    TDiagLogEventEraserState iState;

    /** Indicates if the log event eraser is running a synchronous delete. */
    TBool iIsDeletingSynchronous;

    /** The return code for syncronous operation. */
    TInt iError;
    };

#endif // DIAGLOGEVENTERASER_H

// End of File
