/*
* Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description: 
*
*/

#ifndef CPIX_CPIXERROR_H
#define CPIX_CPIXERROR_H

#include <wchar.h>

#ifdef __cplusplus
extern "C" {
#endif



    /************************************************************************
     * Class cpix_Error and related types
     */
    enum cpix_ErrorType_
        {
            // Unknown exception
            ET_UNKNOWN_EXC         = 0x0000,

            // Standard C++ exceptions
            ET_STD_CPP_EXC         = 0x0100,
            ET_STD_BAD_ALLOC_EXC   = 0x0101,
            // TODO perhaps more std C++ exceptions are of interest?

            // 'Exceptions' that are generated by CPix API impl (CpixExc)
            ET_CPIX_EXC            = 0x0200,

            // LuceneError exception
            ET_CLUCENE_EXC         = 0x0400,

            // Exception(s) from Cpt (cpixtools) component
            ET_CPT_EXC             = 0x0800,
            ET_CPTSYNTAX_EXC       = 0x0801,

            // OS exceptions, if any
            ET_OS_EXC              = 0x1000
        };
    typedef enum cpix_ErrorType_ cpix_ErrorType;

    struct cpix_Error_
    {
        // the type of the error
        cpix_ErrorType   type_;

        // Detailed error message, may be NULL.
        const wchar_t *  msg_;

        // Implementation detail (pointer to derived C++ class instance)
        void          *  impl_;
    };
    typedef cpix_Error_ cpix_Error;

    /**
     * Generates a human readable error report based on the
     * error. Never fails (the report message may be truncated).
     *
     * @param thisError the error object for which to generate error.
     *        Unorthodoxly, it may be NULL, in which case empty string
     *        is generated.
     *
     * @param target the target buffer to render the report to. Must
     *        not be NULL.
     * 
     * @param targetLength the max amount of character to put in
     *        buffer, including the terminating zero. It must be
     *        greater than or equal to 1, obviously.
     */
    void cpix_Error_report(cpix_Error    * thisError,
                           wchar_t       * target,
                           size_t          targetLength);
    
    /**
     * Destroys this error instance.
     * @return always returns NULL
     */
    cpix_Error * cpix_Error_destroy(cpix_Error * thisError);

    /**
     * Tests if the last operation on cpixObj has failed.
     */
#define cpix_Failed(cpixObj)    ((cpixObj)->err_ != NULL)

    /**
     * Tests if the last operation on cpixObj has succeeded.
     */
#define cpix_Succeeded(cpixObj) ((cpixObj)->err_ == NULL)

    /**
     * Clears the error state of cpixObj.
     */
#define cpix_ClearError(cpixObj)  ((cpixObj)->err_ = cpix_Error_destroy((cpixObj)->err_))



    /**
     * Any "static" calls will give the result back via this
     * structure.
     */
    struct cpix_Result_
    {
        // Last error, if any, that resulted from executing a "static"
        // operation.
        // Use macros cpix_Failed, cpix_Succeeded and cpix_ClearError.
        // Do not attempt releasing it.
        cpix_Error * err_;
    };
    typedef cpix_Result_ cpix_Result;



#ifdef __cplusplus
} /* extern "C" */
#endif /* __cplusplus */



#endif /* CPIX_CPIXERROR_H */
