/*
* ============================================================================
*  Name        : hbindicatortestplugin.cpp
*  Part of     : hb / hbcore
*  Description : indicator test plugin implementation
*  Version     : %version: %
*
*  Copyright (c) 2009 Nokia.  All rights reserved.
*  This material, including documentation and any related computer
*  programs, is protected by copyright controlled by Nokia.  All
*  rights are reserved.  Copying, including reproducing, storing,
*  adapting or translating, any or all of this material requires the
*  prior written consent of Nokia.  This material also contains
*  confidential information which may not be disclosed to others
*  without the prior written consent of Nokia.
* ============================================================================
*/

#include <QtPlugin>
#include <QVariant>
#include <HbInstance>

#include "hbindicatortestplugin.h"

Q_EXPORT_PLUGIN(HbIndicatorTestPlugin)

// Constructor
HbIndicatorTestPlugin::HbIndicatorTestPlugin() : mError(0)
{
    for (int i = 0; i < IndicatorCount; ++i) {
        mIndicatorTypes.append(indicatorName(i));
    }
}

// Destructor
HbIndicatorTestPlugin::~HbIndicatorTestPlugin()
{
}

// Return notification types this plugin implements
QStringList HbIndicatorTestPlugin::indicatorTypes() const
{
    return mIndicatorTypes;
}

// Check if client is allowed to use notification widget
bool HbIndicatorTestPlugin::accessAllowed(const QString &indicatorType,
    const QVariantMap &securityInfo) const
{
    Q_UNUSED(indicatorType)
    Q_UNUSED(securityInfo)

    // This plugin doesn't perform operations that may compromise security.
    // All clients are allowed to use.
    return true;
}

HbIndicatorInterface* HbIndicatorTestPlugin::createIndicator(
        const QString &indicatorType)
{
    HbIndicatorInterface *indicator = 0;
    int index(typeIndex(indicatorType));
    if (index >= 0) {
        indicator = new HbTestIndicator(
                indicatorType, index, IndicatorInfos[index].interaction);
    }
    return indicator;
}

int HbIndicatorTestPlugin::error() const
{
    return mError;
}

int HbIndicatorTestPlugin::typeIndex(const QString &indicatorType) const
{
    for (int i = 0; i < mIndicatorTypes.count(); ++i) {
        if (mIndicatorTypes.at(i) == indicatorType) {
            return i;
        }
    }
    return -1;
}

HbTestIndicator::HbTestIndicator(const QString &indicatorType,
                                 int typeIndex,
                                 Interaction interaction) :
    HbIndicatorInterface(indicatorType, IndicatorInfos[typeIndex].category,
    (interaction == InteractionNone) ? NoInteraction : InteractionActivated),
    mPrimaryText(IndicatorInfos[typeIndex].primaryText),
    mSecondaryText(IndicatorInfos[typeIndex].secondaryText),
    mIcon(IndicatorInfos[typeIndex].icon),
    mIconMono(IndicatorInfos[typeIndex].iconMono),
    mTypeIndex(typeIndex), mInteraction(interaction)
{
}

HbTestIndicator::~HbTestIndicator()
{
}

bool HbTestIndicator::handleInteraction(InteractionType type)
{
    if (type == InteractionActivated) {
        switch(mInteraction) {
        case ChangeContent:
            mPrimaryText = "Clicked";
            mSecondaryText = "content changed!";
            mIcon = "qtg_mono_ok.svg";
            break;
        case Deactivate:
            emit deactivate();
            break;
        case ChangeOrientation:
            if (hbInstance->orientation() == Qt::Horizontal) {
                hbInstance->setOrientation(Qt::Vertical);
            } else {
                hbInstance->setOrientation(Qt::Horizontal);
            }
            break;
        default:
            return false;
        }
        emit dataChanged();
    }
    return false;
}

QVariant HbTestIndicator::indicatorData(int role) const
{
    if (role == PrimaryTextRole) {
        if (mParameter.isValid()) {
            return QString("data:").append(mParameter.toString())
                                   .append(" ")
                                   .append(mPrimaryText);
        } else {
            return mPrimaryText;
        }
    } else if (role == SecondaryTextRole && mTypeIndex != 7) {
        if (mParameter.isValid()) {
            return QString("data:").append(mParameter.toString())
                                   .append(" ")
                                   .append(mSecondaryText);
        } else {
            return mSecondaryText;
        }
    } else if (role == IconNameRole) {
        if (mParameter.isValid()) {
            return "qtg_mono_ok.svg";
        } else {
            return mIcon;
        }
    } else if (role == DecorationNameRole) {
        if (mParameter.isValid()) {
            return "qtg_mono_ok.svg";
        } else {
            return mIcon;
        }
    } else if (role == MonoDecorationNameRole) {
        if (mParameter.isValid()) {
            return "qtg_mono_ok.svg";
        } else {
            return mIconMono;
        }
    }
    return QVariant();
}

bool HbTestIndicator::handleClientRequest(RequestType type, const QVariant &parameter)
{
    bool handled(false);
    switch (type) {
    case RequestActivate:
        if (mParameter != parameter) {
            mParameter = parameter;
            emit dataChanged();
        }
        handled =  true;
        break;
    default:
        mParameter.clear();
    }

    return handled;
}

