/*
* Copyright (c) 2010 Sasken Communication Technologies Ltd.
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the "{License}"
* which accompanies  this distribution, and is available
* at the URL "{LicenseUrl}".
*
* Initial Contributors:
* Chandradeep Gandhi, Sasken Communication Technologies Ltd - Initial contribution
*
* Contributors:
* 
* Description:
* Interface spefication for music related services
*
*/

#ifndef SMFMUSIC_H
#define SMMUSIC_H

#include <QObject>

#include <qmobilityglobal.h>
#include <qgeopositioninfo.h>

#include "smfprovider.h"
#include "smfcontact.h"


class SmfProvider; //basic Smf service Provider info
class SmfContact; // Smf contact
class SmfMusicRating;//rating value from 0..31 - services would map accordingly
class SmfMusicProfile; //user profile containing music usage and interest info, extends SmfContact
class SmfTrackInfo; //id, title, album, artist, genre, tag, director,release year, rating, comment info
class SmfMusicFingerPrint; //generation is not in scope of smf
class SmfVenue;
class SmfMusicModel;
class SmfEvents;
class SmfPlaylist;
class SmfLyricsService;
class SmfLyrics;
class SmfSubtitle;
class SmfSubtitleSearchFilter;

typedef QList<SmfMusicProfile> SmfMusicProfileList;
typedef QList<SmfTrackInfo> SmfTrackInfoList;
typedef QList<SmfEvents> SmfEventsList;
typedef QList<SmfProvider> SmfProviderList;
typedef QList<SmfPlaylist> SmfPlaylistList;
typedef QList<SmfVenue> SmfVenueList;
typedef QList<SmfLyrics> SmfLyricsList;
typedef QList<SmfSubtitle> SmfSubtitleList;
/**
  * Basic music service ("org.symbian.smf.client.music.service")
  */
class SMFCLIENT_EXPORT SmfMusicService : public QObject
{
  Q_OBJECT

public:
  /**
   * Constructs SmfMusicService.
   * @param baseProvider The base provider info
   * Seeing as this is a plug-in implementation, these will realistically
   *  be generated by SMF factory of some kind
   */
   SmfMusicService(SmfProvider* baseProvider = 0);
  ~SmfMusicService();

public:
  
  /**
   * Gets self profile information asynchronously.
   * userInfoAvailable() signal is emitted with SmfMusicProfile when the info is arrived
   */
  virtual void userinfo() = 0; 
  
  /**
   * Asynchronously searches information about other service users for a particular venue
   * serachInfoAvailable() signal is emitted with SmfMusicProfileList when the info is arrived
   */
  virtual void searchUser(SmfVenue venue) = 0; 
  
  /**
   * Returns the model
   */
  virtual SmfMusicModel model() = 0; // maybe we can make a QItemModel-derived model?

  //APIs to get/set base provider info (SmfProvider)
  
  /**
   * Gets the base provider info
   */
  virtual SmfProvider* getProvider() = 0;
  
  /**
   * Sets the base provider info
   */
  virtual void setProvider(SmfProvider* provider) = 0;

signals:
	/**
	 * Notification on arrival of the self profile as result of userinfo().
	 * @param profile The self profile
	 */
	void userInfoAvailable(SmfMusicProfile* profile, QString error);
private:
  SmfProvider* m_baseProvider;
};
SMF_GETSERVICES(SmfMusicService, "org.symbian.smf.client.music.service\0.2")


/**
* Provides service ("org.symbian.smf.client.music.search") for music search
*/
class SMFCLIENT_EXPORT SmfMusicSearch : public QObject
{
  Q_OBJECT

public:
  /**
   * Constructs SmfMusicSearch with base provider as arguement.
   * Seeing as this is a plug-in implementation, these will realistically
   * be generated by SMF factory of some kind
   */

  SmfMusicSearch(SmfProvider* baseProvider = 0);
  ~SmfMusicSearch();

public:
  // Get the track listing - might be made asynchrnous later
  
  /**
   * Searches for music recommendations similar to a particulartrack asynchronously.
   * The signal trackSearchAvailable() is emitted with SmfTrackInfoList
   * once its arrived.
   * @param track The track for which similar recommendations need to be fetched.
   */
  virtual void recommendations(SmfTrackInfo track) = 0; // basic list of track objects
  
  /**
   * Searches for tracks similar to a given track asynchronously.
   * The signal trackSearchAvailable() is emitted with SmfTrackInfoList
   * once its arrived.
   * @param track The search criteria, similar tracks are searched
   */
  virtual void tracks(SmfTrackInfo track) = 0; // basic list of track objects
  
  /**
   * Searches for a track having similar finger print asynchronously.
   * The signal trackSearchAvailable() is emitted with SmfTrackInfoList
   * once its arrived.
   * @param signature The search criteria,signature to be searched for
   */
  virtual void trackInfo(SmfMusicFingerPrint signature) = 0; // search by fingerprint object
  
  /**
   * Search information about where to buy this song from asynchronously.
   * The signal storeSearchAvailable() is emitted with SmfProviderList once its arrived.
   * @param track The search criteria for stores
   */
  virtual void stores(SmfTrackInfo track) = 0; 
  
  //APIs to get/set base provider info (SmfProvider)
  
  /**
   * Gets the base provider info
   */
  virtual SmfProvider* getProvider() = 0;
  
  /**
   * Sets the base provider info
   */
  virtual void setProvider(SmfProvider* provider) = 0;

public slots:

	/**
	 * Posts currently playing track.
	 * Success can be checked by checking the signal postFinished()
	 * @param track Track to post
	 */
  virtual void postCurrentPlaying(SmfTrackInfo track) = 0;
  //virtual int postRating(SmfTrackInfo track, SmfMusicRating rate) = 0;
  //virtual int postComments(SmfTrackInfo track, SmfComment comment) = 0;

signals:
	/**
	 * Emitted when the search result for a track is available.
	 * Note if number of tacks in the search is large, then it can download the list page by page.
	 * In that case this signal is emitted multiple times.
	 * @param result The search result
	 */
	void trackSearchAvailable(SmfTrackInfoList* result, QString error, int pageNumber=0);
	
	/**
	 *  Emitted when the search result for a store is available.
	 *  Note if number of tacks in the search is large, then it can download the list page by page.
	 *  In that case this signal is emitted multiple times.
	 *  @param result The search result
	 */
	void storeSearchAvailable(SmfProviderList* result, QString error, int pageNumber=0);
private:
  SmfProvider* m_baseProvider;
};
SMF_GETSERVICES(SmfMusicSearch, "org.symbian.smf.client.music.search\0.2")


/**
 * Remote playlist
 */
class SMFCLIENT_EXPORT SmfPlaylist : public QObject
	{
	Q_OBJECT
	
public:
	SmfPlaylist();
	~SmfPlaylist();
	
	/**
	 * Gets tracks in the playlist
	 */
	SmfTrackInfoList* getTrackList();
	
	/**
	 * Gets playlist title
	 */
	QString getPlayListTitle();
	
	/**
	 * Gets the creation date
	 */
	QDateTime getCreationDate();
	
	/**
	 * Sets tracks in the playlist
	 */
	void setTrackList(SmfTrackInfoList* trackList);
	
	/**
	 * Sets playlist title
	 */
	void setPlayListTitle(QString title);
	
	/**
	 * Sets creation date
	 */
	void setCreationDate(QDateTime time);
	
private:
	SmfTrackInfoList* m_trackList;
	QString m_title;
	QDateTime m_creationDate;
	};


/**
 * provides service ("org.symbian.smf.client.music.playlist")
 *
 * Interface to a remote playlist service. This class
 * provides some basic functionality to allow applications
 * to interact with playlists in some music related service provider (e.g. last.fm).
 *
 * Note that branding information for the particular service implementation
 * is available from getProvider() API. See also:
 * SmfProvider::serviceName(), SmfProvider::serviceIcon()
 *
 * All of the functionality described here should be implemented by a service
 * specific plug-in object.
 *
 */
class SMFCLIENT_EXPORT SmfPlaylistService : public QObject
{
  Q_OBJECT

public:
  /**
   * Constructs SmfPlaylistService with base provider info
   * Seeing as this is a plug-in implementation, these will realistically
   * be generated by SMF factory of some kind
   */
  SmfPlaylistService(SmfProvider* baseProvider = 0);
  ~SmfPlaylistService();

public:
  
  /**
   * Gets the list playlists for the logged-in user asynchronously.
   * The signal playlistsListAvailable() signal is emitted with 
   * SmfPlaylistList once its arrived 
   */
  virtual void playlists() = 0; // basic list of playlist objects 
  
  /**
   * Gets the list playlists for the given user asynchronously.
   * The signal playlistsListAvailable() signal is emitted with 
   * SmfPlaylistList once its arrived.
   * @param user User for which to get the playlists
   */
  virtual void playlistsOf(SmfMusicProfile* user) = 0; 

  //APIs to get/set base provider info (SmfProvider)
  
  /**
   * Gets the base provider info
   */
  virtual SmfProvider* getProvider() = 0;
  
  /**
   * Sets the base provider info
   */
  virtual void setProvider(SmfProvider* provider) = 0;

public slots:

	/**
	 * Upload currently playing track to a playlist. Signal
	 * playlistUpdated() can be checked for success value
	 * @param plst The playlist to be added in
	 * @param tracks The list of tracks to uploaded
	 */
  virtual int addToPlaylist(SmfPlaylist plst, SmfTrackInfoList* tracks) = 0;
  
	/**
	 * Upload currently playing playlist . Signal
	 * playlistUpdated() can be checked for success value
	 * @param plst The playlist to be uploaded
	 */
  virtual int postCurrentPlayingPlaylist(SmfPlaylist plst) = 0;


signals:
	/**
	 * Notification of availability of list of playlists requested.
	 * Note if number of list is large, then it can download the list page by page.
	 * In that case this signal is emitted multiple times.
	 */
	void playlistsListAvailable(SmfPlaylistList*, QString error, int pageNumber=0);
	/**
	 * Signals remote updation of playlist with success value
	 */
    int playlistUpdated(bool success) ;
private:
  SmfProvider* m_baseProvider;
};
SMF_GETSERVICES(SmfPlaylistService, "org.symbian.smf.client.music.playlist\0.2")


/**
* provides service ("org.symbian.smf.client.music.events")
*/
class SMFCLIENT_EXPORT SmfMusicEvents : public QObject
{
  Q_OBJECT

public:
  /**
   * Constructs SmfMusicEvents with base provider info
   * Seeing as this is a plug-in implementation, these will realistically
   *  be generated by SMF factory of some kind
   */
  SmfMusicEvents(SmfProvider* baseProvider = 0);
  ~SmfMusicEvents();

public:
  
  /**
   * Gets list of events in a particular location asynchronously.
   * eventsAvailable() signal is emitted with SmfEventsList once its arrived
   */
  virtual void events(QtMobility::QContactGeolocation location) = 0;
  
  /**
   * Gets list of venues of a particular location asynchronously.
   * venuesAvailable() signal is emitted with SmfVenueList once its arrived.
   */
  virtual void venues(QtMobility::QContactGeolocation location) = 0; // basic list of venue objects
  
  /**
   * Gets list of events in a particular venue asynchronously.
   * eventsAvailable() signal is emitted with SmfEventsList once its arrived
   */
  virtual void events(SmfVenue venue) = 0; // basic list of events objects
  

  //APIs to get/set base provider info (SmfProvider)
  
  /**
   * Gets the base provider info
   */
  virtual SmfProvider* getProvider() = 0;
  
  /**
   * Sets the base provider info
   */
  virtual void setProvider(SmfProvider* provider) = 0;

public slots:

	/**
	 * Updates events. Might not be supported by all service provider.
	 * eventsUpdated() signal can be checked for success value.
	 * @param SmfEventsList List of events to be posted
	 */
	 virtual void postEvents(SmfEventsList events); 

signals:
	
	/**
	 * Notification of the success of request to post an event
	 */
	void eventsUpdated(bool success);
	
	/**
	 * Notification on arrival of event lists
	 * Note if number of list is large, then it can download the list page by page.
	 * In that case this signal is emitted multiple times.
	 */
	void eventsAvailable(SmfEventsList* list, QString error, int pageNumber=0);
	
	/**
	 *  Notification on arrival of venues lists
	 * Note if number of list is large, then it can download the list page by page.
	 * In that case this signal is emitted multiple times.
	 */
	void venuesAvailable(SmfVenueList* list, QString error, int pageNumber=0);
private:
  SmfProvider* m_baseProvider;
};
SMF_GETSERVICES(SmfMusicEvents, "org.symbian.smf.client.music.events\0.2")


/**
* provides service ("org.symbian.smf.client.music.lyrics")
*/
class SMFCLIENT_EXPORT SmfLyricsService : public QObject
{
  Q_OBJECT

public:
  /**
   * Constructs SmfLyricsService with base provider info.
   * Seeing as this is a plug-in implementation, these will realistically
   * be generated by SMF factory of some kind
   */

  SmfLyricsService(SmfProvider* baseProvider = 0);
  ~SmfLyricsService();

public:
  
  /**
   * Get the lyrics lists asynchrnously, it fetches texts without time info.
   * lyricsAvailable() notification comes SmfLyricsList with when the data is available
   * @param track Track for which lyrics needs to be fetched.
   */
  virtual void lyrics(SmfTrackInfo track) = 0; 
  
  /**
   * Get the lyrics lists asynchrnously, it fetches texts with time info.
   * Subtitle search filter can be applied
   * subtitleAvailable() notification comes SmfSubtitleList with when the data is available
   * @param track Track for which subtitle needs to be fetched.
   * @param filter Subtitle search filter
   */
  virtual void subtitles(SmfTrackInfo track, SmfSubtitleSearchFilter filter) = 0; // texts with time information
  //APIs to get/set base provider info (SmfProvider)
  
  /**
   * Gets the base provider info
   */
  virtual SmfProvider* getProvider() = 0;
  
  /**
   * Sets the base provider info
   */
  virtual void setProvider(SmfProvider* provider) = 0;

public slots:


signals:
	
	/**
	 * Notification on arrival of lyrics
	 * Note if the list is large, then it can download the list page by page.
	 * In that case this signal is emitted multiple times.
	 */
	void lyricsAvailable(SmfLyricsList* list, QString error, int pageNumber=0);

	/**
	 * Notification on arrival of subtitle based on filter.
	 * Note if the list is large, then it can download the list page by page.
	 * In that case this signal is emitted multiple times.
	 */
	void subtitleAvailable(SmfSubtitleList* list, QString error, int pageNumber=0);
private:
  SmfProvider* m_baseProvider;
};
SMF_GETSERVICES(SmfLyricsService, "org.symbian.smf.client.music.lyrics\0.2")

#endif // SMFMUSIC_H

