/*
* Copyright (c) 2010 Sasken Communication Technologies Ltd.
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the "{License}"
* which accompanies  this distribution, and is available
* at the URL "{LicenseUrl}".
*
* Initial Contributors:
* Chandradeep Gandhi, Sasken Communication Technologies Ltd - Initial contribution
*
* Contributors:
*
* Description:
* Interface spefication for a remote picture gallery
*
*/

#ifndef SMFGALLERY_H
#define SMFGALLERY_H


#include "smfprovider.h"
#include "../common/SmfClientGlobal.h"
class SmfProvider;
class SmfPicture;
class SmfGalleryModel;
class SmfComment; //user id, string, and url


#include <QObject>
#include <QDateTime>
#include <QStringList>
//List of SmfPicture
typedef QList<SmfPicture> SmfPictureList;
/**
 * Interface to a remote gallery service. This class
 * provides some basic gallery functionality to allow applications
 * to interact with a picture gallery in a social network.
 *
 * Note that branding information for the particular service implementation
 * is available from getProvider() API. See also:
 * SmfProvider::serviceName(), SmfProvider::serviceIcon()
 *
 * All of the functionality described here should be implemented by a service
 * specific plug-in object.
 * Interface name for SmfGallery org.symbian.smf.client.gallery
 */
class SMFCLIENT_EXPORT SmfGallery : public QObject
{
  Q_OBJECT

public:
  /**
   * Constructs SmfGallery.
   * @param baseProvider The base provider info
   * Seeing as this is a plug-in implementation, these will realistically
   be generated by SMF factory of some kind
   */

  SmfGallery(SmfProvider* baseprovider = 0);
  ~SmfGallery();

public:
  /**
   * Get the picture listing asynchronously.
   * The picturesAvailable() signal is emitted with SmfPictureList once the pictures have arrived.
   */
  virtual void pictures() = 0; 
  
  /**
   * Returns model
   */
  virtual SmfGalleryModel model() = 0; // maybe we can make a QItemModel-derived model?
  
  /**
   * Returns a user title/caption for the picture
   */
  virtual QString description() = 0; // A user title or caption, maybe?
  
  //APIs to get/set base provider info (SmfProvider)
  
  /**
   * Gets the base provider info
   */
  virtual SmfProvider* getProvider() = 0;
  
  /**
   * Sets the base provider info
   */
  virtual void setProvider(SmfProvider* provider) = 0;

public slots:
	/**
	 * Upload an image.Implemented as slot to connect to UI controls more easily
	 * uploadFinished() signal is emitted with the success value of the upload
	 * @param image the image to be uploaded
	 */
  virtual void upload(SmfPicture* image) = 0;
  
	/**
	 * Upload an list image.Implemented as slot to connect to UI controls more easily
	 * uploadFinished() signal is emitted with the success value of the upload
	 * @param images the list image to be uploaded
	 */
  virtual void upload(SmfPictureList* images) = 0;
  
  /**
   * Posts a comment for an image. uploadFinished() signal is emitted
   * with success of the post once comment is posted.
   * @param image Image to comment on
   * @param comment Comment to post
   */
  virtual void postComment(SmfPicture image, SmfComment comment) = 0;

signals:
	/*
	 * Notification on arrival of list of SmfPicture as a result of request.
	 * Note if number of friends is large, then it can download the list page by page.
	 * In that case this signal is emitted multiple times.
	 * through pictures().
	 * @param pics Picture list
	 * @param error Error string
	 * @param pageNumber Page number
	 */
	void picturesAvailable(SmfPictureList* pics, QString error, int pageNumber=0);
  
  /**
   * Notification of the success of the uploading of image/comment
   * @param success The success of the post
   */
  void uploadFinished(bool success);
private:
  SmfProvider* m_baseProvider;
};

SMF_GETSERVICES(SmfGallery, "org.symbian.smf.client.gallery\0.2")

#endif // SMFGALLERY_H

