/*
* Copyright (c) 2010 Sasken Communication Technologies Ltd.
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the "{License}"
* which accompanies  this distribution, and is available
* at the URL "{LicenseUrl}".
*
* Initial Contributors:
* Chandradeep Gandhi, Sasken Communication Technologies Ltd - Initial contribution
*
* Contributors:
*
* Description:
* Interface spefication for list of contacts from a site
*
*/

#ifndef SMFCONTACTHETCHER_H
#define SMFCONTACTHETCHER_H

#include <QObject>
#include "smfprovider.h"
#include "smfcontact.h"

class SmfProvider; //base-class for service provider
class SmfContact; //class for Contact in a social network
class SmfGroup; //class for a group in social network
class SmfContactList;

//List of SmfGroup
typedef QList<SmfGroup> SmfGroupList;
/**
 * Interface to search for contacts/connections from a service provider. This class
 * provides basic functionality to allow applications to obtain list of
 * contacts or friends in a social networking service.
 * Note that to get the base provider info like service name, icon, description etc
 * use getProvider().
 * See also:
 * SmfProvider::serviceName(), SmfProvider::serviceIcon(), SmfProvider::description()
 *
 * All of the functionality described here should be implemented by a service
 * specific plug-in object.
 * Interface name:- org.symbian.smf.client.contact.fetcher
 */
class SMFCLIENT_EXPORT SmfContactFetcher : public QObject
{
  Q_OBJECT

public:

  /**
   * Constructs the SmfContactFetcher.
   * @param parent base provider info
   * @param contact Used for searching friends of the given contact
   * Seeing as this is a plug-in implementation, these will realistically
   * be generated by SMF factory of some kind 
   */
  SmfContactFetcher(SmfProvider* baseProvider = 0, SmfContact* contact = 0);
  ~SmfContactFetcher();

public:
  /**
   * Get the friend listing asynchronously. The friendsListAvailable() signal
   * is emitted with SmfContactList once data is arrived. 
   */
  virtual void friends() = 0; // list of contact objects
  
  /**
   * Get the list of followers asynchronously. The followersListAvailable() signal
   * is emitted with SmfContactList once data is arrived. 
   */
  virtual void followers() = 0; // list of contact objects
  
  /**
   * Searches for a contact The searchContactFinished() signal
   * is emitted with SmfContactList once data is arrived.
   */
  virtual void search(SmfContact* contact) = 0; // list of contact objects
  
  /**
   * Get the list of groups. The groupListAvailable() signal
   * is emitted with SmfGroupList once data is arrived. 
   */
  virtual void groups() = 0; // list of group objects
  
  /**
   * Searches for Smf Contacts in an Smf group
   * @param  group The group to be searcged in
   * The nextDataPageAvailable() signal
   * of SmfProvider is emitted with SmfContactList once data is arrived.
   */
  virtual void searchInGroup(SmfGroup group) = 0; // list of contact objects

 
  //APIs to get/set base provider info (SmfProvider)
  
  /**
   * Gets the base provider info
   */
  virtual SmfProvider* getProvider() = 0;
  
  /**
   * Sets the base provider info
   */
  virtual void setProvider(SmfProvider* provider) = 0;


public slots:

Q_SIGNALS:

	/**
	 * This signal is emitted when a request to get friends is completed.
	 * Note if number of friends is large, then it can download the list page by page.
	 * In that case this signal is emitted multiple times.
	 *  @param list list of friends
	 *  @param error error string
	 *  @param pageNumber Page number
	 *  @see friends()
	 */
	void friendsListAvailable(SmfContactList* list, QString error, int pageNumber=0); 

	/**
	 * This signal is emitted when a request to get followers is completed
	 *  Note if number of followers is large, then it can download the list page by page
	 *  In that case this signal is emitted multiple times.
	 *  @param list list of followers
	 *  @param error error string
	 *  @param pageNumber Page number
	 *  @see followers()
	 */
	void followersListAvailable(SmfContactList* list, QString error, int pageNumber=0); 
	
	/**
	 * This signal is emitted when a request to get groups is completed
	 * Note if number of groups is large, then it can download the list page by page
	 *  In that case this signal is emitted multiple times.
	 *  @param list list of groups
	 *  @param error error string
	 *  @param pageNumber Page number
	 *  @see groups()
	 */
	void groupListAvailable(SmfGroupList* list, QString error, int pageNumber=0);
	
	/**
	 * Emitted when search for a contact is finished.
	 * Note if number of contacts in the search is large, then it can download the list page by page
	 * In that case this signal is emitted multiple times.
	 * @param list List of filtered contacts
	 * @param pageNumber Page number
	 * @see search()
	 */
	void searchContactFinished(SmfContactList* list,QString error, int pageNumber=0);
	
	/**
	 * Emitted when search for a contact in a group is finished
	 * Note if number of contacts in the search is large, then it can download the list page by page
	 * In that case this signal is emitted multiple times.
	 * @param list list of filtered contacts
	 * @param pageNumber Page number
	 * @see searchInGroup()
	 */
	void searchInGroupFinished(SmfContactList* list,QString error, int pageNumber=0);

private:
  SmfProvider* m_baseProvider;
};

SMF_GETSERVICES(SmfContactFetcher, "org.symbian.smf.client.contact.fetcher\0.2")
#endif // SMFCONTACTHETCHER_H
