/**
 * Copyright (c) 2010 Sasken Communication Technologies Ltd.
 * All rights reserved.
 * This component and the accompanying materials are made available
 * under the terms of the "Eclipse Public License v1.0" 
 * which accompanies  this distribution, and is available
 * at the URL "http://www.eclipse.org/legal/epl-v10.html"
 *
 * Initial Contributors:
 * Chandradeep Gandhi, Sasken Communication Technologies Ltd - Initial contribution
 *
 * Contributors:
 * Manasij Roy, Nalina Hariharan
 * 
 * Description:
 * The SmfEvent class represents an event
 *
 */

#ifndef SMFCONTACTHETCHER_H
#define SMFCONTACTHETCHER_H

#include <QObject>
#include "smfglobal.h"
#include "smfprovider.h"
#include "smfcontact.h"
#include "smflocation.h"
#include "smfgroup.h"

class SmfProvider; //base-class for service provider
class SmfContact; //class for Contact in a social network
class SmfGroup; //class for a group in social network
class SmfContactFetcherPrivate;

typedef QList<SmfContact>  SmfContactList;
typedef QList<SmfGroup> SmfGroupList;
/**
 * @ingroup smf_client_group 
 * Interface to search for contacts/connections from a service provider. This class
 * provides basic functionality to allow applications to obtain list of
 * contacts or friends in a social networking service.
 * Note that to get the base provider info like service name, icon, description etc
 * use getProvider().
 * See also:
 * SmfProvider::serviceName(), SmfProvider::serviceIcon(), SmfProvider::description()
 *
 * All of the functionality described here should be implemented by a service
 * specific plug-in object.
 * Interface name:- org.symbian.smf.client.contact.fetcher
 */
class  SMFCLIENT_EXPORT SmfContactFetcher : public QObject
{
  Q_OBJECT

public:

  /**
   * Constructs the SmfContactFetcher.
   * @param parent base provider info
   * @param contact Used for searching friends of the given contact
   * Seeing as this is a plug-in implementation, these will realistically
   * be generated by SMF factory of some kind
   */
  SmfContactFetcher(SmfProvider* baseProvider);
  SmfContactFetcher(SmfProvider* baseProvider, SmfContact* contact);
  ~SmfContactFetcher();

public:
  /**
   * Get the friend listing asynchronously. The friendsListAvailable() signal
   * is emitted with SmfContactList once data is arrived.
   * When the list is big user can specify the page number and per page item data.
   * If not supplied by the user default values are used.
   * @param pageNum Page number to download, SMF_FIRST_PAGE denotes fresh query.
   * @param perPage Item per page, default is SMF_ITEMS_PER_PAGE
   */
  bool friends(int pageNum=SMF_FIRST_PAGE,int perPage=SMF_ITEMS_PER_PAGE);

  /**
   * Get the list of followers asynchronously. The followersListAvailable() signal
   * is emitted with SmfContactList once data is arrived. Please note that some
   * service may not support followers/fans - FALSE is returned if not supported.
   * When the list is big user can specify the page number and per page item data.
   * If not supplied by the user default values are used.
   * @param pageNum Page number to download, SMF_FIRST_PAGE denotes fresh query.
   * @param perPage Item per page, default is SMF_ITEMS_PER_PAGE
   */
  bool followers(int pageNum=SMF_FIRST_PAGE,int perPage=SMF_ITEMS_PER_PAGE); // list of contact objects

  /**
   * Searches for a contact The searchContactFinished() signal
   * is emitted with SmfContactList once data is arrived.
   * When the list is big user can specify the page number and per page item data.
   * If not supplied by the user default values are used.
   * @param pageNum Page number to download, SMF_FIRST_PAGE denotes fresh query.
   * @param perPage Item per page, default is SMF_ITEMS_PER_PAGE
   */
  void search(SmfContact* contact,int pageNum=SMF_FIRST_PAGE,int perPage=SMF_ITEMS_PER_PAGE) ; // list of contact objects

  /**
   * Searches for a contacts (friends) who are near the user.
   * Signal searchNearFinished() is emitted with SmfContactList once data is arrived.
   * Proximity defines accuracy level
   * When the list is big user can specify the page number and per page item data.
   * If not supplied by the user default values are used.
   * @param pageNum Page number to download, SMF_FIRST_PAGE denotes fresh query.
   * @param perPage Item per page, default is SMF_ITEMS_PER_PAGE
   */
  bool searchNear(SmfLocation* location,SmfLocationSearchBoundary proximity,int pageNum=SMF_FIRST_PAGE,int perPage=SMF_ITEMS_PER_PAGE) ;


  /**
   * Get the list of groups. The groupListAvailable() signal
   * is emitted with SmfGroupList once data is arrived. False might be returned
   * if this service doesn't support any mode of grouping (very rare).
   * When the list is big user can specify the page number and per page item data.
   * If not supplied by the user default values are used.
   * @param pageNum Page number to download, SMF_FIRST_PAGE denotes fresh query.
   * @param perPage Item per page, default is SMF_ITEMS_PER_PAGE
   */
   bool groups(int pageNum=SMF_FIRST_PAGE,int perPage=SMF_ITEMS_PER_PAGE) ; // list of group objects

  /**
   * Searches for Smf Contacts in an Smf group.
   * When the list is big user can specify the page number and per page item data.
   * If not supplied by the user default values are used.
   * @param pageNum Page number to download, SMF_FIRST_PAGE denotes fresh query.
   * @param perPage Item per page, default is SMF_ITEMS_PER_PAGE
   * @param  group The group to be searcged in
   */
   bool searchInGroup(SmfGroup group,int pageNum=SMF_FIRST_PAGE,int perPage=SMF_ITEMS_PER_PAGE); // list of contact objects
   
   /**
    * Request for a custom operation.
    * @param operationId OperationId
    * @param customData Custom data to be sent
    * Note:-Interpretation of operationId and customData is upto the concerned
    * plugin and client application. service provider should provide some
    * serializing-deserializing utilities for these custom data
    */
   void customRequest(const int& operationId,QByteArray* customData);

  //APIs to get/set base provider info (SmfProvider)

  /**
   * Gets the base provider info
   */
   SmfProvider* getProvider() ;

signals:

	/**
	 * This signal is emitted when a request to get friends is completed.
	 * Note if number of friends is large, then it can download the list page by page.
	 * In that case this signal is emitted multiple times.
	 *  @param list list of friends
	 *  @param error error value
	 *  @param resultPage Page number info
	 *  @see friends()
	 */
	void friendsListAvailable(SmfContactList* list, SmfError error, SmfResultPage resultPage);

	/**
	 * This signal is emitted when a request to get followers is completed
	 *  Note if number of followers is large, then it can download the list page by page
	 *  In that case this signal is emitted multiple times.
	 *  @param list list of followers
	 *  @param error error value
	 *  @param resultPage Page number info
	 *  @see followers()
	 */
	void followersListAvailable(SmfContactList* list, SmfError error, SmfResultPage resultPage);

	/**
	 * This signal is emitted when a request to get groups is completed
	 * Note if number of groups is large, then it can download the list page by page
	 *  In that case this signal is emitted multiple times.
	 *  @param list list of groups
	 *  @param error error value
	 *  @param resultPage Page number info
	 *  @see groups()
	 */
	void groupListAvailable(SmfGroupList* list, SmfError error, SmfResultPage resultPage);

	/**
	 * Emitted when search for a contact is finished.
	 * Note if number of contacts in the search is large, then it can download the list page by page
	 * In that case this signal is emitted multiple times.
	 * @param list List of filtered contacts
	 * @param resultPage Page number info
	 * @see search()
	 */
	void searchContactFinished(SmfContactList* list,SmfError error, SmfResultPage resultPage);


	/**
	 * Emitted when search for contacts who are near a geographic location, is finished.
	 * Note if number of contacts in the search is large, then it can download the list page by page
	 * In that case this signal is emitted multiple times.
	 * @param list List of filtered contacts
	 * @param resultPage Page number info
	 * @see search()
	 */
	void searchNearFinished(SmfContactList* list,SmfError error, SmfResultPage resultPage);

	/**
	 * Emitted when search for a contact in a group is finished
	 * Note if number of contacts in the search is large, then it can download the list page by page
	 * In that case this signal is emitted multiple times.
	 * @param list list of filtered contacts
	 * @param resultPage Page number info
	 * @see searchInGroup()
	 */
	void searchInGroupFinished(SmfContactList* list,SmfError error, SmfResultPage resultPage);
	
	/**
	 * Emitted when custom data is available
	 * @param operationId Requested operation id
	 * @param customData Custom data received, interpretation is not the responsibility of Smf
	 */
	void customDataAvailable(int operationId, QByteArray* customData);
	//so that private impl can directly call emit
	friend class SmfContactFetcherPrivate;
private:
  SmfProvider* m_baseProvider;
  SmfContact* m_frndContact; //used for searching
  //private impl wrapper
  SmfContactFetcherPrivate* m_private;

};

SMF_SERVICE_NAME(SmfContactFetcher, "org.symbian.smf.client.contact.fetcher\0.2")
#endif // SMFCONTACTHETCHER_H
