/*!	\file 
	\brief File containing class description for DataStoreManager class.
	
	Copyright (c) 2010 Sasken Communication Technologies Ltd.
	All rights reserved.
	This component and the accompanying materials are made available
	under the terms of the "{License}"
	which accompanies  this distribution, and is available
	at the URL "{LicenseUrl}".

	\author Jaspinder Singh, Sasken Communication Technologies Ltd - Initial contribution

	\version 0.1

*/

#ifndef DATASTOREMANAGER_H
#define DATASTOREMANAGER_H

#include <QString>
#include <QObject>
#include <QtSql>
#include "smfSns.h"
#include "smfSocialProfile.h"
#include "smfUserProfile.h"

enum DataStoreManagerState{READY, BUSY, CLOSED, ERROR};

//!	\class 	DataStoreManager
/*!
	\brief 	Data Store Manager
	\brief 	Data Store Manager provides the functional interface between the data store and the outside world. 
			It manages access to the data store and encapsulates prebuilt queries for relation management and makes sure 
			that the data in persistent storage is always in a consistent state.
			The DSM implements the Singleton Pattern, which means that only one instance of DSM will be available throughout the system.
	
	\warning	Not Thread Safe
	\warning	Do not subclass
*/
class DataStoreManager : public QObject
{
	Q_OBJECT
	
	public:
        static DataStoreManager* getDataStoreManager();
        ~DataStoreManager();

        QList <SMFSocialProfile> getAllRelated(const SMFUserProfile&);
        SMFSocialProfile getRelatedByService(const SMFUserProfile&, const SMFSocialNetworkingSite&);

        DataStoreManagerState getState() const;
        QString getError() const;



        SMFUserProfile getUserProfile(const QString& name, const QString& contact_id);
        SMFSocialNetworkingSite getSNSEntry(const QString& name);
        void saveUserProfile(const SMFUserProfile& user_profile);
        void saveSocialProfile(const SMFSocialProfile& social_profile);
        void saveSNSEntry(const SMFSocialNetworkingSite& sns);
        void modifyRelation(SMFSocialProfile& sns, SMFUserProfile& new_user_profile);


	public slots:
        int addUserProfile( SMFUserProfile&);
        int deleteUserProfile( SMFUserProfile&);
        int addSocialProfile( SMFSocialProfile&);
		int deleteSocialProfile(SMFSocialProfile&);
        int addSNSEntry( SMFSocialNetworkingSite&);
		int deleteSNSEntry(SMFSocialNetworkingSite&);
        int createRelation(const SMFUserProfile&, SMFSocialProfile&);
        int deleteRelation(const SMFUserProfile&, SMFSocialProfile&);

	private:
        static DataStoreManager* m_dsm_instance;		// Unique Instance for DSM (Singleton Implementation)
        static const QString db_name;
        int m_dsm_instance_count; 				// Track references to the DSM
        static DataStoreManagerState state;				// Current state of DSM
        QString m_last_msg;					// Last message/error generated by the database.
        QSqlDatabase db;

        DataStoreManager(const QString& db_name, QObject* parent = 0); 			// Private Constructor.
        bool InitializeDataBase();				// Initialization code to be called once while lazy construction of the instance
};

#endif
