/*
* Copyright (c) 2007 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:   Font manager allows creation, enumeration and management of
*                platform fonts in Hitchcock UI Toolkit usage.
*
*/



#ifndef HUIFONTMANAGER_H
#define HUIFONTMANAGER_H

#include <e32std.h>

#include <uiacceltk/huifontspecification.h>

/**
 * THuiFontManager allows user to enumerate all fonts supported by the platform
 * that can be used in Hitchcock UI Toolkit user interfaces. THuiFontManager also
 * allows creation of platform supported fonts and manages the lifetime of these
 * font objects. Fonts are referred to by Hithcock UI Toolkit session specific 
 * unique IDs.
 */
NONSHARABLE_CLASS(THuiFontManager)
{
public:

    /**
     * Default constructor. Creates the THuiFontManager object.
     * This is called by CHuiEnv to create the system font manager instance.
     */
    IMPORT_C THuiFontManager();
    
    /**
     * Default destructor.
     */    
    IMPORT_C virtual ~THuiFontManager();
    
    /** @beginAPI */
    
    /**
     * Retrieves the CHuiFontSpecification array that describes all fonts
     * supported by this platform. Any of these specifications can be used to create
     * new font instances. The user has to pass in an array that will be filled by this
     * method with the system font specifications.
     * @see CreateFontL()
     *
     * @param aArray Reference to an array that is going to include all platform supported
     * fonst specifications after this call has exited.
     */
    IMPORT_C void EnumerateFontsL(RArray<THuiFontSpecification>& aArray) const;
    
    /**
     * Creates a new font object based on the font specification passed into this method.
     * If no matching font specification is found from the system the method leaves with 
     * KErrNotSupported leave code. User can make sure that the font is supported by using
     * the EnumerateFontsL - method and using the font specification array generated by
     * the method. A unique ID is constructed based on this font. This font can be accessed 
     * anytime through the FontL - method based on this unique ID.
     * @see FontL()
     * @see EnumerateFontsL()
     * 
     * @param aSpecification A font specification object according to which the font object is created.
     * @return Unique ID describing the newly created font object. If similar font already exists then
     * the ID of the already created font is returned.
     */
    IMPORT_C TInt CreateFontL(const THuiFontSpecification& aSpecification);
    
    /** @endAPI */
    
private:

    /** Registry of all fonts within this manager. These fonts are accessed by font id. */
    RArray<THuiFontSpecification> iFonts;

};

#endif // HUIFONTMANAGER_H
