/*
* Copyright (c) 2006-2007 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:   Curve path layout definition.
*
*/



#ifndef C_ALFCURVEPATHLAYOUT_H
#define C_ALFCURVEPATHLAYOUT_H

#include <alf/alflayout.h>
#include <alf/alfmappingfunctions.h>

class CAlfCurvePath;

/**
 *  Curve path layout can be used to control size and position properties of
 *  a visual using a curve path object.
 *  @see CAlfCurvePath
 *
 *  When using a curve path layout the user has to specify which properties of
 *  a visual should be controlled by the layout. By default the curve path layout
 *  does not affect any properties of its child visuals. Mapping function identifier
 *  retrieved through CAlfCurvePathLayout::MappingFunctionIdentifier() - method
 *  can be used to bind the curve path layout to properties of a visual.
 *
 *  The curve path layout contains a curve path instance that can be used to
 *  customize the path that is used to layout the properties of a visual. Reference
 *  to the curve path can be acquired through the CAlfCurvePathLayout::CurvePath() - method.
 *
 *  The following code example creates a curve path layout, sets its base unit to normalized (0.0 - 1.0)
 *  units, defines a curve path, binds the curve path layouting to image visual's position
 *  property and animates the position of the image visual on the curve path from start point of the curve path (0.0)
 *  to end point of the curve path (2.0) within 3 seconds. Notice that the curve path layout does not
 *  adjust the size of the image visual and because of that the image visual size is explicitly set to (0.25, 0.25).
 *  The size of the image visual is defined in base units of it's parent layout.
 *
 *	@code
 *  // Create a curve path layout
 *  CAlfCurvePathLayout* layout = CAlfCurvePathLayout::AddNewL(*this);
 *  layout->SetBaseUnit(TAlfMetric(1.0, EAlfUnitNormalized));
 *  
 *  // Define a curve path
 *  CAlfCurvePath& curvePath = layout->CurvePath();
 *  curvePath.AppendArcL(TAlfRealPoint(0.5, 0.5), TAlfRealSize(0.5, 0.5), 180.0, 360.0, 2.0);
 *  
 *  // Create an image visual
 *  CAlfImageVisual* imageVisual = CAlfImageVisual::AddNewL(*this, layout);
 *  imageVisual->SetSize(TAlfRealSize(0.25, 0.25));
 *  CAlfTextureManager& defaultTexture = aEnv.TextureManager( ); 
 *  CAlfTexture& texture = defaultTexture.LoadTextureL(_L("album4.jpg"),EAlfTextureFlagDefault,KAlfAutoGeneratedTextureId);
 *  imageVisual->SetImage(TAlfImage(texture));
 *  
 *  // Control the position of the image visual with the curve path
 *  TAlfTimedPoint pos;
 *  pos.SetMappingFunctionIdentifier(layout->MappingFunctionIdentifier());
 *  pos.SetTarget(2.0, 3000);
 *  imageVisual->SetPos(pos);
 *  @endcode
 *
 *  @lib alfclient.lib
 *  @since S60 v3.2
 */
class CAlfCurvePathLayout : public CAlfLayout, public MAlfMappingFunction
    {

public:

    IMPORT_C static CAlfCurvePathLayout* AddNewL(
        CAlfControl& aOwnerControl, 
        CAlfLayout* aParentLayout = 0);

    /**
     * Constructor.
     *
     * @param aOwner  Owner control.
     */
    IMPORT_C CAlfCurvePathLayout();

    /**
     * From CAlfLayout. Second-phase constructor.
     */
    IMPORT_C void ConstructL(CAlfControl& aOwner);

    /**
     * Virtual destructor.
     */
    IMPORT_C ~CAlfCurvePathLayout();
  
    /**
     * Returns the curve path instance of the layout. The returned instance
     * can be used to modify the curve path that is used to control the properties
     * of a layed out visual. The curve path instance is defined in the base units
     * used by this layout.
     * @see CAlfCurvePathLayout::BaseUnit()
     * @see CAlfCurvePathLayout::SetBaseUnit()
     *
     * @return Curve path. This instance cannot be passed
     *         as general mapping function to other instances.
     */
    IMPORT_C CAlfCurvePath& CurvePath();
   
    /**
     * From CAlfVisual
     * @see CAlfVisual
     */
    IMPORT_C void RemoveAndDestroyAllD();
    IMPORT_C void UpdateChildrenLayout(TInt aTransitionTime = 0);
    IMPORT_C CAlfVisual* FindTag(const TDesC8& aTag);

    /** 
     * From CAlfLayout
     * @see CAlfLayout
     */
    IMPORT_C TAlfXYMetric BaseUnit() const; 
    
    /**
     * From MAlfMappingFunction
     * @see MAlfMappingFunction
     */
    IMPORT_C TInt MappingFunctionIdentifier() const;
    IMPORT_C TReal32 MapValue(TReal32 aValue, TInt aMode) const __SOFTFP;


protected:
    /**
     * From CAlfVisual
     * @see CAlfVisual
     */
    IMPORT_C void DoRemoveAndDestroyAllD();
    IMPORT_C void PropertyOwnerExtension(const TUid& aExtensionUid, TAny** aExtensionParams);


private: // data

    struct TCurvePathLayoutPrivateData;
    TCurvePathLayoutPrivateData* iCurvePathLayoutData;

    };


#endif // C_ALFCURVEPATHLAYOUT_H
