/*
* Copyright (c) 2003 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:  Defines the package ID class used to identify skin packages
*
*/


#ifndef AKNSPKGID_H
#define AKNSPKGID_H

//  INCLUDES
#include <e32std.h>
#include <e32base.h>

// CONSTANTS

static const TInt KAknsPkgIDDesSize = 23;

// DATA TYPES

typedef TBuf<KAknsPkgIDDesSize> TAknsPkgIDBuf;

// FORWARD DECLARATIONS

// CLASS DECLARATION

/**
* Package identifier used to identify skin packages. Package ID may be either
* a UID or a timestamp and random number pair.
*
* Note that the class does not define any constructors. This ensures that 
* @c TAknsPkgID can be used in ROM constants.
*
* @lib AknSkinSrv.lib
*
* @since 2.0
*/
class TAknsPkgID
    {    
    public: // New functions

        /**
        * Sets skin package ID by giving a UID.
        *
        * @param aUid New UID to be set as the value of the package ID.
        */
        inline void Set( const TUid aUid );

        /**
        * Sets skin package ID by giving a timestamp and a random number.
        *
        * @param aTimestamp Timestamp as an integer. Must be non-zero.
        *
        * @param aNumber Random number.
        */
        inline void Set( const TInt aTimestamp, const TInt aNumber );
        
        /**
        * Sets skin package ID by an existing package ID.
        *
        * @param aID Skin package ID.
        */
        inline void Set( const TAknsPkgID& aID );

        /**
        * Checks whether the skin package ID is a UID.
        *
        * @return ETrue if the skin package ID is a UID, EFalse otherwise.
        */
        inline TBool IsUid() const;

        /**
        * Returns the skin package ID as a UID. If the package ID is not a
        * UID, the result is undefined.
        *
        * @return Skin packge ID as a UID.
        */
        inline TUid Uid() const;

    public: // New functions - String conversions

        /**
        * Copies the content of the package ID to the given descriptor.
        * The size of the descriptor must be at least the size of
        * @c TAknsPkgIDBuf. Otherwise the method may panic.
        *
        * @param aDes Target descriptor.
        */
        IMPORT_C void CopyToDes( TDes& aDes ) const;

        /**
        * Sets the content of the package ID based on the given character
        * representation. The content of the given descriptor must be
        * valid (as generated by @c CopyToDes). 
        *
        * If the given descriptor doesn't contain a correctly formatted
        * skin package ID the method leaves and the content of the object
        * is undefined.
        *
        * @param aDes Source descriptor.
        */
        IMPORT_C void SetFromDesL( const TDesC& aDes );

    public: // Data

        TInt iNumber;       //!< Number or UID.
        TInt iTimestamp;    //!< Timestamp (or 0 if UID).

    };

// CONSTANTS (continued)

static const TAknsPkgID KAknsNullPkgID = { 0, 0 };

// FUNCTION PROTOTYPES

/**
* C++ equal operator.
* Checks whether two package ID objects have identical content.
*
* @param aFirst First object to be compared.
*
* @param aSecond Second object to be compared.
*
* @return true if objects have identical content, false otherwise.
*/
inline TBool operator==( const TAknsPkgID& aFirst, const TAknsPkgID& aSecond );

/**
* C++ not equal operator.
* Checks whether two package ID objects have identical content.
*
* @param aFirst First object to be compared.
*
* @param aSecond Second object to be compared.
*
* @return false if objects have identical content, true otherwise.
*/
inline TBool operator!=( const TAknsPkgID& aFirst, const TAknsPkgID& aSecond );

// INLINED MEMBERS

#include "AknsPkgID.inl"

#endif // AKNSITEMID_H   

// End of File
