/*
* Copyright (c) 2002-2006 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:  A private accessor API for Accessory Policy internal use. 
*                  This API is used to modify the attribute values of a 
*                  Generic ID instance.
*
*/



#ifndef TACCPOLGENERICIDACCESSOR_H
#define TACCPOLGENERICIDACCESSOR_H

//  INCLUDES
#include <AccPolGenericID.h>
#include <AccPolGenericIDArray.h>
#include <AccPolNameValueRecord.h>

// CONSTANTS

// Used internally by Accessory Policy. This defines the first attribute
// definition. All other definitions are relative to this.
const TUint KAccPolFirstAttributeDefinition = 1;
// Device type
const TUint KAccPolDTDeviceType             = KAccPolFirstAttributeDefinition;
// Physical connection
const TUint KAccPolPCPhysicalConnection     = KAccPolFirstAttributeDefinition + 1;
// Application protocol
const TUint KAccPolAPApplicationProtocol    = KAccPolFirstAttributeDefinition + 2;
// Capabilities subblock
const TUint KAccPolSBCapabilities           = KAccPolFirstAttributeDefinition + 3;

// MACROS

// DATA TYPES

// FUNCTION PROTOTYPES

// FORWARD DECLARATIONS
class CAccConGenericID;

// CLASS DECLARATION

/**
*  This API is used to modify the attribute values of a Generic ID instance
*
*  @lib AccPolicy.lib
*  @since S60 3.1
*/
class TAccPolGenericIDAccessor
    {
    public: // New functions

        /**
        * Set matching Generic ID feature attribute.
        *
        * @since S60 3.1
        * @param aGenericID           Generic ID instance where the features are stored.
        * @param aGIDAttribute        Generic ID attribute to be set.
        * @param aGIDAttributeBitMask Attribute(s) value to be set.
        * @return void
        */
        IMPORT_C static void SetFeatureAttributeL( TAccPolGenericID& aGenericID, 
                                                   const TUint aGIDAttribute, 
                                                   const TUint32 aGIDAttributeBitMask );

        /**
        * Set matching Generic ID feature attribute.
        *
        * @since S60 3.1
        * @param aGenericID           Generic ID instance where the features are stored.
        * @param aGIDAttribute        Generic ID attribute to be set.
        * @param aGIDAttributeBitMask Attribute(s) value to be set.
        * @return void
        */
        IMPORT_C static void SetFeatureAttributeL( TAccPolGenericID& aGenericID, 
                                                   const TUint aGIDAttribute, 
                                                   const TUint64 aGIDAttributeBitMask );

        /**
        * Set vendor specific hardware model identifier for this Generic ID.
        *
        * @since S60 3.1
        * @param aGenericID Generic ID instance where the HWModelID is stored.
        * @param aHWModelID Hardware model identifier to be set.
        * @return void
        */
        IMPORT_C static void SetHWModelID( TAccPolGenericID& aGenericID, 
                                           const TDesC& aHWModelID );

        /**
        * Set vendor specific hardware device identifier for this Generic ID.
        *
        * @since S60 3.1
        * @param aGenericID  Generic ID instance where the HWDeviceID is stored.
        * @param aHWDeviceID Hardware device identifier to be set.
        * @return void
        */
        IMPORT_C static void SetHWDeviceID( TAccPolGenericID& aGenericID, 
                                            const TUint64 aHWDeviceID );

        /**
        * Set device address ID for this Generic ID.
        *
        * @since S60 3.1
        * @param aGenericID      Generic ID instance where the aDeviceAddress is stored.
        * @param aDeviceAddress  Device address to be set, for instance BT Address.
        * @return void
        */
        IMPORT_C static void SetDeviceAddress( TAccPolGenericID& aGenericID,
                                               const TUint64 aDeviceAddress );

        /**
        * Set unique ID from DB for this Generic ID.
        *
        * @since S60 3.1
        * @param aGenericID Generic ID instance where the unique identifier is stored.
        * @param aDBID      Unique identifier to be set, generated by DBMS server.
        * @return void
        */
        IMPORT_C static void SetDBID( TAccPolGenericID& aGenericID, const TInt aDBID );

        /**
        * Compare two Generic ID instances based on static member attributes 
        *   (AccPolGIDHeader).
        *
        * @since S60 3.1
        * @param aLValue A reference to the first (or left) Generic ID to be compared
        * @param aRValue A reference to the second (or right) Generic ID to be compared.
        * @return ETrue if AccPolGIDHeader instance of both Generic 
        *   IDs are exactly the same, else EFalse.
        */
        IMPORT_C static TBool CompareGenericID( const TAccPolGenericID& aLValue, 
                                                const TAccPolGenericID& aRValue );

        /**
        * Set static attributes for this Generic ID.
        *
        * @since S60 3.1
        * @param aLGenericID Generic ID instance where the static attributes are stored.
        * @param aRGenericID Static attributes to be set.        
        * @return void
        */                                        
        IMPORT_C static void SetGenericIDStaticAttributes( CAccConGenericID* const aLGenericID, 
                                                           const TAccPolGenericID& aRGenericID );

        /**
        * Set static attributes for this Generic ID.
        *
        * @since S60 3.1
        * @param aGenericID Generic ID instance where the static attributes are stored.
        * @param aGenericIDHeader Static attributes to be set.        
        * @return void
        */                                        
        static void SetGenericIDStaticAttributes( TAccPolGenericID& aGenericID, 
                                                  const TAccPolGIDHeader &aGenericIDHeader);


                               
        /**
        * Get a pointer to the member name value buffer.
        *
        * @since S60 3.1
        * @param aGenericID Generic ID instance.
        * @return Pointer to name value buffer.
        */
        IMPORT_C static CBufFlat* NameValueBuf( CAccConGenericID* const aGenericID );
     
    
        /**
        * Get reference to Generic ID header.
        *
        * @since S60 3.1
        * @param aGenericID Generic ID instance.
        * @return Reference to Generic ID header.
        */
        IMPORT_C static TAccPolGenericID& GenericIDRef( CAccConGenericID* const aGenericID );
 
        /**
        * Debug print Generic ID using RDebug::Print().
        *
        * @since S60 3.1
        * @param aGenericID Generic ID instance to be print.
        * @return void
        */
        IMPORT_C static void DebugPrint( const TAccPolGenericID& aGenericID );

    private:  // Constructors and destructor
        
        /**
        * C++ default constructor.
        */
        TAccPolGenericIDAccessor();
    };

#endif      // TACCPOLGENERICIDACCESSOR_H

// End of File
