// Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
//


/**
 @file
 @publishedPartner
 @prototype
*/


#include "eglwindow.h"
#include <e32base.h>
#include <w32std.h>


//REglWindowBase-----------------------------------------------------

REglWindowBase::REglWindowBase() :
	iIsRWindow(0xFFFFFFFF)
	{
	//Check that BC is not broken between RWindow and REglWindowBase.
	__ASSERT_COMPILE(sizeof(MWsClientClass) == sizeof(REglWindowBase));
	}


TBool REglWindowBase::IsRWindow() const
    {
    //REglWindowBase works out whether it is really an RWindow or a class
    //derived from REglWindowBase by checking the location where iWsBuffer 
    //would be in an RWindow (iIsRWindow in this class). If this is 
    //0xFFFFFFFF, it is not an RWindow.
    return (iIsRWindow != 0xFFFFFFFF);
    }

//-------------------------------------------------------------------


//Static function used to copy one configuration to another.
static void CopySurfaceConfiguration(const TSurfaceConfiguration& aSrc, TSurfaceConfiguration& aDest)
    {
    TRect extent;
    TRect viewport;
    TSurfaceId surface;
    aSrc.GetExtent(extent);
    aDest.SetExtent(extent);
    aSrc.GetSurfaceId(surface);
    aDest.SetSurfaceId(surface);
    aSrc.GetViewport(viewport);
    aDest.SetViewport(viewport);
    aDest.SetFlip(aSrc.Flip());
    aDest.SetOrientation(aSrc.Orientation());
    }


//REglStandAloneWindow-----------------------------------------------

//Class used to hold surface window data to prevent the 
//possibility of binary break if TSurfaceConfiguration grows.
NONSHARABLE_CLASS(REglStandAloneWindow::TEglStandAloneWindowData)
    {
public:
    TEglStandAloneWindowData(const TSize& aSizeInPixels, const TSize& aSizeInTwips, TInt aScreenNumber, MEglStandAloneWindowObserver& aObserver);
    MEglStandAloneWindowObserver& iObserver;
    TSize iSizeInPixels;
    TSize iSizeInTwips;
    TInt iScreenNumber;
    TRgb iBackgroundColor;
    TSurfaceConfiguration iSurfaceConfig;
    };


REglStandAloneWindow::TEglStandAloneWindowData::TEglStandAloneWindowData(const TSize& aSizeInPixels, const TSize& aSizeInTwips, TInt aScreenNumber, MEglStandAloneWindowObserver& aObserver) :
    iObserver(aObserver),
    iSizeInPixels(aSizeInPixels),
    iSizeInTwips(aSizeInTwips),
    iScreenNumber(aScreenNumber),
    iBackgroundColor(0,0,0,0) //Transparent Black.
    {
    CopySurfaceConfiguration(TSurfaceConfiguration(), iSurfaceConfig);
    }


EXPORT_C REglStandAloneWindow::REglStandAloneWindow() :
    iData(NULL)
	{
	}

	
EXPORT_C TInt REglStandAloneWindow::Create(const TSize& aSizeInPixels, const TSize& aSizeInTwips, TInt aScreenNumber, MEglStandAloneWindowObserver& aObserver)
	{
	if(iData)
	    {
	    return KErrAlreadyExists;
	    }
	
	EGL_WINDOW_ASSERT_DEBUG(aSizeInPixels.iWidth >= 0, EEglWindowPanicSizeInPixelsIsNegative);
	EGL_WINDOW_ASSERT_DEBUG(aSizeInPixels.iHeight >= 0, EEglWindowPanicSizeInPixelsIsNegative);
	EGL_WINDOW_ASSERT_DEBUG(aSizeInTwips.iWidth >= 0, EEglWindowPanicSizeInTwipsIsNegative);
	EGL_WINDOW_ASSERT_DEBUG(aSizeInTwips.iHeight >= 0, EEglWindowPanicSizeInTwipsIsNegative);
	EGL_WINDOW_ASSERT_DEBUG(aScreenNumber >= 0 || aScreenNumber == KAllScreens, EEglWindowPanicInvalidScreenNumber);
	
	//Allocate and initialize data.
    iData = new TEglStandAloneWindowData(aSizeInPixels, aSizeInTwips, aScreenNumber, aObserver);
    if(!iData)
        {
        return KErrNoMemory;
        }
    
	return KErrNone;
	}


EXPORT_C void REglStandAloneWindow::Close()
    {
    if(!iData)
        {
        return;
        }
    
    //Remove background surface with a forced redraw so that 
    //the contents immediately disappear from the screen.
    RemoveBackgroundSurface(ETrue);
    delete iData;
    iData = NULL;
    }


EXPORT_C void REglStandAloneWindow::SetSizeInPixels(const TSize& aSize)
    {
    EGL_WINDOW_ASSERT_ALWAYS(iData, EEglWindowPanicREglStandAloneWindowNotCreated);
    EGL_WINDOW_ASSERT_DEBUG(aSize.iWidth >= 0, EEglWindowPanicSizeInPixelsIsNegative);
    EGL_WINDOW_ASSERT_DEBUG(aSize.iHeight >= 0, EEglWindowPanicSizeInPixelsIsNegative);
    iData->iSizeInPixels = aSize;
    }


EXPORT_C void REglStandAloneWindow::SetSizeInTwips(const TSize& aSize)
    {
    EGL_WINDOW_ASSERT_ALWAYS(iData, EEglWindowPanicREglStandAloneWindowNotCreated);
    EGL_WINDOW_ASSERT_DEBUG(aSize.iWidth >= 0, EEglWindowPanicSizeInTwipsIsNegative);
    EGL_WINDOW_ASSERT_DEBUG(aSize.iHeight >= 0, EEglWindowPanicSizeInTwipsIsNegative);
    iData->iSizeInTwips = aSize;
    }


EXPORT_C TRgb REglStandAloneWindow::BackgroundColor() const
    {
    EGL_WINDOW_ASSERT_ALWAYS(iData, EEglWindowPanicREglStandAloneWindowNotCreated);
    return iData->iBackgroundColor;
    }


EXPORT_C TBool REglStandAloneWindow::GetBackgroundSurface(TSurfaceConfiguration& aConfiguration) const
    {
    EGL_WINDOW_ASSERT_ALWAYS(iData, EEglWindowPanicREglStandAloneWindowNotCreated);
    
    TSurfaceId surface;
    iData->iSurfaceConfig.GetSurfaceId(surface);
    if(surface.IsNull())
        {
        return EFalse;
        }
    
    CopySurfaceConfiguration(iData->iSurfaceConfig, aConfiguration);
    return ETrue;
    }


EXPORT_C TBool REglStandAloneWindow::IsValid() const
    {
    return iData != NULL;
    }


EXPORT_C TSize REglStandAloneWindow::SizeInPixels() const
    {
    EGL_WINDOW_ASSERT_ALWAYS(iData, EEglWindowPanicREglStandAloneWindowNotCreated);
    return iData->iSizeInPixels;
    }


EXPORT_C TSize REglStandAloneWindow::SizeInTwips() const
    {
    EGL_WINDOW_ASSERT_ALWAYS(iData, EEglWindowPanicREglStandAloneWindowNotCreated);
    return iData->iSizeInTwips;
    }


EXPORT_C TInt REglStandAloneWindow::ScreenNumber() const
    {
    EGL_WINDOW_ASSERT_ALWAYS(iData, EEglWindowPanicREglStandAloneWindowNotCreated);
    return iData->iScreenNumber;
    }


EXPORT_C void REglStandAloneWindow::SetBackgroundColor(TRgb aColor, TBool aTriggerRedraw)
    {
    EGL_WINDOW_ASSERT_ALWAYS(iData, EEglWindowPanicREglStandAloneWindowNotCreated);
    iData->iBackgroundColor = aColor;
    iData->iObserver.ColorWasSetForWindow(*this, aTriggerRedraw);
    }


EXPORT_C TInt REglStandAloneWindow::SetBackgroundSurface(const TSurfaceConfiguration& aConfiguration, TBool aTriggerRedraw)
    {
    EGL_WINDOW_ASSERT_ALWAYS(iData, EEglWindowPanicREglStandAloneWindowNotCreated);
    
#ifdef _DEBUG
    //Assert that the surface is not null.
    TSurfaceId surface;
    aConfiguration.GetSurfaceId(surface);
    EGL_WINDOW_ASSERT_DEBUG(!surface.IsNull(), EEglWindowPanicInvalidSurface);
#endif
    
    //Remove the old surface if one exists.
    RemoveBackgroundSurface(EFalse);
    
    //Save the new configuration and alert the observer that a new surface was set.
    TInt err = KErrNone;
    CopySurfaceConfiguration(aConfiguration, iData->iSurfaceConfig);
    err = iData->iObserver.SurfaceWasSetForWindow(*this, aTriggerRedraw);
    
    return err;
    }


EXPORT_C void REglStandAloneWindow::RemoveBackgroundSurface(TBool aTriggerRedraw)
    {
    EGL_WINDOW_ASSERT_ALWAYS(iData, EEglWindowPanicREglStandAloneWindowNotCreated);
    
    //If there is a valid config, alert the observer that it will be removed.
    TSurfaceId surface;
    iData->iSurfaceConfig.GetSurfaceId(surface);
    if(!surface.IsNull())
        {
        iData->iObserver.SurfaceWillBeRemovedForWindow(*this, aTriggerRedraw);
        }
    
    //Zero the configuration.
    CopySurfaceConfiguration(TSurfaceConfiguration(), iData->iSurfaceConfig);
    }

//-------------------------------------------------------------------
