// Copyright (c) 2010 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// This file is intended for use by EGL implementations only. It 
// provides a standardised mechanism for interfacing with window
// objects that are passed to EGL.


/**
 @file
 @publishedPartner
 @prototype
*/


#ifndef __EGLWINDOWINTERFACE_H__
#define __EGLWINDOWINTERFACE_H__


#include <graphics/eglwindowcommon.h>
#include <EGL/egltypes.h>


/**
Forward declarations.
*/
class REglWindowBase;


/**
TEglWindowInterface exposes a proxy interface that forwards actions
to the NativeWindow that the instance is constructed from. The 
interface consists of all the functionality that an EGL 
implementation requires to implement EGL window surfaces.
*/
NONSHARABLE_CLASS(TEglWindowInterface)
    {
public:
    /**
    Constructs a TEglWindowInterface to provide an interface to the
    EGLNativeWindowType handle passed in.

    @param  aNativeWindow   Pointer to the native window. 
    
    @pre    aNativeWindow is either an initialised RWindow or an 
            initialised REglWindowBase derived type.

    @error  If aNativeWindow is not an RWindow or an REglWindowBase 
    derived type, behaviour is undefined.
    */
    IMPORT_C TEglWindowInterface(EGLNativeWindowType aNativeWindow);

    /**
    If the native window that is wrapped by the instance of 
    TEglWindowInterface is valid, this function returns ETrue, else
    EFalse is returned.
    */
    IMPORT_C TBool IsValid() const;
    
    /**
    Returns the size in pixels of the native window.
    */
    IMPORT_C TSize SizeInPixels() const;
    
    /**
    Returns the size in twips of the native window. This can be used
    in conjunction with SizeInPixels() to work out the dot pitch and
    pixel aspect ratio of the EGL surface attached to this window.
    These attributes are queryable by an EGL client.
    */
    IMPORT_C TSize SizeInTwips() const;
    
    /**
    Returns the screen number that the native window is displayed on
    or KAllScreens to indicate that the window must be displayable on 
    all screens in the system. 
    */
    IMPORT_C TInt ScreenNumber() const;
    
    /**
    Sets the window's background color. This is the area of the 
    window not covered by a background surface, either because it
    has no background surface, or because the surface configuration
    specifies that the surface does not cover the whole of the 
    window's area.
    
    @param  aColor          The color that the window's background
                            will be set to.
                            
    @param  aTriggerRedraw  If set causes any affected portions of 
                            the display to be redrawn.
                            
    @note   The appearance on screen may not change until the window
            is redrawn and the next refresh occurs. The 
            aTriggerRedraw parameter can be set to trigger this 
            redrawing.
    */
    IMPORT_C void SetBackgroundColor(TRgb aColor, TBool aTriggerRedraw);
    
    /**
    This sets a surface to appear in front of the native window's 
    background within a given area of that window. The 
    TSurfaceConfiguration object contains the Surface ID and allows 
    various surface presentation attributes to be specified. This 
    describes the mapping from surface co-ordinates to screen 
    co-ordinates, allowing for scaling, cropping, and rotation.
    
    @param  aConfiguration  The set of configuration that applies to
                            the surface. 
    
    @param  aTriggerRedraw  If set causes any affected portions of 
                            the display to be redrawn.  

    @pre    The surface is opaque; otherwise results are not defined. 

    @pre    The surface must not be the null surface ID; otherwise 
            the client is panicked with code ENWPanicInvalidSurface. 

    @note   If the same surface ID is already set as the window 
            background surface, then only the configuration 
            parameters will be updated.

    @note   If the window already has a background surface (that is
            not same as the new surface) then that surface will be 
            removed and the new background surface will be set. The 
            Surface ID will be registered while attached to this 
            window, incrementing the registration count by one.

    @post   The window has a new background surface set to be within 
            the given area. 
     
    @return KErrNone on success or any system-wide error code. 
    */
    IMPORT_C TInt SetBackgroundSurface(const TSurfaceConfiguration &aConfiguration, TBool aTriggerRedraw);
    
    /**
    This removes any background surface that has previously been set 
    to the window. The surface ID registration associated with the 
    window will be released. The aTriggerRedraw parameter triggers a 
    redraw of at least the affected areas of the window at the end of
    this method.

    @param  aTriggerRedraw  If set causes any affected portions of 
                            the display to be redrawn. 

    @post   Any background surface associated with this window has 
            been removed. The appearance on screen may not change 
            until the window is redrawn and the next refresh occurs. 
            The aTriggerRedraw parameter can be set to trigger this 
            redrawing.
    */
    IMPORT_C void RemoveBackgroundSurface(TBool aTriggerRedraw);
    
private:
    REglWindowBase* iEglWindow;
    };


#endif //__EGLWINDOWINTERFACE_H__
