// Copyright (c) 1994-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of the License "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// e32test\buffer\t_collate.cpp
// Overview:
// Test Unicode collations.
// API Information:
// CompareC, TCollationMethod, TCollationKeyTable.
// Details:
// - Check the collation for characters, accents, different cases and standard 
// alphabetical ordering is as expected.
// - Check collation using standard collation method, overriding the collation tables, 
// collation keys is as expected.
// - Check collation for simplified Chinese characters is as expected.
// - Check collation and compare the Chinese ideological characters to other 
// characters which are not included in the Chinese collation table are as expected.
// - Check collation and compare Chinese to English alphabets.
// - Check collation and compare Chinese to Hiragana characters.
// - Check collation and compare Chinese to Katakana characters.
// - Check collation and compare Hiragana to a Katakana characters.
// - Check collation and compare full width and half width digits and letters.
// - Set the current collation key table to standard table, collation keys to default keys,
// constants to specified values, compare data and check it is as expected.
// Platforms/Drives/Compatibility:
// All 
// Assumptions/Requirement/Pre-requisites:
// Failures and causes:
// Base Port information:
// 
//

#include <e32test.h>
#include <collate.h>	  
#include "../../../kernel/eka/euser/unicode/collateimp.h"
#include "u32std.h"

#ifdef __VC32__
    // Solve compilation problem caused by non-English locale
    #pragma setlocale("english")
#endif

LOCAL_D RTest test(_L("T_COLLATE"));


#ifndef _DEBUG
#pragma warning (disable :4710) //Function not expanded 
#pragma warning (disable :4700) //Local variable used without being initialized
#endif

///***************** copied from locale euser source code ***********************
static const TCollationMethod TheCollationMethod[] =
	{
		{
		KUidBasicCollationMethod,				// this is the standard unlocalised method
		NULL,									// null means use the standard table
		NULL,									// there's no override table
		0										// the flags are standard
		}
	};
static const TCollationDataSet TheCollationDataSet =
	{
	TheCollationMethod,
	1
	};
const LCharSet TheCharSet =
	{
	NULL,
	&TheCollationDataSet
	};
const LCharSet* GetLocaleCharSet()
	{
	return &TheCharSet;
	}
///*******************************************************************************




void test_unicode_collation(const TDesC& x,const TDesC& y,TInt desired_order,
							TInt level = 3,const TCollationMethod* method = NULL)
	{
	TInt order = x.CompareC(y,level,method);

	// Normalise order to -1, 0 or 1.
	if (order < 0)
		order = -1;
	else if (order > 0)
		order = 1;

	test(order == desired_order);
	}

#undef USE_SWEDISH_TABLE
#ifdef USE_SWEDISH_TABLE
static const TUint32 TheSwedishKey[] = 
	{
	0x8f60109,0x8f70109,0x8f80109,0x8f60121,0x8f70121,0x8f80121,0x8dd0109,0x8dd0121,
	0x8c50121,0x8c50109,
	};

static const TUint32 TheSwedishIndex[] = 
	{
	0x570008,0x770009,
	};

static const TUint16 TheSwedishStringElement[] = 
	{
	0x2,0x61,0x30a,0x2,0x61,0x308,0x2,0x6f,
	0x308,0x2,0x41,0x30a,0x2,0x41,0x308,0x2,
	0x4f,0x308,0x2,0x75,0x308,0x2,0x55,0x308,
	};

static const TUint32 TheSwedishStringIndex[] = 
	{
	0xc0004,0x90003,0xf0005,0x150007,0x30001,0x0,0x60002,0x120006,
	};

static const TCollationKeyTable TheSwedishTable = 
	{ TheSwedishKey, TheSwedishIndex, 2, TheSwedishStringElement, TheSwedishStringIndex, 8 };

static const TCollationMethod TheSwedishMethod =
	{ 0, NULL, &TheSwedishTable, 0 };
#endif

static const TCollationMethod TheSwapCaseMethod =
	{ 0, NULL, NULL, TCollationMethod::ESwapCase };

static const TUint32 TheKanaKey[] = 
	{
	0x102c013c,0xc949,0x103f013c,0xc949,0x1044013c,0xc949,0x1049013c,0xc949,
	0x104e013c,0xc949,0x1053013c,0xc949,0x1058013c,0xc949,0x105d013c,0xc949,
	0x1062013c,0xc949,0x1067013c,0xc949,0x106c013c,0xc949,0x1071013c,0xc949,
	0x1076013c,0xc949,0x107b013c,0xc949,0x1083013c,0xc949,0x1088013c,0xc949,
	0x109c013c,0xc949,0x10a3013c,0xc949,0x10aa013c,0xc949,0x10b1013c,0xc949,
	0x10b8013c,0xc949,0x109c013c,0xca49,0x10a3013c,0xca49,0x10aa013c,0xca49,
	0x10b1013c,0xca49,0x10b8013c,0xca49,0x10200134,0x10200135,0x10260134,0x10260135,
	0x102c0134,0x102c0135,0x10330134,0x10330135,0x10390134,0x10390135,0x103f0134,0x10200135,
	0x10440134,0x10260135,0x10490134,0x102c0135,0x104e0134,0x10330135,0x10530134,0x10390135,
	0x103f0134,0xc948,0x10200135,0x10440134,0xc948,0x10260135,0x10490134,0xc948,
	0x102c0135,0x104e0134,0xc948,0x10330135,0x10530134,0xc948,0x10390135,0x10580134,
	0x10200135,0x105d0134,0x10260135,0x10620134,0x102c0135,0x10670134,0x10330135,0x106c0134,
	0x10390135,0x10580134,0xc948,0x10200135,0x105d0134,0xc948,0x10260135,0x10620134,
	0xc948,0x102c0135,0x10670134,0xc948,0x10330135,0x106c0134,0xc948,0x10390135,
	0x10710134,0x10200135,0x10760134,0x10260135,0x107b0134,0x102c0135,0x10830134,0x10330135,
	0x10880134,0x10390135,0x10710134,0xc948,0x10200135,0x10760134,0xc948,0x10260135,
	0x107b0134,0xc948,0x102c0135,0x10830134,0xc948,0x10330135,0x10880134,0xc948,
	0x10390135,0x108d0134,0x10200135,0x10900134,0x10260135,0x10930134,0x102c0135,0x10960134,
	0x10330135,0x10990134,0x10390135,0x109c0134,0x10200135,0x10a30134,0x10260135,0x10aa0134,
	0x102c0135,0x10b10134,0x10330135,0x10b80134,0x10390135,0x109c0134,0xc948,0x10200135,
	0x10a30134,0xc948,0x10260135,0x10aa0134,0xc948,0x102c0135,0x10b10134,0xc948,
	0x10330135,0x10b80134,0xc948,0x10390135,0x109c0134,0xca08,0x10200135,0x10a30134,
	0xca08,0x10260135,0x10aa0134,0xca08,0x102c0135,0x10b10134,0xca08,0x10330135,
	0x10b80134,0xca08,0x10390135,0x10bf0134,0x10200135,0x10c20134,0x10260135,0x10c50134,
	0x102c0135,0x10c80134,0x10330135,0x10cb0134,0x10390135,0x10ce0134,0x10200135,0x10d40134,
	0x102c0135,0x10da0134,0x10390135,0x10e00134,0x10200135,0x10e30134,0x10260135,0x10e60134,
	0x102c0135,0x10e90134,0x10330135,0x10ec0134,0x10390135,0x10ef0134,0x10200135,0x103f0144,
	0x10200145,0x10440144,0x10260145,0x10490144,0x102c0145,0x104e0144,0x10330145,0x10530144,
	0x10390145,0x103f0144,0xc948,0x10200145,0x10440144,0xc948,0x10260145,0x10490144,
	0xc948,0x102c0145,0x104e0144,0xc948,0x10330145,0x10530144,0xc948,0x10390145,
	0x10580144,0x10200145,0x105d0144,0x10260145,0x10620144,0x102c0145,0x10670144,0x10330145,
	0x106c0144,0x10390145,0x10580144,0xc948,0x10200145,0x105d0144,0xc948,0x10260145,
	0x10620144,0xc948,0x102c0145,0x10670144,0xc948,0x10330145,0x106c0144,0xc948,
	0x10390145,0x10710144,0x10200145,0x10760144,0x10260145,0x107b0144,0x102c0145,0x10830144,
	0x10330145,0x10880144,0x10390145,0x10710144,0xc948,0x10200145,0x10760144,0xc948,
	0x10260145,0x107b0144,0xc948,0x102c0145,0x10830144,0xc948,0x10330145,0x10880144,
	0xc948,0x10390145,0x108d0144,0x10200145,0x10900144,0x10260145,0x10930144,0x102c0145,
	0x10960144,0x10330145,0x10990144,0x10390145,0x109c0144,0x10200145,0x10a30144,0x10260145,
	0x10aa0144,0x102c0145,0x10b10144,0x10330145,0x10b80144,0x10390145,0x109c0144,0xc948,
	0x10200145,0x10a30144,0xc948,0x10260145,0x10aa0144,0xc948,0x102c0145,0x10b10144,
	0xc948,0x10330145,0x10b80144,0xc948,0x10390145,0x109c0144,0xca08,0x10200145,
	0x10a30144,0xca08,0x10260145,0x10aa0144,0xca08,0x102c0145,0x10b10144,0xca08,
	0x10330145,0x10b80144,0xca08,0x10390145,0x10bf0144,0x10200145,0x10c20144,0x10260145,
	0x10c50144,0x102c0145,0x10c80144,0x10330145,0x10cb0144,0x10390145,0x10ce0144,0x10200145,
	0x10d40144,0x102c0145,0x10da0144,0x10390145,0x10e00144,0x10200145,0x10e30144,0x10260145,
	0x10e60144,0x102c0145,0x10e90144,0x10330145,0x10ec0144,0x10390145,0x10ef0144,0x10200145,
	0x10f40144,0x10260145,0x10f60144,0x10330145,0x10f80144,0x10390145,0x10ef0144,0xc948,
	0x10200145,0x10f40144,0xc948,0x10260145,0x102c0144,0xc948,0x102c0145,0x10f60144,
	0xc948,0x10330145,0x10f80144,0xc948,0x10390145,0x103f013c,0x1020013d,0x1044013c,
	0x1026013d,0x1049013c,0x102c013d,0x104e013c,0x1033013d,0x1053013c,0x1039013d,0x103f013c,
	0xc948,0x1020013d,0x1044013c,0xc948,0x1026013d,0x1049013c,0xc948,0x102c013d,
	0x104e013c,0xc948,0x1033013d,0x1053013c,0xc948,0x1039013d,0x1058013c,0x1020013d,
	0x105d013c,0x1026013d,0x1062013c,0x102c013d,0x1067013c,0x1033013d,0x106c013c,0x1039013d,
	0x1058013c,0xc948,0x1020013d,0x105d013c,0xc948,0x1026013d,0x1062013c,0xc948,
	0x102c013d,0x1067013c,0xc948,0x1033013d,0x106c013c,0xc948,0x1039013d,0x1071013c,
	0x1020013d,0x1076013c,0x1026013d,0x107b013c,0x102c013d,0x1083013c,0x1033013d,0x1088013c,
	0x1039013d,0x1071013c,0xc948,0x1020013d,0x1076013c,0xc948,0x1026013d,0x107b013c,
	0xc948,0x102c013d,0x1083013c,0xc948,0x1033013d,0x1088013c,0xc948,0x1039013d,
	0x108d013c,0x1020013d,0x1090013c,0x1026013d,0x1093013c,0x102c013d,0x1096013c,0x1033013d,
	0x1099013c,0x1039013d,0x109c013c,0x1020013d,0x10a3013c,0x1026013d,0x10aa013c,0x102c013d,
	0x10b1013c,0x1033013d,0x10b8013c,0x1039013d,0x109c013c,0xc948,0x1020013d,0x10a3013c,
	0xc948,0x1026013d,0x10aa013c,0xc948,0x102c013d,0x10b1013c,0xc948,0x1033013d,
	0x10b8013c,0xc948,0x1039013d,0x109c013c,0xca48,0x1020013d,0x10a3013c,0xca48,
	0x1026013d,0x10aa013c,0xca48,0x102c013d,0x10b1013c,0xca48,0x1033013d,0x10b8013c,
	0xca48,0x1039013d,0x10bf013c,0x1020013d,0x10c2013c,0x1026013d,0x10c5013c,0x102c013d,
	0x10c8013c,0x1033013d,0x10cb013c,0x1039013d,0x10ce013c,0x1020013d,0x10d4013c,0x102c013d,
	0x10da013c,0x1039013d,0x10e0013c,0x1020013d,0x10e3013c,0x1026013d,0x10e6013c,0x102c013d,
	0x10e9013c,0x1033013d,0x10ec013c,0x1039013d,0x10ef013c,0x1020013d,0x10ef013c,0xc948,
	0x1020013d,0xca09,0x2010f,0x20113,0x4010b,0x4010f,0x5010b,0x5010f,
	0x8010b,0x8010f,0xa010f,0xa014b,0xc010b,0xc010f,0xe010b,0xe010f,
	0xe013f,0x10010b,0x10010f,0x12010b,0x12010f,0x14c90b,0x14c90f,0x16ca0b,
	0x16ca0f,0x17010b,0x18030f,0x1a060b,0x1a060f,0x1c160f,0x1e0d0b,0x1e0d0f,
	0x20290f,0x22020b,0x22020f,0x24010d,0x28010d,0x2c010d,0x2e010d,0x30010d,
	0x32010d,0x34010d,0x36010b,0x36010f,0x38010f,0x3a010f,0x3c0109,0x3c010d,
	0x3e0109,0x3e010d,0x40010d,0x400109,0x42010d,0x440109,0x44010d,0x46010d,
	0x48010d,0x4a010f,0x4c010f,0x4e010b,0x4f010b,0x50010b,0x50010f,0x52010b,
	0x52010f,0x54010b,0x56010b,0x58010b,0x58010f,0x5a010b,0x5a010f,0x5c010b,
	0x5c010f,0x5e010b,0x5e010f,0x60010b,0x62010b,0x64010b,0x66010b,0x68010f,
	0x68010b,0x6a010f,0x6a010b,0x6c010b,0x6e010b,0x6f010b,0x70010b,0x72010b,
	0x72010f,0x74010b,0x74010f,0x76010f,0x78010f,0x7a010f,0x7c010b,0x7c010f,
	0x80010b,0x80010f,0x82010b,0x82010f,0x84010f,0x86010f,0x88010f,0x8a010f,
	0x8c010b,0x8e010f,0x90010f,0x92010f,0x94010d,0x960109,0x96010d,0x980109,
	0x98010d,0x9a010d,0x9c0109,0x9c010d,0x9e010b,0x9e010f,0x100010b,0x100010f,
	0x102010b,0x102010f,0x104010b,0x104010f,0x106010b,0x106010f,0x107010f,0x108010f,
	0x109010f,0x10a010f,0x10b010f,0x10c010f,0x10d010f,0x10e010f,0x10f010f,0x110010f,
	0x111010f,0x112010f,0x113010f,0x114010f,0x115010f,0x116010f,0x117010f,0x118010f,
	0x119010f,0x11a010f,0x11b010f,0x11c010f,0x11d010f,0x11e010f,0x11f010f,0x120010f,
	0x121010f,0x122010f,0x123010f,0x124010f,0x125010f,0x126010f,0x127010f,0x128010f,
	0x129010f,0x12a010f,0x12b010f,0x12c010f,0x12d010f,0x12e010f,0x12f010f,0x130010f,
	0x131010f,0x132010f,0x133010f,0x134010f,0x135010f,0x136010f,0x137010f,0x138010f,
	0x139010f,0x13b010f,0x13c010d,0x13d010d,0x13e010f,0x13f010f,0x140010f,0x141010f,
	0x142010f,0x143010d,0x144010d,0x145010d,0x146010d,0x147010d,0x148010d,0x149010d,
	0x14a010d,0x14b010d,0x14c010d,0x14d010d,0x14e010d,0x14f010d,0x150010d,0x151010d,
	0x152010d,0x153010d,0x154010d,0x155010d,0x156010d,0x157010d,0x158010d,0x159010d,
	0x15a010d,0x15b010d,0x15c010d,0x15d010d,0x15e010d,0x15f010d,0x160010f,0x161010f,
	0x162010f,0x163010f,0x164010f,0x165010f,0x166010f,0x167010f,0x168010f,0x169010f,
	0x16a010f,0x16b010f,0x16c010f,0x16d010f,0x16e010f,0x16f010f,0x170010f,0x171010f,
	0x172010f,0x174010f,0x175010f,0x176010f,0x177010f,0x178010f,0x179010f,0x17a010d,
	0x17b010d,0x17c010d,0x17d0119,0x17e0119,0x17f0119,0x1800119,0x1810119,0x182010d,
	0x183010d,0x184010d,0x185010d,0x186010d,0x187010d,0x189010f,0x18a010f,0x18b010f,
	0x18c010f,0x18d0111,0x18e0111,0x18f0111,0x1900111,0x1910111,0x1920111,0x1930111,
	0x1940111,0x1950111,0x1960111,0x197010f,0x198010f,0x199010b,0x199010f,0x6c50109,
	0x6c60109,0x6c70109,0x6c80109,0x6c90109,0x6ca0109,0x6cb0109,0x6cc0109,0x6cd0109,
	0x6ce0109,0x6cf0109,0x6cf010d,0x6cf0121,0x6cf0125,0x6e30109,0x6e3010d,0x6e30121,
	0x6e30125,0x6fb0109,0x6fb010d,0x6fb0121,0x6fb0125,0x7070109,0x707010d,0x7070121,
	0x7070125,0x72f0109,0x72f010d,0x72f0121,0x72f0125,0x75b0109,0x75b010d,0x75b0121,
	0x75b0125,0x7630109,0x763010d,0x7630121,0x7630125,0x7830109,0x783010d,0x7830121,
	0x7830125,0x79d0109,0x79d010d,0x79d0121,0x79d0125,0x7b10109,0x7b1010d,0x7b10121,
	0x7b10125,0x7c10109,0x7c1010d,0x7c10121,0x7c10125,0x7cd0109,0x7cd010d,0x7cd0121,
	0x7cd0125,0x7f50109,0x7f5010d,0x7f50121,0x7f50125,0x7fd0109,0x7fd010d,0x7fd0121,
	0x7fd0125,0x8150109,0x815010d,0x8150121,0x8150125,0x82d0109,0x82d010d,0x82d0121,
	0x82d0125,0x8390109,0x839010d,0x8390121,0x8390125,0x8450109,0x845010d,0x8450121,
	0x8450125,0x86d0109,0x86d010d,0x86d0121,0x86d0125,0x8890109,0x889010d,0x8890121,
	0x8890125,0x8ad0109,0x8ad010d,0x8ad0121,0x8ad0125,0x8c50109,0x8c5010d,0x8c50121,
	0x8c50125,0x8d10109,0x8d1010d,0x8d10121,0x8d10125,0x8d90109,0x8d9010d,0x8d90121,
	0x8d90125,0x8dd0109,0x8dd010d,0x8dd0121,0x8dd0125,0x8e90109,0x8e9010d,0x8e90121,
	0x8e90125,0x10200139,0x10200149,0x10200141,0x10260139,0x10260149,0x10260141,0x102c0139,
	0x102c0149,0x102c0141,0x10330139,0x10330149,0x10330141,0x10390139,0x10390149,0x10390141,
	0x103f0135,0x103f0145,0x103f013d,0x10440135,0x10440145,0x1044013d,0x10490135,0x10490145,
	0x1049013d,0x104e0135,0x104e0145,0x104e013d,0x10530135,0x10530145,0x1053013d,0x10580135,
	0x10580145,0x1058013d,0x105d0135,0x105d0145,0x105d013d,0x10620135,0x10620145,0x1062013d,
	0x10670135,0x10670145,0x1067013d,0x106c0135,0x106c0145,0x106c013d,0x10710135,0x10710145,
	0x1071013d,0x10760135,0x10760145,0x1076013d,0x107b0139,0x107b0135,0x107b0149,0x107b0141,
	0x107b0145,0x107b013d,0x10830135,0x10830145,0x1083013d,0x10880135,0x10880145,0x1088013d,
	0x108d0135,0x108d0145,0x108d013d,0x10900135,0x10900145,0x1090013d,0x10930135,0x10930145,
	0x1093013d,0x10960135,0x10960145,0x1096013d,0x10990135,0x10990145,0x1099013d,0x109c0135,
	0x109c0145,0x109c013d,0x10a30135,0x10a30145,0x10a3013d,0x10aa0135,0x10aa0145,0x10aa013d,
	0x10b10135,0x10b10145,0x10b1013d,0x10b80135,0x10b80145,0x10b8013d,0x10bf0135,0x10bf0145,
	0x10bf013d,0x10c20135,0x10c20145,0x10c2013d,0x10c50135,0x10c50145,0x10c5013d,0x10c80135,
	0x10c80145,0x10c8013d,0x10cb0135,0x10cb0145,0x10cb013d,0x10ce0139,0x10ce0135,0x10ce0149,
	0x10ce0141,0x10ce0145,0x10ce013d,0x10d40139,0x10d40135,0x10d40149,0x10d40141,0x10d40145,
	0x10d4013d,0x10da0139,0x10da0135,0x10da0149,0x10da0141,0x10da0145,0x10da013d,0x10e00135,
	0x10e00145,0x10e0013d,0x10e30135,0x10e30145,0x10e3013d,0x10e60135,0x10e60145,0x10e6013d,
	0x10e90135,0x10e90145,0x10e9013d,0x10ec0135,0x10ec0145,0x10ec013d,0x10ef0139,0x10ef0135,
	0x10ef0149,0x10ef0145,0x10ef013d,0x10f40135,0x10f40145,0x10f60135,0x10f60145,0x10f80135,
	0x10f80145,0x10f8013d,0x10fb0135,0x10fb0145,
	};

static const TUint32 TheKanaIndex[] = 
	{
	0x210233,0x22032d,0x230296,0x24028f,0x250294,0x260298,0x270239,0x28025d,
	0x29025f,0x2a029a,0x2b0277,0x2c0226,0x2d0279,0x2e0228,0x2f024e,0x30032f,
	0x310330,0x320331,0x330332,0x340333,0x350334,0x360335,0x370336,0x380337,
	0x390338,0x3a022c,0x3b022e,0x3c0280,0x3d027e,0x3e0282,0x3f0231,0x40029c,
	0x41033b,0x42033f,0x430343,0x440347,0x45034b,0x46034f,0x470353,0x480357,
	0x49035b,0x4a035f,0x4b0363,0x4c0367,0x4d036b,0x4e036f,0x4f0373,0x500377,
	0x51037b,0x52037f,0x530383,0x540387,0x55038b,0x56038f,0x570393,0x580397,
	0x59039b,0x5a039f,0x5b0263,0x5c0250,0x5d0265,0x5e023e,0x5f0241,0x60023b,
	0x610339,0x62033d,0x630341,0x640345,0x650349,0x66034d,0x670351,0x680355,
	0x690359,0x6a035d,0x6b0361,0x6c0365,0x6d0369,0x6e036d,0x6f0371,0x700375,
	0x710379,0x72037d,0x730381,0x740385,0x750389,0x76038d,0x770391,0x780395,
	0x790399,0x7a039d,0x7b0267,0x7c0255,0x7d0269,0x7e0253,0xa20291,0xa30292,
	0xa5028d,0xa7029e,0xa8023d,0xb00289,0xb1027b,0xb4023a,0xb602d7,0xd7027c,
	0xf7027d,0x2010024d,0x2015024c,0x20180259,0x2019025a,0x201c025b,0x201d025c,0x202002d5,
	0x202102d6,0x20250258,0x20260257,0x203002d1,0x2032028a,0x2033028b,0x203b02ac,0x2103028c,
	0x2116030f,0x21210311,0x216002ec,0x216102ed,0x216202ee,0x216302ef,0x216402f0,0x216502f1,
	0x216602f2,0x216702f3,0x216802f4,0x216902f5,0x21700321,0x21710322,0x21720323,0x21730324,
	0x21740325,0x21750326,0x21760327,0x21770328,0x21780329,0x2179032a,0x219002af,0x219102b0,
	0x219202ae,0x219302b1,0x21d202be,0x21d402bf,0x220002c0,0x220202c5,0x220302c1,0x220702c6,
	0x220802b3,0x220b02b4,0x2211031e,0x2212027a,0x221a02cb,0x221d02cd,0x221e0286,0x221f031f,
	0x222002c2,0x22250254,0x222702bb,0x222802bc,0x222902ba,0x222a02b9,0x222b02cf,0x222c02d0,
	0x222e031d,0x22340286,0x223502ce,0x223d02cc,0x225202c8,0x226102c7,0x22660284,0x22670285,
	0x226a02c9,0x226b02ca,0x228202b7,0x228302b8,0x228602b5,0x228702b6,0x22a502c3,0x22bf0320,
	0x231202c4,0x246002d9,0x246102da,0x246202db,0x246302dc,0x246402dd,0x246502de,0x246602df,
	0x246702e0,0x246802e1,0x246902e2,0x246a02e3,0x246b02e3,0x246c02e4,0x246d02e5,0x246e02e6,
	0x246f02e7,0x247002e8,0x247102e9,0x247202ea,0x247302eb,0x25a002a7,0x25a102a6,0x25b202a9,
	0x25b302a8,0x25bc02ab,0x25bd02aa,0x25c602a5,0x25c702a4,0x25cb02a1,0x25ce02a3,0x25cf02a2,
	0x25ef02d8,0x260502a0,0x2606029f,0x26400288,0x26420287,0x266a02d4,0x266d02d3,0x266f02d2,
	0x30010223,0x30020225,0x30030245,0x30050247,0x30060248,0x30070249,0x3008026b,0x3009026c,
	0x300a026d,0x300b026e,0x300c026f,0x300d0271,0x300e0273,0x300f0274,0x30100275,0x30110276,
	0x301202ad,0x301302b2,0x30140261,0x30150262,0x301c0252,0x301d030d,0x301f030e,0x304103a1,
	0x304200d6,0x304303a4,0x304400ce,0x304503a7,0x304600d0,0x304703aa,0x304800d2,0x304903ad,
	0x304a00d4,0x304b03b0,0x304d03b3,0x304f03b6,0x305103b9,0x305303bc,0x305503bf,0x305703c2,
	0x305903c5,0x305b03c8,0x305d03cb,0x305f03ce,0x306103d1,0x306303d4,0x306403d5,0x306603da,
	0x306803dd,0x306a03e0,0x306b03e3,0x306c03e6,0x306d03e9,0x306e03ec,0x306f03ef,0x307203f2,
	0x307503f5,0x307803f8,0x307b03fb,0x307e03fe,0x307f0401,0x30800404,0x30810407,0x3082040a,
	0x3083040d,0x3084040e,0x30850413,0x30860414,0x30870419,0x3088041a,0x3089041f,0x308a0422,
	0x308b0425,0x308c0428,0x308d042b,0x308e042e,0x308f042f,0x30900433,0x30910435,0x30920437,
	0x3093043a,0x30990029,0x309a0221,0x309b0236,0x309c0238,0x309d0244,0x30a103a2,0x30a20178,
	0x30a303a5,0x30a4017b,0x30a503a8,0x30a6017e,0x30a703ab,0x30a80181,0x30a903ae,0x30aa0184,
	0x30ab03b1,0x30ad03b4,0x30af03b7,0x30b103ba,0x30b303bd,0x30b503c0,0x30b703c3,0x30b903c6,
	0x30bb03c9,0x30bd03cc,0x30bf03cf,0x30c103d2,0x30c303d6,0x30c403d8,0x30c603db,0x30c803de,
	0x30ca03e1,0x30cb03e4,0x30cc03e7,0x30cd03ea,0x30ce03ed,0x30cf03f0,0x30d203f3,0x30d503f6,
	0x30d803f9,0x30db03fc,0x30de03ff,0x30df0402,0x30e00405,0x30e10408,0x30e2040b,0x30e3040f,
	0x30e40411,0x30e50415,0x30e60417,0x30e7041b,0x30e8041d,0x30e90420,0x30ea0423,0x30eb0426,
	0x30ec0429,0x30ed042c,0x30ee0430,0x30ef0431,0x30f00434,0x30f10436,0x30f20438,0x30f3043b,
	0x30fb022a,0x30fc024b,0x30fd0243,0x32310317,0x32320318,0x32390319,0x32a40312,0x32a50313,
	0x32a60314,0x32a70315,0x32a80316,0x330302fc,0x330d0300,0x331402f7,0x331802fa,0x332202f8,
	0x33230302,0x33260301,0x332702fb,0x332b0303,0x333602fd,0x333b0305,0x334902f6,0x334a0304,
	0x334d02f9,0x335102fe,0x335702ff,0x337b030c,0x337c031c,0x337d031b,0x337e031a,0x338f0309,
	0x339c0306,0x339d0307,0x339e0308,0x33a1030b,0x33c4030a,0x33cd0310,0x4edd0246,0xfe540230,
	0xff010234,0xff02032e,0xff030297,0xff040290,0xff050295,0xff060299,0xff07032c,0xff08025e,
	0xff090260,0xff0a029b,0xff0b0278,0xff0c0227,0xff0d027a,0xff0e0229,0xff0f024f,0xff1a022d,
	0xff1b022f,0xff1c0281,0xff1d027f,0xff1e0283,0xff1f0232,0xff20029d,0xff21033c,0xff220340,
	0xff230344,0xff240348,0xff25034c,0xff260350,0xff270354,0xff280358,0xff29035c,0xff2a0360,
	0xff2b0364,0xff2c0368,0xff2d036c,0xff2e0370,0xff2f0374,0xff300378,0xff31037c,0xff320380,
	0xff330384,0xff340388,0xff35038c,0xff360390,0xff370394,0xff380398,0xff39039c,0xff3a03a0,
	0xff3b0264,0xff3c0251,0xff3d0266,0xff3e023f,0xff3f0242,0xff40023c,0xff41033a,0xff42033e,
	0xff430342,0xff440346,0xff45034a,0xff46034e,0xff470352,0xff480356,0xff49035a,0xff4a035e,
	0xff4b0362,0xff4c0366,0xff4d036a,0xff4e036e,0xff4f0372,0xff500376,0xff51037a,0xff52037e,
	0xff530382,0xff540386,0xff55038a,0xff56038e,0xff570392,0xff580396,0xff59039a,0xff5a039e,
	0xff5b0268,0xff5c0256,0xff5d026a,0xff5e0252,0xff610224,0xff620270,0xff630272,0xff640222,
	0xff65022b,0xff660439,0xff6703a3,0xff6803a6,0xff6903a9,0xff6a03ac,0xff6b03af,0xff6c0410,
	0xff6d0416,0xff6e041c,0xff6f03d7,0xff70024a,0xff710220,0xff720215,0xff730217,0xff740219,
	0xff75021b,0xff7603b2,0xff7703b5,0xff7803b8,0xff7903bb,0xff7a03be,0xff7b03c1,0xff7c03c4,
	0xff7d03c7,0xff7e03ca,0xff7f03cd,0xff8003d0,0xff8103d3,0xff8203d9,0xff8303dc,0xff8403df,
	0xff8503e2,0xff8603e5,0xff8703e8,0xff8803eb,0xff8903ee,0xff8a03f1,0xff8b03f4,0xff8c03f7,
	0xff8d03fa,0xff8e03fd,0xff8f0400,0xff900403,0xff910406,0xff920409,0xff93040c,0xff940412,
	0xff950418,0xff96041e,0xff970421,0xff980424,0xff990427,0xff9a042a,0xff9b042d,0xff9c0432,
	0xff9e0235,0xff9f0237,0xffe00291,0xffe10293,0xffe202bd,0xffe30240,0xffe4032b,0xffe5028e,
	};

static const TUint16 TheKanaStringElement[] = 
	{
	0x2,0xff73,0xff9e,0x2,0xff76,0xff9e,0x2,0xff77,
	0xff9e,0x2,0xff78,0xff9e,0x2,0xff79,0xff9e,0x2,
	0xff7a,0xff9e,0x2,0xff7b,0xff9e,0x2,0xff7c,0xff9e,
	0x2,0xff7d,0xff9e,0x2,0xff7e,0xff9e,0x2,0xff7f,
	0xff9e,0x2,0xff80,0xff9e,0x2,0xff81,0xff9e,0x2,
	0xff82,0xff9e,0x2,0xff83,0xff9e,0x2,0xff84,0xff9e,
	0x2,0xff8a,0xff9e,0x2,0xff8b,0xff9e,0x2,0xff8c,
	0xff9e,0x2,0xff8d,0xff9e,0x2,0xff8e,0xff9e,0x2,
	0xff8a,0xff9f,0x2,0xff8b,0xff9f,0x2,0xff8c,0xff9f,
	0x2,0xff8d,0xff9f,0x2,0xff8e,0xff9f,0x2,0x3042,
	0x30fc,0x2,0x3044,0x30fc,0x2,0x3046,0x30fc,0x2,
	0x3048,0x30fc,0x2,0x304a,0x30fc,0x2,0x304b,0x30fc,
	0x2,0x304d,0x30fc,0x2,0x304f,0x30fc,0x2,0x3051,
	0x30fc,0x2,0x3053,0x30fc,0x3,0x304b,0x3099,0x30fc,
	0x3,0x304d,0x3099,0x30fc,0x3,0x304f,0x3099,0x30fc,
	0x3,0x3051,0x3099,0x30fc,0x3,0x3053,0x3099,0x30fc,
	0x2,0x3055,0x30fc,0x2,0x3057,0x30fc,0x2,0x3059,
	0x30fc,0x2,0x305b,0x30fc,0x2,0x305d,0x30fc,0x3,
	0x3055,0x3099,0x30fc,0x3,0x3057,0x3099,0x30fc,0x3,
	0x3059,0x3099,0x30fc,0x3,0x305b,0x3099,0x30fc,0x3,
	0x305d,0x3099,0x30fc,0x2,0x305f,0x30fc,0x2,0x3061,
	0x30fc,0x2,0x3064,0x30fc,0x2,0x3066,0x30fc,0x2,
	0x3068,0x30fc,0x3,0x305f,0x3099,0x30fc,0x3,0x3061,
	0x3099,0x30fc,0x3,0x3064,0x3099,0x30fc,0x3,0x3066,
	0x3099,0x30fc,0x3,0x3068,0x3099,0x30fc,0x2,0x306a,
	0x30fc,0x2,0x306b,0x30fc,0x2,0x306c,0x30fc,0x2,
	0x306d,0x30fc,0x2,0x306e,0x30fc,0x2,0x306f,0x30fc,
	0x2,0x3072,0x30fc,0x2,0x3075,0x30fc,0x2,0x3078,
	0x30fc,0x2,0x307b,0x30fc,0x3,0x306f,0x3099,0x30fc,
	0x3,0x3072,0x3099,0x30fc,0x3,0x3075,0x3099,0x30fc,
	0x3,0x3078,0x3099,0x30fc,0x3,0x307b,0x3099,0x30fc,
	0x3,0x306f,0x309a,0x30fc,0x3,0x3072,0x309a,0x30fc,
	0x3,0x3075,0x309a,0x30fc,0x3,0x3078,0x309a,0x30fc,
	0x3,0x307b,0x309a,0x30fc,0x2,0x307e,0x30fc,0x2,
	0x307f,0x30fc,0x2,0x3080,0x30fc,0x2,0x3081,0x30fc,
	0x2,0x3082,0x30fc,0x2,0x3084,0x30fc,0x2,0x3086,
	0x30fc,0x2,0x3088,0x30fc,0x2,0x3089,0x30fc,0x2,
	0x308a,0x30fc,0x2,0x308b,0x30fc,0x2,0x308c,0x30fc,
	0x2,0x308d,0x30fc,0x2,0x308f,0x30fc,0x2,0x30ab,
	0x30fc,0x2,0x30ad,0x30fc,0x2,0x30af,0x30fc,0x2,
	0x30b1,0x30fc,0x2,0x30b3,0x30fc,0x3,0x30ab,0x3099,
	0x30fc,0x3,0x30ad,0x3099,0x30fc,0x3,0x30af,0x3099,
	0x30fc,0x3,0x30b1,0x3099,0x30fc,0x3,0x30b3,0x3099,
	0x30fc,0x2,0x30b5,0x30fc,0x2,0x30b7,0x30fc,0x2,
	0x30b9,0x30fc,0x2,0x30bb,0x30fc,0x2,0x30bd,0x30fc,
	0x3,0x30b5,0x3099,0x30fc,0x3,0x30b7,0x3099,0x30fc,
	0x3,0x30b9,0x3099,0x30fc,0x3,0x30bb,0x3099,0x30fc,
	0x3,0x30bd,0x3099,0x30fc,0x2,0x30bf,0x30fc,0x2,
	0x30c1,0x30fc,0x2,0x30c4,0x30fc,0x2,0x30c6,0x30fc,
	0x2,0x30c8,0x30fc,0x3,0x30bf,0x3099,0x30fc,0x3,
	0x30c1,0x3099,0x30fc,0x3,0x30c4,0x3099,0x30fc,0x3,
	0x30c6,0x3099,0x30fc,0x3,0x30c8,0x3099,0x30fc,0x2,
	0x30ca,0x30fc,0x2,0x30cb,0x30fc,0x2,0x30cc,0x30fc,
	0x2,0x30cd,0x30fc,0x2,0x30ce,0x30fc,0x2,0x30cf,
	0x30fc,0x2,0x30d2,0x30fc,0x2,0x30d5,0x30fc,0x2,
	0x30d8,0x30fc,0x2,0x30db,0x30fc,0x3,0x30cf,0x3099,
	0x30fc,0x3,0x30d2,0x3099,0x30fc,0x3,0x30d5,0x3099,
	0x30fc,0x3,0x30d8,0x3099,0x30fc,0x3,0x30db,0x3099,
	0x30fc,0x3,0x30cf,0x309a,0x30fc,0x3,0x30d2,0x309a,
	0x30fc,0x3,0x30d5,0x309a,0x30fc,0x3,0x30d8,0x309a,
	0x30fc,0x3,0x30db,0x309a,0x30fc,0x2,0x30de,0x30fc,
	0x2,0x30df,0x30fc,0x2,0x30e0,0x30fc,0x2,0x30e1,
	0x30fc,0x2,0x30e2,0x30fc,0x2,0x30e4,0x30fc,0x2,
	0x30e6,0x30fc,0x2,0x30e8,0x30fc,0x2,0x30e9,0x30fc,
	0x2,0x30ea,0x30fc,0x2,0x30eb,0x30fc,0x2,0x30ec,
	0x30fc,0x2,0x30ed,0x30fc,0x2,0x30ef,0x30fc,0x2,
	0x30f0,0x30fc,0x2,0x30f1,0x30fc,0x2,0x30f2,0x30fc,
	0x3,0x30ef,0x3099,0x30fc,0x3,0x30f0,0x3099,0x30fc,
	0x3,0x30ef,0x3099,0x30fc,0x3,0x30f1,0x3099,0x30fc,
	0x3,0x30f2,0x3099,0x30fc,0x2,0xff76,0xff70,0x2,
	0xff77,0xff70,0x2,0xff78,0xff70,0x2,0xff79,0xff70,
	0x2,0xff7a,0xff70,0x3,0xff76,0xff9e,0xff70,0x3,
	0xff77,0xff9e,0xff70,0x3,0xff78,0xff9e,0xff70,0x3,
	0xff79,0xff9e,0xff70,0x3,0xff7a,0xff9e,0xff70,0x2,
	0xff7b,0xff70,0x2,0xff7c,0xff70,0x2,0xff7d,0xff70,
	0x2,0xff7e,0xff70,0x2,0xff7f,0xff70,0x3,0xff7b,
	0xff9e,0xff70,0x3,0xff7c,0xff9e,0xff70,0x3,0xff7d,
	0xff9e,0xff70,0x3,0xff7e,0xff9e,0xff70,0x3,0xff7f,
	0xff9e,0xff70,0x2,0xff80,0xff70,0x2,0xff81,0xff70,
	0x2,0xff82,0xff70,0x2,0xff83,0xff70,0x2,0xff84,
	0xff70,0x3,0xff80,0xff9e,0xff70,0x3,0xff81,0xff9e,
	0xff70,0x3,0xff82,0xff9e,0xff70,0x3,0xff83,0xff9e,
	0xff70,0x3,0xff84,0xff9e,0xff70,0x2,0xff85,0xff70,
	0x2,0xff86,0xff70,0x2,0xff87,0xff70,0x2,0xff88,
	0xff70,0x2,0xff89,0xff70,0x2,0xff8a,0xff70,0x2,
	0xff8b,0xff70,0x2,0xff8c,0xff70,0x2,0xff8d,0xff70,
	0x2,0xff8e,0xff70,0x3,0xff8a,0xff9e,0xff70,0x3,
	0xff8b,0xff9e,0xff70,0x3,0xff8c,0xff9e,0xff70,0x3,
	0xff8d,0xff9e,0xff70,0x3,0xff8e,0xff9e,0xff70,0x3,
	0xff8a,0xff9f,0xff70,0x3,0xff8b,0xff9f,0xff70,0x3,
	0xff8c,0xff9f,0xff70,0x3,0xff8d,0xff9f,0xff70,0x3,
	0xff8e,0xff9f,0xff70,0x2,0xff8f,0xff70,0x2,0xff90,
	0xff70,0x2,0xff91,0xff70,0x2,0xff92,0xff70,0x2,
	0xff93,0xff70,0x2,0xff94,0xff70,0x2,0xff95,0xff70,
	0x2,0xff96,0xff70,0x2,0xff97,0xff70,0x2,0xff98,
	0xff70,0x2,0xff99,0xff70,0x2,0xff9a,0xff70,0x2,
	0xff9b,0xff70,0x2,0xff9c,0xff70,0x3,0xff9c,0xff9e,
	0xff70,
	};

static const TUint32 TheKanaStringIndex[] = 
	{
	0x4e0034,0x510036,0x540038,0x57003a,0x5a003c,0x6c0048,0x5d003e,0x70004b,
	0x600040,0x74004e,0x630042,0x780051,0x660044,0x7c0054,0x690046,0x8f0061,
	0x800057,0x930064,0x830059,0x970067,0x86005b,0x9b006a,0x89005d,0x9f006d,
	0x8c005f,0xb2007a,0xa30070,0xb6007d,0xa60072,0xba0080,0xa90074,0xbe0083,
	0xac0076,0xc20086,0xaf0078,0xc60089,0xc9008b,0xcc008d,0xcf008f,0xd20091,
	0xe4009d,0xf800ac,0xd50093,0xe800a0,0xfc00af,0xd80095,0xec00a3,0x10000b2,
	0xdb0097,0xf000a6,0x10400b5,0xde0099,0xf400a9,0x10800b8,0xe1009b,0x10c00bb,
	0x10f00bd,0x11200bf,0x11500c1,0x11800c3,0x11b00c5,0x11e00c7,0x12100c9,0x12400cb,
	0x12700cd,0x12a00cf,0x12d00d1,0x13000d3,0x13300d5,0x14500e1,0x13600d7,0x14900e4,
	0x13900d9,0x14d00e7,0x13c00db,0x15100ea,0x13f00dd,0x15500ed,0x14200df,0x16800fa,
	0x15900f0,0x16c00fd,0x15c00f2,0x1700100,0x15f00f4,0x1740103,0x16200f6,0x1780106,
	0x16500f8,0x18b0113,0x17c0109,0x18f0116,0x17f010b,0x1930119,0x182010d,0x197011c,
	0x185010f,0x19b011f,0x1880111,0x19f0122,0x1a20124,0x1a50126,0x1a80128,0x1ab012a,
	0x1bd0136,0x1d10145,0x1ae012c,0x1c10139,0x1d50148,0x1b1012e,0x1c5013c,0x1d9014b,
	0x1b40130,0x1c9013f,0x1dd014e,0x1b70132,0x1cd0142,0x1e10151,0x1ba0134,0x1e50154,
	0x1e80156,0x1eb0158,0x1ee015a,0x1f1015c,0x1f4015e,0x1f70160,0x1fa0162,0x1fd0164,
	0x2000166,0x2030168,0x206016a,0x209016c,0x2180176,0x220017c,0x20c016e,0x21c0179,
	0x20f0170,0x224017f,0x2120172,0x2280182,0x2150174,0x0,0x22c0185,0x30002,
	0x23b018f,0x22f0187,0x60004,0x23f0192,0x2320189,0x90006,0x2430195,0x235018b,
	0xc0008,0x2470198,0x238018d,0xf000a,0x24b019b,0x24f019e,0x12000c,0x25e01a8,
	0x25201a0,0x15000e,0x26201ab,0x25501a2,0x180010,0x26601ae,0x25801a4,0x1b0012,
	0x26a01b1,0x25b01a6,0x1e0014,0x26e01b4,0x27201b7,0x210016,0x28101c1,0x27501b9,
	0x240018,0x28501c4,0x27801bb,0x27001a,0x28901c7,0x27b01bd,0x2a001c,0x28d01ca,
	0x27e01bf,0x2d001e,0x29101cd,0x29501d0,0x29801d2,0x29b01d4,0x29e01d6,0x2a101d8,
	0x2a401da,0x300020,0x2b301e4,0x3f002a,0x2c701f3,0x2a701dc,0x330022,0x2b701e7,
	0x42002c,0x2cb01f6,0x2aa01de,0x360024,0x2bb01ea,0x45002e,0x2cf01f9,0x2ad01e0,
	0x390026,0x2bf01ed,0x480030,0x2d301fc,0x2b001e2,0x3c0028,0x2c301f0,0x4b0032,
	0x2d701ff,0x2db0202,0x2de0204,0x2e10206,0x2e40208,0x2e7020a,0x2ea020c,0x2ed020e,
	0x2f00210,0x2f30212,0x2f60214,0x2f90216,0x2fc0218,0x2ff021a,0x302021c,0x305021e,
	};

static const TCollationKeyTable TheKanaTable = 
	{ TheKanaKey, TheKanaIndex, 560, TheKanaStringElement, TheKanaStringIndex, 232 };


static const TCollationMethod TheKanaMethod =
	{ 0, NULL, &TheKanaTable, 0 };

static const TCollationMethod TheSwapKanaMethod =
	{ 0, NULL, &TheKanaTable, TCollationMethod::ESwapKana };

// Simplified Chinese. The data is extracted from loce32\Is_unic_template_chinese_simplified.cpp
static const TUint32 TheChineseKey[] = 
	{
	0x4e01001d,0x4e02001d,0x4e03001d,0x4e04001d,0x4e05001d,0x4e06001d,0x4e07001d,0x4e08001d,
	0x4e09001d,0x4e0a001d,0x4e0b001d,0x4e0c001d,0x4e0d001d,0x4e0e001d,0x4e0f001d,0x4e10001d,
	0x4e11001d,0x4e12001d,0x4e13001d,0x4e14001d,0x4e15001d,0x4e16001d,0x4e17001d,0x4e18001d,
	0x4e19001d,0x4e1a001d,0x4e1b001d,0x4e1c001d,0x4e1d001d,0x4e1e001d,0x4e1f001d,0x4e20001d,
	0x4e21001d,0x4e22001d,0x4e23001d,0x4e24001d,0x4e25001d,0x4e26001d,0x4e27001d,0x4e28001d,
	0x4e29001d,0x4e2a001d,0x4e2b001d,0x4e2c001d,0x4e2d001d,0x4e2e001d,0x4e2f001d,0x4e30001d,
	0x4e31001d,0x4e32001d,0x4e33001d,0x4e34001d,0x4e35001d,0x4e36001d,0x4e37001d,0x4e38001d,
	0x4e39001d,0x4e3a001d,0x4e3b001d,0x4e3c001d,0x4e3d001d,0x4e3e001d,0x4e3f001d,0x4e40001d,
	0x4e41001d,0x4e42001d,0x4e43001d,0x4e44001d,0x4e45001d,0x4e46001d,0x4e47001d,0x4e48001d,
	0x4e49001d,0x4e4a001d,0x4e4b001d,0x4e4c001d,0x4e4d001d,0x4e4e001d,0x4e4f001d,0x4e50001d,
	0x4e51001d,0x4e52001d,0x4e53001d,0x4e54001d,0x4e55001d,0x4e56001d,0x4e57001d,0x4e58001d,
	0x4e59001d,0x4e5a001d,0x4e5b001d,0x4e5c001d,0x4e5d001d,0x4e5e001d,0x4e5f001d,0x4e60001d,
	0x4e61001d,0x4e62001d,0x4e63001d,0x4e64001d,0x4e65001d,0x4e66001d,0x4e67001d,0x4e68001d,
	0x4e69001d,0x4e6a001d,0x4e6b001d,0x4e6c001d,0x4e6d001d,0x4e6e001d,0x4e6f001d,0x4e70001d,
	0x4e71001d,0x4e72001d,0x4e73001d,0x4e74001d,0x4e75001d,0x4e76001d,0x4e77001d,0x4e78001d,
	0x4e79001d,0x4e7a001d,0x4e7b001d,0x4e7c001d,0x4e7d001d,0x4e7e001d,0x4e7f001d,0x4e80001d,
	0x4e81001d,0x4e82001d,0x4e83001d,0x4e84001d,0x4e85001d,0x4e86001d,0x4e87001d,0x4e88001d,
	0x4e89001d,0x4e8a001d,0x4e8b001d,0x4e8c001d,0x4e8d001d,0x4e8e001d,0x4e8f001d,0x4e90001d,
	0x4e91001d,0x4e92001d,0x4e93001d,0x4e94001d,0x4e95001d,0x4e96001d,0x4e97001d,0x4e98001d,
	0x4e99001d,0x4e9a001d,0x4e9b001d,0x4e9c001d,0x4e9d001d,0x4e9e001d,0x4e9f001d,0x4ea0001d,
	0x4ea1001d,0x4ea2001d,0x4ea3001d,0x4ea4001d,0x4ea5001d,0x4ea6001d,0x4ea7001d,0x4ea8001d,
	0x4ea9001d,0x4eaa001d,0x4eab001d,0x4eac001d,0x4ead001d,0x4eae001d,0x4eaf001d,0x4eb0001d,
	0x4eb1001d,0x4eb2001d,0x4eb3001d,0x4eb4001d,0x4eb5001d,0x4eb6001d,0x4eb7001d,0x4eb8001d,
	0x4eb9001d,0x4eba001d,0x4ebb001d,0x4ebc001d,0x4ebd001d,0x4ebe001d,0x4ebf001d,0x4ec0001d,
	0x4ec1001d,0x4ec2001d,0x4ec3001d,0x4ec4001d,0x4ec5001d,0x4ec6001d,0x4ec7001d,0x4ec8001d,
	0x4ec9001d,0x4eca001d,0x4ecb001d,0x4ecc001d,0x4ecd001d,0x4ece001d,0x4ecf001d,0x4ed0001d,
	0x4ed1001d,0x4ed2001d,0x4ed3001d,0x4ed4001d,0x4ed5001d,0x4ed6001d,0x4ed7001d,0x4ed8001d,
	0x4ed9001d,0x4eda001d,0x4edb001d,0x4edc001d,0x4edd001d,0x4ede001d,0x4edf001d,0x4ee0001d,
	0x4ee1001d,0x4ee2001d,0x4ee3001d,0x4ee4001d,0x4ee5001d,0x4ee6001d,0x4ee7001d,0x4ee8001d,
	0x4ee9001d,0x4eea001d,0x4eeb001d,0x4eec001d,0x4eed001d,0x4eee001d,0x4eef001d,0x4ef0001d,
	0x4ef1001d,0x4ef2001d,0x4ef3001d,0x4ef4001d,0x4ef5001d,0x4ef6001d,0x4ef7001d,0x4ef8001d,
	0x4ef9001d,0x4efa001d,0x4efb001d,0x4efc001d,0x4efd001d,0x4efe001d,0x4eff001d,0x4f00001d,
	0x4f01001d,0x4f02001d,0x4f03001d,0x4f04001d,0x4f05001d,0x4f06001d,0x4f07001d,0x4f08001d,
	0x4f09001d,0x4f0a001d,0x4f0b001d,0x4f0c001d,0x4f0d001d,0x4f0e001d,0x4f0f001d,0x4f10001d,
	0x4f11001d,0x4f12001d,0x4f13001d,0x4f14001d,0x4f15001d,0x4f16001d,0x4f17001d,0x4f18001d,
	0x4f19001d,0x4f1a001d,0x4f1b001d,0x4f1c001d,0x4f1d001d,0x4f1e001d,0x4f1f001d,0x4f20001d,
	0x4f21001d,0x4f22001d,0x4f23001d,0x4f24001d,0x4f25001d,0x4f26001d,0x4f27001d,0x4f28001d,
	0x4f29001d,0x4f2a001d,0x4f2b001d,0x4f2c001d,0x4f2d001d,0x4f2e001d,0x4f2f001d,0x4f30001d,
	0x4f31001d,0x4f32001d,0x4f33001d,0x4f34001d,0x4f35001d,0x4f36001d,0x4f37001d,0x4f38001d,
	0x4f39001d,0x4f3a001d,0x4f3b001d,0x4f3c001d,0x4f3d001d,0x4f3e001d,0x4f3f001d,0x4f40001d,
	0x4f41001d,0x4f42001d,0x4f43001d,0x4f44001d,0x4f45001d,0x4f46001d,0x4f47001d,0x4f48001d,
	0x4f49001d,0x4f4a001d,0x4f4b001d,0x4f4c001d,0x4f4d001d,0x4f4e001d,0x4f4f001d,0x4f50001d,
	0x4f51001d,0x4f52001d,0x4f53001d,0x4f54001d,0x4f55001d,0x4f56001d,0x4f57001d,0x4f58001d,
	0x4f59001d,0x4f5a001d,0x4f5b001d,0x4f5c001d,0x4f5d001d,0x4f5e001d,0x4f5f001d,0x4f60001d,
	0x4f61001d,0x4f62001d,0x4f63001d,0x4f64001d,0x4f65001d,0x4f66001d,0x4f67001d,0x4f68001d,
	0x4f69001d,0x4f6a001d,0x4f6b001d,0x4f6c001d,0x4f6d001d,0x4f6e001d,0x4f6f001d,0x4f70001d,
	0x4f71001d,0x4f72001d,0x4f73001d,0x4f74001d,0x4f75001d,0x4f76001d,0x4f77001d,0x4f78001d,
	0x4f79001d,0x4f7a001d,0x4f7b001d,0x4f7c001d,0x4f7d001d,0x4f7e001d,0x4f7f001d,0x4f80001d,
	0x4f81001d,0x4f82001d,0x4f83001d,0x4f84001d,0x4f85001d,0x4f86001d,0x4f87001d,0x4f88001d,
	0x4f89001d,0x4f8a001d,0x4f8b001d,0x4f8c001d,0x4f8d001d,0x4f8e001d,0x4f8f001d,0x4f90001d,
	0x4f91001d,0x4f92001d,0x4f93001d,0x4f94001d,0x4f95001d,0x4f96001d,0x4f97001d,0x4f98001d,
	0x4f99001d,0x4f9a001d,0x4f9b001d,0x4f9c001d,0x4f9d001d,0x4f9e001d,0x4f9f001d,0x4fa0001d,
	0x4fa1001d,0x4fa2001d,0x4fa3001d,0x4fa4001d,0x4fa5001d,0x4fa6001d,0x4fa7001d,0x4fa8001d,
	0x4fa9001d,0x4faa001d,0x4fab001d,0x4fac001d,0x4fad001d,0x4fae001d,0x4faf001d,0x4fb0001d,
	0x4fb1001d,0x4fb2001d,0x4fb3001d,0x4fb4001d,0x4fb5001d,0x4fb6001d,0x4fb7001d,0x4fb8001d,
	0x4fb9001d,0x4fba001d,0x4fbb001d,0x4fbc001d,0x4fbd001d,0x4fbe001d,0x4fbf001d,0x4fc0001d,
	0x4fc1001d,0x4fc2001d,0x4fc3001d,0x4fc4001d,0x4fc5001d,0x4fc6001d,0x4fc7001d,0x4fc8001d,
	0x4fc9001d,0x4fca001d,0x4fcb001d,0x4fcc001d,0x4fcd001d,0x4fce001d,0x4fcf001d,0x4fd0001d,
	0x4fd1001d,0x4fd2001d,0x4fd3001d,0x4fd4001d,0x4fd5001d,0x4fd6001d,0x4fd7001d,0x4fd8001d,
	0x4fd9001d,0x4fda001d,0x4fdb001d,0x4fdc001d,0x4fdd001d,0x4fde001d,0x4fdf001d,0x4fe0001d,
	0x4fe1001d,0x4fe2001d,0x4fe3001d,0x4fe4001d,0x4fe5001d,0x4fe6001d,0x4fe7001d,0x4fe8001d,
	0x4fe9001d,0x4fea001d,0x4feb001d,0x4fec001d,0x4fed001d,0x4fee001d,0x4fef001d,0x4ff0001d,
	0x4ff1001d,0x4ff2001d,0x4ff3001d,0x4ff4001d,0x4ff5001d,0x4ff6001d,0x4ff7001d,0x4ff8001d,
	0x4ff9001d,0x4ffa001d,0x4ffb001d,0x4ffc001d,0x4ffd001d,0x4ffe001d,0x4fff001d,0x5000001d,
	0x5001001d,0x5002001d,0x5003001d,0x5004001d,0x5005001d,0x5006001d,0x5007001d,0x5008001d,
	0x5009001d,0x500a001d,0x500b001d,0x500c001d,0x500d001d,0x500e001d,0x500f001d,0x5010001d,
	0x5011001d,0x5012001d,0x5013001d,0x5014001d,0x5015001d,0x5016001d,0x5017001d,0x5018001d,
	0x5019001d,0x501a001d,0x501b001d,0x501c001d,0x501d001d,0x501e001d,0x501f001d,0x5020001d,
	0x5021001d,0x5022001d,0x5023001d,0x5024001d,0x5025001d,0x5026001d,0x5027001d,0x5028001d,
	0x5029001d,0x502a001d,0x502b001d,0x502c001d,0x502d001d,0x502e001d,0x502f001d,0x5030001d,
	0x5031001d,0x5032001d,0x5033001d,0x5034001d,0x5035001d,0x5036001d,0x5037001d,0x5038001d,
	0x5039001d,0x503a001d,0x503b001d,0x503c001d,0x503d001d,0x503e001d,0x503f001d,0x5040001d,
	0x5041001d,0x5042001d,0x5043001d,0x5044001d,0x5045001d,0x5046001d,0x5047001d,0x5048001d,
	0x5049001d,0x504a001d,0x504b001d,0x504c001d,0x504d001d,0x504e001d,0x504f001d,0x5050001d,
	0x5051001d,0x5052001d,0x5053001d,0x5054001d,0x5055001d,0x5056001d,0x5057001d,0x5058001d,
	0x5059001d,0x505a001d,0x505b001d,0x505c001d,0x505d001d,0x505e001d,0x505f001d,0x5060001d,
	0x5061001d,0x5062001d,0x5063001d,0x5064001d,0x5065001d,0x5066001d,0x5067001d,0x5068001d,
	0x5069001d,0x506a001d,0x506b001d,0x506c001d,0x506d001d,0x506e001d,0x506f001d,0x5070001d,
	0x5071001d,0x5072001d,0x5073001d,0x5074001d,0x5075001d,0x5076001d,0x5077001d,0x5078001d,
	0x5079001d,0x507a001d,0x507b001d,0x507c001d,0x507d001d,0x507e001d,0x507f001d,0x5080001d,
	0x5081001d,0x5082001d,0x5083001d,0x5084001d,0x5085001d,0x5086001d,0x5087001d,0x5088001d,
	0x5089001d,0x508a001d,0x508b001d,0x508c001d,0x508d001d,0x508e001d,0x508f001d,0x5090001d,
	0x5091001d,0x5092001d,0x5093001d,0x5094001d,0x5095001d,0x5096001d,0x5097001d,0x5098001d,
	0x5099001d,0x509a001d,0x509b001d,0x509c001d,0x509d001d,0x509e001d,0x509f001d,0x50a0001d,
	0x50a1001d,0x50a2001d,0x50a3001d,0x50a4001d,0x50a5001d,0x50a6001d,0x50a7001d,0x50a8001d,
	0x50a9001d,0x50aa001d,0x50ab001d,0x50ac001d,0x50ad001d,0x50ae001d,0x50af001d,0x50b0001d,
	0x50b1001d,0x50b2001d,0x50b3001d,0x50b4001d,0x50b5001d,0x50b6001d,0x50b7001d,0x50b8001d,
	0x50b9001d,0x50ba001d,0x50bb001d,0x50bc001d,0x50bd001d,0x50be001d,0x50bf001d,0x50c0001d,
	0x50c1001d,0x50c2001d,0x50c3001d,0x50c4001d,0x50c5001d,0x50c6001d,0x50c7001d,0x50c8001d,
	0x50c9001d,0x50ca001d,0x50cb001d,0x50cc001d,0x50cd001d,0x50ce001d,0x50cf001d,0x50d0001d,
	0x50d1001d,0x50d2001d,0x50d3001d,0x50d4001d,0x50d5001d,0x50d6001d,0x50d7001d,0x50d8001d,
	0x50d9001d,0x50da001d,0x50db001d,0x50dc001d,0x50dd001d,0x50de001d,0x50df001d,0x50e0001d,
	0x50e1001d,0x50e2001d,0x50e3001d,0x50e4001d,0x50e5001d,0x50e6001d,0x50e7001d,0x50e8001d,
	0x50e9001d,0x50ea001d,0x50eb001d,0x50ec001d,0x50ed001d,0x50ee001d,0x50ef001d,0x50f0001d,
	0x50f1001d,0x50f2001d,0x50f3001d,0x50f4001d,0x50f5001d,0x50f6001d,0x50f7001d,0x50f8001d,
	0x50f9001d,0x50fa001d,0x50fb001d,0x50fc001d,0x50fd001d,0x50fe001d,0x50ff001d,0x5100001d,
	0x5101001d,0x5102001d,0x5103001d,0x5104001d,0x5105001d,0x5106001d,0x5107001d,0x5108001d,
	0x5109001d,0x510a001d,0x510b001d,0x510c001d,0x510d001d,0x510e001d,0x510f001d,0x5110001d,
	0x5111001d,0x5112001d,0x5113001d,0x5114001d,0x5115001d,0x5116001d,0x5117001d,0x5118001d,
	0x5119001d,0x511a001d,0x511b001d,0x511c001d,0x511d001d,0x511e001d,0x511f001d,0x5120001d,
	0x5121001d,0x5122001d,0x5123001d,0x5124001d,0x5125001d,0x5126001d,0x5127001d,0x5128001d,
	0x5129001d,0x512a001d,0x512b001d,0x512c001d,0x512d001d,0x512e001d,0x512f001d,0x5130001d,
	0x5131001d,0x5132001d,0x5133001d,0x5134001d,0x5135001d,0x5136001d,0x5137001d,0x5138001d,
	0x5139001d,0x513a001d,0x513b001d,0x513c001d,0x513d001d,0x513e001d,0x513f001d,0x5140001d,
	0x5141001d,0x5142001d,0x5143001d,0x5144001d,0x5145001d,0x5146001d,0x5147001d,0x5148001d,
	0x5149001d,0x514a001d,0x514b001d,0x514c001d,0x514d001d,0x514e001d,0x514f001d,0x5150001d,
	0x5151001d,0x5152001d,0x5153001d,0x5154001d,0x5155001d,0x5156001d,0x5157001d,0x5158001d,
	0x5159001d,0x515a001d,0x515b001d,0x515c001d,0x515d001d,0x515e001d,0x515f001d,0x5160001d,
	0x5161001d,0x5162001d,0x5163001d,0x5164001d,0x5165001d,0x5166001d,0x5167001d,0x5168001d,
	0x5169001d,0x516a001d,0x516b001d,0x516c001d,0x516d001d,0x516e001d,0x516f001d,0x5170001d,
	0x5171001d,0x5172001d,0x5173001d,0x5174001d,0x5175001d,0x5176001d,0x5177001d,0x5178001d,
	0x5179001d,0x517a001d,0x517b001d,0x517c001d,0x517d001d,0x517e001d,0x517f001d,0x5180001d,
	0x5181001d,0x5182001d,0x5183001d,0x5184001d,0x5185001d,0x5186001d,0x5187001d,0x5188001d,
	0x5189001d,0x518a001d,0x518b001d,0x518c001d,0x518d001d,0x518e001d,0x518f001d,0x5190001d,
	0x5191001d,0x5192001d,0x5193001d,0x5194001d,0x5195001d,0x5196001d,0x5197001d,0x5198001d,
	0x5199001d,0x519a001d,0x519b001d,0x519c001d,0x519d001d,0x519e001d,0x519f001d,0x51a0001d,
	0x51a1001d,0x51a2001d,0x51a3001d,0x51a4001d,0x51a5001d,0x51a6001d,0x51a7001d,0x51a8001d,
	0x51a9001d,0x51aa001d,0x51ab001d,0x51ac001d,0x51ad001d,0x51ae001d,0x51af001d,0x51b0001d,
	0x51b1001d,0x51b2001d,0x51b3001d,0x51b4001d,0x51b5001d,0x51b6001d,0x51b7001d,0x51b8001d,
	0x51b9001d,0x51ba001d,0x51bb001d,0x51bc001d,0x51bd001d,0x51be001d,0x51bf001d,0x51c0001d,
	0x51c1001d,0x51c2001d,0x51c3001d,0x51c4001d,0x51c5001d,0x51c6001d,0x51c7001d,0x51c8001d,
	0x51c9001d,0x51ca001d,0x51cb001d,0x51cc001d,0x51cd001d,0x51ce001d,0x51cf001d,0x51d0001d,
	0x51d1001d,0x51d2001d,0x51d3001d,0x51d4001d,0x51d5001d,0x51d6001d,0x51d7001d,0x51d8001d,
	0x51d9001d,0x51da001d,0x51db001d,0x51dc001d,0x51dd001d,0x51de001d,0x51df001d,0x51e0001d,
	0x51e1001d,0x51e2001d,0x51e3001d,0x51e4001d,0x51e5001d,0x51e6001d,0x51e7001d,0x51e8001d,
	0x51e9001d,0x51ea001d,0x51eb001d,0x51ec001d,0x51ed001d,0x51ee001d,0x51ef001d,0x51f0001d,
	0x51f1001d,0x51f2001d,0x51f3001d,0x51f4001d,0x51f5001d,0x51f6001d,0x51f7001d,0x51f8001d,
	0x51f9001d,0x51fa001d,0x51fb001d,0x51fc001d,0x51fd001d,0x51fe001d,0x51ff001d,0x5200001d,
	0x5201001d,0x5202001d,0x5203001d,0x5204001d,0x5205001d,0x5206001d,0x5207001d,0x5208001d,
	0x5209001d,0x520a001d,0x520b001d,0x520c001d,0x520d001d,0x520e001d,0x520f001d,0x5210001d,
	0x5211001d,0x5212001d,0x5213001d,0x5214001d,0x5215001d,0x5216001d,0x5217001d,0x5218001d,
	0x5219001d,0x521a001d,0x521b001d,0x521c001d,0x521d001d,0x521e001d,0x521f001d,0x5220001d,
	0x5221001d,0x5222001d,0x5223001d,0x5224001d,0x5225001d,0x5226001d,0x5227001d,0x5228001d,
	0x5229001d,0x522a001d,0x522b001d,0x522c001d,0x522d001d,0x522e001d,0x522f001d,0x5230001d,
	0x5231001d,0x5232001d,0x5233001d,0x5234001d,0x5235001d,0x5236001d,0x5237001d,0x5238001d,
	0x5239001d,0x523a001d,0x523b001d,0x523c001d,0x523d001d,0x523e001d,0x523f001d,0x5240001d,
	0x5241001d,0x5242001d,0x5243001d,0x5244001d,0x5245001d,0x5246001d,0x5247001d,0x5248001d,
	0x5249001d,0x524a001d,0x524b001d,0x524c001d,0x524d001d,0x524e001d,0x524f001d,0x5250001d,
	0x5251001d,0x5252001d,0x5253001d,0x5254001d,0x5255001d,0x5256001d,0x5257001d,0x5258001d,
	0x5259001d,0x525a001d,0x525b001d,0x525c001d,0x525d001d,0x525e001d,0x525f001d,0x5260001d,
	0x5261001d,0x5262001d,0x5263001d,0x5264001d,0x5265001d,0x5266001d,0x5267001d,0x5268001d,
	0x5269001d,0x526a001d,0x526b001d,0x526c001d,0x526d001d,0x526e001d,0x526f001d,0x5270001d,
	0x5271001d,0x5272001d,0x5273001d,0x5274001d,0x5275001d,0x5276001d,0x5277001d,0x5278001d,
	0x5279001d,0x527a001d,0x527b001d,0x527c001d,0x527d001d,0x527e001d,0x527f001d,0x5280001d,
	0x5281001d,0x5282001d,0x5283001d,0x5284001d,0x5285001d,0x5286001d,0x5287001d,0x5288001d,
	0x5289001d,0x528a001d,0x528b001d,0x528c001d,0x528d001d,0x528e001d,0x528f001d,0x5290001d,
	0x5291001d,0x5292001d,0x5293001d,0x5294001d,0x5295001d,0x5296001d,0x5297001d,0x5298001d,
	0x5299001d,0x529a001d,0x529b001d,0x529c001d,0x529d001d,0x529e001d,0x529f001d,0x52a0001d,
	0x52a1001d,0x52a2001d,0x52a3001d,0x52a4001d,0x52a5001d,0x52a6001d,0x52a7001d,0x52a8001d,
	0x52a9001d,0x52aa001d,0x52ab001d,0x52ac001d,0x52ad001d,0x52ae001d,0x52af001d,0x52b0001d,
	0x52b1001d,0x52b2001d,0x52b3001d,0x52b4001d,0x52b5001d,0x52b6001d,0x52b7001d,0x52b8001d,
	0x52b9001d,0x52ba001d,0x52bb001d,0x52bc001d,0x52bd001d,0x52be001d,0x52bf001d,0x52c0001d,
	0x52c1001d,0x52c2001d,0x52c3001d,0x52c4001d,0x52c5001d,0x52c6001d,0x52c7001d,0x52c8001d,
	0x52c9001d,0x52ca001d,0x52cb001d,0x52cc001d,0x52cd001d,0x52ce001d,0x52cf001d,0x52d0001d,
	0x52d1001d,0x52d2001d,0x52d3001d,0x52d4001d,0x52d5001d,0x52d6001d,0x52d7001d,0x52d8001d,
	0x52d9001d,0x52da001d,0x52db001d,0x52dc001d,0x52dd001d,0x52de001d,0x52df001d,0x52e0001d,
	0x52e1001d,0x52e2001d,0x52e3001d,0x52e4001d,0x52e5001d,0x52e6001d,0x52e7001d,0x52e8001d,
	0x52e9001d,0x52ea001d,0x52eb001d,0x52ec001d,0x52ed001d,0x52ee001d,0x52ef001d,0x52f0001d,
	0x52f1001d,0x52f2001d,0x52f3001d,0x52f4001d,0x52f5001d,0x52f6001d,0x52f7001d,0x52f8001d,
	0x52f9001d,0x52fa001d,0x52fb001d,0x52fc001d,0x52fd001d,0x52fe001d,0x52ff001d,0x5300001d,
	0x5301001d,0x5302001d,0x5303001d,0x5304001d,0x5305001d,0x5306001d,0x5307001d,0x5308001d,
	0x5309001d,0x530a001d,0x530b001d,0x530c001d,0x530d001d,0x530e001d,0x530f001d,0x5310001d,
	0x5311001d,0x5312001d,0x5313001d,0x5314001d,0x5315001d,0x5316001d,0x5317001d,0x5318001d,
	0x5319001d,0x531a001d,0x531b001d,0x531c001d,0x531d001d,0x531e001d,0x531f001d,0x5320001d,
	0x5321001d,0x5322001d,0x5323001d,0x5324001d,0x5325001d,0x5326001d,0x5327001d,0x5328001d,
	0x5329001d,0x532a001d,0x532b001d,0x532c001d,0x532d001d,0x532e001d,0x532f001d,0x5330001d,
	0x5331001d,0x5332001d,0x5333001d,0x5334001d,0x5335001d,0x5336001d,0x5337001d,0x5338001d,
	0x5339001d,0x533a001d,0x533b001d,0x533c001d,0x533d001d,0x533e001d,0x533f001d,0x5340001d,
	0x5341001d,0x5342001d,0x5343001d,0x5344001d,0x5345001d,0x5346001d,0x5347001d,0x5348001d,
	0x5349001d,0x534a001d,0x534b001d,0x534c001d,0x534d001d,0x534e001d,0x534f001d,0x5350001d,
	0x5351001d,0x5352001d,0x5353001d,0x5354001d,0x5355001d,0x5356001d,0x5357001d,0x5358001d,
	0x5359001d,0x535a001d,0x535b001d,0x535c001d,0x535d001d,0x535e001d,0x535f001d,0x5360001d,
	0x5361001d,0x5362001d,0x5363001d,0x5364001d,0x5365001d,0x5366001d,0x5367001d,0x5368001d,
	0x5369001d,0x536a001d,0x536b001d,0x536c001d,0x536d001d,0x536e001d,0x536f001d,0x5370001d,
	0x5371001d,0x5372001d,0x5373001d,0x5374001d,0x5375001d,0x5376001d,0x5377001d,0x5378001d,
	0x5379001d,0x537a001d,0x537b001d,0x537c001d,0x537d001d,0x537e001d,0x537f001d,0x5380001d,
	0x5381001d,0x5382001d,0x5383001d,0x5384001d,0x5385001d,0x5386001d,0x5387001d,0x5388001d,
	0x5389001d,0x538a001d,0x538b001d,0x538c001d,0x538d001d,0x538e001d,0x538f001d,0x5390001d,
	0x5391001d,0x5392001d,0x5393001d,0x5394001d,0x5395001d,0x5396001d,0x5397001d,0x5398001d,
	0x5399001d,0x539a001d,0x539b001d,0x539c001d,0x539d001d,0x539e001d,0x539f001d,0x53a0001d,
	0x53a1001d,0x53a2001d,0x53a3001d,0x53a4001d,0x53a5001d,0x53a6001d,0x53a7001d,0x53a8001d,
	0x53a9001d,0x53aa001d,0x53ab001d,0x53ac001d,0x53ad001d,0x53ae001d,0x53af001d,0x53b0001d,
	0x53b1001d,0x53b2001d,0x53b3001d,0x53b4001d,0x53b5001d,0x53b6001d,0x53b7001d,0x53b8001d,
	0x53b9001d,0x53ba001d,0x53bb001d,0x53bc001d,0x53bd001d,0x53be001d,0x53bf001d,0x53c0001d,
	0x53c1001d,0x53c2001d,0x53c3001d,0x53c4001d,0x53c5001d,0x53c6001d,0x53c7001d,0x53c8001d,
	0x53c9001d,0x53ca001d,0x53cb001d,0x53cc001d,0x53cd001d,0x53ce001d,0x53cf001d,0x53d0001d,
	0x53d1001d,0x53d2001d,0x53d3001d,0x53d4001d,0x53d5001d,0x53d6001d,0x53d7001d,0x53d8001d,
	0x53d9001d,0x53da001d,0x53db001d,0x53dc001d,0x53dd001d,0x53de001d,0x53df001d,0x53e0001d,
	0x53e1001d,0x53e2001d,0x53e3001d,0x53e4001d,0x53e5001d,0x53e6001d,0x53e7001d,0x53e8001d,
	0x53e9001d,0x53ea001d,0x53eb001d,0x53ec001d,0x53ed001d,0x53ee001d,0x53ef001d,0x53f0001d,
	0x53f1001d,0x53f2001d,0x53f3001d,0x53f4001d,0x53f5001d,0x53f6001d,0x53f7001d,0x53f8001d,
	0x53f9001d,0x53fa001d,0x53fb001d,0x53fc001d,0x53fd001d,0x53fe001d,0x53ff001d,0x5400001d,
	0x5401001d,0x5402001d,0x5403001d,0x5404001d,0x5405001d,0x5406001d,0x5407001d,0x5408001d,
	0x5409001d,0x540a001d,0x540b001d,0x540c001d,0x540d001d,0x540e001d,0x540f001d,0x5410001d,
	0x5411001d,0x5412001d,0x5413001d,0x5414001d,0x5415001d,0x5416001d,0x5417001d,0x5418001d,
	0x5419001d,0x541a001d,0x541b001d,0x541c001d,0x541d001d,0x541e001d,0x541f001d,0x5420001d,
	0x5421001d,0x5422001d,0x5423001d,0x5424001d,0x5425001d,0x5426001d,0x5427001d,0x5428001d,
	0x5429001d,0x542a001d,0x542b001d,0x542c001d,0x542d001d,0x542e001d,0x542f001d,0x5430001d,
	0x5431001d,0x5432001d,0x5433001d,0x5434001d,0x5435001d,0x5436001d,0x5437001d,0x5438001d,
	0x5439001d,0x543a001d,0x543b001d,0x543c001d,0x543d001d,0x543e001d,0x543f001d,0x5440001d,
	0x5441001d,0x5442001d,0x5443001d,0x5444001d,0x5445001d,0x5446001d,0x5447001d,0x5448001d,
	0x5449001d,0x544a001d,0x544b001d,0x544c001d,0x544d001d,0x544e001d,0x544f001d,0x5450001d,
	0x5451001d,0x5452001d,0x5453001d,0x5454001d,0x5455001d,0x5456001d,0x5457001d,0x5458001d,
	0x5459001d,0x545a001d,0x545b001d,0x545c001d,0x545d001d,0x545e001d,0x545f001d,0x5460001d,
	0x5461001d,0x5462001d,0x5463001d,0x5464001d,0x5465001d,0x5466001d,0x5467001d,0x5468001d,
	0x5469001d,0x546a001d,0x546b001d,0x546c001d,0x546d001d,0x546e001d,0x546f001d,0x5470001d,
	0x5471001d,0x5472001d,0x5473001d,0x5474001d,0x5475001d,0x5476001d,0x5477001d,0x5478001d,
	0x5479001d,0x547a001d,0x547b001d,0x547c001d,0x547d001d,0x547e001d,0x547f001d,0x5480001d,
	0x5481001d,0x5482001d,0x5483001d,0x5484001d,0x5485001d,0x5486001d,0x5487001d,0x5488001d,
	0x5489001d,0x548a001d,0x548b001d,0x548c001d,0x548d001d,0x548e001d,0x548f001d,0x5490001d,
	0x5491001d,0x5492001d,0x5493001d,0x5494001d,0x5495001d,0x5496001d,0x5497001d,0x5498001d,
	0x5499001d,0x549a001d,0x549b001d,0x549c001d,0x549d001d,0x549e001d,0x549f001d,0x54a0001d,
	0x54a1001d,0x54a2001d,0x54a3001d,0x54a4001d,0x54a5001d,0x54a6001d,0x54a7001d,0x54a8001d,
	0x54a9001d,0x54aa001d,0x54ab001d,0x54ac001d,0x54ad001d,0x54ae001d,0x54af001d,0x54b0001d,
	0x54b1001d,0x54b2001d,0x54b3001d,0x54b4001d,0x54b5001d,0x54b6001d,0x54b7001d,0x54b8001d,
	0x54b9001d,0x54ba001d,0x54bb001d,0x54bc001d,0x54bd001d,0x54be001d,0x54bf001d,0x54c0001d,
	0x54c1001d,0x54c2001d,0x54c3001d,0x54c4001d,0x54c5001d,0x54c6001d,0x54c7001d,0x54c8001d,
	0x54c9001d,0x54ca001d,0x54cb001d,0x54cc001d,0x54cd001d,0x54ce001d,0x54cf001d,0x54d0001d,
	0x54d1001d,0x54d2001d,0x54d3001d,0x54d4001d,0x54d5001d,0x54d6001d,0x54d7001d,0x54d8001d,
	0x54d9001d,0x54da001d,0x54db001d,0x54dc001d,0x54dd001d,0x54de001d,0x54df001d,0x54e0001d,
	0x54e1001d,0x54e2001d,0x54e3001d,0x54e4001d,0x54e5001d,0x54e6001d,0x54e7001d,0x54e8001d,
	0x54e9001d,0x54ea001d,0x54eb001d,0x54ec001d,0x54ed001d,0x54ee001d,0x54ef001d,0x54f0001d,
	0x54f1001d,0x54f2001d,0x54f3001d,0x54f4001d,0x54f5001d,0x54f6001d,0x54f7001d,0x54f8001d,
	0x54f9001d,0x54fa001d,0x54fb001d,0x54fc001d,0x54fd001d,0x54fe001d,0x54ff001d,0x5500001d,
	0x5501001d,0x5502001d,0x5503001d,0x5504001d,0x5505001d,0x5506001d,0x5507001d,0x5508001d,
	0x5509001d,0x550a001d,0x550b001d,0x550c001d,0x550d001d,0x550e001d,0x550f001d,0x5510001d,
	0x5511001d,0x5512001d,0x5513001d,0x5514001d,0x5515001d,0x5516001d,0x5517001d,0x5518001d,
	0x5519001d,0x551a001d,0x551b001d,0x551c001d,0x551d001d,0x551e001d,0x551f001d,0x5520001d,
	0x5521001d,0x5522001d,0x5523001d,0x5524001d,0x5525001d,0x5526001d,0x5527001d,0x5528001d,
	0x5529001d,0x552a001d,0x552b001d,0x552c001d,0x552d001d,0x552e001d,0x552f001d,0x5530001d,
	0x5531001d,0x5532001d,0x5533001d,0x5534001d,0x5535001d,0x5536001d,0x5537001d,0x5538001d,
	0x5539001d,0x553a001d,0x553b001d,0x553c001d,0x553d001d,0x553e001d,0x553f001d,0x5540001d,
	0x5541001d,0x5542001d,0x5543001d,0x5544001d,0x5545001d,0x5546001d,0x5547001d,0x5548001d,
	0x5549001d,0x554a001d,0x554b001d,0x554c001d,0x554d001d,0x554e001d,0x554f001d,0x5550001d,
	0x5551001d,0x5552001d,0x5553001d,0x5554001d,0x5555001d,0x5556001d,0x5557001d,0x5558001d,
	0x5559001d,0x555a001d,0x555b001d,0x555c001d,0x555d001d,0x555e001d,0x555f001d,0x5560001d,
	0x5561001d,0x5562001d,0x5563001d,0x5564001d,0x5565001d,0x5566001d,0x5567001d,0x5568001d,
	0x5569001d,0x556a001d,0x556b001d,0x556c001d,0x556d001d,0x556e001d,0x556f001d,0x5570001d,
	0x5571001d,0x5572001d,0x5573001d,0x5574001d,0x5575001d,0x5576001d,0x5577001d,0x5578001d,
	0x5579001d,0x557a001d,0x557b001d,0x557c001d,0x557d001d,0x557e001d,0x557f001d,0x5580001d,
	0x5581001d,0x5582001d,0x5583001d,0x5584001d,0x5585001d,0x5586001d,0x5587001d,0x5588001d,
	0x5589001d,0x558a001d,0x558b001d,0x558c001d,0x558d001d,0x558e001d,0x558f001d,0x5590001d,
	0x5591001d,0x5592001d,0x5593001d,0x5594001d,0x5595001d,0x5596001d,0x5597001d,0x5598001d,
	0x5599001d,0x559a001d,0x559b001d,0x559c001d,0x559d001d,0x559e001d,0x559f001d,0x55a0001d,
	0x55a1001d,0x55a2001d,0x55a3001d,0x55a4001d,0x55a5001d,0x55a6001d,0x55a7001d,0x55a8001d,
	0x55a9001d,0x55aa001d,0x55ab001d,0x55ac001d,0x55ad001d,0x55ae001d,0x55af001d,0x55b0001d,
	0x55b1001d,0x55b2001d,0x55b3001d,0x55b4001d,0x55b5001d,0x55b6001d,0x55b7001d,0x55b8001d,
	0x55b9001d,0x55ba001d,0x55bb001d,0x55bc001d,0x55bd001d,0x55be001d,0x55bf001d,0x55c0001d,
	0x55c1001d,0x55c2001d,0x55c3001d,0x55c4001d,0x55c5001d,0x55c6001d,0x55c7001d,0x55c8001d,
	0x55c9001d,0x55ca001d,0x55cb001d,0x55cc001d,0x55cd001d,0x55ce001d,0x55cf001d,0x55d0001d,
	0x55d1001d,0x55d2001d,0x55d3001d,0x55d4001d,0x55d5001d,0x55d6001d,0x55d7001d,0x55d8001d,
	0x55d9001d,0x55da001d,0x55db001d,0x55dc001d,0x55dd001d,0x55de001d,0x55df001d,0x55e0001d,
	0x55e1001d,0x55e2001d,0x55e3001d,0x55e4001d,0x55e5001d,0x55e6001d,0x55e7001d,0x55e8001d,
	0x55e9001d,0x55ea001d,0x55eb001d,0x55ec001d,0x55ed001d,0x55ee001d,0x55ef001d,0x55f0001d,
	0x55f1001d,0x55f2001d,0x55f3001d,0x55f4001d,0x55f5001d,0x55f6001d,0x55f7001d,0x55f8001d,
	0x55f9001d,0x55fa001d,0x55fb001d,0x55fc001d,0x55fd001d,0x55fe001d,0x55ff001d,0x5600001d,
	0x5601001d,0x5602001d,0x5603001d,0x5604001d,0x5605001d,0x5606001d,0x5607001d,0x5608001d,
	0x5609001d,0x560a001d,0x560b001d,0x560c001d,0x560d001d,0x560e001d,0x560f001d,0x5610001d,
	0x5611001d,0x5612001d,0x5613001d,0x5614001d,0x5615001d,0x5616001d,0x5617001d,0x5618001d,
	0x5619001d,0x561a001d,0x561b001d,0x561c001d,0x561d001d,0x561e001d,0x561f001d,0x5620001d,
	0x5621001d,0x5622001d,0x5623001d,0x5624001d,0x5625001d,0x5626001d,0x5627001d,0x5628001d,
	0x5629001d,0x562a001d,0x562b001d,0x562c001d,0x562d001d,0x562e001d,0x562f001d,0x5630001d,
	0x5631001d,0x5632001d,0x5633001d,0x5634001d,0x5635001d,0x5636001d,0x5637001d,0x5638001d,
	0x5639001d,0x563a001d,0x563b001d,0x563c001d,0x563d001d,0x563e001d,0x563f001d,0x5640001d,
	0x5641001d,0x5642001d,0x5643001d,0x5644001d,0x5645001d,0x5646001d,0x5647001d,0x5648001d,
	0x5649001d,0x564a001d,0x564b001d,0x564c001d,0x564d001d,0x564e001d,0x564f001d,0x5650001d,
	0x5651001d,0x5652001d,0x5653001d,0x5654001d,0x5655001d,0x5656001d,0x5657001d,0x5658001d,
	0x5659001d,0x565a001d,0x565b001d,0x565c001d,0x565d001d,0x565e001d,0x565f001d,0x5660001d,
	0x5661001d,0x5662001d,0x5663001d,0x5664001d,0x5665001d,0x5666001d,0x5667001d,0x5668001d,
	0x5669001d,0x566a001d,0x566b001d,0x566c001d,0x566d001d,0x566e001d,0x566f001d,0x5670001d,
	0x5671001d,0x5672001d,0x5673001d,0x5674001d,0x5675001d,0x5676001d,0x5677001d,0x5678001d,
	0x5679001d,0x567a001d,0x567b001d,0x567c001d,0x567d001d,0x567e001d,0x567f001d,0x5680001d,
	0x5681001d,0x5682001d,0x5683001d,0x5684001d,0x5685001d,0x5686001d,0x5687001d,0x5688001d,
	0x5689001d,0x568a001d,0x568b001d,0x568c001d,0x568d001d,0x568e001d,0x568f001d,0x5690001d,
	0x5691001d,0x5692001d,0x5693001d,0x5694001d,0x5695001d,0x5696001d,0x5697001d,0x5698001d,
	0x5699001d,0x569a001d,0x569b001d,0x569c001d,0x569d001d,0x569e001d,0x569f001d,0x56a0001d,
	0x56a1001d,0x56a2001d,0x56a3001d,0x56a4001d,0x56a5001d,0x56a6001d,0x56a7001d,0x56a8001d,
	0x56a9001d,0x56aa001d,0x56ab001d,0x56ac001d,0x56ad001d,0x56ae001d,0x56af001d,0x56b0001d,
	0x56b1001d,0x56b2001d,0x56b3001d,0x56b4001d,0x56b5001d,0x56b6001d,0x56b7001d,0x56b8001d,
	0x56b9001d,0x56ba001d,0x56bb001d,0x56bc001d,0x56bd001d,0x56be001d,0x56bf001d,0x56c0001d,
	0x56c1001d,0x56c2001d,0x56c3001d,0x56c4001d,0x56c5001d,0x56c6001d,0x56c7001d,0x56c8001d,
	0x56c9001d,0x56ca001d,0x56cb001d,0x56cc001d,0x56cd001d,0x56ce001d,0x56cf001d,0x56d0001d,
	0x56d1001d,0x56d2001d,0x56d3001d,0x56d4001d,0x56d5001d,0x56d6001d,0x56d7001d,0x56d8001d,
	0x56d9001d,0x56da001d,0x56db001d,0x56dc001d,0x56dd001d,0x56de001d,0x56df001d,0x56e0001d,
	0x56e1001d,0x56e2001d,0x56e3001d,0x56e4001d,0x56e5001d,0x56e6001d,0x56e7001d,0x56e8001d,
	0x56e9001d,0x56ea001d,0x56eb001d,0x56ec001d,0x56ed001d,0x56ee001d,0x56ef001d,0x56f0001d,
	0x56f1001d,0x56f2001d,0x56f3001d,0x56f4001d,0x56f5001d,0x56f6001d,0x56f7001d,0x56f8001d,
	0x56f9001d,0x56fa001d,0x56fb001d,0x56fc001d,0x56fd001d,0x56fe001d,0x56ff001d,0x5700001d,
	0x5701001d,0x5702001d,0x5703001d,0x5704001d,0x5705001d,0x5706001d,0x5707001d,0x5708001d,
	0x5709001d,0x570a001d,0x570b001d,0x570c001d,0x570d001d,0x570e001d,0x570f001d,0x5710001d,
	0x5711001d,0x5712001d,0x5713001d,0x5714001d,0x5715001d,0x5716001d,0x5717001d,0x5718001d,
	0x5719001d,0x571a001d,0x571b001d,0x571c001d,0x571d001d,0x571e001d,0x571f001d,0x5720001d,
	0x5721001d,0x5722001d,0x5723001d,0x5724001d,0x5725001d,0x5726001d,0x5727001d,0x5728001d,
	0x5729001d,0x572a001d,0x572b001d,0x572c001d,0x572d001d,0x572e001d,0x572f001d,0x5730001d,
	0x5731001d,0x5732001d,0x5733001d,0x5734001d,0x5735001d,0x5736001d,0x5737001d,0x5738001d,
	0x5739001d,0x573a001d,0x573b001d,0x573c001d,0x573d001d,0x573e001d,0x573f001d,0x5740001d,
	0x5741001d,0x5742001d,0x5743001d,0x5744001d,0x5745001d,0x5746001d,0x5747001d,0x5748001d,
	0x5749001d,0x574a001d,0x574b001d,0x574c001d,0x574d001d,0x574e001d,0x574f001d,0x5750001d,
	0x5751001d,0x5752001d,0x5753001d,0x5754001d,0x5755001d,0x5756001d,0x5757001d,0x5758001d,
	0x5759001d,0x575a001d,0x575b001d,0x575c001d,0x575d001d,0x575e001d,0x575f001d,0x5760001d,
	0x5761001d,0x5762001d,0x5763001d,0x5764001d,0x5765001d,0x5766001d,0x5767001d,0x5768001d,
	0x5769001d,0x576a001d,0x576b001d,0x576c001d,0x576d001d,0x576e001d,0x576f001d,0x5770001d,
	0x5771001d,0x5772001d,0x5773001d,0x5774001d,0x5775001d,0x5776001d,0x5777001d,0x5778001d,
	0x5779001d,0x577a001d,0x577b001d,0x577c001d,0x577d001d,0x577e001d,0x577f001d,0x5780001d,
	0x5781001d,0x5782001d,0x5783001d,0x5784001d,0x5785001d,0x5786001d,0x5787001d,0x5788001d,
	0x5789001d,0x578a001d,0x578b001d,0x578c001d,0x578d001d,0x578e001d,0x578f001d,0x5790001d,
	0x5791001d,0x5792001d,0x5793001d,0x5794001d,0x5795001d,0x5796001d,0x5797001d,0x5798001d,
	0x5799001d,0x579a001d,0x579b001d,0x579c001d,0x579d001d,0x579e001d,0x579f001d,0x57a0001d,
	0x57a1001d,0x57a2001d,0x57a3001d,0x57a4001d,0x57a5001d,0x57a6001d,0x57a7001d,0x57a8001d,
	0x57a9001d,0x57aa001d,0x57ab001d,0x57ac001d,0x57ad001d,0x57ae001d,0x57af001d,0x57b0001d,
	0x57b1001d,0x57b2001d,0x57b3001d,0x57b4001d,0x57b5001d,0x57b6001d,0x57b7001d,0x57b8001d,
	0x57b9001d,0x57ba001d,0x57bb001d,0x57bc001d,0x57bd001d,0x57be001d,0x57bf001d,0x57c0001d,
	0x57c1001d,0x57c2001d,0x57c3001d,0x57c4001d,0x57c5001d,0x57c6001d,0x57c7001d,0x57c8001d,
	0x57c9001d,0x57ca001d,0x57cb001d,0x57cc001d,0x57cd001d,0x57ce001d,0x57cf001d,0x57d0001d,
	0x57d1001d,0x57d2001d,0x57d3001d,0x57d4001d,0x57d5001d,0x57d6001d,0x57d7001d,0x57d8001d,
	0x57d9001d,0x57da001d,0x57db001d,0x57dc001d,0x57dd001d,0x57de001d,0x57df001d,0x57e0001d,
	0x57e1001d,0x57e2001d,0x57e3001d,0x57e4001d,0x57e5001d,0x57e6001d,0x57e7001d,0x57e8001d,
	0x57e9001d,0x57ea001d,0x57eb001d,0x57ec001d,0x57ed001d,0x57ee001d,0x57ef001d,0x57f0001d,
	0x57f1001d,0x57f2001d,0x57f3001d,0x57f4001d,0x57f5001d,0x57f6001d,0x57f7001d,0x57f8001d,
	0x57f9001d,0x57fa001d,0x57fb001d,0x57fc001d,0x57fd001d,0x57fe001d,0x57ff001d,0x5800001d,
	0x5801001d,0x5802001d,0x5803001d,0x5804001d,0x5805001d,0x5806001d,0x5807001d,0x5808001d,
	0x5809001d,0x580a001d,0x580b001d,0x580c001d,0x580d001d,0x580e001d,0x580f001d,0x5810001d,
	0x5811001d,0x5812001d,0x5813001d,0x5814001d,0x5815001d,0x5816001d,0x5817001d,0x5818001d,
	0x5819001d,0x581a001d,0x581b001d,0x581c001d,0x581d001d,0x581e001d,0x581f001d,0x5820001d,
	0x5821001d,0x5822001d,0x5823001d,0x5824001d,0x5825001d,0x5826001d,0x5827001d,0x5828001d,
	0x5829001d,0x582a001d,0x582b001d,0x582c001d,0x582d001d,0x582e001d,0x582f001d,0x5830001d,
	0x5831001d,0x5832001d,0x5833001d,0x5834001d,0x5835001d,0x5836001d,0x5837001d,0x5838001d,
	0x5839001d,0x583a001d,0x583b001d,0x583c001d,0x583d001d,0x583e001d,0x583f001d,0x5840001d,
	0x5841001d,0x5842001d,0x5843001d,0x5844001d,0x5845001d,0x5846001d,0x5847001d,0x5848001d,
	0x5849001d,0x584a001d,0x584b001d,0x584c001d,0x584d001d,0x584e001d,0x584f001d,0x5850001d,
	0x5851001d,0x5852001d,0x5853001d,0x5854001d,0x5855001d,0x5856001d,0x5857001d,0x5858001d,
	0x5859001d,0x585a001d,0x585b001d,0x585c001d,0x585d001d,0x585e001d,0x585f001d,0x5860001d,
	0x5861001d,0x5862001d,0x5863001d,0x5864001d,0x5865001d,0x5866001d,0x5867001d,0x5868001d,
	0x5869001d,0x586a001d,0x586b001d,0x586c001d,0x586d001d,0x586e001d,0x586f001d,0x5870001d,
	0x5871001d,0x5872001d,0x5873001d,0x5874001d,0x5875001d,0x5876001d,0x5877001d,0x5878001d,
	0x5879001d,0x587a001d,0x587b001d,0x587c001d,0x587d001d,0x587e001d,0x587f001d,0x5880001d,
	0x5881001d,0x5882001d,0x5883001d,0x5884001d,0x5885001d,0x5886001d,0x5887001d,0x5888001d,
	0x5889001d,0x588a001d,0x588b001d,0x588c001d,0x588d001d,0x588e001d,0x588f001d,0x5890001d,
	0x5891001d,0x5892001d,0x5893001d,0x5894001d,0x5895001d,0x5896001d,0x5897001d,0x5898001d,
	0x5899001d,0x589a001d,0x589b001d,0x589c001d,0x589d001d,0x589e001d,0x589f001d,0x58a0001d,
	0x58a1001d,0x58a2001d,0x58a3001d,0x58a4001d,0x58a5001d,0x58a6001d,0x58a7001d,0x58a8001d,
	0x58a9001d,0x58aa001d,0x58ab001d,0x58ac001d,0x58ad001d,0x58ae001d,0x58af001d,0x58b0001d,
	0x58b1001d,0x58b2001d,0x58b3001d,0x58b4001d,0x58b5001d,0x58b6001d,0x58b7001d,0x58b8001d,
	0x58b9001d,0x58ba001d,0x58bb001d,0x58bc001d,0x58bd001d,0x58be001d,0x58bf001d,0x58c0001d,
	0x58c1001d,0x58c2001d,0x58c3001d,0x58c4001d,0x58c5001d,0x58c6001d,0x58c7001d,0x58c8001d,
	0x58c9001d,0x58ca001d,0x58cb001d,0x58cc001d,0x58cd001d,0x58ce001d,0x58cf001d,0x58d0001d,
	0x58d1001d,0x58d2001d,0x58d3001d,0x58d4001d,0x58d5001d,0x58d6001d,0x58d7001d,0x58d8001d,
	0x58d9001d,0x58da001d,0x58db001d,0x58dc001d,0x58dd001d,0x58de001d,0x58df001d,0x58e0001d,
	0x58e1001d,0x58e2001d,0x58e3001d,0x58e4001d,0x58e5001d,0x58e6001d,0x58e7001d,0x58e8001d,
	0x58e9001d,0x58ea001d,0x58eb001d,0x58ec001d,0x58ed001d,0x58ee001d,0x58ef001d,0x58f0001d,
	0x58f1001d,0x58f2001d,0x58f3001d,0x58f4001d,0x58f5001d,0x58f6001d,0x58f7001d,0x58f8001d,
	0x58f9001d,0x58fa001d,0x58fb001d,0x58fc001d,0x58fd001d,0x58fe001d,0x58ff001d,0x5900001d,
	0x5901001d,0x5902001d,0x5903001d,0x5904001d,0x5905001d,0x5906001d,0x5907001d,0x5908001d,
	0x5909001d,0x590a001d,0x590b001d,0x590c001d,0x590d001d,0x590e001d,0x590f001d,0x5910001d,
	0x5911001d,0x5912001d,0x5913001d,0x5914001d,0x5915001d,0x5916001d,0x5917001d,0x5918001d,
	0x5919001d,0x591a001d,0x591b001d,0x591c001d,0x591d001d,0x591e001d,0x591f001d,0x5920001d,
	0x5921001d,0x5922001d,0x5923001d,0x5924001d,0x5925001d,0x5926001d,0x5927001d,0x5928001d,
	0x5929001d,0x592a001d,0x592b001d,0x592c001d,0x592d001d,0x592e001d,0x592f001d,0x5930001d,
	0x5931001d,0x5932001d,0x5933001d,0x5934001d,0x5935001d,0x5936001d,0x5937001d,0x5938001d,
	0x5939001d,0x593a001d,0x593b001d,0x593c001d,0x593d001d,0x593e001d,0x593f001d,0x5940001d,
	0x5941001d,0x5942001d,0x5943001d,0x5944001d,0x5945001d,0x5946001d,0x5947001d,0x5948001d,
	0x5949001d,0x594a001d,0x594b001d,0x594c001d,0x594d001d,0x594e001d,0x594f001d,0x5950001d,
	0x5951001d,0x5952001d,0x5953001d,0x5954001d,0x5955001d,0x5956001d,0x5957001d,0x5958001d,
	0x5959001d,0x595a001d,0x595b001d,0x595c001d,0x595d001d,0x595e001d,0x595f001d,0x5960001d,
	0x5961001d,0x5962001d,0x5963001d,0x5964001d,0x5965001d,0x5966001d,0x5967001d,0x5968001d,
	0x5969001d,0x596a001d,0x596b001d,0x596c001d,0x596d001d,0x596e001d,0x596f001d,0x5970001d,
	0x5971001d,0x5972001d,0x5973001d,0x5974001d,0x5975001d,0x5976001d,0x5977001d,0x5978001d,
	0x5979001d,0x597a001d,0x597b001d,0x597c001d,0x597d001d,0x597e001d,0x597f001d,0x5980001d,
	0x5981001d,0x5982001d,0x5983001d,0x5984001d,0x5985001d,0x5986001d,0x5987001d,0x5988001d,
	0x5989001d,0x598a001d,0x598b001d,0x598c001d,0x598d001d,0x598e001d,0x598f001d,0x5990001d,
	0x5991001d,0x5992001d,0x5993001d,0x5994001d,0x5995001d,0x5996001d,0x5997001d,0x5998001d,
	0x5999001d,0x599a001d,0x599b001d,0x599c001d,0x599d001d,0x599e001d,0x599f001d,0x59a0001d,
	0x59a1001d,0x59a2001d,0x59a3001d,0x59a4001d,0x59a5001d,0x59a6001d,0x59a7001d,0x59a8001d,
	0x59a9001d,0x59aa001d,0x59ab001d,0x59ac001d,0x59ad001d,0x59ae001d,0x59af001d,0x59b0001d,
	0x59b1001d,0x59b2001d,0x59b3001d,0x59b4001d,0x59b5001d,0x59b6001d,0x59b7001d,0x59b8001d,
	0x59b9001d,0x59ba001d,0x59bb001d,0x59bc001d,0x59bd001d,0x59be001d,0x59bf001d,0x59c0001d,
	0x59c1001d,0x59c2001d,0x59c3001d,0x59c4001d,0x59c5001d,0x59c6001d,0x59c7001d,0x59c8001d,
	0x59c9001d,0x59ca001d,0x59cb001d,0x59cc001d,0x59cd001d,0x59ce001d,0x59cf001d,0x59d0001d,
	0x59d1001d,0x59d2001d,0x59d3001d,0x59d4001d,0x59d5001d,0x59d6001d,0x59d7001d,0x59d8001d,
	0x59d9001d,0x59da001d,0x59db001d,0x59dc001d,0x59dd001d,0x59de001d,0x59df001d,0x59e0001d,
	0x59e1001d,0x59e2001d,0x59e3001d,0x59e4001d,0x59e5001d,0x59e6001d,0x59e7001d,0x59e8001d,
	0x59e9001d,0x59ea001d,0x59eb001d,0x59ec001d,0x59ed001d,0x59ee001d,0x59ef001d,0x59f0001d,
	0x59f1001d,0x59f2001d,0x59f3001d,0x59f4001d,0x59f5001d,0x59f6001d,0x59f7001d,0x59f8001d,
	0x59f9001d,0x59fa001d,0x59fb001d,0x59fc001d,0x59fd001d,0x59fe001d,0x59ff001d,0x5a00001d,
	0x5a01001d,0x5a02001d,0x5a03001d,0x5a04001d,0x5a05001d,0x5a06001d,0x5a07001d,0x5a08001d,
	0x5a09001d,0x5a0a001d,0x5a0b001d,0x5a0c001d,0x5a0d001d,0x5a0e001d,0x5a0f001d,0x5a10001d,
	0x5a11001d,0x5a12001d,0x5a13001d,0x5a14001d,0x5a15001d,0x5a16001d,0x5a17001d,0x5a18001d,
	0x5a19001d,0x5a1a001d,0x5a1b001d,0x5a1c001d,0x5a1d001d,0x5a1e001d,0x5a1f001d,0x5a20001d,
	0x5a21001d,0x5a22001d,0x5a23001d,0x5a24001d,0x5a25001d,0x5a26001d,0x5a27001d,0x5a28001d,
	0x5a29001d,0x5a2a001d,0x5a2b001d,0x5a2c001d,0x5a2d001d,0x5a2e001d,0x5a2f001d,0x5a30001d,
	0x5a31001d,0x5a32001d,0x5a33001d,0x5a34001d,0x5a35001d,0x5a36001d,0x5a37001d,0x5a38001d,
	0x5a39001d,0x5a3a001d,0x5a3b001d,0x5a3c001d,0x5a3d001d,0x5a3e001d,0x5a3f001d,0x5a40001d,
	0x5a41001d,0x5a42001d,0x5a43001d,0x5a44001d,0x5a45001d,0x5a46001d,0x5a47001d,0x5a48001d,
	0x5a49001d,0x5a4a001d,0x5a4b001d,0x5a4c001d,0x5a4d001d,0x5a4e001d,0x5a4f001d,0x5a50001d,
	0x5a51001d,0x5a52001d,0x5a53001d,0x5a54001d,0x5a55001d,0x5a56001d,0x5a57001d,0x5a58001d,
	0x5a59001d,0x5a5a001d,0x5a5b001d,0x5a5c001d,0x5a5d001d,0x5a5e001d,0x5a5f001d,0x5a60001d,
	0x5a61001d,0x5a62001d,0x5a63001d,0x5a64001d,0x5a65001d,0x5a66001d,0x5a67001d,0x5a68001d,
	0x5a69001d,0x5a6a001d,0x5a6b001d,0x5a6c001d,0x5a6d001d,0x5a6e001d,0x5a6f001d,0x5a70001d,
	0x5a71001d,0x5a72001d,0x5a73001d,0x5a74001d,0x5a75001d,0x5a76001d,0x5a77001d,0x5a78001d,
	0x5a79001d,0x5a7a001d,0x5a7b001d,0x5a7c001d,0x5a7d001d,0x5a7e001d,0x5a7f001d,0x5a80001d,
	0x5a81001d,0x5a82001d,0x5a83001d,0x5a84001d,0x5a85001d,0x5a86001d,0x5a87001d,0x5a88001d,
	0x5a89001d,0x5a8a001d,0x5a8b001d,0x5a8c001d,0x5a8d001d,0x5a8e001d,0x5a8f001d,0x5a90001d,
	0x5a91001d,0x5a92001d,0x5a93001d,0x5a94001d,0x5a95001d,0x5a96001d,0x5a97001d,0x5a98001d,
	0x5a99001d,0x5a9a001d,0x5a9b001d,0x5a9c001d,0x5a9d001d,0x5a9e001d,0x5a9f001d,0x5aa0001d,
	0x5aa1001d,0x5aa2001d,0x5aa3001d,0x5aa4001d,0x5aa5001d,0x5aa6001d,0x5aa7001d,0x5aa8001d,
	0x5aa9001d,0x5aaa001d,0x5aab001d,0x5aac001d,0x5aad001d,0x5aae001d,0x5aaf001d,0x5ab0001d,
	0x5ab1001d,0x5ab2001d,0x5ab3001d,0x5ab4001d,0x5ab5001d,0x5ab6001d,0x5ab7001d,0x5ab8001d,
	0x5ab9001d,0x5aba001d,0x5abb001d,0x5abc001d,0x5abd001d,0x5abe001d,0x5abf001d,0x5ac0001d,
	0x5ac1001d,0x5ac2001d,0x5ac3001d,0x5ac4001d,0x5ac5001d,0x5ac6001d,0x5ac7001d,0x5ac8001d,
	0x5ac9001d,0x5aca001d,0x5acb001d,0x5acc001d,0x5acd001d,0x5ace001d,0x5acf001d,0x5ad0001d,
	0x5ad1001d,0x5ad2001d,0x5ad3001d,0x5ad4001d,0x5ad5001d,0x5ad6001d,0x5ad7001d,0x5ad8001d,
	0x5ad9001d,0x5ada001d,0x5adb001d,0x5adc001d,0x5add001d,0x5ade001d,0x5adf001d,0x5ae0001d,
	0x5ae1001d,0x5ae2001d,0x5ae3001d,0x5ae4001d,0x5ae5001d,0x5ae6001d,0x5ae7001d,0x5ae8001d,
	0x5ae9001d,0x5aea001d,0x5aeb001d,0x5aec001d,0x5aed001d,0x5aee001d,0x5aef001d,0x5af0001d,
	0x5af1001d,0x5af2001d,0x5af3001d,0x5af4001d,0x5af5001d,0x5af6001d,0x5af7001d,0x5af8001d,
	0x5af9001d,0x5afa001d,0x5afb001d,0x5afc001d,0x5afd001d,0x5afe001d,0x5aff001d,0x5b00001d,
	0x5b01001d,0x5b02001d,0x5b03001d,0x5b04001d,0x5b05001d,0x5b06001d,0x5b07001d,0x5b08001d,
	0x5b09001d,0x5b0a001d,0x5b0b001d,0x5b0c001d,0x5b0d001d,0x5b0e001d,0x5b0f001d,0x5b10001d,
	0x5b11001d,0x5b12001d,0x5b13001d,0x5b14001d,0x5b15001d,0x5b16001d,0x5b17001d,0x5b18001d,
	0x5b19001d,0x5b1a001d,0x5b1b001d,0x5b1c001d,0x5b1d001d,0x5b1e001d,0x5b1f001d,0x5b20001d,
	0x5b21001d,0x5b22001d,0x5b23001d,0x5b24001d,0x5b25001d,0x5b26001d,0x5b27001d,0x5b28001d,
	0x5b29001d,0x5b2a001d,0x5b2b001d,0x5b2c001d,0x5b2d001d,0x5b2e001d,0x5b2f001d,0x5b30001d,
	0x5b31001d,0x5b32001d,0x5b33001d,0x5b34001d,0x5b35001d,0x5b36001d,0x5b37001d,0x5b38001d,
	0x5b39001d,0x5b3a001d,0x5b3b001d,0x5b3c001d,0x5b3d001d,0x5b3e001d,0x5b3f001d,0x5b40001d,
	0x5b41001d,0x5b42001d,0x5b43001d,0x5b44001d,0x5b45001d,0x5b46001d,0x5b47001d,0x5b48001d,
	0x5b49001d,0x5b4a001d,0x5b4b001d,0x5b4c001d,0x5b4d001d,0x5b4e001d,0x5b4f001d,0x5b50001d,
	0x5b51001d,0x5b52001d,0x5b53001d,0x5b54001d,0x5b55001d,0x5b56001d,0x5b57001d,0x5b58001d,
	0x5b59001d,0x5b5a001d,0x5b5b001d,0x5b5c001d,0x5b5d001d,0x5b5e001d,0x5b5f001d,0x5b60001d,
	0x5b61001d,0x5b62001d,0x5b63001d,0x5b64001d,0x5b65001d,0x5b66001d,0x5b67001d,0x5b68001d,
	0x5b69001d,0x5b6a001d,0x5b6b001d,0x5b6c001d,0x5b6d001d,0x5b6e001d,0x5b6f001d,0x5b70001d,
	0x5b71001d,0x5b72001d,0x5b73001d,0x5b74001d,0x5b75001d,0x5b76001d,0x5b77001d,0x5b78001d,
	0x5b79001d,0x5b7a001d,0x5b7b001d,0x5b7c001d,0x5b7d001d,0x5b7e001d,0x5b7f001d,0x5b80001d,
	0x5b81001d,0x5b82001d,0x5b83001d,0x5b84001d,0x5b85001d,0x5b86001d,0x5b87001d,0x5b88001d,
	0x5b89001d,0x5b8a001d,0x5b8b001d,0x5b8c001d,0x5b8d001d,0x5b8e001d,0x5b8f001d,0x5b90001d,
	0x5b91001d,0x5b92001d,0x5b93001d,0x5b94001d,0x5b95001d,0x5b96001d,0x5b97001d,0x5b98001d,
	0x5b99001d,0x5b9a001d,0x5b9b001d,0x5b9c001d,0x5b9d001d,0x5b9e001d,0x5b9f001d,0x5ba0001d,
	0x5ba1001d,0x5ba2001d,0x5ba3001d,0x5ba4001d,0x5ba5001d,0x5ba6001d,0x5ba7001d,0x5ba8001d,
	0x5ba9001d,0x5baa001d,0x5bab001d,0x5bac001d,0x5bad001d,0x5bae001d,0x5baf001d,0x5bb0001d,
	0x5bb1001d,0x5bb2001d,0x5bb3001d,0x5bb4001d,0x5bb5001d,0x5bb6001d,0x5bb7001d,0x5bb8001d,
	0x5bb9001d,0x5bba001d,0x5bbb001d,0x5bbc001d,0x5bbd001d,0x5bbe001d,0x5bbf001d,0x5bc0001d,
	0x5bc1001d,0x5bc2001d,0x5bc3001d,0x5bc4001d,0x5bc5001d,0x5bc6001d,0x5bc7001d,0x5bc8001d,
	0x5bc9001d,0x5bca001d,0x5bcb001d,0x5bcc001d,0x5bcd001d,0x5bce001d,0x5bcf001d,0x5bd0001d,
	0x5bd1001d,0x5bd2001d,0x5bd3001d,0x5bd4001d,0x5bd5001d,0x5bd6001d,0x5bd7001d,0x5bd8001d,
	0x5bd9001d,0x5bda001d,0x5bdb001d,0x5bdc001d,0x5bdd001d,0x5bde001d,0x5bdf001d,0x5be0001d,
	0x5be1001d,0x5be2001d,0x5be3001d,0x5be4001d,0x5be5001d,0x5be6001d,0x5be7001d,0x5be8001d,
	0x5be9001d,0x5bea001d,0x5beb001d,0x5bec001d,0x5bed001d,0x5bee001d,0x5bef001d,0x5bf0001d,
	0x5bf1001d,0x5bf2001d,0x5bf3001d,0x5bf4001d,0x5bf5001d,0x5bf6001d,0x5bf7001d,0x5bf8001d,
	0x5bf9001d,0x5bfa001d,0x5bfb001d,0x5bfc001d,0x5bfd001d,0x5bfe001d,0x5bff001d,0x5c00001d,
	0x5c01001d,0x5c02001d,0x5c03001d,0x5c04001d,0x5c05001d,0x5c06001d,0x5c07001d,0x5c08001d,
	0x5c09001d,0x5c0a001d,0x5c0b001d,0x5c0c001d,0x5c0d001d,0x5c0e001d,0x5c0f001d,0x5c10001d,
	0x5c11001d,0x5c12001d,0x5c13001d,0x5c14001d,0x5c15001d,0x5c16001d,0x5c17001d,0x5c18001d,
	0x5c19001d,0x5c1a001d,0x5c1b001d,0x5c1c001d,0x5c1d001d,0x5c1e001d,0x5c1f001d,0x5c20001d,
	0x5c21001d,0x5c22001d,0x5c23001d,0x5c24001d,0x5c25001d,0x5c26001d,0x5c27001d,0x5c28001d,
	0x5c29001d,0x5c2a001d,0x5c2b001d,0x5c2c001d,0x5c2d001d,0x5c2e001d,0x5c2f001d,0x5c30001d,
	0x5c31001d,0x5c32001d,0x5c33001d,0x5c34001d,0x5c35001d,0x5c36001d,0x5c37001d,0x5c38001d,
	0x5c39001d,0x5c3a001d,0x5c3b001d,0x5c3c001d,0x5c3d001d,0x5c3e001d,0x5c3f001d,0x5c40001d,
	0x5c41001d,0x5c42001d,0x5c43001d,0x5c44001d,0x5c45001d,0x5c46001d,0x5c47001d,0x5c48001d,
	0x5c49001d,0x5c4a001d,0x5c4b001d,0x5c4c001d,0x5c4d001d,0x5c4e001d,0x5c4f001d,0x5c50001d,
	0x5c51001d,0x5c52001d,0x5c53001d,0x5c54001d,0x5c55001d,0x5c56001d,0x5c57001d,0x5c58001d,
	0x5c59001d,0x5c5a001d,0x5c5b001d,0x5c5c001d,0x5c5d001d,0x5c5e001d,0x5c5f001d,0x5c60001d,
	0x5c61001d,0x5c62001d,0x5c63001d,0x5c64001d,0x5c65001d,0x5c66001d,0x5c67001d,0x5c68001d,
	0x5c69001d,0x5c6a001d,0x5c6b001d,0x5c6c001d,0x5c6d001d,0x5c6e001d,0x5c6f001d,0x5c70001d,
	0x5c71001d,0x5c72001d,0x5c73001d,0x5c74001d,0x5c75001d,0x5c76001d,0x5c77001d,0x5c78001d,
	0x5c79001d,0x5c7a001d,0x5c7b001d,0x5c7c001d,0x5c7d001d,0x5c7e001d,0x5c7f001d,0x5c80001d,
	0x5c81001d,0x5c82001d,0x5c83001d,0x5c84001d,0x5c85001d,0x5c86001d,0x5c87001d,0x5c88001d,
	0x5c89001d,0x5c8a001d,0x5c8b001d,0x5c8c001d,0x5c8d001d,0x5c8e001d,0x5c8f001d,0x5c90001d,
	0x5c91001d,0x5c92001d,0x5c93001d,0x5c94001d,0x5c95001d,0x5c96001d,0x5c97001d,0x5c98001d,
	0x5c99001d,0x5c9a001d,0x5c9b001d,0x5c9c001d,0x5c9d001d,0x5c9e001d,0x5c9f001d,0x5ca0001d,
	0x5ca1001d,0x5ca2001d,0x5ca3001d,0x5ca4001d,0x5ca5001d,0x5ca6001d,0x5ca7001d,0x5ca8001d,
	0x5ca9001d,0x5caa001d,0x5cab001d,0x5cac001d,0x5cad001d,0x5cae001d,0x5caf001d,0x5cb0001d,
	0x5cb1001d,0x5cb2001d,0x5cb3001d,0x5cb4001d,0x5cb5001d,0x5cb6001d,0x5cb7001d,0x5cb8001d,
	0x5cb9001d,0x5cba001d,0x5cbb001d,0x5cbc001d,0x5cbd001d,0x5cbe001d,0x5cbf001d,0x5cc0001d,
	0x5cc1001d,0x5cc2001d,0x5cc3001d,0x5cc4001d,0x5cc5001d,0x5cc6001d,0x5cc7001d,0x5cc8001d,
	0x5cc9001d,0x5cca001d,0x5ccb001d,0x5ccc001d,0x5ccd001d,0x5cce001d,0x5ccf001d,0x5cd0001d,
	0x5cd1001d,0x5cd2001d,0x5cd3001d,0x5cd4001d,0x5cd5001d,0x5cd6001d,0x5cd7001d,0x5cd8001d,
	0x5cd9001d,0x5cda001d,0x5cdb001d,0x5cdc001d,0x5cdd001d,0x5cde001d,0x5cdf001d,0x5ce0001d,
	0x5ce1001d,0x5ce2001d,0x5ce3001d,0x5ce4001d,0x5ce5001d,0x5ce6001d,0x5ce7001d,0x5ce8001d,
	0x5ce9001d,0x5cea001d,0x5ceb001d,0x5cec001d,0x5ced001d,0x5cee001d,0x5cef001d,0x5cf0001d,
	0x5cf1001d,0x5cf2001d,0x5cf3001d,0x5cf4001d,0x5cf5001d,0x5cf6001d,0x5cf7001d,0x5cf8001d,
	0x5cf9001d,0x5cfa001d,0x5cfb001d,0x5cfc001d,0x5cfd001d,0x5cfe001d,0x5cff001d,0x5d00001d,
	0x5d01001d,0x5d02001d,0x5d03001d,0x5d04001d,0x5d05001d,0x5d06001d,0x5d07001d,0x5d08001d,
	0x5d09001d,0x5d0a001d,0x5d0b001d,0x5d0c001d,0x5d0d001d,0x5d0e001d,0x5d0f001d,0x5d10001d,
	0x5d11001d,0x5d12001d,0x5d13001d,0x5d14001d,0x5d15001d,0x5d16001d,0x5d17001d,0x5d18001d,
	0x5d19001d,0x5d1a001d,0x5d1b001d,0x5d1c001d,0x5d1d001d,0x5d1e001d,0x5d1f001d,0x5d20001d,
	0x5d21001d,0x5d22001d,0x5d23001d,0x5d24001d,0x5d25001d,0x5d26001d,0x5d27001d,0x5d28001d,
	0x5d29001d,0x5d2a001d,0x5d2b001d,0x5d2c001d,0x5d2d001d,0x5d2e001d,0x5d2f001d,0x5d30001d,
	0x5d31001d,0x5d32001d,0x5d33001d,0x5d34001d,0x5d35001d,0x5d36001d,0x5d37001d,0x5d38001d,
	0x5d39001d,0x5d3a001d,0x5d3b001d,0x5d3c001d,0x5d3d001d,0x5d3e001d,0x5d3f001d,0x5d40001d,
	0x5d41001d,0x5d42001d,0x5d43001d,0x5d44001d,0x5d45001d,0x5d46001d,0x5d47001d,0x5d48001d,
	0x5d49001d,0x5d4a001d,0x5d4b001d,0x5d4c001d,0x5d4d001d,0x5d4e001d,0x5d4f001d,0x5d50001d,
	0x5d51001d,0x5d52001d,0x5d53001d,0x5d54001d,0x5d55001d,0x5d56001d,0x5d57001d,0x5d58001d,
	0x5d59001d,0x5d5a001d,0x5d5b001d,0x5d5c001d,0x5d5d001d,0x5d5e001d,0x5d5f001d,0x5d60001d,
	0x5d61001d,0x5d62001d,0x5d63001d,0x5d64001d,0x5d65001d,0x5d66001d,0x5d67001d,0x5d68001d,
	0x5d69001d,0x5d6a001d,0x5d6b001d,0x5d6c001d,0x5d6d001d,0x5d6e001d,0x5d6f001d,0x5d70001d,
	0x5d71001d,0x5d72001d,0x5d73001d,0x5d74001d,0x5d75001d,0x5d76001d,0x5d77001d,0x5d78001d,
	0x5d79001d,0x5d7a001d,0x5d7b001d,0x5d7c001d,0x5d7d001d,0x5d7e001d,0x5d7f001d,0x5d80001d,
	0x5d81001d,0x5d82001d,0x5d83001d,0x5d84001d,0x5d85001d,0x5d86001d,0x5d87001d,0x5d88001d,
	0x5d89001d,0x5d8a001d,0x5d8b001d,0x5d8c001d,0x5d8d001d,0x5d8e001d,0x5d8f001d,0x5d90001d,
	0x5d91001d,0x5d92001d,0x5d93001d,0x5d94001d,0x5d95001d,0x5d96001d,0x5d97001d,0x5d98001d,
	0x5d99001d,0x5d9a001d,0x5d9b001d,0x5d9c001d,0x5d9d001d,0x5d9e001d,0x5d9f001d,0x5da0001d,
	0x5da1001d,0x5da2001d,0x5da3001d,0x5da4001d,0x5da5001d,0x5da6001d,0x5da7001d,0x5da8001d,
	0x5da9001d,0x5daa001d,0x5dab001d,0x5dac001d,0x5dad001d,0x5dae001d,0x5daf001d,0x5db0001d,
	0x5db1001d,0x5db2001d,0x5db3001d,0x5db4001d,0x5db5001d,0x5db6001d,0x5db7001d,0x5db8001d,
	0x5db9001d,0x5dba001d,0x5dbb001d,0x5dbc001d,0x5dbd001d,0x5dbe001d,0x5dbf001d,0x5dc0001d,
	0x5dc1001d,0x5dc2001d,0x5dc3001d,0x5dc4001d,0x5dc5001d,0x5dc6001d,0x5dc7001d,0x5dc8001d,
	0x5dc9001d,0x5dca001d,0x5dcb001d,0x5dcc001d,0x5dcd001d,0x5dce001d,0x5dcf001d,0x5dd0001d,
	0x5dd1001d,0x5dd2001d,0x5dd3001d,0x5dd4001d,0x5dd5001d,0x5dd6001d,0x5dd7001d,0x5dd8001d,
	0x5dd9001d,0x5dda001d,0x5ddb001d,0x5ddc001d,0x5ddd001d,0x5dde001d,0x5ddf001d,0x5de0001d,
	0x5de1001d,0x5de2001d,0x5de3001d,0x5de4001d,0x5de5001d,0x5de6001d,0x5de7001d,0x5de8001d,
	0x5de9001d,0x5dea001d,0x5deb001d,0x5dec001d,0x5ded001d,0x5dee001d,0x5def001d,0x5df0001d,
	0x5df1001d,0x5df2001d,0x5df3001d,0x5df4001d,0x5df5001d,0x5df6001d,0x5df7001d,0x5df8001d,
	0x5df9001d,0x5dfa001d,0x5dfb001d,0x5dfc001d,0x5dfd001d,0x5dfe001d,0x5dff001d,0x5e00001d,
	0x5e01001d,0x5e02001d,0x5e03001d,0x5e04001d,0x5e05001d,0x5e06001d,0x5e07001d,0x5e08001d,
	0x5e09001d,0x5e0a001d,0x5e0b001d,0x5e0c001d,0x5e0d001d,0x5e0e001d,0x5e0f001d,0x5e10001d,
	0x5e11001d,0x5e12001d,0x5e13001d,0x5e14001d,0x5e15001d,0x5e16001d,0x5e17001d,0x5e18001d,
	0x5e19001d,0x5e1a001d,0x5e1b001d,0x5e1c001d,0x5e1d001d,0x5e1e001d,0x5e1f001d,0x5e20001d,
	0x5e21001d,0x5e22001d,0x5e23001d,0x5e24001d,0x5e25001d,0x5e26001d,0x5e27001d,0x5e28001d,
	0x5e29001d,0x5e2a001d,0x5e2b001d,0x5e2c001d,0x5e2d001d,0x5e2e001d,0x5e2f001d,0x5e30001d,
	0x5e31001d,0x5e32001d,0x5e33001d,0x5e34001d,0x5e35001d,0x5e36001d,0x5e37001d,0x5e38001d,
	0x5e39001d,0x5e3a001d,0x5e3b001d,0x5e3c001d,0x5e3d001d,0x5e3e001d,0x5e3f001d,0x5e40001d,
	0x5e41001d,0x5e42001d,0x5e43001d,0x5e44001d,0x5e45001d,0x5e46001d,0x5e47001d,0x5e48001d,
	0x5e49001d,0x5e4a001d,0x5e4b001d,0x5e4c001d,0x5e4d001d,0x5e4e001d,0x5e4f001d,0x5e50001d,
	0x5e51001d,0x5e52001d,0x5e53001d,0x5e54001d,0x5e55001d,0x5e56001d,0x5e57001d,0x5e58001d,
	0x5e59001d,0x5e5a001d,0x5e5b001d,0x5e5c001d,0x5e5d001d,0x5e5e001d,0x5e5f001d,0x5e60001d,
	0x5e61001d,0x5e62001d,0x5e63001d,0x5e64001d,0x5e65001d,0x5e66001d,0x5e67001d,0x5e68001d,
	0x5e69001d,0x5e6a001d,0x5e6b001d,0x5e6c001d,0x5e6d001d,0x5e6e001d,0x5e6f001d,0x5e70001d,
	0x5e71001d,0x5e72001d,0x5e73001d,0x5e74001d,0x5e75001d,0x5e76001d,0x5e77001d,0x5e78001d,
	0x5e79001d,0x5e7a001d,0x5e7b001d,0x5e7c001d,0x5e7d001d,0x5e7e001d,0x5e7f001d,0x5e80001d,
	0x5e81001d,0x5e82001d,0x5e83001d,0x5e84001d,0x5e85001d,0x5e86001d,0x5e87001d,0x5e88001d,
	0x5e89001d,0x5e8a001d,0x5e8b001d,0x5e8c001d,0x5e8d001d,0x5e8e001d,0x5e8f001d,0x5e90001d,
	0x5e91001d,0x5e92001d,0x5e93001d,0x5e94001d,0x5e95001d,0x5e96001d,0x5e97001d,0x5e98001d,
	0x5e99001d,0x5e9a001d,0x5e9b001d,0x5e9c001d,0x5e9d001d,0x5e9e001d,0x5e9f001d,0x5ea0001d,
	0x5ea1001d,0x5ea2001d,0x5ea3001d,0x5ea4001d,0x5ea5001d,0x5ea6001d,0x5ea7001d,0x5ea8001d,
	0x5ea9001d,0x5eaa001d,0x5eab001d,0x5eac001d,0x5ead001d,0x5eae001d,0x5eaf001d,0x5eb0001d,
	0x5eb1001d,0x5eb2001d,0x5eb3001d,0x5eb4001d,0x5eb5001d,0x5eb6001d,0x5eb7001d,0x5eb8001d,
	0x5eb9001d,0x5eba001d,0x5ebb001d,0x5ebc001d,0x5ebd001d,0x5ebe001d,0x5ebf001d,0x5ec0001d,
	0x5ec1001d,0x5ec2001d,0x5ec3001d,0x5ec4001d,0x5ec5001d,0x5ec6001d,0x5ec7001d,0x5ec8001d,
	0x5ec9001d,0x5eca001d,0x5ecb001d,0x5ecc001d,0x5ecd001d,0x5ece001d,0x5ecf001d,0x5ed0001d,
	0x5ed1001d,0x5ed2001d,0x5ed3001d,0x5ed4001d,0x5ed5001d,0x5ed6001d,0x5ed7001d,0x5ed8001d,
	0x5ed9001d,0x5eda001d,0x5edb001d,0x5edc001d,0x5edd001d,0x5ede001d,0x5edf001d,0x5ee0001d,
	0x5ee1001d,0x5ee2001d,0x5ee3001d,0x5ee4001d,0x5ee5001d,0x5ee6001d,0x5ee7001d,0x5ee8001d,
	0x5ee9001d,0x5eea001d,0x5eeb001d,0x5eec001d,0x5eed001d,0x5eee001d,0x5eef001d,0x5ef0001d,
	0x5ef1001d,0x5ef2001d,0x5ef3001d,0x5ef4001d,0x5ef5001d,0x5ef6001d,0x5ef7001d,0x5ef8001d,
	0x5ef9001d,0x5efa001d,0x5efb001d,0x5efc001d,0x5efd001d,0x5efe001d,0x5eff001d,0x5f00001d,
	0x5f01001d,0x5f02001d,0x5f03001d,0x5f04001d,0x5f05001d,0x5f06001d,0x5f07001d,0x5f08001d,
	0x5f09001d,0x5f0a001d,0x5f0b001d,0x5f0c001d,0x5f0d001d,0x5f0e001d,0x5f0f001d,0x5f10001d,
	0x5f11001d,0x5f12001d,0x5f13001d,0x5f14001d,0x5f15001d,0x5f16001d,0x5f17001d,0x5f18001d,
	0x5f19001d,0x5f1a001d,0x5f1b001d,0x5f1c001d,0x5f1d001d,0x5f1e001d,0x5f1f001d,0x5f20001d,
	0x5f21001d,0x5f22001d,0x5f23001d,0x5f24001d,0x5f25001d,0x5f26001d,0x5f27001d,0x5f28001d,
	0x5f29001d,0x5f2a001d,0x5f2b001d,0x5f2c001d,0x5f2d001d,0x5f2e001d,0x5f2f001d,0x5f30001d,
	0x5f31001d,0x5f32001d,0x5f33001d,0x5f34001d,0x5f35001d,0x5f36001d,0x5f37001d,0x5f38001d,
	0x5f39001d,0x5f3a001d,0x5f3b001d,0x5f3c001d,0x5f3d001d,0x5f3e001d,0x5f3f001d,0x5f40001d,
	0x5f41001d,0x5f42001d,0x5f43001d,0x5f44001d,0x5f45001d,0x5f46001d,0x5f47001d,0x5f48001d,
	0x5f49001d,0x5f4a001d,0x5f4b001d,0x5f4c001d,0x5f4d001d,0x5f4e001d,0x5f4f001d,0x5f50001d,
	0x5f51001d,0x5f52001d,0x5f53001d,0x5f54001d,0x5f55001d,0x5f56001d,0x5f57001d,0x5f58001d,
	0x5f59001d,0x5f5a001d,0x5f5b001d,0x5f5c001d,0x5f5d001d,0x5f5e001d,0x5f5f001d,0x5f60001d,
	0x5f61001d,0x5f62001d,0x5f63001d,0x5f64001d,0x5f65001d,0x5f66001d,0x5f67001d,0x5f68001d,
	0x5f69001d,0x5f6a001d,0x5f6b001d,0x5f6c001d,0x5f6d001d,0x5f6e001d,0x5f6f001d,0x5f70001d,
	0x5f71001d,0x5f72001d,0x5f73001d,0x5f74001d,0x5f75001d,0x5f76001d,0x5f77001d,0x5f78001d,
	0x5f79001d,0x5f7a001d,0x5f7b001d,0x5f7c001d,0x5f7d001d,0x5f7e001d,0x5f7f001d,0x5f80001d,
	0x5f81001d,0x5f82001d,0x5f83001d,0x5f84001d,0x5f85001d,0x5f86001d,0x5f87001d,0x5f88001d,
	0x5f89001d,0x5f8a001d,0x5f8b001d,0x5f8c001d,0x5f8d001d,0x5f8e001d,0x5f8f001d,0x5f90001d,
	0x5f91001d,0x5f92001d,0x5f93001d,0x5f94001d,0x5f95001d,0x5f96001d,0x5f97001d,0x5f98001d,
	0x5f99001d,0x5f9a001d,0x5f9b001d,0x5f9c001d,0x5f9d001d,0x5f9e001d,0x5f9f001d,0x5fa0001d,
	0x5fa1001d,0x5fa2001d,0x5fa3001d,0x5fa4001d,0x5fa5001d,0x5fa6001d,0x5fa7001d,0x5fa8001d,
	0x5fa9001d,0x5faa001d,0x5fab001d,0x5fac001d,0x5fad001d,0x5fae001d,0x5faf001d,0x5fb0001d,
	0x5fb1001d,0x5fb2001d,0x5fb3001d,0x5fb4001d,0x5fb5001d,0x5fb6001d,0x5fb7001d,0x5fb8001d,
	0x5fb9001d,0x5fba001d,0x5fbb001d,0x5fbc001d,0x5fbd001d,0x5fbe001d,0x5fbf001d,0x5fc0001d,
	0x5fc1001d,0x5fc2001d,0x5fc3001d,0x5fc4001d,0x5fc5001d,0x5fc6001d,0x5fc7001d,0x5fc8001d,
	0x5fc9001d,0x5fca001d,0x5fcb001d,0x5fcc001d,0x5fcd001d,0x5fce001d,0x5fcf001d,0x5fd0001d,
	0x5fd1001d,0x5fd2001d,0x5fd3001d,0x5fd4001d,0x5fd5001d,0x5fd6001d,0x5fd7001d,0x5fd8001d,
	0x5fd9001d,0x5fda001d,0x5fdb001d,0x5fdc001d,0x5fdd001d,0x5fde001d,0x5fdf001d,0x5fe0001d,
	0x5fe1001d,0x5fe2001d,0x5fe3001d,0x5fe4001d,0x5fe5001d,0x5fe6001d,0x5fe7001d,0x5fe8001d,
	0x5fe9001d,0x5fea001d,0x5feb001d,0x5fec001d,0x5fed001d,0x5fee001d,0x5fef001d,0x5ff0001d,
	0x5ff1001d,0x5ff2001d,0x5ff3001d,0x5ff4001d,0x5ff5001d,0x5ff6001d,0x5ff7001d,0x5ff8001d,
	0x5ff9001d,0x5ffa001d,0x5ffb001d,0x5ffc001d,0x5ffd001d,0x5ffe001d,0x5fff001d,0x6000001d,
	0x6001001d,0x6002001d,0x6003001d,0x6004001d,0x6005001d,0x6006001d,0x6007001d,0x6008001d,
	0x6009001d,0x600a001d,0x600b001d,0x600c001d,0x600d001d,0x600e001d,0x600f001d,0x6010001d,
	0x6011001d,0x6012001d,0x6013001d,0x6014001d,0x6015001d,0x6016001d,0x6017001d,0x6018001d,
	0x6019001d,0x601a001d,0x601b001d,0x601c001d,0x601d001d,0x601e001d,0x601f001d,0x6020001d,
	0x6021001d,0x6022001d,0x6023001d,0x6024001d,0x6025001d,0x6026001d,0x6027001d,0x6028001d,
	0x6029001d,0x602a001d,0x602b001d,0x602c001d,0x602d001d,0x602e001d,0x602f001d,0x6030001d,
	0x6031001d,0x6032001d,0x6033001d,0x6034001d,0x6035001d,0x6036001d,0x6037001d,0x6038001d,
	0x6039001d,0x603a001d,0x603b001d,0x603c001d,0x603d001d,0x603e001d,0x603f001d,0x6040001d,
	0x6041001d,0x6042001d,0x6043001d,0x6044001d,0x6045001d,0x6046001d,0x6047001d,0x6048001d,
	0x6049001d,0x604a001d,0x604b001d,0x604c001d,0x604d001d,0x604e001d,0x604f001d,0x6050001d,
	0x6051001d,0x6052001d,0x6053001d,0x6054001d,0x6055001d,0x6056001d,0x6057001d,0x6058001d,
	0x6059001d,0x605a001d,0x605b001d,0x605c001d,0x605d001d,0x605e001d,0x605f001d,0x6060001d,
	0x6061001d,0x6062001d,0x6063001d,0x6064001d,0x6065001d,0x6066001d,0x6067001d,0x6068001d,
	0x6069001d,0x606a001d,0x606b001d,0x606c001d,0x606d001d,0x606e001d,0x606f001d,0x6070001d,
	0x6071001d,0x6072001d,0x6073001d,0x6074001d,0x6075001d,0x6076001d,0x6077001d,0x6078001d,
	0x6079001d,0x607a001d,0x607b001d,0x607c001d,0x607d001d,0x607e001d,0x607f001d,0x6080001d,
	0x6081001d,0x6082001d,0x6083001d,0x6084001d,0x6085001d,0x6086001d,0x6087001d,0x6088001d,
	0x6089001d,0x608a001d,0x608b001d,0x608c001d,0x608d001d,0x608e001d,0x608f001d,0x6090001d,
	0x6091001d,0x6092001d,0x6093001d,0x6094001d,0x6095001d,0x6096001d,0x6097001d,0x6098001d,
	0x6099001d,0x609a001d,0x609b001d,0x609c001d,0x609d001d,0x609e001d,0x609f001d,0x60a0001d,
	0x60a1001d,0x60a2001d,0x60a3001d,0x60a4001d,0x60a5001d,0x60a6001d,0x60a7001d,0x60a8001d,
	0x60a9001d,0x60aa001d,0x60ab001d,0x60ac001d,0x60ad001d,0x60ae001d,0x60af001d,0x60b0001d,
	0x60b1001d,0x60b2001d,0x60b3001d,0x60b4001d,0x60b5001d,0x60b6001d,0x60b7001d,0x60b8001d,
	0x60b9001d,0x60ba001d,0x60bb001d,0x60bc001d,0x60bd001d,0x60be001d,0x60bf001d,0x60c0001d,
	0x60c1001d,0x60c2001d,0x60c3001d,0x60c4001d,0x60c5001d,0x60c6001d,0x60c7001d,0x60c8001d,
	0x60c9001d,0x60ca001d,0x60cb001d,0x60cc001d,0x60cd001d,0x60ce001d,0x60cf001d,0x60d0001d,
	0x60d1001d,0x60d2001d,0x60d3001d,0x60d4001d,0x60d5001d,0x60d6001d,0x60d7001d,0x60d8001d,
	0x60d9001d,0x60da001d,0x60db001d,0x60dc001d,0x60dd001d,0x60de001d,0x60df001d,0x60e0001d,
	0x60e1001d,0x60e2001d,0x60e3001d,0x60e4001d,0x60e5001d,0x60e6001d,0x60e7001d,0x60e8001d,
	0x60e9001d,0x60ea001d,0x60eb001d,0x60ec001d,0x60ed001d,0x60ee001d,0x60ef001d,0x60f0001d,
	0x60f1001d,0x60f2001d,0x60f3001d,0x60f4001d,0x60f5001d,0x60f6001d,0x60f7001d,0x60f8001d,
	0x60f9001d,0x60fa001d,0x60fb001d,0x60fc001d,0x60fd001d,0x60fe001d,0x60ff001d,0x6100001d,
	0x6101001d,0x6102001d,0x6103001d,0x6104001d,0x6105001d,0x6106001d,0x6107001d,0x6108001d,
	0x6109001d,0x610a001d,0x610b001d,0x610c001d,0x610d001d,0x610e001d,0x610f001d,0x6110001d,
	0x6111001d,0x6112001d,0x6113001d,0x6114001d,0x6115001d,0x6116001d,0x6117001d,0x6118001d,
	0x6119001d,0x611a001d,0x611b001d,0x611c001d,0x611d001d,0x611e001d,0x611f001d,0x6120001d,
	0x6121001d,0x6122001d,0x6123001d,0x6124001d,0x6125001d,0x6126001d,0x6127001d,0x6128001d,
	0x6129001d,0x612a001d,0x612b001d,0x612c001d,0x612d001d,0x612e001d,0x612f001d,0x6130001d,
	0x6131001d,0x6132001d,0x6133001d,0x6134001d,0x6135001d,0x6136001d,0x6137001d,0x6138001d,
	0x6139001d,0x613a001d,0x613b001d,0x613c001d,0x613d001d,0x613e001d,0x613f001d,0x6140001d,
	0x6141001d,0x6142001d,0x6143001d,0x6144001d,0x6145001d,0x6146001d,0x6147001d,0x6148001d,
	0x6149001d,0x614a001d,0x614b001d,0x614c001d,0x614d001d,0x614e001d,0x614f001d,0x6150001d,
	0x6151001d,0x6152001d,0x6153001d,0x6154001d,0x6155001d,0x6156001d,0x6157001d,0x6158001d,
	0x6159001d,0x615a001d,0x615b001d,0x615c001d,0x615d001d,0x615e001d,0x615f001d,0x6160001d,
	0x6161001d,0x6162001d,0x6163001d,0x6164001d,0x6165001d,0x6166001d,0x6167001d,0x6168001d,
	0x6169001d,0x616a001d,0x616b001d,0x616c001d,0x616d001d,0x616e001d,0x616f001d,0x6170001d,
	0x6171001d,0x6172001d,0x6173001d,0x6174001d,0x6175001d,0x6176001d,0x6177001d,0x6178001d,
	0x6179001d,0x617a001d,0x617b001d,0x617c001d,0x617d001d,0x617e001d,0x617f001d,0x6180001d,
	0x6181001d,0x6182001d,0x6183001d,0x6184001d,0x6185001d,0x6186001d,0x6187001d,0x6188001d,
	0x6189001d,0x618a001d,0x618b001d,0x618c001d,0x618d001d,0x618e001d,0x618f001d,0x6190001d,
	0x6191001d,0x6192001d,0x6193001d,0x6194001d,0x6195001d,0x6196001d,0x6197001d,0x6198001d,
	0x6199001d,0x619a001d,0x619b001d,0x619c001d,0x619d001d,0x619e001d,0x619f001d,0x61a0001d,
	0x61a1001d,0x61a2001d,0x61a3001d,0x61a4001d,0x61a5001d,0x61a6001d,0x61a7001d,0x61a8001d,
	0x61a9001d,0x61aa001d,0x61ab001d,0x61ac001d,0x61ad001d,0x61ae001d,0x61af001d,0x61b0001d,
	0x61b1001d,0x61b2001d,0x61b3001d,0x61b4001d,0x61b5001d,0x61b6001d,0x61b7001d,0x61b8001d,
	0x61b9001d,0x61ba001d,0x61bb001d,0x61bc001d,0x61bd001d,0x61be001d,0x61bf001d,0x61c0001d,
	0x61c1001d,0x61c2001d,0x61c3001d,0x61c4001d,0x61c5001d,0x61c6001d,0x61c7001d,0x61c8001d,
	0x61c9001d,0x61ca001d,0x61cb001d,0x61cc001d,0x61cd001d,0x61ce001d,0x61cf001d,0x61d0001d,
	0x61d1001d,0x61d2001d,0x61d3001d,0x61d4001d,0x61d5001d,0x61d6001d,0x61d7001d,0x61d8001d,
	0x61d9001d,0x61da001d,0x61db001d,0x61dc001d,0x61dd001d,0x61de001d,0x61df001d,0x61e0001d,
	0x61e1001d,0x61e2001d,0x61e3001d,0x61e4001d,0x61e5001d,0x61e6001d,0x61e7001d,0x61e8001d,
	0x61e9001d,0x61ea001d,0x61eb001d,0x61ec001d,0x61ed001d,0x61ee001d,0x61ef001d,0x61f0001d,
	0x61f1001d,0x61f2001d,0x61f3001d,0x61f4001d,0x61f5001d,0x61f6001d,0x61f7001d,0x61f8001d,
	0x61f9001d,0x61fa001d,0x61fb001d,0x61fc001d,0x61fd001d,0x61fe001d,0x61ff001d,0x6200001d,
	0x6201001d,0x6202001d,0x6203001d,0x6204001d,0x6205001d,0x6206001d,0x6207001d,0x6208001d,
	0x6209001d,0x620a001d,0x620b001d,0x620c001d,0x620d001d,0x620e001d,0x620f001d,0x6210001d,
	0x6211001d,0x6212001d,0x6213001d,0x6214001d,0x6215001d,0x6216001d,0x6217001d,0x6218001d,
	0x6219001d,0x621a001d,0x621b001d,0x621c001d,0x621d001d,0x621e001d,0x621f001d,0x6220001d,
	0x6221001d,0x6222001d,0x6223001d,0x6224001d,0x6225001d,0x6226001d,0x6227001d,0x6228001d,
	0x6229001d,0x622a001d,0x622b001d,0x622c001d,0x622d001d,0x622e001d,0x622f001d,0x6230001d,
	0x6231001d,0x6232001d,0x6233001d,0x6234001d,0x6235001d,0x6236001d,0x6237001d,0x6238001d,
	0x6239001d,0x623a001d,0x623b001d,0x623c001d,0x623d001d,0x623e001d,0x623f001d,0x6240001d,
	0x6241001d,0x6242001d,0x6243001d,0x6244001d,0x6245001d,0x6246001d,0x6247001d,0x6248001d,
	0x6249001d,0x624a001d,0x624b001d,0x624c001d,0x624d001d,0x624e001d,0x624f001d,0x6250001d,
	0x6251001d,0x6252001d,0x6253001d,0x6254001d,0x6255001d,0x6256001d,0x6257001d,0x6258001d,
	0x6259001d,0x625a001d,0x625b001d,0x625c001d,0x625d001d,0x625e001d,0x625f001d,0x6260001d,
	0x6261001d,0x6262001d,0x6263001d,0x6264001d,0x6265001d,0x6266001d,0x6267001d,0x6268001d,
	0x6269001d,0x626a001d,0x626b001d,0x626c001d,0x626d001d,0x626e001d,0x626f001d,0x6270001d,
	0x6271001d,0x6272001d,0x6273001d,0x6274001d,0x6275001d,0x6276001d,0x6277001d,0x6278001d,
	0x6279001d,0x627a001d,0x627b001d,0x627c001d,0x627d001d,0x627e001d,0x627f001d,0x6280001d,
	0x6281001d,0x6282001d,0x6283001d,0x6284001d,0x6285001d,0x6286001d,0x6287001d,0x6288001d,
	0x6289001d,0x628a001d,0x628b001d,0x628c001d,0x628d001d,0x628e001d,0x628f001d,0x6290001d,
	0x6291001d,0x6292001d,0x6293001d,0x6294001d,0x6295001d,0x6296001d,0x6297001d,0x6298001d,
	0x6299001d,0x629a001d,0x629b001d,0x629c001d,0x629d001d,0x629e001d,0x629f001d,0x62a0001d,
	0x62a1001d,0x62a2001d,0x62a3001d,0x62a4001d,0x62a5001d,0x62a6001d,0x62a7001d,0x62a8001d,
	0x62a9001d,0x62aa001d,0x62ab001d,0x62ac001d,0x62ad001d,0x62ae001d,0x62af001d,0x62b0001d,
	0x62b1001d,0x62b2001d,0x62b3001d,0x62b4001d,0x62b5001d,0x62b6001d,0x62b7001d,0x62b8001d,
	0x62b9001d,0x62ba001d,0x62bb001d,0x62bc001d,0x62bd001d,0x62be001d,0x62bf001d,0x62c0001d,
	0x62c1001d,0x62c2001d,0x62c3001d,0x62c4001d,0x62c5001d,0x62c6001d,0x62c7001d,0x62c8001d,
	0x62c9001d,0x62ca001d,0x62cb001d,0x62cc001d,0x62cd001d,0x62ce001d,0x62cf001d,0x62d0001d,
	0x62d1001d,0x62d2001d,0x62d3001d,0x62d4001d,0x62d5001d,0x62d6001d,0x62d7001d,0x62d8001d,
	0x62d9001d,0x62da001d,0x62db001d,0x62dc001d,0x62dd001d,0x62de001d,0x62df001d,0x62e0001d,
	0x62e1001d,0x62e2001d,0x62e3001d,0x62e4001d,0x62e5001d,0x62e6001d,0x62e7001d,0x62e8001d,
	0x62e9001d,0x62ea001d,0x62eb001d,0x62ec001d,0x62ed001d,0x62ee001d,0x62ef001d,0x62f0001d,
	0x62f1001d,0x62f2001d,0x62f3001d,0x62f4001d,0x62f5001d,0x62f6001d,0x62f7001d,0x62f8001d,
	0x62f9001d,0x62fa001d,0x62fb001d,0x62fc001d,0x62fd001d,0x62fe001d,0x62ff001d,0x6300001d,
	0x6301001d,0x6302001d,0x6303001d,0x6304001d,0x6305001d,0x6306001d,0x6307001d,0x6308001d,
	0x6309001d,0x630a001d,0x630b001d,0x630c001d,0x630d001d,0x630e001d,0x630f001d,0x6310001d,
	0x6311001d,0x6312001d,0x6313001d,0x6314001d,0x6315001d,0x6316001d,0x6317001d,0x6318001d,
	0x6319001d,0x631a001d,0x631b001d,0x631c001d,0x631d001d,0x631e001d,0x631f001d,0x6320001d,
	0x6321001d,0x6322001d,0x6323001d,0x6324001d,0x6325001d,0x6326001d,0x6327001d,0x6328001d,
	0x6329001d,0x632a001d,0x632b001d,0x632c001d,0x632d001d,0x632e001d,0x632f001d,0x6330001d,
	0x6331001d,0x6332001d,0x6333001d,0x6334001d,0x6335001d,0x6336001d,0x6337001d,0x6338001d,
	0x6339001d,0x633a001d,0x633b001d,0x633c001d,0x633d001d,0x633e001d,0x633f001d,0x6340001d,
	0x6341001d,0x6342001d,0x6343001d,0x6344001d,0x6345001d,0x6346001d,0x6347001d,0x6348001d,
	0x6349001d,0x634a001d,0x634b001d,0x634c001d,0x634d001d,0x634e001d,0x634f001d,0x6350001d,
	0x6351001d,0x6352001d,0x6353001d,0x6354001d,0x6355001d,0x6356001d,0x6357001d,0x6358001d,
	0x6359001d,0x635a001d,0x635b001d,0x635c001d,0x635d001d,0x635e001d,0x635f001d,0x6360001d,
	0x6361001d,0x6362001d,0x6363001d,0x6364001d,0x6365001d,0x6366001d,0x6367001d,0x6368001d,
	0x6369001d,0x636a001d,0x636b001d,0x636c001d,0x636d001d,0x636e001d,0x636f001d,0x6370001d,
	0x6371001d,0x6372001d,0x6373001d,0x6374001d,0x6375001d,0x6376001d,0x6377001d,0x6378001d,
	0x6379001d,0x637a001d,0x637b001d,0x637c001d,0x637d001d,0x637e001d,0x637f001d,0x6380001d,
	0x6381001d,0x6382001d,0x6383001d,0x6384001d,0x6385001d,0x6386001d,0x6387001d,0x6388001d,
	0x6389001d,0x638a001d,0x638b001d,0x638c001d,0x638d001d,0x638e001d,0x638f001d,0x6390001d,
	0x6391001d,0x6392001d,0x6393001d,0x6394001d,0x6395001d,0x6396001d,0x6397001d,0x6398001d,
	0x6399001d,0x639a001d,0x639b001d,0x639c001d,0x639d001d,0x639e001d,0x639f001d,0x63a0001d,
	0x63a1001d,0x63a2001d,0x63a3001d,0x63a4001d,0x63a5001d,0x63a6001d,0x63a7001d,0x63a8001d,
	0x63a9001d,0x63aa001d,0x63ab001d,0x63ac001d,0x63ad001d,0x63ae001d,0x63af001d,0x63b0001d,
	0x63b1001d,0x63b2001d,0x63b3001d,0x63b4001d,0x63b5001d,0x63b6001d,0x63b7001d,0x63b8001d,
	0x63b9001d,0x63ba001d,0x63bb001d,0x63bc001d,0x63bd001d,0x63be001d,0x63bf001d,0x63c0001d,
	0x63c1001d,0x63c2001d,0x63c3001d,0x63c4001d,0x63c5001d,0x63c6001d,0x63c7001d,0x63c8001d,
	0x63c9001d,0x63ca001d,0x63cb001d,0x63cc001d,0x63cd001d,0x63ce001d,0x63cf001d,0x63d0001d,
	0x63d1001d,0x63d2001d,0x63d3001d,0x63d4001d,0x63d5001d,0x63d6001d,0x63d7001d,0x63d8001d,
	0x63d9001d,0x63da001d,0x63db001d,0x63dc001d,0x63dd001d,0x63de001d,0x63df001d,0x63e0001d,
	0x63e1001d,0x63e2001d,0x63e3001d,0x63e4001d,0x63e5001d,0x63e6001d,0x63e7001d,0x63e8001d,
	0x63e9001d,0x63ea001d,0x63eb001d,0x63ec001d,0x63ed001d,0x63ee001d,0x63ef001d,0x63f0001d,
	0x63f1001d,0x63f2001d,0x63f3001d,0x63f4001d,0x63f5001d,0x63f6001d,0x63f7001d,0x63f8001d,
	0x63f9001d,0x63fa001d,0x63fb001d,0x63fc001d,0x63fd001d,0x63fe001d,0x63ff001d,0x6400001d,
	0x6401001d,0x6402001d,0x6403001d,0x6404001d,0x6405001d,0x6406001d,0x6407001d,0x6408001d,
	0x6409001d,0x640a001d,0x640b001d,0x640c001d,0x640d001d,0x640e001d,0x640f001d,0x6410001d,
	0x6411001d,0x6412001d,0x6413001d,0x6414001d,0x6415001d,0x6416001d,0x6417001d,0x6418001d,
	0x6419001d,0x641a001d,0x641b001d,0x641c001d,0x641d001d,0x641e001d,0x641f001d,0x6420001d,
	0x6421001d,0x6422001d,0x6423001d,0x6424001d,0x6425001d,0x6426001d,0x6427001d,0x6428001d,
	0x6429001d,0x642a001d,0x642b001d,0x642c001d,0x642d001d,0x642e001d,0x642f001d,0x6430001d,
	0x6431001d,0x6432001d,0x6433001d,0x6434001d,0x6435001d,0x6436001d,0x6437001d,0x6438001d,
	0x6439001d,0x643a001d,0x643b001d,0x643c001d,0x643d001d,0x643e001d,0x643f001d,0x6440001d,
	0x6441001d,0x6442001d,0x6443001d,0x6444001d,0x6445001d,0x6446001d,0x6447001d,0x6448001d,
	0x6449001d,0x644a001d,0x644b001d,0x644c001d,0x644d001d,0x644e001d,0x644f001d,0x6450001d,
	0x6451001d,0x6452001d,0x6453001d,0x6454001d,0x6455001d,0x6456001d,0x6457001d,0x6458001d,
	0x6459001d,0x645a001d,0x645b001d,0x645c001d,0x645d001d,0x645e001d,0x645f001d,0x6460001d,
	0x6461001d,0x6462001d,0x6463001d,0x6464001d,0x6465001d,0x6466001d,0x6467001d,0x6468001d,
	0x6469001d,0x646a001d,0x646b001d,0x646c001d,0x646d001d,0x646e001d,0x646f001d,0x6470001d,
	0x6471001d,0x6472001d,0x6473001d,0x6474001d,0x6475001d,0x6476001d,0x6477001d,0x6478001d,
	0x6479001d,0x647a001d,0x647b001d,0x647c001d,0x647d001d,0x647e001d,0x647f001d,0x6480001d,
	0x6481001d,0x6482001d,0x6483001d,0x6484001d,0x6485001d,0x6486001d,0x6487001d,0x6488001d,
	0x6489001d,0x648a001d,0x648b001d,0x648c001d,0x648d001d,0x648e001d,0x648f001d,0x6490001d,
	0x6491001d,0x6492001d,0x6493001d,0x6494001d,0x6495001d,0x6496001d,0x6497001d,0x6498001d,
	0x6499001d,0x649a001d,0x649b001d,0x649c001d,0x649d001d,0x649e001d,0x649f001d,0x64a0001d,
	0x64a1001d,0x64a2001d,0x64a3001d,0x64a4001d,0x64a5001d,0x64a6001d,0x64a7001d,0x64a8001d,
	0x64a9001d,0x64aa001d,0x64ab001d,0x64ac001d,0x64ad001d,0x64ae001d,0x64af001d,0x64b0001d,
	0x64b1001d,0x64b2001d,0x64b3001d,0x64b4001d,0x64b5001d,0x64b6001d,0x64b7001d,0x64b8001d,
	0x64b9001d,0x64ba001d,0x64bb001d,0x64bc001d,0x64bd001d,0x64be001d,0x64bf001d,0x64c0001d,
	0x64c1001d,0x64c2001d,0x64c3001d,0x64c4001d,0x64c5001d,0x64c6001d,0x64c7001d,0x64c8001d,
	0x64c9001d,0x64ca001d,0x64cb001d,0x64cc001d,0x64cd001d,0x64ce001d,0x64cf001d,0x64d0001d,
	0x64d1001d,0x64d2001d,0x64d3001d,0x64d4001d,0x64d5001d,0x64d6001d,0x64d7001d,0x64d8001d,
	0x64d9001d,0x64da001d,0x64db001d,0x64dc001d,0x64dd001d,0x64de001d,0x64df001d,0x64e0001d,
	0x64e1001d,0x64e2001d,0x64e3001d,0x64e4001d,0x64e5001d,0x64e6001d,0x64e7001d,0x64e8001d,
	0x64e9001d,0x64ea001d,0x64eb001d,0x64ec001d,0x64ed001d,0x64ee001d,0x64ef001d,0x64f0001d,
	0x64f1001d,0x64f2001d,0x64f3001d,0x64f4001d,0x64f5001d,0x64f6001d,0x64f7001d,0x64f8001d,
	0x64f9001d,0x64fa001d,0x64fb001d,0x64fc001d,0x64fd001d,0x64fe001d,0x64ff001d,0x6500001d,
	0x6501001d,0x6502001d,0x6503001d,0x6504001d,0x6505001d,0x6506001d,0x6507001d,0x6508001d,
	0x6509001d,0x650a001d,0x650b001d,0x650c001d,0x650d001d,0x650e001d,0x650f001d,0x6510001d,
	0x6511001d,0x6512001d,0x6513001d,0x6514001d,0x6515001d,0x6516001d,0x6517001d,0x6518001d,
	0x6519001d,0x651a001d,0x651b001d,0x651c001d,0x651d001d,0x651e001d,0x651f001d,0x6520001d,
	0x6521001d,0x6522001d,0x6523001d,0x6524001d,0x6525001d,0x6526001d,0x6527001d,0x6528001d,
	0x6529001d,0x652a001d,0x652b001d,0x652c001d,0x652d001d,0x652e001d,0x652f001d,0x6530001d,
	0x6531001d,0x6532001d,0x6533001d,0x6534001d,0x6535001d,0x6536001d,0x6537001d,0x6538001d,
	0x6539001d,0x653a001d,0x653b001d,0x653c001d,0x653d001d,0x653e001d,0x653f001d,0x6540001d,
	0x6541001d,0x6542001d,0x6543001d,0x6544001d,0x6545001d,0x6546001d,0x6547001d,0x6548001d,
	0x6549001d,0x654a001d,0x654b001d,0x654c001d,0x654d001d,0x654e001d,0x654f001d,0x6550001d,
	0x6551001d,0x6552001d,0x6553001d,0x6554001d,0x6555001d,0x6556001d,0x6557001d,0x6558001d,
	0x6559001d,0x655a001d,0x655b001d,0x655c001d,0x655d001d,0x655e001d,0x655f001d,0x6560001d,
	0x6561001d,0x6562001d,0x6563001d,0x6564001d,0x6565001d,0x6566001d,0x6567001d,0x6568001d,
	0x6569001d,0x656a001d,0x656b001d,0x656c001d,0x656d001d,0x656e001d,0x656f001d,0x6570001d,
	0x6571001d,0x6572001d,0x6573001d,0x6574001d,0x6575001d,0x6576001d,0x6577001d,0x6578001d,
	0x6579001d,0x657a001d,0x657b001d,0x657c001d,0x657d001d,0x657e001d,0x657f001d,0x6580001d,
	0x6581001d,0x6582001d,0x6583001d,0x6584001d,0x6585001d,0x6586001d,0x6587001d,0x6588001d,
	0x6589001d,0x658a001d,0x658b001d,0x658c001d,0x658d001d,0x658e001d,0x658f001d,0x6590001d,
	0x6591001d,0x6592001d,0x6593001d,0x6594001d,0x6595001d,0x6596001d,0x6597001d,0x6598001d,
	0x6599001d,0x659a001d,0x659b001d,0x659c001d,0x659d001d,0x659e001d,0x659f001d,0x65a0001d,
	0x65a1001d,0x65a2001d,0x65a3001d,0x65a4001d,0x65a5001d,0x65a6001d,0x65a7001d,0x65a8001d,
	0x65a9001d,0x65aa001d,0x65ab001d,0x65ac001d,0x65ad001d,0x65ae001d,0x65af001d,0x65b0001d,
	0x65b1001d,0x65b2001d,0x65b3001d,0x65b4001d,0x65b5001d,0x65b6001d,0x65b7001d,0x65b8001d,
	0x65b9001d,0x65ba001d,0x65bb001d,0x65bc001d,0x65bd001d,0x65be001d,0x65bf001d,0x65c0001d,
	0x65c1001d,0x65c2001d,0x65c3001d,0x65c4001d,0x65c5001d,0x65c6001d,0x65c7001d,0x65c8001d,
	0x65c9001d,0x65ca001d,0x65cb001d,0x65cc001d,0x65cd001d,0x65ce001d,0x65cf001d,0x65d0001d,
	0x65d1001d,0x65d2001d,0x65d3001d,0x65d4001d,0x65d5001d,0x65d6001d,0x65d7001d,0x65d8001d,
	0x65d9001d,0x65da001d,0x65db001d,0x65dc001d,0x65dd001d,0x65de001d,0x65df001d,0x65e0001d,
	0x65e1001d,0x65e2001d,0x65e3001d,0x65e4001d,0x65e5001d,0x65e6001d,0x65e7001d,0x65e8001d,
	0x65e9001d,0x65ea001d,0x65eb001d,0x65ec001d,0x65ed001d,0x65ee001d,0x65ef001d,0x65f0001d,
	0x65f1001d,0x65f2001d,0x65f3001d,0x65f4001d,0x65f5001d,0x65f6001d,0x65f7001d,0x65f8001d,
	0x65f9001d,0x65fa001d,0x65fb001d,0x65fc001d,0x65fd001d,0x65fe001d,0x65ff001d,0x6600001d,
	0x6601001d,0x6602001d,0x6603001d,0x6604001d,0x6605001d,0x6606001d,0x6607001d,0x6608001d,
	0x6609001d,0x660a001d,0x660b001d,0x660c001d,0x660d001d,0x660e001d,0x660f001d,0x6610001d,
	0x6611001d,0x6612001d,0x6613001d,0x6614001d,0x6615001d,0x6616001d,0x6617001d,0x6618001d,
	0x6619001d,0x661a001d,0x661b001d,0x661c001d,0x661d001d,0x661e001d,0x661f001d,0x6620001d,
	0x6621001d,0x6622001d,0x6623001d,0x6624001d,0x6625001d,0x6626001d,0x6627001d,0x6628001d,
	0x6629001d,0x662a001d,0x662b001d,0x662c001d,0x662d001d,0x662e001d,0x662f001d,0x6630001d,
	0x6631001d,0x6632001d,0x6633001d,0x6634001d,0x6635001d,0x6636001d,0x6637001d,0x6638001d,
	0x6639001d,0x663a001d,0x663b001d,0x663c001d,0x663d001d,0x663e001d,0x663f001d,0x6640001d,
	0x6641001d,0x6642001d,0x6643001d,0x6644001d,0x6645001d,0x6646001d,0x6647001d,0x6648001d,
	0x6649001d,0x664a001d,0x664b001d,0x664c001d,0x664d001d,0x664e001d,0x664f001d,0x6650001d,
	0x6651001d,0x6652001d,0x6653001d,0x6654001d,0x6655001d,0x6656001d,0x6657001d,0x6658001d,
	0x6659001d,0x665a001d,0x665b001d,0x665c001d,0x665d001d,0x665e001d,0x665f001d,0x6660001d,
	0x6661001d,0x6662001d,0x6663001d,0x6664001d,0x6665001d,0x6666001d,0x6667001d,0x6668001d,
	0x6669001d,0x666a001d,0x666b001d,0x666c001d,0x666d001d,0x666e001d,0x666f001d,0x6670001d,
	0x6671001d,0x6672001d,0x6673001d,0x6674001d,0x6675001d,0x6676001d,0x6677001d,0x6678001d,
	0x6679001d,0x667a001d,0x667b001d,0x667c001d,0x667d001d,0x667e001d,0x667f001d,0x6680001d,
	0x6681001d,0x6682001d,0x6683001d,0x6684001d,0x6685001d,0x6686001d,0x6687001d,0x6688001d,
	0x6689001d,0x668a001d,0x668b001d,0x668c001d,0x668d001d,0x668e001d,0x668f001d,0x6690001d,
	0x6691001d,0x6692001d,0x6693001d,0x6694001d,0x6695001d,0x6696001d,0x6697001d,0x6698001d,
	0x6699001d,0x669a001d,0x669b001d,0x669c001d,0x669d001d,0x669e001d,0x669f001d,0x66a0001d,
	0x66a1001d,0x66a2001d,0x66a3001d,0x66a4001d,0x66a5001d,0x66a6001d,0x66a7001d,0x66a8001d,
	0x66a9001d,0x66aa001d,0x66ab001d,0x66ac001d,0x66ad001d,0x66ae001d,0x66af001d,0x66b0001d,
	0x66b1001d,0x66b2001d,0x66b3001d,0x66b4001d,0x66b5001d,0x66b6001d,0x66b7001d,0x66b8001d,
	0x66b9001d,0x66ba001d,0x66bb001d,0x66bc001d,0x66bd001d,0x66be001d,0x66bf001d,0x66c0001d,
	0x66c1001d,0x66c2001d,0x66c3001d,0x66c4001d,0x66c5001d,0x66c6001d,0x66c7001d,0x66c8001d,
	0x66c9001d,0x66ca001d,0x66cb001d,0x66cc001d,0x66cd001d,0x66ce001d,0x66cf001d,0x66d0001d,
	0x66d1001d,0x66d2001d,0x66d3001d,0x66d4001d,0x66d5001d,0x66d6001d,0x66d7001d,0x66d8001d,
	0x66d9001d,0x66da001d,0x66db001d,0x66dc001d,0x66dd001d,0x66de001d,0x66df001d,0x66e0001d,
	0x66e1001d,0x66e2001d,0x66e3001d,0x66e4001d,0x66e5001d,0x66e6001d,0x66e7001d,0x66e8001d,
	0x66e9001d,0x66ea001d,0x66eb001d,0x66ec001d,0x66ed001d,0x66ee001d,0x66ef001d,0x66f0001d,
	0x66f1001d,0x66f2001d,0x66f3001d,0x66f4001d,0x66f5001d,0x66f6001d,0x66f7001d,0x66f8001d,
	0x66f9001d,0x66fa001d,0x66fb001d,0x66fc001d,0x66fd001d,0x66fe001d,0x66ff001d,0x6700001d,
	0x6701001d,0x6702001d,0x6703001d,0x6704001d,0x6705001d,0x6706001d,0x6707001d,0x6708001d,
	0x6709001d,0x670a001d,0x670b001d,0x670c001d,0x670d001d,0x670e001d,0x670f001d,0x6710001d,
	0x6711001d,0x6712001d,0x6713001d,0x6714001d,0x6715001d,0x6716001d,0x6717001d,0x6718001d,
	0x6719001d,0x671a001d,0x671b001d,0x671c001d,0x671d001d,0x671e001d,0x671f001d,0x6720001d,
	0x6721001d,0x6722001d,0x6723001d,0x6724001d,0x6725001d,0x6726001d,0x6727001d,0x6728001d,
	0x6729001d,0x672a001d,0x672b001d,0x672c001d,0x672d001d,0x672e001d,0x672f001d,0x6730001d,
	0x6731001d,0x6732001d,0x6733001d,0x6734001d,0x6735001d,0x6736001d,0x6737001d,0x6738001d,
	0x6739001d,0x673a001d,0x673b001d,0x673c001d,0x673d001d,0x673e001d,0x673f001d,0x6740001d,
	0x6741001d,0x6742001d,0x6743001d,0x6744001d,0x6745001d,0x6746001d,0x6747001d,0x6748001d,
	0x6749001d,0x674a001d,0x674b001d,0x674c001d,0x674d001d,0x674e001d,0x674f001d,0x6750001d,
	0x6751001d,0x6752001d,0x6753001d,0x6754001d,0x6755001d,0x6756001d,0x6757001d,0x6758001d,
	0x6759001d,0x675a001d,0x675b001d,0x675c001d,0x675d001d,0x675e001d,0x675f001d,0x6760001d,
	0x6761001d,0x6762001d,0x6763001d,0x6764001d,0x6765001d,0x6766001d,0x6767001d,0x6768001d,
	0x6769001d,0x676a001d,0x676b001d,0x676c001d,0x676d001d,0x676e001d,0x676f001d,0x6770001d,
	0x6771001d,0x6772001d,0x6773001d,0x6774001d,0x6775001d,0x6776001d,0x6777001d,0x6778001d,
	0x6779001d,0x677a001d,0x677b001d,0x677c001d,0x677d001d,0x677e001d,0x677f001d,0x6780001d,
	0x6781001d,0x6782001d,0x6783001d,0x6784001d,0x6785001d,0x6786001d,0x6787001d,0x6788001d,
	0x6789001d,0x678a001d,0x678b001d,0x678c001d,0x678d001d,0x678e001d,0x678f001d,0x6790001d,
	0x6791001d,0x6792001d,0x6793001d,0x6794001d,0x6795001d,0x6796001d,0x6797001d,0x6798001d,
	0x6799001d,0x679a001d,0x679b001d,0x679c001d,0x679d001d,0x679e001d,0x679f001d,0x67a0001d,
	0x67a1001d,0x67a2001d,0x67a3001d,0x67a4001d,0x67a5001d,0x67a6001d,0x67a7001d,0x67a8001d,
	0x67a9001d,0x67aa001d,0x67ab001d,0x67ac001d,0x67ad001d,0x67ae001d,0x67af001d,0x67b0001d,
	0x67b1001d,0x67b2001d,0x67b3001d,0x67b4001d,0x67b5001d,0x67b6001d,0x67b7001d,0x67b8001d,
	0x67b9001d,0x67ba001d,0x67bb001d,0x67bc001d,0x67bd001d,0x67be001d,0x67bf001d,0x67c0001d,
	0x67c1001d,0x67c2001d,0x67c3001d,0x67c4001d,0x67c5001d,0x67c6001d,0x67c7001d,0x67c8001d,
	0x67c9001d,0x67ca001d,0x67cb001d,0x67cc001d,0x67cd001d,0x67ce001d,0x67cf001d,0x67d0001d,
	0x67d1001d,0x67d2001d,0x67d3001d,0x67d4001d,0x67d5001d,0x67d6001d,0x67d7001d,0x67d8001d,
	0x67d9001d,0x67da001d,0x67db001d,0x67dc001d,0x67dd001d,0x67de001d,0x67df001d,0x67e0001d,
	0x67e1001d,0x67e2001d,0x67e3001d,0x67e4001d,0x67e5001d,0x67e6001d,0x67e7001d,0x67e8001d,
	0x67e9001d,0x67ea001d,0x67eb001d,0x67ec001d,0x67ed001d,0x67ee001d,0x67ef001d,0x67f0001d,
	0x67f1001d,0x67f2001d,0x67f3001d,0x67f4001d,0x67f5001d,0x67f6001d,0x67f7001d,0x67f8001d,
	0x67f9001d,0x67fa001d,0x67fb001d,0x67fc001d,0x67fd001d,0x67fe001d,0x67ff001d,0x6800001d,
	0x6801001d,0x6802001d,0x6803001d,0x6804001d,0x6805001d,0x6806001d,0x6807001d,0x6808001d,
	0x6809001d,0x680a001d,0x680b001d,0x680c001d,0x680d001d,0x680e001d,0x680f001d,0x6810001d,
	0x6811001d,0x6812001d,0x6813001d,0x6814001d,0x6815001d,0x6816001d,0x6817001d,0x6818001d,
	0x6819001d,0x681a001d,0x681b001d,0x681c001d,0x681d001d,0x681e001d,0x681f001d,0x6820001d,
	0x6821001d,0x6822001d,0x6823001d,0x6824001d,0x6825001d,0x6826001d,0x6827001d,0x6828001d,
	0x6829001d,0x682a001d,0x682b001d,0x682c001d,0x682d001d,0x682e001d,0x682f001d,0x6830001d,
	0x6831001d,0x6832001d,0x6833001d,0x6834001d,0x6835001d,0x6836001d,0x6837001d,0x6838001d,
	0x6839001d,0x683a001d,0x683b001d,0x683c001d,0x683d001d,0x683e001d,0x683f001d,0x6840001d,
	0x6841001d,0x6842001d,0x6843001d,0x6844001d,0x6845001d,0x6846001d,0x6847001d,0x6848001d,
	0x6849001d,0x684a001d,0x684b001d,0x684c001d,0x684d001d,0x684e001d,0x684f001d,0x6850001d,
	0x6851001d,0x6852001d,0x6853001d,0x6854001d,0x6855001d,0x6856001d,0x6857001d,0x6858001d,
	0x6859001d,0x685a001d,0x685b001d,0x685c001d,0x685d001d,0x685e001d,0x685f001d,0x6860001d,
	0x6861001d,0x6862001d,0x6863001d,0x6864001d,0x6865001d,0x6866001d,0x6867001d,0x6868001d,
	0x6869001d,0x686a001d,0x686b001d,0x21e010f,0x234010f,0x269010f,0x2c4010d,0x26a010f,
	0x268010f,0x22d010f,0x23e010f,0x23f010f,0x263010f,0x369010f,0x214010f,0x20c010f,
	0x226010f,0x265010f,0x6c5010d,0x6c6010d,0x6c7010d,0x6c8010d,0x6c9010d,0x6ca010d,
	0x6cb010d,0x6cc010d,0x6cd010d,0x6ce010d,0x21c010f,0x21a010f,0x36d010f,0x36e010f,
	0x36f010f,0x221010f,0x262010f,0x6cf0125,0x6e30125,0x6f70125,0x7030125,0x71f0125,
	0x74b0125,0x7530125,0x7730125,0x78d0125,0x7a10125,0x7b10125,0x7bd0125,0x7e50125,
	0x7ed0125,0x8050125,0x81d0125,0x8290125,0x8350125,0x85d0125,0x8750125,0x88d0125,
	0x8a50125,0x8b10125,0x8b90125,0x8bd0125,0x8c90125,0x240010f,0x241010f,0x2091d0f,
	0x209020f,0x209160f,0x6cf010d,0x6e3010d,0x6f7010d,0x703010d,0x71f010d,0x74b010d,
	0x753010d,0x773010d,0x78d010d,0x7a1010d,0x7b1010d,0x7bd010d,0x7e5010d,0x7ed010d,
	0x805010d,0x81d010d,0x829010d,0x835010d,0x85d010d,0x875010d,0x88d010d,0x8a5010d,
	0x8b1010d,0x8b9010d,0x8bd010d,0x8c9010d,0x242010f,0x371010f,0x243010f,0x374010f,
	0x228010b,0x219010b,0x22e010b,0x22f010b,0x235010b,
	};

static const TUint32 TheChineseIndex[] = 
	{
	0x20181aca,0x20191acb,0x201c1acc,0x30011ac9,0x30021ac8,0x4e0016ba,0x4e0103eb,0x4e030f2d,
	0x4e0713de,0x4e0818d0,0x4e0910e0,0x4e0a113a,0x4e0b14db,0x4e0c0f3c,0x4e0d0176,0x4e0e17d4,
	0x4e100575,0x4e110284,0x4e1319d8,0x4e140fd3,0x4e150ea7,0x4e1611b0,0x4e181014,0x4e190143,
	0x4e1a16ad,0x4e1b02ea,0x4e1c03ff,0x4e1d122e,0x4e1e0232,0x4e2203fd,0x4e240b87,0x4e25163c,
	0x4e2710e8,0x4e28067b,0x4e2a05cd,0x4e2b160f,0x4e2c0e65,0x4e2d197e,0x4e3004fe,0x4e3202af,
	0x4e340bb1,0x4e3619bb,0x4e3813c9,0x4e390348,0x4e3a13fb,0x4e3b19bc,0x4e3d0b3f,0x4e3e098c,
	0x4e3f0ee9,0x4e430dab,0x4e450968,0x4e47139f,0x4e480ccf,0x4e4916ea,0x4e4b1930,0x4e4c144e,
	0x4e4d189f,0x4e4e071f,0x4e4f049b,0x4e500af8,0x4e520ef6,0x4e530e6e,0x4e540fbe,0x4e560642,
	0x4e580239,0x4e5916df,0x4e5c0d41,0x4e5d0967,0x4e5e0f5b,0x4e5f16aa,0x4e6014b3,0x4e61150d,
	0x4e6611dd,0x4e6907db,0x4e700c92,0x4e710c49,0x4e7310bb,0x4e7e0f8f,0x4e860afd,0x4e8817b4,
	0x4e89191c,0x4e8b11b5,0x4e8c0498,0x4e8d029b,0x4e8e17b3,0x4e8f0a7e,0x4e911836,0x4e92073d,
	0x4e930f3d,0x4e941460,0x4e950941,0x4e9805d6,0x4e9a1623,0x4e9b1547,0x4e9f0801,0x4ea01377,
	0x4ea113e1,0x4ea20a08,0x4ea408ba,0x4ea5069c,0x4ea616f1,0x4ea701e1,0x4ea806f7,0x4ea90d8c,
	0x4eab151d,0x4eac0932,0x4ead1355,0x4eae0b89,0x4eb20fdd,0x4eb3015b,0x4eb5155e,0x4eba1086,
	0x4ebb1087,0x4ebf16eb,0x4ec0116f,0x4ec11088,0x4ec20af7,0x4ec303ec,0x4ec41883,0x4ec5091b,
	0x4ec60f14,0x4ec70278,0x4ec918cd,0x4eca0910,0x4ecb0904,0x4ecd1098,0x4ece02e9,0x4ed10c4e,
	0x4ed30198,0x4ed41a0c,0x4ed511b1,0x4ed612ae,0x4ed718d1,0x4ed80554,0x4ed914df,0x4edd1363,
	0x4ede108f,0x4edf0f78,0x4ee105bd,0x4ee30337,0x4ee40bdc,0x4ee516e1,0x4ee810d4,0x4eea16c7,
	0x4eeb0d91,0x4eec0cee,0x4ef0167d,0x4ef2198b,0x4ef30ec3,0x4ef51462,0x4ef6088e,0x4ef7085b,
	0x4efb1090,0x4efd04f6,0x4eff04c8,0x4f010f5c,0x4f090a09,0x4f0a16bc,0x4f0d1463,0x4f0e081d,
	0x4f0f0520,0x4f10049c,0x4f11159b,0x4f17198c,0x4f181784,0x4f1907c9,0x4f1a07a6,0x4f1b17d5,
	0x4f1e10e3,0x4f1f140b,0x4f2002a8,0x4f221617,0x4f241131,0x4f2501eb,0x4f260c4f,0x4f270199,
	0x4f2a140c,0x4f2b19c3,0x4f2f0156,0x4f30060d,0x4f320de1,0x4f340078,0x4f360bc6,0x4f381164,
	0x4f3a1241,0x4f3c1242,0x4f3d083c,0x4f4303c0,0x4f460358,0x4f4d141d,0x4f4e0390,0x4f4f19c4,
	0x4f501a60,0x4f5117a5,0x4f531325,0x4f5506dc,0x4f5713a5,0x4f581155,0x4f5917b5,0x4f5a16f4,
	0x4f5b0513,0x4f5c1a61,0x4f5d05f9,0x4f5e0e16,0x4f5f1365,0x4f600de2,0x4f63176b,0x4f6413be,
	0x4f650f7d,0x4f6709e6,0x4f690e8c,0x4f6c0aee,0x4f6f1677,0x4f700062,0x4f73083f,0x4f740497,
	0x4f760802,0x4f7b133d,0x4f7c08cd,0x4f7e16fa,0x4f7f11a8,0x4f8309fd,0x4f841942,0x4f88025b,
	0x4f890a5e,0x4f8b0b46,0x4f8d11b6,0x4f8f19a6,0x4f9117a6,0x4f940d85,0x4f970409,0x4f9b05e9,
	0x4f9d16bf,0x4fa014cf,0x4fa30c72,0x4fa508ce,0x4fa618fe,0x4fa701a8,0x4fa80fbf,0x4fa90a65,
	0x4faa01ca,0x4fac0e21,0x4fae1469,0x4faf0710,0x4fb50fde,0x4fbf0115,0x4fc302f4,0x4fc4046c,
	0x4fc51020,0x4fca09db,0x4fce1a4c,0x4fcf0fcb,0x4fd00b4b,0x4fd1177d,0x4fd7126e,0x4fd8052b,
	0x4fda0b2f,0x4fdc0ef7,0x4fdd0096,0x4fde17bb,0x4fdf0f5a,0x4fe1157b,0x4fe317da,0x4fe60279,
	0x4fe8164d,0x4fe90b63,0x4fea0b4c,0x4fed0879,0x4fee159c,0x4fef0549,0x4ff1099b,0x4ff30e56,
	0x4ff80512,0x4ffa0020,0x4ffe00d7,0x500c0649,0x500d00b1,0x500f11e4,0x50120374,0x501409c2,
	0x501812f7,0x5019071c,0x501a16e7,0x501c1328,0x501f090b,0x50210202,0x50250a41,0x502609b2,
	0x5028099c,0x50290f9d,0x502a0ddc,0x502c19f9,0x502d143f,0x502e0c60,0x503a18ab,0x503c1944,
	0x503e0ff6,0x5043164f,0x5047085a,0x5048082c,0x504c10d1,0x504e13f1,0x504f0ed1,0x5055154f,
	0x505a1a6a,0x505c1358,0x50650895,0x506c1a37,0x50760e46,0x50771376,0x507b0c08,0x507e04f9,
	0x507f01f6,0x50800a8f,0x5085055e,0x50880b5c,0x508d0085,0x50a30336,0x50a512f9,0x50a70132,
	0x50a80296,0x50a90e32,0x50ac0304,0x50b20040,0x50ba0269,0x50bb110a,0x50cf1527,0x50d614a4,
	0x50da0b92,0x50e60977,0x50e710fd,0x50ec08c6,0x50ed08a1,0x50ee136d,0x50f3127c,0x50f508aa,
	0x50fb0ece,0x51060947,0x510715d3,0x510b0351,0x511210b2,0x51210b0b,0x513f048c,0x51401470,
	0x5141183d,0x5143180b,0x51441594,0x5145026b,0x514618e0,0x514814e0,0x51490657,0x514b0a2e,
	0x514d0d26,0x51510442,0x51541389,0x51551243,0x5156164b,0x515a0365,0x515c0413,0x5162093e,
	0x516510bd,0x5168104e,0x516b0045,0x516c05e6,0x516d0bf4,0x516e1485,0x51700ac1,0x517105f6,
	0x51730646,0x5174157d,0x51750142,0x51760f43,0x51770998,0x517803bb,0x51791a0e,0x517b167e,
	0x517c0868,0x517d11d8,0x51800838,0x518101e6,0x5182095a,0x51850dd3,0x51880593,0x51891074,
	0x518c01a7,0x518d1858,0x51920cc6,0x51950d2a,0x51960d16,0x519710a8,0x51991557,0x519b09d3,
	0x519c0e20,0x51a00648,0x51a21989,0x51a41807,0x51a50d5b,0x51ab0140,0x51ac0400,0x51af050b,
	0x51b00141,0x51b1073f,0x51b2026c,0x51b309bc,0x51b50a78,0x51b616ab,0x51b70b14,0x51bb0408,
	0x51bc14f7,0x51bd0ba4,0x51c0094a,0x51c40f31,0x51c619f6,0x51c7124f,0x51c90b7f,0x51cb03d1,
	0x51cc0bce,0x51cf087e,0x51d102ed,0x51db0bbd,0x51dd0e14,0x51e007f8,0x51e104a9,0x51e40510,
	0x51eb0521,0x51ed0efa,0x51ef09ec,0x51f0077e,0x51f3038b,0x51f50a00,0x51f61593,0x51f8137c,
	0x51f9002e,0x51fa0287,0x51fb07d8,0x51fc0367,0x51fd06a8,0x51ff186b,0x5200036d,0x520103cf,
	0x5202036e,0x5203108d,0x520604ea,0x52070fd1,0x520816ed,0x520a09f6,0x520d028a,0x520e1432,
	0x52111583,0x5212074b,0x5216182a,0x52170ba2,0x52180be3,0x52191878,0x521a0594,0x521b02b6,
	0x521d0288,0x52201114,0x52240e67,0x52280e79,0x52290b40,0x522b012e,0x522d0943,0x522e0638,
	0x52300378,0x52330a50,0x5236195e,0x5237120a,0x52381061,0x52391101,0x523a02df,0x523b0a2f,
	0x523d0671,0x523f0672,0x524009ed,0x52410463,0x52420824,0x52431327,0x524a15e5,0x524c0aaf,
	0x524d0f89,0x5250063c,0x52510891,0x52541319,0x52560f11,0x525c13c5,0x525e07e4,0x5261164a,
	0x52650150,0x5267099d,0x5269118c,0x526a087f,0x526f055d,0x527205ba,0x527d0edb,0x527f08d9,
	0x52810fb8,0x528209c8,0x52880eae,0x529007c3,0x52931718,0x529b0b2a,0x529d1060,0x529e0076,
	0x529f05e7,0x52a0083d,0x52a11472,0x52a20c94,0x52a30ba3,0x52a80407,0x52a919c5,0x52aa0e2a,
	0x52ab08f2,0x52ac1039,0x52ad114a,0x52b10b41,0x52b20925,0x52b30ae7,0x52be06dd,0x52bf11b7,
	0x52c3015a,0x52c7177e,0x52c90d28,0x52cb15f2,0x52d00cfc,0x52d20afb,0x52d615c1,0x52d809f7,
	0x52df0d98,0x52e40fe6,0x52f01553,0x52f9008b,0x52fa1145,0x52fe05f8,0x52ff1471,0x53001837,
	0x5305008c,0x530602e2,0x53081595,0x530d0f1a,0x530f0e7f,0x53100535,0x531500d1,0x53160753,
	0x531700aa,0x531911ce,0x531a04bc,0x531d184c,0x532008b3,0x53210a6e,0x532314ce,0x5326066b,
	0x532a04db,0x532e0a91,0x53390ec1,0x533a102b,0x533b16be,0x533e010d,0x533f0de7,0x53411199,
	0x53430f77,0x534510d7,0x5347117e,0x53481461,0x534907a4,0x534a0077,0x534e074d,0x534f154b,
	0x535100a4,0x53521a46,0x535319fd,0x53550349,0x53560c97,0x53570db6,0x535a015f,0x535c0171,
	0x535e0110,0x535f0172,0x536018b6,0x536109e5,0x53620c18,0x5363179b,0x53640c24,0x5366063e,
	0x53671447,0x536908ee,0x536b141b,0x536e1932,0x536f0cc0,0x53701740,0x537113ef,0x537307ff,
	0x53741065,0x53750c48,0x537709b1,0x5378155b,0x537a091c,0x537f0ff7,0x538201fb,0x53840475,
	0x53851350,0x53860b3b,0x53890b3c,0x538b1610,0x538c1659,0x538d1158,0x539501a9,0x53980b17,
	0x539a0719,0x539d031f,0x539f1811,0x53a21511,0x53a31650,0x53a509c7,0x53a6110f,0x53a8028c,
	0x53a90973,0x53ae1237,0x53b6122c,0x53bb1048,0x53bf1500,0x53c110e1,0x53c2018d,0x53c817a2,
	0x53c901b2,0x53ca07fa,0x53cb1799,0x53cc1216,0x53cd04b3,0x53d1049a,0x53d411e1,0x53d61044,
	0x53d711d6,0x53d80116,0x53d915be,0x53db0e69,0x53df1265,0x53e003e5,0x53e30a4a,0x53e4061e,
	0x53e50993,0x53e60be0,0x53e8036f,0x53e90a4b,0x53ea193f,0x53eb08dd,0x53ec18df,0x53ed0047,
	0x53ee03ed,0x53ef0a2b,0x53f012bf,0x53f10261,0x53f211a5,0x53f317a3,0x53f50f09,0x53f616ae,
	0x53f706cf,0x53f8122f,0x53f912e2,0x53fb0af9,0x53fc03d0,0x53fd07d9,0x540115b4,0x54030246,
	0x540405ce,0x54061687,0x540806db,0x540907fb,0x540a03d6,0x540c1364,0x540d0d57,0x540e0718,
	0x540f0b3e,0x54101387,0x54111523,0x5412189d,0x541314dc,0x54150c71,0x5416160e,0x54170c85,
	0x541b09d4,0x541d0bc1,0x541e1398,0x541f172d,0x542004e1,0x542100d3,0x54230ff0,0x54260515,
	0x54270049,0x54280447,0x542904eb,0x542b06a6,0x542c1352,0x542d0a3d,0x542e121f,0x542f0f60,
	0x54311935,0x5432173a,0x54341459,0x5435020e,0x54381488,0x543902b8,0x543b1433,0x543c0717,
	0x543e145a,0x54401611,0x54430476,0x54460333,0x54480234,0x544a05ae,0x544b0517,0x54500daa,
	0x54520543,0x545316f5,0x54540334,0x54550e45,0x54560b42,0x545700bd,0x5458180c,0x545913c1,
	0x545b0fa2,0x545c1452,0x54620dd1,0x54640bdd,0x54661787,0x54681992,0x5471061b,0x54721a0b,
	0x5473141e,0x547506d7,0x54760dc3,0x547714cb,0x54780e83,0x547b1166,0x547c0721,0x547d0d62,
	0x5480098a,0x5482184d,0x54840457,0x54860e7a,0x548b189e,0x548c06de,0x548e096f,0x548f177b,
	0x54900558,0x5492199a,0x549409e4,0x5495060e,0x549609e2,0x54990bf7,0x549a0401,0x549b0e0f,
	0x549d1231,0x54a30658,0x54a418a1,0x54a616cf,0x54a70ba1,0x54a81a0f,0x54a90d42,0x54aa0d05,
	0x54ab1954,0x54ac169a,0x54ad07e3,0x54af09e8,0x54b1185c,0x54b30a2a,0x54b40793,0x54b814ef,
	0x54bb159d,0x54bd162b,0x54bf16c0,0x54c00004,0x54c10ef1,0x54c2116c,0x54c40700,0x54c60458,
	0x54c713b7,0x54c80694,0x54c91853,0x54cc0e5a,0x54cd151e,0x54ce0003,0x54cf05d5,0x54d00a70,
	0x54d11620,0x54d20324,0x54d3152c,0x54d400e1,0x54d51828,0x54d7074e,0x54d90a67,0x54da045f,
	0x54dc0826,0x54dd0e22,0x54de0d82,0x54df1769,0x54e505b6,0x54e60e3f,0x54e70247,0x54e8114d,
	0x54e90b62,0x54ea0da2,0x54ed0a52,0x54ee1542,0x54f218eb,0x54f31890,0x54fa0174,0x54fc06f8,
	0x54fd05de,0x54ff05cb,0x5501165c,0x5506129e,0x550702c4,0x55090005,0x550f148f,0x551012e9,
	0x55111a68,0x55140c82,0x551b0c8d,0x55200af2,0x552212a8,0x55231873,0x5524076d,0x552707e5,
	0x552a050f,0x552c073b,0x552e11d9,0x552f1405,0x5530120b,0x55310204,0x55330b57,0x5537176a,
	0x553c110c,0x553e13b5,0x553f0725,0x5541198e,0x55430a3b,0x55441a05,0x55461133,0x55490ace,
	0x554a0001,0x55500309,0x55551309,0x5556035a,0x555c02cc,0x556104d0,0x55640eba,0x55651109,
	0x55660ab4,0x5567187d,0x556a0e4c,0x556c10f8,0x556d19dc,0x556e0e05,0x5575016c,0x557603f9,
	0x55770ad8,0x55781546,0x557b0268,0x557c131e,0x557e0964,0x558009e3,0x55811778,0x55821424,
	0x55830db8,0x55841127,0x55870aae,0x558808ea,0x55890711,0x558a06af,0x558b03e2,0x558f0e34,
	0x5591172b,0x55940e3e,0x559802ae,0x559907b2,0x559c14be,0x559d06d8,0x559f0a92,0x55a715ce,
	0x55b10b1d,0x55b31891,0x55b50d30,0x55b70e92,0x55b90a86,0x55bb17f6,0x55bd0c09,0x55be0a59,
	0x55c4110b,0x55c515ab,0x55c91278,0x55cc0014,0x55cd129d,0x55d10a28,0x55d20327,0x55d310e9,
	0x55d4021d,0x55d6125d,0x55dc11c9,0x55dd05c5,0x55df08eb,0x55e1143a,0x55e3124b,0x55e4024c,
	0x55e506c8,0x55e612ac,0x55e80695,0x55ea0fe7,0x55eb0e06,0x55ec06d9,0x55ef0d9f,0x55f203b5,
	0x55f3000c,0x55f51362,0x55f70031,0x55fd1269,0x55fe1266,0x56000395,0x56010f38,0x560801a1,
	0x5609084a,0x560c0edf,0x560e056a,0x560f062a,0x561815b1,0x561b0c8f,0x561e0b10,0x561f041f,
	0x562300c8,0x56241747,0x56270d20,0x562c1a5d,0x562d0e99,0x563119c0,0x5632020c,0x56341a53,
	0x56361239,0x56390b94,0x563b14a8,0x563f06f2,0x564c01af,0x564d08e3,0x564e16a5,0x56540385,
	0x56570f19,0x565809b9,0x56590fe9,0x565c0c16,0x56620e3d,0x56640931,0x56680f70,0x56690486,
	0x566a1875,0x566b16c5,0x566c11cc,0x567115e8,0x5676056d,0x567b10dd,0x567c0eaf,0x568510b3,
	0x568606c4,0x568e06ca,0x568f132f,0x5693017f,0x56a3153a,0x56af07d4,0x56b7107a,0x56bc08cb,
	0x56ca0dbe,0x56d40dbd,0x56d713fa,0x56da101b,0x56db123e,0x56dd0876,0x56de079c,0x56df157a,
	0x56e01721,0x56e10db4,0x56e2138d,0x56e4044e,0x56eb0729,0x56ed180d,0x56f00aa4,0x56f102e1,
	0x56f413fe,0x56f50c50,0x56f90bc8,0x56fa062d,0x56fd068a,0x56fe137f,0x56ff17a7,0x57030f23,
	0x570417dd,0x57061812,0x5708104d,0x570917de,0x570a0ff8,0x571c0768,0x571f1386,0x57231189,
	0x57281859,0x572913fd,0x572a05b3,0x572c144f,0x572d065f,0x572e0ec4,0x572f16c8,0x573003a9,
	0x57331914,0x57390a76,0x573a01fa,0x573b0f40,0x573e07dc,0x57401950,0x57420071,0x574709d5,
	0x574a04bf,0x574c00c4,0x574d12ce,0x574e09fc,0x574f075c,0x57501a62,0x57510a3e,0x57570a63,
	0x575a0863,0x575b12d4,0x575c0b43,0x575d0058,0x575e146e,0x575f04f0,0x576019ee,0x57610f03,
	0x57640a9a,0x576612de,0x576813a7,0x5769057f,0x576a0efb,0x576b03c3,0x576d0dd9,0x576f0eab,
	0x5773003e,0x57760d94,0x57770a17,0x577b03a3,0x577c0216,0x578202ba,0x57830aaa,0x57840c04,
	0x57850c05,0x57860c1b,0x578b1588,0x578c040a,0x57920b08,0x57930572,0x579b0460,0x57a0172e,
	0x57a1049d,0x57a20607,0x57a3180f,0x57a403df,0x57a60a39,0x57a71137,0x57a9047a,0x57ab03c5,
	0x57ad1626,0x57ae0a5d,0x57b209ee,0x57b40dc9,0x57b81821,0x57c205df,0x57c30006,0x57cb0c90,
	0x57ce0238,0x57cf1638,0x57d20ba6,0x57d40f24,0x57d5023a,0x57d811a3,0x57d915f3,0x57da0683,
	0x57dd0df7,0x57df17f2,0x57e0017c,0x57e40ebb,0x57ed0342,0x57ef0021,0x57f41945,0x57f8170a,
	0x57f90e87,0x57fa07ea,0x57fd10f4,0x58000a53,0x580212ea,0x5806043f,0x58070919,0x580b0e9b,
	0x580d138a,0x58110f9e,0x58150464,0x5819172c,0x581e03e3,0x5820071d,0x58210098,0x58240392,
	0x582a09f9,0x58301662,0x5835042b,0x58440b11,0x584c12b2,0x584d023e,0x58511279,0x585412b4,
	0x585812ec,0x585e10db,0x586505c6,0x586b1336,0x586c1817,0x587e11f2,0x58800257,0x58810ca3,
	0x58830956,0x58851207,0x58891771,0x58921135,0x58930d99,0x58990fad,0x589a0b84,0x589e1889,
	0x589f15b3,0x58a80d7c,0x58a90449,0x58bc07f4,0x58c100f7,0x58c51773,0x58d106f0,0x58d506cb,
	0x58e4107b,0x58eb11ad,0x58ec1089,0x58ee19e6,0x58f01180,0x58f30a29,0x58f6072d,0x58f916c2,
	0x5902194d,0x5904029a,0x590700ae,0x590d055b,0x590f14dd,0x59140a8d,0x59151484,0x591613c3,
	0x5919126f,0x591a0456,0x591c16b2,0x591f0608,0x59241734,0x592507cb,0x59270332,0x59291330,
	0x592a12c7,0x592b0516,0x592d1686,0x592e166c,0x592f06bd,0x5931118f,0x59341378,0x593716c9,
	0x59380a5c,0x5939083e,0x593a045b,0x593c0a74,0x59410b64,0x5942076b,0x5944164c,0x59470f44,
	0x59480daf,0x59490511,0x594b04f7,0x594e0a83,0x594f1a41,0x59510f6c,0x595400be,0x59551701,
	0x595608af,0x5957130e,0x595819e5,0x595a1490,0x596003ca,0x5962114f,0x59650041,0x59730e39,
	0x59740e27,0x59760dac,0x59780861,0x597912ad,0x597d06cd,0x59811224,0x598210af,0x598304ce,
	0x598413ea,0x598619e1,0x59870555,0x59880c83,0x598a1092,0x598d163d,0x5992042f,0x5993081f,
	0x59961688,0x59970926,0x59990d3d,0x599e0e18,0x59a300d4,0x59a417b6,0x59a513b0,0x59a804c4,
	0x59a91464,0x59aa17e8,0x59ab0660,0x59ae0dd7,0x59af1995,0x59b2032a,0x59b90ce5,0x59bb0f2f,
	0x59be0fd5,0x59c60d8e,0x59ca1a25,0x59cb11a9,0x59d00902,0x59d1060f,0x59d21244,0x59d3158e,
	0x59d4140f,0x59d71117,0x59d80eeb,0x59da168e,0x59dc08a5,0x59dd11e3,0x59e308bc,0x59e50aef,
	0x59e816d0,0x59ec07e6,0x59f901c5,0x59fb1723,0x59ff1a10,0x5a0113f0,0x5a0313bc,0x5a040c07,
	0x5a051627,0x5a061081,0x5a0708bd,0x5a080c3f,0x5a090ef5,0x5a0c0b30,0x5a11129f,0x5a131413,
	0x5a180df8,0x5a1c0da6,0x5a1f09aa,0x5a201169,0x5a2303ac,0x5a25046d,0x5a290d29,0x5a3117bf,
	0x5a3213ba,0x5a3414f1,0x5a361045,0x5a3c01ed,0x5a40046a,0x5a460f06,0x5a4913d4,0x5a4a0128,
	0x5a5508f8,0x5a5a07ba,0x5a6200e8,0x5a670951,0x5a6a0ac5,0x5a741745,0x5a7501d5,0x5a761174,
	0x5a771359,0x5a7a147c,0x5a7f15c5,0x5a920cd7,0x5a9a0ce8,0x5a9b1822,0x5aaa003c,0x5ab20ecd,
	0x5ab314b7,0x5ab51768,0x5ab8024d,0x5abe0609,0x5ac1085e,0x5ac210f3,0x5ac907f7,0x5acc14f6,
	0x5ad20015,0x5ad40eee,0x5ad60ee1,0x5ad80aff,0x5adc18c6,0x5ae00b22,0x5ae1039f,0x5ae31637,
	0x5ae601f9,0x5ae90dd4,0x5aeb0d67,0x5af10fae,0x5b0914a9,0x5b1600f8,0x5b17112b,0x5b320dfd,
	0x5b34175d,0x5b370d65,0x5b401218,0x5b501a24,0x5b5108ef,0x5b5309bb,0x5b540a44,0x5b551841,
	0x5b571a2c,0x5b580313,0x5b591295,0x5b5a0522,0x5b5b00bc,0x5b5c1a0d,0x5b5d1541,0x5b5f0d03,
	0x5b62008d,0x5b630825,0x5b640610,0x5b650e28,0x5b6615ea,0x5b690696,0x5b6a0c40,0x5b6c0dc2,
	0x5b7011f0,0x5b7101d7,0x5b731a16,0x5b75051d,0x5b7a10b4,0x5b7d0e0b,0x5b800d22,0x5b810e0e,
	0x5b8312af,0x5b840668,0x5b8518a9,0x5b8717d6,0x5b8811d2,0x5b890019,0x5b8b1259,0x5b8c13cc,
	0x5b8f0706,0x5b930d19,0x5b950368,0x5b971a30,0x5b980656,0x5b99199b,0x5b9a03f8,0x5b9b13d0,
	0x5b9c16cc,0x5b9d0094,0x5b9e119e,0x5ba00274,0x5ba11171,0x5ba20a30,0x5ba315cc,0x5ba411bb,
	0x5ba517a8,0x5ba6076c,0x5baa1506,0x5bab05eb,0x5bb01855,0x5bb3069e,0x5bb4165d,0x5bb5152f,
	0x5bb60844,0x5bb80224,0x5bb910a0,0x5bbd0a6b,0x5bbe0130,0x5bbf1275,0x5bc2082d,0x5bc4082e,
	0x5bc51730,0x5bc60d1d,0x5bc70a4d,0x5bcc055f,0x5bd00ce9,0x5bd206ac,0x5bd317f7,0x5bdd0fef,
	0x5bde0d78,0x5bdf01bd,0x5be1063d,0x5be41480,0x5be50b93,0x5be818ad,0x5bee0b95,0x5bf00765,
	0x5bf80315,0x5bf90441,0x5bfa123f,0x5bfb15f9,0x5bfc0372,0x5bff11d5,0x5c010502,0x5c04115b,
	0x5c0608a6,0x5c091422,0x5c0a1a58,0x5c0f153e,0x5c111149,0x5c140490,0x5c15056e,0x5c160862,
	0x5c18021e,0x5c1a113b,0x5c1c056c,0x5c1d01f5,0x5c22178a,0x5c24178b,0x5c250b9e,0x5c27168c,
	0x5c2c056f,0x5c310975,0x5c340586,0x5c38118e,0x5c391738,0x5c3a025a,0x5c3b0a0e,0x5c3c0dd8,
	0x5c3d0924,0x5c3e13f9,0x5c3f0dfe,0x5c400987,0x5c410ecb,0x5c4201b0,0x5c450979,0x5c481030,
	0x5c491326,0x5c4a0907,0x5c4b1454,0x5c4e11ab,0x5c4f0efd,0x5c5007e7,0x5c51155c,0x5c5518b8,
	0x5c590469,0x5c5e11fb,0x5c601384,0x5c610c76,0x5c6314c0,0x5c650c7a,0x5c6609a6,0x5c6e0214,
	0x5c6f139a,0x5c711112,0x5c7916f2,0x5c7a0f5d,0x5c7f17d7,0x5c81128c,0x5c820f5e,0x5c881618,
	0x5c8c07fc,0x5c8d0f7e,0x5c900f41,0x5c9101ad,0x5c9401c3,0x5c96102d,0x5c97059a,0x5c981501,
	0x5c99003f,0x5c9a0ac2,0x5c9b0373,0x5c9c004a,0x5ca20a2c,0x5ca305ff,0x5ca91640,0x5cab15a6,
	0x5cac0854,0x5cad0bde,0x5cb10338,0x5cb3182b,0x5cb50742,0x5cb70d48,0x5cb80025,0x5cbd0402,
	0x5cbf0a7f,0x5cc10cc1,0x5cc416fb,0x5ccb15fd,0x5cd2040b,0x5cd91965,0x5ce114d1,0x5ce40fbd,
	0x5ce5191f,0x5ce60c41,0x5ce8046e,0x5cea17ee,0x5ced0fcd,0x5cf00505,0x5cfb09dd,0x5d020ae9,
	0x5d030ab6,0x5d060a42,0x5d070273,0x5d0e0f4a,0x5d140303,0x5d16161c,0x5d1b09c3,0x5d1e0685,
	0x5d24153c,0x5d26162f,0x5d271250,0x5d2900c6,0x5d2d18ba,0x5d2e0630,0x5d3413ee,0x5d3d1856,
	0x5d3e169d,0x5d4707f9,0x5d4a118d,0x5d4b0cd8,0x5d4c0f9f,0x5d5810a1,0x5d5b17c7,0x5d5d0c0f,
	0x5d691253,0x5d6b1a17,0x5d6c1409,0x5d6f031a,0x5d740817,0x5d8218d7,0x5d990bb5,0x5d9d038c,
	0x5db716da,0x5dc503b9,0x5dcd13f8,0x5ddb02a4,0x5ddd02a5,0x5dde198f,0x5de115fa,0x5de2020a,
	0x5de505e4,0x5de61a5f,0x5de70fc9,0x5de80994,0x5de905f2,0x5deb1453,0x5dee01c6,0x5def1025,
	0x5df10811,0x5df216e0,0x5df3123d,0x5df40046,0x5df71524,0x5dfd160c,0x5dfe090f,0x5e0100db,
	0x5e0211b2,0x5e030177,0x5e051210,0x5e0604a4,0x5e081190,0x5e0c1489,0x5e0f13ff,0x5e1018d2,
	0x5e1112f6,0x5e140e8d,0x5e150e53,0x5e16134e,0x5e180b66,0x5e19195f,0x5e1a1998,0x5e1b0158,
	0x5e1c1960,0x5e1d03ab,0x5e26033c,0x5e27190c,0x5e2d14b4,0x5e2e007e,0x5e31027a,0x5e371406,
	0x5e3801f7,0x5e3b187e,0x5e3c068b,0x5e3d0cca,0x5e420d1e,0x5e441448,0x5e45053c,0x5e4c0791,
	0x5e540ca4,0x5e550d9a,0x5e5b18d8,0x5e5e0540,0x5e6104a6,0x5e6202b4,0x5e72057b,0x5e730ef8,
	0x5e740ded,0x5e76014a,0x5e78158f,0x5e7a1685,0x5e7b076a,0x5e7c17a4,0x5e7d1788,0x5e7f065b,
	0x5e800ec2,0x5e8419e2,0x5e861007,0x5e8700df,0x5e8a02b3,0x5e8b066a,0x5e8f15bd,0x5e900c19,
	0x5e911465,0x5e930a57,0x5e941743,0x5e9503a4,0x5e960e7b,0x5e9703c4,0x5e990d3e,0x5e9a05da,
	0x5e9c0546,0x5e9e0e71,0x5e9f04e2,0x5ea0151a,0x5ea5159e,0x5ea60432,0x5ea71a69,0x5ead1356,
	0x5eb300a9,0x5eb5001c,0x5eb61204,0x5eb70a04,0x5eb8176f,0x5eb913b1,0x5ebe17df,0x5ec90b6c,
	0x5eca0adb,0x5ed10920,0x5ed20032,0x5ed30aa9,0x5ed60b9c,0x5edb01dc,0x5ee81563,0x5eea0bbe,
	0x5ef41737,0x5ef6163a,0x5ef71354,0x5efa088f,0x5efe05f1,0x5eff0df5,0x5f0009e9,0x5f010111,
	0x5f0216f3,0x5f030f69,0x5f040e25,0x5f081702,0x5f0a00f3,0x5f0b16ec,0x5f0f11b4,0x5f1111c6,
	0x5f1305e5,0x5f151739,0x5f17051f,0x5f180704,0x5f1b0251,0x5f1f03aa,0x5f2018c3,0x5f250d07,
	0x5f2614ed,0x5f27072a,0x5f290e2b,0x5f2a094b,0x5f2d0d13,0x5f2f13c4,0x5f3110d2,0x5f39035b,
	0x5f3a0fac,0x5f3c00eb,0x5f40060a,0x5f5015ed,0x5f52065e,0x5f530362,0x5f550c2d,0x5f561390,
	0x5f5707af,0x5f581972,0x5f5d16db,0x5f611113,0x5f621586,0x5f641366,0x5f66165a,0x5f690188,
	0x5f6a011c,0x5f6c0131,0x5f6d0e9c,0x5f7018c7,0x5f711764,0x5f730260,0x5f770e70,0x5f7916f6,
	0x5f7b0215,0x5f7c00d5,0x5f8013e4,0x5f81191d,0x5f8202f2,0x5f84094c,0x5f85033d,0x5f871609,
	0x5f8806f4,0x5f891679,0x5f8a0758,0x5f8b0c7d,0x5f8c071a,0x5f9015b5,0x5f921380,0x5f950aba,
	0x5f97037f,0x5f980e57,0x5f9914bc,0x5f9c01f8,0x5fa117f8,0x5fa80781,0x5faa1602,0x5fad1693,
	0x5fae13f5,0x5fb51926,0x5fb70381,0x5fbc08db,0x5fbd079a,0x5fc3156d,0x5fc4156c,0x5fc500dc,
	0x5fc616ee,0x5fc90370,0x5fcc0820,0x5fcd108a,0x5fcf01e8,0x5fd012dd,0x5fd1130f,0x5fd21310,
	0x5fd60314,0x5fd7195b,0x5fd813eb,0x5fd90cac,0x5fdd1338,0x5fe0197f,0x5fe1026d,0x5fe41466,
	0x5fe71785,0x5fea124d,0x5feb0a64,0x5fed0112,0x5fee195c,0x5ff10220,0x5ff50df6,0x5ff80e1a,
	0x5ffb156e,0x5ffd0722,0x5ffe09f5,0x5fff04f8,0x60000757,0x600112c9,0x60021254,0x60031467,
	0x60040e49,0x60050200,0x600602b7,0x600a0206,0x600d1a64,0x600e1886,0x600f1681,0x60120e2d,
	0x6014191e,0x60150e54,0x60160179,0x60190743,0x601b032b,0x601c0b67,0x601d1232,0x6020033e,
	0x602116cd,0x60250803,0x60260e95,0x60271590,0x6028181f,0x60290dda,0x602a0645,0x602b0525,
	0x602f0fd6,0x6035029c,0x603b1a36,0x603c0443,0x603f16fc,0x60410dd5,0x604215fe,0x604311bc,
	0x604b0b79,0x604d078e,0x60500a45,0x605206f9,0x60551203,0x60591682,0x605a07ac,0x605d084f,
	0x60620794,0x60631a22,0x606415bf,0x60670e3b,0x606806f6,0x60690489,0x606a0a31,0x606b040c,
	0x606c1333,0x606d05ec,0x606f1491,0x60700f74,0x60730a3a,0x6076047b,0x60781374,0x6079162c,
	0x607a09ef,0x607b01aa,0x607c0dca,0x607d1844,0x607f1780,0x60830aa1,0x60840fc8,0x60891494,
	0x608c1329,0x608d06b3,0x60921706,0x609407a1,0x609600b2,0x609a1255,0x609b104c,0x609d0a80,
	0x609f1479,0x60a01789,0x60a30771,0x60a6182c,0x60a80e0d,0x60ab1066,0x60ac15d6,0x60ad0f81,
	0x60af0d53,0x60b104dd,0x60b200a6,0x60b4030a,0x60b8082f,0x60bb1592,0x60bc0379,0x60c50ffc,
	0x60c6027b,0x60ca0937,0x60cb13d5,0x60d107d1,0x60d5132c,0x60d813e7,0x60da0726,0x60dc1495,
	0x60dd01fd,0x60df1407,0x60e007b3,0x60e603c8,0x60e7099e,0x60e80193,0x60e90242,0x60eb00b4,
	0x60ec0fd9,0x60ed0192,0x60ee035c,0x60ef0650,0x60f00466,0x60f31521,0x60f419f0,0x60f60782,
	0x60f91084,0x60fa157f,0x61000fca,0x6101027e,0x61060f84,0x610817fc,0x610917c8,0x610d0d55,
	0x610e00ec,0x610f170e,0x61150480,0x611a17cc,0x611f058b,0x61201847,0x61230b15,0x612404fa,
	0x61260a93,0x61270a94,0x612b127a,0x613f1825,0x614802d7,0x614a0f99,0x614c077b,0x614e117b,
	0x6151115e,0x61550d9c,0x615d1313,0x61620ca5,0x616707b5,0x616809f1,0x61701428,0x61751772,
	0x61770a05,0x618b012b,0x618e188a,0x61940fc3,0x619d0445,0x61a70270,0x61a806a3,0x61a90f71,
	0x61ac0948,0x61b702a0,0x61be06b9,0x61c20406,0x61c81564,0x61ca0043,0x61cb0cce,0x61d10cf1,
	0x61d20ad5,0x61d40bbf,0x61e60e37,0x61f50d01,0x61ff1720,0x6206059e,0x620805b2,0x620a1473,
	0x620b0860,0x620c15ac,0x620d11fd,0x620e109a,0x620f14c3,0x62100233,0x62111444,0x62120905,
	0x62150fa4,0x621607cd,0x62170fa5,0x621818bd,0x621a0f34,0x621b0850,0x621f0816,0x622109fa,
	0x62220806,0x62240579,0x62250389,0x622a08fc,0x622c0888,0x622e0c37,0x623302cb,0x62340345,
	0x6237073e,0x623d0744,0x623e0b47,0x623f04c5,0x624012a7,0x6241010b,0x6243095b,0x62471126,
	0x62480747,0x624904d3,0x624b11d0,0x624c11d1,0x624d0183,0x624e1896,0x62510f17,0x62520048,
	0x62530330,0x62541097,0x625813a0,0x625b0a07,0x62630a4c,0x62660f7a,0x62671941,0x62690aa5,
	0x626a0cec,0x626b10f2,0x626c1672,0x626d0e1b,0x626e0079,0x626f0213,0x62701082,0x62730068,
	0x62760523,0x62790ea8,0x627c0477,0x627e18dd,0x627f0235,0x62800821,0x62840205,0x628909be,
	0x628a0055,0x629116f7,0x629211df,0x629319d5,0x62951379,0x62960417,0x62970a0a,0x629818e9,
	0x629a0544,0x629b0e77,0x629f138e,0x62a00a47,0x62a10c4d,0x62a20fb2,0x62a40740,0x62a5009b,
	0x62a80e96,0x62ab0eac,0x62ac12c1,0x62b1009c,0x62b503a5,0x62b90d70,0x62bb021a,0x62bc1612,
	0x62bd0276,0x62bf0d50,0x62c20526,0x62c419bd,0x62c5034a,0x62c601c7,0x62c70d8f,0x62c80deb,
	0x62c90aab,0x62ca0547,0x62cc007a,0x62cd0e55,0x62ce0bae,0x62d00644,0x62d20996,0x62d313b3,
	0x62d40050,0x62d613a1,0x62d70e1f,0x62d8097a,0x62d919f8,0x62da0e66,0x62db18db,0x62dc005f,
	0x62df0de3,0x62e20c06,0x62e30877,0x62e5176c,0x62e60ac3,0x62e70e10,0x62e8014d,0x62e91879,
	0x62ec0aa6,0x62ed11bd,0x62ee08f4,0x62ef1927,0x62f105f4,0x62f31053,0x62f41213,0x62f6184e,
	0x62f70a10,0x62fc0eec,0x62fd19d7,0x62fe119f,0x62ff0da0,0x63010256,0x63020640,0x63071955,
	0x63080fd8,0x63090026,0x630e0a5f,0x6311133e,0x631613b8,0x631a1968,0x631b0c42,0x631d143e,
	0x631e12b8,0x631f154e,0x63200dc4,0x63210364,0x632208cf,0x63231929,0x63240813,0x63250795,
	0x63280007,0x632a0e31,0x632b0320,0x632f1917,0x633212a0,0x63391707,0x633a135d,0x633d13d1,
	0x6342146a,0x634309de,0x63451371,0x63460aa2,0x634919fa,0x634b0c7b,0x634c004d,0x634d06b4,
	0x634e113e,0x634f0e00,0x635009ab,0x63550175,0x635e0ae6,0x635f1299,0x6361087c,0x6362076e,
	0x63630375,0x63670ea5,0x63690ba8,0x636d0064,0x636e099f,0x63710009,0x637602bc,0x637708f9,
	0x637a0da9,0x637b0df1,0x638014e6,0x638203b6,0x6387045a,0x638811da,0x638903d9,0x638a0f13,
	0x638c18cf,0x638e0815,0x638f1301,0x63900f72,0x63920e58,0x639616b5,0x639809c4,0x63a00c4a,
	0x63a212e4,0x63a30217,0x63a508e8,0x63a70a46,0x63a81391,0x63a91651,0x63aa0321,0x63ac097f,
	0x63ad133c,0x63ae0f90,0x63b0005e,0x63b30c26,0x63b40643,0x63b7196e,0x63b80354,0x63ba01d3,
	0x63bc0651,0x63be1823,0x63c417c9,0x63c60a87,0x63c910aa,0x63cd1a42,0x63ce15cf,0x63cf0d32,
	0x63d0131f,0x63d201b4,0x63d616c3,0x63de0023,0x63e0162a,0x63e11449,0x63e301cd,0x63e909ea,
	0x63ea0965,0x63ed08ec,0x63f21153,0x63f41814,0x63f616a8,0x63f81892,0x63fd0ad0,0x63ff0ff2,
	0x640001d4,0x640105bb,0x64020c10,0x640508d7,0x640b01c9,0x640c18bb,0x640f0162,0x6410029e,
	0x64130316,0x641410ed,0x641b086f,0x641c125e,0x641e05a8,0x64201229,0x642110ea,0x64260e35,
	0x642a12ed,0x642c006d,0x642d0326,0x64340f87,0x643a1552,0x643d01ba,0x643f05c7,0x6441048b,
	0x6444115f,0x644511eb,0x64460065,0x64471694,0x6448013b,0x644a12d0,0x6452014c,0x6454120e,
	0x645818a8,0x645e0c6b,0x64670305,0x64690d6c,0x646d194b,0x64780d64,0x64790d69,0x647a18ef,
	0x64820ba0,0x64841748,0x648509ba,0x64870ee7,0x64910230,0x649210d5,0x6495123a,0x649606b8,
	0x64991a5c,0x649e19e8,0x64a40218,0x64a90b8d,0x64ac0fcf,0x64ad0154,0x64ae0318,0x64b019de,
	0x64b50df3,0x64b71554,0x64b80c17,0x64ba02fd,0x64bc06ba,0x64c0058e,0x64c20b05,0x64c5112c,
	0x64cd019e,0x64ce0fff,0x64d00778,0x64d20fea,0x64d70ec6,0x64d8016a,0x64de126a,0x64e21a08,
	0x64e4158c,0x64e60180,0x65000e5f,0x650907c5,0x6512185e,0x6518107c,0x65251a52,0x652b09d2,
	0x652e0dc1,0x652f1931,0x65340f15,0x65350f16,0x653611cf,0x65381786,0x65390574,0x653b05e8,
	0x653e04cc,0x653f192e,0x6545062e,0x65481543,0x65490d14,0x654c039a,0x654f0d54,0x65510974,
	0x65550267,0x65560030,0x655908e0,0x655b0b72,0x655d00e9,0x655e01fe,0x6562058a,0x656310e6,
	0x65660448,0x656b08da,0x656c0953,0x65701205,0x65720fb9,0x65741928,0x6577051e,0x6587142c,
	0x658b18a7,0x658c0133,0x659004d7,0x6591006c,0x65930acb,0x65970416,0x65990b9f,0x659b072e,
	0x659c1550,0x659f1905,0x65a1144c,0x65a40911,0x65a50262,0x65a70548,0x65a918b7,0x65ab1a00,
	0x65ad0439,0x65af1234,0x65b01574,0x65b904bd,0x65bc17b1,0x65bd1193,0x65c10e73,0x65c318b1,
	0x65c40cba,0x65c50c73,0x65c60e8e,0x65cb15d7,0x65cc0938,0x65ce0de4,0x65cf1a47,0x65d20be9,
	0x65d616e9,0x65d70f54,0x65e01457,0x65e20827,0x65e51099,0x65e60357,0x65e7096d,0x65e8194f,
	0x65e9186c,0x65ec15fb,0x65ed15bc,0x65ee0569,0x65ef0aad,0x65f0058f,0x65f106b2,0x65f6119c,
	0x65f70a79,0x65fa13ec,0x6600183a,0x6602002c,0x66031884,0x66060a9b,0x660a06d0,0x660c01ec,
	0x660e0d58,0x660f07b8,0x661316fd,0x6614148a,0x66151571,0x661912d5,0x661d185d,0x661f157e,
	0x66201766,0x662502c0,0x66270ce6,0x66281a5e,0x662d18dc,0x662f11be,0x663117ec,0x66340cc3,
	0x66350de5,0x663601fc,0x663c199d,0x663e14f8,0x66410209,0x6643078f,0x664b092a,0x664c1138,
	0x664f165e,0x66521111,0x6653153f,0x665416b3,0x66551845,0x66560796,0x665706a9,0x665a13d2,
	0x665f118a,0x6661016f,0x6664147a,0x666607b0,0x66680225,0x666e0f26,0x666f0946,0x6670149b,
	0x66740ffd,0x6676093a,0x6677066f,0x667a1973,0x667e0b8c,0x66821860,0x668415d1,0x668714d6,
	0x668c0a89,0x669111f4,0x66960e2e,0x66970029,0x669d0d5e,0x66a70017,0x66a80832,0x66ae0d9d,
	0x66b400a0,0x66b914eb,0x66be1399,0x66d911fa,0x66db15f7,0x66dc16a1,0x66dd00a2,0x66e614b1,
	0x66e90dc0,0x66f01826,0x66f2102c,0x66f316af,0x66f405d9,0x66f706e0,0x66f901a0,0x66fc0ca2,
	0x66fe1888,0x66ff132d,0x67001a54,0x67081829,0x6709179a,0x670a10c4,0x670b0e9a,0x670d0527,
	0x6710103a,0x67141226,0x67151918,0x67170ae1,0x671b13ed,0x671d020b,0x671f0f36,0x67260cf8,
	0x67280d90,0x672a141c,0x672b0d71,0x672c00c1,0x672d1897,0x672f11fc,0x673119a5,0x67340f22,
	0x6735045e,0x673a07dd,0x673d15a4,0x674010fe,0x6742184f,0x6743104f,0x6746057d,0x674801b3,
	0x67491115,0x674c1475,0x674e0b2d,0x674f158d,0x67500184,0x67510311,0x67531144,0x675618d3,
	0x675c0430,0x675e0f61,0x675f11fe,0x6760059c,0x67611340,0x67650ab5,0x67681675,0x67690c8b,
	0x676a0d35,0x676d06be,0x676f00a5,0x677008f3,0x677205a7,0x67731699,0x67750294,0x67770e4e,
	0x677c19c7,0x677e124e,0x677f0072,0x67810800,0x67840604,0x67870eb5,0x678913e5,0x678b04c1,
	0x6790148b,0x6795190e,0x67970bb0,0x679810c9,0x679a0cd2,0x679c068e,0x679d1936,0x679e02e4,
	0x67a211e2,0x67a3186d,0x67a50b48,0x67a70878,0x67a80236,0x67aa0fa6,0x67ab0501,0x67ad152b,
	0x67af0a51,0x67b00efe,0x67b31940,0x67b5152d,0x67b6085d,0x67b70841,0x67b80602,0x67c10468,
	0x67c30bcb,0x67c40146,0x67cf0063,0x67d00d89,0x67d10582,0x67d20f30,0x67d31076,0x67d410a9,
	0x67d818f7,0x67d914d2,0x67da17a1,0x67dc0673,0x67dd13b4,0x67de1a65,0x67e00e12,0x67e203a6,
	0x67e501b7,0x67e90971,0x67ec087a,0x67ef0a19,0x67f00db0,0x67f119cb,0x67f30bef,0x67f401cb,
	0x67fd022e,0x67ff11bf,0x6800193a,0x680518a2,0x6807011d,0x680818be,0x68091966,0x680a0bfa,
	0x680b040d,0x680c0c1e,0x680e0b4d,0x680f0ac4,0x68111201,0x68131214,0x68160f32,0x68170b52,
	0x681d063b,0x68211544,0x682915b8,0x682a19ab,0x68320a11,0x68330af0,0x68371683,0x683806e2,
	0x683905d3,0x683c05c0,0x683d1854,0x683e0c43,0x684008f7,0x684106fa,0x68420676,0x68431307,
	0x68440659,0x68451403,0x68460a7c,0x68480027,0x6849001a,0x684a09b3,0x684c19fb,0x684e1969,
	0x68501368,0x685110e7,0x68530762,0x68540901,0x68550972,0x68601614,0x68611080,0x68621901,
	0x6863036b,0x68640f33,0x68650fc1,0x68660756,0x68670679,0x686808b0,0x686919e3,0x686b12a1,
	0x68740536,0x68761372,0x687709c5,0x68810b80,0x6883135e,0x68850cd6,0x6886007f,0x688f0631,
	0x68931a29,0x689705e2,0x68a2113f,0x68a60d04,0x68a7145c,0x68a80b18,0x68ad12a2,0x68af131a,
	0x68b0155d,0x68b311e6,0x68b504bb,0x68c00880,0x68c20bd1,0x68c90d25,0x68cb0f4f,0x68cd0682,
	0x68d20086,0x68d51a32,0x68d80807,0x68da0e9d,0x68e012eb,0x68e303b0,0x68ee10fc,0x68f002bd,
	0x68f10b12,0x68f50a1f,0x68f918e4,0x68fa064a,0x68fc04f2,0x6901068f,0x690516e8,0x690b0b81,
	0x690d1947,0x690e19ec,0x69100980,0x691208c2,0x691f0425,0x69200fa0,0x69240c5a,0x692d13b2,
	0x693016a4,0x6934043b,0x6939117c,0x693d02ac,0x693f02c1,0x69421894,0x69541548,0x6957089a,
	0x695a0298,0x695d0b7c,0x695e0b13,0x69600db9,0x69630cdb,0x696615e2,0x696b080a,0x696e0297,
	0x697102ee,0x697709f3,0x69781018,0x69791758,0x697c0c0b,0x69800ef2,0x6982057a,0x69840ad2,
	0x698617cd,0x6987022b,0x69880c70,0x6989098f,0x698d1561,0x69940add,0x699510a4,0x69980990,
	0x699b1908,0x699c0083,0x69a704de,0x69a818a6,0x69ab129c,0x69ad1562,0x69b10306,0x69b40beb,
	0x69b7106c,0x69bb12bb,0x69c105aa,0x69ca122b,0x69cc02be,0x69ce01bc,0x69d0075a,0x69d405a2,
	0x69db09fb,0x69df0136,0x69e019b1,0x69ed0f64,0x69f20735,0x69fd01a3,0x69ff0922,0x6a0a04ad,
	0x6a170289,0x6a1812f0,0x6a1f18ca,0x6a210d6a,0x6a2814ab,0x6a2a06fc,0x6a2f0fb0,0x6a31174b,
	0x6a350fc5,0x6a3d1a5a,0x6a3e1832,0x6a44058d,0x6a470fbb,0x6a5013ae,0x6a580989,0x6a590241,
	0x6a5b09cc,0x6a611528,0x6a6519b3,0x6a710291,0x6a790c28,0x6a7c181b,0x6a8012dc,0x6a8414b9,
	0x6a8e0feb,0x6a901649,0x6a910b01,0x6a97016b,0x6aa01000,0x6aa90bc0,0x6aab01bf,0x6aac0cf9,
	0x6b200f9a,0x6b2102de,0x6b22075d,0x6b231572,0x6b2417b7,0x6b270e41,0x6b3217f3,0x6b371496,
	0x6b390f3b,0x6b3a0f37,0x6b3e0a6d,0x6b43110d,0x6b461575,0x6b471549,0x6b490fa1,0x6b4c05bc,
	0x6b591162,0x6b62194e,0x6b63192a,0x6b6402dd,0x6b650178,0x6b661468,0x6b670f45,0x6b6a13c2,
	0x6b790335,0x6b7b123c,0x6b7c0864,0x6b810d72,0x6b8202f3,0x6b83166e,0x6b841339,0x6b86033f,
	0x6b871132,0x6b89160b,0x6b8a11e5,0x6b8b0190,0x6b8d0ee3,0x6b921840,0x6b930b7a,0x6b961948,
	0x6b9a034f,0x6b9b0808,0x6ba1013c,0x6baa1719,0x6bb311de,0x6bb40e42,0x6bb50438,0x6bb71728,
	0x6bbf03cb,0x6bc107a2,0x6bc2061c,0x6bc51715,0x6bcb1458,0x6bcd0d8b,0x6bcf0ce1,0x6bd20422,
	0x6bd31800,0x6bd400d2,0x6bd500dd,0x6bd600e2,0x6bd70eb6,0x6bd900e5,0x6bdb0cb6,0x6be118b0,
	0x6bea0d8a,0x6beb06c7,0x6bef12e1,0x6bf3030d,0x6bf510e2,0x6bf911ec,0x6bfd089b,0x6c0501ff,
	0x6c060f29,0x6c070c2a,0x6c0d1041,0x6c0f11ae,0x6c100394,0x6c110d47,0x6c130cb1,0x6c140f65,
	0x6c150ee6,0x6c160dae,0x6c180371,0x6c1914e2,0x6c1a02a6,0x6c1b04ee,0x6c1f052c,0x6c210403,
	0x6c220ff4,0x6c241729,0x6c26069f,0x6c27167f,0x6c28001b,0x6c291629,0x6c2a0a33,0x6c2e0360,
	0x6c2f0c81,0x6c300ffe,0x6c321835,0x6c34121c,0x6c35121b,0x6c381779,0x6c3d139e,0x6c401351,
	0x6c411933,0x6c42101d,0x6c4602fc,0x6c4707a5,0x6c4906b0,0x6c4a01c2,0x6c501486,0x6c540f67,
	0x6c551124,0x6c5706b1,0x6c5b1607,0x6c5c1240,0x6c5d10ba,0x6c5e05f3,0x6c5f08a4,0x6c600252,
	0x6c611450,0x6c6412e6,0x6c680d18,0x6c69061f,0x6c6a13e0,0x6c7012c8,0x6c7207fd,0x6c740113,
	0x6c761437,0x6c791597,0x6c7d0f6a,0x6c7e04f1,0x6c810ff1,0x6c8216ca,0x6c831445,0x6c85180e,
	0x6c8606c2,0x6c881170,0x6c890221,0x6c8c044f,0x6c8f0f2e,0x6c900d93,0x6c9312b7,0x6c940d27,
	0x6c9910ff,0x6c9b0e8b,0x6c9f05fa,0x6ca10cd1,0x6ca30500,0x6ca40e4a,0x6ca50b44,0x6ca60c51,
	0x6ca7019a,0x6ca91400,0x6caa0741,0x6cab0d73,0x6cad11ff,0x6cae098b,0x6cb113a8,0x6cb213a3,
	0x6cb306df,0x6cb804e3,0x6cb9178f,0x6cbb1961,0x6cbc18de,0x6cbd0611,0x6cbe18af,0x6cbf1641,
	0x6cc41558,0x6cc5101f,0x6cc91051,0x6cca0159,0x6ccc0d1a,0x6cd00afa,0x6cd30708,0x6cd40580,
	0x6cd504a1,0x6cd60cc2,0x6cd71246,0x6cdb04b6,0x6cde0e17,0x6ce00bc9,0x6ce10e81,0x6ce2014e,
	0x6ce30f6b,0x6ce50ddb,0x6ce819c8,0x6cea0b0d,0x6ceb15dc,0x6cee0e68,0x6cef0d51,0x6cf012cc,
	0x6cf1166d,0x6cf3177c,0x6cf500ca,0x6cf615eb,0x6cf70bf8,0x6cf80c1c,0x6cfa0c64,0x6cfb1559,
	0x6cfc0f04,0x6cfd187a,0x6cfe0933,0x6d0108f5,0x6d04079d,0x6d071724,0x6d0b167a,0x6d0c0ba5,
	0x6d0e0828,0x6d1210d6,0x6d1714ba,0x6d1919a9,0x6d1a08b5,0x6d1b0c65,0x6d1e040e,0x6d250914,
	0x6d271412,0x6d2a0709,0x6d2b15c0,0x6d2e1305,0x6d310493,0x6d321993,0x6d3310be,0x6d3515ff,
	0x6d390761,0x6d3b07c6,0x6d3c13b9,0x6d3d0f75,0x6d3e0e5b,0x6d410be5,0x6d430842,0x6d450f94,
	0x6d4608a8,0x6d4708be,0x6d4818ff,0x6d4a1a01,0x6d4b01ab,0x6d4d07a8,0x6d4e0829,0x6d4f0be4,
	0x6d5107bc,0x6d52073a,0x6d530e23,0x6d541600,0x6d5918f8,0x6d5a09df,0x6d5c0080,0x6d5e1a02,
	0x6d601492,0x6d63076f,0x6d660f25,0x6d6906d1,0x6d6a0ae4,0x6d6e0531,0x6d6f145d,0x6d7417ef,
	0x6d770699,0x6d78092b,0x6d7c0ce3,0x6d821381,0x6d850e02,0x6d881530,0x6d89115c,0x6d8c177f,
	0x6d8e14f0,0x6d911272,0x6d9309ac,0x6d9401ae,0x6d95132a,0x6d9b12ff,0x6d9d0af3,0x6d9e0ab7,
	0x6d9f0b68,0x6da01404,0x6da11440,0x6da30770,0x6da4039b,0x6da610cf,0x6da70896,0x6da818ce,
	0x6da910f7,0x6daa0537,0x6dab064c,0x6dae1215,0x6daf161d,0x6db216b6,0x6db506aa,0x6db806e5,
	0x6dbf19fc,0x6dc003c9,0x6dc41a13,0x6dc51497,0x6dc6153d,0x6dc70f4b,0x6dcb0bb2,0x6dcc12f8,
	0x6dd111e7,0x6dd60dce,0x6dd8130a,0x6dd902eb,0x6ddd04d9,0x6dde1251,0x6de00ecc,0x6de1035d,
	0x6de417af,0x6de60591,0x6deb1731,0x6dec030b,0x6dee0759,0x6df1116b,0x6df302c6,0x6df707c0,
	0x6df91630,0x6dfb1331,0x6dfc0d38,0x6e050ff9,0x6e0a180a,0x6e0c0c30,0x6e0d1a2e,0x6e0e0424,
	0x6e100898,0x6e111184,0x6e1417c3,0x6e161175,0x6e17117a,0x6e1a19be,0x6e1d17b2,0x6e20103c,
	0x6e210433,0x6e231893,0x6e240160,0x6e25144a,0x6e29142a,0x6e2b155f,0x6e2d1425,0x6e2f059b,
	0x6e3215e1,0x6e340a2d,0x6e381795,0x6e3a0d39,0x6e430e5c,0x6e440cd9,0x6e4d138c,0x6e4e0d2b,
	0x6e530e94,0x6e54086c,0x6e56072f,0x6e581512,0x6e5b18c1,0x6e5f0783,0x6e6b08d8,0x6e6e1634,
	0x6e7e13c6,0x6e7f1195,0x6e830a95,0x6e85089c,0x6e8615c6,0x6e890578,0x6e8f12ee,0x6e901818,
	0x6e980a36,0x6e9c0be1,0x6e9f0d5d,0x6ea2170f,0x6ea50f27,0x6ea70b16,0x6eaa14a1,0x6eaf127b,
	0x6eb10fe8,0x6eb2125f,0x6eb415aa,0x6eb610a2,0x6eb707c1,0x6eba0de8,0x6ebb12b3,0x6ebd10bf,
	0x6ec1028d,0x6ec20e6f,0x6ec703b7,0x6ecb1a18,0x6ecf054e,0x6ed10751,0x6ed31a2b,0x6ed41302,
	0x6ed51317,0x6ed700ee,0x6eda067f,0x6ede1974,0x6edf1666,0x6ee01160,0x6ee10ca0,0x6ee21759,
	0x6ee40c7c,0x6ee50ad7,0x6ee60c46,0x6ee80134,0x6ee912d1,0x6ef40393,0x6ef90727,0x6f020edc,
	0x6f060f39,0x6f090c36,0x6f0f0c13,0x6f130b1f,0x6f141656,0x6f1501a2,0x6f200d79,0x6f240ad3,
	0x6f2915d8,0x6f2a16c4,0x6f2b0ca6,0x6f2d0cb3,0x6f2f0c6c,0x6f311208,0x6f3318c8,0x6f360776,
	0x6f3e1684,0x6f46175b,0x6f471536,0x6f4b0b7d,0x6f4d140a,0x6f580e5e,0x6f5c0f92,0x6f5e0c39,
	0x6f620786,0x6f660b8e,0x6f6d12db,0x6f6e020d,0x6f72114e,0x6f7419b2,0x6f78111e,0x6f7a01dd,
	0x6f7c136e,0x6f840240,0x6f880219,0x6f89058c,0x6f8c123b,0x6f8d1209,0x6f8e0ea1,0x6f9c0ac9,
	0x6fa1186f,0x6fa70b35,0x6fb30042,0x6fb601e0,0x6fb912d3,0x6fc007f5,0x6fc20b6e,0x6fc91287,
	0x6fd10abe,0x6fd20138,0x6fde00fc,0x6fe006cc,0x6fe110b5,0x6fee0f20,0x6fef1a09,0x70110f2c,
	0x701a06bc,0x701b175f,0x70231569,0x703504fd,0x70391834,0x704c0653,0x704f06d5,0x705e005d,
	0x706b07c8,0x706c07c7,0x706d0d43,0x706f0383,0x70700792,0x70750bc7,0x70761871,0x70780969,
	0x707c19fe,0x707e1851,0x707f0195,0x70801676,0x70850674,0x70890c1d,0x708a02b9,0x708e1642,
	0x7092020f,0x70941062,0x70950a0c,0x70960450,0x70991962,0x709c1410,0x709d0fb5,0x70ab15dd,
	0x70ac0999,0x70ad12e3,0x70ae0e7d,0x70af095d,0x70b112c3,0x70b30147,0x70b719cc,0x70b818a3,
	0x70b903bc,0x70bb11a0,0x70bc0b78,0x70bd0265,0x70c00723,0x70c11225,0x70c20ad6,0x70c31353,
	0x70c80ba7,0x70ca167b,0x70d80702,0x70d90af4,0x70db19b6,0x70df162d,0x70e40a12,0x70e604ac,
	0x70e71140,0x70e816b4,0x70e907ad,0x70eb12fd,0x70ec092c,0x70ed1085,0x70ef1498,0x70f713cf,
	0x70f90e98,0x70fd0506,0x71091631,0x710a06b5,0x7110147b,0x711306ab,0x71150772,0x71160cf0,
	0x7118037a,0x711900b5,0x711a04f3,0x712608c3,0x712f19f7,0x71301663,0x71311664,0x71361071,
	0x7145043c,0x714a15d2,0x714c0785,0x714e0870,0x715c17fd,0x715e1108,0x71640cdc,0x716615c8,
	0x716718e5,0x716813f6,0x716e19bf,0x71720090,0x71730732,0x71780106,0x717a1395,0x717d111d,
	0x718414a5,0x718a159a,0x718f15f4,0x719410a5,0x71980be2,0x719914a6,0x719f11f3,0x71a01716,
	0x71a8184a,0x71ac002f,0x71b30ca9,0x71b51136,0x71b914ac,0x71c31073,0x71ce0b98,0x71d404af,
	0x71d5166a,0x71e01803,0x71e51876,0x71e71292,0x71ee1568,0x71f914ff,0x720600a1,0x721d09d1,
	0x72280302,0x722a19d6,0x722c0e4f,0x72301810,0x72310011,0x723509cd,0x72360552,0x723716a6,
	0x72380059,0x723903dc,0x723b168b,0x723d1219,0x723f0e60,0x72470ed9,0x72480073,0x724c0e59,
	0x724d0426,0x725203e6,0x7256179f,0x72591616,0x725b0e19,0x725d0ef3,0x725f0d84,0x72610d8d,
	0x72620ae8,0x72660cb8,0x72670d95,0x72691477,0x726e0892,0x726f0623,0x72721181,0x72750f80,
	0x72791311,0x727a1493,0x727e146b,0x727f0632,0x7280149c,0x72810b1c,0x728407eb,0x728a0427,
	0x728b09a1,0x728d086d,0x728f0ed2,0x72920a14,0x729f08b8,0x72ac105d,0x72ad105c,0x72af04b5,
	0x72b0101c,0x72b40024,0x72b619e7,0x72b7065c,0x72b80c87,0x72b9178d,0x72c1183e,0x72c20a72,
	0x72c30e1c,0x72c40397,0x72c800ac,0x72cd0e7c,0x72ce14d0,0x72d0072b,0x72d204e4,0x72d70600,
	0x72d9097b,0x72de0e11,0x72e006f5,0x72e108d0,0x72e8109c,0x72e911d7,0x72ec0429,0x72ed14d3,
	0x72ee1194,0x72ef0a68,0x72f01920,0x72f117ed,0x72f21296,0x72f317c0,0x72f400e6,0x72f709b4,
	0x72f80b19,0x72fa172f,0x72fb1280,0x72fc0ada,0x73010b53,0x730314fa,0x730a0dde,0x730e0ba9,
	0x73130689,0x73150d0a,0x731601ee,0x731716dc,0x731b0cfd,0x731c0182,0x731d02f5,0x731e1150,
	0x73210c56,0x73220730,0x73251416,0x73291580,0x732a19ae,0x732b0cb5,0x732c1426,0x732e150a,
	0x73310dc7,0x73340712,0x73371797,0x73380cda,0x733901bb,0x733e0752,0x733f1819,0x734d0957,
	0x735018c9,0x73520033,0x735709ca,0x73600b96,0x736c1565,0x736d12b5,0x736f15f5,0x737e075e,
	0x738415d4,0x73870c7f,0x738917e4,0x738b13e2,0x738e03ee,0x739107de,0x7396096a,0x739b0c88,
	0x739f0d49,0x73a2013a,0x73a913cd,0x73ab0cd3,0x73ae1411,0x73af075f,0x73b01503,0x73b20bcc,
	0x73b30340,0x73b703c6,0x73ba14bb,0x73bb014f,0x73c00f0d,0x73c20a1a,0x73c80843,0x73c90d4b,
	0x73ca111a,0x73cd1900,0x73cf09bf,0x73d004a3,0x73d10bfb,0x73d905f5,0x73de0c69,0x73e019ac,
	0x73e50495,0x73e71690,0x73e906fb,0x73ed0069,0x73f20797,0x74031023,0x74050adc,0x74060b32,
	0x74090be7,0x740a161a,0x740f0b73,0x741012aa,0x741a0981,0x741b021c,0x74221a06,0x7425073c,
	0x74260f50,0x74280a9c,0x742a0f51,0x742c13d9,0x742e02ec,0x74301654,0x74330bb3,0x74340fe4,
	0x74350ebc,0x74360e51,0x743c1012,0x74410ccb,0x745514d7,0x74571824,0x74590dcc,0x745a0733,
	0x745b1746,0x745c17ce,0x745e10cc,0x745f10fa,0x746d12ef,0x74700666,0x74761696,0x74770018,
	0x747e0923,0x74800307,0x748102e7,0x74830b23,0x748715d9,0x748b18cb,0x748e174c,0x74900c3a,
	0x749c0787,0x749e0f1f,0x74a700ff,0x74a80197,0x74a9103f,0x74ba1438,0x74d21863,0x74dc0637,
	0x74de03e0,0x74e00748,0x74e20ee0,0x74e3007c,0x74e41078,0x74e613bd,0x74ee143c,0x74ef0e43,
	0x74f40bcd,0x74f60eff,0x74f702d6,0x74ff017d,0x75041906,0x750d0cf6,0x750f00cc,0x7511188e,
	0x75130ecf,0x7518057c,0x75190339,0x751a1178,0x751c1335,0x751f117f,0x75251183,0x75281783,
	0x7529120f,0x752b0545,0x752c177a,0x752d00c9,0x752f0e15,0x75301332,0x7531178c,0x75320855,
	0x75331163,0x753503bf,0x75370db5,0x753803c1,0x753a03f4,0x753b0754,0x753e1852,0x754000e0,
	0x75450201,0x754804ba,0x754b1334,0x754c0908,0x754e105e,0x754f141f,0x75540e6b,0x75590be6,
	0x755a00c3,0x755b1910,0x755c02a3,0x75650c4b,0x75660f4c,0x756a04a5,0x75721152,0x7574027d,
	0x757807ee,0x757913db,0x757f07f1,0x7583138f,0x758608ad,0x758b0ec8,0x758f11e9,0x759116d9,
	0x75920dcf,0x759403ef,0x759608e6,0x75970b90,0x759905b5,0x759a0970,0x759d1125,0x759f0e2f,
	0x75a00b49,0x75a11678,0x75a31790,0x75a4004c,0x75a50909,0x75ab1703,0x75ac0b4e,0x75ae02b1,
	0x75af0503,0x75b019ce,0x75b10e82,0x75b20eb7,0x75b30584,0x75b40a1d,0x75b502d1,0x75b80353,
	0x75b91911,0x75bc1314,0x75bd097e,0x75be0805,0x75c20845,0x75c315d5,0x75c418a4,0x75c5014b,
	0x75c7192f,0x75c8176d,0x75c9094f,0x75ca1055,0x75cd16d5,0x75d21680,0x75d4196f,0x75d506f3,
	0x75d61621,0x75d8041d,0x75db1375,0x75de0ec5,0x75e20b5d,0x75e31975,0x75e4031b,0x75e6147d,
	0x75e71105,0x75e80aea,0x75ea0774,0x75eb14f4,0x75f012d8,0x75f104e7,0x75f4024e,0x75f900ef,
	0x75fc0634,0x75ff1417,0x760017b0,0x7601030e,0x760319b9,0x76050356,0x760a0713,0x760c0ab1,
	0x761017e0,0x76150859,0x76171713,0x76180c14,0x761910f5,0x761b026a,0x761f142b,0x7620080d,
	0x7622006e,0x76240bec,0x762501cf,0x762611dc,0x7629032d,0x762a012d,0x762b12d2,0x762d0122,
	0x76300c62,0x76330277,0x763418d9,0x763518ae,0x76381064,0x763c0d7d,0x763e173e,0x763f1765,
	0x7640078a,0x76430c01,0x764c000b,0x764d006f,0x7654171e,0x76560ec7,0x765c03cd,0x765e0abf,
	0x766315db,0x766b03ba,0x766f1042,0x7678066d,0x767b0384,0x767d0060,0x767e0061,0x76821872,
	0x76840382,0x768608e7,0x7687077d,0x76880663,0x768b059f,0x768e08d3,0x7691000a,0x769306d3,
	0x769613da,0x769914a2,0x76a40f08,0x76ae0eb3,0x76b119a0,0x76b209d7,0x76b40312,0x76bf0d4e,
	0x76c217b8,0x76c51981,0x76c60e93,0x76c81752,0x76ca1708,0x76cd06e3,0x76ce002d,0x76cf18b9,
	0x76d01644,0x76d10869,0x76d206e6,0x76d40a81,0x76d60577,0x76d7037b,0x76d80e61,0x76db118b,
	0x76df0cf5,0x76e50652,0x76ee0d92,0x76ef03f0,0x76f115ad,0x76f20cae,0x76f41943,0x76f8150f,
	0x76f9044c,0x76fc0e6a,0x76fe0451,0x77011186,0x77040d2e,0x77070d36,0x7708034b,0x77090cd4,
	0x770b0a01,0x770d0a49,0x771916d3,0x771a1187,0x771f1902,0x77200d23,0x77221808,0x77261a2f,
	0x7728189b,0x772915df,0x772d067a,0x772f0d06,0x7735024a,0x77360a7d,0x773709b7,0x77380d86,
	0x773a1348,0x773c1652,0x774018fb,0x77411922,0x774312a3,0x774703b1,0x77500abc,0x77510882,
	0x775a161e,0x775b093c,0x7761121e,0x77621286,0x77630420,0x77650101,0x77660d9b,0x77680de9,
	0x776b08fb,0x776c0189,0x7779042d,0x777d0a8b,0x777e05a3,0x777f10cd,0x77800ccc,0x77840d33,
	0x77850285,0x778c0a24,0x778d1267,0x778e14cd,0x77910d5f,0x77920c9c,0x779f0ee4,0x77a00231,
	0x77a20cf7,0x77a50ee8,0x77a70fc6,0x77a919c2,0x77aa038d,0x77ac1222,0x77b00a03,0x77b3136f,
	0x77b50bb9,0x77bb18b5,0x77bd062c,0x77bf1038,0x77cd09d0,0x77d702a2,0x77db0cb7,0x77dc0915,
	0x77e211a6,0x77e316e3,0x77e51937,0x77e71172,0x77e9098d,0x77eb08d4,0x77ec031c,0x77ed0437,
	0x77ee000d,0x77f3119b,0x77f607e1,0x77f80587,0x77fd148c,0x77fe04aa,0x77ff0a7a,0x78000369,
	0x78010c89,0x78021102,0x7809074c,0x780c0f6d,0x780d09fe,0x78111628,0x78120ead,0x78141643,
	0x781619d9,0x78170212,0x78180452,0x781a165b,0x781c0504,0x781d04a2,0x781f189c,0x782313aa,
	0x782503a7,0x782618ac,0x78271903,0x78290532,0x782c0b38,0x782d0103,0x78300e97,0x78340f0e,
	0x7837116a,0x78381850,0x78390012,0x783a0b54,0x783b0bfd,0x783c1369,0x783e0b55,0x78400295,
	0x78450665,0x78470dc5,0x784c05d0,0x784e1589,0x78500411,0x78521499,0x78551228,0x785614d4,
	0x78570fb6,0x785d1534,0x786a144b,0x786b0be8,0x786c1767,0x786d0cb0,0x786e1068,0x78770883,
	0x787c0e9e,0x788703fb,0x788903d3,0x788c0c34,0x788d0016,0x788e1290,0x789100a7,0x78930444,
	0x789713dc,0x789803bd,0x789a00b7,0x789b0f6f,0x789c0227,0x789f03e7,0x78a119a4,0x78a308fd,
	0x78a5010e,0x78a700f4,0x78b00ea6,0x78b10889,0x78b203b4,0x78b312e5,0x78b401be,0x78b915e3,
	0x78be0df4,0x78c102d9,0x78c50089,0x78c910eb,0x78ca0b09,0x78cb0317,0x78d00e62,0x78d418f0,
	0x78d50a25,0x78d90680,0x78e80d6d,0x78ec1009,0x78f2103e,0x78f4038e,0x78f70bba,0x78fa078b,
	0x790108c9,0x7905044a,0x791308ac,0x791e0cfa,0x79240181,0x79340167,0x793a11b3,0x793b11af,
	0x793c0b2c,0x793e115a,0x79401245,0x79410f3e,0x794614e3,0x79480f46,0x79491953,0x7953052d,
	0x79561a4d,0x7957193b,0x795a1a66,0x795b1031,0x795c0745,0x795d19cd,0x795e116e,0x795f128d,
	0x796002d3,0x79620d08,0x7965151b,0x7967133f,0x79680ee5,0x796d0830,0x796f1904,0x79770376,
	0x797807d0,0x797a0f52,0x79800149,0x7981091a,0x79840c33,0x798501d8,0x798a14ca,0x798f053d,
	0x799a1a07,0x79a714c2,0x79b31077,0x79b917db,0x79ba17bc,0x79bb0b1a,0x79bd0fe5,0x79be06da,
	0x79c015a5,0x79c11230,0x79c3137d,0x79c60588,0x79c90145,0x79cb1016,0x79cd1988,0x79d10a1b,
	0x79d20d37,0x79d500d6,0x79d80d1c,0x79df1a43,0x79e30d76,0x79e40245,0x79e60fe3,0x79e7166f,
	0x79e9196a,0x79eb11ef,0x79ed1a26,0x79ef07e8,0x79f0022d,0x79f808e9,0x79fb16d6,0x79fd07b1,
	0x7a00149d,0x7a020ade,0x7a03051b,0x7a060c74,0x7a0b023c,0x7a0d1141,0x7a0e121d,0x7a14108c,
	0x7a170067,0x7a1a1978,0x7a1e0a21,0x7a20027f,0x7a23126d,0x7a331435,0x7a370836,0x7a391913,
	0x7a3b037d,0x7a3c085f,0x7a3d07f2,0x7a3f05ab,0x7a460d9e,0x7a5110fb,0x7a571293,0x7a701079,
	0x7a7415e9,0x7a760960,0x7a77100e,0x7a78148d,0x7a79100f,0x7a7a0a40,0x7a7f02a7,0x7a8019f4,
	0x7a81137e,0x7a830fd7,0x7a8418aa,0x7a86010c,0x7a88169b,0x7a8d0fce,0x7a911691,0x7a921970,
	0x7a951347,0x7a9608e1,0x7a9702b2,0x7a98095e,0x7a9c0300,0x7a9d1442,0x7a9f0a54,0x7aa00a22,
	0x7aa50a82,0x7aa6041e,0x7aa8173f,0x7aac17d1,0x7aad09a4,0x7ab317e1,0x7abf0c02,0x7acb0b3d,
	0x7ad61202,0x7ad918bf,0x7ade0950,0x7adf0952,0x7ae018c4,0x7ae309e1,0x7ae5136b,0x7ae61257,
	0x7aed08fe,0x7aef0436,0x7af919b4,0x7afa19b5,0x7afd17bd,0x7aff0583,0x7b03042a,0x7b0407e9,
	0x7b06004e,0x7b080804,0x7b0a18e3,0x7b0b129a,0x7b0f0746,0x7b111545,0x7b1400d8,0x7b15087d,
	0x7b191182,0x7b1b039d,0x7b1e024b,0x7b200b58,0x7b241342,0x7b251249,0x7b260538,0x7b2800c5,
	0x7b2a0331,0x7b2b1a28,0x7b2c03ae,0x7b2e187f,0x7b310603,0x7b330846,0x7b380f0b,0x7b3a086a,
	0x7b3c0bfe,0x7b3e0104,0x7b4514fc,0x7b471011,0x7b490388,0x7b4b0917,0x7b4c1057,0x7b4f04a0,
	0x7b500a71,0x7b5119d1,0x7b521373,0x7b54032c,0x7b5601ac,0x7b580a4e,0x7b5a00ed,0x7b5b1110,
	0x7b5d1924,0x7b6009d9,0x7b620e52,0x7b6e11ca,0x7b711540,0x7b721142,0x7b751646,0x7b770a6a,
	0x7b790280,0x7b7b059d,0x7b7e0f85,0x7b800886,0x7b8500f5,0x7b8d061a,0x7b901008,0x7b940163,
	0x7b9507f0,0x7b971283,0x7b9c0a43,0x7b9d0f91,0x7ba1064e,0x7ba213dd,0x7ba61881,0x7ba70fda,
	0x7ba813b6,0x7ba90c5c,0x7baa0350,0x7bab1537,0x7bac10d3,0x7bad08a2,0x7bb11515,0x7bb41909,
	0x7bb819d3,0x7bc10788,0x7bc619df,0x7bc70ed3,0x7bcc0714,0x7bd10a98,0x7bd30c11,0x7bd905a5,
	0x7bda04e0,0x7bdd05fd,0x7be10301,0x7be50b61,0x7be600f9,0x7bea0259,0x7bee0acc,0x7bf10b26,
	0x7bf70ea2,0x7bfc0415,0x7bfe0d45,0x7c0702f9,0x7c0b0670,0x7c0c127f,0x7c0f0c3b,0x7c16043e,
	0x7c1f03ce,0x7c260386,0x7c27078c,0x7c2a185b,0x7c380169,0x7c3f017e,0x7c4019a3,0x7c410ac0,
	0x7c4d080f,0x7c730d12,0x7c740398,0x7c7b0b0e,0x7c7c14e4,0x7c7d1a23,0x7c8904f5,0x7c91004f,
	0x7c920b59,0x7c950f0f,0x7c9702f1,0x7c9818b2,0x7c9c1349,0x7c9d0b5a,0x7c9e149e,0x7c9f1276,
	0x7ca21a19,0x7ca41830,0x7ca51994,0x7caa04fb,0x7cae0b82,0x7cb10b83,0x7cb20196,0x7cb3093d,
	0x7cb9030f,0x7cbc0bb4,0x7cbd1a39,0x7cbe093f,0x7cc110e4,0x7cc510ab,0x7cc70715,0x7cc815b9,
	0x7cca0728,0x7ccc185a,0x7ccd02dc,0x7cd505a6,0x7cd612f2,0x7cd7102a,0x7cd9019f,0x7cdc0d0d,
	0x7cdf186a,0x7ce00a06,0x7ce808b9,0x7cef0e38,0x7cf80d17,0x7cfb14c4,0x7d0a1434,0x7d201273,
	0x7d2212a9,0x7d27091d,0x7d2b1a2a,0x7d2f0b04,0x7d6e15c7,0x7d771949,0x7da60f55,0x7dae1006,
	0x7e3b0d0e,0x7e4104b0,0x7e471697,0x7e821a51,0x7e9b037e,0x7e9f122d,0x7ea0095f,0x7ea117ad,
	0x7ea20705,0x7ea31999,0x7ea414e1,0x7ea505b4,0x7ea61827,0x7ea707fe,0x7ea813ca,0x7ea90a77,
	0x7eaa081e,0x7eab1091,0x7eac140d,0x7ead1838,0x7eaf02c3,0x7eb00ea9,0x7eb11100,0x7eb20595,
	0x7eb30da4,0x7eb51a38,0x7eb60c52,0x7eb704ec,0x7eb81951,0x7eb9142d,0x7eba04ca,0x7ebd0e1d,
	0x7ebe11e0,0x7ebf1504,0x7ec00590,0x7ec1155a,0x7ec20528,0x7ec30b77,0x7ec41a4b,0x7ec51167,
	0x7ec614c6,0x7ec71938,0x7ec81980,0x7ec9199c,0x7eca007b,0x7ecb0529,0x7ecc029d,0x7ecd114c,
	0x7ece16fe,0x7ecf0934,0x7ed0033a,0x7ed10082,0x7ed2109d,0x7ed308f6,0x7ed40a58,0x7ed51083,
	0x7ed706bf,0x7ed807a9,0x7ed905d2,0x7eda15de,0x7edb08b6,0x7edc0c66,0x7edd09c0,0x7ede08d1,
	0x7edf1370,0x7ee005e0,0x7ee11531,0x7ee209b5,0x7ee315a7,0x7ee51288,0x7ee61300,0x7ee7082a,
	0x7ee8131d,0x7ee9081a,0x7eea15c2,0x7eeb0bd2,0x7eed15c3,0x7eee0f63,0x7eef04d1,0x7ef002ce,
	0x7ef1113c,0x7ef2067d,0x7ef31185,0x7ef41408,0x7ef50d24,0x7ef611db,0x7ef700c7,0x7ef8027c,
	0x7efa0bf0,0x7efb105f,0x7efc1a31,0x7efd18c0,0x7efe13d6,0x7eff0c80,0x7f0019ef,0x7f011a14,
	0x7f020a35,0x7f031513,0x7f04086e,0x7f050d2c,0x7f060ad1,0x7f071320,0x7f080d3a,0x7f0907ec,
	0x7f0b07b4,0x7f0c1235,0x7f0d0461,0x7f0e043a,0x7f0f0117,0x7f1105fc,0x7f1219f1,0x7f130769,
	0x7f1403b2,0x7f150c77,0x7f160105,0x7f170d4c,0x7f181815,0x7f19092e,0x7f1a0561,0x7f1b10c0,
	0x7f1c1912,0x7f1d050d,0x7f1f05a9,0x7f2001da,0x7f210b20,0x7f221710,0x7f230871,0x7f240135,
	0x7f250ee2,0x7f260ca7,0x7f270b00,0x7f281749,0x7f2912a6,0x7f2a0d3f,0x7f2b10ef,0x7f2c1555,
	0x7f2d0b97,0x7f2e112a,0x7f2f188b,0x7f3008ab,0x7f310f98,0x7f320fbc,0x7f330766,0x7f3408dc,
	0x7f351a50,0x7f360514,0x7f380597,0x7f3a1063,0x7f42174a,0x7f44100a,0x7f4514de,0x7f500655,
	0x7f5113e3,0x7f5413e6,0x7f5506ae,0x7f570c55,0x7f58052e,0x7f5a049e,0x7f5f0625,0x7f610599,
	0x7f62005a,0x7f681655,0x7f6918e6,0x7f6a1a55,0x7f6e1979,0x7f710ad4,0x7f7211f6,0x7f740ebe,
	0x7f790b27,0x7f7e188c,0x7f8107f6,0x7f8a1673,0x7f8c0fa3,0x7f8e0ce2,0x7f9405a0,0x7f9a0bd3,
	0x7f9d0391,0x7f9e159f,0x7f9f0fb3,0x7fa11509,0x7fa4106f,0x7fa712a4,0x7faf0900,0x7fb012e7,
	0x7fb214ad,0x7fb80b03,0x7fb905dd,0x7fbc01ea,0x7fbd17d8,0x7fbf1704,0x7fc11439,0x7fc50266,
	0x7fca170b,0x7fcc170c,0x7fce0bd4,0x7fd4151c,0x7fd5149f,0x7fd80fc7,0x7fdf03a0,0x7fe00310,
	0x7fe104df,0x7fe519d4,0x7fe6088a,0x7fe90ed4,0x7fee06eb,0x7ff006bb,0x7ff10035,0x7ff3171b,
	0x7ffb04a7,0x7ffc171c,0x800016a2,0x80010aed,0x80030a0f,0x80040cc8,0x800518f2,0x80060f47,
	0x800b03e4,0x800c048d,0x800d120c,0x80100db1,0x80120b06,0x80141a27,0x801505db,0x80160210,
	0x801706d2,0x8018183c,0x80190e50,0x801c124a,0x802007c2,0x80220af5,0x802512fa,0x80260e47,
	0x80270c0c,0x80280e26,0x802908b2,0x802a0e75,0x80310d81,0x80330491,0x803503f2,0x803616a7,
	0x80370325,0x80381256,0x803b025d,0x803d034c,0x803f05e1,0x80420e03,0x8043034e,0x80460bd5,
	0x804a0b91,0x804b0bff,0x804c1946,0x804d0e13,0x80520686,0x80540b6a,0x80580ef4,0x805a09a5,
	0x80690a99,0x806a02e8,0x80710036,0x807f17e6,0x808017e3,0x80831271,0x80841711,0x8086124c,
	0x808718e7,0x808910ae,0x808b0b0c,0x808c07df,0x80930779,0x8096153b,0x80981997,0x809a0431,
	0x809b0596,0x809c109b,0x809d057e,0x809f1446,0x80a001f3,0x80a10622,0x80a21939,0x80a40518,
	0x80a504d8,0x80a90866,0x80aa04c6,0x80ab19f3,0x80ad0da5,0x80ae002b,0x80af0a38,0x80b105ea,
	0x80b217ea,0x80b4168d,0x80b70f95,0x80ba04e5,0x80bc0944,0x80bd12ca,0x80be1177,0x80bf1987,
	0x80c018d4,0x80c1154d,0x80c21179,0x80c31420,0x80c4199e,0x80c60352,0x80cc00af,0x80cd0639,
	0x80ce12be,0x80d60e76,0x80d7190b,0x80d91a67,0x80da0e84,0x80db0856,0x80dc1188,0x80dd193c,
	0x80de008f,0x80e1072c,0x80e41742,0x80e515ae,0x80e70bfc,0x80e8040f,0x80e909e7,0x80ea0c1f,
	0x80eb094e,0x80ec0e2c,0x80ed162e,0x80ef0a60,0x80f016d4,0x80f1065a,0x80f2069a,0x80f305b7,
	0x80f40410,0x80f608c1,0x80f81598,0x80fa0028,0x80fc0ed6,0x80fd0dd6,0x8102193d,0x81060308,
	0x81090c98,0x810a0814,0x810d0a69,0x810e10d9,0x810f1867,0x81100f48,0x81110dcb,0x81120d15,
	0x81130e24,0x81140c45,0x81160155,0x811813d7,0x811a08d5,0x811e031e,0x812c0e78,0x812f054b,
	0x813113a2,0x81320dff,0x81360c57,0x81380b74,0x813e0ebd,0x8146133a,0x8148093b,0x814a0ab0,
	0x814b16b8,0x814c1635,0x8150054f,0x8151054d,0x815304da,0x81540fa8,0x815513df,0x81591a33,
	0x815a03fa,0x816002ef,0x81651581,0x81671206,0x81690dbb,0x816d0483,0x816e10dc,0x81701689,
	0x8171089d,0x817417ca,0x81790562,0x817a150b,0x817b0dea,0x817c0d2d,0x817d13c0,0x817e1315,
	0x817f1393,0x81800084,0x81820c78,0x818805c8,0x818a0164,0x818f05a4,0x8191013d,0x81980121,
	0x819b12f1,0x819c0d6b,0x819d14aa,0x81a3197b,0x81a60bc4,0x81a80ea3,0x81aa01d1,0x81b3112d,
	0x81ba174e,0x81bb111f,0x81c0139d,0x81c10b6f,0x81c200fd,0x81c31775,0x81c6171d,0x81ca10f0,
	0x81cc062b,0x81e3021f,0x81e71865,0x81ea1a2d,0x81ec0e04,0x81ed0286,0x81f3195a,0x81f4196b,
	0x81fb190a,0x81fc096e,0x81fe17b9,0x8200169c,0x820117be,0x8202026f,0x820414c8,0x82050976,
	0x820617d2,0x820c1154,0x820d1157,0x821011c2,0x821211ea,0x8214133b,0x821b02ad,0x821c1221,
	0x821e146d,0x821f1990,0x822102a9,0x8222111b,0x822316e5,0x82280075,0x822a06c0,0x822b04cb,
	0x822c006a,0x822d00d9,0x822f1983,0x82300897,0x8231019c,0x823319b8,0x82341880,0x82350465,
	0x8236015e,0x823714f2,0x823805cc,0x823902aa,0x823b0c21,0x823e14a0,0x82441143,0x82471360,
	0x82491418,0x824b0cff,0x824f11d4,0x82581263,0x825a01a4,0x825f0271,0x82680cfb,0x826e05d7,
	0x826f0b7e,0x82700867,0x827210f6,0x8273165f,0x82740539,0x8279003d,0x827a16ef,0x827d101a,
	0x827e0010,0x827f0dad,0x828208f0,0x828413cb,0x82880d40,0x828a0f7b,0x828b17e7,0x828d1146,
	0x828e1596,0x828f042e,0x82910f5f,0x82920cad,0x8297150e,0x82980eb4,0x82990524,0x829c145b,
	0x829d1934,0x829f1116,0x82a10f9b,0x82a40a48,0x82a50906,0x82a60c1a,0x82a807e0,0x82a90fe1,
	0x82aa0f42,0x82ab163e,0x82ac04ed,0x82ad004b,0x82ae10c8,0x82af156f,0x82b00822,0x82b1074a,
	0x82b304c0,0x82b41476,0x82b71952,0x82b81839,0x82b90fe2,0x82bd1619,0x82be04e9,0x82c102e3,
	0x82c40114,0x82c7140e,0x82c80b45,0x82ca0478,0x82cb1502,0x82cc01f4,0x82cd019b,0x82ce19c6,
	0x82cf126b,0x82d1181e,0x82d21075,0x82d30bca,0x82d412c2,0x82d51147,0x82d70d31,0x82d81002,
	0x82db0a18,0x82dc0d96,0x82de008e,0x82df0601,0x82e00d4a,0x82e116e4,0x82e30997,0x82e40eea,
	0x82e510d0,0x82e60a56,0x82eb1120,0x82ef00c2,0x82f11760,0x82f4097c,0x82f70581,0x82f90efc,
	0x82fb052a,0x830119ff,0x83020cc5,0x830304b8,0x83040fd2,0x83050cb9,0x83060cbf,0x83070051,
	0x830802d4,0x83090d74,0x830c0255,0x830e0935,0x830f0bf9,0x83110dfb,0x83141751,0x83151010,
	0x83170d5a,0x831a1741,0x831b05d8,0x831c0f9c,0x8327087b,0x832802d5,0x832b0caf,0x832c01b8,
	0x832d08bf,0x832f052f,0x833119aa,0x833308a7,0x8334079e,0x83351725,0x833601b9,0x8338109e,
	0x833910b0,0x833a026e,0x833c1367,0x83401601,0x83431052,0x83460936,0x83471591,0x834901a6,
	0x834f108b,0x83500893,0x835116d1,0x8352077a,0x83540b4f,0x835a084e,0x835b107e,0x835c00e3,
	0x835e0fc0,0x835f07aa,0x8360083b,0x8361036a,0x8363109f,0x836407b9,0x8365158a,0x83660c67,
	0x83671753,0x83680f8a,0x83690929,0x836a1297,0x836b1726,0x836c0c93,0x836d070a,0x836e199f,
	0x836f169e,0x837706e4,0x837800cf,0x837b039c,0x837c1382,0x837d1285,0x83850b56,0x83860f1b,
	0x83890b39,0x838e1103,0x8392098e,0x83930cd5,0x8398116d,0x839b1357,0x839c1791,0x839e13d3,
	0x83a0179d,0x83a80ae2,0x83a90533,0x83aa046f,0x83ab0d77,0x83b009ff,0x83b10ab8,0x83b20b69,
	0x83b311ac,0x83b41441,0x83b614e5,0x83b707cf,0x83b81792,0x83b91754,0x83ba1744,0x83bc02c5,
	0x83bd0cb2,0x83c013d8,0x83c10939,0x83c5086b,0x83c70614,0x83ca0988,0x83cc09d8,0x83cf06e7,
	0x83d4053a,0x83d601ef,0x83d81252,0x83dc018b,0x83dd0052,0x83df138b,0x83e00153,0x83e106b6,
	0x83e5149a,0x83e90f1c,0x83ea036c,0x83f00615,0x83f10bd6,0x83f204d2,0x83f81632,0x83f91a44,
	0x83fd11e8,0x84010f4d,0x8403030c,0x8404130b,0x840600ea,0x840b0f35,0x840c0cf4,0x840d0f00,
	0x840e1415,0x840f035e,0x84110763,0x84180db2,0x841c134c,0x841d0c58,0x84241755,0x84251756,
	0x84261757,0x84271533,0x842810da,0x843115d0,0x843817c4,0x843c0481,0x843d0c6a,0x84460099,
	0x84510507,0x845719cf,0x84591514,0x845a1176,0x845b05c2,0x845c0f73,0x84610f1d,0x84630405,
	0x84690e4d,0x846b0731,0x846c1868,0x846d0848,0x847102e5,0x847313f4,0x84750a88,0x8476135a,
	0x847814bf,0x847a0f6e,0x848203b3,0x848701e5,0x848809f2,0x84890a96,0x848b08b1,0x848c0c0a,
	0x848e0e5d,0x84970ae5,0x84990cf2,0x849c1282,0x84a10088,0x84af0a62,0x84b20f1e,0x84b4122a,
	0x84b81925,0x84b90872,0x84ba080b,0x84bd048a,0x84bf06c3,0x84c11907,0x84c415c9,0x84c910a3,
	0x84ca143b,0x84cd1196,0x84d010c1,0x84d112a5,0x84d300b8,0x84d600f0,0x84dd0ac7,0x84df0831,
	0x84e00b21,0x84e317fe,0x84e5175a,0x84e60d7a,0x84ec0e9f,0x84f014c1,0x84fc0b9b,0x84ff15ca,
	0x850c127d,0x85110d44,0x85130ca8,0x851718f9,0x851a1427,0x851f02f7,0x8521018c,0x852b0dec,
	0x852c11ee,0x85370faf,0x85380414,0x85390b76,0x853a0bc3,0x853b0a4f,0x853c000e,0x853d00f6,
	0x854304ae,0x8548160d,0x854908c8,0x854a10c7,0x8556103d,0x855907b6,0x855e1a56,0x856410c6,
	0x856809cb,0x85720f57,0x8574184b,0x8579143d,0x857a080e,0x857b070e,0x857e0b0a,0x85840166,
	0x858506c5,0x858713f7,0x858f171a,0x859b15e7,0x859c00fa,0x85a41566,0x85a80703,0x85aa1576,
	0x85ae1268,0x85af11f9,0x85b015f6,0x85b710b6,0x85b912c6,0x85c105ad,0x85c9090e,0x85cf019d,
	0x85d00d3b,0x85d314fe,0x85d50e48,0x85dc0b28,0x85e41318,0x85e904a8,0x85fb1870,0x85ff07d5,
	0x860506fe,0x86110d6e,0x86160e0c,0x86271040,0x862904b2,0x863818c2,0x863c0d11,0x864d0720,
	0x864e0739,0x864f0c25,0x86500e30,0x86510c7e,0x86540f8c,0x865a15b0,0x865e17cf,0x8662068c,
	0x866b0272,0x866c101e,0x866e0812,0x86711191,0x8679070b,0x867a07a0,0x867b0cf3,0x867c05cf,
	0x867d1284,0x867e14cc,0x867f01ce,0x868011a1,0x868116e6,0x86820c8a,0x868a142f,0x868b10ca,
	0x868c0081,0x868d0eb8,0x8693173c,0x86950191,0x869c161b,0x869d06c6,0x86a305ed,0x86a4186e,
	0x86a7090c,0x86a80534,0x86a90248,0x86aa041a,0x86ac14fb,0x86af1017,0x86b01794,0x86b118a5,
	0x86b417aa,0x86b50a16,0x86b606a1,0x86ba1070,0x86c019d0,0x86c40616,0x86c61032,0x86c71156,
	0x86c90bd7,0x86ca0627,0x86cb035f,0x86ce0b5b,0x86cf022f,0x86d01034,0x86d10d83,0x86d4079f,
	0x86d8167c,0x86d913bb,0x86db19b0,0x86de0aa7,0x86df08c4,0x86e405c3,0x86e91013,0x86ed1976,
	0x86ee0c9a,0x86f018ed,0x86f10852,0x86f20dc8,0x86f31236,0x86f40f53,0x86f8152a,0x86f91781,
	0x86fe0472,0x870011f8,0x87020509,0x8703117d,0x870718ea,0x8708145e,0x8709053e,0x870a0b3a,
	0x870d028f,0x87121647,0x8713135b,0x87151396,0x87171443,0x8718193e,0x871a04d4,0x871c0d21,
	0x871e0f56,0x87210ab2,0x87220d00,0x87230fa9,0x872514a7,0x87291344,0x872e1801,0x87310ebf,
	0x87341714,0x87371058,0x873b0ffa,0x873e0690,0x873f13c7,0x8747175c,0x87480688,0x874901db,
	0x874c0a26,0x874e154a,0x875317d3,0x87570789,0x87590107,0x87600541,0x87631798,0x87641028,
	0x87650cbd,0x876e0565,0x87700a8c,0x87740736,0x877603e8,0x877b0dbc,0x877c0c0d,0x877d02c2,
	0x877e10a6,0x87820ae0,0x87830e74,0x878514ae,0x8788181c,0x878b1264,0x878d10a7,0x87930fec,
	0x879712f3,0x879f0d60,0x87a80ca1,0x87ab11cd,0x87ac01a5,0x87ad024f,0x87af0037,0x87b312f4,
	0x87b50ede,0x87ba0c5f,0x87bd1986,0x87c01211,0x87c60c8e,0x87ca0cbe,0x87cb14af,0x87d118cc,
	0x87d20cb4,0x87d31529,0x87db0ea4,0x87e00e64,0x87e5077c,0x87ea07b7,0x87ee112f,0x87f9156a,
	0x87fe01de,0x88030c63,0x880a0b71,0x88130d02,0x881510b8,0x881607d6,0x881b0d46,0x88210b2b,
	0x882202c9,0x883209af,0x88390435,0x883c1043,0x884015f0,0x88440e3c,0x8845157c,0x884c1584,
	0x884d164e,0x885414f3,0x885708ed,0x8859161f,0x886106fd,0x88621047,0x886316bd,0x886416bb,
	0x88650173,0x88680127,0x886901c1,0x886b1118,0x886c0228,0x886e067c,0x8870120d,0x88720da7,
	0x88771984,0x887d1096,0x887e0fe0,0x887f0916,0x88811813,0x88820ce7,0x8884003b,0x88850dfc,
	0x88880847,0x888b0343,0x888d0e7e,0x889212df,0x889615a8,0x889c13bf,0x88a20e6c,0x88a40cc9,
	0x88ab00b3,0x88ad14b5,0x88b1053b,0x88b7084c,0x88bc05b8,0x88c10186,0x88c20baa,0x88c519e4,
	0x88c60363,0x88c90a3c,0x88ce023d,0x88d20f12,0x88d41712,0x88d517f9,0x88d81027,0x88d9106e,
	0x88df1106,0x88e20b6b,0x88e30b75,0x88e40a5a,0x88e50884,0x88e800f1,0x88f00459,0x88f10129,
	0x88f3113d,0x88f40e8a,0x88f80c61,0x88f90691,0x88fc132e,0x88fe0983,0x89020641,0x890a010f,
	0x891006ed,0x89120093,0x8913009a,0x891900b9,0x891a0299,0x891b0c79,0x89210328,0x892510c2,
	0x892a1397,0x892b025f,0x89300f88,0x89340aca,0x893618f5,0x89410fb4,0x89441516,0x895e0100,
	0x895f0918,0x896610b7,0x897b0e6d,0x897f1487,0x8981169f,0x89830fed,0x89860567,0x89c1088d,
	0x89c20647,0x89c40662,0x89c50d1b,0x89c611b8,0x89c701d2,0x89c80acf,0x89c909c1,0x89ca082b,
	0x89cb14b6,0x89cc039e,0x89ce17d0,0x89cf060c,0x89d00930,0x89d1104a,0x89d208cc,0x89d609c6,
	0x89da0617,0x89dc1a1b,0x89de1134,0x89e30903,0x89e505f0,0x89e6029f,0x89eb127e,0x89ef197c,
	0x89f30738,0x8a00163f,0x8a070701,0x8a3e1a21,0x8a480b5e,0x8a7918b3,0x8a8917ff,0x8a8a1316,
	0x8a9311cb,0x8b07088b,0x8b261005,0x8b660949,0x8b6c0ed0,0x8ba01639,0x8ba1081b,0x8ba203f7,
	0x8ba30553,0x8ba4108e,0x8ba507d7,0x8ba608f1,0x8ba7070f,0x8ba8130d,0x8ba9107d,0x8baa1123,
	0x8bab0f66,0x8bad1605,0x8bae16f0,0x8baf1606,0x8bb0081c,0x8bb208ae,0x8bb307a7,0x8bb40e40,
	0x8bb50995,0x8bb61624,0x8bb70dd0,0x8bb815b6,0x8bb9046b,0x8bba0c54,0x8bbc1258,0x8bbd050e,
	0x8bbe1159,0x8bbf04c9,0x8bc009bd,0x8bc1192b,0x8bc20620,0x8bc306d6,0x8bc40ef9,0x8bc51a49,
	0x8bc6119d,0x8bc818a0,0x8bc91270,0x8bca190d,0x8bcb03a1,0x8bcc1991,0x8bcd02d2,0x8bce102e,
	0x8bcf18e1,0x8bd116f8,0x8bd216cb,0x8bd30a6f,0x8bd40b07,0x8bd511b9,0x8bd6063f,0x8bd71192,
	0x8bd80810,0x8bd907a3,0x8bda0237,0x8bdb19a7,0x8bdc1168,0x8bdd0755,0x8bde0359,0x8bdf0605,
	0x8be01050,0x8be1066c,0x8be215fc,0x8be316ff,0x8be4192c,0x8be50570,0x8be61519,0x8be701c4,
	0x8be807bf,0x8be915b7,0x8beb090a,0x8bec1455,0x8bed17dc,0x8bee0fcc,0x8bef1478,0x8bf005af,
	0x8bf117a9,0x8bf207ab,0x8bf30a73,0x8bf41223,0x8bf5125a,0x8bf60488,0x8bf71004,0x8bf819ad,
	0x8bf91a3c,0x8bfa0e33,0x8bfb0423,0x8bfc1a03,0x8bfd04dc,0x8bfe0a32,0x8bff1414,0x8c0017c1,
	0x8c01121a,0x8c021173,0x8c0303d8,0x8c0401e2,0x8c050b8a,0x8c0619f5,0x8c07128e,0x8c0812d6,
	0x8c0a1709,0x8c0b0d87,0x8c0c0226,0x8c0d03e1,0x8c0e0790,0x8c0f0899,0x8c101551,0x8c1115f1,
	0x8c1216b7,0x8c131423,0x8c14047d,0x8c1517f4,0x8c1615cd,0x8c1701d6,0x8c181a15,0x8c19001d,
	0x8c1a1661,0x8c1b03af,0x8c1c0d0b,0x8c1d0ed8,0x8c1f0d66,0x8c200366,0x8c211277,0x8c221560,
	0x8c231692,0x8c240087,0x8c2511c7,0x8c260f83,0x8c270d1f,0x8c28091e,0x8c290c9f,0x8c2a18ee,
	0x8c2b0887,0x8c2c0d63,0x8c2d12da,0x8c2e1887,0x8c2f0fc2,0x8c300ac8,0x8c310f28,0x8c3209c9,
	0x8c331668,0x8c340f97,0x8c3518b4,0x8c36022c,0x8c370621,0x8c4107c4,0x8c46041b,0x8c4708a9,
	0x8c49025e,0x8c4c13c8,0x8c5511a7,0x8c5a139c,0x8c611526,0x8c620775,0x8c6a06c9,0x8c6b1802,
	0x8c730139,0x8c78195d,0x8c79009d,0x8c7a01cc,0x8c8203d2,0x8c8515a1,0x8c8906e9,0x8c8a0d7b,
	0x8c8c0ccd,0x8c940eca,0x8c980d80,0x8d1d00ab,0x8d1e18fc,0x8d1f0556,0x8d2105f7,0x8d220185,
	0x8d23187b,0x8d2414ee,0x8d250066,0x8d2618d5,0x8d2707ce,0x8d281963,0x8d2904b9,0x8d2a12cf,
	0x8d2b0eed,0x8d2c010a,0x8d2d0606,0x8d2e19c9,0x8d2f064f,0x8d300499,0x8d310894,0x8d3200bf,
	0x8d3311c0,0x8d34134b,0x8d350675,0x8d360a7b,0x8d370341,0x8d380cc7,0x8d3904e6,0x8d3a06ec,
	0x8d3b16d2,0x8d3c1885,0x8d3d196c,0x8d3e0857,0x8d3f07ae,0x8d401a11,0x8d410bc2,0x8d420c2e,
	0x8d431864,0x8d441a12,0x8d450573,0x8d46092d,0x8d471024,0x8d481919,0x8d490abb,0x8d4a1151,
	0x8d4b0560,0x8d4c042c,0x8d4d07ed,0x8d4e11f1,0x8d4f1139,0x8d5002e0,0x8d5305dc,0x8d540e88,
	0x8d550355,0x8d560abd,0x8d5819f2,0x8d590564,0x8d5a19dd,0x8d5b10df,0x8d5c1882,0x8d5d166b,
	0x8d5e1861,0x8d60188f,0x8d61112e,0x8d62175e,0x8d630592,0x8d640263,0x8d66115d,0x8d670dba,
	0x8d6b06ee,0x8d6d18f4,0x8d701a40,0x8d730962,0x8d74055c,0x8d7518e2,0x8d760589,0x8d770f62,
	0x8d81022a,0x8d840fd4,0x8d850208,0x8d8a1831,0x8d8b1035,0x8d911a1c,0x8d940bab,0x8d9f12fe,
	0x8da3104b,0x8db1185f,0x8db31a45,0x8db40e4b,0x8db5009e,0x8db8044d,0x8dba0519,0x8dbc0881,
	0x8dbe1957,0x8dbf12b0,0x8dc3182f,0x8dc40fa7,0x8dc612c4,0x8dcb0053,0x8dcc03dd,0x8dce13ac,
	0x8dcf0849,0x8dd10e80,0x8dd6194a,0x8dd7051c,0x8dda111c,0x8ddb0168,0x8ddd09a0,0x8dde0b5f,
	0x8ddf05d4,0x8de314fd,0x8de408c5,0x8de80a61,0x8dea0677,0x8deb100b,0x8dec0a90,0x8def0c35,
	0x8df3134a,0x8df5089e,0x8df70fb7,0x8df800f2,0x8df914e7,0x8dfa0467,0x8dfb07ef,0x8dfd0833,
	0x8e0515ec,0x8e090b85,0x8e0a1782,0x8e0c0282,0x8e0f12bc,0x8e1402ca,0x8e1d075b,0x8e1e09a7,
	0x8e1f0258,0x8e22131c,0x8e230165,0x8e29018a,0x8e2a1a34,0x8e2c197d,0x8e2e03be,0x8e2f194c,
	0x8e31045d,0x8e35198a,0x8e3901d0,0x8e3a08a3,0x8e3d0992,0x8e4003e9,0x8e410ed7,0x8e4210ac,
	0x8e441323,0x8e47088c,0x8e480377,0x8e490319,0x8e4a1483,0x8e4b12bd,0x8e510e0a,0x8e520e63,
	0x8e5902fa,0x8e6600cd,0x8e69012f,0x8e6c0387,0x8e6d01b1,0x8e6f04b1,0x8e700293,0x8e72044b,
	0x8e7402fb,0x8e7609cf,0x8e7c0f2b,0x8e7f02ff,0x8e811877,0x8e8519ba,0x8e870292,0x8e8f0bc5,
	0x8e900bac,0x8e9401df,0x8e9c1a4e,0x8e9e156b,0x8eab1165,0x8eac05ee,0x8eaf1033,0x8eb20462,
	0x8eba12fb,0x8ece1421,0x8f660211,0x8f671898,0x8f680669,0x8f6915cb,0x8f6b1093,0x8f6c19db,
	0x8f6d0479,0x8f6e0c53,0x8f6f10c5,0x8f7006ff,0x8f710612,0x8f720a1c,0x8f730c3e,0x8f741996,
	0x8f751956,0x8f761705,0x8f770724,0x8f78190f,0x8f790b50,0x8f7a168f,0x8f7b0ff5,0x8f7c11c3,
	0x8f7d1857,0x8f7e196d,0x8f7f08de,0x8f811054,0x8f820c2f,0x8f8308df,0x8f8418ec,0x8f85054c,
	0x8f860b8b,0x8f870df2,0x8f8800b6,0x8f890798,0x8f8a067e,0x8f8b13e8,0x8f8d02cf,0x8f8e1a1a,
	0x8f8f02f0,0x8f90053f,0x8f91080c,0x8f9311ed,0x8f940e90,0x8f95181a,0x8f9614d8,0x8f9718bc,
	0x8f980c38,0x8f9918f1,0x8f9a0bb7,0x8f9b1570,0x8f9c0618,0x8f9e02d8,0x8f9f0ec9,0x8fa30ab3,
	0x8fa80119,0x8fa9011a,0x8fab011b,0x8fb00222,0x8fb110bc,0x8fb602cd,0x8fb90102,0x8fbd0b8f,
	0x8fbe0329,0x8fc10f7c,0x8fc217ae,0x8fc40f68,0x8fc51608,0x8fc70692,0x8fc80c95,0x8fce1750,
	0x8fd01842,0x8fd10927,0x8fd31625,0x8fd404b4,0x8fd5146f,0x8fd80698,0x8fd918f6,0x8fdb0928,
	0x8fdc181d,0x8fdd1401,0x8fde0b65,0x8fdf0254,0x8fe21341,0x8fe416dd,0x8fe5095c,0x8fe60840,
	0x8fe8033b,0x8fe90492,0x8fea0399,0x8feb0f0c,0x8fed03de,0x8fee187c,0x8ff01200,0x8ff3094d,
	0x8ff70d09,0x8ff800cb,0x8ff90819,0x8ffd19ea,0x90001394,0x9001125b,0x900211c1,0x90031306,
	0x90040e72,0x9005071b,0x90060de6,0x900915da,0x900a160a,0x900b016d,0x900d1532,0x900f137b,
	0x901019b7,0x90111022,0x901203ad,0x90141383,0x9016132b,0x9017041c,0x901a1361,0x901b065d,
	0x901d11c4,0x901e0243,0x901f1274,0x90201874,0x9021106d,0x9022050c,0x90260b31,0x902d0773,
	0x902e0344,0x902f0c31,0x90350a84,0x903613f2,0x9038170d,0x903b0c59,0x903c00ce,0x903e17cb,
	0x90410455,0x9042128f,0x904402ab,0x904717fa,0x904d0118,0x904f0482,0x905014d5,0x90510784,
	0x90521026,0x9053037c,0x905716d7,0x9058060b,0x905b0bf2,0x906212ba,0x90630f96,0x90651695,
	0x90680034,0x906d1869,0x906e18e8,0x90740bb6,0x90751a59,0x907d09a8,0x907f00fb,0x9080168a,
	0x90821567,0x90831294,0x90880d3c,0x908b0aac,0x909116f9,0x9093038a,0x9095176e,0x909706a5,
	0x90990cab,0x909b100d,0x909d0a75,0x90a104be,0x90a21585,0x90a30da3,0x90a6007d,0x90aa154c,
	0x90ac1451,0x90ae178e,0x90af06a7,0x90b012c0,0x90b11015,0x90b30eaa,0x90b40144,0x90b5114b,
	0x90b600ad,0x90b803a2,0x90b91a3a,0x90ba16b1,0x90bb0baf,0x90be19a8,0x90c117eb,0x90c40fdc,
	0x90c51964,0x90c70760,0x90ca08bb,0x90ce0ad9,0x90cf084d,0x90d00a66,0x90d1192d,0x90d31843,
	0x90d7148e,0x90db0530,0x90dc05b0,0x90dd06ce,0x90e109dc,0x90e21761,0x90e60b51,0x90e7183b,
	0x90e8017b,0x90eb0eb9,0x90ed0684,0x90ef12d7,0x90f4021b,0x90f8034d,0x90fd0412,0x90fe1653,
	0x9102047e,0x910409b8,0x911900da,0x911e1733,0x91221636,0x912318c5,0x912f1129,0x91310f07,
	0x91391a3e,0x91430bdb,0x9146050a,0x9149179c,0x914a03f3,0x914b1021,0x914c1a04,0x914d0e8f,
	0x914e19a1,0x914f16de,0x91500585,0x9152096c,0x915715c4,0x915a04ef,0x915d1846,0x915e12cd,
	0x916113ad,0x916202f6,0x916306a2,0x91640619,0x9165126c,0x91690d61,0x916a0af6,0x916c0281,
	0x916e136c,0x916f1959,0x917014e8,0x917108b7,0x9172023f,0x91741385,0x917508e2,0x91760cdd,
	0x91770a5b,0x91781281,0x91790b0f,0x917d1667,0x917e1197,0x917f0df9,0x91850e85,0x918702c8,
	0x91891a57,0x918b02f8,0x918c0a9f,0x918d1324,0x91900737,0x919115ba,0x9192158b,0x919a0d0c,
	0x919b1059,0x91a2069b,0x91a312f5,0x91aa0aec,0x91ad0170,0x91ae08e4,0x91af14b0,0x91b40b36,
	0x91b509a9,0x91ba15f8,0x91c70187,0x91c917ab,0x91ca11c8,0x91cc0b2e,0x91cd198d,0x91ce16ac,
	0x91cf0b86,0x91d10913,0x91dc054a,0x9274089f,0x928e100c,0x92ae0c47,0x92c81482,0x933e1862,
	0x936a0d88,0x938f0bee,0x93ca0044,0x93d60039,0x943e00bb,0x946b1578,0x94850912,0x9486056b,
	0x948716e2,0x948818fd,0x948903f1,0x948a18da,0x948b0f02,0x948c0b9a,0x948d1388,0x948e0f7f,
	0x948f02b0,0x94901119,0x949204ab,0x949303d7,0x94940ced,0x94950e3a,0x949701c8,0x94990576,
	0x949a017a,0x949b12cb,0x949c099a,0x949d0453,0x949e0207,0x949f1982,0x94a00da8,0x94a100b0,
	0x94a20598,0x94a30074,0x94a40f8b,0x94a516a3,0x94a60fdf,0x94a709d6,0x94a81456,0x94a905fb,
	0x94aa0a0d,0x94ab04c2,0x94ac07ca,0x94ad0418,0x94ae0e1e,0x94af0056,0x94b017f0,0x94b10f8d,
	0x94b21921,0x94b30f8e,0x94b40626,0x94b50151,0x94b60a1e,0x94b70f0a,0x94b8016e,0x94b9015c,
	0x94ba182d,0x94bb1a4f,0x94bc0d97,0x94bd12e0,0x94be0858,0x94bf03c7,0x94c01793,0x94c1134d,
	0x94c2015d,0x94c30bcf,0x94c41227,0x94c50f82,0x94c60cc4,0x94c811c5,0x94c915e0,0x94ca12b1,
	0x94cb00e7,0x94cc0ddd,0x94cd0eb2,0x94ce045c,0x94d00a13,0x94d10af1,0x94d20496,0x94d5179e,
	0x94d6023b,0x94d70851,0x94d816a9,0x94d90dc6,0x94db0361,0x94dc136a,0x94dd0c75,0x94de03da,
	0x94df172a,0x94e009f0,0x94e1189a,0x94e219af,0x94e314bd,0x94e4135f,0x94e503fe,0x94e70750,
	0x94e81056,0x94e91104,0x94ea0693,0x94eb03db,0x94ec05d1,0x94ed0d5c,0x94ee1923,0x94ef10f9,
	0x94f008d6,0x94f116c1,0x94f201e3,0x94f30275,0x94f412fc,0x94f50022,0x94f61732,0x94f710b1,
	0x94f819d2,0x94f90aeb,0x94fa0f18,0x94fc0ab9,0x94fd1312,0x94fe0b7b,0x94ff0a3f,0x95001535,
	0x950112ab,0x95020b33,0x9503188d,0x9504028e,0x95050687,0x950605b1,0x95070470,0x950815a9,
	0x95090322,0x950a0c4c,0x950b0508,0x950c1573,0x950d0bf1,0x950e09eb,0x950f0885,0x951010cb,
	0x9511131b,0x95120adf,0x95130fee,0x95140982,0x95150002,0x95160faa,0x951718f3,0x95180e36,
	0x95190323,0x951a0cbb,0x951b00c0,0x951d0380,0x951e0a37,0x951f0a9d,0x952114a3,0x95220635,
	0x95230c5b,0x952402bf,0x952519ed,0x9526091f,0x952814e9,0x952909b0,0x952a07cc,0x952b0e89,
	0x952c12d9,0x952d03fc,0x952e08a0,0x952f09a3,0x95300cfe,0x95311a1d,0x95320fdb,0x953409f4,
	0x95350fab,0x95361238,0x95370484,0x953801b6,0x95390fba,0x953a1985,0x953b043d,0x953c1262,
	0x953e0764,0x953f0008,0x95400434,0x95410ce4,0x95420c15,0x954404e8,0x95450cde,0x95460d7e,
	0x9547191a,0x954905c9,0x954a0e07,0x954c09ae,0x954d0e08,0x954e0da1,0x954f0bed,0x955005ac,
	0x9551008a,0x95521717,0x9553084b,0x95540137,0x95560123,0x955712e8,0x95580caa,0x95590c5e,
	0x955b1774,0x955c0959,0x955d0396,0x955e1a48,0x955f15e4,0x95611579,0x956209ce,0x95630b9d,
	0x95640f21,0x95650c29,0x95660446,0x95670acd,0x95680f2a,0x956902fe,0x956a0fb1,0x956b038f,
	0x956c07d3,0x956d0b02,0x956f1a0a,0x95700b70,0x9571171f,0x957201c0,0x95730126,0x95761518,
	0x957f01f2,0x95e80ceb,0x95e91212,0x95ea1121,0x95eb163b,0x95ed00de,0x95ee1436,0x95ef02b5,
	0x95f010ce,0x95f11402,0x95f214ec,0x95f30707,0x95f40865,0x95f50d4f,0x95f60a0b,0x95f70cef,
	0x95f81899,0x95f90dcd,0x95fa0664,0x95fb142e,0x95fc12b9,0x95fd0d52,0x95fe0c6f,0x9600049f,
	0x960105be,0x960206e1,0x96030aa3,0x96040963,0x9605182e,0x96060ae3,0x960817f5,0x96091633,
	0x960a01f0,0x960b14c7,0x960c1430,0x960d07bb,0x960e1645,0x960f047f,0x961001e4,0x96110ac6,
	0x96121049,0x96140aa8,0x96151069,0x961606ea,0x96171337,0x9619106a,0x961a0a02,0x961c0559,
	0x961d0551,0x961f0440,0x96210f79,0x96221474,0x962a0070,0x962e10c3,0x96310942,0x963204c3,
	0x96331674,0x96341722,0x96351915,0x963608e5,0x963b1a4a,0x963c1a63,0x963d03c2,0x963f0000,
	0x964013a6,0x964200a3,0x96440557,0x96450823,0x96460c2c,0x96470c03,0x96480223,0x96491587,
	0x964b0c12,0x964c0d75,0x964d08b4,0x96501505,0x96540571,0x96551122,0x965b00e4,0x965f1967,
	0x96610419,0x96621820,0x9664028b,0x96670e01,0x9668183f,0x966914f9,0x966a0e86,0x966c1a3d,
	0x967202bb,0x96740eb1,0x96750bd0,0x96761308,0x96771507,0x968517c5,0x96860c00,0x968813f3,
	0x968b1289,0x968d077f,0x968f128a,0x9690173d,0x969405c4,0x96970a8e,0x96980013,0x969914c9,
	0x969c18d6,0x96a71291,0x96b014b8,0x96b3079b,0x96b60b4a,0x96b919e9,0x96bc129b,0x96bd09b6,
	0x96be0db7,0x96c01067,0x96c11665,0x96c41599,0x96c51622,0x96c60809,0x96c70633,0x96c9197a,
	0x96cc02da,0x96cd1770,0x96ce0984,0x96cf0290,0x96d20c6d,0x96d503d4,0x96e00283,0x96e817d9,
	0x96e917c6,0x96ea15ee,0x96ef1431,0x96f30b60,0x96f60bd8,0x96f70afe,0x96f90092,0x96fe147f,
	0x970015b2,0x97010834,0x97041538,0x9706135c,0x9707191b,0x97080e91,0x97090ce0,0x970d07d2,
	0x970e110e,0x970f04d5,0x97130ddf,0x97160bb8,0x971c1217,0x971e14d9,0x972a1736,0x972d000f,
	0x9730150c,0x97320c2b,0x9738005c,0x97390eb0,0x973e0c91,0x97520ff3,0x97530954,0x97560955,
	0x97590958,0x975b03cc,0x975e04cf,0x97600a15,0x97610d10,0x97620d2f,0x976516b9,0x976905bf,
	0x9773092f,0x977415e6,0x97760057,0x977c032e,0x97851671,0x978b1556,0x978d001f,0x9791032f,
	0x97920fc4,0x97940c9d,0x97980fd0,0x97a00985,0x97a310ad,0x97ab0986,0x97ad0109,0x97af0875,
	0x97b205fe,0x97b400ba,0x97e613fc,0x97e71094,0x97e906ad,0x97ea1419,0x97eb1848,0x97ec1303,
	0x97ed096b,0x97f31727,0x97f51849,0x97f61148,0x987516b0,0x987603f5,0x98771003,0x987806a0,
	0x98791525,0x987a1220,0x987b15af,0x987c15bb,0x987d13ce,0x987e062f,0x987f0454,0x98800f49,
	0x9881006b,0x9882125c,0x988306c1,0x988417f1,0x98850c22,0x98860bdf,0x98870f05,0x98880945,
	0x988908fa,0x988a0853,0x988c06e8,0x988d1762,0x988f0a20,0x989016d8,0x98910eef,0x98931392,
	0x989406b7,0x98961763,0x98970a23,0x98981322,0x989a0474,0x989b19da,0x989c1648,0x989d0473,
	0x989e0e09,0x989f0c99,0x98a003b8,0x98a110ec,0x98a206d4,0x98a401e9,0x98a510b9,0x98a60ef0,
	0x98a7105b,0x98ce04ff,0x98d1011e,0x98d210d8,0x98d309a2,0x98d51261,0x98d80edd,0x98d90124,
	0x98da0125,0x98de04cd,0x98df11a2,0x98e71298,0x98e81520,0x990d1669,0x9910018f,0x992e134f,
	0x99541777,0x99551304,0x9963119a,0x996507da,0x99671582,0x9968139b,0x996914c5,0x996a1095,
	0x996b17e9,0x996c0264,0x996d04b7,0x996e173b,0x996f0890,0x997011ba,0x99710095,0x99721247,
	0x997416ce,0x99750494,0x9976107f,0x9977151f,0x997a08d2,0x997c0148,0x997d0152,0x997f047c,
	0x998017c2,0x99810dd2,0x998407bd,0x99851508,0x9986064d,0x998701b5,0x99880a97,0x998a1260,
	0x998b01d9,0x998d0d68,0x998f0bf3,0x999015a2,0x99910921,0x99920c9b,0x999310e5,0x999419e0,
	0x99950dbf,0x999611d3,0x99970a85,0x9998068d,0x99991510,0x99a50568,0x99a81577,0x9a6c0c86,
	0x9a6d17e5,0x9a6e13a4,0x9a6f1604,0x9a700253,0x9a71102f,0x9a730157,0x9a740c6e,0x9a751866,
	0x9a7611aa,0x9a771248,0x9a78055a,0x9a79097d,0x9a7a1a3b,0x9a7b19ca,0x9a7c13a9,0x9a7d0e29,
	0x9a7e085c,0x9a7f1700,0x9a800347,0x9a81152e,0x9a820c8c,0x9a8408c0,0x9a85074f,0x9a860c68,
	0x9a87069d,0x9a880ed5,0x9a8a0b1b,0x9a8b0244,0x9a8c1660,0x9a8f09e0,0x9a900f4e,0x9a910f3a,
	0x9a920a34,0x9a9319eb,0x9a96018e,0x9a970eda,0x9a981977,0x9a9a10ee,0x9a9b147e,0x9a9c003a,
	0x9a9d0bea,0x9a9e0f86,0x9a9f1128,0x9aa00120,0x9aa10c5d,0x9aa202e6,0x9aa301e7,0x9aa419a2,
	0x9aa5083a,0x9aa71517,0x9aa80624,0x9ab0137a,0x9ab1090d,0x9ab603a8,0x9ab70a55,0x9ab80697,
	0x9aba0716,0x9abc05ca,0x9ac000fe,0x9ac10a27,0x9ac20f76,0x9ac50c0e,0x9acb0a6c,0x9acc013e,
	0x9ad10421,0x9ad3128b,0x9ad805a1,0x9adf011f,0x9ae10a9e,0x9ae60cbc,0x9aeb1345,0x9aed1a1f,
	0x9aef1072,0x9af915a3,0x9afb0839,0x9b031a35,0x9b08105a,0x9b0f0966,0x9b13013f,0x9b1f0767,
	0x9b230bad,0x9b2f0203,0x9b3205c1,0x9b3b1805,0x9b3c066e,0x9b410a8a,0x9b4207be,0x9b430054,
	0x9b440f10,0x9b450cea,0x9b471657,0x9b481539,0x9b490b88,0x9b4d13e9,0x9b4f1429,0x9b510250,
	0x9b540d6f,0x9c7c17ba,0x9c7f1796,0x9c810c27,0x9c8204c7,0x9c85005b,0x9c860f01,0x9c870dee,
	0x9c880c23,0x9c8b0563,0x9c8d009f,0x9c8e071e,0x9c9012c5,0x9c910667,0x9c9208ff,0x9c94141a,
	0x9c95048f,0x9c9a0835,0x9c9b08c7,0x9c9c14ea,0x9c9e1522,0x9c9f1603,0x9ca005e3,0x9ca10b24,
	0x9ca20b6d,0x9ca30873,0x9ca40b34,0x9ca511a4,0x9ca61346,0x9ca70681,0x9ca81107,0x9ca90777,
	0x9cab0837,0x9cad0ffb,0x9cae0bda,0x9cb01a3f,0x9cb104d6,0x9cb20aa0,0x9cb301f1,0x9cb40636,
	0x9cb50de0,0x9cb60def,0x9cb703d5,0x9cb80940,0x9cba1198,0x9cbb1a20,0x9cbc04fc,0x9cbd03ea,
	0x9cc310de,0x9cc40487,0x9cc51019,0x9cc60566,0x9cc7078d,0x9cca0108,0x9ccb10f1,0x9ccc0038,
	0x9ccd0f58,0x9cce12b6,0x9ccf064b,0x9cd01698,0x9cd30afc,0x9cd4012a,0x9cd515ef,0x9cd6012c,
	0x9cd70c9e,0x9cd80d56,0x9cd91776,0x9cdc0678,0x9cdd1130,0x9cde0bbb,0x9cdf1a5b,0x9ce20b37,
	0x9e1f0dfa,0x9e200961,0x9e2107e2,0x9e221806,0x9e230d59,0x9e250e44,0x9e261613,0x9e280097,
	0x9e291916,0x9e2a0613,0x9e2b0404,0x9e2c0c20,0x9e2d1615,0x9e2f1670,0x9e310249,0x9e32103b,
	0x9e331809,0x9e3513ab,0x9e361233,0x9e371971,0x9e38048e,0x9e39063a,0x9e3a15a0,0x9e3d05b9,
	0x9e3e0c44,0x9e3f070c,0x9e410161,0x9e420b1e,0x9e4309ad,0x9e440628,0x9e450471,0x9e4617fb,
	0x9e4714f5,0x9e481321,0x9e49146c,0x9e4a106b,0x9e4b0d34,0x9e4c001e,0x9e4e00a8,0x9e4f0ea0,
	0x9e5102c7,0x9e550734,0x9e570485,0x9e58061d,0x9e5a02db,0x9e5b0cdf,0x9e5c1481,0x9e5e16a0,
	0x9e630874,0x9e6406ef,0x9e66174d,0x9e6718fa,0x9e680bf5,0x9e690b99,0x9e6a08ca,0x9e6b0978,
	0x9e6c1804,0x9e6d0c3c,0x9e70174f,0x9e710749,0x9e730654,0x9e7e031d,0x9e7f0c32,0x9e820818,
	0x9e8709da,0x9e8819c1,0x9e8b0d0f,0x9e920f59,0x9e930c3d,0x9e9d1161,0x9e9f0bbc,0x9ea60c96,
	0x9eb41036,0x9eb8051a,0x9ebb0c84,0x9ebd0cd0,0x9ebe0799,0x9ec40780,0x9ec9070d,0x9ecd11f5,
	0x9ece0b25,0x9ecf0df0,0x9ed106f1,0x9ed40f93,0x9ed80d7f,0x9edb0346,0x9edc02a1,0x9edd17a0,
	0x9edf16c6,0x9ee014da,0x9ee21037,0x9ee51001,0x9ee70b29,0x9ee90428,0x9eea0194,0x9eef002a,
	0x9ef91958,0x9efb0542,0x9efc0550,0x9efe0d4d,0x9f0b1816,0x9f0d13af,0x9f0e03f6,0x9f100db3,
	0x9f130629,0x9f17130c,0x9f190ec0,0x9f2011f7,0x9f2204f4,0x9f2c17ac,0x9f2f145f,0x9f3714b2,
	0x9f391658,0x9f3b00d0,0x9f3d1029,0x9f3e06a4,0x9f441895,0x9f500f3f,0x9f5107f3,0x9f7f025c,
	0x9f800229,0x9f830991,0x9f840bd9,0x9f850091,0x9f861343,0x9f871a1e,0x9f881735,0x9f8917e2,
	0x9f8a02d0,0x9f8b1046,0x9f8c144d,0x9f990bf6,0x9f9a05ef,0x9f9b09f8,0x9f9f0661,0x9fa01833,
	0xff011a6b,0xff021a6c,0xff031a6d,0xff041a6e,0xff051a6f,0xff061a70,0xff071a71,0xff081a72,
	0xff091a73,0xff0a1a74,0xff0b1a75,0xff0c1a76,0xff0d1a77,0xff0e1a78,0xff0f1a79,0xff101a7a,
	0xff111a7b,0xff121a7c,0xff131a7d,0xff141a7e,0xff151a7f,0xff161a80,0xff171a81,0xff181a82,
	0xff191a83,0xff1a1a84,0xff1b1a85,0xff1c1a86,0xff1d1a87,0xff1e1a88,0xff1f1a89,0xff201a8a,
	0xff211a8b,0xff221a8c,0xff231a8d,0xff241a8e,0xff251a8f,0xff261a90,0xff271a91,0xff281a92,
	0xff291a93,0xff2a1a94,0xff2b1a95,0xff2c1a96,0xff2d1a97,0xff2e1a98,0xff2f1a99,0xff301a9a,
	0xff311a9b,0xff321a9c,0xff331a9d,0xff341a9e,0xff351a9f,0xff361aa0,0xff371aa1,0xff381aa2,
	0xff391aa3,0xff3a1aa4,0xff3b1aa5,0xff3d1aa6,0xff3e1aa7,0xff3f1aa8,0xff401aa9,0xff411aaa,
	0xff421aab,0xff431aac,0xff441aad,0xff451aae,0xff461aaf,0xff471ab0,0xff481ab1,0xff491ab2,
	0xff4a1ab3,0xff4b1ab4,0xff4c1ab5,0xff4d1ab6,0xff4e1ab7,0xff4f1ab8,0xff501ab9,0xff511aba,
	0xff521abb,0xff531abc,0xff541abd,0xff551abe,0xff561abf,0xff571ac0,0xff581ac1,0xff591ac2,
	0xff5a1ac3,0xff5b1ac4,0xff5c1ac5,0xff5d1ac6,0xff5e1ac7,
	};

static const TCollationKeyTable TheChineseTable = 
	{ TheChineseKey, TheChineseIndex, 6861, NULL, NULL, 0 };

static const TCollationMethod TheChineseMethod =
	{ 0, NULL, &TheChineseTable, 0 };

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// surrogate
// generated from gb18030_allkeys.txt.
// generated by coltab.exe.


static const TUint32 TheGB18030Key[] = 
	{	 // 133770 keys
	0x4e01001d,0x4e02001d,0x4e03001d,0x4e04001d,0x4e05001d,0x4e06001d,0x4e07001d,0x4e08001d,
	0x4e09001d,0x4e0a001d,0x4e0b001d,0x4e0c001d,0x4e0d001d,0x4e0e001d,0x4e0f001d,0x4e10001d,
	0x4e11001d,0x4e12001d,0x4e13001d,0x4e14001d,0x4e15001d,0x4e16001d,0x4e17001d,0x4e18001d,
	0x4e19001d,0x4e1a001d,0x4e1b001d,0x4e1c001d,0x4e1d001d,0x4e1e001d,0x4e1f001d,0x4e20001d,
	0x4e21001d,0x4e22001d,0x4e23001d,0x4e24001d,0x4e25001d,0x4e26001d,0x4e27001d,0x4e28001d,
	0x4e29001d,0x4e2a001d,0x4e2b001d,0x4e2c001d,0x4e2d001d,0x4e2e001d,0x4e2f001d,0x4e30001d,
	0x4e31001d,0x4e32001d,0x4e33001d,0x4e34001d,0x4e35001d,0x4e36001d,0x4e37001d,0x4e38001d,
	0x4e39001d,0x4e3a001d,0x4e3b001d,0x4e3c001d,0x4e3d001d,0x4e3e001d,0x4e3f001d,0x4e40001d,
	0x4e41001d,0x4e42001d,0x4e43001d,0x4e44001d,0x4e45001d,0x4e46001d,0x4e47001d,0x4e48001d,
	0x4e49001d,0x4e4a001d,0x4e4b001d,0x4e4c001d,0x4e4d001d,0x4e4e001d,0x4e4f001d,0x4e50001d,
	0x4e51001d,0x4e52001d,0x4e53001d,0x4e54001d,0x4e55001d,0x4e56001d,0x4e57001d,0x4e58001d,
	0x4e59001d,0x4e5a001d,0x4e5b001d,0x4e5c001d,0x4e5d001d,0x4e5e001d,0x4e5f001d,0x4e60001d,
	0x4e61001d,0x4e62001d,0x4e63001d,0x4e64001d,0x4e65001d,0x4e66001d,0x4e67001d,0x4e68001d,
	0x4e69001d,0x4e6a001d,0x4e6b001d,0x4e6c001d,0x4e6d001d,0x4e6e001d,0x4e6f001d,0x4e70001d,
	0x4e71001d,0x4e72001d,0x4e73001d,0x4e74001d,0x4e75001d,0x4e76001d,0x4e77001d,0x4e78001d,
	0x4e79001d,0x4e7a001d,0x4e7b001d,0x4e7c001d,0x4e7d001d,0x4e7e001d,0x4e7f001d,0x4e80001d,
	0x4e81001d,0x4e82001d,0x4e83001d,0x4e84001d,0x4e85001d,0x4e86001d,0x4e87001d,0x4e88001d,
	0x4e89001d,0x4e8a001d,0x4e8b001d,0x4e8c001d,0x4e8d001d,0x4e8e001d,0x4e8f001d,0x4e90001d,
	0x4e91001d,0x4e92001d,0x4e93001d,0x4e94001d,0x4e95001d,0x4e96001d,0x4e97001d,0x4e98001d,
	0x4e99001d,0x4e9a001d,0x4e9b001d,0x4e9c001d,0x4e9d001d,0x4e9e001d,0x4e9f001d,0x4ea0001d,
	0x4ea1001d,0x4ea2001d,0x4ea3001d,0x4ea4001d,0x4ea5001d,0x4ea6001d,0x4ea7001d,0x4ea8001d,
	0x4ea9001d,0x4eaa001d,0x4eab001d,0x4eac001d,0x4ead001d,0x4eae001d,0x4eaf001d,0x4eb0001d,
	0x4eb1001d,0x4eb2001d,0x4eb3001d,0x4eb4001d,0x4eb5001d,0x4eb6001d,0x4eb7001d,0x4eb8001d,
	0x4eb9001d,0x4eba001d,0x4ebb001d,0x4ebc001d,0x4ebd001d,0x4ebe001d,0x4ebf001d,0x4ec0001d,
	0x4ec1001d,0x4ec2001d,0x4ec3001d,0x4ec4001d,0x4ec5001d,0x4ec6001d,0x4ec7001d,0x4ec8001d,
	0x4ec9001d,0x4eca001d,0x4ecb001d,0x4ecc001d,0x4ecd001d,0x4ece001d,0x4ecf001d,0x4ed0001d,
	0x4ed1001d,0x4ed2001d,0x4ed3001d,0x4ed4001d,0x4ed5001d,0x4ed6001d,0x4ed7001d,0x4ed8001d,
	0x4ed9001d,0x4eda001d,0x4edb001d,0x4edc001d,0x4edd001d,0x4ede001d,0x4edf001d,0x4ee0001d,
	0x4ee1001d,0x4ee2001d,0x4ee3001d,0x4ee4001d,0x4ee5001d,0x4ee6001d,0x4ee7001d,0x4ee8001d,
	0x4ee9001d,0x4eea001d,0x4eeb001d,0x4eec001d,0x4eed001d,0x4eee001d,0x4eef001d,0x4ef0001d,
	0x4ef1001d,0x4ef2001d,0x4ef3001d,0x4ef4001d,0x4ef5001d,0x4ef6001d,0x4ef7001d,0x4ef8001d,
	0x4ef9001d,0x4efa001d,0x4efb001d,0x4efc001d,0x4efd001d,0x4efe001d,0x4eff001d,0x4f00001d,
	0x4f01001d,0x4f02001d,0x4f03001d,0x4f04001d,0x4f05001d,0x4f06001d,0x4f07001d,0x4f08001d,
	0x4f09001d,0x4f0a001d,0x4f0b001d,0x4f0c001d,0x4f0d001d,0x4f0e001d,0x4f0f001d,0x4f10001d,
	0x4f11001d,0x4f12001d,0x4f13001d,0x4f14001d,0x4f15001d,0x4f16001d,0x4f17001d,0x4f18001d,
	0x4f19001d,0x4f1a001d,0x4f1b001d,0x4f1c001d,0x4f1d001d,0x4f1e001d,0x4f1f001d,0x4f20001d,
	0x4f21001d,0x4f22001d,0x4f23001d,0x4f24001d,0x4f25001d,0x4f26001d,0x4f27001d,0x4f28001d,
	0x4f29001d,0x4f2a001d,0x4f2b001d,0x4f2c001d,0x4f2d001d,0x4f2e001d,0x4f2f001d,0x4f30001d,
	0x4f31001d,0x4f32001d,0x4f33001d,0x4f34001d,0x4f35001d,0x4f36001d,0x4f37001d,0x4f38001d,
	0x4f39001d,0x4f3a001d,0x4f3b001d,0x4f3c001d,0x4f3d001d,0x4f3e001d,0x4f3f001d,0x4f40001d,
	0x4f41001d,0x4f42001d,0x4f43001d,0x4f44001d,0x4f45001d,0x4f46001d,0x4f47001d,0x4f48001d,
	0x4f49001d,0x4f4a001d,0x4f4b001d,0x4f4c001d,0x4f4d001d,0x4f4e001d,0x4f4f001d,0x4f50001d,
	0x4f51001d,0x4f52001d,0x4f53001d,0x4f54001d,0x4f55001d,0x4f56001d,0x4f57001d,0x4f58001d,
	0x4f59001d,0x4f5a001d,0x4f5b001d,0x4f5c001d,0x4f5d001d,0x4f5e001d,0x4f5f001d,0x4f60001d,
	0x4f61001d,0x4f62001d,0x4f63001d,0x4f64001d,0x4f65001d,0x4f66001d,0x4f67001d,0x4f68001d,
	0x4f69001d,0x4f6a001d,0x4f6b001d,0x4f6c001d,0x4f6d001d,0x4f6e001d,0x4f6f001d,0x4f70001d,
	0x4f71001d,0x4f72001d,0x4f73001d,0x4f74001d,0x4f75001d,0x4f76001d,0x4f77001d,0x4f78001d,
	0x4f79001d,0x4f7a001d,0x4f7b001d,0x4f7c001d,0x4f7d001d,0x4f7e001d,0x4f7f001d,0x4f80001d,
	0x4f81001d,0x4f82001d,0x4f83001d,0x4f84001d,0x4f85001d,0x4f86001d,0x4f87001d,0x4f88001d,
	0x4f89001d,0x4f8a001d,0x4f8b001d,0x4f8c001d,0x4f8d001d,0x4f8e001d,0x4f8f001d,0x4f90001d,
	0x4f91001d,0x4f92001d,0x4f93001d,0x4f94001d,0x4f95001d,0x4f96001d,0x4f97001d,0x4f98001d,
	0x4f99001d,0x4f9a001d,0x4f9b001d,0x4f9c001d,0x4f9d001d,0x4f9e001d,0x4f9f001d,0x4fa0001d,
	0x4fa1001d,0x4fa2001d,0x4fa3001d,0x4fa4001d,0x4fa5001d,0x4fa6001d,0x4fa7001d,0x4fa8001d,
	0x4fa9001d,0x4faa001d,0x4fab001d,0x4fac001d,0x4fad001d,0x4fae001d,0x4faf001d,0x4fb0001d,
	0x4fb1001d,0x4fb2001d,0x4fb3001d,0x4fb4001d,0x4fb5001d,0x4fb6001d,0x4fb7001d,0x4fb8001d,
	0x4fb9001d,0x4fba001d,0x4fbb001d,0x4fbc001d,0x4fbd001d,0x4fbe001d,0x4fbf001d,0x4fc0001d,
	0x4fc1001d,0x4fc2001d,0x4fc3001d,0x4fc4001d,0x4fc5001d,0x4fc6001d,0x4fc7001d,0x4fc8001d,
	0x4fc9001d,0x4fca001d,0x4fcb001d,0x4fcc001d,0x4fcd001d,0x4fce001d,0x4fcf001d,0x4fd0001d,
	0x4fd1001d,0x4fd2001d,0x4fd3001d,0x4fd4001d,0x4fd5001d,0x4fd6001d,0x4fd7001d,0x4fd8001d,
	0x4fd9001d,0x4fda001d,0x4fdb001d,0x4fdc001d,0x4fdd001d,0x4fde001d,0x4fdf001d,0x4fe0001d,
	0x4fe1001d,0x4fe2001d,0x4fe3001d,0x4fe4001d,0x4fe5001d,0x4fe6001d,0x4fe7001d,0x4fe8001d,
	0x4fe9001d,0x4fea001d,0x4feb001d,0x4fec001d,0x4fed001d,0x4fee001d,0x4fef001d,0x4ff0001d,
	0x4ff1001d,0x4ff2001d,0x4ff3001d,0x4ff4001d,0x4ff5001d,0x4ff6001d,0x4ff7001d,0x4ff8001d,
	0x4ff9001d,0x4ffa001d,0x4ffb001d,0x4ffc001d,0x4ffd001d,0x4ffe001d,0x4fff001d,0x5000001d,
	0x5001001d,0x5002001d,0x5003001d,0x5004001d,0x5005001d,0x5006001d,0x5007001d,0x5008001d,
	0x5009001d,0x500a001d,0x500b001d,0x500c001d,0x500d001d,0x500e001d,0x500f001d,0x5010001d,
	0x5011001d,0x5012001d,0x5013001d,0x5014001d,0x5015001d,0x5016001d,0x5017001d,0x5018001d,
	0x5019001d,0x501a001d,0x501b001d,0x501c001d,0x501d001d,0x501e001d,0x501f001d,0x5020001d,
	0x5021001d,0x5022001d,0x5023001d,0x5024001d,0x5025001d,0x5026001d,0x5027001d,0x5028001d,
	0x5029001d,0x502a001d,0x502b001d,0x502c001d,0x502d001d,0x502e001d,0x502f001d,0x5030001d,
	0x5031001d,0x5032001d,0x5033001d,0x5034001d,0x5035001d,0x5036001d,0x5037001d,0x5038001d,
	0x5039001d,0x503a001d,0x503b001d,0x503c001d,0x503d001d,0x503e001d,0x503f001d,0x5040001d,
	0x5041001d,0x5042001d,0x5043001d,0x5044001d,0x5045001d,0x5046001d,0x5047001d,0x5048001d,
	0x5049001d,0x504a001d,0x504b001d,0x504c001d,0x504d001d,0x504e001d,0x504f001d,0x5050001d,
	0x5051001d,0x5052001d,0x5053001d,0x5054001d,0x5055001d,0x5056001d,0x5057001d,0x5058001d,
	0x5059001d,0x505a001d,0x505b001d,0x505c001d,0x505d001d,0x505e001d,0x505f001d,0x5060001d,
	0x5061001d,0x5062001d,0x5063001d,0x5064001d,0x5065001d,0x5066001d,0x5067001d,0x5068001d,
	0x5069001d,0x506a001d,0x506b001d,0x506c001d,0x506d001d,0x506e001d,0x506f001d,0x5070001d,
	0x5071001d,0x5072001d,0x5073001d,0x5074001d,0x5075001d,0x5076001d,0x5077001d,0x5078001d,
	0x5079001d,0x507a001d,0x507b001d,0x507c001d,0x507d001d,0x507e001d,0x507f001d,0x5080001d,
	0x5081001d,0x5082001d,0x5083001d,0x5084001d,0x5085001d,0x5086001d,0x5087001d,0x5088001d,
	0x5089001d,0x508a001d,0x508b001d,0x508c001d,0x508d001d,0x508e001d,0x508f001d,0x5090001d,
	0x5091001d,0x5092001d,0x5093001d,0x5094001d,0x5095001d,0x5096001d,0x5097001d,0x5098001d,
	0x5099001d,0x509a001d,0x509b001d,0x509c001d,0x509d001d,0x509e001d,0x509f001d,0x50a0001d,
	0x50a1001d,0x50a2001d,0x50a3001d,0x50a4001d,0x50a5001d,0x50a6001d,0x50a7001d,0x50a8001d,
	0x50a9001d,0x50aa001d,0x50ab001d,0x50ac001d,0x50ad001d,0x50ae001d,0x50af001d,0x50b0001d,
	0x50b1001d,0x50b2001d,0x50b3001d,0x50b4001d,0x50b5001d,0x50b6001d,0x50b7001d,0x50b8001d,
	0x50b9001d,0x50ba001d,0x50bb001d,0x50bc001d,0x50bd001d,0x50be001d,0x50bf001d,0x50c0001d,
	0x50c1001d,0x50c2001d,0x50c3001d,0x50c4001d,0x50c5001d,0x50c6001d,0x50c7001d,0x50c8001d,
	0x50c9001d,0x50ca001d,0x50cb001d,0x50cc001d,0x50cd001d,0x50ce001d,0x50cf001d,0x50d0001d,
	0x50d1001d,0x50d2001d,0x50d3001d,0x50d4001d,0x50d5001d,0x50d6001d,0x50d7001d,0x50d8001d,
	0x50d9001d,0x50da001d,0x50db001d,0x50dc001d,0x50dd001d,0x50de001d,0x50df001d,0x50e0001d,
	0x50e1001d,0x50e2001d,0x50e3001d,0x50e4001d,0x50e5001d,0x50e6001d,0x50e7001d,0x50e8001d,
	0x50e9001d,0x50ea001d,0x50eb001d,0x50ec001d,0x50ed001d,0x50ee001d,0x50ef001d,0x50f0001d,
	0x50f1001d,0x50f2001d,0x50f3001d,0x50f4001d,0x50f5001d,0x50f6001d,0x50f7001d,0x50f8001d,
	0x50f9001d,0x50fa001d,0x50fb001d,0x50fc001d,0x50fd001d,0x50fe001d,0x50ff001d,0x5100001d,
	0x5101001d,0x5102001d,0x5103001d,0x5104001d,0x5105001d,0x5106001d,0x5107001d,0x5108001d,
	0x5109001d,0x510a001d,0x510b001d,0x510c001d,0x510d001d,0x510e001d,0x510f001d,0x5110001d,
	0x5111001d,0x5112001d,0x5113001d,0x5114001d,0x5115001d,0x5116001d,0x5117001d,0x5118001d,
	0x5119001d,0x511a001d,0x511b001d,0x511c001d,0x511d001d,0x511e001d,0x511f001d,0x5120001d,
	0x5121001d,0x5122001d,0x5123001d,0x5124001d,0x5125001d,0x5126001d,0x5127001d,0x5128001d,
	0x5129001d,0x512a001d,0x512b001d,0x512c001d,0x512d001d,0x512e001d,0x512f001d,0x5130001d,
	0x5131001d,0x5132001d,0x5133001d,0x5134001d,0x5135001d,0x5136001d,0x5137001d,0x5138001d,
	0x5139001d,0x513a001d,0x513b001d,0x513c001d,0x513d001d,0x513e001d,0x513f001d,0x5140001d,
	0x5141001d,0x5142001d,0x5143001d,0x5144001d,0x5145001d,0x5146001d,0x5147001d,0x5148001d,
	0x5149001d,0x514a001d,0x514b001d,0x514c001d,0x514d001d,0x514e001d,0x514f001d,0x5150001d,
	0x5151001d,0x5152001d,0x5153001d,0x5154001d,0x5155001d,0x5156001d,0x5157001d,0x5158001d,
	0x5159001d,0x515a001d,0x515b001d,0x515c001d,0x515d001d,0x515e001d,0x515f001d,0x5160001d,
	0x5161001d,0x5162001d,0x5163001d,0x5164001d,0x5165001d,0x5166001d,0x5167001d,0x5168001d,
	0x5169001d,0x516a001d,0x516b001d,0x516c001d,0x516d001d,0x516e001d,0x516f001d,0x5170001d,
	0x5171001d,0x5172001d,0x5173001d,0x5174001d,0x5175001d,0x5176001d,0x5177001d,0x5178001d,
	0x5179001d,0x517a001d,0x517b001d,0x517c001d,0x517d001d,0x517e001d,0x517f001d,0x5180001d,
	0x5181001d,0x5182001d,0x5183001d,0x5184001d,0x5185001d,0x5186001d,0x5187001d,0x5188001d,
	0x5189001d,0x518a001d,0x518b001d,0x518c001d,0x518d001d,0x518e001d,0x518f001d,0x5190001d,
	0x5191001d,0x5192001d,0x5193001d,0x5194001d,0x5195001d,0x5196001d,0x5197001d,0x5198001d,
	0x5199001d,0x519a001d,0x519b001d,0x519c001d,0x519d001d,0x519e001d,0x519f001d,0x51a0001d,
	0x51a1001d,0x51a2001d,0x51a3001d,0x51a4001d,0x51a5001d,0x51a6001d,0x51a7001d,0x51a8001d,
	0x51a9001d,0x51aa001d,0x51ab001d,0x51ac001d,0x51ad001d,0x51ae001d,0x51af001d,0x51b0001d,
	0x51b1001d,0x51b2001d,0x51b3001d,0x51b4001d,0x51b5001d,0x51b6001d,0x51b7001d,0x51b8001d,
	0x51b9001d,0x51ba001d,0x51bb001d,0x51bc001d,0x51bd001d,0x51be001d,0x51bf001d,0x51c0001d,
	0x51c1001d,0x51c2001d,0x51c3001d,0x51c4001d,0x51c5001d,0x51c6001d,0x51c7001d,0x51c8001d,
	0x51c9001d,0x51ca001d,0x51cb001d,0x51cc001d,0x51cd001d,0x51ce001d,0x51cf001d,0x51d0001d,
	0x51d1001d,0x51d2001d,0x51d3001d,0x51d4001d,0x51d5001d,0x51d6001d,0x51d7001d,0x51d8001d,
	0x51d9001d,0x51da001d,0x51db001d,0x51dc001d,0x51dd001d,0x51de001d,0x51df001d,0x51e0001d,
	0x51e1001d,0x51e2001d,0x51e3001d,0x51e4001d,0x51e5001d,0x51e6001d,0x51e7001d,0x51e8001d,
	0x51e9001d,0x51ea001d,0x51eb001d,0x51ec001d,0x51ed001d,0x51ee001d,0x51ef001d,0x51f0001d,
	0x51f1001d,0x51f2001d,0x51f3001d,0x51f4001d,0x51f5001d,0x51f6001d,0x51f7001d,0x51f8001d,
	0x51f9001d,0x51fa001d,0x51fb001d,0x51fc001d,0x51fd001d,0x51fe001d,0x51ff001d,0x5200001d,
	0x5201001d,0x5202001d,0x5203001d,0x5204001d,0x5205001d,0x5206001d,0x5207001d,0x5208001d,
	0x5209001d,0x520a001d,0x520b001d,0x520c001d,0x520d001d,0x520e001d,0x520f001d,0x5210001d,
	0x5211001d,0x5212001d,0x5213001d,0x5214001d,0x5215001d,0x5216001d,0x5217001d,0x5218001d,
	0x5219001d,0x521a001d,0x521b001d,0x521c001d,0x521d001d,0x521e001d,0x521f001d,0x5220001d,
	0x5221001d,0x5222001d,0x5223001d,0x5224001d,0x5225001d,0x5226001d,0x5227001d,0x5228001d,
	0x5229001d,0x522a001d,0x522b001d,0x522c001d,0x522d001d,0x522e001d,0x522f001d,0x5230001d,
	0x5231001d,0x5232001d,0x5233001d,0x5234001d,0x5235001d,0x5236001d,0x5237001d,0x5238001d,
	0x5239001d,0x523a001d,0x523b001d,0x523c001d,0x523d001d,0x523e001d,0x523f001d,0x5240001d,
	0x5241001d,0x5242001d,0x5243001d,0x5244001d,0x5245001d,0x5246001d,0x5247001d,0x5248001d,
	0x5249001d,0x524a001d,0x524b001d,0x524c001d,0x524d001d,0x524e001d,0x524f001d,0x5250001d,
	0x5251001d,0x5252001d,0x5253001d,0x5254001d,0x5255001d,0x5256001d,0x5257001d,0x5258001d,
	0x5259001d,0x525a001d,0x525b001d,0x525c001d,0x525d001d,0x525e001d,0x525f001d,0x5260001d,
	0x5261001d,0x5262001d,0x5263001d,0x5264001d,0x5265001d,0x5266001d,0x5267001d,0x5268001d,
	0x5269001d,0x526a001d,0x526b001d,0x526c001d,0x526d001d,0x526e001d,0x526f001d,0x5270001d,
	0x5271001d,0x5272001d,0x5273001d,0x5274001d,0x5275001d,0x5276001d,0x5277001d,0x5278001d,
	0x5279001d,0x527a001d,0x527b001d,0x527c001d,0x527d001d,0x527e001d,0x527f001d,0x5280001d,
	0x5281001d,0x5282001d,0x5283001d,0x5284001d,0x5285001d,0x5286001d,0x5287001d,0x5288001d,
	0x5289001d,0x528a001d,0x528b001d,0x528c001d,0x528d001d,0x528e001d,0x528f001d,0x5290001d,
	0x5291001d,0x5292001d,0x5293001d,0x5294001d,0x5295001d,0x5296001d,0x5297001d,0x5298001d,
	0x5299001d,0x529a001d,0x529b001d,0x529c001d,0x529d001d,0x529e001d,0x529f001d,0x52a0001d,
	0x52a1001d,0x52a2001d,0x52a3001d,0x52a4001d,0x52a5001d,0x52a6001d,0x52a7001d,0x52a8001d,
	0x52a9001d,0x52aa001d,0x52ab001d,0x52ac001d,0x52ad001d,0x52ae001d,0x52af001d,0x52b0001d,
	0x52b1001d,0x52b2001d,0x52b3001d,0x52b4001d,0x52b5001d,0x52b6001d,0x52b7001d,0x52b8001d,
	0x52b9001d,0x52ba001d,0x52bb001d,0x52bc001d,0x52bd001d,0x52be001d,0x52bf001d,0x52c0001d,
	0x52c1001d,0x52c2001d,0x52c3001d,0x52c4001d,0x52c5001d,0x52c6001d,0x52c7001d,0x52c8001d,
	0x52c9001d,0x52ca001d,0x52cb001d,0x52cc001d,0x52cd001d,0x52ce001d,0x52cf001d,0x52d0001d,
	0x52d1001d,0x52d2001d,0x52d3001d,0x52d4001d,0x52d5001d,0x52d6001d,0x52d7001d,0x52d8001d,
	0x52d9001d,0x52da001d,0x52db001d,0x52dc001d,0x52dd001d,0x52de001d,0x52df001d,0x52e0001d,
	0x52e1001d,0x52e2001d,0x52e3001d,0x52e4001d,0x52e5001d,0x52e6001d,0x52e7001d,0x52e8001d,
	0x52e9001d,0x52ea001d,0x52eb001d,0x52ec001d,0x52ed001d,0x52ee001d,0x52ef001d,0x52f0001d,
	0x52f1001d,0x52f2001d,0x52f3001d,0x52f4001d,0x52f5001d,0x52f6001d,0x52f7001d,0x52f8001d,
	0x52f9001d,0x52fa001d,0x52fb001d,0x52fc001d,0x52fd001d,0x52fe001d,0x52ff001d,0x5300001d,
	0x5301001d,0x5302001d,0x5303001d,0x5304001d,0x5305001d,0x5306001d,0x5307001d,0x5308001d,
	0x5309001d,0x530a001d,0x530b001d,0x530c001d,0x530d001d,0x530e001d,0x530f001d,0x5310001d,
	0x5311001d,0x5312001d,0x5313001d,0x5314001d,0x5315001d,0x5316001d,0x5317001d,0x5318001d,
	0x5319001d,0x531a001d,0x531b001d,0x531c001d,0x531d001d,0x531e001d,0x531f001d,0x5320001d,
	0x5321001d,0x5322001d,0x5323001d,0x5324001d,0x5325001d,0x5326001d,0x5327001d,0x5328001d,
	0x5329001d,0x532a001d,0x532b001d,0x532c001d,0x532d001d,0x532e001d,0x532f001d,0x5330001d,
	0x5331001d,0x5332001d,0x5333001d,0x5334001d,0x5335001d,0x5336001d,0x5337001d,0x5338001d,
	0x5339001d,0x533a001d,0x533b001d,0x533c001d,0x533d001d,0x533e001d,0x533f001d,0x5340001d,
	0x5341001d,0x5342001d,0x5343001d,0x5344001d,0x5345001d,0x5346001d,0x5347001d,0x5348001d,
	0x5349001d,0x534a001d,0x534b001d,0x534c001d,0x534d001d,0x534e001d,0x534f001d,0x5350001d,
	0x5351001d,0x5352001d,0x5353001d,0x5354001d,0x5355001d,0x5356001d,0x5357001d,0x5358001d,
	0x5359001d,0x535a001d,0x535b001d,0x535c001d,0x535d001d,0x535e001d,0x535f001d,0x5360001d,
	0x5361001d,0x5362001d,0x5363001d,0x5364001d,0x5365001d,0x5366001d,0x5367001d,0x5368001d,
	0x5369001d,0x536a001d,0x536b001d,0x536c001d,0x536d001d,0x536e001d,0x536f001d,0x5370001d,
	0x5371001d,0x5372001d,0x5373001d,0x5374001d,0x5375001d,0x5376001d,0x5377001d,0x5378001d,
	0x5379001d,0x537a001d,0x537b001d,0x537c001d,0x537d001d,0x537e001d,0x537f001d,0x5380001d,
	0x5381001d,0x5382001d,0x5383001d,0x5384001d,0x5385001d,0x5386001d,0x5387001d,0x5388001d,
	0x5389001d,0x538a001d,0x538b001d,0x538c001d,0x538d001d,0x538e001d,0x538f001d,0x5390001d,
	0x5391001d,0x5392001d,0x5393001d,0x5394001d,0x5395001d,0x5396001d,0x5397001d,0x5398001d,
	0x5399001d,0x539a001d,0x539b001d,0x539c001d,0x539d001d,0x539e001d,0x539f001d,0x53a0001d,
	0x53a1001d,0x53a2001d,0x53a3001d,0x53a4001d,0x53a5001d,0x53a6001d,0x53a7001d,0x53a8001d,
	0x53a9001d,0x53aa001d,0x53ab001d,0x53ac001d,0x53ad001d,0x53ae001d,0x53af001d,0x53b0001d,
	0x53b1001d,0x53b2001d,0x53b3001d,0x53b4001d,0x53b5001d,0x53b6001d,0x53b7001d,0x53b8001d,
	0x53b9001d,0x53ba001d,0x53bb001d,0x53bc001d,0x53bd001d,0x53be001d,0x53bf001d,0x53c0001d,
	0x53c1001d,0x53c2001d,0x53c3001d,0x53c4001d,0x53c5001d,0x53c6001d,0x53c7001d,0x53c8001d,
	0x53c9001d,0x53ca001d,0x53cb001d,0x53cc001d,0x53cd001d,0x53ce001d,0x53cf001d,0x53d0001d,
	0x53d1001d,0x53d2001d,0x53d3001d,0x53d4001d,0x53d5001d,0x53d6001d,0x53d7001d,0x53d8001d,
	0x53d9001d,0x53da001d,0x53db001d,0x53dc001d,0x53dd001d,0x53de001d,0x53df001d,0x53e0001d,
	0x53e1001d,0x53e2001d,0x53e3001d,0x53e4001d,0x53e5001d,0x53e6001d,0x53e7001d,0x53e8001d,
	0x53e9001d,0x53ea001d,0x53eb001d,0x53ec001d,0x53ed001d,0x53ee001d,0x53ef001d,0x53f0001d,
	0x53f1001d,0x53f2001d,0x53f3001d,0x53f4001d,0x53f5001d,0x53f6001d,0x53f7001d,0x53f8001d,
	0x53f9001d,0x53fa001d,0x53fb001d,0x53fc001d,0x53fd001d,0x53fe001d,0x53ff001d,0x5400001d,
	0x5401001d,0x5402001d,0x5403001d,0x5404001d,0x5405001d,0x5406001d,0x5407001d,0x5408001d,
	0x5409001d,0x540a001d,0x540b001d,0x540c001d,0x540d001d,0x540e001d,0x540f001d,0x5410001d,
	0x5411001d,0x5412001d,0x5413001d,0x5414001d,0x5415001d,0x5416001d,0x5417001d,0x5418001d,
	0x5419001d,0x541a001d,0x541b001d,0x541c001d,0x541d001d,0x541e001d,0x541f001d,0x5420001d,
	0x5421001d,0x5422001d,0x5423001d,0x5424001d,0x5425001d,0x5426001d,0x5427001d,0x5428001d,
	0x5429001d,0x542a001d,0x542b001d,0x542c001d,0x542d001d,0x542e001d,0x542f001d,0x5430001d,
	0x5431001d,0x5432001d,0x5433001d,0x5434001d,0x5435001d,0x5436001d,0x5437001d,0x5438001d,
	0x5439001d,0x543a001d,0x543b001d,0x543c001d,0x543d001d,0x543e001d,0x543f001d,0x5440001d,
	0x5441001d,0x5442001d,0x5443001d,0x5444001d,0x5445001d,0x5446001d,0x5447001d,0x5448001d,
	0x5449001d,0x544a001d,0x544b001d,0x544c001d,0x544d001d,0x544e001d,0x544f001d,0x5450001d,
	0x5451001d,0x5452001d,0x5453001d,0x5454001d,0x5455001d,0x5456001d,0x5457001d,0x5458001d,
	0x5459001d,0x545a001d,0x545b001d,0x545c001d,0x545d001d,0x545e001d,0x545f001d,0x5460001d,
	0x5461001d,0x5462001d,0x5463001d,0x5464001d,0x5465001d,0x5466001d,0x5467001d,0x5468001d,
	0x5469001d,0x546a001d,0x546b001d,0x546c001d,0x546d001d,0x546e001d,0x546f001d,0x5470001d,
	0x5471001d,0x5472001d,0x5473001d,0x5474001d,0x5475001d,0x5476001d,0x5477001d,0x5478001d,
	0x5479001d,0x547a001d,0x547b001d,0x547c001d,0x547d001d,0x547e001d,0x547f001d,0x5480001d,
	0x5481001d,0x5482001d,0x5483001d,0x5484001d,0x5485001d,0x5486001d,0x5487001d,0x5488001d,
	0x5489001d,0x548a001d,0x548b001d,0x548c001d,0x548d001d,0x548e001d,0x548f001d,0x5490001d,
	0x5491001d,0x5492001d,0x5493001d,0x5494001d,0x5495001d,0x5496001d,0x5497001d,0x5498001d,
	0x5499001d,0x549a001d,0x549b001d,0x549c001d,0x549d001d,0x549e001d,0x549f001d,0x54a0001d,
	0x54a1001d,0x54a2001d,0x54a3001d,0x54a4001d,0x54a5001d,0x54a6001d,0x54a7001d,0x54a8001d,
	0x54a9001d,0x54aa001d,0x54ab001d,0x54ac001d,0x54ad001d,0x54ae001d,0x54af001d,0x54b0001d,
	0x54b1001d,0x54b2001d,0x54b3001d,0x54b4001d,0x54b5001d,0x54b6001d,0x54b7001d,0x54b8001d,
	0x54b9001d,0x54ba001d,0x54bb001d,0x54bc001d,0x54bd001d,0x54be001d,0x54bf001d,0x54c0001d,
	0x54c1001d,0x54c2001d,0x54c3001d,0x54c4001d,0x54c5001d,0x54c6001d,0x54c7001d,0x54c8001d,
	0x54c9001d,0x54ca001d,0x54cb001d,0x54cc001d,0x54cd001d,0x54ce001d,0x54cf001d,0x54d0001d,
	0x54d1001d,0x54d2001d,0x54d3001d,0x54d4001d,0x54d5001d,0x54d6001d,0x54d7001d,0x54d8001d,
	0x54d9001d,0x54da001d,0x54db001d,0x54dc001d,0x54dd001d,0x54de001d,0x54df001d,0x54e0001d,
	0x54e1001d,0x54e2001d,0x54e3001d,0x54e4001d,0x54e5001d,0x54e6001d,0x54e7001d,0x54e8001d,
	0x54e9001d,0x54ea001d,0x54eb001d,0x54ec001d,0x54ed001d,0x54ee001d,0x54ef001d,0x54f0001d,
	0x54f1001d,0x54f2001d,0x54f3001d,0x54f4001d,0x54f5001d,0x54f6001d,0x54f7001d,0x54f8001d,
	0x54f9001d,0x54fa001d,0x54fb001d,0x54fc001d,0x54fd001d,0x54fe001d,0x54ff001d,0x5500001d,
	0x5501001d,0x5502001d,0x5503001d,0x5504001d,0x5505001d,0x5506001d,0x5507001d,0x5508001d,
	0x5509001d,0x550a001d,0x550b001d,0x550c001d,0x550d001d,0x550e001d,0x550f001d,0x5510001d,
	0x5511001d,0x5512001d,0x5513001d,0x5514001d,0x5515001d,0x5516001d,0x5517001d,0x5518001d,
	0x5519001d,0x551a001d,0x551b001d,0x551c001d,0x551d001d,0x551e001d,0x551f001d,0x5520001d,
	0x5521001d,0x5522001d,0x5523001d,0x5524001d,0x5525001d,0x5526001d,0x5527001d,0x5528001d,
	0x5529001d,0x552a001d,0x552b001d,0x552c001d,0x552d001d,0x552e001d,0x552f001d,0x5530001d,
	0x5531001d,0x5532001d,0x5533001d,0x5534001d,0x5535001d,0x5536001d,0x5537001d,0x5538001d,
	0x5539001d,0x553a001d,0x553b001d,0x553c001d,0x553d001d,0x553e001d,0x553f001d,0x5540001d,
	0x5541001d,0x5542001d,0x5543001d,0x5544001d,0x5545001d,0x5546001d,0x5547001d,0x5548001d,
	0x5549001d,0x554a001d,0x554b001d,0x554c001d,0x554d001d,0x554e001d,0x554f001d,0x5550001d,
	0x5551001d,0x5552001d,0x5553001d,0x5554001d,0x5555001d,0x5556001d,0x5557001d,0x5558001d,
	0x5559001d,0x555a001d,0x555b001d,0x555c001d,0x555d001d,0x555e001d,0x555f001d,0x5560001d,
	0x5561001d,0x5562001d,0x5563001d,0x5564001d,0x5565001d,0x5566001d,0x5567001d,0x5568001d,
	0x5569001d,0x556a001d,0x556b001d,0x556c001d,0x556d001d,0x556e001d,0x556f001d,0x5570001d,
	0x5571001d,0x5572001d,0x5573001d,0x5574001d,0x5575001d,0x5576001d,0x5577001d,0x5578001d,
	0x5579001d,0x557a001d,0x557b001d,0x557c001d,0x557d001d,0x557e001d,0x557f001d,0x5580001d,
	0x5581001d,0x5582001d,0x5583001d,0x5584001d,0x5585001d,0x5586001d,0x5587001d,0x5588001d,
	0x5589001d,0x558a001d,0x558b001d,0x558c001d,0x558d001d,0x558e001d,0x558f001d,0x5590001d,
	0x5591001d,0x5592001d,0x5593001d,0x5594001d,0x5595001d,0x5596001d,0x5597001d,0x5598001d,
	0x5599001d,0x559a001d,0x559b001d,0x559c001d,0x559d001d,0x559e001d,0x559f001d,0x55a0001d,
	0x55a1001d,0x55a2001d,0x55a3001d,0x55a4001d,0x55a5001d,0x55a6001d,0x55a7001d,0x55a8001d,
	0x55a9001d,0x55aa001d,0x55ab001d,0x55ac001d,0x55ad001d,0x55ae001d,0x55af001d,0x55b0001d,
	0x55b1001d,0x55b2001d,0x55b3001d,0x55b4001d,0x55b5001d,0x55b6001d,0x55b7001d,0x55b8001d,
	0x55b9001d,0x55ba001d,0x55bb001d,0x55bc001d,0x55bd001d,0x55be001d,0x55bf001d,0x55c0001d,
	0x55c1001d,0x55c2001d,0x55c3001d,0x55c4001d,0x55c5001d,0x55c6001d,0x55c7001d,0x55c8001d,
	0x55c9001d,0x55ca001d,0x55cb001d,0x55cc001d,0x55cd001d,0x55ce001d,0x55cf001d,0x55d0001d,
	0x55d1001d,0x55d2001d,0x55d3001d,0x55d4001d,0x55d5001d,0x55d6001d,0x55d7001d,0x55d8001d,
	0x55d9001d,0x55da001d,0x55db001d,0x55dc001d,0x55dd001d,0x55de001d,0x55df001d,0x55e0001d,
	0x55e1001d,0x55e2001d,0x55e3001d,0x55e4001d,0x55e5001d,0x55e6001d,0x55e7001d,0x55e8001d,
	0x55e9001d,0x55ea001d,0x55eb001d,0x55ec001d,0x55ed001d,0x55ee001d,0x55ef001d,0x55f0001d,
	0x55f1001d,0x55f2001d,0x55f3001d,0x55f4001d,0x55f5001d,0x55f6001d,0x55f7001d,0x55f8001d,
	0x55f9001d,0x55fa001d,0x55fb001d,0x55fc001d,0x55fd001d,0x55fe001d,0x55ff001d,0x5600001d,
	0x5601001d,0x5602001d,0x5603001d,0x5604001d,0x5605001d,0x5606001d,0x5607001d,0x5608001d,
	0x5609001d,0x560a001d,0x560b001d,0x560c001d,0x560d001d,0x560e001d,0x560f001d,0x5610001d,
	0x5611001d,0x5612001d,0x5613001d,0x5614001d,0x5615001d,0x5616001d,0x5617001d,0x5618001d,
	0x5619001d,0x561a001d,0x561b001d,0x561c001d,0x561d001d,0x561e001d,0x561f001d,0x5620001d,
	0x5621001d,0x5622001d,0x5623001d,0x5624001d,0x5625001d,0x5626001d,0x5627001d,0x5628001d,
	0x5629001d,0x562a001d,0x562b001d,0x562c001d,0x562d001d,0x562e001d,0x562f001d,0x5630001d,
	0x5631001d,0x5632001d,0x5633001d,0x5634001d,0x5635001d,0x5636001d,0x5637001d,0x5638001d,
	0x5639001d,0x563a001d,0x563b001d,0x563c001d,0x563d001d,0x563e001d,0x563f001d,0x5640001d,
	0x5641001d,0x5642001d,0x5643001d,0x5644001d,0x5645001d,0x5646001d,0x5647001d,0x5648001d,
	0x5649001d,0x564a001d,0x564b001d,0x564c001d,0x564d001d,0x564e001d,0x564f001d,0x5650001d,
	0x5651001d,0x5652001d,0x5653001d,0x5654001d,0x5655001d,0x5656001d,0x5657001d,0x5658001d,
	0x5659001d,0x565a001d,0x565b001d,0x565c001d,0x565d001d,0x565e001d,0x565f001d,0x5660001d,
	0x5661001d,0x5662001d,0x5663001d,0x5664001d,0x5665001d,0x5666001d,0x5667001d,0x5668001d,
	0x5669001d,0x566a001d,0x566b001d,0x566c001d,0x566d001d,0x566e001d,0x566f001d,0x5670001d,
	0x5671001d,0x5672001d,0x5673001d,0x5674001d,0x5675001d,0x5676001d,0x5677001d,0x5678001d,
	0x5679001d,0x567a001d,0x567b001d,0x567c001d,0x567d001d,0x567e001d,0x567f001d,0x5680001d,
	0x5681001d,0x5682001d,0x5683001d,0x5684001d,0x5685001d,0x5686001d,0x5687001d,0x5688001d,
	0x5689001d,0x568a001d,0x568b001d,0x568c001d,0x568d001d,0x568e001d,0x568f001d,0x5690001d,
	0x5691001d,0x5692001d,0x5693001d,0x5694001d,0x5695001d,0x5696001d,0x5697001d,0x5698001d,
	0x5699001d,0x569a001d,0x569b001d,0x569c001d,0x569d001d,0x569e001d,0x569f001d,0x56a0001d,
	0x56a1001d,0x56a2001d,0x56a3001d,0x56a4001d,0x56a5001d,0x56a6001d,0x56a7001d,0x56a8001d,
	0x56a9001d,0x56aa001d,0x56ab001d,0x56ac001d,0x56ad001d,0x56ae001d,0x56af001d,0x56b0001d,
	0x56b1001d,0x56b2001d,0x56b3001d,0x56b4001d,0x56b5001d,0x56b6001d,0x56b7001d,0x56b8001d,
	0x56b9001d,0x56ba001d,0x56bb001d,0x56bc001d,0x56bd001d,0x56be001d,0x56bf001d,0x56c0001d,
	0x56c1001d,0x56c2001d,0x56c3001d,0x56c4001d,0x56c5001d,0x56c6001d,0x56c7001d,0x56c8001d,
	0x56c9001d,0x56ca001d,0x56cb001d,0x56cc001d,0x56cd001d,0x56ce001d,0x56cf001d,0x56d0001d,
	0x56d1001d,0x56d2001d,0x56d3001d,0x56d4001d,0x56d5001d,0x56d6001d,0x56d7001d,0x56d8001d,
	0x56d9001d,0x56da001d,0x56db001d,0x56dc001d,0x56dd001d,0x56de001d,0x56df001d,0x56e0001d,
	0x56e1001d,0x56e2001d,0x56e3001d,0x56e4001d,0x56e5001d,0x56e6001d,0x56e7001d,0x56e8001d,
	0x56e9001d,0x56ea001d,0x56eb001d,0x56ec001d,0x56ed001d,0x56ee001d,0x56ef001d,0x56f0001d,
	0x56f1001d,0x56f2001d,0x56f3001d,0x56f4001d,0x56f5001d,0x56f6001d,0x56f7001d,0x56f8001d,
	0x56f9001d,0x56fa001d,0x56fb001d,0x56fc001d,0x56fd001d,0x56fe001d,0x56ff001d,0x5700001d,
	0x5701001d,0x5702001d,0x5703001d,0x5704001d,0x5705001d,0x5706001d,0x5707001d,0x5708001d,
	0x5709001d,0x570a001d,0x570b001d,0x570c001d,0x570d001d,0x570e001d,0x570f001d,0x5710001d,
	0x5711001d,0x5712001d,0x5713001d,0x5714001d,0x5715001d,0x5716001d,0x5717001d,0x5718001d,
	0x5719001d,0x571a001d,0x571b001d,0x571c001d,0x571d001d,0x571e001d,0x571f001d,0x5720001d,
	0x5721001d,0x5722001d,0x5723001d,0x5724001d,0x5725001d,0x5726001d,0x5727001d,0x5728001d,
	0x5729001d,0x572a001d,0x572b001d,0x572c001d,0x572d001d,0x572e001d,0x572f001d,0x5730001d,
	0x5731001d,0x5732001d,0x5733001d,0x5734001d,0x5735001d,0x5736001d,0x5737001d,0x5738001d,
	0x5739001d,0x573a001d,0x573b001d,0x573c001d,0x573d001d,0x573e001d,0x573f001d,0x5740001d,
	0x5741001d,0x5742001d,0x5743001d,0x5744001d,0x5745001d,0x5746001d,0x5747001d,0x5748001d,
	0x5749001d,0x574a001d,0x574b001d,0x574c001d,0x574d001d,0x574e001d,0x574f001d,0x5750001d,
	0x5751001d,0x5752001d,0x5753001d,0x5754001d,0x5755001d,0x5756001d,0x5757001d,0x5758001d,
	0x5759001d,0x575a001d,0x575b001d,0x575c001d,0x575d001d,0x575e001d,0x575f001d,0x5760001d,
	0x5761001d,0x5762001d,0x5763001d,0x5764001d,0x5765001d,0x5766001d,0x5767001d,0x5768001d,
	0x5769001d,0x576a001d,0x576b001d,0x576c001d,0x576d001d,0x576e001d,0x576f001d,0x5770001d,
	0x5771001d,0x5772001d,0x5773001d,0x5774001d,0x5775001d,0x5776001d,0x5777001d,0x5778001d,
	0x5779001d,0x577a001d,0x577b001d,0x577c001d,0x577d001d,0x577e001d,0x577f001d,0x5780001d,
	0x5781001d,0x5782001d,0x5783001d,0x5784001d,0x5785001d,0x5786001d,0x5787001d,0x5788001d,
	0x5789001d,0x578a001d,0x578b001d,0x578c001d,0x578d001d,0x578e001d,0x578f001d,0x5790001d,
	0x5791001d,0x5792001d,0x5793001d,0x5794001d,0x5795001d,0x5796001d,0x5797001d,0x5798001d,
	0x5799001d,0x579a001d,0x579b001d,0x579c001d,0x579d001d,0x579e001d,0x579f001d,0x57a0001d,
	0x57a1001d,0x57a2001d,0x57a3001d,0x57a4001d,0x57a5001d,0x57a6001d,0x57a7001d,0x57a8001d,
	0x57a9001d,0x57aa001d,0x57ab001d,0x57ac001d,0x57ad001d,0x57ae001d,0x57af001d,0x57b0001d,
	0x57b1001d,0x57b2001d,0x57b3001d,0x57b4001d,0x57b5001d,0x57b6001d,0x57b7001d,0x57b8001d,
	0x57b9001d,0x57ba001d,0x57bb001d,0x57bc001d,0x57bd001d,0x57be001d,0x57bf001d,0x57c0001d,
	0x57c1001d,0x57c2001d,0x57c3001d,0x57c4001d,0x57c5001d,0x57c6001d,0x57c7001d,0x57c8001d,
	0x57c9001d,0x57ca001d,0x57cb001d,0x57cc001d,0x57cd001d,0x57ce001d,0x57cf001d,0x57d0001d,
	0x57d1001d,0x57d2001d,0x57d3001d,0x57d4001d,0x57d5001d,0x57d6001d,0x57d7001d,0x57d8001d,
	0x57d9001d,0x57da001d,0x57db001d,0x57dc001d,0x57dd001d,0x57de001d,0x57df001d,0x57e0001d,
	0x57e1001d,0x57e2001d,0x57e3001d,0x57e4001d,0x57e5001d,0x57e6001d,0x57e7001d,0x57e8001d,
	0x57e9001d,0x57ea001d,0x57eb001d,0x57ec001d,0x57ed001d,0x57ee001d,0x57ef001d,0x57f0001d,
	0x57f1001d,0x57f2001d,0x57f3001d,0x57f4001d,0x57f5001d,0x57f6001d,0x57f7001d,0x57f8001d,
	0x57f9001d,0x57fa001d,0x57fb001d,0x57fc001d,0x57fd001d,0x57fe001d,0x57ff001d,0x5800001d,
	0x5801001d,0x5802001d,0x5803001d,0x5804001d,0x5805001d,0x5806001d,0x5807001d,0x5808001d,
	0x5809001d,0x580a001d,0x580b001d,0x580c001d,0x580d001d,0x580e001d,0x580f001d,0x5810001d,
	0x5811001d,0x5812001d,0x5813001d,0x5814001d,0x5815001d,0x5816001d,0x5817001d,0x5818001d,
	0x5819001d,0x581a001d,0x581b001d,0x581c001d,0x581d001d,0x581e001d,0x581f001d,0x5820001d,
	0x5821001d,0x5822001d,0x5823001d,0x5824001d,0x5825001d,0x5826001d,0x5827001d,0x5828001d,
	0x5829001d,0x582a001d,0x582b001d,0x582c001d,0x582d001d,0x582e001d,0x582f001d,0x5830001d,
	0x5831001d,0x5832001d,0x5833001d,0x5834001d,0x5835001d,0x5836001d,0x5837001d,0x5838001d,
	0x5839001d,0x583a001d,0x583b001d,0x583c001d,0x583d001d,0x583e001d,0x583f001d,0x5840001d,
	0x5841001d,0x5842001d,0x5843001d,0x5844001d,0x5845001d,0x5846001d,0x5847001d,0x5848001d,
	0x5849001d,0x584a001d,0x584b001d,0x584c001d,0x584d001d,0x584e001d,0x584f001d,0x5850001d,
	0x5851001d,0x5852001d,0x5853001d,0x5854001d,0x5855001d,0x5856001d,0x5857001d,0x5858001d,
	0x5859001d,0x585a001d,0x585b001d,0x585c001d,0x585d001d,0x585e001d,0x585f001d,0x5860001d,
	0x5861001d,0x5862001d,0x5863001d,0x5864001d,0x5865001d,0x5866001d,0x5867001d,0x5868001d,
	0x5869001d,0x586a001d,0x586b001d,0x586c001d,0x586d001d,0x586e001d,0x586f001d,0x5870001d,
	0x5871001d,0x5872001d,0x5873001d,0x5874001d,0x5875001d,0x5876001d,0x5877001d,0x5878001d,
	0x5879001d,0x587a001d,0x587b001d,0x587c001d,0x587d001d,0x587e001d,0x587f001d,0x5880001d,
	0x5881001d,0x5882001d,0x5883001d,0x5884001d,0x5885001d,0x5886001d,0x5887001d,0x5888001d,
	0x5889001d,0x588a001d,0x588b001d,0x588c001d,0x588d001d,0x588e001d,0x588f001d,0x5890001d,
	0x5891001d,0x5892001d,0x5893001d,0x5894001d,0x5895001d,0x5896001d,0x5897001d,0x5898001d,
	0x5899001d,0x589a001d,0x589b001d,0x589c001d,0x589d001d,0x589e001d,0x589f001d,0x58a0001d,
	0x58a1001d,0x58a2001d,0x58a3001d,0x58a4001d,0x58a5001d,0x58a6001d,0x58a7001d,0x58a8001d,
	0x58a9001d,0x58aa001d,0x58ab001d,0x58ac001d,0x58ad001d,0x58ae001d,0x58af001d,0x58b0001d,
	0x58b1001d,0x58b2001d,0x58b3001d,0x58b4001d,0x58b5001d,0x58b6001d,0x58b7001d,0x58b8001d,
	0x58b9001d,0x58ba001d,0x58bb001d,0x58bc001d,0x58bd001d,0x58be001d,0x58bf001d,0x58c0001d,
	0x58c1001d,0x58c2001d,0x58c3001d,0x58c4001d,0x58c5001d,0x58c6001d,0x58c7001d,0x58c8001d,
	0x58c9001d,0x58ca001d,0x58cb001d,0x58cc001d,0x58cd001d,0x58ce001d,0x58cf001d,0x58d0001d,
	0x58d1001d,0x58d2001d,0x58d3001d,0x58d4001d,0x58d5001d,0x58d6001d,0x58d7001d,0x58d8001d,
	0x58d9001d,0x58da001d,0x58db001d,0x58dc001d,0x58dd001d,0x58de001d,0x58df001d,0x58e0001d,
	0x58e1001d,0x58e2001d,0x58e3001d,0x58e4001d,0x58e5001d,0x58e6001d,0x58e7001d,0x58e8001d,
	0x58e9001d,0x58ea001d,0x58eb001d,0x58ec001d,0x58ed001d,0x58ee001d,0x58ef001d,0x58f0001d,
	0x58f1001d,0x58f2001d,0x58f3001d,0x58f4001d,0x58f5001d,0x58f6001d,0x58f7001d,0x58f8001d,
	0x58f9001d,0x58fa001d,0x58fb001d,0x58fc001d,0x58fd001d,0x58fe001d,0x58ff001d,0x5900001d,
	0x5901001d,0x5902001d,0x5903001d,0x5904001d,0x5905001d,0x5906001d,0x5907001d,0x5908001d,
	0x5909001d,0x590a001d,0x590b001d,0x590c001d,0x590d001d,0x590e001d,0x590f001d,0x5910001d,
	0x5911001d,0x5912001d,0x5913001d,0x5914001d,0x5915001d,0x5916001d,0x5917001d,0x5918001d,
	0x5919001d,0x591a001d,0x591b001d,0x591c001d,0x591d001d,0x591e001d,0x591f001d,0x5920001d,
	0x5921001d,0x5922001d,0x5923001d,0x5924001d,0x5925001d,0x5926001d,0x5927001d,0x5928001d,
	0x5929001d,0x592a001d,0x592b001d,0x592c001d,0x592d001d,0x592e001d,0x592f001d,0x5930001d,
	0x5931001d,0x5932001d,0x5933001d,0x5934001d,0x5935001d,0x5936001d,0x5937001d,0x5938001d,
	0x5939001d,0x593a001d,0x593b001d,0x593c001d,0x593d001d,0x593e001d,0x593f001d,0x5940001d,
	0x5941001d,0x5942001d,0x5943001d,0x5944001d,0x5945001d,0x5946001d,0x5947001d,0x5948001d,
	0x5949001d,0x594a001d,0x594b001d,0x594c001d,0x594d001d,0x594e001d,0x594f001d,0x5950001d,
	0x5951001d,0x5952001d,0x5953001d,0x5954001d,0x5955001d,0x5956001d,0x5957001d,0x5958001d,
	0x5959001d,0x595a001d,0x595b001d,0x595c001d,0x595d001d,0x595e001d,0x595f001d,0x5960001d,
	0x5961001d,0x5962001d,0x5963001d,0x5964001d,0x5965001d,0x5966001d,0x5967001d,0x5968001d,
	0x5969001d,0x596a001d,0x596b001d,0x596c001d,0x596d001d,0x596e001d,0x596f001d,0x5970001d,
	0x5971001d,0x5972001d,0x5973001d,0x5974001d,0x5975001d,0x5976001d,0x5977001d,0x5978001d,
	0x5979001d,0x597a001d,0x597b001d,0x597c001d,0x597d001d,0x597e001d,0x597f001d,0x5980001d,
	0x5981001d,0x5982001d,0x5983001d,0x5984001d,0x5985001d,0x5986001d,0x5987001d,0x5988001d,
	0x5989001d,0x598a001d,0x598b001d,0x598c001d,0x598d001d,0x598e001d,0x598f001d,0x5990001d,
	0x5991001d,0x5992001d,0x5993001d,0x5994001d,0x5995001d,0x5996001d,0x5997001d,0x5998001d,
	0x5999001d,0x599a001d,0x599b001d,0x599c001d,0x599d001d,0x599e001d,0x599f001d,0x59a0001d,
	0x59a1001d,0x59a2001d,0x59a3001d,0x59a4001d,0x59a5001d,0x59a6001d,0x59a7001d,0x59a8001d,
	0x59a9001d,0x59aa001d,0x59ab001d,0x59ac001d,0x59ad001d,0x59ae001d,0x59af001d,0x59b0001d,
	0x59b1001d,0x59b2001d,0x59b3001d,0x59b4001d,0x59b5001d,0x59b6001d,0x59b7001d,0x59b8001d,
	0x59b9001d,0x59ba001d,0x59bb001d,0x59bc001d,0x59bd001d,0x59be001d,0x59bf001d,0x59c0001d,
	0x59c1001d,0x59c2001d,0x59c3001d,0x59c4001d,0x59c5001d,0x59c6001d,0x59c7001d,0x59c8001d,
	0x59c9001d,0x59ca001d,0x59cb001d,0x59cc001d,0x59cd001d,0x59ce001d,0x59cf001d,0x59d0001d,
	0x59d1001d,0x59d2001d,0x59d3001d,0x59d4001d,0x59d5001d,0x59d6001d,0x59d7001d,0x59d8001d,
	0x59d9001d,0x59da001d,0x59db001d,0x59dc001d,0x59dd001d,0x59de001d,0x59df001d,0x59e0001d,
	0x59e1001d,0x59e2001d,0x59e3001d,0x59e4001d,0x59e5001d,0x59e6001d,0x59e7001d,0x59e8001d,
	0x59e9001d,0x59ea001d,0x59eb001d,0x59ec001d,0x59ed001d,0x59ee001d,0x59ef001d,0x59f0001d,
	0x59f1001d,0x59f2001d,0x59f3001d,0x59f4001d,0x59f5001d,0x59f6001d,0x59f7001d,0x59f8001d,
	0x59f9001d,0x59fa001d,0x59fb001d,0x59fc001d,0x59fd001d,0x59fe001d,0x59ff001d,0x5a00001d,
	0x5a01001d,0x5a02001d,0x5a03001d,0x5a04001d,0x5a05001d,0x5a06001d,0x5a07001d,0x5a08001d,
	0x5a09001d,0x5a0a001d,0x5a0b001d,0x5a0c001d,0x5a0d001d,0x5a0e001d,0x5a0f001d,0x5a10001d,
	0x5a11001d,0x5a12001d,0x5a13001d,0x5a14001d,0x5a15001d,0x5a16001d,0x5a17001d,0x5a18001d,
	0x5a19001d,0x5a1a001d,0x5a1b001d,0x5a1c001d,0x5a1d001d,0x5a1e001d,0x5a1f001d,0x5a20001d,
	0x5a21001d,0x5a22001d,0x5a23001d,0x5a24001d,0x5a25001d,0x5a26001d,0x5a27001d,0x5a28001d,
	0x5a29001d,0x5a2a001d,0x5a2b001d,0x5a2c001d,0x5a2d001d,0x5a2e001d,0x5a2f001d,0x5a30001d,
	0x5a31001d,0x5a32001d,0x5a33001d,0x5a34001d,0x5a35001d,0x5a36001d,0x5a37001d,0x5a38001d,
	0x5a39001d,0x5a3a001d,0x5a3b001d,0x5a3c001d,0x5a3d001d,0x5a3e001d,0x5a3f001d,0x5a40001d,
	0x5a41001d,0x5a42001d,0x5a43001d,0x5a44001d,0x5a45001d,0x5a46001d,0x5a47001d,0x5a48001d,
	0x5a49001d,0x5a4a001d,0x5a4b001d,0x5a4c001d,0x5a4d001d,0x5a4e001d,0x5a4f001d,0x5a50001d,
	0x5a51001d,0x5a52001d,0x5a53001d,0x5a54001d,0x5a55001d,0x5a56001d,0x5a57001d,0x5a58001d,
	0x5a59001d,0x5a5a001d,0x5a5b001d,0x5a5c001d,0x5a5d001d,0x5a5e001d,0x5a5f001d,0x5a60001d,
	0x5a61001d,0x5a62001d,0x5a63001d,0x5a64001d,0x5a65001d,0x5a66001d,0x5a67001d,0x5a68001d,
	0x5a69001d,0x5a6a001d,0x5a6b001d,0x5a6c001d,0x5a6d001d,0x5a6e001d,0x5a6f001d,0x5a70001d,
	0x5a71001d,0x5a72001d,0x5a73001d,0x5a74001d,0x5a75001d,0x5a76001d,0x5a77001d,0x5a78001d,
	0x5a79001d,0x5a7a001d,0x5a7b001d,0x5a7c001d,0x5a7d001d,0x5a7e001d,0x5a7f001d,0x5a80001d,
	0x5a81001d,0x5a82001d,0x5a83001d,0x5a84001d,0x5a85001d,0x5a86001d,0x5a87001d,0x5a88001d,
	0x5a89001d,0x5a8a001d,0x5a8b001d,0x5a8c001d,0x5a8d001d,0x5a8e001d,0x5a8f001d,0x5a90001d,
	0x5a91001d,0x5a92001d,0x5a93001d,0x5a94001d,0x5a95001d,0x5a96001d,0x5a97001d,0x5a98001d,
	0x5a99001d,0x5a9a001d,0x5a9b001d,0x5a9c001d,0x5a9d001d,0x5a9e001d,0x5a9f001d,0x5aa0001d,
	0x5aa1001d,0x5aa2001d,0x5aa3001d,0x5aa4001d,0x5aa5001d,0x5aa6001d,0x5aa7001d,0x5aa8001d,
	0x5aa9001d,0x5aaa001d,0x5aab001d,0x5aac001d,0x5aad001d,0x5aae001d,0x5aaf001d,0x5ab0001d,
	0x5ab1001d,0x5ab2001d,0x5ab3001d,0x5ab4001d,0x5ab5001d,0x5ab6001d,0x5ab7001d,0x5ab8001d,
	0x5ab9001d,0x5aba001d,0x5abb001d,0x5abc001d,0x5abd001d,0x5abe001d,0x5abf001d,0x5ac0001d,
	0x5ac1001d,0x5ac2001d,0x5ac3001d,0x5ac4001d,0x5ac5001d,0x5ac6001d,0x5ac7001d,0x5ac8001d,
	0x5ac9001d,0x5aca001d,0x5acb001d,0x5acc001d,0x5acd001d,0x5ace001d,0x5acf001d,0x5ad0001d,
	0x5ad1001d,0x5ad2001d,0x5ad3001d,0x5ad4001d,0x5ad5001d,0x5ad6001d,0x5ad7001d,0x5ad8001d,
	0x5ad9001d,0x5ada001d,0x5adb001d,0x5adc001d,0x5add001d,0x5ade001d,0x5adf001d,0x5ae0001d,
	0x5ae1001d,0x5ae2001d,0x5ae3001d,0x5ae4001d,0x5ae5001d,0x5ae6001d,0x5ae7001d,0x5ae8001d,
	0x5ae9001d,0x5aea001d,0x5aeb001d,0x5aec001d,0x5aed001d,0x5aee001d,0x5aef001d,0x5af0001d,
	0x5af1001d,0x5af2001d,0x5af3001d,0x5af4001d,0x5af5001d,0x5af6001d,0x5af7001d,0x5af8001d,
	0x5af9001d,0x5afa001d,0x5afb001d,0x5afc001d,0x5afd001d,0x5afe001d,0x5aff001d,0x5b00001d,
	0x5b01001d,0x5b02001d,0x5b03001d,0x5b04001d,0x5b05001d,0x5b06001d,0x5b07001d,0x5b08001d,
	0x5b09001d,0x5b0a001d,0x5b0b001d,0x5b0c001d,0x5b0d001d,0x5b0e001d,0x5b0f001d,0x5b10001d,
	0x5b11001d,0x5b12001d,0x5b13001d,0x5b14001d,0x5b15001d,0x5b16001d,0x5b17001d,0x5b18001d,
	0x5b19001d,0x5b1a001d,0x5b1b001d,0x5b1c001d,0x5b1d001d,0x5b1e001d,0x5b1f001d,0x5b20001d,
	0x5b21001d,0x5b22001d,0x5b23001d,0x5b24001d,0x5b25001d,0x5b26001d,0x5b27001d,0x5b28001d,
	0x5b29001d,0x5b2a001d,0x5b2b001d,0x5b2c001d,0x5b2d001d,0x5b2e001d,0x5b2f001d,0x5b30001d,
	0x5b31001d,0x5b32001d,0x5b33001d,0x5b34001d,0x5b35001d,0x5b36001d,0x5b37001d,0x5b38001d,
	0x5b39001d,0x5b3a001d,0x5b3b001d,0x5b3c001d,0x5b3d001d,0x5b3e001d,0x5b3f001d,0x5b40001d,
	0x5b41001d,0x5b42001d,0x5b43001d,0x5b44001d,0x5b45001d,0x5b46001d,0x5b47001d,0x5b48001d,
	0x5b49001d,0x5b4a001d,0x5b4b001d,0x5b4c001d,0x5b4d001d,0x5b4e001d,0x5b4f001d,0x5b50001d,
	0x5b51001d,0x5b52001d,0x5b53001d,0x5b54001d,0x5b55001d,0x5b56001d,0x5b57001d,0x5b58001d,
	0x5b59001d,0x5b5a001d,0x5b5b001d,0x5b5c001d,0x5b5d001d,0x5b5e001d,0x5b5f001d,0x5b60001d,
	0x5b61001d,0x5b62001d,0x5b63001d,0x5b64001d,0x5b65001d,0x5b66001d,0x5b67001d,0x5b68001d,
	0x5b69001d,0x5b6a001d,0x5b6b001d,0x5b6c001d,0x5b6d001d,0x5b6e001d,0x5b6f001d,0x5b70001d,
	0x5b71001d,0x5b72001d,0x5b73001d,0x5b74001d,0x5b75001d,0x5b76001d,0x5b77001d,0x5b78001d,
	0x5b79001d,0x5b7a001d,0x5b7b001d,0x5b7c001d,0x5b7d001d,0x5b7e001d,0x5b7f001d,0x5b80001d,
	0x5b81001d,0x5b82001d,0x5b83001d,0x5b84001d,0x5b85001d,0x5b86001d,0x5b87001d,0x5b88001d,
	0x5b89001d,0x5b8a001d,0x5b8b001d,0x5b8c001d,0x5b8d001d,0x5b8e001d,0x5b8f001d,0x5b90001d,
	0x5b91001d,0x5b92001d,0x5b93001d,0x5b94001d,0x5b95001d,0x5b96001d,0x5b97001d,0x5b98001d,
	0x5b99001d,0x5b9a001d,0x5b9b001d,0x5b9c001d,0x5b9d001d,0x5b9e001d,0x5b9f001d,0x5ba0001d,
	0x5ba1001d,0x5ba2001d,0x5ba3001d,0x5ba4001d,0x5ba5001d,0x5ba6001d,0x5ba7001d,0x5ba8001d,
	0x5ba9001d,0x5baa001d,0x5bab001d,0x5bac001d,0x5bad001d,0x5bae001d,0x5baf001d,0x5bb0001d,
	0x5bb1001d,0x5bb2001d,0x5bb3001d,0x5bb4001d,0x5bb5001d,0x5bb6001d,0x5bb7001d,0x5bb8001d,
	0x5bb9001d,0x5bba001d,0x5bbb001d,0x5bbc001d,0x5bbd001d,0x5bbe001d,0x5bbf001d,0x5bc0001d,
	0x5bc1001d,0x5bc2001d,0x5bc3001d,0x5bc4001d,0x5bc5001d,0x5bc6001d,0x5bc7001d,0x5bc8001d,
	0x5bc9001d,0x5bca001d,0x5bcb001d,0x5bcc001d,0x5bcd001d,0x5bce001d,0x5bcf001d,0x5bd0001d,
	0x5bd1001d,0x5bd2001d,0x5bd3001d,0x5bd4001d,0x5bd5001d,0x5bd6001d,0x5bd7001d,0x5bd8001d,
	0x5bd9001d,0x5bda001d,0x5bdb001d,0x5bdc001d,0x5bdd001d,0x5bde001d,0x5bdf001d,0x5be0001d,
	0x5be1001d,0x5be2001d,0x5be3001d,0x5be4001d,0x5be5001d,0x5be6001d,0x5be7001d,0x5be8001d,
	0x5be9001d,0x5bea001d,0x5beb001d,0x5bec001d,0x5bed001d,0x5bee001d,0x5bef001d,0x5bf0001d,
	0x5bf1001d,0x5bf2001d,0x5bf3001d,0x5bf4001d,0x5bf5001d,0x5bf6001d,0x5bf7001d,0x5bf8001d,
	0x5bf9001d,0x5bfa001d,0x5bfb001d,0x5bfc001d,0x5bfd001d,0x5bfe001d,0x5bff001d,0x5c00001d,
	0x5c01001d,0x5c02001d,0x5c03001d,0x5c04001d,0x5c05001d,0x5c06001d,0x5c07001d,0x5c08001d,
	0x5c09001d,0x5c0a001d,0x5c0b001d,0x5c0c001d,0x5c0d001d,0x5c0e001d,0x5c0f001d,0x5c10001d,
	0x5c11001d,0x5c12001d,0x5c13001d,0x5c14001d,0x5c15001d,0x5c16001d,0x5c17001d,0x5c18001d,
	0x5c19001d,0x5c1a001d,0x5c1b001d,0x5c1c001d,0x5c1d001d,0x5c1e001d,0x5c1f001d,0x5c20001d,
	0x5c21001d,0x5c22001d,0x5c23001d,0x5c24001d,0x5c25001d,0x5c26001d,0x5c27001d,0x5c28001d,
	0x5c29001d,0x5c2a001d,0x5c2b001d,0x5c2c001d,0x5c2d001d,0x5c2e001d,0x5c2f001d,0x5c30001d,
	0x5c31001d,0x5c32001d,0x5c33001d,0x5c34001d,0x5c35001d,0x5c36001d,0x5c37001d,0x5c38001d,
	0x5c39001d,0x5c3a001d,0x5c3b001d,0x5c3c001d,0x5c3d001d,0x5c3e001d,0x5c3f001d,0x5c40001d,
	0x5c41001d,0x5c42001d,0x5c43001d,0x5c44001d,0x5c45001d,0x5c46001d,0x5c47001d,0x5c48001d,
	0x5c49001d,0x5c4a001d,0x5c4b001d,0x5c4c001d,0x5c4d001d,0x5c4e001d,0x5c4f001d,0x5c50001d,
	0x5c51001d,0x5c52001d,0x5c53001d,0x5c54001d,0x5c55001d,0x5c56001d,0x5c57001d,0x5c58001d,
	0x5c59001d,0x5c5a001d,0x5c5b001d,0x5c5c001d,0x5c5d001d,0x5c5e001d,0x5c5f001d,0x5c60001d,
	0x5c61001d,0x5c62001d,0x5c63001d,0x5c64001d,0x5c65001d,0x5c66001d,0x5c67001d,0x5c68001d,
	0x5c69001d,0x5c6a001d,0x5c6b001d,0x5c6c001d,0x5c6d001d,0x5c6e001d,0x5c6f001d,0x5c70001d,
	0x5c71001d,0x5c72001d,0x5c73001d,0x5c74001d,0x5c75001d,0x5c76001d,0x5c77001d,0x5c78001d,
	0x5c79001d,0x5c7a001d,0x5c7b001d,0x5c7c001d,0x5c7d001d,0x5c7e001d,0x5c7f001d,0x5c80001d,
	0x5c81001d,0x5c82001d,0x5c83001d,0x5c84001d,0x5c85001d,0x5c86001d,0x5c87001d,0x5c88001d,
	0x5c89001d,0x5c8a001d,0x5c8b001d,0x5c8c001d,0x5c8d001d,0x5c8e001d,0x5c8f001d,0x5c90001d,
	0x5c91001d,0x5c92001d,0x5c93001d,0x5c94001d,0x5c95001d,0x5c96001d,0x5c97001d,0x5c98001d,
	0x5c99001d,0x5c9a001d,0x5c9b001d,0x5c9c001d,0x5c9d001d,0x5c9e001d,0x5c9f001d,0x5ca0001d,
	0x5ca1001d,0x5ca2001d,0x5ca3001d,0x5ca4001d,0x5ca5001d,0x5ca6001d,0x5ca7001d,0x5ca8001d,
	0x5ca9001d,0x5caa001d,0x5cab001d,0x5cac001d,0x5cad001d,0x5cae001d,0x5caf001d,0x5cb0001d,
	0x5cb1001d,0x5cb2001d,0x5cb3001d,0x5cb4001d,0x5cb5001d,0x5cb6001d,0x5cb7001d,0x5cb8001d,
	0x5cb9001d,0x5cba001d,0x5cbb001d,0x5cbc001d,0x5cbd001d,0x5cbe001d,0x5cbf001d,0x5cc0001d,
	0x5cc1001d,0x5cc2001d,0x5cc3001d,0x5cc4001d,0x5cc5001d,0x5cc6001d,0x5cc7001d,0x5cc8001d,
	0x5cc9001d,0x5cca001d,0x5ccb001d,0x5ccc001d,0x5ccd001d,0x5cce001d,0x5ccf001d,0x5cd0001d,
	0x5cd1001d,0x5cd2001d,0x5cd3001d,0x5cd4001d,0x5cd5001d,0x5cd6001d,0x5cd7001d,0x5cd8001d,
	0x5cd9001d,0x5cda001d,0x5cdb001d,0x5cdc001d,0x5cdd001d,0x5cde001d,0x5cdf001d,0x5ce0001d,
	0x5ce1001d,0x5ce2001d,0x5ce3001d,0x5ce4001d,0x5ce5001d,0x5ce6001d,0x5ce7001d,0x5ce8001d,
	0x5ce9001d,0x5cea001d,0x5ceb001d,0x5cec001d,0x5ced001d,0x5cee001d,0x5cef001d,0x5cf0001d,
	0x5cf1001d,0x5cf2001d,0x5cf3001d,0x5cf4001d,0x5cf5001d,0x5cf6001d,0x5cf7001d,0x5cf8001d,
	0x5cf9001d,0x5cfa001d,0x5cfb001d,0x5cfc001d,0x5cfd001d,0x5cfe001d,0x5cff001d,0x5d00001d,
	0x5d01001d,0x5d02001d,0x5d03001d,0x5d04001d,0x5d05001d,0x5d06001d,0x5d07001d,0x5d08001d,
	0x5d09001d,0x5d0a001d,0x5d0b001d,0x5d0c001d,0x5d0d001d,0x5d0e001d,0x5d0f001d,0x5d10001d,
	0x5d11001d,0x5d12001d,0x5d13001d,0x5d14001d,0x5d15001d,0x5d16001d,0x5d17001d,0x5d18001d,
	0x5d19001d,0x5d1a001d,0x5d1b001d,0x5d1c001d,0x5d1d001d,0x5d1e001d,0x5d1f001d,0x5d20001d,
	0x5d21001d,0x5d22001d,0x5d23001d,0x5d24001d,0x5d25001d,0x5d26001d,0x5d27001d,0x5d28001d,
	0x5d29001d,0x5d2a001d,0x5d2b001d,0x5d2c001d,0x5d2d001d,0x5d2e001d,0x5d2f001d,0x5d30001d,
	0x5d31001d,0x5d32001d,0x5d33001d,0x5d34001d,0x5d35001d,0x5d36001d,0x5d37001d,0x5d38001d,
	0x5d39001d,0x5d3a001d,0x5d3b001d,0x5d3c001d,0x5d3d001d,0x5d3e001d,0x5d3f001d,0x5d40001d,
	0x5d41001d,0x5d42001d,0x5d43001d,0x5d44001d,0x5d45001d,0x5d46001d,0x5d47001d,0x5d48001d,
	0x5d49001d,0x5d4a001d,0x5d4b001d,0x5d4c001d,0x5d4d001d,0x5d4e001d,0x5d4f001d,0x5d50001d,
	0x5d51001d,0x5d52001d,0x5d53001d,0x5d54001d,0x5d55001d,0x5d56001d,0x5d57001d,0x5d58001d,
	0x5d59001d,0x5d5a001d,0x5d5b001d,0x5d5c001d,0x5d5d001d,0x5d5e001d,0x5d5f001d,0x5d60001d,
	0x5d61001d,0x5d62001d,0x5d63001d,0x5d64001d,0x5d65001d,0x5d66001d,0x5d67001d,0x5d68001d,
	0x5d69001d,0x5d6a001d,0x5d6b001d,0x5d6c001d,0x5d6d001d,0x5d6e001d,0x5d6f001d,0x5d70001d,
	0x5d71001d,0x5d72001d,0x5d73001d,0x5d74001d,0x5d75001d,0x5d76001d,0x5d77001d,0x5d78001d,
	0x5d79001d,0x5d7a001d,0x5d7b001d,0x5d7c001d,0x5d7d001d,0x5d7e001d,0x5d7f001d,0x5d80001d,
	0x5d81001d,0x5d82001d,0x5d83001d,0x5d84001d,0x5d85001d,0x5d86001d,0x5d87001d,0x5d88001d,
	0x5d89001d,0x5d8a001d,0x5d8b001d,0x5d8c001d,0x5d8d001d,0x5d8e001d,0x5d8f001d,0x5d90001d,
	0x5d91001d,0x5d92001d,0x5d93001d,0x5d94001d,0x5d95001d,0x5d96001d,0x5d97001d,0x5d98001d,
	0x5d99001d,0x5d9a001d,0x5d9b001d,0x5d9c001d,0x5d9d001d,0x5d9e001d,0x5d9f001d,0x5da0001d,
	0x5da1001d,0x5da2001d,0x5da3001d,0x5da4001d,0x5da5001d,0x5da6001d,0x5da7001d,0x5da8001d,
	0x5da9001d,0x5daa001d,0x5dab001d,0x5dac001d,0x5dad001d,0x5dae001d,0x5daf001d,0x5db0001d,
	0x5db1001d,0x5db2001d,0x5db3001d,0x5db4001d,0x5db5001d,0x5db6001d,0x5db7001d,0x5db8001d,
	0x5db9001d,0x5dba001d,0x5dbb001d,0x5dbc001d,0x5dbd001d,0x5dbe001d,0x5dbf001d,0x5dc0001d,
	0x5dc1001d,0x5dc2001d,0x5dc3001d,0x5dc4001d,0x5dc5001d,0x5dc6001d,0x5dc7001d,0x5dc8001d,
	0x5dc9001d,0x5dca001d,0x5dcb001d,0x5dcc001d,0x5dcd001d,0x5dce001d,0x5dcf001d,0x5dd0001d,
	0x5dd1001d,0x5dd2001d,0x5dd3001d,0x5dd4001d,0x5dd5001d,0x5dd6001d,0x5dd7001d,0x5dd8001d,
	0x5dd9001d,0x5dda001d,0x5ddb001d,0x5ddc001d,0x5ddd001d,0x5dde001d,0x5ddf001d,0x5de0001d,
	0x5de1001d,0x5de2001d,0x5de3001d,0x5de4001d,0x5de5001d,0x5de6001d,0x5de7001d,0x5de8001d,
	0x5de9001d,0x5dea001d,0x5deb001d,0x5dec001d,0x5ded001d,0x5dee001d,0x5def001d,0x5df0001d,
	0x5df1001d,0x5df2001d,0x5df3001d,0x5df4001d,0x5df5001d,0x5df6001d,0x5df7001d,0x5df8001d,
	0x5df9001d,0x5dfa001d,0x5dfb001d,0x5dfc001d,0x5dfd001d,0x5dfe001d,0x5dff001d,0x5e00001d,
	0x5e01001d,0x5e02001d,0x5e03001d,0x5e04001d,0x5e05001d,0x5e06001d,0x5e07001d,0x5e08001d,
	0x5e09001d,0x5e0a001d,0x5e0b001d,0x5e0c001d,0x5e0d001d,0x5e0e001d,0x5e0f001d,0x5e10001d,
	0x5e11001d,0x5e12001d,0x5e13001d,0x5e14001d,0x5e15001d,0x5e16001d,0x5e17001d,0x5e18001d,
	0x5e19001d,0x5e1a001d,0x5e1b001d,0x5e1c001d,0x5e1d001d,0x5e1e001d,0x5e1f001d,0x5e20001d,
	0x5e21001d,0x5e22001d,0x5e23001d,0x5e24001d,0x5e25001d,0x5e26001d,0x5e27001d,0x5e28001d,
	0x5e29001d,0x5e2a001d,0x5e2b001d,0x5e2c001d,0x5e2d001d,0x5e2e001d,0x5e2f001d,0x5e30001d,
	0x5e31001d,0x5e32001d,0x5e33001d,0x5e34001d,0x5e35001d,0x5e36001d,0x5e37001d,0x5e38001d,
	0x5e39001d,0x5e3a001d,0x5e3b001d,0x5e3c001d,0x5e3d001d,0x5e3e001d,0x5e3f001d,0x5e40001d,
	0x5e41001d,0x5e42001d,0x5e43001d,0x5e44001d,0x5e45001d,0x5e46001d,0x5e47001d,0x5e48001d,
	0x5e49001d,0x5e4a001d,0x5e4b001d,0x5e4c001d,0x5e4d001d,0x5e4e001d,0x5e4f001d,0x5e50001d,
	0x5e51001d,0x5e52001d,0x5e53001d,0x5e54001d,0x5e55001d,0x5e56001d,0x5e57001d,0x5e58001d,
	0x5e59001d,0x5e5a001d,0x5e5b001d,0x5e5c001d,0x5e5d001d,0x5e5e001d,0x5e5f001d,0x5e60001d,
	0x5e61001d,0x5e62001d,0x5e63001d,0x5e64001d,0x5e65001d,0x5e66001d,0x5e67001d,0x5e68001d,
	0x5e69001d,0x5e6a001d,0x5e6b001d,0x5e6c001d,0x5e6d001d,0x5e6e001d,0x5e6f001d,0x5e70001d,
	0x5e71001d,0x5e72001d,0x5e73001d,0x5e74001d,0x5e75001d,0x5e76001d,0x5e77001d,0x5e78001d,
	0x5e79001d,0x5e7a001d,0x5e7b001d,0x5e7c001d,0x5e7d001d,0x5e7e001d,0x5e7f001d,0x5e80001d,
	0x5e81001d,0x5e82001d,0x5e83001d,0x5e84001d,0x5e85001d,0x5e86001d,0x5e87001d,0x5e88001d,
	0x5e89001d,0x5e8a001d,0x5e8b001d,0x5e8c001d,0x5e8d001d,0x5e8e001d,0x5e8f001d,0x5e90001d,
	0x5e91001d,0x5e92001d,0x5e93001d,0x5e94001d,0x5e95001d,0x5e96001d,0x5e97001d,0x5e98001d,
	0x5e99001d,0x5e9a001d,0x5e9b001d,0x5e9c001d,0x5e9d001d,0x5e9e001d,0x5e9f001d,0x5ea0001d,
	0x5ea1001d,0x5ea2001d,0x5ea3001d,0x5ea4001d,0x5ea5001d,0x5ea6001d,0x5ea7001d,0x5ea8001d,
	0x5ea9001d,0x5eaa001d,0x5eab001d,0x5eac001d,0x5ead001d,0x5eae001d,0x5eaf001d,0x5eb0001d,
	0x5eb1001d,0x5eb2001d,0x5eb3001d,0x5eb4001d,0x5eb5001d,0x5eb6001d,0x5eb7001d,0x5eb8001d,
	0x5eb9001d,0x5eba001d,0x5ebb001d,0x5ebc001d,0x5ebd001d,0x5ebe001d,0x5ebf001d,0x5ec0001d,
	0x5ec1001d,0x5ec2001d,0x5ec3001d,0x5ec4001d,0x5ec5001d,0x5ec6001d,0x5ec7001d,0x5ec8001d,
	0x5ec9001d,0x5eca001d,0x5ecb001d,0x5ecc001d,0x5ecd001d,0x5ece001d,0x5ecf001d,0x5ed0001d,
	0x5ed1001d,0x5ed2001d,0x5ed3001d,0x5ed4001d,0x5ed5001d,0x5ed6001d,0x5ed7001d,0x5ed8001d,
	0x5ed9001d,0x5eda001d,0x5edb001d,0x5edc001d,0x5edd001d,0x5ede001d,0x5edf001d,0x5ee0001d,
	0x5ee1001d,0x5ee2001d,0x5ee3001d,0x5ee4001d,0x5ee5001d,0x5ee6001d,0x5ee7001d,0x5ee8001d,
	0x5ee9001d,0x5eea001d,0x5eeb001d,0x5eec001d,0x5eed001d,0x5eee001d,0x5eef001d,0x5ef0001d,
	0x5ef1001d,0x5ef2001d,0x5ef3001d,0x5ef4001d,0x5ef5001d,0x5ef6001d,0x5ef7001d,0x5ef8001d,
	0x5ef9001d,0x5efa001d,0x5efb001d,0x5efc001d,0x5efd001d,0x5efe001d,0x5eff001d,0x5f00001d,
	0x5f01001d,0x5f02001d,0x5f03001d,0x5f04001d,0x5f05001d,0x5f06001d,0x5f07001d,0x5f08001d,
	0x5f09001d,0x5f0a001d,0x5f0b001d,0x5f0c001d,0x5f0d001d,0x5f0e001d,0x5f0f001d,0x5f10001d,
	0x5f11001d,0x5f12001d,0x5f13001d,0x5f14001d,0x5f15001d,0x5f16001d,0x5f17001d,0x5f18001d,
	0x5f19001d,0x5f1a001d,0x5f1b001d,0x5f1c001d,0x5f1d001d,0x5f1e001d,0x5f1f001d,0x5f20001d,
	0x5f21001d,0x5f22001d,0x5f23001d,0x5f24001d,0x5f25001d,0x5f26001d,0x5f27001d,0x5f28001d,
	0x5f29001d,0x5f2a001d,0x5f2b001d,0x5f2c001d,0x5f2d001d,0x5f2e001d,0x5f2f001d,0x5f30001d,
	0x5f31001d,0x5f32001d,0x5f33001d,0x5f34001d,0x5f35001d,0x5f36001d,0x5f37001d,0x5f38001d,
	0x5f39001d,0x5f3a001d,0x5f3b001d,0x5f3c001d,0x5f3d001d,0x5f3e001d,0x5f3f001d,0x5f40001d,
	0x5f41001d,0x5f42001d,0x5f43001d,0x5f44001d,0x5f45001d,0x5f46001d,0x5f47001d,0x5f48001d,
	0x5f49001d,0x5f4a001d,0x5f4b001d,0x5f4c001d,0x5f4d001d,0x5f4e001d,0x5f4f001d,0x5f50001d,
	0x5f51001d,0x5f52001d,0x5f53001d,0x5f54001d,0x5f55001d,0x5f56001d,0x5f57001d,0x5f58001d,
	0x5f59001d,0x5f5a001d,0x5f5b001d,0x5f5c001d,0x5f5d001d,0x5f5e001d,0x5f5f001d,0x5f60001d,
	0x5f61001d,0x5f62001d,0x5f63001d,0x5f64001d,0x5f65001d,0x5f66001d,0x5f67001d,0x5f68001d,
	0x5f69001d,0x5f6a001d,0x5f6b001d,0x5f6c001d,0x5f6d001d,0x5f6e001d,0x5f6f001d,0x5f70001d,
	0x5f71001d,0x5f72001d,0x5f73001d,0x5f74001d,0x5f75001d,0x5f76001d,0x5f77001d,0x5f78001d,
	0x5f79001d,0x5f7a001d,0x5f7b001d,0x5f7c001d,0x5f7d001d,0x5f7e001d,0x5f7f001d,0x5f80001d,
	0x5f81001d,0x5f82001d,0x5f83001d,0x5f84001d,0x5f85001d,0x5f86001d,0x5f87001d,0x5f88001d,
	0x5f89001d,0x5f8a001d,0x5f8b001d,0x5f8c001d,0x5f8d001d,0x5f8e001d,0x5f8f001d,0x5f90001d,
	0x5f91001d,0x5f92001d,0x5f93001d,0x5f94001d,0x5f95001d,0x5f96001d,0x5f97001d,0x5f98001d,
	0x5f99001d,0x5f9a001d,0x5f9b001d,0x5f9c001d,0x5f9d001d,0x5f9e001d,0x5f9f001d,0x5fa0001d,
	0x5fa1001d,0x5fa2001d,0x5fa3001d,0x5fa4001d,0x5fa5001d,0x5fa6001d,0x5fa7001d,0x5fa8001d,
	0x5fa9001d,0x5faa001d,0x5fab001d,0x5fac001d,0x5fad001d,0x5fae001d,0x5faf001d,0x5fb0001d,
	0x5fb1001d,0x5fb2001d,0x5fb3001d,0x5fb4001d,0x5fb5001d,0x5fb6001d,0x5fb7001d,0x5fb8001d,
	0x5fb9001d,0x5fba001d,0x5fbb001d,0x5fbc001d,0x5fbd001d,0x5fbe001d,0x5fbf001d,0x5fc0001d,
	0x5fc1001d,0x5fc2001d,0x5fc3001d,0x5fc4001d,0x5fc5001d,0x5fc6001d,0x5fc7001d,0x5fc8001d,
	0x5fc9001d,0x5fca001d,0x5fcb001d,0x5fcc001d,0x5fcd001d,0x5fce001d,0x5fcf001d,0x5fd0001d,
	0x5fd1001d,0x5fd2001d,0x5fd3001d,0x5fd4001d,0x5fd5001d,0x5fd6001d,0x5fd7001d,0x5fd8001d,
	0x5fd9001d,0x5fda001d,0x5fdb001d,0x5fdc001d,0x5fdd001d,0x5fde001d,0x5fdf001d,0x5fe0001d,
	0x5fe1001d,0x5fe2001d,0x5fe3001d,0x5fe4001d,0x5fe5001d,0x5fe6001d,0x5fe7001d,0x5fe8001d,
	0x5fe9001d,0x5fea001d,0x5feb001d,0x5fec001d,0x5fed001d,0x5fee001d,0x5fef001d,0x5ff0001d,
	0x5ff1001d,0x5ff2001d,0x5ff3001d,0x5ff4001d,0x5ff5001d,0x5ff6001d,0x5ff7001d,0x5ff8001d,
	0x5ff9001d,0x5ffa001d,0x5ffb001d,0x5ffc001d,0x5ffd001d,0x5ffe001d,0x5fff001d,0x6000001d,
	0x6001001d,0x6002001d,0x6003001d,0x6004001d,0x6005001d,0x6006001d,0x6007001d,0x6008001d,
	0x6009001d,0x600a001d,0x600b001d,0x600c001d,0x600d001d,0x600e001d,0x600f001d,0x6010001d,
	0x6011001d,0x6012001d,0x6013001d,0x6014001d,0x6015001d,0x6016001d,0x6017001d,0x6018001d,
	0x6019001d,0x601a001d,0x601b001d,0x601c001d,0x601d001d,0x601e001d,0x601f001d,0x6020001d,
	0x6021001d,0x6022001d,0x6023001d,0x6024001d,0x6025001d,0x6026001d,0x6027001d,0x6028001d,
	0x6029001d,0x602a001d,0x602b001d,0x602c001d,0x602d001d,0x602e001d,0x602f001d,0x6030001d,
	0x6031001d,0x6032001d,0x6033001d,0x6034001d,0x6035001d,0x6036001d,0x6037001d,0x6038001d,
	0x6039001d,0x603a001d,0x603b001d,0x603c001d,0x603d001d,0x603e001d,0x603f001d,0x6040001d,
	0x6041001d,0x6042001d,0x6043001d,0x6044001d,0x6045001d,0x6046001d,0x6047001d,0x6048001d,
	0x6049001d,0x604a001d,0x604b001d,0x604c001d,0x604d001d,0x604e001d,0x604f001d,0x6050001d,
	0x6051001d,0x6052001d,0x6053001d,0x6054001d,0x6055001d,0x6056001d,0x6057001d,0x6058001d,
	0x6059001d,0x605a001d,0x605b001d,0x605c001d,0x605d001d,0x605e001d,0x605f001d,0x6060001d,
	0x6061001d,0x6062001d,0x6063001d,0x6064001d,0x6065001d,0x6066001d,0x6067001d,0x6068001d,
	0x6069001d,0x606a001d,0x606b001d,0x606c001d,0x606d001d,0x606e001d,0x606f001d,0x6070001d,
	0x6071001d,0x6072001d,0x6073001d,0x6074001d,0x6075001d,0x6076001d,0x6077001d,0x6078001d,
	0x6079001d,0x607a001d,0x607b001d,0x607c001d,0x607d001d,0x607e001d,0x607f001d,0x6080001d,
	0x6081001d,0x6082001d,0x6083001d,0x6084001d,0x6085001d,0x6086001d,0x6087001d,0x6088001d,
	0x6089001d,0x608a001d,0x608b001d,0x608c001d,0x608d001d,0x608e001d,0x608f001d,0x6090001d,
	0x6091001d,0x6092001d,0x6093001d,0x6094001d,0x6095001d,0x6096001d,0x6097001d,0x6098001d,
	0x6099001d,0x609a001d,0x609b001d,0x609c001d,0x609d001d,0x609e001d,0x609f001d,0x60a0001d,
	0x60a1001d,0x60a2001d,0x60a3001d,0x60a4001d,0x60a5001d,0x60a6001d,0x60a7001d,0x60a8001d,
	0x60a9001d,0x60aa001d,0x60ab001d,0x60ac001d,0x60ad001d,0x60ae001d,0x60af001d,0x60b0001d,
	0x60b1001d,0x60b2001d,0x60b3001d,0x60b4001d,0x60b5001d,0x60b6001d,0x60b7001d,0x60b8001d,
	0x60b9001d,0x60ba001d,0x60bb001d,0x60bc001d,0x60bd001d,0x60be001d,0x60bf001d,0x60c0001d,
	0x60c1001d,0x60c2001d,0x60c3001d,0x60c4001d,0x60c5001d,0x60c6001d,0x60c7001d,0x60c8001d,
	0x60c9001d,0x60ca001d,0x60cb001d,0x60cc001d,0x60cd001d,0x60ce001d,0x60cf001d,0x60d0001d,
	0x60d1001d,0x60d2001d,0x60d3001d,0x60d4001d,0x60d5001d,0x60d6001d,0x60d7001d,0x60d8001d,
	0x60d9001d,0x60da001d,0x60db001d,0x60dc001d,0x60dd001d,0x60de001d,0x60df001d,0x60e0001d,
	0x60e1001d,0x60e2001d,0x60e3001d,0x60e4001d,0x60e5001d,0x60e6001d,0x60e7001d,0x60e8001d,
	0x60e9001d,0x60ea001d,0x60eb001d,0x60ec001d,0x60ed001d,0x60ee001d,0x60ef001d,0x60f0001d,
	0x60f1001d,0x60f2001d,0x60f3001d,0x60f4001d,0x60f5001d,0x60f6001d,0x60f7001d,0x60f8001d,
	0x60f9001d,0x60fa001d,0x60fb001d,0x60fc001d,0x60fd001d,0x60fe001d,0x60ff001d,0x6100001d,
	0x6101001d,0x6102001d,0x6103001d,0x6104001d,0x6105001d,0x6106001d,0x6107001d,0x6108001d,
	0x6109001d,0x610a001d,0x610b001d,0x610c001d,0x610d001d,0x610e001d,0x610f001d,0x6110001d,
	0x6111001d,0x6112001d,0x6113001d,0x6114001d,0x6115001d,0x6116001d,0x6117001d,0x6118001d,
	0x6119001d,0x611a001d,0x611b001d,0x611c001d,0x611d001d,0x611e001d,0x611f001d,0x6120001d,
	0x6121001d,0x6122001d,0x6123001d,0x6124001d,0x6125001d,0x6126001d,0x6127001d,0x6128001d,
	0x6129001d,0x612a001d,0x612b001d,0x612c001d,0x612d001d,0x612e001d,0x612f001d,0x6130001d,
	0x6131001d,0x6132001d,0x6133001d,0x6134001d,0x6135001d,0x6136001d,0x6137001d,0x6138001d,
	0x6139001d,0x613a001d,0x613b001d,0x613c001d,0x613d001d,0x613e001d,0x613f001d,0x6140001d,
	0x6141001d,0x6142001d,0x6143001d,0x6144001d,0x6145001d,0x6146001d,0x6147001d,0x6148001d,
	0x6149001d,0x614a001d,0x614b001d,0x614c001d,0x614d001d,0x614e001d,0x614f001d,0x6150001d,
	0x6151001d,0x6152001d,0x6153001d,0x6154001d,0x6155001d,0x6156001d,0x6157001d,0x6158001d,
	0x6159001d,0x615a001d,0x615b001d,0x615c001d,0x615d001d,0x615e001d,0x615f001d,0x6160001d,
	0x6161001d,0x6162001d,0x6163001d,0x6164001d,0x6165001d,0x6166001d,0x6167001d,0x6168001d,
	0x6169001d,0x616a001d,0x616b001d,0x616c001d,0x616d001d,0x616e001d,0x616f001d,0x6170001d,
	0x6171001d,0x6172001d,0x6173001d,0x6174001d,0x6175001d,0x6176001d,0x6177001d,0x6178001d,
	0x6179001d,0x617a001d,0x617b001d,0x617c001d,0x617d001d,0x617e001d,0x617f001d,0x6180001d,
	0x6181001d,0x6182001d,0x6183001d,0x6184001d,0x6185001d,0x6186001d,0x6187001d,0x6188001d,
	0x6189001d,0x618a001d,0x618b001d,0x618c001d,0x618d001d,0x618e001d,0x618f001d,0x6190001d,
	0x6191001d,0x6192001d,0x6193001d,0x6194001d,0x6195001d,0x6196001d,0x6197001d,0x6198001d,
	0x6199001d,0x619a001d,0x619b001d,0x619c001d,0x619d001d,0x619e001d,0x619f001d,0x61a0001d,
	0x61a1001d,0x61a2001d,0x61a3001d,0x61a4001d,0x61a5001d,0x61a6001d,0x61a7001d,0x61a8001d,
	0x61a9001d,0x61aa001d,0x61ab001d,0x61ac001d,0x61ad001d,0x61ae001d,0x61af001d,0x61b0001d,
	0x61b1001d,0x61b2001d,0x61b3001d,0x61b4001d,0x61b5001d,0x61b6001d,0x61b7001d,0x61b8001d,
	0x61b9001d,0x61ba001d,0x61bb001d,0x61bc001d,0x61bd001d,0x61be001d,0x61bf001d,0x61c0001d,
	0x61c1001d,0x61c2001d,0x61c3001d,0x61c4001d,0x61c5001d,0x61c6001d,0x61c7001d,0x61c8001d,
	0x61c9001d,0x61ca001d,0x61cb001d,0x61cc001d,0x61cd001d,0x61ce001d,0x61cf001d,0x61d0001d,
	0x61d1001d,0x61d2001d,0x61d3001d,0x61d4001d,0x61d5001d,0x61d6001d,0x61d7001d,0x61d8001d,
	0x61d9001d,0x61da001d,0x61db001d,0x61dc001d,0x61dd001d,0x61de001d,0x61df001d,0x61e0001d,
	0x61e1001d,0x61e2001d,0x61e3001d,0x61e4001d,0x61e5001d,0x61e6001d,0x61e7001d,0x61e8001d,
	0x61e9001d,0x61ea001d,0x61eb001d,0x61ec001d,0x61ed001d,0x61ee001d,0x61ef001d,0x61f0001d,
	0x61f1001d,0x61f2001d,0x61f3001d,0x61f4001d,0x61f5001d,0x61f6001d,0x61f7001d,0x61f8001d,
	0x61f9001d,0x61fa001d,0x61fb001d,0x61fc001d,0x61fd001d,0x61fe001d,0x61ff001d,0x6200001d,
	0x6201001d,0x6202001d,0x6203001d,0x6204001d,0x6205001d,0x6206001d,0x6207001d,0x6208001d,
	0x6209001d,0x620a001d,0x620b001d,0x620c001d,0x620d001d,0x620e001d,0x620f001d,0x6210001d,
	0x6211001d,0x6212001d,0x6213001d,0x6214001d,0x6215001d,0x6216001d,0x6217001d,0x6218001d,
	0x6219001d,0x621a001d,0x621b001d,0x621c001d,0x621d001d,0x621e001d,0x621f001d,0x6220001d,
	0x6221001d,0x6222001d,0x6223001d,0x6224001d,0x6225001d,0x6226001d,0x6227001d,0x6228001d,
	0x6229001d,0x622a001d,0x622b001d,0x622c001d,0x622d001d,0x622e001d,0x622f001d,0x6230001d,
	0x6231001d,0x6232001d,0x6233001d,0x6234001d,0x6235001d,0x6236001d,0x6237001d,0x6238001d,
	0x6239001d,0x623a001d,0x623b001d,0x623c001d,0x623d001d,0x623e001d,0x623f001d,0x6240001d,
	0x6241001d,0x6242001d,0x6243001d,0x6244001d,0x6245001d,0x6246001d,0x6247001d,0x6248001d,
	0x6249001d,0x624a001d,0x624b001d,0x624c001d,0x624d001d,0x624e001d,0x624f001d,0x6250001d,
	0x6251001d,0x6252001d,0x6253001d,0x6254001d,0x6255001d,0x6256001d,0x6257001d,0x6258001d,
	0x6259001d,0x625a001d,0x625b001d,0x625c001d,0x625d001d,0x625e001d,0x625f001d,0x6260001d,
	0x6261001d,0x6262001d,0x6263001d,0x6264001d,0x6265001d,0x6266001d,0x6267001d,0x6268001d,
	0x6269001d,0x626a001d,0x626b001d,0x626c001d,0x626d001d,0x626e001d,0x626f001d,0x6270001d,
	0x6271001d,0x6272001d,0x6273001d,0x6274001d,0x6275001d,0x6276001d,0x6277001d,0x6278001d,
	0x6279001d,0x627a001d,0x627b001d,0x627c001d,0x627d001d,0x627e001d,0x627f001d,0x6280001d,
	0x6281001d,0x6282001d,0x6283001d,0x6284001d,0x6285001d,0x6286001d,0x6287001d,0x6288001d,
	0x6289001d,0x628a001d,0x628b001d,0x628c001d,0x628d001d,0x628e001d,0x628f001d,0x6290001d,
	0x6291001d,0x6292001d,0x6293001d,0x6294001d,0x6295001d,0x6296001d,0x6297001d,0x6298001d,
	0x6299001d,0x629a001d,0x629b001d,0x629c001d,0x629d001d,0x629e001d,0x629f001d,0x62a0001d,
	0x62a1001d,0x62a2001d,0x62a3001d,0x62a4001d,0x62a5001d,0x62a6001d,0x62a7001d,0x62a8001d,
	0x62a9001d,0x62aa001d,0x62ab001d,0x62ac001d,0x62ad001d,0x62ae001d,0x62af001d,0x62b0001d,
	0x62b1001d,0x62b2001d,0x62b3001d,0x62b4001d,0x62b5001d,0x62b6001d,0x62b7001d,0x62b8001d,
	0x62b9001d,0x62ba001d,0x62bb001d,0x62bc001d,0x62bd001d,0x62be001d,0x62bf001d,0x62c0001d,
	0x62c1001d,0x62c2001d,0x62c3001d,0x62c4001d,0x62c5001d,0x62c6001d,0x62c7001d,0x62c8001d,
	0x62c9001d,0x62ca001d,0x62cb001d,0x62cc001d,0x62cd001d,0x62ce001d,0x62cf001d,0x62d0001d,
	0x62d1001d,0x62d2001d,0x62d3001d,0x62d4001d,0x62d5001d,0x62d6001d,0x62d7001d,0x62d8001d,
	0x62d9001d,0x62da001d,0x62db001d,0x62dc001d,0x62dd001d,0x62de001d,0x62df001d,0x62e0001d,
	0x62e1001d,0x62e2001d,0x62e3001d,0x62e4001d,0x62e5001d,0x62e6001d,0x62e7001d,0x62e8001d,
	0x62e9001d,0x62ea001d,0x62eb001d,0x62ec001d,0x62ed001d,0x62ee001d,0x62ef001d,0x62f0001d,
	0x62f1001d,0x62f2001d,0x62f3001d,0x62f4001d,0x62f5001d,0x62f6001d,0x62f7001d,0x62f8001d,
	0x62f9001d,0x62fa001d,0x62fb001d,0x62fc001d,0x62fd001d,0x62fe001d,0x62ff001d,0x6300001d,
	0x6301001d,0x6302001d,0x6303001d,0x6304001d,0x6305001d,0x6306001d,0x6307001d,0x6308001d,
	0x6309001d,0x630a001d,0x630b001d,0x630c001d,0x630d001d,0x630e001d,0x630f001d,0x6310001d,
	0x6311001d,0x6312001d,0x6313001d,0x6314001d,0x6315001d,0x6316001d,0x6317001d,0x6318001d,
	0x6319001d,0x631a001d,0x631b001d,0x631c001d,0x631d001d,0x631e001d,0x631f001d,0x6320001d,
	0x6321001d,0x6322001d,0x6323001d,0x6324001d,0x6325001d,0x6326001d,0x6327001d,0x6328001d,
	0x6329001d,0x632a001d,0x632b001d,0x632c001d,0x632d001d,0x632e001d,0x632f001d,0x6330001d,
	0x6331001d,0x6332001d,0x6333001d,0x6334001d,0x6335001d,0x6336001d,0x6337001d,0x6338001d,
	0x6339001d,0x633a001d,0x633b001d,0x633c001d,0x633d001d,0x633e001d,0x633f001d,0x6340001d,
	0x6341001d,0x6342001d,0x6343001d,0x6344001d,0x6345001d,0x6346001d,0x6347001d,0x6348001d,
	0x6349001d,0x634a001d,0x634b001d,0x634c001d,0x634d001d,0x634e001d,0x634f001d,0x6350001d,
	0x6351001d,0x6352001d,0x6353001d,0x6354001d,0x6355001d,0x6356001d,0x6357001d,0x6358001d,
	0x6359001d,0x635a001d,0x635b001d,0x635c001d,0x635d001d,0x635e001d,0x635f001d,0x6360001d,
	0x6361001d,0x6362001d,0x6363001d,0x6364001d,0x6365001d,0x6366001d,0x6367001d,0x6368001d,
	0x6369001d,0x636a001d,0x636b001d,0x636c001d,0x636d001d,0x636e001d,0x636f001d,0x6370001d,
	0x6371001d,0x6372001d,0x6373001d,0x6374001d,0x6375001d,0x6376001d,0x6377001d,0x6378001d,
	0x6379001d,0x637a001d,0x637b001d,0x637c001d,0x637d001d,0x637e001d,0x637f001d,0x6380001d,
	0x6381001d,0x6382001d,0x6383001d,0x6384001d,0x6385001d,0x6386001d,0x6387001d,0x6388001d,
	0x6389001d,0x638a001d,0x638b001d,0x638c001d,0x638d001d,0x638e001d,0x638f001d,0x6390001d,
	0x6391001d,0x6392001d,0x6393001d,0x6394001d,0x6395001d,0x6396001d,0x6397001d,0x6398001d,
	0x6399001d,0x639a001d,0x639b001d,0x639c001d,0x639d001d,0x639e001d,0x639f001d,0x63a0001d,
	0x63a1001d,0x63a2001d,0x63a3001d,0x63a4001d,0x63a5001d,0x63a6001d,0x63a7001d,0x63a8001d,
	0x63a9001d,0x63aa001d,0x63ab001d,0x63ac001d,0x63ad001d,0x63ae001d,0x63af001d,0x63b0001d,
	0x63b1001d,0x63b2001d,0x63b3001d,0x63b4001d,0x63b5001d,0x63b6001d,0x63b7001d,0x63b8001d,
	0x63b9001d,0x63ba001d,0x63bb001d,0x63bc001d,0x63bd001d,0x63be001d,0x63bf001d,0x63c0001d,
	0x63c1001d,0x63c2001d,0x63c3001d,0x63c4001d,0x63c5001d,0x63c6001d,0x63c7001d,0x63c8001d,
	0x63c9001d,0x63ca001d,0x63cb001d,0x63cc001d,0x63cd001d,0x63ce001d,0x63cf001d,0x63d0001d,
	0x63d1001d,0x63d2001d,0x63d3001d,0x63d4001d,0x63d5001d,0x63d6001d,0x63d7001d,0x63d8001d,
	0x63d9001d,0x63da001d,0x63db001d,0x63dc001d,0x63dd001d,0x63de001d,0x63df001d,0x63e0001d,
	0x63e1001d,0x63e2001d,0x63e3001d,0x63e4001d,0x63e5001d,0x63e6001d,0x63e7001d,0x63e8001d,
	0x63e9001d,0x63ea001d,0x63eb001d,0x63ec001d,0x63ed001d,0x63ee001d,0x63ef001d,0x63f0001d,
	0x63f1001d,0x63f2001d,0x63f3001d,0x63f4001d,0x63f5001d,0x63f6001d,0x63f7001d,0x63f8001d,
	0x63f9001d,0x63fa001d,0x63fb001d,0x63fc001d,0x63fd001d,0x63fe001d,0x63ff001d,0x6400001d,
	0x6401001d,0x6402001d,0x6403001d,0x6404001d,0x6405001d,0x6406001d,0x6407001d,0x6408001d,
	0x6409001d,0x640a001d,0x640b001d,0x640c001d,0x640d001d,0x640e001d,0x640f001d,0x6410001d,
	0x6411001d,0x6412001d,0x6413001d,0x6414001d,0x6415001d,0x6416001d,0x6417001d,0x6418001d,
	0x6419001d,0x641a001d,0x641b001d,0x641c001d,0x641d001d,0x641e001d,0x641f001d,0x6420001d,
	0x6421001d,0x6422001d,0x6423001d,0x6424001d,0x6425001d,0x6426001d,0x6427001d,0x6428001d,
	0x6429001d,0x642a001d,0x642b001d,0x642c001d,0x642d001d,0x642e001d,0x642f001d,0x6430001d,
	0x6431001d,0x6432001d,0x6433001d,0x6434001d,0x6435001d,0x6436001d,0x6437001d,0x6438001d,
	0x6439001d,0x643a001d,0x643b001d,0x643c001d,0x643d001d,0x643e001d,0x643f001d,0x6440001d,
	0x6441001d,0x6442001d,0x6443001d,0x6444001d,0x6445001d,0x6446001d,0x6447001d,0x6448001d,
	0x6449001d,0x644a001d,0x644b001d,0x644c001d,0x644d001d,0x644e001d,0x644f001d,0x6450001d,
	0x6451001d,0x6452001d,0x6453001d,0x6454001d,0x6455001d,0x6456001d,0x6457001d,0x6458001d,
	0x6459001d,0x645a001d,0x645b001d,0x645c001d,0x645d001d,0x645e001d,0x645f001d,0x6460001d,
	0x6461001d,0x6462001d,0x6463001d,0x6464001d,0x6465001d,0x6466001d,0x6467001d,0x6468001d,
	0x6469001d,0x646a001d,0x646b001d,0x646c001d,0x646d001d,0x646e001d,0x646f001d,0x6470001d,
	0x6471001d,0x6472001d,0x6473001d,0x6474001d,0x6475001d,0x6476001d,0x6477001d,0x6478001d,
	0x6479001d,0x647a001d,0x647b001d,0x647c001d,0x647d001d,0x647e001d,0x647f001d,0x6480001d,
	0x6481001d,0x6482001d,0x6483001d,0x6484001d,0x6485001d,0x6486001d,0x6487001d,0x6488001d,
	0x6489001d,0x648a001d,0x648b001d,0x648c001d,0x648d001d,0x648e001d,0x648f001d,0x6490001d,
	0x6491001d,0x6492001d,0x6493001d,0x6494001d,0x6495001d,0x6496001d,0x6497001d,0x6498001d,
	0x6499001d,0x649a001d,0x649b001d,0x649c001d,0x649d001d,0x649e001d,0x649f001d,0x64a0001d,
	0x64a1001d,0x64a2001d,0x64a3001d,0x64a4001d,0x64a5001d,0x64a6001d,0x64a7001d,0x64a8001d,
	0x64a9001d,0x64aa001d,0x64ab001d,0x64ac001d,0x64ad001d,0x64ae001d,0x64af001d,0x64b0001d,
	0x64b1001d,0x64b2001d,0x64b3001d,0x64b4001d,0x64b5001d,0x64b6001d,0x64b7001d,0x64b8001d,
	0x64b9001d,0x64ba001d,0x64bb001d,0x64bc001d,0x64bd001d,0x64be001d,0x64bf001d,0x64c0001d,
	0x64c1001d,0x64c2001d,0x64c3001d,0x64c4001d,0x64c5001d,0x64c6001d,0x64c7001d,0x64c8001d,
	0x64c9001d,0x64ca001d,0x64cb001d,0x64cc001d,0x64cd001d,0x64ce001d,0x64cf001d,0x64d0001d,
	0x64d1001d,0x64d2001d,0x64d3001d,0x64d4001d,0x64d5001d,0x64d6001d,0x64d7001d,0x64d8001d,
	0x64d9001d,0x64da001d,0x64db001d,0x64dc001d,0x64dd001d,0x64de001d,0x64df001d,0x64e0001d,
	0x64e1001d,0x64e2001d,0x64e3001d,0x64e4001d,0x64e5001d,0x64e6001d,0x64e7001d,0x64e8001d,
	0x64e9001d,0x64ea001d,0x64eb001d,0x64ec001d,0x64ed001d,0x64ee001d,0x64ef001d,0x64f0001d,
	0x64f1001d,0x64f2001d,0x64f3001d,0x64f4001d,0x64f5001d,0x64f6001d,0x64f7001d,0x64f8001d,
	0x64f9001d,0x64fa001d,0x64fb001d,0x64fc001d,0x64fd001d,0x64fe001d,0x64ff001d,0x6500001d,
	0x6501001d,0x6502001d,0x6503001d,0x6504001d,0x6505001d,0x6506001d,0x6507001d,0x6508001d,
	0x6509001d,0x650a001d,0x650b001d,0x650c001d,0x650d001d,0x650e001d,0x650f001d,0x6510001d,
	0x6511001d,0x6512001d,0x6513001d,0x6514001d,0x6515001d,0x6516001d,0x6517001d,0x6518001d,
	0x6519001d,0x651a001d,0x651b001d,0x651c001d,0x651d001d,0x651e001d,0x651f001d,0x6520001d,
	0x6521001d,0x6522001d,0x6523001d,0x6524001d,0x6525001d,0x6526001d,0x6527001d,0x6528001d,
	0x6529001d,0x652a001d,0x652b001d,0x652c001d,0x652d001d,0x652e001d,0x652f001d,0x6530001d,
	0x6531001d,0x6532001d,0x6533001d,0x6534001d,0x6535001d,0x6536001d,0x6537001d,0x6538001d,
	0x6539001d,0x653a001d,0x653b001d,0x653c001d,0x653d001d,0x653e001d,0x653f001d,0x6540001d,
	0x6541001d,0x6542001d,0x6543001d,0x6544001d,0x6545001d,0x6546001d,0x6547001d,0x6548001d,
	0x6549001d,0x654a001d,0x654b001d,0x654c001d,0x654d001d,0x654e001d,0x654f001d,0x6550001d,
	0x6551001d,0x6552001d,0x6553001d,0x6554001d,0x6555001d,0x6556001d,0x6557001d,0x6558001d,
	0x6559001d,0x655a001d,0x655b001d,0x655c001d,0x655d001d,0x655e001d,0x655f001d,0x6560001d,
	0x6561001d,0x6562001d,0x6563001d,0x6564001d,0x6565001d,0x6566001d,0x6567001d,0x6568001d,
	0x6569001d,0x656a001d,0x656b001d,0x656c001d,0x656d001d,0x656e001d,0x656f001d,0x6570001d,
	0x6571001d,0x6572001d,0x6573001d,0x6574001d,0x6575001d,0x6576001d,0x6577001d,0x6578001d,
	0x6579001d,0x657a001d,0x657b001d,0x657c001d,0x657d001d,0x657e001d,0x657f001d,0x6580001d,
	0x6581001d,0x6582001d,0x6583001d,0x6584001d,0x6585001d,0x6586001d,0x6587001d,0x6588001d,
	0x6589001d,0x658a001d,0x658b001d,0x658c001d,0x658d001d,0x658e001d,0x658f001d,0x6590001d,
	0x6591001d,0x6592001d,0x6593001d,0x6594001d,0x6595001d,0x6596001d,0x6597001d,0x6598001d,
	0x6599001d,0x659a001d,0x659b001d,0x659c001d,0x659d001d,0x659e001d,0x659f001d,0x65a0001d,
	0x65a1001d,0x65a2001d,0x65a3001d,0x65a4001d,0x65a5001d,0x65a6001d,0x65a7001d,0x65a8001d,
	0x65a9001d,0x65aa001d,0x65ab001d,0x65ac001d,0x65ad001d,0x65ae001d,0x65af001d,0x65b0001d,
	0x65b1001d,0x65b2001d,0x65b3001d,0x65b4001d,0x65b5001d,0x65b6001d,0x65b7001d,0x65b8001d,
	0x65b9001d,0x65ba001d,0x65bb001d,0x65bc001d,0x65bd001d,0x65be001d,0x65bf001d,0x65c0001d,
	0x65c1001d,0x65c2001d,0x65c3001d,0x65c4001d,0x65c5001d,0x65c6001d,0x65c7001d,0x65c8001d,
	0x65c9001d,0x65ca001d,0x65cb001d,0x65cc001d,0x65cd001d,0x65ce001d,0x65cf001d,0x65d0001d,
	0x65d1001d,0x65d2001d,0x65d3001d,0x65d4001d,0x65d5001d,0x65d6001d,0x65d7001d,0x65d8001d,
	0x65d9001d,0x65da001d,0x65db001d,0x65dc001d,0x65dd001d,0x65de001d,0x65df001d,0x65e0001d,
	0x65e1001d,0x65e2001d,0x65e3001d,0x65e4001d,0x65e5001d,0x65e6001d,0x65e7001d,0x65e8001d,
	0x65e9001d,0x65ea001d,0x65eb001d,0x65ec001d,0x65ed001d,0x65ee001d,0x65ef001d,0x65f0001d,
	0x65f1001d,0x65f2001d,0x65f3001d,0x65f4001d,0x65f5001d,0x65f6001d,0x65f7001d,0x65f8001d,
	0x65f9001d,0x65fa001d,0x65fb001d,0x65fc001d,0x65fd001d,0x65fe001d,0x65ff001d,0x6600001d,
	0x6601001d,0x6602001d,0x6603001d,0x6604001d,0x6605001d,0x6606001d,0x6607001d,0x6608001d,
	0x6609001d,0x660a001d,0x660b001d,0x660c001d,0x660d001d,0x660e001d,0x660f001d,0x6610001d,
	0x6611001d,0x6612001d,0x6613001d,0x6614001d,0x6615001d,0x6616001d,0x6617001d,0x6618001d,
	0x6619001d,0x661a001d,0x661b001d,0x661c001d,0x661d001d,0x661e001d,0x661f001d,0x6620001d,
	0x6621001d,0x6622001d,0x6623001d,0x6624001d,0x6625001d,0x6626001d,0x6627001d,0x6628001d,
	0x6629001d,0x662a001d,0x662b001d,0x662c001d,0x662d001d,0x662e001d,0x662f001d,0x6630001d,
	0x6631001d,0x6632001d,0x6633001d,0x6634001d,0x6635001d,0x6636001d,0x6637001d,0x6638001d,
	0x6639001d,0x663a001d,0x663b001d,0x663c001d,0x663d001d,0x663e001d,0x663f001d,0x6640001d,
	0x6641001d,0x6642001d,0x6643001d,0x6644001d,0x6645001d,0x6646001d,0x6647001d,0x6648001d,
	0x6649001d,0x664a001d,0x664b001d,0x664c001d,0x664d001d,0x664e001d,0x664f001d,0x6650001d,
	0x6651001d,0x6652001d,0x6653001d,0x6654001d,0x6655001d,0x6656001d,0x6657001d,0x6658001d,
	0x6659001d,0x665a001d,0x665b001d,0x665c001d,0x665d001d,0x665e001d,0x665f001d,0x6660001d,
	0x6661001d,0x6662001d,0x6663001d,0x6664001d,0x6665001d,0x6666001d,0x6667001d,0x6668001d,
	0x6669001d,0x666a001d,0x666b001d,0x666c001d,0x666d001d,0x666e001d,0x666f001d,0x6670001d,
	0x6671001d,0x6672001d,0x6673001d,0x6674001d,0x6675001d,0x6676001d,0x6677001d,0x6678001d,
	0x6679001d,0x667a001d,0x667b001d,0x667c001d,0x667d001d,0x667e001d,0x667f001d,0x6680001d,
	0x6681001d,0x6682001d,0x6683001d,0x6684001d,0x6685001d,0x6686001d,0x6687001d,0x6688001d,
	0x6689001d,0x668a001d,0x668b001d,0x668c001d,0x668d001d,0x668e001d,0x668f001d,0x6690001d,
	0x6691001d,0x6692001d,0x6693001d,0x6694001d,0x6695001d,0x6696001d,0x6697001d,0x6698001d,
	0x6699001d,0x669a001d,0x669b001d,0x669c001d,0x669d001d,0x669e001d,0x669f001d,0x66a0001d,
	0x66a1001d,0x66a2001d,0x66a3001d,0x66a4001d,0x66a5001d,0x66a6001d,0x66a7001d,0x66a8001d,
	0x66a9001d,0x66aa001d,0x66ab001d,0x66ac001d,0x66ad001d,0x66ae001d,0x66af001d,0x66b0001d,
	0x66b1001d,0x66b2001d,0x66b3001d,0x66b4001d,0x66b5001d,0x66b6001d,0x66b7001d,0x66b8001d,
	0x66b9001d,0x66ba001d,0x66bb001d,0x66bc001d,0x66bd001d,0x66be001d,0x66bf001d,0x66c0001d,
	0x66c1001d,0x66c2001d,0x66c3001d,0x66c4001d,0x66c5001d,0x66c6001d,0x66c7001d,0x66c8001d,
	0x66c9001d,0x66ca001d,0x66cb001d,0x66cc001d,0x66cd001d,0x66ce001d,0x66cf001d,0x66d0001d,
	0x66d1001d,0x66d2001d,0x66d3001d,0x66d4001d,0x66d5001d,0x66d6001d,0x66d7001d,0x66d8001d,
	0x66d9001d,0x66da001d,0x66db001d,0x66dc001d,0x66dd001d,0x66de001d,0x66df001d,0x66e0001d,
	0x66e1001d,0x66e2001d,0x66e3001d,0x66e4001d,0x66e5001d,0x66e6001d,0x66e7001d,0x66e8001d,
	0x66e9001d,0x66ea001d,0x66eb001d,0x66ec001d,0x66ed001d,0x66ee001d,0x66ef001d,0x66f0001d,
	0x66f1001d,0x66f2001d,0x66f3001d,0x66f4001d,0x66f5001d,0x66f6001d,0x66f7001d,0x66f8001d,
	0x66f9001d,0x66fa001d,0x66fb001d,0x66fc001d,0x66fd001d,0x66fe001d,0x66ff001d,0x6700001d,
	0x6701001d,0x6702001d,0x6703001d,0x6704001d,0x6705001d,0x6706001d,0x6707001d,0x6708001d,
	0x6709001d,0x670a001d,0x670b001d,0x670c001d,0x670d001d,0x670e001d,0x670f001d,0x6710001d,
	0x6711001d,0x6712001d,0x6713001d,0x6714001d,0x6715001d,0x6716001d,0x6717001d,0x6718001d,
	0x6719001d,0x671a001d,0x671b001d,0x671c001d,0x671d001d,0x671e001d,0x671f001d,0x6720001d,
	0x6721001d,0x6722001d,0x6723001d,0x6724001d,0x6725001d,0x6726001d,0x6727001d,0x6728001d,
	0x6729001d,0x672a001d,0x672b001d,0x672c001d,0x672d001d,0x672e001d,0x672f001d,0x6730001d,
	0x6731001d,0x6732001d,0x6733001d,0x6734001d,0x6735001d,0x6736001d,0x6737001d,0x6738001d,
	0x6739001d,0x673a001d,0x673b001d,0x673c001d,0x673d001d,0x673e001d,0x673f001d,0x6740001d,
	0x6741001d,0x6742001d,0x6743001d,0x6744001d,0x6745001d,0x6746001d,0x6747001d,0x6748001d,
	0x6749001d,0x674a001d,0x674b001d,0x674c001d,0x674d001d,0x674e001d,0x674f001d,0x6750001d,
	0x6751001d,0x6752001d,0x6753001d,0x6754001d,0x6755001d,0x6756001d,0x6757001d,0x6758001d,
	0x6759001d,0x675a001d,0x675b001d,0x675c001d,0x675d001d,0x675e001d,0x675f001d,0x6760001d,
	0x6761001d,0x6762001d,0x6763001d,0x6764001d,0x6765001d,0x6766001d,0x6767001d,0x6768001d,
	0x6769001d,0x676a001d,0x676b001d,0x676c001d,0x676d001d,0x676e001d,0x676f001d,0x6770001d,
	0x6771001d,0x6772001d,0x6773001d,0x6774001d,0x6775001d,0x6776001d,0x6777001d,0x6778001d,
	0x6779001d,0x677a001d,0x677b001d,0x677c001d,0x677d001d,0x677e001d,0x677f001d,0x6780001d,
	0x6781001d,0x6782001d,0x6783001d,0x6784001d,0x6785001d,0x6786001d,0x6787001d,0x6788001d,
	0x6789001d,0x678a001d,0x678b001d,0x678c001d,0x678d001d,0x678e001d,0x678f001d,0x6790001d,
	0x6791001d,0x6792001d,0x6793001d,0x6794001d,0x6795001d,0x6796001d,0x6797001d,0x6798001d,
	0x6799001d,0x679a001d,0x679b001d,0x679c001d,0x679d001d,0x679e001d,0x679f001d,0x67a0001d,
	0x67a1001d,0x67a2001d,0x67a3001d,0x67a4001d,0x67a5001d,0x67a6001d,0x67a7001d,0x67a8001d,
	0x67a9001d,0x67aa001d,0x67ab001d,0x67ac001d,0x67ad001d,0x67ae001d,0x67af001d,0x67b0001d,
	0x67b1001d,0x67b2001d,0x67b3001d,0x67b4001d,0x67b5001d,0x67b6001d,0x67b7001d,0x67b8001d,
	0x67b9001d,0x67ba001d,0x67bb001d,0x67bc001d,0x67bd001d,0x67be001d,0x67bf001d,0x67c0001d,
	0x67c1001d,0x67c2001d,0x67c3001d,0x67c4001d,0x67c5001d,0x67c6001d,0x67c7001d,0x67c8001d,
	0x67c9001d,0x67ca001d,0x67cb001d,0x67cc001d,0x67cd001d,0x67ce001d,0x67cf001d,0x67d0001d,
	0x67d1001d,0x67d2001d,0x67d3001d,0x67d4001d,0x67d5001d,0x67d6001d,0x67d7001d,0x67d8001d,
	0x67d9001d,0x67da001d,0x67db001d,0x67dc001d,0x67dd001d,0x67de001d,0x67df001d,0x67e0001d,
	0x67e1001d,0x67e2001d,0x67e3001d,0x67e4001d,0x67e5001d,0x67e6001d,0x67e7001d,0x67e8001d,
	0x67e9001d,0x67ea001d,0x67eb001d,0x67ec001d,0x67ed001d,0x67ee001d,0x67ef001d,0x67f0001d,
	0x67f1001d,0x67f2001d,0x67f3001d,0x67f4001d,0x67f5001d,0x67f6001d,0x67f7001d,0x67f8001d,
	0x67f9001d,0x67fa001d,0x67fb001d,0x67fc001d,0x67fd001d,0x67fe001d,0x67ff001d,0x6800001d,
	0x6801001d,0x6802001d,0x6803001d,0x6804001d,0x6805001d,0x6806001d,0x6807001d,0x6808001d,
	0x6809001d,0x680a001d,0x680b001d,0x680c001d,0x680d001d,0x680e001d,0x680f001d,0x6810001d,
	0x6811001d,0x6812001d,0x6813001d,0x6814001d,0x6815001d,0x6816001d,0x6817001d,0x6818001d,
	0x6819001d,0x681a001d,0x681b001d,0x681c001d,0x681d001d,0x681e001d,0x681f001d,0x6820001d,
	0x6821001d,0x6822001d,0x6823001d,0x6824001d,0x6825001d,0x6826001d,0x6827001d,0x6828001d,
	0x6829001d,0x682a001d,0x682b001d,0x682c001d,0x682d001d,0x682e001d,0x682f001d,0x6830001d,
	0x6831001d,0x6832001d,0x6833001d,0x6834001d,0x6835001d,0x6836001d,0x6837001d,0x6838001d,
	0x6839001d,0x683a001d,0x683b001d,0x683c001d,0x683d001d,0x683e001d,0x683f001d,0x6840001d,
	0x6841001d,0x6842001d,0x6843001d,0x6844001d,0x6845001d,0x6846001d,0x6847001d,0x6848001d,
	0x6849001d,0x684a001d,0x684b001d,0x684c001d,0x684d001d,0x684e001d,0x684f001d,0x6850001d,
	0x6851001d,0x6852001d,0x6853001d,0x6854001d,0x6855001d,0x6856001d,0x6857001d,0x6858001d,
	0x6859001d,0x685a001d,0x685b001d,0x685c001d,0x685d001d,0x685e001d,0x685f001d,0x6860001d,
	0x6861001d,0x6862001d,0x6863001d,0x6864001d,0x6865001d,0x6866001d,0x6867001d,0x6868001d,
	0x6869001d,0x686a001d,0x686b001d,0x21e010f,0x234010f,0x269010f,0x2c4010d,0x26a010f,
	0x268010f,0x22d010f,0x23e010f,0x23f010f,0x263010f,0x369010f,0x214010f,0x20c010f,
	0x226010f,0x265010f,0x6c5010d,0x6c6010d,0x6c7010d,0x6c8010d,0x6c9010d,0x6ca010d,
	0x6cb010d,0x6cc010d,0x6cd010d,0x6ce010d,0x21c010f,0x21a010f,0x36d010f,0x36e010f,
	0x36f010f,0x221010f,0x262010f,0x6cf0125,0x6e30125,0x6f70125,0x7030125,0x71f0125,
	0x74b0125,0x7530125,0x7730125,0x78d0125,0x7a10125,0x7b10125,0x7bd0125,0x7e50125,
	0x7ed0125,0x8050125,0x81d0125,0x8290125,0x8350125,0x85d0125,0x8750125,0x88d0125,
	0x8a50125,0x8b10125,0x8b90125,0x8bd0125,0x8c90125,0x240010f,0x241010f,0x2091d0f,
	0x209020f,0x209160f,0x6cf010d,0x6e3010d,0x6f7010d,0x703010d,0x71f010d,0x74b010d,
	0x753010d,0x773010d,0x78d010d,0x7a1010d,0x7b1010d,0x7bd010d,0x7e5010d,0x7ed010d,
	0x805010d,0x81d010d,0x829010d,0x835010d,0x85d010d,0x875010d,0x88d010d,0x8a5010d,
	0x8b1010d,0x8b9010d,0x8bd010d,0x8c9010d,0x242010f,0x371010f,0x243010f,0x374010f,
	0x228010b,0x219010b,0x22e010b,0x22f010b,0x235010b,0x236010b,0xfb400108,0xce020101,
	0xfb400108,0xce040101,0xfb400108,0xce050101,0xfb400108,0xce060101,0xfb400108,0xce0f0101,
	0xfb400108,0xce120101,0xfb400108,0xce170101,0xfb400108,0xce1f0101,0xfb400108,0xce200101,
	0xfb400108,0xce210101,0xfb400108,0xce230101,0xfb400108,0xce260101,0xfb400108,0xce290101,
	0xfb400108,0xce2e0101,0xfb400108,0xce2f0101,0xfb400108,0xce310101,0xfb400108,0xce330101,
	0xfb400108,0xce350101,0xfb400108,0xce370101,0xfb400108,0xce3c0101,0xfb400108,0xce400101,
	0xfb400108,0xce410101,0xfb400108,0xce420101,0xfb400108,0xce440101,0xfb400108,0xce460101,
	0xfb400108,0xce4a0101,0xfb400108,0xce510101,0xfb400108,0xce550101,0xfb400108,0xce570101,
	0xfb400108,0xce5a0101,0xfb400108,0xce5b0101,0xfb400108,0xce620101,0xfb400108,0xce630101,
	0xfb400108,0xce640101,0xfb400108,0xce650101,0xfb400108,0xce670101,0xfb400108,0xce680101,
	0xfb400108,0xce6a0101,0xfb400108,0xce6b0101,0xfb400108,0xce6c0101,0xfb400108,0xce6d0101,
	0xfb400108,0xce6e0101,0xfb400108,0xce6f0101,0xfb400108,0xce720101,0xfb400108,0xce740101,
	0xfb400108,0xce750101,0xfb400108,0xce760101,0xfb400108,0xce770101,0xfb400108,0xce780101,
	0xfb400108,0xce790101,0xfb400108,0xce7a0101,0xfb400108,0xce7b0101,0xfb400108,0xce7c0101,
	0xfb400108,0xce7d0101,0xfb400108,0xce7f0101,0xfb400108,0xce800101,0xfb400108,0xce810101,
	0xfb400108,0xce820101,0xfb400108,0xce830101,0xfb400108,0xce840101,0xfb400108,0xce850101,
	0xfb400108,0xce870101,0xfb400108,0xce8a0101,0xfb400108,0xce900101,0xfb400108,0xce960101,
	0xfb400108,0xce970101,0xfb400108,0xce990101,0xfb400108,0xce9c0101,0xfb400108,0xce9d0101,
	0xfb400108,0xce9e0101,0xfb400108,0xcea30101,0xfb400108,0xceaa0101,0xfb400108,0xceaf0101,
	0xfb400108,0xceb00101,0xfb400108,0xceb10101,0xfb400108,0xceb40101,0xfb400108,0xceb60101,
	0xfb400108,0xceb70101,0xfb400108,0xceb80101,0xfb400108,0xceb90101,0xfb400108,0xcebc0101,
	0xfb400108,0xcebd0101,0xfb400108,0xcebe0101,0xfb400108,0xcec80101,0xfb400108,0xcecc0101,
	0xfb400108,0xcecf0101,0xfb400108,0xced00101,0xfb400108,0xced20101,0xfb400108,0xceda0101,
	0xfb400108,0xcedb0101,0xfb400108,0xcedc0101,0xfb400108,0xcee00101,0xfb400108,0xcee20101,
	0xfb400108,0xcee60101,0xfb400108,0xcee70101,0xfb400108,0xcee90101,0xfb400108,0xceed0101,
	0xfb400108,0xceee0101,0xfb400108,0xceef0101,0xfb400108,0xcef10101,0xfb400108,0xcef40101,
	0xfb400108,0xcef80101,0xfb400108,0xcef90101,0xfb400108,0xcefa0101,0xfb400108,0xcefc0101,
	0xfb400108,0xcefe0101,0xfb400108,0xcf000101,0xfb400108,0xcf020101,0xfb400108,0xcf030101,
	0xfb400108,0xcf040101,0xfb400108,0xcf050101,0xfb400108,0xcf060101,0xfb400108,0xcf070101,
	0xfb400108,0xcf080101,0xfb400108,0xcf0b0101,0xfb400108,0xcf0c0101,0xfb400108,0xcf120101,
	0xfb400108,0xcf130101,0xfb400108,0xcf140101,0xfb400108,0xcf150101,0xfb400108,0xcf160101,
	0xfb400108,0xcf1c0101,0xfb400108,0xcf1d0101,0xfb400108,0xcf210101,0xfb400108,0xcf230101,
	0xfb400108,0xcf280101,0xfb400108,0xcf290101,0xfb400108,0xcf2c0101,0xfb400108,0xcf2d0101,
	0xfb400108,0xcf2e0101,0xfb400108,0xcf310101,0xfb400108,0xcf330101,0xfb400108,0xcf350101,
	0xfb400108,0xcf370101,0xfb400108,0xcf390101,0xfb400108,0xcf3b0101,0xfb400108,0xcf3e0101,
	0xfb400108,0xcf3f0101,0xfb400108,0xcf400101,0xfb400108,0xcf410101,0xfb400108,0xcf420101,
	0xfb400108,0xcf440101,0xfb400108,0xcf450101,0xfb400108,0xcf470101,0xfb400108,0xcf480101,
	0xfb400108,0xcf490101,0xfb400108,0xcf4a0101,0xfb400108,0xcf4b0101,0xfb400108,0xcf4c0101,
	0xfb400108,0xcf520101,0xfb400108,0xcf540101,0xfb400108,0xcf560101,0xfb400108,0xcf610101,
	0xfb400108,0xcf620101,0xfb400108,0xcf660101,0xfb400108,0xcf680101,0xfb400108,0xcf6a0101,
	0xfb400108,0xcf6b0101,0xfb400108,0xcf6d0101,0xfb400108,0xcf6e0101,0xfb400108,0xcf710101,
	0xfb400108,0xcf720101,0xfb400108,0xcf750101,0xfb400108,0xcf770101,0xfb400108,0xcf780101,
	0xfb400108,0xcf790101,0xfb400108,0xcf7a0101,0xfb400108,0xcf7d0101,0xfb400108,0xcf800101,
	0xfb400108,0xcf810101,0xfb400108,0xcf820101,0xfb400108,0xcf850101,0xfb400108,0xcf860101,
	0xfb400108,0xcf870101,0xfb400108,0xcf8a0101,0xfb400108,0xcf8c0101,0xfb400108,0xcf8e0101,
	0xfb400108,0xcf900101,0xfb400108,0xcf920101,0xfb400108,0xcf930101,0xfb400108,0xcf950101,
	0xfb400108,0xcf960101,0xfb400108,0xcf980101,0xfb400108,0xcf990101,0xfb400108,0xcf9a0101,
	0xfb400108,0xcf9c0101,0xfb400108,0xcf9e0101,0xfb400108,0xcf9f0101,0xfb400108,0xcfa10101,
	0xfb400108,0xcfa20101,0xfb400108,0xcfa40101,0xfb400108,0xcfab0101,0xfb400108,0xcfad0101,
	0xfb400108,0xcfb00101,0xfb400108,0xcfb10101,0xfb400108,0xcfb20101,0xfb400108,0xcfb30101,
	0xfb400108,0xcfb40101,0xfb400108,0xcfb60101,0xfb400108,0xcfb70101,0xfb400108,0xcfb80101,
	0xfb400108,0xcfb90101,0xfb400108,0xcfba0101,0xfb400108,0xcfbb0101,0xfb400108,0xcfbc0101,
	0xfb400108,0xcfbd0101,0xfb400108,0xcfbe0101,0xfb400108,0xcfc00101,0xfb400108,0xcfc10101,
	0xfb400108,0xcfc20101,0xfb400108,0xcfc60101,0xfb400108,0xcfc70101,0xfb400108,0xcfc80101,
	0xfb400108,0xcfc90101,0xfb400108,0xcfcb0101,0xfb400108,0xcfcc0101,0xfb400108,0xcfcd0101,
	0xfb400108,0xcfd20101,0xfb400108,0xcfd30101,0xfb400108,0xcfd40101,0xfb400108,0xcfd50101,
	0xfb400108,0xcfd60101,0xfb400108,0xcfd90101,0xfb400108,0xcfdb0101,0xfb400108,0xcfe00101,
	0xfb400108,0xcfe20101,0xfb400108,0xcfe40101,0xfb400108,0xcfe50101,0xfb400108,0xcfe70101,
	0xfb400108,0xcfeb0101,0xfb400108,0xcfec0101,0xfb400108,0xcff00101,0xfb400108,0xcff20101,
	0xfb400108,0xcff40101,0xfb400108,0xcff50101,0xfb400108,0xcff60101,0xfb400108,0xcff70101,
	0xfb400108,0xcff90101,0xfb400108,0xcffb0101,0xfb400108,0xcffc0101,0xfb400108,0xcffd0101,
	0xfb400108,0xcfff0101,0xfb400108,0xd0000101,0xfb400108,0xd0010101,0xfb400108,0xd0020101,
	0xfb400108,0xd0030101,0xfb400108,0xd0040101,0xfb400108,0xd0050101,0xfb400108,0xd0060101,
	0xfb400108,0xd0070101,0xfb400108,0xd0080101,0xfb400108,0xd0090101,0xfb400108,0xd00a0101,
	0xfb400108,0xd00b0101,0xfb400108,0xd00e0101,0xfb400108,0xd0100101,0xfb400108,0xd0110101,
	0xfb400108,0xd0130101,0xfb400108,0xd0150101,0xfb400108,0xd0160101,0xfb400108,0xd0170101,
	0xfb400108,0xd01b0101,0xfb400108,0xd01d0101,0xfb400108,0xd01e0101,0xfb400108,0xd0200101,
	0xfb400108,0xd0220101,0xfb400108,0xd0230101,0xfb400108,0xd0240101,0xfb400108,0xd0270101,
	0xfb400108,0xd02b0101,0xfb400108,0xd02f0101,0xfb400108,0xd0300101,0xfb400108,0xd0310101,
	0xfb400108,0xd0320101,0xfb400108,0xd0330101,0xfb400108,0xd0340101,0xfb400108,0xd0350101,
	0xfb400108,0xd0360101,0xfb400108,0xd0370101,0xfb400108,0xd0380101,0xfb400108,0xd0390101,
	0xfb400108,0xd03b0101,0xfb400108,0xd03d0101,0xfb400108,0xd03f0101,0xfb400108,0xd0400101,
	0xfb400108,0xd0410101,0xfb400108,0xd0420101,0xfb400108,0xd0440101,0xfb400108,0xd0450101,
	0xfb400108,0xd0460101,0xfb400108,0xd0490101,0xfb400108,0xd04a0101,0xfb400108,0xd04b0101,
	0xfb400108,0xd04d0101,0xfb400108,0xd0500101,0xfb400108,0xd0510101,0xfb400108,0xd0520101,
	0xfb400108,0xd0530101,0xfb400108,0xd0540101,0xfb400108,0xd0560101,0xfb400108,0xd0570101,
	0xfb400108,0xd0580101,0xfb400108,0xd0590101,0xfb400108,0xd05b0101,0xfb400108,0xd05d0101,
	0xfb400108,0xd05e0101,0xfb400108,0xd05f0101,0xfb400108,0xd0600101,0xfb400108,0xd0610101,
	0xfb400108,0xd0620101,0xfb400108,0xd0630101,0xfb400108,0xd0640101,0xfb400108,0xd0660101,
	0xfb400108,0xd0670101,0xfb400108,0xd0680101,0xfb400108,0xd0690101,0xfb400108,0xd06a0101,
	0xfb400108,0xd06b0101,0xfb400108,0xd06d0101,0xfb400108,0xd06e0101,0xfb400108,0xd06f0101,
	0xfb400108,0xd0700101,0xfb400108,0xd0710101,0xfb400108,0xd0720101,0xfb400108,0xd0730101,
	0xfb400108,0xd0740101,0xfb400108,0xd0750101,0xfb400108,0xd0780101,0xfb400108,0xd0790101,
	0xfb400108,0xd07a0101,0xfb400108,0xd07c0101,0xfb400108,0xd07d0101,0xfb400108,0xd0810101,
	0xfb400108,0xd0820101,0xfb400108,0xd0830101,0xfb400108,0xd0840101,0xfb400108,0xd0860101,
	0xfb400108,0xd0870101,0xfb400108,0xd0890101,0xfb400108,0xd08a0101,0xfb400108,0xd08b0101,
	0xfb400108,0xd08c0101,0xfb400108,0xd08e0101,0xfb400108,0xd08f0101,0xfb400108,0xd0900101,
	0xfb400108,0xd0910101,0xfb400108,0xd0920101,0xfb400108,0xd0930101,0xfb400108,0xd0940101,
	0xfb400108,0xd0950101,0xfb400108,0xd0960101,0xfb400108,0xd0970101,0xfb400108,0xd0980101,
	0xfb400108,0xd0990101,0xfb400108,0xd09a0101,0xfb400108,0xd09b0101,0xfb400108,0xd09c0101,
	0xfb400108,0xd09d0101,0xfb400108,0xd09e0101,0xfb400108,0xd09f0101,0xfb400108,0xd0a00101,
	0xfb400108,0xd0a10101,0xfb400108,0xd0a20101,0xfb400108,0xd0a40101,0xfb400108,0xd0a60101,
	0xfb400108,0xd0aa0101,0xfb400108,0xd0ab0101,0xfb400108,0xd0ad0101,0xfb400108,0xd0ae0101,
	0xfb400108,0xd0af0101,0xfb400108,0xd0b00101,0xfb400108,0xd0b10101,0xfb400108,0xd0b30101,
	0xfb400108,0xd0b40101,0xfb400108,0xd0b50101,0xfb400108,0xd0b60101,0xfb400108,0xd0b70101,
	0xfb400108,0xd0b80101,0xfb400108,0xd0b90101,0xfb400108,0xd0bc0101,0xfb400108,0xd0bd0101,
	0xfb400108,0xd0be0101,0xfb400108,0xd0bf0101,0xfb400108,0xd0c00101,0xfb400108,0xd0c10101,
	0xfb400108,0xd0c20101,0xfb400108,0xd0c30101,0xfb400108,0xd0c40101,0xfb400108,0xd0c50101,
	0xfb400108,0xd0c60101,0xfb400108,0xd0c70101,0xfb400108,0xd0c80101,0xfb400108,0xd0c90101,
	0xfb400108,0xd0ca0101,0xfb400108,0xd0cb0101,0xfb400108,0xd0cc0101,0xfb400108,0xd0cd0101,
	0xfb400108,0xd0ce0101,0xfb400108,0xd0d00101,0xfb400108,0xd0d10101,0xfb400108,0xd0d20101,
	0xfb400108,0xd0d30101,0xfb400108,0xd0d40101,0xfb400108,0xd0d50101,0xfb400108,0xd0d70101,
	0xfb400108,0xd0d80101,0xfb400108,0xd0d90101,0xfb400108,0xd0db0101,0xfb400108,0xd0dc0101,
	0xfb400108,0xd0dd0101,0xfb400108,0xd0de0101,0xfb400108,0xd0df0101,0xfb400108,0xd0e00101,
	0xfb400108,0xd0e10101,0xfb400108,0xd0e20101,0xfb400108,0xd0e30101,0xfb400108,0xd0e40101,
	0xfb400108,0xd0e50101,0xfb400108,0xd0e80101,0xfb400108,0xd0e90101,0xfb400108,0xd0ea0101,
	0xfb400108,0xd0eb0101,0xfb400108,0xd0ef0101,0xfb400108,0xd0f00101,0xfb400108,0xd0f10101,
	0xfb400108,0xd0f20101,0xfb400108,0xd0f40101,0xfb400108,0xd0f60101,0xfb400108,0xd0f70101,
	0xfb400108,0xd0f80101,0xfb400108,0xd0f90101,0xfb400108,0xd0fa0101,0xfb400108,0xd0fc0101,
	0xfb400108,0xd0fd0101,0xfb400108,0xd0fe0101,0xfb400108,0xd0ff0101,0xfb400108,0xd1000101,
	0xfb400108,0xd1010101,0xfb400108,0xd1020101,0xfb400108,0xd1030101,0xfb400108,0xd1040101,
	0xfb400108,0xd1050101,0xfb400108,0xd1080101,0xfb400108,0xd1090101,0xfb400108,0xd10a0101,
	0xfb400108,0xd10c0101,0xfb400108,0xd10d0101,0xfb400108,0xd10e0101,0xfb400108,0xd10f0101,
	0xfb400108,0xd1100101,0xfb400108,0xd1110101,0xfb400108,0xd1130101,0xfb400108,0xd1140101,
	0xfb400108,0xd1150101,0xfb400108,0xd1160101,0xfb400108,0xd1170101,0xfb400108,0xd1180101,
	0xfb400108,0xd1190101,0xfb400108,0xd11a0101,0xfb400108,0xd11b0101,0xfb400108,0xd11c0101,
	0xfb400108,0xd11d0101,0xfb400108,0xd11e0101,0xfb400108,0xd11f0101,0xfb400108,0xd1200101,
	0xfb400108,0xd1220101,0xfb400108,0xd1230101,0xfb400108,0xd1240101,0xfb400108,0xd1250101,
	0xfb400108,0xd1260101,0xfb400108,0xd1270101,0xfb400108,0xd1280101,0xfb400108,0xd1290101,
	0xfb400108,0xd12a0101,0xfb400108,0xd12b0101,0xfb400108,0xd12c0101,0xfb400108,0xd12d0101,
	0xfb400108,0xd12e0101,0xfb400108,0xd12f0101,0xfb400108,0xd1300101,0xfb400108,0xd1310101,
	0xfb400108,0xd1320101,0xfb400108,0xd1330101,0xfb400108,0xd1340101,0xfb400108,0xd1350101,
	0xfb400108,0xd1360101,0xfb400108,0xd1370101,0xfb400108,0xd1380101,0xfb400108,0xd1390101,
	0xfb400108,0xd13a0101,0xfb400108,0xd13b0101,0xfb400108,0xd13c0101,0xfb400108,0xd13d0101,
	0xfb400108,0xd13e0101,0xfb400108,0xd1420101,0xfb400108,0xd1470101,0xfb400108,0xd14a0101,
	0xfb400108,0xd14c0101,0xfb400108,0xd14e0101,0xfb400108,0xd14f0101,0xfb400108,0xd1500101,
	0xfb400108,0xd1520101,0xfb400108,0xd1530101,0xfb400108,0xd1570101,0xfb400108,0xd1580101,
	0xfb400108,0xd1590101,0xfb400108,0xd15b0101,0xfb400108,0xd15d0101,0xfb400108,0xd15e0101,
	0xfb400108,0xd15f0101,0xfb400108,0xd1600101,0xfb400108,0xd1610101,0xfb400108,0xd1630101,
	0xfb400108,0xd1640101,0xfb400108,0xd1660101,0xfb400108,0xd1670101,0xfb400108,0xd1690101,
	0xfb400108,0xd16a0101,0xfb400108,0xd16f0101,0xfb400108,0xd1720101,0xfb400108,0xd17a0101,
	0xfb400108,0xd17e0101,0xfb400108,0xd17f0101,0xfb400108,0xd1830101,0xfb400108,0xd1840101,
	0xfb400108,0xd1860101,0xfb400108,0xd1870101,0xfb400108,0xd18a0101,0xfb400108,0xd18b0101,
	0xfb400108,0xd18e0101,0xfb400108,0xd18f0101,0xfb400108,0xd1900101,0xfb400108,0xd1910101,
	0xfb400108,0xd1930101,0xfb400108,0xd1940101,0xfb400108,0xd1980101,0xfb400108,0xd19a0101,
	0xfb400108,0xd19d0101,0xfb400108,0xd19e0101,0xfb400108,0xd19f0101,0xfb400108,0xd1a10101,
	0xfb400108,0xd1a30101,0xfb400108,0xd1a60101,0xfb400108,0xd1a70101,0xfb400108,0xd1a80101,
	0xfb400108,0xd1a90101,0xfb400108,0xd1aa0101,0xfb400108,0xd1ad0101,0xfb400108,0xd1ae0101,
	0xfb400108,0xd1b40101,0xfb400108,0xd1b80101,0xfb400108,0xd1b90101,0xfb400108,0xd1ba0101,
	0xfb400108,0xd1be0101,0xfb400108,0xd1bf0101,0xfb400108,0xd1c10101,0xfb400108,0xd1c20101,
	0xfb400108,0xd1c30101,0xfb400108,0xd1c50101,0xfb400108,0xd1c80101,0xfb400108,0xd1ca0101,
	0xfb400108,0xd1cd0101,0xfb400108,0xd1ce0101,0xfb400108,0xd1d00101,0xfb400108,0xd1d20101,
	0xfb400108,0xd1d30101,0xfb400108,0xd1d40101,0xfb400108,0xd1d50101,0xfb400108,0xd1d60101,
	0xfb400108,0xd1d70101,0xfb400108,0xd1d80101,0xfb400108,0xd1d90101,0xfb400108,0xd1da0101,
	0xfb400108,0xd1dc0101,0xfb400108,0xd1de0101,0xfb400108,0xd1df0101,0xfb400108,0xd1e20101,
	0xfb400108,0xd1e30101,0xfb400108,0xd1e50101,0xfb400108,0xd1e60101,0xfb400108,0xd1e70101,
	0xfb400108,0xd1e80101,0xfb400108,0xd1e90101,0xfb400108,0xd1ea0101,0xfb400108,0xd1ec0101,
	0xfb400108,0xd1ee0101,0xfb400108,0xd1f10101,0xfb400108,0xd1f20101,0xfb400108,0xd1f40101,
	0xfb400108,0xd1f70101,0xfb400108,0xd1fe0101,0xfb400108,0xd2040101,0xfb400108,0xd2050101,
	0xfb400108,0xd2090101,0xfb400108,0xd20b0101,0xfb400108,0xd20c0101,0xfb400108,0xd20f0101,
	0xfb400108,0xd2100101,0xfb400108,0xd2130101,0xfb400108,0xd2140101,0xfb400108,0xd2150101,
	0xfb400108,0xd21c0101,0xfb400108,0xd21e0101,0xfb400108,0xd21f0101,0xfb400108,0xd2210101,
	0xfb400108,0xd2220101,0xfb400108,0xd2230101,0xfb400108,0xd2250101,0xfb400108,0xd2260101,
	0xfb400108,0xd2270101,0xfb400108,0xd22a0101,0xfb400108,0xd22c0101,0xfb400108,0xd22f0101,
	0xfb400108,0xd2310101,0xfb400108,0xd2320101,0xfb400108,0xd2340101,0xfb400108,0xd2350101,
	0xfb400108,0xd23c0101,0xfb400108,0xd23e0101,0xfb400108,0xd2440101,0xfb400108,0xd2450101,
	0xfb400108,0xd2460101,0xfb400108,0xd2470101,0xfb400108,0xd2480101,0xfb400108,0xd2490101,
	0xfb400108,0xd24b0101,0xfb400108,0xd24e0101,0xfb400108,0xd24f0101,0xfb400108,0xd2520101,
	0xfb400108,0xd2530101,0xfb400108,0xd2550101,0xfb400108,0xd2570101,0xfb400108,0xd2580101,
	0xfb400108,0xd2590101,0xfb400108,0xd25a0101,0xfb400108,0xd25b0101,0xfb400108,0xd25d0101,
	0xfb400108,0xd25f0101,0xfb400108,0xd2600101,0xfb400108,0xd2620101,0xfb400108,0xd2630101,
	0xfb400108,0xd2640101,0xfb400108,0xd2660101,0xfb400108,0xd2680101,0xfb400108,0xd26b0101,
	0xfb400108,0xd26c0101,0xfb400108,0xd26d0101,0xfb400108,0xd26e0101,0xfb400108,0xd2700101,
	0xfb400108,0xd2710101,0xfb400108,0xd2730101,0xfb400108,0xd2740101,0xfb400108,0xd2750101,
	0xfb400108,0xd2760101,0xfb400108,0xd2770101,0xfb400108,0xd2780101,0xfb400108,0xd2790101,
	0xfb400108,0xd27a0101,0xfb400108,0xd27b0101,0xfb400108,0xd27c0101,0xfb400108,0xd27e0101,
	0xfb400108,0xd2800101,0xfb400108,0xd2830101,0xfb400108,0xd2840101,0xfb400108,0xd2850101,
	0xfb400108,0xd2860101,0xfb400108,0xd2870101,0xfb400108,0xd2890101,0xfb400108,0xd28a0101,
	0xfb400108,0xd28b0101,0xfb400108,0xd28c0101,0xfb400108,0xd28d0101,0xfb400108,0xd28e0101,
	0xfb400108,0xd28f0101,0xfb400108,0xd2910101,0xfb400108,0xd2920101,0xfb400108,0xd2940101,
	0xfb400108,0xd2950101,0xfb400108,0xd2960101,0xfb400108,0xd2970101,0xfb400108,0xd2980101,
	0xfb400108,0xd2990101,0xfb400108,0xd29a0101,0xfb400108,0xd29c0101,0xfb400108,0xd2a40101,
	0xfb400108,0xd2a50101,0xfb400108,0xd2a60101,0xfb400108,0xd2a70101,0xfb400108,0xd2ae0101,
	0xfb400108,0xd2af0101,0xfb400108,0xd2b00101,0xfb400108,0xd2b40101,0xfb400108,0xd2b50101,
	0xfb400108,0xd2b60101,0xfb400108,0xd2b70101,0xfb400108,0xd2b80101,0xfb400108,0xd2b90101,
	0xfb400108,0xd2ba0101,0xfb400108,0xd2bb0101,0xfb400108,0xd2bc0101,0xfb400108,0xd2bd0101,
	0xfb400108,0xd2c00101,0xfb400108,0xd2c10101,0xfb400108,0xd2c20101,0xfb400108,0xd2c40101,
	0xfb400108,0xd2c50101,0xfb400108,0xd2c60101,0xfb400108,0xd2c80101,0xfb400108,0xd2ca0101,
	0xfb400108,0xd2cc0101,0xfb400108,0xd2cd0101,0xfb400108,0xd2ce0101,0xfb400108,0xd2cf0101,
	0xfb400108,0xd2d10101,0xfb400108,0xd2d30101,0xfb400108,0xd2d40101,0xfb400108,0xd2d50101,
	0xfb400108,0xd2d70101,0xfb400108,0xd2d90101,0xfb400108,0xd2da0101,0xfb400108,0xd2db0101,
	0xfb400108,0xd2dc0101,0xfb400108,0xd2dd0101,0xfb400108,0xd2de0101,0xfb400108,0xd2e00101,
	0xfb400108,0xd2e10101,0xfb400108,0xd2e20101,0xfb400108,0xd2e30101,0xfb400108,0xd2e50101,
	0xfb400108,0xd2e60101,0xfb400108,0xd2e70101,0xfb400108,0xd2e80101,0xfb400108,0xd2e90101,
	0xfb400108,0xd2ea0101,0xfb400108,0xd2eb0101,0xfb400108,0xd2ec0101,0xfb400108,0xd2ed0101,
	0xfb400108,0xd2ee0101,0xfb400108,0xd2ef0101,0xfb400108,0xd2f10101,0xfb400108,0xd2f20101,
	0xfb400108,0xd2f30101,0xfb400108,0xd2f40101,0xfb400108,0xd2f50101,0xfb400108,0xd2f60101,
	0xfb400108,0xd2f70101,0xfb400108,0xd2f80101,0xfb400108,0xd2fb0101,0xfb400108,0xd2fc0101,
	0xfb400108,0xd2fd0101,0xfb400108,0xd3010101,0xfb400108,0xd3020101,0xfb400108,0xd3030101,
	0xfb400108,0xd3040101,0xfb400108,0xd3070101,0xfb400108,0xd3090101,0xfb400108,0xd30a0101,
	0xfb400108,0xd30b0101,0xfb400108,0xd30c0101,0xfb400108,0xd30e0101,0xfb400108,0xd3110101,
	0xfb400108,0xd3120101,0xfb400108,0xd3130101,0xfb400108,0xd3140101,0xfb400108,0xd3180101,
	0xfb400108,0xd31b0101,0xfb400108,0xd31c0101,0xfb400108,0xd31e0101,0xfb400108,0xd31f0101,
	0xfb400108,0xd3220101,0xfb400108,0xd3240101,0xfb400108,0xd3250101,0xfb400108,0xd3270101,
	0xfb400108,0xd3280101,0xfb400108,0xd3290101,0xfb400108,0xd32b0101,0xfb400108,0xd32c0101,
	0xfb400108,0xd32d0101,0xfb400108,0xd32f0101,0xfb400108,0xd3300101,0xfb400108,0xd3310101,
	0xfb400108,0xd3320101,0xfb400108,0xd3330101,0xfb400108,0xd3340101,0xfb400108,0xd3350101,
	0xfb400108,0xd3360101,0xfb400108,0xd3370101,0xfb400108,0xd3380101,0xfb400108,0xd33c0101,
	0xfb400108,0xd33d0101,0xfb400108,0xd3400101,0xfb400108,0xd3420101,0xfb400108,0xd3440101,
	0xfb400108,0xd3460101,0xfb400108,0xd34b0101,0xfb400108,0xd34c0101,0xfb400108,0xd34d0101,
	0xfb400108,0xd3500101,0xfb400108,0xd3540101,0xfb400108,0xd3580101,0xfb400108,0xd3590101,
	0xfb400108,0xd35b0101,0xfb400108,0xd35d0101,0xfb400108,0xd3650101,0xfb400108,0xd3680101,
	0xfb400108,0xd36a0101,0xfb400108,0xd36c0101,0xfb400108,0xd36d0101,0xfb400108,0xd3720101,
	0xfb400108,0xd3760101,0xfb400108,0xd3790101,0xfb400108,0xd37b0101,0xfb400108,0xd37c0101,
	0xfb400108,0xd37d0101,0xfb400108,0xd37e0101,0xfb400108,0xd3800101,0xfb400108,0xd3810101,
	0xfb400108,0xd3830101,0xfb400108,0xd3870101,0xfb400108,0xd3880101,0xfb400108,0xd38a0101,
	0xfb400108,0xd38e0101,0xfb400108,0xd38f0101,0xfb400108,0xd3900101,0xfb400108,0xd3910101,
	0xfb400108,0xd3920101,0xfb400108,0xd3930101,0xfb400108,0xd3940101,0xfb400108,0xd3960101,
	0xfb400108,0xd3970101,0xfb400108,0xd3990101,0xfb400108,0xd39b0101,0xfb400108,0xd39c0101,
	0xfb400108,0xd39e0101,0xfb400108,0xd3a00101,0xfb400108,0xd3a10101,0xfb400108,0xd3a40101,
	0xfb400108,0xd3a70101,0xfb400108,0xd3aa0101,0xfb400108,0xd3ab0101,0xfb400108,0xd3ac0101,
	0xfb400108,0xd3ad0101,0xfb400108,0xd3af0101,0xfb400108,0xd3b00101,0xfb400108,0xd3b10101,
	0xfb400108,0xd3b20101,0xfb400108,0xd3b30101,0xfb400108,0xd3b40101,0xfb400108,0xd3b50101,
	0xfb400108,0xd3b70101,0xfb400108,0xd3b80101,0xfb400108,0xd3b90101,0xfb400108,0xd3ba0101,
	0xfb400108,0xd3bc0101,0xfb400108,0xd3bd0101,0xfb400108,0xd3be0101,0xfb400108,0xd3c00101,
	0xfb400108,0xd3c30101,0xfb400108,0xd3c40101,0xfb400108,0xd3c50101,0xfb400108,0xd3c60101,
	0xfb400108,0xd3c70101,0xfb400108,0xd3ce0101,0xfb400108,0xd3cf0101,0xfb400108,0xd3d00101,
	0xfb400108,0xd3d20101,0xfb400108,0xd3d30101,0xfb400108,0xd3d50101,0xfb400108,0xd3da0101,
	0xfb400108,0xd3dc0101,0xfb400108,0xd3dd0101,0xfb400108,0xd3de0101,0xfb400108,0xd3e10101,
	0xfb400108,0xd3e20101,0xfb400108,0xd3e70101,0xfb400108,0xd3f40101,0xfb400108,0xd3fa0101,
	0xfb400108,0xd3fe0101,0xfb400108,0xd3ff0101,0xfb400108,0xd4000101,0xfb400108,0xd4020101,
	0xfb400108,0xd4050101,0xfb400108,0xd4070101,0xfb400108,0xd40b0101,0xfb400108,0xd4140101,
	0xfb400108,0xd4180101,0xfb400108,0xd4190101,0xfb400108,0xd41a0101,0xfb400108,0xd41c0101,
	0xfb400108,0xd4220101,0xfb400108,0xd4240101,0xfb400108,0xd4250101,0xfb400108,0xd42a0101,
	0xfb400108,0xd4300101,0xfb400108,0xd4330101,0xfb400108,0xd4360101,0xfb400108,0xd4370101,
	0xfb400108,0xd43a0101,0xfb400108,0xd43d0101,0xfb400108,0xd43f0101,0xfb400108,0xd4410101,
	0xfb400108,0xd4420101,0xfb400108,0xd4440101,0xfb400108,0xd4450101,0xfb400108,0xd4470101,
	0xfb400108,0xd4490101,0xfb400108,0xd44c0101,0xfb400108,0xd44d0101,0xfb400108,0xd44e0101,
	0xfb400108,0xd44f0101,0xfb400108,0xd4510101,0xfb400108,0xd45a0101,0xfb400108,0xd45d0101,
	0xfb400108,0xd45e0101,0xfb400108,0xd45f0101,0xfb400108,0xd4600101,0xfb400108,0xd4610101,
	0xfb400108,0xd4630101,0xfb400108,0xd4650101,0xfb400108,0xd4670101,0xfb400108,0xd4690101,
	0xfb400108,0xd46a0101,0xfb400108,0xd46b0101,0xfb400108,0xd46c0101,0xfb400108,0xd46d0101,
	0xfb400108,0xd46e0101,0xfb400108,0xd46f0101,0xfb400108,0xd4700101,0xfb400108,0xd4740101,
	0xfb400108,0xd4790101,0xfb400108,0xd47a0101,0xfb400108,0xd47e0101,0xfb400108,0xd47f0101,
	0xfb400108,0xd4810101,0xfb400108,0xd4830101,0xfb400108,0xd4850101,0xfb400108,0xd4870101,
	0xfb400108,0xd4880101,0xfb400108,0xd4890101,0xfb400108,0xd48a0101,0xfb400108,0xd48d0101,
	0xfb400108,0xd4910101,0xfb400108,0xd4930101,0xfb400108,0xd4970101,0xfb400108,0xd4980101,
	0xfb400108,0xd49c0101,0xfb400108,0xd49e0101,0xfb400108,0xd49f0101,0xfb400108,0xd4a00101,
	0xfb400108,0xd4a10101,0xfb400108,0xd4a20101,0xfb400108,0xd4a50101,0xfb400108,0xd4ae0101,
	0xfb400108,0xd4b00101,0xfb400108,0xd4b20101,0xfb400108,0xd4b50101,0xfb400108,0xd4b60101,
	0xfb400108,0xd4b70101,0xfb400108,0xd4b90101,0xfb400108,0xd4ba0101,0xfb400108,0xd4bc0101,
	0xfb400108,0xd4be0101,0xfb400108,0xd4c30101,0xfb400108,0xd4c50101,0xfb400108,0xd4ca0101,
	0xfb400108,0xd4cb0101,0xfb400108,0xd4d60101,0xfb400108,0xd4d80101,0xfb400108,0xd4db0101,
	0xfb400108,0xd4e00101,0xfb400108,0xd4e10101,0xfb400108,0xd4e20101,0xfb400108,0xd4e30101,
	0xfb400108,0xd4e40101,0xfb400108,0xd4eb0101,0xfb400108,0xd4ec0101,0xfb400108,0xd4ef0101,
	0xfb400108,0xd4f00101,0xfb400108,0xd4f10101,0xfb400108,0xd4f40101,0xfb400108,0xd4f50101,
	0xfb400108,0xd4f60101,0xfb400108,0xd4f70101,0xfb400108,0xd4f80101,0xfb400108,0xd4f90101,
	0xfb400108,0xd4fb0101,0xfb400108,0xd4fe0101,0xfb400108,0xd5000101,0xfb400108,0xd5020101,
	0xfb400108,0xd5030101,0xfb400108,0xd5040101,0xfb400108,0xd5050101,0xfb400108,0xd5080101,
	0xfb400108,0xd50a0101,0xfb400108,0xd50b0101,0xfb400108,0xd50c0101,0xfb400108,0xd50d0101,
	0xfb400108,0xd50e0101,0xfb400108,0xd5120101,0xfb400108,0xd5130101,0xfb400108,0xd5150101,
	0xfb400108,0xd5160101,0xfb400108,0xd5170101,0xfb400108,0xd5180101,0xfb400108,0xd5190101,
	0xfb400108,0xd51a0101,0xfb400108,0xd51c0101,0xfb400108,0xd51d0101,0xfb400108,0xd51e0101,
	0xfb400108,0xd51f0101,0xfb400108,0xd5210101,0xfb400108,0xd5250101,0xfb400108,0xd5260101,
	0xfb400108,0xd5280101,0xfb400108,0xd5290101,0xfb400108,0xd52b0101,0xfb400108,0xd52d0101,
	0xfb400108,0xd5320101,0xfb400108,0xd5340101,0xfb400108,0xd5350101,0xfb400108,0xd5360101,
	0xfb400108,0xd5380101,0xfb400108,0xd5390101,0xfb400108,0xd53a0101,0xfb400108,0xd53b0101,
	0xfb400108,0xd53d0101,0xfb400108,0xd5400101,0xfb400108,0xd5420101,0xfb400108,0xd5450101,
	0xfb400108,0xd5470101,0xfb400108,0xd5480101,0xfb400108,0xd54b0101,0xfb400108,0xd54c0101,
	0xfb400108,0xd54d0101,0xfb400108,0xd54e0101,0xfb400108,0xd54f0101,0xfb400108,0xd5510101,
	0xfb400108,0xd5520101,0xfb400108,0xd5530101,0xfb400108,0xd5540101,0xfb400108,0xd5570101,
	0xfb400108,0xd5580101,0xfb400108,0xd5590101,0xfb400108,0xd55a0101,0xfb400108,0xd55b0101,
	0xfb400108,0xd55d0101,0xfb400108,0xd55e0101,0xfb400108,0xd55f0101,0xfb400108,0xd5600101,
	0xfb400108,0xd5620101,0xfb400108,0xd5630101,0xfb400108,0xd5680101,0xfb400108,0xd5690101,
	0xfb400108,0xd56b0101,0xfb400108,0xd56f0101,0xfb400108,0xd5700101,0xfb400108,0xd5710101,
	0xfb400108,0xd5720101,0xfb400108,0xd5730101,0xfb400108,0xd5740101,0xfb400108,0xd5790101,
	0xfb400108,0xd57a0101,0xfb400108,0xd57d0101,0xfb400108,0xd57f0101,0xfb400108,0xd5850101,
	0xfb400108,0xd5860101,0xfb400108,0xd58c0101,0xfb400108,0xd58d0101,0xfb400108,0xd58e0101,
	0xfb400108,0xd5900101,0xfb400108,0xd5920101,0xfb400108,0xd5930101,0xfb400108,0xd5950101,
	0xfb400108,0xd5960101,0xfb400108,0xd5970101,0xfb400108,0xd59a0101,0xfb400108,0xd59b0101,
	0xfb400108,0xd59e0101,0xfb400108,0xd5a00101,0xfb400108,0xd5a10101,0xfb400108,0xd5a20101,
	0xfb400108,0xd5a30101,0xfb400108,0xd5a40101,0xfb400108,0xd5a50101,0xfb400108,0xd5a60101,
	0xfb400108,0xd5a80101,0xfb400108,0xd5a90101,0xfb400108,0xd5aa0101,0xfb400108,0xd5ab0101,
	0xfb400108,0xd5ac0101,0xfb400108,0xd5ad0101,0xfb400108,0xd5ae0101,0xfb400108,0xd5af0101,
	0xfb400108,0xd5b00101,0xfb400108,0xd5b20101,0xfb400108,0xd5b40101,0xfb400108,0xd5b60101,
	0xfb400108,0xd5b80101,0xfb400108,0xd5ba0101,0xfb400108,0xd5bc0101,0xfb400108,0xd5bf0101,
	0xfb400108,0xd5c00101,0xfb400108,0xd5c10101,0xfb400108,0xd5c20101,0xfb400108,0xd5c30101,
	0xfb400108,0xd5c60101,0xfb400108,0xd5c70101,0xfb400108,0xd5c80101,0xfb400108,0xd5ca0101,
	0xfb400108,0xd5cb0101,0xfb400108,0xd5ce0101,0xfb400108,0xd5cf0101,0xfb400108,0xd5d00101,
	0xfb400108,0xd5d50101,0xfb400108,0xd5d70101,0xfb400108,0xd5d80101,0xfb400108,0xd5d90101,
	0xfb400108,0xd5da0101,0xfb400108,0xd5db0101,0xfb400108,0xd5de0101,0xfb400108,0xd5e00101,
	0xfb400108,0xd5e20101,0xfb400108,0xd5e70101,0xfb400108,0xd5e90101,0xfb400108,0xd5ed0101,
	0xfb400108,0xd5ee0101,0xfb400108,0xd5f00101,0xfb400108,0xd5f10101,0xfb400108,0xd5f40101,
	0xfb400108,0xd5f60101,0xfb400108,0xd5f80101,0xfb400108,0xd5f90101,0xfb400108,0xd5fa0101,
	0xfb400108,0xd5fb0101,0xfb400108,0xd5fc0101,0xfb400108,0xd5ff0101,0xfb400108,0xd6020101,
	0xfb400108,0xd6030101,0xfb400108,0xd6040101,0xfb400108,0xd6050101,0xfb400108,0xd6060101,
	0xfb400108,0xd6070101,0xfb400108,0xd60a0101,0xfb400108,0xd60b0101,0xfb400108,0xd60d0101,
	0xfb400108,0xd6100101,0xfb400108,0xd6110101,0xfb400108,0xd6120101,0xfb400108,0xd6130101,
	0xfb400108,0xd6140101,0xfb400108,0xd6150101,0xfb400108,0xd6160101,0xfb400108,0xd6170101,
	0xfb400108,0xd6190101,0xfb400108,0xd61a0101,0xfb400108,0xd61c0101,0xfb400108,0xd61d0101,
	0xfb400108,0xd6200101,0xfb400108,0xd6210101,0xfb400108,0xd6220101,0xfb400108,0xd6250101,
	0xfb400108,0xd6260101,0xfb400108,0xd6280101,0xfb400108,0xd6290101,0xfb400108,0xd62a0101,
	0xfb400108,0xd62b0101,0xfb400108,0xd62e0101,0xfb400108,0xd62f0101,0xfb400108,0xd6300101,
	0xfb400108,0xd6330101,0xfb400108,0xd6350101,0xfb400108,0xd6370101,0xfb400108,0xd6380101,
	0xfb400108,0xd63a0101,0xfb400108,0xd63c0101,0xfb400108,0xd63d0101,0xfb400108,0xd63e0101,
	0xfb400108,0xd6400101,0xfb400108,0xd6410101,0xfb400108,0xd6420101,0xfb400108,0xd6430101,
	0xfb400108,0xd6440101,0xfb400108,0xd6450101,0xfb400108,0xd6460101,0xfb400108,0xd6470101,
	0xfb400108,0xd6480101,0xfb400108,0xd6490101,0xfb400108,0xd64a0101,0xfb400108,0xd64b0101,
	0xfb400108,0xd64f0101,0xfb400108,0xd6500101,0xfb400108,0xd6510101,0xfb400108,0xd6520101,
	0xfb400108,0xd6530101,0xfb400108,0xd6550101,0xfb400108,0xd6560101,0xfb400108,0xd65a0101,
	0xfb400108,0xd65b0101,0xfb400108,0xd65d0101,0xfb400108,0xd65e0101,0xfb400108,0xd65f0101,
	0xfb400108,0xd6600101,0xfb400108,0xd6610101,0xfb400108,0xd6630101,0xfb400108,0xd6650101,
	0xfb400108,0xd6660101,0xfb400108,0xd6670101,0xfb400108,0xd66d0101,0xfb400108,0xd66e0101,
	0xfb400108,0xd66f0101,0xfb400108,0xd6700101,0xfb400108,0xd6720101,0xfb400108,0xd6730101,
	0xfb400108,0xd6740101,0xfb400108,0xd6750101,0xfb400108,0xd6770101,0xfb400108,0xd6780101,
	0xfb400108,0xd6790101,0xfb400108,0xd67a0101,0xfb400108,0xd67d0101,0xfb400108,0xd67e0101,
	0xfb400108,0xd67f0101,0xfb400108,0xd6800101,0xfb400108,0xd6810101,0xfb400108,0xd6820101,
	0xfb400108,0xd6830101,0xfb400108,0xd6840101,0xfb400108,0xd6870101,0xfb400108,0xd6880101,
	0xfb400108,0xd6890101,0xfb400108,0xd68a0101,0xfb400108,0xd68b0101,0xfb400108,0xd68c0101,
	0xfb400108,0xd68d0101,0xfb400108,0xd6900101,0xfb400108,0xd6910101,0xfb400108,0xd6920101,
	0xfb400108,0xd6940101,0xfb400108,0xd6950101,0xfb400108,0xd6960101,0xfb400108,0xd6970101,
	0xfb400108,0xd6980101,0xfb400108,0xd6990101,0xfb400108,0xd69a0101,0xfb400108,0xd69b0101,
	0xfb400108,0xd69c0101,0xfb400108,0xd69d0101,0xfb400108,0xd69e0101,0xfb400108,0xd69f0101,
	0xfb400108,0xd6a00101,0xfb400108,0xd6a10101,0xfb400108,0xd6a20101,0xfb400108,0xd6a40101,
	0xfb400108,0xd6a50101,0xfb400108,0xd6a60101,0xfb400108,0xd6a70101,0xfb400108,0xd6a80101,
	0xfb400108,0xd6a90101,0xfb400108,0xd6aa0101,0xfb400108,0xd6ab0101,0xfb400108,0xd6ac0101,
	0xfb400108,0xd6ad0101,0xfb400108,0xd6ae0101,0xfb400108,0xd6b00101,0xfb400108,0xd6b10101,
	0xfb400108,0xd6b20101,0xfb400108,0xd6b30101,0xfb400108,0xd6b40101,0xfb400108,0xd6b50101,
	0xfb400108,0xd6b60101,0xfb400108,0xd6b80101,0xfb400108,0xd6b90101,0xfb400108,0xd6ba0101,
	0xfb400108,0xd6bb0101,0xfb400108,0xd6bd0101,0xfb400108,0xd6be0101,0xfb400108,0xd6bf0101,
	0xfb400108,0xd6c00101,0xfb400108,0xd6c10101,0xfb400108,0xd6c20101,0xfb400108,0xd6c30101,
	0xfb400108,0xd6c40101,0xfb400108,0xd6c50101,0xfb400108,0xd6c60101,0xfb400108,0xd6c70101,
	0xfb400108,0xd6c80101,0xfb400108,0xd6c90101,0xfb400108,0xd6cb0101,0xfb400108,0xd6cc0101,
	0xfb400108,0xd6cd0101,0xfb400108,0xd6ce0101,0xfb400108,0xd6cf0101,0xfb400108,0xd6d00101,
	0xfb400108,0xd6d10101,0xfb400108,0xd6d20101,0xfb400108,0xd6d30101,0xfb400108,0xd6d50101,
	0xfb400108,0xd6d60101,0xfb400108,0xd6d80101,0xfb400108,0xd6d90101,0xfb400108,0xd6dc0101,
	0xfb400108,0xd6e30101,0xfb400108,0xd6e50101,0xfb400108,0xd6e60101,0xfb400108,0xd6e70101,
	0xfb400108,0xd6e80101,0xfb400108,0xd6e90101,0xfb400108,0xd6ea0101,0xfb400108,0xd6ec0101,
	0xfb400108,0xd6ee0101,0xfb400108,0xd6ef0101,0xfb400108,0xd6f20101,0xfb400108,0xd6f30101,
	0xfb400108,0xd6f60101,0xfb400108,0xd6f70101,0xfb400108,0xd6f80101,0xfb400108,0xd6fb0101,
	0xfb400108,0xd6fc0101,0xfb400108,0xd7000101,0xfb400108,0xd7010101,0xfb400108,0xd7020101,
	0xfb400108,0xd7050101,0xfb400108,0xd7070101,0xfb400108,0xd70b0101,0xfb400108,0xd70c0101,
	0xfb400108,0xd70d0101,0xfb400108,0xd70e0101,0xfb400108,0xd70f0101,0xfb400108,0xd7100101,
	0xfb400108,0xd7110101,0xfb400108,0xd7120101,0xfb400108,0xd7130101,0xfb400108,0xd7140101,
	0xfb400108,0xd7150101,0xfb400108,0xd7160101,0xfb400108,0xd7170101,0xfb400108,0xd7180101,
	0xfb400108,0xd7190101,0xfb400108,0xd71a0101,0xfb400108,0xd71b0101,0xfb400108,0xd71d0101,
	0xfb400108,0xd71e0101,0xfb400108,0xd7200101,0xfb400108,0xd7210101,0xfb400108,0xd7220101,
	0xfb400108,0xd7240101,0xfb400108,0xd7250101,0xfb400108,0xd7260101,0xfb400108,0xd7270101,
	0xfb400108,0xd72b0101,0xfb400108,0xd7310101,0xfb400108,0xd7320101,0xfb400108,0xd7340101,
	0xfb400108,0xd7350101,0xfb400108,0xd7360101,0xfb400108,0xd7370101,0xfb400108,0xd7380101,
	0xfb400108,0xd73c0101,0xfb400108,0xd73d0101,0xfb400108,0xd73f0101,0xfb400108,0xd7410101,
	0xfb400108,0xd7430101,0xfb400108,0xd7440101,0xfb400108,0xd7450101,0xfb400108,0xd7460101,
	0xfb400108,0xd7480101,0xfb400108,0xd7490101,0xfb400108,0xd74b0101,0xfb400108,0xd7520101,
	0xfb400108,0xd7530101,0xfb400108,0xd7540101,0xfb400108,0xd7550101,0xfb400108,0xd7560101,
	0xfb400108,0xd7580101,0xfb400108,0xd7590101,0xfb400108,0xd7620101,0xfb400108,0xd7630101,
	0xfb400108,0xd7650101,0xfb400108,0xd7670101,0xfb400108,0xd76c0101,0xfb400108,0xd76e0101,
	0xfb400108,0xd7700101,0xfb400108,0xd7710101,0xfb400108,0xd7720101,0xfb400108,0xd7740101,
	0xfb400108,0xd7750101,0xfb400108,0xd7780101,0xfb400108,0xd7790101,0xfb400108,0xd77a0101,
	0xfb400108,0xd77d0101,0xfb400108,0xd77e0101,0xfb400108,0xd77f0101,0xfb400108,0xd7800101,
	0xfb400108,0xd7810101,0xfb400108,0xd7870101,0xfb400108,0xd7880101,0xfb400108,0xd7890101,
	0xfb400108,0xd78a0101,0xfb400108,0xd78d0101,0xfb400108,0xd78e0101,0xfb400108,0xd78f0101,
	0xfb400108,0xd7900101,0xfb400108,0xd7910101,0xfb400108,0xd7940101,0xfb400108,0xd7950101,
	0xfb400108,0xd7960101,0xfb400108,0xd7970101,0xfb400108,0xd7980101,0xfb400108,0xd7990101,
	0xfb400108,0xd79a0101,0xfb400108,0xd79c0101,0xfb400108,0xd79d0101,0xfb400108,0xd79e0101,
	0xfb400108,0xd79f0101,0xfb400108,0xd7a50101,0xfb400108,0xd7a80101,0xfb400108,0xd7aa0101,
	0xfb400108,0xd7ac0101,0xfb400108,0xd7af0101,0xfb400108,0xd7b00101,0xfb400108,0xd7b10101,
	0xfb400108,0xd7b30101,0xfb400108,0xd7b50101,0xfb400108,0xd7b60101,0xfb400108,0xd7b70101,
	0xfb400108,0xd7b90101,0xfb400108,0xd7ba0101,0xfb400108,0xd7bb0101,0xfb400108,0xd7bc0101,
	0xfb400108,0xd7bd0101,0xfb400108,0xd7be0101,0xfb400108,0xd7bf0101,0xfb400108,0xd7c00101,
	0xfb400108,0xd7c10101,0xfb400108,0xd7c40101,0xfb400108,0xd7c50101,0xfb400108,0xd7c60101,
	0xfb400108,0xd7c70101,0xfb400108,0xd7c80101,0xfb400108,0xd7c90101,0xfb400108,0xd7ca0101,
	0xfb400108,0xd7cc0101,0xfb400108,0xd7cd0101,0xfb400108,0xd7d00101,0xfb400108,0xd7d10101,
	0xfb400108,0xd7d30101,0xfb400108,0xd7d60101,0xfb400108,0xd7d70101,0xfb400108,0xd7db0101,
	0xfb400108,0xd7dc0101,0xfb400108,0xd7de0101,0xfb400108,0xd7e10101,0xfb400108,0xd7e20101,
	0xfb400108,0xd7e30101,0xfb400108,0xd7e50101,0xfb400108,0xd7e60101,0xfb400108,0xd7e70101,
	0xfb400108,0xd7e80101,0xfb400108,0xd7e90101,0xfb400108,0xd7ea0101,0xfb400108,0xd7eb0101,
	0xfb400108,0xd7ec0101,0xfb400108,0xd7ee0101,0xfb400108,0xd7f00101,0xfb400108,0xd7f10101,
	0xfb400108,0xd7f20101,0xfb400108,0xd7f30101,0xfb400108,0xd7f50101,0xfb400108,0xd7f60101,
	0xfb400108,0xd7f70101,0xfb400108,0xd7fb0101,0xfb400108,0xd7fc0101,0xfb400108,0xd7fe0101,
	0xfb400108,0xd7ff0101,0xfb400108,0xd8010101,0xfb400108,0xd8030101,0xfb400108,0xd8040101,
	0xfb400108,0xd8050101,0xfb400108,0xd8080101,0xfb400108,0xd8090101,0xfb400108,0xd80a0101,
	0xfb400108,0xd80c0101,0xfb400108,0xd80e0101,0xfb400108,0xd80f0101,0xfb400108,0xd8100101,
	0xfb400108,0xd8120101,0xfb400108,0xd8130101,0xfb400108,0xd8140101,0xfb400108,0xd8160101,
	0xfb400108,0xd8170101,0xfb400108,0xd8180101,0xfb400108,0xd81a0101,0xfb400108,0xd81b0101,
	0xfb400108,0xd81c0101,0xfb400108,0xd81d0101,0xfb400108,0xd81f0101,0xfb400108,0xd8220101,
	0xfb400108,0xd8230101,0xfb400108,0xd8250101,0xfb400108,0xd8260101,0xfb400108,0xd8270101,
	0xfb400108,0xd8280101,0xfb400108,0xd8290101,0xfb400108,0xd82b0101,0xfb400108,0xd82c0101,
	0xfb400108,0xd82d0101,0xfb400108,0xd82e0101,0xfb400108,0xd82f0101,0xfb400108,0xd8310101,
	0xfb400108,0xd8320101,0xfb400108,0xd8330101,0xfb400108,0xd8340101,0xfb400108,0xd8360101,
	0xfb400108,0xd8370101,0xfb400108,0xd8380101,0xfb400108,0xd8390101,0xfb400108,0xd83a0101,
	0xfb400108,0xd83b0101,0xfb400108,0xd83c0101,0xfb400108,0xd83d0101,0xfb400108,0xd83e0101,
	0xfb400108,0xd83f0101,0xfb400108,0xd8400101,0xfb400108,0xd8410101,0xfb400108,0xd8420101,
	0xfb400108,0xd8430101,0xfb400108,0xd8450101,0xfb400108,0xd8460101,0xfb400108,0xd8470101,
	0xfb400108,0xd8480101,0xfb400108,0xd8490101,0xfb400108,0xd84a0101,0xfb400108,0xd84b0101,
	0xfb400108,0xd84e0101,0xfb400108,0xd84f0101,0xfb400108,0xd8500101,0xfb400108,0xd8520101,
	0xfb400108,0xd8530101,0xfb400108,0xd8550101,0xfb400108,0xd8560101,0xfb400108,0xd8570101,
	0xfb400108,0xd8590101,0xfb400108,0xd85a0101,0xfb400108,0xd85b0101,0xfb400108,0xd85c0101,
	0xfb400108,0xd85d0101,0xfb400108,0xd85f0101,0xfb400108,0xd8600101,0xfb400108,0xd8610101,
	0xfb400108,0xd8620101,0xfb400108,0xd8630101,0xfb400108,0xd8640101,0xfb400108,0xd8660101,
	0xfb400108,0xd8670101,0xfb400108,0xd8680101,0xfb400108,0xd8690101,0xfb400108,0xd86a0101,
	0xfb400108,0xd86d0101,0xfb400108,0xd86e0101,0xfb400108,0xd86f0101,0xfb400108,0xd8700101,
	0xfb400108,0xd8710101,0xfb400108,0xd8720101,0xfb400108,0xd8730101,0xfb400108,0xd8740101,
	0xfb400108,0xd8750101,0xfb400108,0xd8760101,0xfb400108,0xd8770101,0xfb400108,0xd8780101,
	0xfb400108,0xd8790101,0xfb400108,0xd87a0101,0xfb400108,0xd87b0101,0xfb400108,0xd87c0101,
	0xfb400108,0xd87d0101,0xfb400108,0xd87f0101,0xfb400108,0xd8820101,0xfb400108,0xd8840101,
	0xfb400108,0xd8860101,0xfb400108,0xd8870101,0xfb400108,0xd8880101,0xfb400108,0xd88a0101,
	0xfb400108,0xd88b0101,0xfb400108,0xd88c0101,0xfb400108,0xd88d0101,0xfb400108,0xd88e0101,
	0xfb400108,0xd88f0101,0xfb400108,0xd8900101,0xfb400108,0xd8910101,0xfb400108,0xd8940101,
	0xfb400108,0xd8950101,0xfb400108,0xd8960101,0xfb400108,0xd8970101,0xfb400108,0xd8980101,
	0xfb400108,0xd89b0101,0xfb400108,0xd89c0101,0xfb400108,0xd89d0101,0xfb400108,0xd8a00101,
	0xfb400108,0xd8a10101,0xfb400108,0xd8a20101,0xfb400108,0xd8a30101,0xfb400108,0xd8a40101,
	0xfb400108,0xd8a50101,0xfb400108,0xd8a60101,0xfb400108,0xd8a70101,0xfb400108,0xd8aa0101,
	0xfb400108,0xd8ab0101,0xfb400108,0xd8ac0101,0xfb400108,0xd8ad0101,0xfb400108,0xd8ae0101,
	0xfb400108,0xd8af0101,0xfb400108,0xd8b00101,0xfb400108,0xd8b10101,0xfb400108,0xd8b20101,
	0xfb400108,0xd8b30101,0xfb400108,0xd8b40101,0xfb400108,0xd8b50101,0xfb400108,0xd8b60101,
	0xfb400108,0xd8b70101,0xfb400108,0xd8b80101,0xfb400108,0xd8b90101,0xfb400108,0xd8ba0101,
	0xfb400108,0xd8bb0101,0xfb400108,0xd8bd0101,0xfb400108,0xd8be0101,0xfb400108,0xd8bf0101,
	0xfb400108,0xd8c00101,0xfb400108,0xd8c20101,0xfb400108,0xd8c30101,0xfb400108,0xd8c40101,
	0xfb400108,0xd8c60101,0xfb400108,0xd8c70101,0xfb400108,0xd8c80101,0xfb400108,0xd8c90101,
	0xfb400108,0xd8ca0101,0xfb400108,0xd8cb0101,0xfb400108,0xd8cc0101,0xfb400108,0xd8cd0101,
	0xfb400108,0xd8ce0101,0xfb400108,0xd8cf0101,0xfb400108,0xd8d00101,0xfb400108,0xd8d20101,
	0xfb400108,0xd8d30101,0xfb400108,0xd8d40101,0xfb400108,0xd8d60101,0xfb400108,0xd8d70101,
	0xfb400108,0xd8d80101,0xfb400108,0xd8d90101,0xfb400108,0xd8da0101,0xfb400108,0xd8db0101,
	0xfb400108,0xd8dc0101,0xfb400108,0xd8dd0101,0xfb400108,0xd8de0101,0xfb400108,0xd8df0101,
	0xfb400108,0xd8e00101,0xfb400108,0xd8e10101,0xfb400108,0xd8e20101,0xfb400108,0xd8e30101,
	0xfb400108,0xd8e50101,0xfb400108,0xd8e60101,0xfb400108,0xd8e70101,0xfb400108,0xd8e80101,
	0xfb400108,0xd8e90101,0xfb400108,0xd8ea0101,0xfb400108,0xd8ed0101,0xfb400108,0xd8ef0101,
	0xfb400108,0xd8f10101,0xfb400108,0xd8f20101,0xfb400108,0xd8f40101,0xfb400108,0xd8f50101,
	0xfb400108,0xd8f70101,0xfb400108,0xd8f80101,0xfb400108,0xd8fa0101,0xfb400108,0xd8fb0101,
	0xfb400108,0xd8fc0101,0xfb400108,0xd8fd0101,0xfb400108,0xd8fe0101,0xfb400108,0xd8ff0101,
	0xfb400108,0xd9000101,0xfb400108,0xd9010101,0xfb400108,0xd9030101,0xfb400108,0xd9050101,
	0xfb400108,0xd9060101,0xfb400108,0xd9080101,0xfb400108,0xd9090101,0xfb400108,0xd90a0101,
	0xfb400108,0xd90b0101,0xfb400108,0xd90c0101,0xfb400108,0xd90e0101,0xfb400108,0xd9100101,
	0xfb400108,0xd9110101,0xfb400108,0xd9120101,0xfb400108,0xd9130101,0xfb400108,0xd9170101,
	0xfb400108,0xd9180101,0xfb400108,0xd91b0101,0xfb400108,0xd91d0101,0xfb400108,0xd91e0101,
	0xfb400108,0xd9200101,0xfb400108,0xd9210101,0xfb400108,0xd9220101,0xfb400108,0xd9230101,
	0xfb400108,0xd9260101,0xfb400108,0xd9280101,0xfb400108,0xd92c0101,0xfb400108,0xd9300101,
	0xfb400108,0xd9320101,0xfb400108,0xd9330101,0xfb400108,0xd9350101,0xfb400108,0xd9360101,
	0xfb400108,0xd93b0101,0xfb400108,0xd93d0101,0xfb400108,0xd93e0101,0xfb400108,0xd93f0101,
	0xfb400108,0xd9400101,0xfb400108,0xd9430101,0xfb400108,0xd9450101,0xfb400108,0xd9460101,
	0xfb400108,0xd94a0101,0xfb400108,0xd94c0101,0xfb400108,0xd94d0101,0xfb400108,0xd9500101,
	0xfb400108,0xd9520101,0xfb400108,0xd9530101,0xfb400108,0xd9590101,0xfb400108,0xd95b0101,
	0xfb400108,0xd95c0101,0xfb400108,0xd95d0101,0xfb400108,0xd95e0101,0xfb400108,0xd95f0101,
	0xfb400108,0xd9610101,0xfb400108,0xd9630101,0xfb400108,0xd9640101,0xfb400108,0xd9660101,
	0xfb400108,0xd9670101,0xfb400108,0xd9680101,0xfb400108,0xd9690101,0xfb400108,0xd96a0101,
	0xfb400108,0xd96b0101,0xfb400108,0xd96c0101,0xfb400108,0xd96d0101,0xfb400108,0xd96e0101,
	0xfb400108,0xd96f0101,0xfb400108,0xd9700101,0xfb400108,0xd9710101,0xfb400108,0xd9720101,
	0xfb400108,0xd9750101,0xfb400108,0xd9770101,0xfb400108,0xd97a0101,0xfb400108,0xd97b0101,
	0xfb400108,0xd97c0101,0xfb400108,0xd97e0101,0xfb400108,0xd97f0101,0xfb400108,0xd9800101,
	0xfb400108,0xd9850101,0xfb400108,0xd9890101,0xfb400108,0xd98b0101,0xfb400108,0xd98c0101,
	0xfb400108,0xd98e0101,0xfb400108,0xd98f0101,0xfb400108,0xd9900101,0xfb400108,0xd9910101,
	0xfb400108,0xd9940101,0xfb400108,0xd9950101,0xfb400108,0xd9980101,0xfb400108,0xd99a0101,
	0xfb400108,0xd99b0101,0xfb400108,0xd99c0101,0xfb400108,0xd99d0101,0xfb400108,0xd99f0101,
	0xfb400108,0xd9a00101,0xfb400108,0xd9a10101,0xfb400108,0xd9a20101,0xfb400108,0xd9a60101,
	0xfb400108,0xd9a70101,0xfb400108,0xd9ac0101,0xfb400108,0xd9ad0101,0xfb400108,0xd9b00101,
	0xfb400108,0xd9b10101,0xfb400108,0xd9b30101,0xfb400108,0xd9b40101,0xfb400108,0xd9b50101,
	0xfb400108,0xd9b60101,0xfb400108,0xd9b70101,0xfb400108,0xd9b80101,0xfb400108,0xd9ba0101,
	0xfb400108,0xd9bc0101,0xfb400108,0xd9bd0101,0xfb400108,0xd9bf0101,0xfb400108,0xd9c00101,
	0xfb400108,0xd9c10101,0xfb400108,0xd9c20101,0xfb400108,0xd9c30101,0xfb400108,0xd9c40101,
	0xfb400108,0xd9c50101,0xfb400108,0xd9c70101,0xfb400108,0xd9c80101,0xfb400108,0xd9c90101,
	0xfb400108,0xd9cc0101,0xfb400108,0xd9cd0101,0xfb400108,0xd9ce0101,0xfb400108,0xd9cf0101,
	0xfb400108,0xd9d50101,0xfb400108,0xd9d60101,0xfb400108,0xd9d90101,0xfb400108,0xd9db0101,
	0xfb400108,0xd9de0101,0xfb400108,0xd9df0101,0xfb400108,0xd9e00101,0xfb400108,0xd9e10101,
	0xfb400108,0xd9e20101,0xfb400108,0xd9e40101,0xfb400108,0xd9e60101,0xfb400108,0xd9e70101,
	0xfb400108,0xd9e90101,0xfb400108,0xd9ea0101,0xfb400108,0xd9eb0101,0xfb400108,0xd9ed0101,
	0xfb400108,0xd9ee0101,0xfb400108,0xd9ef0101,0xfb400108,0xd9f00101,0xfb400108,0xd9f10101,
	0xfb400108,0xd9f20101,0xfb400108,0xd9f30101,0xfb400108,0xd9f40101,0xfb400108,0xd9f50101,
	0xfb400108,0xd9f60101,0xfb400108,0xd9f70101,0xfb400108,0xd9f80101,0xfb400108,0xd9fa0101,
	0xfb400108,0xd9fc0101,0xfb400108,0xd9fd0101,0xfb400108,0xd9fe0101,0xfb400108,0xda000101,
	0xfb400108,0xda020101,0xfb400108,0xda0a0101,0xfb400108,0xda0b0101,0xfb400108,0xda0d0101,
	0xfb400108,0xda0e0101,0xfb400108,0xda0f0101,0xfb400108,0xda100101,0xfb400108,0xda120101,
	0xfb400108,0xda140101,0xfb400108,0xda150101,0xfb400108,0xda160101,0xfb400108,0xda170101,
	0xfb400108,0xda190101,0xfb400108,0xda1a0101,0xfb400108,0xda1b0101,0xfb400108,0xda1d0101,
	0xfb400108,0xda1e0101,0xfb400108,0xda210101,0xfb400108,0xda220101,0xfb400108,0xda240101,
	0xfb400108,0xda260101,0xfb400108,0xda270101,0xfb400108,0xda280101,0xfb400108,0xda2a0101,
	0xfb400108,0xda2b0101,0xfb400108,0xda2c0101,0xfb400108,0xda2d0101,0xfb400108,0xda2e0101,
	0xfb400108,0xda2f0101,0xfb400108,0xda300101,0xfb400108,0xda330101,0xfb400108,0xda350101,
	0xfb400108,0xda370101,0xfb400108,0xda380101,0xfb400108,0xda390101,0xfb400108,0xda3a0101,
	0xfb400108,0xda3b0101,0xfb400108,0xda3d0101,0xfb400108,0xda3e0101,0xfb400108,0xda3f0101,
	0xfb400108,0xda410101,0xfb400108,0xda420101,0xfb400108,0xda430101,0xfb400108,0xda440101,
	0xfb400108,0xda450101,0xfb400108,0xda470101,0xfb400108,0xda480101,0xfb400108,0xda4b0101,
	0xfb400108,0xda4c0101,0xfb400108,0xda4d0101,0xfb400108,0xda4e0101,0xfb400108,0xda4f0101,
	0xfb400108,0xda500101,0xfb400108,0xda510101,0xfb400108,0xda520101,0xfb400108,0xda530101,
	0xfb400108,0xda540101,0xfb400108,0xda560101,0xfb400108,0xda570101,0xfb400108,0xda580101,
	0xfb400108,0xda590101,0xfb400108,0xda5b0101,0xfb400108,0xda5c0101,0xfb400108,0xda5d0101,
	0xfb400108,0xda5e0101,0xfb400108,0xda5f0101,0xfb400108,0xda600101,0xfb400108,0xda610101,
	0xfb400108,0xda630101,0xfb400108,0xda640101,0xfb400108,0xda650101,0xfb400108,0xda660101,
	0xfb400108,0xda680101,0xfb400108,0xda690101,0xfb400108,0xda6b0101,0xfb400108,0xda6c0101,
	0xfb400108,0xda6d0101,0xfb400108,0xda6e0101,0xfb400108,0xda6f0101,0xfb400108,0xda700101,
	0xfb400108,0xda710101,0xfb400108,0xda720101,0xfb400108,0xda730101,0xfb400108,0xda780101,
	0xfb400108,0xda790101,0xfb400108,0xda7b0101,0xfb400108,0xda7c0101,0xfb400108,0xda7d0101,
	0xfb400108,0xda7e0101,0xfb400108,0xda800101,0xfb400108,0xda810101,0xfb400108,0xda820101,
	0xfb400108,0xda830101,0xfb400108,0xda840101,0xfb400108,0xda850101,0xfb400108,0xda860101,
	0xfb400108,0xda870101,0xfb400108,0xda880101,0xfb400108,0xda890101,0xfb400108,0xda8a0101,
	0xfb400108,0xda8b0101,0xfb400108,0xda8c0101,0xfb400108,0xda8d0101,0xfb400108,0xda8e0101,
	0xfb400108,0xda8f0101,0xfb400108,0xda900101,0xfb400108,0xda910101,0xfb400108,0xda930101,
	0xfb400108,0xda940101,0xfb400108,0xda950101,0xfb400108,0xda960101,0xfb400108,0xda970101,
	0xfb400108,0xda980101,0xfb400108,0xda990101,0xfb400108,0xda9c0101,0xfb400108,0xda9d0101,
	0xfb400108,0xda9e0101,0xfb400108,0xda9f0101,0xfb400108,0xdaa00101,0xfb400108,0xdaa10101,
	0xfb400108,0xdaa20101,0xfb400108,0xdaa30101,0xfb400108,0xdaa40101,0xfb400108,0xdaa50101,
	0xfb400108,0xdaa60101,0xfb400108,0xdaa70101,0xfb400108,0xdaa80101,0xfb400108,0xdaa90101,
	0xfb400108,0xdaab0101,0xfb400108,0xdaac0101,0xfb400108,0xdaad0101,0xfb400108,0xdaae0101,
	0xfb400108,0xdaaf0101,0xfb400108,0xdab00101,0xfb400108,0xdab10101,0xfb400108,0xdab40101,
	0xfb400108,0xdab60101,0xfb400108,0xdab70101,0xfb400108,0xdab90101,0xfb400108,0xdaba0101,
	0xfb400108,0xdabb0101,0xfb400108,0xdabc0101,0xfb400108,0xdabd0101,0xfb400108,0xdabf0101,
	0xfb400108,0xdac00101,0xfb400108,0xdac30101,0xfb400108,0xdac40101,0xfb400108,0xdac50101,
	0xfb400108,0xdac60101,0xfb400108,0xdac70101,0xfb400108,0xdac80101,0xfb400108,0xdaca0101,
	0xfb400108,0xdacb0101,0xfb400108,0xdacd0101,0xfb400108,0xdace0101,0xfb400108,0xdacf0101,
	0xfb400108,0xdad00101,0xfb400108,0xdad10101,0xfb400108,0xdad30101,0xfb400108,0xdad50101,
	0xfb400108,0xdad70101,0xfb400108,0xdad90101,0xfb400108,0xdada0101,0xfb400108,0xdadb0101,
	0xfb400108,0xdadd0101,0xfb400108,0xdade0101,0xfb400108,0xdadf0101,0xfb400108,0xdae20101,
	0xfb400108,0xdae40101,0xfb400108,0xdae50101,0xfb400108,0xdae70101,0xfb400108,0xdae80101,
	0xfb400108,0xdaea0101,0xfb400108,0xdaec0101,0xfb400108,0xdaed0101,0xfb400108,0xdaee0101,
	0xfb400108,0xdaef0101,0xfb400108,0xdaf00101,0xfb400108,0xdaf20101,0xfb400108,0xdaf30101,
	0xfb400108,0xdaf40101,0xfb400108,0xdaf50101,0xfb400108,0xdaf60101,0xfb400108,0xdaf70101,
	0xfb400108,0xdaf80101,0xfb400108,0xdaf90101,0xfb400108,0xdafa0101,0xfb400108,0xdafb0101,
	0xfb400108,0xdafc0101,0xfb400108,0xdafd0101,0xfb400108,0xdafe0101,0xfb400108,0xdaff0101,
	0xfb400108,0xdb000101,0xfb400108,0xdb010101,0xfb400108,0xdb020101,0xfb400108,0xdb030101,
	0xfb400108,0xdb040101,0xfb400108,0xdb050101,0xfb400108,0xdb060101,0xfb400108,0xdb070101,
	0xfb400108,0xdb080101,0xfb400108,0xdb0a0101,0xfb400108,0xdb0b0101,0xfb400108,0xdb0c0101,
	0xfb400108,0xdb0d0101,0xfb400108,0xdb0e0101,0xfb400108,0xdb0f0101,0xfb400108,0xdb100101,
	0xfb400108,0xdb110101,0xfb400108,0xdb120101,0xfb400108,0xdb130101,0xfb400108,0xdb140101,
	0xfb400108,0xdb150101,0xfb400108,0xdb180101,0xfb400108,0xdb190101,0xfb400108,0xdb1a0101,
	0xfb400108,0xdb1b0101,0xfb400108,0xdb1c0101,0xfb400108,0xdb1d0101,0xfb400108,0xdb1e0101,
	0xfb400108,0xdb1f0101,0xfb400108,0xdb200101,0xfb400108,0xdb210101,0xfb400108,0xdb220101,
	0xfb400108,0xdb230101,0xfb400108,0xdb240101,0xfb400108,0xdb250101,0xfb400108,0xdb260101,
	0xfb400108,0xdb270101,0xfb400108,0xdb280101,0xfb400108,0xdb290101,0xfb400108,0xdb2a0101,
	0xfb400108,0xdb2b0101,0xfb400108,0xdb2c0101,0xfb400108,0xdb2d0101,0xfb400108,0xdb2e0101,
	0xfb400108,0xdb2f0101,0xfb400108,0xdb300101,0xfb400108,0xdb310101,0xfb400108,0xdb330101,
	0xfb400108,0xdb350101,0xfb400108,0xdb360101,0xfb400108,0xdb380101,0xfb400108,0xdb390101,
	0xfb400108,0xdb3a0101,0xfb400108,0xdb3b0101,0xfb400108,0xdb3c0101,0xfb400108,0xdb3d0101,
	0xfb400108,0xdb3e0101,0xfb400108,0xdb3f0101,0xfb400108,0xdb410101,0xfb400108,0xdb420101,
	0xfb400108,0xdb430101,0xfb400108,0xdb440101,0xfb400108,0xdb450101,0xfb400108,0xdb460101,
	0xfb400108,0xdb470101,0xfb400108,0xdb480101,0xfb400108,0xdb490101,0xfb400108,0xdb4a0101,
	0xfb400108,0xdb4b0101,0xfb400108,0xdb4c0101,0xfb400108,0xdb4d0101,0xfb400108,0xdb4e0101,
	0xfb400108,0xdb4f0101,0xfb400108,0xdb520101,0xfb400108,0xdb560101,0xfb400108,0xdb5e0101,
	0xfb400108,0xdb600101,0xfb400108,0xdb610101,0xfb400108,0xdb670101,0xfb400108,0xdb680101,
	0xfb400108,0xdb6b0101,0xfb400108,0xdb6d0101,0xfb400108,0xdb6e0101,0xfb400108,0xdb6f0101,
	0xfb400108,0xdb720101,0xfb400108,0xdb740101,0xfb400108,0xdb760101,0xfb400108,0xdb770101,
	0xfb400108,0xdb780101,0xfb400108,0xdb790101,0xfb400108,0xdb7b0101,0xfb400108,0xdb7c0101,
	0xfb400108,0xdb7e0101,0xfb400108,0xdb7f0101,0xfb400108,0xdb820101,0xfb400108,0xdb860101,
	0xfb400108,0xdb8a0101,0xfb400108,0xdb8d0101,0xfb400108,0xdb8e0101,0xfb400108,0xdb900101,
	0xfb400108,0xdb910101,0xfb400108,0xdb920101,0xfb400108,0xdb940101,0xfb400108,0xdb960101,
	0xfb400108,0xdb9f0101,0xfb400108,0xdba70101,0xfb400108,0xdba80101,0xfb400108,0xdba90101,
	0xfb400108,0xdbac0101,0xfb400108,0xdbad0101,0xfb400108,0xdbae0101,0xfb400108,0xdbaf0101,
	0xfb400108,0xdbb10101,0xfb400108,0xdbb20101,0xfb400108,0xdbb70101,0xfb400108,0xdbba0101,
	0xfb400108,0xdbbb0101,0xfb400108,0xdbbc0101,0xfb400108,0xdbc00101,0xfb400108,0xdbc10101,
	0xfb400108,0xdbc30101,0xfb400108,0xdbc80101,0xfb400108,0xdbc90101,0xfb400108,0xdbca0101,
	0xfb400108,0xdbcb0101,0xfb400108,0xdbcd0101,0xfb400108,0xdbce0101,0xfb400108,0xdbcf0101,
	0xfb400108,0xdbd10101,0xfb400108,0xdbd40101,0xfb400108,0xdbd50101,0xfb400108,0xdbd60101,
	0xfb400108,0xdbd70101,0xfb400108,0xdbd80101,0xfb400108,0xdbd90101,0xfb400108,0xdbda0101,
	0xfb400108,0xdbdb0101,0xfb400108,0xdbdc0101,0xfb400108,0xdbe00101,0xfb400108,0xdbe20101,
	0xfb400108,0xdbe30101,0xfb400108,0xdbe60101,0xfb400108,0xdbe70101,0xfb400108,0xdbe90101,
	0xfb400108,0xdbea0101,0xfb400108,0xdbeb0101,0xfb400108,0xdbec0101,0xfb400108,0xdbed0101,
	0xfb400108,0xdbef0101,0xfb400108,0xdbf10101,0xfb400108,0xdbf20101,0xfb400108,0xdbf30101,
	0xfb400108,0xdbf40101,0xfb400108,0xdbf50101,0xfb400108,0xdbf60101,0xfb400108,0xdbf70101,
	0xfb400108,0xdbfd0101,0xfb400108,0xdbfe0101,0xfb400108,0xdc000101,0xfb400108,0xdc020101,
	0xfb400108,0xdc030101,0xfb400108,0xdc050101,0xfb400108,0xdc070101,0xfb400108,0xdc080101,
	0xfb400108,0xdc0b0101,0xfb400108,0xdc0c0101,0xfb400108,0xdc0d0101,0xfb400108,0xdc0e0101,
	0xfb400108,0xdc100101,0xfb400108,0xdc120101,0xfb400108,0xdc130101,0xfb400108,0xdc170101,
	0xfb400108,0xdc190101,0xfb400108,0xdc1b0101,0xfb400108,0xdc1e0101,0xfb400108,0xdc1f0101,
	0xfb400108,0xdc200101,0xfb400108,0xdc210101,0xfb400108,0xdc230101,0xfb400108,0xdc260101,
	0xfb400108,0xdc280101,0xfb400108,0xdc290101,0xfb400108,0xdc2a0101,0xfb400108,0xdc2b0101,
	0xfb400108,0xdc2d0101,0xfb400108,0xdc2e0101,0xfb400108,0xdc2f0101,0xfb400108,0xdc300101,
	0xfb400108,0xdc320101,0xfb400108,0xdc330101,0xfb400108,0xdc350101,0xfb400108,0xdc360101,
	0xfb400108,0xdc370101,0xfb400108,0xdc430101,0xfb400108,0xdc440101,0xfb400108,0xdc460101,
	0xfb400108,0xdc470101,0xfb400108,0xdc4c0101,0xfb400108,0xdc4d0101,0xfb400108,0xdc520101,
	0xfb400108,0xdc530101,0xfb400108,0xdc540101,0xfb400108,0xdc560101,0xfb400108,0xdc570101,
	0xfb400108,0xdc580101,0xfb400108,0xdc5a0101,0xfb400108,0xdc5b0101,0xfb400108,0xdc5c0101,
	0xfb400108,0xdc5d0101,0xfb400108,0xdc5f0101,0xfb400108,0xdc620101,0xfb400108,0xdc640101,
	0xfb400108,0xdc670101,0xfb400108,0xdc680101,0xfb400108,0xdc690101,0xfb400108,0xdc6a0101,
	0xfb400108,0xdc6b0101,0xfb400108,0xdc6c0101,0xfb400108,0xdc6d0101,0xfb400108,0xdc700101,
	0xfb400108,0xdc720101,0xfb400108,0xdc730101,0xfb400108,0xdc740101,0xfb400108,0xdc750101,
	0xfb400108,0xdc760101,0xfb400108,0xdc770101,0xfb400108,0xdc780101,0xfb400108,0xdc7b0101,
	0xfb400108,0xdc7c0101,0xfb400108,0xdc7d0101,0xfb400108,0xdc7e0101,0xfb400108,0xdc800101,
	0xfb400108,0xdc830101,0xfb400108,0xdc840101,0xfb400108,0xdc850101,0xfb400108,0xdc860101,
	0xfb400108,0xdc870101,0xfb400108,0xdc890101,0xfb400108,0xdc8a0101,0xfb400108,0xdc8b0101,
	0xfb400108,0xdc8e0101,0xfb400108,0xdc8f0101,0xfb400108,0xdc920101,0xfb400108,0xdc930101,
	0xfb400108,0xdc950101,0xfb400108,0xdc9d0101,0xfb400108,0xdc9e0101,0xfb400108,0xdc9f0101,
	0xfb400108,0xdca00101,0xfb400108,0xdca10101,0xfb400108,0xdca40101,0xfb400108,0xdca50101,
	0xfb400108,0xdca60101,0xfb400108,0xdca70101,0xfb400108,0xdca80101,0xfb400108,0xdcaa0101,
	0xfb400108,0xdcae0101,0xfb400108,0xdcaf0101,0xfb400108,0xdcb00101,0xfb400108,0xdcb20101,
	0xfb400108,0xdcb40101,0xfb400108,0xdcb60101,0xfb400108,0xdcb90101,0xfb400108,0xdcba0101,
	0xfb400108,0xdcbb0101,0xfb400108,0xdcbc0101,0xfb400108,0xdcbe0101,0xfb400108,0xdcc00101,
	0xfb400108,0xdcc20101,0xfb400108,0xdcc30101,0xfb400108,0xdcc50101,0xfb400108,0xdcc60101,
	0xfb400108,0xdcc70101,0xfb400108,0xdcc80101,0xfb400108,0xdcc90101,0xfb400108,0xdcca0101,
	0xfb400108,0xdccc0101,0xfb400108,0xdccd0101,0xfb400108,0xdcce0101,0xfb400108,0xdccf0101,
	0xfb400108,0xdcd00101,0xfb400108,0xdcd10101,0xfb400108,0xdcd30101,0xfb400108,0xdcd40101,
	0xfb400108,0xdcd50101,0xfb400108,0xdcd60101,0xfb400108,0xdcd70101,0xfb400108,0xdcd80101,
	0xfb400108,0xdcda0101,0xfb400108,0xdcdb0101,0xfb400108,0xdcdc0101,0xfb400108,0xdcdd0101,
	0xfb400108,0xdcde0101,0xfb400108,0xdcdf0101,0xfb400108,0xdce00101,0xfb400108,0xdce20101,
	0xfb400108,0xdce30101,0xfb400108,0xdce70101,0xfb400108,0xdce90101,0xfb400108,0xdceb0101,
	0xfb400108,0xdcec0101,0xfb400108,0xdcee0101,0xfb400108,0xdcef0101,0xfb400108,0xdcf10101,
	0xfb400108,0xdcf20101,0xfb400108,0xdcf30101,0xfb400108,0xdcf40101,0xfb400108,0xdcf50101,
	0xfb400108,0xdcf60101,0xfb400108,0xdcf70101,0xfb400108,0xdcf80101,0xfb400108,0xdcf90101,
	0xfb400108,0xdcfa0101,0xfb400108,0xdcfc0101,0xfb400108,0xdcfd0101,0xfb400108,0xdcfe0101,
	0xfb400108,0xdcff0101,0xfb400108,0xdd000101,0xfb400108,0xdd010101,0xfb400108,0xdd040101,
	0xfb400108,0xdd050101,0xfb400108,0xdd080101,0xfb400108,0xdd090101,0xfb400108,0xdd0a0101,
	0xfb400108,0xdd0b0101,0xfb400108,0xdd0c0101,0xfb400108,0xdd0d0101,0xfb400108,0xdd0f0101,
	0xfb400108,0xdd100101,0xfb400108,0xdd110101,0xfb400108,0xdd120101,0xfb400108,0xdd130101,
	0xfb400108,0xdd150101,0xfb400108,0xdd170101,0xfb400108,0xdd180101,0xfb400108,0xdd190101,
	0xfb400108,0xdd1a0101,0xfb400108,0xdd1c0101,0xfb400108,0xdd1d0101,0xfb400108,0xdd1f0101,
	0xfb400108,0xdd200101,0xfb400108,0xdd210101,0xfb400108,0xdd220101,0xfb400108,0xdd230101,
	0xfb400108,0xdd250101,0xfb400108,0xdd280101,0xfb400108,0xdd2a0101,0xfb400108,0xdd2b0101,
	0xfb400108,0xdd2c0101,0xfb400108,0xdd2f0101,0xfb400108,0xdd300101,0xfb400108,0xdd310101,
	0xfb400108,0xdd320101,0xfb400108,0xdd330101,0xfb400108,0xdd350101,0xfb400108,0xdd360101,
	0xfb400108,0xdd370101,0xfb400108,0xdd380101,0xfb400108,0xdd390101,0xfb400108,0xdd3a0101,
	0xfb400108,0xdd3b0101,0xfb400108,0xdd3c0101,0xfb400108,0xdd3f0101,0xfb400108,0xdd400101,
	0xfb400108,0xdd410101,0xfb400108,0xdd420101,0xfb400108,0xdd430101,0xfb400108,0xdd440101,
	0xfb400108,0xdd450101,0xfb400108,0xdd460101,0xfb400108,0xdd480101,0xfb400108,0xdd490101,
	0xfb400108,0xdd4d0101,0xfb400108,0xdd4e0101,0xfb400108,0xdd4f0101,0xfb400108,0xdd500101,
	0xfb400108,0xdd510101,0xfb400108,0xdd520101,0xfb400108,0xdd530101,0xfb400108,0xdd540101,
	0xfb400108,0xdd550101,0xfb400108,0xdd560101,0xfb400108,0xdd570101,0xfb400108,0xdd590101,
	0xfb400108,0xdd5a0101,0xfb400108,0xdd5c0101,0xfb400108,0xdd5e0101,0xfb400108,0xdd5f0101,
	0xfb400108,0xdd600101,0xfb400108,0xdd610101,0xfb400108,0xdd620101,0xfb400108,0xdd630101,
	0xfb400108,0xdd640101,0xfb400108,0xdd650101,0xfb400108,0xdd660101,0xfb400108,0xdd670101,
	0xfb400108,0xdd680101,0xfb400108,0xdd6a0101,0xfb400108,0xdd6d0101,0xfb400108,0xdd6e0101,
	0xfb400108,0xdd700101,0xfb400108,0xdd710101,0xfb400108,0xdd720101,0xfb400108,0xdd730101,
	0xfb400108,0xdd750101,0xfb400108,0xdd760101,0xfb400108,0xdd770101,0xfb400108,0xdd780101,
	0xfb400108,0xdd790101,0xfb400108,0xdd7a0101,0xfb400108,0xdd7b0101,0xfb400108,0xdd7c0101,
	0xfb400108,0xdd7d0101,0xfb400108,0xdd7e0101,0xfb400108,0xdd7f0101,0xfb400108,0xdd800101,
	0xfb400108,0xdd810101,0xfb400108,0xdd830101,0xfb400108,0xdd840101,0xfb400108,0xdd850101,
	0xfb400108,0xdd860101,0xfb400108,0xdd870101,0xfb400108,0xdd880101,0xfb400108,0xdd890101,
	0xfb400108,0xdd8a0101,0xfb400108,0xdd8b0101,0xfb400108,0xdd8c0101,0xfb400108,0xdd8d0101,
	0xfb400108,0xdd8e0101,0xfb400108,0xdd8f0101,0xfb400108,0xdd900101,0xfb400108,0xdd910101,
	0xfb400108,0xdd920101,0xfb400108,0xdd930101,0xfb400108,0xdd940101,0xfb400108,0xdd950101,
	0xfb400108,0xdd960101,0xfb400108,0xdd970101,0xfb400108,0xdd980101,0xfb400108,0xdd9a0101,
	0xfb400108,0xdd9b0101,0xfb400108,0xdd9c0101,0xfb400108,0xdd9e0101,0xfb400108,0xdd9f0101,
	0xfb400108,0xdda00101,0xfb400108,0xdda10101,0xfb400108,0xdda20101,0xfb400108,0xdda30101,
	0xfb400108,0xdda40101,0xfb400108,0xdda50101,0xfb400108,0xdda60101,0xfb400108,0xdda70101,
	0xfb400108,0xdda80101,0xfb400108,0xdda90101,0xfb400108,0xddaa0101,0xfb400108,0xddab0101,
	0xfb400108,0xddac0101,0xfb400108,0xddad0101,0xfb400108,0xddae0101,0xfb400108,0xddaf0101,
	0xfb400108,0xddb00101,0xfb400108,0xddb10101,0xfb400108,0xddb20101,0xfb400108,0xddb30101,
	0xfb400108,0xddb40101,0xfb400108,0xddb50101,0xfb400108,0xddb60101,0xfb400108,0xddb80101,
	0xfb400108,0xddb90101,0xfb400108,0xddba0101,0xfb400108,0xddbb0101,0xfb400108,0xddbc0101,
	0xfb400108,0xddbd0101,0xfb400108,0xddbe0101,0xfb400108,0xddbf0101,0xfb400108,0xddc00101,
	0xfb400108,0xddc10101,0xfb400108,0xddc20101,0xfb400108,0xddc30101,0xfb400108,0xddc40101,
	0xfb400108,0xddc60101,0xfb400108,0xddc70101,0xfb400108,0xddc80101,0xfb400108,0xddc90101,
	0xfb400108,0xddca0101,0xfb400108,0xddcb0101,0xfb400108,0xddcc0101,0xfb400108,0xddce0101,
	0xfb400108,0xddcf0101,0xfb400108,0xddd00101,0xfb400108,0xddd10101,0xfb400108,0xddd20101,
	0xfb400108,0xddd30101,0xfb400108,0xddd40101,0xfb400108,0xddd50101,0xfb400108,0xddd60101,
	0xfb400108,0xddd70101,0xfb400108,0xddd80101,0xfb400108,0xddd90101,0xfb400108,0xddda0101,
	0xfb400108,0xdddc0101,0xfb400108,0xdddf0101,0xfb400108,0xdde00101,0xfb400108,0xdde30101,
	0xfb400108,0xdde40101,0xfb400108,0xddea0101,0xfb400108,0xddec0101,0xfb400108,0xdded0101,
	0xfb400108,0xddf00101,0xfb400108,0xddf50101,0xfb400108,0xddf60101,0xfb400108,0xddf80101,
	0xfb400108,0xddf90101,0xfb400108,0xddfa0101,0xfb400108,0xddfb0101,0xfb400108,0xddfc0101,
	0xfb400108,0xddff0101,0xfb400108,0xde000101,0xfb400108,0xde040101,0xfb400108,0xde070101,
	0xfb400108,0xde090101,0xfb400108,0xde0a0101,0xfb400108,0xde0b0101,0xfb400108,0xde0d0101,
	0xfb400108,0xde0e0101,0xfb400108,0xde120101,0xfb400108,0xde130101,0xfb400108,0xde170101,
	0xfb400108,0xde1e0101,0xfb400108,0xde1f0101,0xfb400108,0xde200101,0xfb400108,0xde210101,
	0xfb400108,0xde220101,0xfb400108,0xde230101,0xfb400108,0xde240101,0xfb400108,0xde250101,
	0xfb400108,0xde280101,0xfb400108,0xde290101,0xfb400108,0xde2a0101,0xfb400108,0xde2b0101,
	0xfb400108,0xde2c0101,0xfb400108,0xde2f0101,0xfb400108,0xde300101,0xfb400108,0xde320101,
	0xfb400108,0xde330101,0xfb400108,0xde340101,0xfb400108,0xde350101,0xfb400108,0xde360101,
	0xfb400108,0xde390101,0xfb400108,0xde3a0101,0xfb400108,0xde3e0101,0xfb400108,0xde3f0101,
	0xfb400108,0xde400101,0xfb400108,0xde410101,0xfb400108,0xde430101,0xfb400108,0xde460101,
	0xfb400108,0xde470101,0xfb400108,0xde480101,0xfb400108,0xde490101,0xfb400108,0xde4a0101,
	0xfb400108,0xde4b0101,0xfb400108,0xde4d0101,0xfb400108,0xde4e0101,0xfb400108,0xde4f0101,
	0xfb400108,0xde500101,0xfb400108,0xde510101,0xfb400108,0xde520101,0xfb400108,0xde530101,
	0xfb400108,0xde560101,0xfb400108,0xde570101,0xfb400108,0xde580101,0xfb400108,0xde590101,
	0xfb400108,0xde5a0101,0xfb400108,0xde5c0101,0xfb400108,0xde5d0101,0xfb400108,0xde5f0101,
	0xfb400108,0xde600101,0xfb400108,0xde630101,0xfb400108,0xde640101,0xfb400108,0xde650101,
	0xfb400108,0xde660101,0xfb400108,0xde670101,0xfb400108,0xde680101,0xfb400108,0xde690101,
	0xfb400108,0xde6a0101,0xfb400108,0xde6b0101,0xfb400108,0xde6c0101,0xfb400108,0xde6d0101,
	0xfb400108,0xde6e0101,0xfb400108,0xde6f0101,0xfb400108,0xde700101,0xfb400108,0xde710101,
	0xfb400108,0xde750101,0xfb400108,0xde770101,0xfb400108,0xde790101,0xfb400108,0xde7e0101,
	0xfb400108,0xde810101,0xfb400108,0xde820101,0xfb400108,0xde830101,0xfb400108,0xde850101,
	0xfb400108,0xde880101,0xfb400108,0xde890101,0xfb400108,0xde8c0101,0xfb400108,0xde8d0101,
	0xfb400108,0xde8e0101,0xfb400108,0xde920101,0xfb400108,0xde980101,0xfb400108,0xde9b0101,
	0xfb400108,0xde9d0101,0xfb400108,0xdea10101,0xfb400108,0xdea20101,0xfb400108,0xdea30101,
	0xfb400108,0xdea40101,0xfb400108,0xdea80101,0xfb400108,0xdea90101,0xfb400108,0xdeaa0101,
	0xfb400108,0xdeab0101,0xfb400108,0xdeac0101,0xfb400108,0xdeae0101,0xfb400108,0xdeaf0101,
	0xfb400108,0xdeb00101,0xfb400108,0xdeb10101,0xfb400108,0xdeb20101,0xfb400108,0xdeb40101,
	0xfb400108,0xdeba0101,0xfb400108,0xdebb0101,0xfb400108,0xdebc0101,0xfb400108,0xdebd0101,
	0xfb400108,0xdebf0101,0xfb400108,0xdec00101,0xfb400108,0xdec10101,0xfb400108,0xdec20101,
	0xfb400108,0xdec30101,0xfb400108,0xdec40101,0xfb400108,0xdec50101,0xfb400108,0xdec60101,
	0xfb400108,0xdec70101,0xfb400108,0xdec80101,0xfb400108,0xdecb0101,0xfb400108,0xdecc0101,
	0xfb400108,0xdecd0101,0xfb400108,0xdece0101,0xfb400108,0xdecf0101,0xfb400108,0xded00101,
	0xfb400108,0xded40101,0xfb400108,0xded50101,0xfb400108,0xded70101,0xfb400108,0xded80101,
	0xfb400108,0xded90101,0xfb400108,0xdeda0101,0xfb400108,0xdedc0101,0xfb400108,0xdedd0101,
	0xfb400108,0xdede0101,0xfb400108,0xdedf0101,0xfb400108,0xdee00101,0xfb400108,0xdee10101,
	0xfb400108,0xdee20101,0xfb400108,0xdee30101,0xfb400108,0xdee40101,0xfb400108,0xdee50101,
	0xfb400108,0xdee60101,0xfb400108,0xdee70101,0xfb400108,0xdee90101,0xfb400108,0xdeeb0101,
	0xfb400108,0xdeec0101,0xfb400108,0xdeed0101,0xfb400108,0xdeee0101,0xfb400108,0xdeef0101,
	0xfb400108,0xdef00101,0xfb400108,0xdef10101,0xfb400108,0xdef20101,0xfb400108,0xdef30101,
	0xfb400108,0xdef50101,0xfb400108,0xdef80101,0xfb400108,0xdef90101,0xfb400108,0xdefb0101,
	0xfb400108,0xdefc0101,0xfb400108,0xdefd0101,0xfb400108,0xdf050101,0xfb400108,0xdf060101,
	0xfb400108,0xdf070101,0xfb400108,0xdf090101,0xfb400108,0xdf0c0101,0xfb400108,0xdf0d0101,
	0xfb400108,0xdf0e0101,0xfb400108,0xdf100101,0xfb400108,0xdf120101,0xfb400108,0xdf140101,
	0xfb400108,0xdf160101,0xfb400108,0xdf190101,0xfb400108,0xdf1a0101,0xfb400108,0xdf1c0101,
	0xfb400108,0xdf1d0101,0xfb400108,0xdf1e0101,0xfb400108,0xdf210101,0xfb400108,0xdf220101,
	0xfb400108,0xdf230101,0xfb400108,0xdf240101,0xfb400108,0xdf280101,0xfb400108,0xdf2b0101,
	0xfb400108,0xdf2c0101,0xfb400108,0xdf2e0101,0xfb400108,0xdf300101,0xfb400108,0xdf320101,
	0xfb400108,0xdf330101,0xfb400108,0xdf340101,0xfb400108,0xdf350101,0xfb400108,0xdf360101,
	0xfb400108,0xdf370101,0xfb400108,0xdf380101,0xfb400108,0xdf3b0101,0xfb400108,0xdf3d0101,
	0xfb400108,0xdf3e0101,0xfb400108,0xdf3f0101,0xfb400108,0xdf410101,0xfb400108,0xdf420101,
	0xfb400108,0xdf430101,0xfb400108,0xdf440101,0xfb400108,0xdf450101,0xfb400108,0xdf460101,
	0xfb400108,0xdf470101,0xfb400108,0xdf480101,0xfb400108,0xdf490101,0xfb400108,0xdf4a0101,
	0xfb400108,0xdf4b0101,0xfb400108,0xdf4c0101,0xfb400108,0xdf4d0101,0xfb400108,0xdf4e0101,
	0xfb400108,0xdf4f0101,0xfb400108,0xdf510101,0xfb400108,0xdf540101,0xfb400108,0xdf590101,
	0xfb400108,0xdf5a0101,0xfb400108,0xdf5b0101,0xfb400108,0xdf5c0101,0xfb400108,0xdf5e0101,
	0xfb400108,0xdf5f0101,0xfb400108,0xdf600101,0xfb400108,0xdf630101,0xfb400108,0xdf650101,
	0xfb400108,0xdf670101,0xfb400108,0xdf680101,0xfb400108,0xdf6b0101,0xfb400108,0xdf6e0101,
	0xfb400108,0xdf6f0101,0xfb400108,0xdf720101,0xfb400108,0xdf740101,0xfb400108,0xdf750101,
	0xfb400108,0xdf760101,0xfb400108,0xdf780101,0xfb400108,0xdf7a0101,0xfb400108,0xdf7d0101,
	0xfb400108,0xdf7e0101,0xfb400108,0xdf7f0101,0xfb400108,0xdf830101,0xfb400108,0xdf860101,
	0xfb400108,0xdf8d0101,0xfb400108,0xdf8e0101,0xfb400108,0xdf8f0101,0xfb400108,0xdf910101,
	0xfb400108,0xdf930101,0xfb400108,0xdf940101,0xfb400108,0xdf960101,0xfb400108,0xdf9a0101,
	0xfb400108,0xdf9b0101,0xfb400108,0xdf9d0101,0xfb400108,0xdf9e0101,0xfb400108,0xdf9f0101,
	0xfb400108,0xdfa00101,0xfb400108,0xdfa20101,0xfb400108,0xdfa30101,0xfb400108,0xdfa40101,
	0xfb400108,0xdfa50101,0xfb400108,0xdfa60101,0xfb400108,0xdfa70101,0xfb400108,0xdfa90101,
	0xfb400108,0xdfab0101,0xfb400108,0xdfac0101,0xfb400108,0xdfaf0101,0xfb400108,0xdfb00101,
	0xfb400108,0xdfb10101,0xfb400108,0xdfb20101,0xfb400108,0xdfb30101,0xfb400108,0xdfb40101,
	0xfb400108,0xdfb60101,0xfb400108,0xdfb80101,0xfb400108,0xdfb90101,0xfb400108,0xdfba0101,
	0xfb400108,0xdfbb0101,0xfb400108,0xdfbe0101,0xfb400108,0xdfbf0101,0xfb400108,0xdfc00101,
	0xfb400108,0xdfc10101,0xfb400108,0xdfc20101,0xfb400108,0xdfc70101,0xfb400108,0xdfc80101,
	0xfb400108,0xdfca0101,0xfb400108,0xdfcb0101,0xfb400108,0xdfce0101,0xfb400108,0xdfd30101,
	0xfb400108,0xdfd40101,0xfb400108,0xdfd50101,0xfb400108,0xdfda0101,0xfb400108,0xdfdb0101,
	0xfb400108,0xdfdc0101,0xfb400108,0xdfde0101,0xfb400108,0xdfdf0101,0xfb400108,0xdfe20101,
	0xfb400108,0xdfe30101,0xfb400108,0xdfe50101,0xfb400108,0xdfe60101,0xfb400108,0xdfe80101,
	0xfb400108,0xdfe90101,0xfb400108,0xdfec0101,0xfb400108,0xdfef0101,0xfb400108,0xdff00101,
	0xfb400108,0xdff20101,0xfb400108,0xdff30101,0xfb400108,0xdff40101,0xfb400108,0xdff60101,
	0xfb400108,0xdff70101,0xfb400108,0xdff90101,0xfb400108,0xdffa0101,0xfb400108,0xdffc0101,
	0xfb400108,0xe0070101,0xfb400108,0xe0080101,0xfb400108,0xe0090101,0xfb400108,0xe00b0101,
	0xfb400108,0xe00c0101,0xfb400108,0xe0100101,0xfb400108,0xe0110101,0xfb400108,0xe0130101,
	0xfb400108,0xe0170101,0xfb400108,0xe0180101,0xfb400108,0xe01a0101,0xfb400108,0xe01e0101,
	0xfb400108,0xe01f0101,0xfb400108,0xe0220101,0xfb400108,0xe0230101,0xfb400108,0xe0240101,
	0xfb400108,0xe02c0101,0xfb400108,0xe02d0101,0xfb400108,0xe02e0101,0xfb400108,0xe0300101,
	0xfb400108,0xe0310101,0xfb400108,0xe0320101,0xfb400108,0xe0330101,0xfb400108,0xe0340101,
	0xfb400108,0xe0360101,0xfb400108,0xe0370101,0xfb400108,0xe0380101,0xfb400108,0xe0390101,
	0xfb400108,0xe03a0101,0xfb400108,0xe03d0101,0xfb400108,0xe03e0101,0xfb400108,0xe0400101,
	0xfb400108,0xe0440101,0xfb400108,0xe0450101,0xfb400108,0xe0460101,0xfb400108,0xe0470101,
	0xfb400108,0xe0480101,0xfb400108,0xe0490101,0xfb400108,0xe04a0101,0xfb400108,0xe04c0101,
	0xfb400108,0xe04e0101,0xfb400108,0xe04f0101,0xfb400108,0xe0510101,0xfb400108,0xe0530101,
	0xfb400108,0xe0540101,0xfb400108,0xe0560101,0xfb400108,0xe0570101,0xfb400108,0xe0580101,
	0xfb400108,0xe05b0101,0xfb400108,0xe05c0101,0xfb400108,0xe05e0101,0xfb400108,0xe05f0101,
	0xfb400108,0xe0600101,0xfb400108,0xe0610101,0xfb400108,0xe0650101,0xfb400108,0xe0660101,
	0xfb400108,0xe06e0101,0xfb400108,0xe0710101,0xfb400108,0xe0720101,0xfb400108,0xe0740101,
	0xfb400108,0xe0750101,0xfb400108,0xe0770101,0xfb400108,0xe07e0101,0xfb400108,0xe0800101,
	0xfb400108,0xe0810101,0xfb400108,0xe0820101,0xfb400108,0xe0850101,0xfb400108,0xe0860101,
	0xfb400108,0xe0870101,0xfb400108,0xe0880101,0xfb400108,0xe08a0101,0xfb400108,0xe08b0101,
	0xfb400108,0xe08e0101,0xfb400108,0xe08f0101,0xfb400108,0xe0900101,0xfb400108,0xe0910101,
	0xfb400108,0xe0930101,0xfb400108,0xe0950101,0xfb400108,0xe0970101,0xfb400108,0xe0980101,
	0xfb400108,0xe0990101,0xfb400108,0xe09c0101,0xfb400108,0xe09e0101,0xfb400108,0xe0a10101,
	0xfb400108,0xe0a20101,0xfb400108,0xe0a40101,0xfb400108,0xe0a50101,0xfb400108,0xe0a70101,
	0xfb400108,0xe0a90101,0xfb400108,0xe0aa0101,0xfb400108,0xe0ae0101,0xfb400108,0xe0b00101,
	0xfb400108,0xe0b30101,0xfb400108,0xe0b50101,0xfb400108,0xe0b60101,0xfb400108,0xe0b70101,
	0xfb400108,0xe0b90101,0xfb400108,0xe0ba0101,0xfb400108,0xe0bd0101,0xfb400108,0xe0be0101,
	0xfb400108,0xe0bf0101,0xfb400108,0xe0c00101,0xfb400108,0xe0c10101,0xfb400108,0xe0c20101,
	0xfb400108,0xe0c30101,0xfb400108,0xe0c40101,0xfb400108,0xe0c70101,0xfb400108,0xe0c80101,
	0xfb400108,0xe0c90101,0xfb400108,0xe0cc0101,0xfb400108,0xe0cd0101,0xfb400108,0xe0ce0101,
	0xfb400108,0xe0cf0101,0xfb400108,0xe0d00101,0xfb400108,0xe0d20101,0xfb400108,0xe0d30101,
	0xfb400108,0xe0d40101,0xfb400108,0xe0d60101,0xfb400108,0xe0d70101,0xfb400108,0xe0d90101,
	0xfb400108,0xe0db0101,0xfb400108,0xe0de0101,0xfb400108,0xe0e10101,0xfb400108,0xe0e20101,
	0xfb400108,0xe0e30101,0xfb400108,0xe0e40101,0xfb400108,0xe0e50101,0xfb400108,0xe0ea0101,
	0xfb400108,0xe0f10101,0xfb400108,0xe0f20101,0xfb400108,0xe0f50101,0xfb400108,0xe0f70101,
	0xfb400108,0xe0f80101,0xfb400108,0xe0fb0101,0xfb400108,0xe0fc0101,0xfb400108,0xe0fd0101,
	0xfb400108,0xe0fe0101,0xfb400108,0xe0ff0101,0xfb400108,0xe1020101,0xfb400108,0xe1030101,
	0xfb400108,0xe1040101,0xfb400108,0xe1050101,0xfb400108,0xe1070101,0xfb400108,0xe10a0101,
	0xfb400108,0xe10b0101,0xfb400108,0xe10c0101,0xfb400108,0xe1100101,0xfb400108,0xe1110101,
	0xfb400108,0xe1120101,0xfb400108,0xe1130101,0xfb400108,0xe1140101,0xfb400108,0xe1160101,
	0xfb400108,0xe1170101,0xfb400108,0xe1180101,0xfb400108,0xe1190101,0xfb400108,0xe11b0101,
	0xfb400108,0xe11c0101,0xfb400108,0xe11d0101,0xfb400108,0xe11e0101,0xfb400108,0xe1210101,
	0xfb400108,0xe1220101,0xfb400108,0xe1250101,0xfb400108,0xe1280101,0xfb400108,0xe1290101,
	0xfb400108,0xe12a0101,0xfb400108,0xe12c0101,0xfb400108,0xe12d0101,0xfb400108,0xe12e0101,
	0xfb400108,0xe12f0101,0xfb400108,0xe1300101,0xfb400108,0xe1310101,0xfb400108,0xe1320101,
	0xfb400108,0xe1330101,0xfb400108,0xe1340101,0xfb400108,0xe1350101,0xfb400108,0xe1360101,
	0xfb400108,0xe1370101,0xfb400108,0xe1380101,0xfb400108,0xe1390101,0xfb400108,0xe13a0101,
	0xfb400108,0xe13b0101,0xfb400108,0xe13c0101,0xfb400108,0xe13d0101,0xfb400108,0xe13e0101,
	0xfb400108,0xe1400101,0xfb400108,0xe1410101,0xfb400108,0xe1420101,0xfb400108,0xe1430101,
	0xfb400108,0xe1440101,0xfb400108,0xe1450101,0xfb400108,0xe1460101,0xfb400108,0xe1470101,
	0xfb400108,0xe1490101,0xfb400108,0xe14b0101,0xfb400108,0xe14d0101,0xfb400108,0xe14f0101,
	0xfb400108,0xe1500101,0xfb400108,0xe1520101,0xfb400108,0xe1530101,0xfb400108,0xe1540101,
	0xfb400108,0xe1560101,0xfb400108,0xe1570101,0xfb400108,0xe1580101,0xfb400108,0xe1590101,
	0xfb400108,0xe15a0101,0xfb400108,0xe15b0101,0xfb400108,0xe15c0101,0xfb400108,0xe15e0101,
	0xfb400108,0xe15f0101,0xfb400108,0xe1600101,0xfb400108,0xe1610101,0xfb400108,0xe1630101,
	0xfb400108,0xe1640101,0xfb400108,0xe1650101,0xfb400108,0xe1660101,0xfb400108,0xe1690101,
	0xfb400108,0xe16a0101,0xfb400108,0xe16b0101,0xfb400108,0xe16c0101,0xfb400108,0xe16d0101,
	0xfb400108,0xe16e0101,0xfb400108,0xe16f0101,0xfb400108,0xe1710101,0xfb400108,0xe1720101,
	0xfb400108,0xe1730101,0xfb400108,0xe1740101,0xfb400108,0xe1760101,0xfb400108,0xe1780101,
	0xfb400108,0xe1790101,0xfb400108,0xe17a0101,0xfb400108,0xe17b0101,0xfb400108,0xe17c0101,
	0xfb400108,0xe17d0101,0xfb400108,0xe17e0101,0xfb400108,0xe17f0101,0xfb400108,0xe1800101,
	0xfb400108,0xe1810101,0xfb400108,0xe1820101,0xfb400108,0xe1830101,0xfb400108,0xe1840101,
	0xfb400108,0xe1850101,0xfb400108,0xe1860101,0xfb400108,0xe1870101,0xfb400108,0xe1880101,
	0xfb400108,0xe1890101,0xfb400108,0xe18a0101,0xfb400108,0xe18c0101,0xfb400108,0xe18d0101,
	0xfb400108,0xe18f0101,0xfb400108,0xe1900101,0xfb400108,0xe1910101,0xfb400108,0xe1920101,
	0xfb400108,0xe1930101,0xfb400108,0xe1950101,0xfb400108,0xe1960101,0xfb400108,0xe1970101,
	0xfb400108,0xe1980101,0xfb400108,0xe1990101,0xfb400108,0xe19a0101,0xfb400108,0xe19b0101,
	0xfb400108,0xe19c0101,0xfb400108,0xe19e0101,0xfb400108,0xe19f0101,0xfb400108,0xe1a00101,
	0xfb400108,0xe1a10101,0xfb400108,0xe1a20101,0xfb400108,0xe1a30101,0xfb400108,0xe1a40101,
	0xfb400108,0xe1a50101,0xfb400108,0xe1a60101,0xfb400108,0xe1aa0101,0xfb400108,0xe1ab0101,
	0xfb400108,0xe1ad0101,0xfb400108,0xe1ae0101,0xfb400108,0xe1af0101,0xfb400108,0xe1b00101,
	0xfb400108,0xe1b10101,0xfb400108,0xe1b20101,0xfb400108,0xe1b30101,0xfb400108,0xe1b40101,
	0xfb400108,0xe1b50101,0xfb400108,0xe1b60101,0xfb400108,0xe1b80101,0xfb400108,0xe1b90101,
	0xfb400108,0xe1ba0101,0xfb400108,0xe1bb0101,0xfb400108,0xe1bc0101,0xfb400108,0xe1bd0101,
	0xfb400108,0xe1bf0101,0xfb400108,0xe1c00101,0xfb400108,0xe1c10101,0xfb400108,0xe1c30101,
	0xfb400108,0xe1c40101,0xfb400108,0xe1c50101,0xfb400108,0xe1c60101,0xfb400108,0xe1c70101,
	0xfb400108,0xe1c90101,0xfb400108,0xe1cc0101,0xfb400108,0xe1cd0101,0xfb400108,0xe1ce0101,
	0xfb400108,0xe1cf0101,0xfb400108,0xe1d00101,0xfb400108,0xe1d30101,0xfb400108,0xe1d50101,
	0xfb400108,0xe1d60101,0xfb400108,0xe1d70101,0xfb400108,0xe1d80101,0xfb400108,0xe1d90101,
	0xfb400108,0xe1da0101,0xfb400108,0xe1db0101,0xfb400108,0xe1dc0101,0xfb400108,0xe1dd0101,
	0xfb400108,0xe1de0101,0xfb400108,0xe1df0101,0xfb400108,0xe1e00101,0xfb400108,0xe1e10101,
	0xfb400108,0xe1e20101,0xfb400108,0xe1e30101,0xfb400108,0xe1e40101,0xfb400108,0xe1e50101,
	0xfb400108,0xe1e70101,0xfb400108,0xe1e80101,0xfb400108,0xe1e90101,0xfb400108,0xe1ea0101,
	0xfb400108,0xe1eb0101,0xfb400108,0xe1ec0101,0xfb400108,0xe1ed0101,0xfb400108,0xe1ee0101,
	0xfb400108,0xe1ef0101,0xfb400108,0xe1f00101,0xfb400108,0xe1f10101,0xfb400108,0xe1f20101,
	0xfb400108,0xe1f30101,0xfb400108,0xe1f40101,0xfb400108,0xe1f60101,0xfb400108,0xe1f70101,
	0xfb400108,0xe1f80101,0xfb400108,0xe1f90101,0xfb400108,0xe1fa0101,0xfb400108,0xe1fb0101,
	0xfb400108,0xe1fc0101,0xfb400108,0xe1fd0101,0xfb400108,0xe1fe0101,0xfb400108,0xe2000101,
	0xfb400108,0xe2010101,0xfb400108,0xe2020101,0xfb400108,0xe2030101,0xfb400108,0xe2040101,
	0xfb400108,0xe2050101,0xfb400108,0xe2070101,0xfb400108,0xe2090101,0xfb400108,0xe2130101,
	0xfb400108,0xe2140101,0xfb400108,0xe2190101,0xfb400108,0xe21c0101,0xfb400108,0xe21d0101,
	0xfb400108,0xe21e0101,0xfb400108,0xe2200101,0xfb400108,0xe2230101,0xfb400108,0xe2260101,
	0xfb400108,0xe2270101,0xfb400108,0xe2280101,0xfb400108,0xe2290101,0xfb400108,0xe22b0101,
	0xfb400108,0xe22d0101,0xfb400108,0xe22f0101,0xfb400108,0xe2300101,0xfb400108,0xe2310101,
	0xfb400108,0xe2320101,0xfb400108,0xe2350101,0xfb400108,0xe2360101,0xfb400108,0xe2380101,
	0xfb400108,0xe2390101,0xfb400108,0xe23a0101,0xfb400108,0xe23b0101,0xfb400108,0xe23c0101,
	0xfb400108,0xe2420101,0xfb400108,0xe2440101,0xfb400108,0xe2450101,0xfb400108,0xe2460101,
	0xfb400108,0xe24a0101,0xfb400108,0xe24f0101,0xfb400108,0xe2500101,0xfb400108,0xe2550101,
	0xfb400108,0xe2560101,0xfb400108,0xe2570101,0xfb400108,0xe2590101,0xfb400108,0xe25a0101,
	0xfb400108,0xe25c0101,0xfb400108,0xe25d0101,0xfb400108,0xe25e0101,0xfb400108,0xe25f0101,
	0xfb400108,0xe2600101,0xfb400108,0xe2610101,0xfb400108,0xe2620101,0xfb400108,0xe2640101,
	0xfb400108,0xe2650101,0xfb400108,0xe2680101,0xfb400108,0xe2710101,0xfb400108,0xe2720101,
	0xfb400108,0xe2740101,0xfb400108,0xe2750101,0xfb400108,0xe2770101,0xfb400108,0xe2780101,
	0xfb400108,0xe27a0101,0xfb400108,0xe27b0101,0xfb400108,0xe27d0101,0xfb400108,0xe2810101,
	0xfb400108,0xe2820101,0xfb400108,0xe2830101,0xfb400108,0xe2850101,0xfb400108,0xe2860101,
	0xfb400108,0xe2870101,0xfb400108,0xe2880101,0xfb400108,0xe28b0101,0xfb400108,0xe28c0101,
	0xfb400108,0xe28d0101,0xfb400108,0xe28e0101,0xfb400108,0xe28f0101,0xfb400108,0xe2900101,
	0xfb400108,0xe2940101,0xfb400108,0xe2990101,0xfb400108,0xe29c0101,0xfb400108,0xe29d0101,
	0xfb400108,0xe29e0101,0xfb400108,0xe2a30101,0xfb400108,0xe2a60101,0xfb400108,0xe2a70101,
	0xfb400108,0xe2a90101,0xfb400108,0xe2aa0101,0xfb400108,0xe2ad0101,0xfb400108,0xe2ae0101,
	0xfb400108,0xe2af0101,0xfb400108,0xe2b00101,0xfb400108,0xe2b20101,0xfb400108,0xe2b30101,
	0xfb400108,0xe2b40101,0xfb400108,0xe2b60101,0xfb400108,0xe2b70101,0xfb400108,0xe2b80101,
	0xfb400108,0xe2ba0101,0xfb400108,0xe2be0101,0xfb400108,0xe2c00101,0xfb400108,0xe2c10101,
	0xfb400108,0xe2c30101,0xfb400108,0xe2cb0101,0xfb400108,0xe2cf0101,0xfb400108,0xe2d10101,
	0xfb400108,0xe2d50101,0xfb400108,0xe2dd0101,0xfb400108,0xe2de0101,0xfb400108,0xe2e00101,
	0xfb400108,0xe2e10101,0xfb400108,0xe2e40101,0xfb400108,0xe2ea0101,0xfb400108,0xe2eb0101,
	0xfb400108,0xe2f00101,0xfb400108,0xe2f20101,0xfb400108,0xe2f50101,0xfb400108,0xe2f80101,
	0xfb400108,0xe2f90101,0xfb400108,0xe2fa0101,0xfb400108,0xe2fb0101,0xfb400108,0xe3000101,
	0xfb400108,0xe3030101,0xfb400108,0xe3040101,0xfb400108,0xe3050101,0xfb400108,0xe3060101,
	0xfb400108,0xe30a0101,0xfb400108,0xe30b0101,0xfb400108,0xe30c0101,0xfb400108,0xe30d0101,
	0xfb400108,0xe30f0101,0xfb400108,0xe3100101,0xfb400108,0xe3120101,0xfb400108,0xe3130101,
	0xfb400108,0xe3140101,0xfb400108,0xe3150101,0xfb400108,0xe3170101,0xfb400108,0xe3180101,
	0xfb400108,0xe3190101,0xfb400108,0xe31c0101,0xfb400108,0xe3260101,0xfb400108,0xe3270101,
	0xfb400108,0xe3290101,0xfb400108,0xe32c0101,0xfb400108,0xe32d0101,0xfb400108,0xe32e0101,
	0xfb400108,0xe3300101,0xfb400108,0xe3310101,0xfb400108,0xe3330101,0xfb400108,0xe3340101,
	0xfb400108,0xe3350101,0xfb400108,0xe3360101,0xfb400108,0xe3370101,0xfb400108,0xe3380101,
	0xfb400108,0xe33b0101,0xfb400108,0xe33c0101,0xfb400108,0xe33e0101,0xfb400108,0xe33f0101,
	0xfb400108,0xe3400101,0xfb400108,0xe3410101,0xfb400108,0xe3440101,0xfb400108,0xe3470101,
	0xfb400108,0xe3480101,0xfb400108,0xe34a0101,0xfb400108,0xe3510101,0xfb400108,0xe3520101,
	0xfb400108,0xe3530101,0xfb400108,0xe3540101,0xfb400108,0xe3560101,0xfb400108,0xe3570101,
	0xfb400108,0xe3580101,0xfb400108,0xe3590101,0xfb400108,0xe35a0101,0xfb400108,0xe35b0101,
	0xfb400108,0xe35c0101,0xfb400108,0xe35d0101,0xfb400108,0xe3600101,0xfb400108,0xe3640101,
	0xfb400108,0xe3650101,0xfb400108,0xe3660101,0xfb400108,0xe3680101,0xfb400108,0xe36a0101,
	0xfb400108,0xe36b0101,0xfb400108,0xe36c0101,0xfb400108,0xe36f0101,0xfb400108,0xe3700101,
	0xfb400108,0xe3720101,0xfb400108,0xe3730101,0xfb400108,0xe3740101,0xfb400108,0xe3750101,
	0xfb400108,0xe3780101,0xfb400108,0xe3790101,0xfb400108,0xe37c0101,0xfb400108,0xe37d0101,
	0xfb400108,0xe37e0101,0xfb400108,0xe37f0101,0xfb400108,0xe3810101,0xfb400108,0xe3830101,
	0xfb400108,0xe3840101,0xfb400108,0xe3850101,0xfb400108,0xe3860101,0xfb400108,0xe38b0101,
	0xfb400108,0xe38d0101,0xfb400108,0xe3910101,0xfb400108,0xe3930101,0xfb400108,0xe3940101,
	0xfb400108,0xe3950101,0xfb400108,0xe3970101,0xfb400108,0xe3990101,0xfb400108,0xe39a0101,
	0xfb400108,0xe39b0101,0xfb400108,0xe39c0101,0xfb400108,0xe39d0101,0xfb400108,0xe39e0101,
	0xfb400108,0xe39f0101,0xfb400108,0xe3a10101,0xfb400108,0xe3a40101,0xfb400108,0xe3a60101,
	0xfb400108,0xe3ab0101,0xfb400108,0xe3af0101,0xfb400108,0xe3b10101,0xfb400108,0xe3b20101,
	0xfb400108,0xe3b50101,0xfb400108,0xe3b60101,0xfb400108,0xe3b90101,0xfb400108,0xe3bb0101,
	0xfb400108,0xe3bd0101,0xfb400108,0xe3bf0101,0xfb400108,0xe3c00101,0xfb400108,0xe3c10101,
	0xfb400108,0xe3c20101,0xfb400108,0xe3c30101,0xfb400108,0xe3c50101,0xfb400108,0xe3c70101,
	0xfb400108,0xe3c80101,0xfb400108,0xe3ca0101,0xfb400108,0xe3cb0101,0xfb400108,0xe3cc0101,
	0xfb400108,0xe3d10101,0xfb400108,0xe3d30101,0xfb400108,0xe3d40101,0xfb400108,0xe3d50101,
	0xfb400108,0xe3d70101,0xfb400108,0xe3d80101,0xfb400108,0xe3d90101,0xfb400108,0xe3da0101,
	0xfb400108,0xe3db0101,0xfb400108,0xe3dc0101,0xfb400108,0xe3dd0101,0xfb400108,0xe3df0101,
	0xfb400108,0xe3e20101,0xfb400108,0xe3e40101,0xfb400108,0xe3e50101,0xfb400108,0xe3e60101,
	0xfb400108,0xe3e70101,0xfb400108,0xe3e80101,0xfb400108,0xe3eb0101,0xfb400108,0xe3ec0101,
	0xfb400108,0xe3ee0101,0xfb400108,0xe3ef0101,0xfb400108,0xe3f00101,0xfb400108,0xe3f10101,
	0xfb400108,0xe3f30101,0xfb400108,0xe3f50101,0xfb400108,0xe3f70101,0xfb400108,0xe3f90101,
	0xfb400108,0xe3fa0101,0xfb400108,0xe3fb0101,0xfb400108,0xe3fc0101,0xfb400108,0xe3fe0101,
	0xfb400108,0xe4030101,0xfb400108,0xe4040101,0xfb400108,0xe4060101,0xfb400108,0xe4070101,
	0xfb400108,0xe4080101,0xfb400108,0xe4090101,0xfb400108,0xe40a0101,0xfb400108,0xe40d0101,
	0xfb400108,0xe40e0101,0xfb400108,0xe4110101,0xfb400108,0xe4120101,0xfb400108,0xe4150101,
	0xfb400108,0xe4160101,0xfb400108,0xe4170101,0xfb400108,0xe4180101,0xfb400108,0xe4190101,
	0xfb400108,0xe41a0101,0xfb400108,0xe41d0101,0xfb400108,0xe41f0101,0xfb400108,0xe4220101,
	0xfb400108,0xe4230101,0xfb400108,0xe4240101,0xfb400108,0xe4250101,0xfb400108,0xe4270101,
	0xfb400108,0xe4280101,0xfb400108,0xe4290101,0xfb400108,0xe42b0101,0xfb400108,0xe42e0101,
	0xfb400108,0xe42f0101,0xfb400108,0xe4300101,0xfb400108,0xe4310101,0xfb400108,0xe4320101,
	0xfb400108,0xe4330101,0xfb400108,0xe4350101,0xfb400108,0xe4360101,0xfb400108,0xe4370101,
	0xfb400108,0xe4380101,0xfb400108,0xe4390101,0xfb400108,0xe43b0101,0xfb400108,0xe43c0101,
	0xfb400108,0xe43e0101,0xfb400108,0xe4400101,0xfb400108,0xe4420101,0xfb400108,0xe4430101,
	0xfb400108,0xe4490101,0xfb400108,0xe44b0101,0xfb400108,0xe44c0101,0xfb400108,0xe44d0101,
	0xfb400108,0xe44e0101,0xfb400108,0xe44f0101,0xfb400108,0xe4500101,0xfb400108,0xe4510101,
	0xfb400108,0xe4530101,0xfb400108,0xe4550101,0xfb400108,0xe4560101,0xfb400108,0xe4570101,
	0xfb400108,0xe4590101,0xfb400108,0xe45a0101,0xfb400108,0xe45b0101,0xfb400108,0xe45c0101,
	0xfb400108,0xe45d0101,0xfb400108,0xe45f0101,0xfb400108,0xe4600101,0xfb400108,0xe4610101,
	0xfb400108,0xe4620101,0xfb400108,0xe4630101,0xfb400108,0xe4640101,0xfb400108,0xe4650101,
	0xfb400108,0xe4660101,0xfb400108,0xe4680101,0xfb400108,0xe46a0101,0xfb400108,0xe46b0101,
	0xfb400108,0xe46c0101,0xfb400108,0xe46e0101,0xfb400108,0xe46f0101,0xfb400108,0xe4700101,
	0xfb400108,0xe4710101,0xfb400108,0xe4720101,0xfb400108,0xe4730101,0xfb400108,0xe4740101,
	0xfb400108,0xe4750101,0xfb400108,0xe4760101,0xfb400108,0xe4770101,0xfb400108,0xe47b0101,
	0xfb400108,0xe47c0101,0xfb400108,0xe47d0101,0xfb400108,0xe47e0101,0xfb400108,0xe47f0101,
	0xfb400108,0xe4800101,0xfb400108,0xe4810101,0xfb400108,0xe4830101,0xfb400108,0xe4860101,
	0xfb400108,0xe4880101,0xfb400108,0xe4890101,0xfb400108,0xe48a0101,0xfb400108,0xe48b0101,
	0xfb400108,0xe48c0101,0xfb400108,0xe48d0101,0xfb400108,0xe48e0101,0xfb400108,0xe48f0101,
	0xfb400108,0xe4900101,0xfb400108,0xe4930101,0xfb400108,0xe4940101,0xfb400108,0xe4970101,
	0xfb400108,0xe4980101,0xfb400108,0xe49a0101,0xfb400108,0xe49b0101,0xfb400108,0xe49c0101,
	0xfb400108,0xe49d0101,0xfb400108,0xe49f0101,0xfb400108,0xe4a00101,0xfb400108,0xe4a10101,
	0xfb400108,0xe4a20101,0xfb400108,0xe4a30101,0xfb400108,0xe4a50101,0xfb400108,0xe4a60101,
	0xfb400108,0xe4a70101,0xfb400108,0xe4a80101,0xfb400108,0xe4aa0101,0xfb400108,0xe4ab0101,
	0xfb400108,0xe4af0101,0xfb400108,0xe4b10101,0xfb400108,0xe4b20101,0xfb400108,0xe4b30101,
	0xfb400108,0xe4b40101,0xfb400108,0xe4b60101,0xfb400108,0xe4b90101,0xfb400108,0xe4bb0101,
	0xfb400108,0xe4bd0101,0xfb400108,0xe4be0101,0xfb400108,0xe4bf0101,0xfb400108,0xe4c10101,
	0xfb400108,0xe4c30101,0xfb400108,0xe4c40101,0xfb400108,0xe4c60101,0xfb400108,0xe4c70101,
	0xfb400108,0xe4c80101,0xfb400108,0xe4c90101,0xfb400108,0xe4ca0101,0xfb400108,0xe4cb0101,
	0xfb400108,0xe4cc0101,0xfb400108,0xe4cf0101,0xfb400108,0xe4d10101,0xfb400108,0xe4d30101,
	0xfb400108,0xe4d40101,0xfb400108,0xe4d50101,0xfb400108,0xe4d60101,0xfb400108,0xe4d90101,
	0xfb400108,0xe4da0101,0xfb400108,0xe4db0101,0xfb400108,0xe4dc0101,0xfb400108,0xe4dd0101,
	0xfb400108,0xe4df0101,0xfb400108,0xe4e00101,0xfb400108,0xe4e10101,0xfb400108,0xe4e30101,
	0xfb400108,0xe4e50101,0xfb400108,0xe4e70101,0xfb400108,0xe4e80101,0xfb400108,0xe4e90101,
	0xfb400108,0xe4ea0101,0xfb400108,0xe4eb0101,0xfb400108,0xe4ec0101,0xfb400108,0xe4ed0101,
	0xfb400108,0xe4ee0101,0xfb400108,0xe4ef0101,0xfb400108,0xe4f00101,0xfb400108,0xe4f10101,
	0xfb400108,0xe4f20101,0xfb400108,0xe4f30101,0xfb400108,0xe4f40101,0xfb400108,0xe4f50101,
	0xfb400108,0xe4f60101,0xfb400108,0xe4f70101,0xfb400108,0xe4f80101,0xfb400108,0xe4f90101,
	0xfb400108,0xe4fa0101,0xfb400108,0xe4fb0101,0xfb400108,0xe4fc0101,0xfb400108,0xe4fd0101,
	0xfb400108,0xe4fe0101,0xfb400108,0xe4ff0101,0xfb400108,0xe5010101,0xfb400108,0xe5020101,
	0xfb400108,0xe5030101,0xfb400108,0xe5040101,0xfb400108,0xe5050101,0xfb400108,0xe5060101,
	0xfb400108,0xe5070101,0xfb400108,0xe5080101,0xfb400108,0xe50a0101,0xfb400108,0xe50b0101,
	0xfb400108,0xe50c0101,0xfb400108,0xe50d0101,0xfb400108,0xe50e0101,0xfb400108,0xe50f0101,
	0xfb400108,0xe5100101,0xfb400108,0xe5110101,0xfb400108,0xe5130101,0xfb400108,0xe5140101,
	0xfb400108,0xe5150101,0xfb400108,0xe5160101,0xfb400108,0xe5170101,0xfb400108,0xe5190101,
	0xfb400108,0xe51a0101,0xfb400108,0xe51b0101,0xfb400108,0xe51c0101,0xfb400108,0xe51d0101,
	0xfb400108,0xe51e0101,0xfb400108,0xe51f0101,0xfb400108,0xe5200101,0xfb400108,0xe5210101,
	0xfb400108,0xe5220101,0xfb400108,0xe5230101,0xfb400108,0xe5240101,0xfb400108,0xe5260101,
	0xfb400108,0xe5270101,0xfb400108,0xe5280101,0xfb400108,0xe5290101,0xfb400108,0xe52a0101,
	0xfb400108,0xe52c0101,0xfb400108,0xe52d0101,0xfb400108,0xe5300101,0xfb400108,0xe5310101,
	0xfb400108,0xe5320101,0xfb400108,0xe5330101,0xfb400108,0xe5370101,0xfb400108,0xe53a0101,
	0xfb400108,0xe53c0101,0xfb400108,0xe53d0101,0xfb400108,0xe5400101,0xfb400108,0xe5410101,
	0xfb400108,0xe5420101,0xfb400108,0xe5430101,0xfb400108,0xe5440101,0xfb400108,0xe5460101,
	0xfb400108,0xe5470101,0xfb400108,0xe54a0101,0xfb400108,0xe54b0101,0xfb400108,0xe54d0101,
	0xfb400108,0xe54e0101,0xfb400108,0xe5500101,0xfb400108,0xe5520101,0xfb400108,0xe5530101,
	0xfb400108,0xe5540101,0xfb400108,0xe5570101,0xfb400108,0xe5580101,0xfb400108,0xe55a0101,
	0xfb400108,0xe55c0101,0xfb400108,0xe55f0101,0xfb400108,0xe5600101,0xfb400108,0xe5610101,
	0xfb400108,0xe5640101,0xfb400108,0xe5650101,0xfb400108,0xe5670101,0xfb400108,0xe5680101,
	0xfb400108,0xe5690101,0xfb400108,0xe56a0101,0xfb400108,0xe56d0101,0xfb400108,0xe56e0101,
	0xfb400108,0xe56f0101,0xfb400108,0xe5710101,0xfb400108,0xe5730101,0xfb400108,0xe5750101,
	0xfb400108,0xe5760101,0xfb400108,0xe5780101,0xfb400108,0xe5790101,0xfb400108,0xe57a0101,
	0xfb400108,0xe57b0101,0xfb400108,0xe57c0101,0xfb400108,0xe57d0101,0xfb400108,0xe57e0101,
	0xfb400108,0xe57f0101,0xfb400108,0xe5800101,0xfb400108,0xe5810101,0xfb400108,0xe5820101,
	0xfb400108,0xe5830101,0xfb400108,0xe5840101,0xfb400108,0xe5850101,0xfb400108,0xe5860101,
	0xfb400108,0xe5880101,0xfb400108,0xe5890101,0xfb400108,0xe58a0101,0xfb400108,0xe58d0101,
	0xfb400108,0xe58e0101,0xfb400108,0xe58f0101,0xfb400108,0xe5920101,0xfb400108,0xe5940101,
	0xfb400108,0xe5950101,0xfb400108,0xe5960101,0xfb400108,0xe5980101,0xfb400108,0xe59a0101,
	0xfb400108,0xe59d0101,0xfb400108,0xe59e0101,0xfb400108,0xe5a00101,0xfb400108,0xe5a20101,
	0xfb400108,0xe5a30101,0xfb400108,0xe5a60101,0xfb400108,0xe5a80101,0xfb400108,0xe5aa0101,
	0xfb400108,0xe5ac0101,0xfb400108,0xe5ae0101,0xfb400108,0xe5b10101,0xfb400108,0xe5b20101,
	0xfb400108,0xe5b30101,0xfb400108,0xe5b40101,0xfb400108,0xe5b50101,0xfb400108,0xe5b60101,
	0xfb400108,0xe5b70101,0xfb400108,0xe5b80101,0xfb400108,0xe5ba0101,0xfb400108,0xe5bb0101,
	0xfb400108,0xe5be0101,0xfb400108,0xe5bf0101,0xfb400108,0xe5c00101,0xfb400108,0xe5c20101,
	0xfb400108,0xe5c70101,0xfb400108,0xe5c80101,0xfb400108,0xe5c90101,0xfb400108,0xe5ca0101,
	0xfb400108,0xe5cd0101,0xfb400108,0xe5d00101,0xfb400108,0xe5d10101,0xfb400108,0xe5d30101,
	0xfb400108,0xe5d40101,0xfb400108,0xe5d50101,0xfb400108,0xe5d80101,0xfb400108,0xe5d90101,
	0xfb400108,0xe5da0101,0xfb400108,0xe5db0101,0xfb400108,0xe5dc0101,0xfb400108,0xe5dd0101,
	0xfb400108,0xe5de0101,0xfb400108,0xe5df0101,0xfb400108,0xe5e10101,0xfb400108,0xe5e30101,
	0xfb400108,0xe5e40101,0xfb400108,0xe5ea0101,0xfb400108,0xe5eb0101,0xfb400108,0xe5f20101,
	0xfb400108,0xe5f30101,0xfb400108,0xe5f40101,0xfb400108,0xe5f50101,0xfb400108,0xe5f80101,
	0xfb400108,0xe5f90101,0xfb400108,0xe5fb0101,0xfb400108,0xe5fc0101,0xfb400108,0xe5fd0101,
	0xfb400108,0xe5fe0101,0xfb400108,0xe5ff0101,0xfb400108,0xe6010101,0xfb400108,0xe6040101,
	0xfb400108,0xe6050101,0xfb400108,0xe6070101,0xfb400108,0xe6080101,0xfb400108,0xe6090101,
	0xfb400108,0xe60b0101,0xfb400108,0xe60d0101,0xfb400108,0xe6100101,0xfb400108,0xe6110101,
	0xfb400108,0xe6120101,0xfb400108,0xe6160101,0xfb400108,0xe6170101,0xfb400108,0xe6180101,
	0xfb400108,0xe61a0101,0xfb400108,0xe61b0101,0xfb400108,0xe61c0101,0xfb400108,0xe61e0101,
	0xfb400108,0xe6210101,0xfb400108,0xe6220101,0xfb400108,0xe6230101,0xfb400108,0xe6240101,
	0xfb400108,0xe6260101,0xfb400108,0xe6290101,0xfb400108,0xe62a0101,0xfb400108,0xe62b0101,
	0xfb400108,0xe62c0101,0xfb400108,0xe62e0101,0xfb400108,0xe6300101,0xfb400108,0xe6320101,
	0xfb400108,0xe6330101,0xfb400108,0xe6370101,0xfb400108,0xe6380101,0xfb400108,0xe6390101,
	0xfb400108,0xe63a0101,0xfb400108,0xe63b0101,0xfb400108,0xe63d0101,0xfb400108,0xe63f0101,
	0xfb400108,0xe6400101,0xfb400108,0xe6420101,0xfb400108,0xe6440101,0xfb400108,0xe6450101,
	0xfb400108,0xe6460101,0xfb400108,0xe6470101,0xfb400108,0xe6480101,0xfb400108,0xe6490101,
	0xfb400108,0xe64a0101,0xfb400108,0xe64d0101,0xfb400108,0xe64e0101,0xfb400108,0xe6500101,
	0xfb400108,0xe6510101,0xfb400108,0xe6580101,0xfb400108,0xe6590101,0xfb400108,0xe65b0101,
	0xfb400108,0xe65c0101,0xfb400108,0xe65d0101,0xfb400108,0xe65e0101,0xfb400108,0xe6600101,
	0xfb400108,0xe6620101,0xfb400108,0xe6630101,0xfb400108,0xe6650101,0xfb400108,0xe6670101,
	0xfb400108,0xe6690101,0xfb400108,0xe66a0101,0xfb400108,0xe66b0101,0xfb400108,0xe66c0101,
	0xfb400108,0xe66d0101,0xfb400108,0xe6710101,0xfb400108,0xe6720101,0xfb400108,0xe6730101,
	0xfb400108,0xe6750101,0xfb400108,0xe6780101,0xfb400108,0xe6790101,0xfb400108,0xe67b0101,
	0xfb400108,0xe67c0101,0xfb400108,0xe67d0101,0xfb400108,0xe67f0101,0xfb400108,0xe6800101,
	0xfb400108,0xe6810101,0xfb400108,0xe6830101,0xfb400108,0xe6850101,0xfb400108,0xe6860101,
	0xfb400108,0xe6880101,0xfb400108,0xe6890101,0xfb400108,0xe68a0101,0xfb400108,0xe68b0101,
	0xfb400108,0xe68d0101,0xfb400108,0xe68e0101,0xfb400108,0xe68f0101,0xfb400108,0xe6900101,
	0xfb400108,0xe6920101,0xfb400108,0xe6930101,0xfb400108,0xe6940101,0xfb400108,0xe6950101,
	0xfb400108,0xe6980101,0xfb400108,0xe6990101,0xfb400108,0xe69a0101,0xfb400108,0xe69b0101,
	0xfb400108,0xe69c0101,0xfb400108,0xe69e0101,0xfb400108,0xe69f0101,0xfb400108,0xe6a00101,
	0xfb400108,0xe6a10101,0xfb400108,0xe6a20101,0xfb400108,0xe6a30101,0xfb400108,0xe6a40101,
	0xfb400108,0xe6a50101,0xfb400108,0xe6a60101,0xfb400108,0xe6a90101,0xfb400108,0xe6aa0101,
	0xfb400108,0xe6ab0101,0xfb400108,0xe6ac0101,0xfb400108,0xe6ad0101,0xfb400108,0xe6af0101,
	0xfb400108,0xe6b00101,0xfb400108,0xe6b10101,0xfb400108,0xe6b20101,0xfb400108,0xe6b30101,
	0xfb400108,0xe6b50101,0xfb400108,0xe6b60101,0xfb400108,0xe6b70101,0xfb400108,0xe6b80101,
	0xfb400108,0xe6ba0101,0xfb400108,0xe6bb0101,0xfb400108,0xe6bc0101,0xfb400108,0xe6bd0101,
	0xfb400108,0xe6bf0101,0xfb400108,0xe6c00101,0xfb400108,0xe6c10101,0xfb400108,0xe6c20101,
	0xfb400108,0xe6c30101,0xfb400108,0xe6c40101,0xfb400108,0xe6c50101,0xfb400108,0xe6c60101,
	0xfb400108,0xe6c70101,0xfb400108,0xe6c80101,0xfb400108,0xe6c90101,0xfb400108,0xe6ca0101,
	0xfb400108,0xe6cb0101,0xfb400108,0xe6cc0101,0xfb400108,0xe6cd0101,0xfb400108,0xe6ce0101,
	0xfb400108,0xe6cf0101,0xfb400108,0xe6d00101,0xfb400108,0xe6d10101,0xfb400108,0xe6d20101,
	0xfb400108,0xe6d30101,0xfb400108,0xe6d40101,0xfb400108,0xe6d50101,0xfb400108,0xe6d60101,
	0xfb400108,0xe6d70101,0xfb400108,0xe6d80101,0xfb400108,0xe6da0101,0xfb400108,0xe6de0101,
	0xfb400108,0xe6df0101,0xfb400108,0xe6e00101,0xfb400108,0xe6e10101,0xfb400108,0xe6e20101,
	0xfb400108,0xe6e30101,0xfb400108,0xe6e40101,0xfb400108,0xe6e50101,0xfb400108,0xe6e70101,
	0xfb400108,0xe6e80101,0xfb400108,0xe6ea0101,0xfb400108,0xe6eb0101,0xfb400108,0xe6ec0101,
	0xfb400108,0xe6ed0101,0xfb400108,0xe6ee0101,0xfb400108,0xe6ef0101,0xfb400108,0xe6f10101,
	0xfb400108,0xe6f50101,0xfb400108,0xe6f60101,0xfb400108,0xe6f80101,0xfb400108,0xe6fa0101,
	0xfb400108,0xe6fb0101,0xfb400108,0xe6fd0101,0xfb400108,0xe7010101,0xfb400108,0xe7020101,
	0xfb400108,0xe7030101,0xfb400108,0xe7040101,0xfb400108,0xe7050101,0xfb400108,0xe7060101,
	0xfb400108,0xe7070101,0xfb400108,0xe70c0101,0xfb400108,0xe70e0101,0xfb400108,0xe70f0101,
	0xfb400108,0xe7110101,0xfb400108,0xe7120101,0xfb400108,0xe7130101,0xfb400108,0xe7160101,
	0xfb400108,0xe7180101,0xfb400108,0xe7190101,0xfb400108,0xe71a0101,0xfb400108,0xe71c0101,
	0xfb400108,0xe71e0101,0xfb400108,0xe7200101,0xfb400108,0xe7210101,0xfb400108,0xe7220101,
	0xfb400108,0xe7230101,0xfb400108,0xe7240101,0xfb400108,0xe7250101,0xfb400108,0xe7270101,
	0xfb400108,0xe7290101,0xfb400108,0xe72e0101,0xfb400108,0xe7300101,0xfb400108,0xe7320101,
	0xfb400108,0xe7330101,0xfb400108,0xe7360101,0xfb400108,0xe7370101,0xfb400108,0xe7380101,
	0xfb400108,0xe7390101,0xfb400108,0xe73b0101,0xfb400108,0xe73c0101,0xfb400108,0xe73e0101,
	0xfb400108,0xe73f0101,0xfb400108,0xe7410101,0xfb400108,0xe7440101,0xfb400108,0xe7450101,
	0xfb400108,0xe7470101,0xfb400108,0xe74a0101,0xfb400108,0xe74b0101,0xfb400108,0xe74d0101,
	0xfb400108,0xe7520101,0xfb400108,0xe7540101,0xfb400108,0xe7550101,0xfb400108,0xe7570101,
	0xfb400108,0xe7580101,0xfb400108,0xe7590101,0xfb400108,0xe75a0101,0xfb400108,0xe75b0101,
	0xfb400108,0xe75d0101,0xfb400108,0xe7620101,0xfb400108,0xe7630101,0xfb400108,0xe7640101,
	0xfb400108,0xe7660101,0xfb400108,0xe7670101,0xfb400108,0xe76b0101,0xfb400108,0xe76c0101,
	0xfb400108,0xe76e0101,0xfb400108,0xe7710101,0xfb400108,0xe7740101,0xfb400108,0xe7760101,
	0xfb400108,0xe7780101,0xfb400108,0xe7790101,0xfb400108,0xe77a0101,0xfb400108,0xe77b0101,
	0xfb400108,0xe77d0101,0xfb400108,0xe7800101,0xfb400108,0xe7820101,0xfb400108,0xe7830101,
	0xfb400108,0xe7850101,0xfb400108,0xe7860101,0xfb400108,0xe7880101,0xfb400108,0xe78a0101,
	0xfb400108,0xe78c0101,0xfb400108,0xe78d0101,0xfb400108,0xe78e0101,0xfb400108,0xe78f0101,
	0xfb400108,0xe7910101,0xfb400108,0xe7920101,0xfb400108,0xe7930101,0xfb400108,0xe7940101,
	0xfb400108,0xe7960101,0xfb400108,0xe7990101,0xfb400108,0xe79b0101,0xfb400108,0xe79f0101,
	0xfb400108,0xe7a00101,0xfb400108,0xe7a10101,0xfb400108,0xe7a40101,0xfb400108,0xe7a60101,
	0xfb400108,0xe7a90101,0xfb400108,0xe7ac0101,0xfb400108,0xe7ae0101,0xfb400108,0xe7b10101,
	0xfb400108,0xe7b20101,0xfb400108,0xe7b40101,0xfb400108,0xe7b90101,0xfb400108,0xe7ba0101,
	0xfb400108,0xe7bb0101,0xfb400108,0xe7bc0101,0xfb400108,0xe7bd0101,0xfb400108,0xe7be0101,
	0xfb400108,0xe7bf0101,0xfb400108,0xe7c00101,0xfb400108,0xe7c20101,0xfb400108,0xe7c50101,
	0xfb400108,0xe7c60101,0xfb400108,0xe7c70101,0xfb400108,0xe7c80101,0xfb400108,0xe7c90101,
	0xfb400108,0xe7ca0101,0xfb400108,0xe7cb0101,0xfb400108,0xe7cc0101,0xfb400108,0xe7cd0101,
	0xfb400108,0xe7ce0101,0xfb400108,0xe7d50101,0xfb400108,0xe7d60101,0xfb400108,0xe7d70101,
	0xfb400108,0xe7db0101,0xfb400108,0xe7df0101,0xfb400108,0xe7e10101,0xfb400108,0xe7e30101,
	0xfb400108,0xe7e40101,0xfb400108,0xe7e60101,0xfb400108,0xe7e70101,0xfb400108,0xe7e80101,
	0xfb400108,0xe7ea0101,0xfb400108,0xe7eb0101,0xfb400108,0xe7ed0101,0xfb400108,0xe7ee0101,
	0xfb400108,0xe7f20101,0xfb400108,0xe7f50101,0xfb400108,0xe7f60101,0xfb400108,0xe7f70101,
	0xfb400108,0xe7f80101,0xfb400108,0xe7f90101,0xfb400108,0xe7fa0101,0xfb400108,0xe7fb0101,
	0xfb400108,0xe7fc0101,0xfb400108,0xe7fe0101,0xfb400108,0xe8010101,0xfb400108,0xe8020101,
	0xfb400108,0xe8030101,0xfb400108,0xe8040101,0xfb400108,0xe8060101,0xfb400108,0xe80d0101,
	0xfb400108,0xe8100101,0xfb400108,0xe8120101,0xfb400108,0xe8140101,0xfb400108,0xe8150101,
	0xfb400108,0xe8180101,0xfb400108,0xe8190101,0xfb400108,0xe81a0101,0xfb400108,0xe81b0101,
	0xfb400108,0xe81c0101,0xfb400108,0xe81e0101,0xfb400108,0xe81f0101,0xfb400108,0xe8200101,
	0xfb400108,0xe8220101,0xfb400108,0xe8230101,0xfb400108,0xe8240101,0xfb400108,0xe8250101,
	0xfb400108,0xe8260101,0xfb400108,0xe8270101,0xfb400108,0xe8280101,0xfb400108,0xe82b0101,
	0xfb400108,0xe82c0101,0xfb400108,0xe82d0101,0xfb400108,0xe82e0101,0xfb400108,0xe82f0101,
	0xfb400108,0xe8300101,0xfb400108,0xe8310101,0xfb400108,0xe8340101,0xfb400108,0xe8350101,
	0xfb400108,0xe8360101,0xfb400108,0xe83a0101,0xfb400108,0xe83b0101,0xfb400108,0xe83f0101,
	0xfb400108,0xe8470101,0xfb400108,0xe84b0101,0xfb400108,0xe84d0101,0xfb400108,0xe84f0101,
	0xfb400108,0xe8520101,0xfb400108,0xe8560101,0xfb400108,0xe8570101,0xfb400108,0xe8580101,
	0xfb400108,0xe8590101,0xfb400108,0xe85a0101,0xfb400108,0xe85b0101,0xfb400108,0xe85c0101,
	0xfb400108,0xe85d0101,0xfb400108,0xe85e0101,0xfb400108,0xe85f0101,0xfb400108,0xe86a0101,
	0xfb400108,0xe86c0101,0xfb400108,0xe86d0101,0xfb400108,0xe86e0101,0xfb400108,0xe86f0101,
	0xfb400108,0xe8700101,0xfb400108,0xe8710101,0xfb400108,0xe8720101,0xfb400108,0xe8730101,
	0xfb400108,0xe8750101,0xfb400108,0xe8780101,0xfb400108,0xe8790101,0xfb400108,0xe87a0101,
	0xfb400108,0xe87b0101,0xfb400108,0xe87c0101,0xfb400108,0xe87d0101,0xfb400108,0xe87e0101,
	0xfb400108,0xe87f0101,0xfb400108,0xe8800101,0xfb400108,0xe8820101,0xfb400108,0xe8840101,
	0xfb400108,0xe8870101,0xfb400108,0xe8880101,0xfb400108,0xe8890101,0xfb400108,0xe88a0101,
	0xfb400108,0xe88b0101,0xfb400108,0xe88c0101,0xfb400108,0xe88d0101,0xfb400108,0xe88e0101,
	0xfb400108,0xe8900101,0xfb400108,0xe8910101,0xfb400108,0xe8920101,0xfb400108,0xe8940101,
	0xfb400108,0xe8950101,0xfb400108,0xe8960101,0xfb400108,0xe8980101,0xfb400108,0xe8990101,
	0xfb400108,0xe89a0101,0xfb400108,0xe89b0101,0xfb400108,0xe89c0101,0xfb400108,0xe89d0101,
	0xfb400108,0xe89e0101,0xfb400108,0xe89f0101,0xfb400108,0xe8a00101,0xfb400108,0xe8a10101,
	0xfb400108,0xe8a30101,0xfb400108,0xe8a40101,0xfb400108,0xe8a50101,0xfb400108,0xe8a90101,
	0xfb400108,0xe8aa0101,0xfb400108,0xe8ab0101,0xfb400108,0xe8ac0101,0xfb400108,0xe8ae0101,
	0xfb400108,0xe8b10101,0xfb400108,0xe8b20101,0xfb400108,0xe8b40101,0xfb400108,0xe8b60101,
	0xfb400108,0xe8b70101,0xfb400108,0xe8b80101,0xfb400108,0xe8b90101,0xfb400108,0xe8ba0101,
	0xfb400108,0xe8bb0101,0xfb400108,0xe8bc0101,0xfb400108,0xe8bd0101,0xfb400108,0xe8be0101,
	0xfb400108,0xe8bf0101,0xfb400108,0xe8c10101,0xfb400108,0xe8c30101,0xfb400108,0xe8c40101,
	0xfb400108,0xe8c50101,0xfb400108,0xe8c60101,0xfb400108,0xe8c70101,0xfb400108,0xe8c80101,
	0xfb400108,0xe8ca0101,0xfb400108,0xe8cc0101,0xfb400108,0xe8ce0101,0xfb400108,0xe8cf0101,
	0xfb400108,0xe8d00101,0xfb400108,0xe8d10101,0xfb400108,0xe8d30101,0xfb400108,0xe8d40101,
	0xfb400108,0xe8d60101,0xfb400108,0xe8d70101,0xfb400108,0xe8d90101,0xfb400108,0xe8db0101,
	0xfb400108,0xe8dc0101,0xfb400108,0xe8dd0101,0xfb400108,0xe8de0101,0xfb400108,0xe8df0101,
	0xfb400108,0xe8e10101,0xfb400108,0xe8e20101,0xfb400108,0xe8e40101,0xfb400108,0xe8e50101,
	0xfb400108,0xe8e60101,0xfb400108,0xe8e70101,0xfb400108,0xe8e80101,0xfb400108,0xe8e90101,
	0xfb400108,0xe8ea0101,0xfb400108,0xe8eb0101,0xfb400108,0xe8ec0101,0xfb400108,0xe8ed0101,
	0xfb400108,0xe8ef0101,0xfb400108,0xe8f20101,0xfb400108,0xe8f30101,0xfb400108,0xe8f40101,
	0xfb400108,0xe8f60101,0xfb400108,0xe8f70101,0xfb400108,0xe8f80101,0xfb400108,0xe8fb0101,
	0xfb400108,0xe8fd0101,0xfb400108,0xe8fe0101,0xfb400108,0xe8ff0101,0xfb400108,0xe9000101,
	0xfb400108,0xe9020101,0xfb400108,0xe9030101,0xfb400108,0xe9040101,0xfb400108,0xe9060101,
	0xfb400108,0xe9070101,0xfb400108,0xe9080101,0xfb400108,0xe9090101,0xfb400108,0xe90a0101,
	0xfb400108,0xe90c0101,0xfb400108,0xe90f0101,0xfb400108,0xe9110101,0xfb400108,0xe9130101,
	0xfb400108,0xe9140101,0xfb400108,0xe9150101,0xfb400108,0xe9160101,0xfb400108,0xe9170101,
	0xfb400108,0xe9180101,0xfb400108,0xe9190101,0xfb400108,0xe91a0101,0xfb400108,0xe91b0101,
	0xfb400108,0xe91c0101,0xfb400108,0xe91d0101,0xfb400108,0xe91e0101,0xfb400108,0xe9210101,
	0xfb400108,0xe9220101,0xfb400108,0xe9230101,0xfb400108,0xe9250101,0xfb400108,0xe9260101,
	0xfb400108,0xe9270101,0xfb400108,0xe9280101,0xfb400108,0xe9290101,0xfb400108,0xe92a0101,
	0xfb400108,0xe92b0101,0xfb400108,0xe92c0101,0xfb400108,0xe92e0101,0xfb400108,0xe92f0101,
	0xfb400108,0xe9310101,0xfb400108,0xe9320101,0xfb400108,0xe9330101,0xfb400108,0xe9350101,
	0xfb400108,0xe9360101,0xfb400108,0xe9370101,0xfb400108,0xe9380101,0xfb400108,0xe93a0101,
	0xfb400108,0xe93b0101,0xfb400108,0xe93c0101,0xfb400108,0xe93e0101,0xfb400108,0xe9400101,
	0xfb400108,0xe9410101,0xfb400108,0xe9430101,0xfb400108,0xe9440101,0xfb400108,0xe9450101,
	0xfb400108,0xe9460101,0xfb400108,0xe9470101,0xfb400108,0xe9480101,0xfb400108,0xe9490101,
	0xfb400108,0xe94a0101,0xfb400108,0xe94b0101,0xfb400108,0xe94c0101,0xfb400108,0xe94d0101,
	0xfb400108,0xe94e0101,0xfb400108,0xe94f0101,0xfb400108,0xe9500101,0xfb400108,0xe9510101,
	0xfb400108,0xe9520101,0xfb400108,0xe9530101,0xfb400108,0xe9550101,0xfb400108,0xe9560101,
	0xfb400108,0xe9580101,0xfb400108,0xe9590101,0xfb400108,0xe95b0101,0xfb400108,0xe95c0101,
	0xfb400108,0xe95f0101,0xfb400108,0xe9610101,0xfb400108,0xe9620101,0xfb400108,0xe9640101,
	0xfb400108,0xe9650101,0xfb400108,0xe9670101,0xfb400108,0xe9680101,0xfb400108,0xe9690101,
	0xfb400108,0xe96a0101,0xfb400108,0xe96c0101,0xfb400108,0xe96d0101,0xfb400108,0xe96f0101,
	0xfb400108,0xe9700101,0xfb400108,0xe9720101,0xfb400108,0xe9730101,0xfb400108,0xe9740101,
	0xfb400108,0xe9750101,0xfb400108,0xe9760101,0xfb400108,0xe97a0101,0xfb400108,0xe97b0101,
	0xfb400108,0xe97d0101,0xfb400108,0xe97e0101,0xfb400108,0xe97f0101,0xfb400108,0xe9810101,
	0xfb400108,0xe9830101,0xfb400108,0xe9850101,0xfb400108,0xe98a0101,0xfb400108,0xe98b0101,
	0xfb400108,0xe98c0101,0xfb400108,0xe98e0101,0xfb400108,0xe98f0101,0xfb400108,0xe9900101,
	0xfb400108,0xe9910101,0xfb400108,0xe9920101,0xfb400108,0xe9930101,0xfb400108,0xe9960101,
	0xfb400108,0xe9970101,0xfb400108,0xe9990101,0xfb400108,0xe99a0101,0xfb400108,0xe99d0101,
	0xfb400108,0xe99e0101,0xfb400108,0xe99f0101,0xfb400108,0xe9a00101,0xfb400108,0xe9a10101,
	0xfb400108,0xe9a20101,0xfb400108,0xe9a30101,0xfb400108,0xe9a40101,0xfb400108,0xe9a50101,
	0xfb400108,0xe9a60101,0xfb400108,0xe9a90101,0xfb400108,0xe9aa0101,0xfb400108,0xe9ac0101,
	0xfb400108,0xe9ae0101,0xfb400108,0xe9af0101,0xfb400108,0xe9b00101,0xfb400108,0xe9b20101,
	0xfb400108,0xe9b30101,0xfb400108,0xe9b50101,0xfb400108,0xe9b60101,0xfb400108,0xe9b80101,
	0xfb400108,0xe9b90101,0xfb400108,0xe9ba0101,0xfb400108,0xe9bc0101,0xfb400108,0xe9bd0101,
	0xfb400108,0xe9be0101,0xfb400108,0xe9bf0101,0xfb400108,0xe9c00101,0xfb400108,0xe9c20101,
	0xfb400108,0xe9c30101,0xfb400108,0xe9c40101,0xfb400108,0xe9c50101,0xfb400108,0xe9c60101,
	0xfb400108,0xe9c70101,0xfb400108,0xe9c80101,0xfb400108,0xe9c90101,0xfb400108,0xe9cb0101,
	0xfb400108,0xe9cd0101,0xfb400108,0xe9cf0101,0xfb400108,0xe9d10101,0xfb400108,0xe9d20101,
	0xfb400108,0xe9d30101,0xfb400108,0xe9d50101,0xfb400108,0xe9d60101,0xfb400108,0xe9d70101,
	0xfb400108,0xe9d80101,0xfb400108,0xe9d90101,0xfb400108,0xe9da0101,0xfb400108,0xe9dc0101,
	0xfb400108,0xe9dd0101,0xfb400108,0xe9de0101,0xfb400108,0xe9e10101,0xfb400108,0xe9e20101,
	0xfb400108,0xe9e30101,0xfb400108,0xe9e40101,0xfb400108,0xe9e50101,0xfb400108,0xe9e60101,
	0xfb400108,0xe9e70101,0xfb400108,0xe9e80101,0xfb400108,0xe9e90101,0xfb400108,0xe9ea0101,
	0xfb400108,0xe9eb0101,0xfb400108,0xe9ec0101,0xfb400108,0xe9ee0101,0xfb400108,0xe9ef0101,
	0xfb400108,0xe9f00101,0xfb400108,0xe9f10101,0xfb400108,0xe9f30101,0xfb400108,0xe9f40101,
	0xfb400108,0xe9f50101,0xfb400108,0xe9f60101,0xfb400108,0xe9f70101,0xfb400108,0xe9f80101,
	0xfb400108,0xe9f90101,0xfb400108,0xe9fa0101,0xfb400108,0xe9fb0101,0xfb400108,0xe9fc0101,
	0xfb400108,0xe9fe0101,0xfb400108,0xea000101,0xfb400108,0xea010101,0xfb400108,0xea020101,
	0xfb400108,0xea030101,0xfb400108,0xea040101,0xfb400108,0xea050101,0xfb400108,0xea060101,
	0xfb400108,0xea070101,0xfb400108,0xea080101,0xfb400108,0xea090101,0xfb400108,0xea0b0101,
	0xfb400108,0xea0c0101,0xfb400108,0xea0d0101,0xfb400108,0xea0e0101,0xfb400108,0xea0f0101,
	0xfb400108,0xea100101,0xfb400108,0xea110101,0xfb400108,0xea120101,0xfb400108,0xea130101,
	0xfb400108,0xea140101,0xfb400108,0xea150101,0xfb400108,0xea160101,0xfb400108,0xea190101,
	0xfb400108,0xea1a0101,0xfb400108,0xea1b0101,0xfb400108,0xea1c0101,0xfb400108,0xea1d0101,
	0xfb400108,0xea1e0101,0xfb400108,0xea200101,0xfb400108,0xea220101,0xfb400108,0xea230101,
	0xfb400108,0xea240101,0xfb400108,0xea250101,0xfb400108,0xea260101,0xfb400108,0xea270101,
	0xfb400108,0xea290101,0xfb400108,0xea2b0101,0xfb400108,0xea2c0101,0xfb400108,0xea2d0101,
	0xfb400108,0xea2e0101,0xfb400108,0xea300101,0xfb400108,0xea320101,0xfb400108,0xea330101,
	0xfb400108,0xea340101,0xfb400108,0xea360101,0xfb400108,0xea370101,0xfb400108,0xea380101,
	0xfb400108,0xea390101,0xfb400108,0xea3a0101,0xfb400108,0xea3b0101,0xfb400108,0xea3c0101,
	0xfb400108,0xea3f0101,0xfb400108,0xea400101,0xfb400108,0xea410101,0xfb400108,0xea420101,
	0xfb400108,0xea430101,0xfb400108,0xea450101,0xfb400108,0xea460101,0xfb400108,0xea480101,
	0xfb400108,0xea490101,0xfb400108,0xea4a0101,0xfb400108,0xea4b0101,0xfb400108,0xea4c0101,
	0xfb400108,0xea4d0101,0xfb400108,0xea4e0101,0xfb400108,0xea4f0101,0xfb400108,0xea510101,
	0xfb400108,0xea520101,0xfb400108,0xea530101,0xfb400108,0xea540101,0xfb400108,0xea550101,
	0xfb400108,0xea560101,0xfb400108,0xea570101,0xfb400108,0xea5a0101,0xfb400108,0xea5c0101,
	0xfb400108,0xea5d0101,0xfb400108,0xea5e0101,0xfb400108,0xea5f0101,0xfb400108,0xea600101,
	0xfb400108,0xea620101,0xfb400108,0xea630101,0xfb400108,0xea640101,0xfb400108,0xea660101,
	0xfb400108,0xea670101,0xfb400108,0xea680101,0xfb400108,0xea690101,0xfb400108,0xea6a0101,
	0xfb400108,0xea6b0101,0xfb400108,0xea6c0101,0xfb400108,0xea6d0101,0xfb400108,0xea6e0101,
	0xfb400108,0xea6f0101,0xfb400108,0xea700101,0xfb400108,0xea720101,0xfb400108,0xea730101,
	0xfb400108,0xea740101,0xfb400108,0xea750101,0xfb400108,0xea760101,0xfb400108,0xea770101,
	0xfb400108,0xea780101,0xfb400108,0xea7a0101,0xfb400108,0xea7b0101,0xfb400108,0xea7d0101,
	0xfb400108,0xea7e0101,0xfb400108,0xea7f0101,0xfb400108,0xea810101,0xfb400108,0xea820101,
	0xfb400108,0xea830101,0xfb400108,0xea850101,0xfb400108,0xea860101,0xfb400108,0xea870101,
	0xfb400108,0xea880101,0xfb400108,0xea890101,0xfb400108,0xea8a0101,0xfb400108,0xea8b0101,
	0xfb400108,0xea8c0101,0xfb400108,0xea8d0101,0xfb400108,0xea8f0101,0xfb400108,0xea920101,
	0xfb400108,0xea930101,0xfb400108,0xea940101,0xfb400108,0xea950101,0xfb400108,0xea960101,
	0xfb400108,0xea980101,0xfb400108,0xea990101,0xfb400108,0xea9a0101,0xfb400108,0xea9b0101,
	0xfb400108,0xea9c0101,0xfb400108,0xea9d0101,0xfb400108,0xea9e0101,0xfb400108,0xea9f0101,
	0xfb400108,0xeaa10101,0xfb400108,0xeaa20101,0xfb400108,0xeaa30101,0xfb400108,0xeaa40101,
	0xfb400108,0xeaa50101,0xfb400108,0xeaa60101,0xfb400108,0xeaa70101,0xfb400108,0xeaa80101,
	0xfb400108,0xeaaa0101,0xfb400108,0xeaad0101,0xfb400108,0xeaae0101,0xfb400108,0xeaaf0101,
	0xfb400108,0xeab00101,0xfb400108,0xeab10101,0xfb400108,0xeab20101,0xfb400108,0xeab30101,
	0xfb400108,0xeab40101,0xfb400108,0xeab50101,0xfb400108,0xeab60101,0xfb400108,0xeab70101,
	0xfb400108,0xeab80101,0xfb400108,0xeab90101,0xfb400108,0xeaba0101,0xfb400108,0xeabb0101,
	0xfb400108,0xeabc0101,0xfb400108,0xeabd0101,0xfb400108,0xeabe0101,0xfb400108,0xeabf0101,
	0xfb400108,0xeac00101,0xfb400108,0xeac10101,0xfb400108,0xeac20101,0xfb400108,0xeac30101,
	0xfb400108,0xeac40101,0xfb400108,0xeac50101,0xfb400108,0xeac60101,0xfb400108,0xeac70101,
	0xfb400108,0xeac80101,0xfb400108,0xeac90101,0xfb400108,0xeaca0101,0xfb400108,0xeacb0101,
	0xfb400108,0xeacc0101,0xfb400108,0xeacd0101,0xfb400108,0xeace0101,0xfb400108,0xeacf0101,
	0xfb400108,0xead00101,0xfb400108,0xead10101,0xfb400108,0xead20101,0xfb400108,0xead30101,
	0xfb400108,0xead40101,0xfb400108,0xead50101,0xfb400108,0xead60101,0xfb400108,0xead70101,
	0xfb400108,0xead80101,0xfb400108,0xead90101,0xfb400108,0xeada0101,0xfb400108,0xeadb0101,
	0xfb400108,0xeadc0101,0xfb400108,0xeadd0101,0xfb400108,0xeade0101,0xfb400108,0xeadf0101,
	0xfb400108,0xeae00101,0xfb400108,0xeae10101,0xfb400108,0xeae20101,0xfb400108,0xeae30101,
	0xfb400108,0xeae40101,0xfb400108,0xeae50101,0xfb400108,0xeae60101,0xfb400108,0xeae70101,
	0xfb400108,0xeae80101,0xfb400108,0xeae90101,0xfb400108,0xeaea0101,0xfb400108,0xeaeb0101,
	0xfb400108,0xeaec0101,0xfb400108,0xeaed0101,0xfb400108,0xeaee0101,0xfb400108,0xeaef0101,
	0xfb400108,0xeaf00101,0xfb400108,0xeaf10101,0xfb400108,0xeaf20101,0xfb400108,0xeaf30101,
	0xfb400108,0xeaf40101,0xfb400108,0xeaf50101,0xfb400108,0xeaf60101,0xfb400108,0xeaf70101,
	0xfb400108,0xeaf80101,0xfb400108,0xeaf90101,0xfb400108,0xeafa0101,0xfb400108,0xeafb0101,
	0xfb400108,0xeafc0101,0xfb400108,0xeafd0101,0xfb400108,0xeafe0101,0xfb400108,0xeaff0101,
	0xfb400108,0xeb000101,0xfb400108,0xeb010101,0xfb400108,0xeb020101,0xfb400108,0xeb030101,
	0xfb400108,0xeb040101,0xfb400108,0xeb050101,0xfb400108,0xeb060101,0xfb400108,0xeb070101,
	0xfb400108,0xeb080101,0xfb400108,0xeb090101,0xfb400108,0xeb0a0101,0xfb400108,0xeb0b0101,
	0xfb400108,0xeb0c0101,0xfb400108,0xeb0d0101,0xfb400108,0xeb0e0101,0xfb400108,0xeb0f0101,
	0xfb400108,0xeb100101,0xfb400108,0xeb110101,0xfb400108,0xeb120101,0xfb400108,0xeb130101,
	0xfb400108,0xeb140101,0xfb400108,0xeb150101,0xfb400108,0xeb160101,0xfb400108,0xeb170101,
	0xfb400108,0xeb180101,0xfb400108,0xeb190101,0xfb400108,0xeb1a0101,0xfb400108,0xeb1b0101,
	0xfb400108,0xeb1c0101,0xfb400108,0xeb1d0101,0xfb400108,0xeb1e0101,0xfb400108,0xeb1f0101,
	0xfb400108,0xeb250101,0xfb400108,0xeb260101,0xfb400108,0xeb280101,0xfb400108,0xeb290101,
	0xfb400108,0xeb2a0101,0xfb400108,0xeb2b0101,0xfb400108,0xeb2c0101,0xfb400108,0xeb2d0101,
	0xfb400108,0xeb2e0101,0xfb400108,0xeb2f0101,0xfb400108,0xeb300101,0xfb400108,0xeb310101,
	0xfb400108,0xeb330101,0xfb400108,0xeb340101,0xfb400108,0xeb350101,0xfb400108,0xeb360101,
	0xfb400108,0xeb380101,0xfb400108,0xeb3b0101,0xfb400108,0xeb3c0101,0xfb400108,0xeb3d0101,
	0xfb400108,0xeb3f0101,0xfb400108,0xeb400101,0xfb400108,0xeb410101,0xfb400108,0xeb420101,
	0xfb400108,0xeb440101,0xfb400108,0xeb450101,0xfb400108,0xeb480101,0xfb400108,0xeb4a0101,
	0xfb400108,0xeb4b0101,0xfb400108,0xeb4d0101,0xfb400108,0xeb4e0101,0xfb400108,0xeb4f0101,
	0xfb400108,0xeb500101,0xfb400108,0xeb510101,0xfb400108,0xeb520101,0xfb400108,0xeb530101,
	0xfb400108,0xeb540101,0xfb400108,0xeb550101,0xfb400108,0xeb560101,0xfb400108,0xeb570101,
	0xfb400108,0xeb580101,0xfb400108,0xeb5a0101,0xfb400108,0xeb5b0101,0xfb400108,0xeb5c0101,
	0xfb400108,0xeb5d0101,0xfb400108,0xeb5e0101,0xfb400108,0xeb5f0101,0xfb400108,0xeb600101,
	0xfb400108,0xeb610101,0xfb400108,0xeb680101,0xfb400108,0xeb690101,0xfb400108,0xeb6b0101,
	0xfb400108,0xeb6c0101,0xfb400108,0xeb6d0101,0xfb400108,0xeb6e0101,0xfb400108,0xeb6f0101,
	0xfb400108,0xeb700101,0xfb400108,0xeb710101,0xfb400108,0xeb720101,0xfb400108,0xeb730101,
	0xfb400108,0xeb740101,0xfb400108,0xeb750101,0xfb400108,0xeb760101,0xfb400108,0xeb770101,
	0xfb400108,0xeb780101,0xfb400108,0xeb7a0101,0xfb400108,0xeb7d0101,0xfb400108,0xeb7e0101,
	0xfb400108,0xeb7f0101,0xfb400108,0xeb800101,0xfb400108,0xeb850101,0xfb400108,0xeb880101,
	0xfb400108,0xeb8c0101,0xfb400108,0xeb8e0101,0xfb400108,0xeb8f0101,0xfb400108,0xeb900101,
	0xfb400108,0xeb910101,0xfb400108,0xeb940101,0xfb400108,0xeb950101,0xfb400108,0xeb970101,
	0xfb400108,0xeb980101,0xfb400108,0xeb990101,0xfb400108,0xeb9c0101,0xfb400108,0xeb9d0101,
	0xfb400108,0xeb9e0101,0xfb400108,0xeb9f0101,0xfb400108,0xeba00101,0xfb400108,0xeba20101,
	0xfb400108,0xeba30101,0xfb400108,0xeba40101,0xfb400108,0xeba50101,0xfb400108,0xeba60101,
	0xfb400108,0xeba70101,0xfb400108,0xeba80101,0xfb400108,0xeba90101,0xfb400108,0xebab0101,
	0xfb400108,0xebac0101,0xfb400108,0xebad0101,0xfb400108,0xebae0101,0xfb400108,0xebaf0101,
	0xfb400108,0xebb00101,0xfb400108,0xebb10101,0xfb400108,0xebb20101,0xfb400108,0xebb60101,
	0xfb400108,0xebb80101,0xfb400108,0xebb90101,0xfb400108,0xebba0101,0xfb400108,0xebbb0101,
	0xfb400108,0xebbc0101,0xfb400108,0xebbd0101,0xfb400108,0xebbe0101,0xfb400108,0xebc00101,
	0xfb400108,0xebc30101,0xfb400108,0xebc40101,0xfb400108,0xebc60101,0xfb400108,0xebc70101,
	0xfb400108,0xebc80101,0xfb400108,0xebc90101,0xfb400108,0xebca0101,0xfb400108,0xebcc0101,
	0xfb400108,0xebce0101,0xfb400108,0xebd00101,0xfb400108,0xebd10101,0xfb400108,0xebd80101,
	0xfb400108,0xebda0101,0xfb400108,0xebdc0101,0xfb400108,0xebdd0101,0xfb400108,0xebde0101,
	0xfb400108,0xebdf0101,0xfb400108,0xebe00101,0xfb400108,0xebe20101,0xfb400108,0xebe30101,
	0xfb400108,0xebe40101,0xfb400108,0xebe50101,0xfb400108,0xebe60101,0xfb400108,0xebe70101,
	0xfb400108,0xebe80101,0xfb400108,0xebe90101,0xfb400108,0xebec0101,0xfb400108,0xebed0101,
	0xfb400108,0xebee0101,0xfb400108,0xebf00101,0xfb400108,0xebf10101,0xfb400108,0xebf20101,
	0xfb400108,0xebf40101,0xfb400108,0xebf60101,0xfb400108,0xebf70101,0xfb400108,0xebf80101,
	0xfb400108,0xebfa0101,0xfb400108,0xebfb0101,0xfb400108,0xebfc0101,0xfb400108,0xebfe0101,
	0xfb400108,0xebff0101,0xfb400108,0xec000101,0xfb400108,0xec010101,0xfb400108,0xec020101,
	0xfb400108,0xec030101,0xfb400108,0xec040101,0xfb400108,0xec080101,0xfb400108,0xec090101,
	0xfb400108,0xec0a0101,0xfb400108,0xec0b0101,0xfb400108,0xec0c0101,0xfb400108,0xec0e0101,
	0xfb400108,0xec120101,0xfb400108,0xec170101,0xfb400108,0xec1c0101,0xfb400108,0xec1d0101,
	0xfb400108,0xec1e0101,0xfb400108,0xec200101,0xfb400108,0xec230101,0xfb400108,0xec250101,
	0xfb400108,0xec2b0101,0xfb400108,0xec2c0101,0xfb400108,0xec2d0101,0xfb400108,0xec310101,
	0xfb400108,0xec330101,0xfb400108,0xec360101,0xfb400108,0xec370101,0xfb400108,0xec390101,
	0xfb400108,0xec3a0101,0xfb400108,0xec3b0101,0xfb400108,0xec3c0101,0xfb400108,0xec3e0101,
	0xfb400108,0xec3f0101,0xfb400108,0xec430101,0xfb400108,0xec440101,0xfb400108,0xec450101,
	0xfb400108,0xec480101,0xfb400108,0xec4b0101,0xfb400108,0xec4c0101,0xfb400108,0xec4d0101,
	0xfb400108,0xec4e0101,0xfb400108,0xec4f0101,0xfb400108,0xec510101,0xfb400108,0xec520101,
	0xfb400108,0xec530101,0xfb400108,0xec560101,0xfb400108,0xec580101,0xfb400108,0xec590101,
	0xfb400108,0xec5a0101,0xfb400108,0xec620101,0xfb400108,0xec630101,0xfb400108,0xec650101,
	0xfb400108,0xec660101,0xfb400108,0xec670101,0xfb400108,0xec6b0101,0xfb400108,0xec6c0101,
	0xfb400108,0xec6d0101,0xfb400108,0xec6e0101,0xfb400108,0xec6f0101,0xfb400108,0xec710101,
	0xfb400108,0xec730101,0xfb400108,0xec750101,0xfb400108,0xec770101,0xfb400108,0xec780101,
	0xfb400108,0xec7a0101,0xfb400108,0xec7b0101,0xfb400108,0xec7c0101,0xfb400108,0xec7f0101,
	0xfb400108,0xec800101,0xfb400108,0xec840101,0xfb400108,0xec870101,0xfb400108,0xec8a0101,
	0xfb400108,0xec8b0101,0xfb400108,0xec8d0101,0xfb400108,0xec8e0101,0xfb400108,0xec910101,
	0xfb400108,0xec920101,0xfb400108,0xec950101,0xfb400108,0xec960101,0xfb400108,0xec970101,
	0xfb400108,0xec980101,0xfb400108,0xec9a0101,0xfb400108,0xec9c0101,0xfb400108,0xec9d0101,
	0xfb400108,0xec9e0101,0xfb400108,0xeca00101,0xfb400108,0xeca20101,0xfb400108,0xeca80101,
	0xfb400108,0xecac0101,0xfb400108,0xecaf0101,0xfb400108,0xecb00101,0xfb400108,0xecb40101,
	0xfb400108,0xecb50101,0xfb400108,0xecb60101,0xfb400108,0xecb70101,0xfb400108,0xecba0101,
	0xfb400108,0xecc00101,0xfb400108,0xecc10101,0xfb400108,0xecc20101,0xfb400108,0xecc30101,
	0xfb400108,0xecc60101,0xfb400108,0xecc70101,0xfb400108,0xecc80101,0xfb400108,0xeccb0101,
	0xfb400108,0xeccd0101,0xfb400108,0xecce0101,0xfb400108,0xeccf0101,0xfb400108,0xecd10101,
	0xfb400108,0xecd20101,0xfb400108,0xecd80101,0xfb400108,0xecd90101,0xfb400108,0xecda0101,
	0xfb400108,0xecdc0101,0xfb400108,0xecdd0101,0xfb400108,0xecdf0101,0xfb400108,0xece40101,
	0xfb400108,0xece60101,0xfb400108,0xece70101,0xfb400108,0xece90101,0xfb400108,0xecec0101,
	0xfb400108,0xeced0101,0xfb400108,0xecf20101,0xfb400108,0xecf40101,0xfb400108,0xecf90101,
	0xfb400108,0xecff0101,0xfb400108,0xed000101,0xfb400108,0xed020101,0xfb400108,0xed030101,
	0xfb400108,0xed050101,0xfb400108,0xed060101,0xfb400108,0xed080101,0xfb400108,0xed090101,
	0xfb400108,0xed0a0101,0xfb400108,0xed0d0101,0xfb400108,0xed0f0101,0xfb400108,0xed100101,
	0xfb400108,0xed110101,0xfb400108,0xed130101,0xfb400108,0xed140101,0xfb400108,0xed150101,
	0xfb400108,0xed160101,0xfb400108,0xed180101,0xfb400108,0xed1c0101,0xfb400108,0xed1d0101,
	0xfb400108,0xed1f0101,0xfb400108,0xed200101,0xfb400108,0xed210101,0xfb400108,0xed220101,
	0xfb400108,0xed230101,0xfb400108,0xed240101,0xfb400108,0xed260101,0xfb400108,0xed280101,
	0xfb400108,0xed290101,0xfb400108,0xed2c0101,0xfb400108,0xed2d0101,0xfb400108,0xed2f0101,
	0xfb400108,0xed300101,0xfb400108,0xed340101,0xfb400108,0xed360101,0xfb400108,0xed370101,
	0xfb400108,0xed380101,0xfb400108,0xed3a0101,0xfb400108,0xed3f0101,0xfb400108,0xed400101,
	0xfb400108,0xed420101,0xfb400108,0xed440101,0xfb400108,0xed490101,0xfb400108,0xed4c0101,
	0xfb400108,0xed500101,0xfb400108,0xed550101,0xfb400108,0xed560101,0xfb400108,0xed570101,
	0xfb400108,0xed580101,0xfb400108,0xed5b0101,0xfb400108,0xed5d0101,0xfb400108,0xed5f0101,
	0xfb400108,0xed610101,0xfb400108,0xed620101,0xfb400108,0xed640101,0xfb400108,0xed650101,
	0xfb400108,0xed670101,0xfb400108,0xed680101,0xfb400108,0xed6b0101,0xfb400108,0xed6c0101,
	0xfb400108,0xed6d0101,0xfb400108,0xed700101,0xfb400108,0xed710101,0xfb400108,0xed720101,
	0xfb400108,0xed730101,0xfb400108,0xed750101,0xfb400108,0xed760101,0xfb400108,0xed790101,
	0xfb400108,0xed7a0101,0xfb400108,0xed7b0101,0xfb400108,0xed7d0101,0xfb400108,0xed7e0101,
	0xfb400108,0xed7f0101,0xfb400108,0xed800101,0xfb400108,0xed810101,0xfb400108,0xed830101,
	0xfb400108,0xed840101,0xfb400108,0xed860101,0xfb400108,0xed870101,0xfb400108,0xed8a0101,
	0xfb400108,0xed8b0101,0xfb400108,0xed8d0101,0xfb400108,0xed8f0101,0xfb400108,0xed900101,
	0xfb400108,0xed920101,0xfb400108,0xed960101,0xfb400108,0xed970101,0xfb400108,0xed980101,
	0xfb400108,0xed990101,0xfb400108,0xed9a0101,0xfb400108,0xed9c0101,0xfb400108,0xeda20101,
	0xfb400108,0xeda50101,0xfb400108,0xedac0101,0xfb400108,0xedad0101,0xfb400108,0xedb00101,
	0xfb400108,0xedb10101,0xfb400108,0xedb30101,0xfb400108,0xedb40101,0xfb400108,0xedb60101,
	0xfb400108,0xedb70101,0xfb400108,0xedb90101,0xfb400108,0xedba0101,0xfb400108,0xedbb0101,
	0xfb400108,0xedbc0101,0xfb400108,0xedbd0101,0xfb400108,0xedbe0101,0xfb400108,0xedc10101,
	0xfb400108,0xedc20101,0xfb400108,0xedc30101,0xfb400108,0xedc80101,0xfb400108,0xedc90101,
	0xfb400108,0xedca0101,0xfb400108,0xedcd0101,0xfb400108,0xedce0101,0xfb400108,0xedcf0101,
	0xfb400108,0xedd00101,0xfb400108,0xedd20101,0xfb400108,0xedd30101,0xfb400108,0xedd40101,
	0xfb400108,0xedd50101,0xfb400108,0xedd70101,0xfb400108,0xedda0101,0xfb400108,0xeddb0101,
	0xfb400108,0xeddc0101,0xfb400108,0xeddf0101,0xfb400108,0xede20101,0xfb400108,0xede30101,
	0xfb400108,0xede50101,0xfb400108,0xede70101,0xfb400108,0xede80101,0xfb400108,0xede90101,
	0xfb400108,0xedea0101,0xfb400108,0xeded0101,0xfb400108,0xedef0101,0xfb400108,0xedf00101,
	0xfb400108,0xedf20101,0xfb400108,0xedf40101,0xfb400108,0xedf50101,0xfb400108,0xedf60101,
	0xfb400108,0xedf80101,0xfb400108,0xedfa0101,0xfb400108,0xedfd0101,0xfb400108,0xedfe0101,
	0xfb400108,0xedff0101,0xfb400108,0xee000101,0xfb400108,0xee010101,0xfb400108,0xee020101,
	0xfb400108,0xee030101,0xfb400108,0xee040101,0xfb400108,0xee060101,0xfb400108,0xee070101,
	0xfb400108,0xee080101,0xfb400108,0xee090101,0xfb400108,0xee0b0101,0xfb400108,0xee0f0101,
	0xfb400108,0xee120101,0xfb400108,0xee130101,0xfb400108,0xee150101,0xfb400108,0xee180101,
	0xfb400108,0xee190101,0xfb400108,0xee1b0101,0xfb400108,0xee1c0101,0xfb400108,0xee1e0101,
	0xfb400108,0xee1f0101,0xfb400108,0xee220101,0xfb400108,0xee260101,0xfb400108,0xee270101,
	0xfb400108,0xee280101,0xfb400108,0xee2a0101,0xfb400108,0xee2c0101,0xfb400108,0xee2e0101,
	0xfb400108,0xee300101,0xfb400108,0xee310101,0xfb400108,0xee330101,0xfb400108,0xee350101,
	0xfb400108,0xee360101,0xfb400108,0xee370101,0xfb400108,0xee390101,0xfb400108,0xee3b0101,
	0xfb400108,0xee3c0101,0xfb400108,0xee3d0101,0xfb400108,0xee3e0101,0xfb400108,0xee3f0101,
	0xfb400108,0xee400101,0xfb400108,0xee410101,0xfb400108,0xee420101,0xfb400108,0xee450101,
	0xfb400108,0xee460101,0xfb400108,0xee470101,0xfb400108,0xee480101,0xfb400108,0xee490101,
	0xfb400108,0xee4a0101,0xfb400108,0xee4b0101,0xfb400108,0xee4c0101,0xfb400108,0xee4f0101,
	0xfb400108,0xee500101,0xfb400108,0xee510101,0xfb400108,0xee520101,0xfb400108,0xee550101,
	0xfb400108,0xee570101,0xfb400108,0xee590101,0xfb400108,0xee5a0101,0xfb400108,0xee5c0101,
	0xfb400108,0xee5d0101,0xfb400108,0xee5e0101,0xfb400108,0xee600101,0xfb400108,0xee610101,
	0xfb400108,0xee620101,0xfb400108,0xee630101,0xfb400108,0xee640101,0xfb400108,0xee650101,
	0xfb400108,0xee660101,0xfb400108,0xee670101,0xfb400108,0xee680101,0xfb400108,0xee690101,
	0xfb400108,0xee6a0101,0xfb400108,0xee6c0101,0xfb400108,0xee6d0101,0xfb400108,0xee6f0101,
	0xfb400108,0xee700101,0xfb400108,0xee710101,0xfb400108,0xee720101,0xfb400108,0xee730101,
	0xfb400108,0xee740101,0xfb400108,0xee750101,0xfb400108,0xee760101,0xfb400108,0xee770101,
	0xfb400108,0xee780101,0xfb400108,0xee790101,0xfb400108,0xee7a0101,0xfb400108,0xee7b0101,
	0xfb400108,0xee7c0101,0xfb400108,0xee7d0101,0xfb400108,0xee800101,0xfb400108,0xee810101,
	0xfb400108,0xee820101,0xfb400108,0xee840101,0xfb400108,0xee870101,0xfb400108,0xee880101,
	0xfb400108,0xee8a0101,0xfb400108,0xee8b0101,0xfb400108,0xee8c0101,0xfb400108,0xee8d0101,
	0xfb400108,0xee8e0101,0xfb400108,0xee910101,0xfb400108,0xee920101,0xfb400108,0xee930101,
	0xfb400108,0xee940101,0xfb400108,0xee950101,0xfb400108,0xee960101,0xfb400108,0xee970101,
	0xfb400108,0xee990101,0xfb400108,0xee9a0101,0xfb400108,0xee9b0101,0xfb400108,0xee9d0101,
	0xfb400108,0xee9e0101,0xfb400108,0xeea00101,0xfb400108,0xeea10101,0xfb400108,0xeea30101,
	0xfb400108,0xeea40101,0xfb400108,0xeea60101,0xfb400108,0xeea80101,0xfb400108,0xeea90101,
	0xfb400108,0xeeab0101,0xfb400108,0xeeac0101,0xfb400108,0xeead0101,0xfb400108,0xeeae0101,
	0xfb400108,0xeeb00101,0xfb400108,0xeeb30101,0xfb400108,0xeeb50101,0xfb400108,0xeeb80101,
	0xfb400108,0xeeb90101,0xfb400108,0xeebc0101,0xfb400108,0xeebe0101,0xfb400108,0xeebf0101,
	0xfb400108,0xeec00101,0xfb400108,0xeec30101,0xfb400108,0xeec40101,0xfb400108,0xeec50101,
	0xfb400108,0xeec60101,0xfb400108,0xeec80101,0xfb400108,0xeec90101,0xfb400108,0xeeca0101,
	0xfb400108,0xeecc0101,0xfb400108,0xeecd0101,0xfb400108,0xeece0101,0xfb400108,0xeed00101,
	0xfb400108,0xeed20101,0xfb400108,0xeed60101,0xfb400108,0xeed80101,0xfb400108,0xeed90101,
	0xfb400108,0xeedb0101,0xfb400108,0xeedc0101,0xfb400108,0xeedd0101,0xfb400108,0xeee30101,
	0xfb400108,0xeee70101,0xfb400108,0xeeea0101,0xfb400108,0xeeeb0101,0xfb400108,0xeeec0101,
	0xfb400108,0xeeed0101,0xfb400108,0xeeee0101,0xfb400108,0xeeef0101,0xfb400108,0xeef00101,
	0xfb400108,0xeef10101,0xfb400108,0xeef20101,0xfb400108,0xeef30101,0xfb400108,0xeef50101,
	0xfb400108,0xeef60101,0xfb400108,0xeef70101,0xfb400108,0xeef80101,0xfb400108,0xeefa0101,
	0xfb400108,0xeefb0101,0xfb400108,0xeefc0101,0xfb400108,0xeefd0101,0xfb400108,0xeefe0101,
	0xfb400108,0xeeff0101,0xfb400108,0xef000101,0xfb400108,0xef010101,0xfb400108,0xef030101,
	0xfb400108,0xef040101,0xfb400108,0xef050101,0xfb400108,0xef070101,0xfb400108,0xef080101,
	0xfb400108,0xef0a0101,0xfb400108,0xef0b0101,0xfb400108,0xef0c0101,0xfb400108,0xef0d0101,
	0xfb400108,0xef0e0101,0xfb400108,0xef100101,0xfb400108,0xef110101,0xfb400108,0xef120101,
	0xfb400108,0xef160101,0xfb400108,0xef170101,0xfb400108,0xef180101,0xfb400108,0xef190101,
	0xfb400108,0xef1a0101,0xfb400108,0xef1b0101,0xfb400108,0xef1c0101,0xfb400108,0xef1d0101,
	0xfb400108,0xef1e0101,0xfb400108,0xef1f0101,0xfb400108,0xef210101,0xfb400108,0xef220101,
	0xfb400108,0xef230101,0xfb400108,0xef250101,0xfb400108,0xef260101,0xfb400108,0xef270101,
	0xfb400108,0xef280101,0xfb400108,0xef2c0101,0xfb400108,0xef2e0101,0xfb400108,0xef300101,
	0xfb400108,0xef320101,0xfb400108,0xef340101,0xfb400108,0xef350101,0xfb400108,0xef370101,
	0xfb400108,0xef380101,0xfb400108,0xef390101,0xfb400108,0xef3a0101,0xfb400108,0xef3b0101,
	0xfb400108,0xef3c0101,0xfb400108,0xef3d0101,0xfb400108,0xef3f0101,0xfb400108,0xef400101,
	0xfb400108,0xef410101,0xfb400108,0xef420101,0xfb400108,0xef430101,0xfb400108,0xef440101,
	0xfb400108,0xef450101,0xfb400108,0xef480101,0xfb400108,0xef490101,0xfb400108,0xef4a0101,
	0xfb400108,0xef4c0101,0xfb400108,0xef4e0101,0xfb400108,0xef4f0101,0xfb400108,0xef500101,
	0xfb400108,0xef510101,0xfb400108,0xef520101,0xfb400108,0xef530101,0xfb400108,0xef540101,
	0xfb400108,0xef550101,0xfb400108,0xef560101,0xfb400108,0xef570101,0xfb400108,0xef590101,
	0xfb400108,0xef5a0101,0xfb400108,0xef5b0101,0xfb400108,0xef5d0101,0xfb400108,0xef5f0101,
	0xfb400108,0xef600101,0xfb400108,0xef610101,0xfb400108,0xef630101,0xfb400108,0xef640101,
	0xfb400108,0xef650101,0xfb400108,0xef670101,0xfb400108,0xef680101,0xfb400108,0xef690101,
	0xfb400108,0xef6a0101,0xfb400108,0xef6b0101,0xfb400108,0xef6c0101,0xfb400108,0xef6f0101,
	0xfb400108,0xef700101,0xfb400108,0xef710101,0xfb400108,0xef730101,0xfb400108,0xef750101,
	0xfb400108,0xef760101,0xfb400108,0xef770101,0xfb400108,0xef790101,0xfb400108,0xef7b0101,
	0xfb400108,0xef7d0101,0xfb400108,0xef7e0101,0xfb400108,0xef7f0101,0xfb400108,0xef800101,
	0xfb400108,0xef810101,0xfb400108,0xef820101,0xfb400108,0xef830101,0xfb400108,0xef850101,
	0xfb400108,0xef860101,0xfb400108,0xef870101,0xfb400108,0xef8a0101,0xfb400108,0xef8b0101,
	0xfb400108,0xef8f0101,0xfb400108,0xef900101,0xfb400108,0xef910101,0xfb400108,0xef920101,
	0xfb400108,0xef930101,0xfb400108,0xef940101,0xfb400108,0xef950101,0xfb400108,0xef960101,
	0xfb400108,0xef970101,0xfb400108,0xef980101,0xfb400108,0xef990101,0xfb400108,0xef9a0101,
	0xfb400108,0xef9b0101,0xfb400108,0xef9d0101,0xfb400108,0xef9e0101,0xfb400108,0xef9f0101,
	0xfb400108,0xefa00101,0xfb400108,0xefa20101,0xfb400108,0xefa30101,0xfb400108,0xefa40101,
	0xfb400108,0xefa50101,0xfb400108,0xefa60101,0xfb400108,0xefa80101,0xfb400108,0xefa90101,
	0xfb400108,0xefaa0101,0xfb400108,0xefab0101,0xfb400108,0xefac0101,0xfb400108,0xefad0101,
	0xfb400108,0xefae0101,0xfb400108,0xefaf0101,0xfb400108,0xefb00101,0xfb400108,0xefb10101,
	0xfb400108,0xefb20101,0xfb400108,0xefb40101,0xfb400108,0xefb50101,0xfb400108,0xefb70101,
	0xfb400108,0xefb80101,0xfb400108,0xefba0101,0xfb400108,0xefbb0101,0xfb400108,0xefbc0101,
	0xfb400108,0xefbd0101,0xfb400108,0xefbe0101,0xfb400108,0xefbf0101,0xfb400108,0xefc10101,
	0xfb400108,0xefc30101,0xfb400108,0xefc40101,0xfb400108,0xefc50101,0xfb400108,0xefc60101,
	0xfb400108,0xefc70101,0xfb400108,0xefc80101,0xfb400108,0xefca0101,0xfb400108,0xefcb0101,
	0xfb400108,0xefcc0101,0xfb400108,0xefcd0101,0xfb400108,0xefce0101,0xfb400108,0xefcf0101,
	0xfb400108,0xefd00101,0xfb400108,0xefd30101,0xfb400108,0xefd40101,0xfb400108,0xefd50101,
	0xfb400108,0xefd60101,0xfb400108,0xefd70101,0xfb400108,0xefd80101,0xfb400108,0xefd90101,
	0xfb400108,0xefda0101,0xfb400108,0xefdb0101,0xfb400108,0xefdc0101,0xfb400108,0xefdd0101,
	0xfb400108,0xefdf0101,0xfb400108,0xefe20101,0xfb400108,0xefe30101,0xfb400108,0xefe40101,
	0xfb400108,0xefe50101,0xfb400108,0xefe60101,0xfb400108,0xefe70101,0xfb400108,0xefe80101,
	0xfb400108,0xefe90101,0xfb400108,0xefea0101,0xfb400108,0xefeb0101,0xfb400108,0xefec0101,
	0xfb400108,0xefed0101,0xfb400108,0xeff00101,0xfb400108,0xeff10101,0xfb400108,0xeff20101,
	0xfb400108,0xeff30101,0xfb400108,0xeff40101,0xfb400108,0xeff50101,0xfb400108,0xeff60101,
	0xfb400108,0xeff70101,0xfb400108,0xeff80101,0xfb400108,0xeff90101,0xfb400108,0xeffa0101,
	0xfb400108,0xeffb0101,0xfb400108,0xeffc0101,0xfb400108,0xeffd0101,0xfb400108,0xeffe0101,
	0xfb400108,0xefff0101,0xfb400108,0xf0000101,0xfb400108,0xf0010101,0xfb400108,0xf0020101,
	0xfb400108,0xf0030101,0xfb400108,0xf0040101,0xfb400108,0xf0050101,0xfb400108,0xf0060101,
	0xfb400108,0xf0070101,0xfb400108,0xf0080101,0xfb400108,0xf0090101,0xfb400108,0xf00a0101,
	0xfb400108,0xf00b0101,0xfb400108,0xf00c0101,0xfb400108,0xf00d0101,0xfb400108,0xf00e0101,
	0xfb400108,0xf00f0101,0xfb400108,0xf0100101,0xfb400108,0xf0120101,0xfb400108,0xf0130101,
	0xfb400108,0xf0140101,0xfb400108,0xf0150101,0xfb400108,0xf0160101,0xfb400108,0xf0170101,
	0xfb400108,0xf0180101,0xfb400108,0xf0190101,0xfb400108,0xf01c0101,0xfb400108,0xf01d0101,
	0xfb400108,0xf01e0101,0xfb400108,0xf01f0101,0xfb400108,0xf0200101,0xfb400108,0xf0210101,
	0xfb400108,0xf0220101,0xfb400108,0xf0240101,0xfb400108,0xf0250101,0xfb400108,0xf0260101,
	0xfb400108,0xf0270101,0xfb400108,0xf0280101,0xfb400108,0xf0290101,0xfb400108,0xf02a0101,
	0xfb400108,0xf02b0101,0xfb400108,0xf02c0101,0xfb400108,0xf02d0101,0xfb400108,0xf02e0101,
	0xfb400108,0xf02f0101,0xfb400108,0xf0300101,0xfb400108,0xf0310101,0xfb400108,0xf0320101,
	0xfb400108,0xf0330101,0xfb400108,0xf0340101,0xfb400108,0xf0360101,0xfb400108,0xf0370101,
	0xfb400108,0xf0380101,0xfb400108,0xf03a0101,0xfb400108,0xf03b0101,0xfb400108,0xf03c0101,
	0xfb400108,0xf03d0101,0xfb400108,0xf03e0101,0xfb400108,0xf03f0101,0xfb400108,0xf0400101,
	0xfb400108,0xf0410101,0xfb400108,0xf0420101,0xfb400108,0xf0430101,0xfb400108,0xf0440101,
	0xfb400108,0xf0450101,0xfb400108,0xf0460101,0xfb400108,0xf0470101,0xfb400108,0xf0480101,
	0xfb400108,0xf0490101,0xfb400108,0xf04a0101,0xfb400108,0xf04b0101,0xfb400108,0xf04d0101,
	0xfb400108,0xf04e0101,0xfb400108,0xf0500101,0xfb400108,0xf0510101,0xfb400108,0xf0520101,
	0xfb400108,0xf0530101,0xfb400108,0xf0540101,0xfb400108,0xf0550101,0xfb400108,0xf0560101,
	0xfb400108,0xf0570101,0xfb400108,0xf0580101,0xfb400108,0xf0590101,0xfb400108,0xf05a0101,
	0xfb400108,0xf05b0101,0xfb400108,0xf05c0101,0xfb400108,0xf05d0101,0xfb400108,0xf05f0101,
	0xfb400108,0xf0600101,0xfb400108,0xf0610101,0xfb400108,0xf0620101,0xfb400108,0xf0630101,
	0xfb400108,0xf0640101,0xfb400108,0xf0650101,0xfb400108,0xf0660101,0xfb400108,0xf0670101,
	0xfb400108,0xf0680101,0xfb400108,0xf0690101,0xfb400108,0xf06a0101,0xfb400108,0xf06e0101,
	0xfb400108,0xf0710101,0xfb400108,0xf0720101,0xfb400108,0xf0730101,0xfb400108,0xf0740101,
	0xfb400108,0xf0770101,0xfb400108,0xf0790101,0xfb400108,0xf07a0101,0xfb400108,0xf07b0101,
	0xfb400108,0xf07d0101,0xfb400108,0xf0810101,0xfb400108,0xf0820101,0xfb400108,0xf0830101,
	0xfb400108,0xf0840101,0xfb400108,0xf0860101,0xfb400108,0xf0870101,0xfb400108,0xf0880101,
	0xfb400108,0xf08b0101,0xfb400108,0xf08c0101,0xfb400108,0xf08d0101,0xfb400108,0xf08f0101,
	0xfb400108,0xf0900101,0xfb400108,0xf0910101,0xfb400108,0xf0930101,0xfb400108,0xf0970101,
	0xfb400108,0xf0980101,0xfb400108,0xf09a0101,0xfb400108,0xf09b0101,0xfb400108,0xf09e0101,
	0xfb400108,0xf09f0101,0xfb400108,0xf0a00101,0xfb400108,0xf0a10101,0xfb400108,0xf0a20101,
	0xfb400108,0xf0a30101,0xfb400108,0xf0a40101,0xfb400108,0xf0a50101,0xfb400108,0xf0a60101,
	0xfb400108,0xf0a70101,0xfb400108,0xf0a80101,0xfb400108,0xf0a90101,0xfb400108,0xf0aa0101,
	0xfb400108,0xf0b00101,0xfb400108,0xf0b20101,0xfb400108,0xf0b40101,0xfb400108,0xf0b50101,
	0xfb400108,0xf0b60101,0xfb400108,0xf0ba0101,0xfb400108,0xf0be0101,0xfb400108,0xf0bf0101,
	0xfb400108,0xf0c40101,0xfb400108,0xf0c50101,0xfb400108,0xf0c60101,0xfb400108,0xf0c70101,
	0xfb400108,0xf0c90101,0xfb400108,0xf0cb0101,0xfb400108,0xf0cc0101,0xfb400108,0xf0cd0101,
	0xfb400108,0xf0ce0101,0xfb400108,0xf0cf0101,0xfb400108,0xf0d00101,0xfb400108,0xf0d10101,
	0xfb400108,0xf0d20101,0xfb400108,0xf0d30101,0xfb400108,0xf0d40101,0xfb400108,0xf0d50101,
	0xfb400108,0xf0d60101,0xfb400108,0xf0d70101,0xfb400108,0xf0da0101,0xfb400108,0xf0dc0101,
	0xfb400108,0xf0dd0101,0xfb400108,0xf0de0101,0xfb400108,0xf0e00101,0xfb400108,0xf0e10101,
	0xfb400108,0xf0e20101,0xfb400108,0xf0e30101,0xfb400108,0xf0e50101,0xfb400108,0xf0ea0101,
	0xfb400108,0xf0ee0101,0xfb400108,0xf0f00101,0xfb400108,0xf0f10101,0xfb400108,0xf0f20101,
	0xfb400108,0xf0f30101,0xfb400108,0xf0f40101,0xfb400108,0xf0f50101,0xfb400108,0xf0f60101,
	0xfb400108,0xf0f80101,0xfb400108,0xf0fa0101,0xfb400108,0xf0fb0101,0xfb400108,0xf0fc0101,
	0xfb400108,0xf0fe0101,0xfb400108,0xf0ff0101,0xfb400108,0xf1000101,0xfb400108,0xf1010101,
	0xfb400108,0xf1020101,0xfb400108,0xf1030101,0xfb400108,0xf1040101,0xfb400108,0xf1050101,
	0xfb400108,0xf1060101,0xfb400108,0xf1070101,0xfb400108,0xf1080101,0xfb400108,0xf10b0101,
	0xfb400108,0xf10c0101,0xfb400108,0xf10d0101,0xfb400108,0xf10e0101,0xfb400108,0xf10f0101,
	0xfb400108,0xf1110101,0xfb400108,0xf1120101,0xfb400108,0xf1140101,0xfb400108,0xf1170101,
	0xfb400108,0xf11b0101,0xfb400108,0xf11c0101,0xfb400108,0xf11d0101,0xfb400108,0xf11e0101,
	0xfb400108,0xf11f0101,0xfb400108,0xf1200101,0xfb400108,0xf1210101,0xfb400108,0xf1220101,
	0xfb400108,0xf1230101,0xfb400108,0xf1240101,0xfb400108,0xf1250101,0xfb400108,0xf1270101,
	0xfb400108,0xf1280101,0xfb400108,0xf1290101,0xfb400108,0xf12a0101,0xfb400108,0xf12b0101,
	0xfb400108,0xf12c0101,0xfb400108,0xf12d0101,0xfb400108,0xf12e0101,0xfb400108,0xf1320101,
	0xfb400108,0xf1330101,0xfb400108,0xf1340101,0xfb400108,0xf1350101,0xfb400108,0xf1370101,
	0xfb400108,0xf1380101,0xfb400108,0xf1390101,0xfb400108,0xf13a0101,0xfb400108,0xf13b0101,
	0xfb400108,0xf13c0101,0xfb400108,0xf13d0101,0xfb400108,0xf13e0101,0xfb400108,0xf13f0101,
	0xfb400108,0xf1400101,0xfb400108,0xf1410101,0xfb400108,0xf1420101,0xfb400108,0xf1430101,
	0xfb400108,0xf1440101,0xfb400108,0xf1460101,0xfb400108,0xf1470101,0xfb400108,0xf1480101,
	0xfb400108,0xf1490101,0xfb400108,0xf14b0101,0xfb400108,0xf14d0101,0xfb400108,0xf14f0101,
	0xfb400108,0xf1500101,0xfb400108,0xf1510101,0xfb400108,0xf1520101,0xfb400108,0xf1530101,
	0xfb400108,0xf1540101,0xfb400108,0xf1550101,0xfb400108,0xf1560101,0xfb400108,0xf1570101,
	0xfb400108,0xf1580101,0xfb400108,0xf1590101,0xfb400108,0xf15a0101,0xfb400108,0xf15b0101,
	0xfb400108,0xf15d0101,0xfb400108,0xf15f0101,0xfb400108,0xf1600101,0xfb400108,0xf1610101,
	0xfb400108,0xf1620101,0xfb400108,0xf1630101,0xfb400108,0xf1650101,0xfb400108,0xf1690101,
	0xfb400108,0xf16a0101,0xfb400108,0xf16b0101,0xfb400108,0xf16c0101,0xfb400108,0xf16d0101,
	0xfb400108,0xf16f0101,0xfb400108,0xf1700101,0xfb400108,0xf1710101,0xfb400108,0xf1740101,
	0xfb400108,0xf1750101,0xfb400108,0xf1760101,0xfb400108,0xf1770101,0xfb400108,0xf1790101,
	0xfb400108,0xf17b0101,0xfb400108,0xf17c0101,0xfb400108,0xf17e0101,0xfb400108,0xf17f0101,
	0xfb400108,0xf1800101,0xfb400108,0xf1810101,0xfb400108,0xf1820101,0xfb400108,0xf1830101,
	0xfb400108,0xf1850101,0xfb400108,0xf1860101,0xfb400108,0xf1870101,0xfb400108,0xf1880101,
	0xfb400108,0xf1890101,0xfb400108,0xf18b0101,0xfb400108,0xf18c0101,0xfb400108,0xf18d0101,
	0xfb400108,0xf18e0101,0xfb400108,0xf1900101,0xfb400108,0xf1910101,0xfb400108,0xf1920101,
	0xfb400108,0xf1930101,0xfb400108,0xf1950101,0xfb400108,0xf1960101,0xfb400108,0xf1970101,
	0xfb400108,0xf19a0101,0xfb400108,0xf19b0101,0xfb400108,0xf19c0101,0xfb400108,0xf19d0101,
	0xfb400108,0xf19e0101,0xfb400108,0xf1a10101,0xfb400108,0xf1a20101,0xfb400108,0xf1a30101,
	0xfb400108,0xf1a40101,0xfb400108,0xf1a50101,0xfb400108,0xf1a60101,0xfb400108,0xf1a70101,
	0xfb400108,0xf1a90101,0xfb400108,0xf1aa0101,0xfb400108,0xf1ab0101,0xfb400108,0xf1ad0101,
	0xfb400108,0xf1ae0101,0xfb400108,0xf1af0101,0xfb400108,0xf1b00101,0xfb400108,0xf1b10101,
	0xfb400108,0xf1b20101,0xfb400108,0xf1b40101,0xfb400108,0xf1b60101,0xfb400108,0xf1b70101,
	0xfb400108,0xf1b80101,0xfb400108,0xf1ba0101,0xfb400108,0xf1bb0101,0xfb400108,0xf1bc0101,
	0xfb400108,0xf1bd0101,0xfb400108,0xf1be0101,0xfb400108,0xf1bf0101,0xfb400108,0xf1c00101,
	0xfb400108,0xf1c10101,0xfb400108,0xf1c20101,0xfb400108,0xf1c40101,0xfb400108,0xf1c50101,
	0xfb400108,0xf1c60101,0xfb400108,0xf1c70101,0xfb400108,0xf1c80101,0xfb400108,0xf1c90101,
	0xfb400108,0xf1ca0101,0xfb400108,0xf1cb0101,0xfb400108,0xf1cc0101,0xfb400108,0xf1cd0101,
	0xfb400108,0xf1cf0101,0xfb400108,0xf1d00101,0xfb400108,0xf1d10101,0xfb400108,0xf1d20101,
	0xfb400108,0xf1d30101,0xfb400108,0xf1d60101,0xfb400108,0xf1d70101,0xfb400108,0xf1d80101,
	0xfb400108,0xf1d90101,0xfb400108,0xf1da0101,0xfb400108,0xf1db0101,0xfb400108,0xf1dc0101,
	0xfb400108,0xf1dd0101,0xfb400108,0xf1de0101,0xfb400108,0xf1df0101,0xfb400108,0xf1e10101,
	0xfb400108,0xf1e20101,0xfb400108,0xf1e30101,0xfb400108,0xf1e40101,0xfb400108,0xf1e60101,
	0xfb400108,0xf1e80101,0xfb400108,0xf1e90101,0xfb400108,0xf1ea0101,0xfb400108,0xf1eb0101,
	0xfb400108,0xf1ec0101,0xfb400108,0xf1ed0101,0xfb400108,0xf1ef0101,0xfb400108,0xf1f00101,
	0xfb400108,0xf1f10101,0xfb400108,0xf1f20101,0xfb400108,0xf1f30101,0xfb400108,0xf1f40101,
	0xfb400108,0xf1f50101,0xfb400108,0xf1f60101,0xfb400108,0xf1f70101,0xfb400108,0xf1f80101,
	0xfb400108,0xf1fa0101,0xfb400108,0xf1fb0101,0xfb400108,0xf1fc0101,0xfb400108,0xf1fd0101,
	0xfb400108,0xf1fe0101,0xfb400108,0xf1ff0101,0xfb400108,0xf2000101,0xfb400108,0xf2010101,
	0xfb400108,0xf2020101,0xfb400108,0xf2030101,0xfb400108,0xf2040101,0xfb400108,0xf2050101,
	0xfb400108,0xf2070101,0xfb400108,0xf2080101,0xfb400108,0xf2090101,0xfb400108,0xf20a0101,
	0xfb400108,0xf20b0101,0xfb400108,0xf20c0101,0xfb400108,0xf20d0101,0xfb400108,0xf20e0101,
	0xfb400108,0xf20f0101,0xfb400108,0xf2100101,0xfb400108,0xf2110101,0xfb400108,0xf2120101,
	0xfb400108,0xf2130101,0xfb400108,0xf2140101,0xfb400108,0xf2150101,0xfb400108,0xf2160101,
	0xfb400108,0xf2170101,0xfb400108,0xf2180101,0xfb400108,0xf2190101,0xfb400108,0xf21a0101,
	0xfb400108,0xf21b0101,0xfb400108,0xf21c0101,0xfb400108,0xf21e0101,0xfb400108,0xf21f0101,
	0xfb400108,0xf2200101,0xfb400108,0xf2210101,0xfb400108,0xf2220101,0xfb400108,0xf2230101,
	0xfb400108,0xf2240101,0xfb400108,0xf2250101,0xfb400108,0xf2260101,0xfb400108,0xf2270101,
	0xfb400108,0xf2290101,0xfb400108,0xf22b0101,0xfb400108,0xf22d0101,0xfb400108,0xf22e0101,
	0xfb400108,0xf22f0101,0xfb400108,0xf2320101,0xfb400108,0xf2330101,0xfb400108,0xf2340101,
	0xfb400108,0xf23a0101,0xfb400108,0xf23c0101,0xfb400108,0xf23e0101,0xfb400108,0xf2400101,
	0xfb400108,0xf2410101,0xfb400108,0xf2420101,0xfb400108,0xf2430101,0xfb400108,0xf2440101,
	0xfb400108,0xf2450101,0xfb400108,0xf2460101,0xfb400108,0xf2490101,0xfb400108,0xf24a0101,
	0xfb400108,0xf24b0101,0xfb400108,0xf24e0101,0xfb400108,0xf24f0101,0xfb400108,0xf2500101,
	0xfb400108,0xf2510101,0xfb400108,0xf2530101,0xfb400108,0xf2540101,0xfb400108,0xf2550101,
	0xfb400108,0xf2570101,0xfb400108,0xf2580101,0xfb400108,0xf25a0101,0xfb400108,0xf25c0101,
	0xfb400108,0xf25e0101,0xfb400108,0xf2600101,0xfb400108,0xf2630101,0xfb400108,0xf2640101,
	0xfb400108,0xf2650101,0xfb400108,0xf2680101,0xfb400108,0xf26a0101,0xfb400108,0xf26b0101,
	0xfb400108,0xf26c0101,0xfb400108,0xf26d0101,0xfb400108,0xf2700101,0xfb400108,0xf2710101,
	0xfb400108,0xf2730101,0xfb400108,0xf2740101,0xfb400108,0xf2760101,0xfb400108,0xf2770101,
	0xfb400108,0xf2780101,0xfb400108,0xf27b0101,0xfb400108,0xf27c0101,0xfb400108,0xf27d0101,
	0xfb400108,0xf2820101,0xfb400108,0xf2830101,0xfb400108,0xf2850101,0xfb400108,0xf2860101,
	0xfb400108,0xf2870101,0xfb400108,0xf2880101,0xfb400108,0xf2890101,0xfb400108,0xf28c0101,
	0xfb400108,0xf28e0101,0xfb400108,0xf2900101,0xfb400108,0xf2910101,0xfb400108,0xf2930101,
	0xfb400108,0xf2940101,0xfb400108,0xf2950101,0xfb400108,0xf2960101,0xfb400108,0xf2970101,
	0xfb400108,0xf2980101,0xfb400108,0xf2990101,0xfb400108,0xf29a0101,0xfb400108,0xf29b0101,
	0xfb400108,0xf29c0101,0xfb400108,0xf29d0101,0xfb400108,0xf29e0101,0xfb400108,0xf2a00101,
	0xfb400108,0xf2a10101,0xfb400108,0xf2a20101,0xfb400108,0xf2a30101,0xfb400108,0xf2a40101,
	0xfb400108,0xf2a50101,0xfb400108,0xf2a60101,0xfb400108,0xf2a70101,0xfb400108,0xf2a80101,
	0xfb400108,0xf2a90101,0xfb400108,0xf2aa0101,0xfb400108,0xf2ab0101,0xfb400108,0xf2ae0101,
	0xfb400108,0xf2b10101,0xfb400108,0xf2b20101,0xfb400108,0xf2b30101,0xfb400108,0xf2b50101,
	0xfb400108,0xf2ba0101,0xfb400108,0xf2bb0101,0xfb400108,0xf2bc0101,0xfb400108,0xf2bd0101,
	0xfb400108,0xf2be0101,0xfb400108,0xf2bf0101,0xfb400108,0xf2c00101,0xfb400108,0xf2c50101,
	0xfb400108,0xf2c60101,0xfb400108,0xf2c70101,0xfb400108,0xf2c90101,0xfb400108,0xf2ca0101,
	0xfb400108,0xf2cb0101,0xfb400108,0xf2cc0101,0xfb400108,0xf2cf0101,0xfb400108,0xf2d10101,
	0xfb400108,0xf2d30101,0xfb400108,0xf2d40101,0xfb400108,0xf2d50101,0xfb400108,0xf2d60101,
	0xfb400108,0xf2d80101,0xfb400108,0xf2da0101,0xfb400108,0xf2db0101,0xfb400108,0xf2dc0101,
	0xfb400108,0xf2dd0101,0xfb400108,0xf2df0101,0xfb400108,0xf2e20101,0xfb400108,0xf2e30101,
	0xfb400108,0xf2e40101,0xfb400108,0xf2e50101,0xfb400108,0xf2e60101,0xfb400108,0xf2e70101,
	0xfb400108,0xf2ea0101,0xfb400108,0xf2eb0101,0xfb400108,0xf2f50101,0xfb400108,0xf2f60101,
	0xfb400108,0xf2f90101,0xfb400108,0xf2fd0101,0xfb400108,0xf2fe0101,0xfb400108,0xf2ff0101,
	0xfb400108,0xf3000101,0xfb400108,0xf3020101,0xfb400108,0xf3040101,0xfb400108,0xf3050101,
	0xfb400108,0xf3060101,0xfb400108,0xf3070101,0xfb400108,0xf3080101,0xfb400108,0xf3090101,
	0xfb400108,0xf30b0101,0xfb400108,0xf30c0101,0xfb400108,0xf30d0101,0xfb400108,0xf30f0101,
	0xfb400108,0xf3100101,0xfb400108,0xf3110101,0xfb400108,0xf3120101,0xfb400108,0xf3140101,
	0xfb400108,0xf3180101,0xfb400108,0xf3190101,0xfb400108,0xf31a0101,0xfb400108,0xf31f0101,
	0xfb400108,0xf3200101,0xfb400108,0xf3230101,0xfb400108,0xf3240101,0xfb400108,0xf3260101,
	0xfb400108,0xf3270101,0xfb400108,0xf3280101,0xfb400108,0xf32d0101,0xfb400108,0xf32f0101,
	0xfb400108,0xf3300101,0xfb400108,0xf3320101,0xfb400108,0xf3330101,0xfb400108,0xf3350101,
	0xfb400108,0xf3360101,0xfb400108,0xf33a0101,0xfb400108,0xf33b0101,0xfb400108,0xf33c0101,
	0xfb400108,0xf33d0101,0xfb400108,0xf3400101,0xfb400108,0xf3410101,0xfb400108,0xf3420101,
	0xfb400108,0xf3430101,0xfb400108,0xf3440101,0xfb400108,0xf3450101,0xfb400108,0xf3460101,
	0xfb400108,0xf3470101,0xfb400108,0xf3480101,0xfb400108,0xf3490101,0xfb400108,0xf34a0101,
	0xfb400108,0xf34b0101,0xfb400108,0xf34c0101,0xfb400108,0xf34e0101,0xfb400108,0xf34f0101,
	0xfb400108,0xf3510101,0xfb400108,0xf3530101,0xfb400108,0xf3540101,0xfb400108,0xf3550101,
	0xfb400108,0xf3560101,0xfb400108,0xf3580101,0xfb400108,0xf3590101,0xfb400108,0xf35a0101,
	0xfb400108,0xf35b0101,0xfb400108,0xf35c0101,0xfb400108,0xf35d0101,0xfb400108,0xf35e0101,
	0xfb400108,0xf35f0101,0xfb400108,0xf3610101,0xfb400108,0xf3620101,0xfb400108,0xf3630101,
	0xfb400108,0xf3640101,0xfb400108,0xf3650101,0xfb400108,0xf3660101,0xfb400108,0xf3670101,
	0xfb400108,0xf3680101,0xfb400108,0xf3690101,0xfb400108,0xf36a0101,0xfb400108,0xf36b0101,
	0xfb400108,0xf36e0101,0xfb400108,0xf3700101,0xfb400108,0xf3710101,0xfb400108,0xf3720101,
	0xfb400108,0xf3730101,0xfb400108,0xf3740101,0xfb400108,0xf3750101,0xfb400108,0xf3760101,
	0xfb400108,0xf3770101,0xfb400108,0xf3780101,0xfb400108,0xf3790101,0xfb400108,0xf37a0101,
	0xfb400108,0xf37b0101,0xfb400108,0xf37c0101,0xfb400108,0xf37d0101,0xfb400108,0xf37f0101,
	0xfb400108,0xf3800101,0xfb400108,0xf3810101,0xfb400108,0xf3820101,0xfb400108,0xf3830101,
	0xfb400108,0xf3850101,0xfb400108,0xf3860101,0xfb400108,0xf3880101,0xfb400108,0xf38a0101,
	0xfb400108,0xf38c0101,0xfb400108,0xf38d0101,0xfb400108,0xf38f0101,0xfb400108,0xf3900101,
	0xfb400108,0xf3920101,0xfb400108,0xf3930101,0xfb400108,0xf3940101,0xfb400108,0xf3950101,
	0xfb400108,0xf3970101,0xfb400108,0xf3980101,0xfb400108,0xf3990101,0xfb400108,0xf39a0101,
	0xfb400108,0xf39c0101,0xfb400108,0xf39d0101,0xfb400108,0xf39e0101,0xfb400108,0xf3a00101,
	0xfb400108,0xf3a10101,0xfb400108,0xf3a30101,0xfb400108,0xf3a40101,0xfb400108,0xf3a50101,
	0xfb400108,0xf3a60101,0xfb400108,0xf3a70101,0xfb400108,0xf3a80101,0xfb400108,0xf3aa0101,
	0xfb400108,0xf3ac0101,0xfb400108,0xf3ad0101,0xfb400108,0xf3b10101,0xfb400108,0xf3b40101,
	0xfb400108,0xf3b50101,0xfb400108,0xf3b60101,0xfb400108,0xf3b80101,0xfb400108,0xf3b90101,
	0xfb400108,0xf3bc0101,0xfb400108,0xf3bd0101,0xfb400108,0xf3be0101,0xfb400108,0xf3bf0101,
	0xfb400108,0xf3c10101,0xfb400108,0xf3c30101,0xfb400108,0xf3c40101,0xfb400108,0xf3c50101,
	0xfb400108,0xf3c60101,0xfb400108,0xf3c70101,0xfb400108,0xf3cb0101,0xfb400108,0xf3cc0101,
	0xfb400108,0xf3ce0101,0xfb400108,0xf3d20101,0xfb400108,0xf3d30101,0xfb400108,0xf3d40101,
	0xfb400108,0xf3d50101,0xfb400108,0xf3d60101,0xfb400108,0xf3d70101,0xfb400108,0xf3d80101,
	0xfb400108,0xf3da0101,0xfb400108,0xf3db0101,0xfb400108,0xf3dc0101,0xfb400108,0xf3dd0101,
	0xfb400108,0xf3df0101,0xfb400108,0xf3e10101,0xfb400108,0xf3e20101,0xfb400108,0xf3e30101,
	0xfb400108,0xf3e40101,0xfb400108,0xf3e60101,0xfb400108,0xf3e80101,0xfb400108,0xf3ea0101,
	0xfb400108,0xf3eb0101,0xfb400108,0xf3ec0101,0xfb400108,0xf3ee0101,0xfb400108,0xf3ef0101,
	0xfb400108,0xf3f00101,0xfb400108,0xf3f10101,0xfb400108,0xf3f30101,0xfb400108,0xf3f40101,
	0xfb400108,0xf3f50101,0xfb400108,0xf3f60101,0xfb400108,0xf3f70101,0xfb400108,0xf3f80101,
	0xfb400108,0xf3f90101,0xfb400108,0xf3fa0101,0xfb400108,0xf3fb0101,0xfb400108,0xf3fc0101,
	0xfb400108,0xf3fd0101,0xfb400108,0xf3fe0101,0xfb400108,0xf3ff0101,0xfb400108,0xf4000101,
	0xfb400108,0xf4010101,0xfb400108,0xf4020101,0xfb400108,0xf4040101,0xfb400108,0xf4070101,
	0xfb400108,0xf4080101,0xfb400108,0xf40b0101,0xfb400108,0xf40c0101,0xfb400108,0xf40d0101,
	0xfb400108,0xf40e0101,0xfb400108,0xf4110101,0xfb400108,0xf4120101,0xfb400108,0xf4130101,
	0xfb400108,0xf4140101,0xfb400108,0xf4150101,0xfb400108,0xf4160101,0xfb400108,0xf4170101,
	0xfb400108,0xf4180101,0xfb400108,0xf4190101,0xfb400108,0xf41c0101,0xfb400108,0xf41d0101,
	0xfb400108,0xf41e0101,0xfb400108,0xf41f0101,0xfb400108,0xf4200101,0xfb400108,0xf4210101,
	0xfb400108,0xf4230101,0xfb400108,0xf4240101,0xfb400108,0xf4270101,0xfb400108,0xf4290101,
	0xfb400108,0xf42b0101,0xfb400108,0xf42d0101,0xfb400108,0xf42f0101,0xfb400108,0xf4310101,
	0xfb400108,0xf4320101,0xfb400108,0xf4370101,0xfb400108,0xf4380101,0xfb400108,0xf4390101,
	0xfb400108,0xf43a0101,0xfb400108,0xf43b0101,0xfb400108,0xf43d0101,0xfb400108,0xf43e0101,
	0xfb400108,0xf43f0101,0xfb400108,0xf4400101,0xfb400108,0xf4420101,0xfb400108,0xf4430101,
	0xfb400108,0xf4440101,0xfb400108,0xf4450101,0xfb400108,0xf4460101,0xfb400108,0xf4470101,
	0xfb400108,0xf4480101,0xfb400108,0xf4490101,0xfb400108,0xf44a0101,0xfb400108,0xf44b0101,
	0xfb400108,0xf44c0101,0xfb400108,0xf44d0101,0xfb400108,0xf44e0101,0xfb400108,0xf44f0101,
	0xfb400108,0xf4500101,0xfb400108,0xf4510101,0xfb400108,0xf4520101,0xfb400108,0xf4530101,
	0xfb400108,0xf4540101,0xfb400108,0xf4560101,0xfb400108,0xf4580101,0xfb400108,0xf45d0101,
	0xfb400108,0xf4600101,0xfb400108,0xf4610101,0xfb400108,0xf4620101,0xfb400108,0xf4630101,
	0xfb400108,0xf4640101,0xfb400108,0xf4650101,0xfb400108,0xf4660101,0xfb400108,0xf4670101,
	0xfb400108,0xf4680101,0xfb400108,0xf4690101,0xfb400108,0xf46a0101,0xfb400108,0xf46b0101,
	0xfb400108,0xf46c0101,0xfb400108,0xf46e0101,0xfb400108,0xf46f0101,0xfb400108,0xf4710101,
	0xfb400108,0xf4720101,0xfb400108,0xf4730101,0xfb400108,0xf4740101,0xfb400108,0xf4750101,
	0xfb400108,0xf4780101,0xfb400108,0xf4790101,0xfb400108,0xf47a0101,0xfb400108,0xf47b0101,
	0xfb400108,0xf47c0101,0xfb400108,0xf47d0101,0xfb400108,0xf47f0101,0xfb400108,0xf4820101,
	0xfb400108,0xf4840101,0xfb400108,0xf4850101,0xfb400108,0xf4860101,0xfb400108,0xf4880101,
	0xfb400108,0xf4890101,0xfb400108,0xf48a0101,0xfb400108,0xf48c0101,0xfb400108,0xf48d0101,
	0xfb400108,0xf48f0101,0xfb400108,0xf4910101,0xfb400108,0xf4920101,0xfb400108,0xf4930101,
	0xfb400108,0xf4940101,0xfb400108,0xf4950101,0xfb400108,0xf4960101,0xfb400108,0xf4970101,
	0xfb400108,0xf4980101,0xfb400108,0xf4990101,0xfb400108,0xf49a0101,0xfb400108,0xf49b0101,
	0xfb400108,0xf49d0101,0xfb400108,0xf49f0101,0xfb400108,0xf4a00101,0xfb400108,0xf4a10101,
	0xfb400108,0xf4a20101,0xfb400108,0xf4a30101,0xfb400108,0xf4a40101,0xfb400108,0xf4a50101,
	0xfb400108,0xf4a60101,0xfb400108,0xf4aa0101,0xfb400108,0xf4ab0101,0xfb400108,0xf4ac0101,
	0xfb400108,0xf4ad0101,0xfb400108,0xf4ae0101,0xfb400108,0xf4af0101,0xfb400108,0xf4b00101,
	0xfb400108,0xf4b10101,0xfb400108,0xf4b20101,0xfb400108,0xf4b30101,0xfb400108,0xf4b40101,
	0xfb400108,0xf4b50101,0xfb400108,0xf4b60101,0xfb400108,0xf4b70101,0xfb400108,0xf4b80101,
	0xfb400108,0xf4b90101,0xfb400108,0xf4bb0101,0xfb400108,0xf4bc0101,0xfb400108,0xf4bd0101,
	0xfb400108,0xf4be0101,0xfb400108,0xf4bf0101,0xfb400108,0xf4c00101,0xfb400108,0xf4c10101,
	0xfb400108,0xf4c20101,0xfb400108,0xf4c30101,0xfb400108,0xf4c40101,0xfb400108,0xf4c50101,
	0xfb400108,0xf4c60101,0xfb400108,0xf4c70101,0xfb400108,0xf4c80101,0xfb400108,0xf4c90101,
	0xfb400108,0xf4ca0101,0xfb400108,0xf4cb0101,0xfb400108,0xf4cc0101,0xfb400108,0xf4cd0101,
	0xfb400108,0xf4ce0101,0xfb400108,0xf4cf0101,0xfb400108,0xf4d00101,0xfb400108,0xf4d10101,
	0xfb400108,0xf4d30101,0xfb400108,0xf4d40101,0xfb400108,0xf4d50101,0xfb400108,0xf4d60101,
	0xfb400108,0xf4d70101,0xfb400108,0xf4d80101,0xfb400108,0xf4d90101,0xfb400108,0xf4da0101,
	0xfb400108,0xf4db0101,0xfb400108,0xf4dd0101,0xfb400108,0xf4df0101,0xfb400108,0xf4e10101,
	0xfb400108,0xf4e50101,0xfb400108,0xf4e70101,0xfb400108,0xf4e80101,0xfb400108,0xf4e90101,
	0xfb400108,0xf4ea0101,0xfb400108,0xf4eb0101,0xfb400108,0xf4ec0101,0xfb400108,0xf4ed0101,
	0xfb400108,0xf4f00101,0xfb400108,0xf4f10101,0xfb400108,0xf4f20101,0xfb400108,0xf4f30101,
	0xfb400108,0xf4f50101,0xfb400108,0xf4f80101,0xfb400108,0xf4f90101,0xfb400108,0xf4fa0101,
	0xfb400108,0xf4fb0101,0xfb400108,0xf4fc0101,0xfb400108,0xf4fd0101,0xfb400108,0xf4fe0101,
	0xfb400108,0xf5000101,0xfb400108,0xf5010101,0xfb400108,0xf5020101,0xfb400108,0xf5030101,
	0xfb400108,0xf5050101,0xfb400108,0xf5060101,0xfb400108,0xf5070101,0xfb400108,0xf5080101,
	0xfb400108,0xf5090101,0xfb400108,0xf50a0101,0xfb400108,0xf50b0101,0xfb400108,0xf50c0101,
	0xfb400108,0xf50e0101,0xfb400108,0xf5100101,0xfb400108,0xf5120101,0xfb400108,0xf5140101,
	0xfb400108,0xf5150101,0xfb400108,0xf5160101,0xfb400108,0xf5170101,0xfb400108,0xf51b0101,
	0xfb400108,0xf51d0101,0xfb400108,0xf51e0101,0xfb400108,0xf5200101,0xfb400108,0xf5210101,
	0xfb400108,0xf5220101,0xfb400108,0xf5230101,0xfb400108,0xf5240101,0xfb400108,0xf5260101,
	0xfb400108,0xf5270101,0xfb400108,0xf52a0101,0xfb400108,0xf52e0101,0xfb400108,0xf5340101,
	0xfb400108,0xf5360101,0xfb400108,0xf5390101,0xfb400108,0xf53c0101,0xfb400108,0xf53d0101,
	0xfb400108,0xf53f0101,0xfb400108,0xf5410101,0xfb400108,0xf5420101,0xfb400108,0xf5430101,
	0xfb400108,0xf5440101,0xfb400108,0xf5460101,0xfb400108,0xf5470101,0xfb400108,0xf5490101,
	0xfb400108,0xf54a0101,0xfb400108,0xf54d0101,0xfb400108,0xf5500101,0xfb400108,0xf5510101,
	0xfb400108,0xf5520101,0xfb400108,0xf5530101,0xfb400108,0xf5550101,0xfb400108,0xf5560101,
	0xfb400108,0xf5570101,0xfb400108,0xf5580101,0xfb400108,0xf55d0101,0xfb400108,0xf55e0101,
	0xfb400108,0xf55f0101,0xfb400108,0xf5600101,0xfb400108,0xf5610101,0xfb400108,0xf5620101,
	0xfb400108,0xf5630101,0xfb400108,0xf5640101,0xfb400108,0xf5670101,0xfb400108,0xf5680101,
	0xfb400108,0xf5690101,0xfb400108,0xf56b0101,0xfb400108,0xf56c0101,0xfb400108,0xf56d0101,
	0xfb400108,0xf56e0101,0xfb400108,0xf56f0101,0xfb400108,0xf5700101,0xfb400108,0xf5710101,
	0xfb400108,0xf5730101,0xfb400108,0xf5750101,0xfb400108,0xf5760101,0xfb400108,0xf5770101,
	0xfb400108,0xf57a0101,0xfb400108,0xf57b0101,0xfb400108,0xf57c0101,0xfb400108,0xf57d0101,
	0xfb400108,0xf57e0101,0xfb400108,0xf5800101,0xfb400108,0xf5810101,0xfb400108,0xf5820101,
	0xfb400108,0xf5840101,0xfb400108,0xf5850101,0xfb400108,0xf5870101,0xfb400108,0xf5880101,
	0xfb400108,0xf5890101,0xfb400108,0xf58a0101,0xfb400108,0xf58c0101,0xfb400108,0xf58d0101,
	0xfb400108,0xf58e0101,0xfb400108,0xf5900101,0xfb400108,0xf5930101,0xfb400108,0xf5950101,
	0xfb400108,0xf5980101,0xfb400108,0xf59b0101,0xfb400108,0xf59c0101,0xfb400108,0xf59e0101,
	0xfb400108,0xf5a20101,0xfb400108,0xf5a60101,0xfb400108,0xf5a70101,0xfb400108,0xf5a80101,
	0xfb400108,0xf5a90101,0xfb400108,0xf5aa0101,0xfb400108,0xf5ad0101,0xfb400108,0xf5b60101,
	0xfb400108,0xf5b70101,0xfb400108,0xf5ba0101,0xfb400108,0xf5bb0101,0xfb400108,0xf5bf0101,
	0xfb400108,0xf5c00101,0xfb400108,0xf5c10101,0xfb400108,0xf5c60101,0xfb400108,0xf5cb0101,
	0xfb400108,0xf5cc0101,0xfb400108,0xf5ce0101,0xfb400108,0xf5cf0101,0xfb400108,0xf5d00101,
	0xfb400108,0xf5d10101,0xfb400108,0xf5d30101,0xfb400108,0xf5d70101,0xfb400108,0xf5d90101,
	0xfb400108,0xf5da0101,0xfb400108,0xf5dc0101,0xfb400108,0xf5dd0101,0xfb400108,0xf5df0101,
	0xfb400108,0xf5e00101,0xfb400108,0xf5e10101,0xfb400108,0xf5e50101,0xfb400108,0xf5e90101,
	0xfb400108,0xf5ec0101,0xfb400108,0xf5ed0101,0xfb400108,0xf5ee0101,0xfb400108,0xf5ef0101,
	0xfb400108,0xf5f20101,0xfb400108,0xf5f30101,0xfb400108,0xf5f50101,0xfb400108,0xf5f60101,
	0xfb400108,0xf5f70101,0xfb400108,0xf5f80101,0xfb400108,0xf5fa0101,0xfb400108,0xf5fb0101,
	0xfb400108,0xf5fd0101,0xfb400108,0xf5fe0101,0xfb400108,0xf6020101,0xfb400108,0xf6040101,
	0xfb400108,0xf6060101,0xfb400108,0xf6070101,0xfb400108,0xf6080101,0xfb400108,0xf6090101,
	0xfb400108,0xf60b0101,0xfb400108,0xf60d0101,0xfb400108,0xf60e0101,0xfb400108,0xf60f0101,
	0xfb400108,0xf6110101,0xfb400108,0xf6120101,0xfb400108,0xf6130101,0xfb400108,0xf6140101,
	0xfb400108,0xf6160101,0xfb400108,0xf61a0101,0xfb400108,0xf61c0101,0xfb400108,0xf61d0101,
	0xfb400108,0xf61e0101,0xfb400108,0xf6210101,0xfb400108,0xf6230101,0xfb400108,0xf6270101,
	0xfb400108,0xf6280101,0xfb400108,0xf62c0101,0xfb400108,0xf62e0101,0xfb400108,0xf62f0101,
	0xfb400108,0xf6310101,0xfb400108,0xf6320101,0xfb400108,0xf6360101,0xfb400108,0xf6370101,
	0xfb400108,0xf6390101,0xfb400108,0xf63a0101,0xfb400108,0xf63b0101,0xfb400108,0xf63d0101,
	0xfb400108,0xf6410101,0xfb400108,0xf6420101,0xfb400108,0xf6440101,0xfb400108,0xf6450101,
	0xfb400108,0xf6460101,0xfb400108,0xf6470101,0xfb400108,0xf6480101,0xfb400108,0xf6490101,
	0xfb400108,0xf64a0101,0xfb400108,0xf64b0101,0xfb400108,0xf64e0101,0xfb400108,0xf64f0101,
	0xfb400108,0xf6500101,0xfb400108,0xf6510101,0xfb400108,0xf6520101,0xfb400108,0xf6530101,
	0xfb400108,0xf6550101,0xfb400108,0xf6570101,0xfb400108,0xf6580101,0xfb400108,0xf6590101,
	0xfb400108,0xf65a0101,0xfb400108,0xf65b0101,0xfb400108,0xf65d0101,0xfb400108,0xf65f0101,
	0xfb400108,0xf6600101,0xfb400108,0xf6610101,0xfb400108,0xf6620101,0xfb400108,0xf6640101,
	0xfb400108,0xf6650101,0xfb400108,0xf6660101,0xfb400108,0xf6670101,0xfb400108,0xf6680101,
	0xfb400108,0xf6690101,0xfb400108,0xf66a0101,0xfb400108,0xf66c0101,0xfb400108,0xf66d0101,
	0xfb400108,0xf66e0101,0xfb400108,0xf6700101,0xfb400108,0xf6710101,0xfb400108,0xf6720101,
	0xfb400108,0xf6730101,0xfb400108,0xf6740101,0xfb400108,0xf6750101,0xfb400108,0xf6760101,
	0xfb400108,0xf6770101,0xfb400108,0xf6790101,0xfb400108,0xf67a0101,0xfb400108,0xf67c0101,
	0xfb400108,0xf67f0101,0xfb400108,0xf6800101,0xfb400108,0xf6810101,0xfb400108,0xf6830101,
	0xfb400108,0xf6850101,0xfb400108,0xf6890101,0xfb400108,0xf68a0101,0xfb400108,0xf68c0101,
	0xfb400108,0xf68d0101,0xfb400108,0xf68f0101,0xfb400108,0xf6900101,0xfb400108,0xf6920101,
	0xfb400108,0xf6940101,0xfb400108,0xf6950101,0xfb400108,0xf6970101,0xfb400108,0xf6980101,
	0xfb400108,0xf69a0101,0xfb400108,0xf69b0101,0xfb400108,0xf69c0101,0xfb400108,0xf69d0101,
	0xfb400108,0xf69e0101,0xfb400108,0xf69f0101,0xfb400108,0xf6a00101,0xfb400108,0xf6a10101,
	0xfb400108,0xf6a20101,0xfb400108,0xf6a30101,0xfb400108,0xf6a50101,0xfb400108,0xf6a60101,
	0xfb400108,0xf6a70101,0xfb400108,0xf6a80101,0xfb400108,0xf6a90101,0xfb400108,0xf6aa0101,
	0xfb400108,0xf6ab0101,0xfb400108,0xf6ac0101,0xfb400108,0xf6ad0101,0xfb400108,0xf6af0101,
	0xfb400108,0xf6b00101,0xfb400108,0xf6b30101,0xfb400108,0xf6b50101,0xfb400108,0xf6b60101,
	0xfb400108,0xf6b70101,0xfb400108,0xf6b80101,0xfb400108,0xf6b90101,0xfb400108,0xf6ba0101,
	0xfb400108,0xf6bb0101,0xfb400108,0xf6bc0101,0xfb400108,0xf6bd0101,0xfb400108,0xf6be0101,
	0xfb400108,0xf6c00101,0xfb400108,0xf6c10101,0xfb400108,0xf6c30101,0xfb400108,0xf6c40101,
	0xfb400108,0xf6c70101,0xfb400108,0xf6c90101,0xfb400108,0xf6cb0101,0xfb400108,0xf6cc0101,
	0xfb400108,0xf6d30101,0xfb400108,0xf6d50101,0xfb400108,0xf6d90101,0xfb400108,0xf6da0101,
	0xfb400108,0xf6dc0101,0xfb400108,0xf6dd0101,0xfb400108,0xf6de0101,0xfb400108,0xf6e00101,
	0xfb400108,0xf6e10101,0xfb400108,0xf6e20101,0xfb400108,0xf6e30101,0xfb400108,0xf6e40101,
	0xfb400108,0xf6e60101,0xfb400108,0xf6e70101,0xfb400108,0xf6e80101,0xfb400108,0xf6e90101,
	0xfb400108,0xf6ea0101,0xfb400108,0xf6eb0101,0xfb400108,0xf6ec0101,0xfb400108,0xf6ed0101,
	0xfb400108,0xf6f00101,0xfb400108,0xf6f30101,0xfb400108,0xf6f50101,0xfb400108,0xf6f60101,
	0xfb400108,0xf6f70101,0xfb400108,0xf6fa0101,0xfb400108,0xf6fb0101,0xfb400108,0xf6fd0101,
	0xfb400108,0xf6ff0101,0xfb400108,0xf7000101,0xfb400108,0xf7020101,0xfb400108,0xf7030101,
	0xfb400108,0xf7050101,0xfb400108,0xf7060101,0xfb400108,0xf70a0101,0xfb400108,0xf70c0101,
	0xfb400108,0xf70e0101,0xfb400108,0xf70f0101,0xfb400108,0xf7100101,0xfb400108,0xf7110101,
	0xfb400108,0xf7120101,0xfb400108,0xf7130101,0xfb400108,0xf7140101,0xfb400108,0xf7150101,
	0xfb400108,0xf7160101,0xfb400108,0xf7170101,0xfb400108,0xf7180101,0xfb400108,0xf71b0101,
	0xfb400108,0xf71c0101,0xfb400108,0xf71d0101,0xfb400108,0xf71e0101,0xfb400108,0xf7210101,
	0xfb400108,0xf7230101,0xfb400108,0xf7240101,0xfb400108,0xf7250101,0xfb400108,0xf7270101,
	0xfb400108,0xf72a0101,0xfb400108,0xf72b0101,0xfb400108,0xf72c0101,0xfb400108,0xf72e0101,
	0xfb400108,0xf7300101,0xfb400108,0xf7310101,0xfb400108,0xf7320101,0xfb400108,0xf7330101,
	0xfb400108,0xf7340101,0xfb400108,0xf7390101,0xfb400108,0xf73b0101,0xfb400108,0xf73d0101,
	0xfb400108,0xf73e0101,0xfb400108,0xf73f0101,0xfb400108,0xf7420101,0xfb400108,0xf7440101,
	0xfb400108,0xf7450101,0xfb400108,0xf7460101,0xfb400108,0xf7480101,0xfb400108,0xf7490101,
	0xfb400108,0xf74a0101,0xfb400108,0xf74b0101,0xfb400108,0xf74c0101,0xfb400108,0xf74d0101,
	0xfb400108,0xf74e0101,0xfb400108,0xf74f0101,0xfb400108,0xf7520101,0xfb400108,0xf7530101,
	0xfb400108,0xf7540101,0xfb400108,0xf7550101,0xfb400108,0xf7560101,0xfb400108,0xf7570101,
	0xfb400108,0xf7580101,0xfb400108,0xf7590101,0xfb400108,0xf75c0101,0xfb400108,0xf75d0101,
	0xfb400108,0xf75e0101,0xfb400108,0xf75f0101,0xfb400108,0xf7600101,0xfb400108,0xf7640101,
	0xfb400108,0xf7670101,0xfb400108,0xf7690101,0xfb400108,0xf76a0101,0xfb400108,0xf76d0101,
	0xfb400108,0xf76e0101,0xfb400108,0xf76f0101,0xfb400108,0xf7700101,0xfb400108,0xf7710101,
	0xfb400108,0xf7720101,0xfb400108,0xf7730101,0xfb400108,0xf7740101,0xfb400108,0xf7750101,
	0xfb400108,0xf7760101,0xfb400108,0xf7770101,0xfb400108,0xf7780101,0xfb400108,0xf77a0101,
	0xfb400108,0xf77b0101,0xfb400108,0xf77c0101,0xfb400108,0xf7810101,0xfb400108,0xf7820101,
	0xfb400108,0xf7830101,0xfb400108,0xf7860101,0xfb400108,0xf7870101,0xfb400108,0xf7880101,
	0xfb400108,0xf7890101,0xfb400108,0xf78a0101,0xfb400108,0xf78b0101,0xfb400108,0xf78f0101,
	0xfb400108,0xf7900101,0xfb400108,0xf7930101,0xfb400108,0xf7940101,0xfb400108,0xf7950101,
	0xfb400108,0xf7960101,0xfb400108,0xf7970101,0xfb400108,0xf7980101,0xfb400108,0xf7990101,
	0xfb400108,0xf79a0101,0xfb400108,0xf79b0101,0xfb400108,0xf79c0101,0xfb400108,0xf79d0101,
	0xfb400108,0xf79e0101,0xfb400108,0xf7a10101,0xfb400108,0xf7a30101,0xfb400108,0xf7a40101,
	0xfb400108,0xf7a60101,0xfb400108,0xf7a80101,0xfb400108,0xf7ab0101,0xfb400108,0xf7ad0101,
	0xfb400108,0xf7ae0101,0xfb400108,0xf7af0101,0xfb400108,0xf7b10101,0xfb400108,0xf7b20101,
	0xfb400108,0xf7b40101,0xfb400108,0xf7b60101,0xfb400108,0xf7b70101,0xfb400108,0xf7b80101,
	0xfb400108,0xf7b90101,0xfb400108,0xf7ba0101,0xfb400108,0xf7bc0101,0xfb400108,0xf7be0101,
	0xfb400108,0xf7c00101,0xfb400108,0xf7c10101,0xfb400108,0xf7c20101,0xfb400108,0xf7c30101,
	0xfb400108,0xf7c40101,0xfb400108,0xf7c50101,0xfb400108,0xf7c60101,0xfb400108,0xf7c70101,
	0xfb400108,0xf7c80101,0xfb400108,0xf7c90101,0xfb400108,0xf7ca0101,0xfb400108,0xf7cb0101,
	0xfb400108,0xf7cc0101,0xfb400108,0xf7ce0101,0xfb400108,0xf7cf0101,0xfb400108,0xf7d00101,
	0xfb400108,0xf7d10101,0xfb400108,0xf7d20101,0xfb400108,0xf7d30101,0xfb400108,0xf7d40101,
	0xfb400108,0xf7d50101,0xfb400108,0xf7d60101,0xfb400108,0xf7d80101,0xfb400108,0xf7d90101,
	0xfb400108,0xf7da0101,0xfb400108,0xf7dd0101,0xfb400108,0xf7de0101,0xfb400108,0xf7df0101,
	0xfb400108,0xf7e00101,0xfb400108,0xf7e10101,0xfb400108,0xf7e40101,0xfb400108,0xf7e60101,
	0xfb400108,0xf7e80101,0xfb400108,0xf7ea0101,0xfb400108,0xf7ef0101,0xfb400108,0xf7f00101,
	0xfb400108,0xf7f10101,0xfb400108,0xf7f20101,0xfb400108,0xf7f40101,0xfb400108,0xf7f50101,
	0xfb400108,0xf7f70101,0xfb400108,0xf7f90101,0xfb400108,0xf7fa0101,0xfb400108,0xf7fb0101,
	0xfb400108,0xf7fc0101,0xfb400108,0xf8030101,0xfb400108,0xf8040101,0xfb400108,0xf8050101,
	0xfb400108,0xf8060101,0xfb400108,0xf8070101,0xfb400108,0xf8080101,0xfb400108,0xf80a0101,
	0xfb400108,0xf80b0101,0xfb400108,0xf80e0101,0xfb400108,0xf80f0101,0xfb400108,0xf8100101,
	0xfb400108,0xf8130101,0xfb400108,0xf8150101,0xfb400108,0xf8190101,0xfb400108,0xf81b0101,
	0xfb400108,0xf81e0101,0xfb400108,0xf8200101,0xfb400108,0xf8210101,0xfb400108,0xf8220101,
	0xfb400108,0xf8240101,0xfb400108,0xf8280101,0xfb400108,0xf82a0101,0xfb400108,0xf82b0101,
	0xfb400108,0xf82e0101,0xfb400108,0xf82f0101,0xfb400108,0xf8310101,0xfb400108,0xf8320101,
	0xfb400108,0xf8330101,0xfb400108,0xf8350101,0xfb400108,0xf8360101,0xfb400108,0xf83d0101,
	0xfb400108,0xf83f0101,0xfb400108,0xf8410101,0xfb400108,0xf8420101,0xfb400108,0xf8430101,
	0xfb400108,0xf8440101,0xfb400108,0xf8460101,0xfb400108,0xf8480101,0xfb400108,0xf8490101,
	0xfb400108,0xf84a0101,0xfb400108,0xf84b0101,0xfb400108,0xf84d0101,0xfb400108,0xf84f0101,
	0xfb400108,0xf8510101,0xfb400108,0xf8530101,0xfb400108,0xf8540101,0xfb400108,0xf8580101,
	0xfb400108,0xf8590101,0xfb400108,0xf85a0101,0xfb400108,0xf85b0101,0xfb400108,0xf85c0101,
	0xfb400108,0xf85e0101,0xfb400108,0xf85f0101,0xfb400108,0xf8600101,0xfb400108,0xf8610101,
	0xfb400108,0xf8620101,0xfb400108,0xf8630101,0xfb400108,0xf8640101,0xfb400108,0xf8650101,
	0xfb400108,0xf8660101,0xfb400108,0xf8670101,0xfb400108,0xf8680101,0xfb400108,0xf8690101,
	0xfb400108,0xf86f0101,0xfb400108,0xf8700101,0xfb400108,0xf8710101,0xfb400108,0xf8720101,
	0xfb400108,0xf8730101,0xfb400108,0xf8740101,0xfb400108,0xf8750101,0xfb400108,0xf8760101,
	0xfb400108,0xf8780101,0xfb400108,0xf8790101,0xfb400108,0xf87a0101,0xfb400108,0xf87b0101,
	0xfb400108,0xf87d0101,0xfb400108,0xf87e0101,0xfb400108,0xf87f0101,0xfb400108,0xf8800101,
	0xfb400108,0xf8810101,0xfb400108,0xf8820101,0xfb400108,0xf8830101,0xfb400108,0xf8840101,
	0xfb400108,0xf8850101,0xfb400108,0xf8860101,0xfb400108,0xf8880101,0xfb400108,0xf88a0101,
	0xfb400108,0xf88b0101,0xfb400108,0xf88f0101,0xfb400108,0xf8900101,0xfb400108,0xf8920101,
	0xfb400108,0xf8940101,0xfb400108,0xf8950101,0xfb400108,0xf8960101,0xfb400108,0xf8990101,
	0xfb400108,0xf89d0101,0xfb400108,0xf89e0101,0xfb400108,0xf8a00101,0xfb400108,0xf8a20101,
	0xfb400108,0xf8a40101,0xfb400108,0xf8a60101,0xfb400108,0xf8a80101,0xfb400108,0xf8a90101,
	0xfb400108,0xf8aa0101,0xfb400108,0xf8ab0101,0xfb400108,0xf8ac0101,0xfb400108,0xf8ad0101,
	0xfb400108,0xf8ae0101,0xfb400108,0xf8af0101,0xfb400108,0xf8b50101,0xfb400108,0xf8b60101,
	0xfb400108,0xf8b70101,0xfb400108,0xf8b80101,0xfb400108,0xf8ba0101,0xfb400108,0xf8bb0101,
	0xfb400108,0xf8bc0101,0xfb400108,0xf8bd0101,0xfb400108,0xf8bf0101,0xfb400108,0xf8c00101,
	0xfb400108,0xf8c20101,0xfb400108,0xf8c30101,0xfb400108,0xf8c40101,0xfb400108,0xf8c60101,
	0xfb400108,0xf8c70101,0xfb400108,0xf8c80101,0xfb400108,0xf8cc0101,0xfb400108,0xf8cd0101,
	0xfb400108,0xf8ce0101,0xfb400108,0xf8cf0101,0xfb400108,0xf8d10101,0xfb400108,0xf8d20101,
	0xfb400108,0xf8d30101,0xfb400108,0xf8d60101,0xfb400108,0xf8d70101,0xfb400108,0xf8d80101,
	0xfb400108,0xf8da0101,0xfb400108,0xf8db0101,0xfb400108,0xf8dc0101,0xfb400108,0xf8dd0101,
	0xfb400108,0xf8de0101,0xfb400108,0xf8df0101,0xfb400108,0xf8e00101,0xfb400108,0xf8e10101,
	0xfb400108,0xf8e20101,0xfb400108,0xf8e30101,0xfb400108,0xf8e40101,0xfb400108,0xf8e50101,
	0xfb400108,0xf8e60101,0xfb400108,0xf8e70101,0xfb400108,0xf8e90101,0xfb400108,0xf8ea0101,
	0xfb400108,0xf8eb0101,0xfb400108,0xf8ed0101,0xfb400108,0xf8ee0101,0xfb400108,0xf8ef0101,
	0xfb400108,0xf8f00101,0xfb400108,0xf8f10101,0xfb400108,0xf8f30101,0xfb400108,0xf8f50101,
	0xfb400108,0xf8f60101,0xfb400108,0xf8f80101,0xfb400108,0xf8f90101,0xfb400108,0xf8fb0101,
	0xfb400108,0xf8fc0101,0xfb400108,0xf8fd0101,0xfb400108,0xf8fe0101,0xfb400108,0xf8ff0101,
	0xfb400108,0xf9000101,0xfb400108,0xf9020101,0xfb400108,0xf9030101,0xfb400108,0xf9040101,
	0xfb400108,0xf9060101,0xfb400108,0xf9070101,0xfb400108,0xf9080101,0xfb400108,0xf9090101,
	0xfb400108,0xf90a0101,0xfb400108,0xf90b0101,0xfb400108,0xf90c0101,0xfb400108,0xf90d0101,
	0xfb400108,0xf90e0101,0xfb400108,0xf90f0101,0xfb400108,0xf9100101,0xfb400108,0xf9110101,
	0xfb400108,0xf9120101,0xfb400108,0xf9140101,0xfb400108,0xf9150101,0xfb400108,0xf9160101,
	0xfb400108,0xf9170101,0xfb400108,0xf9180101,0xfb400108,0xf9190101,0xfb400108,0xf91a0101,
	0xfb400108,0xf91b0101,0xfb400108,0xf91c0101,0xfb400108,0xf91d0101,0xfb400108,0xf91f0101,
	0xfb400108,0xf9200101,0xfb400108,0xf9210101,0xfb400108,0xf9220101,0xfb400108,0xf9230101,
	0xfb400108,0xf9250101,0xfb400108,0xf9260101,0xfb400108,0xf9270101,0xfb400108,0xf9280101,
	0xfb400108,0xf9290101,0xfb400108,0xf92a0101,0xfb400108,0xf92b0101,0xfb400108,0xf92c0101,
	0xfb400108,0xf92d0101,0xfb400108,0xf92e0101,0xfb400108,0xf92f0101,0xfb400108,0xf9300101,
	0xfb400108,0xf9310101,0xfb400108,0xf9320101,0xfb400108,0xf9330101,0xfb400108,0xf9350101,
	0xfb400108,0xf9360101,0xfb400108,0xf9370101,0xfb400108,0xf9380101,0xfb400108,0xf9390101,
	0xfb400108,0xf93d0101,0xfb400108,0xf93f0101,0xfb400108,0xf9420101,0xfb400108,0xf9430101,
	0xfb400108,0xf9440101,0xfb400108,0xf9450101,0xfb400108,0xf9470101,0xfb400108,0xf94a0101,
	0xfb400108,0xf94b0101,0xfb400108,0xf94c0101,0xfb400108,0xf94d0101,0xfb400108,0xf94e0101,
	0xfb400108,0xf94f0101,0xfb400108,0xf9500101,0xfb400108,0xf9510101,0xfb400108,0xf9520101,
	0xfb400108,0xf9540101,0xfb400108,0xf9550101,0xfb400108,0xf9580101,0xfb400108,0xf9590101,
	0xfb400108,0xf9610101,0xfb400108,0xf9630101,0xfb400108,0xf9640101,0xfb400108,0xf9660101,
	0xfb400108,0xf9690101,0xfb400108,0xf96a0101,0xfb400108,0xf96b0101,0xfb400108,0xf96c0101,
	0xfb400108,0xf96e0101,0xfb400108,0xf9700101,0xfb400108,0xf9710101,0xfb400108,0xf9720101,
	0xfb400108,0xf9730101,0xfb400108,0xf9740101,0xfb400108,0xf9750101,0xfb400108,0xf9760101,
	0xfb400108,0xf9790101,0xfb400108,0xf97b0101,0xfb400108,0xf97c0101,0xfb400108,0xf97d0101,
	0xfb400108,0xf97e0101,0xfb400108,0xf97f0101,0xfb400108,0xf9820101,0xfb400108,0xf9830101,
	0xfb400108,0xf9860101,0xfb400108,0xf9870101,0xfb400108,0xf9880101,0xfb400108,0xf9890101,
	0xfb400108,0xf98b0101,0xfb400108,0xf98c0101,0xfb400108,0xf98d0101,0xfb400108,0xf98e0101,
	0xfb400108,0xf9900101,0xfb400108,0xf9910101,0xfb400108,0xf9920101,0xfb400108,0xf9930101,
	0xfb400108,0xf9940101,0xfb400108,0xf9950101,0xfb400108,0xf9960101,0xfb400108,0xf9970101,
	0xfb400108,0xf9980101,0xfb400108,0xf9990101,0xfb400108,0xf99b0101,0xfb400108,0xf99c0101,
	0xfb400108,0xf99d0101,0xfb400108,0xf99e0101,0xfb400108,0xf99f0101,0xfb400108,0xf9a00101,
	0xfb400108,0xf9a10101,0xfb400108,0xf9a20101,0xfb400108,0xf9a30101,0xfb400108,0xf9a40101,
	0xfb400108,0xf9a50101,0xfb400108,0xf9a60101,0xfb400108,0xf9a80101,0xfb400108,0xf9a90101,
	0xfb400108,0xf9aa0101,0xfb400108,0xf9ab0101,0xfb400108,0xf9ac0101,0xfb400108,0xf9ad0101,
	0xfb400108,0xf9ae0101,0xfb400108,0xf9af0101,0xfb400108,0xf9b00101,0xfb400108,0xf9b10101,
	0xfb400108,0xf9b20101,0xfb400108,0xf9b40101,0xfb400108,0xf9b50101,0xfb400108,0xf9b60101,
	0xfb400108,0xf9b70101,0xfb400108,0xf9b80101,0xfb400108,0xf9bc0101,0xfb400108,0xf9bf0101,
	0xfb400108,0xf9c20101,0xfb400108,0xf9c40101,0xfb400108,0xf9c50101,0xfb400108,0xf9c70101,
	0xfb400108,0xf9c80101,0xfb400108,0xf9ca0101,0xfb400108,0xf9cc0101,0xfb400108,0xf9ce0101,
	0xfb400108,0xf9cf0101,0xfb400108,0xf9d00101,0xfb400108,0xf9d30101,0xfb400108,0xf9d40101,
	0xfb400108,0xf9d60101,0xfb400108,0xf9d70101,0xfb400108,0xf9d90101,0xfb400108,0xf9da0101,
	0xfb400108,0xf9db0101,0xfb400108,0xf9dc0101,0xfb400108,0xf9dd0101,0xfb400108,0xf9de0101,
	0xfb400108,0xf9e00101,0xfb400108,0xf9e10101,0xfb400108,0xf9e20101,0xfb400108,0xf9e50101,
	0xfb400108,0xf9e80101,0xfb400108,0xf9ea0101,0xfb400108,0xf9ec0101,0xfb400108,0xf9ee0101,
	0xfb400108,0xf9f10101,0xfb400108,0xf9f20101,0xfb400108,0xf9f30101,0xfb400108,0xf9f40101,
	0xfb400108,0xf9f50101,0xfb400108,0xf9f60101,0xfb400108,0xf9f70101,0xfb400108,0xf9f90101,
	0xfb400108,0xf9fa0101,0xfb400108,0xf9fc0101,0xfb400108,0xf9fe0101,0xfb400108,0xf9ff0101,
	0xfb400108,0xfa010101,0xfb400108,0xfa040101,0xfb400108,0xfa050101,0xfb400108,0xfa070101,
	0xfb400108,0xfa080101,0xfb400108,0xfa090101,0xfb400108,0xfa0a0101,0xfb400108,0xfa0c0101,
	0xfb400108,0xfa0f0101,0xfb400108,0xfa100101,0xfb400108,0xfa110101,0xfb400108,0xfa120101,
	0xfb400108,0xfa130101,0xfb400108,0xfa150101,0xfb400108,0xfa160101,0xfb400108,0xfa180101,
	0xfb400108,0xfa190101,0xfb400108,0xfa1b0101,0xfb400108,0xfa1c0101,0xfb400108,0xfa1d0101,
	0xfb400108,0xfa1f0101,0xfb400108,0xfa210101,0xfb400108,0xfa220101,0xfb400108,0xfa240101,
	0xfb400108,0xfa250101,0xfb400108,0xfa260101,0xfb400108,0xfa270101,0xfb400108,0xfa280101,
	0xfb400108,0xfa290101,0xfb400108,0xfa2a0101,0xfb400108,0xfa2b0101,0xfb400108,0xfa2c0101,
	0xfb400108,0xfa2d0101,0xfb400108,0xfa2e0101,0xfb400108,0xfa2f0101,0xfb400108,0xfa300101,
	0xfb400108,0xfa310101,0xfb400108,0xfa320101,0xfb400108,0xfa340101,0xfb400108,0xfa350101,
	0xfb400108,0xfa360101,0xfb400108,0xfa380101,0xfb400108,0xfa3a0101,0xfb400108,0xfa3e0101,
	0xfb400108,0xfa400101,0xfb400108,0xfa410101,0xfb400108,0xfa420101,0xfb400108,0xfa430101,
	0xfb400108,0xfa440101,0xfb400108,0xfa450101,0xfb400108,0xfa470101,0xfb400108,0xfa480101,
	0xfb400108,0xfa490101,0xfb400108,0xfa4a0101,0xfb400108,0xfa4b0101,0xfb400108,0xfa4c0101,
	0xfb400108,0xfa4d0101,0xfb400108,0xfa4e0101,0xfb400108,0xfa4f0101,0xfb400108,0xfa500101,
	0xfb400108,0xfa520101,0xfb400108,0xfa530101,0xfb400108,0xfa540101,0xfb400108,0xfa550101,
	0xfb400108,0xfa560101,0xfb400108,0xfa580101,0xfb400108,0xfa590101,0xfb400108,0xfa5a0101,
	0xfb400108,0xfa5b0101,0xfb400108,0xfa5c0101,0xfb400108,0xfa5d0101,0xfb400108,0xfa5e0101,
	0xfb400108,0xfa5f0101,0xfb400108,0xfa600101,0xfb400108,0xfa610101,0xfb400108,0xfa620101,
	0xfb400108,0xfa630101,0xfb400108,0xfa640101,0xfb400108,0xfa650101,0xfb400108,0xfa660101,
	0xfb400108,0xfa670101,0xfb400108,0xfa680101,0xfb400108,0xfa690101,0xfb400108,0xfa6a0101,
	0xfb400108,0xfa6b0101,0xfb400108,0xfa6c0101,0xfb400108,0xfa6d0101,0xfb400108,0xfa6e0101,
	0xfb400108,0xfa6f0101,0xfb400108,0xfa710101,0xfb400108,0xfa720101,0xfb400108,0xfa730101,
	0xfb400108,0xfa750101,0xfb400108,0xfa7b0101,0xfb400108,0xfa7c0101,0xfb400108,0xfa7d0101,
	0xfb400108,0xfa7e0101,0xfb400108,0xfa820101,0xfb400108,0xfa850101,0xfb400108,0xfa870101,
	0xfb400108,0xfa890101,0xfb400108,0xfa8a0101,0xfb400108,0xfa8b0101,0xfb400108,0xfa8c0101,
	0xfb400108,0xfa8e0101,0xfb400108,0xfa8f0101,0xfb400108,0xfa900101,0xfb400108,0xfa930101,
	0xfb400108,0xfa940101,0xfb400108,0xfa990101,0xfb400108,0xfa9a0101,0xfb400108,0xfa9b0101,
	0xfb400108,0xfa9e0101,0xfb400108,0xfaa10101,0xfb400108,0xfaa20101,0xfb400108,0xfaa30101,
	0xfb400108,0xfaa40101,0xfb400108,0xfaa70101,0xfb400108,0xfaa90101,0xfb400108,0xfaaa0101,
	0xfb400108,0xfaab0101,0xfb400108,0xfaae0101,0xfb400108,0xfaaf0101,0xfb400108,0xfab00101,
	0xfb400108,0xfab10101,0xfb400108,0xfab20101,0xfb400108,0xfab40101,0xfb400108,0xfab50101,
	0xfb400108,0xfab60101,0xfb400108,0xfab70101,0xfb400108,0xfab80101,0xfb400108,0xfab90101,
	0xfb400108,0xfaba0101,0xfb400108,0xfabb0101,0xfb400108,0xfabc0101,0xfb400108,0xfabd0101,
	0xfb400108,0xfabe0101,0xfb400108,0xfac00101,0xfb400108,0xfac10101,0xfb400108,0xfac20101,
	0xfb400108,0xfac30101,0xfb400108,0xfac40101,0xfb400108,0xfac50101,0xfb400108,0xfac60101,
	0xfb400108,0xfac70101,0xfb400108,0xfac80101,0xfb400108,0xfac90101,0xfb400108,0xfaca0101,
	0xfb400108,0xfacc0101,0xfb400108,0xfacd0101,0xfb400108,0xface0101,0xfb400108,0xfacf0101,
	0xfb400108,0xfad00101,0xfb400108,0xfad10101,0xfb400108,0xfad20101,0xfb400108,0xfad30101,
	0xfb400108,0xfad40101,0xfb400108,0xfad50101,0xfb400108,0xfad70101,0xfb400108,0xfad80101,
	0xfb400108,0xfada0101,0xfb400108,0xfadb0101,0xfb400108,0xfadc0101,0xfb400108,0xfadd0101,
	0xfb400108,0xfae10101,0xfb400108,0xfae20101,0xfb400108,0xfae40101,0xfb400108,0xfae70101,
	0xfb400108,0xfae80101,0xfb400108,0xfae90101,0xfb400108,0xfaea0101,0xfb400108,0xfaeb0101,
	0xfb400108,0xfaec0101,0xfb400108,0xfaee0101,0xfb400108,0xfaf00101,0xfb400108,0xfaf10101,
	0xfb400108,0xfaf20101,0xfb400108,0xfaf30101,0xfb400108,0xfaf40101,0xfb400108,0xfaf50101,
	0xfb400108,0xfaf60101,0xfb400108,0xfaf70101,0xfb400108,0xfaf80101,0xfb400108,0xfafb0101,
	0xfb400108,0xfafc0101,0xfb400108,0xfafe0101,0xfb400108,0xfb000101,0xfb400108,0xfb010101,
	0xfb400108,0xfb020101,0xfb400108,0xfb050101,0xfb400108,0xfb070101,0xfb400108,0xfb090101,
	0xfb400108,0xfb0c0101,0xfb400108,0xfb0d0101,0xfb400108,0xfb0e0101,0xfb400108,0xfb100101,
	0xfb400108,0xfb120101,0xfb400108,0xfb130101,0xfb400108,0xfb160101,0xfb400108,0xfb170101,
	0xfb400108,0xfb180101,0xfb400108,0xfb1a0101,0xfb400108,0xfb1c0101,0xfb400108,0xfb1d0101,
	0xfb400108,0xfb1f0101,0xfb400108,0xfb210101,0xfb400108,0xfb220101,0xfb400108,0xfb230101,
	0xfb400108,0xfb270101,0xfb400108,0xfb290101,0xfb400108,0xfb2d0101,0xfb400108,0xfb2f0101,
	0xfb400108,0xfb300101,0xfb400108,0xfb320101,0xfb400108,0xfb340101,0xfb400108,0xfb350101,
	0xfb400108,0xfb360101,0xfb400108,0xfb370101,0xfb400108,0xfb390101,0xfb400108,0xfb3b0101,
	0xfb400108,0xfb3d0101,0xfb400108,0xfb3f0101,0xfb400108,0xfb400101,0xfb400108,0xfb410101,
	0xfb400108,0xfb420101,0xfb400108,0xfb430101,0xfb400108,0xfb440101,0xfb400108,0xfb460101,
	0xfb400108,0xfb480101,0xfb400108,0xfb4a0101,0xfb400108,0xfb4d0101,0xfb400108,0xfb4e0101,
	0xfb400108,0xfb530101,0xfb400108,0xfb550101,0xfb400108,0xfb570101,0xfb400108,0xfb590101,
	0xfb400108,0xfb5c0101,0xfb400108,0xfb5e0101,0xfb400108,0xfb5f0101,0xfb400108,0xfb610101,
	0xfb400108,0xfb630101,0xfb400108,0xfb640101,0xfb400108,0xfb650101,0xfb400108,0xfb660101,
	0xfb400108,0xfb670101,0xfb400108,0xfb680101,0xfb400108,0xfb690101,0xfb400108,0xfb6a0101,
	0xfb400108,0xfb6b0101,0xfb400108,0xfb6c0101,0xfb400108,0xfb6d0101,0xfb400108,0xfb6f0101,
	0xfb400108,0xfb700101,0xfb400108,0xfb730101,0xfb400108,0xfb740101,0xfb400108,0xfb760101,
	0xfb400108,0xfb780101,0xfb400108,0xfb7a0101,0xfb400108,0xfb7c0101,0xfb400108,0xfb7d0101,
	0xfb400108,0xfb7f0101,0xfb400108,0xfb810101,0xfb400108,0xfb820101,0xfb400108,0xfb830101,
	0xfb400108,0xfb840101,0xfb400108,0xfb860101,0xfb400108,0xfb870101,0xfb400108,0xfb880101,
	0xfb400108,0xfb890101,0xfb400108,0xfb8a0101,0xfb400108,0xfb8b0101,0xfb400108,0xfb8c0101,
	0xfb400108,0xfb8e0101,0xfb400108,0xfb8f0101,0xfb400108,0xfb910101,0xfb400108,0xfb920101,
	0xfb400108,0xfb930101,0xfb400108,0xfb960101,0xfb400108,0xfb980101,0xfb400108,0xfb990101,
	0xfb400108,0xfb9a0101,0xfb400108,0xfb9b0101,0xfb400108,0xfb9e0101,0xfb400108,0xfb9f0101,
	0xfb400108,0xfba00101,0xfb400108,0xfba30101,0xfb400108,0xfba40101,0xfb400108,0xfba50101,
	0xfb400108,0xfbae0101,0xfb400108,0xfbaf0101,0xfb400108,0xfbb00101,0xfb400108,0xfbb20101,
	0xfb400108,0xfbb30101,0xfb400108,0xfbb50101,0xfb400108,0xfbb60101,0xfb400108,0xfbb70101,
	0xfb400108,0xfbb90101,0xfb400108,0xfbba0101,0xfb400108,0xfbbb0101,0xfb400108,0xfbbc0101,
	0xfb400108,0xfbbd0101,0xfb400108,0xfbbe0101,0xfb400108,0xfbbf0101,0xfb400108,0xfbc00101,
	0xfb400108,0xfbc20101,0xfb400108,0xfbc30101,0xfb400108,0xfbc40101,0xfb400108,0xfbc50101,
	0xfb400108,0xfbc80101,0xfb400108,0xfbc90101,0xfb400108,0xfbca0101,0xfb400108,0xfbcb0101,
	0xfb400108,0xfbcd0101,0xfb400108,0xfbce0101,0xfb400108,0xfbcf0101,0xfb400108,0xfbd00101,
	0xfb400108,0xfbd20101,0xfb400108,0xfbd40101,0xfb400108,0xfbd50101,0xfb400108,0xfbd60101,
	0xfb400108,0xfbd70101,0xfb400108,0xfbd80101,0xfb400108,0xfbdb0101,0xfb400108,0xfbdc0101,
	0xfb400108,0xfbde0101,0xfb400108,0xfbdf0101,0xfb400108,0xfbe00101,0xfb400108,0xfbe20101,
	0xfb400108,0xfbe30101,0xfb400108,0xfbe40101,0xfb400108,0xfbe70101,0xfb400108,0xfbe80101,
	0xfb400108,0xfbe90101,0xfb400108,0xfbeb0101,0xfb400108,0xfbec0101,0xfb400108,0xfbed0101,
	0xfb400108,0xfbef0101,0xfb400108,0xfbf00101,0xfb400108,0xfbf20101,0xfb400108,0xfbf30101,
	0xfb400108,0xfbf40101,0xfb400108,0xfbf50101,0xfb400108,0xfbf60101,0xfb400108,0xfbf80101,
	0xfb400108,0xfbf90101,0xfb400108,0xfbfa0101,0xfb400108,0xfbfb0101,0xfb400108,0xfbfd0101,
	0xfb400108,0xfbff0101,0xfb400108,0xfc000101,0xfb400108,0xfc010101,0xfb400108,0xfc020101,
	0xfb400108,0xfc030101,0xfb400108,0xfc040101,0xfb400108,0xfc050101,0xfb400108,0xfc060101,
	0xfb400108,0xfc080101,0xfb400108,0xfc090101,0xfb400108,0xfc0a0101,0xfb400108,0xfc0d0101,
	0xfb400108,0xfc0e0101,0xfb400108,0xfc100101,0xfb400108,0xfc110101,0xfb400108,0xfc120101,
	0xfb400108,0xfc130101,0xfb400108,0xfc140101,0xfb400108,0xfc150101,0xfb400108,0xfc170101,
	0xfb400108,0xfc180101,0xfb400108,0xfc190101,0xfb400108,0xfc1a0101,0xfb400108,0xfc1b0101,
	0xfb400108,0xfc1c0101,0xfb400108,0xfc1d0101,0xfb400108,0xfc1e0101,0xfb400108,0xfc200101,
	0xfb400108,0xfc210101,0xfb400108,0xfc220101,0xfb400108,0xfc230101,0xfb400108,0xfc240101,
	0xfb400108,0xfc250101,0xfb400108,0xfc280101,0xfb400108,0xfc290101,0xfb400108,0xfc2b0101,
	0xfb400108,0xfc2c0101,0xfb400108,0xfc2d0101,0xfb400108,0xfc2e0101,0xfb400108,0xfc2f0101,
	0xfb400108,0xfc300101,0xfb400108,0xfc310101,0xfb400108,0xfc320101,0xfb400108,0xfc330101,
	0xfb400108,0xfc340101,0xfb400108,0xfc350101,0xfb400108,0xfc360101,0xfb400108,0xfc370101,
	0xfb400108,0xfc390101,0xfb400108,0xfc3a0101,0xfb400108,0xfc3b0101,0xfb400108,0xfc3c0101,
	0xfb400108,0xfc3d0101,0xfb400108,0xfc3e0101,0xfb400108,0xfc420101,0xfb400108,0xfc430101,
	0xfb400108,0xfc440101,0xfb400108,0xfc450101,0xfb400108,0xfc460101,0xfb400108,0xfc470101,
	0xfb400108,0xfc480101,0xfb400108,0xfc490101,0xfb400108,0xfc4a0101,0xfb400108,0xfc4b0101,
	0xfb400108,0xfc4c0101,0xfb400108,0xfc4e0101,0xfb400108,0xfc4f0101,0xfb400108,0xfc500101,
	0xfb400108,0xfc510101,0xfb400108,0xfc520101,0xfb400108,0xfc530101,0xfb400108,0xfc540101,
	0xfb400108,0xfc550101,0xfb400108,0xfc560101,0xfb400108,0xfc570101,0xfb400108,0xfc580101,
	0xfb400108,0xfc590101,0xfb400108,0xfc5a0101,0xfb400108,0xfc5b0101,0xfb400108,0xfc5c0101,
	0xfb400108,0xfc5d0101,0xfb400108,0xfc5e0101,0xfb400108,0xfc5f0101,0xfb400108,0xfc600101,
	0xfb400108,0xfc610101,0xfb400108,0xfc620101,0xfb400108,0xfc630101,0xfb400108,0xfc640101,
	0xfb400108,0xfc650101,0xfb400108,0xfc660101,0xfb400108,0xfc670101,0xfb400108,0xfc680101,
	0xfb400108,0xfc690101,0xfb400108,0xfc6a0101,0xfb400108,0xfc6b0101,0xfb400108,0xfc6c0101,
	0xfb400108,0xfc6d0101,0xfb400108,0xfc6e0101,0xfb400108,0xfc6f0101,0xfb400108,0xfc700101,
	0xfb400108,0xfc710101,0xfb400108,0xfc720101,0xfb400108,0xfc750101,0xfb400108,0xfc760101,
	0xfb400108,0xfc770101,0xfb400108,0xfc780101,0xfb400108,0xfc790101,0xfb400108,0xfc7a0101,
	0xfb400108,0xfc7e0101,0xfb400108,0xfc7f0101,0xfb400108,0xfc800101,0xfb400108,0xfc810101,
	0xfb400108,0xfc820101,0xfb400108,0xfc830101,0xfb400108,0xfc840101,0xfb400108,0xfc850101,
	0xfb400108,0xfc860101,0xfb400108,0xfc870101,0xfb400108,0xfc880101,0xfb400108,0xfc8a0101,
	0xfb400108,0xfc8b0101,0xfb400108,0xfc8c0101,0xfb400108,0xfc8d0101,0xfb400108,0xfc8e0101,
	0xfb400108,0xfc8f0101,0xfb400108,0xfc900101,0xfb400108,0xfc930101,0xfb400108,0xfc940101,
	0xfb400108,0xfc960101,0xfb400108,0xfc990101,0xfb400108,0xfc9a0101,0xfb400108,0xfc9b0101,
	0xfb400108,0xfca00101,0xfb400108,0xfca10101,0xfb400108,0xfca30101,0xfb400108,0xfca60101,
	0xfb400108,0xfca70101,0xfb400108,0xfca80101,0xfb400108,0xfca90101,0xfb400108,0xfcab0101,
	0xfb400108,0xfcac0101,0xfb400108,0xfcad0101,0xfb400108,0xfcaf0101,0xfb400108,0xfcb00101,
	0xfb400108,0xfcb40101,0xfb400108,0xfcb50101,0xfb400108,0xfcb60101,0xfb400108,0xfcb70101,
	0xfb400108,0xfcb80101,0xfb400108,0xfcba0101,0xfb400108,0xfcbb0101,0xfb400108,0xfcbf0101,
	0xfb400108,0xfcc00101,0xfb400108,0xfcc20101,0xfb400108,0xfcc30101,0xfb400108,0xfcc40101,
	0xfb400108,0xfcc60101,0xfb400108,0xfcc90101,0xfb400108,0xfccb0101,0xfb400108,0xfcce0101,
	0xfb400108,0xfccf0101,0xfb400108,0xfcd00101,0xfb400108,0xfcd10101,0xfb400108,0xfcd20101,
	0xfb400108,0xfcd30101,0xfb400108,0xfcd40101,0xfb400108,0xfcd80101,0xfb400108,0xfcda0101,
	0xfb400108,0xfcdb0101,0xfb400108,0xfcdd0101,0xfb400108,0xfcde0101,0xfb400108,0xfce10101,
	0xfb400108,0xfce20101,0xfb400108,0xfce30101,0xfb400108,0xfce40101,0xfb400108,0xfce50101,
	0xfb400108,0xfce60101,0xfb400108,0xfce70101,0xfb400108,0xfce90101,0xfb400108,0xfcea0101,
	0xfb400108,0xfceb0101,0xfb400108,0xfcec0101,0xfb400108,0xfced0101,0xfb400108,0xfcee0101,
	0xfb400108,0xfcf00101,0xfb400108,0xfcf10101,0xfb400108,0xfcf20101,0xfb400108,0xfcf30101,
	0xfb400108,0xfcf40101,0xfb400108,0xfcf50101,0xfb400108,0xfcf60101,0xfb400108,0xfcf70101,
	0xfb400108,0xfcf90101,0xfb400108,0xfcfa0101,0xfb400108,0xfcfc0101,0xfb400108,0xfcfd0101,
	0xfb400108,0xfcfe0101,0xfb400108,0xfcff0101,0xfb400108,0xfd000101,0xfb400108,0xfd010101,
	0xfb400108,0xfd020101,0xfb400108,0xfd030101,0xfb400108,0xfd040101,0xfb400108,0xfd050101,
	0xfb400108,0xfd060101,0xfb400108,0xfd070101,0xfb400108,0xfd080101,0xfb400108,0xfd090101,
	0xfb400108,0xfd0b0101,0xfb400108,0xfd0c0101,0xfb400108,0xfd0d0101,0xfb400108,0xfd0e0101,
	0xfb400108,0xfd0f0101,0xfb400108,0xfd100101,0xfb400108,0xfd110101,0xfb400108,0xfd120101,
	0xfb400108,0xfd130101,0xfb400108,0xfd140101,0xfb400108,0xfd150101,0xfb400108,0xfd160101,
	0xfb400108,0xfd170101,0xfb400108,0xfd180101,0xfb400108,0xfd190101,0xfb400108,0xfd1a0101,
	0xfb400108,0xfd1b0101,0xfb400108,0xfd1c0101,0xfb400108,0xfd1d0101,0xfb400108,0xfd1e0101,
	0xfb400108,0xfd1f0101,0xfb400108,0xfd210101,0xfb400108,0xfd230101,0xfb400108,0xfd240101,
	0xfb400108,0xfd250101,0xfb400108,0xfd260101,0xfb400108,0xfd280101,0xfb400108,0xfd290101,
	0xfb400108,0xfd2a0101,0xfb400108,0xfd2c0101,0xfb400108,0xfd2d0101,0xfb400108,0xfd2e0101,
	0xfb400108,0xfd300101,0xfb400108,0xfd310101,0xfb400108,0xfd320101,0xfb400108,0xfd330101,
	0xfb400108,0xfd340101,0xfb400108,0xfd350101,0xfb400108,0xfd360101,0xfb400108,0xfd370101,
	0xfb400108,0xfd380101,0xfb400108,0xfd390101,0xfb400108,0xfd3a0101,0xfb400108,0xfd3b0101,
	0xfb400108,0xfd3c0101,0xfb400108,0xfd3d0101,0xfb400108,0xfd3e0101,0xfb400108,0xfd3f0101,
	0xfb400108,0xfd400101,0xfb400108,0xfd410101,0xfb400108,0xfd420101,0xfb400108,0xfd430101,
	0xfb400108,0xfd440101,0xfb400108,0xfd450101,0xfb400108,0xfd460101,0xfb400108,0xfd470101,
	0xfb400108,0xfd480101,0xfb400108,0xfd490101,0xfb400108,0xfd4a0101,0xfb400108,0xfd4b0101,
	0xfb400108,0xfd4c0101,0xfb400108,0xfd4d0101,0xfb400108,0xfd4e0101,0xfb400108,0xfd4f0101,
	0xfb400108,0xfd500101,0xfb400108,0xfd510101,0xfb400108,0xfd520101,0xfb400108,0xfd530101,
	0xfb400108,0xfd540101,0xfb400108,0xfd550101,0xfb400108,0xfd560101,0xfb400108,0xfd570101,
	0xfb400108,0xfd580101,0xfb400108,0xfd590101,0xfb400108,0xfd5a0101,0xfb400108,0xfd5b0101,
	0xfb400108,0xfd5c0101,0xfb400108,0xfd5d0101,0xfb400108,0xfd5e0101,0xfb400108,0xfd5f0101,
	0xfb400108,0xfd600101,0xfb400108,0xfd610101,0xfb400108,0xfd620101,0xfb400108,0xfd630101,
	0xfb400108,0xfd640101,0xfb400108,0xfd650101,0xfb400108,0xfd660101,0xfb400108,0xfd670101,
	0xfb400108,0xfd680101,0xfb400108,0xfd690101,0xfb400108,0xfd6a0101,0xfb400108,0xfd6b0101,
	0xfb400108,0xfd6c0101,0xfb400108,0xfd6d0101,0xfb400108,0xfd6f0101,0xfb400108,0xfd700101,
	0xfb400108,0xfd710101,0xfb400108,0xfd720101,0xfb400108,0xfd730101,0xfb400108,0xfd740101,
	0xfb400108,0xfd750101,0xfb400108,0xfd760101,0xfb400108,0xfd780101,0xfb400108,0xfd790101,
	0xfb400108,0xfd7a0101,0xfb400108,0xfd7b0101,0xfb400108,0xfd7c0101,0xfb400108,0xfd7d0101,
	0xfb400108,0xfd7e0101,0xfb400108,0xfd7f0101,0xfb400108,0xfd800101,0xfb400108,0xfd810101,
	0xfb400108,0xfd820101,0xfb400108,0xfd830101,0xfb400108,0xfd840101,0xfb400108,0xfd850101,
	0xfb400108,0xfd860101,0xfb400108,0xfd870101,0xfb400108,0xfd880101,0xfb400108,0xfd890101,
	0xfb400108,0xfd8a0101,0xfb400108,0xfd8b0101,0xfb400108,0xfd8c0101,0xfb400108,0xfd8d0101,
	0xfb400108,0xfd8e0101,0xfb400108,0xfd8f0101,0xfb400108,0xfd900101,0xfb400108,0xfd910101,
	0xfb400108,0xfd920101,0xfb400108,0xfd930101,0xfb400108,0xfd940101,0xfb400108,0xfd950101,
	0xfb400108,0xfd960101,0xfb400108,0xfd970101,0xfb400108,0xfd980101,0xfb400108,0xfd990101,
	0xfb400108,0xfd9a0101,0xfb400108,0xfd9b0101,0xfb400108,0xfd9c0101,0xfb400108,0xfd9d0101,
	0xfb400108,0xfd9e0101,0xfb400108,0xfd9f0101,0xfb400108,0xfda00101,0xfb400108,0xfda10101,
	0xfb400108,0xfda20101,0xfb400108,0xfda30101,0xfb400108,0xfda40101,0xfb400108,0xfda50101,
	0xfb400108,0xfda70101,0xfb400108,0xfda80101,0xfb400108,0xfda90101,0xfb400108,0xfdaa0101,
	0xfb400108,0xfdab0101,0xfb400108,0xfdac0101,0xfb400108,0xfdad0101,0xfb400108,0xfdaf0101,
	0xfb400108,0xfdb00101,0xfb400108,0xfdb10101,0xfb400108,0xfdb20101,0xfb400108,0xfdb30101,
	0xfb400108,0xfdb40101,0xfb400108,0xfdb50101,0xfb400108,0xfdb60101,0xfb400108,0xfdb70101,
	0xfb400108,0xfdb80101,0xfb400108,0xfdb90101,0xfb400108,0xfdba0101,0xfb400108,0xfdbb0101,
	0xfb400108,0xfdbc0101,0xfb400108,0xfdbd0101,0xfb400108,0xfdbe0101,0xfb400108,0xfdbf0101,
	0xfb400108,0xfdc00101,0xfb400108,0xfdc10101,0xfb400108,0xfdc20101,0xfb400108,0xfdc30101,
	0xfb400108,0xfdc40101,0xfb400108,0xfdc50101,0xfb400108,0xfdc60101,0xfb400108,0xfdc70101,
	0xfb400108,0xfdc80101,0xfb400108,0xfdc90101,0xfb400108,0xfdca0101,0xfb400108,0xfdcb0101,
	0xfb400108,0xfdcc0101,0xfb400108,0xfdcd0101,0xfb400108,0xfdce0101,0xfb400108,0xfdcf0101,
	0xfb400108,0xfdd00101,0xfb400108,0xfdd10101,0xfb400108,0xfdd20101,0xfb400108,0xfdd30101,
	0xfb400108,0xfdd40101,0xfb400108,0xfdd50101,0xfb400108,0xfdd60101,0xfb400108,0xfdd70101,
	0xfb400108,0xfdd80101,0xfb400108,0xfdd90101,0xfb400108,0xfdda0101,0xfb400108,0xfddb0101,
	0xfb400108,0xfddc0101,0xfb400108,0xfddd0101,0xfb400108,0xfdde0101,0xfb400108,0xfddf0101,
	0xfb400108,0xfde00101,0xfb400108,0xfde10101,0xfb400108,0xfde20101,0xfb400108,0xfde30101,
	0xfb400108,0xfde40101,0xfb400108,0xfde50101,0xfb400108,0xfde60101,0xfb400108,0xfde70101,
	0xfb400108,0xfde80101,0xfb400108,0xfde90101,0xfb400108,0xfdea0101,0xfb400108,0xfdeb0101,
	0xfb400108,0xfdec0101,0xfb400108,0xfded0101,0xfb400108,0xfdee0101,0xfb400108,0xfdef0101,
	0xfb400108,0xfdf00101,0xfb400108,0xfdf10101,0xfb400108,0xfdf20101,0xfb400108,0xfdf30101,
	0xfb400108,0xfdf40101,0xfb400108,0xfdf50101,0xfb400108,0xfdf60101,0xfb400108,0xfdf70101,
	0xfb400108,0xfdf80101,0xfb400108,0xfdf90101,0xfb400108,0xfdfa0101,0xfb400108,0xfdfb0101,
	0xfb400108,0xfdfc0101,0xfb400108,0xfdfd0101,0xfb400108,0xfdfe0101,0xfb400108,0xfdff0101,
	0xfb400108,0xfe000101,0xfb400108,0xfe010101,0xfb400108,0xfe020101,0xfb400108,0xfe030101,
	0xfb400108,0xfe040101,0xfb400108,0xfe050101,0xfb400108,0xfe060101,0xfb400108,0xfe070101,
	0xfb400108,0xfe080101,0xfb400108,0xfe090101,0xfb400108,0xfe0a0101,0xfb400108,0xfe0b0101,
	0xfb400108,0xfe0c0101,0xfb400108,0xfe0d0101,0xfb400108,0xfe0e0101,0xfb400108,0xfe0f0101,
	0xfb400108,0xfe100101,0xfb400108,0xfe110101,0xfb400108,0xfe120101,0xfb400108,0xfe130101,
	0xfb400108,0xfe140101,0xfb400108,0xfe150101,0xfb400108,0xfe160101,0xfb400108,0xfe170101,
	0xfb400108,0xfe180101,0xfb400108,0xfe190101,0xfb400108,0xfe1a0101,0xfb400108,0xfe1b0101,
	0xfb400108,0xfe1c0101,0xfb400108,0xfe1d0101,0xfb400108,0xfe1e0101,0xfb400108,0xfe1f0101,
	0xfb400108,0xfe200101,0xfb400108,0xfe210101,0xfb400108,0xfe220101,0xfb400108,0xfe230101,
	0xfb400108,0xfe240101,0xfb400108,0xfe250101,0xfb400108,0xfe260101,0xfb400108,0xfe270101,
	0xfb400108,0xfe280101,0xfb400108,0xfe290101,0xfb400108,0xfe2a0101,0xfb400108,0xfe2b0101,
	0xfb400108,0xfe2c0101,0xfb400108,0xfe2d0101,0xfb400108,0xfe2e0101,0xfb400108,0xfe2f0101,
	0xfb400108,0xfe300101,0xfb400108,0xfe310101,0xfb400108,0xfe320101,0xfb400108,0xfe330101,
	0xfb400108,0xfe340101,0xfb400108,0xfe350101,0xfb400108,0xfe360101,0xfb400108,0xfe370101,
	0xfb400108,0xfe380101,0xfb400108,0xfe390101,0xfb400108,0xfe3a0101,0xfb400108,0xfe3c0101,
	0xfb400108,0xfe3d0101,0xfb400108,0xfe3e0101,0xfb400108,0xfe3f0101,0xfb400108,0xfe400101,
	0xfb400108,0xfe420101,0xfb400108,0xfe430101,0xfb400108,0xfe440101,0xfb400108,0xfe450101,
	0xfb400108,0xfe460101,0xfb400108,0xfe480101,0xfb400108,0xfe490101,0xfb400108,0xfe4a0101,
	0xfb400108,0xfe4b0101,0xfb400108,0xfe4c0101,0xfb400108,0xfe4d0101,0xfb400108,0xfe4e0101,
	0xfb400108,0xfe4f0101,0xfb400108,0xfe500101,0xfb400108,0xfe510101,0xfb400108,0xfe520101,
	0xfb400108,0xfe530101,0xfb400108,0xfe540101,0xfb400108,0xfe550101,0xfb400108,0xfe560101,
	0xfb400108,0xfe570101,0xfb400108,0xfe580101,0xfb400108,0xfe590101,0xfb400108,0xfe5a0101,
	0xfb400108,0xfe5b0101,0xfb400108,0xfe5c0101,0xfb400108,0xfe5d0101,0xfb400108,0xfe5e0101,
	0xfb400108,0xfe5f0101,0xfb400108,0xfe600101,0xfb400108,0xfe610101,0xfb400108,0xfe620101,
	0xfb400108,0xfe630101,0xfb400108,0xfe640101,0xfb400108,0xfe650101,0xfb400108,0xfe660101,
	0xfb400108,0xfe670101,0xfb400108,0xfe680101,0xfb400108,0xfe690101,0xfb400108,0xfe6a0101,
	0xfb400108,0xfe6b0101,0xfb400108,0xfe6c0101,0xfb400108,0xfe6d0101,0xfb400108,0xfe6e0101,
	0xfb400108,0xfe6f0101,0xfb400108,0xfe700101,0xfb400108,0xfe710101,0xfb400108,0xfe720101,
	0xfb400108,0xfe730101,0xfb400108,0xfe740101,0xfb400108,0xfe750101,0xfb400108,0xfe760101,
	0xfb400108,0xfe770101,0xfb400108,0xfe780101,0xfb400108,0xfe790101,0xfb400108,0xfe7a0101,
	0xfb400108,0xfe7b0101,0xfb400108,0xfe7c0101,0xfb400108,0xfe7d0101,0xfb400108,0xfe7e0101,
	0xfb400108,0xfe7f0101,0xfb400108,0xfe800101,0xfb400108,0xfe810101,0xfb400108,0xfe830101,
	0xfb400108,0xfe840101,0xfb400108,0xfe850101,0xfb400108,0xfe860101,0xfb400108,0xfe870101,
	0xfb400108,0xfe880101,0xfb400108,0xfe890101,0xfb400108,0xfe8a0101,0xfb400108,0xfe8b0101,
	0xfb400108,0xfe8c0101,0xfb400108,0xfe8d0101,0xfb400108,0xfe8e0101,0xfb400108,0xfe8f0101,
	0xfb400108,0xfe900101,0xfb400108,0xfe910101,0xfb400108,0xfe920101,0xfb400108,0xfe930101,
	0xfb400108,0xfe940101,0xfb400108,0xfe950101,0xfb400108,0xfe960101,0xfb400108,0xfe970101,
	0xfb400108,0xfe980101,0xfb400108,0xfe990101,0xfb400108,0xfe9a0101,0xfb400108,0xfe9c0101,
	0xfb400108,0xfe9d0101,0xfb400108,0xfe9e0101,0xfb400108,0xfeae0101,0xfb400108,0xfeb40101,
	0xfb400108,0xfebb0101,0xfb400108,0xfebc0101,0xfb400108,0xfed60101,0xfb400108,0xfee40101,
	0xfb400108,0xfeec0101,0xfb400108,0xfef90101,0xfb400108,0xff0a0101,0xfb400108,0xff100101,
	0xfb400108,0xff1e0101,0xfb400108,0xff370101,0xfb400108,0xff390101,0xfb400108,0xff3b0101,
	0xfb400108,0xff3c0101,0xfb400108,0xff3d0101,0xfb400108,0xff3e0101,0xfb400108,0xff3f0101,
	0xfb400108,0xff400101,0xfb400108,0xff410101,0xfb400108,0xff430101,0xfb400108,0xff460101,
	0xfb400108,0xff470101,0xfb400108,0xff480101,0xfb400108,0xff490101,0xfb400108,0xff4a0101,
	0xfb400108,0xff4b0101,0xfb400108,0xff4c0101,0xfb400108,0xff4d0101,0xfb400108,0xff4e0101,
	0xfb400108,0xff4f0101,0xfb400108,0xff520101,0xfb400108,0xff530101,0xfb400108,0xff560101,
	0xfb400108,0xff590101,0xfb400108,0xff5b0101,0xfb400108,0xff5c0101,0xfb400108,0xff5d0101,
	0xfb400108,0xff5e0101,0xfb400108,0xff600101,0xfb400108,0xff630101,0xfb400108,0xff640101,
	0xfb400108,0xff650101,0xfb400108,0xff660101,0xfb400108,0xff670101,0xfb400108,0xff6b0101,
	0xfb400108,0xff6c0101,0xfb400108,0xff6d0101,0xfb400108,0xff6f0101,0xfb400108,0xff700101,
	0xfb400108,0xff730101,0xfb400108,0xff750101,0xfb400108,0xff760101,0xfb400108,0xff770101,
	0xfb400108,0xff780101,0xfb400108,0xff7a0101,0xfb400108,0xff7b0101,0xfb400108,0xff7c0101,
	0xfb400108,0xff7d0101,0xfb400108,0xff7f0101,0xfb400108,0xff800101,0xfb400108,0xff820101,
	0xfb400108,0xff830101,0xfb400108,0xff840101,0xfb400108,0xff850101,0xfb400108,0xff860101,
	0xfb400108,0xff870101,0xfb400108,0xff880101,0xfb400108,0xff890101,0xfb400108,0xff8b0101,
	0xfb400108,0xff8d0101,0xfb400108,0xff8f0101,0xfb400108,0xff900101,0xfb400108,0xff910101,
	0xfb400108,0xff920101,0xfb400108,0xff930101,0xfb400108,0xff950101,0xfb400108,0xff960101,
	0xfb400108,0xff970101,0xfb400108,0xff980101,0xfb400108,0xff990101,0xfb400108,0xff9b0101,
	0xfb400108,0xff9c0101,0xfb400108,0xffa00101,0xfb400108,0xffa20101,0xfb400108,0xffa30101,
	0xfb400108,0xffa50101,0xfb400108,0xffa60101,0xfb400108,0xffa80101,0xfb400108,0xffa90101,
	0xfb400108,0xffaa0101,0xfb400108,0xffab0101,0xfb400108,0xffac0101,0xfb400108,0xffad0101,
	0xfb400108,0xffae0101,0xfb400108,0xffb10101,0xfb400108,0xffb30101,0xfb400108,0xffb40101,
	0xfb400108,0xffb50101,0xfb400108,0xffb60101,0xfb400108,0xffb70101,0xfb400108,0xffba0101,
	0xfb400108,0xffbb0101,0xfb400108,0xffbe0101,0xfb400108,0xffc00101,0xfb400108,0xffc20101,
	0xfb400108,0xffc30101,0xfb400108,0xffc40101,0xfb400108,0xffc60101,0xfb400108,0xffc70101,
	0xfb400108,0xffc80101,0xfb400108,0xffc90101,0xfb400108,0xffcb0101,0xfb400108,0xffcd0101,
	0xfb400108,0xffcf0101,0xfb400108,0xffd00101,0xfb400108,0xffd10101,0xfb400108,0xffd20101,
	0xfb400108,0xffd30101,0xfb400108,0xffd60101,0xfb400108,0xffd70101,0xfb400108,0xffd90101,
	0xfb400108,0xffda0101,0xfb400108,0xffdb0101,0xfb400108,0xffdc0101,0xfb400108,0xffdd0101,
	0xfb400108,0xffde0101,0xfb400108,0xffe20101,0xfb400108,0xffe30101,0xfb400108,0xffe40101,
	0xfb400108,0xffe70101,0xfb400108,0xffe80101,0xfb400108,0xffea0101,0xfb400108,0xffeb0101,
	0xfb400108,0xffec0101,0xfb400108,0xffed0101,0xfb400108,0xffef0101,0xfb400108,0xfff20101,
	0xfb400108,0xfff40101,0xfb400108,0xfff50101,0xfb400108,0xfff60101,0xfb400108,0xfff70101,
	0xfb400108,0xfff80101,0xfb400108,0xfff90101,0xfb400108,0xfffa0101,0xfb400108,0xfffd0101,
	0xfb400108,0xfffe0101,0xfb400108,0xffff0101,0xfb410108,0x80020101,0xfb410108,0x80070101,
	0xfb410108,0x80080101,0xfb410108,0x80090101,0xfb410108,0x800a0101,0xfb410108,0x800e0101,
	0xfb410108,0x800f0101,0xfb410108,0x80110101,0xfb410108,0x80130101,0xfb410108,0x801a0101,
	0xfb410108,0x801b0101,0xfb410108,0x801d0101,0xfb410108,0x801e0101,0xfb410108,0x801f0101,
	0xfb410108,0x80210101,0xfb410108,0x80230101,0xfb410108,0x80240101,0xfb410108,0x802b0101,
	0xfb410108,0x802c0101,0xfb410108,0x802d0101,0xfb410108,0x802e0101,0xfb410108,0x802f0101,
	0xfb410108,0x80300101,0xfb410108,0x80320101,0xfb410108,0x80340101,0xfb410108,0x80390101,
	0xfb410108,0x803a0101,0xfb410108,0x803c0101,0xfb410108,0x803e0101,0xfb410108,0x80400101,
	0xfb410108,0x80410101,0xfb410108,0x80440101,0xfb410108,0x80450101,0xfb410108,0x80470101,
	0xfb410108,0x80480101,0xfb410108,0x80490101,0xfb410108,0x804e0101,0xfb410108,0x804f0101,
	0xfb410108,0x80500101,0xfb410108,0x80510101,0xfb410108,0x80530101,0xfb410108,0x80550101,
	0xfb410108,0x80560101,0xfb410108,0x80570101,0xfb410108,0x80590101,0xfb410108,0x805b0101,
	0xfb410108,0x805c0101,0xfb410108,0x805d0101,0xfb410108,0x805e0101,0xfb410108,0x805f0101,
	0xfb410108,0x80600101,0xfb410108,0x80610101,0xfb410108,0x80620101,0xfb410108,0x80630101,
	0xfb410108,0x80640101,0xfb410108,0x80650101,0xfb410108,0x80660101,0xfb410108,0x80670101,
	0xfb410108,0x80680101,0xfb410108,0x806b0101,0xfb410108,0x806c0101,0xfb410108,0x806d0101,
	0xfb410108,0x806e0101,0xfb410108,0x806f0101,0xfb410108,0x80700101,0xfb410108,0x80720101,
	0xfb410108,0x80730101,0xfb410108,0x80740101,0xfb410108,0x80750101,0xfb410108,0x80760101,
	0xfb410108,0x80770101,0xfb410108,0x80780101,0xfb410108,0x80790101,0xfb410108,0x807a0101,
	0xfb410108,0x807b0101,0xfb410108,0x807c0101,0xfb410108,0x807d0101,0xfb410108,0x807e0101,
	0xfb410108,0x80810101,0xfb410108,0x80820101,0xfb410108,0x80850101,0xfb410108,0x80880101,
	0xfb410108,0x808a0101,0xfb410108,0x808d0101,0xfb410108,0x808e0101,0xfb410108,0x808f0101,
	0xfb410108,0x80900101,0xfb410108,0x80910101,0xfb410108,0x80920101,0xfb410108,0x80940101,
	0xfb410108,0x80950101,0xfb410108,0x80970101,0xfb410108,0x80990101,0xfb410108,0x809e0101,
	0xfb410108,0x80a30101,0xfb410108,0x80a60101,0xfb410108,0x80a70101,0xfb410108,0x80a80101,
	0xfb410108,0x80ac0101,0xfb410108,0x80b00101,0xfb410108,0x80b30101,0xfb410108,0x80b50101,
	0xfb410108,0x80b60101,0xfb410108,0x80b80101,0xfb410108,0x80b90101,0xfb410108,0x80bb0101,
	0xfb410108,0x80c50101,0xfb410108,0x80c70101,0xfb410108,0x80c80101,0xfb410108,0x80c90101,
	0xfb410108,0x80ca0101,0xfb410108,0x80cb0101,0xfb410108,0x80cf0101,0xfb410108,0x80d00101,
	0xfb410108,0x80d10101,0xfb410108,0x80d20101,0xfb410108,0x80d30101,0xfb410108,0x80d40101,
	0xfb410108,0x80d50101,0xfb410108,0x80d80101,0xfb410108,0x80df0101,0xfb410108,0x80e00101,
	0xfb410108,0x80e20101,0xfb410108,0x80e30101,0xfb410108,0x80e60101,0xfb410108,0x80ee0101,
	0xfb410108,0x80f50101,0xfb410108,0x80f70101,0xfb410108,0x80f90101,0xfb410108,0x80fb0101,
	0xfb410108,0x80fe0101,0xfb410108,0x80ff0101,0xfb410108,0x81000101,0xfb410108,0x81010101,
	0xfb410108,0x81030101,0xfb410108,0x81040101,0xfb410108,0x81050101,0xfb410108,0x81070101,
	0xfb410108,0x81080101,0xfb410108,0x810b0101,0xfb410108,0x810c0101,0xfb410108,0x81150101,
	0xfb410108,0x81170101,0xfb410108,0x81190101,0xfb410108,0x811b0101,0xfb410108,0x811c0101,
	0xfb410108,0x811d0101,0xfb410108,0x811f0101,0xfb410108,0x81200101,0xfb410108,0x81210101,
	0xfb410108,0x81220101,0xfb410108,0x81230101,0xfb410108,0x81240101,0xfb410108,0x81250101,
	0xfb410108,0x81260101,0xfb410108,0x81270101,0xfb410108,0x81280101,0xfb410108,0x81290101,
	0xfb410108,0x812a0101,0xfb410108,0x812b0101,0xfb410108,0x812d0101,0xfb410108,0x812e0101,
	0xfb410108,0x81300101,0xfb410108,0x81330101,0xfb410108,0x81340101,0xfb410108,0x81350101,
	0xfb410108,0x81370101,0xfb410108,0x81390101,0xfb410108,0x813a0101,0xfb410108,0x813b0101,
	0xfb410108,0x813c0101,0xfb410108,0x813d0101,0xfb410108,0x813f0101,0xfb410108,0x81400101,
	0xfb410108,0x81410101,0xfb410108,0x81420101,0xfb410108,0x81430101,0xfb410108,0x81440101,
	0xfb410108,0x81450101,0xfb410108,0x81470101,0xfb410108,0x81490101,0xfb410108,0x814d0101,
	0xfb410108,0x814e0101,0xfb410108,0x814f0101,0xfb410108,0x81520101,0xfb410108,0x81560101,
	0xfb410108,0x81570101,0xfb410108,0x81580101,0xfb410108,0x815b0101,0xfb410108,0x815c0101,
	0xfb410108,0x815d0101,0xfb410108,0x815e0101,0xfb410108,0x815f0101,0xfb410108,0x81610101,
	0xfb410108,0x81620101,0xfb410108,0x81630101,0xfb410108,0x81640101,0xfb410108,0x81660101,
	0xfb410108,0x81680101,0xfb410108,0x816a0101,0xfb410108,0x816b0101,0xfb410108,0x816c0101,
	0xfb410108,0x816f0101,0xfb410108,0x81720101,0xfb410108,0x81730101,0xfb410108,0x81750101,
	0xfb410108,0x81760101,0xfb410108,0x81770101,0xfb410108,0x81780101,0xfb410108,0x81810101,
	0xfb410108,0x81830101,0xfb410108,0x81840101,0xfb410108,0x81850101,0xfb410108,0x81860101,
	0xfb410108,0x81870101,0xfb410108,0x81890101,0xfb410108,0x818b0101,0xfb410108,0x818c0101,
	0xfb410108,0x818d0101,0xfb410108,0x818e0101,0xfb410108,0x81900101,0xfb410108,0x81920101,
	0xfb410108,0x81930101,0xfb410108,0x81940101,0xfb410108,0x81950101,0xfb410108,0x81960101,
	0xfb410108,0x81970101,0xfb410108,0x81990101,0xfb410108,0x819a0101,0xfb410108,0x819e0101,
	0xfb410108,0x819f0101,0xfb410108,0x81a00101,0xfb410108,0x81a10101,0xfb410108,0x81a20101,
	0xfb410108,0x81a40101,0xfb410108,0x81a50101,0xfb410108,0x81a70101,0xfb410108,0x81a90101,
	0xfb410108,0x81ab0101,0xfb410108,0x81ac0101,0xfb410108,0x81ad0101,0xfb410108,0x81ae0101,
	0xfb410108,0x81af0101,0xfb410108,0x81b00101,0xfb410108,0x81b10101,0xfb410108,0x81b20101,
	0xfb410108,0x81b40101,0xfb410108,0x81b50101,0xfb410108,0x81b60101,0xfb410108,0x81b70101,
	0xfb410108,0x81b80101,0xfb410108,0x81b90101,0xfb410108,0x81bc0101,0xfb410108,0x81bd0101,
	0xfb410108,0x81be0101,0xfb410108,0x81bf0101,0xfb410108,0x81c40101,0xfb410108,0x81c50101,
	0xfb410108,0x81c70101,0xfb410108,0x81c80101,0xfb410108,0x81c90101,0xfb410108,0x81cb0101,
	0xfb410108,0x81cd0101,0xfb410108,0x81ce0101,0xfb410108,0x81cf0101,0xfb410108,0x81d00101,
	0xfb410108,0x81d10101,0xfb410108,0x81d20101,0xfb410108,0x81d30101,0xfb410108,0x81d40101,
	0xfb410108,0x81d50101,0xfb410108,0x81d60101,0xfb410108,0x81d70101,0xfb410108,0x81d80101,
	0xfb410108,0x81d90101,0xfb410108,0x81da0101,0xfb410108,0x81db0101,0xfb410108,0x81dc0101,
	0xfb410108,0x81dd0101,0xfb410108,0x81de0101,0xfb410108,0x81df0101,0xfb410108,0x81e00101,
	0xfb410108,0x81e10101,0xfb410108,0x81e20101,0xfb410108,0x81e40101,0xfb410108,0x81e50101,
	0xfb410108,0x81e60101,0xfb410108,0x81e80101,0xfb410108,0x81e90101,0xfb410108,0x81eb0101,
	0xfb410108,0x81ee0101,0xfb410108,0x81ef0101,0xfb410108,0x81f00101,0xfb410108,0x81f10101,
	0xfb410108,0x81f20101,0xfb410108,0x81f50101,0xfb410108,0x81f60101,0xfb410108,0x81f70101,
	0xfb410108,0x81f80101,0xfb410108,0x81f90101,0xfb410108,0x81fa0101,0xfb410108,0x81fd0101,
	0xfb410108,0x81ff0101,0xfb410108,0x82030101,0xfb410108,0x82070101,0xfb410108,0x82080101,
	0xfb410108,0x82090101,0xfb410108,0x820a0101,0xfb410108,0x820b0101,0xfb410108,0x820e0101,
	0xfb410108,0x820f0101,0xfb410108,0x82110101,0xfb410108,0x82130101,0xfb410108,0x82150101,
	0xfb410108,0x82160101,0xfb410108,0x82170101,0xfb410108,0x82180101,0xfb410108,0x82190101,
	0xfb410108,0x821a0101,0xfb410108,0x821d0101,0xfb410108,0x82200101,0xfb410108,0x82240101,
	0xfb410108,0x82250101,0xfb410108,0x82260101,0xfb410108,0x82270101,0xfb410108,0x82290101,
	0xfb410108,0x822e0101,0xfb410108,0x82320101,0xfb410108,0x823a0101,0xfb410108,0x823c0101,
	0xfb410108,0x823d0101,0xfb410108,0x823f0101,0xfb410108,0x82400101,0xfb410108,0x82410101,
	0xfb410108,0x82420101,0xfb410108,0x82430101,0xfb410108,0x82450101,0xfb410108,0x82460101,
	0xfb410108,0x82480101,0xfb410108,0x824a0101,0xfb410108,0x824c0101,0xfb410108,0x824d0101,
	0xfb410108,0x824e0101,0xfb410108,0x82500101,0xfb410108,0x82510101,0xfb410108,0x82520101,
	0xfb410108,0x82530101,0xfb410108,0x82540101,0xfb410108,0x82550101,0xfb410108,0x82560101,
	0xfb410108,0x82570101,0xfb410108,0x82590101,0xfb410108,0x825b0101,0xfb410108,0x825c0101,
	0xfb410108,0x825d0101,0xfb410108,0x825e0101,0xfb410108,0x82600101,0xfb410108,0x82610101,
	0xfb410108,0x82620101,0xfb410108,0x82630101,0xfb410108,0x82640101,0xfb410108,0x82650101,
	0xfb410108,0x82660101,0xfb410108,0x82670101,0xfb410108,0x82690101,0xfb410108,0x826a0101,
	0xfb410108,0x826b0101,0xfb410108,0x826c0101,0xfb410108,0x826d0101,0xfb410108,0x82710101,
	0xfb410108,0x82750101,0xfb410108,0x82760101,0xfb410108,0x82770101,0xfb410108,0x82780101,
	0xfb410108,0x827b0101,0xfb410108,0x827c0101,0xfb410108,0x82800101,0xfb410108,0x82810101,
	0xfb410108,0x82830101,0xfb410108,0x82850101,0xfb410108,0x82860101,0xfb410108,0x82870101,
	0xfb410108,0x82890101,0xfb410108,0x828c0101,0xfb410108,0x82900101,0xfb410108,0x82930101,
	0xfb410108,0x82940101,0xfb410108,0x82950101,0xfb410108,0x82960101,0xfb410108,0x829a0101,
	0xfb410108,0x829b0101,0xfb410108,0x829e0101,0xfb410108,0x82a00101,0xfb410108,0x82a20101,
	0xfb410108,0x82a30101,0xfb410108,0x82a70101,0xfb410108,0x82b20101,0xfb410108,0x82b50101,
	0xfb410108,0x82b60101,0xfb410108,0x82ba0101,0xfb410108,0x82bb0101,0xfb410108,0x82bc0101,
	0xfb410108,0x82bf0101,0xfb410108,0x82c00101,0xfb410108,0x82c20101,0xfb410108,0x82c30101,
	0xfb410108,0x82c50101,0xfb410108,0x82c60101,0xfb410108,0x82c90101,0xfb410108,0x82d00101,
	0xfb410108,0x82d60101,0xfb410108,0x82d90101,0xfb410108,0x82da0101,0xfb410108,0x82dd0101,
	0xfb410108,0x82e20101,0xfb410108,0x82e70101,0xfb410108,0x82e80101,0xfb410108,0x82e90101,
	0xfb410108,0x82ea0101,0xfb410108,0x82ec0101,0xfb410108,0x82ed0101,0xfb410108,0x82ee0101,
	0xfb410108,0x82f00101,0xfb410108,0x82f20101,0xfb410108,0x82f30101,0xfb410108,0x82f50101,
	0xfb410108,0x82f60101,0xfb410108,0x82f80101,0xfb410108,0x82fa0101,0xfb410108,0x82fc0101,
	0xfb410108,0x82fd0101,0xfb410108,0x82fe0101,0xfb410108,0x82ff0101,0xfb410108,0x83000101,
	0xfb410108,0x830a0101,0xfb410108,0x830b0101,0xfb410108,0x830d0101,0xfb410108,0x83100101,
	0xfb410108,0x83120101,0xfb410108,0x83130101,0xfb410108,0x83160101,0xfb410108,0x83180101,
	0xfb410108,0x83190101,0xfb410108,0x831d0101,0xfb410108,0x831e0101,0xfb410108,0x831f0101,
	0xfb410108,0x83200101,0xfb410108,0x83210101,0xfb410108,0x83220101,0xfb410108,0x83230101,
	0xfb410108,0x83240101,0xfb410108,0x83250101,0xfb410108,0x83260101,0xfb410108,0x83290101,
	0xfb410108,0x832a0101,0xfb410108,0x832e0101,0xfb410108,0x83300101,0xfb410108,0x83320101,
	0xfb410108,0x83370101,0xfb410108,0x833b0101,0xfb410108,0x833d0101,0xfb410108,0x833e0101,
	0xfb410108,0x833f0101,0xfb410108,0x83410101,0xfb410108,0x83420101,0xfb410108,0x83440101,
	0xfb410108,0x83450101,0xfb410108,0x83480101,0xfb410108,0x834a0101,0xfb410108,0x834b0101,
	0xfb410108,0x834c0101,0xfb410108,0x834d0101,0xfb410108,0x834e0101,0xfb410108,0x83530101,
	0xfb410108,0x83550101,0xfb410108,0x83560101,0xfb410108,0x83570101,0xfb410108,0x83580101,
	0xfb410108,0x83590101,0xfb410108,0x835d0101,0xfb410108,0x83620101,0xfb410108,0x83700101,
	0xfb410108,0x83710101,0xfb410108,0x83720101,0xfb410108,0x83730101,0xfb410108,0x83740101,
	0xfb410108,0x83750101,0xfb410108,0x83760101,0xfb410108,0x83790101,0xfb410108,0x837a0101,
	0xfb410108,0x837e0101,0xfb410108,0x837f0101,0xfb410108,0x83800101,0xfb410108,0x83810101,
	0xfb410108,0x83820101,0xfb410108,0x83830101,0xfb410108,0x83840101,0xfb410108,0x83870101,
	0xfb410108,0x83880101,0xfb410108,0x838a0101,0xfb410108,0x838b0101,0xfb410108,0x838c0101,
	0xfb410108,0x838d0101,0xfb410108,0x838f0101,0xfb410108,0x83900101,0xfb410108,0x83910101,
	0xfb410108,0x83940101,0xfb410108,0x83950101,0xfb410108,0x83960101,0xfb410108,0x83970101,
	0xfb410108,0x83990101,0xfb410108,0x839a0101,0xfb410108,0x839d0101,0xfb410108,0x839f0101,
	0xfb410108,0x83a10101,0xfb410108,0x83a20101,0xfb410108,0x83a30101,0xfb410108,0x83a40101,
	0xfb410108,0x83a50101,0xfb410108,0x83a60101,0xfb410108,0x83a70101,0xfb410108,0x83ac0101,
	0xfb410108,0x83ad0101,0xfb410108,0x83ae0101,0xfb410108,0x83af0101,0xfb410108,0x83b50101,
	0xfb410108,0x83bb0101,0xfb410108,0x83be0101,0xfb410108,0x83bf0101,0xfb410108,0x83c20101,
	0xfb410108,0x83c30101,0xfb410108,0x83c40101,0xfb410108,0x83c60101,0xfb410108,0x83c80101,
	0xfb410108,0x83c90101,0xfb410108,0x83cb0101,0xfb410108,0x83cd0101,0xfb410108,0x83ce0101,
	0xfb410108,0x83d00101,0xfb410108,0x83d10101,0xfb410108,0x83d20101,0xfb410108,0x83d30101,
	0xfb410108,0x83d50101,0xfb410108,0x83d70101,0xfb410108,0x83d90101,0xfb410108,0x83da0101,
	0xfb410108,0x83db0101,0xfb410108,0x83de0101,0xfb410108,0x83e20101,0xfb410108,0x83e30101,
	0xfb410108,0x83e40101,0xfb410108,0x83e60101,0xfb410108,0x83e70101,0xfb410108,0x83e80101,
	0xfb410108,0x83eb0101,0xfb410108,0x83ec0101,0xfb410108,0x83ed0101,0xfb410108,0x83ee0101,
	0xfb410108,0x83ef0101,0xfb410108,0x83f30101,0xfb410108,0x83f40101,0xfb410108,0x83f50101,
	0xfb410108,0x83f60101,0xfb410108,0x83f70101,0xfb410108,0x83fa0101,0xfb410108,0x83fb0101,
	0xfb410108,0x83fc0101,0xfb410108,0x83fe0101,0xfb410108,0x83ff0101,0xfb410108,0x84000101,
	0xfb410108,0x84020101,0xfb410108,0x84050101,0xfb410108,0x84070101,0xfb410108,0x84080101,
	0xfb410108,0x84090101,0xfb410108,0x840a0101,0xfb410108,0x84100101,0xfb410108,0x84120101,
	0xfb410108,0x84130101,0xfb410108,0x84140101,0xfb410108,0x84150101,0xfb410108,0x84160101,
	0xfb410108,0x84170101,0xfb410108,0x84190101,0xfb410108,0x841a0101,0xfb410108,0x841b0101,
	0xfb410108,0x841e0101,0xfb410108,0x841f0101,0xfb410108,0x84200101,0xfb410108,0x84210101,
	0xfb410108,0x84220101,0xfb410108,0x84230101,0xfb410108,0x84290101,0xfb410108,0x842a0101,
	0xfb410108,0x842b0101,0xfb410108,0x842c0101,0xfb410108,0x842d0101,0xfb410108,0x842e0101,
	0xfb410108,0x842f0101,0xfb410108,0x84300101,0xfb410108,0x84320101,0xfb410108,0x84330101,
	0xfb410108,0x84340101,0xfb410108,0x84350101,0xfb410108,0x84360101,0xfb410108,0x84370101,
	0xfb410108,0x84390101,0xfb410108,0x843a0101,0xfb410108,0x843b0101,0xfb410108,0x843e0101,
	0xfb410108,0x843f0101,0xfb410108,0x84400101,0xfb410108,0x84410101,0xfb410108,0x84420101,
	0xfb410108,0x84430101,0xfb410108,0x84440101,0xfb410108,0x84450101,0xfb410108,0x84470101,
	0xfb410108,0x84480101,0xfb410108,0x84490101,0xfb410108,0x844a0101,0xfb410108,0x844b0101,
	0xfb410108,0x844c0101,0xfb410108,0x844d0101,0xfb410108,0x844e0101,0xfb410108,0x844f0101,
	0xfb410108,0x84500101,0xfb410108,0x84520101,0xfb410108,0x84530101,0xfb410108,0x84540101,
	0xfb410108,0x84550101,0xfb410108,0x84560101,0xfb410108,0x84580101,0xfb410108,0x845d0101,
	0xfb410108,0x845e0101,0xfb410108,0x845f0101,0xfb410108,0x84600101,0xfb410108,0x84620101,
	0xfb410108,0x84640101,0xfb410108,0x84650101,0xfb410108,0x84660101,0xfb410108,0x84670101,
	0xfb410108,0x84680101,0xfb410108,0x846a0101,0xfb410108,0x846e0101,0xfb410108,0x846f0101,
	0xfb410108,0x84700101,0xfb410108,0x84720101,0xfb410108,0x84740101,0xfb410108,0x84770101,
	0xfb410108,0x84790101,0xfb410108,0x847b0101,0xfb410108,0x847c0101,0xfb410108,0x847d0101,
	0xfb410108,0x847e0101,0xfb410108,0x847f0101,0xfb410108,0x84800101,0xfb410108,0x84810101,
	0xfb410108,0x84830101,0xfb410108,0x84840101,0xfb410108,0x84850101,0xfb410108,0x84860101,
	0xfb410108,0x848a0101,0xfb410108,0x848d0101,0xfb410108,0x848f0101,0xfb410108,0x84900101,
	0xfb410108,0x84910101,0xfb410108,0x84920101,0xfb410108,0x84930101,0xfb410108,0x84940101,
	0xfb410108,0x84950101,0xfb410108,0x84960101,0xfb410108,0x84980101,0xfb410108,0x849a0101,
	0xfb410108,0x849b0101,0xfb410108,0x849d0101,0xfb410108,0x849e0101,0xfb410108,0x849f0101,
	0xfb410108,0x84a00101,0xfb410108,0x84a20101,0xfb410108,0x84a30101,0xfb410108,0x84a40101,
	0xfb410108,0x84a50101,0xfb410108,0x84a60101,0xfb410108,0x84a70101,0xfb410108,0x84a80101,
	0xfb410108,0x84a90101,0xfb410108,0x84aa0101,0xfb410108,0x84ab0101,0xfb410108,0x84ac0101,
	0xfb410108,0x84ad0101,0xfb410108,0x84ae0101,0xfb410108,0x84b00101,0xfb410108,0x84b10101,
	0xfb410108,0x84b30101,0xfb410108,0x84b50101,0xfb410108,0x84b60101,0xfb410108,0x84b70101,
	0xfb410108,0x84bb0101,0xfb410108,0x84bc0101,0xfb410108,0x84be0101,0xfb410108,0x84c00101,
	0xfb410108,0x84c20101,0xfb410108,0x84c30101,0xfb410108,0x84c50101,0xfb410108,0x84c60101,
	0xfb410108,0x84c70101,0xfb410108,0x84c80101,0xfb410108,0x84cb0101,0xfb410108,0x84cc0101,
	0xfb410108,0x84ce0101,0xfb410108,0x84cf0101,0xfb410108,0x84d20101,0xfb410108,0x84d40101,
	0xfb410108,0x84d50101,0xfb410108,0x84d70101,0xfb410108,0x84d80101,0xfb410108,0x84d90101,
	0xfb410108,0x84da0101,0xfb410108,0x84db0101,0xfb410108,0x84dc0101,0xfb410108,0x84de0101,
	0xfb410108,0x84e10101,0xfb410108,0x84e20101,0xfb410108,0x84e40101,0xfb410108,0x84e70101,
	0xfb410108,0x84e80101,0xfb410108,0x84e90101,0xfb410108,0x84ea0101,0xfb410108,0x84eb0101,
	0xfb410108,0x84ed0101,0xfb410108,0x84ee0101,0xfb410108,0x84ef0101,0xfb410108,0x84f10101,
	0xfb410108,0x84f20101,0xfb410108,0x84f30101,0xfb410108,0x84f40101,0xfb410108,0x84f50101,
	0xfb410108,0x84f60101,0xfb410108,0x84f70101,0xfb410108,0x84f80101,0xfb410108,0x84f90101,
	0xfb410108,0x84fa0101,0xfb410108,0x84fb0101,0xfb410108,0x84fd0101,0xfb410108,0x84fe0101,
	0xfb410108,0x85000101,0xfb410108,0x85010101,0xfb410108,0x85020101,0xfb410108,0x85030101,
	0xfb410108,0x85040101,0xfb410108,0x85050101,0xfb410108,0x85060101,0xfb410108,0x85070101,
	0xfb410108,0x85080101,0xfb410108,0x85090101,0xfb410108,0x850a0101,0xfb410108,0x850b0101,
	0xfb410108,0x850d0101,0xfb410108,0x850e0101,0xfb410108,0x850f0101,0xfb410108,0x85100101,
	0xfb410108,0x85120101,0xfb410108,0x85140101,0xfb410108,0x85150101,0xfb410108,0x85160101,
	0xfb410108,0x85180101,0xfb410108,0x85190101,0xfb410108,0x851b0101,0xfb410108,0x851c0101,
	0xfb410108,0x851d0101,0xfb410108,0x851e0101,0xfb410108,0x85200101,0xfb410108,0x85220101,
	0xfb410108,0x85230101,0xfb410108,0x85240101,0xfb410108,0x85250101,0xfb410108,0x85260101,
	0xfb410108,0x85270101,0xfb410108,0x85280101,0xfb410108,0x85290101,0xfb410108,0x852a0101,
	0xfb410108,0x852d0101,0xfb410108,0x852e0101,0xfb410108,0x852f0101,0xfb410108,0x85300101,
	0xfb410108,0x85310101,0xfb410108,0x85320101,0xfb410108,0x85330101,0xfb410108,0x85340101,
	0xfb410108,0x85350101,0xfb410108,0x85360101,0xfb410108,0x853e0101,0xfb410108,0x853f0101,
	0xfb410108,0x85400101,0xfb410108,0x85410101,0xfb410108,0x85420101,0xfb410108,0x85440101,
	0xfb410108,0x85450101,0xfb410108,0x85460101,0xfb410108,0x85470101,0xfb410108,0x854b0101,
	0xfb410108,0x854c0101,0xfb410108,0x854d0101,0xfb410108,0x854e0101,0xfb410108,0x854f0101,
	0xfb410108,0x85500101,0xfb410108,0x85510101,0xfb410108,0x85520101,0xfb410108,0x85530101,
	0xfb410108,0x85540101,0xfb410108,0x85550101,0xfb410108,0x85570101,0xfb410108,0x85580101,
	0xfb410108,0x855a0101,0xfb410108,0x855b0101,0xfb410108,0x855c0101,0xfb410108,0x855d0101,
	0xfb410108,0x855f0101,0xfb410108,0x85600101,0xfb410108,0x85610101,0xfb410108,0x85620101,
	0xfb410108,0x85630101,0xfb410108,0x85650101,0xfb410108,0x85660101,0xfb410108,0x85670101,
	0xfb410108,0x85690101,0xfb410108,0x856a0101,0xfb410108,0x856b0101,0xfb410108,0x856c0101,
	0xfb410108,0x856d0101,0xfb410108,0x856e0101,0xfb410108,0x856f0101,0xfb410108,0x85700101,
	0xfb410108,0x85710101,0xfb410108,0x85730101,0xfb410108,0x85750101,0xfb410108,0x85760101,
	0xfb410108,0x85770101,0xfb410108,0x85780101,0xfb410108,0x857c0101,0xfb410108,0x857d0101,
	0xfb410108,0x857f0101,0xfb410108,0x85800101,0xfb410108,0x85810101,0xfb410108,0x85820101,
	0xfb410108,0x85830101,0xfb410108,0x85860101,0xfb410108,0x85880101,0xfb410108,0x85890101,
	0xfb410108,0x858a0101,0xfb410108,0x858b0101,0xfb410108,0x858c0101,0xfb410108,0x858d0101,
	0xfb410108,0x858e0101,0xfb410108,0x85900101,0xfb410108,0x85910101,0xfb410108,0x85920101,
	0xfb410108,0x85930101,0xfb410108,0x85940101,0xfb410108,0x85950101,0xfb410108,0x85960101,
	0xfb410108,0x85970101,0xfb410108,0x85980101,0xfb410108,0x85990101,0xfb410108,0x859a0101,
	0xfb410108,0x859d0101,0xfb410108,0x859e0101,0xfb410108,0x859f0101,0xfb410108,0x85a00101,
	0xfb410108,0x85a10101,0xfb410108,0x85a20101,0xfb410108,0x85a30101,0xfb410108,0x85a50101,
	0xfb410108,0x85a60101,0xfb410108,0x85a70101,0xfb410108,0x85a90101,0xfb410108,0x85ab0101,
	0xfb410108,0x85ac0101,0xfb410108,0x85ad0101,0xfb410108,0x85b10101,0xfb410108,0x85b20101,
	0xfb410108,0x85b30101,0xfb410108,0x85b40101,0xfb410108,0x85b50101,0xfb410108,0x85b60101,
	0xfb410108,0x85b80101,0xfb410108,0x85ba0101,0xfb410108,0x85bb0101,0xfb410108,0x85bc0101,
	0xfb410108,0x85bd0101,0xfb410108,0x85be0101,0xfb410108,0x85bf0101,0xfb410108,0x85c00101,
	0xfb410108,0x85c20101,0xfb410108,0x85c30101,0xfb410108,0x85c40101,0xfb410108,0x85c50101,
	0xfb410108,0x85c60101,0xfb410108,0x85c70101,0xfb410108,0x85c80101,0xfb410108,0x85ca0101,
	0xfb410108,0x85cb0101,0xfb410108,0x85cc0101,0xfb410108,0x85cd0101,0xfb410108,0x85ce0101,
	0xfb410108,0x85d10101,0xfb410108,0x85d20101,0xfb410108,0x85d40101,0xfb410108,0x85d60101,
	0xfb410108,0x85d70101,0xfb410108,0x85d80101,0xfb410108,0x85d90101,0xfb410108,0x85da0101,
	0xfb410108,0x85db0101,0xfb410108,0x85dd0101,0xfb410108,0x85de0101,0xfb410108,0x85df0101,
	0xfb410108,0x85e00101,0xfb410108,0x85e10101,0xfb410108,0x85e20101,0xfb410108,0x85e30101,
	0xfb410108,0x85e50101,0xfb410108,0x85e60101,0xfb410108,0x85e70101,0xfb410108,0x85e80101,
	0xfb410108,0x85ea0101,0xfb410108,0x85eb0101,0xfb410108,0x85ec0101,0xfb410108,0x85ed0101,
	0xfb410108,0x85ee0101,0xfb410108,0x85ef0101,0xfb410108,0x85f00101,0xfb410108,0x85f10101,
	0xfb410108,0x85f20101,0xfb410108,0x85f30101,0xfb410108,0x85f40101,0xfb410108,0x85f50101,
	0xfb410108,0x85f60101,0xfb410108,0x85f70101,0xfb410108,0x85f80101,0xfb410108,0x85f90101,
	0xfb410108,0x85fa0101,0xfb410108,0x85fc0101,0xfb410108,0x85fd0101,0xfb410108,0x85fe0101,
	0xfb410108,0x86000101,0xfb410108,0x86010101,0xfb410108,0x86020101,0xfb410108,0x86030101,
	0xfb410108,0x86040101,0xfb410108,0x86060101,0xfb410108,0x86070101,0xfb410108,0x86080101,
	0xfb410108,0x86090101,0xfb410108,0x860a0101,0xfb410108,0x860b0101,0xfb410108,0x860c0101,
	0xfb410108,0x860d0101,0xfb410108,0x860e0101,0xfb410108,0x860f0101,0xfb410108,0x86100101,
	0xfb410108,0x86120101,0xfb410108,0x86130101,0xfb410108,0x86140101,0xfb410108,0x86150101,
	0xfb410108,0x86170101,0xfb410108,0x86180101,0xfb410108,0x86190101,0xfb410108,0x861a0101,
	0xfb410108,0x861b0101,0xfb410108,0x861c0101,0xfb410108,0x861d0101,0xfb410108,0x861e0101,
	0xfb410108,0x861f0101,0xfb410108,0x86200101,0xfb410108,0x86210101,0xfb410108,0x86220101,
	0xfb410108,0x86230101,0xfb410108,0x86240101,0xfb410108,0x86250101,0xfb410108,0x86260101,
	0xfb410108,0x86280101,0xfb410108,0x862a0101,0xfb410108,0x862b0101,0xfb410108,0x862c0101,
	0xfb410108,0x862d0101,0xfb410108,0x862e0101,0xfb410108,0x862f0101,0xfb410108,0x86300101,
	0xfb410108,0x86310101,0xfb410108,0x86320101,0xfb410108,0x86330101,0xfb410108,0x86340101,
	0xfb410108,0x86350101,0xfb410108,0x86360101,0xfb410108,0x86370101,0xfb410108,0x86390101,
	0xfb410108,0x863a0101,0xfb410108,0x863b0101,0xfb410108,0x863d0101,0xfb410108,0x863e0101,
	0xfb410108,0x863f0101,0xfb410108,0x86400101,0xfb410108,0x86410101,0xfb410108,0x86420101,
	0xfb410108,0x86430101,0xfb410108,0x86440101,0xfb410108,0x86450101,0xfb410108,0x86460101,
	0xfb410108,0x86470101,0xfb410108,0x86480101,0xfb410108,0x86490101,0xfb410108,0x864a0101,
	0xfb410108,0x864b0101,0xfb410108,0x864c0101,0xfb410108,0x86520101,0xfb410108,0x86530101,
	0xfb410108,0x86550101,0xfb410108,0x86560101,0xfb410108,0x86570101,0xfb410108,0x86580101,
	0xfb410108,0x86590101,0xfb410108,0x865b0101,0xfb410108,0x865c0101,0xfb410108,0x865d0101,
	0xfb410108,0x865f0101,0xfb410108,0x86600101,0xfb410108,0x86610101,0xfb410108,0x86630101,
	0xfb410108,0x86640101,0xfb410108,0x86650101,0xfb410108,0x86660101,0xfb410108,0x86670101,
	0xfb410108,0x86680101,0xfb410108,0x86690101,0xfb410108,0x866a0101,0xfb410108,0x866d0101,
	0xfb410108,0x866f0101,0xfb410108,0x86700101,0xfb410108,0x86720101,0xfb410108,0x86730101,
	0xfb410108,0x86740101,0xfb410108,0x86750101,0xfb410108,0x86760101,0xfb410108,0x86770101,
	0xfb410108,0x86780101,0xfb410108,0x86830101,0xfb410108,0x86840101,0xfb410108,0x86850101,
	0xfb410108,0x86860101,0xfb410108,0x86870101,0xfb410108,0x86880101,0xfb410108,0x86890101,
	0xfb410108,0x868e0101,0xfb410108,0x868f0101,0xfb410108,0x86900101,0xfb410108,0x86910101,
	0xfb410108,0x86920101,0xfb410108,0x86940101,0xfb410108,0x86960101,0xfb410108,0x86970101,
	0xfb410108,0x86980101,0xfb410108,0x86990101,0xfb410108,0x869a0101,0xfb410108,0x869b0101,
	0xfb410108,0x869e0101,0xfb410108,0x869f0101,0xfb410108,0x86a00101,0xfb410108,0x86a10101,
	0xfb410108,0x86a20101,0xfb410108,0x86a50101,0xfb410108,0x86a60101,0xfb410108,0x86ab0101,
	0xfb410108,0x86ad0101,0xfb410108,0x86ae0101,0xfb410108,0x86b20101,0xfb410108,0x86b30101,
	0xfb410108,0x86b70101,0xfb410108,0x86b80101,0xfb410108,0x86b90101,0xfb410108,0x86bb0101,
	0xfb410108,0x86bc0101,0xfb410108,0x86bd0101,0xfb410108,0x86be0101,0xfb410108,0x86bf0101,
	0xfb410108,0x86c10101,0xfb410108,0x86c20101,0xfb410108,0x86c30101,0xfb410108,0x86c50101,
	0xfb410108,0x86c80101,0xfb410108,0x86cc0101,0xfb410108,0x86cd0101,0xfb410108,0x86d20101,
	0xfb410108,0x86d30101,0xfb410108,0x86d50101,0xfb410108,0x86d60101,0xfb410108,0x86d70101,
	0xfb410108,0x86da0101,0xfb410108,0x86dc0101,0xfb410108,0x86dd0101,0xfb410108,0x86e00101,
	0xfb410108,0x86e10101,0xfb410108,0x86e20101,0xfb410108,0x86e30101,0xfb410108,0x86e50101,
	0xfb410108,0x86e60101,0xfb410108,0x86e70101,0xfb410108,0x86e80101,0xfb410108,0x86ea0101,
	0xfb410108,0x86eb0101,0xfb410108,0x86ec0101,0xfb410108,0x86ef0101,0xfb410108,0x86f50101,
	0xfb410108,0x86f60101,0xfb410108,0x86f70101,0xfb410108,0x86fa0101,0xfb410108,0x86fb0101,
	0xfb410108,0x86fc0101,0xfb410108,0x86fd0101,0xfb410108,0x86ff0101,0xfb410108,0x87010101,
	0xfb410108,0x87040101,0xfb410108,0x87050101,0xfb410108,0x87060101,0xfb410108,0x870b0101,
	0xfb410108,0x870c0101,0xfb410108,0x870e0101,0xfb410108,0x870f0101,0xfb410108,0x87100101,
	0xfb410108,0x87110101,0xfb410108,0x87140101,0xfb410108,0x87160101,0xfb410108,0x87190101,
	0xfb410108,0x871b0101,0xfb410108,0x871d0101,0xfb410108,0x871f0101,0xfb410108,0x87200101,
	0xfb410108,0x87240101,0xfb410108,0x87260101,0xfb410108,0x87270101,0xfb410108,0x87280101,
	0xfb410108,0x872a0101,0xfb410108,0x872b0101,0xfb410108,0x872c0101,0xfb410108,0x872d0101,
	0xfb410108,0x872f0101,0xfb410108,0x87300101,0xfb410108,0x87320101,0xfb410108,0x87330101,
	0xfb410108,0x87350101,0xfb410108,0x87360101,0xfb410108,0x87380101,0xfb410108,0x87390101,
	0xfb410108,0x873a0101,0xfb410108,0x873c0101,0xfb410108,0x873d0101,0xfb410108,0x87400101,
	0xfb410108,0x87410101,0xfb410108,0x87420101,0xfb410108,0x87430101,0xfb410108,0x87440101,
	0xfb410108,0x87450101,0xfb410108,0x87460101,0xfb410108,0x874a0101,0xfb410108,0x874b0101,
	0xfb410108,0x874d0101,0xfb410108,0x874f0101,0xfb410108,0x87500101,0xfb410108,0x87510101,
	0xfb410108,0x87520101,0xfb410108,0x87540101,0xfb410108,0x87550101,0xfb410108,0x87560101,
	0xfb410108,0x87580101,0xfb410108,0x875a0101,0xfb410108,0x875b0101,0xfb410108,0x875c0101,
	0xfb410108,0x875d0101,0xfb410108,0x875e0101,0xfb410108,0x875f0101,0xfb410108,0x87610101,
	0xfb410108,0x87620101,0xfb410108,0x87660101,0xfb410108,0x87670101,0xfb410108,0x87680101,
	0xfb410108,0x87690101,0xfb410108,0x876a0101,0xfb410108,0x876b0101,0xfb410108,0x876c0101,
	0xfb410108,0x876d0101,0xfb410108,0x876f0101,0xfb410108,0x87710101,0xfb410108,0x87720101,
	0xfb410108,0x87730101,0xfb410108,0x87750101,0xfb410108,0x87770101,0xfb410108,0x87780101,
	0xfb410108,0x87790101,0xfb410108,0x877a0101,0xfb410108,0x877f0101,0xfb410108,0x87800101,
	0xfb410108,0x87810101,0xfb410108,0x87840101,0xfb410108,0x87860101,0xfb410108,0x87870101,
	0xfb410108,0x87890101,0xfb410108,0x878a0101,0xfb410108,0x878c0101,0xfb410108,0x878e0101,
	0xfb410108,0x878f0101,0xfb410108,0x87900101,0xfb410108,0x87910101,0xfb410108,0x87920101,
	0xfb410108,0x87940101,0xfb410108,0x87950101,0xfb410108,0x87960101,0xfb410108,0x87980101,
	0xfb410108,0x87990101,0xfb410108,0x879a0101,0xfb410108,0x879b0101,0xfb410108,0x879c0101,
	0xfb410108,0x879d0101,0xfb410108,0x879e0101,0xfb410108,0x87a00101,0xfb410108,0x87a10101,
	0xfb410108,0x87a20101,0xfb410108,0x87a30101,0xfb410108,0x87a40101,0xfb410108,0x87a50101,
	0xfb410108,0x87a60101,0xfb410108,0x87a70101,0xfb410108,0x87a90101,0xfb410108,0x87aa0101,
	0xfb410108,0x87ae0101,0xfb410108,0x87b00101,0xfb410108,0x87b10101,0xfb410108,0x87b20101,
	0xfb410108,0x87b40101,0xfb410108,0x87b60101,0xfb410108,0x87b70101,0xfb410108,0x87b80101,
	0xfb410108,0x87b90101,0xfb410108,0x87bb0101,0xfb410108,0x87bc0101,0xfb410108,0x87be0101,
	0xfb410108,0x87bf0101,0xfb410108,0x87c10101,0xfb410108,0x87c20101,0xfb410108,0x87c30101,
	0xfb410108,0x87c40101,0xfb410108,0x87c50101,0xfb410108,0x87c70101,0xfb410108,0x87c80101,
	0xfb410108,0x87c90101,0xfb410108,0x87cc0101,0xfb410108,0x87cd0101,0xfb410108,0x87ce0101,
	0xfb410108,0x87cf0101,0xfb410108,0x87d00101,0xfb410108,0x87d40101,0xfb410108,0x87d50101,
	0xfb410108,0x87d60101,0xfb410108,0x87d70101,0xfb410108,0x87d80101,0xfb410108,0x87d90101,
	0xfb410108,0x87da0101,0xfb410108,0x87dc0101,0xfb410108,0x87dd0101,0xfb410108,0x87de0101,
	0xfb410108,0x87df0101,0xfb410108,0x87e10101,0xfb410108,0x87e20101,0xfb410108,0x87e30101,
	0xfb410108,0x87e40101,0xfb410108,0x87e60101,0xfb410108,0x87e70101,0xfb410108,0x87e80101,
	0xfb410108,0x87e90101,0xfb410108,0x87eb0101,0xfb410108,0x87ec0101,0xfb410108,0x87ed0101,
	0xfb410108,0x87ef0101,0xfb410108,0x87f00101,0xfb410108,0x87f10101,0xfb410108,0x87f20101,
	0xfb410108,0x87f30101,0xfb410108,0x87f40101,0xfb410108,0x87f50101,0xfb410108,0x87f60101,
	0xfb410108,0x87f70101,0xfb410108,0x87f80101,0xfb410108,0x87fa0101,0xfb410108,0x87fb0101,
	0xfb410108,0x87fc0101,0xfb410108,0x87fd0101,0xfb410108,0x87ff0101,0xfb410108,0x88000101,
	0xfb410108,0x88010101,0xfb410108,0x88020101,0xfb410108,0x88040101,0xfb410108,0x88050101,
	0xfb410108,0x88060101,0xfb410108,0x88070101,0xfb410108,0x88080101,0xfb410108,0x88090101,
	0xfb410108,0x880b0101,0xfb410108,0x880c0101,0xfb410108,0x880d0101,0xfb410108,0x880e0101,
	0xfb410108,0x880f0101,0xfb410108,0x88100101,0xfb410108,0x88110101,0xfb410108,0x88120101,
	0xfb410108,0x88140101,0xfb410108,0x88170101,0xfb410108,0x88180101,0xfb410108,0x88190101,
	0xfb410108,0x881a0101,0xfb410108,0x881c0101,0xfb410108,0x881d0101,0xfb410108,0x881e0101,
	0xfb410108,0x881f0101,0xfb410108,0x88200101,0xfb410108,0x88230101,0xfb410108,0x88240101,
	0xfb410108,0x88250101,0xfb410108,0x88260101,0xfb410108,0x88270101,0xfb410108,0x88280101,
	0xfb410108,0x88290101,0xfb410108,0x882a0101,0xfb410108,0x882b0101,0xfb410108,0x882c0101,
	0xfb410108,0x882d0101,0xfb410108,0x882e0101,0xfb410108,0x882f0101,0xfb410108,0x88300101,
	0xfb410108,0x88310101,0xfb410108,0x88330101,0xfb410108,0x88340101,0xfb410108,0x88350101,
	0xfb410108,0x88360101,0xfb410108,0x88370101,0xfb410108,0x88380101,0xfb410108,0x883a0101,
	0xfb410108,0x883b0101,0xfb410108,0x883d0101,0xfb410108,0x883e0101,0xfb410108,0x883f0101,
	0xfb410108,0x88410101,0xfb410108,0x88420101,0xfb410108,0x88430101,0xfb410108,0x88460101,
	0xfb410108,0x88470101,0xfb410108,0x88480101,0xfb410108,0x88490101,0xfb410108,0x884a0101,
	0xfb410108,0x884b0101,0xfb410108,0x884e0101,0xfb410108,0x884f0101,0xfb410108,0x88500101,
	0xfb410108,0x88510101,0xfb410108,0x88520101,0xfb410108,0x88530101,0xfb410108,0x88550101,
	0xfb410108,0x88560101,0xfb410108,0x88580101,0xfb410108,0x885a0101,0xfb410108,0x885b0101,
	0xfb410108,0x885c0101,0xfb410108,0x885d0101,0xfb410108,0x885e0101,0xfb410108,0x885f0101,
	0xfb410108,0x88600101,0xfb410108,0x88660101,0xfb410108,0x88670101,0xfb410108,0x886a0101,
	0xfb410108,0x886d0101,0xfb410108,0x886f0101,0xfb410108,0x88710101,0xfb410108,0x88730101,
	0xfb410108,0x88740101,0xfb410108,0x88750101,0xfb410108,0x88760101,0xfb410108,0x88780101,
	0xfb410108,0x88790101,0xfb410108,0x887a0101,0xfb410108,0x887b0101,0xfb410108,0x887c0101,
	0xfb410108,0x88800101,0xfb410108,0x88830101,0xfb410108,0x88860101,0xfb410108,0x88870101,
	0xfb410108,0x88890101,0xfb410108,0x888a0101,0xfb410108,0x888c0101,0xfb410108,0x888e0101,
	0xfb410108,0x888f0101,0xfb410108,0x88900101,0xfb410108,0x88910101,0xfb410108,0x88930101,
	0xfb410108,0x88940101,0xfb410108,0x88950101,0xfb410108,0x88970101,0xfb410108,0x88980101,
	0xfb410108,0x88990101,0xfb410108,0x889a0101,0xfb410108,0x889b0101,0xfb410108,0x889d0101,
	0xfb410108,0x889e0101,0xfb410108,0x889f0101,0xfb410108,0x88a00101,0xfb410108,0x88a10101,
	0xfb410108,0x88a30101,0xfb410108,0x88a50101,0xfb410108,0x88a60101,0xfb410108,0x88a70101,
	0xfb410108,0x88a80101,0xfb410108,0x88a90101,0xfb410108,0x88aa0101,0xfb410108,0x88ac0101,
	0xfb410108,0x88ae0101,0xfb410108,0x88af0101,0xfb410108,0x88b00101,0xfb410108,0x88b20101,
	0xfb410108,0x88b30101,0xfb410108,0x88b40101,0xfb410108,0x88b50101,0xfb410108,0x88b60101,
	0xfb410108,0x88b80101,0xfb410108,0x88b90101,0xfb410108,0x88ba0101,0xfb410108,0x88bb0101,
	0xfb410108,0x88bd0101,0xfb410108,0x88be0101,0xfb410108,0x88bf0101,0xfb410108,0x88c00101,
	0xfb410108,0x88c30101,0xfb410108,0x88c40101,0xfb410108,0x88c70101,0xfb410108,0x88c80101,
	0xfb410108,0x88ca0101,0xfb410108,0x88cb0101,0xfb410108,0x88cc0101,0xfb410108,0x88cd0101,
	0xfb410108,0x88cf0101,0xfb410108,0x88d00101,0xfb410108,0x88d10101,0xfb410108,0x88d30101,
	0xfb410108,0x88d60101,0xfb410108,0x88d70101,0xfb410108,0x88da0101,0xfb410108,0x88db0101,
	0xfb410108,0x88dc0101,0xfb410108,0x88dd0101,0xfb410108,0x88de0101,0xfb410108,0x88e00101,
	0xfb410108,0x88e10101,0xfb410108,0x88e60101,0xfb410108,0x88e70101,0xfb410108,0x88e90101,
	0xfb410108,0x88ea0101,0xfb410108,0x88eb0101,0xfb410108,0x88ec0101,0xfb410108,0x88ed0101,
	0xfb410108,0x88ee0101,0xfb410108,0x88ef0101,0xfb410108,0x88f20101,0xfb410108,0x88f50101,
	0xfb410108,0x88f60101,0xfb410108,0x88f70101,0xfb410108,0x88fa0101,0xfb410108,0x88fb0101,
	0xfb410108,0x88fd0101,0xfb410108,0x88ff0101,0xfb410108,0x89000101,0xfb410108,0x89010101,
	0xfb410108,0x89030101,0xfb410108,0x89040101,0xfb410108,0x89050101,0xfb410108,0x89060101,
	0xfb410108,0x89070101,0xfb410108,0x89080101,0xfb410108,0x89090101,0xfb410108,0x890b0101,
	0xfb410108,0x890c0101,0xfb410108,0x890d0101,0xfb410108,0x890e0101,0xfb410108,0x890f0101,
	0xfb410108,0x89110101,0xfb410108,0x89140101,0xfb410108,0x89150101,0xfb410108,0x89160101,
	0xfb410108,0x89170101,0xfb410108,0x89180101,0xfb410108,0x891c0101,0xfb410108,0x891d0101,
	0xfb410108,0x891e0101,0xfb410108,0x891f0101,0xfb410108,0x89200101,0xfb410108,0x89220101,
	0xfb410108,0x89230101,0xfb410108,0x89240101,0xfb410108,0x89260101,0xfb410108,0x89270101,
	0xfb410108,0x89280101,0xfb410108,0x89290101,0xfb410108,0x892c0101,0xfb410108,0x892d0101,
	0xfb410108,0x892e0101,0xfb410108,0x892f0101,0xfb410108,0x89310101,0xfb410108,0x89320101,
	0xfb410108,0x89330101,0xfb410108,0x89350101,0xfb410108,0x89370101,0xfb410108,0x89380101,
	0xfb410108,0x89390101,0xfb410108,0x893a0101,0xfb410108,0x893b0101,0xfb410108,0x893c0101,
	0xfb410108,0x893d0101,0xfb410108,0x893e0101,0xfb410108,0x893f0101,0xfb410108,0x89400101,
	0xfb410108,0x89420101,0xfb410108,0x89430101,0xfb410108,0x89450101,0xfb410108,0x89460101,
	0xfb410108,0x89470101,0xfb410108,0x89480101,0xfb410108,0x89490101,0xfb410108,0x894a0101,
	0xfb410108,0x894b0101,0xfb410108,0x894c0101,0xfb410108,0x894d0101,0xfb410108,0x894e0101,
	0xfb410108,0x894f0101,0xfb410108,0x89500101,0xfb410108,0x89510101,0xfb410108,0x89520101,
	0xfb410108,0x89530101,0xfb410108,0x89540101,0xfb410108,0x89550101,0xfb410108,0x89560101,
	0xfb410108,0x89570101,0xfb410108,0x89580101,0xfb410108,0x89590101,0xfb410108,0x895a0101,
	0xfb410108,0x895b0101,0xfb410108,0x895c0101,0xfb410108,0x895d0101,0xfb410108,0x89600101,
	0xfb410108,0x89610101,0xfb410108,0x89620101,0xfb410108,0x89630101,0xfb410108,0x89640101,
	0xfb410108,0x89650101,0xfb410108,0x89670101,0xfb410108,0x89680101,0xfb410108,0x89690101,
	0xfb410108,0x896a0101,0xfb410108,0x896b0101,0xfb410108,0x896c0101,0xfb410108,0x896d0101,
	0xfb410108,0x896e0101,0xfb410108,0x896f0101,0xfb410108,0x89700101,0xfb410108,0x89710101,
	0xfb410108,0x89720101,0xfb410108,0x89730101,0xfb410108,0x89740101,0xfb410108,0x89750101,
	0xfb410108,0x89760101,0xfb410108,0x89770101,0xfb410108,0x89780101,0xfb410108,0x89790101,
	0xfb410108,0x897a0101,0xfb410108,0x897c0101,0xfb410108,0x897d0101,0xfb410108,0x897e0101,
	0xfb410108,0x89800101,0xfb410108,0x89820101,0xfb410108,0x89840101,0xfb410108,0x89850101,
	0xfb410108,0x89870101,0xfb410108,0x89880101,0xfb410108,0x89890101,0xfb410108,0x898a0101,
	0xfb410108,0x898b0101,0xfb410108,0x898c0101,0xfb410108,0x898d0101,0xfb410108,0x898e0101,
	0xfb410108,0x898f0101,0xfb410108,0x89900101,0xfb410108,0x89910101,0xfb410108,0x89920101,
	0xfb410108,0x89930101,0xfb410108,0x89940101,0xfb410108,0x89950101,0xfb410108,0x89960101,
	0xfb410108,0x89970101,0xfb410108,0x89980101,0xfb410108,0x89990101,0xfb410108,0x899a0101,
	0xfb410108,0x899b0101,0xfb410108,0x899c0101,0xfb410108,0x899d0101,0xfb410108,0x899e0101,
	0xfb410108,0x899f0101,0xfb410108,0x89a00101,0xfb410108,0x89a10101,0xfb410108,0x89a20101,
	0xfb410108,0x89a30101,0xfb410108,0x89a40101,0xfb410108,0x89a50101,0xfb410108,0x89a60101,
	0xfb410108,0x89a70101,0xfb410108,0x89a80101,0xfb410108,0x89a90101,0xfb410108,0x89aa0101,
	0xfb410108,0x89ab0101,0xfb410108,0x89ac0101,0xfb410108,0x89ad0101,0xfb410108,0x89ae0101,
	0xfb410108,0x89af0101,0xfb410108,0x89b00101,0xfb410108,0x89b10101,0xfb410108,0x89b20101,
	0xfb410108,0x89b30101,0xfb410108,0x89b40101,0xfb410108,0x89b50101,0xfb410108,0x89b60101,
	0xfb410108,0x89b70101,0xfb410108,0x89b80101,0xfb410108,0x89b90101,0xfb410108,0x89ba0101,
	0xfb410108,0x89bb0101,0xfb410108,0x89bc0101,0xfb410108,0x89bd0101,0xfb410108,0x89be0101,
	0xfb410108,0x89bf0101,0xfb410108,0x89c00101,0xfb410108,0x89c30101,0xfb410108,0x89cd0101,
	0xfb410108,0x89d30101,0xfb410108,0x89d40101,0xfb410108,0x89d50101,0xfb410108,0x89d70101,
	0xfb410108,0x89d80101,0xfb410108,0x89d90101,0xfb410108,0x89db0101,0xfb410108,0x89dd0101,
	0xfb410108,0x89df0101,0xfb410108,0x89e00101,0xfb410108,0x89e10101,0xfb410108,0x89e20101,
	0xfb410108,0x89e40101,0xfb410108,0x89e70101,0xfb410108,0x89e80101,0xfb410108,0x89e90101,
	0xfb410108,0x89ea0101,0xfb410108,0x89ec0101,0xfb410108,0x89ed0101,0xfb410108,0x89ee0101,
	0xfb410108,0x89f00101,0xfb410108,0x89f10101,0xfb410108,0x89f20101,0xfb410108,0x89f40101,
	0xfb410108,0x89f50101,0xfb410108,0x89f60101,0xfb410108,0x89f70101,0xfb410108,0x89f80101,
	0xfb410108,0x89f90101,0xfb410108,0x89fa0101,0xfb410108,0x89fb0101,0xfb410108,0x89fc0101,
	0xfb410108,0x89fd0101,0xfb410108,0x89fe0101,0xfb410108,0x89ff0101,0xfb410108,0x8a010101,
	0xfb410108,0x8a020101,0xfb410108,0x8a030101,0xfb410108,0x8a040101,0xfb410108,0x8a050101,
	0xfb410108,0x8a060101,0xfb410108,0x8a080101,0xfb410108,0x8a090101,0xfb410108,0x8a0a0101,
	0xfb410108,0x8a0b0101,0xfb410108,0x8a0c0101,0xfb410108,0x8a0d0101,0xfb410108,0x8a0e0101,
	0xfb410108,0x8a0f0101,0xfb410108,0x8a100101,0xfb410108,0x8a110101,0xfb410108,0x8a120101,
	0xfb410108,0x8a130101,0xfb410108,0x8a140101,0xfb410108,0x8a150101,0xfb410108,0x8a160101,
	0xfb410108,0x8a170101,0xfb410108,0x8a180101,0xfb410108,0x8a190101,0xfb410108,0x8a1a0101,
	0xfb410108,0x8a1b0101,0xfb410108,0x8a1c0101,0xfb410108,0x8a1d0101,0xfb410108,0x8a1e0101,
	0xfb410108,0x8a1f0101,0xfb410108,0x8a200101,0xfb410108,0x8a210101,0xfb410108,0x8a220101,
	0xfb410108,0x8a230101,0xfb410108,0x8a240101,0xfb410108,0x8a250101,0xfb410108,0x8a260101,
	0xfb410108,0x8a270101,0xfb410108,0x8a280101,0xfb410108,0x8a290101,0xfb410108,0x8a2a0101,
	0xfb410108,0x8a2b0101,0xfb410108,0x8a2c0101,0xfb410108,0x8a2d0101,0xfb410108,0x8a2e0101,
	0xfb410108,0x8a2f0101,0xfb410108,0x8a300101,0xfb410108,0x8a310101,0xfb410108,0x8a320101,
	0xfb410108,0x8a330101,0xfb410108,0x8a340101,0xfb410108,0x8a350101,0xfb410108,0x8a360101,
	0xfb410108,0x8a370101,0xfb410108,0x8a380101,0xfb410108,0x8a390101,0xfb410108,0x8a3a0101,
	0xfb410108,0x8a3b0101,0xfb410108,0x8a3c0101,0xfb410108,0x8a3d0101,0xfb410108,0x8a3f0101,
	0xfb410108,0x8a400101,0xfb410108,0x8a410101,0xfb410108,0x8a420101,0xfb410108,0x8a430101,
	0xfb410108,0x8a440101,0xfb410108,0x8a450101,0xfb410108,0x8a460101,0xfb410108,0x8a470101,
	0xfb410108,0x8a490101,0xfb410108,0x8a4a0101,0xfb410108,0x8a4b0101,0xfb410108,0x8a4c0101,
	0xfb410108,0x8a4d0101,0xfb410108,0x8a4e0101,0xfb410108,0x8a4f0101,0xfb410108,0x8a500101,
	0xfb410108,0x8a510101,0xfb410108,0x8a520101,0xfb410108,0x8a530101,0xfb410108,0x8a540101,
	0xfb410108,0x8a550101,0xfb410108,0x8a560101,0xfb410108,0x8a570101,0xfb410108,0x8a580101,
	0xfb410108,0x8a590101,0xfb410108,0x8a5a0101,0xfb410108,0x8a5b0101,0xfb410108,0x8a5c0101,
	0xfb410108,0x8a5d0101,0xfb410108,0x8a5e0101,0xfb410108,0x8a5f0101,0xfb410108,0x8a600101,
	0xfb410108,0x8a610101,0xfb410108,0x8a620101,0xfb410108,0x8a630101,0xfb410108,0x8a640101,
	0xfb410108,0x8a650101,0xfb410108,0x8a660101,0xfb410108,0x8a670101,0xfb410108,0x8a680101,
	0xfb410108,0x8a690101,0xfb410108,0x8a6a0101,0xfb410108,0x8a6b0101,0xfb410108,0x8a6c0101,
	0xfb410108,0x8a6d0101,0xfb410108,0x8a6e0101,0xfb410108,0x8a6f0101,0xfb410108,0x8a700101,
	0xfb410108,0x8a710101,0xfb410108,0x8a720101,0xfb410108,0x8a730101,0xfb410108,0x8a740101,
	0xfb410108,0x8a750101,0xfb410108,0x8a760101,0xfb410108,0x8a770101,0xfb410108,0x8a780101,
	0xfb410108,0x8a7a0101,0xfb410108,0x8a7b0101,0xfb410108,0x8a7c0101,0xfb410108,0x8a7d0101,
	0xfb410108,0x8a7e0101,0xfb410108,0x8a7f0101,0xfb410108,0x8a800101,0xfb410108,0x8a810101,
	0xfb410108,0x8a820101,0xfb410108,0x8a830101,0xfb410108,0x8a840101,0xfb410108,0x8a850101,
	0xfb410108,0x8a860101,0xfb410108,0x8a870101,0xfb410108,0x8a880101,0xfb410108,0x8a8b0101,
	0xfb410108,0x8a8c0101,0xfb410108,0x8a8d0101,0xfb410108,0x8a8e0101,0xfb410108,0x8a8f0101,
	0xfb410108,0x8a900101,0xfb410108,0x8a910101,0xfb410108,0x8a920101,0xfb410108,0x8a940101,
	0xfb410108,0x8a950101,0xfb410108,0x8a960101,0xfb410108,0x8a970101,0xfb410108,0x8a980101,
	0xfb410108,0x8a990101,0xfb410108,0x8a9a0101,0xfb410108,0x8a9b0101,0xfb410108,0x8a9c0101,
	0xfb410108,0x8a9d0101,0xfb410108,0x8a9e0101,0xfb410108,0x8a9f0101,0xfb410108,0x8aa00101,
	0xfb410108,0x8aa10101,0xfb410108,0x8aa20101,0xfb410108,0x8aa30101,0xfb410108,0x8aa40101,
	0xfb410108,0x8aa50101,0xfb410108,0x8aa60101,0xfb410108,0x8aa70101,0xfb410108,0x8aa80101,
	0xfb410108,0x8aa90101,0xfb410108,0x8aaa0101,0xfb410108,0x8aab0101,0xfb410108,0x8aac0101,
	0xfb410108,0x8aad0101,0xfb410108,0x8aae0101,0xfb410108,0x8aaf0101,0xfb410108,0x8ab00101,
	0xfb410108,0x8ab10101,0xfb410108,0x8ab20101,0xfb410108,0x8ab30101,0xfb410108,0x8ab40101,
	0xfb410108,0x8ab50101,0xfb410108,0x8ab60101,0xfb410108,0x8ab70101,0xfb410108,0x8ab80101,
	0xfb410108,0x8ab90101,0xfb410108,0x8aba0101,0xfb410108,0x8abb0101,0xfb410108,0x8abc0101,
	0xfb410108,0x8abd0101,0xfb410108,0x8abe0101,0xfb410108,0x8abf0101,0xfb410108,0x8ac00101,
	0xfb410108,0x8ac10101,0xfb410108,0x8ac20101,0xfb410108,0x8ac30101,0xfb410108,0x8ac40101,
	0xfb410108,0x8ac50101,0xfb410108,0x8ac60101,0xfb410108,0x8ac70101,0xfb410108,0x8ac80101,
	0xfb410108,0x8ac90101,0xfb410108,0x8aca0101,0xfb410108,0x8acb0101,0xfb410108,0x8acc0101,
	0xfb410108,0x8acd0101,0xfb410108,0x8ace0101,0xfb410108,0x8acf0101,0xfb410108,0x8ad00101,
	0xfb410108,0x8ad10101,0xfb410108,0x8ad20101,0xfb410108,0x8ad30101,0xfb410108,0x8ad40101,
	0xfb410108,0x8ad50101,0xfb410108,0x8ad60101,0xfb410108,0x8ad70101,0xfb410108,0x8ad80101,
	0xfb410108,0x8ad90101,0xfb410108,0x8ada0101,0xfb410108,0x8adb0101,0xfb410108,0x8adc0101,
	0xfb410108,0x8add0101,0xfb410108,0x8ade0101,0xfb410108,0x8adf0101,0xfb410108,0x8ae00101,
	0xfb410108,0x8ae10101,0xfb410108,0x8ae20101,0xfb410108,0x8ae30101,0xfb410108,0x8ae40101,
	0xfb410108,0x8ae50101,0xfb410108,0x8ae60101,0xfb410108,0x8ae70101,0xfb410108,0x8ae80101,
	0xfb410108,0x8ae90101,0xfb410108,0x8aea0101,0xfb410108,0x8aeb0101,0xfb410108,0x8aec0101,
	0xfb410108,0x8aed0101,0xfb410108,0x8aee0101,0xfb410108,0x8aef0101,0xfb410108,0x8af00101,
	0xfb410108,0x8af10101,0xfb410108,0x8af20101,0xfb410108,0x8af30101,0xfb410108,0x8af40101,
	0xfb410108,0x8af50101,0xfb410108,0x8af60101,0xfb410108,0x8af70101,0xfb410108,0x8af80101,
	0xfb410108,0x8af90101,0xfb410108,0x8afa0101,0xfb410108,0x8afb0101,0xfb410108,0x8afc0101,
	0xfb410108,0x8afd0101,0xfb410108,0x8afe0101,0xfb410108,0x8aff0101,0xfb410108,0x8b000101,
	0xfb410108,0x8b010101,0xfb410108,0x8b020101,0xfb410108,0x8b030101,0xfb410108,0x8b040101,
	0xfb410108,0x8b050101,0xfb410108,0x8b060101,0xfb410108,0x8b080101,0xfb410108,0x8b090101,
	0xfb410108,0x8b0a0101,0xfb410108,0x8b0b0101,0xfb410108,0x8b0c0101,0xfb410108,0x8b0d0101,
	0xfb410108,0x8b0e0101,0xfb410108,0x8b0f0101,0xfb410108,0x8b100101,0xfb410108,0x8b110101,
	0xfb410108,0x8b120101,0xfb410108,0x8b130101,0xfb410108,0x8b140101,0xfb410108,0x8b150101,
	0xfb410108,0x8b160101,0xfb410108,0x8b170101,0xfb410108,0x8b180101,0xfb410108,0x8b190101,
	0xfb410108,0x8b1a0101,0xfb410108,0x8b1b0101,0xfb410108,0x8b1c0101,0xfb410108,0x8b1d0101,
	0xfb410108,0x8b1e0101,0xfb410108,0x8b1f0101,0xfb410108,0x8b200101,0xfb410108,0x8b210101,
	0xfb410108,0x8b220101,0xfb410108,0x8b230101,0xfb410108,0x8b240101,0xfb410108,0x8b250101,
	0xfb410108,0x8b270101,0xfb410108,0x8b280101,0xfb410108,0x8b290101,0xfb410108,0x8b2a0101,
	0xfb410108,0x8b2b0101,0xfb410108,0x8b2c0101,0xfb410108,0x8b2d0101,0xfb410108,0x8b2e0101,
	0xfb410108,0x8b2f0101,0xfb410108,0x8b300101,0xfb410108,0x8b310101,0xfb410108,0x8b320101,
	0xfb410108,0x8b330101,0xfb410108,0x8b340101,0xfb410108,0x8b350101,0xfb410108,0x8b360101,
	0xfb410108,0x8b370101,0xfb410108,0x8b380101,0xfb410108,0x8b390101,0xfb410108,0x8b3a0101,
	0xfb410108,0x8b3b0101,0xfb410108,0x8b3c0101,0xfb410108,0x8b3d0101,0xfb410108,0x8b3e0101,
	0xfb410108,0x8b3f0101,0xfb410108,0x8b400101,0xfb410108,0x8b410101,0xfb410108,0x8b420101,
	0xfb410108,0x8b430101,0xfb410108,0x8b440101,0xfb410108,0x8b450101,0xfb410108,0x8b460101,
	0xfb410108,0x8b470101,0xfb410108,0x8b480101,0xfb410108,0x8b490101,0xfb410108,0x8b4a0101,
	0xfb410108,0x8b4b0101,0xfb410108,0x8b4c0101,0xfb410108,0x8b4d0101,0xfb410108,0x8b4e0101,
	0xfb410108,0x8b4f0101,0xfb410108,0x8b500101,0xfb410108,0x8b510101,0xfb410108,0x8b520101,
	0xfb410108,0x8b530101,0xfb410108,0x8b540101,0xfb410108,0x8b550101,0xfb410108,0x8b560101,
	0xfb410108,0x8b570101,0xfb410108,0x8b580101,0xfb410108,0x8b590101,0xfb410108,0x8b5a0101,
	0xfb410108,0x8b5b0101,0xfb410108,0x8b5c0101,0xfb410108,0x8b5d0101,0xfb410108,0x8b5e0101,
	0xfb410108,0x8b5f0101,0xfb410108,0x8b600101,0xfb410108,0x8b610101,0xfb410108,0x8b620101,
	0xfb410108,0x8b630101,0xfb410108,0x8b640101,0xfb410108,0x8b650101,0xfb410108,0x8b670101,
	0xfb410108,0x8b680101,0xfb410108,0x8b690101,0xfb410108,0x8b6a0101,0xfb410108,0x8b6b0101,
	0xfb410108,0x8b6d0101,0xfb410108,0x8b6e0101,0xfb410108,0x8b6f0101,0xfb410108,0x8b700101,
	0xfb410108,0x8b710101,0xfb410108,0x8b720101,0xfb410108,0x8b730101,0xfb410108,0x8b740101,
	0xfb410108,0x8b750101,0xfb410108,0x8b760101,0xfb410108,0x8b770101,0xfb410108,0x8b780101,
	0xfb410108,0x8b790101,0xfb410108,0x8b7a0101,0xfb410108,0x8b7b0101,0xfb410108,0x8b7c0101,
	0xfb410108,0x8b7d0101,0xfb410108,0x8b7e0101,0xfb410108,0x8b7f0101,0xfb410108,0x8b800101,
	0xfb410108,0x8b810101,0xfb410108,0x8b820101,0xfb410108,0x8b830101,0xfb410108,0x8b840101,
	0xfb410108,0x8b850101,0xfb410108,0x8b860101,0xfb410108,0x8b870101,0xfb410108,0x8b880101,
	0xfb410108,0x8b890101,0xfb410108,0x8b8a0101,0xfb410108,0x8b8b0101,0xfb410108,0x8b8c0101,
	0xfb410108,0x8b8d0101,0xfb410108,0x8b8e0101,0xfb410108,0x8b8f0101,0xfb410108,0x8b900101,
	0xfb410108,0x8b910101,0xfb410108,0x8b920101,0xfb410108,0x8b930101,0xfb410108,0x8b940101,
	0xfb410108,0x8b950101,0xfb410108,0x8b960101,0xfb410108,0x8b970101,0xfb410108,0x8b980101,
	0xfb410108,0x8b990101,0xfb410108,0x8b9a0101,0xfb410108,0x8b9b0101,0xfb410108,0x8b9c0101,
	0xfb410108,0x8b9d0101,0xfb410108,0x8b9e0101,0xfb410108,0x8b9f0101,0xfb410108,0x8bac0101,
	0xfb410108,0x8bb10101,0xfb410108,0x8bbb0101,0xfb410108,0x8bc70101,0xfb410108,0x8bd00101,
	0xfb410108,0x8bea0101,0xfb410108,0x8c090101,0xfb410108,0x8c1e0101,0xfb410108,0x8c380101,
	0xfb410108,0x8c390101,0xfb410108,0x8c3a0101,0xfb410108,0x8c3b0101,0xfb410108,0x8c3c0101,
	0xfb410108,0x8c3d0101,0xfb410108,0x8c3e0101,0xfb410108,0x8c3f0101,0xfb410108,0x8c400101,
	0xfb410108,0x8c420101,0xfb410108,0x8c430101,0xfb410108,0x8c440101,0xfb410108,0x8c450101,
	0xfb410108,0x8c480101,0xfb410108,0x8c4a0101,0xfb410108,0x8c4b0101,0xfb410108,0x8c4d0101,
	0xfb410108,0x8c4e0101,0xfb410108,0x8c4f0101,0xfb410108,0x8c500101,0xfb410108,0x8c510101,
	0xfb410108,0x8c520101,0xfb410108,0x8c530101,0xfb410108,0x8c540101,0xfb410108,0x8c560101,
	0xfb410108,0x8c570101,0xfb410108,0x8c580101,0xfb410108,0x8c590101,0xfb410108,0x8c5b0101,
	0xfb410108,0x8c5c0101,0xfb410108,0x8c5d0101,0xfb410108,0x8c5e0101,0xfb410108,0x8c5f0101,
	0xfb410108,0x8c600101,0xfb410108,0x8c630101,0xfb410108,0x8c640101,0xfb410108,0x8c650101,
	0xfb410108,0x8c660101,0xfb410108,0x8c670101,0xfb410108,0x8c680101,0xfb410108,0x8c690101,
	0xfb410108,0x8c6c0101,0xfb410108,0x8c6d0101,0xfb410108,0x8c6e0101,0xfb410108,0x8c6f0101,
	0xfb410108,0x8c700101,0xfb410108,0x8c710101,0xfb410108,0x8c720101,0xfb410108,0x8c740101,
	0xfb410108,0x8c750101,0xfb410108,0x8c760101,0xfb410108,0x8c770101,0xfb410108,0x8c7b0101,
	0xfb410108,0x8c7c0101,0xfb410108,0x8c7d0101,0xfb410108,0x8c7e0101,0xfb410108,0x8c7f0101,
	0xfb410108,0x8c800101,0xfb410108,0x8c810101,0xfb410108,0x8c830101,0xfb410108,0x8c840101,
	0xfb410108,0x8c860101,0xfb410108,0x8c870101,0xfb410108,0x8c880101,0xfb410108,0x8c8b0101,
	0xfb410108,0x8c8d0101,0xfb410108,0x8c8e0101,0xfb410108,0x8c8f0101,0xfb410108,0x8c900101,
	0xfb410108,0x8c910101,0xfb410108,0x8c920101,0xfb410108,0x8c930101,0xfb410108,0x8c950101,
	0xfb410108,0x8c960101,0xfb410108,0x8c970101,0xfb410108,0x8c990101,0xfb410108,0x8c9a0101,
	0xfb410108,0x8c9b0101,0xfb410108,0x8c9c0101,0xfb410108,0x8c9d0101,0xfb410108,0x8c9e0101,
	0xfb410108,0x8c9f0101,0xfb410108,0x8ca00101,0xfb410108,0x8ca10101,0xfb410108,0x8ca20101,
	0xfb410108,0x8ca30101,0xfb410108,0x8ca40101,0xfb410108,0x8ca50101,0xfb410108,0x8ca60101,
	0xfb410108,0x8ca70101,0xfb410108,0x8ca80101,0xfb410108,0x8ca90101,0xfb410108,0x8caa0101,
	0xfb410108,0x8cab0101,0xfb410108,0x8cac0101,0xfb410108,0x8cad0101,0xfb410108,0x8cae0101,
	0xfb410108,0x8caf0101,0xfb410108,0x8cb00101,0xfb410108,0x8cb10101,0xfb410108,0x8cb20101,
	0xfb410108,0x8cb30101,0xfb410108,0x8cb40101,0xfb410108,0x8cb50101,0xfb410108,0x8cb60101,
	0xfb410108,0x8cb70101,0xfb410108,0x8cb80101,0xfb410108,0x8cb90101,0xfb410108,0x8cba0101,
	0xfb410108,0x8cbb0101,0xfb410108,0x8cbc0101,0xfb410108,0x8cbd0101,0xfb410108,0x8cbe0101,
	0xfb410108,0x8cbf0101,0xfb410108,0x8cc00101,0xfb410108,0x8cc10101,0xfb410108,0x8cc20101,
	0xfb410108,0x8cc30101,0xfb410108,0x8cc40101,0xfb410108,0x8cc50101,0xfb410108,0x8cc60101,
	0xfb410108,0x8cc70101,0xfb410108,0x8cc80101,0xfb410108,0x8cc90101,0xfb410108,0x8cca0101,
	0xfb410108,0x8ccb0101,0xfb410108,0x8ccc0101,0xfb410108,0x8ccd0101,0xfb410108,0x8cce0101,
	0xfb410108,0x8ccf0101,0xfb410108,0x8cd00101,0xfb410108,0x8cd10101,0xfb410108,0x8cd20101,
	0xfb410108,0x8cd30101,0xfb410108,0x8cd40101,0xfb410108,0x8cd50101,0xfb410108,0x8cd60101,
	0xfb410108,0x8cd70101,0xfb410108,0x8cd80101,0xfb410108,0x8cd90101,0xfb410108,0x8cda0101,
	0xfb410108,0x8cdb0101,0xfb410108,0x8cdc0101,0xfb410108,0x8cdd0101,0xfb410108,0x8cde0101,
	0xfb410108,0x8cdf0101,0xfb410108,0x8ce00101,0xfb410108,0x8ce10101,0xfb410108,0x8ce20101,
	0xfb410108,0x8ce30101,0xfb410108,0x8ce40101,0xfb410108,0x8ce50101,0xfb410108,0x8ce60101,
	0xfb410108,0x8ce70101,0xfb410108,0x8ce80101,0xfb410108,0x8ce90101,0xfb410108,0x8cea0101,
	0xfb410108,0x8ceb0101,0xfb410108,0x8cec0101,0xfb410108,0x8ced0101,0xfb410108,0x8cee0101,
	0xfb410108,0x8cef0101,0xfb410108,0x8cf00101,0xfb410108,0x8cf10101,0xfb410108,0x8cf20101,
	0xfb410108,0x8cf30101,0xfb410108,0x8cf40101,0xfb410108,0x8cf50101,0xfb410108,0x8cf60101,
	0xfb410108,0x8cf70101,0xfb410108,0x8cf80101,0xfb410108,0x8cf90101,0xfb410108,0x8cfa0101,
	0xfb410108,0x8cfb0101,0xfb410108,0x8cfc0101,0xfb410108,0x8cfd0101,0xfb410108,0x8cfe0101,
	0xfb410108,0x8cff0101,0xfb410108,0x8d000101,0xfb410108,0x8d010101,0xfb410108,0x8d020101,
	0xfb410108,0x8d030101,0xfb410108,0x8d040101,0xfb410108,0x8d050101,0xfb410108,0x8d060101,
	0xfb410108,0x8d070101,0xfb410108,0x8d080101,0xfb410108,0x8d090101,0xfb410108,0x8d0a0101,
	0xfb410108,0x8d0b0101,0xfb410108,0x8d0c0101,0xfb410108,0x8d0d0101,0xfb410108,0x8d0e0101,
	0xfb410108,0x8d0f0101,0xfb410108,0x8d100101,0xfb410108,0x8d110101,0xfb410108,0x8d120101,
	0xfb410108,0x8d130101,0xfb410108,0x8d140101,0xfb410108,0x8d150101,0xfb410108,0x8d160101,
	0xfb410108,0x8d170101,0xfb410108,0x8d180101,0xfb410108,0x8d190101,0xfb410108,0x8d1a0101,
	0xfb410108,0x8d1b0101,0xfb410108,0x8d1c0101,0xfb410108,0x8d200101,0xfb410108,0x8d510101,
	0xfb410108,0x8d520101,0xfb410108,0x8d570101,0xfb410108,0x8d5f0101,0xfb410108,0x8d650101,
	0xfb410108,0x8d680101,0xfb410108,0x8d690101,0xfb410108,0x8d6a0101,0xfb410108,0x8d6c0101,
	0xfb410108,0x8d6e0101,0xfb410108,0x8d6f0101,0xfb410108,0x8d710101,0xfb410108,0x8d720101,
	0xfb410108,0x8d780101,0xfb410108,0x8d790101,0xfb410108,0x8d7a0101,0xfb410108,0x8d7b0101,
	0xfb410108,0x8d7c0101,0xfb410108,0x8d7d0101,0xfb410108,0x8d7e0101,0xfb410108,0x8d7f0101,
	0xfb410108,0x8d800101,0xfb410108,0x8d820101,0xfb410108,0x8d830101,0xfb410108,0x8d860101,
	0xfb410108,0x8d870101,0xfb410108,0x8d880101,0xfb410108,0x8d890101,0xfb410108,0x8d8c0101,
	0xfb410108,0x8d8d0101,0xfb410108,0x8d8e0101,0xfb410108,0x8d8f0101,0xfb410108,0x8d900101,
	0xfb410108,0x8d920101,0xfb410108,0x8d930101,0xfb410108,0x8d950101,0xfb410108,0x8d960101,
	0xfb410108,0x8d970101,0xfb410108,0x8d980101,0xfb410108,0x8d990101,0xfb410108,0x8d9a0101,
	0xfb410108,0x8d9b0101,0xfb410108,0x8d9c0101,0xfb410108,0x8d9d0101,0xfb410108,0x8d9e0101,
	0xfb410108,0x8da00101,0xfb410108,0x8da10101,0xfb410108,0x8da20101,0xfb410108,0x8da40101,
	0xfb410108,0x8da50101,0xfb410108,0x8da60101,0xfb410108,0x8da70101,0xfb410108,0x8da80101,
	0xfb410108,0x8da90101,0xfb410108,0x8daa0101,0xfb410108,0x8dab0101,0xfb410108,0x8dac0101,
	0xfb410108,0x8dad0101,0xfb410108,0x8dae0101,0xfb410108,0x8daf0101,0xfb410108,0x8db00101,
	0xfb410108,0x8db20101,0xfb410108,0x8db60101,0xfb410108,0x8db70101,0xfb410108,0x8db90101,
	0xfb410108,0x8dbb0101,0xfb410108,0x8dbd0101,0xfb410108,0x8dc00101,0xfb410108,0x8dc10101,
	0xfb410108,0x8dc20101,0xfb410108,0x8dc50101,0xfb410108,0x8dc70101,0xfb410108,0x8dc80101,
	0xfb410108,0x8dc90101,0xfb410108,0x8dca0101,0xfb410108,0x8dcd0101,0xfb410108,0x8dd00101,
	0xfb410108,0x8dd20101,0xfb410108,0x8dd30101,0xfb410108,0x8dd40101,0xfb410108,0x8dd50101,
	0xfb410108,0x8dd80101,0xfb410108,0x8dd90101,0xfb410108,0x8ddc0101,0xfb410108,0x8de00101,
	0xfb410108,0x8de10101,0xfb410108,0x8de20101,0xfb410108,0x8de50101,0xfb410108,0x8de60101,
	0xfb410108,0x8de70101,0xfb410108,0x8de90101,0xfb410108,0x8ded0101,0xfb410108,0x8dee0101,
	0xfb410108,0x8df00101,0xfb410108,0x8df10101,0xfb410108,0x8df20101,0xfb410108,0x8df40101,
	0xfb410108,0x8df60101,0xfb410108,0x8dfc0101,0xfb410108,0x8dfe0101,0xfb410108,0x8dff0101,
	0xfb410108,0x8e000101,0xfb410108,0x8e010101,0xfb410108,0x8e020101,0xfb410108,0x8e030101,
	0xfb410108,0x8e040101,0xfb410108,0x8e060101,0xfb410108,0x8e070101,0xfb410108,0x8e080101,
	0xfb410108,0x8e0b0101,0xfb410108,0x8e0d0101,0xfb410108,0x8e0e0101,0xfb410108,0x8e100101,
	0xfb410108,0x8e110101,0xfb410108,0x8e120101,0xfb410108,0x8e130101,0xfb410108,0x8e150101,
	0xfb410108,0x8e160101,0xfb410108,0x8e170101,0xfb410108,0x8e180101,0xfb410108,0x8e190101,
	0xfb410108,0x8e1a0101,0xfb410108,0x8e1b0101,0xfb410108,0x8e1c0101,0xfb410108,0x8e200101,
	0xfb410108,0x8e210101,0xfb410108,0x8e240101,0xfb410108,0x8e250101,0xfb410108,0x8e260101,
	0xfb410108,0x8e270101,0xfb410108,0x8e280101,0xfb410108,0x8e2b0101,0xfb410108,0x8e2d0101,
	0xfb410108,0x8e300101,0xfb410108,0x8e320101,0xfb410108,0x8e330101,0xfb410108,0x8e340101,
	0xfb410108,0x8e360101,0xfb410108,0x8e370101,0xfb410108,0x8e380101,0xfb410108,0x8e3b0101,
	0xfb410108,0x8e3c0101,0xfb410108,0x8e3e0101,0xfb410108,0x8e3f0101,0xfb410108,0x8e430101,
	0xfb410108,0x8e450101,0xfb410108,0x8e460101,0xfb410108,0x8e4c0101,0xfb410108,0x8e4d0101,
	0xfb410108,0x8e4e0101,0xfb410108,0x8e4f0101,0xfb410108,0x8e500101,0xfb410108,0x8e530101,
	0xfb410108,0x8e540101,0xfb410108,0x8e550101,0xfb410108,0x8e560101,0xfb410108,0x8e570101,
	0xfb410108,0x8e580101,0xfb410108,0x8e5a0101,0xfb410108,0x8e5b0101,0xfb410108,0x8e5c0101,
	0xfb410108,0x8e5d0101,0xfb410108,0x8e5e0101,0xfb410108,0x8e5f0101,0xfb410108,0x8e600101,
	0xfb410108,0x8e610101,0xfb410108,0x8e620101,0xfb410108,0x8e630101,0xfb410108,0x8e640101,
	0xfb410108,0x8e650101,0xfb410108,0x8e670101,0xfb410108,0x8e680101,0xfb410108,0x8e6a0101,
	0xfb410108,0x8e6b0101,0xfb410108,0x8e6e0101,0xfb410108,0x8e710101,0xfb410108,0x8e730101,
	0xfb410108,0x8e750101,0xfb410108,0x8e770101,0xfb410108,0x8e780101,0xfb410108,0x8e790101,
	0xfb410108,0x8e7a0101,0xfb410108,0x8e7b0101,0xfb410108,0x8e7d0101,0xfb410108,0x8e7e0101,
	0xfb410108,0x8e800101,0xfb410108,0x8e820101,0xfb410108,0x8e830101,0xfb410108,0x8e840101,
	0xfb410108,0x8e860101,0xfb410108,0x8e880101,0xfb410108,0x8e890101,0xfb410108,0x8e8a0101,
	0xfb410108,0x8e8b0101,0xfb410108,0x8e8c0101,0xfb410108,0x8e8d0101,0xfb410108,0x8e8e0101,
	0xfb410108,0x8e910101,0xfb410108,0x8e920101,0xfb410108,0x8e930101,0xfb410108,0x8e950101,
	0xfb410108,0x8e960101,0xfb410108,0x8e970101,0xfb410108,0x8e980101,0xfb410108,0x8e990101,
	0xfb410108,0x8e9a0101,0xfb410108,0x8e9b0101,0xfb410108,0x8e9d0101,0xfb410108,0x8e9f0101,
	0xfb410108,0x8ea00101,0xfb410108,0x8ea10101,0xfb410108,0x8ea20101,0xfb410108,0x8ea30101,
	0xfb410108,0x8ea40101,0xfb410108,0x8ea50101,0xfb410108,0x8ea60101,0xfb410108,0x8ea70101,
	0xfb410108,0x8ea80101,0xfb410108,0x8ea90101,0xfb410108,0x8eaa0101,0xfb410108,0x8ead0101,
	0xfb410108,0x8eae0101,0xfb410108,0x8eb00101,0xfb410108,0x8eb10101,0xfb410108,0x8eb30101,
	0xfb410108,0x8eb40101,0xfb410108,0x8eb50101,0xfb410108,0x8eb60101,0xfb410108,0x8eb70101,
	0xfb410108,0x8eb80101,0xfb410108,0x8eb90101,0xfb410108,0x8ebb0101,0xfb410108,0x8ebc0101,
	0xfb410108,0x8ebd0101,0xfb410108,0x8ebe0101,0xfb410108,0x8ebf0101,0xfb410108,0x8ec00101,
	0xfb410108,0x8ec10101,0xfb410108,0x8ec20101,0xfb410108,0x8ec30101,0xfb410108,0x8ec40101,
	0xfb410108,0x8ec50101,0xfb410108,0x8ec60101,0xfb410108,0x8ec70101,0xfb410108,0x8ec80101,
	0xfb410108,0x8ec90101,0xfb410108,0x8eca0101,0xfb410108,0x8ecb0101,0xfb410108,0x8ecc0101,
	0xfb410108,0x8ecd0101,0xfb410108,0x8ecf0101,0xfb410108,0x8ed00101,0xfb410108,0x8ed10101,
	0xfb410108,0x8ed20101,0xfb410108,0x8ed30101,0xfb410108,0x8ed40101,0xfb410108,0x8ed50101,
	0xfb410108,0x8ed60101,0xfb410108,0x8ed70101,0xfb410108,0x8ed80101,0xfb410108,0x8ed90101,
	0xfb410108,0x8eda0101,0xfb410108,0x8edb0101,0xfb410108,0x8edc0101,0xfb410108,0x8edd0101,
	0xfb410108,0x8ede0101,0xfb410108,0x8edf0101,0xfb410108,0x8ee00101,0xfb410108,0x8ee10101,
	0xfb410108,0x8ee20101,0xfb410108,0x8ee30101,0xfb410108,0x8ee40101,0xfb410108,0x8ee50101,
	0xfb410108,0x8ee60101,0xfb410108,0x8ee70101,0xfb410108,0x8ee80101,0xfb410108,0x8ee90101,
	0xfb410108,0x8eea0101,0xfb410108,0x8eeb0101,0xfb410108,0x8eec0101,0xfb410108,0x8eed0101,
	0xfb410108,0x8eee0101,0xfb410108,0x8eef0101,0xfb410108,0x8ef00101,0xfb410108,0x8ef10101,
	0xfb410108,0x8ef20101,0xfb410108,0x8ef30101,0xfb410108,0x8ef40101,0xfb410108,0x8ef50101,
	0xfb410108,0x8ef60101,0xfb410108,0x8ef70101,0xfb410108,0x8ef80101,0xfb410108,0x8ef90101,
	0xfb410108,0x8efa0101,0xfb410108,0x8efb0101,0xfb410108,0x8efc0101,0xfb410108,0x8efd0101,
	0xfb410108,0x8efe0101,0xfb410108,0x8eff0101,0xfb410108,0x8f000101,0xfb410108,0x8f010101,
	0xfb410108,0x8f020101,0xfb410108,0x8f030101,0xfb410108,0x8f040101,0xfb410108,0x8f050101,
	0xfb410108,0x8f060101,0xfb410108,0x8f070101,0xfb410108,0x8f080101,0xfb410108,0x8f090101,
	0xfb410108,0x8f0a0101,0xfb410108,0x8f0b0101,0xfb410108,0x8f0c0101,0xfb410108,0x8f0d0101,
	0xfb410108,0x8f0e0101,0xfb410108,0x8f0f0101,0xfb410108,0x8f100101,0xfb410108,0x8f110101,
	0xfb410108,0x8f120101,0xfb410108,0x8f130101,0xfb410108,0x8f140101,0xfb410108,0x8f150101,
	0xfb410108,0x8f160101,0xfb410108,0x8f170101,0xfb410108,0x8f180101,0xfb410108,0x8f190101,
	0xfb410108,0x8f1a0101,0xfb410108,0x8f1b0101,0xfb410108,0x8f1c0101,0xfb410108,0x8f1d0101,
	0xfb410108,0x8f1e0101,0xfb410108,0x8f1f0101,0xfb410108,0x8f200101,0xfb410108,0x8f210101,
	0xfb410108,0x8f220101,0xfb410108,0x8f230101,0xfb410108,0x8f240101,0xfb410108,0x8f250101,
	0xfb410108,0x8f260101,0xfb410108,0x8f270101,0xfb410108,0x8f280101,0xfb410108,0x8f290101,
	0xfb410108,0x8f2a0101,0xfb410108,0x8f2b0101,0xfb410108,0x8f2c0101,0xfb410108,0x8f2d0101,
	0xfb410108,0x8f2e0101,0xfb410108,0x8f2f0101,0xfb410108,0x8f300101,0xfb410108,0x8f310101,
	0xfb410108,0x8f320101,0xfb410108,0x8f330101,0xfb410108,0x8f340101,0xfb410108,0x8f350101,
	0xfb410108,0x8f360101,0xfb410108,0x8f370101,0xfb410108,0x8f380101,0xfb410108,0x8f390101,
	0xfb410108,0x8f3a0101,0xfb410108,0x8f3b0101,0xfb410108,0x8f3c0101,0xfb410108,0x8f3d0101,
	0xfb410108,0x8f3e0101,0xfb410108,0x8f3f0101,0xfb410108,0x8f400101,0xfb410108,0x8f410101,
	0xfb410108,0x8f420101,0xfb410108,0x8f430101,0xfb410108,0x8f440101,0xfb410108,0x8f450101,
	0xfb410108,0x8f460101,0xfb410108,0x8f470101,0xfb410108,0x8f480101,0xfb410108,0x8f490101,
	0xfb410108,0x8f4a0101,0xfb410108,0x8f4b0101,0xfb410108,0x8f4c0101,0xfb410108,0x8f4d0101,
	0xfb410108,0x8f4e0101,0xfb410108,0x8f4f0101,0xfb410108,0x8f500101,0xfb410108,0x8f510101,
	0xfb410108,0x8f520101,0xfb410108,0x8f530101,0xfb410108,0x8f540101,0xfb410108,0x8f550101,
	0xfb410108,0x8f560101,0xfb410108,0x8f570101,0xfb410108,0x8f580101,0xfb410108,0x8f590101,
	0xfb410108,0x8f5a0101,0xfb410108,0x8f5b0101,0xfb410108,0x8f5c0101,0xfb410108,0x8f5d0101,
	0xfb410108,0x8f5e0101,0xfb410108,0x8f5f0101,0xfb410108,0x8f600101,0xfb410108,0x8f610101,
	0xfb410108,0x8f620101,0xfb410108,0x8f630101,0xfb410108,0x8f640101,0xfb410108,0x8f650101,
	0xfb410108,0x8f6a0101,0xfb410108,0x8f800101,0xfb410108,0x8f8c0101,0xfb410108,0x8f920101,
	0xfb410108,0x8f9d0101,0xfb410108,0x8fa00101,0xfb410108,0x8fa10101,0xfb410108,0x8fa20101,
	0xfb410108,0x8fa40101,0xfb410108,0x8fa50101,0xfb410108,0x8fa60101,0xfb410108,0x8fa70101,
	0xfb410108,0x8faa0101,0xfb410108,0x8fac0101,0xfb410108,0x8fad0101,0xfb410108,0x8fae0101,
	0xfb410108,0x8faf0101,0xfb410108,0x8fb20101,0xfb410108,0x8fb30101,0xfb410108,0x8fb40101,
	0xfb410108,0x8fb50101,0xfb410108,0x8fb70101,0xfb410108,0x8fb80101,0xfb410108,0x8fba0101,
	0xfb410108,0x8fbb0101,0xfb410108,0x8fbc0101,0xfb410108,0x8fbf0101,0xfb410108,0x8fc00101,
	0xfb410108,0x8fc30101,0xfb410108,0x8fc60101,0xfb410108,0x8fc90101,0xfb410108,0x8fca0101,
	0xfb410108,0x8fcb0101,0xfb410108,0x8fcc0101,0xfb410108,0x8fcd0101,0xfb410108,0x8fcf0101,
	0xfb410108,0x8fd20101,0xfb410108,0x8fd60101,0xfb410108,0x8fd70101,0xfb410108,0x8fda0101,
	0xfb410108,0x8fe00101,0xfb410108,0x8fe10101,0xfb410108,0x8fe30101,0xfb410108,0x8fe70101,
	0xfb410108,0x8fec0101,0xfb410108,0x8fef0101,0xfb410108,0x8ff10101,0xfb410108,0x8ff20101,
	0xfb410108,0x8ff40101,0xfb410108,0x8ff50101,0xfb410108,0x8ff60101,0xfb410108,0x8ffa0101,
	0xfb410108,0x8ffb0101,0xfb410108,0x8ffc0101,0xfb410108,0x8ffe0101,0xfb410108,0x8fff0101,
	0xfb410108,0x90070101,0xfb410108,0x90080101,0xfb410108,0x900c0101,0xfb410108,0x900e0101,
	0xfb410108,0x90130101,0xfb410108,0x90150101,0xfb410108,0x90180101,0xfb410108,0x90190101,
	0xfb410108,0x901c0101,0xfb410108,0x90230101,0xfb410108,0x90240101,0xfb410108,0x90250101,
	0xfb410108,0x90270101,0xfb410108,0x90280101,0xfb410108,0x90290101,0xfb410108,0x902a0101,
	0xfb410108,0x902b0101,0xfb410108,0x902c0101,0xfb410108,0x90300101,0xfb410108,0x90310101,
	0xfb410108,0x90320101,0xfb410108,0x90330101,0xfb410108,0x90340101,0xfb410108,0x90370101,
	0xfb410108,0x90390101,0xfb410108,0x903a0101,0xfb410108,0x903d0101,0xfb410108,0x903f0101,
	0xfb410108,0x90400101,0xfb410108,0x90430101,0xfb410108,0x90450101,0xfb410108,0x90460101,
	0xfb410108,0x90480101,0xfb410108,0x90490101,0xfb410108,0x904a0101,0xfb410108,0x904b0101,
	0xfb410108,0x904c0101,0xfb410108,0x904e0101,0xfb410108,0x90540101,0xfb410108,0x90550101,
	0xfb410108,0x90560101,0xfb410108,0x90590101,0xfb410108,0x905a0101,0xfb410108,0x905c0101,
	0xfb410108,0x905d0101,0xfb410108,0x905e0101,0xfb410108,0x905f0101,0xfb410108,0x90600101,
	0xfb410108,0x90610101,0xfb410108,0x90640101,0xfb410108,0x90660101,0xfb410108,0x90670101,
	0xfb410108,0x90690101,0xfb410108,0x906a0101,0xfb410108,0x906b0101,0xfb410108,0x906c0101,
	0xfb410108,0x906f0101,0xfb410108,0x90700101,0xfb410108,0x90710101,0xfb410108,0x90720101,
	0xfb410108,0x90730101,0xfb410108,0x90760101,0xfb410108,0x90770101,0xfb410108,0x90780101,
	0xfb410108,0x90790101,0xfb410108,0x907a0101,0xfb410108,0x907b0101,0xfb410108,0x907c0101,
	0xfb410108,0x907e0101,0xfb410108,0x90810101,0xfb410108,0x90840101,0xfb410108,0x90850101,
	0xfb410108,0x90860101,0xfb410108,0x90870101,0xfb410108,0x90890101,0xfb410108,0x908a0101,
	0xfb410108,0x908c0101,0xfb410108,0x908d0101,0xfb410108,0x908e0101,0xfb410108,0x908f0101,
	0xfb410108,0x90900101,0xfb410108,0x90920101,0xfb410108,0x90940101,0xfb410108,0x90960101,
	0xfb410108,0x90980101,0xfb410108,0x909a0101,0xfb410108,0x909c0101,0xfb410108,0x909e0101,
	0xfb410108,0x909f0101,0xfb410108,0x90a00101,0xfb410108,0x90a40101,0xfb410108,0x90a50101,
	0xfb410108,0x90a70101,0xfb410108,0x90a80101,0xfb410108,0x90a90101,0xfb410108,0x90ab0101,
	0xfb410108,0x90ad0101,0xfb410108,0x90b20101,0xfb410108,0x90b70101,0xfb410108,0x90bc0101,
	0xfb410108,0x90bd0101,0xfb410108,0x90bf0101,0xfb410108,0x90c00101,0xfb410108,0x90c20101,
	0xfb410108,0x90c30101,0xfb410108,0x90c60101,0xfb410108,0x90c80101,0xfb410108,0x90c90101,
	0xfb410108,0x90cb0101,0xfb410108,0x90cc0101,0xfb410108,0x90cd0101,0xfb410108,0x90d20101,
	0xfb410108,0x90d40101,0xfb410108,0x90d50101,0xfb410108,0x90d60101,0xfb410108,0x90d80101,
	0xfb410108,0x90d90101,0xfb410108,0x90da0101,0xfb410108,0x90de0101,0xfb410108,0x90df0101,
	0xfb410108,0x90e00101,0xfb410108,0x90e30101,0xfb410108,0x90e40101,0xfb410108,0x90e50101,
	0xfb410108,0x90e90101,0xfb410108,0x90ea0101,0xfb410108,0x90ec0101,0xfb410108,0x90ee0101,
	0xfb410108,0x90f00101,0xfb410108,0x90f10101,0xfb410108,0x90f20101,0xfb410108,0x90f30101,
	0xfb410108,0x90f50101,0xfb410108,0x90f60101,0xfb410108,0x90f70101,0xfb410108,0x90f90101,
	0xfb410108,0x90fa0101,0xfb410108,0x90fb0101,0xfb410108,0x90fc0101,0xfb410108,0x90ff0101,
	0xfb410108,0x91000101,0xfb410108,0x91010101,0xfb410108,0x91030101,0xfb410108,0x91050101,
	0xfb410108,0x91060101,0xfb410108,0x91070101,0xfb410108,0x91080101,0xfb410108,0x91090101,
	0xfb410108,0x910a0101,0xfb410108,0x910b0101,0xfb410108,0x910c0101,0xfb410108,0x910d0101,
	0xfb410108,0x910e0101,0xfb410108,0x910f0101,0xfb410108,0x91100101,0xfb410108,0x91110101,
	0xfb410108,0x91120101,0xfb410108,0x91130101,0xfb410108,0x91140101,0xfb410108,0x91150101,
	0xfb410108,0x91160101,0xfb410108,0x91170101,0xfb410108,0x91180101,0xfb410108,0x911a0101,
	0xfb410108,0x911b0101,0xfb410108,0x911c0101,0xfb410108,0x911d0101,0xfb410108,0x911f0101,
	0xfb410108,0x91200101,0xfb410108,0x91210101,0xfb410108,0x91240101,0xfb410108,0x91250101,
	0xfb410108,0x91260101,0xfb410108,0x91270101,0xfb410108,0x91280101,0xfb410108,0x91290101,
	0xfb410108,0x912a0101,0xfb410108,0x912b0101,0xfb410108,0x912c0101,0xfb410108,0x912d0101,
	0xfb410108,0x912e0101,0xfb410108,0x91300101,0xfb410108,0x91320101,0xfb410108,0x91330101,
	0xfb410108,0x91340101,0xfb410108,0x91350101,0xfb410108,0x91360101,0xfb410108,0x91370101,
	0xfb410108,0x91380101,0xfb410108,0x913a0101,0xfb410108,0x913b0101,0xfb410108,0x913c0101,
	0xfb410108,0x913d0101,0xfb410108,0x913e0101,0xfb410108,0x913f0101,0xfb410108,0x91400101,
	0xfb410108,0x91410101,0xfb410108,0x91420101,0xfb410108,0x91440101,0xfb410108,0x91450101,
	0xfb410108,0x91470101,0xfb410108,0x91480101,0xfb410108,0x91510101,0xfb410108,0x91530101,
	0xfb410108,0x91540101,0xfb410108,0x91550101,0xfb410108,0x91560101,0xfb410108,0x91580101,
	0xfb410108,0x91590101,0xfb410108,0x915b0101,0xfb410108,0x915c0101,0xfb410108,0x915f0101,
	0xfb410108,0x91600101,0xfb410108,0x91660101,0xfb410108,0x91670101,0xfb410108,0x91680101,
	0xfb410108,0x916b0101,0xfb410108,0x916d0101,0xfb410108,0x91730101,0xfb410108,0x917a0101,
	0xfb410108,0x917b0101,0xfb410108,0x917c0101,0xfb410108,0x91800101,0xfb410108,0x91810101,
	0xfb410108,0x91820101,0xfb410108,0x91830101,0xfb410108,0x91840101,0xfb410108,0x91860101,
	0xfb410108,0x91880101,0xfb410108,0x918a0101,0xfb410108,0x918e0101,0xfb410108,0x918f0101,
	0xfb410108,0x91930101,0xfb410108,0x91940101,0xfb410108,0x91950101,0xfb410108,0x91960101,
	0xfb410108,0x91970101,0xfb410108,0x91980101,0xfb410108,0x91990101,0xfb410108,0x919c0101,
	0xfb410108,0x919d0101,0xfb410108,0x919e0101,0xfb410108,0x919f0101,0xfb410108,0x91a00101,
	0xfb410108,0x91a10101,0xfb410108,0x91a40101,0xfb410108,0x91a50101,0xfb410108,0x91a60101,
	0xfb410108,0x91a70101,0xfb410108,0x91a80101,0xfb410108,0x91a90101,0xfb410108,0x91ab0101,
	0xfb410108,0x91ac0101,0xfb410108,0x91b00101,0xfb410108,0x91b10101,0xfb410108,0x91b20101,
	0xfb410108,0x91b30101,0xfb410108,0x91b60101,0xfb410108,0x91b70101,0xfb410108,0x91b80101,
	0xfb410108,0x91b90101,0xfb410108,0x91bb0101,0xfb410108,0x91bc0101,0xfb410108,0x91bd0101,
	0xfb410108,0x91be0101,0xfb410108,0x91bf0101,0xfb410108,0x91c00101,0xfb410108,0x91c10101,
	0xfb410108,0x91c20101,0xfb410108,0x91c30101,0xfb410108,0x91c40101,0xfb410108,0x91c50101,
	0xfb410108,0x91c60101,0xfb410108,0x91c80101,0xfb410108,0x91cb0101,0xfb410108,0x91d00101,
	0xfb410108,0x91d20101,0xfb410108,0x91d30101,0xfb410108,0x91d40101,0xfb410108,0x91d50101,
	0xfb410108,0x91d60101,0xfb410108,0x91d70101,0xfb410108,0x91d80101,0xfb410108,0x91d90101,
	0xfb410108,0x91da0101,0xfb410108,0x91db0101,0xfb410108,0x91dd0101,0xfb410108,0x91de0101,
	0xfb410108,0x91df0101,0xfb410108,0x91e00101,0xfb410108,0x91e10101,0xfb410108,0x91e20101,
	0xfb410108,0x91e30101,0xfb410108,0x91e40101,0xfb410108,0x91e50101,0xfb410108,0x91e60101,
	0xfb410108,0x91e70101,0xfb410108,0x91e80101,0xfb410108,0x91e90101,0xfb410108,0x91ea0101,
	0xfb410108,0x91eb0101,0xfb410108,0x91ec0101,0xfb410108,0x91ed0101,0xfb410108,0x91ee0101,
	0xfb410108,0x91ef0101,0xfb410108,0x91f00101,0xfb410108,0x91f10101,0xfb410108,0x91f20101,
	0xfb410108,0x91f30101,0xfb410108,0x91f40101,0xfb410108,0x91f50101,0xfb410108,0x91f60101,
	0xfb410108,0x91f70101,0xfb410108,0x91f80101,0xfb410108,0x91f90101,0xfb410108,0x91fa0101,
	0xfb410108,0x91fb0101,0xfb410108,0x91fc0101,0xfb410108,0x91fd0101,0xfb410108,0x91fe0101,
	0xfb410108,0x91ff0101,0xfb410108,0x92000101,0xfb410108,0x92010101,0xfb410108,0x92020101,
	0xfb410108,0x92030101,0xfb410108,0x92040101,0xfb410108,0x92050101,0xfb410108,0x92060101,
	0xfb410108,0x92070101,0xfb410108,0x92080101,0xfb410108,0x92090101,0xfb410108,0x920a0101,
	0xfb410108,0x920b0101,0xfb410108,0x920c0101,0xfb410108,0x920d0101,0xfb410108,0x920e0101,
	0xfb410108,0x920f0101,0xfb410108,0x92100101,0xfb410108,0x92110101,0xfb410108,0x92120101,
	0xfb410108,0x92130101,0xfb410108,0x92140101,0xfb410108,0x92150101,0xfb410108,0x92160101,
	0xfb410108,0x92170101,0xfb410108,0x92180101,0xfb410108,0x92190101,0xfb410108,0x921a0101,
	0xfb410108,0x921b0101,0xfb410108,0x921c0101,0xfb410108,0x921d0101,0xfb410108,0x921e0101,
	0xfb410108,0x921f0101,0xfb410108,0x92200101,0xfb410108,0x92210101,0xfb410108,0x92220101,
	0xfb410108,0x92230101,0xfb410108,0x92240101,0xfb410108,0x92250101,0xfb410108,0x92260101,
	0xfb410108,0x92270101,0xfb410108,0x92280101,0xfb410108,0x92290101,0xfb410108,0x922a0101,
	0xfb410108,0x922b0101,0xfb410108,0x922c0101,0xfb410108,0x922d0101,0xfb410108,0x922e0101,
	0xfb410108,0x922f0101,0xfb410108,0x92300101,0xfb410108,0x92310101,0xfb410108,0x92320101,
	0xfb410108,0x92330101,0xfb410108,0x92340101,0xfb410108,0x92350101,0xfb410108,0x92360101,
	0xfb410108,0x92370101,0xfb410108,0x92380101,0xfb410108,0x92390101,0xfb410108,0x923a0101,
	0xfb410108,0x923b0101,0xfb410108,0x923c0101,0xfb410108,0x923d0101,0xfb410108,0x923e0101,
	0xfb410108,0x923f0101,0xfb410108,0x92400101,0xfb410108,0x92410101,0xfb410108,0x92420101,
	0xfb410108,0x92430101,0xfb410108,0x92440101,0xfb410108,0x92450101,0xfb410108,0x92460101,
	0xfb410108,0x92470101,0xfb410108,0x92480101,0xfb410108,0x92490101,0xfb410108,0x924a0101,
	0xfb410108,0x924b0101,0xfb410108,0x924c0101,0xfb410108,0x924d0101,0xfb410108,0x924e0101,
	0xfb410108,0x924f0101,0xfb410108,0x92500101,0xfb410108,0x92510101,0xfb410108,0x92520101,
	0xfb410108,0x92530101,0xfb410108,0x92540101,0xfb410108,0x92550101,0xfb410108,0x92560101,
	0xfb410108,0x92570101,0xfb410108,0x92580101,0xfb410108,0x92590101,0xfb410108,0x925a0101,
	0xfb410108,0x925b0101,0xfb410108,0x925c0101,0xfb410108,0x925d0101,0xfb410108,0x925e0101,
	0xfb410108,0x925f0101,0xfb410108,0x92600101,0xfb410108,0x92610101,0xfb410108,0x92620101,
	0xfb410108,0x92630101,0xfb410108,0x92640101,0xfb410108,0x92650101,0xfb410108,0x92660101,
	0xfb410108,0x92670101,0xfb410108,0x92680101,0xfb410108,0x92690101,0xfb410108,0x926a0101,
	0xfb410108,0x926b0101,0xfb410108,0x926c0101,0xfb410108,0x926d0101,0xfb410108,0x926e0101,
	0xfb410108,0x926f0101,0xfb410108,0x92700101,0xfb410108,0x92710101,0xfb410108,0x92720101,
	0xfb410108,0x92730101,0xfb410108,0x92750101,0xfb410108,0x92760101,0xfb410108,0x92770101,
	0xfb410108,0x92780101,0xfb410108,0x92790101,0xfb410108,0x927a0101,0xfb410108,0x927b0101,
	0xfb410108,0x927c0101,0xfb410108,0x927d0101,0xfb410108,0x927e0101,0xfb410108,0x927f0101,
	0xfb410108,0x92800101,0xfb410108,0x92810101,0xfb410108,0x92820101,0xfb410108,0x92830101,
	0xfb410108,0x92840101,0xfb410108,0x92850101,0xfb410108,0x92860101,0xfb410108,0x92870101,
	0xfb410108,0x92880101,0xfb410108,0x92890101,0xfb410108,0x928a0101,0xfb410108,0x928b0101,
	0xfb410108,0x928c0101,0xfb410108,0x928d0101,0xfb410108,0x928f0101,0xfb410108,0x92900101,
	0xfb410108,0x92910101,0xfb410108,0x92920101,0xfb410108,0x92930101,0xfb410108,0x92940101,
	0xfb410108,0x92950101,0xfb410108,0x92960101,0xfb410108,0x92970101,0xfb410108,0x92980101,
	0xfb410108,0x92990101,0xfb410108,0x929a0101,0xfb410108,0x929b0101,0xfb410108,0x929c0101,
	0xfb410108,0x929d0101,0xfb410108,0x929e0101,0xfb410108,0x929f0101,0xfb410108,0x92a00101,
	0xfb410108,0x92a10101,0xfb410108,0x92a20101,0xfb410108,0x92a30101,0xfb410108,0x92a40101,
	0xfb410108,0x92a50101,0xfb410108,0x92a60101,0xfb410108,0x92a70101,0xfb410108,0x92a80101,
	0xfb410108,0x92a90101,0xfb410108,0x92aa0101,0xfb410108,0x92ab0101,0xfb410108,0x92ac0101,
	0xfb410108,0x92ad0101,0xfb410108,0x92af0101,0xfb410108,0x92b00101,0xfb410108,0x92b10101,
	0xfb410108,0x92b20101,0xfb410108,0x92b30101,0xfb410108,0x92b40101,0xfb410108,0x92b50101,
	0xfb410108,0x92b60101,0xfb410108,0x92b70101,0xfb410108,0x92b80101,0xfb410108,0x92b90101,
	0xfb410108,0x92ba0101,0xfb410108,0x92bb0101,0xfb410108,0x92bc0101,0xfb410108,0x92bd0101,
	0xfb410108,0x92be0101,0xfb410108,0x92bf0101,0xfb410108,0x92c00101,0xfb410108,0x92c10101,
	0xfb410108,0x92c20101,0xfb410108,0x92c30101,0xfb410108,0x92c40101,0xfb410108,0x92c50101,
	0xfb410108,0x92c60101,0xfb410108,0x92c70101,0xfb410108,0x92c90101,0xfb410108,0x92ca0101,
	0xfb410108,0x92cb0101,0xfb410108,0x92cc0101,0xfb410108,0x92cd0101,0xfb410108,0x92ce0101,
	0xfb410108,0x92cf0101,0xfb410108,0x92d00101,0xfb410108,0x92d10101,0xfb410108,0x92d20101,
	0xfb410108,0x92d30101,0xfb410108,0x92d40101,0xfb410108,0x92d50101,0xfb410108,0x92d60101,
	0xfb410108,0x92d70101,0xfb410108,0x92d80101,0xfb410108,0x92d90101,0xfb410108,0x92da0101,
	0xfb410108,0x92db0101,0xfb410108,0x92dc0101,0xfb410108,0x92dd0101,0xfb410108,0x92de0101,
	0xfb410108,0x92df0101,0xfb410108,0x92e00101,0xfb410108,0x92e10101,0xfb410108,0x92e20101,
	0xfb410108,0x92e30101,0xfb410108,0x92e40101,0xfb410108,0x92e50101,0xfb410108,0x92e60101,
	0xfb410108,0x92e70101,0xfb410108,0x92e80101,0xfb410108,0x92e90101,0xfb410108,0x92ea0101,
	0xfb410108,0x92eb0101,0xfb410108,0x92ec0101,0xfb410108,0x92ed0101,0xfb410108,0x92ee0101,
	0xfb410108,0x92ef0101,0xfb410108,0x92f00101,0xfb410108,0x92f10101,0xfb410108,0x92f20101,
	0xfb410108,0x92f30101,0xfb410108,0x92f40101,0xfb410108,0x92f50101,0xfb410108,0x92f60101,
	0xfb410108,0x92f70101,0xfb410108,0x92f80101,0xfb410108,0x92f90101,0xfb410108,0x92fa0101,
	0xfb410108,0x92fb0101,0xfb410108,0x92fc0101,0xfb410108,0x92fd0101,0xfb410108,0x92fe0101,
	0xfb410108,0x92ff0101,0xfb410108,0x93000101,0xfb410108,0x93010101,0xfb410108,0x93020101,
	0xfb410108,0x93030101,0xfb410108,0x93040101,0xfb410108,0x93050101,0xfb410108,0x93060101,
	0xfb410108,0x93070101,0xfb410108,0x93080101,0xfb410108,0x93090101,0xfb410108,0x930a0101,
	0xfb410108,0x930b0101,0xfb410108,0x930c0101,0xfb410108,0x930d0101,0xfb410108,0x930e0101,
	0xfb410108,0x930f0101,0xfb410108,0x93100101,0xfb410108,0x93110101,0xfb410108,0x93120101,
	0xfb410108,0x93130101,0xfb410108,0x93140101,0xfb410108,0x93150101,0xfb410108,0x93160101,
	0xfb410108,0x93170101,0xfb410108,0x93180101,0xfb410108,0x93190101,0xfb410108,0x931a0101,
	0xfb410108,0x931b0101,0xfb410108,0x931c0101,0xfb410108,0x931d0101,0xfb410108,0x931e0101,
	0xfb410108,0x931f0101,0xfb410108,0x93200101,0xfb410108,0x93210101,0xfb410108,0x93220101,
	0xfb410108,0x93230101,0xfb410108,0x93240101,0xfb410108,0x93250101,0xfb410108,0x93260101,
	0xfb410108,0x93270101,0xfb410108,0x93280101,0xfb410108,0x93290101,0xfb410108,0x932a0101,
	0xfb410108,0x932b0101,0xfb410108,0x932c0101,0xfb410108,0x932d0101,0xfb410108,0x932e0101,
	0xfb410108,0x932f0101,0xfb410108,0x93300101,0xfb410108,0x93310101,0xfb410108,0x93320101,
	0xfb410108,0x93330101,0xfb410108,0x93340101,0xfb410108,0x93350101,0xfb410108,0x93360101,
	0xfb410108,0x93370101,0xfb410108,0x93380101,0xfb410108,0x93390101,0xfb410108,0x933a0101,
	0xfb410108,0x933b0101,0xfb410108,0x933c0101,0xfb410108,0x933d0101,0xfb410108,0x933f0101,
	0xfb410108,0x93400101,0xfb410108,0x93410101,0xfb410108,0x93420101,0xfb410108,0x93430101,
	0xfb410108,0x93440101,0xfb410108,0x93450101,0xfb410108,0x93460101,0xfb410108,0x93470101,
	0xfb410108,0x93480101,0xfb410108,0x93490101,0xfb410108,0x934a0101,0xfb410108,0x934b0101,
	0xfb410108,0x934c0101,0xfb410108,0x934d0101,0xfb410108,0x934e0101,0xfb410108,0x934f0101,
	0xfb410108,0x93500101,0xfb410108,0x93510101,0xfb410108,0x93520101,0xfb410108,0x93530101,
	0xfb410108,0x93540101,0xfb410108,0x93550101,0xfb410108,0x93560101,0xfb410108,0x93570101,
	0xfb410108,0x93580101,0xfb410108,0x93590101,0xfb410108,0x935a0101,0xfb410108,0x935b0101,
	0xfb410108,0x935c0101,0xfb410108,0x935d0101,0xfb410108,0x935e0101,0xfb410108,0x935f0101,
	0xfb410108,0x93600101,0xfb410108,0x93610101,0xfb410108,0x93620101,0xfb410108,0x93630101,
	0xfb410108,0x93640101,0xfb410108,0x93650101,0xfb410108,0x93660101,0xfb410108,0x93670101,
	0xfb410108,0x93680101,0xfb410108,0x93690101,0xfb410108,0x936b0101,0xfb410108,0x936c0101,
	0xfb410108,0x936d0101,0xfb410108,0x936e0101,0xfb410108,0x936f0101,0xfb410108,0x93700101,
	0xfb410108,0x93710101,0xfb410108,0x93720101,0xfb410108,0x93730101,0xfb410108,0x93740101,
	0xfb410108,0x93750101,0xfb410108,0x93760101,0xfb410108,0x93770101,0xfb410108,0x93780101,
	0xfb410108,0x93790101,0xfb410108,0x937a0101,0xfb410108,0x937b0101,0xfb410108,0x937c0101,
	0xfb410108,0x937d0101,0xfb410108,0x937e0101,0xfb410108,0x937f0101,0xfb410108,0x93800101,
	0xfb410108,0x93810101,0xfb410108,0x93820101,0xfb410108,0x93830101,0xfb410108,0x93840101,
	0xfb410108,0x93850101,0xfb410108,0x93860101,0xfb410108,0x93870101,0xfb410108,0x93880101,
	0xfb410108,0x93890101,0xfb410108,0x938a0101,0xfb410108,0x938b0101,0xfb410108,0x938c0101,
	0xfb410108,0x938d0101,0xfb410108,0x938e0101,0xfb410108,0x93900101,0xfb410108,0x93910101,
	0xfb410108,0x93920101,0xfb410108,0x93930101,0xfb410108,0x93940101,0xfb410108,0x93950101,
	0xfb410108,0x93960101,0xfb410108,0x93970101,0xfb410108,0x93980101,0xfb410108,0x93990101,
	0xfb410108,0x939a0101,0xfb410108,0x939b0101,0xfb410108,0x939c0101,0xfb410108,0x939d0101,
	0xfb410108,0x939e0101,0xfb410108,0x939f0101,0xfb410108,0x93a00101,0xfb410108,0x93a10101,
	0xfb410108,0x93a20101,0xfb410108,0x93a30101,0xfb410108,0x93a40101,0xfb410108,0x93a50101,
	0xfb410108,0x93a60101,0xfb410108,0x93a70101,0xfb410108,0x93a80101,0xfb410108,0x93a90101,
	0xfb410108,0x93aa0101,0xfb410108,0x93ab0101,0xfb410108,0x93ac0101,0xfb410108,0x93ad0101,
	0xfb410108,0x93ae0101,0xfb410108,0x93af0101,0xfb410108,0x93b00101,0xfb410108,0x93b10101,
	0xfb410108,0x93b20101,0xfb410108,0x93b30101,0xfb410108,0x93b40101,0xfb410108,0x93b50101,
	0xfb410108,0x93b60101,0xfb410108,0x93b70101,0xfb410108,0x93b80101,0xfb410108,0x93b90101,
	0xfb410108,0x93ba0101,0xfb410108,0x93bb0101,0xfb410108,0x93bc0101,0xfb410108,0x93bd0101,
	0xfb410108,0x93be0101,0xfb410108,0x93bf0101,0xfb410108,0x93c00101,0xfb410108,0x93c10101,
	0xfb410108,0x93c20101,0xfb410108,0x93c30101,0xfb410108,0x93c40101,0xfb410108,0x93c50101,
	0xfb410108,0x93c60101,0xfb410108,0x93c70101,0xfb410108,0x93c80101,0xfb410108,0x93c90101,
	0xfb410108,0x93cb0101,0xfb410108,0x93cc0101,0xfb410108,0x93cd0101,0xfb410108,0x93ce0101,
	0xfb410108,0x93cf0101,0xfb410108,0x93d00101,0xfb410108,0x93d10101,0xfb410108,0x93d20101,
	0xfb410108,0x93d30101,0xfb410108,0x93d40101,0xfb410108,0x93d50101,0xfb410108,0x93d70101,
	0xfb410108,0x93d80101,0xfb410108,0x93d90101,0xfb410108,0x93da0101,0xfb410108,0x93db0101,
	0xfb410108,0x93dc0101,0xfb410108,0x93dd0101,0xfb410108,0x93de0101,0xfb410108,0x93df0101,
	0xfb410108,0x93e00101,0xfb410108,0x93e10101,0xfb410108,0x93e20101,0xfb410108,0x93e30101,
	0xfb410108,0x93e40101,0xfb410108,0x93e50101,0xfb410108,0x93e60101,0xfb410108,0x93e70101,
	0xfb410108,0x93e80101,0xfb410108,0x93e90101,0xfb410108,0x93ea0101,0xfb410108,0x93eb0101,
	0xfb410108,0x93ec0101,0xfb410108,0x93ed0101,0xfb410108,0x93ee0101,0xfb410108,0x93ef0101,
	0xfb410108,0x93f00101,0xfb410108,0x93f10101,0xfb410108,0x93f20101,0xfb410108,0x93f30101,
	0xfb410108,0x93f40101,0xfb410108,0x93f50101,0xfb410108,0x93f60101,0xfb410108,0x93f70101,
	0xfb410108,0x93f80101,0xfb410108,0x93f90101,0xfb410108,0x93fa0101,0xfb410108,0x93fb0101,
	0xfb410108,0x93fc0101,0xfb410108,0x93fd0101,0xfb410108,0x93fe0101,0xfb410108,0x93ff0101,
	0xfb410108,0x94000101,0xfb410108,0x94010101,0xfb410108,0x94020101,0xfb410108,0x94030101,
	0xfb410108,0x94040101,0xfb410108,0x94050101,0xfb410108,0x94060101,0xfb410108,0x94070101,
	0xfb410108,0x94080101,0xfb410108,0x94090101,0xfb410108,0x940a0101,0xfb410108,0x940b0101,
	0xfb410108,0x940c0101,0xfb410108,0x940d0101,0xfb410108,0x940e0101,0xfb410108,0x940f0101,
	0xfb410108,0x94100101,0xfb410108,0x94110101,0xfb410108,0x94120101,0xfb410108,0x94130101,
	0xfb410108,0x94140101,0xfb410108,0x94150101,0xfb410108,0x94160101,0xfb410108,0x94170101,
	0xfb410108,0x94180101,0xfb410108,0x94190101,0xfb410108,0x941a0101,0xfb410108,0x941b0101,
	0xfb410108,0x941c0101,0xfb410108,0x941d0101,0xfb410108,0x941e0101,0xfb410108,0x941f0101,
	0xfb410108,0x94200101,0xfb410108,0x94210101,0xfb410108,0x94220101,0xfb410108,0x94230101,
	0xfb410108,0x94240101,0xfb410108,0x94250101,0xfb410108,0x94260101,0xfb410108,0x94270101,
	0xfb410108,0x94280101,0xfb410108,0x94290101,0xfb410108,0x942a0101,0xfb410108,0x942b0101,
	0xfb410108,0x942c0101,0xfb410108,0x942d0101,0xfb410108,0x942e0101,0xfb410108,0x942f0101,
	0xfb410108,0x94300101,0xfb410108,0x94310101,0xfb410108,0x94320101,0xfb410108,0x94330101,
	0xfb410108,0x94340101,0xfb410108,0x94350101,0xfb410108,0x94360101,0xfb410108,0x94370101,
	0xfb410108,0x94380101,0xfb410108,0x94390101,0xfb410108,0x943a0101,0xfb410108,0x943b0101,
	0xfb410108,0x943c0101,0xfb410108,0x943d0101,0xfb410108,0x943f0101,0xfb410108,0x94400101,
	0xfb410108,0x94410101,0xfb410108,0x94420101,0xfb410108,0x94430101,0xfb410108,0x94440101,
	0xfb410108,0x94450101,0xfb410108,0x94460101,0xfb410108,0x94470101,0xfb410108,0x94480101,
	0xfb410108,0x94490101,0xfb410108,0x944a0101,0xfb410108,0x944b0101,0xfb410108,0x944c0101,
	0xfb410108,0x944d0101,0xfb410108,0x944e0101,0xfb410108,0x944f0101,0xfb410108,0x94500101,
	0xfb410108,0x94510101,0xfb410108,0x94520101,0xfb410108,0x94530101,0xfb410108,0x94540101,
	0xfb410108,0x94550101,0xfb410108,0x94560101,0xfb410108,0x94570101,0xfb410108,0x94580101,
	0xfb410108,0x94590101,0xfb410108,0x945a0101,0xfb410108,0x945b0101,0xfb410108,0x945c0101,
	0xfb410108,0x945d0101,0xfb410108,0x945e0101,0xfb410108,0x945f0101,0xfb410108,0x94600101,
	0xfb410108,0x94610101,0xfb410108,0x94620101,0xfb410108,0x94630101,0xfb410108,0x94640101,
	0xfb410108,0x94650101,0xfb410108,0x94660101,0xfb410108,0x94670101,0xfb410108,0x94680101,
	0xfb410108,0x94690101,0xfb410108,0x946a0101,0xfb410108,0x946c0101,0xfb410108,0x946d0101,
	0xfb410108,0x946e0101,0xfb410108,0x946f0101,0xfb410108,0x94700101,0xfb410108,0x94710101,
	0xfb410108,0x94720101,0xfb410108,0x94730101,0xfb410108,0x94740101,0xfb410108,0x94750101,
	0xfb410108,0x94760101,0xfb410108,0x94770101,0xfb410108,0x94780101,0xfb410108,0x94790101,
	0xfb410108,0x947a0101,0xfb410108,0x947b0101,0xfb410108,0x947c0101,0xfb410108,0x947d0101,
	0xfb410108,0x947e0101,0xfb410108,0x947f0101,0xfb410108,0x94800101,0xfb410108,0x94810101,
	0xfb410108,0x94820101,0xfb410108,0x94830101,0xfb410108,0x94840101,0xfb410108,0x94910101,
	0xfb410108,0x94960101,0xfb410108,0x94980101,0xfb410108,0x94c70101,0xfb410108,0x94cf0101,
	0xfb410108,0x94d30101,0xfb410108,0x94d40101,0xfb410108,0x94da0101,0xfb410108,0x94e60101,
	0xfb410108,0x94fb0101,0xfb410108,0x951c0101,0xfb410108,0x95200101,0xfb410108,0x95270101,
	0xfb410108,0x95330101,0xfb410108,0x953d0101,0xfb410108,0x95430101,0xfb410108,0x95480101,
	0xfb410108,0x954b0101,0xfb410108,0x95550101,0xfb410108,0x955a0101,0xfb410108,0x95600101,
	0xfb410108,0x956e0101,0xfb410108,0x95740101,0xfb410108,0x95750101,0xfb410108,0x95770101,
	0xfb410108,0x95780101,0xfb410108,0x95790101,0xfb410108,0x957a0101,0xfb410108,0x957b0101,
	0xfb410108,0x957c0101,0xfb410108,0x957d0101,0xfb410108,0x957e0101,0xfb410108,0x95800101,
	0xfb410108,0x95810101,0xfb410108,0x95820101,0xfb410108,0x95830101,0xfb410108,0x95840101,
	0xfb410108,0x95850101,0xfb410108,0x95860101,0xfb410108,0x95870101,0xfb410108,0x95880101,
	0xfb410108,0x95890101,0xfb410108,0x958a0101,0xfb410108,0x958b0101,0xfb410108,0x958c0101,
	0xfb410108,0x958d0101,0xfb410108,0x958e0101,0xfb410108,0x958f0101,0xfb410108,0x95900101,
	0xfb410108,0x95910101,0xfb410108,0x95920101,0xfb410108,0x95930101,0xfb410108,0x95940101,
	0xfb410108,0x95950101,0xfb410108,0x95960101,0xfb410108,0x95970101,0xfb410108,0x95980101,
	0xfb410108,0x95990101,0xfb410108,0x959a0101,0xfb410108,0x959b0101,0xfb410108,0x959c0101,
	0xfb410108,0x959d0101,0xfb410108,0x959e0101,0xfb410108,0x959f0101,0xfb410108,0x95a00101,
	0xfb410108,0x95a10101,0xfb410108,0x95a20101,0xfb410108,0x95a30101,0xfb410108,0x95a40101,
	0xfb410108,0x95a50101,0xfb410108,0x95a60101,0xfb410108,0x95a70101,0xfb410108,0x95a80101,
	0xfb410108,0x95a90101,0xfb410108,0x95aa0101,0xfb410108,0x95ab0101,0xfb410108,0x95ac0101,
	0xfb410108,0x95ad0101,0xfb410108,0x95ae0101,0xfb410108,0x95af0101,0xfb410108,0x95b00101,
	0xfb410108,0x95b10101,0xfb410108,0x95b20101,0xfb410108,0x95b30101,0xfb410108,0x95b40101,
	0xfb410108,0x95b50101,0xfb410108,0x95b60101,0xfb410108,0x95b70101,0xfb410108,0x95b80101,
	0xfb410108,0x95b90101,0xfb410108,0x95ba0101,0xfb410108,0x95bb0101,0xfb410108,0x95bc0101,
	0xfb410108,0x95bd0101,0xfb410108,0x95be0101,0xfb410108,0x95bf0101,0xfb410108,0x95c00101,
	0xfb410108,0x95c10101,0xfb410108,0x95c20101,0xfb410108,0x95c30101,0xfb410108,0x95c40101,
	0xfb410108,0x95c50101,0xfb410108,0x95c60101,0xfb410108,0x95c70101,0xfb410108,0x95c80101,
	0xfb410108,0x95c90101,0xfb410108,0x95ca0101,0xfb410108,0x95cb0101,0xfb410108,0x95cc0101,
	0xfb410108,0x95cd0101,0xfb410108,0x95ce0101,0xfb410108,0x95cf0101,0xfb410108,0x95d00101,
	0xfb410108,0x95d10101,0xfb410108,0x95d20101,0xfb410108,0x95d30101,0xfb410108,0x95d40101,
	0xfb410108,0x95d50101,0xfb410108,0x95d60101,0xfb410108,0x95d70101,0xfb410108,0x95d80101,
	0xfb410108,0x95d90101,0xfb410108,0x95da0101,0xfb410108,0x95db0101,0xfb410108,0x95dc0101,
	0xfb410108,0x95dd0101,0xfb410108,0x95de0101,0xfb410108,0x95df0101,0xfb410108,0x95e00101,
	0xfb410108,0x95e10101,0xfb410108,0x95e20101,0xfb410108,0x95e30101,0xfb410108,0x95e40101,
	0xfb410108,0x95e50101,0xfb410108,0x95e60101,0xfb410108,0x95e70101,0xfb410108,0x95ec0101,
	0xfb410108,0x95ff0101,0xfb410108,0x96070101,0xfb410108,0x96130101,0xfb410108,0x96180101,
	0xfb410108,0x961b0101,0xfb410108,0x961e0101,0xfb410108,0x96200101,0xfb410108,0x96230101,
	0xfb410108,0x96240101,0xfb410108,0x96250101,0xfb410108,0x96260101,0xfb410108,0x96270101,
	0xfb410108,0x96280101,0xfb410108,0x96290101,0xfb410108,0x962b0101,0xfb410108,0x962c0101,
	0xfb410108,0x962d0101,0xfb410108,0x962f0101,0xfb410108,0x96300101,0xfb410108,0x96370101,
	0xfb410108,0x96380101,0xfb410108,0x96390101,0xfb410108,0x963a0101,0xfb410108,0x963e0101,
	0xfb410108,0x96410101,0xfb410108,0x96430101,0xfb410108,0x964a0101,0xfb410108,0x964e0101,
	0xfb410108,0x964f0101,0xfb410108,0x96510101,0xfb410108,0x96520101,0xfb410108,0x96530101,
	0xfb410108,0x96560101,0xfb410108,0x96570101,0xfb410108,0x96580101,0xfb410108,0x96590101,
	0xfb410108,0x965a0101,0xfb410108,0x965c0101,0xfb410108,0x965d0101,0xfb410108,0x965e0101,
	0xfb410108,0x96600101,0xfb410108,0x96630101,0xfb410108,0x96650101,0xfb410108,0x96660101,
	0xfb410108,0x966b0101,0xfb410108,0x966d0101,0xfb410108,0x966e0101,0xfb410108,0x966f0101,
	0xfb410108,0x96700101,0xfb410108,0x96710101,0xfb410108,0x96730101,0xfb410108,0x96780101,
	0xfb410108,0x96790101,0xfb410108,0x967a0101,0xfb410108,0x967b0101,0xfb410108,0x967c0101,
	0xfb410108,0x967d0101,0xfb410108,0x967e0101,0xfb410108,0x967f0101,0xfb410108,0x96800101,
	0xfb410108,0x96810101,0xfb410108,0x96820101,0xfb410108,0x96830101,0xfb410108,0x96840101,
	0xfb410108,0x96870101,0xfb410108,0x96890101,0xfb410108,0x968a0101,0xfb410108,0x968c0101,
	0xfb410108,0x968e0101,0xfb410108,0x96910101,0xfb410108,0x96920101,0xfb410108,0x96930101,
	0xfb410108,0x96950101,0xfb410108,0x96960101,0xfb410108,0x969a0101,0xfb410108,0x969b0101,
	0xfb410108,0x969d0101,0xfb410108,0x969e0101,0xfb410108,0x969f0101,0xfb410108,0x96a00101,
	0xfb410108,0x96a10101,0xfb410108,0x96a20101,0xfb410108,0x96a30101,0xfb410108,0x96a40101,
	0xfb410108,0x96a50101,0xfb410108,0x96a60101,0xfb410108,0x96a80101,0xfb410108,0x96a90101,
	0xfb410108,0x96aa0101,0xfb410108,0x96ab0101,0xfb410108,0x96ac0101,0xfb410108,0x96ad0101,
	0xfb410108,0x96ae0101,0xfb410108,0x96af0101,0xfb410108,0x96b10101,0xfb410108,0x96b20101,
	0xfb410108,0x96b40101,0xfb410108,0x96b50101,0xfb410108,0x96b70101,0xfb410108,0x96b80101,
	0xfb410108,0x96ba0101,0xfb410108,0x96bb0101,0xfb410108,0x96bf0101,0xfb410108,0x96c20101,
	0xfb410108,0x96c30101,0xfb410108,0x96c80101,0xfb410108,0x96ca0101,0xfb410108,0x96cb0101,
	0xfb410108,0x96d00101,0xfb410108,0x96d10101,0xfb410108,0x96d30101,0xfb410108,0x96d40101,
	0xfb410108,0x96d60101,0xfb410108,0x96d70101,0xfb410108,0x96d80101,0xfb410108,0x96d90101,
	0xfb410108,0x96da0101,0xfb410108,0x96db0101,0xfb410108,0x96dc0101,0xfb410108,0x96dd0101,
	0xfb410108,0x96de0101,0xfb410108,0x96df0101,0xfb410108,0x96e10101,0xfb410108,0x96e20101,
	0xfb410108,0x96e30101,0xfb410108,0x96e40101,0xfb410108,0x96e50101,0xfb410108,0x96e60101,
	0xfb410108,0x96e70101,0xfb410108,0x96eb0101,0xfb410108,0x96ec0101,0xfb410108,0x96ed0101,
	0xfb410108,0x96ee0101,0xfb410108,0x96f00101,0xfb410108,0x96f10101,0xfb410108,0x96f20101,
	0xfb410108,0x96f40101,0xfb410108,0x96f50101,0xfb410108,0x96f80101,0xfb410108,0x96fa0101,
	0xfb410108,0x96fb0101,0xfb410108,0x96fc0101,0xfb410108,0x96fd0101,0xfb410108,0x96ff0101,
	0xfb410108,0x97020101,0xfb410108,0x97030101,0xfb410108,0x97050101,0xfb410108,0x970a0101,
	0xfb410108,0x970b0101,0xfb410108,0x970c0101,0xfb410108,0x97100101,0xfb410108,0x97110101,
	0xfb410108,0x97120101,0xfb410108,0x97140101,0xfb410108,0x97150101,0xfb410108,0x97170101,
	0xfb410108,0x97180101,0xfb410108,0x97190101,0xfb410108,0x971a0101,0xfb410108,0x971b0101,
	0xfb410108,0x971d0101,0xfb410108,0x971f0101,0xfb410108,0x97200101,0xfb410108,0x97210101,
	0xfb410108,0x97220101,0xfb410108,0x97230101,0xfb410108,0x97240101,0xfb410108,0x97250101,
	0xfb410108,0x97260101,0xfb410108,0x97270101,0xfb410108,0x97280101,0xfb410108,0x97290101,
	0xfb410108,0x972b0101,0xfb410108,0x972c0101,0xfb410108,0x972e0101,0xfb410108,0x972f0101,
	0xfb410108,0x97310101,0xfb410108,0x97330101,0xfb410108,0x97340101,0xfb410108,0x97350101,
	0xfb410108,0x97360101,0xfb410108,0x97370101,0xfb410108,0x973a0101,0xfb410108,0x973b0101,
	0xfb410108,0x973c0101,0xfb410108,0x973d0101,0xfb410108,0x973f0101,0xfb410108,0x97400101,
	0xfb410108,0x97410101,0xfb410108,0x97420101,0xfb410108,0x97430101,0xfb410108,0x97440101,
	0xfb410108,0x97450101,0xfb410108,0x97460101,0xfb410108,0x97470101,0xfb410108,0x97480101,
	0xfb410108,0x97490101,0xfb410108,0x974a0101,0xfb410108,0x974b0101,0xfb410108,0x974c0101,
	0xfb410108,0x974d0101,0xfb410108,0x974e0101,0xfb410108,0x974f0101,0xfb410108,0x97500101,
	0xfb410108,0x97510101,0xfb410108,0x97540101,0xfb410108,0x97550101,0xfb410108,0x97570101,
	0xfb410108,0x97580101,0xfb410108,0x975a0101,0xfb410108,0x975c0101,0xfb410108,0x975d0101,
	0xfb410108,0x975f0101,0xfb410108,0x97630101,0xfb410108,0x97640101,0xfb410108,0x97660101,
	0xfb410108,0x97670101,0xfb410108,0x97680101,0xfb410108,0x976a0101,0xfb410108,0x976b0101,
	0xfb410108,0x976c0101,0xfb410108,0x976d0101,0xfb410108,0x976e0101,0xfb410108,0x976f0101,
	0xfb410108,0x97700101,0xfb410108,0x97710101,0xfb410108,0x97720101,0xfb410108,0x97750101,
	0xfb410108,0x97770101,0xfb410108,0x97780101,0xfb410108,0x97790101,0xfb410108,0x977a0101,
	0xfb410108,0x977b0101,0xfb410108,0x977d0101,0xfb410108,0x977e0101,0xfb410108,0x977f0101,
	0xfb410108,0x97800101,0xfb410108,0x97810101,0xfb410108,0x97820101,0xfb410108,0x97830101,
	0xfb410108,0x97840101,0xfb410108,0x97860101,0xfb410108,0x97870101,0xfb410108,0x97880101,
	0xfb410108,0x97890101,0xfb410108,0x978a0101,0xfb410108,0x978c0101,0xfb410108,0x978e0101,
	0xfb410108,0x978f0101,0xfb410108,0x97900101,0xfb410108,0x97930101,0xfb410108,0x97950101,
	0xfb410108,0x97960101,0xfb410108,0x97970101,0xfb410108,0x97990101,0xfb410108,0x979a0101,
	0xfb410108,0x979b0101,0xfb410108,0x979c0101,0xfb410108,0x979d0101,0xfb410108,0x979e0101,
	0xfb410108,0x979f0101,0xfb410108,0x97a10101,0xfb410108,0x97a20101,0xfb410108,0x97a40101,
	0xfb410108,0x97a50101,0xfb410108,0x97a60101,0xfb410108,0x97a70101,0xfb410108,0x97a80101,
	0xfb410108,0x97a90101,0xfb410108,0x97aa0101,0xfb410108,0x97ac0101,0xfb410108,0x97ae0101,
	0xfb410108,0x97b00101,0xfb410108,0x97b10101,0xfb410108,0x97b30101,0xfb410108,0x97b50101,
	0xfb410108,0x97b60101,0xfb410108,0x97b70101,0xfb410108,0x97b80101,0xfb410108,0x97b90101,
	0xfb410108,0x97ba0101,0xfb410108,0x97bb0101,0xfb410108,0x97bc0101,0xfb410108,0x97bd0101,
	0xfb410108,0x97be0101,0xfb410108,0x97bf0101,0xfb410108,0x97c00101,0xfb410108,0x97c10101,
	0xfb410108,0x97c20101,0xfb410108,0x97c30101,0xfb410108,0x97c40101,0xfb410108,0x97c50101,
	0xfb410108,0x97c60101,0xfb410108,0x97c70101,0xfb410108,0x97c80101,0xfb410108,0x97c90101,
	0xfb410108,0x97ca0101,0xfb410108,0x97cb0101,0xfb410108,0x97cc0101,0xfb410108,0x97cd0101,
	0xfb410108,0x97ce0101,0xfb410108,0x97cf0101,0xfb410108,0x97d00101,0xfb410108,0x97d10101,
	0xfb410108,0x97d20101,0xfb410108,0x97d30101,0xfb410108,0x97d40101,0xfb410108,0x97d50101,
	0xfb410108,0x97d60101,0xfb410108,0x97d70101,0xfb410108,0x97d80101,0xfb410108,0x97d90101,
	0xfb410108,0x97da0101,0xfb410108,0x97db0101,0xfb410108,0x97dc0101,0xfb410108,0x97dd0101,
	0xfb410108,0x97de0101,0xfb410108,0x97df0101,0xfb410108,0x97e00101,0xfb410108,0x97e10101,
	0xfb410108,0x97e20101,0xfb410108,0x97e30101,0xfb410108,0x97e40101,0xfb410108,0x97e50101,
	0xfb410108,0x97e80101,0xfb410108,0x97ee0101,0xfb410108,0x97ef0101,0xfb410108,0x97f00101,
	0xfb410108,0x97f10101,0xfb410108,0x97f20101,0xfb410108,0x97f40101,0xfb410108,0x97f70101,
	0xfb410108,0x97f80101,0xfb410108,0x97f90101,0xfb410108,0x97fa0101,0xfb410108,0x97fb0101,
	0xfb410108,0x97fc0101,0xfb410108,0x97fd0101,0xfb410108,0x97fe0101,0xfb410108,0x97ff0101,
	0xfb410108,0x98000101,0xfb410108,0x98010101,0xfb410108,0x98020101,0xfb410108,0x98030101,
	0xfb410108,0x98040101,0xfb410108,0x98050101,0xfb410108,0x98060101,0xfb410108,0x98070101,
	0xfb410108,0x98080101,0xfb410108,0x98090101,0xfb410108,0x980a0101,0xfb410108,0x980b0101,
	0xfb410108,0x980c0101,0xfb410108,0x980d0101,0xfb410108,0x980e0101,0xfb410108,0x980f0101,
	0xfb410108,0x98100101,0xfb410108,0x98110101,0xfb410108,0x98120101,0xfb410108,0x98130101,
	0xfb410108,0x98140101,0xfb410108,0x98150101,0xfb410108,0x98160101,0xfb410108,0x98170101,
	0xfb410108,0x98180101,0xfb410108,0x98190101,0xfb410108,0x981a0101,0xfb410108,0x981b0101,
	0xfb410108,0x981c0101,0xfb410108,0x981d0101,0xfb410108,0x981e0101,0xfb410108,0x981f0101,
	0xfb410108,0x98200101,0xfb410108,0x98210101,0xfb410108,0x98220101,0xfb410108,0x98230101,
	0xfb410108,0x98240101,0xfb410108,0x98250101,0xfb410108,0x98260101,0xfb410108,0x98270101,
	0xfb410108,0x98280101,0xfb410108,0x98290101,0xfb410108,0x982a0101,0xfb410108,0x982b0101,
	0xfb410108,0x982c0101,0xfb410108,0x982d0101,0xfb410108,0x982e0101,0xfb410108,0x982f0101,
	0xfb410108,0x98300101,0xfb410108,0x98310101,0xfb410108,0x98320101,0xfb410108,0x98330101,
	0xfb410108,0x98340101,0xfb410108,0x98350101,0xfb410108,0x98360101,0xfb410108,0x98370101,
	0xfb410108,0x98380101,0xfb410108,0x98390101,0xfb410108,0x983a0101,0xfb410108,0x983b0101,
	0xfb410108,0x983c0101,0xfb410108,0x983d0101,0xfb410108,0x983e0101,0xfb410108,0x983f0101,
	0xfb410108,0x98400101,0xfb410108,0x98410101,0xfb410108,0x98420101,0xfb410108,0x98430101,
	0xfb410108,0x98440101,0xfb410108,0x98450101,0xfb410108,0x98460101,0xfb410108,0x98470101,
	0xfb410108,0x98480101,0xfb410108,0x98490101,0xfb410108,0x984a0101,0xfb410108,0x984b0101,
	0xfb410108,0x984c0101,0xfb410108,0x984d0101,0xfb410108,0x984e0101,0xfb410108,0x984f0101,
	0xfb410108,0x98500101,0xfb410108,0x98510101,0xfb410108,0x98520101,0xfb410108,0x98530101,
	0xfb410108,0x98540101,0xfb410108,0x98550101,0xfb410108,0x98560101,0xfb410108,0x98570101,
	0xfb410108,0x98580101,0xfb410108,0x98590101,0xfb410108,0x985a0101,0xfb410108,0x985b0101,
	0xfb410108,0x985c0101,0xfb410108,0x985d0101,0xfb410108,0x985e0101,0xfb410108,0x985f0101,
	0xfb410108,0x98600101,0xfb410108,0x98610101,0xfb410108,0x98620101,0xfb410108,0x98630101,
	0xfb410108,0x98640101,0xfb410108,0x98650101,0xfb410108,0x98660101,0xfb410108,0x98670101,
	0xfb410108,0x98680101,0xfb410108,0x98690101,0xfb410108,0x986a0101,0xfb410108,0x986b0101,
	0xfb410108,0x986c0101,0xfb410108,0x986d0101,0xfb410108,0x986e0101,0xfb410108,0x986f0101,
	0xfb410108,0x98700101,0xfb410108,0x98710101,0xfb410108,0x98720101,0xfb410108,0x98730101,
	0xfb410108,0x98740101,0xfb410108,0x988b0101,0xfb410108,0x988e0101,0xfb410108,0x98920101,
	0xfb410108,0x98950101,0xfb410108,0x98990101,0xfb410108,0x98a30101,0xfb410108,0x98a80101,
	0xfb410108,0x98a90101,0xfb410108,0x98aa0101,0xfb410108,0x98ab0101,0xfb410108,0x98ac0101,
	0xfb410108,0x98ad0101,0xfb410108,0x98ae0101,0xfb410108,0x98af0101,0xfb410108,0x98b00101,
	0xfb410108,0x98b10101,0xfb410108,0x98b20101,0xfb410108,0x98b30101,0xfb410108,0x98b40101,
	0xfb410108,0x98b50101,0xfb410108,0x98b60101,0xfb410108,0x98b70101,0xfb410108,0x98b80101,
	0xfb410108,0x98b90101,0xfb410108,0x98ba0101,0xfb410108,0x98bb0101,0xfb410108,0x98bc0101,
	0xfb410108,0x98bd0101,0xfb410108,0x98be0101,0xfb410108,0x98bf0101,0xfb410108,0x98c00101,
	0xfb410108,0x98c10101,0xfb410108,0x98c20101,0xfb410108,0x98c30101,0xfb410108,0x98c40101,
	0xfb410108,0x98c50101,0xfb410108,0x98c60101,0xfb410108,0x98c70101,0xfb410108,0x98c80101,
	0xfb410108,0x98c90101,0xfb410108,0x98ca0101,0xfb410108,0x98cb0101,0xfb410108,0x98cc0101,
	0xfb410108,0x98cd0101,0xfb410108,0x98cf0101,0xfb410108,0x98d00101,0xfb410108,0x98d40101,
	0xfb410108,0x98d60101,0xfb410108,0x98d70101,0xfb410108,0x98db0101,0xfb410108,0x98dc0101,
	0xfb410108,0x98dd0101,0xfb410108,0x98e00101,0xfb410108,0x98e10101,0xfb410108,0x98e20101,
	0xfb410108,0x98e30101,0xfb410108,0x98e40101,0xfb410108,0x98e50101,0xfb410108,0x98e60101,
	0xfb410108,0x98e90101,0xfb410108,0x98ea0101,0xfb410108,0x98eb0101,0xfb410108,0x98ec0101,
	0xfb410108,0x98ed0101,0xfb410108,0x98ee0101,0xfb410108,0x98ef0101,0xfb410108,0x98f00101,
	0xfb410108,0x98f10101,0xfb410108,0x98f20101,0xfb410108,0x98f30101,0xfb410108,0x98f40101,
	0xfb410108,0x98f50101,0xfb410108,0x98f60101,0xfb410108,0x98f70101,0xfb410108,0x98f80101,
	0xfb410108,0x98f90101,0xfb410108,0x98fa0101,0xfb410108,0x98fb0101,0xfb410108,0x98fc0101,
	0xfb410108,0x98fd0101,0xfb410108,0x98fe0101,0xfb410108,0x98ff0101,0xfb410108,0x99000101,
	0xfb410108,0x99010101,0xfb410108,0x99020101,0xfb410108,0x99030101,0xfb410108,0x99040101,
	0xfb410108,0x99050101,0xfb410108,0x99060101,0xfb410108,0x99070101,0xfb410108,0x99080101,
	0xfb410108,0x99090101,0xfb410108,0x990a0101,0xfb410108,0x990b0101,0xfb410108,0x990c0101,
	0xfb410108,0x990e0101,0xfb410108,0x990f0101,0xfb410108,0x99110101,0xfb410108,0x99120101,
	0xfb410108,0x99130101,0xfb410108,0x99140101,0xfb410108,0x99150101,0xfb410108,0x99160101,
	0xfb410108,0x99170101,0xfb410108,0x99180101,0xfb410108,0x99190101,0xfb410108,0x991a0101,
	0xfb410108,0x991b0101,0xfb410108,0x991c0101,0xfb410108,0x991d0101,0xfb410108,0x991e0101,
	0xfb410108,0x991f0101,0xfb410108,0x99200101,0xfb410108,0x99210101,0xfb410108,0x99220101,
	0xfb410108,0x99230101,0xfb410108,0x99240101,0xfb410108,0x99250101,0xfb410108,0x99260101,
	0xfb410108,0x99270101,0xfb410108,0x99280101,0xfb410108,0x99290101,0xfb410108,0x992a0101,
	0xfb410108,0x992b0101,0xfb410108,0x992c0101,0xfb410108,0x992d0101,0xfb410108,0x992f0101,
	0xfb410108,0x99300101,0xfb410108,0x99310101,0xfb410108,0x99320101,0xfb410108,0x99330101,
	0xfb410108,0x99340101,0xfb410108,0x99350101,0xfb410108,0x99360101,0xfb410108,0x99370101,
	0xfb410108,0x99380101,0xfb410108,0x99390101,0xfb410108,0x993a0101,0xfb410108,0x993b0101,
	0xfb410108,0x993c0101,0xfb410108,0x993d0101,0xfb410108,0x993e0101,0xfb410108,0x993f0101,
	0xfb410108,0x99400101,0xfb410108,0x99410101,0xfb410108,0x99420101,0xfb410108,0x99430101,
	0xfb410108,0x99440101,0xfb410108,0x99450101,0xfb410108,0x99460101,0xfb410108,0x99470101,
	0xfb410108,0x99480101,0xfb410108,0x99490101,0xfb410108,0x994a0101,0xfb410108,0x994b0101,
	0xfb410108,0x994c0101,0xfb410108,0x994d0101,0xfb410108,0x994e0101,0xfb410108,0x994f0101,
	0xfb410108,0x99500101,0xfb410108,0x99510101,0xfb410108,0x99520101,0xfb410108,0x99530101,
	0xfb410108,0x99560101,0xfb410108,0x99570101,0xfb410108,0x99580101,0xfb410108,0x99590101,
	0xfb410108,0x995a0101,0xfb410108,0x995b0101,0xfb410108,0x995c0101,0xfb410108,0x995d0101,
	0xfb410108,0x995e0101,0xfb410108,0x995f0101,0xfb410108,0x99600101,0xfb410108,0x99610101,
	0xfb410108,0x99620101,0xfb410108,0x99640101,0xfb410108,0x99660101,0xfb410108,0x99730101,
	0xfb410108,0x99780101,0xfb410108,0x99790101,0xfb410108,0x997b0101,0xfb410108,0x997e0101,
	0xfb410108,0x99820101,0xfb410108,0x99830101,0xfb410108,0x99890101,0xfb410108,0x998c0101,
	0xfb410108,0x998e0101,0xfb410108,0x999a0101,0xfb410108,0x999b0101,0xfb410108,0x999c0101,
	0xfb410108,0x999d0101,0xfb410108,0x999e0101,0xfb410108,0x999f0101,0xfb410108,0x99a00101,
	0xfb410108,0x99a10101,0xfb410108,0x99a20101,0xfb410108,0x99a30101,0xfb410108,0x99a40101,
	0xfb410108,0x99a60101,0xfb410108,0x99a70101,0xfb410108,0x99a90101,0xfb410108,0x99aa0101,
	0xfb410108,0x99ab0101,0xfb410108,0x99ac0101,0xfb410108,0x99ad0101,0xfb410108,0x99ae0101,
	0xfb410108,0x99af0101,0xfb410108,0x99b00101,0xfb410108,0x99b10101,0xfb410108,0x99b20101,
	0xfb410108,0x99b30101,0xfb410108,0x99b40101,0xfb410108,0x99b50101,0xfb410108,0x99b60101,
	0xfb410108,0x99b70101,0xfb410108,0x99b80101,0xfb410108,0x99b90101,0xfb410108,0x99ba0101,
	0xfb410108,0x99bb0101,0xfb410108,0x99bc0101,0xfb410108,0x99bd0101,0xfb410108,0x99be0101,
	0xfb410108,0x99bf0101,0xfb410108,0x99c00101,0xfb410108,0x99c10101,0xfb410108,0x99c20101,
	0xfb410108,0x99c30101,0xfb410108,0x99c40101,0xfb410108,0x99c50101,0xfb410108,0x99c60101,
	0xfb410108,0x99c70101,0xfb410108,0x99c80101,0xfb410108,0x99c90101,0xfb410108,0x99ca0101,
	0xfb410108,0x99cb0101,0xfb410108,0x99cc0101,0xfb410108,0x99cd0101,0xfb410108,0x99ce0101,
	0xfb410108,0x99cf0101,0xfb410108,0x99d00101,0xfb410108,0x99d10101,0xfb410108,0x99d20101,
	0xfb410108,0x99d30101,0xfb410108,0x99d40101,0xfb410108,0x99d50101,0xfb410108,0x99d60101,
	0xfb410108,0x99d70101,0xfb410108,0x99d80101,0xfb410108,0x99d90101,0xfb410108,0x99da0101,
	0xfb410108,0x99db0101,0xfb410108,0x99dc0101,0xfb410108,0x99dd0101,0xfb410108,0x99de0101,
	0xfb410108,0x99df0101,0xfb410108,0x99e00101,0xfb410108,0x99e10101,0xfb410108,0x99e20101,
	0xfb410108,0x99e30101,0xfb410108,0x99e40101,0xfb410108,0x99e50101,0xfb410108,0x99e60101,
	0xfb410108,0x99e70101,0xfb410108,0x99e80101,0xfb410108,0x99e90101,0xfb410108,0x99ea0101,
	0xfb410108,0x99eb0101,0xfb410108,0x99ec0101,0xfb410108,0x99ed0101,0xfb410108,0x99ee0101,
	0xfb410108,0x99ef0101,0xfb410108,0x99f00101,0xfb410108,0x99f10101,0xfb410108,0x99f20101,
	0xfb410108,0x99f30101,0xfb410108,0x99f40101,0xfb410108,0x99f50101,0xfb410108,0x99f60101,
	0xfb410108,0x99f70101,0xfb410108,0x99f80101,0xfb410108,0x99f90101,0xfb410108,0x99fa0101,
	0xfb410108,0x99fb0101,0xfb410108,0x99fc0101,0xfb410108,0x99fd0101,0xfb410108,0x99fe0101,
	0xfb410108,0x99ff0101,0xfb410108,0x9a000101,0xfb410108,0x9a010101,0xfb410108,0x9a020101,
	0xfb410108,0x9a030101,0xfb410108,0x9a040101,0xfb410108,0x9a050101,0xfb410108,0x9a060101,
	0xfb410108,0x9a070101,0xfb410108,0x9a080101,0xfb410108,0x9a090101,0xfb410108,0x9a0a0101,
	0xfb410108,0x9a0b0101,0xfb410108,0x9a0c0101,0xfb410108,0x9a0d0101,0xfb410108,0x9a0e0101,
	0xfb410108,0x9a0f0101,0xfb410108,0x9a100101,0xfb410108,0x9a110101,0xfb410108,0x9a120101,
	0xfb410108,0x9a130101,0xfb410108,0x9a140101,0xfb410108,0x9a150101,0xfb410108,0x9a160101,
	0xfb410108,0x9a170101,0xfb410108,0x9a180101,0xfb410108,0x9a190101,0xfb410108,0x9a1a0101,
	0xfb410108,0x9a1b0101,0xfb410108,0x9a1c0101,0xfb410108,0x9a1d0101,0xfb410108,0x9a1e0101,
	0xfb410108,0x9a1f0101,0xfb410108,0x9a200101,0xfb410108,0x9a210101,0xfb410108,0x9a220101,
	0xfb410108,0x9a230101,0xfb410108,0x9a240101,0xfb410108,0x9a250101,0xfb410108,0x9a260101,
	0xfb410108,0x9a270101,0xfb410108,0x9a280101,0xfb410108,0x9a290101,0xfb410108,0x9a2a0101,
	0xfb410108,0x9a2b0101,0xfb410108,0x9a2c0101,0xfb410108,0x9a2d0101,0xfb410108,0x9a2e0101,
	0xfb410108,0x9a2f0101,0xfb410108,0x9a300101,0xfb410108,0x9a310101,0xfb410108,0x9a320101,
	0xfb410108,0x9a330101,0xfb410108,0x9a340101,0xfb410108,0x9a350101,0xfb410108,0x9a360101,
	0xfb410108,0x9a370101,0xfb410108,0x9a380101,0xfb410108,0x9a390101,0xfb410108,0x9a3a0101,
	0xfb410108,0x9a3b0101,0xfb410108,0x9a3c0101,0xfb410108,0x9a3d0101,0xfb410108,0x9a3e0101,
	0xfb410108,0x9a3f0101,0xfb410108,0x9a400101,0xfb410108,0x9a410101,0xfb410108,0x9a420101,
	0xfb410108,0x9a430101,0xfb410108,0x9a440101,0xfb410108,0x9a450101,0xfb410108,0x9a460101,
	0xfb410108,0x9a470101,0xfb410108,0x9a480101,0xfb410108,0x9a490101,0xfb410108,0x9a4a0101,
	0xfb410108,0x9a4b0101,0xfb410108,0x9a4c0101,0xfb410108,0x9a4d0101,0xfb410108,0x9a4e0101,
	0xfb410108,0x9a4f0101,0xfb410108,0x9a500101,0xfb410108,0x9a510101,0xfb410108,0x9a520101,
	0xfb410108,0x9a530101,0xfb410108,0x9a540101,0xfb410108,0x9a550101,0xfb410108,0x9a560101,
	0xfb410108,0x9a570101,0xfb410108,0x9a580101,0xfb410108,0x9a590101,0xfb410108,0x9a5a0101,
	0xfb410108,0x9a5b0101,0xfb410108,0x9a5c0101,0xfb410108,0x9a5d0101,0xfb410108,0x9a5e0101,
	0xfb410108,0x9a5f0101,0xfb410108,0x9a600101,0xfb410108,0x9a610101,0xfb410108,0x9a620101,
	0xfb410108,0x9a630101,0xfb410108,0x9a640101,0xfb410108,0x9a650101,0xfb410108,0x9a660101,
	0xfb410108,0x9a670101,0xfb410108,0x9a680101,0xfb410108,0x9a690101,0xfb410108,0x9a6a0101,
	0xfb410108,0x9a6b0101,0xfb410108,0x9a720101,0xfb410108,0x9a830101,0xfb410108,0x9a890101,
	0xfb410108,0x9a8d0101,0xfb410108,0x9a8e0101,0xfb410108,0x9a940101,0xfb410108,0x9a950101,
	0xfb410108,0x9a990101,0xfb410108,0x9aa60101,0xfb410108,0x9aa90101,0xfb410108,0x9aaa0101,
	0xfb410108,0x9aab0101,0xfb410108,0x9aac0101,0xfb410108,0x9aad0101,0xfb410108,0x9aae0101,
	0xfb410108,0x9aaf0101,0xfb410108,0x9ab20101,0xfb410108,0x9ab30101,0xfb410108,0x9ab40101,
	0xfb410108,0x9ab50101,0xfb410108,0x9ab90101,0xfb410108,0x9abb0101,0xfb410108,0x9abd0101,
	0xfb410108,0x9abe0101,0xfb410108,0x9abf0101,0xfb410108,0x9ac30101,0xfb410108,0x9ac40101,
	0xfb410108,0x9ac60101,0xfb410108,0x9ac70101,0xfb410108,0x9ac80101,0xfb410108,0x9ac90101,
	0xfb410108,0x9aca0101,0xfb410108,0x9acd0101,0xfb410108,0x9ace0101,0xfb410108,0x9acf0101,
	0xfb410108,0x9ad00101,0xfb410108,0x9ad20101,0xfb410108,0x9ad40101,0xfb410108,0x9ad50101,
	0xfb410108,0x9ad60101,0xfb410108,0x9ad70101,0xfb410108,0x9ad90101,0xfb410108,0x9ada0101,
	0xfb410108,0x9adb0101,0xfb410108,0x9adc0101,0xfb410108,0x9add0101,0xfb410108,0x9ade0101,
	0xfb410108,0x9ae00101,0xfb410108,0x9ae20101,0xfb410108,0x9ae30101,0xfb410108,0x9ae40101,
	0xfb410108,0x9ae50101,0xfb410108,0x9ae70101,0xfb410108,0x9ae80101,0xfb410108,0x9ae90101,
	0xfb410108,0x9aea0101,0xfb410108,0x9aec0101,0xfb410108,0x9aee0101,0xfb410108,0x9af00101,
	0xfb410108,0x9af10101,0xfb410108,0x9af20101,0xfb410108,0x9af30101,0xfb410108,0x9af40101,
	0xfb410108,0x9af50101,0xfb410108,0x9af60101,0xfb410108,0x9af70101,0xfb410108,0x9af80101,
	0xfb410108,0x9afa0101,0xfb410108,0x9afc0101,0xfb410108,0x9afd0101,0xfb410108,0x9afe0101,
	0xfb410108,0x9aff0101,0xfb410108,0x9b000101,0xfb410108,0x9b010101,0xfb410108,0x9b020101,
	0xfb410108,0x9b040101,0xfb410108,0x9b050101,0xfb410108,0x9b060101,0xfb410108,0x9b070101,
	0xfb410108,0x9b090101,0xfb410108,0x9b0a0101,0xfb410108,0x9b0b0101,0xfb410108,0x9b0c0101,
	0xfb410108,0x9b0d0101,0xfb410108,0x9b0e0101,0xfb410108,0x9b100101,0xfb410108,0x9b110101,
	0xfb410108,0x9b120101,0xfb410108,0x9b140101,0xfb410108,0x9b150101,0xfb410108,0x9b160101,
	0xfb410108,0x9b170101,0xfb410108,0x9b180101,0xfb410108,0x9b190101,0xfb410108,0x9b1a0101,
	0xfb410108,0x9b1b0101,0xfb410108,0x9b1c0101,0xfb410108,0x9b1d0101,0xfb410108,0x9b1e0101,
	0xfb410108,0x9b200101,0xfb410108,0x9b210101,0xfb410108,0x9b220101,0xfb410108,0x9b240101,
	0xfb410108,0x9b250101,0xfb410108,0x9b260101,0xfb410108,0x9b270101,0xfb410108,0x9b280101,
	0xfb410108,0x9b290101,0xfb410108,0x9b2a0101,0xfb410108,0x9b2b0101,0xfb410108,0x9b2c0101,
	0xfb410108,0x9b2d0101,0xfb410108,0x9b2e0101,0xfb410108,0x9b300101,0xfb410108,0x9b310101,
	0xfb410108,0x9b330101,0xfb410108,0x9b340101,0xfb410108,0x9b350101,0xfb410108,0x9b360101,
	0xfb410108,0x9b370101,0xfb410108,0x9b380101,0xfb410108,0x9b390101,0xfb410108,0x9b3a0101,
	0xfb410108,0x9b3d0101,0xfb410108,0x9b3e0101,0xfb410108,0x9b3f0101,0xfb410108,0x9b400101,
	0xfb410108,0x9b460101,0xfb410108,0x9b4a0101,0xfb410108,0x9b4b0101,0xfb410108,0x9b4c0101,
	0xfb410108,0x9b4e0101,0xfb410108,0x9b500101,0xfb410108,0x9b520101,0xfb410108,0x9b530101,
	0xfb410108,0x9b550101,0xfb410108,0x9b560101,0xfb410108,0x9b570101,0xfb410108,0x9b580101,
	0xfb410108,0x9b590101,0xfb410108,0x9b5a0101,0xfb410108,0x9b5b0101,0xfb410108,0x9b5c0101,
	0xfb410108,0x9b5d0101,0xfb410108,0x9b5e0101,0xfb410108,0x9b5f0101,0xfb410108,0x9b600101,
	0xfb410108,0x9b610101,0xfb410108,0x9b620101,0xfb410108,0x9b630101,0xfb410108,0x9b640101,
	0xfb410108,0x9b650101,0xfb410108,0x9b660101,0xfb410108,0x9b670101,0xfb410108,0x9b680101,
	0xfb410108,0x9b690101,0xfb410108,0x9b6a0101,0xfb410108,0x9b6b0101,0xfb410108,0x9b6c0101,
	0xfb410108,0x9b6d0101,0xfb410108,0x9b6e0101,0xfb410108,0x9b6f0101,0xfb410108,0x9b700101,
	0xfb410108,0x9b710101,0xfb410108,0x9b720101,0xfb410108,0x9b730101,0xfb410108,0x9b740101,
	0xfb410108,0x9b750101,0xfb410108,0x9b760101,0xfb410108,0x9b770101,0xfb410108,0x9b780101,
	0xfb410108,0x9b790101,0xfb410108,0x9b7a0101,0xfb410108,0x9b7b0101,0xfb410108,0x9b7c0101,
	0xfb410108,0x9b7d0101,0xfb410108,0x9b7e0101,0xfb410108,0x9b7f0101,0xfb410108,0x9b800101,
	0xfb410108,0x9b810101,0xfb410108,0x9b820101,0xfb410108,0x9b830101,0xfb410108,0x9b840101,
	0xfb410108,0x9b850101,0xfb410108,0x9b860101,0xfb410108,0x9b870101,0xfb410108,0x9b880101,
	0xfb410108,0x9b890101,0xfb410108,0x9b8a0101,0xfb410108,0x9b8b0101,0xfb410108,0x9b8c0101,
	0xfb410108,0x9b8d0101,0xfb410108,0x9b8e0101,0xfb410108,0x9b8f0101,0xfb410108,0x9b900101,
	0xfb410108,0x9b910101,0xfb410108,0x9b920101,0xfb410108,0x9b930101,0xfb410108,0x9b940101,
	0xfb410108,0x9b950101,0xfb410108,0x9b960101,0xfb410108,0x9b970101,0xfb410108,0x9b980101,
	0xfb410108,0x9b990101,0xfb410108,0x9b9a0101,0xfb410108,0x9b9b0101,0xfb410108,0x9b9c0101,
	0xfb410108,0x9b9d0101,0xfb410108,0x9b9e0101,0xfb410108,0x9b9f0101,0xfb410108,0x9ba00101,
	0xfb410108,0x9ba10101,0xfb410108,0x9ba20101,0xfb410108,0x9ba30101,0xfb410108,0x9ba40101,
	0xfb410108,0x9ba50101,0xfb410108,0x9ba60101,0xfb410108,0x9ba70101,0xfb410108,0x9ba80101,
	0xfb410108,0x9ba90101,0xfb410108,0x9baa0101,0xfb410108,0x9bab0101,0xfb410108,0x9bac0101,
	0xfb410108,0x9bad0101,0xfb410108,0x9bae0101,0xfb410108,0x9baf0101,0xfb410108,0x9bb00101,
	0xfb410108,0x9bb10101,0xfb410108,0x9bb20101,0xfb410108,0x9bb30101,0xfb410108,0x9bb40101,
	0xfb410108,0x9bb50101,0xfb410108,0x9bb60101,0xfb410108,0x9bb70101,0xfb410108,0x9bb80101,
	0xfb410108,0x9bb90101,0xfb410108,0x9bba0101,0xfb410108,0x9bbb0101,0xfb410108,0x9bbc0101,
	0xfb410108,0x9bbd0101,0xfb410108,0x9bbe0101,0xfb410108,0x9bbf0101,0xfb410108,0x9bc00101,
	0xfb410108,0x9bc10101,0xfb410108,0x9bc20101,0xfb410108,0x9bc30101,0xfb410108,0x9bc40101,
	0xfb410108,0x9bc50101,0xfb410108,0x9bc60101,0xfb410108,0x9bc70101,0xfb410108,0x9bc80101,
	0xfb410108,0x9bc90101,0xfb410108,0x9bca0101,0xfb410108,0x9bcb0101,0xfb410108,0x9bcc0101,
	0xfb410108,0x9bcd0101,0xfb410108,0x9bce0101,0xfb410108,0x9bcf0101,0xfb410108,0x9bd00101,
	0xfb410108,0x9bd10101,0xfb410108,0x9bd20101,0xfb410108,0x9bd30101,0xfb410108,0x9bd40101,
	0xfb410108,0x9bd50101,0xfb410108,0x9bd60101,0xfb410108,0x9bd70101,0xfb410108,0x9bd80101,
	0xfb410108,0x9bd90101,0xfb410108,0x9bda0101,0xfb410108,0x9bdb0101,0xfb410108,0x9bdc0101,
	0xfb410108,0x9bdd0101,0xfb410108,0x9bde0101,0xfb410108,0x9bdf0101,0xfb410108,0x9be00101,
	0xfb410108,0x9be10101,0xfb410108,0x9be20101,0xfb410108,0x9be30101,0xfb410108,0x9be40101,
	0xfb410108,0x9be50101,0xfb410108,0x9be60101,0xfb410108,0x9be70101,0xfb410108,0x9be80101,
	0xfb410108,0x9be90101,0xfb410108,0x9bea0101,0xfb410108,0x9beb0101,0xfb410108,0x9bec0101,
	0xfb410108,0x9bed0101,0xfb410108,0x9bee0101,0xfb410108,0x9bef0101,0xfb410108,0x9bf00101,
	0xfb410108,0x9bf10101,0xfb410108,0x9bf20101,0xfb410108,0x9bf30101,0xfb410108,0x9bf40101,
	0xfb410108,0x9bf50101,0xfb410108,0x9bf60101,0xfb410108,0x9bf70101,0xfb410108,0x9bf80101,
	0xfb410108,0x9bf90101,0xfb410108,0x9bfa0101,0xfb410108,0x9bfb0101,0xfb410108,0x9bfc0101,
	0xfb410108,0x9bfd0101,0xfb410108,0x9bfe0101,0xfb410108,0x9bff0101,0xfb410108,0x9c000101,
	0xfb410108,0x9c010101,0xfb410108,0x9c020101,0xfb410108,0x9c030101,0xfb410108,0x9c040101,
	0xfb410108,0x9c050101,0xfb410108,0x9c060101,0xfb410108,0x9c070101,0xfb410108,0x9c080101,
	0xfb410108,0x9c090101,0xfb410108,0x9c0a0101,0xfb410108,0x9c0b0101,0xfb410108,0x9c0c0101,
	0xfb410108,0x9c0d0101,0xfb410108,0x9c0e0101,0xfb410108,0x9c0f0101,0xfb410108,0x9c100101,
	0xfb410108,0x9c110101,0xfb410108,0x9c120101,0xfb410108,0x9c130101,0xfb410108,0x9c140101,
	0xfb410108,0x9c150101,0xfb410108,0x9c160101,0xfb410108,0x9c170101,0xfb410108,0x9c180101,
	0xfb410108,0x9c190101,0xfb410108,0x9c1a0101,0xfb410108,0x9c1b0101,0xfb410108,0x9c1c0101,
	0xfb410108,0x9c1d0101,0xfb410108,0x9c1e0101,0xfb410108,0x9c1f0101,0xfb410108,0x9c200101,
	0xfb410108,0x9c210101,0xfb410108,0x9c220101,0xfb410108,0x9c230101,0xfb410108,0x9c240101,
	0xfb410108,0x9c250101,0xfb410108,0x9c260101,0xfb410108,0x9c270101,0xfb410108,0x9c280101,
	0xfb410108,0x9c290101,0xfb410108,0x9c2a0101,0xfb410108,0x9c2b0101,0xfb410108,0x9c2c0101,
	0xfb410108,0x9c2d0101,0xfb410108,0x9c2e0101,0xfb410108,0x9c2f0101,0xfb410108,0x9c300101,
	0xfb410108,0x9c310101,0xfb410108,0x9c320101,0xfb410108,0x9c330101,0xfb410108,0x9c340101,
	0xfb410108,0x9c350101,0xfb410108,0x9c360101,0xfb410108,0x9c370101,0xfb410108,0x9c380101,
	0xfb410108,0x9c390101,0xfb410108,0x9c3a0101,0xfb410108,0x9c3b0101,0xfb410108,0x9c3c0101,
	0xfb410108,0x9c3d0101,0xfb410108,0x9c3e0101,0xfb410108,0x9c3f0101,0xfb410108,0x9c400101,
	0xfb410108,0x9c410101,0xfb410108,0x9c420101,0xfb410108,0x9c430101,0xfb410108,0x9c440101,
	0xfb410108,0x9c450101,0xfb410108,0x9c460101,0xfb410108,0x9c470101,0xfb410108,0x9c480101,
	0xfb410108,0x9c490101,0xfb410108,0x9c4a0101,0xfb410108,0x9c4b0101,0xfb410108,0x9c4c0101,
	0xfb410108,0x9c4d0101,0xfb410108,0x9c4e0101,0xfb410108,0x9c4f0101,0xfb410108,0x9c500101,
	0xfb410108,0x9c510101,0xfb410108,0x9c520101,0xfb410108,0x9c530101,0xfb410108,0x9c540101,
	0xfb410108,0x9c550101,0xfb410108,0x9c560101,0xfb410108,0x9c570101,0xfb410108,0x9c580101,
	0xfb410108,0x9c590101,0xfb410108,0x9c5a0101,0xfb410108,0x9c5b0101,0xfb410108,0x9c5c0101,
	0xfb410108,0x9c5d0101,0xfb410108,0x9c5e0101,0xfb410108,0x9c5f0101,0xfb410108,0x9c600101,
	0xfb410108,0x9c610101,0xfb410108,0x9c620101,0xfb410108,0x9c630101,0xfb410108,0x9c640101,
	0xfb410108,0x9c650101,0xfb410108,0x9c660101,0xfb410108,0x9c670101,0xfb410108,0x9c680101,
	0xfb410108,0x9c690101,0xfb410108,0x9c6a0101,0xfb410108,0x9c6b0101,0xfb410108,0x9c6c0101,
	0xfb410108,0x9c6d0101,0xfb410108,0x9c6e0101,0xfb410108,0x9c6f0101,0xfb410108,0x9c700101,
	0xfb410108,0x9c710101,0xfb410108,0x9c720101,0xfb410108,0x9c730101,0xfb410108,0x9c740101,
	0xfb410108,0x9c750101,0xfb410108,0x9c760101,0xfb410108,0x9c770101,0xfb410108,0x9c780101,
	0xfb410108,0x9c790101,0xfb410108,0x9c7a0101,0xfb410108,0x9c7b0101,0xfb410108,0x9c7d0101,
	0xfb410108,0x9c7e0101,0xfb410108,0x9c800101,0xfb410108,0x9c830101,0xfb410108,0x9c840101,
	0xfb410108,0x9c890101,0xfb410108,0x9c8a0101,0xfb410108,0x9c8c0101,0xfb410108,0x9c8f0101,
	0xfb410108,0x9c930101,0xfb410108,0x9c960101,0xfb410108,0x9c970101,0xfb410108,0x9c980101,
	0xfb410108,0x9c990101,0xfb410108,0x9c9d0101,0xfb410108,0x9caa0101,0xfb410108,0x9cac0101,
	0xfb410108,0x9caf0101,0xfb410108,0x9cb90101,0xfb410108,0x9cbe0101,0xfb410108,0x9cbf0101,
	0xfb410108,0x9cc00101,0xfb410108,0x9cc10101,0xfb410108,0x9cc20101,0xfb410108,0x9cc80101,
	0xfb410108,0x9cc90101,0xfb410108,0x9cd10101,0xfb410108,0x9cd20101,0xfb410108,0x9cda0101,
	0xfb410108,0x9cdb0101,0xfb410108,0x9ce00101,0xfb410108,0x9ce10101,0xfb410108,0x9ce30101,
	0xfb410108,0x9ce40101,0xfb410108,0x9ce50101,0xfb410108,0x9ce60101,0xfb410108,0x9ce70101,
	0xfb410108,0x9ce80101,0xfb410108,0x9ce90101,0xfb410108,0x9cea0101,0xfb410108,0x9ceb0101,
	0xfb410108,0x9cec0101,0xfb410108,0x9ced0101,0xfb410108,0x9cee0101,0xfb410108,0x9cef0101,
	0xfb410108,0x9cf00101,0xfb410108,0x9cf10101,0xfb410108,0x9cf20101,0xfb410108,0x9cf30101,
	0xfb410108,0x9cf40101,0xfb410108,0x9cf50101,0xfb410108,0x9cf60101,0xfb410108,0x9cf70101,
	0xfb410108,0x9cf80101,0xfb410108,0x9cf90101,0xfb410108,0x9cfa0101,0xfb410108,0x9cfb0101,
	0xfb410108,0x9cfc0101,0xfb410108,0x9cfd0101,0xfb410108,0x9cfe0101,0xfb410108,0x9cff0101,
	0xfb410108,0x9d000101,0xfb410108,0x9d010101,0xfb410108,0x9d020101,0xfb410108,0x9d030101,
	0xfb410108,0x9d040101,0xfb410108,0x9d050101,0xfb410108,0x9d060101,0xfb410108,0x9d070101,
	0xfb410108,0x9d080101,0xfb410108,0x9d090101,0xfb410108,0x9d0a0101,0xfb410108,0x9d0b0101,
	0xfb410108,0x9d0c0101,0xfb410108,0x9d0d0101,0xfb410108,0x9d0e0101,0xfb410108,0x9d0f0101,
	0xfb410108,0x9d100101,0xfb410108,0x9d110101,0xfb410108,0x9d120101,0xfb410108,0x9d130101,
	0xfb410108,0x9d140101,0xfb410108,0x9d150101,0xfb410108,0x9d160101,0xfb410108,0x9d170101,
	0xfb410108,0x9d180101,0xfb410108,0x9d190101,0xfb410108,0x9d1a0101,0xfb410108,0x9d1b0101,
	0xfb410108,0x9d1c0101,0xfb410108,0x9d1d0101,0xfb410108,0x9d1e0101,0xfb410108,0x9d1f0101,
	0xfb410108,0x9d200101,0xfb410108,0x9d210101,0xfb410108,0x9d220101,0xfb410108,0x9d230101,
	0xfb410108,0x9d240101,0xfb410108,0x9d250101,0xfb410108,0x9d260101,0xfb410108,0x9d270101,
	0xfb410108,0x9d280101,0xfb410108,0x9d290101,0xfb410108,0x9d2a0101,0xfb410108,0x9d2b0101,
	0xfb410108,0x9d2c0101,0xfb410108,0x9d2d0101,0xfb410108,0x9d2e0101,0xfb410108,0x9d2f0101,
	0xfb410108,0x9d300101,0xfb410108,0x9d310101,0xfb410108,0x9d320101,0xfb410108,0x9d330101,
	0xfb410108,0x9d340101,0xfb410108,0x9d350101,0xfb410108,0x9d360101,0xfb410108,0x9d370101,
	0xfb410108,0x9d380101,0xfb410108,0x9d390101,0xfb410108,0x9d3a0101,0xfb410108,0x9d3b0101,
	0xfb410108,0x9d3c0101,0xfb410108,0x9d3d0101,0xfb410108,0x9d3e0101,0xfb410108,0x9d3f0101,
	0xfb410108,0x9d400101,0xfb410108,0x9d410101,0xfb410108,0x9d420101,0xfb410108,0x9d430101,
	0xfb410108,0x9d440101,0xfb410108,0x9d450101,0xfb410108,0x9d460101,0xfb410108,0x9d470101,
	0xfb410108,0x9d480101,0xfb410108,0x9d490101,0xfb410108,0x9d4a0101,0xfb410108,0x9d4b0101,
	0xfb410108,0x9d4c0101,0xfb410108,0x9d4d0101,0xfb410108,0x9d4e0101,0xfb410108,0x9d4f0101,
	0xfb410108,0x9d500101,0xfb410108,0x9d510101,0xfb410108,0x9d520101,0xfb410108,0x9d530101,
	0xfb410108,0x9d540101,0xfb410108,0x9d550101,0xfb410108,0x9d560101,0xfb410108,0x9d570101,
	0xfb410108,0x9d580101,0xfb410108,0x9d590101,0xfb410108,0x9d5a0101,0xfb410108,0x9d5b0101,
	0xfb410108,0x9d5c0101,0xfb410108,0x9d5d0101,0xfb410108,0x9d5e0101,0xfb410108,0x9d5f0101,
	0xfb410108,0x9d600101,0xfb410108,0x9d610101,0xfb410108,0x9d620101,0xfb410108,0x9d630101,
	0xfb410108,0x9d640101,0xfb410108,0x9d650101,0xfb410108,0x9d660101,0xfb410108,0x9d670101,
	0xfb410108,0x9d680101,0xfb410108,0x9d690101,0xfb410108,0x9d6a0101,0xfb410108,0x9d6b0101,
	0xfb410108,0x9d6c0101,0xfb410108,0x9d6d0101,0xfb410108,0x9d6e0101,0xfb410108,0x9d6f0101,
	0xfb410108,0x9d700101,0xfb410108,0x9d710101,0xfb410108,0x9d720101,0xfb410108,0x9d730101,
	0xfb410108,0x9d740101,0xfb410108,0x9d750101,0xfb410108,0x9d760101,0xfb410108,0x9d770101,
	0xfb410108,0x9d780101,0xfb410108,0x9d790101,0xfb410108,0x9d7a0101,0xfb410108,0x9d7b0101,
	0xfb410108,0x9d7c0101,0xfb410108,0x9d7d0101,0xfb410108,0x9d7e0101,0xfb410108,0x9d7f0101,
	0xfb410108,0x9d800101,0xfb410108,0x9d810101,0xfb410108,0x9d820101,0xfb410108,0x9d830101,
	0xfb410108,0x9d840101,0xfb410108,0x9d850101,0xfb410108,0x9d860101,0xfb410108,0x9d870101,
	0xfb410108,0x9d880101,0xfb410108,0x9d890101,0xfb410108,0x9d8a0101,0xfb410108,0x9d8b0101,
	0xfb410108,0x9d8c0101,0xfb410108,0x9d8d0101,0xfb410108,0x9d8e0101,0xfb410108,0x9d8f0101,
	0xfb410108,0x9d900101,0xfb410108,0x9d910101,0xfb410108,0x9d920101,0xfb410108,0x9d930101,
	0xfb410108,0x9d940101,0xfb410108,0x9d950101,0xfb410108,0x9d960101,0xfb410108,0x9d970101,
	0xfb410108,0x9d980101,0xfb410108,0x9d990101,0xfb410108,0x9d9a0101,0xfb410108,0x9d9b0101,
	0xfb410108,0x9d9c0101,0xfb410108,0x9d9d0101,0xfb410108,0x9d9e0101,0xfb410108,0x9d9f0101,
	0xfb410108,0x9da00101,0xfb410108,0x9da10101,0xfb410108,0x9da20101,0xfb410108,0x9da30101,
	0xfb410108,0x9da40101,0xfb410108,0x9da50101,0xfb410108,0x9da60101,0xfb410108,0x9da70101,
	0xfb410108,0x9da80101,0xfb410108,0x9da90101,0xfb410108,0x9daa0101,0xfb410108,0x9dab0101,
	0xfb410108,0x9dac0101,0xfb410108,0x9dad0101,0xfb410108,0x9dae0101,0xfb410108,0x9daf0101,
	0xfb410108,0x9db00101,0xfb410108,0x9db10101,0xfb410108,0x9db20101,0xfb410108,0x9db30101,
	0xfb410108,0x9db40101,0xfb410108,0x9db50101,0xfb410108,0x9db60101,0xfb410108,0x9db70101,
	0xfb410108,0x9db80101,0xfb410108,0x9db90101,0xfb410108,0x9dba0101,0xfb410108,0x9dbb0101,
	0xfb410108,0x9dbc0101,0xfb410108,0x9dbd0101,0xfb410108,0x9dbe0101,0xfb410108,0x9dbf0101,
	0xfb410108,0x9dc00101,0xfb410108,0x9dc10101,0xfb410108,0x9dc20101,0xfb410108,0x9dc30101,
	0xfb410108,0x9dc40101,0xfb410108,0x9dc50101,0xfb410108,0x9dc60101,0xfb410108,0x9dc70101,
	0xfb410108,0x9dc80101,0xfb410108,0x9dc90101,0xfb410108,0x9dca0101,0xfb410108,0x9dcb0101,
	0xfb410108,0x9dcc0101,0xfb410108,0x9dcd0101,0xfb410108,0x9dce0101,0xfb410108,0x9dcf0101,
	0xfb410108,0x9dd00101,0xfb410108,0x9dd10101,0xfb410108,0x9dd20101,0xfb410108,0x9dd30101,
	0xfb410108,0x9dd40101,0xfb410108,0x9dd50101,0xfb410108,0x9dd60101,0xfb410108,0x9dd70101,
	0xfb410108,0x9dd80101,0xfb410108,0x9dd90101,0xfb410108,0x9dda0101,0xfb410108,0x9ddb0101,
	0xfb410108,0x9ddc0101,0xfb410108,0x9ddd0101,0xfb410108,0x9dde0101,0xfb410108,0x9ddf0101,
	0xfb410108,0x9de00101,0xfb410108,0x9de10101,0xfb410108,0x9de20101,0xfb410108,0x9de30101,
	0xfb410108,0x9de40101,0xfb410108,0x9de50101,0xfb410108,0x9de60101,0xfb410108,0x9de70101,
	0xfb410108,0x9de80101,0xfb410108,0x9de90101,0xfb410108,0x9dea0101,0xfb410108,0x9deb0101,
	0xfb410108,0x9dec0101,0xfb410108,0x9ded0101,0xfb410108,0x9dee0101,0xfb410108,0x9def0101,
	0xfb410108,0x9df00101,0xfb410108,0x9df10101,0xfb410108,0x9df20101,0xfb410108,0x9df30101,
	0xfb410108,0x9df40101,0xfb410108,0x9df50101,0xfb410108,0x9df60101,0xfb410108,0x9df70101,
	0xfb410108,0x9df80101,0xfb410108,0x9df90101,0xfb410108,0x9dfa0101,0xfb410108,0x9dfb0101,
	0xfb410108,0x9dfc0101,0xfb410108,0x9dfd0101,0xfb410108,0x9dfe0101,0xfb410108,0x9dff0101,
	0xfb410108,0x9e000101,0xfb410108,0x9e010101,0xfb410108,0x9e020101,0xfb410108,0x9e030101,
	0xfb410108,0x9e040101,0xfb410108,0x9e050101,0xfb410108,0x9e060101,0xfb410108,0x9e070101,
	0xfb410108,0x9e080101,0xfb410108,0x9e090101,0xfb410108,0x9e0a0101,0xfb410108,0x9e0b0101,
	0xfb410108,0x9e0c0101,0xfb410108,0x9e0d0101,0xfb410108,0x9e0e0101,0xfb410108,0x9e0f0101,
	0xfb410108,0x9e100101,0xfb410108,0x9e110101,0xfb410108,0x9e120101,0xfb410108,0x9e130101,
	0xfb410108,0x9e140101,0xfb410108,0x9e150101,0xfb410108,0x9e160101,0xfb410108,0x9e170101,
	0xfb410108,0x9e180101,0xfb410108,0x9e190101,0xfb410108,0x9e1a0101,0xfb410108,0x9e1b0101,
	0xfb410108,0x9e1c0101,0xfb410108,0x9e1d0101,0xfb410108,0x9e1e0101,0xfb410108,0x9e240101,
	0xfb410108,0x9e270101,0xfb410108,0x9e2e0101,0xfb410108,0x9e300101,0xfb410108,0x9e340101,
	0xfb410108,0x9e3b0101,0xfb410108,0x9e3c0101,0xfb410108,0x9e400101,0xfb410108,0x9e4d0101,
	0xfb410108,0x9e500101,0xfb410108,0x9e520101,0xfb410108,0x9e530101,0xfb410108,0x9e540101,
	0xfb410108,0x9e560101,0xfb410108,0x9e590101,0xfb410108,0x9e5d0101,0xfb410108,0x9e5f0101,
	0xfb410108,0x9e600101,0xfb410108,0x9e610101,0xfb410108,0x9e620101,0xfb410108,0x9e650101,
	0xfb410108,0x9e6e0101,0xfb410108,0x9e6f0101,0xfb410108,0x9e720101,0xfb410108,0x9e740101,
	0xfb410108,0x9e750101,0xfb410108,0x9e760101,0xfb410108,0x9e770101,0xfb410108,0x9e780101,
	0xfb410108,0x9e790101,0xfb410108,0x9e7a0101,0xfb410108,0x9e7b0101,0xfb410108,0x9e7c0101,
	0xfb410108,0x9e7d0101,0xfb410108,0x9e800101,0xfb410108,0x9e810101,0xfb410108,0x9e830101,
	0xfb410108,0x9e840101,0xfb410108,0x9e850101,0xfb410108,0x9e860101,0xfb410108,0x9e890101,
	0xfb410108,0x9e8a0101,0xfb410108,0x9e8c0101,0xfb410108,0x9e8d0101,0xfb410108,0x9e8e0101,
	0xfb410108,0x9e8f0101,0xfb410108,0x9e900101,0xfb410108,0x9e910101,0xfb410108,0x9e940101,
	0xfb410108,0x9e950101,0xfb410108,0x9e960101,0xfb410108,0x9e970101,0xfb410108,0x9e980101,
	0xfb410108,0x9e990101,0xfb410108,0x9e9a0101,0xfb410108,0x9e9b0101,0xfb410108,0x9e9c0101,
	0xfb410108,0x9e9e0101,0xfb410108,0x9ea00101,0xfb410108,0x9ea10101,0xfb410108,0x9ea20101,
	0xfb410108,0x9ea30101,0xfb410108,0x9ea40101,0xfb410108,0x9ea50101,0xfb410108,0x9ea70101,
	0xfb410108,0x9ea80101,0xfb410108,0x9ea90101,0xfb410108,0x9eaa0101,0xfb410108,0x9eab0101,
	0xfb410108,0x9eac0101,0xfb410108,0x9ead0101,0xfb410108,0x9eae0101,0xfb410108,0x9eaf0101,
	0xfb410108,0x9eb00101,0xfb410108,0x9eb10101,0xfb410108,0x9eb20101,0xfb410108,0x9eb30101,
	0xfb410108,0x9eb50101,0xfb410108,0x9eb60101,0xfb410108,0x9eb70101,0xfb410108,0x9eb90101,
	0xfb410108,0x9eba0101,0xfb410108,0x9ebc0101,0xfb410108,0x9ebf0101,0xfb410108,0x9ec00101,
	0xfb410108,0x9ec10101,0xfb410108,0x9ec20101,0xfb410108,0x9ec30101,0xfb410108,0x9ec50101,
	0xfb410108,0x9ec60101,0xfb410108,0x9ec70101,0xfb410108,0x9ec80101,0xfb410108,0x9eca0101,
	0xfb410108,0x9ecb0101,0xfb410108,0x9ecc0101,0xfb410108,0x9ed00101,0xfb410108,0x9ed20101,
	0xfb410108,0x9ed30101,0xfb410108,0x9ed50101,0xfb410108,0x9ed60101,0xfb410108,0x9ed70101,
	0xfb410108,0x9ed90101,0xfb410108,0x9eda0101,0xfb410108,0x9ede0101,0xfb410108,0x9ee10101,
	0xfb410108,0x9ee30101,0xfb410108,0x9ee40101,0xfb410108,0x9ee60101,0xfb410108,0x9ee80101,
	0xfb410108,0x9eeb0101,0xfb410108,0x9eec0101,0xfb410108,0x9eed0101,0xfb410108,0x9eee0101,
	0xfb410108,0x9ef00101,0xfb410108,0x9ef10101,0xfb410108,0x9ef20101,0xfb410108,0x9ef30101,
	0xfb410108,0x9ef40101,0xfb410108,0x9ef50101,0xfb410108,0x9ef60101,0xfb410108,0x9ef70101,
	0xfb410108,0x9ef80101,0xfb410108,0x9efa0101,0xfb410108,0x9efd0101,0xfb410108,0x9eff0101,
	0xfb410108,0x9f000101,0xfb410108,0x9f010101,0xfb410108,0x9f020101,0xfb410108,0x9f030101,
	0xfb410108,0x9f040101,0xfb410108,0x9f050101,0xfb410108,0x9f060101,0xfb410108,0x9f070101,
	0xfb410108,0x9f080101,0xfb410108,0x9f090101,0xfb410108,0x9f0a0101,0xfb410108,0x9f0c0101,
	0xfb410108,0x9f0f0101,0xfb410108,0x9f110101,0xfb410108,0x9f120101,0xfb410108,0x9f140101,
	0xfb410108,0x9f150101,0xfb410108,0x9f160101,0xfb410108,0x9f180101,0xfb410108,0x9f1a0101,
	0xfb410108,0x9f1b0101,0xfb410108,0x9f1c0101,0xfb410108,0x9f1d0101,0xfb410108,0x9f1e0101,
	0xfb410108,0x9f1f0101,0xfb410108,0x9f210101,0xfb410108,0x9f230101,0xfb410108,0x9f240101,
	0xfb410108,0x9f250101,0xfb410108,0x9f260101,0xfb410108,0x9f270101,0xfb410108,0x9f280101,
	0xfb410108,0x9f290101,0xfb410108,0x9f2a0101,0xfb410108,0x9f2b0101,0xfb410108,0x9f2d0101,
	0xfb410108,0x9f2e0101,0xfb410108,0x9f300101,0xfb410108,0x9f310101,0xfb410108,0x9f320101,
	0xfb410108,0x9f330101,0xfb410108,0x9f340101,0xfb410108,0x9f350101,0xfb410108,0x9f360101,
	0xfb410108,0x9f380101,0xfb410108,0x9f3a0101,0xfb410108,0x9f3c0101,0xfb410108,0x9f3f0101,
	0xfb410108,0x9f400101,0xfb410108,0x9f410101,0xfb410108,0x9f420101,0xfb410108,0x9f430101,
	0xfb410108,0x9f450101,0xfb410108,0x9f460101,0xfb410108,0x9f470101,0xfb410108,0x9f480101,
	0xfb410108,0x9f490101,0xfb410108,0x9f4a0101,0xfb410108,0x9f4b0101,0xfb410108,0x9f4c0101,
	0xfb410108,0x9f4d0101,0xfb410108,0x9f4e0101,0xfb410108,0x9f4f0101,0xfb410108,0x9f520101,
	0xfb410108,0x9f530101,0xfb410108,0x9f540101,0xfb410108,0x9f550101,0xfb410108,0x9f560101,
	0xfb410108,0x9f570101,0xfb410108,0x9f580101,0xfb410108,0x9f590101,0xfb410108,0x9f5a0101,
	0xfb410108,0x9f5b0101,0xfb410108,0x9f5c0101,0xfb410108,0x9f5d0101,0xfb410108,0x9f5e0101,
	0xfb410108,0x9f5f0101,0xfb410108,0x9f600101,0xfb410108,0x9f610101,0xfb410108,0x9f620101,
	0xfb410108,0x9f630101,0xfb410108,0x9f640101,0xfb410108,0x9f650101,0xfb410108,0x9f660101,
	0xfb410108,0x9f670101,0xfb410108,0x9f680101,0xfb410108,0x9f690101,0xfb410108,0x9f6a0101,
	0xfb410108,0x9f6b0101,0xfb410108,0x9f6c0101,0xfb410108,0x9f6d0101,0xfb410108,0x9f6e0101,
	0xfb410108,0x9f6f0101,0xfb410108,0x9f700101,0xfb410108,0x9f710101,0xfb410108,0x9f720101,
	0xfb410108,0x9f730101,0xfb410108,0x9f740101,0xfb410108,0x9f750101,0xfb410108,0x9f760101,
	0xfb410108,0x9f770101,0xfb410108,0x9f780101,0xfb410108,0x9f790101,0xfb410108,0x9f7a0101,
	0xfb410108,0x9f7b0101,0xfb410108,0x9f7c0101,0xfb410108,0x9f7d0101,0xfb410108,0x9f7e0101,
	0xfb410108,0x9f810101,0xfb410108,0x9f820101,0xfb410108,0x9f8d0101,0xfb410108,0x9f8e0101,
	0xfb410108,0x9f8f0101,0xfb410108,0x9f900101,0xfb410108,0x9f910101,0xfb410108,0x9f920101,
	0xfb410108,0x9f930101,0xfb410108,0x9f940101,0xfb410108,0x9f950101,0xfb410108,0x9f960101,
	0xfb410108,0x9f970101,0xfb410108,0x9f980101,0xfb410108,0x9f9c0101,0xfb410108,0x9f9d0101,
	0xfb410108,0x9f9e0101,0xfb410108,0x9fa10101,0xfb410108,0x9fa20101,0xfb410108,0x9fa30101,
	0xfb410108,0x9fa40101,0xfb410108,0x9fa50101,0xfb410108,0x9fa60101,0xfb410108,0x9fa70101,
	0xfb410108,0x9fa80101,0xfb410108,0x9fa90101,0xfb410108,0x9faa0101,0xfb410108,0x9fab0101,
	0xfb410108,0x9fac0101,0xfb410108,0x9fad0101,0xfb410108,0x9fae0101,0xfb410108,0x9faf0101,
	0xfb410108,0x9fb00101,0xfb410108,0x9fb10101,0xfb410108,0x9fb20101,0xfb410108,0x9fb30101,
	0xfb410108,0x9fb40101,0xfb410108,0x9fb50101,0xfb410108,0x9fb60101,0xfb410108,0x9fb70101,
	0xfb410108,0x9fb80101,0xfb410108,0x9fb90101,0xfb410108,0x9fba0101,0xfb410108,0x9fbb0101,
	0xfb800108,0xb4000101,0xfb800108,0xb4010101,0xfb800108,0xb4020101,0xfb800108,0xb4030101,
	0xfb800108,0xb4040101,0xfb800108,0xb4050101,0xfb800108,0xb4060101,0xfb800108,0xb4070101,
	0xfb800108,0xb4080101,0xfb800108,0xb4090101,0xfb800108,0xb40a0101,0xfb800108,0xb40b0101,
	0xfb800108,0xb40c0101,0xfb800108,0xb40d0101,0xfb800108,0xb40e0101,0xfb800108,0xb40f0101,
	0xfb800108,0xb4100101,0xfb800108,0xb4110101,0xfb800108,0xb4120101,0xfb800108,0xb4130101,
	0xfb800108,0xb4140101,0xfb800108,0xb4150101,0xfb800108,0xb4160101,0xfb800108,0xb4170101,
	0xfb800108,0xb4180101,0xfb800108,0xb4190101,0xfb800108,0xb41a0101,0xfb800108,0xb41b0101,
	0xfb800108,0xb41c0101,0xfb800108,0xb41d0101,0xfb800108,0xb41e0101,0xfb800108,0xb41f0101,
	0xfb800108,0xb4200101,0xfb800108,0xb4210101,0xfb800108,0xb4220101,0xfb800108,0xb4230101,
	0xfb800108,0xb4240101,0xfb800108,0xb4250101,0xfb800108,0xb4260101,0xfb800108,0xb4270101,
	0xfb800108,0xb4280101,0xfb800108,0xb4290101,0xfb800108,0xb42a0101,0xfb800108,0xb42b0101,
	0xfb800108,0xb42c0101,0xfb800108,0xb42d0101,0xfb800108,0xb42e0101,0xfb800108,0xb42f0101,
	0xfb800108,0xb4300101,0xfb800108,0xb4310101,0xfb800108,0xb4320101,0xfb800108,0xb4330101,
	0xfb800108,0xb4340101,0xfb800108,0xb4350101,0xfb800108,0xb4360101,0xfb800108,0xb4370101,
	0xfb800108,0xb4380101,0xfb800108,0xb4390101,0xfb800108,0xb43a0101,0xfb800108,0xb43b0101,
	0xfb800108,0xb43c0101,0xfb800108,0xb43d0101,0xfb800108,0xb43e0101,0xfb800108,0xb43f0101,
	0xfb800108,0xb4400101,0xfb800108,0xb4410101,0xfb800108,0xb4420101,0xfb800108,0xb4430101,
	0xfb800108,0xb4440101,0xfb800108,0xb4450101,0xfb800108,0xb4460101,0xfb800108,0xb4470101,
	0xfb800108,0xb4480101,0xfb800108,0xb4490101,0xfb800108,0xb44a0101,0xfb800108,0xb44b0101,
	0xfb800108,0xb44c0101,0xfb800108,0xb44d0101,0xfb800108,0xb44e0101,0xfb800108,0xb44f0101,
	0xfb800108,0xb4500101,0xfb800108,0xb4510101,0xfb800108,0xb4520101,0xfb800108,0xb4530101,
	0xfb800108,0xb4540101,0xfb800108,0xb4550101,0xfb800108,0xb4560101,0xfb800108,0xb4570101,
	0xfb800108,0xb4580101,0xfb800108,0xb4590101,0xfb800108,0xb45a0101,0xfb800108,0xb45b0101,
	0xfb800108,0xb45c0101,0xfb800108,0xb45d0101,0xfb800108,0xb45e0101,0xfb800108,0xb45f0101,
	0xfb800108,0xb4600101,0xfb800108,0xb4610101,0xfb800108,0xb4620101,0xfb800108,0xb4630101,
	0xfb800108,0xb4640101,0xfb800108,0xb4650101,0xfb800108,0xb4660101,0xfb800108,0xb4670101,
	0xfb800108,0xb4680101,0xfb800108,0xb4690101,0xfb800108,0xb46a0101,0xfb800108,0xb46b0101,
	0xfb800108,0xb46c0101,0xfb800108,0xb46d0101,0xfb800108,0xb46e0101,0xfb800108,0xb46f0101,
	0xfb800108,0xb4700101,0xfb800108,0xb4710101,0xfb800108,0xb4720101,0xfb800108,0xb4730101,
	0xfb800108,0xb4740101,0xfb800108,0xb4750101,0xfb800108,0xb4760101,0xfb800108,0xb4770101,
	0xfb800108,0xb4780101,0xfb800108,0xb4790101,0xfb800108,0xb47a0101,0xfb800108,0xb47b0101,
	0xfb800108,0xb47c0101,0xfb800108,0xb47d0101,0xfb800108,0xb47e0101,0xfb800108,0xb47f0101,
	0xfb800108,0xb4800101,0xfb800108,0xb4810101,0xfb800108,0xb4820101,0xfb800108,0xb4830101,
	0xfb800108,0xb4840101,0xfb800108,0xb4850101,0xfb800108,0xb4860101,0xfb800108,0xb4870101,
	0xfb800108,0xb4880101,0xfb800108,0xb4890101,0xfb800108,0xb48a0101,0xfb800108,0xb48b0101,
	0xfb800108,0xb48c0101,0xfb800108,0xb48d0101,0xfb800108,0xb48e0101,0xfb800108,0xb48f0101,
	0xfb800108,0xb4900101,0xfb800108,0xb4910101,0xfb800108,0xb4920101,0xfb800108,0xb4930101,
	0xfb800108,0xb4940101,0xfb800108,0xb4950101,0xfb800108,0xb4960101,0xfb800108,0xb4970101,
	0xfb800108,0xb4980101,0xfb800108,0xb4990101,0xfb800108,0xb49a0101,0xfb800108,0xb49b0101,
	0xfb800108,0xb49c0101,0xfb800108,0xb49d0101,0xfb800108,0xb49e0101,0xfb800108,0xb49f0101,
	0xfb800108,0xb4a00101,0xfb800108,0xb4a10101,0xfb800108,0xb4a20101,0xfb800108,0xb4a30101,
	0xfb800108,0xb4a40101,0xfb800108,0xb4a50101,0xfb800108,0xb4a60101,0xfb800108,0xb4a70101,
	0xfb800108,0xb4a80101,0xfb800108,0xb4a90101,0xfb800108,0xb4aa0101,0xfb800108,0xb4ab0101,
	0xfb800108,0xb4ac0101,0xfb800108,0xb4ad0101,0xfb800108,0xb4ae0101,0xfb800108,0xb4af0101,
	0xfb800108,0xb4b00101,0xfb800108,0xb4b10101,0xfb800108,0xb4b20101,0xfb800108,0xb4b30101,
	0xfb800108,0xb4b40101,0xfb800108,0xb4b50101,0xfb800108,0xb4b60101,0xfb800108,0xb4b70101,
	0xfb800108,0xb4b80101,0xfb800108,0xb4b90101,0xfb800108,0xb4ba0101,0xfb800108,0xb4bb0101,
	0xfb800108,0xb4bc0101,0xfb800108,0xb4bd0101,0xfb800108,0xb4be0101,0xfb800108,0xb4bf0101,
	0xfb800108,0xb4c00101,0xfb800108,0xb4c10101,0xfb800108,0xb4c20101,0xfb800108,0xb4c30101,
	0xfb800108,0xb4c40101,0xfb800108,0xb4c50101,0xfb800108,0xb4c60101,0xfb800108,0xb4c70101,
	0xfb800108,0xb4c80101,0xfb800108,0xb4c90101,0xfb800108,0xb4ca0101,0xfb800108,0xb4cb0101,
	0xfb800108,0xb4cc0101,0xfb800108,0xb4cd0101,0xfb800108,0xb4ce0101,0xfb800108,0xb4cf0101,
	0xfb800108,0xb4d00101,0xfb800108,0xb4d10101,0xfb800108,0xb4d20101,0xfb800108,0xb4d30101,
	0xfb800108,0xb4d40101,0xfb800108,0xb4d50101,0xfb800108,0xb4d60101,0xfb800108,0xb4d70101,
	0xfb800108,0xb4d80101,0xfb800108,0xb4d90101,0xfb800108,0xb4da0101,0xfb800108,0xb4db0101,
	0xfb800108,0xb4dc0101,0xfb800108,0xb4dd0101,0xfb800108,0xb4de0101,0xfb800108,0xb4df0101,
	0xfb800108,0xb4e00101,0xfb800108,0xb4e10101,0xfb800108,0xb4e20101,0xfb800108,0xb4e30101,
	0xfb800108,0xb4e40101,0xfb800108,0xb4e50101,0xfb800108,0xb4e60101,0xfb800108,0xb4e70101,
	0xfb800108,0xb4e80101,0xfb800108,0xb4e90101,0xfb800108,0xb4ea0101,0xfb800108,0xb4eb0101,
	0xfb800108,0xb4ec0101,0xfb800108,0xb4ed0101,0xfb800108,0xb4ee0101,0xfb800108,0xb4ef0101,
	0xfb800108,0xb4f00101,0xfb800108,0xb4f10101,0xfb800108,0xb4f20101,0xfb800108,0xb4f30101,
	0xfb800108,0xb4f40101,0xfb800108,0xb4f50101,0xfb800108,0xb4f60101,0xfb800108,0xb4f70101,
	0xfb800108,0xb4f80101,0xfb800108,0xb4f90101,0xfb800108,0xb4fa0101,0xfb800108,0xb4fb0101,
	0xfb800108,0xb4fc0101,0xfb800108,0xb4fd0101,0xfb800108,0xb4fe0101,0xfb800108,0xb4ff0101,
	0xfb800108,0xb5000101,0xfb800108,0xb5010101,0xfb800108,0xb5020101,0xfb800108,0xb5030101,
	0xfb800108,0xb5040101,0xfb800108,0xb5050101,0xfb800108,0xb5060101,0xfb800108,0xb5070101,
	0xfb800108,0xb5080101,0xfb800108,0xb5090101,0xfb800108,0xb50a0101,0xfb800108,0xb50b0101,
	0xfb800108,0xb50c0101,0xfb800108,0xb50d0101,0xfb800108,0xb50e0101,0xfb800108,0xb50f0101,
	0xfb800108,0xb5100101,0xfb800108,0xb5110101,0xfb800108,0xb5120101,0xfb800108,0xb5130101,
	0xfb800108,0xb5140101,0xfb800108,0xb5150101,0xfb800108,0xb5160101,0xfb800108,0xb5170101,
	0xfb800108,0xb5180101,0xfb800108,0xb5190101,0xfb800108,0xb51a0101,0xfb800108,0xb51b0101,
	0xfb800108,0xb51c0101,0xfb800108,0xb51d0101,0xfb800108,0xb51e0101,0xfb800108,0xb51f0101,
	0xfb800108,0xb5200101,0xfb800108,0xb5210101,0xfb800108,0xb5220101,0xfb800108,0xb5230101,
	0xfb800108,0xb5240101,0xfb800108,0xb5250101,0xfb800108,0xb5260101,0xfb800108,0xb5270101,
	0xfb800108,0xb5280101,0xfb800108,0xb5290101,0xfb800108,0xb52a0101,0xfb800108,0xb52b0101,
	0xfb800108,0xb52c0101,0xfb800108,0xb52d0101,0xfb800108,0xb52e0101,0xfb800108,0xb52f0101,
	0xfb800108,0xb5300101,0xfb800108,0xb5310101,0xfb800108,0xb5320101,0xfb800108,0xb5330101,
	0xfb800108,0xb5340101,0xfb800108,0xb5350101,0xfb800108,0xb5360101,0xfb800108,0xb5370101,
	0xfb800108,0xb5380101,0xfb800108,0xb5390101,0xfb800108,0xb53a0101,0xfb800108,0xb53b0101,
	0xfb800108,0xb53c0101,0xfb800108,0xb53d0101,0xfb800108,0xb53e0101,0xfb800108,0xb53f0101,
	0xfb800108,0xb5400101,0xfb800108,0xb5410101,0xfb800108,0xb5420101,0xfb800108,0xb5430101,
	0xfb800108,0xb5440101,0xfb800108,0xb5450101,0xfb800108,0xb5460101,0xfb800108,0xb5470101,
	0xfb800108,0xb5480101,0xfb800108,0xb5490101,0xfb800108,0xb54a0101,0xfb800108,0xb54b0101,
	0xfb800108,0xb54c0101,0xfb800108,0xb54d0101,0xfb800108,0xb54e0101,0xfb800108,0xb54f0101,
	0xfb800108,0xb5500101,0xfb800108,0xb5510101,0xfb800108,0xb5520101,0xfb800108,0xb5530101,
	0xfb800108,0xb5540101,0xfb800108,0xb5550101,0xfb800108,0xb5560101,0xfb800108,0xb5570101,
	0xfb800108,0xb5580101,0xfb800108,0xb5590101,0xfb800108,0xb55a0101,0xfb800108,0xb55b0101,
	0xfb800108,0xb55c0101,0xfb800108,0xb55d0101,0xfb800108,0xb55e0101,0xfb800108,0xb55f0101,
	0xfb800108,0xb5600101,0xfb800108,0xb5610101,0xfb800108,0xb5620101,0xfb800108,0xb5630101,
	0xfb800108,0xb5640101,0xfb800108,0xb5650101,0xfb800108,0xb5660101,0xfb800108,0xb5670101,
	0xfb800108,0xb5680101,0xfb800108,0xb5690101,0xfb800108,0xb56a0101,0xfb800108,0xb56b0101,
	0xfb800108,0xb56c0101,0xfb800108,0xb56d0101,0xfb800108,0xb56e0101,0xfb800108,0xb56f0101,
	0xfb800108,0xb5700101,0xfb800108,0xb5710101,0xfb800108,0xb5720101,0xfb800108,0xb5730101,
	0xfb800108,0xb5740101,0xfb800108,0xb5750101,0xfb800108,0xb5760101,0xfb800108,0xb5770101,
	0xfb800108,0xb5780101,0xfb800108,0xb5790101,0xfb800108,0xb57a0101,0xfb800108,0xb57b0101,
	0xfb800108,0xb57c0101,0xfb800108,0xb57d0101,0xfb800108,0xb57e0101,0xfb800108,0xb57f0101,
	0xfb800108,0xb5800101,0xfb800108,0xb5810101,0xfb800108,0xb5820101,0xfb800108,0xb5830101,
	0xfb800108,0xb5840101,0xfb800108,0xb5850101,0xfb800108,0xb5860101,0xfb800108,0xb5870101,
	0xfb800108,0xb5880101,0xfb800108,0xb5890101,0xfb800108,0xb58a0101,0xfb800108,0xb58b0101,
	0xfb800108,0xb58c0101,0xfb800108,0xb58d0101,0xfb800108,0xb58e0101,0xfb800108,0xb58f0101,
	0xfb800108,0xb5900101,0xfb800108,0xb5910101,0xfb800108,0xb5920101,0xfb800108,0xb5930101,
	0xfb800108,0xb5940101,0xfb800108,0xb5950101,0xfb800108,0xb5960101,0xfb800108,0xb5970101,
	0xfb800108,0xb5980101,0xfb800108,0xb5990101,0xfb800108,0xb59a0101,0xfb800108,0xb59b0101,
	0xfb800108,0xb59c0101,0xfb800108,0xb59d0101,0xfb800108,0xb59e0101,0xfb800108,0xb59f0101,
	0xfb800108,0xb5a00101,0xfb800108,0xb5a10101,0xfb800108,0xb5a20101,0xfb800108,0xb5a30101,
	0xfb800108,0xb5a40101,0xfb800108,0xb5a50101,0xfb800108,0xb5a60101,0xfb800108,0xb5a70101,
	0xfb800108,0xb5a80101,0xfb800108,0xb5a90101,0xfb800108,0xb5aa0101,0xfb800108,0xb5ab0101,
	0xfb800108,0xb5ac0101,0xfb800108,0xb5ad0101,0xfb800108,0xb5ae0101,0xfb800108,0xb5af0101,
	0xfb800108,0xb5b00101,0xfb800108,0xb5b10101,0xfb800108,0xb5b20101,0xfb800108,0xb5b30101,
	0xfb800108,0xb5b40101,0xfb800108,0xb5b50101,0xfb800108,0xb5b60101,0xfb800108,0xb5b70101,
	0xfb800108,0xb5b80101,0xfb800108,0xb5b90101,0xfb800108,0xb5ba0101,0xfb800108,0xb5bb0101,
	0xfb800108,0xb5bc0101,0xfb800108,0xb5bd0101,0xfb800108,0xb5be0101,0xfb800108,0xb5bf0101,
	0xfb800108,0xb5c00101,0xfb800108,0xb5c10101,0xfb800108,0xb5c20101,0xfb800108,0xb5c30101,
	0xfb800108,0xb5c40101,0xfb800108,0xb5c50101,0xfb800108,0xb5c60101,0xfb800108,0xb5c70101,
	0xfb800108,0xb5c80101,0xfb800108,0xb5c90101,0xfb800108,0xb5ca0101,0xfb800108,0xb5cb0101,
	0xfb800108,0xb5cc0101,0xfb800108,0xb5cd0101,0xfb800108,0xb5ce0101,0xfb800108,0xb5cf0101,
	0xfb800108,0xb5d00101,0xfb800108,0xb5d10101,0xfb800108,0xb5d20101,0xfb800108,0xb5d30101,
	0xfb800108,0xb5d40101,0xfb800108,0xb5d50101,0xfb800108,0xb5d60101,0xfb800108,0xb5d70101,
	0xfb800108,0xb5d80101,0xfb800108,0xb5d90101,0xfb800108,0xb5da0101,0xfb800108,0xb5db0101,
	0xfb800108,0xb5dc0101,0xfb800108,0xb5dd0101,0xfb800108,0xb5de0101,0xfb800108,0xb5df0101,
	0xfb800108,0xb5e00101,0xfb800108,0xb5e10101,0xfb800108,0xb5e20101,0xfb800108,0xb5e30101,
	0xfb800108,0xb5e40101,0xfb800108,0xb5e50101,0xfb800108,0xb5e60101,0xfb800108,0xb5e70101,
	0xfb800108,0xb5e80101,0xfb800108,0xb5e90101,0xfb800108,0xb5ea0101,0xfb800108,0xb5eb0101,
	0xfb800108,0xb5ec0101,0xfb800108,0xb5ed0101,0xfb800108,0xb5ee0101,0xfb800108,0xb5ef0101,
	0xfb800108,0xb5f00101,0xfb800108,0xb5f10101,0xfb800108,0xb5f20101,0xfb800108,0xb5f30101,
	0xfb800108,0xb5f40101,0xfb800108,0xb5f50101,0xfb800108,0xb5f60101,0xfb800108,0xb5f70101,
	0xfb800108,0xb5f80101,0xfb800108,0xb5f90101,0xfb800108,0xb5fa0101,0xfb800108,0xb5fb0101,
	0xfb800108,0xb5fc0101,0xfb800108,0xb5fd0101,0xfb800108,0xb5fe0101,0xfb800108,0xb5ff0101,
	0xfb800108,0xb6000101,0xfb800108,0xb6010101,0xfb800108,0xb6020101,0xfb800108,0xb6030101,
	0xfb800108,0xb6040101,0xfb800108,0xb6050101,0xfb800108,0xb6060101,0xfb800108,0xb6070101,
	0xfb800108,0xb6080101,0xfb800108,0xb6090101,0xfb800108,0xb60a0101,0xfb800108,0xb60b0101,
	0xfb800108,0xb60c0101,0xfb800108,0xb60d0101,0xfb800108,0xb60e0101,0xfb800108,0xb60f0101,
	0xfb800108,0xb6100101,0xfb800108,0xb6110101,0xfb800108,0xb6120101,0xfb800108,0xb6130101,
	0xfb800108,0xb6140101,0xfb800108,0xb6150101,0xfb800108,0xb6160101,0xfb800108,0xb6170101,
	0xfb800108,0xb6180101,0xfb800108,0xb6190101,0xfb800108,0xb61a0101,0xfb800108,0xb61b0101,
	0xfb800108,0xb61c0101,0xfb800108,0xb61d0101,0xfb800108,0xb61e0101,0xfb800108,0xb61f0101,
	0xfb800108,0xb6200101,0xfb800108,0xb6210101,0xfb800108,0xb6220101,0xfb800108,0xb6230101,
	0xfb800108,0xb6240101,0xfb800108,0xb6250101,0xfb800108,0xb6260101,0xfb800108,0xb6270101,
	0xfb800108,0xb6280101,0xfb800108,0xb6290101,0xfb800108,0xb62a0101,0xfb800108,0xb62b0101,
	0xfb800108,0xb62c0101,0xfb800108,0xb62d0101,0xfb800108,0xb62e0101,0xfb800108,0xb62f0101,
	0xfb800108,0xb6300101,0xfb800108,0xb6310101,0xfb800108,0xb6320101,0xfb800108,0xb6330101,
	0xfb800108,0xb6340101,0xfb800108,0xb6350101,0xfb800108,0xb6360101,0xfb800108,0xb6370101,
	0xfb800108,0xb6380101,0xfb800108,0xb6390101,0xfb800108,0xb63a0101,0xfb800108,0xb63b0101,
	0xfb800108,0xb63c0101,0xfb800108,0xb63d0101,0xfb800108,0xb63e0101,0xfb800108,0xb63f0101,
	0xfb800108,0xb6400101,0xfb800108,0xb6410101,0xfb800108,0xb6420101,0xfb800108,0xb6430101,
	0xfb800108,0xb6440101,0xfb800108,0xb6450101,0xfb800108,0xb6460101,0xfb800108,0xb6470101,
	0xfb800108,0xb6480101,0xfb800108,0xb6490101,0xfb800108,0xb64a0101,0xfb800108,0xb64b0101,
	0xfb800108,0xb64c0101,0xfb800108,0xb64d0101,0xfb800108,0xb64e0101,0xfb800108,0xb64f0101,
	0xfb800108,0xb6500101,0xfb800108,0xb6510101,0xfb800108,0xb6520101,0xfb800108,0xb6530101,
	0xfb800108,0xb6540101,0xfb800108,0xb6550101,0xfb800108,0xb6560101,0xfb800108,0xb6570101,
	0xfb800108,0xb6580101,0xfb800108,0xb6590101,0xfb800108,0xb65a0101,0xfb800108,0xb65b0101,
	0xfb800108,0xb65c0101,0xfb800108,0xb65d0101,0xfb800108,0xb65e0101,0xfb800108,0xb65f0101,
	0xfb800108,0xb6600101,0xfb800108,0xb6610101,0xfb800108,0xb6620101,0xfb800108,0xb6630101,
	0xfb800108,0xb6640101,0xfb800108,0xb6650101,0xfb800108,0xb6660101,0xfb800108,0xb6670101,
	0xfb800108,0xb6680101,0xfb800108,0xb6690101,0xfb800108,0xb66a0101,0xfb800108,0xb66b0101,
	0xfb800108,0xb66c0101,0xfb800108,0xb66d0101,0xfb800108,0xb66e0101,0xfb800108,0xb66f0101,
	0xfb800108,0xb6700101,0xfb800108,0xb6710101,0xfb800108,0xb6720101,0xfb800108,0xb6730101,
	0xfb800108,0xb6740101,0xfb800108,0xb6750101,0xfb800108,0xb6760101,0xfb800108,0xb6770101,
	0xfb800108,0xb6780101,0xfb800108,0xb6790101,0xfb800108,0xb67a0101,0xfb800108,0xb67b0101,
	0xfb800108,0xb67c0101,0xfb800108,0xb67d0101,0xfb800108,0xb67e0101,0xfb800108,0xb67f0101,
	0xfb800108,0xb6800101,0xfb800108,0xb6810101,0xfb800108,0xb6820101,0xfb800108,0xb6830101,
	0xfb800108,0xb6840101,0xfb800108,0xb6850101,0xfb800108,0xb6860101,0xfb800108,0xb6870101,
	0xfb800108,0xb6880101,0xfb800108,0xb6890101,0xfb800108,0xb68a0101,0xfb800108,0xb68b0101,
	0xfb800108,0xb68c0101,0xfb800108,0xb68d0101,0xfb800108,0xb68e0101,0xfb800108,0xb68f0101,
	0xfb800108,0xb6900101,0xfb800108,0xb6910101,0xfb800108,0xb6920101,0xfb800108,0xb6930101,
	0xfb800108,0xb6940101,0xfb800108,0xb6950101,0xfb800108,0xb6960101,0xfb800108,0xb6970101,
	0xfb800108,0xb6980101,0xfb800108,0xb6990101,0xfb800108,0xb69a0101,0xfb800108,0xb69b0101,
	0xfb800108,0xb69c0101,0xfb800108,0xb69d0101,0xfb800108,0xb69e0101,0xfb800108,0xb69f0101,
	0xfb800108,0xb6a00101,0xfb800108,0xb6a10101,0xfb800108,0xb6a20101,0xfb800108,0xb6a30101,
	0xfb800108,0xb6a40101,0xfb800108,0xb6a50101,0xfb800108,0xb6a60101,0xfb800108,0xb6a70101,
	0xfb800108,0xb6a80101,0xfb800108,0xb6a90101,0xfb800108,0xb6aa0101,0xfb800108,0xb6ab0101,
	0xfb800108,0xb6ac0101,0xfb800108,0xb6ad0101,0xfb800108,0xb6ae0101,0xfb800108,0xb6af0101,
	0xfb800108,0xb6b00101,0xfb800108,0xb6b10101,0xfb800108,0xb6b20101,0xfb800108,0xb6b30101,
	0xfb800108,0xb6b40101,0xfb800108,0xb6b50101,0xfb800108,0xb6b60101,0xfb800108,0xb6b70101,
	0xfb800108,0xb6b80101,0xfb800108,0xb6b90101,0xfb800108,0xb6ba0101,0xfb800108,0xb6bb0101,
	0xfb800108,0xb6bc0101,0xfb800108,0xb6bd0101,0xfb800108,0xb6be0101,0xfb800108,0xb6bf0101,
	0xfb800108,0xb6c00101,0xfb800108,0xb6c10101,0xfb800108,0xb6c20101,0xfb800108,0xb6c30101,
	0xfb800108,0xb6c40101,0xfb800108,0xb6c50101,0xfb800108,0xb6c60101,0xfb800108,0xb6c70101,
	0xfb800108,0xb6c80101,0xfb800108,0xb6c90101,0xfb800108,0xb6ca0101,0xfb800108,0xb6cb0101,
	0xfb800108,0xb6cc0101,0xfb800108,0xb6cd0101,0xfb800108,0xb6ce0101,0xfb800108,0xb6cf0101,
	0xfb800108,0xb6d00101,0xfb800108,0xb6d10101,0xfb800108,0xb6d20101,0xfb800108,0xb6d30101,
	0xfb800108,0xb6d40101,0xfb800108,0xb6d50101,0xfb800108,0xb6d60101,0xfb800108,0xb6d70101,
	0xfb800108,0xb6d80101,0xfb800108,0xb6d90101,0xfb800108,0xb6da0101,0xfb800108,0xb6db0101,
	0xfb800108,0xb6dc0101,0xfb800108,0xb6dd0101,0xfb800108,0xb6de0101,0xfb800108,0xb6df0101,
	0xfb800108,0xb6e00101,0xfb800108,0xb6e10101,0xfb800108,0xb6e20101,0xfb800108,0xb6e30101,
	0xfb800108,0xb6e40101,0xfb800108,0xb6e50101,0xfb800108,0xb6e60101,0xfb800108,0xb6e70101,
	0xfb800108,0xb6e80101,0xfb800108,0xb6e90101,0xfb800108,0xb6ea0101,0xfb800108,0xb6eb0101,
	0xfb800108,0xb6ec0101,0xfb800108,0xb6ed0101,0xfb800108,0xb6ee0101,0xfb800108,0xb6ef0101,
	0xfb800108,0xb6f00101,0xfb800108,0xb6f10101,0xfb800108,0xb6f20101,0xfb800108,0xb6f30101,
	0xfb800108,0xb6f40101,0xfb800108,0xb6f50101,0xfb800108,0xb6f60101,0xfb800108,0xb6f70101,
	0xfb800108,0xb6f80101,0xfb800108,0xb6f90101,0xfb800108,0xb6fa0101,0xfb800108,0xb6fb0101,
	0xfb800108,0xb6fc0101,0xfb800108,0xb6fd0101,0xfb800108,0xb6fe0101,0xfb800108,0xb6ff0101,
	0xfb800108,0xb7000101,0xfb800108,0xb7010101,0xfb800108,0xb7020101,0xfb800108,0xb7030101,
	0xfb800108,0xb7040101,0xfb800108,0xb7050101,0xfb800108,0xb7060101,0xfb800108,0xb7070101,
	0xfb800108,0xb7080101,0xfb800108,0xb7090101,0xfb800108,0xb70a0101,0xfb800108,0xb70b0101,
	0xfb800108,0xb70c0101,0xfb800108,0xb70d0101,0xfb800108,0xb70e0101,0xfb800108,0xb70f0101,
	0xfb800108,0xb7100101,0xfb800108,0xb7110101,0xfb800108,0xb7120101,0xfb800108,0xb7130101,
	0xfb800108,0xb7140101,0xfb800108,0xb7150101,0xfb800108,0xb7160101,0xfb800108,0xb7170101,
	0xfb800108,0xb7180101,0xfb800108,0xb7190101,0xfb800108,0xb71a0101,0xfb800108,0xb71b0101,
	0xfb800108,0xb71c0101,0xfb800108,0xb71d0101,0xfb800108,0xb71e0101,0xfb800108,0xb71f0101,
	0xfb800108,0xb7200101,0xfb800108,0xb7210101,0xfb800108,0xb7220101,0xfb800108,0xb7230101,
	0xfb800108,0xb7240101,0xfb800108,0xb7250101,0xfb800108,0xb7260101,0xfb800108,0xb7270101,
	0xfb800108,0xb7280101,0xfb800108,0xb7290101,0xfb800108,0xb72a0101,0xfb800108,0xb72b0101,
	0xfb800108,0xb72c0101,0xfb800108,0xb72d0101,0xfb800108,0xb72e0101,0xfb800108,0xb72f0101,
	0xfb800108,0xb7300101,0xfb800108,0xb7310101,0xfb800108,0xb7320101,0xfb800108,0xb7330101,
	0xfb800108,0xb7340101,0xfb800108,0xb7350101,0xfb800108,0xb7360101,0xfb800108,0xb7370101,
	0xfb800108,0xb7380101,0xfb800108,0xb7390101,0xfb800108,0xb73a0101,0xfb800108,0xb73b0101,
	0xfb800108,0xb73c0101,0xfb800108,0xb73d0101,0xfb800108,0xb73e0101,0xfb800108,0xb73f0101,
	0xfb800108,0xb7400101,0xfb800108,0xb7410101,0xfb800108,0xb7420101,0xfb800108,0xb7430101,
	0xfb800108,0xb7440101,0xfb800108,0xb7450101,0xfb800108,0xb7460101,0xfb800108,0xb7470101,
	0xfb800108,0xb7480101,0xfb800108,0xb7490101,0xfb800108,0xb74a0101,0xfb800108,0xb74b0101,
	0xfb800108,0xb74c0101,0xfb800108,0xb74d0101,0xfb800108,0xb74e0101,0xfb800108,0xb74f0101,
	0xfb800108,0xb7500101,0xfb800108,0xb7510101,0xfb800108,0xb7520101,0xfb800108,0xb7530101,
	0xfb800108,0xb7540101,0xfb800108,0xb7550101,0xfb800108,0xb7560101,0xfb800108,0xb7570101,
	0xfb800108,0xb7580101,0xfb800108,0xb7590101,0xfb800108,0xb75a0101,0xfb800108,0xb75b0101,
	0xfb800108,0xb75c0101,0xfb800108,0xb75d0101,0xfb800108,0xb75e0101,0xfb800108,0xb75f0101,
	0xfb800108,0xb7600101,0xfb800108,0xb7610101,0xfb800108,0xb7620101,0xfb800108,0xb7630101,
	0xfb800108,0xb7640101,0xfb800108,0xb7650101,0xfb800108,0xb7660101,0xfb800108,0xb7670101,
	0xfb800108,0xb7680101,0xfb800108,0xb7690101,0xfb800108,0xb76a0101,0xfb800108,0xb76b0101,
	0xfb800108,0xb76c0101,0xfb800108,0xb76d0101,0xfb800108,0xb76e0101,0xfb800108,0xb76f0101,
	0xfb800108,0xb7700101,0xfb800108,0xb7710101,0xfb800108,0xb7720101,0xfb800108,0xb7730101,
	0xfb800108,0xb7740101,0xfb800108,0xb7750101,0xfb800108,0xb7760101,0xfb800108,0xb7770101,
	0xfb800108,0xb7780101,0xfb800108,0xb7790101,0xfb800108,0xb77a0101,0xfb800108,0xb77b0101,
	0xfb800108,0xb77c0101,0xfb800108,0xb77d0101,0xfb800108,0xb77e0101,0xfb800108,0xb77f0101,
	0xfb800108,0xb7800101,0xfb800108,0xb7810101,0xfb800108,0xb7820101,0xfb800108,0xb7830101,
	0xfb800108,0xb7840101,0xfb800108,0xb7850101,0xfb800108,0xb7860101,0xfb800108,0xb7870101,
	0xfb800108,0xb7880101,0xfb800108,0xb7890101,0xfb800108,0xb78a0101,0xfb800108,0xb78b0101,
	0xfb800108,0xb78c0101,0xfb800108,0xb78d0101,0xfb800108,0xb78e0101,0xfb800108,0xb78f0101,
	0xfb800108,0xb7900101,0xfb800108,0xb7910101,0xfb800108,0xb7920101,0xfb800108,0xb7930101,
	0xfb800108,0xb7940101,0xfb800108,0xb7950101,0xfb800108,0xb7960101,0xfb800108,0xb7970101,
	0xfb800108,0xb7980101,0xfb800108,0xb7990101,0xfb800108,0xb79a0101,0xfb800108,0xb79b0101,
	0xfb800108,0xb79c0101,0xfb800108,0xb79d0101,0xfb800108,0xb79e0101,0xfb800108,0xb79f0101,
	0xfb800108,0xb7a00101,0xfb800108,0xb7a10101,0xfb800108,0xb7a20101,0xfb800108,0xb7a30101,
	0xfb800108,0xb7a40101,0xfb800108,0xb7a50101,0xfb800108,0xb7a60101,0xfb800108,0xb7a70101,
	0xfb800108,0xb7a80101,0xfb800108,0xb7a90101,0xfb800108,0xb7aa0101,0xfb800108,0xb7ab0101,
	0xfb800108,0xb7ac0101,0xfb800108,0xb7ad0101,0xfb800108,0xb7ae0101,0xfb800108,0xb7af0101,
	0xfb800108,0xb7b00101,0xfb800108,0xb7b10101,0xfb800108,0xb7b20101,0xfb800108,0xb7b30101,
	0xfb800108,0xb7b40101,0xfb800108,0xb7b50101,0xfb800108,0xb7b60101,0xfb800108,0xb7b70101,
	0xfb800108,0xb7b80101,0xfb800108,0xb7b90101,0xfb800108,0xb7ba0101,0xfb800108,0xb7bb0101,
	0xfb800108,0xb7bc0101,0xfb800108,0xb7bd0101,0xfb800108,0xb7be0101,0xfb800108,0xb7bf0101,
	0xfb800108,0xb7c00101,0xfb800108,0xb7c10101,0xfb800108,0xb7c20101,0xfb800108,0xb7c30101,
	0xfb800108,0xb7c40101,0xfb800108,0xb7c50101,0xfb800108,0xb7c60101,0xfb800108,0xb7c70101,
	0xfb800108,0xb7c80101,0xfb800108,0xb7c90101,0xfb800108,0xb7ca0101,0xfb800108,0xb7cb0101,
	0xfb800108,0xb7cc0101,0xfb800108,0xb7cd0101,0xfb800108,0xb7ce0101,0xfb800108,0xb7cf0101,
	0xfb800108,0xb7d00101,0xfb800108,0xb7d10101,0xfb800108,0xb7d20101,0xfb800108,0xb7d30101,
	0xfb800108,0xb7d40101,0xfb800108,0xb7d50101,0xfb800108,0xb7d60101,0xfb800108,0xb7d70101,
	0xfb800108,0xb7d80101,0xfb800108,0xb7d90101,0xfb800108,0xb7da0101,0xfb800108,0xb7db0101,
	0xfb800108,0xb7dc0101,0xfb800108,0xb7dd0101,0xfb800108,0xb7de0101,0xfb800108,0xb7df0101,
	0xfb800108,0xb7e00101,0xfb800108,0xb7e10101,0xfb800108,0xb7e20101,0xfb800108,0xb7e30101,
	0xfb800108,0xb7e40101,0xfb800108,0xb7e50101,0xfb800108,0xb7e60101,0xfb800108,0xb7e70101,
	0xfb800108,0xb7e80101,0xfb800108,0xb7e90101,0xfb800108,0xb7ea0101,0xfb800108,0xb7eb0101,
	0xfb800108,0xb7ec0101,0xfb800108,0xb7ed0101,0xfb800108,0xb7ee0101,0xfb800108,0xb7ef0101,
	0xfb800108,0xb7f00101,0xfb800108,0xb7f10101,0xfb800108,0xb7f20101,0xfb800108,0xb7f30101,
	0xfb800108,0xb7f40101,0xfb800108,0xb7f50101,0xfb800108,0xb7f60101,0xfb800108,0xb7f70101,
	0xfb800108,0xb7f80101,0xfb800108,0xb7f90101,0xfb800108,0xb7fa0101,0xfb800108,0xb7fb0101,
	0xfb800108,0xb7fc0101,0xfb800108,0xb7fd0101,0xfb800108,0xb7fe0101,0xfb800108,0xb7ff0101,
	0xfb800108,0xb8000101,0xfb800108,0xb8010101,0xfb800108,0xb8020101,0xfb800108,0xb8030101,
	0xfb800108,0xb8040101,0xfb800108,0xb8050101,0xfb800108,0xb8060101,0xfb800108,0xb8070101,
	0xfb800108,0xb8080101,0xfb800108,0xb8090101,0xfb800108,0xb80a0101,0xfb800108,0xb80b0101,
	0xfb800108,0xb80c0101,0xfb800108,0xb80d0101,0xfb800108,0xb80e0101,0xfb800108,0xb80f0101,
	0xfb800108,0xb8100101,0xfb800108,0xb8110101,0xfb800108,0xb8120101,0xfb800108,0xb8130101,
	0xfb800108,0xb8140101,0xfb800108,0xb8150101,0xfb800108,0xb8160101,0xfb800108,0xb8170101,
	0xfb800108,0xb8180101,0xfb800108,0xb8190101,0xfb800108,0xb81a0101,0xfb800108,0xb81b0101,
	0xfb800108,0xb81c0101,0xfb800108,0xb81d0101,0xfb800108,0xb81e0101,0xfb800108,0xb81f0101,
	0xfb800108,0xb8200101,0xfb800108,0xb8210101,0xfb800108,0xb8220101,0xfb800108,0xb8230101,
	0xfb800108,0xb8240101,0xfb800108,0xb8250101,0xfb800108,0xb8260101,0xfb800108,0xb8270101,
	0xfb800108,0xb8280101,0xfb800108,0xb8290101,0xfb800108,0xb82a0101,0xfb800108,0xb82b0101,
	0xfb800108,0xb82c0101,0xfb800108,0xb82d0101,0xfb800108,0xb82e0101,0xfb800108,0xb82f0101,
	0xfb800108,0xb8300101,0xfb800108,0xb8310101,0xfb800108,0xb8320101,0xfb800108,0xb8330101,
	0xfb800108,0xb8340101,0xfb800108,0xb8350101,0xfb800108,0xb8360101,0xfb800108,0xb8370101,
	0xfb800108,0xb8380101,0xfb800108,0xb8390101,0xfb800108,0xb83a0101,0xfb800108,0xb83b0101,
	0xfb800108,0xb83c0101,0xfb800108,0xb83d0101,0xfb800108,0xb83e0101,0xfb800108,0xb83f0101,
	0xfb800108,0xb8400101,0xfb800108,0xb8410101,0xfb800108,0xb8420101,0xfb800108,0xb8430101,
	0xfb800108,0xb8440101,0xfb800108,0xb8450101,0xfb800108,0xb8460101,0xfb800108,0xb8470101,
	0xfb800108,0xb8480101,0xfb800108,0xb8490101,0xfb800108,0xb84a0101,0xfb800108,0xb84b0101,
	0xfb800108,0xb84c0101,0xfb800108,0xb84d0101,0xfb800108,0xb84e0101,0xfb800108,0xb84f0101,
	0xfb800108,0xb8500101,0xfb800108,0xb8510101,0xfb800108,0xb8520101,0xfb800108,0xb8530101,
	0xfb800108,0xb8540101,0xfb800108,0xb8550101,0xfb800108,0xb8560101,0xfb800108,0xb8570101,
	0xfb800108,0xb8580101,0xfb800108,0xb8590101,0xfb800108,0xb85a0101,0xfb800108,0xb85b0101,
	0xfb800108,0xb85c0101,0xfb800108,0xb85d0101,0xfb800108,0xb85e0101,0xfb800108,0xb85f0101,
	0xfb800108,0xb8600101,0xfb800108,0xb8610101,0xfb800108,0xb8620101,0xfb800108,0xb8630101,
	0xfb800108,0xb8640101,0xfb800108,0xb8650101,0xfb800108,0xb8660101,0xfb800108,0xb8670101,
	0xfb800108,0xb8680101,0xfb800108,0xb8690101,0xfb800108,0xb86a0101,0xfb800108,0xb86b0101,
	0xfb800108,0xb86c0101,0xfb800108,0xb86d0101,0xfb800108,0xb86e0101,0xfb800108,0xb86f0101,
	0xfb800108,0xb8700101,0xfb800108,0xb8710101,0xfb800108,0xb8720101,0xfb800108,0xb8730101,
	0xfb800108,0xb8740101,0xfb800108,0xb8750101,0xfb800108,0xb8760101,0xfb800108,0xb8770101,
	0xfb800108,0xb8780101,0xfb800108,0xb8790101,0xfb800108,0xb87a0101,0xfb800108,0xb87b0101,
	0xfb800108,0xb87c0101,0xfb800108,0xb87d0101,0xfb800108,0xb87e0101,0xfb800108,0xb87f0101,
	0xfb800108,0xb8800101,0xfb800108,0xb8810101,0xfb800108,0xb8820101,0xfb800108,0xb8830101,
	0xfb800108,0xb8840101,0xfb800108,0xb8850101,0xfb800108,0xb8860101,0xfb800108,0xb8870101,
	0xfb800108,0xb8880101,0xfb800108,0xb8890101,0xfb800108,0xb88a0101,0xfb800108,0xb88b0101,
	0xfb800108,0xb88c0101,0xfb800108,0xb88d0101,0xfb800108,0xb88e0101,0xfb800108,0xb88f0101,
	0xfb800108,0xb8900101,0xfb800108,0xb8910101,0xfb800108,0xb8920101,0xfb800108,0xb8930101,
	0xfb800108,0xb8940101,0xfb800108,0xb8950101,0xfb800108,0xb8960101,0xfb800108,0xb8970101,
	0xfb800108,0xb8980101,0xfb800108,0xb8990101,0xfb800108,0xb89a0101,0xfb800108,0xb89b0101,
	0xfb800108,0xb89c0101,0xfb800108,0xb89d0101,0xfb800108,0xb89e0101,0xfb800108,0xb89f0101,
	0xfb800108,0xb8a00101,0xfb800108,0xb8a10101,0xfb800108,0xb8a20101,0xfb800108,0xb8a30101,
	0xfb800108,0xb8a40101,0xfb800108,0xb8a50101,0xfb800108,0xb8a60101,0xfb800108,0xb8a70101,
	0xfb800108,0xb8a80101,0xfb800108,0xb8a90101,0xfb800108,0xb8aa0101,0xfb800108,0xb8ab0101,
	0xfb800108,0xb8ac0101,0xfb800108,0xb8ad0101,0xfb800108,0xb8ae0101,0xfb800108,0xb8af0101,
	0xfb800108,0xb8b00101,0xfb800108,0xb8b10101,0xfb800108,0xb8b20101,0xfb800108,0xb8b30101,
	0xfb800108,0xb8b40101,0xfb800108,0xb8b50101,0xfb800108,0xb8b60101,0xfb800108,0xb8b70101,
	0xfb800108,0xb8b80101,0xfb800108,0xb8b90101,0xfb800108,0xb8ba0101,0xfb800108,0xb8bb0101,
	0xfb800108,0xb8bc0101,0xfb800108,0xb8bd0101,0xfb800108,0xb8be0101,0xfb800108,0xb8bf0101,
	0xfb800108,0xb8c00101,0xfb800108,0xb8c10101,0xfb800108,0xb8c20101,0xfb800108,0xb8c30101,
	0xfb800108,0xb8c40101,0xfb800108,0xb8c50101,0xfb800108,0xb8c60101,0xfb800108,0xb8c70101,
	0xfb800108,0xb8c80101,0xfb800108,0xb8c90101,0xfb800108,0xb8ca0101,0xfb800108,0xb8cb0101,
	0xfb800108,0xb8cc0101,0xfb800108,0xb8cd0101,0xfb800108,0xb8ce0101,0xfb800108,0xb8cf0101,
	0xfb800108,0xb8d00101,0xfb800108,0xb8d10101,0xfb800108,0xb8d20101,0xfb800108,0xb8d30101,
	0xfb800108,0xb8d40101,0xfb800108,0xb8d50101,0xfb800108,0xb8d60101,0xfb800108,0xb8d70101,
	0xfb800108,0xb8d80101,0xfb800108,0xb8d90101,0xfb800108,0xb8da0101,0xfb800108,0xb8db0101,
	0xfb800108,0xb8dc0101,0xfb800108,0xb8dd0101,0xfb800108,0xb8de0101,0xfb800108,0xb8df0101,
	0xfb800108,0xb8e00101,0xfb800108,0xb8e10101,0xfb800108,0xb8e20101,0xfb800108,0xb8e30101,
	0xfb800108,0xb8e40101,0xfb800108,0xb8e50101,0xfb800108,0xb8e60101,0xfb800108,0xb8e70101,
	0xfb800108,0xb8e80101,0xfb800108,0xb8e90101,0xfb800108,0xb8ea0101,0xfb800108,0xb8eb0101,
	0xfb800108,0xb8ec0101,0xfb800108,0xb8ed0101,0xfb800108,0xb8ee0101,0xfb800108,0xb8ef0101,
	0xfb800108,0xb8f00101,0xfb800108,0xb8f10101,0xfb800108,0xb8f20101,0xfb800108,0xb8f30101,
	0xfb800108,0xb8f40101,0xfb800108,0xb8f50101,0xfb800108,0xb8f60101,0xfb800108,0xb8f70101,
	0xfb800108,0xb8f80101,0xfb800108,0xb8f90101,0xfb800108,0xb8fa0101,0xfb800108,0xb8fb0101,
	0xfb800108,0xb8fc0101,0xfb800108,0xb8fd0101,0xfb800108,0xb8fe0101,0xfb800108,0xb8ff0101,
	0xfb800108,0xb9000101,0xfb800108,0xb9010101,0xfb800108,0xb9020101,0xfb800108,0xb9030101,
	0xfb800108,0xb9040101,0xfb800108,0xb9050101,0xfb800108,0xb9060101,0xfb800108,0xb9070101,
	0xfb800108,0xb9080101,0xfb800108,0xb9090101,0xfb800108,0xb90a0101,0xfb800108,0xb90b0101,
	0xfb800108,0xb90c0101,0xfb800108,0xb90d0101,0xfb800108,0xb90e0101,0xfb800108,0xb90f0101,
	0xfb800108,0xb9100101,0xfb800108,0xb9110101,0xfb800108,0xb9120101,0xfb800108,0xb9130101,
	0xfb800108,0xb9140101,0xfb800108,0xb9150101,0xfb800108,0xb9160101,0xfb800108,0xb9170101,
	0xfb800108,0xb9180101,0xfb800108,0xb9190101,0xfb800108,0xb91a0101,0xfb800108,0xb91b0101,
	0xfb800108,0xb91c0101,0xfb800108,0xb91d0101,0xfb800108,0xb91e0101,0xfb800108,0xb91f0101,
	0xfb800108,0xb9200101,0xfb800108,0xb9210101,0xfb800108,0xb9220101,0xfb800108,0xb9230101,
	0xfb800108,0xb9240101,0xfb800108,0xb9250101,0xfb800108,0xb9260101,0xfb800108,0xb9270101,
	0xfb800108,0xb9280101,0xfb800108,0xb9290101,0xfb800108,0xb92a0101,0xfb800108,0xb92b0101,
	0xfb800108,0xb92c0101,0xfb800108,0xb92d0101,0xfb800108,0xb92e0101,0xfb800108,0xb92f0101,
	0xfb800108,0xb9300101,0xfb800108,0xb9310101,0xfb800108,0xb9320101,0xfb800108,0xb9330101,
	0xfb800108,0xb9340101,0xfb800108,0xb9350101,0xfb800108,0xb9360101,0xfb800108,0xb9370101,
	0xfb800108,0xb9380101,0xfb800108,0xb9390101,0xfb800108,0xb93a0101,0xfb800108,0xb93b0101,
	0xfb800108,0xb93c0101,0xfb800108,0xb93d0101,0xfb800108,0xb93e0101,0xfb800108,0xb93f0101,
	0xfb800108,0xb9400101,0xfb800108,0xb9410101,0xfb800108,0xb9420101,0xfb800108,0xb9430101,
	0xfb800108,0xb9440101,0xfb800108,0xb9450101,0xfb800108,0xb9460101,0xfb800108,0xb9470101,
	0xfb800108,0xb9480101,0xfb800108,0xb9490101,0xfb800108,0xb94a0101,0xfb800108,0xb94b0101,
	0xfb800108,0xb94c0101,0xfb800108,0xb94d0101,0xfb800108,0xb94e0101,0xfb800108,0xb94f0101,
	0xfb800108,0xb9500101,0xfb800108,0xb9510101,0xfb800108,0xb9520101,0xfb800108,0xb9530101,
	0xfb800108,0xb9540101,0xfb800108,0xb9550101,0xfb800108,0xb9560101,0xfb800108,0xb9570101,
	0xfb800108,0xb9580101,0xfb800108,0xb9590101,0xfb800108,0xb95a0101,0xfb800108,0xb95b0101,
	0xfb800108,0xb95c0101,0xfb800108,0xb95d0101,0xfb800108,0xb95e0101,0xfb800108,0xb95f0101,
	0xfb800108,0xb9600101,0xfb800108,0xb9610101,0xfb800108,0xb9620101,0xfb800108,0xb9630101,
	0xfb800108,0xb9640101,0xfb800108,0xb9650101,0xfb800108,0xb9660101,0xfb800108,0xb9670101,
	0xfb800108,0xb9680101,0xfb800108,0xb9690101,0xfb800108,0xb96a0101,0xfb800108,0xb96b0101,
	0xfb800108,0xb96c0101,0xfb800108,0xb96d0101,0xfb800108,0xb96e0101,0xfb800108,0xb96f0101,
	0xfb800108,0xb9700101,0xfb800108,0xb9710101,0xfb800108,0xb9720101,0xfb800108,0xb9730101,
	0xfb800108,0xb9740101,0xfb800108,0xb9750101,0xfb800108,0xb9760101,0xfb800108,0xb9770101,
	0xfb800108,0xb9780101,0xfb800108,0xb9790101,0xfb800108,0xb97a0101,0xfb800108,0xb97b0101,
	0xfb800108,0xb97c0101,0xfb800108,0xb97d0101,0xfb800108,0xb97e0101,0xfb800108,0xb97f0101,
	0xfb800108,0xb9800101,0xfb800108,0xb9810101,0xfb800108,0xb9820101,0xfb800108,0xb9830101,
	0xfb800108,0xb9840101,0xfb800108,0xb9850101,0xfb800108,0xb9860101,0xfb800108,0xb9870101,
	0xfb800108,0xb9880101,0xfb800108,0xb9890101,0xfb800108,0xb98a0101,0xfb800108,0xb98b0101,
	0xfb800108,0xb98c0101,0xfb800108,0xb98d0101,0xfb800108,0xb98e0101,0xfb800108,0xb98f0101,
	0xfb800108,0xb9900101,0xfb800108,0xb9910101,0xfb800108,0xb9920101,0xfb800108,0xb9930101,
	0xfb800108,0xb9940101,0xfb800108,0xb9950101,0xfb800108,0xb9960101,0xfb800108,0xb9970101,
	0xfb800108,0xb9980101,0xfb800108,0xb9990101,0xfb800108,0xb99a0101,0xfb800108,0xb99b0101,
	0xfb800108,0xb99c0101,0xfb800108,0xb99d0101,0xfb800108,0xb99e0101,0xfb800108,0xb99f0101,
	0xfb800108,0xb9a00101,0xfb800108,0xb9a10101,0xfb800108,0xb9a20101,0xfb800108,0xb9a30101,
	0xfb800108,0xb9a40101,0xfb800108,0xb9a50101,0xfb800108,0xb9a60101,0xfb800108,0xb9a70101,
	0xfb800108,0xb9a80101,0xfb800108,0xb9a90101,0xfb800108,0xb9aa0101,0xfb800108,0xb9ab0101,
	0xfb800108,0xb9ac0101,0xfb800108,0xb9ad0101,0xfb800108,0xb9ae0101,0xfb800108,0xb9af0101,
	0xfb800108,0xb9b00101,0xfb800108,0xb9b10101,0xfb800108,0xb9b20101,0xfb800108,0xb9b30101,
	0xfb800108,0xb9b40101,0xfb800108,0xb9b50101,0xfb800108,0xb9b60101,0xfb800108,0xb9b70101,
	0xfb800108,0xb9b80101,0xfb800108,0xb9b90101,0xfb800108,0xb9ba0101,0xfb800108,0xb9bb0101,
	0xfb800108,0xb9bc0101,0xfb800108,0xb9bd0101,0xfb800108,0xb9be0101,0xfb800108,0xb9bf0101,
	0xfb800108,0xb9c00101,0xfb800108,0xb9c10101,0xfb800108,0xb9c20101,0xfb800108,0xb9c30101,
	0xfb800108,0xb9c40101,0xfb800108,0xb9c50101,0xfb800108,0xb9c60101,0xfb800108,0xb9c70101,
	0xfb800108,0xb9c80101,0xfb800108,0xb9c90101,0xfb800108,0xb9ca0101,0xfb800108,0xb9cb0101,
	0xfb800108,0xb9cc0101,0xfb800108,0xb9cd0101,0xfb800108,0xb9ce0101,0xfb800108,0xb9cf0101,
	0xfb800108,0xb9d00101,0xfb800108,0xb9d10101,0xfb800108,0xb9d20101,0xfb800108,0xb9d30101,
	0xfb800108,0xb9d40101,0xfb800108,0xb9d50101,0xfb800108,0xb9d60101,0xfb800108,0xb9d70101,
	0xfb800108,0xb9d80101,0xfb800108,0xb9d90101,0xfb800108,0xb9da0101,0xfb800108,0xb9db0101,
	0xfb800108,0xb9dc0101,0xfb800108,0xb9dd0101,0xfb800108,0xb9de0101,0xfb800108,0xb9df0101,
	0xfb800108,0xb9e00101,0xfb800108,0xb9e10101,0xfb800108,0xb9e20101,0xfb800108,0xb9e30101,
	0xfb800108,0xb9e40101,0xfb800108,0xb9e50101,0xfb800108,0xb9e60101,0xfb800108,0xb9e70101,
	0xfb800108,0xb9e80101,0xfb800108,0xb9e90101,0xfb800108,0xb9ea0101,0xfb800108,0xb9eb0101,
	0xfb800108,0xb9ec0101,0xfb800108,0xb9ed0101,0xfb800108,0xb9ee0101,0xfb800108,0xb9ef0101,
	0xfb800108,0xb9f00101,0xfb800108,0xb9f10101,0xfb800108,0xb9f20101,0xfb800108,0xb9f30101,
	0xfb800108,0xb9f40101,0xfb800108,0xb9f50101,0xfb800108,0xb9f60101,0xfb800108,0xb9f70101,
	0xfb800108,0xb9f80101,0xfb800108,0xb9f90101,0xfb800108,0xb9fa0101,0xfb800108,0xb9fb0101,
	0xfb800108,0xb9fc0101,0xfb800108,0xb9fd0101,0xfb800108,0xb9fe0101,0xfb800108,0xb9ff0101,
	0xfb800108,0xba000101,0xfb800108,0xba010101,0xfb800108,0xba020101,0xfb800108,0xba030101,
	0xfb800108,0xba040101,0xfb800108,0xba050101,0xfb800108,0xba060101,0xfb800108,0xba070101,
	0xfb800108,0xba080101,0xfb800108,0xba090101,0xfb800108,0xba0a0101,0xfb800108,0xba0b0101,
	0xfb800108,0xba0c0101,0xfb800108,0xba0d0101,0xfb800108,0xba0e0101,0xfb800108,0xba0f0101,
	0xfb800108,0xba100101,0xfb800108,0xba110101,0xfb800108,0xba120101,0xfb800108,0xba130101,
	0xfb800108,0xba140101,0xfb800108,0xba150101,0xfb800108,0xba160101,0xfb800108,0xba170101,
	0xfb800108,0xba180101,0xfb800108,0xba190101,0xfb800108,0xba1a0101,0xfb800108,0xba1b0101,
	0xfb800108,0xba1c0101,0xfb800108,0xba1d0101,0xfb800108,0xba1e0101,0xfb800108,0xba1f0101,
	0xfb800108,0xba200101,0xfb800108,0xba210101,0xfb800108,0xba220101,0xfb800108,0xba230101,
	0xfb800108,0xba240101,0xfb800108,0xba250101,0xfb800108,0xba260101,0xfb800108,0xba270101,
	0xfb800108,0xba280101,0xfb800108,0xba290101,0xfb800108,0xba2a0101,0xfb800108,0xba2b0101,
	0xfb800108,0xba2c0101,0xfb800108,0xba2d0101,0xfb800108,0xba2e0101,0xfb800108,0xba2f0101,
	0xfb800108,0xba300101,0xfb800108,0xba310101,0xfb800108,0xba320101,0xfb800108,0xba330101,
	0xfb800108,0xba340101,0xfb800108,0xba350101,0xfb800108,0xba360101,0xfb800108,0xba370101,
	0xfb800108,0xba380101,0xfb800108,0xba390101,0xfb800108,0xba3a0101,0xfb800108,0xba3b0101,
	0xfb800108,0xba3c0101,0xfb800108,0xba3d0101,0xfb800108,0xba3e0101,0xfb800108,0xba3f0101,
	0xfb800108,0xba400101,0xfb800108,0xba410101,0xfb800108,0xba420101,0xfb800108,0xba430101,
	0xfb800108,0xba440101,0xfb800108,0xba450101,0xfb800108,0xba460101,0xfb800108,0xba470101,
	0xfb800108,0xba480101,0xfb800108,0xba490101,0xfb800108,0xba4a0101,0xfb800108,0xba4b0101,
	0xfb800108,0xba4c0101,0xfb800108,0xba4d0101,0xfb800108,0xba4e0101,0xfb800108,0xba4f0101,
	0xfb800108,0xba500101,0xfb800108,0xba510101,0xfb800108,0xba520101,0xfb800108,0xba530101,
	0xfb800108,0xba540101,0xfb800108,0xba550101,0xfb800108,0xba560101,0xfb800108,0xba570101,
	0xfb800108,0xba580101,0xfb800108,0xba590101,0xfb800108,0xba5a0101,0xfb800108,0xba5b0101,
	0xfb800108,0xba5c0101,0xfb800108,0xba5d0101,0xfb800108,0xba5e0101,0xfb800108,0xba5f0101,
	0xfb800108,0xba600101,0xfb800108,0xba610101,0xfb800108,0xba620101,0xfb800108,0xba630101,
	0xfb800108,0xba640101,0xfb800108,0xba650101,0xfb800108,0xba660101,0xfb800108,0xba670101,
	0xfb800108,0xba680101,0xfb800108,0xba690101,0xfb800108,0xba6a0101,0xfb800108,0xba6b0101,
	0xfb800108,0xba6c0101,0xfb800108,0xba6d0101,0xfb800108,0xba6e0101,0xfb800108,0xba6f0101,
	0xfb800108,0xba700101,0xfb800108,0xba710101,0xfb800108,0xba720101,0xfb800108,0xba730101,
	0xfb800108,0xba740101,0xfb800108,0xba750101,0xfb800108,0xba760101,0xfb800108,0xba770101,
	0xfb800108,0xba780101,0xfb800108,0xba790101,0xfb800108,0xba7a0101,0xfb800108,0xba7b0101,
	0xfb800108,0xba7c0101,0xfb800108,0xba7d0101,0xfb800108,0xba7e0101,0xfb800108,0xba7f0101,
	0xfb800108,0xba800101,0xfb800108,0xba810101,0xfb800108,0xba820101,0xfb800108,0xba830101,
	0xfb800108,0xba840101,0xfb800108,0xba850101,0xfb800108,0xba860101,0xfb800108,0xba870101,
	0xfb800108,0xba880101,0xfb800108,0xba890101,0xfb800108,0xba8a0101,0xfb800108,0xba8b0101,
	0xfb800108,0xba8c0101,0xfb800108,0xba8d0101,0xfb800108,0xba8e0101,0xfb800108,0xba8f0101,
	0xfb800108,0xba900101,0xfb800108,0xba910101,0xfb800108,0xba920101,0xfb800108,0xba930101,
	0xfb800108,0xba940101,0xfb800108,0xba950101,0xfb800108,0xba960101,0xfb800108,0xba970101,
	0xfb800108,0xba980101,0xfb800108,0xba990101,0xfb800108,0xba9a0101,0xfb800108,0xba9b0101,
	0xfb800108,0xba9c0101,0xfb800108,0xba9d0101,0xfb800108,0xba9e0101,0xfb800108,0xba9f0101,
	0xfb800108,0xbaa00101,0xfb800108,0xbaa10101,0xfb800108,0xbaa20101,0xfb800108,0xbaa30101,
	0xfb800108,0xbaa40101,0xfb800108,0xbaa50101,0xfb800108,0xbaa60101,0xfb800108,0xbaa70101,
	0xfb800108,0xbaa80101,0xfb800108,0xbaa90101,0xfb800108,0xbaaa0101,0xfb800108,0xbaab0101,
	0xfb800108,0xbaac0101,0xfb800108,0xbaad0101,0xfb800108,0xbaae0101,0xfb800108,0xbaaf0101,
	0xfb800108,0xbab00101,0xfb800108,0xbab10101,0xfb800108,0xbab20101,0xfb800108,0xbab30101,
	0xfb800108,0xbab40101,0xfb800108,0xbab50101,0xfb800108,0xbab60101,0xfb800108,0xbab70101,
	0xfb800108,0xbab80101,0xfb800108,0xbab90101,0xfb800108,0xbaba0101,0xfb800108,0xbabb0101,
	0xfb800108,0xbabc0101,0xfb800108,0xbabd0101,0xfb800108,0xbabe0101,0xfb800108,0xbabf0101,
	0xfb800108,0xbac00101,0xfb800108,0xbac10101,0xfb800108,0xbac20101,0xfb800108,0xbac30101,
	0xfb800108,0xbac40101,0xfb800108,0xbac50101,0xfb800108,0xbac60101,0xfb800108,0xbac70101,
	0xfb800108,0xbac80101,0xfb800108,0xbac90101,0xfb800108,0xbaca0101,0xfb800108,0xbacb0101,
	0xfb800108,0xbacc0101,0xfb800108,0xbacd0101,0xfb800108,0xbace0101,0xfb800108,0xbacf0101,
	0xfb800108,0xbad00101,0xfb800108,0xbad10101,0xfb800108,0xbad20101,0xfb800108,0xbad30101,
	0xfb800108,0xbad40101,0xfb800108,0xbad50101,0xfb800108,0xbad60101,0xfb800108,0xbad70101,
	0xfb800108,0xbad80101,0xfb800108,0xbad90101,0xfb800108,0xbada0101,0xfb800108,0xbadb0101,
	0xfb800108,0xbadc0101,0xfb800108,0xbadd0101,0xfb800108,0xbade0101,0xfb800108,0xbadf0101,
	0xfb800108,0xbae00101,0xfb800108,0xbae10101,0xfb800108,0xbae20101,0xfb800108,0xbae30101,
	0xfb800108,0xbae40101,0xfb800108,0xbae50101,0xfb800108,0xbae60101,0xfb800108,0xbae70101,
	0xfb800108,0xbae80101,0xfb800108,0xbae90101,0xfb800108,0xbaea0101,0xfb800108,0xbaeb0101,
	0xfb800108,0xbaec0101,0xfb800108,0xbaed0101,0xfb800108,0xbaee0101,0xfb800108,0xbaef0101,
	0xfb800108,0xbaf00101,0xfb800108,0xbaf10101,0xfb800108,0xbaf20101,0xfb800108,0xbaf30101,
	0xfb800108,0xbaf40101,0xfb800108,0xbaf50101,0xfb800108,0xbaf60101,0xfb800108,0xbaf70101,
	0xfb800108,0xbaf80101,0xfb800108,0xbaf90101,0xfb800108,0xbafa0101,0xfb800108,0xbafb0101,
	0xfb800108,0xbafc0101,0xfb800108,0xbafd0101,0xfb800108,0xbafe0101,0xfb800108,0xbaff0101,
	0xfb800108,0xbb000101,0xfb800108,0xbb010101,0xfb800108,0xbb020101,0xfb800108,0xbb030101,
	0xfb800108,0xbb040101,0xfb800108,0xbb050101,0xfb800108,0xbb060101,0xfb800108,0xbb070101,
	0xfb800108,0xbb080101,0xfb800108,0xbb090101,0xfb800108,0xbb0a0101,0xfb800108,0xbb0b0101,
	0xfb800108,0xbb0c0101,0xfb800108,0xbb0d0101,0xfb800108,0xbb0e0101,0xfb800108,0xbb0f0101,
	0xfb800108,0xbb100101,0xfb800108,0xbb110101,0xfb800108,0xbb120101,0xfb800108,0xbb130101,
	0xfb800108,0xbb140101,0xfb800108,0xbb150101,0xfb800108,0xbb160101,0xfb800108,0xbb170101,
	0xfb800108,0xbb180101,0xfb800108,0xbb190101,0xfb800108,0xbb1a0101,0xfb800108,0xbb1b0101,
	0xfb800108,0xbb1c0101,0xfb800108,0xbb1d0101,0xfb800108,0xbb1e0101,0xfb800108,0xbb1f0101,
	0xfb800108,0xbb200101,0xfb800108,0xbb210101,0xfb800108,0xbb220101,0xfb800108,0xbb230101,
	0xfb800108,0xbb240101,0xfb800108,0xbb250101,0xfb800108,0xbb260101,0xfb800108,0xbb270101,
	0xfb800108,0xbb280101,0xfb800108,0xbb290101,0xfb800108,0xbb2a0101,0xfb800108,0xbb2b0101,
	0xfb800108,0xbb2c0101,0xfb800108,0xbb2d0101,0xfb800108,0xbb2e0101,0xfb800108,0xbb2f0101,
	0xfb800108,0xbb300101,0xfb800108,0xbb310101,0xfb800108,0xbb320101,0xfb800108,0xbb330101,
	0xfb800108,0xbb340101,0xfb800108,0xbb350101,0xfb800108,0xbb360101,0xfb800108,0xbb370101,
	0xfb800108,0xbb380101,0xfb800108,0xbb390101,0xfb800108,0xbb3a0101,0xfb800108,0xbb3b0101,
	0xfb800108,0xbb3c0101,0xfb800108,0xbb3d0101,0xfb800108,0xbb3e0101,0xfb800108,0xbb3f0101,
	0xfb800108,0xbb400101,0xfb800108,0xbb410101,0xfb800108,0xbb420101,0xfb800108,0xbb430101,
	0xfb800108,0xbb440101,0xfb800108,0xbb450101,0xfb800108,0xbb460101,0xfb800108,0xbb470101,
	0xfb800108,0xbb480101,0xfb800108,0xbb490101,0xfb800108,0xbb4a0101,0xfb800108,0xbb4b0101,
	0xfb800108,0xbb4c0101,0xfb800108,0xbb4d0101,0xfb800108,0xbb4e0101,0xfb800108,0xbb4f0101,
	0xfb800108,0xbb500101,0xfb800108,0xbb510101,0xfb800108,0xbb520101,0xfb800108,0xbb530101,
	0xfb800108,0xbb540101,0xfb800108,0xbb550101,0xfb800108,0xbb560101,0xfb800108,0xbb570101,
	0xfb800108,0xbb580101,0xfb800108,0xbb590101,0xfb800108,0xbb5a0101,0xfb800108,0xbb5b0101,
	0xfb800108,0xbb5c0101,0xfb800108,0xbb5d0101,0xfb800108,0xbb5e0101,0xfb800108,0xbb5f0101,
	0xfb800108,0xbb600101,0xfb800108,0xbb610101,0xfb800108,0xbb620101,0xfb800108,0xbb630101,
	0xfb800108,0xbb640101,0xfb800108,0xbb650101,0xfb800108,0xbb660101,0xfb800108,0xbb670101,
	0xfb800108,0xbb680101,0xfb800108,0xbb690101,0xfb800108,0xbb6a0101,0xfb800108,0xbb6b0101,
	0xfb800108,0xbb6c0101,0xfb800108,0xbb6d0101,0xfb800108,0xbb6e0101,0xfb800108,0xbb6f0101,
	0xfb800108,0xbb700101,0xfb800108,0xbb710101,0xfb800108,0xbb720101,0xfb800108,0xbb730101,
	0xfb800108,0xbb740101,0xfb800108,0xbb750101,0xfb800108,0xbb760101,0xfb800108,0xbb770101,
	0xfb800108,0xbb780101,0xfb800108,0xbb790101,0xfb800108,0xbb7a0101,0xfb800108,0xbb7b0101,
	0xfb800108,0xbb7c0101,0xfb800108,0xbb7d0101,0xfb800108,0xbb7e0101,0xfb800108,0xbb7f0101,
	0xfb800108,0xbb800101,0xfb800108,0xbb810101,0xfb800108,0xbb820101,0xfb800108,0xbb830101,
	0xfb800108,0xbb840101,0xfb800108,0xbb850101,0xfb800108,0xbb860101,0xfb800108,0xbb870101,
	0xfb800108,0xbb880101,0xfb800108,0xbb890101,0xfb800108,0xbb8a0101,0xfb800108,0xbb8b0101,
	0xfb800108,0xbb8c0101,0xfb800108,0xbb8d0101,0xfb800108,0xbb8e0101,0xfb800108,0xbb8f0101,
	0xfb800108,0xbb900101,0xfb800108,0xbb910101,0xfb800108,0xbb920101,0xfb800108,0xbb930101,
	0xfb800108,0xbb940101,0xfb800108,0xbb950101,0xfb800108,0xbb960101,0xfb800108,0xbb970101,
	0xfb800108,0xbb980101,0xfb800108,0xbb990101,0xfb800108,0xbb9a0101,0xfb800108,0xbb9b0101,
	0xfb800108,0xbb9c0101,0xfb800108,0xbb9d0101,0xfb800108,0xbb9e0101,0xfb800108,0xbb9f0101,
	0xfb800108,0xbba00101,0xfb800108,0xbba10101,0xfb800108,0xbba20101,0xfb800108,0xbba30101,
	0xfb800108,0xbba40101,0xfb800108,0xbba50101,0xfb800108,0xbba60101,0xfb800108,0xbba70101,
	0xfb800108,0xbba80101,0xfb800108,0xbba90101,0xfb800108,0xbbaa0101,0xfb800108,0xbbab0101,
	0xfb800108,0xbbac0101,0xfb800108,0xbbad0101,0xfb800108,0xbbae0101,0xfb800108,0xbbaf0101,
	0xfb800108,0xbbb00101,0xfb800108,0xbbb10101,0xfb800108,0xbbb20101,0xfb800108,0xbbb30101,
	0xfb800108,0xbbb40101,0xfb800108,0xbbb50101,0xfb800108,0xbbb60101,0xfb800108,0xbbb70101,
	0xfb800108,0xbbb80101,0xfb800108,0xbbb90101,0xfb800108,0xbbba0101,0xfb800108,0xbbbb0101,
	0xfb800108,0xbbbc0101,0xfb800108,0xbbbd0101,0xfb800108,0xbbbe0101,0xfb800108,0xbbbf0101,
	0xfb800108,0xbbc00101,0xfb800108,0xbbc10101,0xfb800108,0xbbc20101,0xfb800108,0xbbc30101,
	0xfb800108,0xbbc40101,0xfb800108,0xbbc50101,0xfb800108,0xbbc60101,0xfb800108,0xbbc70101,
	0xfb800108,0xbbc80101,0xfb800108,0xbbc90101,0xfb800108,0xbbca0101,0xfb800108,0xbbcb0101,
	0xfb800108,0xbbcc0101,0xfb800108,0xbbcd0101,0xfb800108,0xbbce0101,0xfb800108,0xbbcf0101,
	0xfb800108,0xbbd00101,0xfb800108,0xbbd10101,0xfb800108,0xbbd20101,0xfb800108,0xbbd30101,
	0xfb800108,0xbbd40101,0xfb800108,0xbbd50101,0xfb800108,0xbbd60101,0xfb800108,0xbbd70101,
	0xfb800108,0xbbd80101,0xfb800108,0xbbd90101,0xfb800108,0xbbda0101,0xfb800108,0xbbdb0101,
	0xfb800108,0xbbdc0101,0xfb800108,0xbbdd0101,0xfb800108,0xbbde0101,0xfb800108,0xbbdf0101,
	0xfb800108,0xbbe00101,0xfb800108,0xbbe10101,0xfb800108,0xbbe20101,0xfb800108,0xbbe30101,
	0xfb800108,0xbbe40101,0xfb800108,0xbbe50101,0xfb800108,0xbbe60101,0xfb800108,0xbbe70101,
	0xfb800108,0xbbe80101,0xfb800108,0xbbe90101,0xfb800108,0xbbea0101,0xfb800108,0xbbeb0101,
	0xfb800108,0xbbec0101,0xfb800108,0xbbed0101,0xfb800108,0xbbee0101,0xfb800108,0xbbef0101,
	0xfb800108,0xbbf00101,0xfb800108,0xbbf10101,0xfb800108,0xbbf20101,0xfb800108,0xbbf30101,
	0xfb800108,0xbbf40101,0xfb800108,0xbbf50101,0xfb800108,0xbbf60101,0xfb800108,0xbbf70101,
	0xfb800108,0xbbf80101,0xfb800108,0xbbf90101,0xfb800108,0xbbfa0101,0xfb800108,0xbbfb0101,
	0xfb800108,0xbbfc0101,0xfb800108,0xbbfd0101,0xfb800108,0xbbfe0101,0xfb800108,0xbbff0101,
	0xfb800108,0xbc000101,0xfb800108,0xbc010101,0xfb800108,0xbc020101,0xfb800108,0xbc030101,
	0xfb800108,0xbc040101,0xfb800108,0xbc050101,0xfb800108,0xbc060101,0xfb800108,0xbc070101,
	0xfb800108,0xbc080101,0xfb800108,0xbc090101,0xfb800108,0xbc0a0101,0xfb800108,0xbc0b0101,
	0xfb800108,0xbc0c0101,0xfb800108,0xbc0d0101,0xfb800108,0xbc0e0101,0xfb800108,0xbc0f0101,
	0xfb800108,0xbc100101,0xfb800108,0xbc110101,0xfb800108,0xbc120101,0xfb800108,0xbc130101,
	0xfb800108,0xbc140101,0xfb800108,0xbc150101,0xfb800108,0xbc160101,0xfb800108,0xbc170101,
	0xfb800108,0xbc180101,0xfb800108,0xbc190101,0xfb800108,0xbc1a0101,0xfb800108,0xbc1b0101,
	0xfb800108,0xbc1c0101,0xfb800108,0xbc1d0101,0xfb800108,0xbc1e0101,0xfb800108,0xbc1f0101,
	0xfb800108,0xbc200101,0xfb800108,0xbc210101,0xfb800108,0xbc220101,0xfb800108,0xbc230101,
	0xfb800108,0xbc240101,0xfb800108,0xbc250101,0xfb800108,0xbc260101,0xfb800108,0xbc270101,
	0xfb800108,0xbc280101,0xfb800108,0xbc290101,0xfb800108,0xbc2a0101,0xfb800108,0xbc2b0101,
	0xfb800108,0xbc2c0101,0xfb800108,0xbc2d0101,0xfb800108,0xbc2e0101,0xfb800108,0xbc2f0101,
	0xfb800108,0xbc300101,0xfb800108,0xbc310101,0xfb800108,0xbc320101,0xfb800108,0xbc330101,
	0xfb800108,0xbc340101,0xfb800108,0xbc350101,0xfb800108,0xbc360101,0xfb800108,0xbc370101,
	0xfb800108,0xbc380101,0xfb800108,0xbc390101,0xfb800108,0xbc3a0101,0xfb800108,0xbc3b0101,
	0xfb800108,0xbc3c0101,0xfb800108,0xbc3d0101,0xfb800108,0xbc3e0101,0xfb800108,0xbc3f0101,
	0xfb800108,0xbc400101,0xfb800108,0xbc410101,0xfb800108,0xbc420101,0xfb800108,0xbc430101,
	0xfb800108,0xbc440101,0xfb800108,0xbc450101,0xfb800108,0xbc460101,0xfb800108,0xbc470101,
	0xfb800108,0xbc480101,0xfb800108,0xbc490101,0xfb800108,0xbc4a0101,0xfb800108,0xbc4b0101,
	0xfb800108,0xbc4c0101,0xfb800108,0xbc4d0101,0xfb800108,0xbc4e0101,0xfb800108,0xbc4f0101,
	0xfb800108,0xbc500101,0xfb800108,0xbc510101,0xfb800108,0xbc520101,0xfb800108,0xbc530101,
	0xfb800108,0xbc540101,0xfb800108,0xbc550101,0xfb800108,0xbc560101,0xfb800108,0xbc570101,
	0xfb800108,0xbc580101,0xfb800108,0xbc590101,0xfb800108,0xbc5a0101,0xfb800108,0xbc5b0101,
	0xfb800108,0xbc5c0101,0xfb800108,0xbc5d0101,0xfb800108,0xbc5e0101,0xfb800108,0xbc5f0101,
	0xfb800108,0xbc600101,0xfb800108,0xbc610101,0xfb800108,0xbc620101,0xfb800108,0xbc630101,
	0xfb800108,0xbc640101,0xfb800108,0xbc650101,0xfb800108,0xbc660101,0xfb800108,0xbc670101,
	0xfb800108,0xbc680101,0xfb800108,0xbc690101,0xfb800108,0xbc6a0101,0xfb800108,0xbc6b0101,
	0xfb800108,0xbc6c0101,0xfb800108,0xbc6d0101,0xfb800108,0xbc6e0101,0xfb800108,0xbc6f0101,
	0xfb800108,0xbc700101,0xfb800108,0xbc710101,0xfb800108,0xbc720101,0xfb800108,0xbc730101,
	0xfb800108,0xbc740101,0xfb800108,0xbc750101,0xfb800108,0xbc760101,0xfb800108,0xbc770101,
	0xfb800108,0xbc780101,0xfb800108,0xbc790101,0xfb800108,0xbc7a0101,0xfb800108,0xbc7b0101,
	0xfb800108,0xbc7c0101,0xfb800108,0xbc7d0101,0xfb800108,0xbc7e0101,0xfb800108,0xbc7f0101,
	0xfb800108,0xbc800101,0xfb800108,0xbc810101,0xfb800108,0xbc820101,0xfb800108,0xbc830101,
	0xfb800108,0xbc840101,0xfb800108,0xbc850101,0xfb800108,0xbc860101,0xfb800108,0xbc870101,
	0xfb800108,0xbc880101,0xfb800108,0xbc890101,0xfb800108,0xbc8a0101,0xfb800108,0xbc8b0101,
	0xfb800108,0xbc8c0101,0xfb800108,0xbc8d0101,0xfb800108,0xbc8e0101,0xfb800108,0xbc8f0101,
	0xfb800108,0xbc900101,0xfb800108,0xbc910101,0xfb800108,0xbc920101,0xfb800108,0xbc930101,
	0xfb800108,0xbc940101,0xfb800108,0xbc950101,0xfb800108,0xbc960101,0xfb800108,0xbc970101,
	0xfb800108,0xbc980101,0xfb800108,0xbc990101,0xfb800108,0xbc9a0101,0xfb800108,0xbc9b0101,
	0xfb800108,0xbc9c0101,0xfb800108,0xbc9d0101,0xfb800108,0xbc9e0101,0xfb800108,0xbc9f0101,
	0xfb800108,0xbca00101,0xfb800108,0xbca10101,0xfb800108,0xbca20101,0xfb800108,0xbca30101,
	0xfb800108,0xbca40101,0xfb800108,0xbca50101,0xfb800108,0xbca60101,0xfb800108,0xbca70101,
	0xfb800108,0xbca80101,0xfb800108,0xbca90101,0xfb800108,0xbcaa0101,0xfb800108,0xbcab0101,
	0xfb800108,0xbcac0101,0xfb800108,0xbcad0101,0xfb800108,0xbcae0101,0xfb800108,0xbcaf0101,
	0xfb800108,0xbcb00101,0xfb800108,0xbcb10101,0xfb800108,0xbcb20101,0xfb800108,0xbcb30101,
	0xfb800108,0xbcb40101,0xfb800108,0xbcb50101,0xfb800108,0xbcb60101,0xfb800108,0xbcb70101,
	0xfb800108,0xbcb80101,0xfb800108,0xbcb90101,0xfb800108,0xbcba0101,0xfb800108,0xbcbb0101,
	0xfb800108,0xbcbc0101,0xfb800108,0xbcbd0101,0xfb800108,0xbcbe0101,0xfb800108,0xbcbf0101,
	0xfb800108,0xbcc00101,0xfb800108,0xbcc10101,0xfb800108,0xbcc20101,0xfb800108,0xbcc30101,
	0xfb800108,0xbcc40101,0xfb800108,0xbcc50101,0xfb800108,0xbcc60101,0xfb800108,0xbcc70101,
	0xfb800108,0xbcc80101,0xfb800108,0xbcc90101,0xfb800108,0xbcca0101,0xfb800108,0xbccb0101,
	0xfb800108,0xbccc0101,0xfb800108,0xbccd0101,0xfb800108,0xbcce0101,0xfb800108,0xbccf0101,
	0xfb800108,0xbcd00101,0xfb800108,0xbcd10101,0xfb800108,0xbcd20101,0xfb800108,0xbcd30101,
	0xfb800108,0xbcd40101,0xfb800108,0xbcd50101,0xfb800108,0xbcd60101,0xfb800108,0xbcd70101,
	0xfb800108,0xbcd80101,0xfb800108,0xbcd90101,0xfb800108,0xbcda0101,0xfb800108,0xbcdb0101,
	0xfb800108,0xbcdc0101,0xfb800108,0xbcdd0101,0xfb800108,0xbcde0101,0xfb800108,0xbcdf0101,
	0xfb800108,0xbce00101,0xfb800108,0xbce10101,0xfb800108,0xbce20101,0xfb800108,0xbce30101,
	0xfb800108,0xbce40101,0xfb800108,0xbce50101,0xfb800108,0xbce60101,0xfb800108,0xbce70101,
	0xfb800108,0xbce80101,0xfb800108,0xbce90101,0xfb800108,0xbcea0101,0xfb800108,0xbceb0101,
	0xfb800108,0xbcec0101,0xfb800108,0xbced0101,0xfb800108,0xbcee0101,0xfb800108,0xbcef0101,
	0xfb800108,0xbcf00101,0xfb800108,0xbcf10101,0xfb800108,0xbcf20101,0xfb800108,0xbcf30101,
	0xfb800108,0xbcf40101,0xfb800108,0xbcf50101,0xfb800108,0xbcf60101,0xfb800108,0xbcf70101,
	0xfb800108,0xbcf80101,0xfb800108,0xbcf90101,0xfb800108,0xbcfa0101,0xfb800108,0xbcfb0101,
	0xfb800108,0xbcfc0101,0xfb800108,0xbcfd0101,0xfb800108,0xbcfe0101,0xfb800108,0xbcff0101,
	0xfb800108,0xbd000101,0xfb800108,0xbd010101,0xfb800108,0xbd020101,0xfb800108,0xbd030101,
	0xfb800108,0xbd040101,0xfb800108,0xbd050101,0xfb800108,0xbd060101,0xfb800108,0xbd070101,
	0xfb800108,0xbd080101,0xfb800108,0xbd090101,0xfb800108,0xbd0a0101,0xfb800108,0xbd0b0101,
	0xfb800108,0xbd0c0101,0xfb800108,0xbd0d0101,0xfb800108,0xbd0e0101,0xfb800108,0xbd0f0101,
	0xfb800108,0xbd100101,0xfb800108,0xbd110101,0xfb800108,0xbd120101,0xfb800108,0xbd130101,
	0xfb800108,0xbd140101,0xfb800108,0xbd150101,0xfb800108,0xbd160101,0xfb800108,0xbd170101,
	0xfb800108,0xbd180101,0xfb800108,0xbd190101,0xfb800108,0xbd1a0101,0xfb800108,0xbd1b0101,
	0xfb800108,0xbd1c0101,0xfb800108,0xbd1d0101,0xfb800108,0xbd1e0101,0xfb800108,0xbd1f0101,
	0xfb800108,0xbd200101,0xfb800108,0xbd210101,0xfb800108,0xbd220101,0xfb800108,0xbd230101,
	0xfb800108,0xbd240101,0xfb800108,0xbd250101,0xfb800108,0xbd260101,0xfb800108,0xbd270101,
	0xfb800108,0xbd280101,0xfb800108,0xbd290101,0xfb800108,0xbd2a0101,0xfb800108,0xbd2b0101,
	0xfb800108,0xbd2c0101,0xfb800108,0xbd2d0101,0xfb800108,0xbd2e0101,0xfb800108,0xbd2f0101,
	0xfb800108,0xbd300101,0xfb800108,0xbd310101,0xfb800108,0xbd320101,0xfb800108,0xbd330101,
	0xfb800108,0xbd340101,0xfb800108,0xbd350101,0xfb800108,0xbd360101,0xfb800108,0xbd370101,
	0xfb800108,0xbd380101,0xfb800108,0xbd390101,0xfb800108,0xbd3a0101,0xfb800108,0xbd3b0101,
	0xfb800108,0xbd3c0101,0xfb800108,0xbd3d0101,0xfb800108,0xbd3e0101,0xfb800108,0xbd3f0101,
	0xfb800108,0xbd400101,0xfb800108,0xbd410101,0xfb800108,0xbd420101,0xfb800108,0xbd430101,
	0xfb800108,0xbd440101,0xfb800108,0xbd450101,0xfb800108,0xbd460101,0xfb800108,0xbd470101,
	0xfb800108,0xbd480101,0xfb800108,0xbd490101,0xfb800108,0xbd4a0101,0xfb800108,0xbd4b0101,
	0xfb800108,0xbd4c0101,0xfb800108,0xbd4d0101,0xfb800108,0xbd4e0101,0xfb800108,0xbd4f0101,
	0xfb800108,0xbd500101,0xfb800108,0xbd510101,0xfb800108,0xbd520101,0xfb800108,0xbd530101,
	0xfb800108,0xbd540101,0xfb800108,0xbd550101,0xfb800108,0xbd560101,0xfb800108,0xbd570101,
	0xfb800108,0xbd580101,0xfb800108,0xbd590101,0xfb800108,0xbd5a0101,0xfb800108,0xbd5b0101,
	0xfb800108,0xbd5c0101,0xfb800108,0xbd5d0101,0xfb800108,0xbd5e0101,0xfb800108,0xbd5f0101,
	0xfb800108,0xbd600101,0xfb800108,0xbd610101,0xfb800108,0xbd620101,0xfb800108,0xbd630101,
	0xfb800108,0xbd640101,0xfb800108,0xbd650101,0xfb800108,0xbd660101,0xfb800108,0xbd670101,
	0xfb800108,0xbd680101,0xfb800108,0xbd690101,0xfb800108,0xbd6a0101,0xfb800108,0xbd6b0101,
	0xfb800108,0xbd6c0101,0xfb800108,0xbd6d0101,0xfb800108,0xbd6e0101,0xfb800108,0xbd6f0101,
	0xfb800108,0xbd700101,0xfb800108,0xbd710101,0xfb800108,0xbd720101,0xfb800108,0xbd730101,
	0xfb800108,0xbd740101,0xfb800108,0xbd750101,0xfb800108,0xbd760101,0xfb800108,0xbd770101,
	0xfb800108,0xbd780101,0xfb800108,0xbd790101,0xfb800108,0xbd7a0101,0xfb800108,0xbd7b0101,
	0xfb800108,0xbd7c0101,0xfb800108,0xbd7d0101,0xfb800108,0xbd7e0101,0xfb800108,0xbd7f0101,
	0xfb800108,0xbd800101,0xfb800108,0xbd810101,0xfb800108,0xbd820101,0xfb800108,0xbd830101,
	0xfb800108,0xbd840101,0xfb800108,0xbd850101,0xfb800108,0xbd860101,0xfb800108,0xbd870101,
	0xfb800108,0xbd880101,0xfb800108,0xbd890101,0xfb800108,0xbd8a0101,0xfb800108,0xbd8b0101,
	0xfb800108,0xbd8c0101,0xfb800108,0xbd8d0101,0xfb800108,0xbd8e0101,0xfb800108,0xbd8f0101,
	0xfb800108,0xbd900101,0xfb800108,0xbd910101,0xfb800108,0xbd920101,0xfb800108,0xbd930101,
	0xfb800108,0xbd940101,0xfb800108,0xbd950101,0xfb800108,0xbd960101,0xfb800108,0xbd970101,
	0xfb800108,0xbd980101,0xfb800108,0xbd990101,0xfb800108,0xbd9a0101,0xfb800108,0xbd9b0101,
	0xfb800108,0xbd9c0101,0xfb800108,0xbd9d0101,0xfb800108,0xbd9e0101,0xfb800108,0xbd9f0101,
	0xfb800108,0xbda00101,0xfb800108,0xbda10101,0xfb800108,0xbda20101,0xfb800108,0xbda30101,
	0xfb800108,0xbda40101,0xfb800108,0xbda50101,0xfb800108,0xbda60101,0xfb800108,0xbda70101,
	0xfb800108,0xbda80101,0xfb800108,0xbda90101,0xfb800108,0xbdaa0101,0xfb800108,0xbdab0101,
	0xfb800108,0xbdac0101,0xfb800108,0xbdad0101,0xfb800108,0xbdae0101,0xfb800108,0xbdaf0101,
	0xfb800108,0xbdb00101,0xfb800108,0xbdb10101,0xfb800108,0xbdb20101,0xfb800108,0xbdb30101,
	0xfb800108,0xbdb40101,0xfb800108,0xbdb50101,0xfb800108,0xbdb60101,0xfb800108,0xbdb70101,
	0xfb800108,0xbdb80101,0xfb800108,0xbdb90101,0xfb800108,0xbdba0101,0xfb800108,0xbdbb0101,
	0xfb800108,0xbdbc0101,0xfb800108,0xbdbd0101,0xfb800108,0xbdbe0101,0xfb800108,0xbdbf0101,
	0xfb800108,0xbdc00101,0xfb800108,0xbdc10101,0xfb800108,0xbdc20101,0xfb800108,0xbdc30101,
	0xfb800108,0xbdc40101,0xfb800108,0xbdc50101,0xfb800108,0xbdc60101,0xfb800108,0xbdc70101,
	0xfb800108,0xbdc80101,0xfb800108,0xbdc90101,0xfb800108,0xbdca0101,0xfb800108,0xbdcb0101,
	0xfb800108,0xbdcc0101,0xfb800108,0xbdcd0101,0xfb800108,0xbdce0101,0xfb800108,0xbdcf0101,
	0xfb800108,0xbdd00101,0xfb800108,0xbdd10101,0xfb800108,0xbdd20101,0xfb800108,0xbdd30101,
	0xfb800108,0xbdd40101,0xfb800108,0xbdd50101,0xfb800108,0xbdd60101,0xfb800108,0xbdd70101,
	0xfb800108,0xbdd80101,0xfb800108,0xbdd90101,0xfb800108,0xbdda0101,0xfb800108,0xbddb0101,
	0xfb800108,0xbddc0101,0xfb800108,0xbddd0101,0xfb800108,0xbdde0101,0xfb800108,0xbddf0101,
	0xfb800108,0xbde00101,0xfb800108,0xbde10101,0xfb800108,0xbde20101,0xfb800108,0xbde30101,
	0xfb800108,0xbde40101,0xfb800108,0xbde50101,0xfb800108,0xbde60101,0xfb800108,0xbde70101,
	0xfb800108,0xbde80101,0xfb800108,0xbde90101,0xfb800108,0xbdea0101,0xfb800108,0xbdeb0101,
	0xfb800108,0xbdec0101,0xfb800108,0xbded0101,0xfb800108,0xbdee0101,0xfb800108,0xbdef0101,
	0xfb800108,0xbdf00101,0xfb800108,0xbdf10101,0xfb800108,0xbdf20101,0xfb800108,0xbdf30101,
	0xfb800108,0xbdf40101,0xfb800108,0xbdf50101,0xfb800108,0xbdf60101,0xfb800108,0xbdf70101,
	0xfb800108,0xbdf80101,0xfb800108,0xbdf90101,0xfb800108,0xbdfa0101,0xfb800108,0xbdfb0101,
	0xfb800108,0xbdfc0101,0xfb800108,0xbdfd0101,0xfb800108,0xbdfe0101,0xfb800108,0xbdff0101,
	0xfb800108,0xbe000101,0xfb800108,0xbe010101,0xfb800108,0xbe020101,0xfb800108,0xbe030101,
	0xfb800108,0xbe040101,0xfb800108,0xbe050101,0xfb800108,0xbe060101,0xfb800108,0xbe070101,
	0xfb800108,0xbe080101,0xfb800108,0xbe090101,0xfb800108,0xbe0a0101,0xfb800108,0xbe0b0101,
	0xfb800108,0xbe0c0101,0xfb800108,0xbe0d0101,0xfb800108,0xbe0e0101,0xfb800108,0xbe0f0101,
	0xfb800108,0xbe100101,0xfb800108,0xbe110101,0xfb800108,0xbe120101,0xfb800108,0xbe130101,
	0xfb800108,0xbe140101,0xfb800108,0xbe150101,0xfb800108,0xbe160101,0xfb800108,0xbe170101,
	0xfb800108,0xbe180101,0xfb800108,0xbe190101,0xfb800108,0xbe1a0101,0xfb800108,0xbe1b0101,
	0xfb800108,0xbe1c0101,0xfb800108,0xbe1d0101,0xfb800108,0xbe1e0101,0xfb800108,0xbe1f0101,
	0xfb800108,0xbe200101,0xfb800108,0xbe210101,0xfb800108,0xbe220101,0xfb800108,0xbe230101,
	0xfb800108,0xbe240101,0xfb800108,0xbe250101,0xfb800108,0xbe260101,0xfb800108,0xbe270101,
	0xfb800108,0xbe280101,0xfb800108,0xbe290101,0xfb800108,0xbe2a0101,0xfb800108,0xbe2b0101,
	0xfb800108,0xbe2c0101,0xfb800108,0xbe2d0101,0xfb800108,0xbe2e0101,0xfb800108,0xbe2f0101,
	0xfb800108,0xbe300101,0xfb800108,0xbe310101,0xfb800108,0xbe320101,0xfb800108,0xbe330101,
	0xfb800108,0xbe340101,0xfb800108,0xbe350101,0xfb800108,0xbe360101,0xfb800108,0xbe370101,
	0xfb800108,0xbe380101,0xfb800108,0xbe390101,0xfb800108,0xbe3a0101,0xfb800108,0xbe3b0101,
	0xfb800108,0xbe3c0101,0xfb800108,0xbe3d0101,0xfb800108,0xbe3e0101,0xfb800108,0xbe3f0101,
	0xfb800108,0xbe400101,0xfb800108,0xbe410101,0xfb800108,0xbe420101,0xfb800108,0xbe430101,
	0xfb800108,0xbe440101,0xfb800108,0xbe450101,0xfb800108,0xbe460101,0xfb800108,0xbe470101,
	0xfb800108,0xbe480101,0xfb800108,0xbe490101,0xfb800108,0xbe4a0101,0xfb800108,0xbe4b0101,
	0xfb800108,0xbe4c0101,0xfb800108,0xbe4d0101,0xfb800108,0xbe4e0101,0xfb800108,0xbe4f0101,
	0xfb800108,0xbe500101,0xfb800108,0xbe510101,0xfb800108,0xbe520101,0xfb800108,0xbe530101,
	0xfb800108,0xbe540101,0xfb800108,0xbe550101,0xfb800108,0xbe560101,0xfb800108,0xbe570101,
	0xfb800108,0xbe580101,0xfb800108,0xbe590101,0xfb800108,0xbe5a0101,0xfb800108,0xbe5b0101,
	0xfb800108,0xbe5c0101,0xfb800108,0xbe5d0101,0xfb800108,0xbe5e0101,0xfb800108,0xbe5f0101,
	0xfb800108,0xbe600101,0xfb800108,0xbe610101,0xfb800108,0xbe620101,0xfb800108,0xbe630101,
	0xfb800108,0xbe640101,0xfb800108,0xbe650101,0xfb800108,0xbe660101,0xfb800108,0xbe670101,
	0xfb800108,0xbe680101,0xfb800108,0xbe690101,0xfb800108,0xbe6a0101,0xfb800108,0xbe6b0101,
	0xfb800108,0xbe6c0101,0xfb800108,0xbe6d0101,0xfb800108,0xbe6e0101,0xfb800108,0xbe6f0101,
	0xfb800108,0xbe700101,0xfb800108,0xbe710101,0xfb800108,0xbe720101,0xfb800108,0xbe730101,
	0xfb800108,0xbe740101,0xfb800108,0xbe750101,0xfb800108,0xbe760101,0xfb800108,0xbe770101,
	0xfb800108,0xbe780101,0xfb800108,0xbe790101,0xfb800108,0xbe7a0101,0xfb800108,0xbe7b0101,
	0xfb800108,0xbe7c0101,0xfb800108,0xbe7d0101,0xfb800108,0xbe7e0101,0xfb800108,0xbe7f0101,
	0xfb800108,0xbe800101,0xfb800108,0xbe810101,0xfb800108,0xbe820101,0xfb800108,0xbe830101,
	0xfb800108,0xbe840101,0xfb800108,0xbe850101,0xfb800108,0xbe860101,0xfb800108,0xbe870101,
	0xfb800108,0xbe880101,0xfb800108,0xbe890101,0xfb800108,0xbe8a0101,0xfb800108,0xbe8b0101,
	0xfb800108,0xbe8c0101,0xfb800108,0xbe8d0101,0xfb800108,0xbe8e0101,0xfb800108,0xbe8f0101,
	0xfb800108,0xbe900101,0xfb800108,0xbe910101,0xfb800108,0xbe920101,0xfb800108,0xbe930101,
	0xfb800108,0xbe940101,0xfb800108,0xbe950101,0xfb800108,0xbe960101,0xfb800108,0xbe970101,
	0xfb800108,0xbe980101,0xfb800108,0xbe990101,0xfb800108,0xbe9a0101,0xfb800108,0xbe9b0101,
	0xfb800108,0xbe9c0101,0xfb800108,0xbe9d0101,0xfb800108,0xbe9e0101,0xfb800108,0xbe9f0101,
	0xfb800108,0xbea00101,0xfb800108,0xbea10101,0xfb800108,0xbea20101,0xfb800108,0xbea30101,
	0xfb800108,0xbea40101,0xfb800108,0xbea50101,0xfb800108,0xbea60101,0xfb800108,0xbea70101,
	0xfb800108,0xbea80101,0xfb800108,0xbea90101,0xfb800108,0xbeaa0101,0xfb800108,0xbeab0101,
	0xfb800108,0xbeac0101,0xfb800108,0xbead0101,0xfb800108,0xbeae0101,0xfb800108,0xbeaf0101,
	0xfb800108,0xbeb00101,0xfb800108,0xbeb10101,0xfb800108,0xbeb20101,0xfb800108,0xbeb30101,
	0xfb800108,0xbeb40101,0xfb800108,0xbeb50101,0xfb800108,0xbeb60101,0xfb800108,0xbeb70101,
	0xfb800108,0xbeb80101,0xfb800108,0xbeb90101,0xfb800108,0xbeba0101,0xfb800108,0xbebb0101,
	0xfb800108,0xbebc0101,0xfb800108,0xbebd0101,0xfb800108,0xbebe0101,0xfb800108,0xbebf0101,
	0xfb800108,0xbec00101,0xfb800108,0xbec10101,0xfb800108,0xbec20101,0xfb800108,0xbec30101,
	0xfb800108,0xbec40101,0xfb800108,0xbec50101,0xfb800108,0xbec60101,0xfb800108,0xbec70101,
	0xfb800108,0xbec80101,0xfb800108,0xbec90101,0xfb800108,0xbeca0101,0xfb800108,0xbecb0101,
	0xfb800108,0xbecc0101,0xfb800108,0xbecd0101,0xfb800108,0xbece0101,0xfb800108,0xbecf0101,
	0xfb800108,0xbed00101,0xfb800108,0xbed10101,0xfb800108,0xbed20101,0xfb800108,0xbed30101,
	0xfb800108,0xbed40101,0xfb800108,0xbed50101,0xfb800108,0xbed60101,0xfb800108,0xbed70101,
	0xfb800108,0xbed80101,0xfb800108,0xbed90101,0xfb800108,0xbeda0101,0xfb800108,0xbedb0101,
	0xfb800108,0xbedc0101,0xfb800108,0xbedd0101,0xfb800108,0xbede0101,0xfb800108,0xbedf0101,
	0xfb800108,0xbee00101,0xfb800108,0xbee10101,0xfb800108,0xbee20101,0xfb800108,0xbee30101,
	0xfb800108,0xbee40101,0xfb800108,0xbee50101,0xfb800108,0xbee60101,0xfb800108,0xbee70101,
	0xfb800108,0xbee80101,0xfb800108,0xbee90101,0xfb800108,0xbeea0101,0xfb800108,0xbeeb0101,
	0xfb800108,0xbeec0101,0xfb800108,0xbeed0101,0xfb800108,0xbeee0101,0xfb800108,0xbeef0101,
	0xfb800108,0xbef00101,0xfb800108,0xbef10101,0xfb800108,0xbef20101,0xfb800108,0xbef30101,
	0xfb800108,0xbef40101,0xfb800108,0xbef50101,0xfb800108,0xbef60101,0xfb800108,0xbef70101,
	0xfb800108,0xbef80101,0xfb800108,0xbef90101,0xfb800108,0xbefa0101,0xfb800108,0xbefb0101,
	0xfb800108,0xbefc0101,0xfb800108,0xbefd0101,0xfb800108,0xbefe0101,0xfb800108,0xbeff0101,
	0xfb800108,0xbf000101,0xfb800108,0xbf010101,0xfb800108,0xbf020101,0xfb800108,0xbf030101,
	0xfb800108,0xbf040101,0xfb800108,0xbf050101,0xfb800108,0xbf060101,0xfb800108,0xbf070101,
	0xfb800108,0xbf080101,0xfb800108,0xbf090101,0xfb800108,0xbf0a0101,0xfb800108,0xbf0b0101,
	0xfb800108,0xbf0c0101,0xfb800108,0xbf0d0101,0xfb800108,0xbf0e0101,0xfb800108,0xbf0f0101,
	0xfb800108,0xbf100101,0xfb800108,0xbf110101,0xfb800108,0xbf120101,0xfb800108,0xbf130101,
	0xfb800108,0xbf140101,0xfb800108,0xbf150101,0xfb800108,0xbf160101,0xfb800108,0xbf170101,
	0xfb800108,0xbf180101,0xfb800108,0xbf190101,0xfb800108,0xbf1a0101,0xfb800108,0xbf1b0101,
	0xfb800108,0xbf1c0101,0xfb800108,0xbf1d0101,0xfb800108,0xbf1e0101,0xfb800108,0xbf1f0101,
	0xfb800108,0xbf200101,0xfb800108,0xbf210101,0xfb800108,0xbf220101,0xfb800108,0xbf230101,
	0xfb800108,0xbf240101,0xfb800108,0xbf250101,0xfb800108,0xbf260101,0xfb800108,0xbf270101,
	0xfb800108,0xbf280101,0xfb800108,0xbf290101,0xfb800108,0xbf2a0101,0xfb800108,0xbf2b0101,
	0xfb800108,0xbf2c0101,0xfb800108,0xbf2d0101,0xfb800108,0xbf2e0101,0xfb800108,0xbf2f0101,
	0xfb800108,0xbf300101,0xfb800108,0xbf310101,0xfb800108,0xbf320101,0xfb800108,0xbf330101,
	0xfb800108,0xbf340101,0xfb800108,0xbf350101,0xfb800108,0xbf360101,0xfb800108,0xbf370101,
	0xfb800108,0xbf380101,0xfb800108,0xbf390101,0xfb800108,0xbf3a0101,0xfb800108,0xbf3b0101,
	0xfb800108,0xbf3c0101,0xfb800108,0xbf3d0101,0xfb800108,0xbf3e0101,0xfb800108,0xbf3f0101,
	0xfb800108,0xbf400101,0xfb800108,0xbf410101,0xfb800108,0xbf420101,0xfb800108,0xbf430101,
	0xfb800108,0xbf440101,0xfb800108,0xbf450101,0xfb800108,0xbf460101,0xfb800108,0xbf470101,
	0xfb800108,0xbf480101,0xfb800108,0xbf490101,0xfb800108,0xbf4a0101,0xfb800108,0xbf4b0101,
	0xfb800108,0xbf4c0101,0xfb800108,0xbf4d0101,0xfb800108,0xbf4e0101,0xfb800108,0xbf4f0101,
	0xfb800108,0xbf500101,0xfb800108,0xbf510101,0xfb800108,0xbf520101,0xfb800108,0xbf530101,
	0xfb800108,0xbf540101,0xfb800108,0xbf550101,0xfb800108,0xbf560101,0xfb800108,0xbf570101,
	0xfb800108,0xbf580101,0xfb800108,0xbf590101,0xfb800108,0xbf5a0101,0xfb800108,0xbf5b0101,
	0xfb800108,0xbf5c0101,0xfb800108,0xbf5d0101,0xfb800108,0xbf5e0101,0xfb800108,0xbf5f0101,
	0xfb800108,0xbf600101,0xfb800108,0xbf610101,0xfb800108,0xbf620101,0xfb800108,0xbf630101,
	0xfb800108,0xbf640101,0xfb800108,0xbf650101,0xfb800108,0xbf660101,0xfb800108,0xbf670101,
	0xfb800108,0xbf680101,0xfb800108,0xbf690101,0xfb800108,0xbf6a0101,0xfb800108,0xbf6b0101,
	0xfb800108,0xbf6c0101,0xfb800108,0xbf6d0101,0xfb800108,0xbf6e0101,0xfb800108,0xbf6f0101,
	0xfb800108,0xbf700101,0xfb800108,0xbf710101,0xfb800108,0xbf720101,0xfb800108,0xbf730101,
	0xfb800108,0xbf740101,0xfb800108,0xbf750101,0xfb800108,0xbf760101,0xfb800108,0xbf770101,
	0xfb800108,0xbf780101,0xfb800108,0xbf790101,0xfb800108,0xbf7a0101,0xfb800108,0xbf7b0101,
	0xfb800108,0xbf7c0101,0xfb800108,0xbf7d0101,0xfb800108,0xbf7e0101,0xfb800108,0xbf7f0101,
	0xfb800108,0xbf800101,0xfb800108,0xbf810101,0xfb800108,0xbf820101,0xfb800108,0xbf830101,
	0xfb800108,0xbf840101,0xfb800108,0xbf850101,0xfb800108,0xbf860101,0xfb800108,0xbf870101,
	0xfb800108,0xbf880101,0xfb800108,0xbf890101,0xfb800108,0xbf8a0101,0xfb800108,0xbf8b0101,
	0xfb800108,0xbf8c0101,0xfb800108,0xbf8d0101,0xfb800108,0xbf8e0101,0xfb800108,0xbf8f0101,
	0xfb800108,0xbf900101,0xfb800108,0xbf910101,0xfb800108,0xbf920101,0xfb800108,0xbf930101,
	0xfb800108,0xbf940101,0xfb800108,0xbf950101,0xfb800108,0xbf960101,0xfb800108,0xbf970101,
	0xfb800108,0xbf980101,0xfb800108,0xbf990101,0xfb800108,0xbf9a0101,0xfb800108,0xbf9b0101,
	0xfb800108,0xbf9c0101,0xfb800108,0xbf9d0101,0xfb800108,0xbf9e0101,0xfb800108,0xbf9f0101,
	0xfb800108,0xbfa00101,0xfb800108,0xbfa10101,0xfb800108,0xbfa20101,0xfb800108,0xbfa30101,
	0xfb800108,0xbfa40101,0xfb800108,0xbfa50101,0xfb800108,0xbfa60101,0xfb800108,0xbfa70101,
	0xfb800108,0xbfa80101,0xfb800108,0xbfa90101,0xfb800108,0xbfaa0101,0xfb800108,0xbfab0101,
	0xfb800108,0xbfac0101,0xfb800108,0xbfad0101,0xfb800108,0xbfae0101,0xfb800108,0xbfaf0101,
	0xfb800108,0xbfb00101,0xfb800108,0xbfb10101,0xfb800108,0xbfb20101,0xfb800108,0xbfb30101,
	0xfb800108,0xbfb40101,0xfb800108,0xbfb50101,0xfb800108,0xbfb60101,0xfb800108,0xbfb70101,
	0xfb800108,0xbfb80101,0xfb800108,0xbfb90101,0xfb800108,0xbfba0101,0xfb800108,0xbfbb0101,
	0xfb800108,0xbfbc0101,0xfb800108,0xbfbd0101,0xfb800108,0xbfbe0101,0xfb800108,0xbfbf0101,
	0xfb800108,0xbfc00101,0xfb800108,0xbfc10101,0xfb800108,0xbfc20101,0xfb800108,0xbfc30101,
	0xfb800108,0xbfc40101,0xfb800108,0xbfc50101,0xfb800108,0xbfc60101,0xfb800108,0xbfc70101,
	0xfb800108,0xbfc80101,0xfb800108,0xbfc90101,0xfb800108,0xbfca0101,0xfb800108,0xbfcb0101,
	0xfb800108,0xbfcc0101,0xfb800108,0xbfcd0101,0xfb800108,0xbfce0101,0xfb800108,0xbfcf0101,
	0xfb800108,0xbfd00101,0xfb800108,0xbfd10101,0xfb800108,0xbfd20101,0xfb800108,0xbfd30101,
	0xfb800108,0xbfd40101,0xfb800108,0xbfd50101,0xfb800108,0xbfd60101,0xfb800108,0xbfd70101,
	0xfb800108,0xbfd80101,0xfb800108,0xbfd90101,0xfb800108,0xbfda0101,0xfb800108,0xbfdb0101,
	0xfb800108,0xbfdc0101,0xfb800108,0xbfdd0101,0xfb800108,0xbfde0101,0xfb800108,0xbfdf0101,
	0xfb800108,0xbfe00101,0xfb800108,0xbfe10101,0xfb800108,0xbfe20101,0xfb800108,0xbfe30101,
	0xfb800108,0xbfe40101,0xfb800108,0xbfe50101,0xfb800108,0xbfe60101,0xfb800108,0xbfe70101,
	0xfb800108,0xbfe80101,0xfb800108,0xbfe90101,0xfb800108,0xbfea0101,0xfb800108,0xbfeb0101,
	0xfb800108,0xbfec0101,0xfb800108,0xbfed0101,0xfb800108,0xbfee0101,0xfb800108,0xbfef0101,
	0xfb800108,0xbff00101,0xfb800108,0xbff10101,0xfb800108,0xbff20101,0xfb800108,0xbff30101,
	0xfb800108,0xbff40101,0xfb800108,0xbff50101,0xfb800108,0xbff60101,0xfb800108,0xbff70101,
	0xfb800108,0xbff80101,0xfb800108,0xbff90101,0xfb800108,0xbffa0101,0xfb800108,0xbffb0101,
	0xfb800108,0xbffc0101,0xfb800108,0xbffd0101,0xfb800108,0xbffe0101,0xfb800108,0xbfff0101,
	0xfb800108,0xc0000101,0xfb800108,0xc0010101,0xfb800108,0xc0020101,0xfb800108,0xc0030101,
	0xfb800108,0xc0040101,0xfb800108,0xc0050101,0xfb800108,0xc0060101,0xfb800108,0xc0070101,
	0xfb800108,0xc0080101,0xfb800108,0xc0090101,0xfb800108,0xc00a0101,0xfb800108,0xc00b0101,
	0xfb800108,0xc00c0101,0xfb800108,0xc00d0101,0xfb800108,0xc00e0101,0xfb800108,0xc00f0101,
	0xfb800108,0xc0100101,0xfb800108,0xc0110101,0xfb800108,0xc0120101,0xfb800108,0xc0130101,
	0xfb800108,0xc0140101,0xfb800108,0xc0150101,0xfb800108,0xc0160101,0xfb800108,0xc0170101,
	0xfb800108,0xc0180101,0xfb800108,0xc0190101,0xfb800108,0xc01a0101,0xfb800108,0xc01b0101,
	0xfb800108,0xc01c0101,0xfb800108,0xc01d0101,0xfb800108,0xc01e0101,0xfb800108,0xc01f0101,
	0xfb800108,0xc0200101,0xfb800108,0xc0210101,0xfb800108,0xc0220101,0xfb800108,0xc0230101,
	0xfb800108,0xc0240101,0xfb800108,0xc0250101,0xfb800108,0xc0260101,0xfb800108,0xc0270101,
	0xfb800108,0xc0280101,0xfb800108,0xc0290101,0xfb800108,0xc02a0101,0xfb800108,0xc02b0101,
	0xfb800108,0xc02c0101,0xfb800108,0xc02d0101,0xfb800108,0xc02e0101,0xfb800108,0xc02f0101,
	0xfb800108,0xc0300101,0xfb800108,0xc0310101,0xfb800108,0xc0320101,0xfb800108,0xc0330101,
	0xfb800108,0xc0340101,0xfb800108,0xc0350101,0xfb800108,0xc0360101,0xfb800108,0xc0370101,
	0xfb800108,0xc0380101,0xfb800108,0xc0390101,0xfb800108,0xc03a0101,0xfb800108,0xc03b0101,
	0xfb800108,0xc03c0101,0xfb800108,0xc03d0101,0xfb800108,0xc03e0101,0xfb800108,0xc03f0101,
	0xfb800108,0xc0400101,0xfb800108,0xc0410101,0xfb800108,0xc0420101,0xfb800108,0xc0430101,
	0xfb800108,0xc0440101,0xfb800108,0xc0450101,0xfb800108,0xc0460101,0xfb800108,0xc0470101,
	0xfb800108,0xc0480101,0xfb800108,0xc0490101,0xfb800108,0xc04a0101,0xfb800108,0xc04b0101,
	0xfb800108,0xc04c0101,0xfb800108,0xc04d0101,0xfb800108,0xc04e0101,0xfb800108,0xc04f0101,
	0xfb800108,0xc0500101,0xfb800108,0xc0510101,0xfb800108,0xc0520101,0xfb800108,0xc0530101,
	0xfb800108,0xc0540101,0xfb800108,0xc0550101,0xfb800108,0xc0560101,0xfb800108,0xc0570101,
	0xfb800108,0xc0580101,0xfb800108,0xc0590101,0xfb800108,0xc05a0101,0xfb800108,0xc05b0101,
	0xfb800108,0xc05c0101,0xfb800108,0xc05d0101,0xfb800108,0xc05e0101,0xfb800108,0xc05f0101,
	0xfb800108,0xc0600101,0xfb800108,0xc0610101,0xfb800108,0xc0620101,0xfb800108,0xc0630101,
	0xfb800108,0xc0640101,0xfb800108,0xc0650101,0xfb800108,0xc0660101,0xfb800108,0xc0670101,
	0xfb800108,0xc0680101,0xfb800108,0xc0690101,0xfb800108,0xc06a0101,0xfb800108,0xc06b0101,
	0xfb800108,0xc06c0101,0xfb800108,0xc06d0101,0xfb800108,0xc06e0101,0xfb800108,0xc06f0101,
	0xfb800108,0xc0700101,0xfb800108,0xc0710101,0xfb800108,0xc0720101,0xfb800108,0xc0730101,
	0xfb800108,0xc0740101,0xfb800108,0xc0750101,0xfb800108,0xc0760101,0xfb800108,0xc0770101,
	0xfb800108,0xc0780101,0xfb800108,0xc0790101,0xfb800108,0xc07a0101,0xfb800108,0xc07b0101,
	0xfb800108,0xc07c0101,0xfb800108,0xc07d0101,0xfb800108,0xc07e0101,0xfb800108,0xc07f0101,
	0xfb800108,0xc0800101,0xfb800108,0xc0810101,0xfb800108,0xc0820101,0xfb800108,0xc0830101,
	0xfb800108,0xc0840101,0xfb800108,0xc0850101,0xfb800108,0xc0860101,0xfb800108,0xc0870101,
	0xfb800108,0xc0880101,0xfb800108,0xc0890101,0xfb800108,0xc08a0101,0xfb800108,0xc08b0101,
	0xfb800108,0xc08c0101,0xfb800108,0xc08d0101,0xfb800108,0xc08e0101,0xfb800108,0xc08f0101,
	0xfb800108,0xc0900101,0xfb800108,0xc0910101,0xfb800108,0xc0920101,0xfb800108,0xc0930101,
	0xfb800108,0xc0940101,0xfb800108,0xc0950101,0xfb800108,0xc0960101,0xfb800108,0xc0970101,
	0xfb800108,0xc0980101,0xfb800108,0xc0990101,0xfb800108,0xc09a0101,0xfb800108,0xc09b0101,
	0xfb800108,0xc09c0101,0xfb800108,0xc09d0101,0xfb800108,0xc09e0101,0xfb800108,0xc09f0101,
	0xfb800108,0xc0a00101,0xfb800108,0xc0a10101,0xfb800108,0xc0a20101,0xfb800108,0xc0a30101,
	0xfb800108,0xc0a40101,0xfb800108,0xc0a50101,0xfb800108,0xc0a60101,0xfb800108,0xc0a70101,
	0xfb800108,0xc0a80101,0xfb800108,0xc0a90101,0xfb800108,0xc0aa0101,0xfb800108,0xc0ab0101,
	0xfb800108,0xc0ac0101,0xfb800108,0xc0ad0101,0xfb800108,0xc0ae0101,0xfb800108,0xc0af0101,
	0xfb800108,0xc0b00101,0xfb800108,0xc0b10101,0xfb800108,0xc0b20101,0xfb800108,0xc0b30101,
	0xfb800108,0xc0b40101,0xfb800108,0xc0b50101,0xfb800108,0xc0b60101,0xfb800108,0xc0b70101,
	0xfb800108,0xc0b80101,0xfb800108,0xc0b90101,0xfb800108,0xc0ba0101,0xfb800108,0xc0bb0101,
	0xfb800108,0xc0bc0101,0xfb800108,0xc0bd0101,0xfb800108,0xc0be0101,0xfb800108,0xc0bf0101,
	0xfb800108,0xc0c00101,0xfb800108,0xc0c10101,0xfb800108,0xc0c20101,0xfb800108,0xc0c30101,
	0xfb800108,0xc0c40101,0xfb800108,0xc0c50101,0xfb800108,0xc0c60101,0xfb800108,0xc0c70101,
	0xfb800108,0xc0c80101,0xfb800108,0xc0c90101,0xfb800108,0xc0ca0101,0xfb800108,0xc0cb0101,
	0xfb800108,0xc0cc0101,0xfb800108,0xc0cd0101,0xfb800108,0xc0ce0101,0xfb800108,0xc0cf0101,
	0xfb800108,0xc0d00101,0xfb800108,0xc0d10101,0xfb800108,0xc0d20101,0xfb800108,0xc0d30101,
	0xfb800108,0xc0d40101,0xfb800108,0xc0d50101,0xfb800108,0xc0d60101,0xfb800108,0xc0d70101,
	0xfb800108,0xc0d80101,0xfb800108,0xc0d90101,0xfb800108,0xc0da0101,0xfb800108,0xc0db0101,
	0xfb800108,0xc0dc0101,0xfb800108,0xc0dd0101,0xfb800108,0xc0de0101,0xfb800108,0xc0df0101,
	0xfb800108,0xc0e00101,0xfb800108,0xc0e10101,0xfb800108,0xc0e20101,0xfb800108,0xc0e30101,
	0xfb800108,0xc0e40101,0xfb800108,0xc0e50101,0xfb800108,0xc0e60101,0xfb800108,0xc0e70101,
	0xfb800108,0xc0e80101,0xfb800108,0xc0e90101,0xfb800108,0xc0ea0101,0xfb800108,0xc0eb0101,
	0xfb800108,0xc0ec0101,0xfb800108,0xc0ed0101,0xfb800108,0xc0ee0101,0xfb800108,0xc0ef0101,
	0xfb800108,0xc0f00101,0xfb800108,0xc0f10101,0xfb800108,0xc0f20101,0xfb800108,0xc0f30101,
	0xfb800108,0xc0f40101,0xfb800108,0xc0f50101,0xfb800108,0xc0f60101,0xfb800108,0xc0f70101,
	0xfb800108,0xc0f80101,0xfb800108,0xc0f90101,0xfb800108,0xc0fa0101,0xfb800108,0xc0fb0101,
	0xfb800108,0xc0fc0101,0xfb800108,0xc0fd0101,0xfb800108,0xc0fe0101,0xfb800108,0xc0ff0101,
	0xfb800108,0xc1000101,0xfb800108,0xc1010101,0xfb800108,0xc1020101,0xfb800108,0xc1030101,
	0xfb800108,0xc1040101,0xfb800108,0xc1050101,0xfb800108,0xc1060101,0xfb800108,0xc1070101,
	0xfb800108,0xc1080101,0xfb800108,0xc1090101,0xfb800108,0xc10a0101,0xfb800108,0xc10b0101,
	0xfb800108,0xc10c0101,0xfb800108,0xc10d0101,0xfb800108,0xc10e0101,0xfb800108,0xc10f0101,
	0xfb800108,0xc1100101,0xfb800108,0xc1110101,0xfb800108,0xc1120101,0xfb800108,0xc1130101,
	0xfb800108,0xc1140101,0xfb800108,0xc1150101,0xfb800108,0xc1160101,0xfb800108,0xc1170101,
	0xfb800108,0xc1180101,0xfb800108,0xc1190101,0xfb800108,0xc11a0101,0xfb800108,0xc11b0101,
	0xfb800108,0xc11c0101,0xfb800108,0xc11d0101,0xfb800108,0xc11e0101,0xfb800108,0xc11f0101,
	0xfb800108,0xc1200101,0xfb800108,0xc1210101,0xfb800108,0xc1220101,0xfb800108,0xc1230101,
	0xfb800108,0xc1240101,0xfb800108,0xc1250101,0xfb800108,0xc1260101,0xfb800108,0xc1270101,
	0xfb800108,0xc1280101,0xfb800108,0xc1290101,0xfb800108,0xc12a0101,0xfb800108,0xc12b0101,
	0xfb800108,0xc12c0101,0xfb800108,0xc12d0101,0xfb800108,0xc12e0101,0xfb800108,0xc12f0101,
	0xfb800108,0xc1300101,0xfb800108,0xc1310101,0xfb800108,0xc1320101,0xfb800108,0xc1330101,
	0xfb800108,0xc1340101,0xfb800108,0xc1350101,0xfb800108,0xc1360101,0xfb800108,0xc1370101,
	0xfb800108,0xc1380101,0xfb800108,0xc1390101,0xfb800108,0xc13a0101,0xfb800108,0xc13b0101,
	0xfb800108,0xc13c0101,0xfb800108,0xc13d0101,0xfb800108,0xc13e0101,0xfb800108,0xc13f0101,
	0xfb800108,0xc1400101,0xfb800108,0xc1410101,0xfb800108,0xc1420101,0xfb800108,0xc1430101,
	0xfb800108,0xc1440101,0xfb800108,0xc1450101,0xfb800108,0xc1460101,0xfb800108,0xc1470101,
	0xfb800108,0xc1480101,0xfb800108,0xc1490101,0xfb800108,0xc14a0101,0xfb800108,0xc14b0101,
	0xfb800108,0xc14c0101,0xfb800108,0xc14d0101,0xfb800108,0xc14e0101,0xfb800108,0xc14f0101,
	0xfb800108,0xc1500101,0xfb800108,0xc1510101,0xfb800108,0xc1520101,0xfb800108,0xc1530101,
	0xfb800108,0xc1540101,0xfb800108,0xc1550101,0xfb800108,0xc1560101,0xfb800108,0xc1570101,
	0xfb800108,0xc1580101,0xfb800108,0xc1590101,0xfb800108,0xc15a0101,0xfb800108,0xc15b0101,
	0xfb800108,0xc15c0101,0xfb800108,0xc15d0101,0xfb800108,0xc15e0101,0xfb800108,0xc15f0101,
	0xfb800108,0xc1600101,0xfb800108,0xc1610101,0xfb800108,0xc1620101,0xfb800108,0xc1630101,
	0xfb800108,0xc1640101,0xfb800108,0xc1650101,0xfb800108,0xc1660101,0xfb800108,0xc1670101,
	0xfb800108,0xc1680101,0xfb800108,0xc1690101,0xfb800108,0xc16a0101,0xfb800108,0xc16b0101,
	0xfb800108,0xc16c0101,0xfb800108,0xc16d0101,0xfb800108,0xc16e0101,0xfb800108,0xc16f0101,
	0xfb800108,0xc1700101,0xfb800108,0xc1710101,0xfb800108,0xc1720101,0xfb800108,0xc1730101,
	0xfb800108,0xc1740101,0xfb800108,0xc1750101,0xfb800108,0xc1760101,0xfb800108,0xc1770101,
	0xfb800108,0xc1780101,0xfb800108,0xc1790101,0xfb800108,0xc17a0101,0xfb800108,0xc17b0101,
	0xfb800108,0xc17c0101,0xfb800108,0xc17d0101,0xfb800108,0xc17e0101,0xfb800108,0xc17f0101,
	0xfb800108,0xc1800101,0xfb800108,0xc1810101,0xfb800108,0xc1820101,0xfb800108,0xc1830101,
	0xfb800108,0xc1840101,0xfb800108,0xc1850101,0xfb800108,0xc1860101,0xfb800108,0xc1870101,
	0xfb800108,0xc1880101,0xfb800108,0xc1890101,0xfb800108,0xc18a0101,0xfb800108,0xc18b0101,
	0xfb800108,0xc18c0101,0xfb800108,0xc18d0101,0xfb800108,0xc18e0101,0xfb800108,0xc18f0101,
	0xfb800108,0xc1900101,0xfb800108,0xc1910101,0xfb800108,0xc1920101,0xfb800108,0xc1930101,
	0xfb800108,0xc1940101,0xfb800108,0xc1950101,0xfb800108,0xc1960101,0xfb800108,0xc1970101,
	0xfb800108,0xc1980101,0xfb800108,0xc1990101,0xfb800108,0xc19a0101,0xfb800108,0xc19b0101,
	0xfb800108,0xc19c0101,0xfb800108,0xc19d0101,0xfb800108,0xc19e0101,0xfb800108,0xc19f0101,
	0xfb800108,0xc1a00101,0xfb800108,0xc1a10101,0xfb800108,0xc1a20101,0xfb800108,0xc1a30101,
	0xfb800108,0xc1a40101,0xfb800108,0xc1a50101,0xfb800108,0xc1a60101,0xfb800108,0xc1a70101,
	0xfb800108,0xc1a80101,0xfb800108,0xc1a90101,0xfb800108,0xc1aa0101,0xfb800108,0xc1ab0101,
	0xfb800108,0xc1ac0101,0xfb800108,0xc1ad0101,0xfb800108,0xc1ae0101,0xfb800108,0xc1af0101,
	0xfb800108,0xc1b00101,0xfb800108,0xc1b10101,0xfb800108,0xc1b20101,0xfb800108,0xc1b30101,
	0xfb800108,0xc1b40101,0xfb800108,0xc1b50101,0xfb800108,0xc1b60101,0xfb800108,0xc1b70101,
	0xfb800108,0xc1b80101,0xfb800108,0xc1b90101,0xfb800108,0xc1ba0101,0xfb800108,0xc1bb0101,
	0xfb800108,0xc1bc0101,0xfb800108,0xc1bd0101,0xfb800108,0xc1be0101,0xfb800108,0xc1bf0101,
	0xfb800108,0xc1c00101,0xfb800108,0xc1c10101,0xfb800108,0xc1c20101,0xfb800108,0xc1c30101,
	0xfb800108,0xc1c40101,0xfb800108,0xc1c50101,0xfb800108,0xc1c60101,0xfb800108,0xc1c70101,
	0xfb800108,0xc1c80101,0xfb800108,0xc1c90101,0xfb800108,0xc1ca0101,0xfb800108,0xc1cb0101,
	0xfb800108,0xc1cc0101,0xfb800108,0xc1cd0101,0xfb800108,0xc1ce0101,0xfb800108,0xc1cf0101,
	0xfb800108,0xc1d00101,0xfb800108,0xc1d10101,0xfb800108,0xc1d20101,0xfb800108,0xc1d30101,
	0xfb800108,0xc1d40101,0xfb800108,0xc1d50101,0xfb800108,0xc1d60101,0xfb800108,0xc1d70101,
	0xfb800108,0xc1d80101,0xfb800108,0xc1d90101,0xfb800108,0xc1da0101,0xfb800108,0xc1db0101,
	0xfb800108,0xc1dc0101,0xfb800108,0xc1dd0101,0xfb800108,0xc1de0101,0xfb800108,0xc1df0101,
	0xfb800108,0xc1e00101,0xfb800108,0xc1e10101,0xfb800108,0xc1e20101,0xfb800108,0xc1e30101,
	0xfb800108,0xc1e40101,0xfb800108,0xc1e50101,0xfb800108,0xc1e60101,0xfb800108,0xc1e70101,
	0xfb800108,0xc1e80101,0xfb800108,0xc1e90101,0xfb800108,0xc1ea0101,0xfb800108,0xc1eb0101,
	0xfb800108,0xc1ec0101,0xfb800108,0xc1ed0101,0xfb800108,0xc1ee0101,0xfb800108,0xc1ef0101,
	0xfb800108,0xc1f00101,0xfb800108,0xc1f10101,0xfb800108,0xc1f20101,0xfb800108,0xc1f30101,
	0xfb800108,0xc1f40101,0xfb800108,0xc1f50101,0xfb800108,0xc1f60101,0xfb800108,0xc1f70101,
	0xfb800108,0xc1f80101,0xfb800108,0xc1f90101,0xfb800108,0xc1fa0101,0xfb800108,0xc1fb0101,
	0xfb800108,0xc1fc0101,0xfb800108,0xc1fd0101,0xfb800108,0xc1fe0101,0xfb800108,0xc1ff0101,
	0xfb800108,0xc2000101,0xfb800108,0xc2010101,0xfb800108,0xc2020101,0xfb800108,0xc2030101,
	0xfb800108,0xc2040101,0xfb800108,0xc2050101,0xfb800108,0xc2060101,0xfb800108,0xc2070101,
	0xfb800108,0xc2080101,0xfb800108,0xc2090101,0xfb800108,0xc20a0101,0xfb800108,0xc20b0101,
	0xfb800108,0xc20c0101,0xfb800108,0xc20d0101,0xfb800108,0xc20e0101,0xfb800108,0xc20f0101,
	0xfb800108,0xc2100101,0xfb800108,0xc2110101,0xfb800108,0xc2120101,0xfb800108,0xc2130101,
	0xfb800108,0xc2140101,0xfb800108,0xc2150101,0xfb800108,0xc2160101,0xfb800108,0xc2170101,
	0xfb800108,0xc2180101,0xfb800108,0xc2190101,0xfb800108,0xc21a0101,0xfb800108,0xc21b0101,
	0xfb800108,0xc21c0101,0xfb800108,0xc21d0101,0xfb800108,0xc21e0101,0xfb800108,0xc21f0101,
	0xfb800108,0xc2200101,0xfb800108,0xc2210101,0xfb800108,0xc2220101,0xfb800108,0xc2230101,
	0xfb800108,0xc2240101,0xfb800108,0xc2250101,0xfb800108,0xc2260101,0xfb800108,0xc2270101,
	0xfb800108,0xc2280101,0xfb800108,0xc2290101,0xfb800108,0xc22a0101,0xfb800108,0xc22b0101,
	0xfb800108,0xc22c0101,0xfb800108,0xc22d0101,0xfb800108,0xc22e0101,0xfb800108,0xc22f0101,
	0xfb800108,0xc2300101,0xfb800108,0xc2310101,0xfb800108,0xc2320101,0xfb800108,0xc2330101,
	0xfb800108,0xc2340101,0xfb800108,0xc2350101,0xfb800108,0xc2360101,0xfb800108,0xc2370101,
	0xfb800108,0xc2380101,0xfb800108,0xc2390101,0xfb800108,0xc23a0101,0xfb800108,0xc23b0101,
	0xfb800108,0xc23c0101,0xfb800108,0xc23d0101,0xfb800108,0xc23e0101,0xfb800108,0xc23f0101,
	0xfb800108,0xc2400101,0xfb800108,0xc2410101,0xfb800108,0xc2420101,0xfb800108,0xc2430101,
	0xfb800108,0xc2440101,0xfb800108,0xc2450101,0xfb800108,0xc2460101,0xfb800108,0xc2470101,
	0xfb800108,0xc2480101,0xfb800108,0xc2490101,0xfb800108,0xc24a0101,0xfb800108,0xc24b0101,
	0xfb800108,0xc24c0101,0xfb800108,0xc24d0101,0xfb800108,0xc24e0101,0xfb800108,0xc24f0101,
	0xfb800108,0xc2500101,0xfb800108,0xc2510101,0xfb800108,0xc2520101,0xfb800108,0xc2530101,
	0xfb800108,0xc2540101,0xfb800108,0xc2550101,0xfb800108,0xc2560101,0xfb800108,0xc2570101,
	0xfb800108,0xc2580101,0xfb800108,0xc2590101,0xfb800108,0xc25a0101,0xfb800108,0xc25b0101,
	0xfb800108,0xc25c0101,0xfb800108,0xc25d0101,0xfb800108,0xc25e0101,0xfb800108,0xc25f0101,
	0xfb800108,0xc2600101,0xfb800108,0xc2610101,0xfb800108,0xc2620101,0xfb800108,0xc2630101,
	0xfb800108,0xc2640101,0xfb800108,0xc2650101,0xfb800108,0xc2660101,0xfb800108,0xc2670101,
	0xfb800108,0xc2680101,0xfb800108,0xc2690101,0xfb800108,0xc26a0101,0xfb800108,0xc26b0101,
	0xfb800108,0xc26c0101,0xfb800108,0xc26d0101,0xfb800108,0xc26e0101,0xfb800108,0xc26f0101,
	0xfb800108,0xc2700101,0xfb800108,0xc2710101,0xfb800108,0xc2720101,0xfb800108,0xc2730101,
	0xfb800108,0xc2740101,0xfb800108,0xc2750101,0xfb800108,0xc2760101,0xfb800108,0xc2770101,
	0xfb800108,0xc2780101,0xfb800108,0xc2790101,0xfb800108,0xc27a0101,0xfb800108,0xc27b0101,
	0xfb800108,0xc27c0101,0xfb800108,0xc27d0101,0xfb800108,0xc27e0101,0xfb800108,0xc27f0101,
	0xfb800108,0xc2800101,0xfb800108,0xc2810101,0xfb800108,0xc2820101,0xfb800108,0xc2830101,
	0xfb800108,0xc2840101,0xfb800108,0xc2850101,0xfb800108,0xc2860101,0xfb800108,0xc2870101,
	0xfb800108,0xc2880101,0xfb800108,0xc2890101,0xfb800108,0xc28a0101,0xfb800108,0xc28b0101,
	0xfb800108,0xc28c0101,0xfb800108,0xc28d0101,0xfb800108,0xc28e0101,0xfb800108,0xc28f0101,
	0xfb800108,0xc2900101,0xfb800108,0xc2910101,0xfb800108,0xc2920101,0xfb800108,0xc2930101,
	0xfb800108,0xc2940101,0xfb800108,0xc2950101,0xfb800108,0xc2960101,0xfb800108,0xc2970101,
	0xfb800108,0xc2980101,0xfb800108,0xc2990101,0xfb800108,0xc29a0101,0xfb800108,0xc29b0101,
	0xfb800108,0xc29c0101,0xfb800108,0xc29d0101,0xfb800108,0xc29e0101,0xfb800108,0xc29f0101,
	0xfb800108,0xc2a00101,0xfb800108,0xc2a10101,0xfb800108,0xc2a20101,0xfb800108,0xc2a30101,
	0xfb800108,0xc2a40101,0xfb800108,0xc2a50101,0xfb800108,0xc2a60101,0xfb800108,0xc2a70101,
	0xfb800108,0xc2a80101,0xfb800108,0xc2a90101,0xfb800108,0xc2aa0101,0xfb800108,0xc2ab0101,
	0xfb800108,0xc2ac0101,0xfb800108,0xc2ad0101,0xfb800108,0xc2ae0101,0xfb800108,0xc2af0101,
	0xfb800108,0xc2b00101,0xfb800108,0xc2b10101,0xfb800108,0xc2b20101,0xfb800108,0xc2b30101,
	0xfb800108,0xc2b40101,0xfb800108,0xc2b50101,0xfb800108,0xc2b60101,0xfb800108,0xc2b70101,
	0xfb800108,0xc2b80101,0xfb800108,0xc2b90101,0xfb800108,0xc2ba0101,0xfb800108,0xc2bb0101,
	0xfb800108,0xc2bc0101,0xfb800108,0xc2bd0101,0xfb800108,0xc2be0101,0xfb800108,0xc2bf0101,
	0xfb800108,0xc2c00101,0xfb800108,0xc2c10101,0xfb800108,0xc2c20101,0xfb800108,0xc2c30101,
	0xfb800108,0xc2c40101,0xfb800108,0xc2c50101,0xfb800108,0xc2c60101,0xfb800108,0xc2c70101,
	0xfb800108,0xc2c80101,0xfb800108,0xc2c90101,0xfb800108,0xc2ca0101,0xfb800108,0xc2cb0101,
	0xfb800108,0xc2cc0101,0xfb800108,0xc2cd0101,0xfb800108,0xc2ce0101,0xfb800108,0xc2cf0101,
	0xfb800108,0xc2d00101,0xfb800108,0xc2d10101,0xfb800108,0xc2d20101,0xfb800108,0xc2d30101,
	0xfb800108,0xc2d40101,0xfb800108,0xc2d50101,0xfb800108,0xc2d60101,0xfb800108,0xc2d70101,
	0xfb800108,0xc2d80101,0xfb800108,0xc2d90101,0xfb800108,0xc2da0101,0xfb800108,0xc2db0101,
	0xfb800108,0xc2dc0101,0xfb800108,0xc2dd0101,0xfb800108,0xc2de0101,0xfb800108,0xc2df0101,
	0xfb800108,0xc2e00101,0xfb800108,0xc2e10101,0xfb800108,0xc2e20101,0xfb800108,0xc2e30101,
	0xfb800108,0xc2e40101,0xfb800108,0xc2e50101,0xfb800108,0xc2e60101,0xfb800108,0xc2e70101,
	0xfb800108,0xc2e80101,0xfb800108,0xc2e90101,0xfb800108,0xc2ea0101,0xfb800108,0xc2eb0101,
	0xfb800108,0xc2ec0101,0xfb800108,0xc2ed0101,0xfb800108,0xc2ee0101,0xfb800108,0xc2ef0101,
	0xfb800108,0xc2f00101,0xfb800108,0xc2f10101,0xfb800108,0xc2f20101,0xfb800108,0xc2f30101,
	0xfb800108,0xc2f40101,0xfb800108,0xc2f50101,0xfb800108,0xc2f60101,0xfb800108,0xc2f70101,
	0xfb800108,0xc2f80101,0xfb800108,0xc2f90101,0xfb800108,0xc2fa0101,0xfb800108,0xc2fb0101,
	0xfb800108,0xc2fc0101,0xfb800108,0xc2fd0101,0xfb800108,0xc2fe0101,0xfb800108,0xc2ff0101,
	0xfb800108,0xc3000101,0xfb800108,0xc3010101,0xfb800108,0xc3020101,0xfb800108,0xc3030101,
	0xfb800108,0xc3040101,0xfb800108,0xc3050101,0xfb800108,0xc3060101,0xfb800108,0xc3070101,
	0xfb800108,0xc3080101,0xfb800108,0xc3090101,0xfb800108,0xc30a0101,0xfb800108,0xc30b0101,
	0xfb800108,0xc30c0101,0xfb800108,0xc30d0101,0xfb800108,0xc30e0101,0xfb800108,0xc30f0101,
	0xfb800108,0xc3100101,0xfb800108,0xc3110101,0xfb800108,0xc3120101,0xfb800108,0xc3130101,
	0xfb800108,0xc3140101,0xfb800108,0xc3150101,0xfb800108,0xc3160101,0xfb800108,0xc3170101,
	0xfb800108,0xc3180101,0xfb800108,0xc3190101,0xfb800108,0xc31a0101,0xfb800108,0xc31b0101,
	0xfb800108,0xc31c0101,0xfb800108,0xc31d0101,0xfb800108,0xc31e0101,0xfb800108,0xc31f0101,
	0xfb800108,0xc3200101,0xfb800108,0xc3210101,0xfb800108,0xc3220101,0xfb800108,0xc3230101,
	0xfb800108,0xc3240101,0xfb800108,0xc3250101,0xfb800108,0xc3260101,0xfb800108,0xc3270101,
	0xfb800108,0xc3280101,0xfb800108,0xc3290101,0xfb800108,0xc32a0101,0xfb800108,0xc32b0101,
	0xfb800108,0xc32c0101,0xfb800108,0xc32d0101,0xfb800108,0xc32e0101,0xfb800108,0xc32f0101,
	0xfb800108,0xc3300101,0xfb800108,0xc3310101,0xfb800108,0xc3320101,0xfb800108,0xc3330101,
	0xfb800108,0xc3340101,0xfb800108,0xc3350101,0xfb800108,0xc3360101,0xfb800108,0xc3370101,
	0xfb800108,0xc3380101,0xfb800108,0xc3390101,0xfb800108,0xc33a0101,0xfb800108,0xc33b0101,
	0xfb800108,0xc33c0101,0xfb800108,0xc33d0101,0xfb800108,0xc33e0101,0xfb800108,0xc33f0101,
	0xfb800108,0xc3400101,0xfb800108,0xc3410101,0xfb800108,0xc3420101,0xfb800108,0xc3430101,
	0xfb800108,0xc3440101,0xfb800108,0xc3450101,0xfb800108,0xc3460101,0xfb800108,0xc3470101,
	0xfb800108,0xc3480101,0xfb800108,0xc3490101,0xfb800108,0xc34a0101,0xfb800108,0xc34b0101,
	0xfb800108,0xc34c0101,0xfb800108,0xc34d0101,0xfb800108,0xc34e0101,0xfb800108,0xc34f0101,
	0xfb800108,0xc3500101,0xfb800108,0xc3510101,0xfb800108,0xc3520101,0xfb800108,0xc3530101,
	0xfb800108,0xc3540101,0xfb800108,0xc3550101,0xfb800108,0xc3560101,0xfb800108,0xc3570101,
	0xfb800108,0xc3580101,0xfb800108,0xc3590101,0xfb800108,0xc35a0101,0xfb800108,0xc35b0101,
	0xfb800108,0xc35c0101,0xfb800108,0xc35d0101,0xfb800108,0xc35e0101,0xfb800108,0xc35f0101,
	0xfb800108,0xc3600101,0xfb800108,0xc3610101,0xfb800108,0xc3620101,0xfb800108,0xc3630101,
	0xfb800108,0xc3640101,0xfb800108,0xc3650101,0xfb800108,0xc3660101,0xfb800108,0xc3670101,
	0xfb800108,0xc3680101,0xfb800108,0xc3690101,0xfb800108,0xc36a0101,0xfb800108,0xc36b0101,
	0xfb800108,0xc36c0101,0xfb800108,0xc36d0101,0xfb800108,0xc36e0101,0xfb800108,0xc36f0101,
	0xfb800108,0xc3700101,0xfb800108,0xc3710101,0xfb800108,0xc3720101,0xfb800108,0xc3730101,
	0xfb800108,0xc3740101,0xfb800108,0xc3750101,0xfb800108,0xc3760101,0xfb800108,0xc3770101,
	0xfb800108,0xc3780101,0xfb800108,0xc3790101,0xfb800108,0xc37a0101,0xfb800108,0xc37b0101,
	0xfb800108,0xc37c0101,0xfb800108,0xc37d0101,0xfb800108,0xc37e0101,0xfb800108,0xc37f0101,
	0xfb800108,0xc3800101,0xfb800108,0xc3810101,0xfb800108,0xc3820101,0xfb800108,0xc3830101,
	0xfb800108,0xc3840101,0xfb800108,0xc3850101,0xfb800108,0xc3860101,0xfb800108,0xc3870101,
	0xfb800108,0xc3880101,0xfb800108,0xc3890101,0xfb800108,0xc38a0101,0xfb800108,0xc38b0101,
	0xfb800108,0xc38c0101,0xfb800108,0xc38d0101,0xfb800108,0xc38e0101,0xfb800108,0xc38f0101,
	0xfb800108,0xc3900101,0xfb800108,0xc3910101,0xfb800108,0xc3920101,0xfb800108,0xc3930101,
	0xfb800108,0xc3940101,0xfb800108,0xc3950101,0xfb800108,0xc3960101,0xfb800108,0xc3970101,
	0xfb800108,0xc3980101,0xfb800108,0xc3990101,0xfb800108,0xc39a0101,0xfb800108,0xc39b0101,
	0xfb800108,0xc39c0101,0xfb800108,0xc39d0101,0xfb800108,0xc39e0101,0xfb800108,0xc39f0101,
	0xfb800108,0xc3a00101,0xfb800108,0xc3a10101,0xfb800108,0xc3a20101,0xfb800108,0xc3a30101,
	0xfb800108,0xc3a40101,0xfb800108,0xc3a50101,0xfb800108,0xc3a60101,0xfb800108,0xc3a70101,
	0xfb800108,0xc3a80101,0xfb800108,0xc3a90101,0xfb800108,0xc3aa0101,0xfb800108,0xc3ab0101,
	0xfb800108,0xc3ac0101,0xfb800108,0xc3ad0101,0xfb800108,0xc3ae0101,0xfb800108,0xc3af0101,
	0xfb800108,0xc3b00101,0xfb800108,0xc3b10101,0xfb800108,0xc3b20101,0xfb800108,0xc3b30101,
	0xfb800108,0xc3b40101,0xfb800108,0xc3b50101,0xfb800108,0xc3b60101,0xfb800108,0xc3b70101,
	0xfb800108,0xc3b80101,0xfb800108,0xc3b90101,0xfb800108,0xc3ba0101,0xfb800108,0xc3bb0101,
	0xfb800108,0xc3bc0101,0xfb800108,0xc3bd0101,0xfb800108,0xc3be0101,0xfb800108,0xc3bf0101,
	0xfb800108,0xc3c00101,0xfb800108,0xc3c10101,0xfb800108,0xc3c20101,0xfb800108,0xc3c30101,
	0xfb800108,0xc3c40101,0xfb800108,0xc3c50101,0xfb800108,0xc3c60101,0xfb800108,0xc3c70101,
	0xfb800108,0xc3c80101,0xfb800108,0xc3c90101,0xfb800108,0xc3ca0101,0xfb800108,0xc3cb0101,
	0xfb800108,0xc3cc0101,0xfb800108,0xc3cd0101,0xfb800108,0xc3ce0101,0xfb800108,0xc3cf0101,
	0xfb800108,0xc3d00101,0xfb800108,0xc3d10101,0xfb800108,0xc3d20101,0xfb800108,0xc3d30101,
	0xfb800108,0xc3d40101,0xfb800108,0xc3d50101,0xfb800108,0xc3d60101,0xfb800108,0xc3d70101,
	0xfb800108,0xc3d80101,0xfb800108,0xc3d90101,0xfb800108,0xc3da0101,0xfb800108,0xc3db0101,
	0xfb800108,0xc3dc0101,0xfb800108,0xc3dd0101,0xfb800108,0xc3de0101,0xfb800108,0xc3df0101,
	0xfb800108,0xc3e00101,0xfb800108,0xc3e10101,0xfb800108,0xc3e20101,0xfb800108,0xc3e30101,
	0xfb800108,0xc3e40101,0xfb800108,0xc3e50101,0xfb800108,0xc3e60101,0xfb800108,0xc3e70101,
	0xfb800108,0xc3e80101,0xfb800108,0xc3e90101,0xfb800108,0xc3ea0101,0xfb800108,0xc3eb0101,
	0xfb800108,0xc3ec0101,0xfb800108,0xc3ed0101,0xfb800108,0xc3ee0101,0xfb800108,0xc3ef0101,
	0xfb800108,0xc3f00101,0xfb800108,0xc3f10101,0xfb800108,0xc3f20101,0xfb800108,0xc3f30101,
	0xfb800108,0xc3f40101,0xfb800108,0xc3f50101,0xfb800108,0xc3f60101,0xfb800108,0xc3f70101,
	0xfb800108,0xc3f80101,0xfb800108,0xc3f90101,0xfb800108,0xc3fa0101,0xfb800108,0xc3fb0101,
	0xfb800108,0xc3fc0101,0xfb800108,0xc3fd0101,0xfb800108,0xc3fe0101,0xfb800108,0xc3ff0101,
	0xfb800108,0xc4000101,0xfb800108,0xc4010101,0xfb800108,0xc4020101,0xfb800108,0xc4030101,
	0xfb800108,0xc4040101,0xfb800108,0xc4050101,0xfb800108,0xc4060101,0xfb800108,0xc4070101,
	0xfb800108,0xc4080101,0xfb800108,0xc4090101,0xfb800108,0xc40a0101,0xfb800108,0xc40b0101,
	0xfb800108,0xc40c0101,0xfb800108,0xc40d0101,0xfb800108,0xc40e0101,0xfb800108,0xc40f0101,
	0xfb800108,0xc4100101,0xfb800108,0xc4110101,0xfb800108,0xc4120101,0xfb800108,0xc4130101,
	0xfb800108,0xc4140101,0xfb800108,0xc4150101,0xfb800108,0xc4160101,0xfb800108,0xc4170101,
	0xfb800108,0xc4180101,0xfb800108,0xc4190101,0xfb800108,0xc41a0101,0xfb800108,0xc41b0101,
	0xfb800108,0xc41c0101,0xfb800108,0xc41d0101,0xfb800108,0xc41e0101,0xfb800108,0xc41f0101,
	0xfb800108,0xc4200101,0xfb800108,0xc4210101,0xfb800108,0xc4220101,0xfb800108,0xc4230101,
	0xfb800108,0xc4240101,0xfb800108,0xc4250101,0xfb800108,0xc4260101,0xfb800108,0xc4270101,
	0xfb800108,0xc4280101,0xfb800108,0xc4290101,0xfb800108,0xc42a0101,0xfb800108,0xc42b0101,
	0xfb800108,0xc42c0101,0xfb800108,0xc42d0101,0xfb800108,0xc42e0101,0xfb800108,0xc42f0101,
	0xfb800108,0xc4300101,0xfb800108,0xc4310101,0xfb800108,0xc4320101,0xfb800108,0xc4330101,
	0xfb800108,0xc4340101,0xfb800108,0xc4350101,0xfb800108,0xc4360101,0xfb800108,0xc4370101,
	0xfb800108,0xc4380101,0xfb800108,0xc4390101,0xfb800108,0xc43a0101,0xfb800108,0xc43b0101,
	0xfb800108,0xc43c0101,0xfb800108,0xc43d0101,0xfb800108,0xc43e0101,0xfb800108,0xc43f0101,
	0xfb800108,0xc4400101,0xfb800108,0xc4410101,0xfb800108,0xc4420101,0xfb800108,0xc4430101,
	0xfb800108,0xc4440101,0xfb800108,0xc4450101,0xfb800108,0xc4460101,0xfb800108,0xc4470101,
	0xfb800108,0xc4480101,0xfb800108,0xc4490101,0xfb800108,0xc44a0101,0xfb800108,0xc44b0101,
	0xfb800108,0xc44c0101,0xfb800108,0xc44d0101,0xfb800108,0xc44e0101,0xfb800108,0xc44f0101,
	0xfb800108,0xc4500101,0xfb800108,0xc4510101,0xfb800108,0xc4520101,0xfb800108,0xc4530101,
	0xfb800108,0xc4540101,0xfb800108,0xc4550101,0xfb800108,0xc4560101,0xfb800108,0xc4570101,
	0xfb800108,0xc4580101,0xfb800108,0xc4590101,0xfb800108,0xc45a0101,0xfb800108,0xc45b0101,
	0xfb800108,0xc45c0101,0xfb800108,0xc45d0101,0xfb800108,0xc45e0101,0xfb800108,0xc45f0101,
	0xfb800108,0xc4600101,0xfb800108,0xc4610101,0xfb800108,0xc4620101,0xfb800108,0xc4630101,
	0xfb800108,0xc4640101,0xfb800108,0xc4650101,0xfb800108,0xc4660101,0xfb800108,0xc4670101,
	0xfb800108,0xc4680101,0xfb800108,0xc4690101,0xfb800108,0xc46a0101,0xfb800108,0xc46b0101,
	0xfb800108,0xc46c0101,0xfb800108,0xc46d0101,0xfb800108,0xc46e0101,0xfb800108,0xc46f0101,
	0xfb800108,0xc4700101,0xfb800108,0xc4710101,0xfb800108,0xc4720101,0xfb800108,0xc4730101,
	0xfb800108,0xc4740101,0xfb800108,0xc4750101,0xfb800108,0xc4760101,0xfb800108,0xc4770101,
	0xfb800108,0xc4780101,0xfb800108,0xc4790101,0xfb800108,0xc47a0101,0xfb800108,0xc47b0101,
	0xfb800108,0xc47c0101,0xfb800108,0xc47d0101,0xfb800108,0xc47e0101,0xfb800108,0xc47f0101,
	0xfb800108,0xc4800101,0xfb800108,0xc4810101,0xfb800108,0xc4820101,0xfb800108,0xc4830101,
	0xfb800108,0xc4840101,0xfb800108,0xc4850101,0xfb800108,0xc4860101,0xfb800108,0xc4870101,
	0xfb800108,0xc4880101,0xfb800108,0xc4890101,0xfb800108,0xc48a0101,0xfb800108,0xc48b0101,
	0xfb800108,0xc48c0101,0xfb800108,0xc48d0101,0xfb800108,0xc48e0101,0xfb800108,0xc48f0101,
	0xfb800108,0xc4900101,0xfb800108,0xc4910101,0xfb800108,0xc4920101,0xfb800108,0xc4930101,
	0xfb800108,0xc4940101,0xfb800108,0xc4950101,0xfb800108,0xc4960101,0xfb800108,0xc4970101,
	0xfb800108,0xc4980101,0xfb800108,0xc4990101,0xfb800108,0xc49a0101,0xfb800108,0xc49b0101,
	0xfb800108,0xc49c0101,0xfb800108,0xc49d0101,0xfb800108,0xc49e0101,0xfb800108,0xc49f0101,
	0xfb800108,0xc4a00101,0xfb800108,0xc4a10101,0xfb800108,0xc4a20101,0xfb800108,0xc4a30101,
	0xfb800108,0xc4a40101,0xfb800108,0xc4a50101,0xfb800108,0xc4a60101,0xfb800108,0xc4a70101,
	0xfb800108,0xc4a80101,0xfb800108,0xc4a90101,0xfb800108,0xc4aa0101,0xfb800108,0xc4ab0101,
	0xfb800108,0xc4ac0101,0xfb800108,0xc4ad0101,0xfb800108,0xc4ae0101,0xfb800108,0xc4af0101,
	0xfb800108,0xc4b00101,0xfb800108,0xc4b10101,0xfb800108,0xc4b20101,0xfb800108,0xc4b30101,
	0xfb800108,0xc4b40101,0xfb800108,0xc4b50101,0xfb800108,0xc4b60101,0xfb800108,0xc4b70101,
	0xfb800108,0xc4b80101,0xfb800108,0xc4b90101,0xfb800108,0xc4ba0101,0xfb800108,0xc4bb0101,
	0xfb800108,0xc4bc0101,0xfb800108,0xc4bd0101,0xfb800108,0xc4be0101,0xfb800108,0xc4bf0101,
	0xfb800108,0xc4c00101,0xfb800108,0xc4c10101,0xfb800108,0xc4c20101,0xfb800108,0xc4c30101,
	0xfb800108,0xc4c40101,0xfb800108,0xc4c50101,0xfb800108,0xc4c60101,0xfb800108,0xc4c70101,
	0xfb800108,0xc4c80101,0xfb800108,0xc4c90101,0xfb800108,0xc4ca0101,0xfb800108,0xc4cb0101,
	0xfb800108,0xc4cc0101,0xfb800108,0xc4cd0101,0xfb800108,0xc4ce0101,0xfb800108,0xc4cf0101,
	0xfb800108,0xc4d00101,0xfb800108,0xc4d10101,0xfb800108,0xc4d20101,0xfb800108,0xc4d30101,
	0xfb800108,0xc4d40101,0xfb800108,0xc4d50101,0xfb800108,0xc4d60101,0xfb800108,0xc4d70101,
	0xfb800108,0xc4d80101,0xfb800108,0xc4d90101,0xfb800108,0xc4da0101,0xfb800108,0xc4db0101,
	0xfb800108,0xc4dc0101,0xfb800108,0xc4dd0101,0xfb800108,0xc4de0101,0xfb800108,0xc4df0101,
	0xfb800108,0xc4e00101,0xfb800108,0xc4e10101,0xfb800108,0xc4e20101,0xfb800108,0xc4e30101,
	0xfb800108,0xc4e40101,0xfb800108,0xc4e50101,0xfb800108,0xc4e60101,0xfb800108,0xc4e70101,
	0xfb800108,0xc4e80101,0xfb800108,0xc4e90101,0xfb800108,0xc4ea0101,0xfb800108,0xc4eb0101,
	0xfb800108,0xc4ec0101,0xfb800108,0xc4ed0101,0xfb800108,0xc4ee0101,0xfb800108,0xc4ef0101,
	0xfb800108,0xc4f00101,0xfb800108,0xc4f10101,0xfb800108,0xc4f20101,0xfb800108,0xc4f30101,
	0xfb800108,0xc4f40101,0xfb800108,0xc4f50101,0xfb800108,0xc4f60101,0xfb800108,0xc4f70101,
	0xfb800108,0xc4f80101,0xfb800108,0xc4f90101,0xfb800108,0xc4fa0101,0xfb800108,0xc4fb0101,
	0xfb800108,0xc4fc0101,0xfb800108,0xc4fd0101,0xfb800108,0xc4fe0101,0xfb800108,0xc4ff0101,
	0xfb800108,0xc5000101,0xfb800108,0xc5010101,0xfb800108,0xc5020101,0xfb800108,0xc5030101,
	0xfb800108,0xc5040101,0xfb800108,0xc5050101,0xfb800108,0xc5060101,0xfb800108,0xc5070101,
	0xfb800108,0xc5080101,0xfb800108,0xc5090101,0xfb800108,0xc50a0101,0xfb800108,0xc50b0101,
	0xfb800108,0xc50c0101,0xfb800108,0xc50d0101,0xfb800108,0xc50e0101,0xfb800108,0xc50f0101,
	0xfb800108,0xc5100101,0xfb800108,0xc5110101,0xfb800108,0xc5120101,0xfb800108,0xc5130101,
	0xfb800108,0xc5140101,0xfb800108,0xc5150101,0xfb800108,0xc5160101,0xfb800108,0xc5170101,
	0xfb800108,0xc5180101,0xfb800108,0xc5190101,0xfb800108,0xc51a0101,0xfb800108,0xc51b0101,
	0xfb800108,0xc51c0101,0xfb800108,0xc51d0101,0xfb800108,0xc51e0101,0xfb800108,0xc51f0101,
	0xfb800108,0xc5200101,0xfb800108,0xc5210101,0xfb800108,0xc5220101,0xfb800108,0xc5230101,
	0xfb800108,0xc5240101,0xfb800108,0xc5250101,0xfb800108,0xc5260101,0xfb800108,0xc5270101,
	0xfb800108,0xc5280101,0xfb800108,0xc5290101,0xfb800108,0xc52a0101,0xfb800108,0xc52b0101,
	0xfb800108,0xc52c0101,0xfb800108,0xc52d0101,0xfb800108,0xc52e0101,0xfb800108,0xc52f0101,
	0xfb800108,0xc5300101,0xfb800108,0xc5310101,0xfb800108,0xc5320101,0xfb800108,0xc5330101,
	0xfb800108,0xc5340101,0xfb800108,0xc5350101,0xfb800108,0xc5360101,0xfb800108,0xc5370101,
	0xfb800108,0xc5380101,0xfb800108,0xc5390101,0xfb800108,0xc53a0101,0xfb800108,0xc53b0101,
	0xfb800108,0xc53c0101,0xfb800108,0xc53d0101,0xfb800108,0xc53e0101,0xfb800108,0xc53f0101,
	0xfb800108,0xc5400101,0xfb800108,0xc5410101,0xfb800108,0xc5420101,0xfb800108,0xc5430101,
	0xfb800108,0xc5440101,0xfb800108,0xc5450101,0xfb800108,0xc5460101,0xfb800108,0xc5470101,
	0xfb800108,0xc5480101,0xfb800108,0xc5490101,0xfb800108,0xc54a0101,0xfb800108,0xc54b0101,
	0xfb800108,0xc54c0101,0xfb800108,0xc54d0101,0xfb800108,0xc54e0101,0xfb800108,0xc54f0101,
	0xfb800108,0xc5500101,0xfb800108,0xc5510101,0xfb800108,0xc5520101,0xfb800108,0xc5530101,
	0xfb800108,0xc5540101,0xfb800108,0xc5550101,0xfb800108,0xc5560101,0xfb800108,0xc5570101,
	0xfb800108,0xc5580101,0xfb800108,0xc5590101,0xfb800108,0xc55a0101,0xfb800108,0xc55b0101,
	0xfb800108,0xc55c0101,0xfb800108,0xc55d0101,0xfb800108,0xc55e0101,0xfb800108,0xc55f0101,
	0xfb800108,0xc5600101,0xfb800108,0xc5610101,0xfb800108,0xc5620101,0xfb800108,0xc5630101,
	0xfb800108,0xc5640101,0xfb800108,0xc5650101,0xfb800108,0xc5660101,0xfb800108,0xc5670101,
	0xfb800108,0xc5680101,0xfb800108,0xc5690101,0xfb800108,0xc56a0101,0xfb800108,0xc56b0101,
	0xfb800108,0xc56c0101,0xfb800108,0xc56d0101,0xfb800108,0xc56e0101,0xfb800108,0xc56f0101,
	0xfb800108,0xc5700101,0xfb800108,0xc5710101,0xfb800108,0xc5720101,0xfb800108,0xc5730101,
	0xfb800108,0xc5740101,0xfb800108,0xc5750101,0xfb800108,0xc5760101,0xfb800108,0xc5770101,
	0xfb800108,0xc5780101,0xfb800108,0xc5790101,0xfb800108,0xc57a0101,0xfb800108,0xc57b0101,
	0xfb800108,0xc57c0101,0xfb800108,0xc57d0101,0xfb800108,0xc57e0101,0xfb800108,0xc57f0101,
	0xfb800108,0xc5800101,0xfb800108,0xc5810101,0xfb800108,0xc5820101,0xfb800108,0xc5830101,
	0xfb800108,0xc5840101,0xfb800108,0xc5850101,0xfb800108,0xc5860101,0xfb800108,0xc5870101,
	0xfb800108,0xc5880101,0xfb800108,0xc5890101,0xfb800108,0xc58a0101,0xfb800108,0xc58b0101,
	0xfb800108,0xc58c0101,0xfb800108,0xc58d0101,0xfb800108,0xc58e0101,0xfb800108,0xc58f0101,
	0xfb800108,0xc5900101,0xfb800108,0xc5910101,0xfb800108,0xc5920101,0xfb800108,0xc5930101,
	0xfb800108,0xc5940101,0xfb800108,0xc5950101,0xfb800108,0xc5960101,0xfb800108,0xc5970101,
	0xfb800108,0xc5980101,0xfb800108,0xc5990101,0xfb800108,0xc59a0101,0xfb800108,0xc59b0101,
	0xfb800108,0xc59c0101,0xfb800108,0xc59d0101,0xfb800108,0xc59e0101,0xfb800108,0xc59f0101,
	0xfb800108,0xc5a00101,0xfb800108,0xc5a10101,0xfb800108,0xc5a20101,0xfb800108,0xc5a30101,
	0xfb800108,0xc5a40101,0xfb800108,0xc5a50101,0xfb800108,0xc5a60101,0xfb800108,0xc5a70101,
	0xfb800108,0xc5a80101,0xfb800108,0xc5a90101,0xfb800108,0xc5aa0101,0xfb800108,0xc5ab0101,
	0xfb800108,0xc5ac0101,0xfb800108,0xc5ad0101,0xfb800108,0xc5ae0101,0xfb800108,0xc5af0101,
	0xfb800108,0xc5b00101,0xfb800108,0xc5b10101,0xfb800108,0xc5b20101,0xfb800108,0xc5b30101,
	0xfb800108,0xc5b40101,0xfb800108,0xc5b50101,0xfb800108,0xc5b60101,0xfb800108,0xc5b70101,
	0xfb800108,0xc5b80101,0xfb800108,0xc5b90101,0xfb800108,0xc5ba0101,0xfb800108,0xc5bb0101,
	0xfb800108,0xc5bc0101,0xfb800108,0xc5bd0101,0xfb800108,0xc5be0101,0xfb800108,0xc5bf0101,
	0xfb800108,0xc5c00101,0xfb800108,0xc5c10101,0xfb800108,0xc5c20101,0xfb800108,0xc5c30101,
	0xfb800108,0xc5c40101,0xfb800108,0xc5c50101,0xfb800108,0xc5c60101,0xfb800108,0xc5c70101,
	0xfb800108,0xc5c80101,0xfb800108,0xc5c90101,0xfb800108,0xc5ca0101,0xfb800108,0xc5cb0101,
	0xfb800108,0xc5cc0101,0xfb800108,0xc5cd0101,0xfb800108,0xc5ce0101,0xfb800108,0xc5cf0101,
	0xfb800108,0xc5d00101,0xfb800108,0xc5d10101,0xfb800108,0xc5d20101,0xfb800108,0xc5d30101,
	0xfb800108,0xc5d40101,0xfb800108,0xc5d50101,0xfb800108,0xc5d60101,0xfb800108,0xc5d70101,
	0xfb800108,0xc5d80101,0xfb800108,0xc5d90101,0xfb800108,0xc5da0101,0xfb800108,0xc5db0101,
	0xfb800108,0xc5dc0101,0xfb800108,0xc5dd0101,0xfb800108,0xc5de0101,0xfb800108,0xc5df0101,
	0xfb800108,0xc5e00101,0xfb800108,0xc5e10101,0xfb800108,0xc5e20101,0xfb800108,0xc5e30101,
	0xfb800108,0xc5e40101,0xfb800108,0xc5e50101,0xfb800108,0xc5e60101,0xfb800108,0xc5e70101,
	0xfb800108,0xc5e80101,0xfb800108,0xc5e90101,0xfb800108,0xc5ea0101,0xfb800108,0xc5eb0101,
	0xfb800108,0xc5ec0101,0xfb800108,0xc5ed0101,0xfb800108,0xc5ee0101,0xfb800108,0xc5ef0101,
	0xfb800108,0xc5f00101,0xfb800108,0xc5f10101,0xfb800108,0xc5f20101,0xfb800108,0xc5f30101,
	0xfb800108,0xc5f40101,0xfb800108,0xc5f50101,0xfb800108,0xc5f60101,0xfb800108,0xc5f70101,
	0xfb800108,0xc5f80101,0xfb800108,0xc5f90101,0xfb800108,0xc5fa0101,0xfb800108,0xc5fb0101,
	0xfb800108,0xc5fc0101,0xfb800108,0xc5fd0101,0xfb800108,0xc5fe0101,0xfb800108,0xc5ff0101,
	0xfb800108,0xc6000101,0xfb800108,0xc6010101,0xfb800108,0xc6020101,0xfb800108,0xc6030101,
	0xfb800108,0xc6040101,0xfb800108,0xc6050101,0xfb800108,0xc6060101,0xfb800108,0xc6070101,
	0xfb800108,0xc6080101,0xfb800108,0xc6090101,0xfb800108,0xc60a0101,0xfb800108,0xc60b0101,
	0xfb800108,0xc60c0101,0xfb800108,0xc60d0101,0xfb800108,0xc60e0101,0xfb800108,0xc60f0101,
	0xfb800108,0xc6100101,0xfb800108,0xc6110101,0xfb800108,0xc6120101,0xfb800108,0xc6130101,
	0xfb800108,0xc6140101,0xfb800108,0xc6150101,0xfb800108,0xc6160101,0xfb800108,0xc6170101,
	0xfb800108,0xc6180101,0xfb800108,0xc6190101,0xfb800108,0xc61a0101,0xfb800108,0xc61b0101,
	0xfb800108,0xc61c0101,0xfb800108,0xc61d0101,0xfb800108,0xc61e0101,0xfb800108,0xc61f0101,
	0xfb800108,0xc6200101,0xfb800108,0xc6210101,0xfb800108,0xc6220101,0xfb800108,0xc6230101,
	0xfb800108,0xc6240101,0xfb800108,0xc6250101,0xfb800108,0xc6260101,0xfb800108,0xc6270101,
	0xfb800108,0xc6280101,0xfb800108,0xc6290101,0xfb800108,0xc62a0101,0xfb800108,0xc62b0101,
	0xfb800108,0xc62c0101,0xfb800108,0xc62d0101,0xfb800108,0xc62e0101,0xfb800108,0xc62f0101,
	0xfb800108,0xc6300101,0xfb800108,0xc6310101,0xfb800108,0xc6320101,0xfb800108,0xc6330101,
	0xfb800108,0xc6340101,0xfb800108,0xc6350101,0xfb800108,0xc6360101,0xfb800108,0xc6370101,
	0xfb800108,0xc6380101,0xfb800108,0xc6390101,0xfb800108,0xc63a0101,0xfb800108,0xc63b0101,
	0xfb800108,0xc63c0101,0xfb800108,0xc63d0101,0xfb800108,0xc63e0101,0xfb800108,0xc63f0101,
	0xfb800108,0xc6400101,0xfb800108,0xc6410101,0xfb800108,0xc6420101,0xfb800108,0xc6430101,
	0xfb800108,0xc6440101,0xfb800108,0xc6450101,0xfb800108,0xc6460101,0xfb800108,0xc6470101,
	0xfb800108,0xc6480101,0xfb800108,0xc6490101,0xfb800108,0xc64a0101,0xfb800108,0xc64b0101,
	0xfb800108,0xc64c0101,0xfb800108,0xc64d0101,0xfb800108,0xc64e0101,0xfb800108,0xc64f0101,
	0xfb800108,0xc6500101,0xfb800108,0xc6510101,0xfb800108,0xc6520101,0xfb800108,0xc6530101,
	0xfb800108,0xc6540101,0xfb800108,0xc6550101,0xfb800108,0xc6560101,0xfb800108,0xc6570101,
	0xfb800108,0xc6580101,0xfb800108,0xc6590101,0xfb800108,0xc65a0101,0xfb800108,0xc65b0101,
	0xfb800108,0xc65c0101,0xfb800108,0xc65d0101,0xfb800108,0xc65e0101,0xfb800108,0xc65f0101,
	0xfb800108,0xc6600101,0xfb800108,0xc6610101,0xfb800108,0xc6620101,0xfb800108,0xc6630101,
	0xfb800108,0xc6640101,0xfb800108,0xc6650101,0xfb800108,0xc6660101,0xfb800108,0xc6670101,
	0xfb800108,0xc6680101,0xfb800108,0xc6690101,0xfb800108,0xc66a0101,0xfb800108,0xc66b0101,
	0xfb800108,0xc66c0101,0xfb800108,0xc66d0101,0xfb800108,0xc66e0101,0xfb800108,0xc66f0101,
	0xfb800108,0xc6700101,0xfb800108,0xc6710101,0xfb800108,0xc6720101,0xfb800108,0xc6730101,
	0xfb800108,0xc6740101,0xfb800108,0xc6750101,0xfb800108,0xc6760101,0xfb800108,0xc6770101,
	0xfb800108,0xc6780101,0xfb800108,0xc6790101,0xfb800108,0xc67a0101,0xfb800108,0xc67b0101,
	0xfb800108,0xc67c0101,0xfb800108,0xc67d0101,0xfb800108,0xc67e0101,0xfb800108,0xc67f0101,
	0xfb800108,0xc6800101,0xfb800108,0xc6810101,0xfb800108,0xc6820101,0xfb800108,0xc6830101,
	0xfb800108,0xc6840101,0xfb800108,0xc6850101,0xfb800108,0xc6860101,0xfb800108,0xc6870101,
	0xfb800108,0xc6880101,0xfb800108,0xc6890101,0xfb800108,0xc68a0101,0xfb800108,0xc68b0101,
	0xfb800108,0xc68c0101,0xfb800108,0xc68d0101,0xfb800108,0xc68e0101,0xfb800108,0xc68f0101,
	0xfb800108,0xc6900101,0xfb800108,0xc6910101,0xfb800108,0xc6920101,0xfb800108,0xc6930101,
	0xfb800108,0xc6940101,0xfb800108,0xc6950101,0xfb800108,0xc6960101,0xfb800108,0xc6970101,
	0xfb800108,0xc6980101,0xfb800108,0xc6990101,0xfb800108,0xc69a0101,0xfb800108,0xc69b0101,
	0xfb800108,0xc69c0101,0xfb800108,0xc69d0101,0xfb800108,0xc69e0101,0xfb800108,0xc69f0101,
	0xfb800108,0xc6a00101,0xfb800108,0xc6a10101,0xfb800108,0xc6a20101,0xfb800108,0xc6a30101,
	0xfb800108,0xc6a40101,0xfb800108,0xc6a50101,0xfb800108,0xc6a60101,0xfb800108,0xc6a70101,
	0xfb800108,0xc6a80101,0xfb800108,0xc6a90101,0xfb800108,0xc6aa0101,0xfb800108,0xc6ab0101,
	0xfb800108,0xc6ac0101,0xfb800108,0xc6ad0101,0xfb800108,0xc6ae0101,0xfb800108,0xc6af0101,
	0xfb800108,0xc6b00101,0xfb800108,0xc6b10101,0xfb800108,0xc6b20101,0xfb800108,0xc6b30101,
	0xfb800108,0xc6b40101,0xfb800108,0xc6b50101,0xfb800108,0xc6b60101,0xfb800108,0xc6b70101,
	0xfb800108,0xc6b80101,0xfb800108,0xc6b90101,0xfb800108,0xc6ba0101,0xfb800108,0xc6bb0101,
	0xfb800108,0xc6bc0101,0xfb800108,0xc6bd0101,0xfb800108,0xc6be0101,0xfb800108,0xc6bf0101,
	0xfb800108,0xc6c00101,0xfb800108,0xc6c10101,0xfb800108,0xc6c20101,0xfb800108,0xc6c30101,
	0xfb800108,0xc6c40101,0xfb800108,0xc6c50101,0xfb800108,0xc6c60101,0xfb800108,0xc6c70101,
	0xfb800108,0xc6c80101,0xfb800108,0xc6c90101,0xfb800108,0xc6ca0101,0xfb800108,0xc6cb0101,
	0xfb800108,0xc6cc0101,0xfb800108,0xc6cd0101,0xfb800108,0xc6ce0101,0xfb800108,0xc6cf0101,
	0xfb800108,0xc6d00101,0xfb800108,0xc6d10101,0xfb800108,0xc6d20101,0xfb800108,0xc6d30101,
	0xfb800108,0xc6d40101,0xfb800108,0xc6d50101,0xfb800108,0xc6d60101,0xfb800108,0xc6d70101,
	0xfb800108,0xc6d80101,0xfb800108,0xc6d90101,0xfb800108,0xc6da0101,0xfb800108,0xc6db0101,
	0xfb800108,0xc6dc0101,0xfb800108,0xc6dd0101,0xfb800108,0xc6de0101,0xfb800108,0xc6df0101,
	0xfb800108,0xc6e00101,0xfb800108,0xc6e10101,0xfb800108,0xc6e20101,0xfb800108,0xc6e30101,
	0xfb800108,0xc6e40101,0xfb800108,0xc6e50101,0xfb800108,0xc6e60101,0xfb800108,0xc6e70101,
	0xfb800108,0xc6e80101,0xfb800108,0xc6e90101,0xfb800108,0xc6ea0101,0xfb800108,0xc6eb0101,
	0xfb800108,0xc6ec0101,0xfb800108,0xc6ed0101,0xfb800108,0xc6ee0101,0xfb800108,0xc6ef0101,
	0xfb800108,0xc6f00101,0xfb800108,0xc6f10101,0xfb800108,0xc6f20101,0xfb800108,0xc6f30101,
	0xfb800108,0xc6f40101,0xfb800108,0xc6f50101,0xfb800108,0xc6f60101,0xfb800108,0xc6f70101,
	0xfb800108,0xc6f80101,0xfb800108,0xc6f90101,0xfb800108,0xc6fa0101,0xfb800108,0xc6fb0101,
	0xfb800108,0xc6fc0101,0xfb800108,0xc6fd0101,0xfb800108,0xc6fe0101,0xfb800108,0xc6ff0101,
	0xfb800108,0xc7000101,0xfb800108,0xc7010101,0xfb800108,0xc7020101,0xfb800108,0xc7030101,
	0xfb800108,0xc7040101,0xfb800108,0xc7050101,0xfb800108,0xc7060101,0xfb800108,0xc7070101,
	0xfb800108,0xc7080101,0xfb800108,0xc7090101,0xfb800108,0xc70a0101,0xfb800108,0xc70b0101,
	0xfb800108,0xc70c0101,0xfb800108,0xc70d0101,0xfb800108,0xc70e0101,0xfb800108,0xc70f0101,
	0xfb800108,0xc7100101,0xfb800108,0xc7110101,0xfb800108,0xc7120101,0xfb800108,0xc7130101,
	0xfb800108,0xc7140101,0xfb800108,0xc7150101,0xfb800108,0xc7160101,0xfb800108,0xc7170101,
	0xfb800108,0xc7180101,0xfb800108,0xc7190101,0xfb800108,0xc71a0101,0xfb800108,0xc71b0101,
	0xfb800108,0xc71c0101,0xfb800108,0xc71d0101,0xfb800108,0xc71e0101,0xfb800108,0xc71f0101,
	0xfb800108,0xc7200101,0xfb800108,0xc7210101,0xfb800108,0xc7220101,0xfb800108,0xc7230101,
	0xfb800108,0xc7240101,0xfb800108,0xc7250101,0xfb800108,0xc7260101,0xfb800108,0xc7270101,
	0xfb800108,0xc7280101,0xfb800108,0xc7290101,0xfb800108,0xc72a0101,0xfb800108,0xc72b0101,
	0xfb800108,0xc72c0101,0xfb800108,0xc72d0101,0xfb800108,0xc72e0101,0xfb800108,0xc72f0101,
	0xfb800108,0xc7300101,0xfb800108,0xc7310101,0xfb800108,0xc7320101,0xfb800108,0xc7330101,
	0xfb800108,0xc7340101,0xfb800108,0xc7350101,0xfb800108,0xc7360101,0xfb800108,0xc7370101,
	0xfb800108,0xc7380101,0xfb800108,0xc7390101,0xfb800108,0xc73a0101,0xfb800108,0xc73b0101,
	0xfb800108,0xc73c0101,0xfb800108,0xc73d0101,0xfb800108,0xc73e0101,0xfb800108,0xc73f0101,
	0xfb800108,0xc7400101,0xfb800108,0xc7410101,0xfb800108,0xc7420101,0xfb800108,0xc7430101,
	0xfb800108,0xc7440101,0xfb800108,0xc7450101,0xfb800108,0xc7460101,0xfb800108,0xc7470101,
	0xfb800108,0xc7480101,0xfb800108,0xc7490101,0xfb800108,0xc74a0101,0xfb800108,0xc74b0101,
	0xfb800108,0xc74c0101,0xfb800108,0xc74d0101,0xfb800108,0xc74e0101,0xfb800108,0xc74f0101,
	0xfb800108,0xc7500101,0xfb800108,0xc7510101,0xfb800108,0xc7520101,0xfb800108,0xc7530101,
	0xfb800108,0xc7540101,0xfb800108,0xc7550101,0xfb800108,0xc7560101,0xfb800108,0xc7570101,
	0xfb800108,0xc7580101,0xfb800108,0xc7590101,0xfb800108,0xc75a0101,0xfb800108,0xc75b0101,
	0xfb800108,0xc75c0101,0xfb800108,0xc75d0101,0xfb800108,0xc75e0101,0xfb800108,0xc75f0101,
	0xfb800108,0xc7600101,0xfb800108,0xc7610101,0xfb800108,0xc7620101,0xfb800108,0xc7630101,
	0xfb800108,0xc7640101,0xfb800108,0xc7650101,0xfb800108,0xc7660101,0xfb800108,0xc7670101,
	0xfb800108,0xc7680101,0xfb800108,0xc7690101,0xfb800108,0xc76a0101,0xfb800108,0xc76b0101,
	0xfb800108,0xc76c0101,0xfb800108,0xc76d0101,0xfb800108,0xc76e0101,0xfb800108,0xc76f0101,
	0xfb800108,0xc7700101,0xfb800108,0xc7710101,0xfb800108,0xc7720101,0xfb800108,0xc7730101,
	0xfb800108,0xc7740101,0xfb800108,0xc7750101,0xfb800108,0xc7760101,0xfb800108,0xc7770101,
	0xfb800108,0xc7780101,0xfb800108,0xc7790101,0xfb800108,0xc77a0101,0xfb800108,0xc77b0101,
	0xfb800108,0xc77c0101,0xfb800108,0xc77d0101,0xfb800108,0xc77e0101,0xfb800108,0xc77f0101,
	0xfb800108,0xc7800101,0xfb800108,0xc7810101,0xfb800108,0xc7820101,0xfb800108,0xc7830101,
	0xfb800108,0xc7840101,0xfb800108,0xc7850101,0xfb800108,0xc7860101,0xfb800108,0xc7870101,
	0xfb800108,0xc7880101,0xfb800108,0xc7890101,0xfb800108,0xc78a0101,0xfb800108,0xc78b0101,
	0xfb800108,0xc78c0101,0xfb800108,0xc78d0101,0xfb800108,0xc78e0101,0xfb800108,0xc78f0101,
	0xfb800108,0xc7900101,0xfb800108,0xc7910101,0xfb800108,0xc7920101,0xfb800108,0xc7930101,
	0xfb800108,0xc7940101,0xfb800108,0xc7950101,0xfb800108,0xc7960101,0xfb800108,0xc7970101,
	0xfb800108,0xc7980101,0xfb800108,0xc7990101,0xfb800108,0xc79a0101,0xfb800108,0xc79b0101,
	0xfb800108,0xc79c0101,0xfb800108,0xc79d0101,0xfb800108,0xc79e0101,0xfb800108,0xc79f0101,
	0xfb800108,0xc7a00101,0xfb800108,0xc7a10101,0xfb800108,0xc7a20101,0xfb800108,0xc7a30101,
	0xfb800108,0xc7a40101,0xfb800108,0xc7a50101,0xfb800108,0xc7a60101,0xfb800108,0xc7a70101,
	0xfb800108,0xc7a80101,0xfb800108,0xc7a90101,0xfb800108,0xc7aa0101,0xfb800108,0xc7ab0101,
	0xfb800108,0xc7ac0101,0xfb800108,0xc7ad0101,0xfb800108,0xc7ae0101,0xfb800108,0xc7af0101,
	0xfb800108,0xc7b00101,0xfb800108,0xc7b10101,0xfb800108,0xc7b20101,0xfb800108,0xc7b30101,
	0xfb800108,0xc7b40101,0xfb800108,0xc7b50101,0xfb800108,0xc7b60101,0xfb800108,0xc7b70101,
	0xfb800108,0xc7b80101,0xfb800108,0xc7b90101,0xfb800108,0xc7ba0101,0xfb800108,0xc7bb0101,
	0xfb800108,0xc7bc0101,0xfb800108,0xc7bd0101,0xfb800108,0xc7be0101,0xfb800108,0xc7bf0101,
	0xfb800108,0xc7c00101,0xfb800108,0xc7c10101,0xfb800108,0xc7c20101,0xfb800108,0xc7c30101,
	0xfb800108,0xc7c40101,0xfb800108,0xc7c50101,0xfb800108,0xc7c60101,0xfb800108,0xc7c70101,
	0xfb800108,0xc7c80101,0xfb800108,0xc7c90101,0xfb800108,0xc7ca0101,0xfb800108,0xc7cb0101,
	0xfb800108,0xc7cc0101,0xfb800108,0xc7cd0101,0xfb800108,0xc7ce0101,0xfb800108,0xc7cf0101,
	0xfb800108,0xc7d00101,0xfb800108,0xc7d10101,0xfb800108,0xc7d20101,0xfb800108,0xc7d30101,
	0xfb800108,0xc7d40101,0xfb800108,0xc7d50101,0xfb800108,0xc7d60101,0xfb800108,0xc7d70101,
	0xfb800108,0xc7d80101,0xfb800108,0xc7d90101,0xfb800108,0xc7da0101,0xfb800108,0xc7db0101,
	0xfb800108,0xc7dc0101,0xfb800108,0xc7dd0101,0xfb800108,0xc7de0101,0xfb800108,0xc7df0101,
	0xfb800108,0xc7e00101,0xfb800108,0xc7e10101,0xfb800108,0xc7e20101,0xfb800108,0xc7e30101,
	0xfb800108,0xc7e40101,0xfb800108,0xc7e50101,0xfb800108,0xc7e60101,0xfb800108,0xc7e70101,
	0xfb800108,0xc7e80101,0xfb800108,0xc7e90101,0xfb800108,0xc7ea0101,0xfb800108,0xc7eb0101,
	0xfb800108,0xc7ec0101,0xfb800108,0xc7ed0101,0xfb800108,0xc7ee0101,0xfb800108,0xc7ef0101,
	0xfb800108,0xc7f00101,0xfb800108,0xc7f10101,0xfb800108,0xc7f20101,0xfb800108,0xc7f30101,
	0xfb800108,0xc7f40101,0xfb800108,0xc7f50101,0xfb800108,0xc7f60101,0xfb800108,0xc7f70101,
	0xfb800108,0xc7f80101,0xfb800108,0xc7f90101,0xfb800108,0xc7fa0101,0xfb800108,0xc7fb0101,
	0xfb800108,0xc7fc0101,0xfb800108,0xc7fd0101,0xfb800108,0xc7fe0101,0xfb800108,0xc7ff0101,
	0xfb800108,0xc8000101,0xfb800108,0xc8010101,0xfb800108,0xc8020101,0xfb800108,0xc8030101,
	0xfb800108,0xc8040101,0xfb800108,0xc8050101,0xfb800108,0xc8060101,0xfb800108,0xc8070101,
	0xfb800108,0xc8080101,0xfb800108,0xc8090101,0xfb800108,0xc80a0101,0xfb800108,0xc80b0101,
	0xfb800108,0xc80c0101,0xfb800108,0xc80d0101,0xfb800108,0xc80e0101,0xfb800108,0xc80f0101,
	0xfb800108,0xc8100101,0xfb800108,0xc8110101,0xfb800108,0xc8120101,0xfb800108,0xc8130101,
	0xfb800108,0xc8140101,0xfb800108,0xc8150101,0xfb800108,0xc8160101,0xfb800108,0xc8170101,
	0xfb800108,0xc8180101,0xfb800108,0xc8190101,0xfb800108,0xc81a0101,0xfb800108,0xc81b0101,
	0xfb800108,0xc81c0101,0xfb800108,0xc81d0101,0xfb800108,0xc81e0101,0xfb800108,0xc81f0101,
	0xfb800108,0xc8200101,0xfb800108,0xc8210101,0xfb800108,0xc8220101,0xfb800108,0xc8230101,
	0xfb800108,0xc8240101,0xfb800108,0xc8250101,0xfb800108,0xc8260101,0xfb800108,0xc8270101,
	0xfb800108,0xc8280101,0xfb800108,0xc8290101,0xfb800108,0xc82a0101,0xfb800108,0xc82b0101,
	0xfb800108,0xc82c0101,0xfb800108,0xc82d0101,0xfb800108,0xc82e0101,0xfb800108,0xc82f0101,
	0xfb800108,0xc8300101,0xfb800108,0xc8310101,0xfb800108,0xc8320101,0xfb800108,0xc8330101,
	0xfb800108,0xc8340101,0xfb800108,0xc8350101,0xfb800108,0xc8360101,0xfb800108,0xc8370101,
	0xfb800108,0xc8380101,0xfb800108,0xc8390101,0xfb800108,0xc83a0101,0xfb800108,0xc83b0101,
	0xfb800108,0xc83c0101,0xfb800108,0xc83d0101,0xfb800108,0xc83e0101,0xfb800108,0xc83f0101,
	0xfb800108,0xc8400101,0xfb800108,0xc8410101,0xfb800108,0xc8420101,0xfb800108,0xc8430101,
	0xfb800108,0xc8440101,0xfb800108,0xc8450101,0xfb800108,0xc8460101,0xfb800108,0xc8470101,
	0xfb800108,0xc8480101,0xfb800108,0xc8490101,0xfb800108,0xc84a0101,0xfb800108,0xc84b0101,
	0xfb800108,0xc84c0101,0xfb800108,0xc84d0101,0xfb800108,0xc84e0101,0xfb800108,0xc84f0101,
	0xfb800108,0xc8500101,0xfb800108,0xc8510101,0xfb800108,0xc8520101,0xfb800108,0xc8530101,
	0xfb800108,0xc8540101,0xfb800108,0xc8550101,0xfb800108,0xc8560101,0xfb800108,0xc8570101,
	0xfb800108,0xc8580101,0xfb800108,0xc8590101,0xfb800108,0xc85a0101,0xfb800108,0xc85b0101,
	0xfb800108,0xc85c0101,0xfb800108,0xc85d0101,0xfb800108,0xc85e0101,0xfb800108,0xc85f0101,
	0xfb800108,0xc8600101,0xfb800108,0xc8610101,0xfb800108,0xc8620101,0xfb800108,0xc8630101,
	0xfb800108,0xc8640101,0xfb800108,0xc8650101,0xfb800108,0xc8660101,0xfb800108,0xc8670101,
	0xfb800108,0xc8680101,0xfb800108,0xc8690101,0xfb800108,0xc86a0101,0xfb800108,0xc86b0101,
	0xfb800108,0xc86c0101,0xfb800108,0xc86d0101,0xfb800108,0xc86e0101,0xfb800108,0xc86f0101,
	0xfb800108,0xc8700101,0xfb800108,0xc8710101,0xfb800108,0xc8720101,0xfb800108,0xc8730101,
	0xfb800108,0xc8740101,0xfb800108,0xc8750101,0xfb800108,0xc8760101,0xfb800108,0xc8770101,
	0xfb800108,0xc8780101,0xfb800108,0xc8790101,0xfb800108,0xc87a0101,0xfb800108,0xc87b0101,
	0xfb800108,0xc87c0101,0xfb800108,0xc87d0101,0xfb800108,0xc87e0101,0xfb800108,0xc87f0101,
	0xfb800108,0xc8800101,0xfb800108,0xc8810101,0xfb800108,0xc8820101,0xfb800108,0xc8830101,
	0xfb800108,0xc8840101,0xfb800108,0xc8850101,0xfb800108,0xc8860101,0xfb800108,0xc8870101,
	0xfb800108,0xc8880101,0xfb800108,0xc8890101,0xfb800108,0xc88a0101,0xfb800108,0xc88b0101,
	0xfb800108,0xc88c0101,0xfb800108,0xc88d0101,0xfb800108,0xc88e0101,0xfb800108,0xc88f0101,
	0xfb800108,0xc8900101,0xfb800108,0xc8910101,0xfb800108,0xc8920101,0xfb800108,0xc8930101,
	0xfb800108,0xc8940101,0xfb800108,0xc8950101,0xfb800108,0xc8960101,0xfb800108,0xc8970101,
	0xfb800108,0xc8980101,0xfb800108,0xc8990101,0xfb800108,0xc89a0101,0xfb800108,0xc89b0101,
	0xfb800108,0xc89c0101,0xfb800108,0xc89d0101,0xfb800108,0xc89e0101,0xfb800108,0xc89f0101,
	0xfb800108,0xc8a00101,0xfb800108,0xc8a10101,0xfb800108,0xc8a20101,0xfb800108,0xc8a30101,
	0xfb800108,0xc8a40101,0xfb800108,0xc8a50101,0xfb800108,0xc8a60101,0xfb800108,0xc8a70101,
	0xfb800108,0xc8a80101,0xfb800108,0xc8a90101,0xfb800108,0xc8aa0101,0xfb800108,0xc8ab0101,
	0xfb800108,0xc8ac0101,0xfb800108,0xc8ad0101,0xfb800108,0xc8ae0101,0xfb800108,0xc8af0101,
	0xfb800108,0xc8b00101,0xfb800108,0xc8b10101,0xfb800108,0xc8b20101,0xfb800108,0xc8b30101,
	0xfb800108,0xc8b40101,0xfb800108,0xc8b50101,0xfb800108,0xc8b60101,0xfb800108,0xc8b70101,
	0xfb800108,0xc8b80101,0xfb800108,0xc8b90101,0xfb800108,0xc8ba0101,0xfb800108,0xc8bb0101,
	0xfb800108,0xc8bc0101,0xfb800108,0xc8bd0101,0xfb800108,0xc8be0101,0xfb800108,0xc8bf0101,
	0xfb800108,0xc8c00101,0xfb800108,0xc8c10101,0xfb800108,0xc8c20101,0xfb800108,0xc8c30101,
	0xfb800108,0xc8c40101,0xfb800108,0xc8c50101,0xfb800108,0xc8c60101,0xfb800108,0xc8c70101,
	0xfb800108,0xc8c80101,0xfb800108,0xc8c90101,0xfb800108,0xc8ca0101,0xfb800108,0xc8cb0101,
	0xfb800108,0xc8cc0101,0xfb800108,0xc8cd0101,0xfb800108,0xc8ce0101,0xfb800108,0xc8cf0101,
	0xfb800108,0xc8d00101,0xfb800108,0xc8d10101,0xfb800108,0xc8d20101,0xfb800108,0xc8d30101,
	0xfb800108,0xc8d40101,0xfb800108,0xc8d50101,0xfb800108,0xc8d60101,0xfb800108,0xc8d70101,
	0xfb800108,0xc8d80101,0xfb800108,0xc8d90101,0xfb800108,0xc8da0101,0xfb800108,0xc8db0101,
	0xfb800108,0xc8dc0101,0xfb800108,0xc8dd0101,0xfb800108,0xc8de0101,0xfb800108,0xc8df0101,
	0xfb800108,0xc8e00101,0xfb800108,0xc8e10101,0xfb800108,0xc8e20101,0xfb800108,0xc8e30101,
	0xfb800108,0xc8e40101,0xfb800108,0xc8e50101,0xfb800108,0xc8e60101,0xfb800108,0xc8e70101,
	0xfb800108,0xc8e80101,0xfb800108,0xc8e90101,0xfb800108,0xc8ea0101,0xfb800108,0xc8eb0101,
	0xfb800108,0xc8ec0101,0xfb800108,0xc8ed0101,0xfb800108,0xc8ee0101,0xfb800108,0xc8ef0101,
	0xfb800108,0xc8f00101,0xfb800108,0xc8f10101,0xfb800108,0xc8f20101,0xfb800108,0xc8f30101,
	0xfb800108,0xc8f40101,0xfb800108,0xc8f50101,0xfb800108,0xc8f60101,0xfb800108,0xc8f70101,
	0xfb800108,0xc8f80101,0xfb800108,0xc8f90101,0xfb800108,0xc8fa0101,0xfb800108,0xc8fb0101,
	0xfb800108,0xc8fc0101,0xfb800108,0xc8fd0101,0xfb800108,0xc8fe0101,0xfb800108,0xc8ff0101,
	0xfb800108,0xc9000101,0xfb800108,0xc9010101,0xfb800108,0xc9020101,0xfb800108,0xc9030101,
	0xfb800108,0xc9040101,0xfb800108,0xc9050101,0xfb800108,0xc9060101,0xfb800108,0xc9070101,
	0xfb800108,0xc9080101,0xfb800108,0xc9090101,0xfb800108,0xc90a0101,0xfb800108,0xc90b0101,
	0xfb800108,0xc90c0101,0xfb800108,0xc90d0101,0xfb800108,0xc90e0101,0xfb800108,0xc90f0101,
	0xfb800108,0xc9100101,0xfb800108,0xc9110101,0xfb800108,0xc9120101,0xfb800108,0xc9130101,
	0xfb800108,0xc9140101,0xfb800108,0xc9150101,0xfb800108,0xc9160101,0xfb800108,0xc9170101,
	0xfb800108,0xc9180101,0xfb800108,0xc9190101,0xfb800108,0xc91a0101,0xfb800108,0xc91b0101,
	0xfb800108,0xc91c0101,0xfb800108,0xc91d0101,0xfb800108,0xc91e0101,0xfb800108,0xc91f0101,
	0xfb800108,0xc9200101,0xfb800108,0xc9210101,0xfb800108,0xc9220101,0xfb800108,0xc9230101,
	0xfb800108,0xc9240101,0xfb800108,0xc9250101,0xfb800108,0xc9260101,0xfb800108,0xc9270101,
	0xfb800108,0xc9280101,0xfb800108,0xc9290101,0xfb800108,0xc92a0101,0xfb800108,0xc92b0101,
	0xfb800108,0xc92c0101,0xfb800108,0xc92d0101,0xfb800108,0xc92e0101,0xfb800108,0xc92f0101,
	0xfb800108,0xc9300101,0xfb800108,0xc9310101,0xfb800108,0xc9320101,0xfb800108,0xc9330101,
	0xfb800108,0xc9340101,0xfb800108,0xc9350101,0xfb800108,0xc9360101,0xfb800108,0xc9370101,
	0xfb800108,0xc9380101,0xfb800108,0xc9390101,0xfb800108,0xc93a0101,0xfb800108,0xc93b0101,
	0xfb800108,0xc93c0101,0xfb800108,0xc93d0101,0xfb800108,0xc93e0101,0xfb800108,0xc93f0101,
	0xfb800108,0xc9400101,0xfb800108,0xc9410101,0xfb800108,0xc9420101,0xfb800108,0xc9430101,
	0xfb800108,0xc9440101,0xfb800108,0xc9450101,0xfb800108,0xc9460101,0xfb800108,0xc9470101,
	0xfb800108,0xc9480101,0xfb800108,0xc9490101,0xfb800108,0xc94a0101,0xfb800108,0xc94b0101,
	0xfb800108,0xc94c0101,0xfb800108,0xc94d0101,0xfb800108,0xc94e0101,0xfb800108,0xc94f0101,
	0xfb800108,0xc9500101,0xfb800108,0xc9510101,0xfb800108,0xc9520101,0xfb800108,0xc9530101,
	0xfb800108,0xc9540101,0xfb800108,0xc9550101,0xfb800108,0xc9560101,0xfb800108,0xc9570101,
	0xfb800108,0xc9580101,0xfb800108,0xc9590101,0xfb800108,0xc95a0101,0xfb800108,0xc95b0101,
	0xfb800108,0xc95c0101,0xfb800108,0xc95d0101,0xfb800108,0xc95e0101,0xfb800108,0xc95f0101,
	0xfb800108,0xc9600101,0xfb800108,0xc9610101,0xfb800108,0xc9620101,0xfb800108,0xc9630101,
	0xfb800108,0xc9640101,0xfb800108,0xc9650101,0xfb800108,0xc9660101,0xfb800108,0xc9670101,
	0xfb800108,0xc9680101,0xfb800108,0xc9690101,0xfb800108,0xc96a0101,0xfb800108,0xc96b0101,
	0xfb800108,0xc96c0101,0xfb800108,0xc96d0101,0xfb800108,0xc96e0101,0xfb800108,0xc96f0101,
	0xfb800108,0xc9700101,0xfb800108,0xc9710101,0xfb800108,0xc9720101,0xfb800108,0xc9730101,
	0xfb800108,0xc9740101,0xfb800108,0xc9750101,0xfb800108,0xc9760101,0xfb800108,0xc9770101,
	0xfb800108,0xc9780101,0xfb800108,0xc9790101,0xfb800108,0xc97a0101,0xfb800108,0xc97b0101,
	0xfb800108,0xc97c0101,0xfb800108,0xc97d0101,0xfb800108,0xc97e0101,0xfb800108,0xc97f0101,
	0xfb800108,0xc9800101,0xfb800108,0xc9810101,0xfb800108,0xc9820101,0xfb800108,0xc9830101,
	0xfb800108,0xc9840101,0xfb800108,0xc9850101,0xfb800108,0xc9860101,0xfb800108,0xc9870101,
	0xfb800108,0xc9880101,0xfb800108,0xc9890101,0xfb800108,0xc98a0101,0xfb800108,0xc98b0101,
	0xfb800108,0xc98c0101,0xfb800108,0xc98d0101,0xfb800108,0xc98e0101,0xfb800108,0xc98f0101,
	0xfb800108,0xc9900101,0xfb800108,0xc9910101,0xfb800108,0xc9920101,0xfb800108,0xc9930101,
	0xfb800108,0xc9940101,0xfb800108,0xc9950101,0xfb800108,0xc9960101,0xfb800108,0xc9970101,
	0xfb800108,0xc9980101,0xfb800108,0xc9990101,0xfb800108,0xc99a0101,0xfb800108,0xc99b0101,
	0xfb800108,0xc99c0101,0xfb800108,0xc99d0101,0xfb800108,0xc99e0101,0xfb800108,0xc99f0101,
	0xfb800108,0xc9a00101,0xfb800108,0xc9a10101,0xfb800108,0xc9a20101,0xfb800108,0xc9a30101,
	0xfb800108,0xc9a40101,0xfb800108,0xc9a50101,0xfb800108,0xc9a60101,0xfb800108,0xc9a70101,
	0xfb800108,0xc9a80101,0xfb800108,0xc9a90101,0xfb800108,0xc9aa0101,0xfb800108,0xc9ab0101,
	0xfb800108,0xc9ac0101,0xfb800108,0xc9ad0101,0xfb800108,0xc9ae0101,0xfb800108,0xc9af0101,
	0xfb800108,0xc9b00101,0xfb800108,0xc9b10101,0xfb800108,0xc9b20101,0xfb800108,0xc9b30101,
	0xfb800108,0xc9b40101,0xfb800108,0xc9b50101,0xfb800108,0xc9b60101,0xfb800108,0xc9b70101,
	0xfb800108,0xc9b80101,0xfb800108,0xc9b90101,0xfb800108,0xc9ba0101,0xfb800108,0xc9bb0101,
	0xfb800108,0xc9bc0101,0xfb800108,0xc9bd0101,0xfb800108,0xc9be0101,0xfb800108,0xc9bf0101,
	0xfb800108,0xc9c00101,0xfb800108,0xc9c10101,0xfb800108,0xc9c20101,0xfb800108,0xc9c30101,
	0xfb800108,0xc9c40101,0xfb800108,0xc9c50101,0xfb800108,0xc9c60101,0xfb800108,0xc9c70101,
	0xfb800108,0xc9c80101,0xfb800108,0xc9c90101,0xfb800108,0xc9ca0101,0xfb800108,0xc9cb0101,
	0xfb800108,0xc9cc0101,0xfb800108,0xc9cd0101,0xfb800108,0xc9ce0101,0xfb800108,0xc9cf0101,
	0xfb800108,0xc9d00101,0xfb800108,0xc9d10101,0xfb800108,0xc9d20101,0xfb800108,0xc9d30101,
	0xfb800108,0xc9d40101,0xfb800108,0xc9d50101,0xfb800108,0xc9d60101,0xfb800108,0xc9d70101,
	0xfb800108,0xc9d80101,0xfb800108,0xc9d90101,0xfb800108,0xc9da0101,0xfb800108,0xc9db0101,
	0xfb800108,0xc9dc0101,0xfb800108,0xc9dd0101,0xfb800108,0xc9de0101,0xfb800108,0xc9df0101,
	0xfb800108,0xc9e00101,0xfb800108,0xc9e10101,0xfb800108,0xc9e20101,0xfb800108,0xc9e30101,
	0xfb800108,0xc9e40101,0xfb800108,0xc9e50101,0xfb800108,0xc9e60101,0xfb800108,0xc9e70101,
	0xfb800108,0xc9e80101,0xfb800108,0xc9e90101,0xfb800108,0xc9ea0101,0xfb800108,0xc9eb0101,
	0xfb800108,0xc9ec0101,0xfb800108,0xc9ed0101,0xfb800108,0xc9ee0101,0xfb800108,0xc9ef0101,
	0xfb800108,0xc9f00101,0xfb800108,0xc9f10101,0xfb800108,0xc9f20101,0xfb800108,0xc9f30101,
	0xfb800108,0xc9f40101,0xfb800108,0xc9f50101,0xfb800108,0xc9f60101,0xfb800108,0xc9f70101,
	0xfb800108,0xc9f80101,0xfb800108,0xc9f90101,0xfb800108,0xc9fa0101,0xfb800108,0xc9fb0101,
	0xfb800108,0xc9fc0101,0xfb800108,0xc9fd0101,0xfb800108,0xc9fe0101,0xfb800108,0xc9ff0101,
	0xfb800108,0xca000101,0xfb800108,0xca010101,0xfb800108,0xca020101,0xfb800108,0xca030101,
	0xfb800108,0xca040101,0xfb800108,0xca050101,0xfb800108,0xca060101,0xfb800108,0xca070101,
	0xfb800108,0xca080101,0xfb800108,0xca090101,0xfb800108,0xca0a0101,0xfb800108,0xca0b0101,
	0xfb800108,0xca0c0101,0xfb800108,0xca0d0101,0xfb800108,0xca0e0101,0xfb800108,0xca0f0101,
	0xfb800108,0xca100101,0xfb800108,0xca110101,0xfb800108,0xca120101,0xfb800108,0xca130101,
	0xfb800108,0xca140101,0xfb800108,0xca150101,0xfb800108,0xca160101,0xfb800108,0xca170101,
	0xfb800108,0xca180101,0xfb800108,0xca190101,0xfb800108,0xca1a0101,0xfb800108,0xca1b0101,
	0xfb800108,0xca1c0101,0xfb800108,0xca1d0101,0xfb800108,0xca1e0101,0xfb800108,0xca1f0101,
	0xfb800108,0xca200101,0xfb800108,0xca210101,0xfb800108,0xca220101,0xfb800108,0xca230101,
	0xfb800108,0xca240101,0xfb800108,0xca250101,0xfb800108,0xca260101,0xfb800108,0xca270101,
	0xfb800108,0xca280101,0xfb800108,0xca290101,0xfb800108,0xca2a0101,0xfb800108,0xca2b0101,
	0xfb800108,0xca2c0101,0xfb800108,0xca2d0101,0xfb800108,0xca2e0101,0xfb800108,0xca2f0101,
	0xfb800108,0xca300101,0xfb800108,0xca310101,0xfb800108,0xca320101,0xfb800108,0xca330101,
	0xfb800108,0xca340101,0xfb800108,0xca350101,0xfb800108,0xca360101,0xfb800108,0xca370101,
	0xfb800108,0xca380101,0xfb800108,0xca390101,0xfb800108,0xca3a0101,0xfb800108,0xca3b0101,
	0xfb800108,0xca3c0101,0xfb800108,0xca3d0101,0xfb800108,0xca3e0101,0xfb800108,0xca3f0101,
	0xfb800108,0xca400101,0xfb800108,0xca410101,0xfb800108,0xca420101,0xfb800108,0xca430101,
	0xfb800108,0xca440101,0xfb800108,0xca450101,0xfb800108,0xca460101,0xfb800108,0xca470101,
	0xfb800108,0xca480101,0xfb800108,0xca490101,0xfb800108,0xca4a0101,0xfb800108,0xca4b0101,
	0xfb800108,0xca4c0101,0xfb800108,0xca4d0101,0xfb800108,0xca4e0101,0xfb800108,0xca4f0101,
	0xfb800108,0xca500101,0xfb800108,0xca510101,0xfb800108,0xca520101,0xfb800108,0xca530101,
	0xfb800108,0xca540101,0xfb800108,0xca550101,0xfb800108,0xca560101,0xfb800108,0xca570101,
	0xfb800108,0xca580101,0xfb800108,0xca590101,0xfb800108,0xca5a0101,0xfb800108,0xca5b0101,
	0xfb800108,0xca5c0101,0xfb800108,0xca5d0101,0xfb800108,0xca5e0101,0xfb800108,0xca5f0101,
	0xfb800108,0xca600101,0xfb800108,0xca610101,0xfb800108,0xca620101,0xfb800108,0xca630101,
	0xfb800108,0xca640101,0xfb800108,0xca650101,0xfb800108,0xca660101,0xfb800108,0xca670101,
	0xfb800108,0xca680101,0xfb800108,0xca690101,0xfb800108,0xca6a0101,0xfb800108,0xca6b0101,
	0xfb800108,0xca6c0101,0xfb800108,0xca6d0101,0xfb800108,0xca6e0101,0xfb800108,0xca6f0101,
	0xfb800108,0xca700101,0xfb800108,0xca710101,0xfb800108,0xca720101,0xfb800108,0xca730101,
	0xfb800108,0xca740101,0xfb800108,0xca750101,0xfb800108,0xca760101,0xfb800108,0xca770101,
	0xfb800108,0xca780101,0xfb800108,0xca790101,0xfb800108,0xca7a0101,0xfb800108,0xca7b0101,
	0xfb800108,0xca7c0101,0xfb800108,0xca7d0101,0xfb800108,0xca7e0101,0xfb800108,0xca7f0101,
	0xfb800108,0xca800101,0xfb800108,0xca810101,0xfb800108,0xca820101,0xfb800108,0xca830101,
	0xfb800108,0xca840101,0xfb800108,0xca850101,0xfb800108,0xca860101,0xfb800108,0xca870101,
	0xfb800108,0xca880101,0xfb800108,0xca890101,0xfb800108,0xca8a0101,0xfb800108,0xca8b0101,
	0xfb800108,0xca8c0101,0xfb800108,0xca8d0101,0xfb800108,0xca8e0101,0xfb800108,0xca8f0101,
	0xfb800108,0xca900101,0xfb800108,0xca910101,0xfb800108,0xca920101,0xfb800108,0xca930101,
	0xfb800108,0xca940101,0xfb800108,0xca950101,0xfb800108,0xca960101,0xfb800108,0xca970101,
	0xfb800108,0xca980101,0xfb800108,0xca990101,0xfb800108,0xca9a0101,0xfb800108,0xca9b0101,
	0xfb800108,0xca9c0101,0xfb800108,0xca9d0101,0xfb800108,0xca9e0101,0xfb800108,0xca9f0101,
	0xfb800108,0xcaa00101,0xfb800108,0xcaa10101,0xfb800108,0xcaa20101,0xfb800108,0xcaa30101,
	0xfb800108,0xcaa40101,0xfb800108,0xcaa50101,0xfb800108,0xcaa60101,0xfb800108,0xcaa70101,
	0xfb800108,0xcaa80101,0xfb800108,0xcaa90101,0xfb800108,0xcaaa0101,0xfb800108,0xcaab0101,
	0xfb800108,0xcaac0101,0xfb800108,0xcaad0101,0xfb800108,0xcaae0101,0xfb800108,0xcaaf0101,
	0xfb800108,0xcab00101,0xfb800108,0xcab10101,0xfb800108,0xcab20101,0xfb800108,0xcab30101,
	0xfb800108,0xcab40101,0xfb800108,0xcab50101,0xfb800108,0xcab60101,0xfb800108,0xcab70101,
	0xfb800108,0xcab80101,0xfb800108,0xcab90101,0xfb800108,0xcaba0101,0xfb800108,0xcabb0101,
	0xfb800108,0xcabc0101,0xfb800108,0xcabd0101,0xfb800108,0xcabe0101,0xfb800108,0xcabf0101,
	0xfb800108,0xcac00101,0xfb800108,0xcac10101,0xfb800108,0xcac20101,0xfb800108,0xcac30101,
	0xfb800108,0xcac40101,0xfb800108,0xcac50101,0xfb800108,0xcac60101,0xfb800108,0xcac70101,
	0xfb800108,0xcac80101,0xfb800108,0xcac90101,0xfb800108,0xcaca0101,0xfb800108,0xcacb0101,
	0xfb800108,0xcacc0101,0xfb800108,0xcacd0101,0xfb800108,0xcace0101,0xfb800108,0xcacf0101,
	0xfb800108,0xcad00101,0xfb800108,0xcad10101,0xfb800108,0xcad20101,0xfb800108,0xcad30101,
	0xfb800108,0xcad40101,0xfb800108,0xcad50101,0xfb800108,0xcad60101,0xfb800108,0xcad70101,
	0xfb800108,0xcad80101,0xfb800108,0xcad90101,0xfb800108,0xcada0101,0xfb800108,0xcadb0101,
	0xfb800108,0xcadc0101,0xfb800108,0xcadd0101,0xfb800108,0xcade0101,0xfb800108,0xcadf0101,
	0xfb800108,0xcae00101,0xfb800108,0xcae10101,0xfb800108,0xcae20101,0xfb800108,0xcae30101,
	0xfb800108,0xcae40101,0xfb800108,0xcae50101,0xfb800108,0xcae60101,0xfb800108,0xcae70101,
	0xfb800108,0xcae80101,0xfb800108,0xcae90101,0xfb800108,0xcaea0101,0xfb800108,0xcaeb0101,
	0xfb800108,0xcaec0101,0xfb800108,0xcaed0101,0xfb800108,0xcaee0101,0xfb800108,0xcaef0101,
	0xfb800108,0xcaf00101,0xfb800108,0xcaf10101,0xfb800108,0xcaf20101,0xfb800108,0xcaf30101,
	0xfb800108,0xcaf40101,0xfb800108,0xcaf50101,0xfb800108,0xcaf60101,0xfb800108,0xcaf70101,
	0xfb800108,0xcaf80101,0xfb800108,0xcaf90101,0xfb800108,0xcafa0101,0xfb800108,0xcafb0101,
	0xfb800108,0xcafc0101,0xfb800108,0xcafd0101,0xfb800108,0xcafe0101,0xfb800108,0xcaff0101,
	0xfb800108,0xcb000101,0xfb800108,0xcb010101,0xfb800108,0xcb020101,0xfb800108,0xcb030101,
	0xfb800108,0xcb040101,0xfb800108,0xcb050101,0xfb800108,0xcb060101,0xfb800108,0xcb070101,
	0xfb800108,0xcb080101,0xfb800108,0xcb090101,0xfb800108,0xcb0a0101,0xfb800108,0xcb0b0101,
	0xfb800108,0xcb0c0101,0xfb800108,0xcb0d0101,0xfb800108,0xcb0e0101,0xfb800108,0xcb0f0101,
	0xfb800108,0xcb100101,0xfb800108,0xcb110101,0xfb800108,0xcb120101,0xfb800108,0xcb130101,
	0xfb800108,0xcb140101,0xfb800108,0xcb150101,0xfb800108,0xcb160101,0xfb800108,0xcb170101,
	0xfb800108,0xcb180101,0xfb800108,0xcb190101,0xfb800108,0xcb1a0101,0xfb800108,0xcb1b0101,
	0xfb800108,0xcb1c0101,0xfb800108,0xcb1d0101,0xfb800108,0xcb1e0101,0xfb800108,0xcb1f0101,
	0xfb800108,0xcb200101,0xfb800108,0xcb210101,0xfb800108,0xcb220101,0xfb800108,0xcb230101,
	0xfb800108,0xcb240101,0xfb800108,0xcb250101,0xfb800108,0xcb260101,0xfb800108,0xcb270101,
	0xfb800108,0xcb280101,0xfb800108,0xcb290101,0xfb800108,0xcb2a0101,0xfb800108,0xcb2b0101,
	0xfb800108,0xcb2c0101,0xfb800108,0xcb2d0101,0xfb800108,0xcb2e0101,0xfb800108,0xcb2f0101,
	0xfb800108,0xcb300101,0xfb800108,0xcb310101,0xfb800108,0xcb320101,0xfb800108,0xcb330101,
	0xfb800108,0xcb340101,0xfb800108,0xcb350101,0xfb800108,0xcb360101,0xfb800108,0xcb370101,
	0xfb800108,0xcb380101,0xfb800108,0xcb390101,0xfb800108,0xcb3a0101,0xfb800108,0xcb3b0101,
	0xfb800108,0xcb3c0101,0xfb800108,0xcb3d0101,0xfb800108,0xcb3e0101,0xfb800108,0xcb3f0101,
	0xfb800108,0xcb400101,0xfb800108,0xcb410101,0xfb800108,0xcb420101,0xfb800108,0xcb430101,
	0xfb800108,0xcb440101,0xfb800108,0xcb450101,0xfb800108,0xcb460101,0xfb800108,0xcb470101,
	0xfb800108,0xcb480101,0xfb800108,0xcb490101,0xfb800108,0xcb4a0101,0xfb800108,0xcb4b0101,
	0xfb800108,0xcb4c0101,0xfb800108,0xcb4d0101,0xfb800108,0xcb4e0101,0xfb800108,0xcb4f0101,
	0xfb800108,0xcb500101,0xfb800108,0xcb510101,0xfb800108,0xcb520101,0xfb800108,0xcb530101,
	0xfb800108,0xcb540101,0xfb800108,0xcb550101,0xfb800108,0xcb560101,0xfb800108,0xcb570101,
	0xfb800108,0xcb580101,0xfb800108,0xcb590101,0xfb800108,0xcb5a0101,0xfb800108,0xcb5b0101,
	0xfb800108,0xcb5c0101,0xfb800108,0xcb5d0101,0xfb800108,0xcb5e0101,0xfb800108,0xcb5f0101,
	0xfb800108,0xcb600101,0xfb800108,0xcb610101,0xfb800108,0xcb620101,0xfb800108,0xcb630101,
	0xfb800108,0xcb640101,0xfb800108,0xcb650101,0xfb800108,0xcb660101,0xfb800108,0xcb670101,
	0xfb800108,0xcb680101,0xfb800108,0xcb690101,0xfb800108,0xcb6a0101,0xfb800108,0xcb6b0101,
	0xfb800108,0xcb6c0101,0xfb800108,0xcb6d0101,0xfb800108,0xcb6e0101,0xfb800108,0xcb6f0101,
	0xfb800108,0xcb700101,0xfb800108,0xcb710101,0xfb800108,0xcb720101,0xfb800108,0xcb730101,
	0xfb800108,0xcb740101,0xfb800108,0xcb750101,0xfb800108,0xcb760101,0xfb800108,0xcb770101,
	0xfb800108,0xcb780101,0xfb800108,0xcb790101,0xfb800108,0xcb7a0101,0xfb800108,0xcb7b0101,
	0xfb800108,0xcb7c0101,0xfb800108,0xcb7d0101,0xfb800108,0xcb7e0101,0xfb800108,0xcb7f0101,
	0xfb800108,0xcb800101,0xfb800108,0xcb810101,0xfb800108,0xcb820101,0xfb800108,0xcb830101,
	0xfb800108,0xcb840101,0xfb800108,0xcb850101,0xfb800108,0xcb860101,0xfb800108,0xcb870101,
	0xfb800108,0xcb880101,0xfb800108,0xcb890101,0xfb800108,0xcb8a0101,0xfb800108,0xcb8b0101,
	0xfb800108,0xcb8c0101,0xfb800108,0xcb8d0101,0xfb800108,0xcb8e0101,0xfb800108,0xcb8f0101,
	0xfb800108,0xcb900101,0xfb800108,0xcb910101,0xfb800108,0xcb920101,0xfb800108,0xcb930101,
	0xfb800108,0xcb940101,0xfb800108,0xcb950101,0xfb800108,0xcb960101,0xfb800108,0xcb970101,
	0xfb800108,0xcb980101,0xfb800108,0xcb990101,0xfb800108,0xcb9a0101,0xfb800108,0xcb9b0101,
	0xfb800108,0xcb9c0101,0xfb800108,0xcb9d0101,0xfb800108,0xcb9e0101,0xfb800108,0xcb9f0101,
	0xfb800108,0xcba00101,0xfb800108,0xcba10101,0xfb800108,0xcba20101,0xfb800108,0xcba30101,
	0xfb800108,0xcba40101,0xfb800108,0xcba50101,0xfb800108,0xcba60101,0xfb800108,0xcba70101,
	0xfb800108,0xcba80101,0xfb800108,0xcba90101,0xfb800108,0xcbaa0101,0xfb800108,0xcbab0101,
	0xfb800108,0xcbac0101,0xfb800108,0xcbad0101,0xfb800108,0xcbae0101,0xfb800108,0xcbaf0101,
	0xfb800108,0xcbb00101,0xfb800108,0xcbb10101,0xfb800108,0xcbb20101,0xfb800108,0xcbb30101,
	0xfb800108,0xcbb40101,0xfb800108,0xcbb50101,0xfb800108,0xcbb60101,0xfb800108,0xcbb70101,
	0xfb800108,0xcbb80101,0xfb800108,0xcbb90101,0xfb800108,0xcbba0101,0xfb800108,0xcbbb0101,
	0xfb800108,0xcbbc0101,0xfb800108,0xcbbd0101,0xfb800108,0xcbbe0101,0xfb800108,0xcbbf0101,
	0xfb800108,0xcbc00101,0xfb800108,0xcbc10101,0xfb800108,0xcbc20101,0xfb800108,0xcbc30101,
	0xfb800108,0xcbc40101,0xfb800108,0xcbc50101,0xfb800108,0xcbc60101,0xfb800108,0xcbc70101,
	0xfb800108,0xcbc80101,0xfb800108,0xcbc90101,0xfb800108,0xcbca0101,0xfb800108,0xcbcb0101,
	0xfb800108,0xcbcc0101,0xfb800108,0xcbcd0101,0xfb800108,0xcbce0101,0xfb800108,0xcbcf0101,
	0xfb800108,0xcbd00101,0xfb800108,0xcbd10101,0xfb800108,0xcbd20101,0xfb800108,0xcbd30101,
	0xfb800108,0xcbd40101,0xfb800108,0xcbd50101,0xfb800108,0xcbd60101,0xfb800108,0xcbd70101,
	0xfb800108,0xcbd80101,0xfb800108,0xcbd90101,0xfb800108,0xcbda0101,0xfb800108,0xcbdb0101,
	0xfb800108,0xcbdc0101,0xfb800108,0xcbdd0101,0xfb800108,0xcbde0101,0xfb800108,0xcbdf0101,
	0xfb800108,0xcbe00101,0xfb800108,0xcbe10101,0xfb800108,0xcbe20101,0xfb800108,0xcbe30101,
	0xfb800108,0xcbe40101,0xfb800108,0xcbe50101,0xfb800108,0xcbe60101,0xfb800108,0xcbe70101,
	0xfb800108,0xcbe80101,0xfb800108,0xcbe90101,0xfb800108,0xcbea0101,0xfb800108,0xcbeb0101,
	0xfb800108,0xcbec0101,0xfb800108,0xcbed0101,0xfb800108,0xcbee0101,0xfb800108,0xcbef0101,
	0xfb800108,0xcbf00101,0xfb800108,0xcbf10101,0xfb800108,0xcbf20101,0xfb800108,0xcbf30101,
	0xfb800108,0xcbf40101,0xfb800108,0xcbf50101,0xfb800108,0xcbf60101,0xfb800108,0xcbf70101,
	0xfb800108,0xcbf80101,0xfb800108,0xcbf90101,0xfb800108,0xcbfa0101,0xfb800108,0xcbfb0101,
	0xfb800108,0xcbfc0101,0xfb800108,0xcbfd0101,0xfb800108,0xcbfe0101,0xfb800108,0xcbff0101,
	0xfb800108,0xcc000101,0xfb800108,0xcc010101,0xfb800108,0xcc020101,0xfb800108,0xcc030101,
	0xfb800108,0xcc040101,0xfb800108,0xcc050101,0xfb800108,0xcc060101,0xfb800108,0xcc070101,
	0xfb800108,0xcc080101,0xfb800108,0xcc090101,0xfb800108,0xcc0a0101,0xfb800108,0xcc0b0101,
	0xfb800108,0xcc0c0101,0xfb800108,0xcc0d0101,0xfb800108,0xcc0e0101,0xfb800108,0xcc0f0101,
	0xfb800108,0xcc100101,0xfb800108,0xcc110101,0xfb800108,0xcc120101,0xfb800108,0xcc130101,
	0xfb800108,0xcc140101,0xfb800108,0xcc150101,0xfb800108,0xcc160101,0xfb800108,0xcc170101,
	0xfb800108,0xcc180101,0xfb800108,0xcc190101,0xfb800108,0xcc1a0101,0xfb800108,0xcc1b0101,
	0xfb800108,0xcc1c0101,0xfb800108,0xcc1d0101,0xfb800108,0xcc1e0101,0xfb800108,0xcc1f0101,
	0xfb800108,0xcc200101,0xfb800108,0xcc210101,0xfb800108,0xcc220101,0xfb800108,0xcc230101,
	0xfb800108,0xcc240101,0xfb800108,0xcc250101,0xfb800108,0xcc260101,0xfb800108,0xcc270101,
	0xfb800108,0xcc280101,0xfb800108,0xcc290101,0xfb800108,0xcc2a0101,0xfb800108,0xcc2b0101,
	0xfb800108,0xcc2c0101,0xfb800108,0xcc2d0101,0xfb800108,0xcc2e0101,0xfb800108,0xcc2f0101,
	0xfb800108,0xcc300101,0xfb800108,0xcc310101,0xfb800108,0xcc320101,0xfb800108,0xcc330101,
	0xfb800108,0xcc340101,0xfb800108,0xcc350101,0xfb800108,0xcc360101,0xfb800108,0xcc370101,
	0xfb800108,0xcc380101,0xfb800108,0xcc390101,0xfb800108,0xcc3a0101,0xfb800108,0xcc3b0101,
	0xfb800108,0xcc3c0101,0xfb800108,0xcc3d0101,0xfb800108,0xcc3e0101,0xfb800108,0xcc3f0101,
	0xfb800108,0xcc400101,0xfb800108,0xcc410101,0xfb800108,0xcc420101,0xfb800108,0xcc430101,
	0xfb800108,0xcc440101,0xfb800108,0xcc450101,0xfb800108,0xcc460101,0xfb800108,0xcc470101,
	0xfb800108,0xcc480101,0xfb800108,0xcc490101,0xfb800108,0xcc4a0101,0xfb800108,0xcc4b0101,
	0xfb800108,0xcc4c0101,0xfb800108,0xcc4d0101,0xfb800108,0xcc4e0101,0xfb800108,0xcc4f0101,
	0xfb800108,0xcc500101,0xfb800108,0xcc510101,0xfb800108,0xcc520101,0xfb800108,0xcc530101,
	0xfb800108,0xcc540101,0xfb800108,0xcc550101,0xfb800108,0xcc560101,0xfb800108,0xcc570101,
	0xfb800108,0xcc580101,0xfb800108,0xcc590101,0xfb800108,0xcc5a0101,0xfb800108,0xcc5b0101,
	0xfb800108,0xcc5c0101,0xfb800108,0xcc5d0101,0xfb800108,0xcc5e0101,0xfb800108,0xcc5f0101,
	0xfb800108,0xcc600101,0xfb800108,0xcc610101,0xfb800108,0xcc620101,0xfb800108,0xcc630101,
	0xfb800108,0xcc640101,0xfb800108,0xcc650101,0xfb800108,0xcc660101,0xfb800108,0xcc670101,
	0xfb800108,0xcc680101,0xfb800108,0xcc690101,0xfb800108,0xcc6a0101,0xfb800108,0xcc6b0101,
	0xfb800108,0xcc6c0101,0xfb800108,0xcc6d0101,0xfb800108,0xcc6e0101,0xfb800108,0xcc6f0101,
	0xfb800108,0xcc700101,0xfb800108,0xcc710101,0xfb800108,0xcc720101,0xfb800108,0xcc730101,
	0xfb800108,0xcc740101,0xfb800108,0xcc750101,0xfb800108,0xcc760101,0xfb800108,0xcc770101,
	0xfb800108,0xcc780101,0xfb800108,0xcc790101,0xfb800108,0xcc7a0101,0xfb800108,0xcc7b0101,
	0xfb800108,0xcc7c0101,0xfb800108,0xcc7d0101,0xfb800108,0xcc7e0101,0xfb800108,0xcc7f0101,
	0xfb800108,0xcc800101,0xfb800108,0xcc810101,0xfb800108,0xcc820101,0xfb800108,0xcc830101,
	0xfb800108,0xcc840101,0xfb800108,0xcc850101,0xfb800108,0xcc860101,0xfb800108,0xcc870101,
	0xfb800108,0xcc880101,0xfb800108,0xcc890101,0xfb800108,0xcc8a0101,0xfb800108,0xcc8b0101,
	0xfb800108,0xcc8c0101,0xfb800108,0xcc8d0101,0xfb800108,0xcc8e0101,0xfb800108,0xcc8f0101,
	0xfb800108,0xcc900101,0xfb800108,0xcc910101,0xfb800108,0xcc920101,0xfb800108,0xcc930101,
	0xfb800108,0xcc940101,0xfb800108,0xcc950101,0xfb800108,0xcc960101,0xfb800108,0xcc970101,
	0xfb800108,0xcc980101,0xfb800108,0xcc990101,0xfb800108,0xcc9a0101,0xfb800108,0xcc9b0101,
	0xfb800108,0xcc9c0101,0xfb800108,0xcc9d0101,0xfb800108,0xcc9e0101,0xfb800108,0xcc9f0101,
	0xfb800108,0xcca00101,0xfb800108,0xcca10101,0xfb800108,0xcca20101,0xfb800108,0xcca30101,
	0xfb800108,0xcca40101,0xfb800108,0xcca50101,0xfb800108,0xcca60101,0xfb800108,0xcca70101,
	0xfb800108,0xcca80101,0xfb800108,0xcca90101,0xfb800108,0xccaa0101,0xfb800108,0xccab0101,
	0xfb800108,0xccac0101,0xfb800108,0xccad0101,0xfb800108,0xccae0101,0xfb800108,0xccaf0101,
	0xfb800108,0xccb00101,0xfb800108,0xccb10101,0xfb800108,0xccb20101,0xfb800108,0xccb30101,
	0xfb800108,0xccb40101,0xfb800108,0xccb50101,0xfb800108,0xccb60101,0xfb800108,0xccb70101,
	0xfb800108,0xccb80101,0xfb800108,0xccb90101,0xfb800108,0xccba0101,0xfb800108,0xccbb0101,
	0xfb800108,0xccbc0101,0xfb800108,0xccbd0101,0xfb800108,0xccbe0101,0xfb800108,0xccbf0101,
	0xfb800108,0xccc00101,0xfb800108,0xccc10101,0xfb800108,0xccc20101,0xfb800108,0xccc30101,
	0xfb800108,0xccc40101,0xfb800108,0xccc50101,0xfb800108,0xccc60101,0xfb800108,0xccc70101,
	0xfb800108,0xccc80101,0xfb800108,0xccc90101,0xfb800108,0xccca0101,0xfb800108,0xcccb0101,
	0xfb800108,0xcccc0101,0xfb800108,0xcccd0101,0xfb800108,0xccce0101,0xfb800108,0xcccf0101,
	0xfb800108,0xccd00101,0xfb800108,0xccd10101,0xfb800108,0xccd20101,0xfb800108,0xccd30101,
	0xfb800108,0xccd40101,0xfb800108,0xccd50101,0xfb800108,0xccd60101,0xfb800108,0xccd70101,
	0xfb800108,0xccd80101,0xfb800108,0xccd90101,0xfb800108,0xccda0101,0xfb800108,0xccdb0101,
	0xfb800108,0xccdc0101,0xfb800108,0xccdd0101,0xfb800108,0xccde0101,0xfb800108,0xccdf0101,
	0xfb800108,0xcce00101,0xfb800108,0xcce10101,0xfb800108,0xcce20101,0xfb800108,0xcce30101,
	0xfb800108,0xcce40101,0xfb800108,0xcce50101,0xfb800108,0xcce60101,0xfb800108,0xcce70101,
	0xfb800108,0xcce80101,0xfb800108,0xcce90101,0xfb800108,0xccea0101,0xfb800108,0xcceb0101,
	0xfb800108,0xccec0101,0xfb800108,0xcced0101,0xfb800108,0xccee0101,0xfb800108,0xccef0101,
	0xfb800108,0xccf00101,0xfb800108,0xccf10101,0xfb800108,0xccf20101,0xfb800108,0xccf30101,
	0xfb800108,0xccf40101,0xfb800108,0xccf50101,0xfb800108,0xccf60101,0xfb800108,0xccf70101,
	0xfb800108,0xccf80101,0xfb800108,0xccf90101,0xfb800108,0xccfa0101,0xfb800108,0xccfb0101,
	0xfb800108,0xccfc0101,0xfb800108,0xccfd0101,0xfb800108,0xccfe0101,0xfb800108,0xccff0101,
	0xfb800108,0xcd000101,0xfb800108,0xcd010101,0xfb800108,0xcd020101,0xfb800108,0xcd030101,
	0xfb800108,0xcd040101,0xfb800108,0xcd050101,0xfb800108,0xcd060101,0xfb800108,0xcd070101,
	0xfb800108,0xcd080101,0xfb800108,0xcd090101,0xfb800108,0xcd0a0101,0xfb800108,0xcd0b0101,
	0xfb800108,0xcd0c0101,0xfb800108,0xcd0d0101,0xfb800108,0xcd0e0101,0xfb800108,0xcd0f0101,
	0xfb800108,0xcd100101,0xfb800108,0xcd110101,0xfb800108,0xcd120101,0xfb800108,0xcd130101,
	0xfb800108,0xcd140101,0xfb800108,0xcd150101,0xfb800108,0xcd160101,0xfb800108,0xcd170101,
	0xfb800108,0xcd180101,0xfb800108,0xcd190101,0xfb800108,0xcd1a0101,0xfb800108,0xcd1b0101,
	0xfb800108,0xcd1c0101,0xfb800108,0xcd1d0101,0xfb800108,0xcd1e0101,0xfb800108,0xcd1f0101,
	0xfb800108,0xcd200101,0xfb800108,0xcd210101,0xfb800108,0xcd220101,0xfb800108,0xcd230101,
	0xfb800108,0xcd240101,0xfb800108,0xcd250101,0xfb800108,0xcd260101,0xfb800108,0xcd270101,
	0xfb800108,0xcd280101,0xfb800108,0xcd290101,0xfb800108,0xcd2a0101,0xfb800108,0xcd2b0101,
	0xfb800108,0xcd2c0101,0xfb800108,0xcd2d0101,0xfb800108,0xcd2e0101,0xfb800108,0xcd2f0101,
	0xfb800108,0xcd300101,0xfb800108,0xcd310101,0xfb800108,0xcd320101,0xfb800108,0xcd330101,
	0xfb800108,0xcd340101,0xfb800108,0xcd350101,0xfb800108,0xcd360101,0xfb800108,0xcd370101,
	0xfb800108,0xcd380101,0xfb800108,0xcd390101,0xfb800108,0xcd3a0101,0xfb800108,0xcd3b0101,
	0xfb800108,0xcd3c0101,0xfb800108,0xcd3d0101,0xfb800108,0xcd3e0101,0xfb800108,0xcd3f0101,
	0xfb800108,0xcd400101,0xfb800108,0xcd410101,0xfb800108,0xcd420101,0xfb800108,0xcd430101,
	0xfb800108,0xcd440101,0xfb800108,0xcd450101,0xfb800108,0xcd460101,0xfb800108,0xcd470101,
	0xfb800108,0xcd480101,0xfb800108,0xcd490101,0xfb800108,0xcd4a0101,0xfb800108,0xcd4b0101,
	0xfb800108,0xcd4c0101,0xfb800108,0xcd4d0101,0xfb800108,0xcd4e0101,0xfb800108,0xcd4f0101,
	0xfb800108,0xcd500101,0xfb800108,0xcd510101,0xfb800108,0xcd520101,0xfb800108,0xcd530101,
	0xfb800108,0xcd540101,0xfb800108,0xcd550101,0xfb800108,0xcd560101,0xfb800108,0xcd570101,
	0xfb800108,0xcd580101,0xfb800108,0xcd590101,0xfb800108,0xcd5a0101,0xfb800108,0xcd5b0101,
	0xfb800108,0xcd5c0101,0xfb800108,0xcd5d0101,0xfb800108,0xcd5e0101,0xfb800108,0xcd5f0101,
	0xfb800108,0xcd600101,0xfb800108,0xcd610101,0xfb800108,0xcd620101,0xfb800108,0xcd630101,
	0xfb800108,0xcd640101,0xfb800108,0xcd650101,0xfb800108,0xcd660101,0xfb800108,0xcd670101,
	0xfb800108,0xcd680101,0xfb800108,0xcd690101,0xfb800108,0xcd6a0101,0xfb800108,0xcd6b0101,
	0xfb800108,0xcd6c0101,0xfb800108,0xcd6d0101,0xfb800108,0xcd6e0101,0xfb800108,0xcd6f0101,
	0xfb800108,0xcd700101,0xfb800108,0xcd710101,0xfb800108,0xcd720101,0xfb800108,0xcd730101,
	0xfb800108,0xcd740101,0xfb800108,0xcd750101,0xfb800108,0xcd760101,0xfb800108,0xcd770101,
	0xfb800108,0xcd780101,0xfb800108,0xcd790101,0xfb800108,0xcd7a0101,0xfb800108,0xcd7b0101,
	0xfb800108,0xcd7c0101,0xfb800108,0xcd7d0101,0xfb800108,0xcd7e0101,0xfb800108,0xcd7f0101,
	0xfb800108,0xcd800101,0xfb800108,0xcd810101,0xfb800108,0xcd820101,0xfb800108,0xcd830101,
	0xfb800108,0xcd840101,0xfb800108,0xcd850101,0xfb800108,0xcd860101,0xfb800108,0xcd870101,
	0xfb800108,0xcd880101,0xfb800108,0xcd890101,0xfb800108,0xcd8a0101,0xfb800108,0xcd8b0101,
	0xfb800108,0xcd8c0101,0xfb800108,0xcd8d0101,0xfb800108,0xcd8e0101,0xfb800108,0xcd8f0101,
	0xfb800108,0xcd900101,0xfb800108,0xcd910101,0xfb800108,0xcd920101,0xfb800108,0xcd930101,
	0xfb800108,0xcd940101,0xfb800108,0xcd950101,0xfb800108,0xcd960101,0xfb800108,0xcd970101,
	0xfb800108,0xcd980101,0xfb800108,0xcd990101,0xfb800108,0xcd9a0101,0xfb800108,0xcd9b0101,
	0xfb800108,0xcd9c0101,0xfb800108,0xcd9d0101,0xfb800108,0xcd9e0101,0xfb800108,0xcd9f0101,
	0xfb800108,0xcda00101,0xfb800108,0xcda10101,0xfb800108,0xcda20101,0xfb800108,0xcda30101,
	0xfb800108,0xcda40101,0xfb800108,0xcda50101,0xfb800108,0xcda60101,0xfb800108,0xcda70101,
	0xfb800108,0xcda80101,0xfb800108,0xcda90101,0xfb800108,0xcdaa0101,0xfb800108,0xcdab0101,
	0xfb800108,0xcdac0101,0xfb800108,0xcdad0101,0xfb800108,0xcdae0101,0xfb800108,0xcdaf0101,
	0xfb800108,0xcdb00101,0xfb800108,0xcdb10101,0xfb800108,0xcdb20101,0xfb800108,0xcdb30101,
	0xfb800108,0xcdb40101,0xfb800108,0xcdb50101,0xfb840108,0x80000101,0xfb840108,0x80010101,
	0xfb840108,0x80020101,0xfb840108,0x80030101,0xfb840108,0x80040101,0xfb840108,0x80050101,
	0xfb840108,0x80060101,0xfb840108,0x80070101,0xfb840108,0x80080101,0xfb840108,0x80090101,
	0xfb840108,0x800a0101,0xfb840108,0x800b0101,0xfb840108,0x800c0101,0xfb840108,0x800d0101,
	0xfb840108,0x800e0101,0xfb840108,0x800f0101,0xfb840108,0x80100101,0xfb840108,0x80110101,
	0xfb840108,0x80120101,0xfb840108,0x80130101,0xfb840108,0x80140101,0xfb840108,0x80150101,
	0xfb840108,0x80160101,0xfb840108,0x80170101,0xfb840108,0x80180101,0xfb840108,0x80190101,
	0xfb840108,0x801a0101,0xfb840108,0x801b0101,0xfb840108,0x801c0101,0xfb840108,0x801d0101,
	0xfb840108,0x801e0101,0xfb840108,0x801f0101,0xfb840108,0x80200101,0xfb840108,0x80210101,
	0xfb840108,0x80220101,0xfb840108,0x80230101,0xfb840108,0x80240101,0xfb840108,0x80250101,
	0xfb840108,0x80260101,0xfb840108,0x80270101,0xfb840108,0x80280101,0xfb840108,0x80290101,
	0xfb840108,0x802a0101,0xfb840108,0x802b0101,0xfb840108,0x802c0101,0xfb840108,0x802d0101,
	0xfb840108,0x802e0101,0xfb840108,0x802f0101,0xfb840108,0x80300101,0xfb840108,0x80310101,
	0xfb840108,0x80320101,0xfb840108,0x80330101,0xfb840108,0x80340101,0xfb840108,0x80350101,
	0xfb840108,0x80360101,0xfb840108,0x80370101,0xfb840108,0x80380101,0xfb840108,0x80390101,
	0xfb840108,0x803a0101,0xfb840108,0x803b0101,0xfb840108,0x803c0101,0xfb840108,0x803d0101,
	0xfb840108,0x803e0101,0xfb840108,0x803f0101,0xfb840108,0x80400101,0xfb840108,0x80410101,
	0xfb840108,0x80420101,0xfb840108,0x80430101,0xfb840108,0x80440101,0xfb840108,0x80450101,
	0xfb840108,0x80460101,0xfb840108,0x80470101,0xfb840108,0x80480101,0xfb840108,0x80490101,
	0xfb840108,0x804a0101,0xfb840108,0x804b0101,0xfb840108,0x804c0101,0xfb840108,0x804d0101,
	0xfb840108,0x804e0101,0xfb840108,0x804f0101,0xfb840108,0x80500101,0xfb840108,0x80510101,
	0xfb840108,0x80520101,0xfb840108,0x80530101,0xfb840108,0x80540101,0xfb840108,0x80550101,
	0xfb840108,0x80560101,0xfb840108,0x80570101,0xfb840108,0x80580101,0xfb840108,0x80590101,
	0xfb840108,0x805a0101,0xfb840108,0x805b0101,0xfb840108,0x805c0101,0xfb840108,0x805d0101,
	0xfb840108,0x805e0101,0xfb840108,0x805f0101,0xfb840108,0x80600101,0xfb840108,0x80610101,
	0xfb840108,0x80620101,0xfb840108,0x80630101,0xfb840108,0x80640101,0xfb840108,0x80650101,
	0xfb840108,0x80660101,0xfb840108,0x80670101,0xfb840108,0x80680101,0xfb840108,0x80690101,
	0xfb840108,0x806a0101,0xfb840108,0x806b0101,0xfb840108,0x806c0101,0xfb840108,0x806d0101,
	0xfb840108,0x806e0101,0xfb840108,0x806f0101,0xfb840108,0x80700101,0xfb840108,0x80710101,
	0xfb840108,0x80720101,0xfb840108,0x80730101,0xfb840108,0x80740101,0xfb840108,0x80750101,
	0xfb840108,0x80760101,0xfb840108,0x80770101,0xfb840108,0x80780101,0xfb840108,0x80790101,
	0xfb840108,0x807a0101,0xfb840108,0x807b0101,0xfb840108,0x807c0101,0xfb840108,0x807d0101,
	0xfb840108,0x807e0101,0xfb840108,0x807f0101,0xfb840108,0x80800101,0xfb840108,0x80810101,
	0xfb840108,0x80820101,0xfb840108,0x80830101,0xfb840108,0x80840101,0xfb840108,0x80850101,
	0xfb840108,0x80860101,0xfb840108,0x80870101,0xfb840108,0x80880101,0xfb840108,0x80890101,
	0xfb840108,0x808a0101,0xfb840108,0x808b0101,0xfb840108,0x808c0101,0xfb840108,0x808d0101,
	0xfb840108,0x808e0101,0xfb840108,0x808f0101,0xfb840108,0x80900101,0xfb840108,0x80910101,
	0xfb840108,0x80920101,0xfb840108,0x80930101,0xfb840108,0x80940101,0xfb840108,0x80950101,
	0xfb840108,0x80960101,0xfb840108,0x80970101,0xfb840108,0x80980101,0xfb840108,0x80990101,
	0xfb840108,0x809a0101,0xfb840108,0x809b0101,0xfb840108,0x809c0101,0xfb840108,0x809d0101,
	0xfb840108,0x809e0101,0xfb840108,0x809f0101,0xfb840108,0x80a00101,0xfb840108,0x80a10101,
	0xfb840108,0x80a20101,0xfb840108,0x80a30101,0xfb840108,0x80a40101,0xfb840108,0x80a50101,
	0xfb840108,0x80a60101,0xfb840108,0x80a70101,0xfb840108,0x80a80101,0xfb840108,0x80a90101,
	0xfb840108,0x80aa0101,0xfb840108,0x80ab0101,0xfb840108,0x80ac0101,0xfb840108,0x80ad0101,
	0xfb840108,0x80ae0101,0xfb840108,0x80af0101,0xfb840108,0x80b00101,0xfb840108,0x80b10101,
	0xfb840108,0x80b20101,0xfb840108,0x80b30101,0xfb840108,0x80b40101,0xfb840108,0x80b50101,
	0xfb840108,0x80b60101,0xfb840108,0x80b70101,0xfb840108,0x80b80101,0xfb840108,0x80b90101,
	0xfb840108,0x80ba0101,0xfb840108,0x80bb0101,0xfb840108,0x80bc0101,0xfb840108,0x80bd0101,
	0xfb840108,0x80be0101,0xfb840108,0x80bf0101,0xfb840108,0x80c00101,0xfb840108,0x80c10101,
	0xfb840108,0x80c20101,0xfb840108,0x80c30101,0xfb840108,0x80c40101,0xfb840108,0x80c50101,
	0xfb840108,0x80c60101,0xfb840108,0x80c70101,0xfb840108,0x80c80101,0xfb840108,0x80c90101,
	0xfb840108,0x80ca0101,0xfb840108,0x80cb0101,0xfb840108,0x80cc0101,0xfb840108,0x80cd0101,
	0xfb840108,0x80ce0101,0xfb840108,0x80cf0101,0xfb840108,0x80d00101,0xfb840108,0x80d10101,
	0xfb840108,0x80d20101,0xfb840108,0x80d30101,0xfb840108,0x80d40101,0xfb840108,0x80d50101,
	0xfb840108,0x80d60101,0xfb840108,0x80d70101,0xfb840108,0x80d80101,0xfb840108,0x80d90101,
	0xfb840108,0x80da0101,0xfb840108,0x80db0101,0xfb840108,0x80dc0101,0xfb840108,0x80dd0101,
	0xfb840108,0x80de0101,0xfb840108,0x80df0101,0xfb840108,0x80e00101,0xfb840108,0x80e10101,
	0xfb840108,0x80e20101,0xfb840108,0x80e30101,0xfb840108,0x80e40101,0xfb840108,0x80e50101,
	0xfb840108,0x80e60101,0xfb840108,0x80e70101,0xfb840108,0x80e80101,0xfb840108,0x80e90101,
	0xfb840108,0x80ea0101,0xfb840108,0x80eb0101,0xfb840108,0x80ec0101,0xfb840108,0x80ed0101,
	0xfb840108,0x80ee0101,0xfb840108,0x80ef0101,0xfb840108,0x80f00101,0xfb840108,0x80f10101,
	0xfb840108,0x80f20101,0xfb840108,0x80f30101,0xfb840108,0x80f40101,0xfb840108,0x80f50101,
	0xfb840108,0x80f60101,0xfb840108,0x80f70101,0xfb840108,0x80f80101,0xfb840108,0x80f90101,
	0xfb840108,0x80fa0101,0xfb840108,0x80fb0101,0xfb840108,0x80fc0101,0xfb840108,0x80fd0101,
	0xfb840108,0x80fe0101,0xfb840108,0x80ff0101,0xfb840108,0x81000101,0xfb840108,0x81010101,
	0xfb840108,0x81020101,0xfb840108,0x81030101,0xfb840108,0x81040101,0xfb840108,0x81050101,
	0xfb840108,0x81060101,0xfb840108,0x81070101,0xfb840108,0x81080101,0xfb840108,0x81090101,
	0xfb840108,0x810a0101,0xfb840108,0x810b0101,0xfb840108,0x810c0101,0xfb840108,0x810d0101,
	0xfb840108,0x810e0101,0xfb840108,0x810f0101,0xfb840108,0x81100101,0xfb840108,0x81110101,
	0xfb840108,0x81120101,0xfb840108,0x81130101,0xfb840108,0x81140101,0xfb840108,0x81150101,
	0xfb840108,0x81160101,0xfb840108,0x81170101,0xfb840108,0x81180101,0xfb840108,0x81190101,
	0xfb840108,0x811a0101,0xfb840108,0x811b0101,0xfb840108,0x811c0101,0xfb840108,0x811d0101,
	0xfb840108,0x811e0101,0xfb840108,0x811f0101,0xfb840108,0x81200101,0xfb840108,0x81210101,
	0xfb840108,0x81220101,0xfb840108,0x81230101,0xfb840108,0x81240101,0xfb840108,0x81250101,
	0xfb840108,0x81260101,0xfb840108,0x81270101,0xfb840108,0x81280101,0xfb840108,0x81290101,
	0xfb840108,0x812a0101,0xfb840108,0x812b0101,0xfb840108,0x812c0101,0xfb840108,0x812d0101,
	0xfb840108,0x812e0101,0xfb840108,0x812f0101,0xfb840108,0x81300101,0xfb840108,0x81310101,
	0xfb840108,0x81320101,0xfb840108,0x81330101,0xfb840108,0x81340101,0xfb840108,0x81350101,
	0xfb840108,0x81360101,0xfb840108,0x81370101,0xfb840108,0x81380101,0xfb840108,0x81390101,
	0xfb840108,0x813a0101,0xfb840108,0x813b0101,0xfb840108,0x813c0101,0xfb840108,0x813d0101,
	0xfb840108,0x813e0101,0xfb840108,0x813f0101,0xfb840108,0x81400101,0xfb840108,0x81410101,
	0xfb840108,0x81420101,0xfb840108,0x81430101,0xfb840108,0x81440101,0xfb840108,0x81450101,
	0xfb840108,0x81460101,0xfb840108,0x81470101,0xfb840108,0x81480101,0xfb840108,0x81490101,
	0xfb840108,0x814a0101,0xfb840108,0x814b0101,0xfb840108,0x814c0101,0xfb840108,0x814d0101,
	0xfb840108,0x814e0101,0xfb840108,0x814f0101,0xfb840108,0x81500101,0xfb840108,0x81510101,
	0xfb840108,0x81520101,0xfb840108,0x81530101,0xfb840108,0x81540101,0xfb840108,0x81550101,
	0xfb840108,0x81560101,0xfb840108,0x81570101,0xfb840108,0x81580101,0xfb840108,0x81590101,
	0xfb840108,0x815a0101,0xfb840108,0x815b0101,0xfb840108,0x815c0101,0xfb840108,0x815d0101,
	0xfb840108,0x815e0101,0xfb840108,0x815f0101,0xfb840108,0x81600101,0xfb840108,0x81610101,
	0xfb840108,0x81620101,0xfb840108,0x81630101,0xfb840108,0x81640101,0xfb840108,0x81650101,
	0xfb840108,0x81660101,0xfb840108,0x81670101,0xfb840108,0x81680101,0xfb840108,0x81690101,
	0xfb840108,0x816a0101,0xfb840108,0x816b0101,0xfb840108,0x816c0101,0xfb840108,0x816d0101,
	0xfb840108,0x816e0101,0xfb840108,0x816f0101,0xfb840108,0x81700101,0xfb840108,0x81710101,
	0xfb840108,0x81720101,0xfb840108,0x81730101,0xfb840108,0x81740101,0xfb840108,0x81750101,
	0xfb840108,0x81760101,0xfb840108,0x81770101,0xfb840108,0x81780101,0xfb840108,0x81790101,
	0xfb840108,0x817a0101,0xfb840108,0x817b0101,0xfb840108,0x817c0101,0xfb840108,0x817d0101,
	0xfb840108,0x817e0101,0xfb840108,0x817f0101,0xfb840108,0x81800101,0xfb840108,0x81810101,
	0xfb840108,0x81820101,0xfb840108,0x81830101,0xfb840108,0x81840101,0xfb840108,0x81850101,
	0xfb840108,0x81860101,0xfb840108,0x81870101,0xfb840108,0x81880101,0xfb840108,0x81890101,
	0xfb840108,0x818a0101,0xfb840108,0x818b0101,0xfb840108,0x818c0101,0xfb840108,0x818d0101,
	0xfb840108,0x818e0101,0xfb840108,0x818f0101,0xfb840108,0x81900101,0xfb840108,0x81910101,
	0xfb840108,0x81920101,0xfb840108,0x81930101,0xfb840108,0x81940101,0xfb840108,0x81950101,
	0xfb840108,0x81960101,0xfb840108,0x81970101,0xfb840108,0x81980101,0xfb840108,0x81990101,
	0xfb840108,0x819a0101,0xfb840108,0x819b0101,0xfb840108,0x819c0101,0xfb840108,0x819d0101,
	0xfb840108,0x819e0101,0xfb840108,0x819f0101,0xfb840108,0x81a00101,0xfb840108,0x81a10101,
	0xfb840108,0x81a20101,0xfb840108,0x81a30101,0xfb840108,0x81a40101,0xfb840108,0x81a50101,
	0xfb840108,0x81a60101,0xfb840108,0x81a70101,0xfb840108,0x81a80101,0xfb840108,0x81a90101,
	0xfb840108,0x81aa0101,0xfb840108,0x81ab0101,0xfb840108,0x81ac0101,0xfb840108,0x81ad0101,
	0xfb840108,0x81ae0101,0xfb840108,0x81af0101,0xfb840108,0x81b00101,0xfb840108,0x81b10101,
	0xfb840108,0x81b20101,0xfb840108,0x81b30101,0xfb840108,0x81b40101,0xfb840108,0x81b50101,
	0xfb840108,0x81b60101,0xfb840108,0x81b70101,0xfb840108,0x81b80101,0xfb840108,0x81b90101,
	0xfb840108,0x81ba0101,0xfb840108,0x81bb0101,0xfb840108,0x81bc0101,0xfb840108,0x81bd0101,
	0xfb840108,0x81be0101,0xfb840108,0x81bf0101,0xfb840108,0x81c00101,0xfb840108,0x81c10101,
	0xfb840108,0x81c20101,0xfb840108,0x81c30101,0xfb840108,0x81c40101,0xfb840108,0x81c50101,
	0xfb840108,0x81c60101,0xfb840108,0x81c70101,0xfb840108,0x81c80101,0xfb840108,0x81c90101,
	0xfb840108,0x81ca0101,0xfb840108,0x81cb0101,0xfb840108,0x81cc0101,0xfb840108,0x81cd0101,
	0xfb840108,0x81ce0101,0xfb840108,0x81cf0101,0xfb840108,0x81d00101,0xfb840108,0x81d10101,
	0xfb840108,0x81d20101,0xfb840108,0x81d30101,0xfb840108,0x81d40101,0xfb840108,0x81d50101,
	0xfb840108,0x81d60101,0xfb840108,0x81d70101,0xfb840108,0x81d80101,0xfb840108,0x81d90101,
	0xfb840108,0x81da0101,0xfb840108,0x81db0101,0xfb840108,0x81dc0101,0xfb840108,0x81dd0101,
	0xfb840108,0x81de0101,0xfb840108,0x81df0101,0xfb840108,0x81e00101,0xfb840108,0x81e10101,
	0xfb840108,0x81e20101,0xfb840108,0x81e30101,0xfb840108,0x81e40101,0xfb840108,0x81e50101,
	0xfb840108,0x81e60101,0xfb840108,0x81e70101,0xfb840108,0x81e80101,0xfb840108,0x81e90101,
	0xfb840108,0x81ea0101,0xfb840108,0x81eb0101,0xfb840108,0x81ec0101,0xfb840108,0x81ed0101,
	0xfb840108,0x81ee0101,0xfb840108,0x81ef0101,0xfb840108,0x81f00101,0xfb840108,0x81f10101,
	0xfb840108,0x81f20101,0xfb840108,0x81f30101,0xfb840108,0x81f40101,0xfb840108,0x81f50101,
	0xfb840108,0x81f60101,0xfb840108,0x81f70101,0xfb840108,0x81f80101,0xfb840108,0x81f90101,
	0xfb840108,0x81fa0101,0xfb840108,0x81fb0101,0xfb840108,0x81fc0101,0xfb840108,0x81fd0101,
	0xfb840108,0x81fe0101,0xfb840108,0x81ff0101,0xfb840108,0x82000101,0xfb840108,0x82010101,
	0xfb840108,0x82020101,0xfb840108,0x82030101,0xfb840108,0x82040101,0xfb840108,0x82050101,
	0xfb840108,0x82060101,0xfb840108,0x82070101,0xfb840108,0x82080101,0xfb840108,0x82090101,
	0xfb840108,0x820a0101,0xfb840108,0x820b0101,0xfb840108,0x820c0101,0xfb840108,0x820d0101,
	0xfb840108,0x820e0101,0xfb840108,0x820f0101,0xfb840108,0x82100101,0xfb840108,0x82110101,
	0xfb840108,0x82120101,0xfb840108,0x82130101,0xfb840108,0x82140101,0xfb840108,0x82150101,
	0xfb840108,0x82160101,0xfb840108,0x82170101,0xfb840108,0x82180101,0xfb840108,0x82190101,
	0xfb840108,0x821a0101,0xfb840108,0x821b0101,0xfb840108,0x821c0101,0xfb840108,0x821d0101,
	0xfb840108,0x821e0101,0xfb840108,0x821f0101,0xfb840108,0x82200101,0xfb840108,0x82210101,
	0xfb840108,0x82220101,0xfb840108,0x82230101,0xfb840108,0x82240101,0xfb840108,0x82250101,
	0xfb840108,0x82260101,0xfb840108,0x82270101,0xfb840108,0x82280101,0xfb840108,0x82290101,
	0xfb840108,0x822a0101,0xfb840108,0x822b0101,0xfb840108,0x822c0101,0xfb840108,0x822d0101,
	0xfb840108,0x822e0101,0xfb840108,0x822f0101,0xfb840108,0x82300101,0xfb840108,0x82310101,
	0xfb840108,0x82320101,0xfb840108,0x82330101,0xfb840108,0x82340101,0xfb840108,0x82350101,
	0xfb840108,0x82360101,0xfb840108,0x82370101,0xfb840108,0x82380101,0xfb840108,0x82390101,
	0xfb840108,0x823a0101,0xfb840108,0x823b0101,0xfb840108,0x823c0101,0xfb840108,0x823d0101,
	0xfb840108,0x823e0101,0xfb840108,0x823f0101,0xfb840108,0x82400101,0xfb840108,0x82410101,
	0xfb840108,0x82420101,0xfb840108,0x82430101,0xfb840108,0x82440101,0xfb840108,0x82450101,
	0xfb840108,0x82460101,0xfb840108,0x82470101,0xfb840108,0x82480101,0xfb840108,0x82490101,
	0xfb840108,0x824a0101,0xfb840108,0x824b0101,0xfb840108,0x824c0101,0xfb840108,0x824d0101,
	0xfb840108,0x824e0101,0xfb840108,0x824f0101,0xfb840108,0x82500101,0xfb840108,0x82510101,
	0xfb840108,0x82520101,0xfb840108,0x82530101,0xfb840108,0x82540101,0xfb840108,0x82550101,
	0xfb840108,0x82560101,0xfb840108,0x82570101,0xfb840108,0x82580101,0xfb840108,0x82590101,
	0xfb840108,0x825a0101,0xfb840108,0x825b0101,0xfb840108,0x825c0101,0xfb840108,0x825d0101,
	0xfb840108,0x825e0101,0xfb840108,0x825f0101,0xfb840108,0x82600101,0xfb840108,0x82610101,
	0xfb840108,0x82620101,0xfb840108,0x82630101,0xfb840108,0x82640101,0xfb840108,0x82650101,
	0xfb840108,0x82660101,0xfb840108,0x82670101,0xfb840108,0x82680101,0xfb840108,0x82690101,
	0xfb840108,0x826a0101,0xfb840108,0x826b0101,0xfb840108,0x826c0101,0xfb840108,0x826d0101,
	0xfb840108,0x826e0101,0xfb840108,0x826f0101,0xfb840108,0x82700101,0xfb840108,0x82710101,
	0xfb840108,0x82720101,0xfb840108,0x82730101,0xfb840108,0x82740101,0xfb840108,0x82750101,
	0xfb840108,0x82760101,0xfb840108,0x82770101,0xfb840108,0x82780101,0xfb840108,0x82790101,
	0xfb840108,0x827a0101,0xfb840108,0x827b0101,0xfb840108,0x827c0101,0xfb840108,0x827d0101,
	0xfb840108,0x827e0101,0xfb840108,0x827f0101,0xfb840108,0x82800101,0xfb840108,0x82810101,
	0xfb840108,0x82820101,0xfb840108,0x82830101,0xfb840108,0x82840101,0xfb840108,0x82850101,
	0xfb840108,0x82860101,0xfb840108,0x82870101,0xfb840108,0x82880101,0xfb840108,0x82890101,
	0xfb840108,0x828a0101,0xfb840108,0x828b0101,0xfb840108,0x828c0101,0xfb840108,0x828d0101,
	0xfb840108,0x828e0101,0xfb840108,0x828f0101,0xfb840108,0x82900101,0xfb840108,0x82910101,
	0xfb840108,0x82920101,0xfb840108,0x82930101,0xfb840108,0x82940101,0xfb840108,0x82950101,
	0xfb840108,0x82960101,0xfb840108,0x82970101,0xfb840108,0x82980101,0xfb840108,0x82990101,
	0xfb840108,0x829a0101,0xfb840108,0x829b0101,0xfb840108,0x829c0101,0xfb840108,0x829d0101,
	0xfb840108,0x829e0101,0xfb840108,0x829f0101,0xfb840108,0x82a00101,0xfb840108,0x82a10101,
	0xfb840108,0x82a20101,0xfb840108,0x82a30101,0xfb840108,0x82a40101,0xfb840108,0x82a50101,
	0xfb840108,0x82a60101,0xfb840108,0x82a70101,0xfb840108,0x82a80101,0xfb840108,0x82a90101,
	0xfb840108,0x82aa0101,0xfb840108,0x82ab0101,0xfb840108,0x82ac0101,0xfb840108,0x82ad0101,
	0xfb840108,0x82ae0101,0xfb840108,0x82af0101,0xfb840108,0x82b00101,0xfb840108,0x82b10101,
	0xfb840108,0x82b20101,0xfb840108,0x82b30101,0xfb840108,0x82b40101,0xfb840108,0x82b50101,
	0xfb840108,0x82b60101,0xfb840108,0x82b70101,0xfb840108,0x82b80101,0xfb840108,0x82b90101,
	0xfb840108,0x82ba0101,0xfb840108,0x82bb0101,0xfb840108,0x82bc0101,0xfb840108,0x82bd0101,
	0xfb840108,0x82be0101,0xfb840108,0x82bf0101,0xfb840108,0x82c00101,0xfb840108,0x82c10101,
	0xfb840108,0x82c20101,0xfb840108,0x82c30101,0xfb840108,0x82c40101,0xfb840108,0x82c50101,
	0xfb840108,0x82c60101,0xfb840108,0x82c70101,0xfb840108,0x82c80101,0xfb840108,0x82c90101,
	0xfb840108,0x82ca0101,0xfb840108,0x82cb0101,0xfb840108,0x82cc0101,0xfb840108,0x82cd0101,
	0xfb840108,0x82ce0101,0xfb840108,0x82cf0101,0xfb840108,0x82d00101,0xfb840108,0x82d10101,
	0xfb840108,0x82d20101,0xfb840108,0x82d30101,0xfb840108,0x82d40101,0xfb840108,0x82d50101,
	0xfb840108,0x82d60101,0xfb840108,0x82d70101,0xfb840108,0x82d80101,0xfb840108,0x82d90101,
	0xfb840108,0x82da0101,0xfb840108,0x82db0101,0xfb840108,0x82dc0101,0xfb840108,0x82dd0101,
	0xfb840108,0x82de0101,0xfb840108,0x82df0101,0xfb840108,0x82e00101,0xfb840108,0x82e10101,
	0xfb840108,0x82e20101,0xfb840108,0x82e30101,0xfb840108,0x82e40101,0xfb840108,0x82e50101,
	0xfb840108,0x82e60101,0xfb840108,0x82e70101,0xfb840108,0x82e80101,0xfb840108,0x82e90101,
	0xfb840108,0x82ea0101,0xfb840108,0x82eb0101,0xfb840108,0x82ec0101,0xfb840108,0x82ed0101,
	0xfb840108,0x82ee0101,0xfb840108,0x82ef0101,0xfb840108,0x82f00101,0xfb840108,0x82f10101,
	0xfb840108,0x82f20101,0xfb840108,0x82f30101,0xfb840108,0x82f40101,0xfb840108,0x82f50101,
	0xfb840108,0x82f60101,0xfb840108,0x82f70101,0xfb840108,0x82f80101,0xfb840108,0x82f90101,
	0xfb840108,0x82fa0101,0xfb840108,0x82fb0101,0xfb840108,0x82fc0101,0xfb840108,0x82fd0101,
	0xfb840108,0x82fe0101,0xfb840108,0x82ff0101,0xfb840108,0x83000101,0xfb840108,0x83010101,
	0xfb840108,0x83020101,0xfb840108,0x83030101,0xfb840108,0x83040101,0xfb840108,0x83050101,
	0xfb840108,0x83060101,0xfb840108,0x83070101,0xfb840108,0x83080101,0xfb840108,0x83090101,
	0xfb840108,0x830a0101,0xfb840108,0x830b0101,0xfb840108,0x830c0101,0xfb840108,0x830d0101,
	0xfb840108,0x830e0101,0xfb840108,0x830f0101,0xfb840108,0x83100101,0xfb840108,0x83110101,
	0xfb840108,0x83120101,0xfb840108,0x83130101,0xfb840108,0x83140101,0xfb840108,0x83150101,
	0xfb840108,0x83160101,0xfb840108,0x83170101,0xfb840108,0x83180101,0xfb840108,0x83190101,
	0xfb840108,0x831a0101,0xfb840108,0x831b0101,0xfb840108,0x831c0101,0xfb840108,0x831d0101,
	0xfb840108,0x831e0101,0xfb840108,0x831f0101,0xfb840108,0x83200101,0xfb840108,0x83210101,
	0xfb840108,0x83220101,0xfb840108,0x83230101,0xfb840108,0x83240101,0xfb840108,0x83250101,
	0xfb840108,0x83260101,0xfb840108,0x83270101,0xfb840108,0x83280101,0xfb840108,0x83290101,
	0xfb840108,0x832a0101,0xfb840108,0x832b0101,0xfb840108,0x832c0101,0xfb840108,0x832d0101,
	0xfb840108,0x832e0101,0xfb840108,0x832f0101,0xfb840108,0x83300101,0xfb840108,0x83310101,
	0xfb840108,0x83320101,0xfb840108,0x83330101,0xfb840108,0x83340101,0xfb840108,0x83350101,
	0xfb840108,0x83360101,0xfb840108,0x83370101,0xfb840108,0x83380101,0xfb840108,0x83390101,
	0xfb840108,0x833a0101,0xfb840108,0x833b0101,0xfb840108,0x833c0101,0xfb840108,0x833d0101,
	0xfb840108,0x833e0101,0xfb840108,0x833f0101,0xfb840108,0x83400101,0xfb840108,0x83410101,
	0xfb840108,0x83420101,0xfb840108,0x83430101,0xfb840108,0x83440101,0xfb840108,0x83450101,
	0xfb840108,0x83460101,0xfb840108,0x83470101,0xfb840108,0x83480101,0xfb840108,0x83490101,
	0xfb840108,0x834a0101,0xfb840108,0x834b0101,0xfb840108,0x834c0101,0xfb840108,0x834d0101,
	0xfb840108,0x834e0101,0xfb840108,0x834f0101,0xfb840108,0x83500101,0xfb840108,0x83510101,
	0xfb840108,0x83520101,0xfb840108,0x83530101,0xfb840108,0x83540101,0xfb840108,0x83550101,
	0xfb840108,0x83560101,0xfb840108,0x83570101,0xfb840108,0x83580101,0xfb840108,0x83590101,
	0xfb840108,0x835a0101,0xfb840108,0x835b0101,0xfb840108,0x835c0101,0xfb840108,0x835d0101,
	0xfb840108,0x835e0101,0xfb840108,0x835f0101,0xfb840108,0x83600101,0xfb840108,0x83610101,
	0xfb840108,0x83620101,0xfb840108,0x83630101,0xfb840108,0x83640101,0xfb840108,0x83650101,
	0xfb840108,0x83660101,0xfb840108,0x83670101,0xfb840108,0x83680101,0xfb840108,0x83690101,
	0xfb840108,0x836a0101,0xfb840108,0x836b0101,0xfb840108,0x836c0101,0xfb840108,0x836d0101,
	0xfb840108,0x836e0101,0xfb840108,0x836f0101,0xfb840108,0x83700101,0xfb840108,0x83710101,
	0xfb840108,0x83720101,0xfb840108,0x83730101,0xfb840108,0x83740101,0xfb840108,0x83750101,
	0xfb840108,0x83760101,0xfb840108,0x83770101,0xfb840108,0x83780101,0xfb840108,0x83790101,
	0xfb840108,0x837a0101,0xfb840108,0x837b0101,0xfb840108,0x837c0101,0xfb840108,0x837d0101,
	0xfb840108,0x837e0101,0xfb840108,0x837f0101,0xfb840108,0x83800101,0xfb840108,0x83810101,
	0xfb840108,0x83820101,0xfb840108,0x83830101,0xfb840108,0x83840101,0xfb840108,0x83850101,
	0xfb840108,0x83860101,0xfb840108,0x83870101,0xfb840108,0x83880101,0xfb840108,0x83890101,
	0xfb840108,0x838a0101,0xfb840108,0x838b0101,0xfb840108,0x838c0101,0xfb840108,0x838d0101,
	0xfb840108,0x838e0101,0xfb840108,0x838f0101,0xfb840108,0x83900101,0xfb840108,0x83910101,
	0xfb840108,0x83920101,0xfb840108,0x83930101,0xfb840108,0x83940101,0xfb840108,0x83950101,
	0xfb840108,0x83960101,0xfb840108,0x83970101,0xfb840108,0x83980101,0xfb840108,0x83990101,
	0xfb840108,0x839a0101,0xfb840108,0x839b0101,0xfb840108,0x839c0101,0xfb840108,0x839d0101,
	0xfb840108,0x839e0101,0xfb840108,0x839f0101,0xfb840108,0x83a00101,0xfb840108,0x83a10101,
	0xfb840108,0x83a20101,0xfb840108,0x83a30101,0xfb840108,0x83a40101,0xfb840108,0x83a50101,
	0xfb840108,0x83a60101,0xfb840108,0x83a70101,0xfb840108,0x83a80101,0xfb840108,0x83a90101,
	0xfb840108,0x83aa0101,0xfb840108,0x83ab0101,0xfb840108,0x83ac0101,0xfb840108,0x83ad0101,
	0xfb840108,0x83ae0101,0xfb840108,0x83af0101,0xfb840108,0x83b00101,0xfb840108,0x83b10101,
	0xfb840108,0x83b20101,0xfb840108,0x83b30101,0xfb840108,0x83b40101,0xfb840108,0x83b50101,
	0xfb840108,0x83b60101,0xfb840108,0x83b70101,0xfb840108,0x83b80101,0xfb840108,0x83b90101,
	0xfb840108,0x83ba0101,0xfb840108,0x83bb0101,0xfb840108,0x83bc0101,0xfb840108,0x83bd0101,
	0xfb840108,0x83be0101,0xfb840108,0x83bf0101,0xfb840108,0x83c00101,0xfb840108,0x83c10101,
	0xfb840108,0x83c20101,0xfb840108,0x83c30101,0xfb840108,0x83c40101,0xfb840108,0x83c50101,
	0xfb840108,0x83c60101,0xfb840108,0x83c70101,0xfb840108,0x83c80101,0xfb840108,0x83c90101,
	0xfb840108,0x83ca0101,0xfb840108,0x83cb0101,0xfb840108,0x83cc0101,0xfb840108,0x83cd0101,
	0xfb840108,0x83ce0101,0xfb840108,0x83cf0101,0xfb840108,0x83d00101,0xfb840108,0x83d10101,
	0xfb840108,0x83d20101,0xfb840108,0x83d30101,0xfb840108,0x83d40101,0xfb840108,0x83d50101,
	0xfb840108,0x83d60101,0xfb840108,0x83d70101,0xfb840108,0x83d80101,0xfb840108,0x83d90101,
	0xfb840108,0x83da0101,0xfb840108,0x83db0101,0xfb840108,0x83dc0101,0xfb840108,0x83dd0101,
	0xfb840108,0x83de0101,0xfb840108,0x83df0101,0xfb840108,0x83e00101,0xfb840108,0x83e10101,
	0xfb840108,0x83e20101,0xfb840108,0x83e30101,0xfb840108,0x83e40101,0xfb840108,0x83e50101,
	0xfb840108,0x83e60101,0xfb840108,0x83e70101,0xfb840108,0x83e80101,0xfb840108,0x83e90101,
	0xfb840108,0x83ea0101,0xfb840108,0x83eb0101,0xfb840108,0x83ec0101,0xfb840108,0x83ed0101,
	0xfb840108,0x83ee0101,0xfb840108,0x83ef0101,0xfb840108,0x83f00101,0xfb840108,0x83f10101,
	0xfb840108,0x83f20101,0xfb840108,0x83f30101,0xfb840108,0x83f40101,0xfb840108,0x83f50101,
	0xfb840108,0x83f60101,0xfb840108,0x83f70101,0xfb840108,0x83f80101,0xfb840108,0x83f90101,
	0xfb840108,0x83fa0101,0xfb840108,0x83fb0101,0xfb840108,0x83fc0101,0xfb840108,0x83fd0101,
	0xfb840108,0x83fe0101,0xfb840108,0x83ff0101,0xfb840108,0x84000101,0xfb840108,0x84010101,
	0xfb840108,0x84020101,0xfb840108,0x84030101,0xfb840108,0x84040101,0xfb840108,0x84050101,
	0xfb840108,0x84060101,0xfb840108,0x84070101,0xfb840108,0x84080101,0xfb840108,0x84090101,
	0xfb840108,0x840a0101,0xfb840108,0x840b0101,0xfb840108,0x840c0101,0xfb840108,0x840d0101,
	0xfb840108,0x840e0101,0xfb840108,0x840f0101,0xfb840108,0x84100101,0xfb840108,0x84110101,
	0xfb840108,0x84120101,0xfb840108,0x84130101,0xfb840108,0x84140101,0xfb840108,0x84150101,
	0xfb840108,0x84160101,0xfb840108,0x84170101,0xfb840108,0x84180101,0xfb840108,0x84190101,
	0xfb840108,0x841a0101,0xfb840108,0x841b0101,0xfb840108,0x841c0101,0xfb840108,0x841d0101,
	0xfb840108,0x841e0101,0xfb840108,0x841f0101,0xfb840108,0x84200101,0xfb840108,0x84210101,
	0xfb840108,0x84220101,0xfb840108,0x84230101,0xfb840108,0x84240101,0xfb840108,0x84250101,
	0xfb840108,0x84260101,0xfb840108,0x84270101,0xfb840108,0x84280101,0xfb840108,0x84290101,
	0xfb840108,0x842a0101,0xfb840108,0x842b0101,0xfb840108,0x842c0101,0xfb840108,0x842d0101,
	0xfb840108,0x842e0101,0xfb840108,0x842f0101,0xfb840108,0x84300101,0xfb840108,0x84310101,
	0xfb840108,0x84320101,0xfb840108,0x84330101,0xfb840108,0x84340101,0xfb840108,0x84350101,
	0xfb840108,0x84360101,0xfb840108,0x84370101,0xfb840108,0x84380101,0xfb840108,0x84390101,
	0xfb840108,0x843a0101,0xfb840108,0x843b0101,0xfb840108,0x843c0101,0xfb840108,0x843d0101,
	0xfb840108,0x843e0101,0xfb840108,0x843f0101,0xfb840108,0x84400101,0xfb840108,0x84410101,
	0xfb840108,0x84420101,0xfb840108,0x84430101,0xfb840108,0x84440101,0xfb840108,0x84450101,
	0xfb840108,0x84460101,0xfb840108,0x84470101,0xfb840108,0x84480101,0xfb840108,0x84490101,
	0xfb840108,0x844a0101,0xfb840108,0x844b0101,0xfb840108,0x844c0101,0xfb840108,0x844d0101,
	0xfb840108,0x844e0101,0xfb840108,0x844f0101,0xfb840108,0x84500101,0xfb840108,0x84510101,
	0xfb840108,0x84520101,0xfb840108,0x84530101,0xfb840108,0x84540101,0xfb840108,0x84550101,
	0xfb840108,0x84560101,0xfb840108,0x84570101,0xfb840108,0x84580101,0xfb840108,0x84590101,
	0xfb840108,0x845a0101,0xfb840108,0x845b0101,0xfb840108,0x845c0101,0xfb840108,0x845d0101,
	0xfb840108,0x845e0101,0xfb840108,0x845f0101,0xfb840108,0x84600101,0xfb840108,0x84610101,
	0xfb840108,0x84620101,0xfb840108,0x84630101,0xfb840108,0x84640101,0xfb840108,0x84650101,
	0xfb840108,0x84660101,0xfb840108,0x84670101,0xfb840108,0x84680101,0xfb840108,0x84690101,
	0xfb840108,0x846a0101,0xfb840108,0x846b0101,0xfb840108,0x846c0101,0xfb840108,0x846d0101,
	0xfb840108,0x846e0101,0xfb840108,0x846f0101,0xfb840108,0x84700101,0xfb840108,0x84710101,
	0xfb840108,0x84720101,0xfb840108,0x84730101,0xfb840108,0x84740101,0xfb840108,0x84750101,
	0xfb840108,0x84760101,0xfb840108,0x84770101,0xfb840108,0x84780101,0xfb840108,0x84790101,
	0xfb840108,0x847a0101,0xfb840108,0x847b0101,0xfb840108,0x847c0101,0xfb840108,0x847d0101,
	0xfb840108,0x847e0101,0xfb840108,0x847f0101,0xfb840108,0x84800101,0xfb840108,0x84810101,
	0xfb840108,0x84820101,0xfb840108,0x84830101,0xfb840108,0x84840101,0xfb840108,0x84850101,
	0xfb840108,0x84860101,0xfb840108,0x84870101,0xfb840108,0x84880101,0xfb840108,0x84890101,
	0xfb840108,0x848a0101,0xfb840108,0x848b0101,0xfb840108,0x848c0101,0xfb840108,0x848d0101,
	0xfb840108,0x848e0101,0xfb840108,0x848f0101,0xfb840108,0x84900101,0xfb840108,0x84910101,
	0xfb840108,0x84920101,0xfb840108,0x84930101,0xfb840108,0x84940101,0xfb840108,0x84950101,
	0xfb840108,0x84960101,0xfb840108,0x84970101,0xfb840108,0x84980101,0xfb840108,0x84990101,
	0xfb840108,0x849a0101,0xfb840108,0x849b0101,0xfb840108,0x849c0101,0xfb840108,0x849d0101,
	0xfb840108,0x849e0101,0xfb840108,0x849f0101,0xfb840108,0x84a00101,0xfb840108,0x84a10101,
	0xfb840108,0x84a20101,0xfb840108,0x84a30101,0xfb840108,0x84a40101,0xfb840108,0x84a50101,
	0xfb840108,0x84a60101,0xfb840108,0x84a70101,0xfb840108,0x84a80101,0xfb840108,0x84a90101,
	0xfb840108,0x84aa0101,0xfb840108,0x84ab0101,0xfb840108,0x84ac0101,0xfb840108,0x84ad0101,
	0xfb840108,0x84ae0101,0xfb840108,0x84af0101,0xfb840108,0x84b00101,0xfb840108,0x84b10101,
	0xfb840108,0x84b20101,0xfb840108,0x84b30101,0xfb840108,0x84b40101,0xfb840108,0x84b50101,
	0xfb840108,0x84b60101,0xfb840108,0x84b70101,0xfb840108,0x84b80101,0xfb840108,0x84b90101,
	0xfb840108,0x84ba0101,0xfb840108,0x84bb0101,0xfb840108,0x84bc0101,0xfb840108,0x84bd0101,
	0xfb840108,0x84be0101,0xfb840108,0x84bf0101,0xfb840108,0x84c00101,0xfb840108,0x84c10101,
	0xfb840108,0x84c20101,0xfb840108,0x84c30101,0xfb840108,0x84c40101,0xfb840108,0x84c50101,
	0xfb840108,0x84c60101,0xfb840108,0x84c70101,0xfb840108,0x84c80101,0xfb840108,0x84c90101,
	0xfb840108,0x84ca0101,0xfb840108,0x84cb0101,0xfb840108,0x84cc0101,0xfb840108,0x84cd0101,
	0xfb840108,0x84ce0101,0xfb840108,0x84cf0101,0xfb840108,0x84d00101,0xfb840108,0x84d10101,
	0xfb840108,0x84d20101,0xfb840108,0x84d30101,0xfb840108,0x84d40101,0xfb840108,0x84d50101,
	0xfb840108,0x84d60101,0xfb840108,0x84d70101,0xfb840108,0x84d80101,0xfb840108,0x84d90101,
	0xfb840108,0x84da0101,0xfb840108,0x84db0101,0xfb840108,0x84dc0101,0xfb840108,0x84dd0101,
	0xfb840108,0x84de0101,0xfb840108,0x84df0101,0xfb840108,0x84e00101,0xfb840108,0x84e10101,
	0xfb840108,0x84e20101,0xfb840108,0x84e30101,0xfb840108,0x84e40101,0xfb840108,0x84e50101,
	0xfb840108,0x84e60101,0xfb840108,0x84e70101,0xfb840108,0x84e80101,0xfb840108,0x84e90101,
	0xfb840108,0x84ea0101,0xfb840108,0x84eb0101,0xfb840108,0x84ec0101,0xfb840108,0x84ed0101,
	0xfb840108,0x84ee0101,0xfb840108,0x84ef0101,0xfb840108,0x84f00101,0xfb840108,0x84f10101,
	0xfb840108,0x84f20101,0xfb840108,0x84f30101,0xfb840108,0x84f40101,0xfb840108,0x84f50101,
	0xfb840108,0x84f60101,0xfb840108,0x84f70101,0xfb840108,0x84f80101,0xfb840108,0x84f90101,
	0xfb840108,0x84fa0101,0xfb840108,0x84fb0101,0xfb840108,0x84fc0101,0xfb840108,0x84fd0101,
	0xfb840108,0x84fe0101,0xfb840108,0x84ff0101,0xfb840108,0x85000101,0xfb840108,0x85010101,
	0xfb840108,0x85020101,0xfb840108,0x85030101,0xfb840108,0x85040101,0xfb840108,0x85050101,
	0xfb840108,0x85060101,0xfb840108,0x85070101,0xfb840108,0x85080101,0xfb840108,0x85090101,
	0xfb840108,0x850a0101,0xfb840108,0x850b0101,0xfb840108,0x850c0101,0xfb840108,0x850d0101,
	0xfb840108,0x850e0101,0xfb840108,0x850f0101,0xfb840108,0x85100101,0xfb840108,0x85110101,
	0xfb840108,0x85120101,0xfb840108,0x85130101,0xfb840108,0x85140101,0xfb840108,0x85150101,
	0xfb840108,0x85160101,0xfb840108,0x85170101,0xfb840108,0x85180101,0xfb840108,0x85190101,
	0xfb840108,0x851a0101,0xfb840108,0x851b0101,0xfb840108,0x851c0101,0xfb840108,0x851d0101,
	0xfb840108,0x851e0101,0xfb840108,0x851f0101,0xfb840108,0x85200101,0xfb840108,0x85210101,
	0xfb840108,0x85220101,0xfb840108,0x85230101,0xfb840108,0x85240101,0xfb840108,0x85250101,
	0xfb840108,0x85260101,0xfb840108,0x85270101,0xfb840108,0x85280101,0xfb840108,0x85290101,
	0xfb840108,0x852a0101,0xfb840108,0x852b0101,0xfb840108,0x852c0101,0xfb840108,0x852d0101,
	0xfb840108,0x852e0101,0xfb840108,0x852f0101,0xfb840108,0x85300101,0xfb840108,0x85310101,
	0xfb840108,0x85320101,0xfb840108,0x85330101,0xfb840108,0x85340101,0xfb840108,0x85350101,
	0xfb840108,0x85360101,0xfb840108,0x85370101,0xfb840108,0x85380101,0xfb840108,0x85390101,
	0xfb840108,0x853a0101,0xfb840108,0x853b0101,0xfb840108,0x853c0101,0xfb840108,0x853d0101,
	0xfb840108,0x853e0101,0xfb840108,0x853f0101,0xfb840108,0x85400101,0xfb840108,0x85410101,
	0xfb840108,0x85420101,0xfb840108,0x85430101,0xfb840108,0x85440101,0xfb840108,0x85450101,
	0xfb840108,0x85460101,0xfb840108,0x85470101,0xfb840108,0x85480101,0xfb840108,0x85490101,
	0xfb840108,0x854a0101,0xfb840108,0x854b0101,0xfb840108,0x854c0101,0xfb840108,0x854d0101,
	0xfb840108,0x854e0101,0xfb840108,0x854f0101,0xfb840108,0x85500101,0xfb840108,0x85510101,
	0xfb840108,0x85520101,0xfb840108,0x85530101,0xfb840108,0x85540101,0xfb840108,0x85550101,
	0xfb840108,0x85560101,0xfb840108,0x85570101,0xfb840108,0x85580101,0xfb840108,0x85590101,
	0xfb840108,0x855a0101,0xfb840108,0x855b0101,0xfb840108,0x855c0101,0xfb840108,0x855d0101,
	0xfb840108,0x855e0101,0xfb840108,0x855f0101,0xfb840108,0x85600101,0xfb840108,0x85610101,
	0xfb840108,0x85620101,0xfb840108,0x85630101,0xfb840108,0x85640101,0xfb840108,0x85650101,
	0xfb840108,0x85660101,0xfb840108,0x85670101,0xfb840108,0x85680101,0xfb840108,0x85690101,
	0xfb840108,0x856a0101,0xfb840108,0x856b0101,0xfb840108,0x856c0101,0xfb840108,0x856d0101,
	0xfb840108,0x856e0101,0xfb840108,0x856f0101,0xfb840108,0x85700101,0xfb840108,0x85710101,
	0xfb840108,0x85720101,0xfb840108,0x85730101,0xfb840108,0x85740101,0xfb840108,0x85750101,
	0xfb840108,0x85760101,0xfb840108,0x85770101,0xfb840108,0x85780101,0xfb840108,0x85790101,
	0xfb840108,0x857a0101,0xfb840108,0x857b0101,0xfb840108,0x857c0101,0xfb840108,0x857d0101,
	0xfb840108,0x857e0101,0xfb840108,0x857f0101,0xfb840108,0x85800101,0xfb840108,0x85810101,
	0xfb840108,0x85820101,0xfb840108,0x85830101,0xfb840108,0x85840101,0xfb840108,0x85850101,
	0xfb840108,0x85860101,0xfb840108,0x85870101,0xfb840108,0x85880101,0xfb840108,0x85890101,
	0xfb840108,0x858a0101,0xfb840108,0x858b0101,0xfb840108,0x858c0101,0xfb840108,0x858d0101,
	0xfb840108,0x858e0101,0xfb840108,0x858f0101,0xfb840108,0x85900101,0xfb840108,0x85910101,
	0xfb840108,0x85920101,0xfb840108,0x85930101,0xfb840108,0x85940101,0xfb840108,0x85950101,
	0xfb840108,0x85960101,0xfb840108,0x85970101,0xfb840108,0x85980101,0xfb840108,0x85990101,
	0xfb840108,0x859a0101,0xfb840108,0x859b0101,0xfb840108,0x859c0101,0xfb840108,0x859d0101,
	0xfb840108,0x859e0101,0xfb840108,0x859f0101,0xfb840108,0x85a00101,0xfb840108,0x85a10101,
	0xfb840108,0x85a20101,0xfb840108,0x85a30101,0xfb840108,0x85a40101,0xfb840108,0x85a50101,
	0xfb840108,0x85a60101,0xfb840108,0x85a70101,0xfb840108,0x85a80101,0xfb840108,0x85a90101,
	0xfb840108,0x85aa0101,0xfb840108,0x85ab0101,0xfb840108,0x85ac0101,0xfb840108,0x85ad0101,
	0xfb840108,0x85ae0101,0xfb840108,0x85af0101,0xfb840108,0x85b00101,0xfb840108,0x85b10101,
	0xfb840108,0x85b20101,0xfb840108,0x85b30101,0xfb840108,0x85b40101,0xfb840108,0x85b50101,
	0xfb840108,0x85b60101,0xfb840108,0x85b70101,0xfb840108,0x85b80101,0xfb840108,0x85b90101,
	0xfb840108,0x85ba0101,0xfb840108,0x85bb0101,0xfb840108,0x85bc0101,0xfb840108,0x85bd0101,
	0xfb840108,0x85be0101,0xfb840108,0x85bf0101,0xfb840108,0x85c00101,0xfb840108,0x85c10101,
	0xfb840108,0x85c20101,0xfb840108,0x85c30101,0xfb840108,0x85c40101,0xfb840108,0x85c50101,
	0xfb840108,0x85c60101,0xfb840108,0x85c70101,0xfb840108,0x85c80101,0xfb840108,0x85c90101,
	0xfb840108,0x85ca0101,0xfb840108,0x85cb0101,0xfb840108,0x85cc0101,0xfb840108,0x85cd0101,
	0xfb840108,0x85ce0101,0xfb840108,0x85cf0101,0xfb840108,0x85d00101,0xfb840108,0x85d10101,
	0xfb840108,0x85d20101,0xfb840108,0x85d30101,0xfb840108,0x85d40101,0xfb840108,0x85d50101,
	0xfb840108,0x85d60101,0xfb840108,0x85d70101,0xfb840108,0x85d80101,0xfb840108,0x85d90101,
	0xfb840108,0x85da0101,0xfb840108,0x85db0101,0xfb840108,0x85dc0101,0xfb840108,0x85dd0101,
	0xfb840108,0x85de0101,0xfb840108,0x85df0101,0xfb840108,0x85e00101,0xfb840108,0x85e10101,
	0xfb840108,0x85e20101,0xfb840108,0x85e30101,0xfb840108,0x85e40101,0xfb840108,0x85e50101,
	0xfb840108,0x85e60101,0xfb840108,0x85e70101,0xfb840108,0x85e80101,0xfb840108,0x85e90101,
	0xfb840108,0x85ea0101,0xfb840108,0x85eb0101,0xfb840108,0x85ec0101,0xfb840108,0x85ed0101,
	0xfb840108,0x85ee0101,0xfb840108,0x85ef0101,0xfb840108,0x85f00101,0xfb840108,0x85f10101,
	0xfb840108,0x85f20101,0xfb840108,0x85f30101,0xfb840108,0x85f40101,0xfb840108,0x85f50101,
	0xfb840108,0x85f60101,0xfb840108,0x85f70101,0xfb840108,0x85f80101,0xfb840108,0x85f90101,
	0xfb840108,0x85fa0101,0xfb840108,0x85fb0101,0xfb840108,0x85fc0101,0xfb840108,0x85fd0101,
	0xfb840108,0x85fe0101,0xfb840108,0x85ff0101,0xfb840108,0x86000101,0xfb840108,0x86010101,
	0xfb840108,0x86020101,0xfb840108,0x86030101,0xfb840108,0x86040101,0xfb840108,0x86050101,
	0xfb840108,0x86060101,0xfb840108,0x86070101,0xfb840108,0x86080101,0xfb840108,0x86090101,
	0xfb840108,0x860a0101,0xfb840108,0x860b0101,0xfb840108,0x860c0101,0xfb840108,0x860d0101,
	0xfb840108,0x860e0101,0xfb840108,0x860f0101,0xfb840108,0x86100101,0xfb840108,0x86110101,
	0xfb840108,0x86120101,0xfb840108,0x86130101,0xfb840108,0x86140101,0xfb840108,0x86150101,
	0xfb840108,0x86160101,0xfb840108,0x86170101,0xfb840108,0x86180101,0xfb840108,0x86190101,
	0xfb840108,0x861a0101,0xfb840108,0x861b0101,0xfb840108,0x861c0101,0xfb840108,0x861d0101,
	0xfb840108,0x861e0101,0xfb840108,0x861f0101,0xfb840108,0x86200101,0xfb840108,0x86210101,
	0xfb840108,0x86220101,0xfb840108,0x86230101,0xfb840108,0x86240101,0xfb840108,0x86250101,
	0xfb840108,0x86260101,0xfb840108,0x86270101,0xfb840108,0x86280101,0xfb840108,0x86290101,
	0xfb840108,0x862a0101,0xfb840108,0x862b0101,0xfb840108,0x862c0101,0xfb840108,0x862d0101,
	0xfb840108,0x862e0101,0xfb840108,0x862f0101,0xfb840108,0x86300101,0xfb840108,0x86310101,
	0xfb840108,0x86320101,0xfb840108,0x86330101,0xfb840108,0x86340101,0xfb840108,0x86350101,
	0xfb840108,0x86360101,0xfb840108,0x86370101,0xfb840108,0x86380101,0xfb840108,0x86390101,
	0xfb840108,0x863a0101,0xfb840108,0x863b0101,0xfb840108,0x863c0101,0xfb840108,0x863d0101,
	0xfb840108,0x863e0101,0xfb840108,0x863f0101,0xfb840108,0x86400101,0xfb840108,0x86410101,
	0xfb840108,0x86420101,0xfb840108,0x86430101,0xfb840108,0x86440101,0xfb840108,0x86450101,
	0xfb840108,0x86460101,0xfb840108,0x86470101,0xfb840108,0x86480101,0xfb840108,0x86490101,
	0xfb840108,0x864a0101,0xfb840108,0x864b0101,0xfb840108,0x864c0101,0xfb840108,0x864d0101,
	0xfb840108,0x864e0101,0xfb840108,0x864f0101,0xfb840108,0x86500101,0xfb840108,0x86510101,
	0xfb840108,0x86520101,0xfb840108,0x86530101,0xfb840108,0x86540101,0xfb840108,0x86550101,
	0xfb840108,0x86560101,0xfb840108,0x86570101,0xfb840108,0x86580101,0xfb840108,0x86590101,
	0xfb840108,0x865a0101,0xfb840108,0x865b0101,0xfb840108,0x865c0101,0xfb840108,0x865d0101,
	0xfb840108,0x865e0101,0xfb840108,0x865f0101,0xfb840108,0x86600101,0xfb840108,0x86610101,
	0xfb840108,0x86620101,0xfb840108,0x86630101,0xfb840108,0x86640101,0xfb840108,0x86650101,
	0xfb840108,0x86660101,0xfb840108,0x86670101,0xfb840108,0x86680101,0xfb840108,0x86690101,
	0xfb840108,0x866a0101,0xfb840108,0x866b0101,0xfb840108,0x866c0101,0xfb840108,0x866d0101,
	0xfb840108,0x866e0101,0xfb840108,0x866f0101,0xfb840108,0x86700101,0xfb840108,0x86710101,
	0xfb840108,0x86720101,0xfb840108,0x86730101,0xfb840108,0x86740101,0xfb840108,0x86750101,
	0xfb840108,0x86760101,0xfb840108,0x86770101,0xfb840108,0x86780101,0xfb840108,0x86790101,
	0xfb840108,0x867a0101,0xfb840108,0x867b0101,0xfb840108,0x867c0101,0xfb840108,0x867d0101,
	0xfb840108,0x867e0101,0xfb840108,0x867f0101,0xfb840108,0x86800101,0xfb840108,0x86810101,
	0xfb840108,0x86820101,0xfb840108,0x86830101,0xfb840108,0x86840101,0xfb840108,0x86850101,
	0xfb840108,0x86860101,0xfb840108,0x86870101,0xfb840108,0x86880101,0xfb840108,0x86890101,
	0xfb840108,0x868a0101,0xfb840108,0x868b0101,0xfb840108,0x868c0101,0xfb840108,0x868d0101,
	0xfb840108,0x868e0101,0xfb840108,0x868f0101,0xfb840108,0x86900101,0xfb840108,0x86910101,
	0xfb840108,0x86920101,0xfb840108,0x86930101,0xfb840108,0x86940101,0xfb840108,0x86950101,
	0xfb840108,0x86960101,0xfb840108,0x86970101,0xfb840108,0x86980101,0xfb840108,0x86990101,
	0xfb840108,0x869a0101,0xfb840108,0x869b0101,0xfb840108,0x869c0101,0xfb840108,0x869d0101,
	0xfb840108,0x869e0101,0xfb840108,0x869f0101,0xfb840108,0x86a00101,0xfb840108,0x86a10101,
	0xfb840108,0x86a20101,0xfb840108,0x86a30101,0xfb840108,0x86a40101,0xfb840108,0x86a50101,
	0xfb840108,0x86a60101,0xfb840108,0x86a70101,0xfb840108,0x86a80101,0xfb840108,0x86a90101,
	0xfb840108,0x86aa0101,0xfb840108,0x86ab0101,0xfb840108,0x86ac0101,0xfb840108,0x86ad0101,
	0xfb840108,0x86ae0101,0xfb840108,0x86af0101,0xfb840108,0x86b00101,0xfb840108,0x86b10101,
	0xfb840108,0x86b20101,0xfb840108,0x86b30101,0xfb840108,0x86b40101,0xfb840108,0x86b50101,
	0xfb840108,0x86b60101,0xfb840108,0x86b70101,0xfb840108,0x86b80101,0xfb840108,0x86b90101,
	0xfb840108,0x86ba0101,0xfb840108,0x86bb0101,0xfb840108,0x86bc0101,0xfb840108,0x86bd0101,
	0xfb840108,0x86be0101,0xfb840108,0x86bf0101,0xfb840108,0x86c00101,0xfb840108,0x86c10101,
	0xfb840108,0x86c20101,0xfb840108,0x86c30101,0xfb840108,0x86c40101,0xfb840108,0x86c50101,
	0xfb840108,0x86c60101,0xfb840108,0x86c70101,0xfb840108,0x86c80101,0xfb840108,0x86c90101,
	0xfb840108,0x86ca0101,0xfb840108,0x86cb0101,0xfb840108,0x86cc0101,0xfb840108,0x86cd0101,
	0xfb840108,0x86ce0101,0xfb840108,0x86cf0101,0xfb840108,0x86d00101,0xfb840108,0x86d10101,
	0xfb840108,0x86d20101,0xfb840108,0x86d30101,0xfb840108,0x86d40101,0xfb840108,0x86d50101,
	0xfb840108,0x86d60101,0xfb840108,0x86d70101,0xfb840108,0x86d80101,0xfb840108,0x86d90101,
	0xfb840108,0x86da0101,0xfb840108,0x86db0101,0xfb840108,0x86dc0101,0xfb840108,0x86dd0101,
	0xfb840108,0x86de0101,0xfb840108,0x86df0101,0xfb840108,0x86e00101,0xfb840108,0x86e10101,
	0xfb840108,0x86e20101,0xfb840108,0x86e30101,0xfb840108,0x86e40101,0xfb840108,0x86e50101,
	0xfb840108,0x86e60101,0xfb840108,0x86e70101,0xfb840108,0x86e80101,0xfb840108,0x86e90101,
	0xfb840108,0x86ea0101,0xfb840108,0x86eb0101,0xfb840108,0x86ec0101,0xfb840108,0x86ed0101,
	0xfb840108,0x86ee0101,0xfb840108,0x86ef0101,0xfb840108,0x86f00101,0xfb840108,0x86f10101,
	0xfb840108,0x86f20101,0xfb840108,0x86f30101,0xfb840108,0x86f40101,0xfb840108,0x86f50101,
	0xfb840108,0x86f60101,0xfb840108,0x86f70101,0xfb840108,0x86f80101,0xfb840108,0x86f90101,
	0xfb840108,0x86fa0101,0xfb840108,0x86fb0101,0xfb840108,0x86fc0101,0xfb840108,0x86fd0101,
	0xfb840108,0x86fe0101,0xfb840108,0x86ff0101,0xfb840108,0x87000101,0xfb840108,0x87010101,
	0xfb840108,0x87020101,0xfb840108,0x87030101,0xfb840108,0x87040101,0xfb840108,0x87050101,
	0xfb840108,0x87060101,0xfb840108,0x87070101,0xfb840108,0x87080101,0xfb840108,0x87090101,
	0xfb840108,0x870a0101,0xfb840108,0x870b0101,0xfb840108,0x870c0101,0xfb840108,0x870d0101,
	0xfb840108,0x870e0101,0xfb840108,0x870f0101,0xfb840108,0x87100101,0xfb840108,0x87110101,
	0xfb840108,0x87120101,0xfb840108,0x87130101,0xfb840108,0x87140101,0xfb840108,0x87150101,
	0xfb840108,0x87160101,0xfb840108,0x87170101,0xfb840108,0x87180101,0xfb840108,0x87190101,
	0xfb840108,0x871a0101,0xfb840108,0x871b0101,0xfb840108,0x871c0101,0xfb840108,0x871d0101,
	0xfb840108,0x871e0101,0xfb840108,0x871f0101,0xfb840108,0x87200101,0xfb840108,0x87210101,
	0xfb840108,0x87220101,0xfb840108,0x87230101,0xfb840108,0x87240101,0xfb840108,0x87250101,
	0xfb840108,0x87260101,0xfb840108,0x87270101,0xfb840108,0x87280101,0xfb840108,0x87290101,
	0xfb840108,0x872a0101,0xfb840108,0x872b0101,0xfb840108,0x872c0101,0xfb840108,0x872d0101,
	0xfb840108,0x872e0101,0xfb840108,0x872f0101,0xfb840108,0x87300101,0xfb840108,0x87310101,
	0xfb840108,0x87320101,0xfb840108,0x87330101,0xfb840108,0x87340101,0xfb840108,0x87350101,
	0xfb840108,0x87360101,0xfb840108,0x87370101,0xfb840108,0x87380101,0xfb840108,0x87390101,
	0xfb840108,0x873a0101,0xfb840108,0x873b0101,0xfb840108,0x873c0101,0xfb840108,0x873d0101,
	0xfb840108,0x873e0101,0xfb840108,0x873f0101,0xfb840108,0x87400101,0xfb840108,0x87410101,
	0xfb840108,0x87420101,0xfb840108,0x87430101,0xfb840108,0x87440101,0xfb840108,0x87450101,
	0xfb840108,0x87460101,0xfb840108,0x87470101,0xfb840108,0x87480101,0xfb840108,0x87490101,
	0xfb840108,0x874a0101,0xfb840108,0x874b0101,0xfb840108,0x874c0101,0xfb840108,0x874d0101,
	0xfb840108,0x874e0101,0xfb840108,0x874f0101,0xfb840108,0x87500101,0xfb840108,0x87510101,
	0xfb840108,0x87520101,0xfb840108,0x87530101,0xfb840108,0x87540101,0xfb840108,0x87550101,
	0xfb840108,0x87560101,0xfb840108,0x87570101,0xfb840108,0x87580101,0xfb840108,0x87590101,
	0xfb840108,0x875a0101,0xfb840108,0x875b0101,0xfb840108,0x875c0101,0xfb840108,0x875d0101,
	0xfb840108,0x875e0101,0xfb840108,0x875f0101,0xfb840108,0x87600101,0xfb840108,0x87610101,
	0xfb840108,0x87620101,0xfb840108,0x87630101,0xfb840108,0x87640101,0xfb840108,0x87650101,
	0xfb840108,0x87660101,0xfb840108,0x87670101,0xfb840108,0x87680101,0xfb840108,0x87690101,
	0xfb840108,0x876a0101,0xfb840108,0x876b0101,0xfb840108,0x876c0101,0xfb840108,0x876d0101,
	0xfb840108,0x876e0101,0xfb840108,0x876f0101,0xfb840108,0x87700101,0xfb840108,0x87710101,
	0xfb840108,0x87720101,0xfb840108,0x87730101,0xfb840108,0x87740101,0xfb840108,0x87750101,
	0xfb840108,0x87760101,0xfb840108,0x87770101,0xfb840108,0x87780101,0xfb840108,0x87790101,
	0xfb840108,0x877a0101,0xfb840108,0x877b0101,0xfb840108,0x877c0101,0xfb840108,0x877d0101,
	0xfb840108,0x877e0101,0xfb840108,0x877f0101,0xfb840108,0x87800101,0xfb840108,0x87810101,
	0xfb840108,0x87820101,0xfb840108,0x87830101,0xfb840108,0x87840101,0xfb840108,0x87850101,
	0xfb840108,0x87860101,0xfb840108,0x87870101,0xfb840108,0x87880101,0xfb840108,0x87890101,
	0xfb840108,0x878a0101,0xfb840108,0x878b0101,0xfb840108,0x878c0101,0xfb840108,0x878d0101,
	0xfb840108,0x878e0101,0xfb840108,0x878f0101,0xfb840108,0x87900101,0xfb840108,0x87910101,
	0xfb840108,0x87920101,0xfb840108,0x87930101,0xfb840108,0x87940101,0xfb840108,0x87950101,
	0xfb840108,0x87960101,0xfb840108,0x87970101,0xfb840108,0x87980101,0xfb840108,0x87990101,
	0xfb840108,0x879a0101,0xfb840108,0x879b0101,0xfb840108,0x879c0101,0xfb840108,0x879d0101,
	0xfb840108,0x879e0101,0xfb840108,0x879f0101,0xfb840108,0x87a00101,0xfb840108,0x87a10101,
	0xfb840108,0x87a20101,0xfb840108,0x87a30101,0xfb840108,0x87a40101,0xfb840108,0x87a50101,
	0xfb840108,0x87a60101,0xfb840108,0x87a70101,0xfb840108,0x87a80101,0xfb840108,0x87a90101,
	0xfb840108,0x87aa0101,0xfb840108,0x87ab0101,0xfb840108,0x87ac0101,0xfb840108,0x87ad0101,
	0xfb840108,0x87ae0101,0xfb840108,0x87af0101,0xfb840108,0x87b00101,0xfb840108,0x87b10101,
	0xfb840108,0x87b20101,0xfb840108,0x87b30101,0xfb840108,0x87b40101,0xfb840108,0x87b50101,
	0xfb840108,0x87b60101,0xfb840108,0x87b70101,0xfb840108,0x87b80101,0xfb840108,0x87b90101,
	0xfb840108,0x87ba0101,0xfb840108,0x87bb0101,0xfb840108,0x87bc0101,0xfb840108,0x87bd0101,
	0xfb840108,0x87be0101,0xfb840108,0x87bf0101,0xfb840108,0x87c00101,0xfb840108,0x87c10101,
	0xfb840108,0x87c20101,0xfb840108,0x87c30101,0xfb840108,0x87c40101,0xfb840108,0x87c50101,
	0xfb840108,0x87c60101,0xfb840108,0x87c70101,0xfb840108,0x87c80101,0xfb840108,0x87c90101,
	0xfb840108,0x87ca0101,0xfb840108,0x87cb0101,0xfb840108,0x87cc0101,0xfb840108,0x87cd0101,
	0xfb840108,0x87ce0101,0xfb840108,0x87cf0101,0xfb840108,0x87d00101,0xfb840108,0x87d10101,
	0xfb840108,0x87d20101,0xfb840108,0x87d30101,0xfb840108,0x87d40101,0xfb840108,0x87d50101,
	0xfb840108,0x87d60101,0xfb840108,0x87d70101,0xfb840108,0x87d80101,0xfb840108,0x87d90101,
	0xfb840108,0x87da0101,0xfb840108,0x87db0101,0xfb840108,0x87dc0101,0xfb840108,0x87dd0101,
	0xfb840108,0x87de0101,0xfb840108,0x87df0101,0xfb840108,0x87e00101,0xfb840108,0x87e10101,
	0xfb840108,0x87e20101,0xfb840108,0x87e30101,0xfb840108,0x87e40101,0xfb840108,0x87e50101,
	0xfb840108,0x87e60101,0xfb840108,0x87e70101,0xfb840108,0x87e80101,0xfb840108,0x87e90101,
	0xfb840108,0x87ea0101,0xfb840108,0x87eb0101,0xfb840108,0x87ec0101,0xfb840108,0x87ed0101,
	0xfb840108,0x87ee0101,0xfb840108,0x87ef0101,0xfb840108,0x87f00101,0xfb840108,0x87f10101,
	0xfb840108,0x87f20101,0xfb840108,0x87f30101,0xfb840108,0x87f40101,0xfb840108,0x87f50101,
	0xfb840108,0x87f60101,0xfb840108,0x87f70101,0xfb840108,0x87f80101,0xfb840108,0x87f90101,
	0xfb840108,0x87fa0101,0xfb840108,0x87fb0101,0xfb840108,0x87fc0101,0xfb840108,0x87fd0101,
	0xfb840108,0x87fe0101,0xfb840108,0x87ff0101,0xfb840108,0x88000101,0xfb840108,0x88010101,
	0xfb840108,0x88020101,0xfb840108,0x88030101,0xfb840108,0x88040101,0xfb840108,0x88050101,
	0xfb840108,0x88060101,0xfb840108,0x88070101,0xfb840108,0x88080101,0xfb840108,0x88090101,
	0xfb840108,0x880a0101,0xfb840108,0x880b0101,0xfb840108,0x880c0101,0xfb840108,0x880d0101,
	0xfb840108,0x880e0101,0xfb840108,0x880f0101,0xfb840108,0x88100101,0xfb840108,0x88110101,
	0xfb840108,0x88120101,0xfb840108,0x88130101,0xfb840108,0x88140101,0xfb840108,0x88150101,
	0xfb840108,0x88160101,0xfb840108,0x88170101,0xfb840108,0x88180101,0xfb840108,0x88190101,
	0xfb840108,0x881a0101,0xfb840108,0x881b0101,0xfb840108,0x881c0101,0xfb840108,0x881d0101,
	0xfb840108,0x881e0101,0xfb840108,0x881f0101,0xfb840108,0x88200101,0xfb840108,0x88210101,
	0xfb840108,0x88220101,0xfb840108,0x88230101,0xfb840108,0x88240101,0xfb840108,0x88250101,
	0xfb840108,0x88260101,0xfb840108,0x88270101,0xfb840108,0x88280101,0xfb840108,0x88290101,
	0xfb840108,0x882a0101,0xfb840108,0x882b0101,0xfb840108,0x882c0101,0xfb840108,0x882d0101,
	0xfb840108,0x882e0101,0xfb840108,0x882f0101,0xfb840108,0x88300101,0xfb840108,0x88310101,
	0xfb840108,0x88320101,0xfb840108,0x88330101,0xfb840108,0x88340101,0xfb840108,0x88350101,
	0xfb840108,0x88360101,0xfb840108,0x88370101,0xfb840108,0x88380101,0xfb840108,0x88390101,
	0xfb840108,0x883a0101,0xfb840108,0x883b0101,0xfb840108,0x883c0101,0xfb840108,0x883d0101,
	0xfb840108,0x883e0101,0xfb840108,0x883f0101,0xfb840108,0x88400101,0xfb840108,0x88410101,
	0xfb840108,0x88420101,0xfb840108,0x88430101,0xfb840108,0x88440101,0xfb840108,0x88450101,
	0xfb840108,0x88460101,0xfb840108,0x88470101,0xfb840108,0x88480101,0xfb840108,0x88490101,
	0xfb840108,0x884a0101,0xfb840108,0x884b0101,0xfb840108,0x884c0101,0xfb840108,0x884d0101,
	0xfb840108,0x884e0101,0xfb840108,0x884f0101,0xfb840108,0x88500101,0xfb840108,0x88510101,
	0xfb840108,0x88520101,0xfb840108,0x88530101,0xfb840108,0x88540101,0xfb840108,0x88550101,
	0xfb840108,0x88560101,0xfb840108,0x88570101,0xfb840108,0x88580101,0xfb840108,0x88590101,
	0xfb840108,0x885a0101,0xfb840108,0x885b0101,0xfb840108,0x885c0101,0xfb840108,0x885d0101,
	0xfb840108,0x885e0101,0xfb840108,0x885f0101,0xfb840108,0x88600101,0xfb840108,0x88610101,
	0xfb840108,0x88620101,0xfb840108,0x88630101,0xfb840108,0x88640101,0xfb840108,0x88650101,
	0xfb840108,0x88660101,0xfb840108,0x88670101,0xfb840108,0x88680101,0xfb840108,0x88690101,
	0xfb840108,0x886a0101,0xfb840108,0x886b0101,0xfb840108,0x886c0101,0xfb840108,0x886d0101,
	0xfb840108,0x886e0101,0xfb840108,0x886f0101,0xfb840108,0x88700101,0xfb840108,0x88710101,
	0xfb840108,0x88720101,0xfb840108,0x88730101,0xfb840108,0x88740101,0xfb840108,0x88750101,
	0xfb840108,0x88760101,0xfb840108,0x88770101,0xfb840108,0x88780101,0xfb840108,0x88790101,
	0xfb840108,0x887a0101,0xfb840108,0x887b0101,0xfb840108,0x887c0101,0xfb840108,0x887d0101,
	0xfb840108,0x887e0101,0xfb840108,0x887f0101,0xfb840108,0x88800101,0xfb840108,0x88810101,
	0xfb840108,0x88820101,0xfb840108,0x88830101,0xfb840108,0x88840101,0xfb840108,0x88850101,
	0xfb840108,0x88860101,0xfb840108,0x88870101,0xfb840108,0x88880101,0xfb840108,0x88890101,
	0xfb840108,0x888a0101,0xfb840108,0x888b0101,0xfb840108,0x888c0101,0xfb840108,0x888d0101,
	0xfb840108,0x888e0101,0xfb840108,0x888f0101,0xfb840108,0x88900101,0xfb840108,0x88910101,
	0xfb840108,0x88920101,0xfb840108,0x88930101,0xfb840108,0x88940101,0xfb840108,0x88950101,
	0xfb840108,0x88960101,0xfb840108,0x88970101,0xfb840108,0x88980101,0xfb840108,0x88990101,
	0xfb840108,0x889a0101,0xfb840108,0x889b0101,0xfb840108,0x889c0101,0xfb840108,0x889d0101,
	0xfb840108,0x889e0101,0xfb840108,0x889f0101,0xfb840108,0x88a00101,0xfb840108,0x88a10101,
	0xfb840108,0x88a20101,0xfb840108,0x88a30101,0xfb840108,0x88a40101,0xfb840108,0x88a50101,
	0xfb840108,0x88a60101,0xfb840108,0x88a70101,0xfb840108,0x88a80101,0xfb840108,0x88a90101,
	0xfb840108,0x88aa0101,0xfb840108,0x88ab0101,0xfb840108,0x88ac0101,0xfb840108,0x88ad0101,
	0xfb840108,0x88ae0101,0xfb840108,0x88af0101,0xfb840108,0x88b00101,0xfb840108,0x88b10101,
	0xfb840108,0x88b20101,0xfb840108,0x88b30101,0xfb840108,0x88b40101,0xfb840108,0x88b50101,
	0xfb840108,0x88b60101,0xfb840108,0x88b70101,0xfb840108,0x88b80101,0xfb840108,0x88b90101,
	0xfb840108,0x88ba0101,0xfb840108,0x88bb0101,0xfb840108,0x88bc0101,0xfb840108,0x88bd0101,
	0xfb840108,0x88be0101,0xfb840108,0x88bf0101,0xfb840108,0x88c00101,0xfb840108,0x88c10101,
	0xfb840108,0x88c20101,0xfb840108,0x88c30101,0xfb840108,0x88c40101,0xfb840108,0x88c50101,
	0xfb840108,0x88c60101,0xfb840108,0x88c70101,0xfb840108,0x88c80101,0xfb840108,0x88c90101,
	0xfb840108,0x88ca0101,0xfb840108,0x88cb0101,0xfb840108,0x88cc0101,0xfb840108,0x88cd0101,
	0xfb840108,0x88ce0101,0xfb840108,0x88cf0101,0xfb840108,0x88d00101,0xfb840108,0x88d10101,
	0xfb840108,0x88d20101,0xfb840108,0x88d30101,0xfb840108,0x88d40101,0xfb840108,0x88d50101,
	0xfb840108,0x88d60101,0xfb840108,0x88d70101,0xfb840108,0x88d80101,0xfb840108,0x88d90101,
	0xfb840108,0x88da0101,0xfb840108,0x88db0101,0xfb840108,0x88dc0101,0xfb840108,0x88dd0101,
	0xfb840108,0x88de0101,0xfb840108,0x88df0101,0xfb840108,0x88e00101,0xfb840108,0x88e10101,
	0xfb840108,0x88e20101,0xfb840108,0x88e30101,0xfb840108,0x88e40101,0xfb840108,0x88e50101,
	0xfb840108,0x88e60101,0xfb840108,0x88e70101,0xfb840108,0x88e80101,0xfb840108,0x88e90101,
	0xfb840108,0x88ea0101,0xfb840108,0x88eb0101,0xfb840108,0x88ec0101,0xfb840108,0x88ed0101,
	0xfb840108,0x88ee0101,0xfb840108,0x88ef0101,0xfb840108,0x88f00101,0xfb840108,0x88f10101,
	0xfb840108,0x88f20101,0xfb840108,0x88f30101,0xfb840108,0x88f40101,0xfb840108,0x88f50101,
	0xfb840108,0x88f60101,0xfb840108,0x88f70101,0xfb840108,0x88f80101,0xfb840108,0x88f90101,
	0xfb840108,0x88fa0101,0xfb840108,0x88fb0101,0xfb840108,0x88fc0101,0xfb840108,0x88fd0101,
	0xfb840108,0x88fe0101,0xfb840108,0x88ff0101,0xfb840108,0x89000101,0xfb840108,0x89010101,
	0xfb840108,0x89020101,0xfb840108,0x89030101,0xfb840108,0x89040101,0xfb840108,0x89050101,
	0xfb840108,0x89060101,0xfb840108,0x89070101,0xfb840108,0x89080101,0xfb840108,0x89090101,
	0xfb840108,0x890a0101,0xfb840108,0x890b0101,0xfb840108,0x890c0101,0xfb840108,0x890d0101,
	0xfb840108,0x890e0101,0xfb840108,0x890f0101,0xfb840108,0x89100101,0xfb840108,0x89110101,
	0xfb840108,0x89120101,0xfb840108,0x89130101,0xfb840108,0x89140101,0xfb840108,0x89150101,
	0xfb840108,0x89160101,0xfb840108,0x89170101,0xfb840108,0x89180101,0xfb840108,0x89190101,
	0xfb840108,0x891a0101,0xfb840108,0x891b0101,0xfb840108,0x891c0101,0xfb840108,0x891d0101,
	0xfb840108,0x891e0101,0xfb840108,0x891f0101,0xfb840108,0x89200101,0xfb840108,0x89210101,
	0xfb840108,0x89220101,0xfb840108,0x89230101,0xfb840108,0x89240101,0xfb840108,0x89250101,
	0xfb840108,0x89260101,0xfb840108,0x89270101,0xfb840108,0x89280101,0xfb840108,0x89290101,
	0xfb840108,0x892a0101,0xfb840108,0x892b0101,0xfb840108,0x892c0101,0xfb840108,0x892d0101,
	0xfb840108,0x892e0101,0xfb840108,0x892f0101,0xfb840108,0x89300101,0xfb840108,0x89310101,
	0xfb840108,0x89320101,0xfb840108,0x89330101,0xfb840108,0x89340101,0xfb840108,0x89350101,
	0xfb840108,0x89360101,0xfb840108,0x89370101,0xfb840108,0x89380101,0xfb840108,0x89390101,
	0xfb840108,0x893a0101,0xfb840108,0x893b0101,0xfb840108,0x893c0101,0xfb840108,0x893d0101,
	0xfb840108,0x893e0101,0xfb840108,0x893f0101,0xfb840108,0x89400101,0xfb840108,0x89410101,
	0xfb840108,0x89420101,0xfb840108,0x89430101,0xfb840108,0x89440101,0xfb840108,0x89450101,
	0xfb840108,0x89460101,0xfb840108,0x89470101,0xfb840108,0x89480101,0xfb840108,0x89490101,
	0xfb840108,0x894a0101,0xfb840108,0x894b0101,0xfb840108,0x894c0101,0xfb840108,0x894d0101,
	0xfb840108,0x894e0101,0xfb840108,0x894f0101,0xfb840108,0x89500101,0xfb840108,0x89510101,
	0xfb840108,0x89520101,0xfb840108,0x89530101,0xfb840108,0x89540101,0xfb840108,0x89550101,
	0xfb840108,0x89560101,0xfb840108,0x89570101,0xfb840108,0x89580101,0xfb840108,0x89590101,
	0xfb840108,0x895a0101,0xfb840108,0x895b0101,0xfb840108,0x895c0101,0xfb840108,0x895d0101,
	0xfb840108,0x895e0101,0xfb840108,0x895f0101,0xfb840108,0x89600101,0xfb840108,0x89610101,
	0xfb840108,0x89620101,0xfb840108,0x89630101,0xfb840108,0x89640101,0xfb840108,0x89650101,
	0xfb840108,0x89660101,0xfb840108,0x89670101,0xfb840108,0x89680101,0xfb840108,0x89690101,
	0xfb840108,0x896a0101,0xfb840108,0x896b0101,0xfb840108,0x896c0101,0xfb840108,0x896d0101,
	0xfb840108,0x896e0101,0xfb840108,0x896f0101,0xfb840108,0x89700101,0xfb840108,0x89710101,
	0xfb840108,0x89720101,0xfb840108,0x89730101,0xfb840108,0x89740101,0xfb840108,0x89750101,
	0xfb840108,0x89760101,0xfb840108,0x89770101,0xfb840108,0x89780101,0xfb840108,0x89790101,
	0xfb840108,0x897a0101,0xfb840108,0x897b0101,0xfb840108,0x897c0101,0xfb840108,0x897d0101,
	0xfb840108,0x897e0101,0xfb840108,0x897f0101,0xfb840108,0x89800101,0xfb840108,0x89810101,
	0xfb840108,0x89820101,0xfb840108,0x89830101,0xfb840108,0x89840101,0xfb840108,0x89850101,
	0xfb840108,0x89860101,0xfb840108,0x89870101,0xfb840108,0x89880101,0xfb840108,0x89890101,
	0xfb840108,0x898a0101,0xfb840108,0x898b0101,0xfb840108,0x898c0101,0xfb840108,0x898d0101,
	0xfb840108,0x898e0101,0xfb840108,0x898f0101,0xfb840108,0x89900101,0xfb840108,0x89910101,
	0xfb840108,0x89920101,0xfb840108,0x89930101,0xfb840108,0x89940101,0xfb840108,0x89950101,
	0xfb840108,0x89960101,0xfb840108,0x89970101,0xfb840108,0x89980101,0xfb840108,0x89990101,
	0xfb840108,0x899a0101,0xfb840108,0x899b0101,0xfb840108,0x899c0101,0xfb840108,0x899d0101,
	0xfb840108,0x899e0101,0xfb840108,0x899f0101,0xfb840108,0x89a00101,0xfb840108,0x89a10101,
	0xfb840108,0x89a20101,0xfb840108,0x89a30101,0xfb840108,0x89a40101,0xfb840108,0x89a50101,
	0xfb840108,0x89a60101,0xfb840108,0x89a70101,0xfb840108,0x89a80101,0xfb840108,0x89a90101,
	0xfb840108,0x89aa0101,0xfb840108,0x89ab0101,0xfb840108,0x89ac0101,0xfb840108,0x89ad0101,
	0xfb840108,0x89ae0101,0xfb840108,0x89af0101,0xfb840108,0x89b00101,0xfb840108,0x89b10101,
	0xfb840108,0x89b20101,0xfb840108,0x89b30101,0xfb840108,0x89b40101,0xfb840108,0x89b50101,
	0xfb840108,0x89b60101,0xfb840108,0x89b70101,0xfb840108,0x89b80101,0xfb840108,0x89b90101,
	0xfb840108,0x89ba0101,0xfb840108,0x89bb0101,0xfb840108,0x89bc0101,0xfb840108,0x89bd0101,
	0xfb840108,0x89be0101,0xfb840108,0x89bf0101,0xfb840108,0x89c00101,0xfb840108,0x89c10101,
	0xfb840108,0x89c20101,0xfb840108,0x89c30101,0xfb840108,0x89c40101,0xfb840108,0x89c50101,
	0xfb840108,0x89c60101,0xfb840108,0x89c70101,0xfb840108,0x89c80101,0xfb840108,0x89c90101,
	0xfb840108,0x89ca0101,0xfb840108,0x89cb0101,0xfb840108,0x89cc0101,0xfb840108,0x89cd0101,
	0xfb840108,0x89ce0101,0xfb840108,0x89cf0101,0xfb840108,0x89d00101,0xfb840108,0x89d10101,
	0xfb840108,0x89d20101,0xfb840108,0x89d30101,0xfb840108,0x89d40101,0xfb840108,0x89d50101,
	0xfb840108,0x89d60101,0xfb840108,0x89d70101,0xfb840108,0x89d80101,0xfb840108,0x89d90101,
	0xfb840108,0x89da0101,0xfb840108,0x89db0101,0xfb840108,0x89dc0101,0xfb840108,0x89dd0101,
	0xfb840108,0x89de0101,0xfb840108,0x89df0101,0xfb840108,0x89e00101,0xfb840108,0x89e10101,
	0xfb840108,0x89e20101,0xfb840108,0x89e30101,0xfb840108,0x89e40101,0xfb840108,0x89e50101,
	0xfb840108,0x89e60101,0xfb840108,0x89e70101,0xfb840108,0x89e80101,0xfb840108,0x89e90101,
	0xfb840108,0x89ea0101,0xfb840108,0x89eb0101,0xfb840108,0x89ec0101,0xfb840108,0x89ed0101,
	0xfb840108,0x89ee0101,0xfb840108,0x89ef0101,0xfb840108,0x89f00101,0xfb840108,0x89f10101,
	0xfb840108,0x89f20101,0xfb840108,0x89f30101,0xfb840108,0x89f40101,0xfb840108,0x89f50101,
	0xfb840108,0x89f60101,0xfb840108,0x89f70101,0xfb840108,0x89f80101,0xfb840108,0x89f90101,
	0xfb840108,0x89fa0101,0xfb840108,0x89fb0101,0xfb840108,0x89fc0101,0xfb840108,0x89fd0101,
	0xfb840108,0x89fe0101,0xfb840108,0x89ff0101,0xfb840108,0x8a000101,0xfb840108,0x8a010101,
	0xfb840108,0x8a020101,0xfb840108,0x8a030101,0xfb840108,0x8a040101,0xfb840108,0x8a050101,
	0xfb840108,0x8a060101,0xfb840108,0x8a070101,0xfb840108,0x8a080101,0xfb840108,0x8a090101,
	0xfb840108,0x8a0a0101,0xfb840108,0x8a0b0101,0xfb840108,0x8a0c0101,0xfb840108,0x8a0d0101,
	0xfb840108,0x8a0e0101,0xfb840108,0x8a0f0101,0xfb840108,0x8a100101,0xfb840108,0x8a110101,
	0xfb840108,0x8a120101,0xfb840108,0x8a130101,0xfb840108,0x8a140101,0xfb840108,0x8a150101,
	0xfb840108,0x8a160101,0xfb840108,0x8a170101,0xfb840108,0x8a180101,0xfb840108,0x8a190101,
	0xfb840108,0x8a1a0101,0xfb840108,0x8a1b0101,0xfb840108,0x8a1c0101,0xfb840108,0x8a1d0101,
	0xfb840108,0x8a1e0101,0xfb840108,0x8a1f0101,0xfb840108,0x8a200101,0xfb840108,0x8a210101,
	0xfb840108,0x8a220101,0xfb840108,0x8a230101,0xfb840108,0x8a240101,0xfb840108,0x8a250101,
	0xfb840108,0x8a260101,0xfb840108,0x8a270101,0xfb840108,0x8a280101,0xfb840108,0x8a290101,
	0xfb840108,0x8a2a0101,0xfb840108,0x8a2b0101,0xfb840108,0x8a2c0101,0xfb840108,0x8a2d0101,
	0xfb840108,0x8a2e0101,0xfb840108,0x8a2f0101,0xfb840108,0x8a300101,0xfb840108,0x8a310101,
	0xfb840108,0x8a320101,0xfb840108,0x8a330101,0xfb840108,0x8a340101,0xfb840108,0x8a350101,
	0xfb840108,0x8a360101,0xfb840108,0x8a370101,0xfb840108,0x8a380101,0xfb840108,0x8a390101,
	0xfb840108,0x8a3a0101,0xfb840108,0x8a3b0101,0xfb840108,0x8a3c0101,0xfb840108,0x8a3d0101,
	0xfb840108,0x8a3e0101,0xfb840108,0x8a3f0101,0xfb840108,0x8a400101,0xfb840108,0x8a410101,
	0xfb840108,0x8a420101,0xfb840108,0x8a430101,0xfb840108,0x8a440101,0xfb840108,0x8a450101,
	0xfb840108,0x8a460101,0xfb840108,0x8a470101,0xfb840108,0x8a480101,0xfb840108,0x8a490101,
	0xfb840108,0x8a4a0101,0xfb840108,0x8a4b0101,0xfb840108,0x8a4c0101,0xfb840108,0x8a4d0101,
	0xfb840108,0x8a4e0101,0xfb840108,0x8a4f0101,0xfb840108,0x8a500101,0xfb840108,0x8a510101,
	0xfb840108,0x8a520101,0xfb840108,0x8a530101,0xfb840108,0x8a540101,0xfb840108,0x8a550101,
	0xfb840108,0x8a560101,0xfb840108,0x8a570101,0xfb840108,0x8a580101,0xfb840108,0x8a590101,
	0xfb840108,0x8a5a0101,0xfb840108,0x8a5b0101,0xfb840108,0x8a5c0101,0xfb840108,0x8a5d0101,
	0xfb840108,0x8a5e0101,0xfb840108,0x8a5f0101,0xfb840108,0x8a600101,0xfb840108,0x8a610101,
	0xfb840108,0x8a620101,0xfb840108,0x8a630101,0xfb840108,0x8a640101,0xfb840108,0x8a650101,
	0xfb840108,0x8a660101,0xfb840108,0x8a670101,0xfb840108,0x8a680101,0xfb840108,0x8a690101,
	0xfb840108,0x8a6a0101,0xfb840108,0x8a6b0101,0xfb840108,0x8a6c0101,0xfb840108,0x8a6d0101,
	0xfb840108,0x8a6e0101,0xfb840108,0x8a6f0101,0xfb840108,0x8a700101,0xfb840108,0x8a710101,
	0xfb840108,0x8a720101,0xfb840108,0x8a730101,0xfb840108,0x8a740101,0xfb840108,0x8a750101,
	0xfb840108,0x8a760101,0xfb840108,0x8a770101,0xfb840108,0x8a780101,0xfb840108,0x8a790101,
	0xfb840108,0x8a7a0101,0xfb840108,0x8a7b0101,0xfb840108,0x8a7c0101,0xfb840108,0x8a7d0101,
	0xfb840108,0x8a7e0101,0xfb840108,0x8a7f0101,0xfb840108,0x8a800101,0xfb840108,0x8a810101,
	0xfb840108,0x8a820101,0xfb840108,0x8a830101,0xfb840108,0x8a840101,0xfb840108,0x8a850101,
	0xfb840108,0x8a860101,0xfb840108,0x8a870101,0xfb840108,0x8a880101,0xfb840108,0x8a890101,
	0xfb840108,0x8a8a0101,0xfb840108,0x8a8b0101,0xfb840108,0x8a8c0101,0xfb840108,0x8a8d0101,
	0xfb840108,0x8a8e0101,0xfb840108,0x8a8f0101,0xfb840108,0x8a900101,0xfb840108,0x8a910101,
	0xfb840108,0x8a920101,0xfb840108,0x8a930101,0xfb840108,0x8a940101,0xfb840108,0x8a950101,
	0xfb840108,0x8a960101,0xfb840108,0x8a970101,0xfb840108,0x8a980101,0xfb840108,0x8a990101,
	0xfb840108,0x8a9a0101,0xfb840108,0x8a9b0101,0xfb840108,0x8a9c0101,0xfb840108,0x8a9d0101,
	0xfb840108,0x8a9e0101,0xfb840108,0x8a9f0101,0xfb840108,0x8aa00101,0xfb840108,0x8aa10101,
	0xfb840108,0x8aa20101,0xfb840108,0x8aa30101,0xfb840108,0x8aa40101,0xfb840108,0x8aa50101,
	0xfb840108,0x8aa60101,0xfb840108,0x8aa70101,0xfb840108,0x8aa80101,0xfb840108,0x8aa90101,
	0xfb840108,0x8aaa0101,0xfb840108,0x8aab0101,0xfb840108,0x8aac0101,0xfb840108,0x8aad0101,
	0xfb840108,0x8aae0101,0xfb840108,0x8aaf0101,0xfb840108,0x8ab00101,0xfb840108,0x8ab10101,
	0xfb840108,0x8ab20101,0xfb840108,0x8ab30101,0xfb840108,0x8ab40101,0xfb840108,0x8ab50101,
	0xfb840108,0x8ab60101,0xfb840108,0x8ab70101,0xfb840108,0x8ab80101,0xfb840108,0x8ab90101,
	0xfb840108,0x8aba0101,0xfb840108,0x8abb0101,0xfb840108,0x8abc0101,0xfb840108,0x8abd0101,
	0xfb840108,0x8abe0101,0xfb840108,0x8abf0101,0xfb840108,0x8ac00101,0xfb840108,0x8ac10101,
	0xfb840108,0x8ac20101,0xfb840108,0x8ac30101,0xfb840108,0x8ac40101,0xfb840108,0x8ac50101,
	0xfb840108,0x8ac60101,0xfb840108,0x8ac70101,0xfb840108,0x8ac80101,0xfb840108,0x8ac90101,
	0xfb840108,0x8aca0101,0xfb840108,0x8acb0101,0xfb840108,0x8acc0101,0xfb840108,0x8acd0101,
	0xfb840108,0x8ace0101,0xfb840108,0x8acf0101,0xfb840108,0x8ad00101,0xfb840108,0x8ad10101,
	0xfb840108,0x8ad20101,0xfb840108,0x8ad30101,0xfb840108,0x8ad40101,0xfb840108,0x8ad50101,
	0xfb840108,0x8ad60101,0xfb840108,0x8ad70101,0xfb840108,0x8ad80101,0xfb840108,0x8ad90101,
	0xfb840108,0x8ada0101,0xfb840108,0x8adb0101,0xfb840108,0x8adc0101,0xfb840108,0x8add0101,
	0xfb840108,0x8ade0101,0xfb840108,0x8adf0101,0xfb840108,0x8ae00101,0xfb840108,0x8ae10101,
	0xfb840108,0x8ae20101,0xfb840108,0x8ae30101,0xfb840108,0x8ae40101,0xfb840108,0x8ae50101,
	0xfb840108,0x8ae60101,0xfb840108,0x8ae70101,0xfb840108,0x8ae80101,0xfb840108,0x8ae90101,
	0xfb840108,0x8aea0101,0xfb840108,0x8aeb0101,0xfb840108,0x8aec0101,0xfb840108,0x8aed0101,
	0xfb840108,0x8aee0101,0xfb840108,0x8aef0101,0xfb840108,0x8af00101,0xfb840108,0x8af10101,
	0xfb840108,0x8af20101,0xfb840108,0x8af30101,0xfb840108,0x8af40101,0xfb840108,0x8af50101,
	0xfb840108,0x8af60101,0xfb840108,0x8af70101,0xfb840108,0x8af80101,0xfb840108,0x8af90101,
	0xfb840108,0x8afa0101,0xfb840108,0x8afb0101,0xfb840108,0x8afc0101,0xfb840108,0x8afd0101,
	0xfb840108,0x8afe0101,0xfb840108,0x8aff0101,0xfb840108,0x8b000101,0xfb840108,0x8b010101,
	0xfb840108,0x8b020101,0xfb840108,0x8b030101,0xfb840108,0x8b040101,0xfb840108,0x8b050101,
	0xfb840108,0x8b060101,0xfb840108,0x8b070101,0xfb840108,0x8b080101,0xfb840108,0x8b090101,
	0xfb840108,0x8b0a0101,0xfb840108,0x8b0b0101,0xfb840108,0x8b0c0101,0xfb840108,0x8b0d0101,
	0xfb840108,0x8b0e0101,0xfb840108,0x8b0f0101,0xfb840108,0x8b100101,0xfb840108,0x8b110101,
	0xfb840108,0x8b120101,0xfb840108,0x8b130101,0xfb840108,0x8b140101,0xfb840108,0x8b150101,
	0xfb840108,0x8b160101,0xfb840108,0x8b170101,0xfb840108,0x8b180101,0xfb840108,0x8b190101,
	0xfb840108,0x8b1a0101,0xfb840108,0x8b1b0101,0xfb840108,0x8b1c0101,0xfb840108,0x8b1d0101,
	0xfb840108,0x8b1e0101,0xfb840108,0x8b1f0101,0xfb840108,0x8b200101,0xfb840108,0x8b210101,
	0xfb840108,0x8b220101,0xfb840108,0x8b230101,0xfb840108,0x8b240101,0xfb840108,0x8b250101,
	0xfb840108,0x8b260101,0xfb840108,0x8b270101,0xfb840108,0x8b280101,0xfb840108,0x8b290101,
	0xfb840108,0x8b2a0101,0xfb840108,0x8b2b0101,0xfb840108,0x8b2c0101,0xfb840108,0x8b2d0101,
	0xfb840108,0x8b2e0101,0xfb840108,0x8b2f0101,0xfb840108,0x8b300101,0xfb840108,0x8b310101,
	0xfb840108,0x8b320101,0xfb840108,0x8b330101,0xfb840108,0x8b340101,0xfb840108,0x8b350101,
	0xfb840108,0x8b360101,0xfb840108,0x8b370101,0xfb840108,0x8b380101,0xfb840108,0x8b390101,
	0xfb840108,0x8b3a0101,0xfb840108,0x8b3b0101,0xfb840108,0x8b3c0101,0xfb840108,0x8b3d0101,
	0xfb840108,0x8b3e0101,0xfb840108,0x8b3f0101,0xfb840108,0x8b400101,0xfb840108,0x8b410101,
	0xfb840108,0x8b420101,0xfb840108,0x8b430101,0xfb840108,0x8b440101,0xfb840108,0x8b450101,
	0xfb840108,0x8b460101,0xfb840108,0x8b470101,0xfb840108,0x8b480101,0xfb840108,0x8b490101,
	0xfb840108,0x8b4a0101,0xfb840108,0x8b4b0101,0xfb840108,0x8b4c0101,0xfb840108,0x8b4d0101,
	0xfb840108,0x8b4e0101,0xfb840108,0x8b4f0101,0xfb840108,0x8b500101,0xfb840108,0x8b510101,
	0xfb840108,0x8b520101,0xfb840108,0x8b530101,0xfb840108,0x8b540101,0xfb840108,0x8b550101,
	0xfb840108,0x8b560101,0xfb840108,0x8b570101,0xfb840108,0x8b580101,0xfb840108,0x8b590101,
	0xfb840108,0x8b5a0101,0xfb840108,0x8b5b0101,0xfb840108,0x8b5c0101,0xfb840108,0x8b5d0101,
	0xfb840108,0x8b5e0101,0xfb840108,0x8b5f0101,0xfb840108,0x8b600101,0xfb840108,0x8b610101,
	0xfb840108,0x8b620101,0xfb840108,0x8b630101,0xfb840108,0x8b640101,0xfb840108,0x8b650101,
	0xfb840108,0x8b660101,0xfb840108,0x8b670101,0xfb840108,0x8b680101,0xfb840108,0x8b690101,
	0xfb840108,0x8b6a0101,0xfb840108,0x8b6b0101,0xfb840108,0x8b6c0101,0xfb840108,0x8b6d0101,
	0xfb840108,0x8b6e0101,0xfb840108,0x8b6f0101,0xfb840108,0x8b700101,0xfb840108,0x8b710101,
	0xfb840108,0x8b720101,0xfb840108,0x8b730101,0xfb840108,0x8b740101,0xfb840108,0x8b750101,
	0xfb840108,0x8b760101,0xfb840108,0x8b770101,0xfb840108,0x8b780101,0xfb840108,0x8b790101,
	0xfb840108,0x8b7a0101,0xfb840108,0x8b7b0101,0xfb840108,0x8b7c0101,0xfb840108,0x8b7d0101,
	0xfb840108,0x8b7e0101,0xfb840108,0x8b7f0101,0xfb840108,0x8b800101,0xfb840108,0x8b810101,
	0xfb840108,0x8b820101,0xfb840108,0x8b830101,0xfb840108,0x8b840101,0xfb840108,0x8b850101,
	0xfb840108,0x8b860101,0xfb840108,0x8b870101,0xfb840108,0x8b880101,0xfb840108,0x8b890101,
	0xfb840108,0x8b8a0101,0xfb840108,0x8b8b0101,0xfb840108,0x8b8c0101,0xfb840108,0x8b8d0101,
	0xfb840108,0x8b8e0101,0xfb840108,0x8b8f0101,0xfb840108,0x8b900101,0xfb840108,0x8b910101,
	0xfb840108,0x8b920101,0xfb840108,0x8b930101,0xfb840108,0x8b940101,0xfb840108,0x8b950101,
	0xfb840108,0x8b960101,0xfb840108,0x8b970101,0xfb840108,0x8b980101,0xfb840108,0x8b990101,
	0xfb840108,0x8b9a0101,0xfb840108,0x8b9b0101,0xfb840108,0x8b9c0101,0xfb840108,0x8b9d0101,
	0xfb840108,0x8b9e0101,0xfb840108,0x8b9f0101,0xfb840108,0x8ba00101,0xfb840108,0x8ba10101,
	0xfb840108,0x8ba20101,0xfb840108,0x8ba30101,0xfb840108,0x8ba40101,0xfb840108,0x8ba50101,
	0xfb840108,0x8ba60101,0xfb840108,0x8ba70101,0xfb840108,0x8ba80101,0xfb840108,0x8ba90101,
	0xfb840108,0x8baa0101,0xfb840108,0x8bab0101,0xfb840108,0x8bac0101,0xfb840108,0x8bad0101,
	0xfb840108,0x8bae0101,0xfb840108,0x8baf0101,0xfb840108,0x8bb00101,0xfb840108,0x8bb10101,
	0xfb840108,0x8bb20101,0xfb840108,0x8bb30101,0xfb840108,0x8bb40101,0xfb840108,0x8bb50101,
	0xfb840108,0x8bb60101,0xfb840108,0x8bb70101,0xfb840108,0x8bb80101,0xfb840108,0x8bb90101,
	0xfb840108,0x8bba0101,0xfb840108,0x8bbb0101,0xfb840108,0x8bbc0101,0xfb840108,0x8bbd0101,
	0xfb840108,0x8bbe0101,0xfb840108,0x8bbf0101,0xfb840108,0x8bc00101,0xfb840108,0x8bc10101,
	0xfb840108,0x8bc20101,0xfb840108,0x8bc30101,0xfb840108,0x8bc40101,0xfb840108,0x8bc50101,
	0xfb840108,0x8bc60101,0xfb840108,0x8bc70101,0xfb840108,0x8bc80101,0xfb840108,0x8bc90101,
	0xfb840108,0x8bca0101,0xfb840108,0x8bcb0101,0xfb840108,0x8bcc0101,0xfb840108,0x8bcd0101,
	0xfb840108,0x8bce0101,0xfb840108,0x8bcf0101,0xfb840108,0x8bd00101,0xfb840108,0x8bd10101,
	0xfb840108,0x8bd20101,0xfb840108,0x8bd30101,0xfb840108,0x8bd40101,0xfb840108,0x8bd50101,
	0xfb840108,0x8bd60101,0xfb840108,0x8bd70101,0xfb840108,0x8bd80101,0xfb840108,0x8bd90101,
	0xfb840108,0x8bda0101,0xfb840108,0x8bdb0101,0xfb840108,0x8bdc0101,0xfb840108,0x8bdd0101,
	0xfb840108,0x8bde0101,0xfb840108,0x8bdf0101,0xfb840108,0x8be00101,0xfb840108,0x8be10101,
	0xfb840108,0x8be20101,0xfb840108,0x8be30101,0xfb840108,0x8be40101,0xfb840108,0x8be50101,
	0xfb840108,0x8be60101,0xfb840108,0x8be70101,0xfb840108,0x8be80101,0xfb840108,0x8be90101,
	0xfb840108,0x8bea0101,0xfb840108,0x8beb0101,0xfb840108,0x8bec0101,0xfb840108,0x8bed0101,
	0xfb840108,0x8bee0101,0xfb840108,0x8bef0101,0xfb840108,0x8bf00101,0xfb840108,0x8bf10101,
	0xfb840108,0x8bf20101,0xfb840108,0x8bf30101,0xfb840108,0x8bf40101,0xfb840108,0x8bf50101,
	0xfb840108,0x8bf60101,0xfb840108,0x8bf70101,0xfb840108,0x8bf80101,0xfb840108,0x8bf90101,
	0xfb840108,0x8bfa0101,0xfb840108,0x8bfb0101,0xfb840108,0x8bfc0101,0xfb840108,0x8bfd0101,
	0xfb840108,0x8bfe0101,0xfb840108,0x8bff0101,0xfb840108,0x8c000101,0xfb840108,0x8c010101,
	0xfb840108,0x8c020101,0xfb840108,0x8c030101,0xfb840108,0x8c040101,0xfb840108,0x8c050101,
	0xfb840108,0x8c060101,0xfb840108,0x8c070101,0xfb840108,0x8c080101,0xfb840108,0x8c090101,
	0xfb840108,0x8c0a0101,0xfb840108,0x8c0b0101,0xfb840108,0x8c0c0101,0xfb840108,0x8c0d0101,
	0xfb840108,0x8c0e0101,0xfb840108,0x8c0f0101,0xfb840108,0x8c100101,0xfb840108,0x8c110101,
	0xfb840108,0x8c120101,0xfb840108,0x8c130101,0xfb840108,0x8c140101,0xfb840108,0x8c150101,
	0xfb840108,0x8c160101,0xfb840108,0x8c170101,0xfb840108,0x8c180101,0xfb840108,0x8c190101,
	0xfb840108,0x8c1a0101,0xfb840108,0x8c1b0101,0xfb840108,0x8c1c0101,0xfb840108,0x8c1d0101,
	0xfb840108,0x8c1e0101,0xfb840108,0x8c1f0101,0xfb840108,0x8c200101,0xfb840108,0x8c210101,
	0xfb840108,0x8c220101,0xfb840108,0x8c230101,0xfb840108,0x8c240101,0xfb840108,0x8c250101,
	0xfb840108,0x8c260101,0xfb840108,0x8c270101,0xfb840108,0x8c280101,0xfb840108,0x8c290101,
	0xfb840108,0x8c2a0101,0xfb840108,0x8c2b0101,0xfb840108,0x8c2c0101,0xfb840108,0x8c2d0101,
	0xfb840108,0x8c2e0101,0xfb840108,0x8c2f0101,0xfb840108,0x8c300101,0xfb840108,0x8c310101,
	0xfb840108,0x8c320101,0xfb840108,0x8c330101,0xfb840108,0x8c340101,0xfb840108,0x8c350101,
	0xfb840108,0x8c360101,0xfb840108,0x8c370101,0xfb840108,0x8c380101,0xfb840108,0x8c390101,
	0xfb840108,0x8c3a0101,0xfb840108,0x8c3b0101,0xfb840108,0x8c3c0101,0xfb840108,0x8c3d0101,
	0xfb840108,0x8c3e0101,0xfb840108,0x8c3f0101,0xfb840108,0x8c400101,0xfb840108,0x8c410101,
	0xfb840108,0x8c420101,0xfb840108,0x8c430101,0xfb840108,0x8c440101,0xfb840108,0x8c450101,
	0xfb840108,0x8c460101,0xfb840108,0x8c470101,0xfb840108,0x8c480101,0xfb840108,0x8c490101,
	0xfb840108,0x8c4a0101,0xfb840108,0x8c4b0101,0xfb840108,0x8c4c0101,0xfb840108,0x8c4d0101,
	0xfb840108,0x8c4e0101,0xfb840108,0x8c4f0101,0xfb840108,0x8c500101,0xfb840108,0x8c510101,
	0xfb840108,0x8c520101,0xfb840108,0x8c530101,0xfb840108,0x8c540101,0xfb840108,0x8c550101,
	0xfb840108,0x8c560101,0xfb840108,0x8c570101,0xfb840108,0x8c580101,0xfb840108,0x8c590101,
	0xfb840108,0x8c5a0101,0xfb840108,0x8c5b0101,0xfb840108,0x8c5c0101,0xfb840108,0x8c5d0101,
	0xfb840108,0x8c5e0101,0xfb840108,0x8c5f0101,0xfb840108,0x8c600101,0xfb840108,0x8c610101,
	0xfb840108,0x8c620101,0xfb840108,0x8c630101,0xfb840108,0x8c640101,0xfb840108,0x8c650101,
	0xfb840108,0x8c660101,0xfb840108,0x8c670101,0xfb840108,0x8c680101,0xfb840108,0x8c690101,
	0xfb840108,0x8c6a0101,0xfb840108,0x8c6b0101,0xfb840108,0x8c6c0101,0xfb840108,0x8c6d0101,
	0xfb840108,0x8c6e0101,0xfb840108,0x8c6f0101,0xfb840108,0x8c700101,0xfb840108,0x8c710101,
	0xfb840108,0x8c720101,0xfb840108,0x8c730101,0xfb840108,0x8c740101,0xfb840108,0x8c750101,
	0xfb840108,0x8c760101,0xfb840108,0x8c770101,0xfb840108,0x8c780101,0xfb840108,0x8c790101,
	0xfb840108,0x8c7a0101,0xfb840108,0x8c7b0101,0xfb840108,0x8c7c0101,0xfb840108,0x8c7d0101,
	0xfb840108,0x8c7e0101,0xfb840108,0x8c7f0101,0xfb840108,0x8c800101,0xfb840108,0x8c810101,
	0xfb840108,0x8c820101,0xfb840108,0x8c830101,0xfb840108,0x8c840101,0xfb840108,0x8c850101,
	0xfb840108,0x8c860101,0xfb840108,0x8c870101,0xfb840108,0x8c880101,0xfb840108,0x8c890101,
	0xfb840108,0x8c8a0101,0xfb840108,0x8c8b0101,0xfb840108,0x8c8c0101,0xfb840108,0x8c8d0101,
	0xfb840108,0x8c8e0101,0xfb840108,0x8c8f0101,0xfb840108,0x8c900101,0xfb840108,0x8c910101,
	0xfb840108,0x8c920101,0xfb840108,0x8c930101,0xfb840108,0x8c940101,0xfb840108,0x8c950101,
	0xfb840108,0x8c960101,0xfb840108,0x8c970101,0xfb840108,0x8c980101,0xfb840108,0x8c990101,
	0xfb840108,0x8c9a0101,0xfb840108,0x8c9b0101,0xfb840108,0x8c9c0101,0xfb840108,0x8c9d0101,
	0xfb840108,0x8c9e0101,0xfb840108,0x8c9f0101,0xfb840108,0x8ca00101,0xfb840108,0x8ca10101,
	0xfb840108,0x8ca20101,0xfb840108,0x8ca30101,0xfb840108,0x8ca40101,0xfb840108,0x8ca50101,
	0xfb840108,0x8ca60101,0xfb840108,0x8ca70101,0xfb840108,0x8ca80101,0xfb840108,0x8ca90101,
	0xfb840108,0x8caa0101,0xfb840108,0x8cab0101,0xfb840108,0x8cac0101,0xfb840108,0x8cad0101,
	0xfb840108,0x8cae0101,0xfb840108,0x8caf0101,0xfb840108,0x8cb00101,0xfb840108,0x8cb10101,
	0xfb840108,0x8cb20101,0xfb840108,0x8cb30101,0xfb840108,0x8cb40101,0xfb840108,0x8cb50101,
	0xfb840108,0x8cb60101,0xfb840108,0x8cb70101,0xfb840108,0x8cb80101,0xfb840108,0x8cb90101,
	0xfb840108,0x8cba0101,0xfb840108,0x8cbb0101,0xfb840108,0x8cbc0101,0xfb840108,0x8cbd0101,
	0xfb840108,0x8cbe0101,0xfb840108,0x8cbf0101,0xfb840108,0x8cc00101,0xfb840108,0x8cc10101,
	0xfb840108,0x8cc20101,0xfb840108,0x8cc30101,0xfb840108,0x8cc40101,0xfb840108,0x8cc50101,
	0xfb840108,0x8cc60101,0xfb840108,0x8cc70101,0xfb840108,0x8cc80101,0xfb840108,0x8cc90101,
	0xfb840108,0x8cca0101,0xfb840108,0x8ccb0101,0xfb840108,0x8ccc0101,0xfb840108,0x8ccd0101,
	0xfb840108,0x8cce0101,0xfb840108,0x8ccf0101,0xfb840108,0x8cd00101,0xfb840108,0x8cd10101,
	0xfb840108,0x8cd20101,0xfb840108,0x8cd30101,0xfb840108,0x8cd40101,0xfb840108,0x8cd50101,
	0xfb840108,0x8cd60101,0xfb840108,0x8cd70101,0xfb840108,0x8cd80101,0xfb840108,0x8cd90101,
	0xfb840108,0x8cda0101,0xfb840108,0x8cdb0101,0xfb840108,0x8cdc0101,0xfb840108,0x8cdd0101,
	0xfb840108,0x8cde0101,0xfb840108,0x8cdf0101,0xfb840108,0x8ce00101,0xfb840108,0x8ce10101,
	0xfb840108,0x8ce20101,0xfb840108,0x8ce30101,0xfb840108,0x8ce40101,0xfb840108,0x8ce50101,
	0xfb840108,0x8ce60101,0xfb840108,0x8ce70101,0xfb840108,0x8ce80101,0xfb840108,0x8ce90101,
	0xfb840108,0x8cea0101,0xfb840108,0x8ceb0101,0xfb840108,0x8cec0101,0xfb840108,0x8ced0101,
	0xfb840108,0x8cee0101,0xfb840108,0x8cef0101,0xfb840108,0x8cf00101,0xfb840108,0x8cf10101,
	0xfb840108,0x8cf20101,0xfb840108,0x8cf30101,0xfb840108,0x8cf40101,0xfb840108,0x8cf50101,
	0xfb840108,0x8cf60101,0xfb840108,0x8cf70101,0xfb840108,0x8cf80101,0xfb840108,0x8cf90101,
	0xfb840108,0x8cfa0101,0xfb840108,0x8cfb0101,0xfb840108,0x8cfc0101,0xfb840108,0x8cfd0101,
	0xfb840108,0x8cfe0101,0xfb840108,0x8cff0101,0xfb840108,0x8d000101,0xfb840108,0x8d010101,
	0xfb840108,0x8d020101,0xfb840108,0x8d030101,0xfb840108,0x8d040101,0xfb840108,0x8d050101,
	0xfb840108,0x8d060101,0xfb840108,0x8d070101,0xfb840108,0x8d080101,0xfb840108,0x8d090101,
	0xfb840108,0x8d0a0101,0xfb840108,0x8d0b0101,0xfb840108,0x8d0c0101,0xfb840108,0x8d0d0101,
	0xfb840108,0x8d0e0101,0xfb840108,0x8d0f0101,0xfb840108,0x8d100101,0xfb840108,0x8d110101,
	0xfb840108,0x8d120101,0xfb840108,0x8d130101,0xfb840108,0x8d140101,0xfb840108,0x8d150101,
	0xfb840108,0x8d160101,0xfb840108,0x8d170101,0xfb840108,0x8d180101,0xfb840108,0x8d190101,
	0xfb840108,0x8d1a0101,0xfb840108,0x8d1b0101,0xfb840108,0x8d1c0101,0xfb840108,0x8d1d0101,
	0xfb840108,0x8d1e0101,0xfb840108,0x8d1f0101,0xfb840108,0x8d200101,0xfb840108,0x8d210101,
	0xfb840108,0x8d220101,0xfb840108,0x8d230101,0xfb840108,0x8d240101,0xfb840108,0x8d250101,
	0xfb840108,0x8d260101,0xfb840108,0x8d270101,0xfb840108,0x8d280101,0xfb840108,0x8d290101,
	0xfb840108,0x8d2a0101,0xfb840108,0x8d2b0101,0xfb840108,0x8d2c0101,0xfb840108,0x8d2d0101,
	0xfb840108,0x8d2e0101,0xfb840108,0x8d2f0101,0xfb840108,0x8d300101,0xfb840108,0x8d310101,
	0xfb840108,0x8d320101,0xfb840108,0x8d330101,0xfb840108,0x8d340101,0xfb840108,0x8d350101,
	0xfb840108,0x8d360101,0xfb840108,0x8d370101,0xfb840108,0x8d380101,0xfb840108,0x8d390101,
	0xfb840108,0x8d3a0101,0xfb840108,0x8d3b0101,0xfb840108,0x8d3c0101,0xfb840108,0x8d3d0101,
	0xfb840108,0x8d3e0101,0xfb840108,0x8d3f0101,0xfb840108,0x8d400101,0xfb840108,0x8d410101,
	0xfb840108,0x8d420101,0xfb840108,0x8d430101,0xfb840108,0x8d440101,0xfb840108,0x8d450101,
	0xfb840108,0x8d460101,0xfb840108,0x8d470101,0xfb840108,0x8d480101,0xfb840108,0x8d490101,
	0xfb840108,0x8d4a0101,0xfb840108,0x8d4b0101,0xfb840108,0x8d4c0101,0xfb840108,0x8d4d0101,
	0xfb840108,0x8d4e0101,0xfb840108,0x8d4f0101,0xfb840108,0x8d500101,0xfb840108,0x8d510101,
	0xfb840108,0x8d520101,0xfb840108,0x8d530101,0xfb840108,0x8d540101,0xfb840108,0x8d550101,
	0xfb840108,0x8d560101,0xfb840108,0x8d570101,0xfb840108,0x8d580101,0xfb840108,0x8d590101,
	0xfb840108,0x8d5a0101,0xfb840108,0x8d5b0101,0xfb840108,0x8d5c0101,0xfb840108,0x8d5d0101,
	0xfb840108,0x8d5e0101,0xfb840108,0x8d5f0101,0xfb840108,0x8d600101,0xfb840108,0x8d610101,
	0xfb840108,0x8d620101,0xfb840108,0x8d630101,0xfb840108,0x8d640101,0xfb840108,0x8d650101,
	0xfb840108,0x8d660101,0xfb840108,0x8d670101,0xfb840108,0x8d680101,0xfb840108,0x8d690101,
	0xfb840108,0x8d6a0101,0xfb840108,0x8d6b0101,0xfb840108,0x8d6c0101,0xfb840108,0x8d6d0101,
	0xfb840108,0x8d6e0101,0xfb840108,0x8d6f0101,0xfb840108,0x8d700101,0xfb840108,0x8d710101,
	0xfb840108,0x8d720101,0xfb840108,0x8d730101,0xfb840108,0x8d740101,0xfb840108,0x8d750101,
	0xfb840108,0x8d760101,0xfb840108,0x8d770101,0xfb840108,0x8d780101,0xfb840108,0x8d790101,
	0xfb840108,0x8d7a0101,0xfb840108,0x8d7b0101,0xfb840108,0x8d7c0101,0xfb840108,0x8d7d0101,
	0xfb840108,0x8d7e0101,0xfb840108,0x8d7f0101,0xfb840108,0x8d800101,0xfb840108,0x8d810101,
	0xfb840108,0x8d820101,0xfb840108,0x8d830101,0xfb840108,0x8d840101,0xfb840108,0x8d850101,
	0xfb840108,0x8d860101,0xfb840108,0x8d870101,0xfb840108,0x8d880101,0xfb840108,0x8d890101,
	0xfb840108,0x8d8a0101,0xfb840108,0x8d8b0101,0xfb840108,0x8d8c0101,0xfb840108,0x8d8d0101,
	0xfb840108,0x8d8e0101,0xfb840108,0x8d8f0101,0xfb840108,0x8d900101,0xfb840108,0x8d910101,
	0xfb840108,0x8d920101,0xfb840108,0x8d930101,0xfb840108,0x8d940101,0xfb840108,0x8d950101,
	0xfb840108,0x8d960101,0xfb840108,0x8d970101,0xfb840108,0x8d980101,0xfb840108,0x8d990101,
	0xfb840108,0x8d9a0101,0xfb840108,0x8d9b0101,0xfb840108,0x8d9c0101,0xfb840108,0x8d9d0101,
	0xfb840108,0x8d9e0101,0xfb840108,0x8d9f0101,0xfb840108,0x8da00101,0xfb840108,0x8da10101,
	0xfb840108,0x8da20101,0xfb840108,0x8da30101,0xfb840108,0x8da40101,0xfb840108,0x8da50101,
	0xfb840108,0x8da60101,0xfb840108,0x8da70101,0xfb840108,0x8da80101,0xfb840108,0x8da90101,
	0xfb840108,0x8daa0101,0xfb840108,0x8dab0101,0xfb840108,0x8dac0101,0xfb840108,0x8dad0101,
	0xfb840108,0x8dae0101,0xfb840108,0x8daf0101,0xfb840108,0x8db00101,0xfb840108,0x8db10101,
	0xfb840108,0x8db20101,0xfb840108,0x8db30101,0xfb840108,0x8db40101,0xfb840108,0x8db50101,
	0xfb840108,0x8db60101,0xfb840108,0x8db70101,0xfb840108,0x8db80101,0xfb840108,0x8db90101,
	0xfb840108,0x8dba0101,0xfb840108,0x8dbb0101,0xfb840108,0x8dbc0101,0xfb840108,0x8dbd0101,
	0xfb840108,0x8dbe0101,0xfb840108,0x8dbf0101,0xfb840108,0x8dc00101,0xfb840108,0x8dc10101,
	0xfb840108,0x8dc20101,0xfb840108,0x8dc30101,0xfb840108,0x8dc40101,0xfb840108,0x8dc50101,
	0xfb840108,0x8dc60101,0xfb840108,0x8dc70101,0xfb840108,0x8dc80101,0xfb840108,0x8dc90101,
	0xfb840108,0x8dca0101,0xfb840108,0x8dcb0101,0xfb840108,0x8dcc0101,0xfb840108,0x8dcd0101,
	0xfb840108,0x8dce0101,0xfb840108,0x8dcf0101,0xfb840108,0x8dd00101,0xfb840108,0x8dd10101,
	0xfb840108,0x8dd20101,0xfb840108,0x8dd30101,0xfb840108,0x8dd40101,0xfb840108,0x8dd50101,
	0xfb840108,0x8dd60101,0xfb840108,0x8dd70101,0xfb840108,0x8dd80101,0xfb840108,0x8dd90101,
	0xfb840108,0x8dda0101,0xfb840108,0x8ddb0101,0xfb840108,0x8ddc0101,0xfb840108,0x8ddd0101,
	0xfb840108,0x8dde0101,0xfb840108,0x8ddf0101,0xfb840108,0x8de00101,0xfb840108,0x8de10101,
	0xfb840108,0x8de20101,0xfb840108,0x8de30101,0xfb840108,0x8de40101,0xfb840108,0x8de50101,
	0xfb840108,0x8de60101,0xfb840108,0x8de70101,0xfb840108,0x8de80101,0xfb840108,0x8de90101,
	0xfb840108,0x8dea0101,0xfb840108,0x8deb0101,0xfb840108,0x8dec0101,0xfb840108,0x8ded0101,
	0xfb840108,0x8dee0101,0xfb840108,0x8def0101,0xfb840108,0x8df00101,0xfb840108,0x8df10101,
	0xfb840108,0x8df20101,0xfb840108,0x8df30101,0xfb840108,0x8df40101,0xfb840108,0x8df50101,
	0xfb840108,0x8df60101,0xfb840108,0x8df70101,0xfb840108,0x8df80101,0xfb840108,0x8df90101,
	0xfb840108,0x8dfa0101,0xfb840108,0x8dfb0101,0xfb840108,0x8dfc0101,0xfb840108,0x8dfd0101,
	0xfb840108,0x8dfe0101,0xfb840108,0x8dff0101,0xfb840108,0x8e000101,0xfb840108,0x8e010101,
	0xfb840108,0x8e020101,0xfb840108,0x8e030101,0xfb840108,0x8e040101,0xfb840108,0x8e050101,
	0xfb840108,0x8e060101,0xfb840108,0x8e070101,0xfb840108,0x8e080101,0xfb840108,0x8e090101,
	0xfb840108,0x8e0a0101,0xfb840108,0x8e0b0101,0xfb840108,0x8e0c0101,0xfb840108,0x8e0d0101,
	0xfb840108,0x8e0e0101,0xfb840108,0x8e0f0101,0xfb840108,0x8e100101,0xfb840108,0x8e110101,
	0xfb840108,0x8e120101,0xfb840108,0x8e130101,0xfb840108,0x8e140101,0xfb840108,0x8e150101,
	0xfb840108,0x8e160101,0xfb840108,0x8e170101,0xfb840108,0x8e180101,0xfb840108,0x8e190101,
	0xfb840108,0x8e1a0101,0xfb840108,0x8e1b0101,0xfb840108,0x8e1c0101,0xfb840108,0x8e1d0101,
	0xfb840108,0x8e1e0101,0xfb840108,0x8e1f0101,0xfb840108,0x8e200101,0xfb840108,0x8e210101,
	0xfb840108,0x8e220101,0xfb840108,0x8e230101,0xfb840108,0x8e240101,0xfb840108,0x8e250101,
	0xfb840108,0x8e260101,0xfb840108,0x8e270101,0xfb840108,0x8e280101,0xfb840108,0x8e290101,
	0xfb840108,0x8e2a0101,0xfb840108,0x8e2b0101,0xfb840108,0x8e2c0101,0xfb840108,0x8e2d0101,
	0xfb840108,0x8e2e0101,0xfb840108,0x8e2f0101,0xfb840108,0x8e300101,0xfb840108,0x8e310101,
	0xfb840108,0x8e320101,0xfb840108,0x8e330101,0xfb840108,0x8e340101,0xfb840108,0x8e350101,
	0xfb840108,0x8e360101,0xfb840108,0x8e370101,0xfb840108,0x8e380101,0xfb840108,0x8e390101,
	0xfb840108,0x8e3a0101,0xfb840108,0x8e3b0101,0xfb840108,0x8e3c0101,0xfb840108,0x8e3d0101,
	0xfb840108,0x8e3e0101,0xfb840108,0x8e3f0101,0xfb840108,0x8e400101,0xfb840108,0x8e410101,
	0xfb840108,0x8e420101,0xfb840108,0x8e430101,0xfb840108,0x8e440101,0xfb840108,0x8e450101,
	0xfb840108,0x8e460101,0xfb840108,0x8e470101,0xfb840108,0x8e480101,0xfb840108,0x8e490101,
	0xfb840108,0x8e4a0101,0xfb840108,0x8e4b0101,0xfb840108,0x8e4c0101,0xfb840108,0x8e4d0101,
	0xfb840108,0x8e4e0101,0xfb840108,0x8e4f0101,0xfb840108,0x8e500101,0xfb840108,0x8e510101,
	0xfb840108,0x8e520101,0xfb840108,0x8e530101,0xfb840108,0x8e540101,0xfb840108,0x8e550101,
	0xfb840108,0x8e560101,0xfb840108,0x8e570101,0xfb840108,0x8e580101,0xfb840108,0x8e590101,
	0xfb840108,0x8e5a0101,0xfb840108,0x8e5b0101,0xfb840108,0x8e5c0101,0xfb840108,0x8e5d0101,
	0xfb840108,0x8e5e0101,0xfb840108,0x8e5f0101,0xfb840108,0x8e600101,0xfb840108,0x8e610101,
	0xfb840108,0x8e620101,0xfb840108,0x8e630101,0xfb840108,0x8e640101,0xfb840108,0x8e650101,
	0xfb840108,0x8e660101,0xfb840108,0x8e670101,0xfb840108,0x8e680101,0xfb840108,0x8e690101,
	0xfb840108,0x8e6a0101,0xfb840108,0x8e6b0101,0xfb840108,0x8e6c0101,0xfb840108,0x8e6d0101,
	0xfb840108,0x8e6e0101,0xfb840108,0x8e6f0101,0xfb840108,0x8e700101,0xfb840108,0x8e710101,
	0xfb840108,0x8e720101,0xfb840108,0x8e730101,0xfb840108,0x8e740101,0xfb840108,0x8e750101,
	0xfb840108,0x8e760101,0xfb840108,0x8e770101,0xfb840108,0x8e780101,0xfb840108,0x8e790101,
	0xfb840108,0x8e7a0101,0xfb840108,0x8e7b0101,0xfb840108,0x8e7c0101,0xfb840108,0x8e7d0101,
	0xfb840108,0x8e7e0101,0xfb840108,0x8e7f0101,0xfb840108,0x8e800101,0xfb840108,0x8e810101,
	0xfb840108,0x8e820101,0xfb840108,0x8e830101,0xfb840108,0x8e840101,0xfb840108,0x8e850101,
	0xfb840108,0x8e860101,0xfb840108,0x8e870101,0xfb840108,0x8e880101,0xfb840108,0x8e890101,
	0xfb840108,0x8e8a0101,0xfb840108,0x8e8b0101,0xfb840108,0x8e8c0101,0xfb840108,0x8e8d0101,
	0xfb840108,0x8e8e0101,0xfb840108,0x8e8f0101,0xfb840108,0x8e900101,0xfb840108,0x8e910101,
	0xfb840108,0x8e920101,0xfb840108,0x8e930101,0xfb840108,0x8e940101,0xfb840108,0x8e950101,
	0xfb840108,0x8e960101,0xfb840108,0x8e970101,0xfb840108,0x8e980101,0xfb840108,0x8e990101,
	0xfb840108,0x8e9a0101,0xfb840108,0x8e9b0101,0xfb840108,0x8e9c0101,0xfb840108,0x8e9d0101,
	0xfb840108,0x8e9e0101,0xfb840108,0x8e9f0101,0xfb840108,0x8ea00101,0xfb840108,0x8ea10101,
	0xfb840108,0x8ea20101,0xfb840108,0x8ea30101,0xfb840108,0x8ea40101,0xfb840108,0x8ea50101,
	0xfb840108,0x8ea60101,0xfb840108,0x8ea70101,0xfb840108,0x8ea80101,0xfb840108,0x8ea90101,
	0xfb840108,0x8eaa0101,0xfb840108,0x8eab0101,0xfb840108,0x8eac0101,0xfb840108,0x8ead0101,
	0xfb840108,0x8eae0101,0xfb840108,0x8eaf0101,0xfb840108,0x8eb00101,0xfb840108,0x8eb10101,
	0xfb840108,0x8eb20101,0xfb840108,0x8eb30101,0xfb840108,0x8eb40101,0xfb840108,0x8eb50101,
	0xfb840108,0x8eb60101,0xfb840108,0x8eb70101,0xfb840108,0x8eb80101,0xfb840108,0x8eb90101,
	0xfb840108,0x8eba0101,0xfb840108,0x8ebb0101,0xfb840108,0x8ebc0101,0xfb840108,0x8ebd0101,
	0xfb840108,0x8ebe0101,0xfb840108,0x8ebf0101,0xfb840108,0x8ec00101,0xfb840108,0x8ec10101,
	0xfb840108,0x8ec20101,0xfb840108,0x8ec30101,0xfb840108,0x8ec40101,0xfb840108,0x8ec50101,
	0xfb840108,0x8ec60101,0xfb840108,0x8ec70101,0xfb840108,0x8ec80101,0xfb840108,0x8ec90101,
	0xfb840108,0x8eca0101,0xfb840108,0x8ecb0101,0xfb840108,0x8ecc0101,0xfb840108,0x8ecd0101,
	0xfb840108,0x8ece0101,0xfb840108,0x8ecf0101,0xfb840108,0x8ed00101,0xfb840108,0x8ed10101,
	0xfb840108,0x8ed20101,0xfb840108,0x8ed30101,0xfb840108,0x8ed40101,0xfb840108,0x8ed50101,
	0xfb840108,0x8ed60101,0xfb840108,0x8ed70101,0xfb840108,0x8ed80101,0xfb840108,0x8ed90101,
	0xfb840108,0x8eda0101,0xfb840108,0x8edb0101,0xfb840108,0x8edc0101,0xfb840108,0x8edd0101,
	0xfb840108,0x8ede0101,0xfb840108,0x8edf0101,0xfb840108,0x8ee00101,0xfb840108,0x8ee10101,
	0xfb840108,0x8ee20101,0xfb840108,0x8ee30101,0xfb840108,0x8ee40101,0xfb840108,0x8ee50101,
	0xfb840108,0x8ee60101,0xfb840108,0x8ee70101,0xfb840108,0x8ee80101,0xfb840108,0x8ee90101,
	0xfb840108,0x8eea0101,0xfb840108,0x8eeb0101,0xfb840108,0x8eec0101,0xfb840108,0x8eed0101,
	0xfb840108,0x8eee0101,0xfb840108,0x8eef0101,0xfb840108,0x8ef00101,0xfb840108,0x8ef10101,
	0xfb840108,0x8ef20101,0xfb840108,0x8ef30101,0xfb840108,0x8ef40101,0xfb840108,0x8ef50101,
	0xfb840108,0x8ef60101,0xfb840108,0x8ef70101,0xfb840108,0x8ef80101,0xfb840108,0x8ef90101,
	0xfb840108,0x8efa0101,0xfb840108,0x8efb0101,0xfb840108,0x8efc0101,0xfb840108,0x8efd0101,
	0xfb840108,0x8efe0101,0xfb840108,0x8eff0101,0xfb840108,0x8f000101,0xfb840108,0x8f010101,
	0xfb840108,0x8f020101,0xfb840108,0x8f030101,0xfb840108,0x8f040101,0xfb840108,0x8f050101,
	0xfb840108,0x8f060101,0xfb840108,0x8f070101,0xfb840108,0x8f080101,0xfb840108,0x8f090101,
	0xfb840108,0x8f0a0101,0xfb840108,0x8f0b0101,0xfb840108,0x8f0c0101,0xfb840108,0x8f0d0101,
	0xfb840108,0x8f0e0101,0xfb840108,0x8f0f0101,0xfb840108,0x8f100101,0xfb840108,0x8f110101,
	0xfb840108,0x8f120101,0xfb840108,0x8f130101,0xfb840108,0x8f140101,0xfb840108,0x8f150101,
	0xfb840108,0x8f160101,0xfb840108,0x8f170101,0xfb840108,0x8f180101,0xfb840108,0x8f190101,
	0xfb840108,0x8f1a0101,0xfb840108,0x8f1b0101,0xfb840108,0x8f1c0101,0xfb840108,0x8f1d0101,
	0xfb840108,0x8f1e0101,0xfb840108,0x8f1f0101,0xfb840108,0x8f200101,0xfb840108,0x8f210101,
	0xfb840108,0x8f220101,0xfb840108,0x8f230101,0xfb840108,0x8f240101,0xfb840108,0x8f250101,
	0xfb840108,0x8f260101,0xfb840108,0x8f270101,0xfb840108,0x8f280101,0xfb840108,0x8f290101,
	0xfb840108,0x8f2a0101,0xfb840108,0x8f2b0101,0xfb840108,0x8f2c0101,0xfb840108,0x8f2d0101,
	0xfb840108,0x8f2e0101,0xfb840108,0x8f2f0101,0xfb840108,0x8f300101,0xfb840108,0x8f310101,
	0xfb840108,0x8f320101,0xfb840108,0x8f330101,0xfb840108,0x8f340101,0xfb840108,0x8f350101,
	0xfb840108,0x8f360101,0xfb840108,0x8f370101,0xfb840108,0x8f380101,0xfb840108,0x8f390101,
	0xfb840108,0x8f3a0101,0xfb840108,0x8f3b0101,0xfb840108,0x8f3c0101,0xfb840108,0x8f3d0101,
	0xfb840108,0x8f3e0101,0xfb840108,0x8f3f0101,0xfb840108,0x8f400101,0xfb840108,0x8f410101,
	0xfb840108,0x8f420101,0xfb840108,0x8f430101,0xfb840108,0x8f440101,0xfb840108,0x8f450101,
	0xfb840108,0x8f460101,0xfb840108,0x8f470101,0xfb840108,0x8f480101,0xfb840108,0x8f490101,
	0xfb840108,0x8f4a0101,0xfb840108,0x8f4b0101,0xfb840108,0x8f4c0101,0xfb840108,0x8f4d0101,
	0xfb840108,0x8f4e0101,0xfb840108,0x8f4f0101,0xfb840108,0x8f500101,0xfb840108,0x8f510101,
	0xfb840108,0x8f520101,0xfb840108,0x8f530101,0xfb840108,0x8f540101,0xfb840108,0x8f550101,
	0xfb840108,0x8f560101,0xfb840108,0x8f570101,0xfb840108,0x8f580101,0xfb840108,0x8f590101,
	0xfb840108,0x8f5a0101,0xfb840108,0x8f5b0101,0xfb840108,0x8f5c0101,0xfb840108,0x8f5d0101,
	0xfb840108,0x8f5e0101,0xfb840108,0x8f5f0101,0xfb840108,0x8f600101,0xfb840108,0x8f610101,
	0xfb840108,0x8f620101,0xfb840108,0x8f630101,0xfb840108,0x8f640101,0xfb840108,0x8f650101,
	0xfb840108,0x8f660101,0xfb840108,0x8f670101,0xfb840108,0x8f680101,0xfb840108,0x8f690101,
	0xfb840108,0x8f6a0101,0xfb840108,0x8f6b0101,0xfb840108,0x8f6c0101,0xfb840108,0x8f6d0101,
	0xfb840108,0x8f6e0101,0xfb840108,0x8f6f0101,0xfb840108,0x8f700101,0xfb840108,0x8f710101,
	0xfb840108,0x8f720101,0xfb840108,0x8f730101,0xfb840108,0x8f740101,0xfb840108,0x8f750101,
	0xfb840108,0x8f760101,0xfb840108,0x8f770101,0xfb840108,0x8f780101,0xfb840108,0x8f790101,
	0xfb840108,0x8f7a0101,0xfb840108,0x8f7b0101,0xfb840108,0x8f7c0101,0xfb840108,0x8f7d0101,
	0xfb840108,0x8f7e0101,0xfb840108,0x8f7f0101,0xfb840108,0x8f800101,0xfb840108,0x8f810101,
	0xfb840108,0x8f820101,0xfb840108,0x8f830101,0xfb840108,0x8f840101,0xfb840108,0x8f850101,
	0xfb840108,0x8f860101,0xfb840108,0x8f870101,0xfb840108,0x8f880101,0xfb840108,0x8f890101,
	0xfb840108,0x8f8a0101,0xfb840108,0x8f8b0101,0xfb840108,0x8f8c0101,0xfb840108,0x8f8d0101,
	0xfb840108,0x8f8e0101,0xfb840108,0x8f8f0101,0xfb840108,0x8f900101,0xfb840108,0x8f910101,
	0xfb840108,0x8f920101,0xfb840108,0x8f930101,0xfb840108,0x8f940101,0xfb840108,0x8f950101,
	0xfb840108,0x8f960101,0xfb840108,0x8f970101,0xfb840108,0x8f980101,0xfb840108,0x8f990101,
	0xfb840108,0x8f9a0101,0xfb840108,0x8f9b0101,0xfb840108,0x8f9c0101,0xfb840108,0x8f9d0101,
	0xfb840108,0x8f9e0101,0xfb840108,0x8f9f0101,0xfb840108,0x8fa00101,0xfb840108,0x8fa10101,
	0xfb840108,0x8fa20101,0xfb840108,0x8fa30101,0xfb840108,0x8fa40101,0xfb840108,0x8fa50101,
	0xfb840108,0x8fa60101,0xfb840108,0x8fa70101,0xfb840108,0x8fa80101,0xfb840108,0x8fa90101,
	0xfb840108,0x8faa0101,0xfb840108,0x8fab0101,0xfb840108,0x8fac0101,0xfb840108,0x8fad0101,
	0xfb840108,0x8fae0101,0xfb840108,0x8faf0101,0xfb840108,0x8fb00101,0xfb840108,0x8fb10101,
	0xfb840108,0x8fb20101,0xfb840108,0x8fb30101,0xfb840108,0x8fb40101,0xfb840108,0x8fb50101,
	0xfb840108,0x8fb60101,0xfb840108,0x8fb70101,0xfb840108,0x8fb80101,0xfb840108,0x8fb90101,
	0xfb840108,0x8fba0101,0xfb840108,0x8fbb0101,0xfb840108,0x8fbc0101,0xfb840108,0x8fbd0101,
	0xfb840108,0x8fbe0101,0xfb840108,0x8fbf0101,0xfb840108,0x8fc00101,0xfb840108,0x8fc10101,
	0xfb840108,0x8fc20101,0xfb840108,0x8fc30101,0xfb840108,0x8fc40101,0xfb840108,0x8fc50101,
	0xfb840108,0x8fc60101,0xfb840108,0x8fc70101,0xfb840108,0x8fc80101,0xfb840108,0x8fc90101,
	0xfb840108,0x8fca0101,0xfb840108,0x8fcb0101,0xfb840108,0x8fcc0101,0xfb840108,0x8fcd0101,
	0xfb840108,0x8fce0101,0xfb840108,0x8fcf0101,0xfb840108,0x8fd00101,0xfb840108,0x8fd10101,
	0xfb840108,0x8fd20101,0xfb840108,0x8fd30101,0xfb840108,0x8fd40101,0xfb840108,0x8fd50101,
	0xfb840108,0x8fd60101,0xfb840108,0x8fd70101,0xfb840108,0x8fd80101,0xfb840108,0x8fd90101,
	0xfb840108,0x8fda0101,0xfb840108,0x8fdb0101,0xfb840108,0x8fdc0101,0xfb840108,0x8fdd0101,
	0xfb840108,0x8fde0101,0xfb840108,0x8fdf0101,0xfb840108,0x8fe00101,0xfb840108,0x8fe10101,
	0xfb840108,0x8fe20101,0xfb840108,0x8fe30101,0xfb840108,0x8fe40101,0xfb840108,0x8fe50101,
	0xfb840108,0x8fe60101,0xfb840108,0x8fe70101,0xfb840108,0x8fe80101,0xfb840108,0x8fe90101,
	0xfb840108,0x8fea0101,0xfb840108,0x8feb0101,0xfb840108,0x8fec0101,0xfb840108,0x8fed0101,
	0xfb840108,0x8fee0101,0xfb840108,0x8fef0101,0xfb840108,0x8ff00101,0xfb840108,0x8ff10101,
	0xfb840108,0x8ff20101,0xfb840108,0x8ff30101,0xfb840108,0x8ff40101,0xfb840108,0x8ff50101,
	0xfb840108,0x8ff60101,0xfb840108,0x8ff70101,0xfb840108,0x8ff80101,0xfb840108,0x8ff90101,
	0xfb840108,0x8ffa0101,0xfb840108,0x8ffb0101,0xfb840108,0x8ffc0101,0xfb840108,0x8ffd0101,
	0xfb840108,0x8ffe0101,0xfb840108,0x8fff0101,0xfb840108,0x90000101,0xfb840108,0x90010101,
	0xfb840108,0x90020101,0xfb840108,0x90030101,0xfb840108,0x90040101,0xfb840108,0x90050101,
	0xfb840108,0x90060101,0xfb840108,0x90070101,0xfb840108,0x90080101,0xfb840108,0x90090101,
	0xfb840108,0x900a0101,0xfb840108,0x900b0101,0xfb840108,0x900c0101,0xfb840108,0x900d0101,
	0xfb840108,0x900e0101,0xfb840108,0x900f0101,0xfb840108,0x90100101,0xfb840108,0x90110101,
	0xfb840108,0x90120101,0xfb840108,0x90130101,0xfb840108,0x90140101,0xfb840108,0x90150101,
	0xfb840108,0x90160101,0xfb840108,0x90170101,0xfb840108,0x90180101,0xfb840108,0x90190101,
	0xfb840108,0x901a0101,0xfb840108,0x901b0101,0xfb840108,0x901c0101,0xfb840108,0x901d0101,
	0xfb840108,0x901e0101,0xfb840108,0x901f0101,0xfb840108,0x90200101,0xfb840108,0x90210101,
	0xfb840108,0x90220101,0xfb840108,0x90230101,0xfb840108,0x90240101,0xfb840108,0x90250101,
	0xfb840108,0x90260101,0xfb840108,0x90270101,0xfb840108,0x90280101,0xfb840108,0x90290101,
	0xfb840108,0x902a0101,0xfb840108,0x902b0101,0xfb840108,0x902c0101,0xfb840108,0x902d0101,
	0xfb840108,0x902e0101,0xfb840108,0x902f0101,0xfb840108,0x90300101,0xfb840108,0x90310101,
	0xfb840108,0x90320101,0xfb840108,0x90330101,0xfb840108,0x90340101,0xfb840108,0x90350101,
	0xfb840108,0x90360101,0xfb840108,0x90370101,0xfb840108,0x90380101,0xfb840108,0x90390101,
	0xfb840108,0x903a0101,0xfb840108,0x903b0101,0xfb840108,0x903c0101,0xfb840108,0x903d0101,
	0xfb840108,0x903e0101,0xfb840108,0x903f0101,0xfb840108,0x90400101,0xfb840108,0x90410101,
	0xfb840108,0x90420101,0xfb840108,0x90430101,0xfb840108,0x90440101,0xfb840108,0x90450101,
	0xfb840108,0x90460101,0xfb840108,0x90470101,0xfb840108,0x90480101,0xfb840108,0x90490101,
	0xfb840108,0x904a0101,0xfb840108,0x904b0101,0xfb840108,0x904c0101,0xfb840108,0x904d0101,
	0xfb840108,0x904e0101,0xfb840108,0x904f0101,0xfb840108,0x90500101,0xfb840108,0x90510101,
	0xfb840108,0x90520101,0xfb840108,0x90530101,0xfb840108,0x90540101,0xfb840108,0x90550101,
	0xfb840108,0x90560101,0xfb840108,0x90570101,0xfb840108,0x90580101,0xfb840108,0x90590101,
	0xfb840108,0x905a0101,0xfb840108,0x905b0101,0xfb840108,0x905c0101,0xfb840108,0x905d0101,
	0xfb840108,0x905e0101,0xfb840108,0x905f0101,0xfb840108,0x90600101,0xfb840108,0x90610101,
	0xfb840108,0x90620101,0xfb840108,0x90630101,0xfb840108,0x90640101,0xfb840108,0x90650101,
	0xfb840108,0x90660101,0xfb840108,0x90670101,0xfb840108,0x90680101,0xfb840108,0x90690101,
	0xfb840108,0x906a0101,0xfb840108,0x906b0101,0xfb840108,0x906c0101,0xfb840108,0x906d0101,
	0xfb840108,0x906e0101,0xfb840108,0x906f0101,0xfb840108,0x90700101,0xfb840108,0x90710101,
	0xfb840108,0x90720101,0xfb840108,0x90730101,0xfb840108,0x90740101,0xfb840108,0x90750101,
	0xfb840108,0x90760101,0xfb840108,0x90770101,0xfb840108,0x90780101,0xfb840108,0x90790101,
	0xfb840108,0x907a0101,0xfb840108,0x907b0101,0xfb840108,0x907c0101,0xfb840108,0x907d0101,
	0xfb840108,0x907e0101,0xfb840108,0x907f0101,0xfb840108,0x90800101,0xfb840108,0x90810101,
	0xfb840108,0x90820101,0xfb840108,0x90830101,0xfb840108,0x90840101,0xfb840108,0x90850101,
	0xfb840108,0x90860101,0xfb840108,0x90870101,0xfb840108,0x90880101,0xfb840108,0x90890101,
	0xfb840108,0x908a0101,0xfb840108,0x908b0101,0xfb840108,0x908c0101,0xfb840108,0x908d0101,
	0xfb840108,0x908e0101,0xfb840108,0x908f0101,0xfb840108,0x90900101,0xfb840108,0x90910101,
	0xfb840108,0x90920101,0xfb840108,0x90930101,0xfb840108,0x90940101,0xfb840108,0x90950101,
	0xfb840108,0x90960101,0xfb840108,0x90970101,0xfb840108,0x90980101,0xfb840108,0x90990101,
	0xfb840108,0x909a0101,0xfb840108,0x909b0101,0xfb840108,0x909c0101,0xfb840108,0x909d0101,
	0xfb840108,0x909e0101,0xfb840108,0x909f0101,0xfb840108,0x90a00101,0xfb840108,0x90a10101,
	0xfb840108,0x90a20101,0xfb840108,0x90a30101,0xfb840108,0x90a40101,0xfb840108,0x90a50101,
	0xfb840108,0x90a60101,0xfb840108,0x90a70101,0xfb840108,0x90a80101,0xfb840108,0x90a90101,
	0xfb840108,0x90aa0101,0xfb840108,0x90ab0101,0xfb840108,0x90ac0101,0xfb840108,0x90ad0101,
	0xfb840108,0x90ae0101,0xfb840108,0x90af0101,0xfb840108,0x90b00101,0xfb840108,0x90b10101,
	0xfb840108,0x90b20101,0xfb840108,0x90b30101,0xfb840108,0x90b40101,0xfb840108,0x90b50101,
	0xfb840108,0x90b60101,0xfb840108,0x90b70101,0xfb840108,0x90b80101,0xfb840108,0x90b90101,
	0xfb840108,0x90ba0101,0xfb840108,0x90bb0101,0xfb840108,0x90bc0101,0xfb840108,0x90bd0101,
	0xfb840108,0x90be0101,0xfb840108,0x90bf0101,0xfb840108,0x90c00101,0xfb840108,0x90c10101,
	0xfb840108,0x90c20101,0xfb840108,0x90c30101,0xfb840108,0x90c40101,0xfb840108,0x90c50101,
	0xfb840108,0x90c60101,0xfb840108,0x90c70101,0xfb840108,0x90c80101,0xfb840108,0x90c90101,
	0xfb840108,0x90ca0101,0xfb840108,0x90cb0101,0xfb840108,0x90cc0101,0xfb840108,0x90cd0101,
	0xfb840108,0x90ce0101,0xfb840108,0x90cf0101,0xfb840108,0x90d00101,0xfb840108,0x90d10101,
	0xfb840108,0x90d20101,0xfb840108,0x90d30101,0xfb840108,0x90d40101,0xfb840108,0x90d50101,
	0xfb840108,0x90d60101,0xfb840108,0x90d70101,0xfb840108,0x90d80101,0xfb840108,0x90d90101,
	0xfb840108,0x90da0101,0xfb840108,0x90db0101,0xfb840108,0x90dc0101,0xfb840108,0x90dd0101,
	0xfb840108,0x90de0101,0xfb840108,0x90df0101,0xfb840108,0x90e00101,0xfb840108,0x90e10101,
	0xfb840108,0x90e20101,0xfb840108,0x90e30101,0xfb840108,0x90e40101,0xfb840108,0x90e50101,
	0xfb840108,0x90e60101,0xfb840108,0x90e70101,0xfb840108,0x90e80101,0xfb840108,0x90e90101,
	0xfb840108,0x90ea0101,0xfb840108,0x90eb0101,0xfb840108,0x90ec0101,0xfb840108,0x90ed0101,
	0xfb840108,0x90ee0101,0xfb840108,0x90ef0101,0xfb840108,0x90f00101,0xfb840108,0x90f10101,
	0xfb840108,0x90f20101,0xfb840108,0x90f30101,0xfb840108,0x90f40101,0xfb840108,0x90f50101,
	0xfb840108,0x90f60101,0xfb840108,0x90f70101,0xfb840108,0x90f80101,0xfb840108,0x90f90101,
	0xfb840108,0x90fa0101,0xfb840108,0x90fb0101,0xfb840108,0x90fc0101,0xfb840108,0x90fd0101,
	0xfb840108,0x90fe0101,0xfb840108,0x90ff0101,0xfb840108,0x91000101,0xfb840108,0x91010101,
	0xfb840108,0x91020101,0xfb840108,0x91030101,0xfb840108,0x91040101,0xfb840108,0x91050101,
	0xfb840108,0x91060101,0xfb840108,0x91070101,0xfb840108,0x91080101,0xfb840108,0x91090101,
	0xfb840108,0x910a0101,0xfb840108,0x910b0101,0xfb840108,0x910c0101,0xfb840108,0x910d0101,
	0xfb840108,0x910e0101,0xfb840108,0x910f0101,0xfb840108,0x91100101,0xfb840108,0x91110101,
	0xfb840108,0x91120101,0xfb840108,0x91130101,0xfb840108,0x91140101,0xfb840108,0x91150101,
	0xfb840108,0x91160101,0xfb840108,0x91170101,0xfb840108,0x91180101,0xfb840108,0x91190101,
	0xfb840108,0x911a0101,0xfb840108,0x911b0101,0xfb840108,0x911c0101,0xfb840108,0x911d0101,
	0xfb840108,0x911e0101,0xfb840108,0x911f0101,0xfb840108,0x91200101,0xfb840108,0x91210101,
	0xfb840108,0x91220101,0xfb840108,0x91230101,0xfb840108,0x91240101,0xfb840108,0x91250101,
	0xfb840108,0x91260101,0xfb840108,0x91270101,0xfb840108,0x91280101,0xfb840108,0x91290101,
	0xfb840108,0x912a0101,0xfb840108,0x912b0101,0xfb840108,0x912c0101,0xfb840108,0x912d0101,
	0xfb840108,0x912e0101,0xfb840108,0x912f0101,0xfb840108,0x91300101,0xfb840108,0x91310101,
	0xfb840108,0x91320101,0xfb840108,0x91330101,0xfb840108,0x91340101,0xfb840108,0x91350101,
	0xfb840108,0x91360101,0xfb840108,0x91370101,0xfb840108,0x91380101,0xfb840108,0x91390101,
	0xfb840108,0x913a0101,0xfb840108,0x913b0101,0xfb840108,0x913c0101,0xfb840108,0x913d0101,
	0xfb840108,0x913e0101,0xfb840108,0x913f0101,0xfb840108,0x91400101,0xfb840108,0x91410101,
	0xfb840108,0x91420101,0xfb840108,0x91430101,0xfb840108,0x91440101,0xfb840108,0x91450101,
	0xfb840108,0x91460101,0xfb840108,0x91470101,0xfb840108,0x91480101,0xfb840108,0x91490101,
	0xfb840108,0x914a0101,0xfb840108,0x914b0101,0xfb840108,0x914c0101,0xfb840108,0x914d0101,
	0xfb840108,0x914e0101,0xfb840108,0x914f0101,0xfb840108,0x91500101,0xfb840108,0x91510101,
	0xfb840108,0x91520101,0xfb840108,0x91530101,0xfb840108,0x91540101,0xfb840108,0x91550101,
	0xfb840108,0x91560101,0xfb840108,0x91570101,0xfb840108,0x91580101,0xfb840108,0x91590101,
	0xfb840108,0x915a0101,0xfb840108,0x915b0101,0xfb840108,0x915c0101,0xfb840108,0x915d0101,
	0xfb840108,0x915e0101,0xfb840108,0x915f0101,0xfb840108,0x91600101,0xfb840108,0x91610101,
	0xfb840108,0x91620101,0xfb840108,0x91630101,0xfb840108,0x91640101,0xfb840108,0x91650101,
	0xfb840108,0x91660101,0xfb840108,0x91670101,0xfb840108,0x91680101,0xfb840108,0x91690101,
	0xfb840108,0x916a0101,0xfb840108,0x916b0101,0xfb840108,0x916c0101,0xfb840108,0x916d0101,
	0xfb840108,0x916e0101,0xfb840108,0x916f0101,0xfb840108,0x91700101,0xfb840108,0x91710101,
	0xfb840108,0x91720101,0xfb840108,0x91730101,0xfb840108,0x91740101,0xfb840108,0x91750101,
	0xfb840108,0x91760101,0xfb840108,0x91770101,0xfb840108,0x91780101,0xfb840108,0x91790101,
	0xfb840108,0x917a0101,0xfb840108,0x917b0101,0xfb840108,0x917c0101,0xfb840108,0x917d0101,
	0xfb840108,0x917e0101,0xfb840108,0x917f0101,0xfb840108,0x91800101,0xfb840108,0x91810101,
	0xfb840108,0x91820101,0xfb840108,0x91830101,0xfb840108,0x91840101,0xfb840108,0x91850101,
	0xfb840108,0x91860101,0xfb840108,0x91870101,0xfb840108,0x91880101,0xfb840108,0x91890101,
	0xfb840108,0x918a0101,0xfb840108,0x918b0101,0xfb840108,0x918c0101,0xfb840108,0x918d0101,
	0xfb840108,0x918e0101,0xfb840108,0x918f0101,0xfb840108,0x91900101,0xfb840108,0x91910101,
	0xfb840108,0x91920101,0xfb840108,0x91930101,0xfb840108,0x91940101,0xfb840108,0x91950101,
	0xfb840108,0x91960101,0xfb840108,0x91970101,0xfb840108,0x91980101,0xfb840108,0x91990101,
	0xfb840108,0x919a0101,0xfb840108,0x919b0101,0xfb840108,0x919c0101,0xfb840108,0x919d0101,
	0xfb840108,0x919e0101,0xfb840108,0x919f0101,0xfb840108,0x91a00101,0xfb840108,0x91a10101,
	0xfb840108,0x91a20101,0xfb840108,0x91a30101,0xfb840108,0x91a40101,0xfb840108,0x91a50101,
	0xfb840108,0x91a60101,0xfb840108,0x91a70101,0xfb840108,0x91a80101,0xfb840108,0x91a90101,
	0xfb840108,0x91aa0101,0xfb840108,0x91ab0101,0xfb840108,0x91ac0101,0xfb840108,0x91ad0101,
	0xfb840108,0x91ae0101,0xfb840108,0x91af0101,0xfb840108,0x91b00101,0xfb840108,0x91b10101,
	0xfb840108,0x91b20101,0xfb840108,0x91b30101,0xfb840108,0x91b40101,0xfb840108,0x91b50101,
	0xfb840108,0x91b60101,0xfb840108,0x91b70101,0xfb840108,0x91b80101,0xfb840108,0x91b90101,
	0xfb840108,0x91ba0101,0xfb840108,0x91bb0101,0xfb840108,0x91bc0101,0xfb840108,0x91bd0101,
	0xfb840108,0x91be0101,0xfb840108,0x91bf0101,0xfb840108,0x91c00101,0xfb840108,0x91c10101,
	0xfb840108,0x91c20101,0xfb840108,0x91c30101,0xfb840108,0x91c40101,0xfb840108,0x91c50101,
	0xfb840108,0x91c60101,0xfb840108,0x91c70101,0xfb840108,0x91c80101,0xfb840108,0x91c90101,
	0xfb840108,0x91ca0101,0xfb840108,0x91cb0101,0xfb840108,0x91cc0101,0xfb840108,0x91cd0101,
	0xfb840108,0x91ce0101,0xfb840108,0x91cf0101,0xfb840108,0x91d00101,0xfb840108,0x91d10101,
	0xfb840108,0x91d20101,0xfb840108,0x91d30101,0xfb840108,0x91d40101,0xfb840108,0x91d50101,
	0xfb840108,0x91d60101,0xfb840108,0x91d70101,0xfb840108,0x91d80101,0xfb840108,0x91d90101,
	0xfb840108,0x91da0101,0xfb840108,0x91db0101,0xfb840108,0x91dc0101,0xfb840108,0x91dd0101,
	0xfb840108,0x91de0101,0xfb840108,0x91df0101,0xfb840108,0x91e00101,0xfb840108,0x91e10101,
	0xfb840108,0x91e20101,0xfb840108,0x91e30101,0xfb840108,0x91e40101,0xfb840108,0x91e50101,
	0xfb840108,0x91e60101,0xfb840108,0x91e70101,0xfb840108,0x91e80101,0xfb840108,0x91e90101,
	0xfb840108,0x91ea0101,0xfb840108,0x91eb0101,0xfb840108,0x91ec0101,0xfb840108,0x91ed0101,
	0xfb840108,0x91ee0101,0xfb840108,0x91ef0101,0xfb840108,0x91f00101,0xfb840108,0x91f10101,
	0xfb840108,0x91f20101,0xfb840108,0x91f30101,0xfb840108,0x91f40101,0xfb840108,0x91f50101,
	0xfb840108,0x91f60101,0xfb840108,0x91f70101,0xfb840108,0x91f80101,0xfb840108,0x91f90101,
	0xfb840108,0x91fa0101,0xfb840108,0x91fb0101,0xfb840108,0x91fc0101,0xfb840108,0x91fd0101,
	0xfb840108,0x91fe0101,0xfb840108,0x91ff0101,0xfb840108,0x92000101,0xfb840108,0x92010101,
	0xfb840108,0x92020101,0xfb840108,0x92030101,0xfb840108,0x92040101,0xfb840108,0x92050101,
	0xfb840108,0x92060101,0xfb840108,0x92070101,0xfb840108,0x92080101,0xfb840108,0x92090101,
	0xfb840108,0x920a0101,0xfb840108,0x920b0101,0xfb840108,0x920c0101,0xfb840108,0x920d0101,
	0xfb840108,0x920e0101,0xfb840108,0x920f0101,0xfb840108,0x92100101,0xfb840108,0x92110101,
	0xfb840108,0x92120101,0xfb840108,0x92130101,0xfb840108,0x92140101,0xfb840108,0x92150101,
	0xfb840108,0x92160101,0xfb840108,0x92170101,0xfb840108,0x92180101,0xfb840108,0x92190101,
	0xfb840108,0x921a0101,0xfb840108,0x921b0101,0xfb840108,0x921c0101,0xfb840108,0x921d0101,
	0xfb840108,0x921e0101,0xfb840108,0x921f0101,0xfb840108,0x92200101,0xfb840108,0x92210101,
	0xfb840108,0x92220101,0xfb840108,0x92230101,0xfb840108,0x92240101,0xfb840108,0x92250101,
	0xfb840108,0x92260101,0xfb840108,0x92270101,0xfb840108,0x92280101,0xfb840108,0x92290101,
	0xfb840108,0x922a0101,0xfb840108,0x922b0101,0xfb840108,0x922c0101,0xfb840108,0x922d0101,
	0xfb840108,0x922e0101,0xfb840108,0x922f0101,0xfb840108,0x92300101,0xfb840108,0x92310101,
	0xfb840108,0x92320101,0xfb840108,0x92330101,0xfb840108,0x92340101,0xfb840108,0x92350101,
	0xfb840108,0x92360101,0xfb840108,0x92370101,0xfb840108,0x92380101,0xfb840108,0x92390101,
	0xfb840108,0x923a0101,0xfb840108,0x923b0101,0xfb840108,0x923c0101,0xfb840108,0x923d0101,
	0xfb840108,0x923e0101,0xfb840108,0x923f0101,0xfb840108,0x92400101,0xfb840108,0x92410101,
	0xfb840108,0x92420101,0xfb840108,0x92430101,0xfb840108,0x92440101,0xfb840108,0x92450101,
	0xfb840108,0x92460101,0xfb840108,0x92470101,0xfb840108,0x92480101,0xfb840108,0x92490101,
	0xfb840108,0x924a0101,0xfb840108,0x924b0101,0xfb840108,0x924c0101,0xfb840108,0x924d0101,
	0xfb840108,0x924e0101,0xfb840108,0x924f0101,0xfb840108,0x92500101,0xfb840108,0x92510101,
	0xfb840108,0x92520101,0xfb840108,0x92530101,0xfb840108,0x92540101,0xfb840108,0x92550101,
	0xfb840108,0x92560101,0xfb840108,0x92570101,0xfb840108,0x92580101,0xfb840108,0x92590101,
	0xfb840108,0x925a0101,0xfb840108,0x925b0101,0xfb840108,0x925c0101,0xfb840108,0x925d0101,
	0xfb840108,0x925e0101,0xfb840108,0x925f0101,0xfb840108,0x92600101,0xfb840108,0x92610101,
	0xfb840108,0x92620101,0xfb840108,0x92630101,0xfb840108,0x92640101,0xfb840108,0x92650101,
	0xfb840108,0x92660101,0xfb840108,0x92670101,0xfb840108,0x92680101,0xfb840108,0x92690101,
	0xfb840108,0x926a0101,0xfb840108,0x926b0101,0xfb840108,0x926c0101,0xfb840108,0x926d0101,
	0xfb840108,0x926e0101,0xfb840108,0x926f0101,0xfb840108,0x92700101,0xfb840108,0x92710101,
	0xfb840108,0x92720101,0xfb840108,0x92730101,0xfb840108,0x92740101,0xfb840108,0x92750101,
	0xfb840108,0x92760101,0xfb840108,0x92770101,0xfb840108,0x92780101,0xfb840108,0x92790101,
	0xfb840108,0x927a0101,0xfb840108,0x927b0101,0xfb840108,0x927c0101,0xfb840108,0x927d0101,
	0xfb840108,0x927e0101,0xfb840108,0x927f0101,0xfb840108,0x92800101,0xfb840108,0x92810101,
	0xfb840108,0x92820101,0xfb840108,0x92830101,0xfb840108,0x92840101,0xfb840108,0x92850101,
	0xfb840108,0x92860101,0xfb840108,0x92870101,0xfb840108,0x92880101,0xfb840108,0x92890101,
	0xfb840108,0x928a0101,0xfb840108,0x928b0101,0xfb840108,0x928c0101,0xfb840108,0x928d0101,
	0xfb840108,0x928e0101,0xfb840108,0x928f0101,0xfb840108,0x92900101,0xfb840108,0x92910101,
	0xfb840108,0x92920101,0xfb840108,0x92930101,0xfb840108,0x92940101,0xfb840108,0x92950101,
	0xfb840108,0x92960101,0xfb840108,0x92970101,0xfb840108,0x92980101,0xfb840108,0x92990101,
	0xfb840108,0x929a0101,0xfb840108,0x929b0101,0xfb840108,0x929c0101,0xfb840108,0x929d0101,
	0xfb840108,0x929e0101,0xfb840108,0x929f0101,0xfb840108,0x92a00101,0xfb840108,0x92a10101,
	0xfb840108,0x92a20101,0xfb840108,0x92a30101,0xfb840108,0x92a40101,0xfb840108,0x92a50101,
	0xfb840108,0x92a60101,0xfb840108,0x92a70101,0xfb840108,0x92a80101,0xfb840108,0x92a90101,
	0xfb840108,0x92aa0101,0xfb840108,0x92ab0101,0xfb840108,0x92ac0101,0xfb840108,0x92ad0101,
	0xfb840108,0x92ae0101,0xfb840108,0x92af0101,0xfb840108,0x92b00101,0xfb840108,0x92b10101,
	0xfb840108,0x92b20101,0xfb840108,0x92b30101,0xfb840108,0x92b40101,0xfb840108,0x92b50101,
	0xfb840108,0x92b60101,0xfb840108,0x92b70101,0xfb840108,0x92b80101,0xfb840108,0x92b90101,
	0xfb840108,0x92ba0101,0xfb840108,0x92bb0101,0xfb840108,0x92bc0101,0xfb840108,0x92bd0101,
	0xfb840108,0x92be0101,0xfb840108,0x92bf0101,0xfb840108,0x92c00101,0xfb840108,0x92c10101,
	0xfb840108,0x92c20101,0xfb840108,0x92c30101,0xfb840108,0x92c40101,0xfb840108,0x92c50101,
	0xfb840108,0x92c60101,0xfb840108,0x92c70101,0xfb840108,0x92c80101,0xfb840108,0x92c90101,
	0xfb840108,0x92ca0101,0xfb840108,0x92cb0101,0xfb840108,0x92cc0101,0xfb840108,0x92cd0101,
	0xfb840108,0x92ce0101,0xfb840108,0x92cf0101,0xfb840108,0x92d00101,0xfb840108,0x92d10101,
	0xfb840108,0x92d20101,0xfb840108,0x92d30101,0xfb840108,0x92d40101,0xfb840108,0x92d50101,
	0xfb840108,0x92d60101,0xfb840108,0x92d70101,0xfb840108,0x92d80101,0xfb840108,0x92d90101,
	0xfb840108,0x92da0101,0xfb840108,0x92db0101,0xfb840108,0x92dc0101,0xfb840108,0x92dd0101,
	0xfb840108,0x92de0101,0xfb840108,0x92df0101,0xfb840108,0x92e00101,0xfb840108,0x92e10101,
	0xfb840108,0x92e20101,0xfb840108,0x92e30101,0xfb840108,0x92e40101,0xfb840108,0x92e50101,
	0xfb840108,0x92e60101,0xfb840108,0x92e70101,0xfb840108,0x92e80101,0xfb840108,0x92e90101,
	0xfb840108,0x92ea0101,0xfb840108,0x92eb0101,0xfb840108,0x92ec0101,0xfb840108,0x92ed0101,
	0xfb840108,0x92ee0101,0xfb840108,0x92ef0101,0xfb840108,0x92f00101,0xfb840108,0x92f10101,
	0xfb840108,0x92f20101,0xfb840108,0x92f30101,0xfb840108,0x92f40101,0xfb840108,0x92f50101,
	0xfb840108,0x92f60101,0xfb840108,0x92f70101,0xfb840108,0x92f80101,0xfb840108,0x92f90101,
	0xfb840108,0x92fa0101,0xfb840108,0x92fb0101,0xfb840108,0x92fc0101,0xfb840108,0x92fd0101,
	0xfb840108,0x92fe0101,0xfb840108,0x92ff0101,0xfb840108,0x93000101,0xfb840108,0x93010101,
	0xfb840108,0x93020101,0xfb840108,0x93030101,0xfb840108,0x93040101,0xfb840108,0x93050101,
	0xfb840108,0x93060101,0xfb840108,0x93070101,0xfb840108,0x93080101,0xfb840108,0x93090101,
	0xfb840108,0x930a0101,0xfb840108,0x930b0101,0xfb840108,0x930c0101,0xfb840108,0x930d0101,
	0xfb840108,0x930e0101,0xfb840108,0x930f0101,0xfb840108,0x93100101,0xfb840108,0x93110101,
	0xfb840108,0x93120101,0xfb840108,0x93130101,0xfb840108,0x93140101,0xfb840108,0x93150101,
	0xfb840108,0x93160101,0xfb840108,0x93170101,0xfb840108,0x93180101,0xfb840108,0x93190101,
	0xfb840108,0x931a0101,0xfb840108,0x931b0101,0xfb840108,0x931c0101,0xfb840108,0x931d0101,
	0xfb840108,0x931e0101,0xfb840108,0x931f0101,0xfb840108,0x93200101,0xfb840108,0x93210101,
	0xfb840108,0x93220101,0xfb840108,0x93230101,0xfb840108,0x93240101,0xfb840108,0x93250101,
	0xfb840108,0x93260101,0xfb840108,0x93270101,0xfb840108,0x93280101,0xfb840108,0x93290101,
	0xfb840108,0x932a0101,0xfb840108,0x932b0101,0xfb840108,0x932c0101,0xfb840108,0x932d0101,
	0xfb840108,0x932e0101,0xfb840108,0x932f0101,0xfb840108,0x93300101,0xfb840108,0x93310101,
	0xfb840108,0x93320101,0xfb840108,0x93330101,0xfb840108,0x93340101,0xfb840108,0x93350101,
	0xfb840108,0x93360101,0xfb840108,0x93370101,0xfb840108,0x93380101,0xfb840108,0x93390101,
	0xfb840108,0x933a0101,0xfb840108,0x933b0101,0xfb840108,0x933c0101,0xfb840108,0x933d0101,
	0xfb840108,0x933e0101,0xfb840108,0x933f0101,0xfb840108,0x93400101,0xfb840108,0x93410101,
	0xfb840108,0x93420101,0xfb840108,0x93430101,0xfb840108,0x93440101,0xfb840108,0x93450101,
	0xfb840108,0x93460101,0xfb840108,0x93470101,0xfb840108,0x93480101,0xfb840108,0x93490101,
	0xfb840108,0x934a0101,0xfb840108,0x934b0101,0xfb840108,0x934c0101,0xfb840108,0x934d0101,
	0xfb840108,0x934e0101,0xfb840108,0x934f0101,0xfb840108,0x93500101,0xfb840108,0x93510101,
	0xfb840108,0x93520101,0xfb840108,0x93530101,0xfb840108,0x93540101,0xfb840108,0x93550101,
	0xfb840108,0x93560101,0xfb840108,0x93570101,0xfb840108,0x93580101,0xfb840108,0x93590101,
	0xfb840108,0x935a0101,0xfb840108,0x935b0101,0xfb840108,0x935c0101,0xfb840108,0x935d0101,
	0xfb840108,0x935e0101,0xfb840108,0x935f0101,0xfb840108,0x93600101,0xfb840108,0x93610101,
	0xfb840108,0x93620101,0xfb840108,0x93630101,0xfb840108,0x93640101,0xfb840108,0x93650101,
	0xfb840108,0x93660101,0xfb840108,0x93670101,0xfb840108,0x93680101,0xfb840108,0x93690101,
	0xfb840108,0x936a0101,0xfb840108,0x936b0101,0xfb840108,0x936c0101,0xfb840108,0x936d0101,
	0xfb840108,0x936e0101,0xfb840108,0x936f0101,0xfb840108,0x93700101,0xfb840108,0x93710101,
	0xfb840108,0x93720101,0xfb840108,0x93730101,0xfb840108,0x93740101,0xfb840108,0x93750101,
	0xfb840108,0x93760101,0xfb840108,0x93770101,0xfb840108,0x93780101,0xfb840108,0x93790101,
	0xfb840108,0x937a0101,0xfb840108,0x937b0101,0xfb840108,0x937c0101,0xfb840108,0x937d0101,
	0xfb840108,0x937e0101,0xfb840108,0x937f0101,0xfb840108,0x93800101,0xfb840108,0x93810101,
	0xfb840108,0x93820101,0xfb840108,0x93830101,0xfb840108,0x93840101,0xfb840108,0x93850101,
	0xfb840108,0x93860101,0xfb840108,0x93870101,0xfb840108,0x93880101,0xfb840108,0x93890101,
	0xfb840108,0x938a0101,0xfb840108,0x938b0101,0xfb840108,0x938c0101,0xfb840108,0x938d0101,
	0xfb840108,0x938e0101,0xfb840108,0x938f0101,0xfb840108,0x93900101,0xfb840108,0x93910101,
	0xfb840108,0x93920101,0xfb840108,0x93930101,0xfb840108,0x93940101,0xfb840108,0x93950101,
	0xfb840108,0x93960101,0xfb840108,0x93970101,0xfb840108,0x93980101,0xfb840108,0x93990101,
	0xfb840108,0x939a0101,0xfb840108,0x939b0101,0xfb840108,0x939c0101,0xfb840108,0x939d0101,
	0xfb840108,0x939e0101,0xfb840108,0x939f0101,0xfb840108,0x93a00101,0xfb840108,0x93a10101,
	0xfb840108,0x93a20101,0xfb840108,0x93a30101,0xfb840108,0x93a40101,0xfb840108,0x93a50101,
	0xfb840108,0x93a60101,0xfb840108,0x93a70101,0xfb840108,0x93a80101,0xfb840108,0x93a90101,
	0xfb840108,0x93aa0101,0xfb840108,0x93ab0101,0xfb840108,0x93ac0101,0xfb840108,0x93ad0101,
	0xfb840108,0x93ae0101,0xfb840108,0x93af0101,0xfb840108,0x93b00101,0xfb840108,0x93b10101,
	0xfb840108,0x93b20101,0xfb840108,0x93b30101,0xfb840108,0x93b40101,0xfb840108,0x93b50101,
	0xfb840108,0x93b60101,0xfb840108,0x93b70101,0xfb840108,0x93b80101,0xfb840108,0x93b90101,
	0xfb840108,0x93ba0101,0xfb840108,0x93bb0101,0xfb840108,0x93bc0101,0xfb840108,0x93bd0101,
	0xfb840108,0x93be0101,0xfb840108,0x93bf0101,0xfb840108,0x93c00101,0xfb840108,0x93c10101,
	0xfb840108,0x93c20101,0xfb840108,0x93c30101,0xfb840108,0x93c40101,0xfb840108,0x93c50101,
	0xfb840108,0x93c60101,0xfb840108,0x93c70101,0xfb840108,0x93c80101,0xfb840108,0x93c90101,
	0xfb840108,0x93ca0101,0xfb840108,0x93cb0101,0xfb840108,0x93cc0101,0xfb840108,0x93cd0101,
	0xfb840108,0x93ce0101,0xfb840108,0x93cf0101,0xfb840108,0x93d00101,0xfb840108,0x93d10101,
	0xfb840108,0x93d20101,0xfb840108,0x93d30101,0xfb840108,0x93d40101,0xfb840108,0x93d50101,
	0xfb840108,0x93d60101,0xfb840108,0x93d70101,0xfb840108,0x93d80101,0xfb840108,0x93d90101,
	0xfb840108,0x93da0101,0xfb840108,0x93db0101,0xfb840108,0x93dc0101,0xfb840108,0x93dd0101,
	0xfb840108,0x93de0101,0xfb840108,0x93df0101,0xfb840108,0x93e00101,0xfb840108,0x93e10101,
	0xfb840108,0x93e20101,0xfb840108,0x93e30101,0xfb840108,0x93e40101,0xfb840108,0x93e50101,
	0xfb840108,0x93e60101,0xfb840108,0x93e70101,0xfb840108,0x93e80101,0xfb840108,0x93e90101,
	0xfb840108,0x93ea0101,0xfb840108,0x93eb0101,0xfb840108,0x93ec0101,0xfb840108,0x93ed0101,
	0xfb840108,0x93ee0101,0xfb840108,0x93ef0101,0xfb840108,0x93f00101,0xfb840108,0x93f10101,
	0xfb840108,0x93f20101,0xfb840108,0x93f30101,0xfb840108,0x93f40101,0xfb840108,0x93f50101,
	0xfb840108,0x93f60101,0xfb840108,0x93f70101,0xfb840108,0x93f80101,0xfb840108,0x93f90101,
	0xfb840108,0x93fa0101,0xfb840108,0x93fb0101,0xfb840108,0x93fc0101,0xfb840108,0x93fd0101,
	0xfb840108,0x93fe0101,0xfb840108,0x93ff0101,0xfb840108,0x94000101,0xfb840108,0x94010101,
	0xfb840108,0x94020101,0xfb840108,0x94030101,0xfb840108,0x94040101,0xfb840108,0x94050101,
	0xfb840108,0x94060101,0xfb840108,0x94070101,0xfb840108,0x94080101,0xfb840108,0x94090101,
	0xfb840108,0x940a0101,0xfb840108,0x940b0101,0xfb840108,0x940c0101,0xfb840108,0x940d0101,
	0xfb840108,0x940e0101,0xfb840108,0x940f0101,0xfb840108,0x94100101,0xfb840108,0x94110101,
	0xfb840108,0x94120101,0xfb840108,0x94130101,0xfb840108,0x94140101,0xfb840108,0x94150101,
	0xfb840108,0x94160101,0xfb840108,0x94170101,0xfb840108,0x94180101,0xfb840108,0x94190101,
	0xfb840108,0x941a0101,0xfb840108,0x941b0101,0xfb840108,0x941c0101,0xfb840108,0x941d0101,
	0xfb840108,0x941e0101,0xfb840108,0x941f0101,0xfb840108,0x94200101,0xfb840108,0x94210101,
	0xfb840108,0x94220101,0xfb840108,0x94230101,0xfb840108,0x94240101,0xfb840108,0x94250101,
	0xfb840108,0x94260101,0xfb840108,0x94270101,0xfb840108,0x94280101,0xfb840108,0x94290101,
	0xfb840108,0x942a0101,0xfb840108,0x942b0101,0xfb840108,0x942c0101,0xfb840108,0x942d0101,
	0xfb840108,0x942e0101,0xfb840108,0x942f0101,0xfb840108,0x94300101,0xfb840108,0x94310101,
	0xfb840108,0x94320101,0xfb840108,0x94330101,0xfb840108,0x94340101,0xfb840108,0x94350101,
	0xfb840108,0x94360101,0xfb840108,0x94370101,0xfb840108,0x94380101,0xfb840108,0x94390101,
	0xfb840108,0x943a0101,0xfb840108,0x943b0101,0xfb840108,0x943c0101,0xfb840108,0x943d0101,
	0xfb840108,0x943e0101,0xfb840108,0x943f0101,0xfb840108,0x94400101,0xfb840108,0x94410101,
	0xfb840108,0x94420101,0xfb840108,0x94430101,0xfb840108,0x94440101,0xfb840108,0x94450101,
	0xfb840108,0x94460101,0xfb840108,0x94470101,0xfb840108,0x94480101,0xfb840108,0x94490101,
	0xfb840108,0x944a0101,0xfb840108,0x944b0101,0xfb840108,0x944c0101,0xfb840108,0x944d0101,
	0xfb840108,0x944e0101,0xfb840108,0x944f0101,0xfb840108,0x94500101,0xfb840108,0x94510101,
	0xfb840108,0x94520101,0xfb840108,0x94530101,0xfb840108,0x94540101,0xfb840108,0x94550101,
	0xfb840108,0x94560101,0xfb840108,0x94570101,0xfb840108,0x94580101,0xfb840108,0x94590101,
	0xfb840108,0x945a0101,0xfb840108,0x945b0101,0xfb840108,0x945c0101,0xfb840108,0x945d0101,
	0xfb840108,0x945e0101,0xfb840108,0x945f0101,0xfb840108,0x94600101,0xfb840108,0x94610101,
	0xfb840108,0x94620101,0xfb840108,0x94630101,0xfb840108,0x94640101,0xfb840108,0x94650101,
	0xfb840108,0x94660101,0xfb840108,0x94670101,0xfb840108,0x94680101,0xfb840108,0x94690101,
	0xfb840108,0x946a0101,0xfb840108,0x946b0101,0xfb840108,0x946c0101,0xfb840108,0x946d0101,
	0xfb840108,0x946e0101,0xfb840108,0x946f0101,0xfb840108,0x94700101,0xfb840108,0x94710101,
	0xfb840108,0x94720101,0xfb840108,0x94730101,0xfb840108,0x94740101,0xfb840108,0x94750101,
	0xfb840108,0x94760101,0xfb840108,0x94770101,0xfb840108,0x94780101,0xfb840108,0x94790101,
	0xfb840108,0x947a0101,0xfb840108,0x947b0101,0xfb840108,0x947c0101,0xfb840108,0x947d0101,
	0xfb840108,0x947e0101,0xfb840108,0x947f0101,0xfb840108,0x94800101,0xfb840108,0x94810101,
	0xfb840108,0x94820101,0xfb840108,0x94830101,0xfb840108,0x94840101,0xfb840108,0x94850101,
	0xfb840108,0x94860101,0xfb840108,0x94870101,0xfb840108,0x94880101,0xfb840108,0x94890101,
	0xfb840108,0x948a0101,0xfb840108,0x948b0101,0xfb840108,0x948c0101,0xfb840108,0x948d0101,
	0xfb840108,0x948e0101,0xfb840108,0x948f0101,0xfb840108,0x94900101,0xfb840108,0x94910101,
	0xfb840108,0x94920101,0xfb840108,0x94930101,0xfb840108,0x94940101,0xfb840108,0x94950101,
	0xfb840108,0x94960101,0xfb840108,0x94970101,0xfb840108,0x94980101,0xfb840108,0x94990101,
	0xfb840108,0x949a0101,0xfb840108,0x949b0101,0xfb840108,0x949c0101,0xfb840108,0x949d0101,
	0xfb840108,0x949e0101,0xfb840108,0x949f0101,0xfb840108,0x94a00101,0xfb840108,0x94a10101,
	0xfb840108,0x94a20101,0xfb840108,0x94a30101,0xfb840108,0x94a40101,0xfb840108,0x94a50101,
	0xfb840108,0x94a60101,0xfb840108,0x94a70101,0xfb840108,0x94a80101,0xfb840108,0x94a90101,
	0xfb840108,0x94aa0101,0xfb840108,0x94ab0101,0xfb840108,0x94ac0101,0xfb840108,0x94ad0101,
	0xfb840108,0x94ae0101,0xfb840108,0x94af0101,0xfb840108,0x94b00101,0xfb840108,0x94b10101,
	0xfb840108,0x94b20101,0xfb840108,0x94b30101,0xfb840108,0x94b40101,0xfb840108,0x94b50101,
	0xfb840108,0x94b60101,0xfb840108,0x94b70101,0xfb840108,0x94b80101,0xfb840108,0x94b90101,
	0xfb840108,0x94ba0101,0xfb840108,0x94bb0101,0xfb840108,0x94bc0101,0xfb840108,0x94bd0101,
	0xfb840108,0x94be0101,0xfb840108,0x94bf0101,0xfb840108,0x94c00101,0xfb840108,0x94c10101,
	0xfb840108,0x94c20101,0xfb840108,0x94c30101,0xfb840108,0x94c40101,0xfb840108,0x94c50101,
	0xfb840108,0x94c60101,0xfb840108,0x94c70101,0xfb840108,0x94c80101,0xfb840108,0x94c90101,
	0xfb840108,0x94ca0101,0xfb840108,0x94cb0101,0xfb840108,0x94cc0101,0xfb840108,0x94cd0101,
	0xfb840108,0x94ce0101,0xfb840108,0x94cf0101,0xfb840108,0x94d00101,0xfb840108,0x94d10101,
	0xfb840108,0x94d20101,0xfb840108,0x94d30101,0xfb840108,0x94d40101,0xfb840108,0x94d50101,
	0xfb840108,0x94d60101,0xfb840108,0x94d70101,0xfb840108,0x94d80101,0xfb840108,0x94d90101,
	0xfb840108,0x94da0101,0xfb840108,0x94db0101,0xfb840108,0x94dc0101,0xfb840108,0x94dd0101,
	0xfb840108,0x94de0101,0xfb840108,0x94df0101,0xfb840108,0x94e00101,0xfb840108,0x94e10101,
	0xfb840108,0x94e20101,0xfb840108,0x94e30101,0xfb840108,0x94e40101,0xfb840108,0x94e50101,
	0xfb840108,0x94e60101,0xfb840108,0x94e70101,0xfb840108,0x94e80101,0xfb840108,0x94e90101,
	0xfb840108,0x94ea0101,0xfb840108,0x94eb0101,0xfb840108,0x94ec0101,0xfb840108,0x94ed0101,
	0xfb840108,0x94ee0101,0xfb840108,0x94ef0101,0xfb840108,0x94f00101,0xfb840108,0x94f10101,
	0xfb840108,0x94f20101,0xfb840108,0x94f30101,0xfb840108,0x94f40101,0xfb840108,0x94f50101,
	0xfb840108,0x94f60101,0xfb840108,0x94f70101,0xfb840108,0x94f80101,0xfb840108,0x94f90101,
	0xfb840108,0x94fa0101,0xfb840108,0x94fb0101,0xfb840108,0x94fc0101,0xfb840108,0x94fd0101,
	0xfb840108,0x94fe0101,0xfb840108,0x94ff0101,0xfb840108,0x95000101,0xfb840108,0x95010101,
	0xfb840108,0x95020101,0xfb840108,0x95030101,0xfb840108,0x95040101,0xfb840108,0x95050101,
	0xfb840108,0x95060101,0xfb840108,0x95070101,0xfb840108,0x95080101,0xfb840108,0x95090101,
	0xfb840108,0x950a0101,0xfb840108,0x950b0101,0xfb840108,0x950c0101,0xfb840108,0x950d0101,
	0xfb840108,0x950e0101,0xfb840108,0x950f0101,0xfb840108,0x95100101,0xfb840108,0x95110101,
	0xfb840108,0x95120101,0xfb840108,0x95130101,0xfb840108,0x95140101,0xfb840108,0x95150101,
	0xfb840108,0x95160101,0xfb840108,0x95170101,0xfb840108,0x95180101,0xfb840108,0x95190101,
	0xfb840108,0x951a0101,0xfb840108,0x951b0101,0xfb840108,0x951c0101,0xfb840108,0x951d0101,
	0xfb840108,0x951e0101,0xfb840108,0x951f0101,0xfb840108,0x95200101,0xfb840108,0x95210101,
	0xfb840108,0x95220101,0xfb840108,0x95230101,0xfb840108,0x95240101,0xfb840108,0x95250101,
	0xfb840108,0x95260101,0xfb840108,0x95270101,0xfb840108,0x95280101,0xfb840108,0x95290101,
	0xfb840108,0x952a0101,0xfb840108,0x952b0101,0xfb840108,0x952c0101,0xfb840108,0x952d0101,
	0xfb840108,0x952e0101,0xfb840108,0x952f0101,0xfb840108,0x95300101,0xfb840108,0x95310101,
	0xfb840108,0x95320101,0xfb840108,0x95330101,0xfb840108,0x95340101,0xfb840108,0x95350101,
	0xfb840108,0x95360101,0xfb840108,0x95370101,0xfb840108,0x95380101,0xfb840108,0x95390101,
	0xfb840108,0x953a0101,0xfb840108,0x953b0101,0xfb840108,0x953c0101,0xfb840108,0x953d0101,
	0xfb840108,0x953e0101,0xfb840108,0x953f0101,0xfb840108,0x95400101,0xfb840108,0x95410101,
	0xfb840108,0x95420101,0xfb840108,0x95430101,0xfb840108,0x95440101,0xfb840108,0x95450101,
	0xfb840108,0x95460101,0xfb840108,0x95470101,0xfb840108,0x95480101,0xfb840108,0x95490101,
	0xfb840108,0x954a0101,0xfb840108,0x954b0101,0xfb840108,0x954c0101,0xfb840108,0x954d0101,
	0xfb840108,0x954e0101,0xfb840108,0x954f0101,0xfb840108,0x95500101,0xfb840108,0x95510101,
	0xfb840108,0x95520101,0xfb840108,0x95530101,0xfb840108,0x95540101,0xfb840108,0x95550101,
	0xfb840108,0x95560101,0xfb840108,0x95570101,0xfb840108,0x95580101,0xfb840108,0x95590101,
	0xfb840108,0x955a0101,0xfb840108,0x955b0101,0xfb840108,0x955c0101,0xfb840108,0x955d0101,
	0xfb840108,0x955e0101,0xfb840108,0x955f0101,0xfb840108,0x95600101,0xfb840108,0x95610101,
	0xfb840108,0x95620101,0xfb840108,0x95630101,0xfb840108,0x95640101,0xfb840108,0x95650101,
	0xfb840108,0x95660101,0xfb840108,0x95670101,0xfb840108,0x95680101,0xfb840108,0x95690101,
	0xfb840108,0x956a0101,0xfb840108,0x956b0101,0xfb840108,0x956c0101,0xfb840108,0x956d0101,
	0xfb840108,0x956e0101,0xfb840108,0x956f0101,0xfb840108,0x95700101,0xfb840108,0x95710101,
	0xfb840108,0x95720101,0xfb840108,0x95730101,0xfb840108,0x95740101,0xfb840108,0x95750101,
	0xfb840108,0x95760101,0xfb840108,0x95770101,0xfb840108,0x95780101,0xfb840108,0x95790101,
	0xfb840108,0x957a0101,0xfb840108,0x957b0101,0xfb840108,0x957c0101,0xfb840108,0x957d0101,
	0xfb840108,0x957e0101,0xfb840108,0x957f0101,0xfb840108,0x95800101,0xfb840108,0x95810101,
	0xfb840108,0x95820101,0xfb840108,0x95830101,0xfb840108,0x95840101,0xfb840108,0x95850101,
	0xfb840108,0x95860101,0xfb840108,0x95870101,0xfb840108,0x95880101,0xfb840108,0x95890101,
	0xfb840108,0x958a0101,0xfb840108,0x958b0101,0xfb840108,0x958c0101,0xfb840108,0x958d0101,
	0xfb840108,0x958e0101,0xfb840108,0x958f0101,0xfb840108,0x95900101,0xfb840108,0x95910101,
	0xfb840108,0x95920101,0xfb840108,0x95930101,0xfb840108,0x95940101,0xfb840108,0x95950101,
	0xfb840108,0x95960101,0xfb840108,0x95970101,0xfb840108,0x95980101,0xfb840108,0x95990101,
	0xfb840108,0x959a0101,0xfb840108,0x959b0101,0xfb840108,0x959c0101,0xfb840108,0x959d0101,
	0xfb840108,0x959e0101,0xfb840108,0x959f0101,0xfb840108,0x95a00101,0xfb840108,0x95a10101,
	0xfb840108,0x95a20101,0xfb840108,0x95a30101,0xfb840108,0x95a40101,0xfb840108,0x95a50101,
	0xfb840108,0x95a60101,0xfb840108,0x95a70101,0xfb840108,0x95a80101,0xfb840108,0x95a90101,
	0xfb840108,0x95aa0101,0xfb840108,0x95ab0101,0xfb840108,0x95ac0101,0xfb840108,0x95ad0101,
	0xfb840108,0x95ae0101,0xfb840108,0x95af0101,0xfb840108,0x95b00101,0xfb840108,0x95b10101,
	0xfb840108,0x95b20101,0xfb840108,0x95b30101,0xfb840108,0x95b40101,0xfb840108,0x95b50101,
	0xfb840108,0x95b60101,0xfb840108,0x95b70101,0xfb840108,0x95b80101,0xfb840108,0x95b90101,
	0xfb840108,0x95ba0101,0xfb840108,0x95bb0101,0xfb840108,0x95bc0101,0xfb840108,0x95bd0101,
	0xfb840108,0x95be0101,0xfb840108,0x95bf0101,0xfb840108,0x95c00101,0xfb840108,0x95c10101,
	0xfb840108,0x95c20101,0xfb840108,0x95c30101,0xfb840108,0x95c40101,0xfb840108,0x95c50101,
	0xfb840108,0x95c60101,0xfb840108,0x95c70101,0xfb840108,0x95c80101,0xfb840108,0x95c90101,
	0xfb840108,0x95ca0101,0xfb840108,0x95cb0101,0xfb840108,0x95cc0101,0xfb840108,0x95cd0101,
	0xfb840108,0x95ce0101,0xfb840108,0x95cf0101,0xfb840108,0x95d00101,0xfb840108,0x95d10101,
	0xfb840108,0x95d20101,0xfb840108,0x95d30101,0xfb840108,0x95d40101,0xfb840108,0x95d50101,
	0xfb840108,0x95d60101,0xfb840108,0x95d70101,0xfb840108,0x95d80101,0xfb840108,0x95d90101,
	0xfb840108,0x95da0101,0xfb840108,0x95db0101,0xfb840108,0x95dc0101,0xfb840108,0x95dd0101,
	0xfb840108,0x95de0101,0xfb840108,0x95df0101,0xfb840108,0x95e00101,0xfb840108,0x95e10101,
	0xfb840108,0x95e20101,0xfb840108,0x95e30101,0xfb840108,0x95e40101,0xfb840108,0x95e50101,
	0xfb840108,0x95e60101,0xfb840108,0x95e70101,0xfb840108,0x95e80101,0xfb840108,0x95e90101,
	0xfb840108,0x95ea0101,0xfb840108,0x95eb0101,0xfb840108,0x95ec0101,0xfb840108,0x95ed0101,
	0xfb840108,0x95ee0101,0xfb840108,0x95ef0101,0xfb840108,0x95f00101,0xfb840108,0x95f10101,
	0xfb840108,0x95f20101,0xfb840108,0x95f30101,0xfb840108,0x95f40101,0xfb840108,0x95f50101,
	0xfb840108,0x95f60101,0xfb840108,0x95f70101,0xfb840108,0x95f80101,0xfb840108,0x95f90101,
	0xfb840108,0x95fa0101,0xfb840108,0x95fb0101,0xfb840108,0x95fc0101,0xfb840108,0x95fd0101,
	0xfb840108,0x95fe0101,0xfb840108,0x95ff0101,0xfb840108,0x96000101,0xfb840108,0x96010101,
	0xfb840108,0x96020101,0xfb840108,0x96030101,0xfb840108,0x96040101,0xfb840108,0x96050101,
	0xfb840108,0x96060101,0xfb840108,0x96070101,0xfb840108,0x96080101,0xfb840108,0x96090101,
	0xfb840108,0x960a0101,0xfb840108,0x960b0101,0xfb840108,0x960c0101,0xfb840108,0x960d0101,
	0xfb840108,0x960e0101,0xfb840108,0x960f0101,0xfb840108,0x96100101,0xfb840108,0x96110101,
	0xfb840108,0x96120101,0xfb840108,0x96130101,0xfb840108,0x96140101,0xfb840108,0x96150101,
	0xfb840108,0x96160101,0xfb840108,0x96170101,0xfb840108,0x96180101,0xfb840108,0x96190101,
	0xfb840108,0x961a0101,0xfb840108,0x961b0101,0xfb840108,0x961c0101,0xfb840108,0x961d0101,
	0xfb840108,0x961e0101,0xfb840108,0x961f0101,0xfb840108,0x96200101,0xfb840108,0x96210101,
	0xfb840108,0x96220101,0xfb840108,0x96230101,0xfb840108,0x96240101,0xfb840108,0x96250101,
	0xfb840108,0x96260101,0xfb840108,0x96270101,0xfb840108,0x96280101,0xfb840108,0x96290101,
	0xfb840108,0x962a0101,0xfb840108,0x962b0101,0xfb840108,0x962c0101,0xfb840108,0x962d0101,
	0xfb840108,0x962e0101,0xfb840108,0x962f0101,0xfb840108,0x96300101,0xfb840108,0x96310101,
	0xfb840108,0x96320101,0xfb840108,0x96330101,0xfb840108,0x96340101,0xfb840108,0x96350101,
	0xfb840108,0x96360101,0xfb840108,0x96370101,0xfb840108,0x96380101,0xfb840108,0x96390101,
	0xfb840108,0x963a0101,0xfb840108,0x963b0101,0xfb840108,0x963c0101,0xfb840108,0x963d0101,
	0xfb840108,0x963e0101,0xfb840108,0x963f0101,0xfb840108,0x96400101,0xfb840108,0x96410101,
	0xfb840108,0x96420101,0xfb840108,0x96430101,0xfb840108,0x96440101,0xfb840108,0x96450101,
	0xfb840108,0x96460101,0xfb840108,0x96470101,0xfb840108,0x96480101,0xfb840108,0x96490101,
	0xfb840108,0x964a0101,0xfb840108,0x964b0101,0xfb840108,0x964c0101,0xfb840108,0x964d0101,
	0xfb840108,0x964e0101,0xfb840108,0x964f0101,0xfb840108,0x96500101,0xfb840108,0x96510101,
	0xfb840108,0x96520101,0xfb840108,0x96530101,0xfb840108,0x96540101,0xfb840108,0x96550101,
	0xfb840108,0x96560101,0xfb840108,0x96570101,0xfb840108,0x96580101,0xfb840108,0x96590101,
	0xfb840108,0x965a0101,0xfb840108,0x965b0101,0xfb840108,0x965c0101,0xfb840108,0x965d0101,
	0xfb840108,0x965e0101,0xfb840108,0x965f0101,0xfb840108,0x96600101,0xfb840108,0x96610101,
	0xfb840108,0x96620101,0xfb840108,0x96630101,0xfb840108,0x96640101,0xfb840108,0x96650101,
	0xfb840108,0x96660101,0xfb840108,0x96670101,0xfb840108,0x96680101,0xfb840108,0x96690101,
	0xfb840108,0x966a0101,0xfb840108,0x966b0101,0xfb840108,0x966c0101,0xfb840108,0x966d0101,
	0xfb840108,0x966e0101,0xfb840108,0x966f0101,0xfb840108,0x96700101,0xfb840108,0x96710101,
	0xfb840108,0x96720101,0xfb840108,0x96730101,0xfb840108,0x96740101,0xfb840108,0x96750101,
	0xfb840108,0x96760101,0xfb840108,0x96770101,0xfb840108,0x96780101,0xfb840108,0x96790101,
	0xfb840108,0x967a0101,0xfb840108,0x967b0101,0xfb840108,0x967c0101,0xfb840108,0x967d0101,
	0xfb840108,0x967e0101,0xfb840108,0x967f0101,0xfb840108,0x96800101,0xfb840108,0x96810101,
	0xfb840108,0x96820101,0xfb840108,0x96830101,0xfb840108,0x96840101,0xfb840108,0x96850101,
	0xfb840108,0x96860101,0xfb840108,0x96870101,0xfb840108,0x96880101,0xfb840108,0x96890101,
	0xfb840108,0x968a0101,0xfb840108,0x968b0101,0xfb840108,0x968c0101,0xfb840108,0x968d0101,
	0xfb840108,0x968e0101,0xfb840108,0x968f0101,0xfb840108,0x96900101,0xfb840108,0x96910101,
	0xfb840108,0x96920101,0xfb840108,0x96930101,0xfb840108,0x96940101,0xfb840108,0x96950101,
	0xfb840108,0x96960101,0xfb840108,0x96970101,0xfb840108,0x96980101,0xfb840108,0x96990101,
	0xfb840108,0x969a0101,0xfb840108,0x969b0101,0xfb840108,0x969c0101,0xfb840108,0x969d0101,
	0xfb840108,0x969e0101,0xfb840108,0x969f0101,0xfb840108,0x96a00101,0xfb840108,0x96a10101,
	0xfb840108,0x96a20101,0xfb840108,0x96a30101,0xfb840108,0x96a40101,0xfb840108,0x96a50101,
	0xfb840108,0x96a60101,0xfb840108,0x96a70101,0xfb840108,0x96a80101,0xfb840108,0x96a90101,
	0xfb840108,0x96aa0101,0xfb840108,0x96ab0101,0xfb840108,0x96ac0101,0xfb840108,0x96ad0101,
	0xfb840108,0x96ae0101,0xfb840108,0x96af0101,0xfb840108,0x96b00101,0xfb840108,0x96b10101,
	0xfb840108,0x96b20101,0xfb840108,0x96b30101,0xfb840108,0x96b40101,0xfb840108,0x96b50101,
	0xfb840108,0x96b60101,0xfb840108,0x96b70101,0xfb840108,0x96b80101,0xfb840108,0x96b90101,
	0xfb840108,0x96ba0101,0xfb840108,0x96bb0101,0xfb840108,0x96bc0101,0xfb840108,0x96bd0101,
	0xfb840108,0x96be0101,0xfb840108,0x96bf0101,0xfb840108,0x96c00101,0xfb840108,0x96c10101,
	0xfb840108,0x96c20101,0xfb840108,0x96c30101,0xfb840108,0x96c40101,0xfb840108,0x96c50101,
	0xfb840108,0x96c60101,0xfb840108,0x96c70101,0xfb840108,0x96c80101,0xfb840108,0x96c90101,
	0xfb840108,0x96ca0101,0xfb840108,0x96cb0101,0xfb840108,0x96cc0101,0xfb840108,0x96cd0101,
	0xfb840108,0x96ce0101,0xfb840108,0x96cf0101,0xfb840108,0x96d00101,0xfb840108,0x96d10101,
	0xfb840108,0x96d20101,0xfb840108,0x96d30101,0xfb840108,0x96d40101,0xfb840108,0x96d50101,
	0xfb840108,0x96d60101,0xfb840108,0x96d70101,0xfb840108,0x96d80101,0xfb840108,0x96d90101,
	0xfb840108,0x96da0101,0xfb840108,0x96db0101,0xfb840108,0x96dc0101,0xfb840108,0x96dd0101,
	0xfb840108,0x96de0101,0xfb840108,0x96df0101,0xfb840108,0x96e00101,0xfb840108,0x96e10101,
	0xfb840108,0x96e20101,0xfb840108,0x96e30101,0xfb840108,0x96e40101,0xfb840108,0x96e50101,
	0xfb840108,0x96e60101,0xfb840108,0x96e70101,0xfb840108,0x96e80101,0xfb840108,0x96e90101,
	0xfb840108,0x96ea0101,0xfb840108,0x96eb0101,0xfb840108,0x96ec0101,0xfb840108,0x96ed0101,
	0xfb840108,0x96ee0101,0xfb840108,0x96ef0101,0xfb840108,0x96f00101,0xfb840108,0x96f10101,
	0xfb840108,0x96f20101,0xfb840108,0x96f30101,0xfb840108,0x96f40101,0xfb840108,0x96f50101,
	0xfb840108,0x96f60101,0xfb840108,0x96f70101,0xfb840108,0x96f80101,0xfb840108,0x96f90101,
	0xfb840108,0x96fa0101,0xfb840108,0x96fb0101,0xfb840108,0x96fc0101,0xfb840108,0x96fd0101,
	0xfb840108,0x96fe0101,0xfb840108,0x96ff0101,0xfb840108,0x97000101,0xfb840108,0x97010101,
	0xfb840108,0x97020101,0xfb840108,0x97030101,0xfb840108,0x97040101,0xfb840108,0x97050101,
	0xfb840108,0x97060101,0xfb840108,0x97070101,0xfb840108,0x97080101,0xfb840108,0x97090101,
	0xfb840108,0x970a0101,0xfb840108,0x970b0101,0xfb840108,0x970c0101,0xfb840108,0x970d0101,
	0xfb840108,0x970e0101,0xfb840108,0x970f0101,0xfb840108,0x97100101,0xfb840108,0x97110101,
	0xfb840108,0x97120101,0xfb840108,0x97130101,0xfb840108,0x97140101,0xfb840108,0x97150101,
	0xfb840108,0x97160101,0xfb840108,0x97170101,0xfb840108,0x97180101,0xfb840108,0x97190101,
	0xfb840108,0x971a0101,0xfb840108,0x971b0101,0xfb840108,0x971c0101,0xfb840108,0x971d0101,
	0xfb840108,0x971e0101,0xfb840108,0x971f0101,0xfb840108,0x97200101,0xfb840108,0x97210101,
	0xfb840108,0x97220101,0xfb840108,0x97230101,0xfb840108,0x97240101,0xfb840108,0x97250101,
	0xfb840108,0x97260101,0xfb840108,0x97270101,0xfb840108,0x97280101,0xfb840108,0x97290101,
	0xfb840108,0x972a0101,0xfb840108,0x972b0101,0xfb840108,0x972c0101,0xfb840108,0x972d0101,
	0xfb840108,0x972e0101,0xfb840108,0x972f0101,0xfb840108,0x97300101,0xfb840108,0x97310101,
	0xfb840108,0x97320101,0xfb840108,0x97330101,0xfb840108,0x97340101,0xfb840108,0x97350101,
	0xfb840108,0x97360101,0xfb840108,0x97370101,0xfb840108,0x97380101,0xfb840108,0x97390101,
	0xfb840108,0x973a0101,0xfb840108,0x973b0101,0xfb840108,0x973c0101,0xfb840108,0x973d0101,
	0xfb840108,0x973e0101,0xfb840108,0x973f0101,0xfb840108,0x97400101,0xfb840108,0x97410101,
	0xfb840108,0x97420101,0xfb840108,0x97430101,0xfb840108,0x97440101,0xfb840108,0x97450101,
	0xfb840108,0x97460101,0xfb840108,0x97470101,0xfb840108,0x97480101,0xfb840108,0x97490101,
	0xfb840108,0x974a0101,0xfb840108,0x974b0101,0xfb840108,0x974c0101,0xfb840108,0x974d0101,
	0xfb840108,0x974e0101,0xfb840108,0x974f0101,0xfb840108,0x97500101,0xfb840108,0x97510101,
	0xfb840108,0x97520101,0xfb840108,0x97530101,0xfb840108,0x97540101,0xfb840108,0x97550101,
	0xfb840108,0x97560101,0xfb840108,0x97570101,0xfb840108,0x97580101,0xfb840108,0x97590101,
	0xfb840108,0x975a0101,0xfb840108,0x975b0101,0xfb840108,0x975c0101,0xfb840108,0x975d0101,
	0xfb840108,0x975e0101,0xfb840108,0x975f0101,0xfb840108,0x97600101,0xfb840108,0x97610101,
	0xfb840108,0x97620101,0xfb840108,0x97630101,0xfb840108,0x97640101,0xfb840108,0x97650101,
	0xfb840108,0x97660101,0xfb840108,0x97670101,0xfb840108,0x97680101,0xfb840108,0x97690101,
	0xfb840108,0x976a0101,0xfb840108,0x976b0101,0xfb840108,0x976c0101,0xfb840108,0x976d0101,
	0xfb840108,0x976e0101,0xfb840108,0x976f0101,0xfb840108,0x97700101,0xfb840108,0x97710101,
	0xfb840108,0x97720101,0xfb840108,0x97730101,0xfb840108,0x97740101,0xfb840108,0x97750101,
	0xfb840108,0x97760101,0xfb840108,0x97770101,0xfb840108,0x97780101,0xfb840108,0x97790101,
	0xfb840108,0x977a0101,0xfb840108,0x977b0101,0xfb840108,0x977c0101,0xfb840108,0x977d0101,
	0xfb840108,0x977e0101,0xfb840108,0x977f0101,0xfb840108,0x97800101,0xfb840108,0x97810101,
	0xfb840108,0x97820101,0xfb840108,0x97830101,0xfb840108,0x97840101,0xfb840108,0x97850101,
	0xfb840108,0x97860101,0xfb840108,0x97870101,0xfb840108,0x97880101,0xfb840108,0x97890101,
	0xfb840108,0x978a0101,0xfb840108,0x978b0101,0xfb840108,0x978c0101,0xfb840108,0x978d0101,
	0xfb840108,0x978e0101,0xfb840108,0x978f0101,0xfb840108,0x97900101,0xfb840108,0x97910101,
	0xfb840108,0x97920101,0xfb840108,0x97930101,0xfb840108,0x97940101,0xfb840108,0x97950101,
	0xfb840108,0x97960101,0xfb840108,0x97970101,0xfb840108,0x97980101,0xfb840108,0x97990101,
	0xfb840108,0x979a0101,0xfb840108,0x979b0101,0xfb840108,0x979c0101,0xfb840108,0x979d0101,
	0xfb840108,0x979e0101,0xfb840108,0x979f0101,0xfb840108,0x97a00101,0xfb840108,0x97a10101,
	0xfb840108,0x97a20101,0xfb840108,0x97a30101,0xfb840108,0x97a40101,0xfb840108,0x97a50101,
	0xfb840108,0x97a60101,0xfb840108,0x97a70101,0xfb840108,0x97a80101,0xfb840108,0x97a90101,
	0xfb840108,0x97aa0101,0xfb840108,0x97ab0101,0xfb840108,0x97ac0101,0xfb840108,0x97ad0101,
	0xfb840108,0x97ae0101,0xfb840108,0x97af0101,0xfb840108,0x97b00101,0xfb840108,0x97b10101,
	0xfb840108,0x97b20101,0xfb840108,0x97b30101,0xfb840108,0x97b40101,0xfb840108,0x97b50101,
	0xfb840108,0x97b60101,0xfb840108,0x97b70101,0xfb840108,0x97b80101,0xfb840108,0x97b90101,
	0xfb840108,0x97ba0101,0xfb840108,0x97bb0101,0xfb840108,0x97bc0101,0xfb840108,0x97bd0101,
	0xfb840108,0x97be0101,0xfb840108,0x97bf0101,0xfb840108,0x97c00101,0xfb840108,0x97c10101,
	0xfb840108,0x97c20101,0xfb840108,0x97c30101,0xfb840108,0x97c40101,0xfb840108,0x97c50101,
	0xfb840108,0x97c60101,0xfb840108,0x97c70101,0xfb840108,0x97c80101,0xfb840108,0x97c90101,
	0xfb840108,0x97ca0101,0xfb840108,0x97cb0101,0xfb840108,0x97cc0101,0xfb840108,0x97cd0101,
	0xfb840108,0x97ce0101,0xfb840108,0x97cf0101,0xfb840108,0x97d00101,0xfb840108,0x97d10101,
	0xfb840108,0x97d20101,0xfb840108,0x97d30101,0xfb840108,0x97d40101,0xfb840108,0x97d50101,
	0xfb840108,0x97d60101,0xfb840108,0x97d70101,0xfb840108,0x97d80101,0xfb840108,0x97d90101,
	0xfb840108,0x97da0101,0xfb840108,0x97db0101,0xfb840108,0x97dc0101,0xfb840108,0x97dd0101,
	0xfb840108,0x97de0101,0xfb840108,0x97df0101,0xfb840108,0x97e00101,0xfb840108,0x97e10101,
	0xfb840108,0x97e20101,0xfb840108,0x97e30101,0xfb840108,0x97e40101,0xfb840108,0x97e50101,
	0xfb840108,0x97e60101,0xfb840108,0x97e70101,0xfb840108,0x97e80101,0xfb840108,0x97e90101,
	0xfb840108,0x97ea0101,0xfb840108,0x97eb0101,0xfb840108,0x97ec0101,0xfb840108,0x97ed0101,
	0xfb840108,0x97ee0101,0xfb840108,0x97ef0101,0xfb840108,0x97f00101,0xfb840108,0x97f10101,
	0xfb840108,0x97f20101,0xfb840108,0x97f30101,0xfb840108,0x97f40101,0xfb840108,0x97f50101,
	0xfb840108,0x97f60101,0xfb840108,0x97f70101,0xfb840108,0x97f80101,0xfb840108,0x97f90101,
	0xfb840108,0x97fa0101,0xfb840108,0x97fb0101,0xfb840108,0x97fc0101,0xfb840108,0x97fd0101,
	0xfb840108,0x97fe0101,0xfb840108,0x97ff0101,0xfb840108,0x98000101,0xfb840108,0x98010101,
	0xfb840108,0x98020101,0xfb840108,0x98030101,0xfb840108,0x98040101,0xfb840108,0x98050101,
	0xfb840108,0x98060101,0xfb840108,0x98070101,0xfb840108,0x98080101,0xfb840108,0x98090101,
	0xfb840108,0x980a0101,0xfb840108,0x980b0101,0xfb840108,0x980c0101,0xfb840108,0x980d0101,
	0xfb840108,0x980e0101,0xfb840108,0x980f0101,0xfb840108,0x98100101,0xfb840108,0x98110101,
	0xfb840108,0x98120101,0xfb840108,0x98130101,0xfb840108,0x98140101,0xfb840108,0x98150101,
	0xfb840108,0x98160101,0xfb840108,0x98170101,0xfb840108,0x98180101,0xfb840108,0x98190101,
	0xfb840108,0x981a0101,0xfb840108,0x981b0101,0xfb840108,0x981c0101,0xfb840108,0x981d0101,
	0xfb840108,0x981e0101,0xfb840108,0x981f0101,0xfb840108,0x98200101,0xfb840108,0x98210101,
	0xfb840108,0x98220101,0xfb840108,0x98230101,0xfb840108,0x98240101,0xfb840108,0x98250101,
	0xfb840108,0x98260101,0xfb840108,0x98270101,0xfb840108,0x98280101,0xfb840108,0x98290101,
	0xfb840108,0x982a0101,0xfb840108,0x982b0101,0xfb840108,0x982c0101,0xfb840108,0x982d0101,
	0xfb840108,0x982e0101,0xfb840108,0x982f0101,0xfb840108,0x98300101,0xfb840108,0x98310101,
	0xfb840108,0x98320101,0xfb840108,0x98330101,0xfb840108,0x98340101,0xfb840108,0x98350101,
	0xfb840108,0x98360101,0xfb840108,0x98370101,0xfb840108,0x98380101,0xfb840108,0x98390101,
	0xfb840108,0x983a0101,0xfb840108,0x983b0101,0xfb840108,0x983c0101,0xfb840108,0x983d0101,
	0xfb840108,0x983e0101,0xfb840108,0x983f0101,0xfb840108,0x98400101,0xfb840108,0x98410101,
	0xfb840108,0x98420101,0xfb840108,0x98430101,0xfb840108,0x98440101,0xfb840108,0x98450101,
	0xfb840108,0x98460101,0xfb840108,0x98470101,0xfb840108,0x98480101,0xfb840108,0x98490101,
	0xfb840108,0x984a0101,0xfb840108,0x984b0101,0xfb840108,0x984c0101,0xfb840108,0x984d0101,
	0xfb840108,0x984e0101,0xfb840108,0x984f0101,0xfb840108,0x98500101,0xfb840108,0x98510101,
	0xfb840108,0x98520101,0xfb840108,0x98530101,0xfb840108,0x98540101,0xfb840108,0x98550101,
	0xfb840108,0x98560101,0xfb840108,0x98570101,0xfb840108,0x98580101,0xfb840108,0x98590101,
	0xfb840108,0x985a0101,0xfb840108,0x985b0101,0xfb840108,0x985c0101,0xfb840108,0x985d0101,
	0xfb840108,0x985e0101,0xfb840108,0x985f0101,0xfb840108,0x98600101,0xfb840108,0x98610101,
	0xfb840108,0x98620101,0xfb840108,0x98630101,0xfb840108,0x98640101,0xfb840108,0x98650101,
	0xfb840108,0x98660101,0xfb840108,0x98670101,0xfb840108,0x98680101,0xfb840108,0x98690101,
	0xfb840108,0x986a0101,0xfb840108,0x986b0101,0xfb840108,0x986c0101,0xfb840108,0x986d0101,
	0xfb840108,0x986e0101,0xfb840108,0x986f0101,0xfb840108,0x98700101,0xfb840108,0x98710101,
	0xfb840108,0x98720101,0xfb840108,0x98730101,0xfb840108,0x98740101,0xfb840108,0x98750101,
	0xfb840108,0x98760101,0xfb840108,0x98770101,0xfb840108,0x98780101,0xfb840108,0x98790101,
	0xfb840108,0x987a0101,0xfb840108,0x987b0101,0xfb840108,0x987c0101,0xfb840108,0x987d0101,
	0xfb840108,0x987e0101,0xfb840108,0x987f0101,0xfb840108,0x98800101,0xfb840108,0x98810101,
	0xfb840108,0x98820101,0xfb840108,0x98830101,0xfb840108,0x98840101,0xfb840108,0x98850101,
	0xfb840108,0x98860101,0xfb840108,0x98870101,0xfb840108,0x98880101,0xfb840108,0x98890101,
	0xfb840108,0x988a0101,0xfb840108,0x988b0101,0xfb840108,0x988c0101,0xfb840108,0x988d0101,
	0xfb840108,0x988e0101,0xfb840108,0x988f0101,0xfb840108,0x98900101,0xfb840108,0x98910101,
	0xfb840108,0x98920101,0xfb840108,0x98930101,0xfb840108,0x98940101,0xfb840108,0x98950101,
	0xfb840108,0x98960101,0xfb840108,0x98970101,0xfb840108,0x98980101,0xfb840108,0x98990101,
	0xfb840108,0x989a0101,0xfb840108,0x989b0101,0xfb840108,0x989c0101,0xfb840108,0x989d0101,
	0xfb840108,0x989e0101,0xfb840108,0x989f0101,0xfb840108,0x98a00101,0xfb840108,0x98a10101,
	0xfb840108,0x98a20101,0xfb840108,0x98a30101,0xfb840108,0x98a40101,0xfb840108,0x98a50101,
	0xfb840108,0x98a60101,0xfb840108,0x98a70101,0xfb840108,0x98a80101,0xfb840108,0x98a90101,
	0xfb840108,0x98aa0101,0xfb840108,0x98ab0101,0xfb840108,0x98ac0101,0xfb840108,0x98ad0101,
	0xfb840108,0x98ae0101,0xfb840108,0x98af0101,0xfb840108,0x98b00101,0xfb840108,0x98b10101,
	0xfb840108,0x98b20101,0xfb840108,0x98b30101,0xfb840108,0x98b40101,0xfb840108,0x98b50101,
	0xfb840108,0x98b60101,0xfb840108,0x98b70101,0xfb840108,0x98b80101,0xfb840108,0x98b90101,
	0xfb840108,0x98ba0101,0xfb840108,0x98bb0101,0xfb840108,0x98bc0101,0xfb840108,0x98bd0101,
	0xfb840108,0x98be0101,0xfb840108,0x98bf0101,0xfb840108,0x98c00101,0xfb840108,0x98c10101,
	0xfb840108,0x98c20101,0xfb840108,0x98c30101,0xfb840108,0x98c40101,0xfb840108,0x98c50101,
	0xfb840108,0x98c60101,0xfb840108,0x98c70101,0xfb840108,0x98c80101,0xfb840108,0x98c90101,
	0xfb840108,0x98ca0101,0xfb840108,0x98cb0101,0xfb840108,0x98cc0101,0xfb840108,0x98cd0101,
	0xfb840108,0x98ce0101,0xfb840108,0x98cf0101,0xfb840108,0x98d00101,0xfb840108,0x98d10101,
	0xfb840108,0x98d20101,0xfb840108,0x98d30101,0xfb840108,0x98d40101,0xfb840108,0x98d50101,
	0xfb840108,0x98d60101,0xfb840108,0x98d70101,0xfb840108,0x98d80101,0xfb840108,0x98d90101,
	0xfb840108,0x98da0101,0xfb840108,0x98db0101,0xfb840108,0x98dc0101,0xfb840108,0x98dd0101,
	0xfb840108,0x98de0101,0xfb840108,0x98df0101,0xfb840108,0x98e00101,0xfb840108,0x98e10101,
	0xfb840108,0x98e20101,0xfb840108,0x98e30101,0xfb840108,0x98e40101,0xfb840108,0x98e50101,
	0xfb840108,0x98e60101,0xfb840108,0x98e70101,0xfb840108,0x98e80101,0xfb840108,0x98e90101,
	0xfb840108,0x98ea0101,0xfb840108,0x98eb0101,0xfb840108,0x98ec0101,0xfb840108,0x98ed0101,
	0xfb840108,0x98ee0101,0xfb840108,0x98ef0101,0xfb840108,0x98f00101,0xfb840108,0x98f10101,
	0xfb840108,0x98f20101,0xfb840108,0x98f30101,0xfb840108,0x98f40101,0xfb840108,0x98f50101,
	0xfb840108,0x98f60101,0xfb840108,0x98f70101,0xfb840108,0x98f80101,0xfb840108,0x98f90101,
	0xfb840108,0x98fa0101,0xfb840108,0x98fb0101,0xfb840108,0x98fc0101,0xfb840108,0x98fd0101,
	0xfb840108,0x98fe0101,0xfb840108,0x98ff0101,0xfb840108,0x99000101,0xfb840108,0x99010101,
	0xfb840108,0x99020101,0xfb840108,0x99030101,0xfb840108,0x99040101,0xfb840108,0x99050101,
	0xfb840108,0x99060101,0xfb840108,0x99070101,0xfb840108,0x99080101,0xfb840108,0x99090101,
	0xfb840108,0x990a0101,0xfb840108,0x990b0101,0xfb840108,0x990c0101,0xfb840108,0x990d0101,
	0xfb840108,0x990e0101,0xfb840108,0x990f0101,0xfb840108,0x99100101,0xfb840108,0x99110101,
	0xfb840108,0x99120101,0xfb840108,0x99130101,0xfb840108,0x99140101,0xfb840108,0x99150101,
	0xfb840108,0x99160101,0xfb840108,0x99170101,0xfb840108,0x99180101,0xfb840108,0x99190101,
	0xfb840108,0x991a0101,0xfb840108,0x991b0101,0xfb840108,0x991c0101,0xfb840108,0x991d0101,
	0xfb840108,0x991e0101,0xfb840108,0x991f0101,0xfb840108,0x99200101,0xfb840108,0x99210101,
	0xfb840108,0x99220101,0xfb840108,0x99230101,0xfb840108,0x99240101,0xfb840108,0x99250101,
	0xfb840108,0x99260101,0xfb840108,0x99270101,0xfb840108,0x99280101,0xfb840108,0x99290101,
	0xfb840108,0x992a0101,0xfb840108,0x992b0101,0xfb840108,0x992c0101,0xfb840108,0x992d0101,
	0xfb840108,0x992e0101,0xfb840108,0x992f0101,0xfb840108,0x99300101,0xfb840108,0x99310101,
	0xfb840108,0x99320101,0xfb840108,0x99330101,0xfb840108,0x99340101,0xfb840108,0x99350101,
	0xfb840108,0x99360101,0xfb840108,0x99370101,0xfb840108,0x99380101,0xfb840108,0x99390101,
	0xfb840108,0x993a0101,0xfb840108,0x993b0101,0xfb840108,0x993c0101,0xfb840108,0x993d0101,
	0xfb840108,0x993e0101,0xfb840108,0x993f0101,0xfb840108,0x99400101,0xfb840108,0x99410101,
	0xfb840108,0x99420101,0xfb840108,0x99430101,0xfb840108,0x99440101,0xfb840108,0x99450101,
	0xfb840108,0x99460101,0xfb840108,0x99470101,0xfb840108,0x99480101,0xfb840108,0x99490101,
	0xfb840108,0x994a0101,0xfb840108,0x994b0101,0xfb840108,0x994c0101,0xfb840108,0x994d0101,
	0xfb840108,0x994e0101,0xfb840108,0x994f0101,0xfb840108,0x99500101,0xfb840108,0x99510101,
	0xfb840108,0x99520101,0xfb840108,0x99530101,0xfb840108,0x99540101,0xfb840108,0x99550101,
	0xfb840108,0x99560101,0xfb840108,0x99570101,0xfb840108,0x99580101,0xfb840108,0x99590101,
	0xfb840108,0x995a0101,0xfb840108,0x995b0101,0xfb840108,0x995c0101,0xfb840108,0x995d0101,
	0xfb840108,0x995e0101,0xfb840108,0x995f0101,0xfb840108,0x99600101,0xfb840108,0x99610101,
	0xfb840108,0x99620101,0xfb840108,0x99630101,0xfb840108,0x99640101,0xfb840108,0x99650101,
	0xfb840108,0x99660101,0xfb840108,0x99670101,0xfb840108,0x99680101,0xfb840108,0x99690101,
	0xfb840108,0x996a0101,0xfb840108,0x996b0101,0xfb840108,0x996c0101,0xfb840108,0x996d0101,
	0xfb840108,0x996e0101,0xfb840108,0x996f0101,0xfb840108,0x99700101,0xfb840108,0x99710101,
	0xfb840108,0x99720101,0xfb840108,0x99730101,0xfb840108,0x99740101,0xfb840108,0x99750101,
	0xfb840108,0x99760101,0xfb840108,0x99770101,0xfb840108,0x99780101,0xfb840108,0x99790101,
	0xfb840108,0x997a0101,0xfb840108,0x997b0101,0xfb840108,0x997c0101,0xfb840108,0x997d0101,
	0xfb840108,0x997e0101,0xfb840108,0x997f0101,0xfb840108,0x99800101,0xfb840108,0x99810101,
	0xfb840108,0x99820101,0xfb840108,0x99830101,0xfb840108,0x99840101,0xfb840108,0x99850101,
	0xfb840108,0x99860101,0xfb840108,0x99870101,0xfb840108,0x99880101,0xfb840108,0x99890101,
	0xfb840108,0x998a0101,0xfb840108,0x998b0101,0xfb840108,0x998c0101,0xfb840108,0x998d0101,
	0xfb840108,0x998e0101,0xfb840108,0x998f0101,0xfb840108,0x99900101,0xfb840108,0x99910101,
	0xfb840108,0x99920101,0xfb840108,0x99930101,0xfb840108,0x99940101,0xfb840108,0x99950101,
	0xfb840108,0x99960101,0xfb840108,0x99970101,0xfb840108,0x99980101,0xfb840108,0x99990101,
	0xfb840108,0x999a0101,0xfb840108,0x999b0101,0xfb840108,0x999c0101,0xfb840108,0x999d0101,
	0xfb840108,0x999e0101,0xfb840108,0x999f0101,0xfb840108,0x99a00101,0xfb840108,0x99a10101,
	0xfb840108,0x99a20101,0xfb840108,0x99a30101,0xfb840108,0x99a40101,0xfb840108,0x99a50101,
	0xfb840108,0x99a60101,0xfb840108,0x99a70101,0xfb840108,0x99a80101,0xfb840108,0x99a90101,
	0xfb840108,0x99aa0101,0xfb840108,0x99ab0101,0xfb840108,0x99ac0101,0xfb840108,0x99ad0101,
	0xfb840108,0x99ae0101,0xfb840108,0x99af0101,0xfb840108,0x99b00101,0xfb840108,0x99b10101,
	0xfb840108,0x99b20101,0xfb840108,0x99b30101,0xfb840108,0x99b40101,0xfb840108,0x99b50101,
	0xfb840108,0x99b60101,0xfb840108,0x99b70101,0xfb840108,0x99b80101,0xfb840108,0x99b90101,
	0xfb840108,0x99ba0101,0xfb840108,0x99bb0101,0xfb840108,0x99bc0101,0xfb840108,0x99bd0101,
	0xfb840108,0x99be0101,0xfb840108,0x99bf0101,0xfb840108,0x99c00101,0xfb840108,0x99c10101,
	0xfb840108,0x99c20101,0xfb840108,0x99c30101,0xfb840108,0x99c40101,0xfb840108,0x99c50101,
	0xfb840108,0x99c60101,0xfb840108,0x99c70101,0xfb840108,0x99c80101,0xfb840108,0x99c90101,
	0xfb840108,0x99ca0101,0xfb840108,0x99cb0101,0xfb840108,0x99cc0101,0xfb840108,0x99cd0101,
	0xfb840108,0x99ce0101,0xfb840108,0x99cf0101,0xfb840108,0x99d00101,0xfb840108,0x99d10101,
	0xfb840108,0x99d20101,0xfb840108,0x99d30101,0xfb840108,0x99d40101,0xfb840108,0x99d50101,
	0xfb840108,0x99d60101,0xfb840108,0x99d70101,0xfb840108,0x99d80101,0xfb840108,0x99d90101,
	0xfb840108,0x99da0101,0xfb840108,0x99db0101,0xfb840108,0x99dc0101,0xfb840108,0x99dd0101,
	0xfb840108,0x99de0101,0xfb840108,0x99df0101,0xfb840108,0x99e00101,0xfb840108,0x99e10101,
	0xfb840108,0x99e20101,0xfb840108,0x99e30101,0xfb840108,0x99e40101,0xfb840108,0x99e50101,
	0xfb840108,0x99e60101,0xfb840108,0x99e70101,0xfb840108,0x99e80101,0xfb840108,0x99e90101,
	0xfb840108,0x99ea0101,0xfb840108,0x99eb0101,0xfb840108,0x99ec0101,0xfb840108,0x99ed0101,
	0xfb840108,0x99ee0101,0xfb840108,0x99ef0101,0xfb840108,0x99f00101,0xfb840108,0x99f10101,
	0xfb840108,0x99f20101,0xfb840108,0x99f30101,0xfb840108,0x99f40101,0xfb840108,0x99f50101,
	0xfb840108,0x99f60101,0xfb840108,0x99f70101,0xfb840108,0x99f80101,0xfb840108,0x99f90101,
	0xfb840108,0x99fa0101,0xfb840108,0x99fb0101,0xfb840108,0x99fc0101,0xfb840108,0x99fd0101,
	0xfb840108,0x99fe0101,0xfb840108,0x99ff0101,0xfb840108,0x9a000101,0xfb840108,0x9a010101,
	0xfb840108,0x9a020101,0xfb840108,0x9a030101,0xfb840108,0x9a040101,0xfb840108,0x9a050101,
	0xfb840108,0x9a060101,0xfb840108,0x9a070101,0xfb840108,0x9a080101,0xfb840108,0x9a090101,
	0xfb840108,0x9a0a0101,0xfb840108,0x9a0b0101,0xfb840108,0x9a0c0101,0xfb840108,0x9a0d0101,
	0xfb840108,0x9a0e0101,0xfb840108,0x9a0f0101,0xfb840108,0x9a100101,0xfb840108,0x9a110101,
	0xfb840108,0x9a120101,0xfb840108,0x9a130101,0xfb840108,0x9a140101,0xfb840108,0x9a150101,
	0xfb840108,0x9a160101,0xfb840108,0x9a170101,0xfb840108,0x9a180101,0xfb840108,0x9a190101,
	0xfb840108,0x9a1a0101,0xfb840108,0x9a1b0101,0xfb840108,0x9a1c0101,0xfb840108,0x9a1d0101,
	0xfb840108,0x9a1e0101,0xfb840108,0x9a1f0101,0xfb840108,0x9a200101,0xfb840108,0x9a210101,
	0xfb840108,0x9a220101,0xfb840108,0x9a230101,0xfb840108,0x9a240101,0xfb840108,0x9a250101,
	0xfb840108,0x9a260101,0xfb840108,0x9a270101,0xfb840108,0x9a280101,0xfb840108,0x9a290101,
	0xfb840108,0x9a2a0101,0xfb840108,0x9a2b0101,0xfb840108,0x9a2c0101,0xfb840108,0x9a2d0101,
	0xfb840108,0x9a2e0101,0xfb840108,0x9a2f0101,0xfb840108,0x9a300101,0xfb840108,0x9a310101,
	0xfb840108,0x9a320101,0xfb840108,0x9a330101,0xfb840108,0x9a340101,0xfb840108,0x9a350101,
	0xfb840108,0x9a360101,0xfb840108,0x9a370101,0xfb840108,0x9a380101,0xfb840108,0x9a390101,
	0xfb840108,0x9a3a0101,0xfb840108,0x9a3b0101,0xfb840108,0x9a3c0101,0xfb840108,0x9a3d0101,
	0xfb840108,0x9a3e0101,0xfb840108,0x9a3f0101,0xfb840108,0x9a400101,0xfb840108,0x9a410101,
	0xfb840108,0x9a420101,0xfb840108,0x9a430101,0xfb840108,0x9a440101,0xfb840108,0x9a450101,
	0xfb840108,0x9a460101,0xfb840108,0x9a470101,0xfb840108,0x9a480101,0xfb840108,0x9a490101,
	0xfb840108,0x9a4a0101,0xfb840108,0x9a4b0101,0xfb840108,0x9a4c0101,0xfb840108,0x9a4d0101,
	0xfb840108,0x9a4e0101,0xfb840108,0x9a4f0101,0xfb840108,0x9a500101,0xfb840108,0x9a510101,
	0xfb840108,0x9a520101,0xfb840108,0x9a530101,0xfb840108,0x9a540101,0xfb840108,0x9a550101,
	0xfb840108,0x9a560101,0xfb840108,0x9a570101,0xfb840108,0x9a580101,0xfb840108,0x9a590101,
	0xfb840108,0x9a5a0101,0xfb840108,0x9a5b0101,0xfb840108,0x9a5c0101,0xfb840108,0x9a5d0101,
	0xfb840108,0x9a5e0101,0xfb840108,0x9a5f0101,0xfb840108,0x9a600101,0xfb840108,0x9a610101,
	0xfb840108,0x9a620101,0xfb840108,0x9a630101,0xfb840108,0x9a640101,0xfb840108,0x9a650101,
	0xfb840108,0x9a660101,0xfb840108,0x9a670101,0xfb840108,0x9a680101,0xfb840108,0x9a690101,
	0xfb840108,0x9a6a0101,0xfb840108,0x9a6b0101,0xfb840108,0x9a6c0101,0xfb840108,0x9a6d0101,
	0xfb840108,0x9a6e0101,0xfb840108,0x9a6f0101,0xfb840108,0x9a700101,0xfb840108,0x9a710101,
	0xfb840108,0x9a720101,0xfb840108,0x9a730101,0xfb840108,0x9a740101,0xfb840108,0x9a750101,
	0xfb840108,0x9a760101,0xfb840108,0x9a770101,0xfb840108,0x9a780101,0xfb840108,0x9a790101,
	0xfb840108,0x9a7a0101,0xfb840108,0x9a7b0101,0xfb840108,0x9a7c0101,0xfb840108,0x9a7d0101,
	0xfb840108,0x9a7e0101,0xfb840108,0x9a7f0101,0xfb840108,0x9a800101,0xfb840108,0x9a810101,
	0xfb840108,0x9a820101,0xfb840108,0x9a830101,0xfb840108,0x9a840101,0xfb840108,0x9a850101,
	0xfb840108,0x9a860101,0xfb840108,0x9a870101,0xfb840108,0x9a880101,0xfb840108,0x9a890101,
	0xfb840108,0x9a8a0101,0xfb840108,0x9a8b0101,0xfb840108,0x9a8c0101,0xfb840108,0x9a8d0101,
	0xfb840108,0x9a8e0101,0xfb840108,0x9a8f0101,0xfb840108,0x9a900101,0xfb840108,0x9a910101,
	0xfb840108,0x9a920101,0xfb840108,0x9a930101,0xfb840108,0x9a940101,0xfb840108,0x9a950101,
	0xfb840108,0x9a960101,0xfb840108,0x9a970101,0xfb840108,0x9a980101,0xfb840108,0x9a990101,
	0xfb840108,0x9a9a0101,0xfb840108,0x9a9b0101,0xfb840108,0x9a9c0101,0xfb840108,0x9a9d0101,
	0xfb840108,0x9a9e0101,0xfb840108,0x9a9f0101,0xfb840108,0x9aa00101,0xfb840108,0x9aa10101,
	0xfb840108,0x9aa20101,0xfb840108,0x9aa30101,0xfb840108,0x9aa40101,0xfb840108,0x9aa50101,
	0xfb840108,0x9aa60101,0xfb840108,0x9aa70101,0xfb840108,0x9aa80101,0xfb840108,0x9aa90101,
	0xfb840108,0x9aaa0101,0xfb840108,0x9aab0101,0xfb840108,0x9aac0101,0xfb840108,0x9aad0101,
	0xfb840108,0x9aae0101,0xfb840108,0x9aaf0101,0xfb840108,0x9ab00101,0xfb840108,0x9ab10101,
	0xfb840108,0x9ab20101,0xfb840108,0x9ab30101,0xfb840108,0x9ab40101,0xfb840108,0x9ab50101,
	0xfb840108,0x9ab60101,0xfb840108,0x9ab70101,0xfb840108,0x9ab80101,0xfb840108,0x9ab90101,
	0xfb840108,0x9aba0101,0xfb840108,0x9abb0101,0xfb840108,0x9abc0101,0xfb840108,0x9abd0101,
	0xfb840108,0x9abe0101,0xfb840108,0x9abf0101,0xfb840108,0x9ac00101,0xfb840108,0x9ac10101,
	0xfb840108,0x9ac20101,0xfb840108,0x9ac30101,0xfb840108,0x9ac40101,0xfb840108,0x9ac50101,
	0xfb840108,0x9ac60101,0xfb840108,0x9ac70101,0xfb840108,0x9ac80101,0xfb840108,0x9ac90101,
	0xfb840108,0x9aca0101,0xfb840108,0x9acb0101,0xfb840108,0x9acc0101,0xfb840108,0x9acd0101,
	0xfb840108,0x9ace0101,0xfb840108,0x9acf0101,0xfb840108,0x9ad00101,0xfb840108,0x9ad10101,
	0xfb840108,0x9ad20101,0xfb840108,0x9ad30101,0xfb840108,0x9ad40101,0xfb840108,0x9ad50101,
	0xfb840108,0x9ad60101,0xfb840108,0x9ad70101,0xfb840108,0x9ad80101,0xfb840108,0x9ad90101,
	0xfb840108,0x9ada0101,0xfb840108,0x9adb0101,0xfb840108,0x9adc0101,0xfb840108,0x9add0101,
	0xfb840108,0x9ade0101,0xfb840108,0x9adf0101,0xfb840108,0x9ae00101,0xfb840108,0x9ae10101,
	0xfb840108,0x9ae20101,0xfb840108,0x9ae30101,0xfb840108,0x9ae40101,0xfb840108,0x9ae50101,
	0xfb840108,0x9ae60101,0xfb840108,0x9ae70101,0xfb840108,0x9ae80101,0xfb840108,0x9ae90101,
	0xfb840108,0x9aea0101,0xfb840108,0x9aeb0101,0xfb840108,0x9aec0101,0xfb840108,0x9aed0101,
	0xfb840108,0x9aee0101,0xfb840108,0x9aef0101,0xfb840108,0x9af00101,0xfb840108,0x9af10101,
	0xfb840108,0x9af20101,0xfb840108,0x9af30101,0xfb840108,0x9af40101,0xfb840108,0x9af50101,
	0xfb840108,0x9af60101,0xfb840108,0x9af70101,0xfb840108,0x9af80101,0xfb840108,0x9af90101,
	0xfb840108,0x9afa0101,0xfb840108,0x9afb0101,0xfb840108,0x9afc0101,0xfb840108,0x9afd0101,
	0xfb840108,0x9afe0101,0xfb840108,0x9aff0101,0xfb840108,0x9b000101,0xfb840108,0x9b010101,
	0xfb840108,0x9b020101,0xfb840108,0x9b030101,0xfb840108,0x9b040101,0xfb840108,0x9b050101,
	0xfb840108,0x9b060101,0xfb840108,0x9b070101,0xfb840108,0x9b080101,0xfb840108,0x9b090101,
	0xfb840108,0x9b0a0101,0xfb840108,0x9b0b0101,0xfb840108,0x9b0c0101,0xfb840108,0x9b0d0101,
	0xfb840108,0x9b0e0101,0xfb840108,0x9b0f0101,0xfb840108,0x9b100101,0xfb840108,0x9b110101,
	0xfb840108,0x9b120101,0xfb840108,0x9b130101,0xfb840108,0x9b140101,0xfb840108,0x9b150101,
	0xfb840108,0x9b160101,0xfb840108,0x9b170101,0xfb840108,0x9b180101,0xfb840108,0x9b190101,
	0xfb840108,0x9b1a0101,0xfb840108,0x9b1b0101,0xfb840108,0x9b1c0101,0xfb840108,0x9b1d0101,
	0xfb840108,0x9b1e0101,0xfb840108,0x9b1f0101,0xfb840108,0x9b200101,0xfb840108,0x9b210101,
	0xfb840108,0x9b220101,0xfb840108,0x9b230101,0xfb840108,0x9b240101,0xfb840108,0x9b250101,
	0xfb840108,0x9b260101,0xfb840108,0x9b270101,0xfb840108,0x9b280101,0xfb840108,0x9b290101,
	0xfb840108,0x9b2a0101,0xfb840108,0x9b2b0101,0xfb840108,0x9b2c0101,0xfb840108,0x9b2d0101,
	0xfb840108,0x9b2e0101,0xfb840108,0x9b2f0101,0xfb840108,0x9b300101,0xfb840108,0x9b310101,
	0xfb840108,0x9b320101,0xfb840108,0x9b330101,0xfb840108,0x9b340101,0xfb840108,0x9b350101,
	0xfb840108,0x9b360101,0xfb840108,0x9b370101,0xfb840108,0x9b380101,0xfb840108,0x9b390101,
	0xfb840108,0x9b3a0101,0xfb840108,0x9b3b0101,0xfb840108,0x9b3c0101,0xfb840108,0x9b3d0101,
	0xfb840108,0x9b3e0101,0xfb840108,0x9b3f0101,0xfb840108,0x9b400101,0xfb840108,0x9b410101,
	0xfb840108,0x9b420101,0xfb840108,0x9b430101,0xfb840108,0x9b440101,0xfb840108,0x9b450101,
	0xfb840108,0x9b460101,0xfb840108,0x9b470101,0xfb840108,0x9b480101,0xfb840108,0x9b490101,
	0xfb840108,0x9b4a0101,0xfb840108,0x9b4b0101,0xfb840108,0x9b4c0101,0xfb840108,0x9b4d0101,
	0xfb840108,0x9b4e0101,0xfb840108,0x9b4f0101,0xfb840108,0x9b500101,0xfb840108,0x9b510101,
	0xfb840108,0x9b520101,0xfb840108,0x9b530101,0xfb840108,0x9b540101,0xfb840108,0x9b550101,
	0xfb840108,0x9b560101,0xfb840108,0x9b570101,0xfb840108,0x9b580101,0xfb840108,0x9b590101,
	0xfb840108,0x9b5a0101,0xfb840108,0x9b5b0101,0xfb840108,0x9b5c0101,0xfb840108,0x9b5d0101,
	0xfb840108,0x9b5e0101,0xfb840108,0x9b5f0101,0xfb840108,0x9b600101,0xfb840108,0x9b610101,
	0xfb840108,0x9b620101,0xfb840108,0x9b630101,0xfb840108,0x9b640101,0xfb840108,0x9b650101,
	0xfb840108,0x9b660101,0xfb840108,0x9b670101,0xfb840108,0x9b680101,0xfb840108,0x9b690101,
	0xfb840108,0x9b6a0101,0xfb840108,0x9b6b0101,0xfb840108,0x9b6c0101,0xfb840108,0x9b6d0101,
	0xfb840108,0x9b6e0101,0xfb840108,0x9b6f0101,0xfb840108,0x9b700101,0xfb840108,0x9b710101,
	0xfb840108,0x9b720101,0xfb840108,0x9b730101,0xfb840108,0x9b740101,0xfb840108,0x9b750101,
	0xfb840108,0x9b760101,0xfb840108,0x9b770101,0xfb840108,0x9b780101,0xfb840108,0x9b790101,
	0xfb840108,0x9b7a0101,0xfb840108,0x9b7b0101,0xfb840108,0x9b7c0101,0xfb840108,0x9b7d0101,
	0xfb840108,0x9b7e0101,0xfb840108,0x9b7f0101,0xfb840108,0x9b800101,0xfb840108,0x9b810101,
	0xfb840108,0x9b820101,0xfb840108,0x9b830101,0xfb840108,0x9b840101,0xfb840108,0x9b850101,
	0xfb840108,0x9b860101,0xfb840108,0x9b870101,0xfb840108,0x9b880101,0xfb840108,0x9b890101,
	0xfb840108,0x9b8a0101,0xfb840108,0x9b8b0101,0xfb840108,0x9b8c0101,0xfb840108,0x9b8d0101,
	0xfb840108,0x9b8e0101,0xfb840108,0x9b8f0101,0xfb840108,0x9b900101,0xfb840108,0x9b910101,
	0xfb840108,0x9b920101,0xfb840108,0x9b930101,0xfb840108,0x9b940101,0xfb840108,0x9b950101,
	0xfb840108,0x9b960101,0xfb840108,0x9b970101,0xfb840108,0x9b980101,0xfb840108,0x9b990101,
	0xfb840108,0x9b9a0101,0xfb840108,0x9b9b0101,0xfb840108,0x9b9c0101,0xfb840108,0x9b9d0101,
	0xfb840108,0x9b9e0101,0xfb840108,0x9b9f0101,0xfb840108,0x9ba00101,0xfb840108,0x9ba10101,
	0xfb840108,0x9ba20101,0xfb840108,0x9ba30101,0xfb840108,0x9ba40101,0xfb840108,0x9ba50101,
	0xfb840108,0x9ba60101,0xfb840108,0x9ba70101,0xfb840108,0x9ba80101,0xfb840108,0x9ba90101,
	0xfb840108,0x9baa0101,0xfb840108,0x9bab0101,0xfb840108,0x9bac0101,0xfb840108,0x9bad0101,
	0xfb840108,0x9bae0101,0xfb840108,0x9baf0101,0xfb840108,0x9bb00101,0xfb840108,0x9bb10101,
	0xfb840108,0x9bb20101,0xfb840108,0x9bb30101,0xfb840108,0x9bb40101,0xfb840108,0x9bb50101,
	0xfb840108,0x9bb60101,0xfb840108,0x9bb70101,0xfb840108,0x9bb80101,0xfb840108,0x9bb90101,
	0xfb840108,0x9bba0101,0xfb840108,0x9bbb0101,0xfb840108,0x9bbc0101,0xfb840108,0x9bbd0101,
	0xfb840108,0x9bbe0101,0xfb840108,0x9bbf0101,0xfb840108,0x9bc00101,0xfb840108,0x9bc10101,
	0xfb840108,0x9bc20101,0xfb840108,0x9bc30101,0xfb840108,0x9bc40101,0xfb840108,0x9bc50101,
	0xfb840108,0x9bc60101,0xfb840108,0x9bc70101,0xfb840108,0x9bc80101,0xfb840108,0x9bc90101,
	0xfb840108,0x9bca0101,0xfb840108,0x9bcb0101,0xfb840108,0x9bcc0101,0xfb840108,0x9bcd0101,
	0xfb840108,0x9bce0101,0xfb840108,0x9bcf0101,0xfb840108,0x9bd00101,0xfb840108,0x9bd10101,
	0xfb840108,0x9bd20101,0xfb840108,0x9bd30101,0xfb840108,0x9bd40101,0xfb840108,0x9bd50101,
	0xfb840108,0x9bd60101,0xfb840108,0x9bd70101,0xfb840108,0x9bd80101,0xfb840108,0x9bd90101,
	0xfb840108,0x9bda0101,0xfb840108,0x9bdb0101,0xfb840108,0x9bdc0101,0xfb840108,0x9bdd0101,
	0xfb840108,0x9bde0101,0xfb840108,0x9bdf0101,0xfb840108,0x9be00101,0xfb840108,0x9be10101,
	0xfb840108,0x9be20101,0xfb840108,0x9be30101,0xfb840108,0x9be40101,0xfb840108,0x9be50101,
	0xfb840108,0x9be60101,0xfb840108,0x9be70101,0xfb840108,0x9be80101,0xfb840108,0x9be90101,
	0xfb840108,0x9bea0101,0xfb840108,0x9beb0101,0xfb840108,0x9bec0101,0xfb840108,0x9bed0101,
	0xfb840108,0x9bee0101,0xfb840108,0x9bef0101,0xfb840108,0x9bf00101,0xfb840108,0x9bf10101,
	0xfb840108,0x9bf20101,0xfb840108,0x9bf30101,0xfb840108,0x9bf40101,0xfb840108,0x9bf50101,
	0xfb840108,0x9bf60101,0xfb840108,0x9bf70101,0xfb840108,0x9bf80101,0xfb840108,0x9bf90101,
	0xfb840108,0x9bfa0101,0xfb840108,0x9bfb0101,0xfb840108,0x9bfc0101,0xfb840108,0x9bfd0101,
	0xfb840108,0x9bfe0101,0xfb840108,0x9bff0101,0xfb840108,0x9c000101,0xfb840108,0x9c010101,
	0xfb840108,0x9c020101,0xfb840108,0x9c030101,0xfb840108,0x9c040101,0xfb840108,0x9c050101,
	0xfb840108,0x9c060101,0xfb840108,0x9c070101,0xfb840108,0x9c080101,0xfb840108,0x9c090101,
	0xfb840108,0x9c0a0101,0xfb840108,0x9c0b0101,0xfb840108,0x9c0c0101,0xfb840108,0x9c0d0101,
	0xfb840108,0x9c0e0101,0xfb840108,0x9c0f0101,0xfb840108,0x9c100101,0xfb840108,0x9c110101,
	0xfb840108,0x9c120101,0xfb840108,0x9c130101,0xfb840108,0x9c140101,0xfb840108,0x9c150101,
	0xfb840108,0x9c160101,0xfb840108,0x9c170101,0xfb840108,0x9c180101,0xfb840108,0x9c190101,
	0xfb840108,0x9c1a0101,0xfb840108,0x9c1b0101,0xfb840108,0x9c1c0101,0xfb840108,0x9c1d0101,
	0xfb840108,0x9c1e0101,0xfb840108,0x9c1f0101,0xfb840108,0x9c200101,0xfb840108,0x9c210101,
	0xfb840108,0x9c220101,0xfb840108,0x9c230101,0xfb840108,0x9c240101,0xfb840108,0x9c250101,
	0xfb840108,0x9c260101,0xfb840108,0x9c270101,0xfb840108,0x9c280101,0xfb840108,0x9c290101,
	0xfb840108,0x9c2a0101,0xfb840108,0x9c2b0101,0xfb840108,0x9c2c0101,0xfb840108,0x9c2d0101,
	0xfb840108,0x9c2e0101,0xfb840108,0x9c2f0101,0xfb840108,0x9c300101,0xfb840108,0x9c310101,
	0xfb840108,0x9c320101,0xfb840108,0x9c330101,0xfb840108,0x9c340101,0xfb840108,0x9c350101,
	0xfb840108,0x9c360101,0xfb840108,0x9c370101,0xfb840108,0x9c380101,0xfb840108,0x9c390101,
	0xfb840108,0x9c3a0101,0xfb840108,0x9c3b0101,0xfb840108,0x9c3c0101,0xfb840108,0x9c3d0101,
	0xfb840108,0x9c3e0101,0xfb840108,0x9c3f0101,0xfb840108,0x9c400101,0xfb840108,0x9c410101,
	0xfb840108,0x9c420101,0xfb840108,0x9c430101,0xfb840108,0x9c440101,0xfb840108,0x9c450101,
	0xfb840108,0x9c460101,0xfb840108,0x9c470101,0xfb840108,0x9c480101,0xfb840108,0x9c490101,
	0xfb840108,0x9c4a0101,0xfb840108,0x9c4b0101,0xfb840108,0x9c4c0101,0xfb840108,0x9c4d0101,
	0xfb840108,0x9c4e0101,0xfb840108,0x9c4f0101,0xfb840108,0x9c500101,0xfb840108,0x9c510101,
	0xfb840108,0x9c520101,0xfb840108,0x9c530101,0xfb840108,0x9c540101,0xfb840108,0x9c550101,
	0xfb840108,0x9c560101,0xfb840108,0x9c570101,0xfb840108,0x9c580101,0xfb840108,0x9c590101,
	0xfb840108,0x9c5a0101,0xfb840108,0x9c5b0101,0xfb840108,0x9c5c0101,0xfb840108,0x9c5d0101,
	0xfb840108,0x9c5e0101,0xfb840108,0x9c5f0101,0xfb840108,0x9c600101,0xfb840108,0x9c610101,
	0xfb840108,0x9c620101,0xfb840108,0x9c630101,0xfb840108,0x9c640101,0xfb840108,0x9c650101,
	0xfb840108,0x9c660101,0xfb840108,0x9c670101,0xfb840108,0x9c680101,0xfb840108,0x9c690101,
	0xfb840108,0x9c6a0101,0xfb840108,0x9c6b0101,0xfb840108,0x9c6c0101,0xfb840108,0x9c6d0101,
	0xfb840108,0x9c6e0101,0xfb840108,0x9c6f0101,0xfb840108,0x9c700101,0xfb840108,0x9c710101,
	0xfb840108,0x9c720101,0xfb840108,0x9c730101,0xfb840108,0x9c740101,0xfb840108,0x9c750101,
	0xfb840108,0x9c760101,0xfb840108,0x9c770101,0xfb840108,0x9c780101,0xfb840108,0x9c790101,
	0xfb840108,0x9c7a0101,0xfb840108,0x9c7b0101,0xfb840108,0x9c7c0101,0xfb840108,0x9c7d0101,
	0xfb840108,0x9c7e0101,0xfb840108,0x9c7f0101,0xfb840108,0x9c800101,0xfb840108,0x9c810101,
	0xfb840108,0x9c820101,0xfb840108,0x9c830101,0xfb840108,0x9c840101,0xfb840108,0x9c850101,
	0xfb840108,0x9c860101,0xfb840108,0x9c870101,0xfb840108,0x9c880101,0xfb840108,0x9c890101,
	0xfb840108,0x9c8a0101,0xfb840108,0x9c8b0101,0xfb840108,0x9c8c0101,0xfb840108,0x9c8d0101,
	0xfb840108,0x9c8e0101,0xfb840108,0x9c8f0101,0xfb840108,0x9c900101,0xfb840108,0x9c910101,
	0xfb840108,0x9c920101,0xfb840108,0x9c930101,0xfb840108,0x9c940101,0xfb840108,0x9c950101,
	0xfb840108,0x9c960101,0xfb840108,0x9c970101,0xfb840108,0x9c980101,0xfb840108,0x9c990101,
	0xfb840108,0x9c9a0101,0xfb840108,0x9c9b0101,0xfb840108,0x9c9c0101,0xfb840108,0x9c9d0101,
	0xfb840108,0x9c9e0101,0xfb840108,0x9c9f0101,0xfb840108,0x9ca00101,0xfb840108,0x9ca10101,
	0xfb840108,0x9ca20101,0xfb840108,0x9ca30101,0xfb840108,0x9ca40101,0xfb840108,0x9ca50101,
	0xfb840108,0x9ca60101,0xfb840108,0x9ca70101,0xfb840108,0x9ca80101,0xfb840108,0x9ca90101,
	0xfb840108,0x9caa0101,0xfb840108,0x9cab0101,0xfb840108,0x9cac0101,0xfb840108,0x9cad0101,
	0xfb840108,0x9cae0101,0xfb840108,0x9caf0101,0xfb840108,0x9cb00101,0xfb840108,0x9cb10101,
	0xfb840108,0x9cb20101,0xfb840108,0x9cb30101,0xfb840108,0x9cb40101,0xfb840108,0x9cb50101,
	0xfb840108,0x9cb60101,0xfb840108,0x9cb70101,0xfb840108,0x9cb80101,0xfb840108,0x9cb90101,
	0xfb840108,0x9cba0101,0xfb840108,0x9cbb0101,0xfb840108,0x9cbc0101,0xfb840108,0x9cbd0101,
	0xfb840108,0x9cbe0101,0xfb840108,0x9cbf0101,0xfb840108,0x9cc00101,0xfb840108,0x9cc10101,
	0xfb840108,0x9cc20101,0xfb840108,0x9cc30101,0xfb840108,0x9cc40101,0xfb840108,0x9cc50101,
	0xfb840108,0x9cc60101,0xfb840108,0x9cc70101,0xfb840108,0x9cc80101,0xfb840108,0x9cc90101,
	0xfb840108,0x9cca0101,0xfb840108,0x9ccb0101,0xfb840108,0x9ccc0101,0xfb840108,0x9ccd0101,
	0xfb840108,0x9cce0101,0xfb840108,0x9ccf0101,0xfb840108,0x9cd00101,0xfb840108,0x9cd10101,
	0xfb840108,0x9cd20101,0xfb840108,0x9cd30101,0xfb840108,0x9cd40101,0xfb840108,0x9cd50101,
	0xfb840108,0x9cd60101,0xfb840108,0x9cd70101,0xfb840108,0x9cd80101,0xfb840108,0x9cd90101,
	0xfb840108,0x9cda0101,0xfb840108,0x9cdb0101,0xfb840108,0x9cdc0101,0xfb840108,0x9cdd0101,
	0xfb840108,0x9cde0101,0xfb840108,0x9cdf0101,0xfb840108,0x9ce00101,0xfb840108,0x9ce10101,
	0xfb840108,0x9ce20101,0xfb840108,0x9ce30101,0xfb840108,0x9ce40101,0xfb840108,0x9ce50101,
	0xfb840108,0x9ce60101,0xfb840108,0x9ce70101,0xfb840108,0x9ce80101,0xfb840108,0x9ce90101,
	0xfb840108,0x9cea0101,0xfb840108,0x9ceb0101,0xfb840108,0x9cec0101,0xfb840108,0x9ced0101,
	0xfb840108,0x9cee0101,0xfb840108,0x9cef0101,0xfb840108,0x9cf00101,0xfb840108,0x9cf10101,
	0xfb840108,0x9cf20101,0xfb840108,0x9cf30101,0xfb840108,0x9cf40101,0xfb840108,0x9cf50101,
	0xfb840108,0x9cf60101,0xfb840108,0x9cf70101,0xfb840108,0x9cf80101,0xfb840108,0x9cf90101,
	0xfb840108,0x9cfa0101,0xfb840108,0x9cfb0101,0xfb840108,0x9cfc0101,0xfb840108,0x9cfd0101,
	0xfb840108,0x9cfe0101,0xfb840108,0x9cff0101,0xfb840108,0x9d000101,0xfb840108,0x9d010101,
	0xfb840108,0x9d020101,0xfb840108,0x9d030101,0xfb840108,0x9d040101,0xfb840108,0x9d050101,
	0xfb840108,0x9d060101,0xfb840108,0x9d070101,0xfb840108,0x9d080101,0xfb840108,0x9d090101,
	0xfb840108,0x9d0a0101,0xfb840108,0x9d0b0101,0xfb840108,0x9d0c0101,0xfb840108,0x9d0d0101,
	0xfb840108,0x9d0e0101,0xfb840108,0x9d0f0101,0xfb840108,0x9d100101,0xfb840108,0x9d110101,
	0xfb840108,0x9d120101,0xfb840108,0x9d130101,0xfb840108,0x9d140101,0xfb840108,0x9d150101,
	0xfb840108,0x9d160101,0xfb840108,0x9d170101,0xfb840108,0x9d180101,0xfb840108,0x9d190101,
	0xfb840108,0x9d1a0101,0xfb840108,0x9d1b0101,0xfb840108,0x9d1c0101,0xfb840108,0x9d1d0101,
	0xfb840108,0x9d1e0101,0xfb840108,0x9d1f0101,0xfb840108,0x9d200101,0xfb840108,0x9d210101,
	0xfb840108,0x9d220101,0xfb840108,0x9d230101,0xfb840108,0x9d240101,0xfb840108,0x9d250101,
	0xfb840108,0x9d260101,0xfb840108,0x9d270101,0xfb840108,0x9d280101,0xfb840108,0x9d290101,
	0xfb840108,0x9d2a0101,0xfb840108,0x9d2b0101,0xfb840108,0x9d2c0101,0xfb840108,0x9d2d0101,
	0xfb840108,0x9d2e0101,0xfb840108,0x9d2f0101,0xfb840108,0x9d300101,0xfb840108,0x9d310101,
	0xfb840108,0x9d320101,0xfb840108,0x9d330101,0xfb840108,0x9d340101,0xfb840108,0x9d350101,
	0xfb840108,0x9d360101,0xfb840108,0x9d370101,0xfb840108,0x9d380101,0xfb840108,0x9d390101,
	0xfb840108,0x9d3a0101,0xfb840108,0x9d3b0101,0xfb840108,0x9d3c0101,0xfb840108,0x9d3d0101,
	0xfb840108,0x9d3e0101,0xfb840108,0x9d3f0101,0xfb840108,0x9d400101,0xfb840108,0x9d410101,
	0xfb840108,0x9d420101,0xfb840108,0x9d430101,0xfb840108,0x9d440101,0xfb840108,0x9d450101,
	0xfb840108,0x9d460101,0xfb840108,0x9d470101,0xfb840108,0x9d480101,0xfb840108,0x9d490101,
	0xfb840108,0x9d4a0101,0xfb840108,0x9d4b0101,0xfb840108,0x9d4c0101,0xfb840108,0x9d4d0101,
	0xfb840108,0x9d4e0101,0xfb840108,0x9d4f0101,0xfb840108,0x9d500101,0xfb840108,0x9d510101,
	0xfb840108,0x9d520101,0xfb840108,0x9d530101,0xfb840108,0x9d540101,0xfb840108,0x9d550101,
	0xfb840108,0x9d560101,0xfb840108,0x9d570101,0xfb840108,0x9d580101,0xfb840108,0x9d590101,
	0xfb840108,0x9d5a0101,0xfb840108,0x9d5b0101,0xfb840108,0x9d5c0101,0xfb840108,0x9d5d0101,
	0xfb840108,0x9d5e0101,0xfb840108,0x9d5f0101,0xfb840108,0x9d600101,0xfb840108,0x9d610101,
	0xfb840108,0x9d620101,0xfb840108,0x9d630101,0xfb840108,0x9d640101,0xfb840108,0x9d650101,
	0xfb840108,0x9d660101,0xfb840108,0x9d670101,0xfb840108,0x9d680101,0xfb840108,0x9d690101,
	0xfb840108,0x9d6a0101,0xfb840108,0x9d6b0101,0xfb840108,0x9d6c0101,0xfb840108,0x9d6d0101,
	0xfb840108,0x9d6e0101,0xfb840108,0x9d6f0101,0xfb840108,0x9d700101,0xfb840108,0x9d710101,
	0xfb840108,0x9d720101,0xfb840108,0x9d730101,0xfb840108,0x9d740101,0xfb840108,0x9d750101,
	0xfb840108,0x9d760101,0xfb840108,0x9d770101,0xfb840108,0x9d780101,0xfb840108,0x9d790101,
	0xfb840108,0x9d7a0101,0xfb840108,0x9d7b0101,0xfb840108,0x9d7c0101,0xfb840108,0x9d7d0101,
	0xfb840108,0x9d7e0101,0xfb840108,0x9d7f0101,0xfb840108,0x9d800101,0xfb840108,0x9d810101,
	0xfb840108,0x9d820101,0xfb840108,0x9d830101,0xfb840108,0x9d840101,0xfb840108,0x9d850101,
	0xfb840108,0x9d860101,0xfb840108,0x9d870101,0xfb840108,0x9d880101,0xfb840108,0x9d890101,
	0xfb840108,0x9d8a0101,0xfb840108,0x9d8b0101,0xfb840108,0x9d8c0101,0xfb840108,0x9d8d0101,
	0xfb840108,0x9d8e0101,0xfb840108,0x9d8f0101,0xfb840108,0x9d900101,0xfb840108,0x9d910101,
	0xfb840108,0x9d920101,0xfb840108,0x9d930101,0xfb840108,0x9d940101,0xfb840108,0x9d950101,
	0xfb840108,0x9d960101,0xfb840108,0x9d970101,0xfb840108,0x9d980101,0xfb840108,0x9d990101,
	0xfb840108,0x9d9a0101,0xfb840108,0x9d9b0101,0xfb840108,0x9d9c0101,0xfb840108,0x9d9d0101,
	0xfb840108,0x9d9e0101,0xfb840108,0x9d9f0101,0xfb840108,0x9da00101,0xfb840108,0x9da10101,
	0xfb840108,0x9da20101,0xfb840108,0x9da30101,0xfb840108,0x9da40101,0xfb840108,0x9da50101,
	0xfb840108,0x9da60101,0xfb840108,0x9da70101,0xfb840108,0x9da80101,0xfb840108,0x9da90101,
	0xfb840108,0x9daa0101,0xfb840108,0x9dab0101,0xfb840108,0x9dac0101,0xfb840108,0x9dad0101,
	0xfb840108,0x9dae0101,0xfb840108,0x9daf0101,0xfb840108,0x9db00101,0xfb840108,0x9db10101,
	0xfb840108,0x9db20101,0xfb840108,0x9db30101,0xfb840108,0x9db40101,0xfb840108,0x9db50101,
	0xfb840108,0x9db60101,0xfb840108,0x9db70101,0xfb840108,0x9db80101,0xfb840108,0x9db90101,
	0xfb840108,0x9dba0101,0xfb840108,0x9dbb0101,0xfb840108,0x9dbc0101,0xfb840108,0x9dbd0101,
	0xfb840108,0x9dbe0101,0xfb840108,0x9dbf0101,0xfb840108,0x9dc00101,0xfb840108,0x9dc10101,
	0xfb840108,0x9dc20101,0xfb840108,0x9dc30101,0xfb840108,0x9dc40101,0xfb840108,0x9dc50101,
	0xfb840108,0x9dc60101,0xfb840108,0x9dc70101,0xfb840108,0x9dc80101,0xfb840108,0x9dc90101,
	0xfb840108,0x9dca0101,0xfb840108,0x9dcb0101,0xfb840108,0x9dcc0101,0xfb840108,0x9dcd0101,
	0xfb840108,0x9dce0101,0xfb840108,0x9dcf0101,0xfb840108,0x9dd00101,0xfb840108,0x9dd10101,
	0xfb840108,0x9dd20101,0xfb840108,0x9dd30101,0xfb840108,0x9dd40101,0xfb840108,0x9dd50101,
	0xfb840108,0x9dd60101,0xfb840108,0x9dd70101,0xfb840108,0x9dd80101,0xfb840108,0x9dd90101,
	0xfb840108,0x9dda0101,0xfb840108,0x9ddb0101,0xfb840108,0x9ddc0101,0xfb840108,0x9ddd0101,
	0xfb840108,0x9dde0101,0xfb840108,0x9ddf0101,0xfb840108,0x9de00101,0xfb840108,0x9de10101,
	0xfb840108,0x9de20101,0xfb840108,0x9de30101,0xfb840108,0x9de40101,0xfb840108,0x9de50101,
	0xfb840108,0x9de60101,0xfb840108,0x9de70101,0xfb840108,0x9de80101,0xfb840108,0x9de90101,
	0xfb840108,0x9dea0101,0xfb840108,0x9deb0101,0xfb840108,0x9dec0101,0xfb840108,0x9ded0101,
	0xfb840108,0x9dee0101,0xfb840108,0x9def0101,0xfb840108,0x9df00101,0xfb840108,0x9df10101,
	0xfb840108,0x9df20101,0xfb840108,0x9df30101,0xfb840108,0x9df40101,0xfb840108,0x9df50101,
	0xfb840108,0x9df60101,0xfb840108,0x9df70101,0xfb840108,0x9df80101,0xfb840108,0x9df90101,
	0xfb840108,0x9dfa0101,0xfb840108,0x9dfb0101,0xfb840108,0x9dfc0101,0xfb840108,0x9dfd0101,
	0xfb840108,0x9dfe0101,0xfb840108,0x9dff0101,0xfb840108,0x9e000101,0xfb840108,0x9e010101,
	0xfb840108,0x9e020101,0xfb840108,0x9e030101,0xfb840108,0x9e040101,0xfb840108,0x9e050101,
	0xfb840108,0x9e060101,0xfb840108,0x9e070101,0xfb840108,0x9e080101,0xfb840108,0x9e090101,
	0xfb840108,0x9e0a0101,0xfb840108,0x9e0b0101,0xfb840108,0x9e0c0101,0xfb840108,0x9e0d0101,
	0xfb840108,0x9e0e0101,0xfb840108,0x9e0f0101,0xfb840108,0x9e100101,0xfb840108,0x9e110101,
	0xfb840108,0x9e120101,0xfb840108,0x9e130101,0xfb840108,0x9e140101,0xfb840108,0x9e150101,
	0xfb840108,0x9e160101,0xfb840108,0x9e170101,0xfb840108,0x9e180101,0xfb840108,0x9e190101,
	0xfb840108,0x9e1a0101,0xfb840108,0x9e1b0101,0xfb840108,0x9e1c0101,0xfb840108,0x9e1d0101,
	0xfb840108,0x9e1e0101,0xfb840108,0x9e1f0101,0xfb840108,0x9e200101,0xfb840108,0x9e210101,
	0xfb840108,0x9e220101,0xfb840108,0x9e230101,0xfb840108,0x9e240101,0xfb840108,0x9e250101,
	0xfb840108,0x9e260101,0xfb840108,0x9e270101,0xfb840108,0x9e280101,0xfb840108,0x9e290101,
	0xfb840108,0x9e2a0101,0xfb840108,0x9e2b0101,0xfb840108,0x9e2c0101,0xfb840108,0x9e2d0101,
	0xfb840108,0x9e2e0101,0xfb840108,0x9e2f0101,0xfb840108,0x9e300101,0xfb840108,0x9e310101,
	0xfb840108,0x9e320101,0xfb840108,0x9e330101,0xfb840108,0x9e340101,0xfb840108,0x9e350101,
	0xfb840108,0x9e360101,0xfb840108,0x9e370101,0xfb840108,0x9e380101,0xfb840108,0x9e390101,
	0xfb840108,0x9e3a0101,0xfb840108,0x9e3b0101,0xfb840108,0x9e3c0101,0xfb840108,0x9e3d0101,
	0xfb840108,0x9e3e0101,0xfb840108,0x9e3f0101,0xfb840108,0x9e400101,0xfb840108,0x9e410101,
	0xfb840108,0x9e420101,0xfb840108,0x9e430101,0xfb840108,0x9e440101,0xfb840108,0x9e450101,
	0xfb840108,0x9e460101,0xfb840108,0x9e470101,0xfb840108,0x9e480101,0xfb840108,0x9e490101,
	0xfb840108,0x9e4a0101,0xfb840108,0x9e4b0101,0xfb840108,0x9e4c0101,0xfb840108,0x9e4d0101,
	0xfb840108,0x9e4e0101,0xfb840108,0x9e4f0101,0xfb840108,0x9e500101,0xfb840108,0x9e510101,
	0xfb840108,0x9e520101,0xfb840108,0x9e530101,0xfb840108,0x9e540101,0xfb840108,0x9e550101,
	0xfb840108,0x9e560101,0xfb840108,0x9e570101,0xfb840108,0x9e580101,0xfb840108,0x9e590101,
	0xfb840108,0x9e5a0101,0xfb840108,0x9e5b0101,0xfb840108,0x9e5c0101,0xfb840108,0x9e5d0101,
	0xfb840108,0x9e5e0101,0xfb840108,0x9e5f0101,0xfb840108,0x9e600101,0xfb840108,0x9e610101,
	0xfb840108,0x9e620101,0xfb840108,0x9e630101,0xfb840108,0x9e640101,0xfb840108,0x9e650101,
	0xfb840108,0x9e660101,0xfb840108,0x9e670101,0xfb840108,0x9e680101,0xfb840108,0x9e690101,
	0xfb840108,0x9e6a0101,0xfb840108,0x9e6b0101,0xfb840108,0x9e6c0101,0xfb840108,0x9e6d0101,
	0xfb840108,0x9e6e0101,0xfb840108,0x9e6f0101,0xfb840108,0x9e700101,0xfb840108,0x9e710101,
	0xfb840108,0x9e720101,0xfb840108,0x9e730101,0xfb840108,0x9e740101,0xfb840108,0x9e750101,
	0xfb840108,0x9e760101,0xfb840108,0x9e770101,0xfb840108,0x9e780101,0xfb840108,0x9e790101,
	0xfb840108,0x9e7a0101,0xfb840108,0x9e7b0101,0xfb840108,0x9e7c0101,0xfb840108,0x9e7d0101,
	0xfb840108,0x9e7e0101,0xfb840108,0x9e7f0101,0xfb840108,0x9e800101,0xfb840108,0x9e810101,
	0xfb840108,0x9e820101,0xfb840108,0x9e830101,0xfb840108,0x9e840101,0xfb840108,0x9e850101,
	0xfb840108,0x9e860101,0xfb840108,0x9e870101,0xfb840108,0x9e880101,0xfb840108,0x9e890101,
	0xfb840108,0x9e8a0101,0xfb840108,0x9e8b0101,0xfb840108,0x9e8c0101,0xfb840108,0x9e8d0101,
	0xfb840108,0x9e8e0101,0xfb840108,0x9e8f0101,0xfb840108,0x9e900101,0xfb840108,0x9e910101,
	0xfb840108,0x9e920101,0xfb840108,0x9e930101,0xfb840108,0x9e940101,0xfb840108,0x9e950101,
	0xfb840108,0x9e960101,0xfb840108,0x9e970101,0xfb840108,0x9e980101,0xfb840108,0x9e990101,
	0xfb840108,0x9e9a0101,0xfb840108,0x9e9b0101,0xfb840108,0x9e9c0101,0xfb840108,0x9e9d0101,
	0xfb840108,0x9e9e0101,0xfb840108,0x9e9f0101,0xfb840108,0x9ea00101,0xfb840108,0x9ea10101,
	0xfb840108,0x9ea20101,0xfb840108,0x9ea30101,0xfb840108,0x9ea40101,0xfb840108,0x9ea50101,
	0xfb840108,0x9ea60101,0xfb840108,0x9ea70101,0xfb840108,0x9ea80101,0xfb840108,0x9ea90101,
	0xfb840108,0x9eaa0101,0xfb840108,0x9eab0101,0xfb840108,0x9eac0101,0xfb840108,0x9ead0101,
	0xfb840108,0x9eae0101,0xfb840108,0x9eaf0101,0xfb840108,0x9eb00101,0xfb840108,0x9eb10101,
	0xfb840108,0x9eb20101,0xfb840108,0x9eb30101,0xfb840108,0x9eb40101,0xfb840108,0x9eb50101,
	0xfb840108,0x9eb60101,0xfb840108,0x9eb70101,0xfb840108,0x9eb80101,0xfb840108,0x9eb90101,
	0xfb840108,0x9eba0101,0xfb840108,0x9ebb0101,0xfb840108,0x9ebc0101,0xfb840108,0x9ebd0101,
	0xfb840108,0x9ebe0101,0xfb840108,0x9ebf0101,0xfb840108,0x9ec00101,0xfb840108,0x9ec10101,
	0xfb840108,0x9ec20101,0xfb840108,0x9ec30101,0xfb840108,0x9ec40101,0xfb840108,0x9ec50101,
	0xfb840108,0x9ec60101,0xfb840108,0x9ec70101,0xfb840108,0x9ec80101,0xfb840108,0x9ec90101,
	0xfb840108,0x9eca0101,0xfb840108,0x9ecb0101,0xfb840108,0x9ecc0101,0xfb840108,0x9ecd0101,
	0xfb840108,0x9ece0101,0xfb840108,0x9ecf0101,0xfb840108,0x9ed00101,0xfb840108,0x9ed10101,
	0xfb840108,0x9ed20101,0xfb840108,0x9ed30101,0xfb840108,0x9ed40101,0xfb840108,0x9ed50101,
	0xfb840108,0x9ed60101,0xfb840108,0x9ed70101,0xfb840108,0x9ed80101,0xfb840108,0x9ed90101,
	0xfb840108,0x9eda0101,0xfb840108,0x9edb0101,0xfb840108,0x9edc0101,0xfb840108,0x9edd0101,
	0xfb840108,0x9ede0101,0xfb840108,0x9edf0101,0xfb840108,0x9ee00101,0xfb840108,0x9ee10101,
	0xfb840108,0x9ee20101,0xfb840108,0x9ee30101,0xfb840108,0x9ee40101,0xfb840108,0x9ee50101,
	0xfb840108,0x9ee60101,0xfb840108,0x9ee70101,0xfb840108,0x9ee80101,0xfb840108,0x9ee90101,
	0xfb840108,0x9eea0101,0xfb840108,0x9eeb0101,0xfb840108,0x9eec0101,0xfb840108,0x9eed0101,
	0xfb840108,0x9eee0101,0xfb840108,0x9eef0101,0xfb840108,0x9ef00101,0xfb840108,0x9ef10101,
	0xfb840108,0x9ef20101,0xfb840108,0x9ef30101,0xfb840108,0x9ef40101,0xfb840108,0x9ef50101,
	0xfb840108,0x9ef60101,0xfb840108,0x9ef70101,0xfb840108,0x9ef80101,0xfb840108,0x9ef90101,
	0xfb840108,0x9efa0101,0xfb840108,0x9efb0101,0xfb840108,0x9efc0101,0xfb840108,0x9efd0101,
	0xfb840108,0x9efe0101,0xfb840108,0x9eff0101,0xfb840108,0x9f000101,0xfb840108,0x9f010101,
	0xfb840108,0x9f020101,0xfb840108,0x9f030101,0xfb840108,0x9f040101,0xfb840108,0x9f050101,
	0xfb840108,0x9f060101,0xfb840108,0x9f070101,0xfb840108,0x9f080101,0xfb840108,0x9f090101,
	0xfb840108,0x9f0a0101,0xfb840108,0x9f0b0101,0xfb840108,0x9f0c0101,0xfb840108,0x9f0d0101,
	0xfb840108,0x9f0e0101,0xfb840108,0x9f0f0101,0xfb840108,0x9f100101,0xfb840108,0x9f110101,
	0xfb840108,0x9f120101,0xfb840108,0x9f130101,0xfb840108,0x9f140101,0xfb840108,0x9f150101,
	0xfb840108,0x9f160101,0xfb840108,0x9f170101,0xfb840108,0x9f180101,0xfb840108,0x9f190101,
	0xfb840108,0x9f1a0101,0xfb840108,0x9f1b0101,0xfb840108,0x9f1c0101,0xfb840108,0x9f1d0101,
	0xfb840108,0x9f1e0101,0xfb840108,0x9f1f0101,0xfb840108,0x9f200101,0xfb840108,0x9f210101,
	0xfb840108,0x9f220101,0xfb840108,0x9f230101,0xfb840108,0x9f240101,0xfb840108,0x9f250101,
	0xfb840108,0x9f260101,0xfb840108,0x9f270101,0xfb840108,0x9f280101,0xfb840108,0x9f290101,
	0xfb840108,0x9f2a0101,0xfb840108,0x9f2b0101,0xfb840108,0x9f2c0101,0xfb840108,0x9f2d0101,
	0xfb840108,0x9f2e0101,0xfb840108,0x9f2f0101,0xfb840108,0x9f300101,0xfb840108,0x9f310101,
	0xfb840108,0x9f320101,0xfb840108,0x9f330101,0xfb840108,0x9f340101,0xfb840108,0x9f350101,
	0xfb840108,0x9f360101,0xfb840108,0x9f370101,0xfb840108,0x9f380101,0xfb840108,0x9f390101,
	0xfb840108,0x9f3a0101,0xfb840108,0x9f3b0101,0xfb840108,0x9f3c0101,0xfb840108,0x9f3d0101,
	0xfb840108,0x9f3e0101,0xfb840108,0x9f3f0101,0xfb840108,0x9f400101,0xfb840108,0x9f410101,
	0xfb840108,0x9f420101,0xfb840108,0x9f430101,0xfb840108,0x9f440101,0xfb840108,0x9f450101,
	0xfb840108,0x9f460101,0xfb840108,0x9f470101,0xfb840108,0x9f480101,0xfb840108,0x9f490101,
	0xfb840108,0x9f4a0101,0xfb840108,0x9f4b0101,0xfb840108,0x9f4c0101,0xfb840108,0x9f4d0101,
	0xfb840108,0x9f4e0101,0xfb840108,0x9f4f0101,0xfb840108,0x9f500101,0xfb840108,0x9f510101,
	0xfb840108,0x9f520101,0xfb840108,0x9f530101,0xfb840108,0x9f540101,0xfb840108,0x9f550101,
	0xfb840108,0x9f560101,0xfb840108,0x9f570101,0xfb840108,0x9f580101,0xfb840108,0x9f590101,
	0xfb840108,0x9f5a0101,0xfb840108,0x9f5b0101,0xfb840108,0x9f5c0101,0xfb840108,0x9f5d0101,
	0xfb840108,0x9f5e0101,0xfb840108,0x9f5f0101,0xfb840108,0x9f600101,0xfb840108,0x9f610101,
	0xfb840108,0x9f620101,0xfb840108,0x9f630101,0xfb840108,0x9f640101,0xfb840108,0x9f650101,
	0xfb840108,0x9f660101,0xfb840108,0x9f670101,0xfb840108,0x9f680101,0xfb840108,0x9f690101,
	0xfb840108,0x9f6a0101,0xfb840108,0x9f6b0101,0xfb840108,0x9f6c0101,0xfb840108,0x9f6d0101,
	0xfb840108,0x9f6e0101,0xfb840108,0x9f6f0101,0xfb840108,0x9f700101,0xfb840108,0x9f710101,
	0xfb840108,0x9f720101,0xfb840108,0x9f730101,0xfb840108,0x9f740101,0xfb840108,0x9f750101,
	0xfb840108,0x9f760101,0xfb840108,0x9f770101,0xfb840108,0x9f780101,0xfb840108,0x9f790101,
	0xfb840108,0x9f7a0101,0xfb840108,0x9f7b0101,0xfb840108,0x9f7c0101,0xfb840108,0x9f7d0101,
	0xfb840108,0x9f7e0101,0xfb840108,0x9f7f0101,0xfb840108,0x9f800101,0xfb840108,0x9f810101,
	0xfb840108,0x9f820101,0xfb840108,0x9f830101,0xfb840108,0x9f840101,0xfb840108,0x9f850101,
	0xfb840108,0x9f860101,0xfb840108,0x9f870101,0xfb840108,0x9f880101,0xfb840108,0x9f890101,
	0xfb840108,0x9f8a0101,0xfb840108,0x9f8b0101,0xfb840108,0x9f8c0101,0xfb840108,0x9f8d0101,
	0xfb840108,0x9f8e0101,0xfb840108,0x9f8f0101,0xfb840108,0x9f900101,0xfb840108,0x9f910101,
	0xfb840108,0x9f920101,0xfb840108,0x9f930101,0xfb840108,0x9f940101,0xfb840108,0x9f950101,
	0xfb840108,0x9f960101,0xfb840108,0x9f970101,0xfb840108,0x9f980101,0xfb840108,0x9f990101,
	0xfb840108,0x9f9a0101,0xfb840108,0x9f9b0101,0xfb840108,0x9f9c0101,0xfb840108,0x9f9d0101,
	0xfb840108,0x9f9e0101,0xfb840108,0x9f9f0101,0xfb840108,0x9fa00101,0xfb840108,0x9fa10101,
	0xfb840108,0x9fa20101,0xfb840108,0x9fa30101,0xfb840108,0x9fa40101,0xfb840108,0x9fa50101,
	0xfb840108,0x9fa60101,0xfb840108,0x9fa70101,0xfb840108,0x9fa80101,0xfb840108,0x9fa90101,
	0xfb840108,0x9faa0101,0xfb840108,0x9fab0101,0xfb840108,0x9fac0101,0xfb840108,0x9fad0101,
	0xfb840108,0x9fae0101,0xfb840108,0x9faf0101,0xfb840108,0x9fb00101,0xfb840108,0x9fb10101,
	0xfb840108,0x9fb20101,0xfb840108,0x9fb30101,0xfb840108,0x9fb40101,0xfb840108,0x9fb50101,
	0xfb840108,0x9fb60101,0xfb840108,0x9fb70101,0xfb840108,0x9fb80101,0xfb840108,0x9fb90101,
	0xfb840108,0x9fba0101,0xfb840108,0x9fbb0101,0xfb840108,0x9fbc0101,0xfb840108,0x9fbd0101,
	0xfb840108,0x9fbe0101,0xfb840108,0x9fbf0101,0xfb840108,0x9fc00101,0xfb840108,0x9fc10101,
	0xfb840108,0x9fc20101,0xfb840108,0x9fc30101,0xfb840108,0x9fc40101,0xfb840108,0x9fc50101,
	0xfb840108,0x9fc60101,0xfb840108,0x9fc70101,0xfb840108,0x9fc80101,0xfb840108,0x9fc90101,
	0xfb840108,0x9fca0101,0xfb840108,0x9fcb0101,0xfb840108,0x9fcc0101,0xfb840108,0x9fcd0101,
	0xfb840108,0x9fce0101,0xfb840108,0x9fcf0101,0xfb840108,0x9fd00101,0xfb840108,0x9fd10101,
	0xfb840108,0x9fd20101,0xfb840108,0x9fd30101,0xfb840108,0x9fd40101,0xfb840108,0x9fd50101,
	0xfb840108,0x9fd60101,0xfb840108,0x9fd70101,0xfb840108,0x9fd80101,0xfb840108,0x9fd90101,
	0xfb840108,0x9fda0101,0xfb840108,0x9fdb0101,0xfb840108,0x9fdc0101,0xfb840108,0x9fdd0101,
	0xfb840108,0x9fde0101,0xfb840108,0x9fdf0101,0xfb840108,0x9fe00101,0xfb840108,0x9fe10101,
	0xfb840108,0x9fe20101,0xfb840108,0x9fe30101,0xfb840108,0x9fe40101,0xfb840108,0x9fe50101,
	0xfb840108,0x9fe60101,0xfb840108,0x9fe70101,0xfb840108,0x9fe80101,0xfb840108,0x9fe90101,
	0xfb840108,0x9fea0101,0xfb840108,0x9feb0101,0xfb840108,0x9fec0101,0xfb840108,0x9fed0101,
	0xfb840108,0x9fee0101,0xfb840108,0x9fef0101,0xfb840108,0x9ff00101,0xfb840108,0x9ff10101,
	0xfb840108,0x9ff20101,0xfb840108,0x9ff30101,0xfb840108,0x9ff40101,0xfb840108,0x9ff50101,
	0xfb840108,0x9ff60101,0xfb840108,0x9ff70101,0xfb840108,0x9ff80101,0xfb840108,0x9ff90101,
	0xfb840108,0x9ffa0101,0xfb840108,0x9ffb0101,0xfb840108,0x9ffc0101,0xfb840108,0x9ffd0101,
	0xfb840108,0x9ffe0101,0xfb840108,0x9fff0101,0xfb840108,0xa0000101,0xfb840108,0xa0010101,
	0xfb840108,0xa0020101,0xfb840108,0xa0030101,0xfb840108,0xa0040101,0xfb840108,0xa0050101,
	0xfb840108,0xa0060101,0xfb840108,0xa0070101,0xfb840108,0xa0080101,0xfb840108,0xa0090101,
	0xfb840108,0xa00a0101,0xfb840108,0xa00b0101,0xfb840108,0xa00c0101,0xfb840108,0xa00d0101,
	0xfb840108,0xa00e0101,0xfb840108,0xa00f0101,0xfb840108,0xa0100101,0xfb840108,0xa0110101,
	0xfb840108,0xa0120101,0xfb840108,0xa0130101,0xfb840108,0xa0140101,0xfb840108,0xa0150101,
	0xfb840108,0xa0160101,0xfb840108,0xa0170101,0xfb840108,0xa0180101,0xfb840108,0xa0190101,
	0xfb840108,0xa01a0101,0xfb840108,0xa01b0101,0xfb840108,0xa01c0101,0xfb840108,0xa01d0101,
	0xfb840108,0xa01e0101,0xfb840108,0xa01f0101,0xfb840108,0xa0200101,0xfb840108,0xa0210101,
	0xfb840108,0xa0220101,0xfb840108,0xa0230101,0xfb840108,0xa0240101,0xfb840108,0xa0250101,
	0xfb840108,0xa0260101,0xfb840108,0xa0270101,0xfb840108,0xa0280101,0xfb840108,0xa0290101,
	0xfb840108,0xa02a0101,0xfb840108,0xa02b0101,0xfb840108,0xa02c0101,0xfb840108,0xa02d0101,
	0xfb840108,0xa02e0101,0xfb840108,0xa02f0101,0xfb840108,0xa0300101,0xfb840108,0xa0310101,
	0xfb840108,0xa0320101,0xfb840108,0xa0330101,0xfb840108,0xa0340101,0xfb840108,0xa0350101,
	0xfb840108,0xa0360101,0xfb840108,0xa0370101,0xfb840108,0xa0380101,0xfb840108,0xa0390101,
	0xfb840108,0xa03a0101,0xfb840108,0xa03b0101,0xfb840108,0xa03c0101,0xfb840108,0xa03d0101,
	0xfb840108,0xa03e0101,0xfb840108,0xa03f0101,0xfb840108,0xa0400101,0xfb840108,0xa0410101,
	0xfb840108,0xa0420101,0xfb840108,0xa0430101,0xfb840108,0xa0440101,0xfb840108,0xa0450101,
	0xfb840108,0xa0460101,0xfb840108,0xa0470101,0xfb840108,0xa0480101,0xfb840108,0xa0490101,
	0xfb840108,0xa04a0101,0xfb840108,0xa04b0101,0xfb840108,0xa04c0101,0xfb840108,0xa04d0101,
	0xfb840108,0xa04e0101,0xfb840108,0xa04f0101,0xfb840108,0xa0500101,0xfb840108,0xa0510101,
	0xfb840108,0xa0520101,0xfb840108,0xa0530101,0xfb840108,0xa0540101,0xfb840108,0xa0550101,
	0xfb840108,0xa0560101,0xfb840108,0xa0570101,0xfb840108,0xa0580101,0xfb840108,0xa0590101,
	0xfb840108,0xa05a0101,0xfb840108,0xa05b0101,0xfb840108,0xa05c0101,0xfb840108,0xa05d0101,
	0xfb840108,0xa05e0101,0xfb840108,0xa05f0101,0xfb840108,0xa0600101,0xfb840108,0xa0610101,
	0xfb840108,0xa0620101,0xfb840108,0xa0630101,0xfb840108,0xa0640101,0xfb840108,0xa0650101,
	0xfb840108,0xa0660101,0xfb840108,0xa0670101,0xfb840108,0xa0680101,0xfb840108,0xa0690101,
	0xfb840108,0xa06a0101,0xfb840108,0xa06b0101,0xfb840108,0xa06c0101,0xfb840108,0xa06d0101,
	0xfb840108,0xa06e0101,0xfb840108,0xa06f0101,0xfb840108,0xa0700101,0xfb840108,0xa0710101,
	0xfb840108,0xa0720101,0xfb840108,0xa0730101,0xfb840108,0xa0740101,0xfb840108,0xa0750101,
	0xfb840108,0xa0760101,0xfb840108,0xa0770101,0xfb840108,0xa0780101,0xfb840108,0xa0790101,
	0xfb840108,0xa07a0101,0xfb840108,0xa07b0101,0xfb840108,0xa07c0101,0xfb840108,0xa07d0101,
	0xfb840108,0xa07e0101,0xfb840108,0xa07f0101,0xfb840108,0xa0800101,0xfb840108,0xa0810101,
	0xfb840108,0xa0820101,0xfb840108,0xa0830101,0xfb840108,0xa0840101,0xfb840108,0xa0850101,
	0xfb840108,0xa0860101,0xfb840108,0xa0870101,0xfb840108,0xa0880101,0xfb840108,0xa0890101,
	0xfb840108,0xa08a0101,0xfb840108,0xa08b0101,0xfb840108,0xa08c0101,0xfb840108,0xa08d0101,
	0xfb840108,0xa08e0101,0xfb840108,0xa08f0101,0xfb840108,0xa0900101,0xfb840108,0xa0910101,
	0xfb840108,0xa0920101,0xfb840108,0xa0930101,0xfb840108,0xa0940101,0xfb840108,0xa0950101,
	0xfb840108,0xa0960101,0xfb840108,0xa0970101,0xfb840108,0xa0980101,0xfb840108,0xa0990101,
	0xfb840108,0xa09a0101,0xfb840108,0xa09b0101,0xfb840108,0xa09c0101,0xfb840108,0xa09d0101,
	0xfb840108,0xa09e0101,0xfb840108,0xa09f0101,0xfb840108,0xa0a00101,0xfb840108,0xa0a10101,
	0xfb840108,0xa0a20101,0xfb840108,0xa0a30101,0xfb840108,0xa0a40101,0xfb840108,0xa0a50101,
	0xfb840108,0xa0a60101,0xfb840108,0xa0a70101,0xfb840108,0xa0a80101,0xfb840108,0xa0a90101,
	0xfb840108,0xa0aa0101,0xfb840108,0xa0ab0101,0xfb840108,0xa0ac0101,0xfb840108,0xa0ad0101,
	0xfb840108,0xa0ae0101,0xfb840108,0xa0af0101,0xfb840108,0xa0b00101,0xfb840108,0xa0b10101,
	0xfb840108,0xa0b20101,0xfb840108,0xa0b30101,0xfb840108,0xa0b40101,0xfb840108,0xa0b50101,
	0xfb840108,0xa0b60101,0xfb840108,0xa0b70101,0xfb840108,0xa0b80101,0xfb840108,0xa0b90101,
	0xfb840108,0xa0ba0101,0xfb840108,0xa0bb0101,0xfb840108,0xa0bc0101,0xfb840108,0xa0bd0101,
	0xfb840108,0xa0be0101,0xfb840108,0xa0bf0101,0xfb840108,0xa0c00101,0xfb840108,0xa0c10101,
	0xfb840108,0xa0c20101,0xfb840108,0xa0c30101,0xfb840108,0xa0c40101,0xfb840108,0xa0c50101,
	0xfb840108,0xa0c60101,0xfb840108,0xa0c70101,0xfb840108,0xa0c80101,0xfb840108,0xa0c90101,
	0xfb840108,0xa0ca0101,0xfb840108,0xa0cb0101,0xfb840108,0xa0cc0101,0xfb840108,0xa0cd0101,
	0xfb840108,0xa0ce0101,0xfb840108,0xa0cf0101,0xfb840108,0xa0d00101,0xfb840108,0xa0d10101,
	0xfb840108,0xa0d20101,0xfb840108,0xa0d30101,0xfb840108,0xa0d40101,0xfb840108,0xa0d50101,
	0xfb840108,0xa0d60101,0xfb840108,0xa0d70101,0xfb840108,0xa0d80101,0xfb840108,0xa0d90101,
	0xfb840108,0xa0da0101,0xfb840108,0xa0db0101,0xfb840108,0xa0dc0101,0xfb840108,0xa0dd0101,
	0xfb840108,0xa0de0101,0xfb840108,0xa0df0101,0xfb840108,0xa0e00101,0xfb840108,0xa0e10101,
	0xfb840108,0xa0e20101,0xfb840108,0xa0e30101,0xfb840108,0xa0e40101,0xfb840108,0xa0e50101,
	0xfb840108,0xa0e60101,0xfb840108,0xa0e70101,0xfb840108,0xa0e80101,0xfb840108,0xa0e90101,
	0xfb840108,0xa0ea0101,0xfb840108,0xa0eb0101,0xfb840108,0xa0ec0101,0xfb840108,0xa0ed0101,
	0xfb840108,0xa0ee0101,0xfb840108,0xa0ef0101,0xfb840108,0xa0f00101,0xfb840108,0xa0f10101,
	0xfb840108,0xa0f20101,0xfb840108,0xa0f30101,0xfb840108,0xa0f40101,0xfb840108,0xa0f50101,
	0xfb840108,0xa0f60101,0xfb840108,0xa0f70101,0xfb840108,0xa0f80101,0xfb840108,0xa0f90101,
	0xfb840108,0xa0fa0101,0xfb840108,0xa0fb0101,0xfb840108,0xa0fc0101,0xfb840108,0xa0fd0101,
	0xfb840108,0xa0fe0101,0xfb840108,0xa0ff0101,0xfb840108,0xa1000101,0xfb840108,0xa1010101,
	0xfb840108,0xa1020101,0xfb840108,0xa1030101,0xfb840108,0xa1040101,0xfb840108,0xa1050101,
	0xfb840108,0xa1060101,0xfb840108,0xa1070101,0xfb840108,0xa1080101,0xfb840108,0xa1090101,
	0xfb840108,0xa10a0101,0xfb840108,0xa10b0101,0xfb840108,0xa10c0101,0xfb840108,0xa10d0101,
	0xfb840108,0xa10e0101,0xfb840108,0xa10f0101,0xfb840108,0xa1100101,0xfb840108,0xa1110101,
	0xfb840108,0xa1120101,0xfb840108,0xa1130101,0xfb840108,0xa1140101,0xfb840108,0xa1150101,
	0xfb840108,0xa1160101,0xfb840108,0xa1170101,0xfb840108,0xa1180101,0xfb840108,0xa1190101,
	0xfb840108,0xa11a0101,0xfb840108,0xa11b0101,0xfb840108,0xa11c0101,0xfb840108,0xa11d0101,
	0xfb840108,0xa11e0101,0xfb840108,0xa11f0101,0xfb840108,0xa1200101,0xfb840108,0xa1210101,
	0xfb840108,0xa1220101,0xfb840108,0xa1230101,0xfb840108,0xa1240101,0xfb840108,0xa1250101,
	0xfb840108,0xa1260101,0xfb840108,0xa1270101,0xfb840108,0xa1280101,0xfb840108,0xa1290101,
	0xfb840108,0xa12a0101,0xfb840108,0xa12b0101,0xfb840108,0xa12c0101,0xfb840108,0xa12d0101,
	0xfb840108,0xa12e0101,0xfb840108,0xa12f0101,0xfb840108,0xa1300101,0xfb840108,0xa1310101,
	0xfb840108,0xa1320101,0xfb840108,0xa1330101,0xfb840108,0xa1340101,0xfb840108,0xa1350101,
	0xfb840108,0xa1360101,0xfb840108,0xa1370101,0xfb840108,0xa1380101,0xfb840108,0xa1390101,
	0xfb840108,0xa13a0101,0xfb840108,0xa13b0101,0xfb840108,0xa13c0101,0xfb840108,0xa13d0101,
	0xfb840108,0xa13e0101,0xfb840108,0xa13f0101,0xfb840108,0xa1400101,0xfb840108,0xa1410101,
	0xfb840108,0xa1420101,0xfb840108,0xa1430101,0xfb840108,0xa1440101,0xfb840108,0xa1450101,
	0xfb840108,0xa1460101,0xfb840108,0xa1470101,0xfb840108,0xa1480101,0xfb840108,0xa1490101,
	0xfb840108,0xa14a0101,0xfb840108,0xa14b0101,0xfb840108,0xa14c0101,0xfb840108,0xa14d0101,
	0xfb840108,0xa14e0101,0xfb840108,0xa14f0101,0xfb840108,0xa1500101,0xfb840108,0xa1510101,
	0xfb840108,0xa1520101,0xfb840108,0xa1530101,0xfb840108,0xa1540101,0xfb840108,0xa1550101,
	0xfb840108,0xa1560101,0xfb840108,0xa1570101,0xfb840108,0xa1580101,0xfb840108,0xa1590101,
	0xfb840108,0xa15a0101,0xfb840108,0xa15b0101,0xfb840108,0xa15c0101,0xfb840108,0xa15d0101,
	0xfb840108,0xa15e0101,0xfb840108,0xa15f0101,0xfb840108,0xa1600101,0xfb840108,0xa1610101,
	0xfb840108,0xa1620101,0xfb840108,0xa1630101,0xfb840108,0xa1640101,0xfb840108,0xa1650101,
	0xfb840108,0xa1660101,0xfb840108,0xa1670101,0xfb840108,0xa1680101,0xfb840108,0xa1690101,
	0xfb840108,0xa16a0101,0xfb840108,0xa16b0101,0xfb840108,0xa16c0101,0xfb840108,0xa16d0101,
	0xfb840108,0xa16e0101,0xfb840108,0xa16f0101,0xfb840108,0xa1700101,0xfb840108,0xa1710101,
	0xfb840108,0xa1720101,0xfb840108,0xa1730101,0xfb840108,0xa1740101,0xfb840108,0xa1750101,
	0xfb840108,0xa1760101,0xfb840108,0xa1770101,0xfb840108,0xa1780101,0xfb840108,0xa1790101,
	0xfb840108,0xa17a0101,0xfb840108,0xa17b0101,0xfb840108,0xa17c0101,0xfb840108,0xa17d0101,
	0xfb840108,0xa17e0101,0xfb840108,0xa17f0101,0xfb840108,0xa1800101,0xfb840108,0xa1810101,
	0xfb840108,0xa1820101,0xfb840108,0xa1830101,0xfb840108,0xa1840101,0xfb840108,0xa1850101,
	0xfb840108,0xa1860101,0xfb840108,0xa1870101,0xfb840108,0xa1880101,0xfb840108,0xa1890101,
	0xfb840108,0xa18a0101,0xfb840108,0xa18b0101,0xfb840108,0xa18c0101,0xfb840108,0xa18d0101,
	0xfb840108,0xa18e0101,0xfb840108,0xa18f0101,0xfb840108,0xa1900101,0xfb840108,0xa1910101,
	0xfb840108,0xa1920101,0xfb840108,0xa1930101,0xfb840108,0xa1940101,0xfb840108,0xa1950101,
	0xfb840108,0xa1960101,0xfb840108,0xa1970101,0xfb840108,0xa1980101,0xfb840108,0xa1990101,
	0xfb840108,0xa19a0101,0xfb840108,0xa19b0101,0xfb840108,0xa19c0101,0xfb840108,0xa19d0101,
	0xfb840108,0xa19e0101,0xfb840108,0xa19f0101,0xfb840108,0xa1a00101,0xfb840108,0xa1a10101,
	0xfb840108,0xa1a20101,0xfb840108,0xa1a30101,0xfb840108,0xa1a40101,0xfb840108,0xa1a50101,
	0xfb840108,0xa1a60101,0xfb840108,0xa1a70101,0xfb840108,0xa1a80101,0xfb840108,0xa1a90101,
	0xfb840108,0xa1aa0101,0xfb840108,0xa1ab0101,0xfb840108,0xa1ac0101,0xfb840108,0xa1ad0101,
	0xfb840108,0xa1ae0101,0xfb840108,0xa1af0101,0xfb840108,0xa1b00101,0xfb840108,0xa1b10101,
	0xfb840108,0xa1b20101,0xfb840108,0xa1b30101,0xfb840108,0xa1b40101,0xfb840108,0xa1b50101,
	0xfb840108,0xa1b60101,0xfb840108,0xa1b70101,0xfb840108,0xa1b80101,0xfb840108,0xa1b90101,
	0xfb840108,0xa1ba0101,0xfb840108,0xa1bb0101,0xfb840108,0xa1bc0101,0xfb840108,0xa1bd0101,
	0xfb840108,0xa1be0101,0xfb840108,0xa1bf0101,0xfb840108,0xa1c00101,0xfb840108,0xa1c10101,
	0xfb840108,0xa1c20101,0xfb840108,0xa1c30101,0xfb840108,0xa1c40101,0xfb840108,0xa1c50101,
	0xfb840108,0xa1c60101,0xfb840108,0xa1c70101,0xfb840108,0xa1c80101,0xfb840108,0xa1c90101,
	0xfb840108,0xa1ca0101,0xfb840108,0xa1cb0101,0xfb840108,0xa1cc0101,0xfb840108,0xa1cd0101,
	0xfb840108,0xa1ce0101,0xfb840108,0xa1cf0101,0xfb840108,0xa1d00101,0xfb840108,0xa1d10101,
	0xfb840108,0xa1d20101,0xfb840108,0xa1d30101,0xfb840108,0xa1d40101,0xfb840108,0xa1d50101,
	0xfb840108,0xa1d60101,0xfb840108,0xa1d70101,0xfb840108,0xa1d80101,0xfb840108,0xa1d90101,
	0xfb840108,0xa1da0101,0xfb840108,0xa1db0101,0xfb840108,0xa1dc0101,0xfb840108,0xa1dd0101,
	0xfb840108,0xa1de0101,0xfb840108,0xa1df0101,0xfb840108,0xa1e00101,0xfb840108,0xa1e10101,
	0xfb840108,0xa1e20101,0xfb840108,0xa1e30101,0xfb840108,0xa1e40101,0xfb840108,0xa1e50101,
	0xfb840108,0xa1e60101,0xfb840108,0xa1e70101,0xfb840108,0xa1e80101,0xfb840108,0xa1e90101,
	0xfb840108,0xa1ea0101,0xfb840108,0xa1eb0101,0xfb840108,0xa1ec0101,0xfb840108,0xa1ed0101,
	0xfb840108,0xa1ee0101,0xfb840108,0xa1ef0101,0xfb840108,0xa1f00101,0xfb840108,0xa1f10101,
	0xfb840108,0xa1f20101,0xfb840108,0xa1f30101,0xfb840108,0xa1f40101,0xfb840108,0xa1f50101,
	0xfb840108,0xa1f60101,0xfb840108,0xa1f70101,0xfb840108,0xa1f80101,0xfb840108,0xa1f90101,
	0xfb840108,0xa1fa0101,0xfb840108,0xa1fb0101,0xfb840108,0xa1fc0101,0xfb840108,0xa1fd0101,
	0xfb840108,0xa1fe0101,0xfb840108,0xa1ff0101,0xfb840108,0xa2000101,0xfb840108,0xa2010101,
	0xfb840108,0xa2020101,0xfb840108,0xa2030101,0xfb840108,0xa2040101,0xfb840108,0xa2050101,
	0xfb840108,0xa2060101,0xfb840108,0xa2070101,0xfb840108,0xa2080101,0xfb840108,0xa2090101,
	0xfb840108,0xa20a0101,0xfb840108,0xa20b0101,0xfb840108,0xa20c0101,0xfb840108,0xa20d0101,
	0xfb840108,0xa20e0101,0xfb840108,0xa20f0101,0xfb840108,0xa2100101,0xfb840108,0xa2110101,
	0xfb840108,0xa2120101,0xfb840108,0xa2130101,0xfb840108,0xa2140101,0xfb840108,0xa2150101,
	0xfb840108,0xa2160101,0xfb840108,0xa2170101,0xfb840108,0xa2180101,0xfb840108,0xa2190101,
	0xfb840108,0xa21a0101,0xfb840108,0xa21b0101,0xfb840108,0xa21c0101,0xfb840108,0xa21d0101,
	0xfb840108,0xa21e0101,0xfb840108,0xa21f0101,0xfb840108,0xa2200101,0xfb840108,0xa2210101,
	0xfb840108,0xa2220101,0xfb840108,0xa2230101,0xfb840108,0xa2240101,0xfb840108,0xa2250101,
	0xfb840108,0xa2260101,0xfb840108,0xa2270101,0xfb840108,0xa2280101,0xfb840108,0xa2290101,
	0xfb840108,0xa22a0101,0xfb840108,0xa22b0101,0xfb840108,0xa22c0101,0xfb840108,0xa22d0101,
	0xfb840108,0xa22e0101,0xfb840108,0xa22f0101,0xfb840108,0xa2300101,0xfb840108,0xa2310101,
	0xfb840108,0xa2320101,0xfb840108,0xa2330101,0xfb840108,0xa2340101,0xfb840108,0xa2350101,
	0xfb840108,0xa2360101,0xfb840108,0xa2370101,0xfb840108,0xa2380101,0xfb840108,0xa2390101,
	0xfb840108,0xa23a0101,0xfb840108,0xa23b0101,0xfb840108,0xa23c0101,0xfb840108,0xa23d0101,
	0xfb840108,0xa23e0101,0xfb840108,0xa23f0101,0xfb840108,0xa2400101,0xfb840108,0xa2410101,
	0xfb840108,0xa2420101,0xfb840108,0xa2430101,0xfb840108,0xa2440101,0xfb840108,0xa2450101,
	0xfb840108,0xa2460101,0xfb840108,0xa2470101,0xfb840108,0xa2480101,0xfb840108,0xa2490101,
	0xfb840108,0xa24a0101,0xfb840108,0xa24b0101,0xfb840108,0xa24c0101,0xfb840108,0xa24d0101,
	0xfb840108,0xa24e0101,0xfb840108,0xa24f0101,0xfb840108,0xa2500101,0xfb840108,0xa2510101,
	0xfb840108,0xa2520101,0xfb840108,0xa2530101,0xfb840108,0xa2540101,0xfb840108,0xa2550101,
	0xfb840108,0xa2560101,0xfb840108,0xa2570101,0xfb840108,0xa2580101,0xfb840108,0xa2590101,
	0xfb840108,0xa25a0101,0xfb840108,0xa25b0101,0xfb840108,0xa25c0101,0xfb840108,0xa25d0101,
	0xfb840108,0xa25e0101,0xfb840108,0xa25f0101,0xfb840108,0xa2600101,0xfb840108,0xa2610101,
	0xfb840108,0xa2620101,0xfb840108,0xa2630101,0xfb840108,0xa2640101,0xfb840108,0xa2650101,
	0xfb840108,0xa2660101,0xfb840108,0xa2670101,0xfb840108,0xa2680101,0xfb840108,0xa2690101,
	0xfb840108,0xa26a0101,0xfb840108,0xa26b0101,0xfb840108,0xa26c0101,0xfb840108,0xa26d0101,
	0xfb840108,0xa26e0101,0xfb840108,0xa26f0101,0xfb840108,0xa2700101,0xfb840108,0xa2710101,
	0xfb840108,0xa2720101,0xfb840108,0xa2730101,0xfb840108,0xa2740101,0xfb840108,0xa2750101,
	0xfb840108,0xa2760101,0xfb840108,0xa2770101,0xfb840108,0xa2780101,0xfb840108,0xa2790101,
	0xfb840108,0xa27a0101,0xfb840108,0xa27b0101,0xfb840108,0xa27c0101,0xfb840108,0xa27d0101,
	0xfb840108,0xa27e0101,0xfb840108,0xa27f0101,0xfb840108,0xa2800101,0xfb840108,0xa2810101,
	0xfb840108,0xa2820101,0xfb840108,0xa2830101,0xfb840108,0xa2840101,0xfb840108,0xa2850101,
	0xfb840108,0xa2860101,0xfb840108,0xa2870101,0xfb840108,0xa2880101,0xfb840108,0xa2890101,
	0xfb840108,0xa28a0101,0xfb840108,0xa28b0101,0xfb840108,0xa28c0101,0xfb840108,0xa28d0101,
	0xfb840108,0xa28e0101,0xfb840108,0xa28f0101,0xfb840108,0xa2900101,0xfb840108,0xa2910101,
	0xfb840108,0xa2920101,0xfb840108,0xa2930101,0xfb840108,0xa2940101,0xfb840108,0xa2950101,
	0xfb840108,0xa2960101,0xfb840108,0xa2970101,0xfb840108,0xa2980101,0xfb840108,0xa2990101,
	0xfb840108,0xa29a0101,0xfb840108,0xa29b0101,0xfb840108,0xa29c0101,0xfb840108,0xa29d0101,
	0xfb840108,0xa29e0101,0xfb840108,0xa29f0101,0xfb840108,0xa2a00101,0xfb840108,0xa2a10101,
	0xfb840108,0xa2a20101,0xfb840108,0xa2a30101,0xfb840108,0xa2a40101,0xfb840108,0xa2a50101,
	0xfb840108,0xa2a60101,0xfb840108,0xa2a70101,0xfb840108,0xa2a80101,0xfb840108,0xa2a90101,
	0xfb840108,0xa2aa0101,0xfb840108,0xa2ab0101,0xfb840108,0xa2ac0101,0xfb840108,0xa2ad0101,
	0xfb840108,0xa2ae0101,0xfb840108,0xa2af0101,0xfb840108,0xa2b00101,0xfb840108,0xa2b10101,
	0xfb840108,0xa2b20101,0xfb840108,0xa2b30101,0xfb840108,0xa2b40101,0xfb840108,0xa2b50101,
	0xfb840108,0xa2b60101,0xfb840108,0xa2b70101,0xfb840108,0xa2b80101,0xfb840108,0xa2b90101,
	0xfb840108,0xa2ba0101,0xfb840108,0xa2bb0101,0xfb840108,0xa2bc0101,0xfb840108,0xa2bd0101,
	0xfb840108,0xa2be0101,0xfb840108,0xa2bf0101,0xfb840108,0xa2c00101,0xfb840108,0xa2c10101,
	0xfb840108,0xa2c20101,0xfb840108,0xa2c30101,0xfb840108,0xa2c40101,0xfb840108,0xa2c50101,
	0xfb840108,0xa2c60101,0xfb840108,0xa2c70101,0xfb840108,0xa2c80101,0xfb840108,0xa2c90101,
	0xfb840108,0xa2ca0101,0xfb840108,0xa2cb0101,0xfb840108,0xa2cc0101,0xfb840108,0xa2cd0101,
	0xfb840108,0xa2ce0101,0xfb840108,0xa2cf0101,0xfb840108,0xa2d00101,0xfb840108,0xa2d10101,
	0xfb840108,0xa2d20101,0xfb840108,0xa2d30101,0xfb840108,0xa2d40101,0xfb840108,0xa2d50101,
	0xfb840108,0xa2d60101,0xfb840108,0xa2d70101,0xfb840108,0xa2d80101,0xfb840108,0xa2d90101,
	0xfb840108,0xa2da0101,0xfb840108,0xa2db0101,0xfb840108,0xa2dc0101,0xfb840108,0xa2dd0101,
	0xfb840108,0xa2de0101,0xfb840108,0xa2df0101,0xfb840108,0xa2e00101,0xfb840108,0xa2e10101,
	0xfb840108,0xa2e20101,0xfb840108,0xa2e30101,0xfb840108,0xa2e40101,0xfb840108,0xa2e50101,
	0xfb840108,0xa2e60101,0xfb840108,0xa2e70101,0xfb840108,0xa2e80101,0xfb840108,0xa2e90101,
	0xfb840108,0xa2ea0101,0xfb840108,0xa2eb0101,0xfb840108,0xa2ec0101,0xfb840108,0xa2ed0101,
	0xfb840108,0xa2ee0101,0xfb840108,0xa2ef0101,0xfb840108,0xa2f00101,0xfb840108,0xa2f10101,
	0xfb840108,0xa2f20101,0xfb840108,0xa2f30101,0xfb840108,0xa2f40101,0xfb840108,0xa2f50101,
	0xfb840108,0xa2f60101,0xfb840108,0xa2f70101,0xfb840108,0xa2f80101,0xfb840108,0xa2f90101,
	0xfb840108,0xa2fa0101,0xfb840108,0xa2fb0101,0xfb840108,0xa2fc0101,0xfb840108,0xa2fd0101,
	0xfb840108,0xa2fe0101,0xfb840108,0xa2ff0101,0xfb840108,0xa3000101,0xfb840108,0xa3010101,
	0xfb840108,0xa3020101,0xfb840108,0xa3030101,0xfb840108,0xa3040101,0xfb840108,0xa3050101,
	0xfb840108,0xa3060101,0xfb840108,0xa3070101,0xfb840108,0xa3080101,0xfb840108,0xa3090101,
	0xfb840108,0xa30a0101,0xfb840108,0xa30b0101,0xfb840108,0xa30c0101,0xfb840108,0xa30d0101,
	0xfb840108,0xa30e0101,0xfb840108,0xa30f0101,0xfb840108,0xa3100101,0xfb840108,0xa3110101,
	0xfb840108,0xa3120101,0xfb840108,0xa3130101,0xfb840108,0xa3140101,0xfb840108,0xa3150101,
	0xfb840108,0xa3160101,0xfb840108,0xa3170101,0xfb840108,0xa3180101,0xfb840108,0xa3190101,
	0xfb840108,0xa31a0101,0xfb840108,0xa31b0101,0xfb840108,0xa31c0101,0xfb840108,0xa31d0101,
	0xfb840108,0xa31e0101,0xfb840108,0xa31f0101,0xfb840108,0xa3200101,0xfb840108,0xa3210101,
	0xfb840108,0xa3220101,0xfb840108,0xa3230101,0xfb840108,0xa3240101,0xfb840108,0xa3250101,
	0xfb840108,0xa3260101,0xfb840108,0xa3270101,0xfb840108,0xa3280101,0xfb840108,0xa3290101,
	0xfb840108,0xa32a0101,0xfb840108,0xa32b0101,0xfb840108,0xa32c0101,0xfb840108,0xa32d0101,
	0xfb840108,0xa32e0101,0xfb840108,0xa32f0101,0xfb840108,0xa3300101,0xfb840108,0xa3310101,
	0xfb840108,0xa3320101,0xfb840108,0xa3330101,0xfb840108,0xa3340101,0xfb840108,0xa3350101,
	0xfb840108,0xa3360101,0xfb840108,0xa3370101,0xfb840108,0xa3380101,0xfb840108,0xa3390101,
	0xfb840108,0xa33a0101,0xfb840108,0xa33b0101,0xfb840108,0xa33c0101,0xfb840108,0xa33d0101,
	0xfb840108,0xa33e0101,0xfb840108,0xa33f0101,0xfb840108,0xa3400101,0xfb840108,0xa3410101,
	0xfb840108,0xa3420101,0xfb840108,0xa3430101,0xfb840108,0xa3440101,0xfb840108,0xa3450101,
	0xfb840108,0xa3460101,0xfb840108,0xa3470101,0xfb840108,0xa3480101,0xfb840108,0xa3490101,
	0xfb840108,0xa34a0101,0xfb840108,0xa34b0101,0xfb840108,0xa34c0101,0xfb840108,0xa34d0101,
	0xfb840108,0xa34e0101,0xfb840108,0xa34f0101,0xfb840108,0xa3500101,0xfb840108,0xa3510101,
	0xfb840108,0xa3520101,0xfb840108,0xa3530101,0xfb840108,0xa3540101,0xfb840108,0xa3550101,
	0xfb840108,0xa3560101,0xfb840108,0xa3570101,0xfb840108,0xa3580101,0xfb840108,0xa3590101,
	0xfb840108,0xa35a0101,0xfb840108,0xa35b0101,0xfb840108,0xa35c0101,0xfb840108,0xa35d0101,
	0xfb840108,0xa35e0101,0xfb840108,0xa35f0101,0xfb840108,0xa3600101,0xfb840108,0xa3610101,
	0xfb840108,0xa3620101,0xfb840108,0xa3630101,0xfb840108,0xa3640101,0xfb840108,0xa3650101,
	0xfb840108,0xa3660101,0xfb840108,0xa3670101,0xfb840108,0xa3680101,0xfb840108,0xa3690101,
	0xfb840108,0xa36a0101,0xfb840108,0xa36b0101,0xfb840108,0xa36c0101,0xfb840108,0xa36d0101,
	0xfb840108,0xa36e0101,0xfb840108,0xa36f0101,0xfb840108,0xa3700101,0xfb840108,0xa3710101,
	0xfb840108,0xa3720101,0xfb840108,0xa3730101,0xfb840108,0xa3740101,0xfb840108,0xa3750101,
	0xfb840108,0xa3760101,0xfb840108,0xa3770101,0xfb840108,0xa3780101,0xfb840108,0xa3790101,
	0xfb840108,0xa37a0101,0xfb840108,0xa37b0101,0xfb840108,0xa37c0101,0xfb840108,0xa37d0101,
	0xfb840108,0xa37e0101,0xfb840108,0xa37f0101,0xfb840108,0xa3800101,0xfb840108,0xa3810101,
	0xfb840108,0xa3820101,0xfb840108,0xa3830101,0xfb840108,0xa3840101,0xfb840108,0xa3850101,
	0xfb840108,0xa3860101,0xfb840108,0xa3870101,0xfb840108,0xa3880101,0xfb840108,0xa3890101,
	0xfb840108,0xa38a0101,0xfb840108,0xa38b0101,0xfb840108,0xa38c0101,0xfb840108,0xa38d0101,
	0xfb840108,0xa38e0101,0xfb840108,0xa38f0101,0xfb840108,0xa3900101,0xfb840108,0xa3910101,
	0xfb840108,0xa3920101,0xfb840108,0xa3930101,0xfb840108,0xa3940101,0xfb840108,0xa3950101,
	0xfb840108,0xa3960101,0xfb840108,0xa3970101,0xfb840108,0xa3980101,0xfb840108,0xa3990101,
	0xfb840108,0xa39a0101,0xfb840108,0xa39b0101,0xfb840108,0xa39c0101,0xfb840108,0xa39d0101,
	0xfb840108,0xa39e0101,0xfb840108,0xa39f0101,0xfb840108,0xa3a00101,0xfb840108,0xa3a10101,
	0xfb840108,0xa3a20101,0xfb840108,0xa3a30101,0xfb840108,0xa3a40101,0xfb840108,0xa3a50101,
	0xfb840108,0xa3a60101,0xfb840108,0xa3a70101,0xfb840108,0xa3a80101,0xfb840108,0xa3a90101,
	0xfb840108,0xa3aa0101,0xfb840108,0xa3ab0101,0xfb840108,0xa3ac0101,0xfb840108,0xa3ad0101,
	0xfb840108,0xa3ae0101,0xfb840108,0xa3af0101,0xfb840108,0xa3b00101,0xfb840108,0xa3b10101,
	0xfb840108,0xa3b20101,0xfb840108,0xa3b30101,0xfb840108,0xa3b40101,0xfb840108,0xa3b50101,
	0xfb840108,0xa3b60101,0xfb840108,0xa3b70101,0xfb840108,0xa3b80101,0xfb840108,0xa3b90101,
	0xfb840108,0xa3ba0101,0xfb840108,0xa3bb0101,0xfb840108,0xa3bc0101,0xfb840108,0xa3bd0101,
	0xfb840108,0xa3be0101,0xfb840108,0xa3bf0101,0xfb840108,0xa3c00101,0xfb840108,0xa3c10101,
	0xfb840108,0xa3c20101,0xfb840108,0xa3c30101,0xfb840108,0xa3c40101,0xfb840108,0xa3c50101,
	0xfb840108,0xa3c60101,0xfb840108,0xa3c70101,0xfb840108,0xa3c80101,0xfb840108,0xa3c90101,
	0xfb840108,0xa3ca0101,0xfb840108,0xa3cb0101,0xfb840108,0xa3cc0101,0xfb840108,0xa3cd0101,
	0xfb840108,0xa3ce0101,0xfb840108,0xa3cf0101,0xfb840108,0xa3d00101,0xfb840108,0xa3d10101,
	0xfb840108,0xa3d20101,0xfb840108,0xa3d30101,0xfb840108,0xa3d40101,0xfb840108,0xa3d50101,
	0xfb840108,0xa3d60101,0xfb840108,0xa3d70101,0xfb840108,0xa3d80101,0xfb840108,0xa3d90101,
	0xfb840108,0xa3da0101,0xfb840108,0xa3db0101,0xfb840108,0xa3dc0101,0xfb840108,0xa3dd0101,
	0xfb840108,0xa3de0101,0xfb840108,0xa3df0101,0xfb840108,0xa3e00101,0xfb840108,0xa3e10101,
	0xfb840108,0xa3e20101,0xfb840108,0xa3e30101,0xfb840108,0xa3e40101,0xfb840108,0xa3e50101,
	0xfb840108,0xa3e60101,0xfb840108,0xa3e70101,0xfb840108,0xa3e80101,0xfb840108,0xa3e90101,
	0xfb840108,0xa3ea0101,0xfb840108,0xa3eb0101,0xfb840108,0xa3ec0101,0xfb840108,0xa3ed0101,
	0xfb840108,0xa3ee0101,0xfb840108,0xa3ef0101,0xfb840108,0xa3f00101,0xfb840108,0xa3f10101,
	0xfb840108,0xa3f20101,0xfb840108,0xa3f30101,0xfb840108,0xa3f40101,0xfb840108,0xa3f50101,
	0xfb840108,0xa3f60101,0xfb840108,0xa3f70101,0xfb840108,0xa3f80101,0xfb840108,0xa3f90101,
	0xfb840108,0xa3fa0101,0xfb840108,0xa3fb0101,0xfb840108,0xa3fc0101,0xfb840108,0xa3fd0101,
	0xfb840108,0xa3fe0101,0xfb840108,0xa3ff0101,0xfb840108,0xa4000101,0xfb840108,0xa4010101,
	0xfb840108,0xa4020101,0xfb840108,0xa4030101,0xfb840108,0xa4040101,0xfb840108,0xa4050101,
	0xfb840108,0xa4060101,0xfb840108,0xa4070101,0xfb840108,0xa4080101,0xfb840108,0xa4090101,
	0xfb840108,0xa40a0101,0xfb840108,0xa40b0101,0xfb840108,0xa40c0101,0xfb840108,0xa40d0101,
	0xfb840108,0xa40e0101,0xfb840108,0xa40f0101,0xfb840108,0xa4100101,0xfb840108,0xa4110101,
	0xfb840108,0xa4120101,0xfb840108,0xa4130101,0xfb840108,0xa4140101,0xfb840108,0xa4150101,
	0xfb840108,0xa4160101,0xfb840108,0xa4170101,0xfb840108,0xa4180101,0xfb840108,0xa4190101,
	0xfb840108,0xa41a0101,0xfb840108,0xa41b0101,0xfb840108,0xa41c0101,0xfb840108,0xa41d0101,
	0xfb840108,0xa41e0101,0xfb840108,0xa41f0101,0xfb840108,0xa4200101,0xfb840108,0xa4210101,
	0xfb840108,0xa4220101,0xfb840108,0xa4230101,0xfb840108,0xa4240101,0xfb840108,0xa4250101,
	0xfb840108,0xa4260101,0xfb840108,0xa4270101,0xfb840108,0xa4280101,0xfb840108,0xa4290101,
	0xfb840108,0xa42a0101,0xfb840108,0xa42b0101,0xfb840108,0xa42c0101,0xfb840108,0xa42d0101,
	0xfb840108,0xa42e0101,0xfb840108,0xa42f0101,0xfb840108,0xa4300101,0xfb840108,0xa4310101,
	0xfb840108,0xa4320101,0xfb840108,0xa4330101,0xfb840108,0xa4340101,0xfb840108,0xa4350101,
	0xfb840108,0xa4360101,0xfb840108,0xa4370101,0xfb840108,0xa4380101,0xfb840108,0xa4390101,
	0xfb840108,0xa43a0101,0xfb840108,0xa43b0101,0xfb840108,0xa43c0101,0xfb840108,0xa43d0101,
	0xfb840108,0xa43e0101,0xfb840108,0xa43f0101,0xfb840108,0xa4400101,0xfb840108,0xa4410101,
	0xfb840108,0xa4420101,0xfb840108,0xa4430101,0xfb840108,0xa4440101,0xfb840108,0xa4450101,
	0xfb840108,0xa4460101,0xfb840108,0xa4470101,0xfb840108,0xa4480101,0xfb840108,0xa4490101,
	0xfb840108,0xa44a0101,0xfb840108,0xa44b0101,0xfb840108,0xa44c0101,0xfb840108,0xa44d0101,
	0xfb840108,0xa44e0101,0xfb840108,0xa44f0101,0xfb840108,0xa4500101,0xfb840108,0xa4510101,
	0xfb840108,0xa4520101,0xfb840108,0xa4530101,0xfb840108,0xa4540101,0xfb840108,0xa4550101,
	0xfb840108,0xa4560101,0xfb840108,0xa4570101,0xfb840108,0xa4580101,0xfb840108,0xa4590101,
	0xfb840108,0xa45a0101,0xfb840108,0xa45b0101,0xfb840108,0xa45c0101,0xfb840108,0xa45d0101,
	0xfb840108,0xa45e0101,0xfb840108,0xa45f0101,0xfb840108,0xa4600101,0xfb840108,0xa4610101,
	0xfb840108,0xa4620101,0xfb840108,0xa4630101,0xfb840108,0xa4640101,0xfb840108,0xa4650101,
	0xfb840108,0xa4660101,0xfb840108,0xa4670101,0xfb840108,0xa4680101,0xfb840108,0xa4690101,
	0xfb840108,0xa46a0101,0xfb840108,0xa46b0101,0xfb840108,0xa46c0101,0xfb840108,0xa46d0101,
	0xfb840108,0xa46e0101,0xfb840108,0xa46f0101,0xfb840108,0xa4700101,0xfb840108,0xa4710101,
	0xfb840108,0xa4720101,0xfb840108,0xa4730101,0xfb840108,0xa4740101,0xfb840108,0xa4750101,
	0xfb840108,0xa4760101,0xfb840108,0xa4770101,0xfb840108,0xa4780101,0xfb840108,0xa4790101,
	0xfb840108,0xa47a0101,0xfb840108,0xa47b0101,0xfb840108,0xa47c0101,0xfb840108,0xa47d0101,
	0xfb840108,0xa47e0101,0xfb840108,0xa47f0101,0xfb840108,0xa4800101,0xfb840108,0xa4810101,
	0xfb840108,0xa4820101,0xfb840108,0xa4830101,0xfb840108,0xa4840101,0xfb840108,0xa4850101,
	0xfb840108,0xa4860101,0xfb840108,0xa4870101,0xfb840108,0xa4880101,0xfb840108,0xa4890101,
	0xfb840108,0xa48a0101,0xfb840108,0xa48b0101,0xfb840108,0xa48c0101,0xfb840108,0xa48d0101,
	0xfb840108,0xa48e0101,0xfb840108,0xa48f0101,0xfb840108,0xa4900101,0xfb840108,0xa4910101,
	0xfb840108,0xa4920101,0xfb840108,0xa4930101,0xfb840108,0xa4940101,0xfb840108,0xa4950101,
	0xfb840108,0xa4960101,0xfb840108,0xa4970101,0xfb840108,0xa4980101,0xfb840108,0xa4990101,
	0xfb840108,0xa49a0101,0xfb840108,0xa49b0101,0xfb840108,0xa49c0101,0xfb840108,0xa49d0101,
	0xfb840108,0xa49e0101,0xfb840108,0xa49f0101,0xfb840108,0xa4a00101,0xfb840108,0xa4a10101,
	0xfb840108,0xa4a20101,0xfb840108,0xa4a30101,0xfb840108,0xa4a40101,0xfb840108,0xa4a50101,
	0xfb840108,0xa4a60101,0xfb840108,0xa4a70101,0xfb840108,0xa4a80101,0xfb840108,0xa4a90101,
	0xfb840108,0xa4aa0101,0xfb840108,0xa4ab0101,0xfb840108,0xa4ac0101,0xfb840108,0xa4ad0101,
	0xfb840108,0xa4ae0101,0xfb840108,0xa4af0101,0xfb840108,0xa4b00101,0xfb840108,0xa4b10101,
	0xfb840108,0xa4b20101,0xfb840108,0xa4b30101,0xfb840108,0xa4b40101,0xfb840108,0xa4b50101,
	0xfb840108,0xa4b60101,0xfb840108,0xa4b70101,0xfb840108,0xa4b80101,0xfb840108,0xa4b90101,
	0xfb840108,0xa4ba0101,0xfb840108,0xa4bb0101,0xfb840108,0xa4bc0101,0xfb840108,0xa4bd0101,
	0xfb840108,0xa4be0101,0xfb840108,0xa4bf0101,0xfb840108,0xa4c00101,0xfb840108,0xa4c10101,
	0xfb840108,0xa4c20101,0xfb840108,0xa4c30101,0xfb840108,0xa4c40101,0xfb840108,0xa4c50101,
	0xfb840108,0xa4c60101,0xfb840108,0xa4c70101,0xfb840108,0xa4c80101,0xfb840108,0xa4c90101,
	0xfb840108,0xa4ca0101,0xfb840108,0xa4cb0101,0xfb840108,0xa4cc0101,0xfb840108,0xa4cd0101,
	0xfb840108,0xa4ce0101,0xfb840108,0xa4cf0101,0xfb840108,0xa4d00101,0xfb840108,0xa4d10101,
	0xfb840108,0xa4d20101,0xfb840108,0xa4d30101,0xfb840108,0xa4d40101,0xfb840108,0xa4d50101,
	0xfb840108,0xa4d60101,0xfb840108,0xa4d70101,0xfb840108,0xa4d80101,0xfb840108,0xa4d90101,
	0xfb840108,0xa4da0101,0xfb840108,0xa4db0101,0xfb840108,0xa4dc0101,0xfb840108,0xa4dd0101,
	0xfb840108,0xa4de0101,0xfb840108,0xa4df0101,0xfb840108,0xa4e00101,0xfb840108,0xa4e10101,
	0xfb840108,0xa4e20101,0xfb840108,0xa4e30101,0xfb840108,0xa4e40101,0xfb840108,0xa4e50101,
	0xfb840108,0xa4e60101,0xfb840108,0xa4e70101,0xfb840108,0xa4e80101,0xfb840108,0xa4e90101,
	0xfb840108,0xa4ea0101,0xfb840108,0xa4eb0101,0xfb840108,0xa4ec0101,0xfb840108,0xa4ed0101,
	0xfb840108,0xa4ee0101,0xfb840108,0xa4ef0101,0xfb840108,0xa4f00101,0xfb840108,0xa4f10101,
	0xfb840108,0xa4f20101,0xfb840108,0xa4f30101,0xfb840108,0xa4f40101,0xfb840108,0xa4f50101,
	0xfb840108,0xa4f60101,0xfb840108,0xa4f70101,0xfb840108,0xa4f80101,0xfb840108,0xa4f90101,
	0xfb840108,0xa4fa0101,0xfb840108,0xa4fb0101,0xfb840108,0xa4fc0101,0xfb840108,0xa4fd0101,
	0xfb840108,0xa4fe0101,0xfb840108,0xa4ff0101,0xfb840108,0xa5000101,0xfb840108,0xa5010101,
	0xfb840108,0xa5020101,0xfb840108,0xa5030101,0xfb840108,0xa5040101,0xfb840108,0xa5050101,
	0xfb840108,0xa5060101,0xfb840108,0xa5070101,0xfb840108,0xa5080101,0xfb840108,0xa5090101,
	0xfb840108,0xa50a0101,0xfb840108,0xa50b0101,0xfb840108,0xa50c0101,0xfb840108,0xa50d0101,
	0xfb840108,0xa50e0101,0xfb840108,0xa50f0101,0xfb840108,0xa5100101,0xfb840108,0xa5110101,
	0xfb840108,0xa5120101,0xfb840108,0xa5130101,0xfb840108,0xa5140101,0xfb840108,0xa5150101,
	0xfb840108,0xa5160101,0xfb840108,0xa5170101,0xfb840108,0xa5180101,0xfb840108,0xa5190101,
	0xfb840108,0xa51a0101,0xfb840108,0xa51b0101,0xfb840108,0xa51c0101,0xfb840108,0xa51d0101,
	0xfb840108,0xa51e0101,0xfb840108,0xa51f0101,0xfb840108,0xa5200101,0xfb840108,0xa5210101,
	0xfb840108,0xa5220101,0xfb840108,0xa5230101,0xfb840108,0xa5240101,0xfb840108,0xa5250101,
	0xfb840108,0xa5260101,0xfb840108,0xa5270101,0xfb840108,0xa5280101,0xfb840108,0xa5290101,
	0xfb840108,0xa52a0101,0xfb840108,0xa52b0101,0xfb840108,0xa52c0101,0xfb840108,0xa52d0101,
	0xfb840108,0xa52e0101,0xfb840108,0xa52f0101,0xfb840108,0xa5300101,0xfb840108,0xa5310101,
	0xfb840108,0xa5320101,0xfb840108,0xa5330101,0xfb840108,0xa5340101,0xfb840108,0xa5350101,
	0xfb840108,0xa5360101,0xfb840108,0xa5370101,0xfb840108,0xa5380101,0xfb840108,0xa5390101,
	0xfb840108,0xa53a0101,0xfb840108,0xa53b0101,0xfb840108,0xa53c0101,0xfb840108,0xa53d0101,
	0xfb840108,0xa53e0101,0xfb840108,0xa53f0101,0xfb840108,0xa5400101,0xfb840108,0xa5410101,
	0xfb840108,0xa5420101,0xfb840108,0xa5430101,0xfb840108,0xa5440101,0xfb840108,0xa5450101,
	0xfb840108,0xa5460101,0xfb840108,0xa5470101,0xfb840108,0xa5480101,0xfb840108,0xa5490101,
	0xfb840108,0xa54a0101,0xfb840108,0xa54b0101,0xfb840108,0xa54c0101,0xfb840108,0xa54d0101,
	0xfb840108,0xa54e0101,0xfb840108,0xa54f0101,0xfb840108,0xa5500101,0xfb840108,0xa5510101,
	0xfb840108,0xa5520101,0xfb840108,0xa5530101,0xfb840108,0xa5540101,0xfb840108,0xa5550101,
	0xfb840108,0xa5560101,0xfb840108,0xa5570101,0xfb840108,0xa5580101,0xfb840108,0xa5590101,
	0xfb840108,0xa55a0101,0xfb840108,0xa55b0101,0xfb840108,0xa55c0101,0xfb840108,0xa55d0101,
	0xfb840108,0xa55e0101,0xfb840108,0xa55f0101,0xfb840108,0xa5600101,0xfb840108,0xa5610101,
	0xfb840108,0xa5620101,0xfb840108,0xa5630101,0xfb840108,0xa5640101,0xfb840108,0xa5650101,
	0xfb840108,0xa5660101,0xfb840108,0xa5670101,0xfb840108,0xa5680101,0xfb840108,0xa5690101,
	0xfb840108,0xa56a0101,0xfb840108,0xa56b0101,0xfb840108,0xa56c0101,0xfb840108,0xa56d0101,
	0xfb840108,0xa56e0101,0xfb840108,0xa56f0101,0xfb840108,0xa5700101,0xfb840108,0xa5710101,
	0xfb840108,0xa5720101,0xfb840108,0xa5730101,0xfb840108,0xa5740101,0xfb840108,0xa5750101,
	0xfb840108,0xa5760101,0xfb840108,0xa5770101,0xfb840108,0xa5780101,0xfb840108,0xa5790101,
	0xfb840108,0xa57a0101,0xfb840108,0xa57b0101,0xfb840108,0xa57c0101,0xfb840108,0xa57d0101,
	0xfb840108,0xa57e0101,0xfb840108,0xa57f0101,0xfb840108,0xa5800101,0xfb840108,0xa5810101,
	0xfb840108,0xa5820101,0xfb840108,0xa5830101,0xfb840108,0xa5840101,0xfb840108,0xa5850101,
	0xfb840108,0xa5860101,0xfb840108,0xa5870101,0xfb840108,0xa5880101,0xfb840108,0xa5890101,
	0xfb840108,0xa58a0101,0xfb840108,0xa58b0101,0xfb840108,0xa58c0101,0xfb840108,0xa58d0101,
	0xfb840108,0xa58e0101,0xfb840108,0xa58f0101,0xfb840108,0xa5900101,0xfb840108,0xa5910101,
	0xfb840108,0xa5920101,0xfb840108,0xa5930101,0xfb840108,0xa5940101,0xfb840108,0xa5950101,
	0xfb840108,0xa5960101,0xfb840108,0xa5970101,0xfb840108,0xa5980101,0xfb840108,0xa5990101,
	0xfb840108,0xa59a0101,0xfb840108,0xa59b0101,0xfb840108,0xa59c0101,0xfb840108,0xa59d0101,
	0xfb840108,0xa59e0101,0xfb840108,0xa59f0101,0xfb840108,0xa5a00101,0xfb840108,0xa5a10101,
	0xfb840108,0xa5a20101,0xfb840108,0xa5a30101,0xfb840108,0xa5a40101,0xfb840108,0xa5a50101,
	0xfb840108,0xa5a60101,0xfb840108,0xa5a70101,0xfb840108,0xa5a80101,0xfb840108,0xa5a90101,
	0xfb840108,0xa5aa0101,0xfb840108,0xa5ab0101,0xfb840108,0xa5ac0101,0xfb840108,0xa5ad0101,
	0xfb840108,0xa5ae0101,0xfb840108,0xa5af0101,0xfb840108,0xa5b00101,0xfb840108,0xa5b10101,
	0xfb840108,0xa5b20101,0xfb840108,0xa5b30101,0xfb840108,0xa5b40101,0xfb840108,0xa5b50101,
	0xfb840108,0xa5b60101,0xfb840108,0xa5b70101,0xfb840108,0xa5b80101,0xfb840108,0xa5b90101,
	0xfb840108,0xa5ba0101,0xfb840108,0xa5bb0101,0xfb840108,0xa5bc0101,0xfb840108,0xa5bd0101,
	0xfb840108,0xa5be0101,0xfb840108,0xa5bf0101,0xfb840108,0xa5c00101,0xfb840108,0xa5c10101,
	0xfb840108,0xa5c20101,0xfb840108,0xa5c30101,0xfb840108,0xa5c40101,0xfb840108,0xa5c50101,
	0xfb840108,0xa5c60101,0xfb840108,0xa5c70101,0xfb840108,0xa5c80101,0xfb840108,0xa5c90101,
	0xfb840108,0xa5ca0101,0xfb840108,0xa5cb0101,0xfb840108,0xa5cc0101,0xfb840108,0xa5cd0101,
	0xfb840108,0xa5ce0101,0xfb840108,0xa5cf0101,0xfb840108,0xa5d00101,0xfb840108,0xa5d10101,
	0xfb840108,0xa5d20101,0xfb840108,0xa5d30101,0xfb840108,0xa5d40101,0xfb840108,0xa5d50101,
	0xfb840108,0xa5d60101,0xfb840108,0xa5d70101,0xfb840108,0xa5d80101,0xfb840108,0xa5d90101,
	0xfb840108,0xa5da0101,0xfb840108,0xa5db0101,0xfb840108,0xa5dc0101,0xfb840108,0xa5dd0101,
	0xfb840108,0xa5de0101,0xfb840108,0xa5df0101,0xfb840108,0xa5e00101,0xfb840108,0xa5e10101,
	0xfb840108,0xa5e20101,0xfb840108,0xa5e30101,0xfb840108,0xa5e40101,0xfb840108,0xa5e50101,
	0xfb840108,0xa5e60101,0xfb840108,0xa5e70101,0xfb840108,0xa5e80101,0xfb840108,0xa5e90101,
	0xfb840108,0xa5ea0101,0xfb840108,0xa5eb0101,0xfb840108,0xa5ec0101,0xfb840108,0xa5ed0101,
	0xfb840108,0xa5ee0101,0xfb840108,0xa5ef0101,0xfb840108,0xa5f00101,0xfb840108,0xa5f10101,
	0xfb840108,0xa5f20101,0xfb840108,0xa5f30101,0xfb840108,0xa5f40101,0xfb840108,0xa5f50101,
	0xfb840108,0xa5f60101,0xfb840108,0xa5f70101,0xfb840108,0xa5f80101,0xfb840108,0xa5f90101,
	0xfb840108,0xa5fa0101,0xfb840108,0xa5fb0101,0xfb840108,0xa5fc0101,0xfb840108,0xa5fd0101,
	0xfb840108,0xa5fe0101,0xfb840108,0xa5ff0101,0xfb840108,0xa6000101,0xfb840108,0xa6010101,
	0xfb840108,0xa6020101,0xfb840108,0xa6030101,0xfb840108,0xa6040101,0xfb840108,0xa6050101,
	0xfb840108,0xa6060101,0xfb840108,0xa6070101,0xfb840108,0xa6080101,0xfb840108,0xa6090101,
	0xfb840108,0xa60a0101,0xfb840108,0xa60b0101,0xfb840108,0xa60c0101,0xfb840108,0xa60d0101,
	0xfb840108,0xa60e0101,0xfb840108,0xa60f0101,0xfb840108,0xa6100101,0xfb840108,0xa6110101,
	0xfb840108,0xa6120101,0xfb840108,0xa6130101,0xfb840108,0xa6140101,0xfb840108,0xa6150101,
	0xfb840108,0xa6160101,0xfb840108,0xa6170101,0xfb840108,0xa6180101,0xfb840108,0xa6190101,
	0xfb840108,0xa61a0101,0xfb840108,0xa61b0101,0xfb840108,0xa61c0101,0xfb840108,0xa61d0101,
	0xfb840108,0xa61e0101,0xfb840108,0xa61f0101,0xfb840108,0xa6200101,0xfb840108,0xa6210101,
	0xfb840108,0xa6220101,0xfb840108,0xa6230101,0xfb840108,0xa6240101,0xfb840108,0xa6250101,
	0xfb840108,0xa6260101,0xfb840108,0xa6270101,0xfb840108,0xa6280101,0xfb840108,0xa6290101,
	0xfb840108,0xa62a0101,0xfb840108,0xa62b0101,0xfb840108,0xa62c0101,0xfb840108,0xa62d0101,
	0xfb840108,0xa62e0101,0xfb840108,0xa62f0101,0xfb840108,0xa6300101,0xfb840108,0xa6310101,
	0xfb840108,0xa6320101,0xfb840108,0xa6330101,0xfb840108,0xa6340101,0xfb840108,0xa6350101,
	0xfb840108,0xa6360101,0xfb840108,0xa6370101,0xfb840108,0xa6380101,0xfb840108,0xa6390101,
	0xfb840108,0xa63a0101,0xfb840108,0xa63b0101,0xfb840108,0xa63c0101,0xfb840108,0xa63d0101,
	0xfb840108,0xa63e0101,0xfb840108,0xa63f0101,0xfb840108,0xa6400101,0xfb840108,0xa6410101,
	0xfb840108,0xa6420101,0xfb840108,0xa6430101,0xfb840108,0xa6440101,0xfb840108,0xa6450101,
	0xfb840108,0xa6460101,0xfb840108,0xa6470101,0xfb840108,0xa6480101,0xfb840108,0xa6490101,
	0xfb840108,0xa64a0101,0xfb840108,0xa64b0101,0xfb840108,0xa64c0101,0xfb840108,0xa64d0101,
	0xfb840108,0xa64e0101,0xfb840108,0xa64f0101,0xfb840108,0xa6500101,0xfb840108,0xa6510101,
	0xfb840108,0xa6520101,0xfb840108,0xa6530101,0xfb840108,0xa6540101,0xfb840108,0xa6550101,
	0xfb840108,0xa6560101,0xfb840108,0xa6570101,0xfb840108,0xa6580101,0xfb840108,0xa6590101,
	0xfb840108,0xa65a0101,0xfb840108,0xa65b0101,0xfb840108,0xa65c0101,0xfb840108,0xa65d0101,
	0xfb840108,0xa65e0101,0xfb840108,0xa65f0101,0xfb840108,0xa6600101,0xfb840108,0xa6610101,
	0xfb840108,0xa6620101,0xfb840108,0xa6630101,0xfb840108,0xa6640101,0xfb840108,0xa6650101,
	0xfb840108,0xa6660101,0xfb840108,0xa6670101,0xfb840108,0xa6680101,0xfb840108,0xa6690101,
	0xfb840108,0xa66a0101,0xfb840108,0xa66b0101,0xfb840108,0xa66c0101,0xfb840108,0xa66d0101,
	0xfb840108,0xa66e0101,0xfb840108,0xa66f0101,0xfb840108,0xa6700101,0xfb840108,0xa6710101,
	0xfb840108,0xa6720101,0xfb840108,0xa6730101,0xfb840108,0xa6740101,0xfb840108,0xa6750101,
	0xfb840108,0xa6760101,0xfb840108,0xa6770101,0xfb840108,0xa6780101,0xfb840108,0xa6790101,
	0xfb840108,0xa67a0101,0xfb840108,0xa67b0101,0xfb840108,0xa67c0101,0xfb840108,0xa67d0101,
	0xfb840108,0xa67e0101,0xfb840108,0xa67f0101,0xfb840108,0xa6800101,0xfb840108,0xa6810101,
	0xfb840108,0xa6820101,0xfb840108,0xa6830101,0xfb840108,0xa6840101,0xfb840108,0xa6850101,
	0xfb840108,0xa6860101,0xfb840108,0xa6870101,0xfb840108,0xa6880101,0xfb840108,0xa6890101,
	0xfb840108,0xa68a0101,0xfb840108,0xa68b0101,0xfb840108,0xa68c0101,0xfb840108,0xa68d0101,
	0xfb840108,0xa68e0101,0xfb840108,0xa68f0101,0xfb840108,0xa6900101,0xfb840108,0xa6910101,
	0xfb840108,0xa6920101,0xfb840108,0xa6930101,0xfb840108,0xa6940101,0xfb840108,0xa6950101,
	0xfb840108,0xa6960101,0xfb840108,0xa6970101,0xfb840108,0xa6980101,0xfb840108,0xa6990101,
	0xfb840108,0xa69a0101,0xfb840108,0xa69b0101,0xfb840108,0xa69c0101,0xfb840108,0xa69d0101,
	0xfb840108,0xa69e0101,0xfb840108,0xa69f0101,0xfb840108,0xa6a00101,0xfb840108,0xa6a10101,
	0xfb840108,0xa6a20101,0xfb840108,0xa6a30101,0xfb840108,0xa6a40101,0xfb840108,0xa6a50101,
	0xfb840108,0xa6a60101,0xfb840108,0xa6a70101,0xfb840108,0xa6a80101,0xfb840108,0xa6a90101,
	0xfb840108,0xa6aa0101,0xfb840108,0xa6ab0101,0xfb840108,0xa6ac0101,0xfb840108,0xa6ad0101,
	0xfb840108,0xa6ae0101,0xfb840108,0xa6af0101,0xfb840108,0xa6b00101,0xfb840108,0xa6b10101,
	0xfb840108,0xa6b20101,0xfb840108,0xa6b30101,0xfb840108,0xa6b40101,0xfb840108,0xa6b50101,
	0xfb840108,0xa6b60101,0xfb840108,0xa6b70101,0xfb840108,0xa6b80101,0xfb840108,0xa6b90101,
	0xfb840108,0xa6ba0101,0xfb840108,0xa6bb0101,0xfb840108,0xa6bc0101,0xfb840108,0xa6bd0101,
	0xfb840108,0xa6be0101,0xfb840108,0xa6bf0101,0xfb840108,0xa6c00101,0xfb840108,0xa6c10101,
	0xfb840108,0xa6c20101,0xfb840108,0xa6c30101,0xfb840108,0xa6c40101,0xfb840108,0xa6c50101,
	0xfb840108,0xa6c60101,0xfb840108,0xa6c70101,0xfb840108,0xa6c80101,0xfb840108,0xa6c90101,
	0xfb840108,0xa6ca0101,0xfb840108,0xa6cb0101,0xfb840108,0xa6cc0101,0xfb840108,0xa6cd0101,
	0xfb840108,0xa6ce0101,0xfb840108,0xa6cf0101,0xfb840108,0xa6d00101,0xfb840108,0xa6d10101,
	0xfb840108,0xa6d20101,0xfb840108,0xa6d30101,0xfb840108,0xa6d40101,0xfb840108,0xa6d50101,
	0xfb840108,0xa6d60101,0xfb840108,0xa6d70101,0xfb840108,0xa6d80101,0xfb840108,0xa6d90101,
	0xfb840108,0xa6da0101,0xfb840108,0xa6db0101,0xfb840108,0xa6dc0101,0xfb840108,0xa6dd0101,
	0xfb840108,0xa6de0101,0xfb840108,0xa6df0101,0xfb840108,0xa6e00101,0xfb840108,0xa6e10101,
	0xfb840108,0xa6e20101,0xfb840108,0xa6e30101,0xfb840108,0xa6e40101,0xfb840108,0xa6e50101,
	0xfb840108,0xa6e60101,0xfb840108,0xa6e70101,0xfb840108,0xa6e80101,0xfb840108,0xa6e90101,
	0xfb840108,0xa6ea0101,0xfb840108,0xa6eb0101,0xfb840108,0xa6ec0101,0xfb840108,0xa6ed0101,
	0xfb840108,0xa6ee0101,0xfb840108,0xa6ef0101,0xfb840108,0xa6f00101,0xfb840108,0xa6f10101,
	0xfb840108,0xa6f20101,0xfb840108,0xa6f30101,0xfb840108,0xa6f40101,0xfb840108,0xa6f50101,
	0xfb840108,0xa6f60101,0xfb840108,0xa6f70101,0xfb840108,0xa6f80101,0xfb840108,0xa6f90101,
	0xfb840108,0xa6fa0101,0xfb840108,0xa6fb0101,0xfb840108,0xa6fc0101,0xfb840108,0xa6fd0101,
	0xfb840108,0xa6fe0101,0xfb840108,0xa6ff0101,0xfb840108,0xa7000101,0xfb840108,0xa7010101,
	0xfb840108,0xa7020101,0xfb840108,0xa7030101,0xfb840108,0xa7040101,0xfb840108,0xa7050101,
	0xfb840108,0xa7060101,0xfb840108,0xa7070101,0xfb840108,0xa7080101,0xfb840108,0xa7090101,
	0xfb840108,0xa70a0101,0xfb840108,0xa70b0101,0xfb840108,0xa70c0101,0xfb840108,0xa70d0101,
	0xfb840108,0xa70e0101,0xfb840108,0xa70f0101,0xfb840108,0xa7100101,0xfb840108,0xa7110101,
	0xfb840108,0xa7120101,0xfb840108,0xa7130101,0xfb840108,0xa7140101,0xfb840108,0xa7150101,
	0xfb840108,0xa7160101,0xfb840108,0xa7170101,0xfb840108,0xa7180101,0xfb840108,0xa7190101,
	0xfb840108,0xa71a0101,0xfb840108,0xa71b0101,0xfb840108,0xa71c0101,0xfb840108,0xa71d0101,
	0xfb840108,0xa71e0101,0xfb840108,0xa71f0101,0xfb840108,0xa7200101,0xfb840108,0xa7210101,
	0xfb840108,0xa7220101,0xfb840108,0xa7230101,0xfb840108,0xa7240101,0xfb840108,0xa7250101,
	0xfb840108,0xa7260101,0xfb840108,0xa7270101,0xfb840108,0xa7280101,0xfb840108,0xa7290101,
	0xfb840108,0xa72a0101,0xfb840108,0xa72b0101,0xfb840108,0xa72c0101,0xfb840108,0xa72d0101,
	0xfb840108,0xa72e0101,0xfb840108,0xa72f0101,0xfb840108,0xa7300101,0xfb840108,0xa7310101,
	0xfb840108,0xa7320101,0xfb840108,0xa7330101,0xfb840108,0xa7340101,0xfb840108,0xa7350101,
	0xfb840108,0xa7360101,0xfb840108,0xa7370101,0xfb840108,0xa7380101,0xfb840108,0xa7390101,
	0xfb840108,0xa73a0101,0xfb840108,0xa73b0101,0xfb840108,0xa73c0101,0xfb840108,0xa73d0101,
	0xfb840108,0xa73e0101,0xfb840108,0xa73f0101,0xfb840108,0xa7400101,0xfb840108,0xa7410101,
	0xfb840108,0xa7420101,0xfb840108,0xa7430101,0xfb840108,0xa7440101,0xfb840108,0xa7450101,
	0xfb840108,0xa7460101,0xfb840108,0xa7470101,0xfb840108,0xa7480101,0xfb840108,0xa7490101,
	0xfb840108,0xa74a0101,0xfb840108,0xa74b0101,0xfb840108,0xa74c0101,0xfb840108,0xa74d0101,
	0xfb840108,0xa74e0101,0xfb840108,0xa74f0101,0xfb840108,0xa7500101,0xfb840108,0xa7510101,
	0xfb840108,0xa7520101,0xfb840108,0xa7530101,0xfb840108,0xa7540101,0xfb840108,0xa7550101,
	0xfb840108,0xa7560101,0xfb840108,0xa7570101,0xfb840108,0xa7580101,0xfb840108,0xa7590101,
	0xfb840108,0xa75a0101,0xfb840108,0xa75b0101,0xfb840108,0xa75c0101,0xfb840108,0xa75d0101,
	0xfb840108,0xa75e0101,0xfb840108,0xa75f0101,0xfb840108,0xa7600101,0xfb840108,0xa7610101,
	0xfb840108,0xa7620101,0xfb840108,0xa7630101,0xfb840108,0xa7640101,0xfb840108,0xa7650101,
	0xfb840108,0xa7660101,0xfb840108,0xa7670101,0xfb840108,0xa7680101,0xfb840108,0xa7690101,
	0xfb840108,0xa76a0101,0xfb840108,0xa76b0101,0xfb840108,0xa76c0101,0xfb840108,0xa76d0101,
	0xfb840108,0xa76e0101,0xfb840108,0xa76f0101,0xfb840108,0xa7700101,0xfb840108,0xa7710101,
	0xfb840108,0xa7720101,0xfb840108,0xa7730101,0xfb840108,0xa7740101,0xfb840108,0xa7750101,
	0xfb840108,0xa7760101,0xfb840108,0xa7770101,0xfb840108,0xa7780101,0xfb840108,0xa7790101,
	0xfb840108,0xa77a0101,0xfb840108,0xa77b0101,0xfb840108,0xa77c0101,0xfb840108,0xa77d0101,
	0xfb840108,0xa77e0101,0xfb840108,0xa77f0101,0xfb840108,0xa7800101,0xfb840108,0xa7810101,
	0xfb840108,0xa7820101,0xfb840108,0xa7830101,0xfb840108,0xa7840101,0xfb840108,0xa7850101,
	0xfb840108,0xa7860101,0xfb840108,0xa7870101,0xfb840108,0xa7880101,0xfb840108,0xa7890101,
	0xfb840108,0xa78a0101,0xfb840108,0xa78b0101,0xfb840108,0xa78c0101,0xfb840108,0xa78d0101,
	0xfb840108,0xa78e0101,0xfb840108,0xa78f0101,0xfb840108,0xa7900101,0xfb840108,0xa7910101,
	0xfb840108,0xa7920101,0xfb840108,0xa7930101,0xfb840108,0xa7940101,0xfb840108,0xa7950101,
	0xfb840108,0xa7960101,0xfb840108,0xa7970101,0xfb840108,0xa7980101,0xfb840108,0xa7990101,
	0xfb840108,0xa79a0101,0xfb840108,0xa79b0101,0xfb840108,0xa79c0101,0xfb840108,0xa79d0101,
	0xfb840108,0xa79e0101,0xfb840108,0xa79f0101,0xfb840108,0xa7a00101,0xfb840108,0xa7a10101,
	0xfb840108,0xa7a20101,0xfb840108,0xa7a30101,0xfb840108,0xa7a40101,0xfb840108,0xa7a50101,
	0xfb840108,0xa7a60101,0xfb840108,0xa7a70101,0xfb840108,0xa7a80101,0xfb840108,0xa7a90101,
	0xfb840108,0xa7aa0101,0xfb840108,0xa7ab0101,0xfb840108,0xa7ac0101,0xfb840108,0xa7ad0101,
	0xfb840108,0xa7ae0101,0xfb840108,0xa7af0101,0xfb840108,0xa7b00101,0xfb840108,0xa7b10101,
	0xfb840108,0xa7b20101,0xfb840108,0xa7b30101,0xfb840108,0xa7b40101,0xfb840108,0xa7b50101,
	0xfb840108,0xa7b60101,0xfb840108,0xa7b70101,0xfb840108,0xa7b80101,0xfb840108,0xa7b90101,
	0xfb840108,0xa7ba0101,0xfb840108,0xa7bb0101,0xfb840108,0xa7bc0101,0xfb840108,0xa7bd0101,
	0xfb840108,0xa7be0101,0xfb840108,0xa7bf0101,0xfb840108,0xa7c00101,0xfb840108,0xa7c10101,
	0xfb840108,0xa7c20101,0xfb840108,0xa7c30101,0xfb840108,0xa7c40101,0xfb840108,0xa7c50101,
	0xfb840108,0xa7c60101,0xfb840108,0xa7c70101,0xfb840108,0xa7c80101,0xfb840108,0xa7c90101,
	0xfb840108,0xa7ca0101,0xfb840108,0xa7cb0101,0xfb840108,0xa7cc0101,0xfb840108,0xa7cd0101,
	0xfb840108,0xa7ce0101,0xfb840108,0xa7cf0101,0xfb840108,0xa7d00101,0xfb840108,0xa7d10101,
	0xfb840108,0xa7d20101,0xfb840108,0xa7d30101,0xfb840108,0xa7d40101,0xfb840108,0xa7d50101,
	0xfb840108,0xa7d60101,0xfb840108,0xa7d70101,0xfb840108,0xa7d80101,0xfb840108,0xa7d90101,
	0xfb840108,0xa7da0101,0xfb840108,0xa7db0101,0xfb840108,0xa7dc0101,0xfb840108,0xa7dd0101,
	0xfb840108,0xa7de0101,0xfb840108,0xa7df0101,0xfb840108,0xa7e00101,0xfb840108,0xa7e10101,
	0xfb840108,0xa7e20101,0xfb840108,0xa7e30101,0xfb840108,0xa7e40101,0xfb840108,0xa7e50101,
	0xfb840108,0xa7e60101,0xfb840108,0xa7e70101,0xfb840108,0xa7e80101,0xfb840108,0xa7e90101,
	0xfb840108,0xa7ea0101,0xfb840108,0xa7eb0101,0xfb840108,0xa7ec0101,0xfb840108,0xa7ed0101,
	0xfb840108,0xa7ee0101,0xfb840108,0xa7ef0101,0xfb840108,0xa7f00101,0xfb840108,0xa7f10101,
	0xfb840108,0xa7f20101,0xfb840108,0xa7f30101,0xfb840108,0xa7f40101,0xfb840108,0xa7f50101,
	0xfb840108,0xa7f60101,0xfb840108,0xa7f70101,0xfb840108,0xa7f80101,0xfb840108,0xa7f90101,
	0xfb840108,0xa7fa0101,0xfb840108,0xa7fb0101,0xfb840108,0xa7fc0101,0xfb840108,0xa7fd0101,
	0xfb840108,0xa7fe0101,0xfb840108,0xa7ff0101,0xfb840108,0xa8000101,0xfb840108,0xa8010101,
	0xfb840108,0xa8020101,0xfb840108,0xa8030101,0xfb840108,0xa8040101,0xfb840108,0xa8050101,
	0xfb840108,0xa8060101,0xfb840108,0xa8070101,0xfb840108,0xa8080101,0xfb840108,0xa8090101,
	0xfb840108,0xa80a0101,0xfb840108,0xa80b0101,0xfb840108,0xa80c0101,0xfb840108,0xa80d0101,
	0xfb840108,0xa80e0101,0xfb840108,0xa80f0101,0xfb840108,0xa8100101,0xfb840108,0xa8110101,
	0xfb840108,0xa8120101,0xfb840108,0xa8130101,0xfb840108,0xa8140101,0xfb840108,0xa8150101,
	0xfb840108,0xa8160101,0xfb840108,0xa8170101,0xfb840108,0xa8180101,0xfb840108,0xa8190101,
	0xfb840108,0xa81a0101,0xfb840108,0xa81b0101,0xfb840108,0xa81c0101,0xfb840108,0xa81d0101,
	0xfb840108,0xa81e0101,0xfb840108,0xa81f0101,0xfb840108,0xa8200101,0xfb840108,0xa8210101,
	0xfb840108,0xa8220101,0xfb840108,0xa8230101,0xfb840108,0xa8240101,0xfb840108,0xa8250101,
	0xfb840108,0xa8260101,0xfb840108,0xa8270101,0xfb840108,0xa8280101,0xfb840108,0xa8290101,
	0xfb840108,0xa82a0101,0xfb840108,0xa82b0101,0xfb840108,0xa82c0101,0xfb840108,0xa82d0101,
	0xfb840108,0xa82e0101,0xfb840108,0xa82f0101,0xfb840108,0xa8300101,0xfb840108,0xa8310101,
	0xfb840108,0xa8320101,0xfb840108,0xa8330101,0xfb840108,0xa8340101,0xfb840108,0xa8350101,
	0xfb840108,0xa8360101,0xfb840108,0xa8370101,0xfb840108,0xa8380101,0xfb840108,0xa8390101,
	0xfb840108,0xa83a0101,0xfb840108,0xa83b0101,0xfb840108,0xa83c0101,0xfb840108,0xa83d0101,
	0xfb840108,0xa83e0101,0xfb840108,0xa83f0101,0xfb840108,0xa8400101,0xfb840108,0xa8410101,
	0xfb840108,0xa8420101,0xfb840108,0xa8430101,0xfb840108,0xa8440101,0xfb840108,0xa8450101,
	0xfb840108,0xa8460101,0xfb840108,0xa8470101,0xfb840108,0xa8480101,0xfb840108,0xa8490101,
	0xfb840108,0xa84a0101,0xfb840108,0xa84b0101,0xfb840108,0xa84c0101,0xfb840108,0xa84d0101,
	0xfb840108,0xa84e0101,0xfb840108,0xa84f0101,0xfb840108,0xa8500101,0xfb840108,0xa8510101,
	0xfb840108,0xa8520101,0xfb840108,0xa8530101,0xfb840108,0xa8540101,0xfb840108,0xa8550101,
	0xfb840108,0xa8560101,0xfb840108,0xa8570101,0xfb840108,0xa8580101,0xfb840108,0xa8590101,
	0xfb840108,0xa85a0101,0xfb840108,0xa85b0101,0xfb840108,0xa85c0101,0xfb840108,0xa85d0101,
	0xfb840108,0xa85e0101,0xfb840108,0xa85f0101,0xfb840108,0xa8600101,0xfb840108,0xa8610101,
	0xfb840108,0xa8620101,0xfb840108,0xa8630101,0xfb840108,0xa8640101,0xfb840108,0xa8650101,
	0xfb840108,0xa8660101,0xfb840108,0xa8670101,0xfb840108,0xa8680101,0xfb840108,0xa8690101,
	0xfb840108,0xa86a0101,0xfb840108,0xa86b0101,0xfb840108,0xa86c0101,0xfb840108,0xa86d0101,
	0xfb840108,0xa86e0101,0xfb840108,0xa86f0101,0xfb840108,0xa8700101,0xfb840108,0xa8710101,
	0xfb840108,0xa8720101,0xfb840108,0xa8730101,0xfb840108,0xa8740101,0xfb840108,0xa8750101,
	0xfb840108,0xa8760101,0xfb840108,0xa8770101,0xfb840108,0xa8780101,0xfb840108,0xa8790101,
	0xfb840108,0xa87a0101,0xfb840108,0xa87b0101,0xfb840108,0xa87c0101,0xfb840108,0xa87d0101,
	0xfb840108,0xa87e0101,0xfb840108,0xa87f0101,0xfb840108,0xa8800101,0xfb840108,0xa8810101,
	0xfb840108,0xa8820101,0xfb840108,0xa8830101,0xfb840108,0xa8840101,0xfb840108,0xa8850101,
	0xfb840108,0xa8860101,0xfb840108,0xa8870101,0xfb840108,0xa8880101,0xfb840108,0xa8890101,
	0xfb840108,0xa88a0101,0xfb840108,0xa88b0101,0xfb840108,0xa88c0101,0xfb840108,0xa88d0101,
	0xfb840108,0xa88e0101,0xfb840108,0xa88f0101,0xfb840108,0xa8900101,0xfb840108,0xa8910101,
	0xfb840108,0xa8920101,0xfb840108,0xa8930101,0xfb840108,0xa8940101,0xfb840108,0xa8950101,
	0xfb840108,0xa8960101,0xfb840108,0xa8970101,0xfb840108,0xa8980101,0xfb840108,0xa8990101,
	0xfb840108,0xa89a0101,0xfb840108,0xa89b0101,0xfb840108,0xa89c0101,0xfb840108,0xa89d0101,
	0xfb840108,0xa89e0101,0xfb840108,0xa89f0101,0xfb840108,0xa8a00101,0xfb840108,0xa8a10101,
	0xfb840108,0xa8a20101,0xfb840108,0xa8a30101,0xfb840108,0xa8a40101,0xfb840108,0xa8a50101,
	0xfb840108,0xa8a60101,0xfb840108,0xa8a70101,0xfb840108,0xa8a80101,0xfb840108,0xa8a90101,
	0xfb840108,0xa8aa0101,0xfb840108,0xa8ab0101,0xfb840108,0xa8ac0101,0xfb840108,0xa8ad0101,
	0xfb840108,0xa8ae0101,0xfb840108,0xa8af0101,0xfb840108,0xa8b00101,0xfb840108,0xa8b10101,
	0xfb840108,0xa8b20101,0xfb840108,0xa8b30101,0xfb840108,0xa8b40101,0xfb840108,0xa8b50101,
	0xfb840108,0xa8b60101,0xfb840108,0xa8b70101,0xfb840108,0xa8b80101,0xfb840108,0xa8b90101,
	0xfb840108,0xa8ba0101,0xfb840108,0xa8bb0101,0xfb840108,0xa8bc0101,0xfb840108,0xa8bd0101,
	0xfb840108,0xa8be0101,0xfb840108,0xa8bf0101,0xfb840108,0xa8c00101,0xfb840108,0xa8c10101,
	0xfb840108,0xa8c20101,0xfb840108,0xa8c30101,0xfb840108,0xa8c40101,0xfb840108,0xa8c50101,
	0xfb840108,0xa8c60101,0xfb840108,0xa8c70101,0xfb840108,0xa8c80101,0xfb840108,0xa8c90101,
	0xfb840108,0xa8ca0101,0xfb840108,0xa8cb0101,0xfb840108,0xa8cc0101,0xfb840108,0xa8cd0101,
	0xfb840108,0xa8ce0101,0xfb840108,0xa8cf0101,0xfb840108,0xa8d00101,0xfb840108,0xa8d10101,
	0xfb840108,0xa8d20101,0xfb840108,0xa8d30101,0xfb840108,0xa8d40101,0xfb840108,0xa8d50101,
	0xfb840108,0xa8d60101,0xfb840108,0xa8d70101,0xfb840108,0xa8d80101,0xfb840108,0xa8d90101,
	0xfb840108,0xa8da0101,0xfb840108,0xa8db0101,0xfb840108,0xa8dc0101,0xfb840108,0xa8dd0101,
	0xfb840108,0xa8de0101,0xfb840108,0xa8df0101,0xfb840108,0xa8e00101,0xfb840108,0xa8e10101,
	0xfb840108,0xa8e20101,0xfb840108,0xa8e30101,0xfb840108,0xa8e40101,0xfb840108,0xa8e50101,
	0xfb840108,0xa8e60101,0xfb840108,0xa8e70101,0xfb840108,0xa8e80101,0xfb840108,0xa8e90101,
	0xfb840108,0xa8ea0101,0xfb840108,0xa8eb0101,0xfb840108,0xa8ec0101,0xfb840108,0xa8ed0101,
	0xfb840108,0xa8ee0101,0xfb840108,0xa8ef0101,0xfb840108,0xa8f00101,0xfb840108,0xa8f10101,
	0xfb840108,0xa8f20101,0xfb840108,0xa8f30101,0xfb840108,0xa8f40101,0xfb840108,0xa8f50101,
	0xfb840108,0xa8f60101,0xfb840108,0xa8f70101,0xfb840108,0xa8f80101,0xfb840108,0xa8f90101,
	0xfb840108,0xa8fa0101,0xfb840108,0xa8fb0101,0xfb840108,0xa8fc0101,0xfb840108,0xa8fd0101,
	0xfb840108,0xa8fe0101,0xfb840108,0xa8ff0101,0xfb840108,0xa9000101,0xfb840108,0xa9010101,
	0xfb840108,0xa9020101,0xfb840108,0xa9030101,0xfb840108,0xa9040101,0xfb840108,0xa9050101,
	0xfb840108,0xa9060101,0xfb840108,0xa9070101,0xfb840108,0xa9080101,0xfb840108,0xa9090101,
	0xfb840108,0xa90a0101,0xfb840108,0xa90b0101,0xfb840108,0xa90c0101,0xfb840108,0xa90d0101,
	0xfb840108,0xa90e0101,0xfb840108,0xa90f0101,0xfb840108,0xa9100101,0xfb840108,0xa9110101,
	0xfb840108,0xa9120101,0xfb840108,0xa9130101,0xfb840108,0xa9140101,0xfb840108,0xa9150101,
	0xfb840108,0xa9160101,0xfb840108,0xa9170101,0xfb840108,0xa9180101,0xfb840108,0xa9190101,
	0xfb840108,0xa91a0101,0xfb840108,0xa91b0101,0xfb840108,0xa91c0101,0xfb840108,0xa91d0101,
	0xfb840108,0xa91e0101,0xfb840108,0xa91f0101,0xfb840108,0xa9200101,0xfb840108,0xa9210101,
	0xfb840108,0xa9220101,0xfb840108,0xa9230101,0xfb840108,0xa9240101,0xfb840108,0xa9250101,
	0xfb840108,0xa9260101,0xfb840108,0xa9270101,0xfb840108,0xa9280101,0xfb840108,0xa9290101,
	0xfb840108,0xa92a0101,0xfb840108,0xa92b0101,0xfb840108,0xa92c0101,0xfb840108,0xa92d0101,
	0xfb840108,0xa92e0101,0xfb840108,0xa92f0101,0xfb840108,0xa9300101,0xfb840108,0xa9310101,
	0xfb840108,0xa9320101,0xfb840108,0xa9330101,0xfb840108,0xa9340101,0xfb840108,0xa9350101,
	0xfb840108,0xa9360101,0xfb840108,0xa9370101,0xfb840108,0xa9380101,0xfb840108,0xa9390101,
	0xfb840108,0xa93a0101,0xfb840108,0xa93b0101,0xfb840108,0xa93c0101,0xfb840108,0xa93d0101,
	0xfb840108,0xa93e0101,0xfb840108,0xa93f0101,0xfb840108,0xa9400101,0xfb840108,0xa9410101,
	0xfb840108,0xa9420101,0xfb840108,0xa9430101,0xfb840108,0xa9440101,0xfb840108,0xa9450101,
	0xfb840108,0xa9460101,0xfb840108,0xa9470101,0xfb840108,0xa9480101,0xfb840108,0xa9490101,
	0xfb840108,0xa94a0101,0xfb840108,0xa94b0101,0xfb840108,0xa94c0101,0xfb840108,0xa94d0101,
	0xfb840108,0xa94e0101,0xfb840108,0xa94f0101,0xfb840108,0xa9500101,0xfb840108,0xa9510101,
	0xfb840108,0xa9520101,0xfb840108,0xa9530101,0xfb840108,0xa9540101,0xfb840108,0xa9550101,
	0xfb840108,0xa9560101,0xfb840108,0xa9570101,0xfb840108,0xa9580101,0xfb840108,0xa9590101,
	0xfb840108,0xa95a0101,0xfb840108,0xa95b0101,0xfb840108,0xa95c0101,0xfb840108,0xa95d0101,
	0xfb840108,0xa95e0101,0xfb840108,0xa95f0101,0xfb840108,0xa9600101,0xfb840108,0xa9610101,
	0xfb840108,0xa9620101,0xfb840108,0xa9630101,0xfb840108,0xa9640101,0xfb840108,0xa9650101,
	0xfb840108,0xa9660101,0xfb840108,0xa9670101,0xfb840108,0xa9680101,0xfb840108,0xa9690101,
	0xfb840108,0xa96a0101,0xfb840108,0xa96b0101,0xfb840108,0xa96c0101,0xfb840108,0xa96d0101,
	0xfb840108,0xa96e0101,0xfb840108,0xa96f0101,0xfb840108,0xa9700101,0xfb840108,0xa9710101,
	0xfb840108,0xa9720101,0xfb840108,0xa9730101,0xfb840108,0xa9740101,0xfb840108,0xa9750101,
	0xfb840108,0xa9760101,0xfb840108,0xa9770101,0xfb840108,0xa9780101,0xfb840108,0xa9790101,
	0xfb840108,0xa97a0101,0xfb840108,0xa97b0101,0xfb840108,0xa97c0101,0xfb840108,0xa97d0101,
	0xfb840108,0xa97e0101,0xfb840108,0xa97f0101,0xfb840108,0xa9800101,0xfb840108,0xa9810101,
	0xfb840108,0xa9820101,0xfb840108,0xa9830101,0xfb840108,0xa9840101,0xfb840108,0xa9850101,
	0xfb840108,0xa9860101,0xfb840108,0xa9870101,0xfb840108,0xa9880101,0xfb840108,0xa9890101,
	0xfb840108,0xa98a0101,0xfb840108,0xa98b0101,0xfb840108,0xa98c0101,0xfb840108,0xa98d0101,
	0xfb840108,0xa98e0101,0xfb840108,0xa98f0101,0xfb840108,0xa9900101,0xfb840108,0xa9910101,
	0xfb840108,0xa9920101,0xfb840108,0xa9930101,0xfb840108,0xa9940101,0xfb840108,0xa9950101,
	0xfb840108,0xa9960101,0xfb840108,0xa9970101,0xfb840108,0xa9980101,0xfb840108,0xa9990101,
	0xfb840108,0xa99a0101,0xfb840108,0xa99b0101,0xfb840108,0xa99c0101,0xfb840108,0xa99d0101,
	0xfb840108,0xa99e0101,0xfb840108,0xa99f0101,0xfb840108,0xa9a00101,0xfb840108,0xa9a10101,
	0xfb840108,0xa9a20101,0xfb840108,0xa9a30101,0xfb840108,0xa9a40101,0xfb840108,0xa9a50101,
	0xfb840108,0xa9a60101,0xfb840108,0xa9a70101,0xfb840108,0xa9a80101,0xfb840108,0xa9a90101,
	0xfb840108,0xa9aa0101,0xfb840108,0xa9ab0101,0xfb840108,0xa9ac0101,0xfb840108,0xa9ad0101,
	0xfb840108,0xa9ae0101,0xfb840108,0xa9af0101,0xfb840108,0xa9b00101,0xfb840108,0xa9b10101,
	0xfb840108,0xa9b20101,0xfb840108,0xa9b30101,0xfb840108,0xa9b40101,0xfb840108,0xa9b50101,
	0xfb840108,0xa9b60101,0xfb840108,0xa9b70101,0xfb840108,0xa9b80101,0xfb840108,0xa9b90101,
	0xfb840108,0xa9ba0101,0xfb840108,0xa9bb0101,0xfb840108,0xa9bc0101,0xfb840108,0xa9bd0101,
	0xfb840108,0xa9be0101,0xfb840108,0xa9bf0101,0xfb840108,0xa9c00101,0xfb840108,0xa9c10101,
	0xfb840108,0xa9c20101,0xfb840108,0xa9c30101,0xfb840108,0xa9c40101,0xfb840108,0xa9c50101,
	0xfb840108,0xa9c60101,0xfb840108,0xa9c70101,0xfb840108,0xa9c80101,0xfb840108,0xa9c90101,
	0xfb840108,0xa9ca0101,0xfb840108,0xa9cb0101,0xfb840108,0xa9cc0101,0xfb840108,0xa9cd0101,
	0xfb840108,0xa9ce0101,0xfb840108,0xa9cf0101,0xfb840108,0xa9d00101,0xfb840108,0xa9d10101,
	0xfb840108,0xa9d20101,0xfb840108,0xa9d30101,0xfb840108,0xa9d40101,0xfb840108,0xa9d50101,
	0xfb840108,0xa9d60101,0xfb840108,0xa9d70101,0xfb840108,0xa9d80101,0xfb840108,0xa9d90101,
	0xfb840108,0xa9da0101,0xfb840108,0xa9db0101,0xfb840108,0xa9dc0101,0xfb840108,0xa9dd0101,
	0xfb840108,0xa9de0101,0xfb840108,0xa9df0101,0xfb840108,0xa9e00101,0xfb840108,0xa9e10101,
	0xfb840108,0xa9e20101,0xfb840108,0xa9e30101,0xfb840108,0xa9e40101,0xfb840108,0xa9e50101,
	0xfb840108,0xa9e60101,0xfb840108,0xa9e70101,0xfb840108,0xa9e80101,0xfb840108,0xa9e90101,
	0xfb840108,0xa9ea0101,0xfb840108,0xa9eb0101,0xfb840108,0xa9ec0101,0xfb840108,0xa9ed0101,
	0xfb840108,0xa9ee0101,0xfb840108,0xa9ef0101,0xfb840108,0xa9f00101,0xfb840108,0xa9f10101,
	0xfb840108,0xa9f20101,0xfb840108,0xa9f30101,0xfb840108,0xa9f40101,0xfb840108,0xa9f50101,
	0xfb840108,0xa9f60101,0xfb840108,0xa9f70101,0xfb840108,0xa9f80101,0xfb840108,0xa9f90101,
	0xfb840108,0xa9fa0101,0xfb840108,0xa9fb0101,0xfb840108,0xa9fc0101,0xfb840108,0xa9fd0101,
	0xfb840108,0xa9fe0101,0xfb840108,0xa9ff0101,0xfb840108,0xaa000101,0xfb840108,0xaa010101,
	0xfb840108,0xaa020101,0xfb840108,0xaa030101,0xfb840108,0xaa040101,0xfb840108,0xaa050101,
	0xfb840108,0xaa060101,0xfb840108,0xaa070101,0xfb840108,0xaa080101,0xfb840108,0xaa090101,
	0xfb840108,0xaa0a0101,0xfb840108,0xaa0b0101,0xfb840108,0xaa0c0101,0xfb840108,0xaa0d0101,
	0xfb840108,0xaa0e0101,0xfb840108,0xaa0f0101,0xfb840108,0xaa100101,0xfb840108,0xaa110101,
	0xfb840108,0xaa120101,0xfb840108,0xaa130101,0xfb840108,0xaa140101,0xfb840108,0xaa150101,
	0xfb840108,0xaa160101,0xfb840108,0xaa170101,0xfb840108,0xaa180101,0xfb840108,0xaa190101,
	0xfb840108,0xaa1a0101,0xfb840108,0xaa1b0101,0xfb840108,0xaa1c0101,0xfb840108,0xaa1d0101,
	0xfb840108,0xaa1e0101,0xfb840108,0xaa1f0101,0xfb840108,0xaa200101,0xfb840108,0xaa210101,
	0xfb840108,0xaa220101,0xfb840108,0xaa230101,0xfb840108,0xaa240101,0xfb840108,0xaa250101,
	0xfb840108,0xaa260101,0xfb840108,0xaa270101,0xfb840108,0xaa280101,0xfb840108,0xaa290101,
	0xfb840108,0xaa2a0101,0xfb840108,0xaa2b0101,0xfb840108,0xaa2c0101,0xfb840108,0xaa2d0101,
	0xfb840108,0xaa2e0101,0xfb840108,0xaa2f0101,0xfb840108,0xaa300101,0xfb840108,0xaa310101,
	0xfb840108,0xaa320101,0xfb840108,0xaa330101,0xfb840108,0xaa340101,0xfb840108,0xaa350101,
	0xfb840108,0xaa360101,0xfb840108,0xaa370101,0xfb840108,0xaa380101,0xfb840108,0xaa390101,
	0xfb840108,0xaa3a0101,0xfb840108,0xaa3b0101,0xfb840108,0xaa3c0101,0xfb840108,0xaa3d0101,
	0xfb840108,0xaa3e0101,0xfb840108,0xaa3f0101,0xfb840108,0xaa400101,0xfb840108,0xaa410101,
	0xfb840108,0xaa420101,0xfb840108,0xaa430101,0xfb840108,0xaa440101,0xfb840108,0xaa450101,
	0xfb840108,0xaa460101,0xfb840108,0xaa470101,0xfb840108,0xaa480101,0xfb840108,0xaa490101,
	0xfb840108,0xaa4a0101,0xfb840108,0xaa4b0101,0xfb840108,0xaa4c0101,0xfb840108,0xaa4d0101,
	0xfb840108,0xaa4e0101,0xfb840108,0xaa4f0101,0xfb840108,0xaa500101,0xfb840108,0xaa510101,
	0xfb840108,0xaa520101,0xfb840108,0xaa530101,0xfb840108,0xaa540101,0xfb840108,0xaa550101,
	0xfb840108,0xaa560101,0xfb840108,0xaa570101,0xfb840108,0xaa580101,0xfb840108,0xaa590101,
	0xfb840108,0xaa5a0101,0xfb840108,0xaa5b0101,0xfb840108,0xaa5c0101,0xfb840108,0xaa5d0101,
	0xfb840108,0xaa5e0101,0xfb840108,0xaa5f0101,0xfb840108,0xaa600101,0xfb840108,0xaa610101,
	0xfb840108,0xaa620101,0xfb840108,0xaa630101,0xfb840108,0xaa640101,0xfb840108,0xaa650101,
	0xfb840108,0xaa660101,0xfb840108,0xaa670101,0xfb840108,0xaa680101,0xfb840108,0xaa690101,
	0xfb840108,0xaa6a0101,0xfb840108,0xaa6b0101,0xfb840108,0xaa6c0101,0xfb840108,0xaa6d0101,
	0xfb840108,0xaa6e0101,0xfb840108,0xaa6f0101,0xfb840108,0xaa700101,0xfb840108,0xaa710101,
	0xfb840108,0xaa720101,0xfb840108,0xaa730101,0xfb840108,0xaa740101,0xfb840108,0xaa750101,
	0xfb840108,0xaa760101,0xfb840108,0xaa770101,0xfb840108,0xaa780101,0xfb840108,0xaa790101,
	0xfb840108,0xaa7a0101,0xfb840108,0xaa7b0101,0xfb840108,0xaa7c0101,0xfb840108,0xaa7d0101,
	0xfb840108,0xaa7e0101,0xfb840108,0xaa7f0101,0xfb840108,0xaa800101,0xfb840108,0xaa810101,
	0xfb840108,0xaa820101,0xfb840108,0xaa830101,0xfb840108,0xaa840101,0xfb840108,0xaa850101,
	0xfb840108,0xaa860101,0xfb840108,0xaa870101,0xfb840108,0xaa880101,0xfb840108,0xaa890101,
	0xfb840108,0xaa8a0101,0xfb840108,0xaa8b0101,0xfb840108,0xaa8c0101,0xfb840108,0xaa8d0101,
	0xfb840108,0xaa8e0101,0xfb840108,0xaa8f0101,0xfb840108,0xaa900101,0xfb840108,0xaa910101,
	0xfb840108,0xaa920101,0xfb840108,0xaa930101,0xfb840108,0xaa940101,0xfb840108,0xaa950101,
	0xfb840108,0xaa960101,0xfb840108,0xaa970101,0xfb840108,0xaa980101,0xfb840108,0xaa990101,
	0xfb840108,0xaa9a0101,0xfb840108,0xaa9b0101,0xfb840108,0xaa9c0101,0xfb840108,0xaa9d0101,
	0xfb840108,0xaa9e0101,0xfb840108,0xaa9f0101,0xfb840108,0xaaa00101,0xfb840108,0xaaa10101,
	0xfb840108,0xaaa20101,0xfb840108,0xaaa30101,0xfb840108,0xaaa40101,0xfb840108,0xaaa50101,
	0xfb840108,0xaaa60101,0xfb840108,0xaaa70101,0xfb840108,0xaaa80101,0xfb840108,0xaaa90101,
	0xfb840108,0xaaaa0101,0xfb840108,0xaaab0101,0xfb840108,0xaaac0101,0xfb840108,0xaaad0101,
	0xfb840108,0xaaae0101,0xfb840108,0xaaaf0101,0xfb840108,0xaab00101,0xfb840108,0xaab10101,
	0xfb840108,0xaab20101,0xfb840108,0xaab30101,0xfb840108,0xaab40101,0xfb840108,0xaab50101,
	0xfb840108,0xaab60101,0xfb840108,0xaab70101,0xfb840108,0xaab80101,0xfb840108,0xaab90101,
	0xfb840108,0xaaba0101,0xfb840108,0xaabb0101,0xfb840108,0xaabc0101,0xfb840108,0xaabd0101,
	0xfb840108,0xaabe0101,0xfb840108,0xaabf0101,0xfb840108,0xaac00101,0xfb840108,0xaac10101,
	0xfb840108,0xaac20101,0xfb840108,0xaac30101,0xfb840108,0xaac40101,0xfb840108,0xaac50101,
	0xfb840108,0xaac60101,0xfb840108,0xaac70101,0xfb840108,0xaac80101,0xfb840108,0xaac90101,
	0xfb840108,0xaaca0101,0xfb840108,0xaacb0101,0xfb840108,0xaacc0101,0xfb840108,0xaacd0101,
	0xfb840108,0xaace0101,0xfb840108,0xaacf0101,0xfb840108,0xaad00101,0xfb840108,0xaad10101,
	0xfb840108,0xaad20101,0xfb840108,0xaad30101,0xfb840108,0xaad40101,0xfb840108,0xaad50101,
	0xfb840108,0xaad60101,0xfb840108,0xaad70101,0xfb840108,0xaad80101,0xfb840108,0xaad90101,
	0xfb840108,0xaada0101,0xfb840108,0xaadb0101,0xfb840108,0xaadc0101,0xfb840108,0xaadd0101,
	0xfb840108,0xaade0101,0xfb840108,0xaadf0101,0xfb840108,0xaae00101,0xfb840108,0xaae10101,
	0xfb840108,0xaae20101,0xfb840108,0xaae30101,0xfb840108,0xaae40101,0xfb840108,0xaae50101,
	0xfb840108,0xaae60101,0xfb840108,0xaae70101,0xfb840108,0xaae80101,0xfb840108,0xaae90101,
	0xfb840108,0xaaea0101,0xfb840108,0xaaeb0101,0xfb840108,0xaaec0101,0xfb840108,0xaaed0101,
	0xfb840108,0xaaee0101,0xfb840108,0xaaef0101,0xfb840108,0xaaf00101,0xfb840108,0xaaf10101,
	0xfb840108,0xaaf20101,0xfb840108,0xaaf30101,0xfb840108,0xaaf40101,0xfb840108,0xaaf50101,
	0xfb840108,0xaaf60101,0xfb840108,0xaaf70101,0xfb840108,0xaaf80101,0xfb840108,0xaaf90101,
	0xfb840108,0xaafa0101,0xfb840108,0xaafb0101,0xfb840108,0xaafc0101,0xfb840108,0xaafd0101,
	0xfb840108,0xaafe0101,0xfb840108,0xaaff0101,0xfb840108,0xab000101,0xfb840108,0xab010101,
	0xfb840108,0xab020101,0xfb840108,0xab030101,0xfb840108,0xab040101,0xfb840108,0xab050101,
	0xfb840108,0xab060101,0xfb840108,0xab070101,0xfb840108,0xab080101,0xfb840108,0xab090101,
	0xfb840108,0xab0a0101,0xfb840108,0xab0b0101,0xfb840108,0xab0c0101,0xfb840108,0xab0d0101,
	0xfb840108,0xab0e0101,0xfb840108,0xab0f0101,0xfb840108,0xab100101,0xfb840108,0xab110101,
	0xfb840108,0xab120101,0xfb840108,0xab130101,0xfb840108,0xab140101,0xfb840108,0xab150101,
	0xfb840108,0xab160101,0xfb840108,0xab170101,0xfb840108,0xab180101,0xfb840108,0xab190101,
	0xfb840108,0xab1a0101,0xfb840108,0xab1b0101,0xfb840108,0xab1c0101,0xfb840108,0xab1d0101,
	0xfb840108,0xab1e0101,0xfb840108,0xab1f0101,0xfb840108,0xab200101,0xfb840108,0xab210101,
	0xfb840108,0xab220101,0xfb840108,0xab230101,0xfb840108,0xab240101,0xfb840108,0xab250101,
	0xfb840108,0xab260101,0xfb840108,0xab270101,0xfb840108,0xab280101,0xfb840108,0xab290101,
	0xfb840108,0xab2a0101,0xfb840108,0xab2b0101,0xfb840108,0xab2c0101,0xfb840108,0xab2d0101,
	0xfb840108,0xab2e0101,0xfb840108,0xab2f0101,0xfb840108,0xab300101,0xfb840108,0xab310101,
	0xfb840108,0xab320101,0xfb840108,0xab330101,0xfb840108,0xab340101,0xfb840108,0xab350101,
	0xfb840108,0xab360101,0xfb840108,0xab370101,0xfb840108,0xab380101,0xfb840108,0xab390101,
	0xfb840108,0xab3a0101,0xfb840108,0xab3b0101,0xfb840108,0xab3c0101,0xfb840108,0xab3d0101,
	0xfb840108,0xab3e0101,0xfb840108,0xab3f0101,0xfb840108,0xab400101,0xfb840108,0xab410101,
	0xfb840108,0xab420101,0xfb840108,0xab430101,0xfb840108,0xab440101,0xfb840108,0xab450101,
	0xfb840108,0xab460101,0xfb840108,0xab470101,0xfb840108,0xab480101,0xfb840108,0xab490101,
	0xfb840108,0xab4a0101,0xfb840108,0xab4b0101,0xfb840108,0xab4c0101,0xfb840108,0xab4d0101,
	0xfb840108,0xab4e0101,0xfb840108,0xab4f0101,0xfb840108,0xab500101,0xfb840108,0xab510101,
	0xfb840108,0xab520101,0xfb840108,0xab530101,0xfb840108,0xab540101,0xfb840108,0xab550101,
	0xfb840108,0xab560101,0xfb840108,0xab570101,0xfb840108,0xab580101,0xfb840108,0xab590101,
	0xfb840108,0xab5a0101,0xfb840108,0xab5b0101,0xfb840108,0xab5c0101,0xfb840108,0xab5d0101,
	0xfb840108,0xab5e0101,0xfb840108,0xab5f0101,0xfb840108,0xab600101,0xfb840108,0xab610101,
	0xfb840108,0xab620101,0xfb840108,0xab630101,0xfb840108,0xab640101,0xfb840108,0xab650101,
	0xfb840108,0xab660101,0xfb840108,0xab670101,0xfb840108,0xab680101,0xfb840108,0xab690101,
	0xfb840108,0xab6a0101,0xfb840108,0xab6b0101,0xfb840108,0xab6c0101,0xfb840108,0xab6d0101,
	0xfb840108,0xab6e0101,0xfb840108,0xab6f0101,0xfb840108,0xab700101,0xfb840108,0xab710101,
	0xfb840108,0xab720101,0xfb840108,0xab730101,0xfb840108,0xab740101,0xfb840108,0xab750101,
	0xfb840108,0xab760101,0xfb840108,0xab770101,0xfb840108,0xab780101,0xfb840108,0xab790101,
	0xfb840108,0xab7a0101,0xfb840108,0xab7b0101,0xfb840108,0xab7c0101,0xfb840108,0xab7d0101,
	0xfb840108,0xab7e0101,0xfb840108,0xab7f0101,0xfb840108,0xab800101,0xfb840108,0xab810101,
	0xfb840108,0xab820101,0xfb840108,0xab830101,0xfb840108,0xab840101,0xfb840108,0xab850101,
	0xfb840108,0xab860101,0xfb840108,0xab870101,0xfb840108,0xab880101,0xfb840108,0xab890101,
	0xfb840108,0xab8a0101,0xfb840108,0xab8b0101,0xfb840108,0xab8c0101,0xfb840108,0xab8d0101,
	0xfb840108,0xab8e0101,0xfb840108,0xab8f0101,0xfb840108,0xab900101,0xfb840108,0xab910101,
	0xfb840108,0xab920101,0xfb840108,0xab930101,0xfb840108,0xab940101,0xfb840108,0xab950101,
	0xfb840108,0xab960101,0xfb840108,0xab970101,0xfb840108,0xab980101,0xfb840108,0xab990101,
	0xfb840108,0xab9a0101,0xfb840108,0xab9b0101,0xfb840108,0xab9c0101,0xfb840108,0xab9d0101,
	0xfb840108,0xab9e0101,0xfb840108,0xab9f0101,0xfb840108,0xaba00101,0xfb840108,0xaba10101,
	0xfb840108,0xaba20101,0xfb840108,0xaba30101,0xfb840108,0xaba40101,0xfb840108,0xaba50101,
	0xfb840108,0xaba60101,0xfb840108,0xaba70101,0xfb840108,0xaba80101,0xfb840108,0xaba90101,
	0xfb840108,0xabaa0101,0xfb840108,0xabab0101,0xfb840108,0xabac0101,0xfb840108,0xabad0101,
	0xfb840108,0xabae0101,0xfb840108,0xabaf0101,0xfb840108,0xabb00101,0xfb840108,0xabb10101,
	0xfb840108,0xabb20101,0xfb840108,0xabb30101,0xfb840108,0xabb40101,0xfb840108,0xabb50101,
	0xfb840108,0xabb60101,0xfb840108,0xabb70101,0xfb840108,0xabb80101,0xfb840108,0xabb90101,
	0xfb840108,0xabba0101,0xfb840108,0xabbb0101,0xfb840108,0xabbc0101,0xfb840108,0xabbd0101,
	0xfb840108,0xabbe0101,0xfb840108,0xabbf0101,0xfb840108,0xabc00101,0xfb840108,0xabc10101,
	0xfb840108,0xabc20101,0xfb840108,0xabc30101,0xfb840108,0xabc40101,0xfb840108,0xabc50101,
	0xfb840108,0xabc60101,0xfb840108,0xabc70101,0xfb840108,0xabc80101,0xfb840108,0xabc90101,
	0xfb840108,0xabca0101,0xfb840108,0xabcb0101,0xfb840108,0xabcc0101,0xfb840108,0xabcd0101,
	0xfb840108,0xabce0101,0xfb840108,0xabcf0101,0xfb840108,0xabd00101,0xfb840108,0xabd10101,
	0xfb840108,0xabd20101,0xfb840108,0xabd30101,0xfb840108,0xabd40101,0xfb840108,0xabd50101,
	0xfb840108,0xabd60101,0xfb840108,0xabd70101,0xfb840108,0xabd80101,0xfb840108,0xabd90101,
	0xfb840108,0xabda0101,0xfb840108,0xabdb0101,0xfb840108,0xabdc0101,0xfb840108,0xabdd0101,
	0xfb840108,0xabde0101,0xfb840108,0xabdf0101,0xfb840108,0xabe00101,0xfb840108,0xabe10101,
	0xfb840108,0xabe20101,0xfb840108,0xabe30101,0xfb840108,0xabe40101,0xfb840108,0xabe50101,
	0xfb840108,0xabe60101,0xfb840108,0xabe70101,0xfb840108,0xabe80101,0xfb840108,0xabe90101,
	0xfb840108,0xabea0101,0xfb840108,0xabeb0101,0xfb840108,0xabec0101,0xfb840108,0xabed0101,
	0xfb840108,0xabee0101,0xfb840108,0xabef0101,0xfb840108,0xabf00101,0xfb840108,0xabf10101,
	0xfb840108,0xabf20101,0xfb840108,0xabf30101,0xfb840108,0xabf40101,0xfb840108,0xabf50101,
	0xfb840108,0xabf60101,0xfb840108,0xabf70101,0xfb840108,0xabf80101,0xfb840108,0xabf90101,
	0xfb840108,0xabfa0101,0xfb840108,0xabfb0101,0xfb840108,0xabfc0101,0xfb840108,0xabfd0101,
	0xfb840108,0xabfe0101,0xfb840108,0xabff0101,0xfb840108,0xac000101,0xfb840108,0xac010101,
	0xfb840108,0xac020101,0xfb840108,0xac030101,0xfb840108,0xac040101,0xfb840108,0xac050101,
	0xfb840108,0xac060101,0xfb840108,0xac070101,0xfb840108,0xac080101,0xfb840108,0xac090101,
	0xfb840108,0xac0a0101,0xfb840108,0xac0b0101,0xfb840108,0xac0c0101,0xfb840108,0xac0d0101,
	0xfb840108,0xac0e0101,0xfb840108,0xac0f0101,0xfb840108,0xac100101,0xfb840108,0xac110101,
	0xfb840108,0xac120101,0xfb840108,0xac130101,0xfb840108,0xac140101,0xfb840108,0xac150101,
	0xfb840108,0xac160101,0xfb840108,0xac170101,0xfb840108,0xac180101,0xfb840108,0xac190101,
	0xfb840108,0xac1a0101,0xfb840108,0xac1b0101,0xfb840108,0xac1c0101,0xfb840108,0xac1d0101,
	0xfb840108,0xac1e0101,0xfb840108,0xac1f0101,0xfb840108,0xac200101,0xfb840108,0xac210101,
	0xfb840108,0xac220101,0xfb840108,0xac230101,0xfb840108,0xac240101,0xfb840108,0xac250101,
	0xfb840108,0xac260101,0xfb840108,0xac270101,0xfb840108,0xac280101,0xfb840108,0xac290101,
	0xfb840108,0xac2a0101,0xfb840108,0xac2b0101,0xfb840108,0xac2c0101,0xfb840108,0xac2d0101,
	0xfb840108,0xac2e0101,0xfb840108,0xac2f0101,0xfb840108,0xac300101,0xfb840108,0xac310101,
	0xfb840108,0xac320101,0xfb840108,0xac330101,0xfb840108,0xac340101,0xfb840108,0xac350101,
	0xfb840108,0xac360101,0xfb840108,0xac370101,0xfb840108,0xac380101,0xfb840108,0xac390101,
	0xfb840108,0xac3a0101,0xfb840108,0xac3b0101,0xfb840108,0xac3c0101,0xfb840108,0xac3d0101,
	0xfb840108,0xac3e0101,0xfb840108,0xac3f0101,0xfb840108,0xac400101,0xfb840108,0xac410101,
	0xfb840108,0xac420101,0xfb840108,0xac430101,0xfb840108,0xac440101,0xfb840108,0xac450101,
	0xfb840108,0xac460101,0xfb840108,0xac470101,0xfb840108,0xac480101,0xfb840108,0xac490101,
	0xfb840108,0xac4a0101,0xfb840108,0xac4b0101,0xfb840108,0xac4c0101,0xfb840108,0xac4d0101,
	0xfb840108,0xac4e0101,0xfb840108,0xac4f0101,0xfb840108,0xac500101,0xfb840108,0xac510101,
	0xfb840108,0xac520101,0xfb840108,0xac530101,0xfb840108,0xac540101,0xfb840108,0xac550101,
	0xfb840108,0xac560101,0xfb840108,0xac570101,0xfb840108,0xac580101,0xfb840108,0xac590101,
	0xfb840108,0xac5a0101,0xfb840108,0xac5b0101,0xfb840108,0xac5c0101,0xfb840108,0xac5d0101,
	0xfb840108,0xac5e0101,0xfb840108,0xac5f0101,0xfb840108,0xac600101,0xfb840108,0xac610101,
	0xfb840108,0xac620101,0xfb840108,0xac630101,0xfb840108,0xac640101,0xfb840108,0xac650101,
	0xfb840108,0xac660101,0xfb840108,0xac670101,0xfb840108,0xac680101,0xfb840108,0xac690101,
	0xfb840108,0xac6a0101,0xfb840108,0xac6b0101,0xfb840108,0xac6c0101,0xfb840108,0xac6d0101,
	0xfb840108,0xac6e0101,0xfb840108,0xac6f0101,0xfb840108,0xac700101,0xfb840108,0xac710101,
	0xfb840108,0xac720101,0xfb840108,0xac730101,0xfb840108,0xac740101,0xfb840108,0xac750101,
	0xfb840108,0xac760101,0xfb840108,0xac770101,0xfb840108,0xac780101,0xfb840108,0xac790101,
	0xfb840108,0xac7a0101,0xfb840108,0xac7b0101,0xfb840108,0xac7c0101,0xfb840108,0xac7d0101,
	0xfb840108,0xac7e0101,0xfb840108,0xac7f0101,0xfb840108,0xac800101,0xfb840108,0xac810101,
	0xfb840108,0xac820101,0xfb840108,0xac830101,0xfb840108,0xac840101,0xfb840108,0xac850101,
	0xfb840108,0xac860101,0xfb840108,0xac870101,0xfb840108,0xac880101,0xfb840108,0xac890101,
	0xfb840108,0xac8a0101,0xfb840108,0xac8b0101,0xfb840108,0xac8c0101,0xfb840108,0xac8d0101,
	0xfb840108,0xac8e0101,0xfb840108,0xac8f0101,0xfb840108,0xac900101,0xfb840108,0xac910101,
	0xfb840108,0xac920101,0xfb840108,0xac930101,0xfb840108,0xac940101,0xfb840108,0xac950101,
	0xfb840108,0xac960101,0xfb840108,0xac970101,0xfb840108,0xac980101,0xfb840108,0xac990101,
	0xfb840108,0xac9a0101,0xfb840108,0xac9b0101,0xfb840108,0xac9c0101,0xfb840108,0xac9d0101,
	0xfb840108,0xac9e0101,0xfb840108,0xac9f0101,0xfb840108,0xaca00101,0xfb840108,0xaca10101,
	0xfb840108,0xaca20101,0xfb840108,0xaca30101,0xfb840108,0xaca40101,0xfb840108,0xaca50101,
	0xfb840108,0xaca60101,0xfb840108,0xaca70101,0xfb840108,0xaca80101,0xfb840108,0xaca90101,
	0xfb840108,0xacaa0101,0xfb840108,0xacab0101,0xfb840108,0xacac0101,0xfb840108,0xacad0101,
	0xfb840108,0xacae0101,0xfb840108,0xacaf0101,0xfb840108,0xacb00101,0xfb840108,0xacb10101,
	0xfb840108,0xacb20101,0xfb840108,0xacb30101,0xfb840108,0xacb40101,0xfb840108,0xacb50101,
	0xfb840108,0xacb60101,0xfb840108,0xacb70101,0xfb840108,0xacb80101,0xfb840108,0xacb90101,
	0xfb840108,0xacba0101,0xfb840108,0xacbb0101,0xfb840108,0xacbc0101,0xfb840108,0xacbd0101,
	0xfb840108,0xacbe0101,0xfb840108,0xacbf0101,0xfb840108,0xacc00101,0xfb840108,0xacc10101,
	0xfb840108,0xacc20101,0xfb840108,0xacc30101,0xfb840108,0xacc40101,0xfb840108,0xacc50101,
	0xfb840108,0xacc60101,0xfb840108,0xacc70101,0xfb840108,0xacc80101,0xfb840108,0xacc90101,
	0xfb840108,0xacca0101,0xfb840108,0xaccb0101,0xfb840108,0xaccc0101,0xfb840108,0xaccd0101,
	0xfb840108,0xacce0101,0xfb840108,0xaccf0101,0xfb840108,0xacd00101,0xfb840108,0xacd10101,
	0xfb840108,0xacd20101,0xfb840108,0xacd30101,0xfb840108,0xacd40101,0xfb840108,0xacd50101,
	0xfb840108,0xacd60101,0xfb840108,0xacd70101,0xfb840108,0xacd80101,0xfb840108,0xacd90101,
	0xfb840108,0xacda0101,0xfb840108,0xacdb0101,0xfb840108,0xacdc0101,0xfb840108,0xacdd0101,
	0xfb840108,0xacde0101,0xfb840108,0xacdf0101,0xfb840108,0xace00101,0xfb840108,0xace10101,
	0xfb840108,0xace20101,0xfb840108,0xace30101,0xfb840108,0xace40101,0xfb840108,0xace50101,
	0xfb840108,0xace60101,0xfb840108,0xace70101,0xfb840108,0xace80101,0xfb840108,0xace90101,
	0xfb840108,0xacea0101,0xfb840108,0xaceb0101,0xfb840108,0xacec0101,0xfb840108,0xaced0101,
	0xfb840108,0xacee0101,0xfb840108,0xacef0101,0xfb840108,0xacf00101,0xfb840108,0xacf10101,
	0xfb840108,0xacf20101,0xfb840108,0xacf30101,0xfb840108,0xacf40101,0xfb840108,0xacf50101,
	0xfb840108,0xacf60101,0xfb840108,0xacf70101,0xfb840108,0xacf80101,0xfb840108,0xacf90101,
	0xfb840108,0xacfa0101,0xfb840108,0xacfb0101,0xfb840108,0xacfc0101,0xfb840108,0xacfd0101,
	0xfb840108,0xacfe0101,0xfb840108,0xacff0101,0xfb840108,0xad000101,0xfb840108,0xad010101,
	0xfb840108,0xad020101,0xfb840108,0xad030101,0xfb840108,0xad040101,0xfb840108,0xad050101,
	0xfb840108,0xad060101,0xfb840108,0xad070101,0xfb840108,0xad080101,0xfb840108,0xad090101,
	0xfb840108,0xad0a0101,0xfb840108,0xad0b0101,0xfb840108,0xad0c0101,0xfb840108,0xad0d0101,
	0xfb840108,0xad0e0101,0xfb840108,0xad0f0101,0xfb840108,0xad100101,0xfb840108,0xad110101,
	0xfb840108,0xad120101,0xfb840108,0xad130101,0xfb840108,0xad140101,0xfb840108,0xad150101,
	0xfb840108,0xad160101,0xfb840108,0xad170101,0xfb840108,0xad180101,0xfb840108,0xad190101,
	0xfb840108,0xad1a0101,0xfb840108,0xad1b0101,0xfb840108,0xad1c0101,0xfb840108,0xad1d0101,
	0xfb840108,0xad1e0101,0xfb840108,0xad1f0101,0xfb840108,0xad200101,0xfb840108,0xad210101,
	0xfb840108,0xad220101,0xfb840108,0xad230101,0xfb840108,0xad240101,0xfb840108,0xad250101,
	0xfb840108,0xad260101,0xfb840108,0xad270101,0xfb840108,0xad280101,0xfb840108,0xad290101,
	0xfb840108,0xad2a0101,0xfb840108,0xad2b0101,0xfb840108,0xad2c0101,0xfb840108,0xad2d0101,
	0xfb840108,0xad2e0101,0xfb840108,0xad2f0101,0xfb840108,0xad300101,0xfb840108,0xad310101,
	0xfb840108,0xad320101,0xfb840108,0xad330101,0xfb840108,0xad340101,0xfb840108,0xad350101,
	0xfb840108,0xad360101,0xfb840108,0xad370101,0xfb840108,0xad380101,0xfb840108,0xad390101,
	0xfb840108,0xad3a0101,0xfb840108,0xad3b0101,0xfb840108,0xad3c0101,0xfb840108,0xad3d0101,
	0xfb840108,0xad3e0101,0xfb840108,0xad3f0101,0xfb840108,0xad400101,0xfb840108,0xad410101,
	0xfb840108,0xad420101,0xfb840108,0xad430101,0xfb840108,0xad440101,0xfb840108,0xad450101,
	0xfb840108,0xad460101,0xfb840108,0xad470101,0xfb840108,0xad480101,0xfb840108,0xad490101,
	0xfb840108,0xad4a0101,0xfb840108,0xad4b0101,0xfb840108,0xad4c0101,0xfb840108,0xad4d0101,
	0xfb840108,0xad4e0101,0xfb840108,0xad4f0101,0xfb840108,0xad500101,0xfb840108,0xad510101,
	0xfb840108,0xad520101,0xfb840108,0xad530101,0xfb840108,0xad540101,0xfb840108,0xad550101,
	0xfb840108,0xad560101,0xfb840108,0xad570101,0xfb840108,0xad580101,0xfb840108,0xad590101,
	0xfb840108,0xad5a0101,0xfb840108,0xad5b0101,0xfb840108,0xad5c0101,0xfb840108,0xad5d0101,
	0xfb840108,0xad5e0101,0xfb840108,0xad5f0101,0xfb840108,0xad600101,0xfb840108,0xad610101,
	0xfb840108,0xad620101,0xfb840108,0xad630101,0xfb840108,0xad640101,0xfb840108,0xad650101,
	0xfb840108,0xad660101,0xfb840108,0xad670101,0xfb840108,0xad680101,0xfb840108,0xad690101,
	0xfb840108,0xad6a0101,0xfb840108,0xad6b0101,0xfb840108,0xad6c0101,0xfb840108,0xad6d0101,
	0xfb840108,0xad6e0101,0xfb840108,0xad6f0101,0xfb840108,0xad700101,0xfb840108,0xad710101,
	0xfb840108,0xad720101,0xfb840108,0xad730101,0xfb840108,0xad740101,0xfb840108,0xad750101,
	0xfb840108,0xad760101,0xfb840108,0xad770101,0xfb840108,0xad780101,0xfb840108,0xad790101,
	0xfb840108,0xad7a0101,0xfb840108,0xad7b0101,0xfb840108,0xad7c0101,0xfb840108,0xad7d0101,
	0xfb840108,0xad7e0101,0xfb840108,0xad7f0101,0xfb840108,0xad800101,0xfb840108,0xad810101,
	0xfb840108,0xad820101,0xfb840108,0xad830101,0xfb840108,0xad840101,0xfb840108,0xad850101,
	0xfb840108,0xad860101,0xfb840108,0xad870101,0xfb840108,0xad880101,0xfb840108,0xad890101,
	0xfb840108,0xad8a0101,0xfb840108,0xad8b0101,0xfb840108,0xad8c0101,0xfb840108,0xad8d0101,
	0xfb840108,0xad8e0101,0xfb840108,0xad8f0101,0xfb840108,0xad900101,0xfb840108,0xad910101,
	0xfb840108,0xad920101,0xfb840108,0xad930101,0xfb840108,0xad940101,0xfb840108,0xad950101,
	0xfb840108,0xad960101,0xfb840108,0xad970101,0xfb840108,0xad980101,0xfb840108,0xad990101,
	0xfb840108,0xad9a0101,0xfb840108,0xad9b0101,0xfb840108,0xad9c0101,0xfb840108,0xad9d0101,
	0xfb840108,0xad9e0101,0xfb840108,0xad9f0101,0xfb840108,0xada00101,0xfb840108,0xada10101,
	0xfb840108,0xada20101,0xfb840108,0xada30101,0xfb840108,0xada40101,0xfb840108,0xada50101,
	0xfb840108,0xada60101,0xfb840108,0xada70101,0xfb840108,0xada80101,0xfb840108,0xada90101,
	0xfb840108,0xadaa0101,0xfb840108,0xadab0101,0xfb840108,0xadac0101,0xfb840108,0xadad0101,
	0xfb840108,0xadae0101,0xfb840108,0xadaf0101,0xfb840108,0xadb00101,0xfb840108,0xadb10101,
	0xfb840108,0xadb20101,0xfb840108,0xadb30101,0xfb840108,0xadb40101,0xfb840108,0xadb50101,
	0xfb840108,0xadb60101,0xfb840108,0xadb70101,0xfb840108,0xadb80101,0xfb840108,0xadb90101,
	0xfb840108,0xadba0101,0xfb840108,0xadbb0101,0xfb840108,0xadbc0101,0xfb840108,0xadbd0101,
	0xfb840108,0xadbe0101,0xfb840108,0xadbf0101,0xfb840108,0xadc00101,0xfb840108,0xadc10101,
	0xfb840108,0xadc20101,0xfb840108,0xadc30101,0xfb840108,0xadc40101,0xfb840108,0xadc50101,
	0xfb840108,0xadc60101,0xfb840108,0xadc70101,0xfb840108,0xadc80101,0xfb840108,0xadc90101,
	0xfb840108,0xadca0101,0xfb840108,0xadcb0101,0xfb840108,0xadcc0101,0xfb840108,0xadcd0101,
	0xfb840108,0xadce0101,0xfb840108,0xadcf0101,0xfb840108,0xadd00101,0xfb840108,0xadd10101,
	0xfb840108,0xadd20101,0xfb840108,0xadd30101,0xfb840108,0xadd40101,0xfb840108,0xadd50101,
	0xfb840108,0xadd60101,0xfb840108,0xadd70101,0xfb840108,0xadd80101,0xfb840108,0xadd90101,
	0xfb840108,0xadda0101,0xfb840108,0xaddb0101,0xfb840108,0xaddc0101,0xfb840108,0xaddd0101,
	0xfb840108,0xadde0101,0xfb840108,0xaddf0101,0xfb840108,0xade00101,0xfb840108,0xade10101,
	0xfb840108,0xade20101,0xfb840108,0xade30101,0xfb840108,0xade40101,0xfb840108,0xade50101,
	0xfb840108,0xade60101,0xfb840108,0xade70101,0xfb840108,0xade80101,0xfb840108,0xade90101,
	0xfb840108,0xadea0101,0xfb840108,0xadeb0101,0xfb840108,0xadec0101,0xfb840108,0xaded0101,
	0xfb840108,0xadee0101,0xfb840108,0xadef0101,0xfb840108,0xadf00101,0xfb840108,0xadf10101,
	0xfb840108,0xadf20101,0xfb840108,0xadf30101,0xfb840108,0xadf40101,0xfb840108,0xadf50101,
	0xfb840108,0xadf60101,0xfb840108,0xadf70101,0xfb840108,0xadf80101,0xfb840108,0xadf90101,
	0xfb840108,0xadfa0101,0xfb840108,0xadfb0101,0xfb840108,0xadfc0101,0xfb840108,0xadfd0101,
	0xfb840108,0xadfe0101,0xfb840108,0xadff0101,0xfb840108,0xae000101,0xfb840108,0xae010101,
	0xfb840108,0xae020101,0xfb840108,0xae030101,0xfb840108,0xae040101,0xfb840108,0xae050101,
	0xfb840108,0xae060101,0xfb840108,0xae070101,0xfb840108,0xae080101,0xfb840108,0xae090101,
	0xfb840108,0xae0a0101,0xfb840108,0xae0b0101,0xfb840108,0xae0c0101,0xfb840108,0xae0d0101,
	0xfb840108,0xae0e0101,0xfb840108,0xae0f0101,0xfb840108,0xae100101,0xfb840108,0xae110101,
	0xfb840108,0xae120101,0xfb840108,0xae130101,0xfb840108,0xae140101,0xfb840108,0xae150101,
	0xfb840108,0xae160101,0xfb840108,0xae170101,0xfb840108,0xae180101,0xfb840108,0xae190101,
	0xfb840108,0xae1a0101,0xfb840108,0xae1b0101,0xfb840108,0xae1c0101,0xfb840108,0xae1d0101,
	0xfb840108,0xae1e0101,0xfb840108,0xae1f0101,0xfb840108,0xae200101,0xfb840108,0xae210101,
	0xfb840108,0xae220101,0xfb840108,0xae230101,0xfb840108,0xae240101,0xfb840108,0xae250101,
	0xfb840108,0xae260101,0xfb840108,0xae270101,0xfb840108,0xae280101,0xfb840108,0xae290101,
	0xfb840108,0xae2a0101,0xfb840108,0xae2b0101,0xfb840108,0xae2c0101,0xfb840108,0xae2d0101,
	0xfb840108,0xae2e0101,0xfb840108,0xae2f0101,0xfb840108,0xae300101,0xfb840108,0xae310101,
	0xfb840108,0xae320101,0xfb840108,0xae330101,0xfb840108,0xae340101,0xfb840108,0xae350101,
	0xfb840108,0xae360101,0xfb840108,0xae370101,0xfb840108,0xae380101,0xfb840108,0xae390101,
	0xfb840108,0xae3a0101,0xfb840108,0xae3b0101,0xfb840108,0xae3c0101,0xfb840108,0xae3d0101,
	0xfb840108,0xae3e0101,0xfb840108,0xae3f0101,0xfb840108,0xae400101,0xfb840108,0xae410101,
	0xfb840108,0xae420101,0xfb840108,0xae430101,0xfb840108,0xae440101,0xfb840108,0xae450101,
	0xfb840108,0xae460101,0xfb840108,0xae470101,0xfb840108,0xae480101,0xfb840108,0xae490101,
	0xfb840108,0xae4a0101,0xfb840108,0xae4b0101,0xfb840108,0xae4c0101,0xfb840108,0xae4d0101,
	0xfb840108,0xae4e0101,0xfb840108,0xae4f0101,0xfb840108,0xae500101,0xfb840108,0xae510101,
	0xfb840108,0xae520101,0xfb840108,0xae530101,0xfb840108,0xae540101,0xfb840108,0xae550101,
	0xfb840108,0xae560101,0xfb840108,0xae570101,0xfb840108,0xae580101,0xfb840108,0xae590101,
	0xfb840108,0xae5a0101,0xfb840108,0xae5b0101,0xfb840108,0xae5c0101,0xfb840108,0xae5d0101,
	0xfb840108,0xae5e0101,0xfb840108,0xae5f0101,0xfb840108,0xae600101,0xfb840108,0xae610101,
	0xfb840108,0xae620101,0xfb840108,0xae630101,0xfb840108,0xae640101,0xfb840108,0xae650101,
	0xfb840108,0xae660101,0xfb840108,0xae670101,0xfb840108,0xae680101,0xfb840108,0xae690101,
	0xfb840108,0xae6a0101,0xfb840108,0xae6b0101,0xfb840108,0xae6c0101,0xfb840108,0xae6d0101,
	0xfb840108,0xae6e0101,0xfb840108,0xae6f0101,0xfb840108,0xae700101,0xfb840108,0xae710101,
	0xfb840108,0xae720101,0xfb840108,0xae730101,0xfb840108,0xae740101,0xfb840108,0xae750101,
	0xfb840108,0xae760101,0xfb840108,0xae770101,0xfb840108,0xae780101,0xfb840108,0xae790101,
	0xfb840108,0xae7a0101,0xfb840108,0xae7b0101,0xfb840108,0xae7c0101,0xfb840108,0xae7d0101,
	0xfb840108,0xae7e0101,0xfb840108,0xae7f0101,0xfb840108,0xae800101,0xfb840108,0xae810101,
	0xfb840108,0xae820101,0xfb840108,0xae830101,0xfb840108,0xae840101,0xfb840108,0xae850101,
	0xfb840108,0xae860101,0xfb840108,0xae870101,0xfb840108,0xae880101,0xfb840108,0xae890101,
	0xfb840108,0xae8a0101,0xfb840108,0xae8b0101,0xfb840108,0xae8c0101,0xfb840108,0xae8d0101,
	0xfb840108,0xae8e0101,0xfb840108,0xae8f0101,0xfb840108,0xae900101,0xfb840108,0xae910101,
	0xfb840108,0xae920101,0xfb840108,0xae930101,0xfb840108,0xae940101,0xfb840108,0xae950101,
	0xfb840108,0xae960101,0xfb840108,0xae970101,0xfb840108,0xae980101,0xfb840108,0xae990101,
	0xfb840108,0xae9a0101,0xfb840108,0xae9b0101,0xfb840108,0xae9c0101,0xfb840108,0xae9d0101,
	0xfb840108,0xae9e0101,0xfb840108,0xae9f0101,0xfb840108,0xaea00101,0xfb840108,0xaea10101,
	0xfb840108,0xaea20101,0xfb840108,0xaea30101,0xfb840108,0xaea40101,0xfb840108,0xaea50101,
	0xfb840108,0xaea60101,0xfb840108,0xaea70101,0xfb840108,0xaea80101,0xfb840108,0xaea90101,
	0xfb840108,0xaeaa0101,0xfb840108,0xaeab0101,0xfb840108,0xaeac0101,0xfb840108,0xaead0101,
	0xfb840108,0xaeae0101,0xfb840108,0xaeaf0101,0xfb840108,0xaeb00101,0xfb840108,0xaeb10101,
	0xfb840108,0xaeb20101,0xfb840108,0xaeb30101,0xfb840108,0xaeb40101,0xfb840108,0xaeb50101,
	0xfb840108,0xaeb60101,0xfb840108,0xaeb70101,0xfb840108,0xaeb80101,0xfb840108,0xaeb90101,
	0xfb840108,0xaeba0101,0xfb840108,0xaebb0101,0xfb840108,0xaebc0101,0xfb840108,0xaebd0101,
	0xfb840108,0xaebe0101,0xfb840108,0xaebf0101,0xfb840108,0xaec00101,0xfb840108,0xaec10101,
	0xfb840108,0xaec20101,0xfb840108,0xaec30101,0xfb840108,0xaec40101,0xfb840108,0xaec50101,
	0xfb840108,0xaec60101,0xfb840108,0xaec70101,0xfb840108,0xaec80101,0xfb840108,0xaec90101,
	0xfb840108,0xaeca0101,0xfb840108,0xaecb0101,0xfb840108,0xaecc0101,0xfb840108,0xaecd0101,
	0xfb840108,0xaece0101,0xfb840108,0xaecf0101,0xfb840108,0xaed00101,0xfb840108,0xaed10101,
	0xfb840108,0xaed20101,0xfb840108,0xaed30101,0xfb840108,0xaed40101,0xfb840108,0xaed50101,
	0xfb840108,0xaed60101,0xfb840108,0xaed70101,0xfb840108,0xaed80101,0xfb840108,0xaed90101,
	0xfb840108,0xaeda0101,0xfb840108,0xaedb0101,0xfb840108,0xaedc0101,0xfb840108,0xaedd0101,
	0xfb840108,0xaede0101,0xfb840108,0xaedf0101,0xfb840108,0xaee00101,0xfb840108,0xaee10101,
	0xfb840108,0xaee20101,0xfb840108,0xaee30101,0xfb840108,0xaee40101,0xfb840108,0xaee50101,
	0xfb840108,0xaee60101,0xfb840108,0xaee70101,0xfb840108,0xaee80101,0xfb840108,0xaee90101,
	0xfb840108,0xaeea0101,0xfb840108,0xaeeb0101,0xfb840108,0xaeec0101,0xfb840108,0xaeed0101,
	0xfb840108,0xaeee0101,0xfb840108,0xaeef0101,0xfb840108,0xaef00101,0xfb840108,0xaef10101,
	0xfb840108,0xaef20101,0xfb840108,0xaef30101,0xfb840108,0xaef40101,0xfb840108,0xaef50101,
	0xfb840108,0xaef60101,0xfb840108,0xaef70101,0xfb840108,0xaef80101,0xfb840108,0xaef90101,
	0xfb840108,0xaefa0101,0xfb840108,0xaefb0101,0xfb840108,0xaefc0101,0xfb840108,0xaefd0101,
	0xfb840108,0xaefe0101,0xfb840108,0xaeff0101,0xfb840108,0xaf000101,0xfb840108,0xaf010101,
	0xfb840108,0xaf020101,0xfb840108,0xaf030101,0xfb840108,0xaf040101,0xfb840108,0xaf050101,
	0xfb840108,0xaf060101,0xfb840108,0xaf070101,0xfb840108,0xaf080101,0xfb840108,0xaf090101,
	0xfb840108,0xaf0a0101,0xfb840108,0xaf0b0101,0xfb840108,0xaf0c0101,0xfb840108,0xaf0d0101,
	0xfb840108,0xaf0e0101,0xfb840108,0xaf0f0101,0xfb840108,0xaf100101,0xfb840108,0xaf110101,
	0xfb840108,0xaf120101,0xfb840108,0xaf130101,0xfb840108,0xaf140101,0xfb840108,0xaf150101,
	0xfb840108,0xaf160101,0xfb840108,0xaf170101,0xfb840108,0xaf180101,0xfb840108,0xaf190101,
	0xfb840108,0xaf1a0101,0xfb840108,0xaf1b0101,0xfb840108,0xaf1c0101,0xfb840108,0xaf1d0101,
	0xfb840108,0xaf1e0101,0xfb840108,0xaf1f0101,0xfb840108,0xaf200101,0xfb840108,0xaf210101,
	0xfb840108,0xaf220101,0xfb840108,0xaf230101,0xfb840108,0xaf240101,0xfb840108,0xaf250101,
	0xfb840108,0xaf260101,0xfb840108,0xaf270101,0xfb840108,0xaf280101,0xfb840108,0xaf290101,
	0xfb840108,0xaf2a0101,0xfb840108,0xaf2b0101,0xfb840108,0xaf2c0101,0xfb840108,0xaf2d0101,
	0xfb840108,0xaf2e0101,0xfb840108,0xaf2f0101,0xfb840108,0xaf300101,0xfb840108,0xaf310101,
	0xfb840108,0xaf320101,0xfb840108,0xaf330101,0xfb840108,0xaf340101,0xfb840108,0xaf350101,
	0xfb840108,0xaf360101,0xfb840108,0xaf370101,0xfb840108,0xaf380101,0xfb840108,0xaf390101,
	0xfb840108,0xaf3a0101,0xfb840