// Copyright (c) 2001-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of the License "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
//
// Fold and decomposition tables.
//
// These tables are linked in the following way:
// KUnicodeToIndexHash is a hash table using double hashing for
// conflict resolution. The functions DecompositionHashStart and
// DecompositionHashStep give the start and step values for accessing
// the table. The first probe is at DecompositionHashStart and each
// subsequent probe is offset by DecompositionHashStep. Probes
// continue until either 0 is found (indicating that the Unicode value
// sought has no decompostion (i.e. decomposes to itself)) or a value
// is found that has the sought Unicode value in its lower 20 bits.
//
// In this latter case, the upper 12 bits contain an index into
// one of the following tables, according to the following rules:
//
// In the case of folding:
// If the Index is less than the length of KNonSingletonFolds / 2,
// it is an index into KNonSingletonFolds. If the Index is
// greater than the length of KNonSingletonFolds / 2, then it is an
// index into KSingletonFolds.
//
// In the case of decomposition:
// If the Index is less than the length of KNonSingletonDecompositions / 2,
// it is an index into KNonSingletonDecompositions. If the Index is
// greater than the length of KNonSingletonDecompositions / 2, then it is an
// index into KSingletonDecompositions.
//
// In summary:
// Let Knsf be the length of KNonSingletonFolds / 2,
// let Knsd be the length of KNonSingletonDecompositions / 2,
// let Ksd be the length of KSingletonDecompositions and
// let Ksf be the length of KSingletonFolds.
// Now if you want to fold a character and you have found
// its index 'i' from the KUnicodeToIndexHash, then;
// if (i < Knsf) then look up
//		KNonSingletonFolds[i * 2] and KNonSingletonFolds[i * 2 + 1]
// else if (Knsf <= i < Knsf + Ksf) look up KSingletonFolds[i - Knsf]
// else there is no fold for this character.
//
// Or if you want to decompose the same character, then;
// if (i < Knsd) then look up KNonSingletonDecompositions[i * 2]
//		and KNonSingletonDecompositions[i * 2 + 1]
// else if (Knsd <= i < Knsd + Ksd) look up KSingletonDecompositions[i - Knsd]
// else there is no decomposition for this character.
//
// Your index into KSingletonDecompositions or KSingletonFolds
// yields a single value which is the decomposition or fold.
//
// The KNonSingletonFolds and KNonSingletonDecomposition
// tables are made up of pairs of values. Each pair is either a pair
// of Unicode values that constitute the fold or decomposition, or
// the first value is KLongD and the second has its top 4 bits as the
// length of the decomposition (or folded decomposition) minus 3,
// and its bottom 12 bits as the index into KLongDecompositions
// of where you can find this decomposition.
//
// KLongDecompositions simply contains UTF-16 (Unicode) for
// all the decomposed and folded sequences longer than 4 bytes long.

// Hash table mapping unicode values to indices into the other tables
// in use = 2597 entries
const unsigned long KUnicodeToIndexHash[4096] =
	{
	0x4a702000, 0x4a802001, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x8212fa1b, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7eb2f9d2, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x8012f9f0, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x7e32f9c8, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x33e01026, 0x7bd2f99a,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x79f2f96f, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x82300041, 0x82400042, 0x82500043, 0x82600044, 0x82700045, 0x82800046, 0x82900047,
	0x7632f919, 0x82a00048, 0x82b00049, 0x82c0004a, 0x37a0304c, 0x82d0004b, 0x37b0304e, 0x82e0004c,
	0x37c03050, 0x82f0004d, 0x37d03052, 0x8300004e, 0x37e03054, 0x8310004f, 0x37f03056, 0x83200050,
	0x38003058, 0x83300051, 0x3810305a, 0x83400052, 0x3820305c, 0x83500053, 0x3830305e, 0x83600054,
	0x38403060, 0x83700055, 0x38503062, 0x83800056, 0x83900057, 0x38603065, 0x83a00058, 0x38703067,
	0x83b00059, 0x38803069, 0x83c0005a, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x38903070, 0x38a03071, 0x00000000, 0x38b03073, 0x38c03074, 0x00000000, 0x38d03076, 0x38e03077,
	0x00000000, 0x38f03079, 0x3900307a, 0x7e12f9c6, 0x3910307c, 0x3920307d, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7be2f99b, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x5800f9d3, 0x00000000,
	0x00000000, 0x7a02f970, 0x7c02f99d, 0x00000000, 0x37903094, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x77f2f945, 0x00000000, 0x3930309e, 0x00000000,
	0x950010a0, 0x951010a1, 0x952010a2, 0x4031d160, 0x954010a4, 0x955010a5, 0x956010a6, 0x7642f91a,
	0x958010a8, 0x959010a9, 0x95a010aa, 0x4011d15e, 0x395030ac, 0x95d010ad, 0x396030ae, 0x95f010af,
	0x397030b0, 0x961010b1, 0x398030b2, 0x963010b3, 0x399030b4, 0x83d000b5, 0x39a030b6, 0x967010b7,
	0x39b030b8, 0x969010b9, 0x39c030ba, 0x96b010bb, 0x39d030bc, 0x96d010bd, 0x39e030be, 0x96f010bf,
	0x003000c0, 0x004000c1, 0x005000c2, 0x00a000c3, 0x013000c4, 0x016000c5, 0x83e000c6, 0x028000c7,
	0x030000c8, 0x031000c9, 0x032000ca, 0x03d000cb, 0x058000cc, 0x059000cd, 0x05a000ce, 0x05f000cf,
	0x3a4030d0, 0x07a000d1, 0x081000d2, 0x082000d3, 0x083000d4, 0x088000d5, 0x092000d6, 0x3a9030d7,
	0x83f000d0, 0x0bf000d9, 0x0c0000da, 0x0c1000db, 0x0c7000dc, 0x0e8000dd, 0x840000d8, 0x5010f954,
	0x0f6000e0, 0x0f7000e1, 0x0f8000e2, 0x0fd000e3, 0x106000e4, 0x109000e5, 0x4021d15f, 0x11a000e7,
	0x122000e8, 0x123000e9, 0x124000ea, 0x12f000eb, 0x14b000ec, 0x14c000ed, 0x14d000ee, 0x151000ef,
	0x7a12f971, 0x16d000f1, 0x174000f2, 0x175000f3, 0x176000f4, 0x17b000f5, 0x185000f6, 0x3ae030f7,
	0x3af030f8, 0x1b3000f9, 0x1b4000fa, 0x1b5000fb, 0x1bb000fc, 0x1dd000fd, 0x3b2030fe, 0x1e2000ff,
	0x00b00100, 0x0fe00101, 0x00c00102, 0x0ff00103, 0x02000104, 0x11200105, 0x02400106, 0x11600107,
	0x02500108, 0x11700109, 0x0260010a, 0x1180010b, 0x0270010c, 0x1190010d, 0x02b0010e, 0x11d0010f,
	0x841000de, 0x84200110, 0x03800112, 0x12a00113, 0x03b00114, 0x12d00115, 0x03c00116, 0x12e00117,
	0x04600118, 0x13800119, 0x03f0011a, 0x1310011b, 0x04b0011c, 0x13d0011d, 0x04d0011e, 0x13f0011f,
	0x04e00120, 0x14000121, 0x05000122, 0x14200123, 0x05100124, 0x14300125, 0x4a902126, 0x84300126,
	0x05b00128, 0x14e00129, 0x05c0012a, 0x0170212b, 0x05d0012c, 0x14f0012b, 0x0660012e, 0x1500012d,
	0x05e00130, 0x1580012f, 0x84400131, 0x84500132, 0x06800134, 0x15a00135, 0x06c00136, 0x15f00137,
	0x00000000, 0x06e00139, 0x1610013a, 0x0720013b, 0x1650013c, 0x06f0013d, 0x1620013e, 0x8460013f,
	0x00000000, 0x84700141, 0x00000000, 0x07900143, 0x16c00144, 0x07e00145, 0x17100146, 0x07c00147,
	0x16f00148, 0x00000000, 0x8480014a, 0x7a72f97d, 0x08c0014c, 0x17f0014d, 0x08f0014e, 0x1820014f,
	0x09500150, 0x18800151, 0x4041d161, 0x84900152, 0x0a500154, 0x19800155, 0x0ac00156, 0x19f00157,
	0x0a700158, 0x19a00159, 0x0ae0015a, 0x1a10015b, 0x0b00015c, 0x1a30015d, 0x0b70015e, 0x1aa0015f,
	0x0b200160, 0x1a500161, 0x0bc00162, 0x1b000163, 0x0b900164, 0x1ad00165, 0x84a00166, 0x97e02167,
	0x0c200168, 0x1b600169, 0x0c40016a, 0x1b80016b, 0x0c60016c, 0x1ba0016d, 0x0cd0016e, 0x1c10016f,
	0x0ce00170, 0x1c200171, 0x0da00172, 0x1ce00173, 0x0e100174, 0x1d500175, 0x0e900176, 0x1de00177,
	0x0ed00178, 0x0f000179, 0x1e60017a, 0x0f20017b, 0x1e80017c, 0x0f30017d, 0x1e90017e, 0x84b0017f,
	0x7812f947, 0x84c00181, 0x84d00182, 0x98702183, 0x84e00184, 0x00000000, 0x84f00186, 0x85000187,
	0x00000000, 0x85100189, 0x8520018a, 0x8530018b, 0x00000000, 0x8042f9f4, 0x8540018e, 0x8550018f,
	0x85600190, 0x85700191, 0x00000000, 0x85800193, 0x85900194, 0x00000000, 0x85a00196, 0x4051d162,
	0x7e42f9c9, 0x85b00197, 0x3500219a, 0x3510219b, 0x85c00198, 0x85d0019c, 0x85e0019d, 0x85f0019f,
	0x099001a0, 0x18c001a1, 0x860001a2, 0x7c12f99e, 0x861001a4, 0x5890f9dc, 0x862001a6, 0x863001a7,
	0x00000000, 0x5050f958, 0x864001a9, 0x00000000, 0x865001ac, 0x00000000, 0x352021ae, 0x0d2001af,
	0x1c6001b0, 0x866001ae, 0x867001b1, 0x868001b2, 0x869001b3, 0x86a001b5, 0x8052f9f5, 0x86b001b7,
	0x86c001b8, 0x7822f948, 0x00000000, 0x4081d1bb, 0x4091d1bc, 0x40a1d1bd, 0x40b1d1be, 0x40c1d1bf,
	0x40d1d1c0, 0x86d001bc, 0x00000000, 0x9a1024cf, 0x58a0f9dd, 0x86e001c4, 0x86f001c5, 0x7c92f9a9,
	0x5060f959, 0x870001c7, 0x871001c8, 0x872001ca, 0x873001cb, 0x019001cd, 0x10b001ce, 0x062001cf,
	0x154001d0, 0x096001d1, 0x189001d2, 0x0cf001d3, 0x1c3001d4, 0x0ca001d5, 0x1be001d6, 0x0c9001d7,
	0x1bd001d8, 0x0cb001d9, 0x1bf001da, 0x0c8001db, 0x1bc001dc, 0x00000000, 0x014001de, 0x107001df,
	0x012001e0, 0x105001e1, 0x1f2001e2, 0x1f5001e3, 0x874001e4, 0x00000000, 0x04f001e6, 0x141001e7,
	0x06a001e8, 0x15d001e9, 0x0a1001ea, 0x194001eb, 0x0a2001ec, 0x195001ed, 0x1f8001ee, 0x1f9001ef,
	0x15b001f0, 0x875001f1, 0x876001f2, 0x00000000, 0x04a001f4, 0x13c001f5, 0x877001f6, 0x7652f91c,
	0x078001f8, 0x16b001f9, 0x018001fa, 0x10a001fb, 0x1f1001fc, 0x1f4001fd, 0x1f3001fe, 0x1f6001ff,
	0x01a00200, 0x10c00201, 0x01b00202, 0x10d00203, 0x04000204, 0x13200205, 0x04100206, 0x13300207,
	0x06300208, 0x15500209, 0x0640020a, 0x1560020b, 0x0970020c, 0x18a0020d, 0x0980020e, 0x18b0020f,
	0x0a800210, 0x19b00211, 0x0a900212, 0x19c00213, 0x0d000214, 0x1c400215, 0x0d100216, 0x1c500217,
	0x0b600218, 0x1a900219, 0x0bb0021a, 0x1af0021b, 0x878001f7, 0x8790021c, 0x0540021e, 0x1460021f,
	0x87a00220, 0x58d0f9e0, 0x87b00222, 0x7662f91e, 0x35902224, 0x5090f95c, 0x01100226, 0x10400227,
	0x04400228, 0x13600229, 0x0930022a, 0x1860022b, 0x08a0022c, 0x17d0022d, 0x0900022e, 0x1830022f,
	0x09100230, 0x18400231, 0x0eb00232, 0x1e000233, 0x87c00224, 0x96c010bc, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x87d0023a, 0x87e0023b, 0x00000000, 0x87f0023d, 0x8800023e, 0x00000000,
	0x58e0f9e1, 0x35b02241, 0x88100241, 0x7ed2f9d5, 0x35c02244, 0x88200243, 0x88300244, 0x35d02247,
	0x88400245, 0x35e02249, 0x88500246, 0x88600248, 0x8870024a, 0x8880024c, 0x8890024e, 0x00000000,
	0x975010c5, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7bc2f999, 0x4612f9cb, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00102260, 0x7c32f9a0, 0x36002262, 0x50b0f95e, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x35f0226d, 0x0000226e, 0x0020226f,
	0x36102270, 0x36202271, 0x9820216b, 0x00000000, 0x36302274, 0x36402275, 0x00000000, 0x7842f94a,
	0x36502278, 0x36602279, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x5900f9e3, 0x00000000,
	0x36702280, 0x36802281, 0x50c0f95f, 0x00000000, 0x36b02284, 0x36c02285, 0x00000000, 0x00000000,
	0x36d02288, 0x36e02289, 0x00000000, 0x00000000, 0x9f602cba, 0x00000000, 0x5290f97c, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xa220ff38, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x371022ac, 0x372022ad, 0x373022ae, 0x374022af,
	0x4061d163, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x4622f9cc, 0x4071d164, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x5920f9e5, 0x00000000, 0x00000000, 0x00000000,
	0x50e0f961, 0x00000000, 0x7dc2f9c0, 0x00000000, 0x00000000, 0x7a82f97f, 0x00000000, 0x00000000,
	0x77c2f93e, 0x00000000, 0x00000000, 0x7a32f976, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7852f94b, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x369022e0, 0x36a022e1, 0x36f022e2, 0x370022e3, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x375022ea, 0x376022eb, 0x377022ec, 0x378022ed, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x81d2fa17, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x7b92f995, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x75d2f913, 0x9fd02cc8, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x4e80f93b, 0x00000000, 0x00000000, 0x7e62f9cd, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x9fe02cca, 0x00000000, 0x00000000, 0x7c52f9a2,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x4ab02329, 0x4ac0232a, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x953010a3, 0x00000000, 0x00000000, 0x00000000, 0x7862f94c, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0xa120ff28, 0x5120f965, 0x00000000, 0x7fe2f9ec, 0x7a22f974,
	0x49f00340, 0x4a000341, 0x00000000, 0x4a100343, 0x3b300344, 0x88a00345, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x4e90f93c, 0x00000000,
	0x00000000, 0x00000000, 0xa0102cd0, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x95b010ab, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x95c010ac, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x8072f9f9, 0x00000000, 0x00000000, 0x00000000, 0x6d72f875, 0x9e202c92, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x960010b0, 0x7e72f9ce, 0x4a200374, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x962010b2, 0x00000000, 0x4a30037e, 0x7dd2f9c1,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x1ed00385, 0x1fb00386, 0x4a400387,
	0x21100388, 0x21a00389, 0x22e0038a, 0x4ea0f93d, 0x23c0038c, 0x00000000, 0x2460038e, 0x2500038f,
	0x2a500390, 0x81f2fa19, 0x88b00391, 0x88c00392, 0x88d00393, 0x88e00394, 0x88f00395, 0x89000396,
	0x89100397, 0x89200398, 0x89300399, 0x8940039a, 0x8950039b, 0x8960039c, 0x8970039d, 0x7692f922,
	0x8980039e, 0x8990039f, 0x89a003a0, 0x89b003a1, 0x89c003a3, 0x89d003a4, 0x89e003a5, 0x89f003a6,
	0x8a0003a7, 0x8a1003a8, 0x232003aa, 0x24a003ab, 0x265003ac, 0x27e003ad, 0x288003ae, 0x29f003af,
	0x2c3003b0, 0x8a2003a9, 0x00000000, 0x00000000, 0x59a0f9ed, 0x00000000, 0x00000000, 0x00000000,
	0x96e010be, 0x00000000, 0x81e2fa18, 0x00000000, 0x00000000, 0x9eb02ca4, 0x00000000, 0x00000000,
	0x7ba2f996, 0x00000000, 0x8a3003c2, 0x00000000, 0x00000000, 0x00000000, 0x75e2f914, 0x8082f9fa,
	0x00000000, 0x00000000, 0x2a3003ca, 0x2c1003cb, 0x2b2003cc, 0x2bd003cd, 0x2d1003ce, 0x00000000,
	0x8a4003d0, 0x8a5003d1, 0x7e82f9cf, 0x2e7003d3, 0x2e8003d4, 0x8a6003d5, 0x8a7003d6, 0x00000000,
	0x8a8003d8, 0x00000000, 0x8a9003da, 0x00000000, 0x8aa003dc, 0x00000000, 0x8ab003de, 0x00000000,
	0x8ac003e0, 0x00000000, 0x7872f94e, 0x8ad003e2, 0x8ae003e4, 0x00000000, 0x8af003e6, 0x00000000,
	0x7a52f979, 0x8b0003e8, 0x8b1003ea, 0x00000000, 0x8b2003ec, 0x00000000, 0x8b3003ee, 0x00000000,
	0x8b4003f0, 0x8b5003f1, 0x00000000, 0x5720f9c5, 0x8b6003f4, 0x8b7003f5, 0x5180f96b, 0x8b8003f7,
	0x00000000, 0x8b9003f9, 0x8ba003fa, 0x00000000, 0x00000000, 0x8bb003fd, 0x8bc003fe, 0x8bd003ff,
	0x2ed00400, 0x2ef00401, 0x47710400, 0x2ec00403, 0x47b10404, 0x47c10405, 0x47d10406, 0x2e900407,
	0x47f10408, 0x48010409, 0x4811040a, 0x4821040b, 0x2f70040c, 0x2f30040d, 0x2fa0040e, 0x4861040f,
	0x48710410, 0x48810411, 0x48910412, 0x48a10413, 0x48b10414, 0x48c10415, 0x48d10416, 0x48e10417,
	0x48f10418, 0x2f500419, 0x4911041a, 0x4921041b, 0x4931041c, 0x4941041d, 0x4951041e, 0x4961041f,
	0x49710420, 0x49810421, 0x47810401, 0x49a10423, 0x49b10424, 0x49c10425, 0x49d10426, 0x49e10427,
	0x8be00402, 0x8bf00404, 0x8c000405, 0x8c100406, 0x8c200408, 0x8c300409, 0x8c40040a, 0x8c50040b,
	0x59e0f9f1, 0x7e92f9d0, 0x8c60040f, 0x8c700410, 0x8c800411, 0x8c900412, 0x8ca00413, 0x8cb00414,
	0x8cc00415, 0x30b00439, 0x8cd00416, 0x8ce00417, 0x7de2f9c2, 0x8cf00418, 0x8d00041a, 0x8d10041b,
	0x8d20041c, 0x8d30041d, 0x77e2f940, 0x47910402, 0x8d40041e, 0x8d50041f, 0x8d600420, 0x7a62f97a,
	0x8d700421, 0x8d800422, 0x8d900423, 0x8da00424, 0x8db00425, 0x8dc00426, 0x8dd00427, 0x8de00428,
	0x30300450, 0x30500451, 0x7882f94f, 0x30200453, 0x8df00429, 0x8e00042a, 0x8e10042b, 0x31600457,
	0x8e20042c, 0x8e30042d, 0x8e40042e, 0x8e50042f, 0x30d0045c, 0x3090045d, 0x3100045e, 0x00000000,
	0x8e600460, 0x00000000, 0x8e700462, 0x00000000, 0x47a10403, 0x8e800464, 0x8e900466, 0x00000000,
	0x8ea00468, 0x00000000, 0x8eb0046a, 0x00000000, 0x8ec0046c, 0x00000000, 0x8ed0046e, 0x00000000,
	0x8ee00470, 0x00000000, 0x8ef00472, 0x00000000, 0x8f000474, 0x00000000, 0x31700476, 0x31800477,
	0x8f100478, 0x00000000, 0x8f20047a, 0x00000000, 0x8f30047c, 0x00000000, 0x8f40047e, 0x4ee0f941,
	0x8f500480, 0x00000000, 0x00000000, 0x75f2f915, 0x00000000, 0x8092f9fc, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x8f60048a, 0x00000000, 0x8f70048c, 0x00000000, 0x8f80048e, 0x00000000,
	0x7ea2f9d1, 0x8f900490, 0x8fa00492, 0x00000000, 0x8fb00494, 0x00000000, 0x8fc00496, 0x00000000,
	0x8fd00498, 0x00000000, 0x8fe0049a, 0x00000000, 0x8ff0049c, 0x00000000, 0x9000049e, 0x00000000,
	0x901004a0, 0x00000000, 0x902004a2, 0x00000000, 0x903004a4, 0x00000000, 0x904004a6, 0x00000000,
	0x905004a8, 0x00000000, 0x906004aa, 0x00000000, 0x907004ac, 0x00000000, 0x908004ae, 0x00000000,
	0x909004b0, 0x7892f950, 0x90a004b2, 0x00000000, 0x73d2f8e9, 0x90b004b4, 0x90c004b6, 0x989024b7,
	0x7ff2f9ee, 0x90d004b8, 0x90e004ba, 0x98d024bb, 0x76b2f925, 0x90f004bc, 0x79c2f96c, 0x910004be,
	0x911004c0, 0x2f0004c1, 0x306004c2, 0x912004c3, 0x996024c4, 0x913004c5, 0x998024c6, 0x914004c7,
	0x99a024c8, 0x915004c9, 0x6cc2f868, 0x5a30f9f6, 0x916004cb, 0x917004cd, 0x9a0024ce, 0x51f0f972,
	0x2ea004d0, 0x300004d1, 0x2eb004d2, 0x301004d3, 0x918004d4, 0x00000000, 0x2ee004d6, 0x304004d7,
	0x919004d8, 0x00000000, 0x319004da, 0x31a004db, 0x2f1004dc, 0x307004dd, 0x2f2004de, 0x308004df,
	0x91a004e0, 0x00000000, 0x2f4004e2, 0x30a004e3, 0x2f6004e4, 0x30c004e5, 0x2f8004e6, 0x30e004e7,
	0x47e10407, 0x91b004e8, 0x31b004ea, 0x31c004eb, 0x2ff004ec, 0x315004ed, 0x2f9004ee, 0x30f004ef,
	0x2fb004f0, 0x311004f1, 0x2fc004f2, 0x312004f3, 0x2fd004f4, 0x313004f5, 0x80f2fa04, 0x91c004f6,
	0x2fe004f8, 0x314004f9, 0x7c72f9a7, 0x91d004fa, 0x7aa2f982, 0x91e004fc, 0x91f004fe, 0x00000000,
	0x92000500, 0x00000000, 0x92100502, 0x00000000, 0x92200504, 0x6160fa7c, 0x92300506, 0x00000000,
	0x92400508, 0x00000000, 0x9250050a, 0x968010b8, 0x9260050c, 0x00000000, 0x9270050e, 0x00000000,
	0x78a2f951, 0x92800510, 0x92900512, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x96a010ba, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xa0c0ff22,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x5220f975, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x92a00531, 0x92b00532, 0x92c00533, 0x8202fa1a, 0x92d00534, 0x4f10f944, 0x92e00535,
	0x92f00536, 0x93000537, 0x7bb2f998, 0x93100538, 0x93200539, 0x9330053a, 0x9340053b, 0x9350053c,
	0x7602f916, 0x39f030c0, 0x9360053d, 0x80a2f9fe, 0x9370053e, 0x9380053f, 0x6f02f894, 0x6dc2f87a,
	0x93900540, 0x93a00541, 0x93b00542, 0x93c00543, 0x93d00544, 0x93e00545, 0x93f00546, 0x3a0030c2,
	0x94000547, 0x94100548, 0x7bf2f99c, 0x94200549, 0x9430054a, 0x9440054b, 0x9450054c, 0x9460054d,
	0x9470054e, 0x7c82f9a8, 0x9480054f, 0x94900550, 0x94a00551, 0x94b00552, 0x94c00553, 0x94d00554,
	0x94e00555, 0x5780f9cb, 0x94f00556, 0x00000000, 0x3a1030c5, 0x00000000, 0x5a80f9fb, 0x00000000,
	0x00000000, 0x00000000, 0x5240f977, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x3a2030c7, 0x00000000, 0x00000000, 0x8002f9ef, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x7742f930, 0x79d2f96d, 0xa0f0ff25, 0x00000000, 0x00000000, 0x00000000,
	0x3a3030c9, 0x73f2f8eb, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x4831040c, 0x7902f95a, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0xa100ff26, 0x00000000, 0x00000000, 0x5790f9cc, 0x00000000,
	0x00000000, 0x00000000, 0x80b2f9ff, 0x00000000, 0x5aa0f9fd, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x9f802cbe, 0x00000000, 0x00000000, 0x00000000, 0x7ec2f9d4, 0x4841040d, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7e02f9c4, 0x00000000,
	0x3a5030d1, 0x00000000, 0xa110ff27, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x5a40f9f7, 0x00000000, 0x00000000, 0x3a6030d3, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x3a7030d4, 0x78b2f953, 0x4851040e,
	0x00000000, 0x00000000, 0x00000000, 0x4aa0212a, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x76c2f928, 0x00000000, 0x3a8030d6, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x957010a7, 0x74b2f8fd, 0x00000000, 0x5280f97b, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x3aa030d9, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x3ab030da,
	0xa130ff29, 0x00000000, 0x00000000, 0x97602132, 0x00000000, 0x61e0fa84, 0x00000000, 0x00000000,
	0x00000000, 0x80c2fa00, 0x00000000, 0x00000000, 0x00000000, 0x3ac030dc, 0x00000000, 0x00000000,
	0x76a2f924, 0x00000000, 0x00000000, 0x00000000, 0x3ad030dd, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7ca2f9aa,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x5ae0fa01, 0x00000000, 0x31d00622, 0x31e00623, 0x32000624, 0x31f00625, 0x32100626, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xa150ff2b, 0x00000000,
	0x76d2f929, 0x00000000, 0x00000000, 0x00000000, 0x7612f917, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x52b0f97e, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x81c2fa16, 0x00000000, 0x00000000,
	0x964010b4, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x965010b5, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x966010b6, 0x7ee2f9d6, 0x00000000, 0x00000000, 0xa1b0ff31, 0x00000000,
	0x8112fa06, 0x00000000, 0x00000000, 0x00000000, 0xa170ff2d, 0x00000000, 0x00000000, 0x3fd0fb4b,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x52d0f980, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0xa180ff2e, 0x00000000, 0x00000000, 0x00000000, 0x76e2f92a,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xa0b0ff21, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x74d2f8ff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x394030f4, 0x8222fa1c, 0x00000000,
	0x00000000, 0x7f12f9db, 0xa190ff2f, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x78f2f959,
	0x00000000, 0x00000000, 0x7622f918, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x80d2fa02,
	0x324006c0, 0x00000000, 0x322006c2, 0x00000000, 0x970010c0, 0x00000000, 0x9ea02ca2, 0x00000000,
	0x00000000, 0x00000000, 0x7ef2f9d7, 0x00000000, 0x7fd2f9eb, 0x971010c1, 0x00000000, 0x00000000,
	0x3b0030f9, 0xa1a0ff30, 0x00000000, 0x323006d3, 0x00000000, 0x7cb2f9ac, 0x972010c2, 0x3b1030fa,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x8032f9f3,
	0x7a92f981, 0x97702160, 0x00000000, 0x00000000, 0x00000000, 0x5e50fa46, 0x97802161, 0x00000000,
	0x974010c4, 0x00000000, 0x00000000, 0x78c2f956, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x3fe0fb4c, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x97b02164, 0x76f2f92b, 0x00000000,
	0x7722f92e, 0x00000000, 0x97c02165, 0x00000000, 0x00000000, 0x5310f984, 0x00000000, 0x97d02166,
	0x00000000, 0x74e2f900, 0x00000000, 0x5bd0fa15, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x97f02168, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x98002169, 0xa1c0ff32,
	0x00000000, 0x00000000, 0x00000000, 0x9810216a, 0x00000000, 0xa0602cda, 0x00000000, 0x00000000,
	0x5b60fa09, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x9830216c, 0x80e2fa03, 0x00000000,
	0x00000000, 0x00000000, 0x9840216d, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x9850216e,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x9860216f, 0x8122fa07, 0xa1d0ff33, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x7ad2f985, 0x7c62f9a3, 0x00000000, 0x00000000, 0x5f80fa59,
	0x00000000, 0x7512f903, 0x49010419, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x4af0f902,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x78d2f957, 0x00000000, 0x00000000, 0xa1e0ff34, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7702f92c, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x5340f987, 0x4fa0f94d, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x74f2f901, 0x00000000, 0x00000000, 0xa0902ce0, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xa1f0ff35, 0x00000000, 0x7f22f9dc, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x5f90fa5a, 0x00000000, 0x00000000, 0x7f32f9de,
	0x00000000, 0x5350f988, 0x00000000, 0x00000000, 0x00000000, 0x4b10f904, 0x00000000, 0x95e010ae,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7802f946, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0xa200ff36, 0x00000000, 0x00000000, 0x00000000, 0x988024b6,
	0x00000000, 0x00000000, 0x00000000, 0x7cc2f9ae, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x5360f989, 0x98a024b8, 0x00000000, 0x00000000, 0x7912f95b, 0x00000000, 0x7ab2f983, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x98c024ba, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x78e2f958, 0xa210ff37, 0x00000000, 0x8022f9f2, 0x98e024bc, 0x00000000, 0x00000000,
	0x00000000, 0x5fa0fa5b, 0x98f024bd, 0x00000000, 0x7712f92d, 0x7732f92f, 0x00000000, 0x5370f98a,
	0x00000000, 0x00000000, 0x00000000, 0x4b30f906, 0x991024bf, 0x00000000, 0x00000000, 0x7502f902,
	0x00000000, 0x992024c0, 0x00000000, 0x00000000, 0x00000000, 0x8142fa0a, 0x993024c1, 0x00000000,
	0x00000000, 0x7ce2f9b2, 0x00000000, 0x994024c2, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x7cf2f9b3, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7f42f9df, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x4b40f907, 0x00000000, 0x8102fa05, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x999024c7, 0x00000000, 0x7532f905, 0x7f02f9da,
	0xa230ff39, 0x00000000, 0x8132fa08, 0x00000000, 0x00000000, 0x7e22f9c7, 0x99b024c9, 0x00000000,
	0x7ae2f986, 0x5bb0fa10, 0x7cd2f9af, 0x99c024ca, 0x00000000, 0x00000000, 0x7522f904, 0x00000000,
	0x99d024cb, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7ac2f984, 0x00000000, 0x7752f931,
	0x66e2f800, 0x66f2f801, 0x6702f802, 0x40e2f803, 0x6712f804, 0x6722f805, 0x6732f806, 0x6742f807,
	0x6752f808, 0x6762f809, 0x6772f80a, 0x6782f80b, 0x6792f80c, 0x40f2f80d, 0x67a2f80e, 0x67b2f80f,
	0x67c2f810, 0x67d2f811, 0x4102f812, 0x67e2f813, 0x67f2f814, 0x6802f815, 0x4112f816, 0x6812f817,
	0x6822f818, 0x6832f819, 0x6842f81a, 0x6852f81b, 0x4122f81c, 0x6862f81d, 0x6872f81e, 0x6882f81f,
	0x6892f820, 0x68a2f821, 0x68b2f822, 0x68c2f823, 0x68d2f824, 0x68e2f825, 0x68f2f826, 0x6902f827,
	0x6912f828, 0x6922f829, 0x6932f82a, 0x6942f82b, 0x6952f82c, 0x6962f82d, 0x6972f82e, 0x5bc0fa12,
	0x6992f830, 0x69a2f831, 0x69b2f832, 0x69c2f833, 0x4132f834, 0x69d2f835, 0x69e2f836, 0x69f2f837,
	0x4142f838, 0x6a02f839, 0x6a12f83a, 0x6a22f83b, 0x6a32f83c, 0x6a42f83d, 0x6a52f83e, 0x6a62f83f,
	0x6a72f840, 0x6a82f841, 0x6a92f842, 0x6aa2f843, 0x6ab2f844, 0x6ac2f845, 0x6ad2f846, 0x6ae2f847,
	0x6af2f848, 0x6b02f849, 0x6b12f84a, 0x6b22f84b, 0x6b32f84c, 0x6b42f84d, 0x6b52f84e, 0x6b62f84f,
	0x6b72f850, 0x6b82f851, 0x6b92f852, 0x6ba2f853, 0x6bb2f854, 0x6bc2f855, 0x6bd2f856, 0x6be2f857,
	0x6bf2f858, 0x4152f859, 0x6c02f85a, 0x6c12f85b, 0x6c22f85c, 0x6c32f85d, 0x6c42f85e, 0x6c52f85f,
	0x4162f860, 0x4172f861, 0x6c62f862, 0x49910422, 0x6c82f864, 0x6c92f865, 0x6ca2f866, 0x6cb2f867,
	0x5fd0fa5e, 0x6cd2f869, 0x6ce2f86a, 0x6cf2f86b, 0x4182f86c, 0x6d02f86d, 0x6d12f86e, 0x6d22f86f,
	0x6d32f870, 0x4192f871, 0x6d42f872, 0x6d52f873, 0x6d62f874, 0x4b90f90c, 0x6d82f876, 0x6d92f877,
	0x6da2f878, 0x6db2f879, 0x5850f9d8, 0x41a2f87b, 0x6dd2f87c, 0x41b2f87d, 0x6de2f87e, 0x6df2f87f,
	0x6e02f880, 0x6e12f881, 0x6e22f882, 0x6e32f883, 0x6e42f884, 0x6e52f885, 0x6e62f886, 0x6e72f887,
	0x6e82f888, 0x41c2f889, 0x6e92f88a, 0x6ea2f88b, 0x4ff0f952, 0x6ec2f88d, 0x6ed2f88e, 0x41d2f88f,
	0x6ee2f890, 0x41e2f891, 0x41f2f892, 0x6ef2f893, 0x4ba0f90d, 0x6f12f895, 0x6f22f896, 0x4202f897,
	0x4212f898, 0x6f32f899, 0x6f42f89a, 0x6f52f89b, 0x6f62f89c, 0x6f72f89d, 0x6f82f89e, 0x6f92f89f,
	0x6fa2f8a0, 0x6fb2f8a1, 0x6fc2f8a2, 0x6fd2f8a3, 0x4222f8a4, 0x6fe2f8a5, 0x6ff2f8a6, 0x7002f8a7,
	0x7012f8a8, 0x7022f8a9, 0x7032f8aa, 0x7042f8ab, 0x7052f8ac, 0x7062f8ad, 0x7072f8ae, 0x7082f8af,
	0x7092f8b0, 0x70a2f8b1, 0x70b2f8b2, 0x70c2f8b3, 0x70d2f8b4, 0x70e2f8b5, 0x70f2f8b6, 0x5860f9d9,
	0x4232f8b8, 0x7112f8b9, 0x7122f8ba, 0x7132f8bb, 0x7142f8bc, 0x7152f8bd, 0x4242f8be, 0x7162f8bf,
	0x7172f8c0, 0x7182f8c1, 0x7192f8c2, 0x71a2f8c3, 0x71b2f8c4, 0x71c2f8c5, 0x71d2f8c6, 0x71e2f8c7,
	0x71f2f8c8, 0x7202f8c9, 0x4252f8ca, 0x7212f8cb, 0x7222f8cc, 0x7232f8cd, 0x7242f8ce, 0x7252f8cf,
	0x7262f8d0, 0x7272f8d1, 0x7282f8d2, 0x7292f8d3, 0x72a2f8d4, 0x72b2f8d5, 0x72c2f8d6, 0x72d2f8d7,
	0x72e2f8d8, 0x72f2f8d9, 0x7302f8da, 0x7312f8db, 0x7322f8dc, 0x4262f8dd, 0x7332f8de, 0x7342f8df,
	0x7352f8e0, 0x7362f8e1, 0x7372f8e2, 0x4272f8e3, 0x7382f8e4, 0x7392f8e5, 0x73a2f8e6, 0x73b2f8e7,
	0x73c2f8e8, 0x5020f955, 0x73e2f8ea, 0x5c50fa1d, 0x4282f8ec, 0x7402f8ed, 0x7412f8ee, 0x7422f8ef,
	0x4292f8f0, 0x4bd0f910, 0x7442f8f2, 0x7452f8f3, 0x7462f8f4, 0x7472f8f5, 0x7482f8f6, 0x42a2f8f7,
	0x42b2f8f8, 0x42c2f8f9, 0x7492f8fa, 0x42d2f8fb, 0x74a2f8fc, 0x5440f997, 0x74c2f8fe, 0x4be0f911,
	0x4ad0f900, 0x4ae0f901, 0x353021cd, 0x4b00f903, 0x4662f9dd, 0x4b20f905, 0x42e2f906, 0x355021ce,
	0x4b50f908, 0x4b60f909, 0x4b70f90a, 0x4b80f90b, 0x354021cf, 0x42f2f90d, 0x4bb0f90e, 0x4bc0f90f,
	0x4302f910, 0x4312f911, 0x4bf0f912, 0x4c00f913, 0x4c10f914, 0x4c20f915, 0x4c30f916, 0x4c40f917,
	0x4c50f918, 0x4c60f919, 0x4c70f91a, 0x4322f91b, 0x4c90f91c, 0x4332f91d, 0x4cb0f91e, 0x4342f91f,
	0x4cd0f920, 0x4ce0f921, 0x4cf0f922, 0x4352f923, 0x4d10f924, 0x4462f95c, 0x4362f926, 0x4372f927,
	0x4d50f928, 0x32500929, 0x4d70f92a, 0x4d80f92b, 0x4d90f92c, 0x4da0f92d, 0x4db0f92e, 0x4dc0f92f,
	0x4dd0f930, 0x32600931, 0x4df0f932, 0x4e00f933, 0x32700934, 0x4382f935, 0x4e30f936, 0x4392f937,
	0x4e50f938, 0x43a2f939, 0x4e70f93a, 0x43b2f93b, 0x43c2f93c, 0x43d2f93d, 0x4eb0f93e, 0x4ec0f93f,
	0x4ed0f940, 0x43e2f941, 0x43f2f942, 0x4402f943, 0x4412f944, 0x4f20f945, 0x4f30f946, 0x4f40f947,
	0x4f50f948, 0x4f60f949, 0x4f70f94a, 0x4f80f94b, 0x4f90f94c, 0x4422f94d, 0x4fb0f94e, 0x4fc0f94f,
	0x4fd0f950, 0x4fe0f951, 0x4432f952, 0x5000f953, 0x4442f954, 0x4452f955, 0x5030f956, 0x5040f957,
	0x3b400958, 0x3b500959, 0x3b60095a, 0x3b70095b, 0x3b80095c, 0x3b90095d, 0x3ba0095e, 0x3bb0095f,
	0x50d0f960, 0x4492f961, 0x50f0f962, 0x5100f963, 0x5110f964, 0x44a2f965, 0x5130f966, 0x5140f967,
	0x5150f968, 0x5160f969, 0x5170f96a, 0x44b2f96b, 0x5190f96c, 0x51a0f96d, 0x51b0f96e, 0x51c0f96f,
	0x51d0f970, 0x51e0f971, 0x44c2f972, 0x44d2f973, 0x5210f974, 0x44e2f975, 0x5230f976, 0x44f2f977,
	0x5250f978, 0x5260f979, 0x5270f97a, 0x4502f97b, 0x4512f97c, 0x52a0f97d, 0x4522f97e, 0x52c0f97f,
	0x4532f980, 0x52e0f981, 0x52f0f982, 0x5300f983, 0x4472f95d, 0x5320f985, 0x5330f986, 0x4542f987,
	0x4552f988, 0x4562f989, 0x4572f98a, 0x5380f98b, 0x5390f98c, 0x53a0f98d, 0x53b0f98e, 0x53c0f98f,
	0x53d0f990, 0x53e0f991, 0x53f0f992, 0x5400f993, 0x5410f994, 0x5420f995, 0x5430f996, 0x4582f997,
	0x5450f998, 0x5460f999, 0x5470f99a, 0x5480f99b, 0x5490f99c, 0x54a0f99d, 0x54b0f99e, 0x54c0f99f,
	0x54d0f9a0, 0x54e0f9a1, 0x54f0f9a2, 0x5500f9a3, 0x4592f9a4, 0x45a2f9a5, 0x45b2f9a6, 0x5540f9a7,
	0x5550f9a8, 0x5560f9a9, 0x5570f9aa, 0x45c2f9ab, 0x5590f9ac, 0x45d2f9ad, 0x55b0f9ae, 0x55c0f9af,
	0x45e2f9b0, 0x45f2f9b1, 0x55f0f9b2, 0x5600f9b3, 0x5610f9b4, 0x5620f9b5, 0x5630f9b6, 0x5640f9b7,
	0x5650f9b8, 0x5660f9b9, 0x5670f9ba, 0x5680f9bb, 0x5690f9bc, 0x56a0f9bd, 0x56b0f9be, 0x56c0f9bf,
	0x56d0f9c0, 0x56e0f9c1, 0x56f0f9c2, 0x5700f9c3, 0x5710f9c4, 0x4602f9c5, 0x5730f9c6, 0x5740f9c7,
	0x5750f9c8, 0x5760f9c9, 0x5770f9ca, 0x328009cb, 0x329009cc, 0x57a0f9cd, 0x57b0f9ce, 0x57c0f9cf,
	0x57d0f9d0, 0x57e0f9d1, 0x57f0f9d2, 0x4632f9d3, 0x5810f9d4, 0x5820f9d5, 0x5830f9d6, 0x5840f9d7,
	0x4642f9d8, 0x4652f9d9, 0x5870f9da, 0x5880f9db, 0x3bc009dc, 0x3bd009dd, 0x58b0f9de, 0x3be009df,
	0x4672f9e0, 0x4682f9e1, 0x58f0f9e2, 0x4482f95e, 0x5910f9e4, 0x4692f9e5, 0x5930f9e6, 0x5940f9e7,
	0x5950f9e8, 0x5960f9e9, 0x5970f9ea, 0x5980f9eb, 0x5990f9ec, 0x46a2f9ed, 0x59b0f9ee, 0x59c0f9ef,
	0x59d0f9f0, 0x46b2f9f1, 0x59f0f9f2, 0x5a00f9f3, 0x5a10f9f4, 0x5a20f9f5, 0x46c2f9f6, 0x46d2f9f7,
	0x5a50f9f8, 0x5a60f9f9, 0x5a70f9fa, 0x46e2f9fb, 0x5a90f9fc, 0x46f2f9fd, 0x5ab0f9fe, 0x5ac0f9ff,
	0x5ad0fa00, 0x4702fa01, 0x5af0fa02, 0x5b00fa03, 0x5b10fa04, 0x5b20fa05, 0x5b30fa06, 0x5b40fa07,
	0x5b50fa08, 0x4712fa09, 0x5b70fa0a, 0x5b80fa0b, 0x5b90fa0c, 0x5ba0fa0d, 0x8182fa0e, 0x8192fa0f,
	0x4722fa10, 0x81a2fa11, 0x4732fa12, 0x4742fa13, 0x4752fa14, 0x35602204, 0x5be0fa16, 0x5bf0fa17,
	0x5c00fa18, 0x5c10fa19, 0x5c20fa1a, 0x5c30fa1b, 0x5c40fa1c, 0x4762fa1d, 0x5c60fa1e, 0x00000000,
	0x5c70fa20, 0x8152fa0b, 0x5c80fa22, 0x00000000, 0x7e52f9ca, 0x58c0f9df, 0x5ca0fa26, 0x00000000,
	0x00000000, 0x00000000, 0x5cb0fa2a, 0x5cc0fa2b, 0x5cd0fa2c, 0x5ce0fa2d, 0x35702209, 0x00000000,
	0x5cf0fa30, 0x5d00fa31, 0x5d10fa32, 0x3bf00a33, 0x5d30fa34, 0x5d40fa35, 0x3c000a36, 0x5d60fa37,
	0x5d70fa38, 0x5d80fa39, 0x5d90fa3a, 0x5da0fa3b, 0x5db0fa3c, 0x3580220c, 0x5dd0fa3e, 0x5de0fa3f,
	0x5df0fa40, 0x5e00fa41, 0x5e10fa42, 0x5e20fa43, 0x5e30fa44, 0x5e40fa45, 0x4c80f91b, 0x5e60fa47,
	0x5e70fa48, 0x5e80fa49, 0x5e90fa4a, 0x5ea0fa4b, 0x5eb0fa4c, 0x5ec0fa4d, 0x5ed0fa4e, 0x5ee0fa4f,
	0x5ef0fa50, 0x5f00fa51, 0x5f10fa52, 0x5f20fa53, 0x5f30fa54, 0x5f40fa55, 0x5f50fa56, 0x5f60fa57,
	0x5f70fa58, 0x3c100a59, 0x3c200a5a, 0x3c300a5b, 0x5fb0fa5c, 0x5fc0fa5d, 0x3c400a5e, 0x5fe0fa5f,
	0x5ff0fa60, 0x6000fa61, 0x6010fa62, 0x6020fa63, 0x6030fa64, 0x6040fa65, 0x6050fa66, 0x6060fa67,
	0x6070fa68, 0x6080fa69, 0x6090fa6a, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x60a0fa70, 0x60b0fa71, 0x60c0fa72, 0x60d0fa73, 0x5070f95a, 0x60f0fa75, 0x6100fa76, 0x6110fa77,
	0x6120fa78, 0x6130fa79, 0x6140fa7a, 0x6150fa7b, 0x5c90fa25, 0x6170fa7d, 0x6180fa7e, 0x6190fa7f,
	0x61a0fa80, 0x61b0fa81, 0x61c0fa82, 0x61d0fa83, 0x4ca0f91d, 0x61f0fa85, 0x6200fa86, 0x6210fa87,
	0x6220fa88, 0x6230fa89, 0x6240fa8a, 0x6250fa8b, 0x6260fa8c, 0x6270fa8d, 0x6280fa8e, 0x6290fa8f,
	0x62a0fa90, 0x62b0fa91, 0x62c0fa92, 0x62d0fa93, 0x62e0fa94, 0x62f0fa95, 0x6300fa96, 0x6310fa97,
	0x6320fa98, 0x6330fa99, 0x6340fa9a, 0x6350fa9b, 0x6360fa9c, 0x6370fa9d, 0x6380fa9e, 0x6390fa9f,
	0x63a0faa0, 0x63b0faa1, 0x63c0faa2, 0x63d0faa3, 0x63e0faa4, 0x63f0faa5, 0x6400faa6, 0x6410faa7,
	0x6420faa8, 0x6430faa9, 0x6440faaa, 0x6450faab, 0x6460faac, 0x6470faad, 0x6480faae, 0x6490faaf,
	0x64a0fab0, 0x5080f95b, 0x64c0fab2, 0x64d0fab3, 0x64e0fab4, 0x64f0fab5, 0x6500fab6, 0x6510fab7,
	0x6520fab8, 0x6530fab9, 0x6540faba, 0x6550fabb, 0x6560fabc, 0x6570fabd, 0x6580fabe, 0x35a02226,
	0x65a0fac0, 0x65b0fac1, 0x4cc0f91f, 0x65d0fac3, 0x65e0fac4, 0x65f0fac5, 0x6600fac6, 0x6610fac7,
	0x6620fac8, 0x6630fac9, 0x6640faca, 0x6650facb, 0x6660facc, 0x6670facd, 0x6680face, 0x3d90facf,
	0x3da0fad0, 0x3db0fad1, 0x6690fad2, 0x66a0fad3, 0x66b0fad4, 0x3dc0fad5, 0x3dd0fad6, 0x3de0fad7,
	0x66c0fad8, 0x66d0fad9, 0x00000000, 0x00000000, 0x3d802adc, 0x5510f9a4, 0x8162fa0c, 0x7f52f9e2,
	0x7d02f9b4, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7832f949,
	0x00000000, 0x00000000, 0x7552f908, 0x00000000, 0x00000000, 0x7432f8f1, 0x00000000, 0x997024c5,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7b02f98c, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x5520f9a5, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x7762f932, 0x00000000, 0x00000000, 0x33f01b06, 0x00000000,
	0x34001b08, 0x00000000, 0x34101b0a, 0x7792f936, 0x34201b0c, 0x00000000, 0x34301b0e, 0x00000000,
	0x00000000, 0x00000000, 0x34401b12, 0x00000000, 0x00000000, 0x00000000, 0x7582f90b, 0x00000000,
	0x00000000, 0x00000000, 0x7d12f9b5, 0x5530f9a6, 0x00000000, 0x3df0fb1d, 0x00000000, 0x3e00fb1f,
	0x00000000, 0x00000000, 0x00000000, 0x99e024cc, 0x00000000, 0x7922f95f, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x3e10fb2a, 0x3e20fb2b, 0x3e30fb2c, 0x3e40fb2d, 0x3e50fb2e, 0x3e60fb2f,
	0x3e70fb30, 0x3e80fb31, 0x3e90fb32, 0x3ea0fb33, 0x3eb0fb34, 0x3ec0fb35, 0x3ed0fb36, 0x99f024cd,
	0x3ee0fb38, 0x3ef0fb39, 0x3f00fb3a, 0x34501b3b, 0x3f20fb3c, 0x34601b3d, 0x3f30fb3e, 0x00000000,
	0x34701b40, 0x34801b41, 0x00000000, 0x34901b43, 0x3f70fb44, 0x00000000, 0x3f80fb46, 0x3f90fb47,
	0x32b00b48, 0x3fb0fb49, 0x3fc0fb4a, 0x32a00b4b, 0x32c00b4c, 0x3ff0fb4d, 0x4000fb4e, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x7542f907, 0x7b12f98d, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0xa0d0ff23, 0x00000000, 0x00000000, 0x3c500b5c, 0x3c600b5d, 0x00000000, 0x7942f962,
	0x7d22f9b6, 0x00000000, 0x00000000, 0x00000000, 0x7772f933, 0x00000000, 0x7782f934, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x64b0fab1, 0x00000000, 0x00000000, 0x00000000, 0x973010c3, 0x7592f90c, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x4d20f925, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x32d00b94, 0x00000000, 0xa0e0ff24, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x4d30f926, 0x00000000, 0x7f72f9e4, 0x00000000, 0x00000000,
	0x00000000, 0x7af2f98b, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7562f909,
	0x7d52f9b9, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x7b22f98e, 0x00000000, 0x00000000, 0x5580f9ab, 0x00000000,
	0x00000000, 0x00000000, 0x4d40f927, 0x00000000, 0x00000000, 0x00000000, 0x7952f963, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x9f902cc0, 0x00000000, 0x00000000,
	0x00000000, 0x77a2f938, 0x32e00bca, 0x33000bcb, 0x32f00bcc, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7c42f9a1,
	0x00000000, 0x00000000, 0x7932f960, 0x00000000, 0x5200f973, 0x79b2f96a, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x7102f8b7, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x7c22f99f, 0x00000000, 0x00000000, 0x55a0f9ad, 0x6eb2f88c, 0x00000000, 0x00000000,
	0x4d60f929, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x9a202c00, 0x9a302c01, 0x9a402c02, 0x9a502c03, 0x9a602c04, 0x9a702c05, 0x9a802c06, 0x7d62f9ba,
	0x9aa02c08, 0x9ab02c09, 0x9ac02c0a, 0x9ad02c0b, 0x9ae02c0c, 0x9af02c0d, 0x9b002c0e, 0x9b102c0f,
	0x9b202c10, 0x9b302c11, 0x7b32f98f, 0x9b502c13, 0x9b602c14, 0x9b702c15, 0x9b802c16, 0x9b902c17,
	0x9ba02c18, 0x9bb02c19, 0x8062f9f8, 0x9bd02c1b, 0x9be02c1c, 0x7962f964, 0x9c002c1e, 0x9c102c1f,
	0x9c202c20, 0x9c302c21, 0x9c402c22, 0x9c502c23, 0x9a902c07, 0x9c702c25, 0x3f10fb3b, 0x9c902c27,
	0x9ca02c28, 0x9cb02c29, 0x9cc02c2a, 0x9cd02c2b, 0x9ce02c2c, 0x9cf02c2d, 0x5d20fa33, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x75a2f90e, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x33100c48, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x55d0f9b0, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x7f82f9e6, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x9d102c60, 0x00000000, 0x8172fa0d, 0x9d302c63, 0x7572f90a, 0x00000000, 0x7d72f9bb, 0x9d502c67,
	0x00000000, 0x9d602c69, 0x00000000, 0x9d702c6b, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x55e0f9b1, 0x7b42f990, 0x00000000, 0x00000000, 0x97902162, 0x9d802c75, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x97a02163, 0x00000000, 0x00000000,
	0x9d902c80, 0x00000000, 0x9da02c82, 0x9bc02c1a, 0x9db02c84, 0x00000000, 0x9dc02c86, 0x77b2f93a,
	0x9dd02c88, 0x00000000, 0x9de02c8a, 0x5d50fa36, 0x9df02c8c, 0x00000000, 0x9e002c8e, 0x00000000,
	0x9e102c90, 0x00000000, 0x75b2f90f, 0x00000000, 0x9e302c94, 0x00000000, 0x9e402c96, 0x00000000,
	0x9e502c98, 0x7f62f9e3, 0x9e602c9a, 0x00000000, 0x9e702c9c, 0x79e2f96e, 0x9e802c9e, 0x00000000,
	0x9e902ca0, 0x00000000, 0x7672f920, 0x00000000, 0x9b402c12, 0x00000000, 0x9ec02ca6, 0x00000000,
	0x9ed02ca8, 0x00000000, 0x9ee02caa, 0x00000000, 0x9ef02cac, 0x00000000, 0x9f002cae, 0x00000000,
	0x9f102cb0, 0x00000000, 0x9f202cb2, 0x00000000, 0x9f302cb4, 0x9c602c24, 0x9f402cb6, 0x00000000,
	0x9f502cb8, 0x00000000, 0x7f92f9e7, 0x00000000, 0x9f702cbc, 0x00000000, 0x6c72f863, 0x9c802c26,
	0x33200cc0, 0x3f40fb40, 0x9fa02cc2, 0x00000000, 0x9fb02cc4, 0x7d82f9bc, 0x9fc02cc6, 0x33500cc7,
	0x33600cc8, 0x00000000, 0x33300cca, 0x33400ccb, 0x9ff02ccc, 0x00000000, 0xa0002cce, 0x00000000,
	0x7b52f991, 0x00000000, 0xa0202cd2, 0x00000000, 0xa0302cd4, 0xa240ff3a, 0xa0402cd6, 0x00000000,
	0xa0502cd8, 0x00000000, 0x7d42f9b8, 0x7972f966, 0xa0702cdc, 0x00000000, 0xa0802cde, 0x00000000,
	0x3f50fb41, 0x00000000, 0xa0a02ce2, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x9d002c2e,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x4de0f931, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x50a0f95d, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xa140ff2a, 0x00000000, 0x00000000, 0x9bf02c1d,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x7fa2f9e8, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x3f60fb43, 0x00000000,
	0x00000000, 0x00000000, 0x6590fabf, 0x00000000, 0x7d92f9bd, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7b62f992,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x7982f967, 0x6982f82f, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x33700d4a, 0x33900d4b, 0x33800d4c, 0x4e10f934, 0x00000000, 0x00000000,
	0x7d32f9b7, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x4d00f923, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7682f921,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x5dc0fa3d, 0x00000000, 0x00000000, 0x00000000,
	0x98b024b9, 0x00000000, 0x00000000, 0x00000000, 0x4e20f935, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x7fb2f9e9, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xa160ff2c, 0x65c0fac2,
	0x00000000, 0x00000000, 0x00000000, 0x7da2f9be, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7b72f993, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x7992f968, 0x7a42f978, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x60e0fa74, 0x00000000,
	0x00000000, 0x990024be, 0x4e40f937, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x75c2f912,
	0x00000000, 0x00000000, 0x4a501fbe, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x3fa0fb48, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x77d2f93f, 0x81b2fa15, 0x00000000, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x7fc2f9ea,
	0x00000000, 0x00000000, 0x33a00dda, 0x00000000, 0x33b00ddc, 0x33c00ddd, 0x33d00dde, 0x4a601fef,
	0x00000000, 0x00000000, 0x7db2f9bf, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x4e60f939, 0x00000000, 0x995024c3, 0x9d202c62, 0x00000000, 0x7b82f994, 0x00000000, 0x00000000,
	0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x9d402c64, 0x00000000, 0x00000000,
	0x79a2f969, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x01f01e00, 0x11101e01, 0x02101e02, 0x11301e03, 0x02201e04, 0x11401e05, 0x02301e06, 0x11501e07,
	0x02901e08, 0x11b01e09, 0x02a01e0a, 0x11c01e0b, 0x02c01e0c, 0x11e01e0d, 0x02f01e0e, 0x12101e0f,
	0x02d01e10, 0x11f01e11, 0x02e01e12, 0x12001e13, 0x03901e14, 0x12b01e15, 0x03a01e16, 0x12c01e17,
	0x04701e18, 0x13901e19, 0x04801e1a, 0x13a01e1b, 0x04501e1c, 0x13701e1d, 0x04901e1e, 0x13b01e1f,
	0x04c01e20, 0x13e01e21, 0x05201e22, 0x14401e23, 0x05501e24, 0x14701e25, 0x05301e26, 0x14501e27,
	0x05601e28, 0x14801e29, 0x05701e2a, 0x14901e2b, 0x06701e2c, 0x15901e2d, 0x06001e2e, 0x15201e2f,
	0x06901e30, 0x15c01e31, 0x06b01e32, 0x15e01e33, 0x06d01e34, 0x16001e35, 0x07001e36, 0x16301e37,
	0x07101e38, 0x16401e39, 0x07401e3a, 0x16701e3b, 0x07301e3c, 0x16601e3d, 0x07501e3e, 0x16801e3f,
	0x07601e40, 0x16901e41, 0x07701e42, 0x16a01e43, 0x07b01e44, 0x16e01e45, 0x07d01e46, 0x17001e47,
	0x08001e48, 0x17301e49, 0x07f01e4a, 0x17201e4b, 0x08901e4c, 0x17c01e4d, 0x08b01e4e, 0x17e01e4f,
	0x08d01e50, 0x18001e51, 0x08e01e52, 0x18101e53, 0x0a301e54, 0x19601e55, 0x0a401e56, 0x19701e57,
	0x0a601e58, 0x19901e59, 0x0aa01e5a, 0x19d01e5b, 0x0ab01e5c, 0x19e01e5d, 0x0ad01e5e, 0x1a001e5f,
	0x0b101e60, 0x1a401e61, 0x0b401e62, 0x1a701e63, 0x0af01e64, 0x1a201e65, 0x0b301e66, 0x1a601e67,
	0x0b501e68, 0x1a801e69, 0x0b801e6a, 0x1ab01e6b, 0x0ba01e6c, 0x1ae01e6d, 0x0be01e6e, 0x1b201e6f,
	0x0bd01e70, 0x1b101e71, 0x0d901e72, 0x1cd01e73, 0x0dc01e74, 0x1d001e75, 0x0db01e76, 0x1cf01e77,
	0x0c301e78, 0x1b701e79, 0x0c501e7a, 0x1b901e7b, 0x0dd01e7c, 0x1d101e7d, 0x0de01e7e, 0x1d201e7f,
	0x0df01e80, 0x1d301e81, 0x0e001e82, 0x1d401e83, 0x0e301e84, 0x1d701e85, 0x0e201e86, 0x1d601e87,
	0x0e401e88, 0x1d901e89, 0x0e501e8a, 0x1da01e8b, 0x0e601e8c, 0x1db01e8d, 0x0ec01e8e, 0x1e101e8f,
	0x0f101e90, 0x1e701e91, 0x0f401e92, 0x1ea01e93, 0x0f501e94, 0x1eb01e95, 0x14a01e96, 0x1ac01e97,
	0x1d801e98, 0x1e401e99, 0x00000000, 0x1f701e9b, 0x00000000, 0x00000000, 0x00000000, 0x00000000,
	0x01c01ea0, 0x10e01ea1, 0x01501ea2, 0x10801ea3, 0x00701ea4, 0x0fa01ea5, 0x00601ea6, 0x0f901ea7,
	0x00901ea8, 0x0fc01ea9, 0x00801eaa, 0x0fb01eab, 0x01d01eac, 0x10f01ead, 0x00e01eae, 0x10101eaf,
	0x00d01eb0, 0x10001eb1, 0x01001eb2, 0x10301eb3, 0x00f01eb4, 0x10201eb5, 0x01e01eb6, 0x11001eb7,
	0x04201eb8, 0x13401eb9, 0x03e01eba, 0x13001ebb, 0x03701ebc, 0x12901ebd, 0x03401ebe, 0x12601ebf,
	0x03301ec0, 0x12501ec1, 0x03601ec2, 0x12801ec3, 0x03501ec4, 0x12701ec5, 0x04301ec6, 0x13501ec7,
	0x06101ec8, 0x15301ec9, 0x06501eca, 0x15701ecb, 0x09f01ecc, 0x19201ecd, 0x09401ece, 0x18701ecf,
	0x08501ed0, 0x17801ed1, 0x08401ed2, 0x17701ed3, 0x08701ed4, 0x17a01ed5, 0x08601ed6, 0x17901ed7,
	0x0a001ed8, 0x19301ed9, 0x09b01eda, 0x18e01edb, 0x09a01edc, 0x18d01edd, 0x09d01ede, 0x19001edf,
	0x09c01ee0, 0x18f01ee1, 0x09e01ee2, 0x19101ee3, 0x0d801ee4, 0x1cc01ee5, 0x0cc01ee6, 0x1c001ee7,
	0x0d401ee8, 0x1c801ee9, 0x0d301eea, 0x1c701eeb, 0x0d601eec, 0x1ca01eed, 0x0d501eee, 0x1c901eef,
	0x0d701ef0, 0x1cb01ef1, 0x0e701ef2, 0x1dc01ef3, 0x0ef01ef4, 0x1e501ef5, 0x0ee01ef6, 0x1e301ef7,
	0x0ea01ef8, 0x1df01ef9, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x4ef0f942,
	0x26a01f00, 0x27201f01, 0x26b01f02, 0x27301f03, 0x26d01f04, 0x27501f05, 0x26f01f06, 0x27701f07,
	0x1ff01f08, 0x20701f09, 0x20001f0a, 0x20801f0b, 0x20201f0c, 0x20a01f0d, 0x20401f0e, 0x20c01f0f,
	0x28001f10, 0x28301f11, 0x28101f12, 0x28401f13, 0x28201f14, 0x28501f15, 0x00000000, 0x00000000,
	0x21301f18, 0x21601f19, 0x21401f1a, 0x21701f1b, 0x21501f1c, 0x21801f1d, 0x4f00f943, 0x00000000,
	0x28b01f20, 0x29301f21, 0x28c01f22, 0x29401f23, 0x28e01f24, 0x29601f25, 0x29001f26, 0x29801f27,
	0x21c01f28, 0x22401f29, 0x21d01f2a, 0x22501f2b, 0x21f01f2c, 0x22701f2d, 0x22101f2e, 0x22901f2f,
	0x2a801f30, 0x2ac01f31, 0x2a901f32, 0x2ad01f33, 0x2aa01f34, 0x2ae01f35, 0x2ab01f36, 0x2af01f37,
	0x23301f38, 0x23701f39, 0x23401f3a, 0x23801f3b, 0x23501f3c, 0x23901f3d, 0x23601f3e, 0x23a01f3f,
	0x2b401f40, 0x2b701f41, 0x2b501f42, 0x2b801f43, 0x2b601f44, 0x2b901f45, 0x3c700f43, 0x00000000,
	0x23e01f48, 0x24101f49, 0x23f01f4a, 0x24201f4b, 0x24001f4c, 0x24301f4d, 0x3c800f4d, 0x00000000,
	0x2c601f50, 0x2ca01f51, 0x2c701f52, 0x2cb01f53, 0x2c801f54, 0x2cc01f55, 0x2c901f56, 0x2cd01f57,
	0x3c900f52, 0x24b01f59, 0x3ca00f57, 0x24c01f5b, 0x3cb00f5c, 0x24d01f5d, 0x7df2f9c3, 0x24e01f5f,
	0x2d401f60, 0x2dc01f61, 0x2d501f62, 0x2dd01f63, 0x2d701f64, 0x2df01f65, 0x2d901f66, 0x2e101f67,
	0x25201f68, 0x25a01f69, 0x25301f6a, 0x25b01f6b, 0x25501f6c, 0x25d01f6d, 0x25701f6e, 0x25f01f6f,
	0x26301f70, 0x26601f71, 0x27d01f72, 0x27f01f73, 0x28601f74, 0x28901f75, 0x29e01f76, 0x2a001f77,
	0x2b101f78, 0x2b301f79, 0x2bc01f7a, 0x2be01f7b, 0x2cf01f7c, 0x2d201f7d, 0x3cc00f69, 0x3cd00f73,
	0x27101f80, 0x27901f81, 0x26c01f82, 0x27401f83, 0x26e01f84, 0x27601f85, 0x27001f86, 0x27801f87,
	0x20601f88, 0x20e01f89, 0x20101f8a, 0x20901f8b, 0x20301f8c, 0x20b01f8d, 0x20501f8e, 0x20d01f8f,
	0x29201f90, 0x29a01f91, 0x28d01f92, 0x29501f93, 0x28f01f94, 0x29701f95, 0x29101f96, 0x29901f97,
	0x22301f98, 0x22b01f99, 0x21e01f9a, 0x22601f9b, 0x22001f9c, 0x22801f9d, 0x22201f9e, 0x22a01f9f,
	0x2db01fa0, 0x2e301fa1, 0x2d601fa2, 0x2de01fa3, 0x2d801fa4, 0x2e001fa5, 0x2da01fa6, 0x2e201fa7,
	0x25901fa8, 0x26101fa9, 0x25401faa, 0x25c01fab, 0x25601fac, 0x25e01fad, 0x25801fae, 0x26001faf,
	0x26901fb0, 0x26801fb1, 0x26401fb2, 0x27c01fb3, 0x26701fb4, 0x3ce00f75, 0x27a01fb6, 0x27b01fb7,
	0x1fe01fb8, 0x1fd01fb9, 0x1fa01fba, 0x1fc01fbb, 0x20f01fbc, 0x3cf00f76, 0x3d000f78, 0x3d100f81,
	0x3d200f93, 0x1ef01fc1, 0x28701fc2, 0x29d01fc3, 0x28a01fc4, 0x3d300f9d, 0x29b01fc6, 0x29c01fc7,
	0x21001fc8, 0x21201fc9, 0x21901fca, 0x21b01fcb, 0x22c01fcc, 0x34a01fcd, 0x34b01fce, 0x34c01fcf,
	0x2a201fd0, 0x2a101fd1, 0x2a401fd2, 0x2a601fd3, 0x3d400fa2, 0x3d500fa7, 0x2b001fd6, 0x2a701fd7,
	0x23101fd8, 0x23001fd9, 0x22d01fda, 0x22f01fdb, 0x3d600fac, 0x34d01fdd, 0x34e01fde, 0x34f01fdf,
	0x2c001fe0, 0x2bf01fe1, 0x2c201fe2, 0x2c401fe3, 0x2ba01fe4, 0x2bb01fe5, 0x2ce01fe6, 0x2c501fe7,
	0x24901fe8, 0x24801fe9, 0x24501fea, 0x24701feb, 0x24401fec, 0x1ec01fed, 0x1ee01fee, 0x3d700fb9,
	0x00000000, 0x00000000, 0x2d001ff2, 0x2e601ff3, 0x2d301ff4, 0x00000000, 0x2e401ff6, 0x2e501ff7,
	0x23b01ff8, 0x23d01ff9, 0x24f01ffa, 0x25101ffb, 0x26201ffc, 0x1f001ffd, 0x00000000, 0x00000000
	};

// Hash table access functions
const int KDecompositionHashBitmask = 0xfff;

inline int DecompositionHashStart(long a)
	{
	return a & 0xfff;
	}

inline int DecompositionHashStep(long a)
	{
	a *= a >> 12;
	return ((a<<1) + 1) & 0xfff;
	}

// Table mapping KNonSingletonDecompositions to the hash table entry that
// indexes it
const unsigned short KCompositionMapping[] =
	{
	0x026e, 0x0260, 0x026f, 0x00c0, 0x00c1, 0x00c2, 0x0ea6, 0x0ea4,
	0x0eaa, 0x0ea8, 0x00c3, 0x0100, 0x0102, 0x0eb0, 0x0eae, 0x0eb4,
	0x0eb2, 0x0226, 0x01e0, 0x00c4, 0x01de, 0x0ea2, 0x00c5, 0x012b,
	0x01fa, 0x01cd, 0x0200, 0x0202, 0x0ea0, 0x0eac, 0x0eb6, 0x0e00,
	0x0104, 0x0e02, 0x0e04, 0x0e06, 0x0106, 0x0108, 0x010a, 0x010c,
	0x00c7, 0x0e08, 0x0e0a, 0x010e, 0x0e0c, 0x0e10, 0x0e12, 0x0e0e,
	0x00c8, 0x00c9, 0x00ca, 0x0ec0, 0x0ebe, 0x0ec4, 0x0ec2, 0x0ebc,
	0x0112, 0x0e14, 0x0e16, 0x0114, 0x0116, 0x00cb, 0x0eba, 0x011a,
	0x0204, 0x0206, 0x0eb8, 0x0ec6, 0x0228, 0x0e1c, 0x0118, 0x0e18,
	0x0e1a, 0x0e1e, 0x01f4, 0x011c, 0x0e20, 0x011e, 0x0120, 0x01e6,
	0x0122, 0x0124, 0x0e22, 0x0e26, 0x021e, 0x0e24, 0x0e28, 0x0e2a,
	0x00cc, 0x00cd, 0x00ce, 0x0128, 0x012a, 0x012c, 0x0130, 0x00cf,
	0x0e2e, 0x0ec8, 0x01cf, 0x0208, 0x020a, 0x0eca, 0x012e, 0x0e2c,
	0x0134, 0x0e30, 0x01e8, 0x0e32, 0x0136, 0x0e34, 0x0139, 0x013d,
	0x0e36, 0x0e38, 0x013b, 0x0e3c, 0x0e3a, 0x0e3e, 0x0e40, 0x0e42,
	0x01f8, 0x0143, 0x00d1, 0x0e44, 0x0147, 0x0e46, 0x0145, 0x0e4a,
	0x0e48, 0x00d2, 0x00d3, 0x00d4, 0x0ed2, 0x0ed0, 0x0ed6, 0x0ed4,
	0x00d5, 0x0e4c, 0x022c, 0x0e4e, 0x014c, 0x0e50, 0x0e52, 0x014e,
	0x022e, 0x0230, 0x00d6, 0x022a, 0x0ece, 0x0150, 0x01d1, 0x020c,
	0x020e, 0x01a0, 0x0edc, 0x0eda, 0x0ee0, 0x0ede, 0x0ee2, 0x0ecc,
	0x0ed8, 0x01ea, 0x01ec, 0x0e54, 0x0e56, 0x0154, 0x0e58, 0x0158,
	0x0210, 0x0212, 0x0e5a, 0x0e5c, 0x0156, 0x0e5e, 0x015a, 0x0e64,
	0x015c, 0x0e60, 0x0160, 0x0e66, 0x0e62, 0x0e68, 0x0218, 0x015e,
	0x0e6a, 0x0164, 0x0e6c, 0x021a, 0x0162, 0x0e70, 0x0e6e, 0x00d9,
	0x00da, 0x00db, 0x0168, 0x0e78, 0x016a, 0x0e7a, 0x016c, 0x00dc,
	0x01db, 0x01d7, 0x01d5, 0x01d9, 0x0ee6, 0x016e, 0x0170, 0x01d3,
	0x0214, 0x0216, 0x01af, 0x0eea, 0x0ee8, 0x0eee, 0x0eec, 0x0ef0,
	0x0ee4, 0x0e72, 0x0172, 0x0e76, 0x0e74, 0x0e7c, 0x0e7e, 0x0e80,
	0x0e82, 0x0174, 0x0e86, 0x0e84, 0x0e88, 0x0e8a, 0x0e8c, 0x0ef2,
	0x00dd, 0x0176, 0x0ef8, 0x0232, 0x0e8e, 0x0178, 0x0ef6, 0x0ef4,
	0x0179, 0x0e90, 0x017b, 0x017d, 0x0e92, 0x0e94, 0x00e0, 0x00e1,
	0x00e2, 0x0ea7, 0x0ea5, 0x0eab, 0x0ea9, 0x00e3, 0x0101, 0x0103,
	0x0eb1, 0x0eaf, 0x0eb5, 0x0eb3, 0x0227, 0x01e1, 0x00e4, 0x01df,
	0x0ea3, 0x00e5, 0x01fb, 0x01ce, 0x0201, 0x0203, 0x0ea1, 0x0ead,
	0x0eb7, 0x0e01, 0x0105, 0x0e03, 0x0e05, 0x0e07, 0x0107, 0x0109,
	0x010b, 0x010d, 0x00e7, 0x0e09, 0x0e0b, 0x010f, 0x0e0d, 0x0e11,
	0x0e13, 0x0e0f, 0x00e8, 0x00e9, 0x00ea, 0x0ec1, 0x0ebf, 0x0ec5,
	0x0ec3, 0x0ebd, 0x0113, 0x0e15, 0x0e17, 0x0115, 0x0117, 0x00eb,
	0x0ebb, 0x011b, 0x0205, 0x0207, 0x0eb9, 0x0ec7, 0x0229, 0x0e1d,
	0x0119, 0x0e19, 0x0e1b, 0x0e1f, 0x01f5, 0x011d, 0x0e21, 0x011f,
	0x0121, 0x01e7, 0x0123, 0x0125, 0x0e23, 0x0e27, 0x021f, 0x0e25,
	0x0e29, 0x0e2b, 0x0e96, 0x00ec, 0x00ed, 0x00ee, 0x0129, 0x012d,
	0x012f, 0x00ef, 0x0e2f, 0x0ec9, 0x01d0, 0x0209, 0x020b, 0x0ecb,
	0x0131, 0x0e2d, 0x0135, 0x01f0, 0x0e31, 0x01e9, 0x0e33, 0x0137,
	0x0e35, 0x013a, 0x013e, 0x0e37, 0x0e39, 0x013c, 0x0e3d, 0x0e3b,
	0x0e3f, 0x0e41, 0x0e43, 0x01f9, 0x0144, 0x00f1, 0x0e45, 0x0148,
	0x0e47, 0x0146, 0x0e4b, 0x0e49, 0x00f2, 0x00f3, 0x00f4, 0x0ed3,
	0x0ed1, 0x0ed7, 0x0ed5, 0x00f5, 0x0e4d, 0x022d, 0x0e4f, 0x014d,
	0x0e51, 0x0e53, 0x014f, 0x022f, 0x0231, 0x00f6, 0x022b, 0x0ecf,
	0x0151, 0x01d2, 0x020d, 0x020f, 0x01a1, 0x0edd, 0x0edb, 0x0ee1,
	0x0edf, 0x0ee3, 0x0ecd, 0x0ed9, 0x01eb, 0x01ed, 0x0e55, 0x0e57,
	0x0155, 0x0e59, 0x0159, 0x0211, 0x0213, 0x0e5b, 0x0e5d, 0x0157,
	0x0e5f, 0x015b, 0x0e65, 0x015d, 0x0e61, 0x0161, 0x0e67, 0x0e63,
	0x0e69, 0x0219, 0x015f, 0x0e6b, 0x0e97, 0x0165, 0x0e6d, 0x021b,
	0x0163, 0x0e71, 0x0e6f, 0x00f9, 0x00fa, 0x00fb, 0x0169, 0x0e79,
	0x016b, 0x0e7b, 0x016d, 0x00fc, 0x01dc, 0x01d8, 0x01d6, 0x01da,
	0x0ee7, 0x016f, 0x0171, 0x01d4, 0x0215, 0x0217, 0x01b0, 0x0eeb,
	0x0ee9, 0x0eef, 0x0eed, 0x0ef1, 0x0ee5, 0x0e73, 0x0173, 0x0e77,
	0x0e75, 0x0e7d, 0x0e7f, 0x0e81, 0x0e83, 0x0175, 0x0e87, 0x0e85,
	0x0e98, 0x0e89, 0x0e8b, 0x0e8d, 0x0ef3, 0x00fd, 0x0177, 0x0ef9,
	0x0233, 0x0e8f, 0x00ff, 0x0ef7, 0x0e99, 0x0ef5, 0x017a, 0x0e91,
	0x017c, 0x017e, 0x0e93, 0x0e95, 0x0fed, 0x0385, 0x0fee, 0x0fc1,
	0x0ffd, 0x01fc, 0x01e2, 0x01fe, 0x01fd, 0x01e3, 0x01ff, 0x0e9b,
	0x01ee, 0x01ef, 0x0fba, 0x0386, 0x0fbb, 0x0fb9, 0x0fb8, 0x0f08,
	0x0f0a, 0x0f8a, 0x0f0c, 0x0f8c, 0x0f0e, 0x0f8e, 0x0f88, 0x0f09,
	0x0f0b, 0x0f8b, 0x0f0d, 0x0f8d, 0x0f0f, 0x0f8f, 0x0f89, 0x0fbc,
	0x0fc8, 0x0388, 0x0fc9, 0x0f18, 0x0f1a, 0x0f1c, 0x0f19, 0x0f1b,
	0x0f1d, 0x0fca, 0x0389, 0x0fcb, 0x0f28, 0x0f2a, 0x0f9a, 0x0f2c,
	0x0f9c, 0x0f2e, 0x0f9e, 0x0f98, 0x0f29, 0x0f2b, 0x0f9b, 0x0f2d,
	0x0f9d, 0x0f2f, 0x0f9f, 0x0f99, 0x0fcc, 0x0fda, 0x038a, 0x0fdb,
	0x0fd9, 0x0fd8, 0x03aa, 0x0f38, 0x0f3a, 0x0f3c, 0x0f3e, 0x0f39,
	0x0f3b, 0x0f3d, 0x0f3f, 0x0ff8, 0x038c, 0x0ff9, 0x0f48, 0x0f4a,
	0x0f4c, 0x0f49, 0x0f4b, 0x0f4d, 0x0fec, 0x0fea, 0x038e, 0x0feb,
	0x0fe9, 0x0fe8, 0x03ab, 0x0f59, 0x0f5b, 0x0f5d, 0x0f5f, 0x0ffa,
	0x038f, 0x0ffb, 0x0f68, 0x0f6a, 0x0faa, 0x0f6c, 0x0fac, 0x0f6e,
	0x0fae, 0x0fa8, 0x0f69, 0x0f6b, 0x0fab, 0x0f6d, 0x0fad, 0x0f6f,
	0x0faf, 0x0fa9, 0x0ffc, 0x0f70, 0x0fb2, 0x03ac, 0x0f71, 0x0fb4,
	0x0fb1, 0x0fb0, 0x0f00, 0x0f02, 0x0f82, 0x0f04, 0x0f84, 0x0f06,
	0x0f86, 0x0f80, 0x0f01, 0x0f03, 0x0f83, 0x0f05, 0x0f85, 0x0f07,
	0x0f87, 0x0f81, 0x0fb6, 0x0fb7, 0x0fb3, 0x0f72, 0x03ad, 0x0f73,
	0x0f10, 0x0f12, 0x0f14, 0x0f11, 0x0f13, 0x0f15, 0x0f74, 0x0fc2,
	0x03ae, 0x0f75, 0x0fc4, 0x0f20, 0x0f22, 0x0f92, 0x0f24, 0x0f94,
	0x0f26, 0x0f96, 0x0f90, 0x0f21, 0x0f23, 0x0f93, 0x0f25, 0x0f95,
	0x0f27, 0x0f97, 0x0f91, 0x0fc6, 0x0fc7, 0x0fc3, 0x0f76, 0x03af,
	0x0f77, 0x0fd1, 0x0fd0, 0x03ca, 0x0fd2, 0x0390, 0x0fd3, 0x0fd7,
	0x0f30, 0x0f32, 0x0f34, 0x0f36, 0x0f31, 0x0f33, 0x0f35, 0x0f37,
	0x0fd6, 0x0f78, 0x03cc, 0x0f79, 0x0f40, 0x0f42, 0x0f44, 0x0f41,
	0x0f43, 0x0f45, 0x0fe4, 0x0fe5, 0x0f7a, 0x03cd, 0x0f7b, 0x0fe1,
	0x0fe0, 0x03cb, 0x0fe2, 0x03b0, 0x0fe3, 0x0fe7, 0x0f50, 0x0f52,
	0x0f54, 0x0f56, 0x0f51, 0x0f53, 0x0f55, 0x0f57, 0x0fe6, 0x0f7c,
	0x0ff2, 0x03ce, 0x0f7d, 0x0ff4, 0x0f60, 0x0f62, 0x0fa2, 0x0f64,
	0x0fa4, 0x0f66, 0x0fa6, 0x0fa0, 0x0f61, 0x0f63, 0x0fa3, 0x0f65,
	0x0fa5, 0x0f67, 0x0fa7, 0x0fa1, 0x0ff6, 0x0ff7, 0x0ff3, 0x03d3,
	0x03d4, 0x0407, 0x04d0, 0x04d2, 0x0403, 0x0400, 0x04d6, 0x0401,
	0x04c1, 0x04dc, 0x04de, 0x040d, 0x04e2, 0x0419, 0x04e4, 0x040c,
	0x04e6, 0x04ee, 0x040e, 0x04f0, 0x04f2, 0x04f4, 0x04f8, 0x04ec,
	0x04d1, 0x04d3, 0x0453, 0x0450, 0x04d7, 0x0451, 0x04c2, 0x04dd,
	0x04df, 0x045d, 0x04e3, 0x0439, 0x04e5, 0x045c, 0x04e7, 0x04ef,
	0x045e, 0x04f1, 0x04f3, 0x04f5, 0x04f9, 0x04ed, 0x0457, 0x0476,
	0x0477, 0x04da, 0x04db, 0x04ea, 0x04eb, 0x0622, 0x0623, 0x0625,
	0x0624, 0x0626, 0x06c2, 0x06d3, 0x06c0, 0x0929, 0x0931, 0x0934,
	0x09cb, 0x09cc, 0x0b4b, 0x0b48, 0x0b4c, 0x0b94, 0x0bca, 0x0bcc,
	0x0bcb, 0x0c48, 0x0cc0, 0x0cca, 0x0ccb, 0x0cc7, 0x0cc8, 0x0d4a,
	0x0d4c, 0x0d4b, 0x0dda, 0x0ddc, 0x0ddd, 0x0dde, 0x0026, 0x0b06,
	0x0b08, 0x0b0a, 0x0b0c, 0x0b0e, 0x0b12, 0x0b3b, 0x0b3d, 0x0b40,
	0x0b41, 0x0b43, 0x0fcd, 0x0fce, 0x0fcf, 0x0fdd, 0x0fde, 0x0fdf,
	0x019a, 0x019b, 0x01ae, 0x0902, 0x090c, 0x0907, 0x0a15, 0x0a2e,
	0x0a3d, 0x0224, 0x0abf, 0x0241, 0x0244, 0x0247, 0x0249, 0x026d,
	0x0262, 0x0270, 0x0271, 0x0274, 0x0275, 0x0278, 0x0279, 0x0280,
	0x0281, 0x02e0, 0x02e1, 0x0284, 0x0285, 0x0288, 0x0289, 0x02e2,
	0x02e3, 0x02ac, 0x02ad, 0x02ae, 0x02af, 0x02ea, 0x02eb, 0x02ec,
	0x02ed, 0x0094, 0x004c, 0x004e, 0x0050, 0x0052, 0x0054, 0x0056,
	0x0058, 0x005a, 0x005c, 0x005e, 0x0060, 0x0062, 0x0065, 0x0067,
	0x0069, 0x0070, 0x0071, 0x0073, 0x0074, 0x0076, 0x0077, 0x0079,
	0x007a, 0x007c, 0x007d, 0x009e, 0x06ad, 0x00ac, 0x00ae, 0x00b0,
	0x00b2, 0x00b4, 0x00b6, 0x00b8, 0x00ba, 0x00bc, 0x00be, 0x0541,
	0x054f, 0x0564, 0x0572, 0x0580, 0x00d0, 0x05b8, 0x05c6, 0x05cd,
	0x05db, 0x00d7, 0x05f0, 0x05f7, 0x0605, 0x060c, 0x00f7, 0x00f8,
	0x06d0, 0x06d7, 0x00fe
	};

// Table containing all the decomposition and folding strings longer
// than 2 UTF16 characters
const unsigned short KLongDecompositions[] =
	{
	0xD834, 0xDDBA, 0xD834, 0xDD65, 0xD834, 0xDD6E, 0xD834, 0xDD58,
	0xD834, 0xDD65, 0xD834, 0xDD72, 0xD834, 0xDDBA, 0xD834, 0xDD65,
	0xD834, 0xDD6F, 0xD834, 0xDDB9, 0xD834, 0xDD65, 0xD834, 0xDD6F,
	0xD834, 0xDD58, 0xD834, 0xDD65, 0xD834, 0xDD71, 0xD834, 0xDDB9,
	0xD834, 0xDD65, 0xD834, 0xDD6E, 0xD834, 0xDD58, 0xD834, 0xDD65,
	0xD834, 0xDD6E, 0xD834, 0xDD58, 0xD834, 0xDD65, 0xD834, 0xDD6F,
	0xD834, 0xDD58, 0xD834, 0xDD65, 0xD834, 0xDD70, 0x03B1, 0x0313,
	0x0300, 0x0345, 0x03A9, 0x0314, 0x0300, 0x0345, 0x03C9, 0x0313,
	0x0342, 0x0345, 0xD834, 0xDD57, 0xD834, 0xDD65, 0x0391, 0x0313,
	0x0301, 0x0345, 0x03C9, 0x0313, 0x0301, 0x0345, 0x03B1, 0x0314,
	0x0300, 0x0345, 0x0397, 0x0313, 0x0342, 0x0345, 0x0391, 0x0314,
	0x0301, 0x0345, 0x03B7, 0x0314, 0x0300, 0x0345, 0x03A9, 0x0314,
	0x0342, 0x0345, 0x0397, 0x0314, 0x0342, 0x0345, 0x03A9, 0x0313,
	0x0300, 0x0345, 0x03B1, 0x0313, 0x0342, 0x0345, 0x03A9, 0x0313,
	0x0342, 0x0345, 0x03A9, 0x0314, 0x0301, 0x0345, 0x0397, 0x0313,
	0x0301, 0x0345, 0x03C9, 0x0314, 0x0300, 0x0345, 0x03B1, 0x0313,
	0x0301, 0x0345, 0x03B7, 0x0313, 0x0300, 0x0345, 0x03B1, 0x0314,
	0x0301, 0x0345, 0x0397, 0x0314, 0x0301, 0x0345, 0x03B1, 0x0314,
	0x0342, 0x0345, 0x03B7, 0x0313, 0x0301, 0x0345, 0x03C9, 0x0314,
	0x0301, 0x0345, 0x0397, 0x0314, 0x0300, 0x0345, 0x0391, 0x0313,
	0x0342, 0x0345, 0x0391, 0x0314, 0x0342, 0x0345, 0x0397, 0x0313,
	0x0300, 0x0345, 0x03C9, 0x0313, 0x0300, 0x0345, 0x0391, 0x0313,
	0x0300, 0x0345, 0x03B7, 0x0314, 0x0342, 0x0345, 0x03B7, 0x0314,
	0x0301, 0x0345, 0x03C9, 0x0314, 0x0342, 0x0345, 0x0391, 0x0314,
	0x0300, 0x0345, 0x03A9, 0x0313, 0x0301, 0x0345, 0x03B7, 0x0313,
	0x0342, 0x0345, 0x006F, 0x0328, 0x0304, 0x0399, 0x0314, 0x0342,
	0x006F, 0x0302, 0x0309, 0x0041, 0x0308, 0x0304, 0x03B9, 0x0314,
	0x0300, 0x006F, 0x031B, 0x0303, 0x03B7, 0x0314, 0x0345, 0x03B7,
	0x0301, 0x0345, 0x0065, 0x0304, 0x0300, 0x0061, 0x0323, 0x0306,
	0x0061, 0x0306, 0x0303, 0x0065, 0x0323, 0x0302, 0x0041, 0x0307,
	0x0304, 0x004F, 0x031B, 0x0309, 0x03B9, 0x0313, 0x0342, 0x03BF,
	0x0313, 0x0300, 0x006F, 0x0307, 0x0304, 0x0075, 0x031B, 0x0303,
	0x03C5, 0x0308, 0x0301, 0x0041, 0x0306, 0x0303, 0x05E9, 0x05BC,
	0x05C1, 0x0055, 0x0308, 0x030C, 0x0399, 0x0313, 0x0300, 0x006F,
	0x0303, 0x0304, 0x03C5, 0x0314, 0x0342, 0x03B9, 0x0314, 0x0301,
	0x0045, 0x0302, 0x0300, 0x03B5, 0x0314, 0x0301, 0x0395, 0x0313,
	0x0301, 0x0041, 0x0302, 0x0300, 0x0055, 0x0308, 0x0300, 0x03B1,
	0x0313, 0x0345, 0x004F, 0x0303, 0x0301, 0x0075, 0x031B, 0x0323,
	0x03A9, 0x0314, 0x0345, 0x0041, 0x0306, 0x0300, 0x0061, 0x0306,
	0x0300, 0x006F, 0x0302, 0x0303, 0x039F, 0x0313, 0x0301, 0x006F,
	0x031B, 0x0323, 0x006F, 0x0308, 0x0304, 0x03B9, 0x0308, 0x0342,
	0x004F, 0x031B, 0x0301, 0x0043, 0x0327, 0x0301, 0x004F, 0x0308,
	0x0304, 0x0041, 0x0302, 0x0309, 0x0DD9, 0x0DCF, 0x0DCA, 0x0061,
	0x0302, 0x0303, 0x006F, 0x0302, 0x0301, 0x0065, 0x0302, 0x0309,
	0x006F, 0x031B, 0x0300, 0x0061, 0x0308, 0x0304, 0x03B9, 0x0313,
	0x0300, 0x03B1, 0x0301, 0x0345, 0x03B1, 0x0342, 0x0345, 0x006F,
	0x0304, 0x0301, 0x03B5, 0x0313, 0x0301, 0x039F, 0x0314, 0x0300,
	0x0041, 0x0302, 0x0303, 0x0055, 0x0304, 0x0308, 0x006F, 0x031B,
	0x0309, 0x0075, 0x0308, 0x030C, 0x03B9, 0x0308, 0x0301, 0x004C,
	0x0323, 0x0304, 0x006F, 0x0304, 0x0300, 0x006F, 0x0303, 0x0301,
	0x0045, 0x0302, 0x0301, 0x0391, 0x0314, 0x0345, 0x0045, 0x0327,
	0x0306, 0x03C9, 0x0314, 0x0345, 0x0073, 0x0323, 0x0307, 0x0065,
	0x0302, 0x0301, 0x0053, 0x0323, 0x0307, 0x0075, 0x0308, 0x0301,
	0x0045, 0x0302, 0x0309, 0x0055, 0x0308, 0x0301, 0x0061, 0x0307,
	0x0304, 0x03C5, 0x0308, 0x0300, 0x004F, 0x0303, 0x0308, 0x03BF,
	0x0314, 0x0300, 0x03B1, 0x0314, 0x0345, 0x0399, 0x0313, 0x0301,
	0x05E9, 0x05BC, 0x05C2, 0x0063, 0x0327, 0x0301, 0x0055, 0x031B,
	0x0303, 0x03B1, 0x0300, 0x0345, 0x004F, 0x0302, 0x0303, 0x0061,
	0x0302, 0x0309, 0x004F, 0x0304, 0x0300, 0x0045, 0x0323, 0x0302,
	0x0391, 0x0313, 0x0345, 0x0053, 0x0301, 0x0307, 0x0055, 0x031B,
	0x0323, 0x006C, 0x0323, 0x0304, 0x0073, 0x0301, 0x0307, 0x0055,
	0x031B, 0x0309, 0x03B9, 0x0314, 0x0342, 0x0061, 0x0302, 0x0300,
	0x004F, 0x031B, 0x0303, 0x03B7, 0x0313, 0x0345, 0x0075, 0x0308,
	0x0304, 0x03C5, 0x0313, 0x0300, 0x0055, 0x031B, 0x0301, 0x0073,
	0x030C, 0x0307, 0x004F, 0x0304, 0x0301, 0x0041, 0x0323, 0x0302,
	0x006F, 0x031B, 0x0301, 0x0397, 0x0313, 0x0345, 0x0055, 0x0303,
	0x0301, 0x0061, 0x0306, 0x0301, 0x0061, 0x0323, 0x0302, 0x0065,
	0x0302, 0x0300, 0x004F, 0x0302, 0x0300, 0x03C5, 0x0314, 0x0301,
	0x0041, 0x0306, 0x0301, 0x004F, 0x031B, 0x0323, 0x03A9, 0x0313,
	0x0345, 0x0072, 0x0323, 0x0304, 0x0061, 0x030A, 0x0301, 0x03C9,
	0x0300, 0x0345, 0x0065, 0x0327, 0x0306, 0x0075, 0x031B, 0x0309,
	0x039F, 0x0314, 0x0301, 0x0055, 0x0308, 0x0304, 0x0061, 0x0306,
	0x0309, 0x0049, 0x0308, 0x0301, 0x0399, 0x0314, 0x0301, 0x03C9,
	0x0301, 0x0345, 0x03C9, 0x0313, 0x0345, 0x0041, 0x030A, 0x0301,
	0x006F, 0x0323, 0x0302, 0x0075, 0x031B, 0x0300, 0x03A5, 0x0314,
	0x0342, 0x0041, 0x0306, 0x0309, 0x03A5, 0x0314, 0x0300, 0x03C9,
	0x0342, 0x0345, 0x0075, 0x0308, 0x0300, 0x004F, 0x0303, 0x0304,
	0x006F, 0x0303, 0x0308, 0x0045, 0x0302, 0x0303, 0x0075, 0x0303,
	0x0301, 0x0065, 0x0304, 0x0301, 0x004F, 0x0302, 0x0309, 0x0041,
	0x0323, 0x0306, 0x004F, 0x0323, 0x0302, 0x0053, 0x030C, 0x0307,
	0x03BF, 0x0314, 0x0301, 0x0397, 0x0314, 0x0345, 0x03B7, 0x0342,
	0x0345, 0x0045, 0x0304, 0x0301, 0x0055, 0x031B, 0x0300, 0x0069,
	0x0308, 0x0301, 0x0061, 0x0302, 0x0301, 0x03BF, 0x0313, 0x0301,
	0x0399, 0x0314, 0x0300, 0x004F, 0x0302, 0x0301, 0x03B5, 0x0314,
	0x0300, 0x03B9, 0x0308, 0x0300, 0x03C5, 0x0314, 0x0300, 0x004F,
	0x0307, 0x0304, 0x0052, 0x0323, 0x0304, 0x03B7, 0x0300, 0x0345,
	0x0395, 0x0314, 0x0300, 0x0399, 0x0313, 0x0342, 0x0CC6, 0x0CC2,
	0x0CD5, 0x0075, 0x0304, 0x0308, 0x004F, 0x031B, 0x0300, 0x006F,
	0x0302, 0x0300, 0x004F, 0x0328, 0x0304, 0x0395, 0x0313, 0x0300,
	0x03C5, 0x0313, 0x0301, 0x03C5, 0x0313, 0x0342, 0x03A5, 0x0314,
	0x0301, 0x039F, 0x0313, 0x0300, 0x03B5, 0x0313, 0x0300, 0x0041,
	0x0302, 0x0301, 0x0065, 0x0302, 0x0303, 0x0395, 0x0314, 0x0301,
	0x03B9, 0x0313, 0x0301, 0x0075, 0x031B, 0x0301, 0x0045, 0x0304,
	0x0300, 0x03C5, 0x0308, 0x0342
	};

// Table containing decompositions longer than one UTF16 character.
// The top of the table contains all compositions, sorted lexicographically.
// Any decompositions of length 2 are in the table as a pair of values,
// decompositions longer than that are represented by a KLongD followed by
// a value whose top four bits indicate the length of the decomposition minus
// three and whose bottom 12 bits indicate an index into the KLongDecompositions
// array where the decomposition starts.
const long KLongD = 0;
// sizeof/2 = 1143
const unsigned short KNonSingletonDecompositions[] =
	{
	0x003C, 0x0338, 0x003D, 0x0338, 0x003E, 0x0338, 0x0041, 0x0300,
	0x0041, 0x0301, 0x0041, 0x0302, KLongD, 0x0121, KLongD, 0x02D7,
	KLongD, 0x0178, KLongD, 0x0151, 0x0041, 0x0303, 0x0041, 0x0304,
	0x0041, 0x0306, KLongD, 0x0133, KLongD, 0x0220, KLongD, 0x0103,
	KLongD, 0x0259, 0x0041, 0x0307, KLongD, 0x00EE, 0x0041, 0x0308,
	KLongD, 0x00D3, 0x0041, 0x0309, 0x0041, 0x030A, 0x0041, 0x030A,
	KLongD, 0x024D, 0x0041, 0x030C, 0x0041, 0x030F, 0x0041, 0x0311,
	0x0041, 0x0323, KLongD, 0x0205, KLongD, 0x0277, 0x0041, 0x0325,
	0x0041, 0x0328, 0x0042, 0x0307, 0x0042, 0x0323, 0x0042, 0x0331,
	0x0043, 0x0301, 0x0043, 0x0302, 0x0043, 0x0307, 0x0043, 0x030C,
	0x0043, 0x0327, KLongD, 0x014B, 0x0044, 0x0307, 0x0044, 0x030C,
	0x0044, 0x0323, 0x0044, 0x0327, 0x0044, 0x032D, 0x0044, 0x0331,
	0x0045, 0x0300, 0x0045, 0x0301, 0x0045, 0x0302, KLongD, 0x0118,
	KLongD, 0x0190, KLongD, 0x026B, KLongD, 0x01A8, 0x0045, 0x0303,
	0x0045, 0x0304, KLongD, 0x02E6, KLongD, 0x0289, 0x0045, 0x0306,
	0x0045, 0x0307, 0x0045, 0x0308, 0x0045, 0x0309, 0x0045, 0x030C,
	0x0045, 0x030F, 0x0045, 0x0311, 0x0045, 0x0323, KLongD, 0x01D5,
	0x0045, 0x0327, KLongD, 0x0196, 0x0045, 0x0328, 0x0045, 0x032D,
	0x0045, 0x0330, 0x0046, 0x0307, 0x0047, 0x0301, 0x0047, 0x0302,
	0x0047, 0x0304, 0x0047, 0x0306, 0x0047, 0x0307, 0x0047, 0x030C,
	0x0047, 0x0327, 0x0048, 0x0302, 0x0048, 0x0307, 0x0048, 0x0308,
	0x0048, 0x030C, 0x0048, 0x0323, 0x0048, 0x0327, 0x0048, 0x032E,
	0x0049, 0x0300, 0x0049, 0x0301, 0x0049, 0x0302, 0x0049, 0x0303,
	0x0049, 0x0304, 0x0049, 0x0306, 0x0049, 0x0307, 0x0049, 0x0308,
	KLongD, 0x0241, 0x0049, 0x0309, 0x0049, 0x030C, 0x0049, 0x030F,
	0x0049, 0x0311, 0x0049, 0x0323, 0x0049, 0x0328, 0x0049, 0x0330,
	0x004A, 0x0302, 0x004B, 0x0301, 0x004B, 0x030C, 0x004B, 0x0323,
	0x004B, 0x0327, 0x004B, 0x0331, 0x004C, 0x0301, 0x004C, 0x030C,
	0x004C, 0x0323, KLongD, 0x0187, 0x004C, 0x0327, 0x004C, 0x032D,
	0x004C, 0x0331, 0x004D, 0x0301, 0x004D, 0x0307, 0x004D, 0x0323,
	0x004E, 0x0300, 0x004E, 0x0301, 0x004E, 0x0303, 0x004E, 0x0307,
	0x004E, 0x030C, 0x004E, 0x0323, 0x004E, 0x0327, 0x004E, 0x032D,
	0x004E, 0x0331, 0x004F, 0x0300, 0x004F, 0x0301, 0x004F, 0x0302,
	KLongD, 0x021A, KLongD, 0x029B, KLongD, 0x01CC, KLongD, 0x0274,
	0x004F, 0x0303, KLongD, 0x012A, KLongD, 0x0265, KLongD, 0x01B4,
	0x004F, 0x0304, KLongD, 0x01D2, KLongD, 0x0202, 0x004F, 0x0306,
	0x004F, 0x0307, KLongD, 0x02A7, 0x004F, 0x0308, KLongD, 0x014E,
	0x004F, 0x0309, 0x004F, 0x030B, 0x004F, 0x030C, 0x004F, 0x030F,
	0x004F, 0x0311, 0x004F, 0x031B, KLongD, 0x02BC, KLongD, 0x0148,
	KLongD, 0x01F0, KLongD, 0x00F1, KLongD, 0x0223, 0x004F, 0x0323,
	KLongD, 0x027A, 0x004F, 0x0328, KLongD, 0x02C2, 0x0050, 0x0301,
	0x0050, 0x0307, 0x0052, 0x0301, 0x0052, 0x0307, 0x0052, 0x030C,
	0x0052, 0x030F, 0x0052, 0x0311, 0x0052, 0x0323, KLongD, 0x02AA,
	0x0052, 0x0327, 0x0052, 0x0331, 0x0053, 0x0301, KLongD, 0x01DB,
	0x0053, 0x0302, 0x0053, 0x0307, 0x0053, 0x030C, KLongD, 0x027D,
	0x0053, 0x0323, KLongD, 0x01A2, 0x0053, 0x0326, 0x0053, 0x0327,
	0x0054, 0x0307, 0x0054, 0x030C, 0x0054, 0x0323, 0x0054, 0x0326,
	0x0054, 0x0327, 0x0054, 0x032D, 0x0054, 0x0331, 0x0055, 0x0300,
	0x0055, 0x0301, 0x0055, 0x0302, 0x0055, 0x0303, KLongD, 0x020E,
	0x0055, 0x0304, KLongD, 0x017B, 0x0055, 0x0306, 0x0055, 0x0308,
	KLongD, 0x0124, KLongD, 0x01AB, KLongD, 0x023B, KLongD, 0x0109,
	0x0055, 0x0309, 0x0055, 0x030A, 0x0055, 0x030B, 0x0055, 0x030C,
	0x0055, 0x030F, 0x0055, 0x0311, 0x0055, 0x031B, KLongD, 0x028C,
	KLongD, 0x01FC, KLongD, 0x01C6, KLongD, 0x01E7, KLongD, 0x01DE,
	0x0055, 0x0323, 0x0055, 0x0324, 0x0055, 0x0328, 0x0055, 0x032D,
	0x0055, 0x0330, 0x0056, 0x0303, 0x0056, 0x0323, 0x0057, 0x0300,
	0x0057, 0x0301, 0x0057, 0x0302, 0x0057, 0x0307, 0x0057, 0x0308,
	0x0057, 0x0323, 0x0058, 0x0307, 0x0058, 0x0308, 0x0059, 0x0300,
	0x0059, 0x0301, 0x0059, 0x0302, 0x0059, 0x0303, 0x0059, 0x0304,
	0x0059, 0x0307, 0x0059, 0x0308, 0x0059, 0x0309, 0x0059, 0x0323,
	0x005A, 0x0301, 0x005A, 0x0302, 0x005A, 0x0307, 0x005A, 0x030C,
	0x005A, 0x0323, 0x005A, 0x0331, 0x0061, 0x0300, 0x0061, 0x0301,
	0x0061, 0x0302, KLongD, 0x01ED, KLongD, 0x0292, KLongD, 0x0157,
	KLongD, 0x01CF, 0x0061, 0x0303, 0x0061, 0x0304, 0x0061, 0x0306,
	KLongD, 0x0136, KLongD, 0x0211, KLongD, 0x00E8, KLongD, 0x023E,
	0x0061, 0x0307, KLongD, 0x01AE, 0x0061, 0x0308, KLongD, 0x0163,
	0x0061, 0x0309, 0x0061, 0x030A, KLongD, 0x022C, 0x0061, 0x030C,
	0x0061, 0x030F, 0x0061, 0x0311, 0x0061, 0x0323, KLongD, 0x0214,
	KLongD, 0x00E5, 0x0061, 0x0325, 0x0061, 0x0328, 0x0062, 0x0307,
	0x0062, 0x0323, 0x0062, 0x0331, 0x0063, 0x0301, 0x0063, 0x0302,
	0x0063, 0x0307, 0x0063, 0x030C, 0x0063, 0x0327, KLongD, 0x01C3,
	0x0064, 0x0307, 0x0064, 0x030C, 0x0064, 0x0323, 0x0064, 0x0327,
	0x0064, 0x032D, 0x0064, 0x0331, 0x0065, 0x0300, 0x0065, 0x0301,
	0x0065, 0x0302, KLongD, 0x0217, KLongD, 0x019F, KLongD, 0x02DA,
	KLongD, 0x015D, 0x0065, 0x0303, 0x0065, 0x0304, KLongD, 0x00E2,
	KLongD, 0x0271, 0x0065, 0x0306, 0x0065, 0x0307, 0x0065, 0x0308,
	0x0065, 0x0309, 0x0065, 0x030C, 0x0065, 0x030F, 0x0065, 0x0311,
	0x0065, 0x0323, KLongD, 0x00EB, 0x0065, 0x0327, KLongD, 0x0232,
	0x0065, 0x0328, 0x0065, 0x032D, 0x0065, 0x0330, 0x0066, 0x0307,
	0x0067, 0x0301, 0x0067, 0x0302, 0x0067, 0x0304, 0x0067, 0x0306,
	0x0067, 0x0307, 0x0067, 0x030C, 0x0067, 0x0327, 0x0068, 0x0302,
	0x0068, 0x0307, 0x0068, 0x0308, 0x0068, 0x030C, 0x0068, 0x0323,
	0x0068, 0x0327, 0x0068, 0x032E, 0x0068, 0x0331, 0x0069, 0x0300,
	0x0069, 0x0301, 0x0069, 0x0302, 0x0069, 0x0303, 0x0069, 0x0304,
	0x0069, 0x0306, 0x0069, 0x0308, KLongD, 0x028F, 0x0069, 0x0309,
	0x0069, 0x030C, 0x0069, 0x030F, 0x0069, 0x0311, 0x0069, 0x0323,
	0x0069, 0x0328, 0x0069, 0x0330, 0x006A, 0x0302, 0x006A, 0x030C,
	0x006B, 0x0301, 0x006B, 0x030C, 0x006B, 0x0323, 0x006B, 0x0327,
	0x006B, 0x0331, 0x006C, 0x0301, 0x006C, 0x030C, 0x006C, 0x0323,
	KLongD, 0x01E1, 0x006C, 0x0327, 0x006C, 0x032D, 0x006C, 0x0331,
	0x006D, 0x0301, 0x006D, 0x0307, 0x006D, 0x0323, 0x006E, 0x0300,
	0x006E, 0x0301, 0x006E, 0x0303, 0x006E, 0x0307, 0x006E, 0x030C,
	0x006E, 0x0323, 0x006E, 0x0327, 0x006E, 0x032D, 0x006E, 0x0331,
	0x006F, 0x0300, 0x006F, 0x0301, 0x006F, 0x0302, KLongD, 0x02BF,
	KLongD, 0x015A, KLongD, 0x0139, KLongD, 0x00D0, 0x006F, 0x0303,
	KLongD, 0x018D, KLongD, 0x010F, KLongD, 0x0268, 0x006F, 0x0304,
	KLongD, 0x018A, KLongD, 0x016F, 0x006F, 0x0306, 0x006F, 0x0307,
	KLongD, 0x00FA, 0x006F, 0x0308, KLongD, 0x0142, 0x006F, 0x0309,
	0x006F, 0x030B, 0x006F, 0x030C, 0x006F, 0x030F, 0x006F, 0x0311,
	0x006F, 0x031B, KLongD, 0x0160, KLongD, 0x0208, KLongD, 0x00D9,
	KLongD, 0x017E, KLongD, 0x013F, 0x006F, 0x0323, KLongD, 0x0250,
	0x006F, 0x0328, KLongD, 0x00CA, 0x0070, 0x0301, 0x0070, 0x0307,
	0x0072, 0x0301, 0x0072, 0x0307, 0x0072, 0x030C, 0x0072, 0x030F,
	0x0072, 0x0311, 0x0072, 0x0323, KLongD, 0x0229, 0x0072, 0x0327,
	0x0072, 0x0331, 0x0073, 0x0301, KLongD, 0x01E4, 0x0073, 0x0302,
	0x0073, 0x0307, 0x0073, 0x030C, KLongD, 0x01FF, 0x0073, 0x0323,
	KLongD, 0x019C, 0x0073, 0x0326, 0x0073, 0x0327, 0x0074, 0x0307,
	0x0074, 0x0308, 0x0074, 0x030C, 0x0074, 0x0323, 0x0074, 0x0326,
	0x0074, 0x0327, 0x0074, 0x032D, 0x0074, 0x0331, 0x0075, 0x0300,
	0x0075, 0x0301, 0x0075, 0x0302, 0x0075, 0x0303, KLongD, 0x026E,
	0x0075, 0x0304, KLongD, 0x02B9, 0x0075, 0x0306, 0x0075, 0x0308,
	KLongD, 0x0262, KLongD, 0x01A5, KLongD, 0x01F6, KLongD, 0x0181,
	0x0075, 0x0309, 0x0075, 0x030A, 0x0075, 0x030B, 0x0075, 0x030C,
	0x0075, 0x030F, 0x0075, 0x0311, 0x0075, 0x031B, KLongD, 0x0253,
	KLongD, 0x02E3, KLongD, 0x00FD, KLongD, 0x0235, KLongD, 0x012D,
	0x0075, 0x0323, 0x0075, 0x0324, 0x0075, 0x0328, 0x0075, 0x032D,
	0x0075, 0x0330, 0x0076, 0x0303, 0x0076, 0x0323, 0x0077, 0x0300,
	0x0077, 0x0301, 0x0077, 0x0302, 0x0077, 0x0307, 0x0077, 0x0308,
	0x0077, 0x030A, 0x0077, 0x0323, 0x0078, 0x0307, 0x0078, 0x0308,
	0x0079, 0x0300, 0x0079, 0x0301, 0x0079, 0x0302, 0x0079, 0x0303,
	0x0079, 0x0304, 0x0079, 0x0307, 0x0079, 0x0308, 0x0079, 0x0309,
	0x0079, 0x030A, 0x0079, 0x0323, 0x007A, 0x0301, 0x007A, 0x0302,
	0x007A, 0x0307, 0x007A, 0x030C, 0x007A, 0x0323, 0x007A, 0x0331,
	0x00A8, 0x0300, 0x00A8, 0x0301, 0x00A8, 0x0301, 0x00A8, 0x0342,
	0x00B4, 0xFFFF, 0x00C6, 0x0301, 0x00C6, 0x0304, 0x00D8, 0x0301,
	0x00E6, 0x0301, 0x00E6, 0x0304, 0x00F8, 0x0301, 0x017F, 0x0307,
	0x01B7, 0x030C, 0x0292, 0x030C, 0x0391, 0x0300, 0x0391, 0x0301,
	0x0391, 0x0301, 0x0391, 0x0304, 0x0391, 0x0306, 0x0391, 0x0313,
	KLongD, 0x00AE, KLongD, 0x10AE, KLongD, 0x0046, KLongD, 0x1046,
	KLongD, 0x009E, KLongD, 0x109E, KLongD, 0x01D8, 0x0391, 0x0314,
	KLongD, 0x00BE, KLongD, 0x10BE, KLongD, 0x0056, KLongD, 0x1056,
	KLongD, 0x00A2, KLongD, 0x10A2, KLongD, 0x0193, 0x0391, 0x0345,
	0x0395, 0x0300, 0x0395, 0x0301, 0x0395, 0x0301, 0x0395, 0x0313,
	KLongD, 0x02C5, KLongD, 0x011E, 0x0395, 0x0314, KLongD, 0x02B0,
	KLongD, 0x02DD, 0x0397, 0x0300, 0x0397, 0x0301, 0x0397, 0x0301,
	0x0397, 0x0313, KLongD, 0x00A6, KLongD, 0x10A6, KLongD, 0x0076,
	KLongD, 0x1076, KLongD, 0x0052, KLongD, 0x1052, KLongD, 0x020B,
	0x0397, 0x0314, KLongD, 0x009A, KLongD, 0x109A, KLongD, 0x008A,
	KLongD, 0x108A, KLongD, 0x0062, KLongD, 0x1062, KLongD, 0x0283,
	0x0397, 0x0345, 0x0399, 0x0300, 0x0399, 0x0301, 0x0399, 0x0301,
	0x0399, 0x0304, 0x0399, 0x0306, 0x0399, 0x0308, 0x0399, 0x0313,
	KLongD, 0x010C, KLongD, 0x01BD, KLongD, 0x02B3, 0x0399, 0x0314,
	KLongD, 0x0298, KLongD, 0x0244, KLongD, 0x00CD, 0x039F, 0x0300,
	0x039F, 0x0301, 0x039F, 0x0301, 0x039F, 0x0313, KLongD, 0x02D1,
	KLongD, 0x013C, 0x039F, 0x0314, KLongD, 0x0175, KLongD, 0x0238,
	0x03A1, 0x0314, 0x03A5, 0x0300, 0x03A5, 0x0301, 0x03A5, 0x0301,
	0x03A5, 0x0304, 0x03A5, 0x0306, 0x03A5, 0x0308, 0x03A5, 0x0314,
	KLongD, 0x025C, KLongD, 0x02CE, KLongD, 0x0256, 0x03A9, 0x0300,
	0x03A9, 0x0301, 0x03A9, 0x0301, 0x03A9, 0x0313, KLongD, 0x0066,
	KLongD, 0x1066, KLongD, 0x00C2, KLongD, 0x10C2, KLongD, 0x006E,
	KLongD, 0x106E, KLongD, 0x0226, 0x03A9, 0x0314, KLongD, 0x003A,
	KLongD, 0x103A, KLongD, 0x0072, KLongD, 0x1072, KLongD, 0x005E,
	KLongD, 0x105E, KLongD, 0x0130, 0x03A9, 0x0345, 0x03B1, 0x0300,
	KLongD, 0x01C9, 0x03B1, 0x0301, 0x03B1, 0x0301, KLongD, 0x0169,
	0x03B1, 0x0304, 0x03B1, 0x0306, 0x03B1, 0x0313, KLongD, 0x0036,
	KLongD, 0x1036, KLongD, 0x007E, KLongD, 0x107E, KLongD, 0x006A,
	KLongD, 0x106A, KLongD, 0x0127, 0x03B1, 0x0314, KLongD, 0x004E,
	KLongD, 0x104E, KLongD, 0x0086, KLongD, 0x1086, KLongD, 0x008E,
	KLongD, 0x108E, KLongD, 0x01BA, 0x03B1, 0x0342, KLongD, 0x016C,
	0x03B1, 0x0345, 0x03B5, 0x0300, 0x03B5, 0x0301, 0x03B5, 0x0301,
	0x03B5, 0x0313, KLongD, 0x02D4, KLongD, 0x0172, 0x03B5, 0x0314,
	KLongD, 0x029E, KLongD, 0x011B, 0x03B7, 0x0300, KLongD, 0x02AD,
	0x03B7, 0x0301, 0x03B7, 0x0301, KLongD, 0x00DF, 0x03B7, 0x0313,
	KLongD, 0x0082, KLongD, 0x1082, KLongD, 0x0092, KLongD, 0x1092,
	KLongD, 0x00C6, KLongD, 0x10C6, KLongD, 0x01F3, 0x03B7, 0x0314,
	KLongD, 0x005A, KLongD, 0x105A, KLongD, 0x00B6, KLongD, 0x10B6,
	KLongD, 0x00B2, KLongD, 0x10B2, KLongD, 0x00DC, 0x03B7, 0x0342,
	KLongD, 0x0286, 0x03B7, 0x0345, 0x03B9, 0x0300, 0x03B9, 0x0301,
	0x03B9, 0x0301, 0x03B9, 0x0304, 0x03B9, 0x0306, 0x03B9, 0x0308,
	KLongD, 0x02A1, KLongD, 0x0184, KLongD, 0x0184, KLongD, 0x0145,
	0x03B9, 0x0313, KLongD, 0x0166, KLongD, 0x02E0, KLongD, 0x00F4,
	0x03B9, 0x0314, KLongD, 0x00D6, KLongD, 0x0115, KLongD, 0x01EA,
	0x03B9, 0x0342, 0x03BF, 0x0300, 0x03BF, 0x0301, 0x03BF, 0x0301,
	0x03BF, 0x0313, KLongD, 0x00F7, KLongD, 0x0295, 0x03BF, 0x0314,
	KLongD, 0x01B7, KLongD, 0x0280, 0x03C1, 0x0313, 0x03C1, 0x0314,
	0x03C5, 0x0300, 0x03C5, 0x0301, 0x03C5, 0x0301, 0x03C5, 0x0304,
	0x03C5, 0x0306, 0x03C5, 0x0308, KLongD, 0x01B1, KLongD, 0x0100,
	KLongD, 0x0100, KLongD, 0x02E9, 0x03C5, 0x0313, KLongD, 0x01F9,
	KLongD, 0x02C8, KLongD, 0x02CB, 0x03C5, 0x0314, KLongD, 0x02A4,
	KLongD, 0x021D, KLongD, 0x0112, 0x03C5, 0x0342, 0x03C9, 0x0300,
	KLongD, 0x022F, 0x03C9, 0x0301, 0x03C9, 0x0301, KLongD, 0x0247,
	0x03C9, 0x0313, KLongD, 0x00AA, KLongD, 0x10AA, KLongD, 0x004A,
	KLongD, 0x104A, KLongD, 0x003E, KLongD, 0x103E, KLongD, 0x024A,
	0x03C9, 0x0314, KLongD, 0x007A, KLongD, 0x107A, KLongD, 0x0096,
	KLongD, 0x1096, KLongD, 0x00BA, KLongD, 0x10BA, KLongD, 0x0199,
	0x03C9, 0x0342, KLongD, 0x025F, 0x03C9, 0x0345, 0x03D2, 0x0301,
	0x03D2, 0x0308, 0x0406, 0x0308, 0x0410, 0x0306, 0x0410, 0x0308,
	0x0413, 0x0301, 0x0415, 0x0300, 0x0415, 0x0306, 0x0415, 0x0308,
	0x0416, 0x0306, 0x0416, 0x0308, 0x0417, 0x0308, 0x0418, 0x0300,
	0x0418, 0x0304, 0x0418, 0x0306, 0x0418, 0x0308, 0x041A, 0x0301,
	0x041E, 0x0308, 0x0423, 0x0304, 0x0423, 0x0306, 0x0423, 0x0308,
	0x0423, 0x030B, 0x0427, 0x0308, 0x042B, 0x0308, 0x042D, 0x0308,
	0x0430, 0x0306, 0x0430, 0x0308, 0x0433, 0x0301, 0x0435, 0x0300,
	0x0435, 0x0306, 0x0435, 0x0308, 0x0436, 0x0306, 0x0436, 0x0308,
	0x0437, 0x0308, 0x0438, 0x0300, 0x0438, 0x0304, 0x0438, 0x0306,
	0x0438, 0x0308, 0x043A, 0x0301, 0x043E, 0x0308, 0x0443, 0x0304,
	0x0443, 0x0306, 0x0443, 0x0308, 0x0443, 0x030B, 0x0447, 0x0308,
	0x044B, 0x0308, 0x044D, 0x0308, 0x0456, 0x0308, 0x0474, 0x030F,
	0x0475, 0x030F, 0x04D8, 0x0308, 0x04D9, 0x0308, 0x04E8, 0x0308,
	0x04E9, 0x0308, 0x0627, 0x0653, 0x0627, 0x0654, 0x0627, 0x0655,
	0x0648, 0x0654, 0x064A, 0x0654, 0x06C1, 0x0654, 0x06D2, 0x0654,
	0x06D5, 0x0654, 0x0928, 0x093C, 0x0930, 0x093C, 0x0933, 0x093C,
	0x09C7, 0x09BE, 0x09C7, 0x09D7, 0x0B47, 0x0B3E, 0x0B47, 0x0B56,
	0x0B47, 0x0B57, 0x0B92, 0x0BD7, 0x0BC6, 0x0BBE, 0x0BC6, 0x0BD7,
	0x0BC7, 0x0BBE, 0x0C46, 0x0C56, 0x0CBF, 0x0CD5, 0x0CC6, 0x0CC2,
	KLongD, 0x02B6, 0x0CC6, 0x0CD5, 0x0CC6, 0x0CD6, 0x0D46, 0x0D3E,
	0x0D46, 0x0D57, 0x0D47, 0x0D3E, 0x0DD9, 0x0DCA, 0x0DD9, 0x0DCF,
	KLongD, 0x0154, 0x0DD9, 0x0DDF, 0x1025, 0x102E, 0x1B05, 0x1B35,
	0x1B07, 0x1B35, 0x1B09, 0x1B35, 0x1B0B, 0x1B35, 0x1B0D, 0x1B35,
	0x1B11, 0x1B35, 0x1B3A, 0x1B35, 0x1B3C, 0x1B35, 0x1B3E, 0x1B35,
	0x1B3F, 0x1B35, 0x1B42, 0x1B35, 0x1FBF, 0x0300, 0x1FBF, 0x0301,
	0x1FBF, 0x0342, 0x1FFE, 0x0300, 0x1FFE, 0x0301, 0x1FFE, 0x0342,
	0x2190, 0x0338, 0x2192, 0x0338, 0x2194, 0x0338, 0x21D0, 0x0338,
	0x21D2, 0x0338, 0x21D4, 0x0338, 0x2203, 0x0338, 0x2208, 0x0338,
	0x220B, 0x0338, 0x2223, 0x0338, 0x2225, 0x0338, 0x223C, 0x0338,
	0x2243, 0x0338, 0x2245, 0x0338, 0x2248, 0x0338, 0x224D, 0x0338,
	0x2261, 0x0338, 0x2264, 0x0338, 0x2265, 0x0338, 0x2272, 0x0338,
	0x2273, 0x0338, 0x2276, 0x0338, 0x2277, 0x0338, 0x227A, 0x0338,
	0x227B, 0x0338, 0x227C, 0x0338, 0x227D, 0x0338, 0x2282, 0x0338,
	0x2283, 0x0338, 0x2286, 0x0338, 0x2287, 0x0338, 0x2291, 0x0338,
	0x2292, 0x0338, 0x22A2, 0x0338, 0x22A8, 0x0338, 0x22A9, 0x0338,
	0x22AB, 0x0338, 0x22B2, 0x0338, 0x22B3, 0x0338, 0x22B4, 0x0338,
	0x22B5, 0x0338, 0x3046, 0x3099, 0x304B, 0x3099, 0x304D, 0x3099,
	0x304F, 0x3099, 0x3051, 0x3099, 0x3053, 0x3099, 0x3055, 0x3099,
	0x3057, 0x3099, 0x3059, 0x3099, 0x305B, 0x3099, 0x305D, 0x3099,
	0x305F, 0x3099, 0x3061, 0x3099, 0x3064, 0x3099, 0x3066, 0x3099,
	0x3068, 0x3099, 0x306F, 0x3099, 0x306F, 0x309A, 0x3072, 0x3099,
	0x3072, 0x309A, 0x3075, 0x3099, 0x3075, 0x309A, 0x3078, 0x3099,
	0x3078, 0x309A, 0x307B, 0x3099, 0x307B, 0x309A, 0x309D, 0x3099,
	0x30A6, 0x3099, 0x30AB, 0x3099, 0x30AD, 0x3099, 0x30AF, 0x3099,
	0x30B1, 0x3099, 0x30B3, 0x3099, 0x30B5, 0x3099, 0x30B7, 0x3099,
	0x30B9, 0x3099, 0x30BB, 0x3099, 0x30BD, 0x3099, 0x30BF, 0x3099,
	0x30C1, 0x3099, 0x30C4, 0x3099, 0x30C6, 0x3099, 0x30C8, 0x3099,
	0x30CF, 0x3099, 0x30CF, 0x309A, 0x30D2, 0x3099, 0x30D2, 0x309A,
	0x30D5, 0x3099, 0x30D5, 0x309A, 0x30D8, 0x3099, 0x30D8, 0x309A,
	0x30DB, 0x3099, 0x30DB, 0x309A, 0x30EF, 0x3099, 0x30F0, 0x3099,
	0x30F1, 0x3099, 0x30F2, 0x3099, 0x30FD, 0x3099, 0x0308, 0x0301,
	0x0915, 0x093C, 0x0916, 0x093C, 0x0917, 0x093C, 0x091C, 0x093C,
	0x0921, 0x093C, 0x0922, 0x093C, 0x092B, 0x093C, 0x092F, 0x093C,
	0x09A1, 0x09BC, 0x09A2, 0x09BC, 0x09AF, 0x09BC, 0x0A32, 0x0A3C,
	0x0A38, 0x0A3C, 0x0A16, 0x0A3C, 0x0A17, 0x0A3C, 0x0A1C, 0x0A3C,
	0x0A2B, 0x0A3C, 0x0B21, 0x0B3C, 0x0B22, 0x0B3C, 0x0F42, 0x0FB7,
	0x0F4C, 0x0FB7, 0x0F51, 0x0FB7, 0x0F56, 0x0FB7, 0x0F5B, 0x0FB7,
	0x0F40, 0x0FB5, 0x0F71, 0x0F72, 0x0F71, 0x0F74, 0x0FB2, 0x0F80,
	0x0FB3, 0x0F80, 0x0F71, 0x0F80, 0x0F92, 0x0FB7, 0x0F9C, 0x0FB7,
	0x0FA1, 0x0FB7, 0x0FA6, 0x0FB7, 0x0FAB, 0x0FB7, 0x0F90, 0x0FB5,
	0x2ADD, 0x0338, 0xD84A, 0xDC4A, 0xD84A, 0xDC44, 0xD84C, 0xDFD5,
	0xD854, 0xDE49, 0xD857, 0xDCD0, 0xD85F, 0xDED3, 0x05D9, 0x05B4,
	0x05F2, 0x05B7, 0x05E9, 0x05C1, 0x05E9, 0x05C2, KLongD, 0x0106,
	KLongD, 0x01C0, 0x05D0, 0x05B7, 0x05D0, 0x05B8, 0x05D0, 0x05BC,
	0x05D1, 0x05BC, 0x05D2, 0x05BC, 0x05D3, 0x05BC, 0x05D4, 0x05BC,
	0x05D5, 0x05BC, 0x05D6, 0x05BC, 0x05D8, 0x05BC, 0x05D9, 0x05BC,
	0x05DA, 0x05BC, 0x05DB, 0x05BC, 0x05DC, 0x05BC, 0x05DE, 0x05BC,
	0x05E0, 0x05BC, 0x05E1, 0x05BC, 0x05E3, 0x05BC, 0x05E4, 0x05BC,
	0x05E6, 0x05BC, 0x05E7, 0x05BC, 0x05E8, 0x05BC, 0x05E9, 0x05BC,
	0x05EA, 0x05BC, 0x05D5, 0x05B9, 0x05D1, 0x05BF, 0x05DB, 0x05BF,
	0x05E4, 0x05BF, KLongD, 0x1042, KLongD, 0x1030, KLongD, 0x3024,
	KLongD, 0x302A, KLongD, 0x3030, KLongD, 0x3018, KLongD, 0x3006,
	KLongD, 0x101E, KLongD, 0x100C, KLongD, 0x301E, KLongD, 0x3000,
	KLongD, 0x3012, KLongD, 0x300C, 0xD840, 0xDD22, 0xD841, 0xDE3A,
	0xD841, 0xDD1C, 0xD841, 0xDD4B, 0xD864, 0xDDDF, 0xD842, 0xDE2C,
	0xD842, 0xDF63, 0xD845, 0xDCE4, 0xD845, 0xDEA8, 0xD845, 0xDEEA,
	0xD846, 0xDDC8, 0xD846, 0xDF18, 0xD847, 0xDDE4, 0xD847, 0xDDE6,
	0xD848, 0xDD83, 0xD868, 0xDF92, 0xD848, 0xDF31, 0xD848, 0xDF31,
	0xD84C, 0xDEB8, 0xD858, 0xDDDA, 0xD849, 0xDED4, 0xD84A, 0xDF0C,
	0xD84A, 0xDFF1, 0xD84C, 0xDC0A, 0xD84C, 0xDFC3, 0xD84D, 0xDC6D,
	0xD84D, 0xDEA3, 0xD84E, 0xDCA7, 0xD84E, 0xDE8D, 0xD847, 0xDD0B,
	0xD84E, 0xDEFA, 0xD84F, 0xDCBC, 0xD84F, 0xDD1E, 0xD84F, 0xDED1,
	0xD84F, 0xDF5E, 0xD84F, 0xDF8E, 0xD841, 0xDD25, 0xD850, 0xDE63,
	0xD850, 0xDFAB, 0xD851, 0xDE08, 0xD851, 0xDF35, 0xD852, 0xDC14,
	0xD853, 0xDC36, 0xD853, 0xDC92, 0xD848, 0xDD9F, 0xD853, 0xDFA1,
	0xD853, 0xDFB8, 0xD854, 0xDC44, 0xD854, 0xDCF3, 0xD854, 0xDCF2,
	0xD854, 0xDD19, 0xD854, 0xDD33, 0xD855, 0xDC1D, 0xD855, 0xDE26,
	0xD855, 0xDE9A, 0xD855, 0xDEC5, 0xD856, 0xDD7C, 0xD856, 0xDEA7,
	0xD856, 0xDEA7, 0xD856, 0xDFAB, 0xD857, 0xDC80, 0xD857, 0xDF86,
	0xD858, 0xDE28, 0xD858, 0xDE47, 0xD858, 0xDED9, 0xD858, 0xDF3E,
	0xD859, 0xDCDA, 0xD859, 0xDD23, 0xD859, 0xDDA8, 0xD84C, 0xDF5F,
	0xD859, 0xDFA7, 0xD859, 0xDFB5, 0xD84C, 0xDF93, 0xD84C, 0xDF9C,
	0xD85A, 0xDF3C, 0xD85B, 0xDC36, 0xD85B, 0xDD6B, 0xD85B, 0xDCD5,
	0xD85C, 0xDFCA, 0xD85B, 0xDF2C, 0xD85B, 0xDFB1, 0xD85C, 0xDCD2,
	0xD85D, 0xDE67, 0xD85E, 0xDCAE, 0xD85E, 0xDD66, 0xD85F, 0xDCA8,
	0xD85F, 0xDF2F, 0xD842, 0xDC04, 0xD842, 0xDCDE, 0xD861, 0xDDD2,
	0xD861, 0xDDED, 0xD861, 0xDF2E, 0xD862, 0xDFFA, 0xD863, 0xDD77,
	0xD864, 0xDD45, 0xD864, 0xDE1A, 0xD865, 0xDC0A, 0xD865, 0xDC96,
	0xD865, 0xDDB6, 0xD866, 0xDF30, 0xD868, 0xDCCE, 0xD868, 0xDD05,
	0xD868, 0xDE0E, 0xD868, 0xDE91, 0xD869, 0xDE00
	};

// Table of folded decompositions which either have more than one UTF16, or
// their normal decompositions have more than one UTF16
// sizeof/2 = 1183
const unsigned short KNonSingletonFolds[] =
	{
	0x003C, 0x0338, 0x003D, 0x0338, 0x003E, 0x0338, 0x0061, 0x0300,
	0x0061, 0x0301, 0x0061, 0x0302, KLongD, 0x01ED, KLongD, 0x0292,
	KLongD, 0x0157, KLongD, 0x01CF, 0x0061, 0x0303, 0x0061, 0x0304,
	0x0061, 0x0306, KLongD, 0x0136, KLongD, 0x0211, KLongD, 0x00E8,
	KLongD, 0x023E, 0x0061, 0x0307, KLongD, 0x01AE, 0x0061, 0x0308,
	KLongD, 0x0163, 0x0061, 0x0309, 0x0061, 0x030A, 0x0061, 0x030A,
	KLongD, 0x022C, 0x0061, 0x030C, 0x0061, 0x030F, 0x0061, 0x0311,
	0x0061, 0x0323, KLongD, 0x0214, KLongD, 0x00E5, 0x0061, 0x0325,
	0x0061, 0x0328, 0x0062, 0x0307, 0x0062, 0x0323, 0x0062, 0x0331,
	0x0063, 0x0301, 0x0063, 0x0302, 0x0063, 0x0307, 0x0063, 0x030C,
	0x0063, 0x0327, KLongD, 0x01C3, 0x0064, 0x0307, 0x0064, 0x030C,
	0x0064, 0x0323, 0x0064, 0x0327, 0x0064, 0x032D, 0x0064, 0x0331,
	0x0065, 0x0300, 0x0065, 0x0301, 0x0065, 0x0302, KLongD, 0x0217,
	KLongD, 0x019F, KLongD, 0x02DA, KLongD, 0x015D, 0x0065, 0x0303,
	0x0065, 0x0304, KLongD, 0x00E2, KLongD, 0x0271, 0x0065, 0x0306,
	0x0065, 0x0307, 0x0065, 0x0308, 0x0065, 0x0309, 0x0065, 0x030C,
	0x0065, 0x030F, 0x0065, 0x0311, 0x0065, 0x0323, KLongD, 0x00EB,
	0x0065, 0x0327, KLongD, 0x0232, 0x0065, 0x0328, 0x0065, 0x032D,
	0x0065, 0x0330, 0x0066, 0x0307, 0x0067, 0x0301, 0x0067, 0x0302,
	0x0067, 0x0304, 0x0067, 0x0306, 0x0067, 0x0307, 0x0067, 0x030C,
	0x0067, 0x0327, 0x0068, 0x0302, 0x0068, 0x0307, 0x0068, 0x0308,
	0x0068, 0x030C, 0x0068, 0x0323, 0x0068, 0x0327, 0x0068, 0x032E,
	0x0069, 0x0300, 0x0069, 0x0301, 0x0069, 0x0302, 0x0069, 0x0303,
	0x0069, 0x0304, 0x0069, 0x0306, 0x0069, 0xFFFF, 0x0069, 0x0308,
	KLongD, 0x028F, 0x0069, 0x0309, 0x0069, 0x030C, 0x0069, 0x030F,
	0x0069, 0x0311, 0x0069, 0x0323, 0x0069, 0x0328, 0x0069, 0x0330,
	0x006A, 0x0302, 0x006B, 0x0301, 0x006B, 0x030C, 0x006B, 0x0323,
	0x006B, 0x0327, 0x006B, 0x0331, 0x006C, 0x0301, 0x006C, 0x030C,
	0x006C, 0x0323, KLongD, 0x01E1, 0x006C, 0x0327, 0x006C, 0x032D,
	0x006C, 0x0331, 0x006D, 0x0301, 0x006D, 0x0307, 0x006D, 0x0323,
	0x006E, 0x0300, 0x006E, 0x0301, 0x006E, 0x0303, 0x006E, 0x0307,
	0x006E, 0x030C, 0x006E, 0x0323, 0x006E, 0x0327, 0x006E, 0x032D,
	0x006E, 0x0331, 0x006F, 0x0300, 0x006F, 0x0301, 0x006F, 0x0302,
	KLongD, 0x02BF, KLongD, 0x015A, KLongD, 0x0139, KLongD, 0x00D0,
	0x006F, 0x0303, KLongD, 0x018D, KLongD, 0x010F, KLongD, 0x0268,
	0x006F, 0x0304, KLongD, 0x018A, KLongD, 0x016F, 0x006F, 0x0306,
	0x006F, 0x0307, KLongD, 0x00FA, 0x006F, 0x0308, KLongD, 0x0142,
	0x006F, 0x0309, 0x006F, 0x030B, 0x006F, 0x030C, 0x006F, 0x030F,
	0x006F, 0x0311, 0x006F, 0x031B, KLongD, 0x0160, KLongD, 0x0208,
	KLongD, 0x00D9, KLongD, 0x017E, KLongD, 0x013F, 0x006F, 0x0323,
	KLongD, 0x0250, 0x006F, 0x0328, KLongD, 0x00CA, 0x0070, 0x0301,
	0x0070, 0x0307, 0x0072, 0x0301, 0x0072, 0x0307, 0x0072, 0x030C,
	0x0072, 0x030F, 0x0072, 0x0311, 0x0072, 0x0323, KLongD, 0x0229,
	0x0072, 0x0327, 0x0072, 0x0331, 0x0073, 0x0301, KLongD, 0x01E4,
	0x0073, 0x0302, 0x0073, 0x0307, 0x0073, 0x030C, KLongD, 0x01FF,
	0x0073, 0x0323, KLongD, 0x019C, 0x0073, 0x0326, 0x0073, 0x0327,
	0x0074, 0x0307, 0x0074, 0x030C, 0x0074, 0x0323, 0x0074, 0x0326,
	0x0074, 0x0327, 0x0074, 0x032D, 0x0074, 0x0331, 0x0075, 0x0300,
	0x0075, 0x0301, 0x0075, 0x0302, 0x0075, 0x0303, KLongD, 0x026E,
	0x0075, 0x0304, KLongD, 0x02B9, 0x0075, 0x0306, 0x0075, 0x0308,
	KLongD, 0x0262, KLongD, 0x01A5, KLongD, 0x01F6, KLongD, 0x0181,
	0x0075, 0x0309, 0x0075, 0x030A, 0x0075, 0x030B, 0x0075, 0x030C,
	0x0075, 0x030F, 0x0075, 0x0311, 0x0075, 0x031B, KLongD, 0x0253,
	KLongD, 0x02E3, KLongD, 0x00FD, KLongD, 0x0235, KLongD, 0x012D,
	0x0075, 0x0323, 0x0075, 0x0324, 0x0075, 0x0328, 0x0075, 0x032D,
	0x0075, 0x0330, 0x0076, 0x0303, 0x0076, 0x0323, 0x0077, 0x0300,
	0x0077, 0x0301, 0x0077, 0x0302, 0x0077, 0x0307, 0x0077, 0x0308,
	0x0077, 0x0323, 0x0078, 0x0307, 0x0078, 0x0308, 0x0079, 0x0300,
	0x0079, 0x0301, 0x0079, 0x0302, 0x0079, 0x0303, 0x0079, 0x0304,
	0x0079, 0x0307, 0x0079, 0x0308, 0x0079, 0x0309, 0x0079, 0x0323,
	0x007A, 0x0301, 0x007A, 0x0302, 0x007A, 0x0307, 0x007A, 0x030C,
	0x007A, 0x0323, 0x007A, 0x0331, 0x0061, 0x0300, 0x0061, 0x0301,
	0x0061, 0x0302, KLongD, 0x01ED, KLongD, 0x0292, KLongD, 0x0157,
	KLongD, 0x01CF, 0x0061, 0x0303, 0x0061, 0x0304, 0x0061, 0x0306,
	KLongD, 0x0136, KLongD, 0x0211, KLongD, 0x00E8, KLongD, 0x023E,
	0x0061, 0x0307, KLongD, 0x01AE, 0x0061, 0x0308, KLongD, 0x0163,
	0x0061, 0x0309, 0x0061, 0x030A, KLongD, 0x022C, 0x0061, 0x030C,
	0x0061, 0x030F, 0x0061, 0x0311, 0x0061, 0x0323, KLongD, 0x0214,
	KLongD, 0x00E5, 0x0061, 0x0325, 0x0061, 0x0328, 0x0062, 0x0307,
	0x0062, 0x0323, 0x0062, 0x0331, 0x0063, 0x0301, 0x0063, 0x0302,
	0x0063, 0x0307, 0x0063, 0x030C, 0x0063, 0x0327, KLongD, 0x01C3,
	0x0064, 0x0307, 0x0064, 0x030C, 0x0064, 0x0323, 0x0064, 0x0327,
	0x0064, 0x032D, 0x0064, 0x0331, 0x0065, 0x0300, 0x0065, 0x0301,
	0x0065, 0x0302, KLongD, 0x0217, KLongD, 0x019F, KLongD, 0x02DA,
	KLongD, 0x015D, 0x0065, 0x0303, 0x0065, 0x0304, KLongD, 0x00E2,
	KLongD, 0x0271, 0x0065, 0x0306, 0x0065, 0x0307, 0x0065, 0x0308,
	0x0065, 0x0309, 0x0065, 0x030C, 0x0065, 0x030F, 0x0065, 0x0311,
	0x0065, 0x0323, KLongD, 0x00EB, 0x0065, 0x0327, KLongD, 0x0232,
	0x0065, 0x0328, 0x0065, 0x032D, 0x0065, 0x0330, 0x0066, 0x0307,
	0x0067, 0x0301, 0x0067, 0x0302, 0x0067, 0x0304, 0x0067, 0x0306,
	0x0067, 0x0307, 0x0067, 0x030C, 0x0067, 0x0327, 0x0068, 0x0302,
	0x0068, 0x0307, 0x0068, 0x0308, 0x0068, 0x030C, 0x0068, 0x0323,
	0x0068, 0x0327, 0x0068, 0x032E, 0x0068, 0x0331, 0x0069, 0x0300,
	0x0069, 0x0301, 0x0069, 0x0302, 0x0069, 0x0303, 0x0069, 0x0304,
	0x0069, 0x0306, 0x0069, 0x0308, KLongD, 0x028F, 0x0069, 0x0309,
	0x0069, 0x030C, 0x0069, 0x030F, 0x0069, 0x0311, 0x0069, 0x0323,
	0x0069, 0x0328, 0x0069, 0x0330, 0x006A, 0x0302, 0x006A, 0x030C,
	0x006B, 0x0301, 0x006B, 0x030C, 0x006B, 0x0323, 0x006B, 0x0327,
	0x006B, 0x0331, 0x006C, 0x0301, 0x006C, 0x030C, 0x006C, 0x0323,
	KLongD, 0x01E1, 0x006C, 0x0327, 0x006C, 0x032D, 0x006C, 0x0331,
	0x006D, 0x0301, 0x006D, 0x0307, 0x006D, 0x0323, 0x006E, 0x0300,
	0x006E, 0x0301, 0x006E, 0x0303, 0x006E, 0x0307, 0x006E, 0x030C,
	0x006E, 0x0323, 0x006E, 0x0327, 0x006E, 0x032D, 0x006E, 0x0331,
	0x006F, 0x0300, 0x006F, 0x0301, 0x006F, 0x0302, KLongD, 0x02BF,
	KLongD, 0x015A, KLongD, 0x0139, KLongD, 0x00D0, 0x006F, 0x0303,
	KLongD, 0x018D, KLongD, 0x010F, KLongD, 0x0268, 0x006F, 0x0304,
	KLongD, 0x018A, KLongD, 0x016F, 0x006F, 0x0306, 0x006F, 0x0307,
	KLongD, 0x00FA, 0x006F, 0x0308, KLongD, 0x0142, 0x006F, 0x0309,
	0x006F, 0x030B, 0x006F, 0x030C, 0x006F, 0x030F, 0x006F, 0x0311,
	0x006F, 0x031B, KLongD, 0x0160, KLongD, 0x0208, KLongD, 0x00D9,
	KLongD, 0x017E, KLongD, 0x013F, 0x006F, 0x0323, KLongD, 0x0250,
	0x006F, 0x0328, KLongD, 0x00CA, 0x0070, 0x0301, 0x0070, 0x0307,
	0x0072, 0x0301, 0x0072, 0x0307, 0x0072, 0x030C, 0x0072, 0x030F,
	0x0072, 0x0311, 0x0072, 0x0323, KLongD, 0x0229, 0x0072, 0x0327,
	0x0072, 0x0331, 0x0073, 0x0301, KLongD, 0x01E4, 0x0073, 0x0302,
	0x0073, 0x0307, 0x0073, 0x030C, KLongD, 0x01FF, 0x0073, 0x0323,
	KLongD, 0x019C, 0x0073, 0x0326, 0x0073, 0x0327, 0x0074, 0x0307,
	0x0074, 0x0308, 0x0074, 0x030C, 0x0074, 0x0323, 0x0074, 0x0326,
	0x0074, 0x0327, 0x0074, 0x032D, 0x0074, 0x0331, 0x0075, 0x0300,
	0x0075, 0x0301, 0x0075, 0x0302, 0x0075, 0x0303, KLongD, 0x026E,
	0x0075, 0x0304, KLongD, 0x02B9, 0x0075, 0x0306, 0x0075, 0x0308,
	KLongD, 0x0262, KLongD, 0x01A5, KLongD, 0x01F6, KLongD, 0x0181,
	0x0075, 0x0309, 0x0075, 0x030A, 0x0075, 0x030B, 0x0075, 0x030C,
	0x0075, 0x030F, 0x0075, 0x0311, 0x0075, 0x031B, KLongD, 0x0253,
	KLongD, 0x02E3, KLongD, 0x00FD, KLongD, 0x0235, KLongD, 0x012D,
	0x0075, 0x0323, 0x0075, 0x0324, 0x0075, 0x0328, 0x0075, 0x032D,
	0x0075, 0x0330, 0x0076, 0x0303, 0x0076, 0x0323, 0x0077, 0x0300,
	0x0077, 0x0301, 0x0077, 0x0302, 0x0077, 0x0307, 0x0077, 0x0308,
	0x0077, 0x030A, 0x0077, 0x0323, 0x0078, 0x0307, 0x0078, 0x0308,
	0x0079, 0x0300, 0x0079, 0x0301, 0x0079, 0x0302, 0x0079, 0x0303,
	0x0079, 0x0304, 0x0079, 0x0307, 0x0079, 0x0308, 0x0079, 0x0309,
	0x0079, 0x030A, 0x0079, 0x0323, 0x007A, 0x0301, 0x007A, 0x0302,
	0x007A, 0x0307, 0x007A, 0x030C, 0x007A, 0x0323, 0x007A, 0x0331,
	0x00A8, 0x0300, 0x00A8, 0x0301, 0x00A8, 0x0301, 0x00A8, 0x0342,
	0x00B4, 0xFFFF, 0x00E6, 0x0301, 0x00E6, 0x0304, 0x00F8, 0x0301,
	0x00E6, 0x0301, 0x00E6, 0x0304, 0x00F8, 0x0301, 0x0073, 0x0307,
	0x0292, 0x030C, 0x0292, 0x030C, 0x03B1, 0x0300, 0x03B1, 0x0301,
	0x03B1, 0x0301, 0x03B1, 0x0304, 0x03B1, 0x0306, 0x03B1, 0x0313,
	KLongD, 0x0036, KLongD, 0x1036, KLongD, 0x007E, KLongD, 0x107E,
	KLongD, 0x006A, KLongD, 0x106A, KLongD, 0x0127, 0x03B1, 0x0314,
	KLongD, 0x004E, KLongD, 0x104E, KLongD, 0x0086, KLongD, 0x1086,
	KLongD, 0x008E, KLongD, 0x108E, KLongD, 0x01BA, 0x03B1, 0x0345,
	0x03B5, 0x0300, 0x03B5, 0x0301, 0x03B5, 0x0301, 0x03B5, 0x0313,
	KLongD, 0x02D4, KLongD, 0x0172, 0x03B5, 0x0314, KLongD, 0x029E,
	KLongD, 0x011B, 0x03B7, 0x0300, 0x03B7, 0x0301, 0x03B7, 0x0301,
	0x03B7, 0x0313, KLongD, 0x0082, KLongD, 0x1082, KLongD, 0x0092,
	KLongD, 0x1092, KLongD, 0x00C6, KLongD, 0x10C6, KLongD, 0x01F3,
	0x03B7, 0x0314, KLongD, 0x005A, KLongD, 0x105A, KLongD, 0x00B6,
	KLongD, 0x10B6, KLongD, 0x00B2, KLongD, 0x10B2, KLongD, 0x00DC,
	0x03B7, 0x0345, 0x03B9, 0x0300, 0x03B9, 0x0301, 0x03B9, 0x0301,
	0x03B9, 0x0304, 0x03B9, 0x0306, 0x03B9, 0x0308, 0x03B9, 0x0313,
	KLongD, 0x0166, KLongD, 0x02E0, KLongD, 0x00F4, 0x03B9, 0x0314,
	KLongD, 0x00D6, KLongD, 0x0115, KLongD, 0x01EA, 0x03BF, 0x0300,
	0x03BF, 0x0301, 0x03BF, 0x0301, 0x03BF, 0x0313, KLongD, 0x00F7,
	KLongD, 0x0295, 0x03BF, 0x0314, KLongD, 0x01B7, KLongD, 0x0280,
	0x03C1, 0x0314, 0x03C5, 0x0300, 0x03C5, 0x0301, 0x03C5, 0x0301,
	0x03C5, 0x0304, 0x03C5, 0x0306, 0x03C5, 0x0308, 0x03C5, 0x0314,
	KLongD, 0x02A4, KLongD, 0x021D, KLongD, 0x0112, 0x03C9, 0x0300,
	0x03C9, 0x0301, 0x03C9, 0x0301, 0x03C9, 0x0313, KLongD, 0x00AA,
	KLongD, 0x10AA, KLongD, 0x004A, KLongD, 0x104A, KLongD, 0x003E,
	KLongD, 0x103E, KLongD, 0x024A, 0x03C9, 0x0314, KLongD, 0x007A,
	KLongD, 0x107A, KLongD, 0x0096, KLongD, 0x1096, KLongD, 0x00BA,
	KLongD, 0x10BA, KLongD, 0x0199, 0x03C9, 0x0345, 0x03B1, 0x0300,
	KLongD, 0x01C9, 0x03B1, 0x0301, 0x03B1, 0x0301, KLongD, 0x0169,
	0x03B1, 0x0304, 0x03B1, 0x0306, 0x03B1, 0x0313, KLongD, 0x0036,
	KLongD, 0x1036, KLongD, 0x007E, KLongD, 0x107E, KLongD, 0x006A,
	KLongD, 0x106A, KLongD, 0x0127, 0x03B1, 0x0314, KLongD, 0x004E,
	KLongD, 0x104E, KLongD, 0x0086, KLongD, 0x1086, KLongD, 0x008E,
	KLongD, 0x108E, KLongD, 0x01BA, 0x03B1, 0x0342, KLongD, 0x016C,
	0x03B1, 0x0345, 0x03B5, 0x0300, 0x03B5, 0x0301, 0x03B5, 0x0301,
	0x03B5, 0x0313, KLongD, 0x02D4, KLongD, 0x0172, 0x03B5, 0x0314,
	KLongD, 0x029E, KLongD, 0x011B, 0x03B7, 0x0300, KLongD, 0x02AD,
	0x03B7, 0x0301, 0x03B7, 0x0301, KLongD, 0x00DF, 0x03B7, 0x0313,
	KLongD, 0x0082, KLongD, 0x1082, KLongD, 0x0092, KLongD, 0x1092,
	KLongD, 0x00C6, KLongD, 0x10C6, KLongD, 0x01F3, 0x03B7, 0x0314,
	KLongD, 0x005A, KLongD, 0x105A, KLongD, 0x00B6, KLongD, 0x10B6,
	KLongD, 0x00B2, KLongD, 0x10B2, KLongD, 0x00DC, 0x03B7, 0x0342,
	KLongD, 0x0286, 0x03B7, 0x0345, 0x03B9, 0x0300, 0x03B9, 0x0301,
	0x03B9, 0x0301, 0x03B9, 0x0304, 0x03B9, 0x0306, 0x03B9, 0x0308,
	KLongD, 0x02A1, KLongD, 0x0184, KLongD, 0x0184, KLongD, 0x0145,
	0x03B9, 0x0313, KLongD, 0x0166, KLongD, 0x02E0, KLongD, 0x00F4,
	0x03B9, 0x0314, KLongD, 0x00D6, KLongD, 0x0115, KLongD, 0x01EA,
	0x03B9, 0x0342, 0x03BF, 0x0300, 0x03BF, 0x0301, 0x03BF, 0x0301,
	0x03BF, 0x0313, KLongD, 0x00F7, KLongD, 0x0295, 0x03BF, 0x0314,
	KLongD, 0x01B7, KLongD, 0x0280, 0x03C1, 0x0313, 0x03C1, 0x0314,
	0x03C5, 0x0300, 0x03C5, 0x0301, 0x03C5, 0x0301, 0x03C5, 0x0304,
	0x03C5, 0x0306, 0x03C5, 0x0308, KLongD, 0x01B1, KLongD, 0x0100,
	KLongD, 0x0100, KLongD, 0x02E9, 0x03C5, 0x0313, KLongD, 0x01F9,
	KLongD, 0x02C8, KLongD, 0x02CB, 0x03C5, 0x0314, KLongD, 0x02A4,
	KLongD, 0x021D, KLongD, 0x0112, 0x03C5, 0x0342, 0x03C9, 0x0300,
	KLongD, 0x022F, 0x03C9, 0x0301, 0x03C9, 0x0301, KLongD, 0x0247,
	0x03C9, 0x0313, KLongD, 0x00AA, KLongD, 0x10AA, KLongD, 0x004A,
	KLongD, 0x104A, KLongD, 0x003E, KLongD, 0x103E, KLongD, 0x024A,
	0x03C9, 0x0314, KLongD, 0x007A, KLongD, 0x107A, KLongD, 0x0096,
	KLongD, 0x1096, KLongD, 0x00BA, KLongD, 0x10BA, KLongD, 0x0199,
	0x03C9, 0x0342, KLongD, 0x025F, 0x03C9, 0x0345, 0x03D2, 0x0301,
	0x03D2, 0x0308, 0x0456, 0x0308, 0x0430, 0x0306, 0x0430, 0x0308,
	0x0433, 0x0301, 0x0435, 0x0300, 0x0435, 0x0306, 0x0435, 0x0308,
	0x0436, 0x0306, 0x0436, 0x0308, 0x0437, 0x0308, 0x0438, 0x0300,
	0x0438, 0x0304, 0x0438, 0x0306, 0x0438, 0x0308, 0x043A, 0x0301,
	0x043E, 0x0308, 0x0443, 0x0304, 0x0443, 0x0306, 0x0443, 0x0308,
	0x0443, 0x030B, 0x0447, 0x0308, 0x044B, 0x0308, 0x044D, 0x0308,
	0x0430, 0x0306, 0x0430, 0x0308, 0x0433, 0x0301, 0x0435, 0x0300,
	0x0435, 0x0306, 0x0435, 0x0308, 0x0436, 0x0306, 0x0436, 0x0308,
	0x0437, 0x0308, 0x0438, 0x0300, 0x0438, 0x0304, 0x0438, 0x0306,
	0x0438, 0x0308, 0x043A, 0x0301, 0x043E, 0x0308, 0x0443, 0x0304,
	0x0443, 0x0306, 0x0443, 0x0308, 0x0443, 0x030B, 0x0447, 0x0308,
	0x044B, 0x0308, 0x044D, 0x0308, 0x0456, 0x0308, 0x0475, 0x030F,
	0x0475, 0x030F, 0x04D9, 0x0308, 0x04D9, 0x0308, 0x04E9, 0x0308,
	0x04E9, 0x0308, 0x0627, 0x0653, 0x0627, 0x0654, 0x0627, 0x0655,
	0x0648, 0x0654, 0x064A, 0x0654, 0x06C1, 0x0654, 0x06D2, 0x0654,
	0x06D5, 0x0654, 0x0928, 0x093C, 0x0930, 0x093C, 0x0933, 0x093C,
	0x09C7, 0x09BE, 0x09C7, 0x09D7, 0x0B47, 0x0B3E, 0x0B47, 0x0B56,
	0x0B47, 0x0B57, 0x0B92, 0x0BD7, 0x0BC6, 0x0BBE, 0x0BC6, 0x0BD7,
	0x0BC7, 0x0BBE, 0x0C46, 0x0C56, 0x0CBF, 0x0CD5, 0x0CC6, 0x0CC2,
	KLongD, 0x02B6, 0x0CC6, 0x0CD5, 0x0CC6, 0x0CD6, 0x0D46, 0x0D3E,
	0x0D46, 0x0D57, 0x0D47, 0x0D3E, 0x0DD9, 0x0DCA, 0x0DD9, 0x0DCF,
	KLongD, 0x0154, 0x0DD9, 0x0DDF, 0x1025, 0x102E, 0x1B05, 0x1B35,
	0x1B07, 0x1B35, 0x1B09, 0x1B35, 0x1B0B, 0x1B35, 0x1B0D, 0x1B35,
	0x1B11, 0x1B35, 0x1B3A, 0x1B35, 0x1B3C, 0x1B35, 0x1B3E, 0x1B35,
	0x1B3F, 0x1B35, 0x1B42, 0x1B35, 0x1FBF, 0x0300, 0x1FBF, 0x0301,
	0x1FBF, 0x0342, 0x1FFE, 0x0300, 0x1FFE, 0x0301, 0x1FFE, 0x0342,
	0x2190, 0x0338, 0x2192, 0x0338, 0x2194, 0x0338, 0x21D0, 0x0338,
	0x21D2, 0x0338, 0x21D4, 0x0338, 0x2203, 0x0338, 0x2208, 0x0338,
	0x220B, 0x0338, 0x2223, 0x0338, 0x2225, 0x0338, 0x223C, 0x0338,
	0x2243, 0x0338, 0x2245, 0x0338, 0x2248, 0x0338, 0x224D, 0x0338,
	0x2261, 0x0338, 0x2264, 0x0338, 0x2265, 0x0338, 0x2272, 0x0338,
	0x2273, 0x0338, 0x2276, 0x0338, 0x2277, 0x0338, 0x227A, 0x0338,
	0x227B, 0x0338, 0x227C, 0x0338, 0x227D, 0x0338, 0x2282, 0x0338,
	0x2283, 0x0338, 0x2286, 0x0338, 0x2287, 0x0338, 0x2291, 0x0338,
	0x2292, 0x0338, 0x22A2, 0x0338, 0x22A8, 0x0338, 0x22A9, 0x0338,
	0x22AB, 0x0338, 0x22B2, 0x0338, 0x22B3, 0x0338, 0x22B4, 0x0338,
	0x22B5, 0x0338, 0x3046, 0x3099, 0x304B, 0x3099, 0x304D, 0x3099,
	0x304F, 0x3099, 0x3051, 0x3099, 0x3053, 0x3099, 0x3055, 0x3099,
	0x3057, 0x3099, 0x3059, 0x3099, 0x305B, 0x3099, 0x305D, 0x3099,
	0x305F, 0x3099, 0x3061, 0x3099, 0x3064, 0x3099, 0x3066, 0x3099,
	0x3068, 0x3099, 0x306F, 0x3099, 0x306F, 0x309A, 0x3072, 0x3099,
	0x3072, 0x309A, 0x3075, 0x3099, 0x3075, 0x309A, 0x3078, 0x3099,
	0x3078, 0x309A, 0x307B, 0x3099, 0x307B, 0x309A, 0x309D, 0x3099,
	0x30A6, 0x3099, 0x30AB, 0x3099, 0x30AD, 0x3099, 0x30AF, 0x3099,
	0x30B1, 0x3099, 0x30B3, 0x3099, 0x30B5, 0x3099, 0x30B7, 0x3099,
	0x30B9, 0x3099, 0x30BB, 0x3099, 0x30BD, 0x3099, 0x30BF, 0x3099,
	0x30C1, 0x3099, 0x30C4, 0x3099, 0x30C6, 0x3099, 0x30C8, 0x3099,
	0x30CF, 0x3099, 0x30CF, 0x309A, 0x30D2, 0x3099, 0x30D2, 0x309A,
	0x30D5, 0x3099, 0x30D5, 0x309A, 0x30D8, 0x3099, 0x30D8, 0x309A,
	0x30DB, 0x3099, 0x30DB, 0x309A, 0x30EF, 0x3099, 0x30F0, 0x3099,
	0x30F1, 0x3099, 0x30F2, 0x3099, 0x30FD, 0x3099, 0x0308, 0x0301,
	0x0915, 0x093C, 0x0916, 0x093C, 0x0917, 0x093C, 0x091C, 0x093C,
	0x0921, 0x093C, 0x0922, 0x093C, 0x092B, 0x093C, 0x092F, 0x093C,
	0x09A1, 0x09BC, 0x09A2, 0x09BC, 0x09AF, 0x09BC, 0x0A32, 0x0A3C,
	0x0A38, 0x0A3C, 0x0A16, 0x0A3C, 0x0A17, 0x0A3C, 0x0A1C, 0x0A3C,
	0x0A2B, 0x0A3C, 0x0B21, 0x0B3C, 0x0B22, 0x0B3C, 0x0F42, 0x0FB7,
	0x0F4C, 0x0FB7, 0x0F51, 0x0FB7, 0x0F56, 0x0FB7, 0x0F5B, 0x0FB7,
	0x0F40, 0x0FB5, 0x0F71, 0x0F72, 0x0F71, 0x0F74, 0x0FB2, 0x0F80,
	0x0FB3, 0x0F80, 0x0F71, 0x0F80, 0x0F92, 0x0FB7, 0x0F9C, 0x0FB7,
	0x0FA1, 0x0FB7, 0x0FA6, 0x0FB7, 0x0FAB, 0x0FB7, 0x0F90, 0x0FB5,
	0x2ADD, 0x0338, 0xD84A, 0xDC4A, 0xD84A, 0xDC44, 0xD84C, 0xDFD5,
	0xD854, 0xDE49, 0xD857, 0xDCD0, 0xD85F, 0xDED3, 0x05D9, 0x05B4,
	0x05F2, 0x05B7, 0x05E9, 0x05C1, 0x05E9, 0x05C2, KLongD, 0x0106,
	KLongD, 0x01C0, 0x05D0, 0x05B7, 0x05D0, 0x05B8, 0x05D0, 0x05BC,
	0x05D1, 0x05BC, 0x05D2, 0x05BC, 0x05D3, 0x05BC, 0x05D4, 0x05BC,
	0x05D5, 0x05BC, 0x05D6, 0x05BC, 0x05D8, 0x05BC, 0x05D9, 0x05BC,
	0x05DA, 0x05BC, 0x05DB, 0x05BC, 0x05DC, 0x05BC, 0x05DE, 0x05BC,
	0x05E0, 0x05BC, 0x05E1, 0x05BC, 0x05E3, 0x05BC, 0x05E4, 0x05BC,
	0x05E6, 0x05BC, 0x05E7, 0x05BC, 0x05E8, 0x05BC, 0x05E9, 0x05BC,
	0x05EA, 0x05BC, 0x05D5, 0x05B9, 0x05D1, 0x05BF, 0x05DB, 0x05BF,
	0x05E4, 0x05BF, KLongD, 0x1042, KLongD, 0x1030, KLongD, 0x3024,
	KLongD, 0x302A, KLongD, 0x3030, KLongD, 0x3018, KLongD, 0x3006,
	KLongD, 0x101E, KLongD, 0x100C, KLongD, 0x301E, KLongD, 0x3000,
	KLongD, 0x3012, KLongD, 0x300C, 0xD840, 0xDD22, 0xD841, 0xDE3A,
	0xD841, 0xDD1C, 0xD841, 0xDD4B, 0xD864, 0xDDDF, 0xD842, 0xDE2C,
	0xD842, 0xDF63, 0xD845, 0xDCE4, 0xD845, 0xDEA8, 0xD845, 0xDEEA,
	0xD846, 0xDDC8, 0xD846, 0xDF18, 0xD847, 0xDDE4, 0xD847, 0xDDE6,
	0xD848, 0xDD83, 0xD868, 0xDF92, 0xD848, 0xDF31, 0xD848, 0xDF31,
	0xD84C, 0xDEB8, 0xD858, 0xDDDA, 0xD849, 0xDED4, 0xD84A, 0xDF0C,
	0xD84A, 0xDFF1, 0xD84C, 0xDC0A, 0xD84C, 0xDFC3, 0xD84D, 0xDC6D,
	0xD84D, 0xDEA3, 0xD84E, 0xDCA7, 0xD84E, 0xDE8D, 0xD847, 0xDD0B,
	0xD84E, 0xDEFA, 0xD84F, 0xDCBC, 0xD84F, 0xDD1E, 0xD84F, 0xDED1,
	0xD84F, 0xDF5E, 0xD84F, 0xDF8E, 0xD841, 0xDD25, 0xD850, 0xDE63,
	0xD850, 0xDFAB, 0xD851, 0xDE08, 0xD851, 0xDF35, 0xD852, 0xDC14,
	0xD853, 0xDC36, 0xD853, 0xDC92, 0xD848, 0xDD9F, 0xD853, 0xDFA1,
	0xD853, 0xDFB8, 0xD854, 0xDC44, 0xD854, 0xDCF3, 0xD854, 0xDCF2,
	0xD854, 0xDD19, 0xD854, 0xDD33, 0xD855, 0xDC1D, 0xD855, 0xDE26,
	0xD855, 0xDE9A, 0xD855, 0xDEC5, 0xD856, 0xDD7C, 0xD856, 0xDEA7,
	0xD856, 0xDEA7, 0xD856, 0xDFAB, 0xD857, 0xDC80, 0xD857, 0xDF86,
	0xD858, 0xDE28, 0xD858, 0xDE47, 0xD858, 0xDED9, 0xD858, 0xDF3E,
	0xD859, 0xDCDA, 0xD859, 0xDD23, 0xD859, 0xDDA8, 0xD84C, 0xDF5F,
	0xD859, 0xDFA7, 0xD859, 0xDFB5, 0xD84C, 0xDF93, 0xD84C, 0xDF9C,
	0xD85A, 0xDF3C, 0xD85B, 0xDC36, 0xD85B, 0xDD6B, 0xD85B, 0xDCD5,
	0xD85C, 0xDFCA, 0xD85B, 0xDF2C, 0xD85B, 0xDFB1, 0xD85C, 0xDCD2,
	0xD85D, 0xDE67, 0xD85E, 0xDCAE, 0xD85E, 0xDD66, 0xD85F, 0xDCA8,
	0xD85F, 0xDF2F, 0xD842, 0xDC04, 0xD842, 0xDCDE, 0xD861, 0xDDD2,
	0xD861, 0xDDED, 0xD861, 0xDF2E, 0xD862, 0xDFFA, 0xD863, 0xDD77,
	0xD864, 0xDD45, 0xD864, 0xDE1A, 0xD865, 0xDC0A, 0xD865, 0xDC96,
	0xD865, 0xDDB6, 0xD866, 0xDF30, 0xD868, 0xDCCE, 0xD868, 0xDD05,
	0xD868, 0xDE0E, 0xD868, 0xDE91, 0xD869, 0xDE00, 0xD801, 0xDC28,
	0xD801, 0xDC29, 0xD801, 0xDC2A, 0xD801, 0xDC2B, 0xD801, 0xDC2C,
	0xD801, 0xDC2D, 0xD801, 0xDC2E, 0xD801, 0xDC2F, 0xD801, 0xDC30,
	0xD801, 0xDC31, 0xD801, 0xDC32, 0xD801, 0xDC33, 0xD801, 0xDC34,
	0xD801, 0xDC35, 0xD801, 0xDC36, 0xD801, 0xDC37, 0xD801, 0xDC38,
	0xD801, 0xDC39, 0xD801, 0xDC3A, 0xD801, 0xDC3B, 0xD801, 0xDC3C,
	0xD801, 0xDC3D, 0xD801, 0xDC3E, 0xD801, 0xDC3F, 0xD801, 0xDC40,
	0xD801, 0xDC41, 0xD801, 0xDC42, 0xD801, 0xDC43, 0xD801, 0xDC44,
	0xD801, 0xDC45, 0xD801, 0xDC46, 0xD801, 0xDC47, 0xD801, 0xDC48,
	0xD801, 0xDC49, 0xD801, 0xDC4A, 0xD801, 0xDC4B, 0xD801, 0xDC4C,
	0xD801, 0xDC4D, 0xD801, 0xDC4E, 0xD801, 0xDC4F
	};

// Table of singleton decompositions and characters with singleton folds
// Note for Unicode 5.0:
// Unicode 5.0 contains some non-BMP characters have non-BMP "singleton" folds.
// As per the algorithm of this file, the non-BMP character should be stored in 
// this table. "Unsigned short" is not big enough to hold them. However, this 
// "character" information is not useful. So we just store 0xFFFF instead. 
// Please do check 0xFFFF when access this table. If meet 0xFFFF, that means 
// your character has no decomposition.
// See the variable "ShortDecompsLongFolds" in FoldAndDecompTables.pl if you 
// want to know more.
// sizeof = 940
const unsigned short KSingletonDecompositions[] =
	{
	0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF, 0xFFFF,
	0x0300, 0x0301, 0x0313, 0x02B9, 0x003B, 0x00B7, 0x03B9, 0x0060,
	0x2002, 0x2003, 0x03A9, 0x004B, 0x3008, 0x3009, 0x8C48, 0x66F4,
	0x8ECA, 0x8CC8, 0x6ED1, 0x4E32, 0x53E5, 0x9F9C, 0x9F9C, 0x5951,
	0x91D1, 0x5587, 0x5948, 0x61F6, 0x7669, 0x7F85, 0x863F, 0x87BA,
	0x88F8, 0x908F, 0x6A02, 0x6D1B, 0x70D9, 0x73DE, 0x843D, 0x916A,
	0x99F1, 0x4E82, 0x5375, 0x6B04, 0x721B, 0x862D, 0x9E1E, 0x5D50,
	0x6FEB, 0x85CD, 0x8964, 0x62C9, 0x81D8, 0x881F, 0x5ECA, 0x6717,
	0x6D6A, 0x72FC, 0x90CE, 0x4F86, 0x51B7, 0x52DE, 0x64C4, 0x6AD3,
	0x7210, 0x76E7, 0x8001, 0x8606, 0x865C, 0x8DEF, 0x9732, 0x9B6F,
	0x9DFA, 0x788C, 0x797F, 0x7DA0, 0x83C9, 0x9304, 0x9E7F, 0x8AD6,
	0x58DF, 0x5F04, 0x7C60, 0x807E, 0x7262, 0x78CA, 0x8CC2, 0x96F7,
	0x58D8, 0x5C62, 0x6A13, 0x6DDA, 0x6F0F, 0x7D2F, 0x7E37, 0x964B,
	0x52D2, 0x808B, 0x51DC, 0x51CC, 0x7A1C, 0x7DBE, 0x83F1, 0x9675,
	0x8B80, 0x62CF, 0x6A02, 0x8AFE, 0x4E39, 0x5BE7, 0x6012, 0x7387,
	0x7570, 0x5317, 0x78FB, 0x4FBF, 0x5FA9, 0x4E0D, 0x6CCC, 0x6578,
	0x7D22, 0x53C3, 0x585E, 0x7701, 0x8449, 0x8AAA, 0x6BBA, 0x8FB0,
	0x6C88, 0x62FE, 0x82E5, 0x63A0, 0x7565, 0x4EAE, 0x5169, 0x51C9,
	0x6881, 0x7CE7, 0x826F, 0x8AD2, 0x91CF, 0x52F5, 0x5442, 0x5973,
	0x5EEC, 0x65C5, 0x6FFE, 0x792A, 0x95AD, 0x9A6A, 0x9E97, 0x9ECE,
	0x529B, 0x66C6, 0x6B77, 0x8F62, 0x5E74, 0x6190, 0x6200, 0x649A,
	0x6F23, 0x7149, 0x7489, 0x79CA, 0x7DF4, 0x806F, 0x8F26, 0x84EE,
	0x9023, 0x934A, 0x5217, 0x52A3, 0x54BD, 0x70C8, 0x88C2, 0x8AAA,
	0x5EC9, 0x5FF5, 0x637B, 0x6BAE, 0x7C3E, 0x7375, 0x4EE4, 0x56F9,
	0x5BE7, 0x5DBA, 0x601C, 0x73B2, 0x7469, 0x7F9A, 0x8046, 0x9234,
	0x96F6, 0x9748, 0x9818, 0x4F8B, 0x79AE, 0x91B4, 0x96B8, 0x60E1,
	0x4E86, 0x50DA, 0x5BEE, 0x5C3F, 0x6599, 0x6A02, 0x71CE, 0x7642,
	0x84FC, 0x907C, 0x9F8D, 0x6688, 0x962E, 0x5289, 0x677B, 0x67F3,
	0x6D41, 0x6E9C, 0x7409, 0x7559, 0x786B, 0x7D10, 0x985E, 0x516D,
	0x622E, 0x9678, 0x502B, 0x5D19, 0x6DEA, 0x8F2A, 0x5F8B, 0x6144,
	0x6817, 0x7387, 0x9686, 0x5229, 0x540F, 0x5C65, 0x6613, 0x674E,
	0x68A8, 0x6CE5, 0x7406, 0x75E2, 0x7F79, 0x88CF, 0x88E1, 0x91CC,
	0x96E2, 0x533F, 0x6EBA, 0x541D, 0x71D0, 0x7498, 0x85FA, 0x96A3,
	0x9C57, 0x9E9F, 0x6797, 0x6DCB, 0x81E8, 0x7ACB, 0x7B20, 0x7C92,
	0x72C0, 0x7099, 0x8B58, 0x4EC0, 0x8336, 0x523A, 0x5207, 0x5EA6,
	0x62D3, 0x7CD6, 0x5B85, 0x6D1E, 0x66B4, 0x8F3B, 0x884C, 0x964D,
	0x898B, 0x5ED3, 0x5140, 0x55C0, 0x585A, 0x6674, 0x51DE, 0x732A,
	0x76CA, 0x793C, 0x795E, 0x7965, 0x798F, 0x9756, 0x7CBE, 0x7FBD,
	0x8612, 0x8AF8, 0x9038, 0x90FD, 0x98EF, 0x98FC, 0x9928, 0x9DB4,
	0x4FAE, 0x50E7, 0x514D, 0x52C9, 0x52E4, 0x5351, 0x559D, 0x5606,
	0x5668, 0x5840, 0x58A8, 0x5C64, 0x5C6E, 0x6094, 0x6168, 0x618E,
	0x61F2, 0x654F, 0x65E2, 0x6691, 0x6885, 0x6D77, 0x6E1A, 0x6F22,
	0x716E, 0x722B, 0x7422, 0x7891, 0x793E, 0x7949, 0x7948, 0x7950,
	0x7956, 0x795D, 0x798D, 0x798E, 0x7A40, 0x7A81, 0x7BC0, 0x7DF4,
	0x7E09, 0x7E41, 0x7F72, 0x8005, 0x81ED, 0x8279, 0x8279, 0x8457,
	0x8910, 0x8996, 0x8B01, 0x8B39, 0x8CD3, 0x8D08, 0x8FB6, 0x9038,
	0x96E3, 0x97FF, 0x983B, 0x4E26, 0x51B5, 0x5168, 0x4F80, 0x5145,
	0x5180, 0x52C7, 0x52FA, 0x559D, 0x5555, 0x5599, 0x55E2, 0x585A,
	0x58B3, 0x5944, 0x5954, 0x5A62, 0x5B28, 0x5ED2, 0x5ED9, 0x5F69,
	0x5FAD, 0x60D8, 0x614E, 0x6108, 0x618E, 0x6160, 0x61F2, 0x6234,
	0x63C4, 0x641C, 0x6452, 0x6556, 0x6674, 0x6717, 0x671B, 0x6756,
	0x6B79, 0x6BBA, 0x6D41, 0x6EDB, 0x6ECB, 0x6F22, 0x701E, 0x716E,
	0x77A7, 0x7235, 0x72AF, 0x732A, 0x7471, 0x7506, 0x753B, 0x761D,
	0x761F, 0x76CA, 0x76DB, 0x76F4, 0x774A, 0x7740, 0x78CC, 0x7AB1,
	0x7BC0, 0x7C7B, 0x7D5B, 0x7DF4, 0x7F3E, 0x8005, 0x8352, 0x83EF,
	0x8779, 0x8941, 0x8986, 0x8996, 0x8ABF, 0x8AF8, 0x8ACB, 0x8B01,
	0x8AFE, 0x8AED, 0x8B39, 0x8B8A, 0x8D08, 0x8F38, 0x9072, 0x9199,
	0x9276, 0x967C, 0x96E3, 0x9756, 0x97DB, 0x97FF, 0x980B, 0x983B,
	0x9B12, 0x9F9C, 0x3B9D, 0x4018, 0x4039, 0x9F43, 0x9F8E, 0x4E3D,
	0x4E38, 0x4E41, 0x4F60, 0x4FAE, 0x4FBB, 0x5002, 0x507A, 0x5099,
	0x50E7, 0x50CF, 0x349E, 0x514D, 0x5154, 0x5164, 0x5177, 0x34B9,
	0x5167, 0x518D, 0x5197, 0x51A4, 0x4ECC, 0x51AC, 0x51B5, 0x51F5,
	0x5203, 0x34DF, 0x523B, 0x5246, 0x5272, 0x5277, 0x3515, 0x52C7,
	0x52C9, 0x52E4, 0x52FA, 0x5305, 0x5306, 0x5317, 0x5349, 0x5351,
	0x535A, 0x5373, 0x537D, 0x537F, 0x537F, 0x537F, 0x7070, 0x53CA,
	0x53DF, 0x53EB, 0x53F1, 0x5406, 0x549E, 0x5438, 0x5448, 0x5468,
	0x54A2, 0x54F6, 0x5510, 0x5553, 0x5563, 0x5584, 0x5584, 0x5599,
	0x55AB, 0x55B3, 0x55C2, 0x5716, 0x5606, 0x5717, 0x5651, 0x5674,
	0x5207, 0x58EE, 0x57CE, 0x57F4, 0x580D, 0x578B, 0x5832, 0x5831,
	0x58AC, 0x58F2, 0x58F7, 0x5906, 0x591A, 0x5922, 0x5962, 0x59EC,
	0x5A1B, 0x5A27, 0x59D8, 0x5A66, 0x36EE, 0x36FC, 0x5B08, 0x5B3E,
	0x5B3E, 0x5BC3, 0x5BD8, 0x5BE7, 0x5BF3, 0x5BFF, 0x5C06, 0x5F53,
	0x5C22, 0x3781, 0x5C60, 0x5C6E, 0x5CC0, 0x5C8D, 0x5D43, 0x5D6E,
	0x5D6B, 0x5D7C, 0x5DE1, 0x5DE2, 0x382F, 0x5DFD, 0x5E28, 0x5E3D,
	0x5E69, 0x3862, 0x387C, 0x5EB0, 0x5EB3, 0x5EB6, 0x5ECA, 0x5EFE,
	0x8201, 0x5F22, 0x5F22, 0x38C7, 0x5F62, 0x5F6B, 0x38E3, 0x5F9A,
	0x5FCD, 0x5FD7, 0x5FF9, 0x6081, 0x393A, 0x391C, 0x6094, 0x60C7,
	0x6148, 0x614C, 0x614E, 0x614C, 0x617A, 0x618E, 0x61B2, 0x61A4,
	0x61AF, 0x61DE, 0x61F2, 0x61F6, 0x6210, 0x621B, 0x625D, 0x62B1,
	0x62D4, 0x6350, 0x633D, 0x62FC, 0x6368, 0x6383, 0x63E4, 0x6422,
	0x63C5, 0x63A9, 0x3A2E, 0x6469, 0x647E, 0x649D, 0x6477, 0x3A6C,
	0x654F, 0x656C, 0x65E3, 0x66F8, 0x6649, 0x3B19, 0x6691, 0x3B08,
	0x3AE4, 0x5192, 0x5195, 0x6700, 0x669C, 0x80AD, 0x43D9, 0x6717,
	0x671B, 0x6721, 0x675E, 0x6753, 0x3B49, 0x67FA, 0x6785, 0x6852,
	0x6885, 0x688E, 0x681F, 0x6914, 0x3B9D, 0x6942, 0x69A3, 0x69EA,
	0x6AA8, 0x6ADB, 0x3C18, 0x6B21, 0x6B54, 0x3C4E, 0x6B72, 0x6B9F,
	0x6BBA, 0x6BBB, 0x6C4E, 0x6CBF, 0x6CCD, 0x6C67, 0x6D16, 0x6D3E,
	0x6D77, 0x6D41, 0x6D69, 0x6D78, 0x6D85, 0x6D34, 0x6E2F, 0x6E6E,
	0x3D33, 0x6ECB, 0x6EC7, 0x6DF9, 0x6F6E, 0x6FC6, 0x7039, 0x701E,
	0x701B, 0x3D96, 0x704A, 0x707D, 0x7077, 0x70AD, 0x7145, 0x719C,
	0x7228, 0x7235, 0x7250, 0x7280, 0x7295, 0x737A, 0x738B, 0x3EAC,
	0x73A5, 0x3EB8, 0x3EB8, 0x7447, 0x745C, 0x7471, 0x7485, 0x74CA,
	0x3F1B, 0x7524, 0x753E, 0x7570, 0x7610, 0x3FFC, 0x4008, 0x76F4,
	0x771E, 0x771F, 0x771F, 0x774A, 0x4039, 0x778B, 0x4046, 0x4096,
	0x784E, 0x788C, 0x78CC, 0x40E3, 0x7956, 0x798F, 0x79EB, 0x412F,
	0x7A40, 0x7A4A, 0x7A4F, 0x7AEE, 0x4202, 0x7BC6, 0x7BC9, 0x4227,
	0x7CD2, 0x42A0, 0x7CE8, 0x7CE3, 0x7D00, 0x7D63, 0x4301, 0x7DC7,
	0x7E02, 0x7E45, 0x4334, 0x4359, 0x7F7A, 0x7F95, 0x7FFA, 0x8005,
	0x8060, 0x8070, 0x43D5, 0x80B2, 0x8103, 0x440B, 0x813E, 0x5AB5,
	0x8201, 0x8204, 0x8F9E, 0x446B, 0x8291, 0x828B, 0x829D, 0x52B3,
	0x82B1, 0x82B3, 0x82BD, 0x82E6, 0x82E5, 0x831D, 0x8363, 0x83AD,
	0x8323, 0x83BD, 0x83E7, 0x8457, 0x8353, 0x83CA, 0x83CC, 0x83DC,
	0x452B, 0x84F1, 0x84F3, 0x8516, 0x8564, 0x455D, 0x4561, 0x456B,
	0x8650, 0x865C, 0x8667, 0x8669, 0x86A9, 0x8688, 0x870E, 0x86E2,
	0x8779, 0x8728, 0x876B, 0x8786, 0x45D7, 0x87E1, 0x8801, 0x45F9,
	0x8860, 0x8863, 0x88D7, 0x88DE, 0x4635, 0x88FA, 0x34BB, 0x46BE,
	0x46C7, 0x8AA0, 0x8AED, 0x8B8A, 0x8C55, 0x8CAB, 0x8CC1, 0x8D1B,
	0x8D77, 0x8DCB, 0x8DBC, 0x8DF0, 0x8ED4, 0x8F38, 0x9094, 0x90F1,
	0x9111, 0x911B, 0x9238, 0x92D7, 0x92D8, 0x927C, 0x93F9, 0x9415,
	0x958B, 0x4995, 0x95B7, 0x49E6, 0x96C3, 0x5DB2, 0x9723, 0x4A6E,
	0x4A76, 0x97E0, 0x4AB2, 0x980B, 0x980B, 0x9829, 0x98E2, 0x4B33,
	0x9929, 0x99A7, 0x99C2, 0x99FE, 0x4BCE, 0x9B12, 0x9C40, 0x9CFD,
	0x4CCE, 0x4CED, 0x9D67, 0x4CF8, 0x9EBB, 0x4D56, 0x9EF9, 0x9EFE,
	0x9F05, 0x9F0F, 0x9F16, 0x9F3B
	};

// Table of singleton folds
// sizeof = 1414
const unsigned short KSingletonFolds[] =
	{
	0x0300, 0x0301, 0x0313, 0x02B9, 0x003B, 0x00B7, 0x03B9, 0x0060,
	0x2002, 0x2003, 0x03C9, 0x006B, 0x3008, 0x3009, 0x8C48, 0x66F4,
	0x8ECA, 0x8CC8, 0x6ED1, 0x4E32, 0x53E5, 0x9F9C, 0x9F9C, 0x5951,
	0x91D1, 0x5587, 0x5948, 0x61F6, 0x7669, 0x7F85, 0x863F, 0x87BA,
	0x88F8, 0x908F, 0x6A02, 0x6D1B, 0x70D9, 0x73DE, 0x843D, 0x916A,
	0x99F1, 0x4E82, 0x5375, 0x6B04, 0x721B, 0x862D, 0x9E1E, 0x5D50,
	0x6FEB, 0x85CD, 0x8964, 0x62C9, 0x81D8, 0x881F, 0x5ECA, 0x6717,
	0x6D6A, 0x72FC, 0x90CE, 0x4F86, 0x51B7, 0x52DE, 0x64C4, 0x6AD3,
	0x7210, 0x76E7, 0x8001, 0x8606, 0x865C, 0x8DEF, 0x9732, 0x9B6F,
	0x9DFA, 0x788C, 0x797F, 0x7DA0, 0x83C9, 0x9304, 0x9E7F, 0x8AD6,
	0x58DF, 0x5F04, 0x7C60, 0x807E, 0x7262, 0x78CA, 0x8CC2, 0x96F7,
	0x58D8, 0x5C62, 0x6A13, 0x6DDA, 0x6F0F, 0x7D2F, 0x7E37, 0x964B,
	0x52D2, 0x808B, 0x51DC, 0x51CC, 0x7A1C, 0x7DBE, 0x83F1, 0x9675,
	0x8B80, 0x62CF, 0x6A02, 0x8AFE, 0x4E39, 0x5BE7, 0x6012, 0x7387,
	0x7570, 0x5317, 0x78FB, 0x4FBF, 0x5FA9, 0x4E0D, 0x6CCC, 0x6578,
	0x7D22, 0x53C3, 0x585E, 0x7701, 0x8449, 0x8AAA, 0x6BBA, 0x8FB0,
	0x6C88, 0x62FE, 0x82E5, 0x63A0, 0x7565, 0x4EAE, 0x5169, 0x51C9,
	0x6881, 0x7CE7, 0x826F, 0x8AD2, 0x91CF, 0x52F5, 0x5442, 0x5973,
	0x5EEC, 0x65C5, 0x6FFE, 0x792A, 0x95AD, 0x9A6A, 0x9E97, 0x9ECE,
	0x529B, 0x66C6, 0x6B77, 0x8F62, 0x5E74, 0x6190, 0x6200, 0x649A,
	0x6F23, 0x7149, 0x7489, 0x79CA, 0x7DF4, 0x806F, 0x8F26, 0x84EE,
	0x9023, 0x934A, 0x5217, 0x52A3, 0x54BD, 0x70C8, 0x88C2, 0x8AAA,
	0x5EC9, 0x5FF5, 0x637B, 0x6BAE, 0x7C3E, 0x7375, 0x4EE4, 0x56F9,
	0x5BE7, 0x5DBA, 0x601C, 0x73B2, 0x7469, 0x7F9A, 0x8046, 0x9234,
	0x96F6, 0x9748, 0x9818, 0x4F8B, 0x79AE, 0x91B4, 0x96B8, 0x60E1,
	0x4E86, 0x50DA, 0x5BEE, 0x5C3F, 0x6599, 0x6A02, 0x71CE, 0x7642,
	0x84FC, 0x907C, 0x9F8D, 0x6688, 0x962E, 0x5289, 0x677B, 0x67F3,
	0x6D41, 0x6E9C, 0x7409, 0x7559, 0x786B, 0x7D10, 0x985E, 0x516D,
	0x622E, 0x9678, 0x502B, 0x5D19, 0x6DEA, 0x8F2A, 0x5F8B, 0x6144,
	0x6817, 0x7387, 0x9686, 0x5229, 0x540F, 0x5C65, 0x6613, 0x674E,
	0x68A8, 0x6CE5, 0x7406, 0x75E2, 0x7F79, 0x88CF, 0x88E1, 0x91CC,
	0x96E2, 0x533F, 0x6EBA, 0x541D, 0x71D0, 0x7498, 0x85FA, 0x96A3,
	0x9C57, 0x9E9F, 0x6797, 0x6DCB, 0x81E8, 0x7ACB, 0x7B20, 0x7C92,
	0x72C0, 0x7099, 0x8B58, 0x4EC0, 0x8336, 0x523A, 0x5207, 0x5EA6,
	0x62D3, 0x7CD6, 0x5B85, 0x6D1E, 0x66B4, 0x8F3B, 0x884C, 0x964D,
	0x898B, 0x5ED3, 0x5140, 0x55C0, 0x585A, 0x6674, 0x51DE, 0x732A,
	0x76CA, 0x793C, 0x795E, 0x7965, 0x798F, 0x9756, 0x7CBE, 0x7FBD,
	0x8612, 0x8AF8, 0x9038, 0x90FD, 0x98EF, 0x98FC, 0x9928, 0x9DB4,
	0x4FAE, 0x50E7, 0x514D, 0x52C9, 0x52E4, 0x5351, 0x559D, 0x5606,
	0x5668, 0x5840, 0x58A8, 0x5C64, 0x5C6E, 0x6094, 0x6168, 0x618E,
	0x61F2, 0x654F, 0x65E2, 0x6691, 0x6885, 0x6D77, 0x6E1A, 0x6F22,
	0x716E, 0x722B, 0x7422, 0x7891, 0x793E, 0x7949, 0x7948, 0x7950,
	0x7956, 0x795D, 0x798D, 0x798E, 0x7A40, 0x7A81, 0x7BC0, 0x7DF4,
	0x7E09, 0x7E41, 0x7F72, 0x8005, 0x81ED, 0x8279, 0x8279, 0x8457,
	0x8910, 0x8996, 0x8B01, 0x8B39, 0x8CD3, 0x8D08, 0x8FB6, 0x9038,
	0x96E3, 0x97FF, 0x983B, 0x4E26, 0x51B5, 0x5168, 0x4F80, 0x5145,
	0x5180, 0x52C7, 0x52FA, 0x559D, 0x5555, 0x5599, 0x55E2, 0x585A,
	0x58B3, 0x5944, 0x5954, 0x5A62, 0x5B28, 0x5ED2, 0x5ED9, 0x5F69,
	0x5FAD, 0x60D8, 0x614E, 0x6108, 0x618E, 0x6160, 0x61F2, 0x6234,
	0x63C4, 0x641C, 0x6452, 0x6556, 0x6674, 0x6717, 0x671B, 0x6756,
	0x6B79, 0x6BBA, 0x6D41, 0x6EDB, 0x6ECB, 0x6F22, 0x701E, 0x716E,
	0x77A7, 0x7235, 0x72AF, 0x732A, 0x7471, 0x7506, 0x753B, 0x761D,
	0x761F, 0x76CA, 0x76DB, 0x76F4, 0x774A, 0x7740, 0x78CC, 0x7AB1,
	0x7BC0, 0x7C7B, 0x7D5B, 0x7DF4, 0x7F3E, 0x8005, 0x8352, 0x83EF,
	0x8779, 0x8941, 0x8986, 0x8996, 0x8ABF, 0x8AF8, 0x8ACB, 0x8B01,
	0x8AFE, 0x8AED, 0x8B39, 0x8B8A, 0x8D08, 0x8F38, 0x9072, 0x9199,
	0x9276, 0x967C, 0x96E3, 0x9756, 0x97DB, 0x97FF, 0x980B, 0x983B,
	0x9B12, 0x9F9C, 0x3B9D, 0x4018, 0x4039, 0x9F43, 0x9F8E, 0x4E3D,
	0x4E38, 0x4E41, 0x4F60, 0x4FAE, 0x4FBB, 0x5002, 0x507A, 0x5099,
	0x50E7, 0x50CF, 0x349E, 0x514D, 0x5154, 0x5164, 0x5177, 0x34B9,
	0x5167, 0x518D, 0x5197, 0x51A4, 0x4ECC, 0x51AC, 0x51B5, 0x51F5,
	0x5203, 0x34DF, 0x523B, 0x5246, 0x5272, 0x5277, 0x3515, 0x52C7,
	0x52C9, 0x52E4, 0x52FA, 0x5305, 0x5306, 0x5317, 0x5349, 0x5351,
	0x535A, 0x5373, 0x537D, 0x537F, 0x537F, 0x537F, 0x7070, 0x53CA,
	0x53DF, 0x53EB, 0x53F1, 0x5406, 0x549E, 0x5438, 0x5448, 0x5468,
	0x54A2, 0x54F6, 0x5510, 0x5553, 0x5563, 0x5584, 0x5584, 0x5599,
	0x55AB, 0x55B3, 0x55C2, 0x5716, 0x5606, 0x5717, 0x5651, 0x5674,
	0x5207, 0x58EE, 0x57CE, 0x57F4, 0x580D, 0x578B, 0x5832, 0x5831,
	0x58AC, 0x58F2, 0x58F7, 0x5906, 0x591A, 0x5922, 0x5962, 0x59EC,
	0x5A1B, 0x5A27, 0x59D8, 0x5A66, 0x36EE, 0x36FC, 0x5B08, 0x5B3E,
	0x5B3E, 0x5BC3, 0x5BD8, 0x5BE7, 0x5BF3, 0x5BFF, 0x5C06, 0x5F53,
	0x5C22, 0x3781, 0x5C60, 0x5C6E, 0x5CC0, 0x5C8D, 0x5D43, 0x5D6E,
	0x5D6B, 0x5D7C, 0x5DE1, 0x5DE2, 0x382F, 0x5DFD, 0x5E28, 0x5E3D,
	0x5E69, 0x3862, 0x387C, 0x5EB0, 0x5EB3, 0x5EB6, 0x5ECA, 0x5EFE,
	0x8201, 0x5F22, 0x5F22, 0x38C7, 0x5F62, 0x5F6B, 0x38E3, 0x5F9A,
	0x5FCD, 0x5FD7, 0x5FF9, 0x6081, 0x393A, 0x391C, 0x6094, 0x60C7,
	0x6148, 0x614C, 0x614E, 0x614C, 0x617A, 0x618E, 0x61B2, 0x61A4,
	0x61AF, 0x61DE, 0x61F2, 0x61F6, 0x6210, 0x621B, 0x625D, 0x62B1,
	0x62D4, 0x6350, 0x633D, 0x62FC, 0x6368, 0x6383, 0x63E4, 0x6422,
	0x63C5, 0x63A9, 0x3A2E, 0x6469, 0x647E, 0x649D, 0x6477, 0x3A6C,
	0x654F, 0x656C, 0x65E3, 0x66F8, 0x6649, 0x3B19, 0x6691, 0x3B08,
	0x3AE4, 0x5192, 0x5195, 0x6700, 0x669C, 0x80AD, 0x43D9, 0x6717,
	0x671B, 0x6721, 0x675E, 0x6753, 0x3B49, 0x67FA, 0x6785, 0x6852,
	0x6885, 0x688E, 0x681F, 0x6914, 0x3B9D, 0x6942, 0x69A3, 0x69EA,
	0x6AA8, 0x6ADB, 0x3C18, 0x6B21, 0x6B54, 0x3C4E, 0x6B72, 0x6B9F,
	0x6BBA, 0x6BBB, 0x6C4E, 0x6CBF, 0x6CCD, 0x6C67, 0x6D16, 0x6D3E,
	0x6D77, 0x6D41, 0x6D69, 0x6D78, 0x6D85, 0x6D34, 0x6E2F, 0x6E6E,
	0x3D33, 0x6ECB, 0x6EC7, 0x6DF9, 0x6F6E, 0x6FC6, 0x7039, 0x701E,
	0x701B, 0x3D96, 0x704A, 0x707D, 0x7077, 0x70AD, 0x7145, 0x719C,
	0x7228, 0x7235, 0x7250, 0x7280, 0x7295, 0x737A, 0x738B, 0x3EAC,
	0x73A5, 0x3EB8, 0x3EB8, 0x7447, 0x745C, 0x7471, 0x7485, 0x74CA,
	0x3F1B, 0x7524, 0x753E, 0x7570, 0x7610, 0x3FFC, 0x4008, 0x76F4,
	0x771E, 0x771F, 0x771F, 0x774A, 0x4039, 0x778B, 0x4046, 0x4096,
	0x784E, 0x788C, 0x78CC, 0x40E3, 0x7956, 0x798F, 0x79EB, 0x412F,
	0x7A40, 0x7A4A, 0x7A4F, 0x7AEE, 0x4202, 0x7BC6, 0x7BC9, 0x4227,
	0x7CD2, 0x42A0, 0x7CE8, 0x7CE3, 0x7D00, 0x7D63, 0x4301, 0x7DC7,
	0x7E02, 0x7E45, 0x4334, 0x4359, 0x7F7A, 0x7F95, 0x7FFA, 0x8005,
	0x8060, 0x8070, 0x43D5, 0x80B2, 0x8103, 0x440B, 0x813E, 0x5AB5,
	0x8201, 0x8204, 0x8F9E, 0x446B, 0x8291, 0x828B, 0x829D, 0x52B3,
	0x82B1, 0x82B3, 0x82BD, 0x82E6, 0x82E5, 0x831D, 0x8363, 0x83AD,
	0x8323, 0x83BD, 0x83E7, 0x8457, 0x8353, 0x83CA, 0x83CC, 0x83DC,
	0x452B, 0x84F1, 0x84F3, 0x8516, 0x8564, 0x455D, 0x4561, 0x456B,
	0x8650, 0x865C, 0x8667, 0x8669, 0x86A9, 0x8688, 0x870E, 0x86E2,
	0x8779, 0x8728, 0x876B, 0x8786, 0x45D7, 0x87E1, 0x8801, 0x45F9,
	0x8860, 0x8863, 0x88D7, 0x88DE, 0x4635, 0x88FA, 0x34BB, 0x46BE,
	0x46C7, 0x8AA0, 0x8AED, 0x8B8A, 0x8C55, 0x8CAB, 0x8CC1, 0x8D1B,
	0x8D77, 0x8DCB, 0x8DBC, 0x8DF0, 0x8ED4, 0x8F38, 0x9094, 0x90F1,
	0x9111, 0x911B, 0x9238, 0x92D7, 0x92D8, 0x927C, 0x93F9, 0x9415,
	0x958B, 0x4995, 0x95B7, 0x49E6, 0x96C3, 0x5DB2, 0x9723, 0x4A6E,
	0x4A76, 0x97E0, 0x4AB2, 0x980B, 0x980B, 0x9829, 0x98E2, 0x4B33,
	0x9929, 0x99A7, 0x99C2, 0x99FE, 0x4BCE, 0x9B12, 0x9C40, 0x9CFD,
	0x4CCE, 0x4CED, 0x9D67, 0x4CF8, 0x9EBB, 0x4D56, 0x9EF9, 0x9EFE,
	0x9F05, 0x9F0F, 0x9F16, 0x9F3B, 0x0061, 0x0062, 0x0063, 0x0064,
	0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x006A, 0x006B, 0x006C,
	0x006D, 0x006E, 0x006F, 0x0070, 0x0071, 0x0072, 0x0073, 0x0074,
	0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007A, 0x03BC, 0x00E6,
	0x00F0, 0x00F8, 0x00FE, 0x0111, 0x0127, 0x0069, 0x0133, 0x0140,
	0x0142, 0x014B, 0x0153, 0x0167, 0x0073, 0x0253, 0x0183, 0x0185,
	0x0254, 0x0188, 0x0256, 0x0257, 0x018C, 0x01DD, 0x0259, 0x025B,
	0x0192, 0x0260, 0x0263, 0x0269, 0x0268, 0x0199, 0x026F, 0x0272,
	0x0275, 0x01A3, 0x01A5, 0x0280, 0x01A8, 0x0283, 0x01AD, 0x0288,
	0x028A, 0x028B, 0x01B4, 0x01B6, 0x0292, 0x01B9, 0x01BD, 0x01C6,
	0x01C6, 0x01C9, 0x01C9, 0x01CC, 0x01CC, 0x01E5, 0x01F3, 0x01F3,
	0x0195, 0x01BF, 0x021D, 0x019E, 0x0223, 0x0225, 0x2C65, 0x023C,
	0x019A, 0x2C66, 0x0242, 0x0180, 0x0289, 0x028C, 0x0247, 0x0249,
	0x024B, 0x024D, 0x024F, 0x03B9, 0x03B1, 0x03B2, 0x03B3, 0x03B4,
	0x03B5, 0x03B6, 0x03B7, 0x03B8, 0x03B9, 0x03BA, 0x03BB, 0x03BC,
	0x03BD, 0x03BE, 0x03BF, 0x03C0, 0x03C1, 0x03C3, 0x03C4, 0x03C5,
	0x03C6, 0x03C7, 0x03C8, 0x03C9, 0x03C3, 0x03B2, 0x03B8, 0x03C6,
	0x03C0, 0x03D9, 0x03DB, 0x03DD, 0x03DF, 0x03E1, 0x03E3, 0x03E5,
	0x03E7, 0x03E9, 0x03EB, 0x03ED, 0x03EF, 0x03BA, 0x03C1, 0x03B8,
	0x03B5, 0x03F8, 0x03F2, 0x03FB, 0x037B, 0x037C, 0x037D, 0x0452,
	0x0454, 0x0455, 0x0456, 0x0458, 0x0459, 0x045A, 0x045B, 0x045F,
	0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0436, 0x0437,
	0x0438, 0x043A, 0x043B, 0x043C, 0x043D, 0x043E, 0x043F, 0x0440,
	0x0441, 0x0442, 0x0443, 0x0444, 0x0445, 0x0446, 0x0447, 0x0448,
	0x0449, 0x044A, 0x044B, 0x044C, 0x044D, 0x044E, 0x044F, 0x0461,
	0x0463, 0x0465, 0x0467, 0x0469, 0x046B, 0x046D, 0x046F, 0x0471,
	0x0473, 0x0475, 0x0479, 0x047B, 0x047D, 0x047F, 0x0481, 0x048B,
	0x048D, 0x048F, 0x0491, 0x0493, 0x0495, 0x0497, 0x0499, 0x049B,
	0x049D, 0x049F, 0x04A1, 0x04A3, 0x04A5, 0x04A7, 0x04A9, 0x04AB,
	0x04AD, 0x04AF, 0x04B1, 0x04B3, 0x04B5, 0x04B7, 0x04B9, 0x04BB,
	0x04BD, 0x04BF, 0x04CF, 0x04C4, 0x04C6, 0x04C8, 0x04CA, 0x04CC,
	0x04CE, 0x04D5, 0x04D9, 0x04E1, 0x04E9, 0x04F7, 0x04FB, 0x04FD,
	0x04FF, 0x0501, 0x0503, 0x0505, 0x0507, 0x0509, 0x050B, 0x050D,
	0x050F, 0x0511, 0x0513, 0x0561, 0x0562, 0x0563, 0x0564, 0x0565,
	0x0566, 0x0567, 0x0568, 0x0569, 0x056A, 0x056B, 0x056C, 0x056D,
	0x056E, 0x056F, 0x0570, 0x0571, 0x0572, 0x0573, 0x0574, 0x0575,
	0x0576, 0x0577, 0x0578, 0x0579, 0x057A, 0x057B, 0x057C, 0x057D,
	0x057E, 0x057F, 0x0580, 0x0581, 0x0582, 0x0583, 0x0584, 0x0585,
	0x0586, 0x2D00, 0x2D01, 0x2D02, 0x2D03, 0x2D04, 0x2D05, 0x2D06,
	0x2D07, 0x2D08, 0x2D09, 0x2D0A, 0x2D0B, 0x2D0C, 0x2D0D, 0x2D0E,
	0x2D0F, 0x2D10, 0x2D11, 0x2D12, 0x2D13, 0x2D14, 0x2D15, 0x2D16,
	0x2D17, 0x2D18, 0x2D19, 0x2D1A, 0x2D1B, 0x2D1C, 0x2D1D, 0x2D1E,
	0x2D1F, 0x2D20, 0x2D21, 0x2D22, 0x2D23, 0x2D24, 0x2D25, 0x214E,
	0x2170, 0x2171, 0x2172, 0x2173, 0x2174, 0x2175, 0x2176, 0x2177,
	0x2178, 0x2179, 0x217A, 0x217B, 0x217C, 0x217D, 0x217E, 0x217F,
	0x2184, 0x24D0, 0x24D1, 0x24D2, 0x24D3, 0x24D4, 0x24D5, 0x24D6,
	0x24D7, 0x24D8, 0x24D9, 0x24DA, 0x24DB, 0x24DC, 0x24DD, 0x24DE,
	0x24DF, 0x24E0, 0x24E1, 0x24E2, 0x24E3, 0x24E4, 0x24E5, 0x24E6,
	0x24E7, 0x24E8, 0x24E9, 0x2C30, 0x2C31, 0x2C32, 0x2C33, 0x2C34,
	0x2C35, 0x2C36, 0x2C37, 0x2C38, 0x2C39, 0x2C3A, 0x2C3B, 0x2C3C,
	0x2C3D, 0x2C3E, 0x2C3F, 0x2C40, 0x2C41, 0x2C42, 0x2C43, 0x2C44,
	0x2C45, 0x2C46, 0x2C47, 0x2C48, 0x2C49, 0x2C4A, 0x2C4B, 0x2C4C,
	0x2C4D, 0x2C4E, 0x2C4F, 0x2C50, 0x2C51, 0x2C52, 0x2C53, 0x2C54,
	0x2C55, 0x2C56, 0x2C57, 0x2C58, 0x2C59, 0x2C5A, 0x2C5B, 0x2C5C,
	0x2C5D, 0x2C5E, 0x2C61, 0x026B, 0x1D7D, 0x027D, 0x2C68, 0x2C6A,
	0x2C6C, 0x2C76, 0x2C81, 0x2C83, 0x2C85, 0x2C87, 0x2C89, 0x2C8B,
	0x2C8D, 0x2C8F, 0x2C91, 0x2C93, 0x2C95, 0x2C97, 0x2C99, 0x2C9B,
	0x2C9D, 0x2C9F, 0x2CA1, 0x2CA3, 0x2CA5, 0x2CA7, 0x2CA9, 0x2CAB,
	0x2CAD, 0x2CAF, 0x2CB1, 0x2CB3, 0x2CB5, 0x2CB7, 0x2CB9, 0x2CBB,
	0x2CBD, 0x2CBF, 0x2CC1, 0x2CC3, 0x2CC5, 0x2CC7, 0x2CC9, 0x2CCB,
	0x2CCD, 0x2CCF, 0x2CD1, 0x2CD3, 0x2CD5, 0x2CD7, 0x2CD9, 0x2CDB,
	0x2CDD, 0x2CDF, 0x2CE1, 0x2CE3, 0xFF41, 0xFF42, 0xFF43, 0xFF44,
	0xFF45, 0xFF46, 0xFF47, 0xFF48, 0xFF49, 0xFF4A, 0xFF4B, 0xFF4C,
	0xFF4D, 0xFF4E, 0xFF4F, 0xFF50, 0xFF51, 0xFF52, 0xFF53, 0xFF54,
	0xFF55, 0xFF56, 0xFF57, 0xFF58, 0xFF59, 0xFF5A
	};

// Total size: 38494 bytes
