/*****************************************************************************/
/*                                                                           */
/* PROJECT : ANYSTORE II                                                     */
/* MODULE  : ONBL2                                                           */
/* NAME    : OneNAND Boot Loader stage 2                                     */
/* FILE    : ONBL2.c(cpp)                                                    */
/* PURPOSE : This file implements for OneNAND Bootloader                     */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/*        COPYRIGHT 2003-2007, SAMSUNG ELECTRONICS CO., LTD.                 */
/*                      ALL RIGHTS RESERVED                                  */
/*                                                                           */
/*   Permission is hereby granted to licensees of Samsung Electronics        */
/*   Co., Ltd. products to use or abstract this computer program for the     */
/*   sole purpose of implementing a product based on Samsung                 */
/*   Electronics Co., Ltd. products. No other rights to reproduce, use,      */
/*   or disseminate this computer program, whether in part or in whole,      */
/*   are granted.                                                            */
/*                                                                           */
/*   Samsung Electronics Co., Ltd. makes no representation or warranties     */
/*   with respect to the performance of this computer program, and           */
/*   specifically disclaims any responsibility for any damages,              */
/*   special or consequential, connected with the use of this program.       */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* REVISION HISTORY                                                          */
/*                                                                           */
/* - 26/MAY/2003 [Janghwan Kim] : first writing                              */
/* - 19/Nov/2003 [SeWook Na]    : Add BML Operations                         */
/* - 21/Nov/2003 [MinSoo Moon]  : Reorganization                             */
/* - 24/DEC/2003 [MinSoo Moon]  : Removed platform relative feature          */
/* - 04/MAY/2004 [Janghwan Kim] : Removed platform relative feature          */
/* - 08/JUN/2007 [WooYoung Yang] : Reorganization                            */
/*                                                                           */ 
/*****************************************************************************/


/*****************************************************************************/
/* Note                                                                      */
/*                                                                           */
// Copyright (c) 1996-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of the License "Symbian Foundation License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.symbianfoundation.org/legal/sfl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// Coreloader code for boot from Samsung OneNAND                             
// Adapted from Samsung code distributed with Unistore-II XSR 1.5.2p2.       
// File modified as follows:                                                 
// - Remove platform specific code											 
// - Remove NORMAL_BOOT_MODE / TERTIARY_BL_MODE config options,              
// GetBootMode() and ShadowTBL() and always run in normal boot mode        
// - Don't use OS_SHADOW_ADDR macro. Use PSL GetRomTargetAddress()           
// instead.                                                                
// - Remove macro ONBL2_LOG_MSG_ON which is not used                         
// - Insert code to handle compressed roms                                   
// - Remove void parameter lists                                             
// - Remove procedure definitions for PrintBanner() amd _ONBL2()             
// - Modify _ONBL2() to return pointer to CoreOS image as UINT32             
// - Replace Main() with LoadMain() and call PSL bootos() to jump to         
// - OS image instead of jumping directly from this module                   
// Header file declararation                                                 
// 
//

/*****************************************************************************/
#include <XsrTypes.h>
#include <OAM.h>
#include <PAM.h>
#include <BML.h>
#include <LLD.h>
#include <ONLDReg.h>
#include <ONLD.h>

#include <e32def.h>
#include <e32def_private.h>
#include <e32cmn.h>
#include <e32cmn_private.h>
#include <e32rom.h>
#include <e32ldr.h>

#include "nand_plat.h"

#include "inflate.h"
#include "coreldr.h"

/*****************************************************************************/
/* Debug Configuration                                                       */
/*****************************************************************************/
#define ONBL2_ERR_MSG_ON             1 
#define ONBL2_INF_MSG_ON             1

#define ONBL2_RTL_PRINT(x)           XSR_RTL_PRINT(x)

#if ONBL2_ERR_MSG_ON
#define ONBL2_ERR_PRINT(x)           XSR_RTL_PRINT(x)
#else
#define ONBL2_ERR_PRINT(x)        
#endif /* #if ONBL2_ERR_MSG_ON */

#if ONBL2_INF_MSG_ON
#define ONBL2_INF_PRINT(x)           XSR_DBG_PRINT(x)
#else
#define ONBL2_INF_PRINT(x)        
#endif  /* #if ONBL2_INF_MSG_ON */

#define SYNC_MODE
#undef  CHK_SUM
#undef  LOCK_FOREVER
//#undef  HEX_DUMP
#define HEX_DUMP

extern "C" void charout(UINT8);
extern "C" void bootos(TUint32 aBootAddress);

/*****************************************************************************/
/* Code Implementation                                                       */
/*****************************************************************************/
void _PrintBanner()
	{
    ONBL2_RTL_PRINT((TEXT("\r\n")));
    ONBL2_RTL_PRINT((TEXT("/*===*===*/\r\n")));
    ONBL2_RTL_PRINT((TEXT("/* ONBL2 */\r\n")));
    ONBL2_RTL_PRINT((TEXT("/*===*===*/\r\n")));
	}

/*****************************************************************************/
/*                                                                           */
/* NAME                                                                      */
/*      ErrorLoop                                                            */
/* DESCRIPTION                                                               */
/*      This function spins forever (error case it can be called)            */
/* PARAMETERS                                                                */
/*      None                                                                 */
/* RETURN VALUES                                                             */
/*      None                                                                 */
/* NOTES                                                                     */
/*                                                                           */
/*****************************************************************************/
VOID 
ErrorLoop()
	{
    ONBL2_ERR_PRINT((TEXT("Error Occurs During Boot. Please ReBoot.\r\n")));
    while(1);
	}
	
#if defined(HEX_DUMP)	
/*****************************************************************************/
/*                                                                           */
/* NAME                                                                      */
/*      HexDump                                                              */
/* DESCRIPTION                                                               */
/*      Dump Array of Memory for debug purposes                              */
/* PARAMETERS                                                                */
/*      aStartAddress                                                        */
/*          Memory address of tart point of dump                             */
/*      aLength																 */
/*   		Length of dump													 */
/* RETURN VALUES                                                             */
/*      None                                                                 */
/* NOTES                                                                     */
/*                                                                           */
/*****************************************************************************/
VOID
HexDump(UINT8 * aStartAddress, UINT32 aLength)
	{
	UINT32 nIndex;
	for(nIndex = 0; nIndex != aLength ; ++nIndex )
		{
		if(nIndex % 16 == 0)	
			{
			ONBL2_RTL_PRINT((TEXT("\r\n0x%08x: "), aStartAddress + nIndex));	
			}
		ONBL2_RTL_PRINT((TEXT("%02x "), *(aStartAddress + nIndex) ));
		
		}
	ONBL2_RTL_PRINT((TEXT("\r\n\r\n")));
	}
#endif // HEX_DUMP

/*****************************************************************************/
/*                                                                           */
/* NAME                                                                      */
/*      ShadowOS                                                             */
/* DESCRIPTION                                                               */
/*      Copy / decompress OS image                                           */
/* PARAMETERS                                                                */
/*      nVol                                                                 */
/*          Volume Number                                                    */
/* RETURN VALUES                                                             */
/*      Start Address of OS Image                                            */
/* NOTES                                                                     */
/*      After Image Launch, Never return                                     */
/*                                                                           */
/*****************************************************************************/
UINT32
ShadowOS(UINT32 nVol)
	{
    UINT32          nPpn;
    UINT8           *pSrc = (UINT8*)(GetRomTargetAddress());
    
    INT32           nRet;
    UINT8           aSBuf[64];
    UINT32          n1stVSN;
    UINT32          nNumOfScts;
    UINT32        	nPgsPerBlk;
    UINT32        	nSctsPerPg;
    UINT32		  	nSctsPerBlk;
    UINT32          nPer;
    XSRPartEntry    stPEntry;
    BMLVolSpec      stVolSpec;

#if defined(LOCK_FOREVER)
    UINT32          nRev;
#endif //#if defined(LOCK_FOREVER)

#if defined(CHK_SUM)
	/* Check sum Related variables */
	BOOL32		bChk_Sum_Err;
	UINT32		nLen, nCnt, nCrcSector, pCrcTst;
	UINT32		nMakeCrc[256], nReadCrc[256];

    nCrcSector = 0;
    bChk_Sum_Err = FALSE32;
#endif //#if defined(CHK_SUM)
        
    ONBL2_INF_PRINT((TEXT("++ ShadowOS ++\r\n")));  

    nRet = BML_Init();
    if (nRet != BML_SUCCESS)
		{
		ONBL2_ERR_PRINT((TEXT("BML_Init Error\r\n")));
        ErrorLoop();
		}
	
    nRet = BML_Open(nVol);
    if (nRet != BML_SUCCESS)
		{
		ONBL2_ERR_PRINT((TEXT("BML_Open Error\r\n")));
        ErrorLoop();
		}

    if (BML_GetVolInfo(nVol, &stVolSpec) != BML_SUCCESS)
		{
		ONBL2_ERR_PRINT((TEXT("BML_GetVolInfo Error\r\n")));
        ErrorLoop();
		}

    if (BML_LoadPIEntry(nVol, PARTITION_ID_COPIEDOS, &stPEntry) != BML_SUCCESS)
		{
		ONBL2_ERR_PRINT((TEXT("BML_LoadPIEntry Error! P_ID: %d\r\n"), PARTITION_ID_COPIEDOS));
        ErrorLoop();
		}

    nPgsPerBlk	 = stVolSpec.nPgsPerBlk;
    nSctsPerPg   = stVolSpec.nSctsPerPg;
    nSctsPerBlk  = nPgsPerBlk * nSctsPerPg;
    
    ONBL2_INF_PRINT((TEXT("SctsPerPg: %d, PgsPerBlk: %d\r\n"), nSctsPerPg, nPgsPerBlk));  
      
    n1stVSN      = stPEntry.n1stVbn    * stVolSpec.nPgsPerBlk * nSctsPerPg;
    nNumOfScts   = stPEntry.nNumOfBlks * stVolSpec.nPgsPerBlk * nSctsPerPg;
 
    ONBL2_INF_PRINT((TEXT("Shadow Proceed: n1stVSN: %0x, nNumOfScts: %d\r\n"), n1stVSN, nNumOfScts));

	// setup to investigate rom header
	TRomHeader& romHeader = * (TRomHeader *)pSrc;

	// first sector to get rom header
	nRet = BML_Read(nVol,                                   /* Volume Number    */
                    n1stVSN,                                /* Vsn              */
                    1,										/* Sector Number    */
                    pSrc,                                   /* Pointer to Data  */
                    NULL,                                   /* Not used         */
                    (BML_FLAG_SYNC_OP | BML_FLAG_ECC_ON));  /* Op Flag          */ 
    if (nRet != BML_SUCCESS)
		{                 

       ONBL2_ERR_PRINT((TEXT("BML_Read Error\r\n")));
		}

	UINT32 romHeaderSize = sizeof(TRomHeader);
    ONBL2_INF_PRINT((TEXT("RomHeader size: %d\r\n"), romHeaderSize));
	
	ONBL2_INF_PRINT((TEXT("CompType: 0x%08x CompSz %d (0x%08x) UncompSz %d (0x%08x)\r\n")
									, romHeader.iCompressionType
									, romHeader.iCompressedSize
									, romHeader.iCompressedSize
									, romHeader.iUncompressedSize
									, romHeader.iUncompressedSize));
	
	ONBL2_INF_PRINT((TEXT("Unpaged CompType: 0x%08x CompSz %d (0x%08x) UncompSz %d (0x%08x)\r\n")
									, romHeader.iCompressionType
									, romHeader.iUnpagedCompressedSize
									, romHeader.iUnpagedCompressedSize
									, romHeader.iUnpagedUncompressedSize
									, romHeader.iUnpagedUncompressedSize));
	
	UINT32 compType = romHeader.iCompressionType;
	UINT32 compressedImageLen = romHeader.iUnpagedCompressedSize;
	UINT32 uncompressedImageLen = romHeader.iUnpagedUncompressedSize;
	
	ONBL2_INF_PRINT((TEXT("Type: 0x%08x CompSz %d (0x%08x) UncompSz %d (0x%08x)\r\n")
									, compType
									, compressedImageLen
									, compressedImageLen
									, uncompressedImageLen
									, uncompressedImageLen));
																
	if (compType == 0x0)
		{
		// uncompressed image
		ONBL2_INF_PRINT((TEXT("Uncompressed image.\r\n")));	
		if(romHeader.iPageableRomStart)
			{
			ONBL2_INF_PRINT((TEXT("Detected pageable rom (iPageableRomStart:0x%08x).\r\n"),romHeader.iPageableRomStart ));
			nNumOfScts = romHeader.iPageableRomStart / XSR_SECTOR_SIZE;
			if (romHeader.iPageableRomStart % XSR_SECTOR_SIZE) nNumOfScts++;
			}
		}
    else if (compType == 0x101F7AFC)
		{
		// Read whole un-paged section (RomHeader + bootstrap + Page Index Table + compressed un-paged section)
		ONBL2_INF_PRINT((TEXT("Deflated un-paged image.\r\n")));
		ONBL2_INF_PRINT((TEXT("iCompressedUnpagedStart:0x%08x.\r\n"),romHeader.iCompressedUnpagedStart ));
		ONBL2_INF_PRINT((TEXT("iUnpagedCompressedSize :0x%08x.\r\n"),romHeader.iUnpagedCompressedSize ));
		
		// Read whole un-paged section (memcpy is faster than read from NAND)
		nNumOfScts = (romHeader.iCompressedUnpagedStart + compressedImageLen) / XSR_SECTOR_SIZE;
		if ((romHeader.iCompressedUnpagedStart + compressedImageLen) % XSR_SECTOR_SIZE) nNumOfScts++;
		
		// Read into	
		ONBL2_INF_PRINT((TEXT("pSrc                   :0x%08x.\r\n"),pSrc ));
		ONBL2_INF_PRINT((TEXT("iCompressedUnpagedStart:0x%08x.\r\n"),romHeader.iCompressedUnpagedStart));
		ONBL2_INF_PRINT((TEXT("uncompressedImageLen   :0x%08x.\r\n"),uncompressedImageLen ));
		pSrc += (romHeader.iCompressedUnpagedStart + uncompressedImageLen);
		ONBL2_INF_PRINT((TEXT("New pSrc               :0x%08x.\r\n"),pSrc ));
		
		}
	else
		{
		// compression type not handled
   		ONBL2_ERR_PRINT((TEXT("Bad Compression Type Error\r\n")));
		ErrorLoop();
		}

	// now read appropriate nNumOfScts to pSrc
	UINT8           *pRAM=pSrc;
	UINT8           aMBuf[2048];
	UINT8			progress = 10;
	ONBL2_RTL_PRINT((TEXT("ONBL2: loading Core OS into RAM\r\n")));
	for (UINT32 nVS = n1stVSN; nVS < (n1stVSN + nNumOfScts); nVS+=4, pRAM+=2048)
		{
			// read data from the particular sector
			nRet = BML_Read(nVol,                           /* Volume Number    */
                    nVS,                                    /* Vsn              */
                    4,										/* Sector Number    */
                    aMBuf,                                  /* Pointer to Data  */
                    NULL,                                   /* Not used         */
                    (BML_FLAG_SYNC_OP | BML_FLAG_ECC_ON));  /* Op Flag          */ 
			if (nRet != BML_SUCCESS)
				{                 
				ONBL2_ERR_PRINT((TEXT("BML_Read Error\r\n")));
				}
			OAM_Memcpy(pRAM,aMBuf,2048);
			if (((100*(nVS-n1stVSN))/nNumOfScts)>progress)
				{
				ONBL2_RTL_PRINT((TEXT("ONBL2: %d%% loaded\r\n"),progress));
				progress += 10;
				}
		}
		ONBL2_RTL_PRINT((TEXT("ONBL2: 100%% loaded\r\n"),progress));
  
#if defined(CHK_SUM)
#define CHK_SUM_BLK	465
	// CRC check
    for (nPpn = 0; nPpn < nNumOfScts; nPpn += nSctsPerPg)
		{
 
        pCrcTst = (UINT32)pSrc;

    	if ((nPpn % nSctsPerBlk) == 0)
    		{
        	for (nCnt = 0; nCnt < nSctsPerBlk; nCnt++)
        		{
        		nMakeCrc[nCnt] = 0;
        		}
    		}
            	    	
    	/* Calculate 4 Sector's Sum value */
     	for (nCnt = 0; nCnt < nSctsPerPg; nCnt++)
    		{   	
	    	for (nLen = 0; nLen < XSR_SECTOR_SIZE; nLen++)
	    		{
	    		nMakeCrc[nPpn % nSctsPerBlk + nCnt] += *(UINT8 *)pCrcTst++;
	    		}
    		}
    	
    	/* Read Block's Sum value */
    	if ((nPpn % nSctsPerBlk) == (nSctsPerBlk - nSctsPerPg))
    		{
	        nRet = BML_Read(nVol,
	                        CHK_SUM_BLK * nSctsPerBlk + nCrcSector,
	                        sizeof(nReadCrc) / XSR_SECTOR_SIZE,
	                        (UINT8 *)nReadCrc,
	                        aSBuf, 
	                        (BML_FLAG_SYNC_OP | BML_FLAG_ECC_ON));
	                        
	        if (nRet != BML_SUCCESS)
				{                 
	            ONBL2_ERR_PRINT((TEXT("BML_Read Error\r\n")));
				}
	        
	        /* Verify CRC */
	        for (nCnt = 0; nCnt < sizeof(nReadCrc) / sizeof(UINT32); nCnt++)
				{
	        	if (nMakeCrc[nCnt] != nReadCrc[nCnt])
	        		{
	        		Uart_Printf("CHK_SUM ERR, BLK NUM: %d, SCT NUM: %d \n", nPpn / nSctsPerBlk, nCnt);
	        		bChk_Sum_Err = TRUE32;	        		
	        		}	
				}
	        
	        nCrcSector += nSctsPerPg;
			}
		}
	if (bChk_Sum_Err)
		{
		ErrorLoop();
		}
#endif /* #if defined(CHK_SUM) */
    
	// decompress image as required
	if (compType == KUidCompressionDeflate)
		{
		// Copy uncompressed un-paged part (bootstrap + Page Index Table) to the proper place if it longer than the romHeader
		if( romHeader.iCompressedUnpagedStart > romHeaderSize )
			{
			ONBL2_INF_PRINT((TEXT("Copy uncompressed un-paged part ...\r\n")));			
			ONBL2_INF_PRINT((TEXT("from :0x%08x.\r\n"),(pSrc+romHeaderSize) ));
			ONBL2_INF_PRINT((TEXT("to   :0x%08x.\r\n"),(GetRomTargetAddress() + romHeaderSize) ));
			ONBL2_INF_PRINT((TEXT("len  :0x%08x.\r\n"),(romHeader.iCompressedUnpagedStart - romHeaderSize) ));
			memcpy1((TUint8*)(GetRomTargetAddress() + romHeaderSize), (TUint8*)pSrc+romHeaderSize, (romHeader.iCompressedUnpagedStart - romHeaderSize) );
			}
		
		ONBL2_INF_PRINT((TEXT("Decompressing rom ...\r\n")));			
		ONBL2_INF_PRINT((TEXT("from :0x%08x.\r\n"),(pSrc+romHeader.iCompressedUnpagedStart) ));
		ONBL2_INF_PRINT((TEXT("to   :0x%08x.\r\n"),(GetRomTargetAddress() + romHeader.iCompressedUnpagedStart) ));
		ONBL2_INF_PRINT((TEXT("len  :0x%08x.\r\n"),uncompressedImageLen ));
		
		TBitInput image;
		
		image.Set((TUint8*)pSrc+romHeader.iCompressedUnpagedStart, compressedImageLen*8); //Skip any uncompressed data at ROM start
		UINT32 nChars = Inflater::Inflate(
			image,
			(TUint8*)(GetRomTargetAddress() + romHeader.iCompressedUnpagedStart), 
			uncompressedImageLen
			);
		ONBL2_INF_PRINT((TEXT("Decompressed %d bytes\r\n"), nChars));			
		}	
	
	ONBL2_INF_PRINT((TEXT("\r\n-- ShadowOS --\r\n")));

	ONBL2_INF_PRINT((TEXT("\r\nAbout to start Symbian OS\r\n")));
	
#if defined(HEX_DUMP1)	
	UINT8 * startAddress;
	startAddress = (UINT8*)GetRomTargetAddress();
	HexDump(startAddress, 0x200);  //Dump Header
	
	startAddress +=0x0200; // Skip the TRomHeader
	HexDump(startAddress, 64);  //Dump beyond Header
	
	startAddress = (UINT8*)GetRomTargetAddress();
	startAddress += romHeader.iRomPageIndex;  //Dump the Rom Page Index table
	HexDump(startAddress, 256);
	
	startAddress = (UINT8*)GetRomTargetAddress();
	startAddress += 0x0001bf88;
	HexDump(startAddress, 256);
#endif	
	
#if defined(LOCK_FOREVER)    
    nRet = BML_IOCtl(nVol, BML_IOCTL_LOCK_FOREVER, NULL, 0, NULL, 0, &nRev);
    if (nRet != BML_SUCCESS)
		{
    	ONBL2_ERR_PRINT((TEXT("BML_IOCtl(ID:BML_IOCTL_LOCK_FOREVER) Error\r\n")));
		ErrorLoop();
		}
#endif
		
    return ((UINT32)GetRomTargetAddress());
}


/*****************************************************************************/
/*                                                                           */
/* NAME                                                                      */
/*      ShadowHCR                                                            */
/* DESCRIPTION                                                               */
/*      Copy / decompress HCR imagesmn                                       */
/* PARAMETERS                                                                */
/*      nVol                                                                 */
/*          Volume Number                                                    */
/* RETURN VALUES                                                             */
/*      Start Address SMRIB Table, or 0 if not supported                      */
/*                                                                           */
/*****************************************************************************/
#ifdef SUPPORTS_SMR
void
ShadowHCR(UINT32 nVol)
	{
    UINT32          nPpn;
    UINT8           *pSrc = (UINT8*)(GetSMRTargetStartAddress());
    UINT32          *pSMRIB = (UINT32*)(GetSMRIBTargetAddress());
    UINT32			smribSize; 
	   
    INT32           nRet;
    UINT8           aSBuf[64];
    UINT32          n1stVSN;
    UINT32          nNumOfScts;
    UINT32        	nPgsPerBlk;
    UINT32        	nSctsPerPg;
    UINT32		  	nSctsPerBlk;
    UINT32          nPer;
	UINT32			partID = 0;
    XSRPartEntry    stPEntry;
    BMLVolSpec      stVolSpec;
    UINT32			nPartSize;

	INT32			nUDPartFound = 0;

       
    ONBL2_INF_PRINT((TEXT("++ ShadowHCR ++\r\n")));  

    if (BML_GetVolInfo(nVol, &stVolSpec) != BML_SUCCESS)
		{
		ONBL2_ERR_PRINT((TEXT("BML_GetVolInfo Error\r\n")));
        ErrorLoop();
		}

	smribSize = 0; 
	*pSMRIB = 0; pSMRIB++;	// +0: Space to store the number of partitions, used in coreldrmmu.s to enable negative test case
	*pSMRIB = 0; pSMRIB++;  // +4: Will hold size of SMRIB
							// +8: The SMRIB entry 1 etc...	 
	for (partID = PARTITION_ID_FILESYSTEM; partID < PARTITION_ID_FILESYSTEM9; partID++)
		{

		INT32 rc =  BML_LoadPIEntry(nVol, partID, &stPEntry);
		if (rc == BML_NO_PIENTRY)
			break;
		if (rc != BML_SUCCESS)
			{
			ONBL2_ERR_PRINT((TEXT("BML_LoadPIEntry Error! P_ID: %0x\r\n"), partID));
	        ErrorLoop();
			}
			
		nUDPartFound++;	
		
	    nPgsPerBlk	 = stVolSpec.nPgsPerBlk;
	    nSctsPerPg   = stVolSpec.nSctsPerPg;
	    nSctsPerBlk  = nPgsPerBlk * nSctsPerPg;
	    
	    ONBL2_INF_PRINT((TEXT("Partition FILESYSTEM%d found: SctsPerPg: %d, PgsPerBlk: %d\r\n"), partID-PARTITION_ID_FILESYSTEM, nSctsPerPg, nPgsPerBlk));  
	      
	    n1stVSN      = stPEntry.n1stVbn    * stVolSpec.nPgsPerBlk * nSctsPerPg;
	    nNumOfScts   = stPEntry.nNumOfBlks * stVolSpec.nPgsPerBlk * nSctsPerPg;
	 	nPartSize = nNumOfScts*XSR_SECTOR_SIZE;
	 	if (nPartSize > 0x800000)
	 		{
	    	ONBL2_INF_PRINT((TEXT("FILESYSTEM%d to large for testing, size=%x, will skip\r\n"), partID-PARTITION_ID_FILESYSTEM, nPartSize));
			continue;		// Skip partition if greater than 8Mb
	 		}
	 
	    ONBL2_INF_PRINT((TEXT("Shadow HCR Proceed: FILESYSTEM%d, n1stVSN: %0x, nNumOfScts: %d\r\n"), partID-PARTITION_ID_FILESYSTEM, n1stVSN, nNumOfScts));
	
											
		// now read appropriate nNumOfScts to pSrc
		nRet = BML_MRead(nVol,                                   /* Volume Number    */
	                     n1stVSN,                                /* Vsn              */
	                     nNumOfScts,                             /* Sector Number    */
	                     pSrc,                                   /* Pointer to Data  */
	                     NULL,                                   /* Not used         */
	                     (BML_FLAG_SYNC_OP | BML_FLAG_ECC_ON));  /* Op Flag          */ 
	    if (nRet != BML_SUCCESS)
			{                 
	        ONBL2_ERR_PRINT((TEXT("BML_MRead Error\r\n")));
	        ErrorLoop();
			}
		else
		    {
			ONBL2_INF_PRINT((TEXT("BML_MRead Success, pSrc = 0x%0x\r\n"),pSrc ));
			
			// Now Setup the SMRIB to record the shadowing done above


			pSMRIB[smribSize+0] = (UINT32)pSrc;
			pSMRIB[smribSize+1] = nPartSize;
//			pSMRIB[smribSize+1] = nPartSize+13; // TC S03.04.00 - SMR size not multiple of page size in bytes
		    pSMRIB[smribSize+2] = 0x00286AB8;
		    pSMRIB[smribSize+3] = partID;	
			smribSize +=4;				
			pSrc = (UINT8*) (((UINT32)pSrc)+(nPartSize+4095) & 0xFFFFF000); 
	        }
	        
	    }
	
	pSMRIB[smribSize+0] = 0;
	pSMRIB[smribSize+1] = 0;    
	pSMRIB[smribSize+2] = 0;
	pSMRIB[smribSize+3] = 0;

	if (smribSize > 0)
		{
		smribSize +=4;
		}
	*(pSMRIB-2) = nUDPartFound;
	*(pSMRIB-1) = smribSize<<2;     // Number of bytes in SMRIB	
//	*(pSMRIB-1) = 128;     			// TC S03.03.02 - size to big test
			
	#if defined(HEX_DUMP)	
		ONBL2_INF_PRINT((TEXT("==> ShadowHCR SMRIB\r\n")));
	   	UINT8 * startAddress = (UINT8*)(pSMRIB-2);
		HexDump(startAddress, 0x88);  //Dump SMRIB
	#endif	
	
	ONBL2_INF_PRINT((TEXT("\r\n-- ShadowHCR --\r\n")));	
    return;
}
#endif

/*****************************************************************************/
/*                                                                           */
/* NAME                                                                      */
/*      _ONBL2                                                               */
/* DESCRIPTION                                                               */
/*      Call ShadowOS() to copy / decompress OS image                        */
/* PARAMETERS                                                                */
/*      nVol                                                                 */
/*          Volume Number                                                    */
/* RETURN VALUES                                                             */
/*      None                                                                 */
/* NOTES                                                                     */
/*                                                                           */
/*****************************************************************************/
UINT32 
_ONBL2(UINT32 nVol)
	{
    UINT32 pJumpAddr;
	
	ONBL2_INF_PRINT((TEXT("\r\nEntering ONBL2\r\n")));
    pJumpAddr = ShadowOS(nVol);
    ONBL2_INF_PRINT((TEXT("Image Shadowing finished. Image located at 0x%x.\r\n"), pJumpAddr));
    
#ifdef SUPPORTS_SMR
    ShadowHCR(nVol);
    ONBL2_INF_PRINT((TEXT("HCR Image Shadowing finished.\r\n")));
#endif    	

    return pJumpAddr;
	}

/*****************************************************************************/
/*                                                                           */
/* NAME                                                                      */
/*      Main                                                                 */
/* DESCRIPTION                                                               */
/*      Main NBL2 function                                                   */
/* PARAMETERS                                                                */
/*      None                                                                 */
/* RETURN VALUES                                                             */
/*      None                                                                 */
/* NOTES                                                                     */
/*                                                                           */
/*****************************************************************************/
extern "C" void loadmain()
	{
    _PrintBanner();

    bootos((TUint32)_ONBL2(0)); /* Volume = 0 */

    while(1);
	}   
