/**
* Copyright (c) 2007-2009 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:
* Trace API
*
* WARNING: This file contains some APIs which are internal and are subject
*          to change without notice. Such APIs should therefore not be used
*          outside the Kernel and Hardware Services package.
*/



/**
 @file
 @publishedPartner
 @prototype
*/


#ifndef E32UTRACE_BASIC_TYPES_H
#define E32UTRACE_BASIC_TYPES_H

#ifdef __KERNEL_MODE__
#include <kernel/kernel.h>
#else //__KERNEL_MODE__
#include <e32std.h>
#endif //__KERNEL_MODE__


/**
@file
@publishedPartner
@prototype
*/

namespace UTF
{

/**
 * Every trace point must be statically assigned a classification. This is
 * essentially a label that allows identification of the intended use of
 * the trace packets generated by the trace point. The classification either
 * reflects a system-wide Tracing use-case, such as highlighting reasons why
 * a Panic occurred, or is assigned to trace points in performance critical
 * software such as the kernel or other key services.
 * @see TClassificationRange for more information.
 */
typedef TUint8  TClassification;

/**
 * The maximum possible value for TClassification
 */
const static TClassification KMaxClassification = 255; // 2^8 - 1

/**
 * Each trace point must be statically assigned a ModuleUid to indicate the
 * module in which the trace point is defined. It is recommended that this
 * value is the UID3 of the associated binary file.
 *
 * The ModuleUid and Classification attributes of a trace point are independent.
 */
typedef TUint32 TModuleUid;

/**
 * The maximum possible value for TModuleUid
 */
const static TModuleUid KMaxModuleUid = 0xFFFFFFFF; // 2^32 - 1, or 4294967295 - 1

/**
 * The EXECUTABLE_DEFAULT_MODULEUID can be used to define your own default
 * specific ModuleUid. This is done by defining the macro to be your new
 * default value.
 */
#ifdef EXECUTABLE_DEFAULT_MODULEUID
#define FW_DEFAULT_MODULEUID EXECUTABLE_DEFAULT_MODULEUID
#else
#define FW_DEFAULT_MODULEUID TTraceContext::DefaultModuleUid()
#endif

/**
 * This is a numerical value statically assigned to a trace point which will be used
 * on the host to look-up the format of the associated trace packets.
 *
 * The meaning of a FormatId is specific to the ModuleUid and Category of the
 * associated trace point.
 */
typedef TUint16 TFormatId;

/**
 * The maximum possible value for TFormatId
 */
const static TFormatId KMaxFormatId = 65535; // 2^16 - 1

/**
 * TFormatId used in packets produced by the Print and Printf
 * functions.
 *
 * Note that this format should not be used on the
 * device by clients of UTrace. This symbol is only marked
 * as published to partners to give host side tools access to
 * it.
 *
 * @see TFormatId
 */
const static TFormatId KFormatPrintf = 0;

/**
 * TFormatId used in packets produced by the Print and Printf
 * functions for unicode descriptors.
 *
 * Note that this format should not be used on the
 * device by clients of UTrace. This symbol is only marked
 * as published to partners to give host side tools access to
 * it.
 *
 * @see TFormatId
 */
const static TFormatId KFormatPrintfUnicode = 1;

/**
 * This value is intended to be used by clients to specify the
 * start of the range of enums used to define their format ids.
 *
 * Any values between 0 up to this should not be used directly
 * by clients of UTrace on the device.
 *
 * @see TFormatId
 */
const static TFormatId KInitialClientFormat = 512;

/**
 * Include the thread identification into the trace packet at run-time.
 * The thread identification is used as an identifier to resolve
 * thread and process names in conjunction with
 * classificiation EThreadIdentification = 3.
 */
enum THasThreadIdentification
	{
	/**Do add the thread identification to the trace packet*/
	EAddThreadIdentification = ETrue,
	/**Don't add the thread identification*/
	ENoThreadIdentification = EFalse
	};


/**
 * The EXECUTABLE_DEFAULT_HAS_THREAD_IDENTIFICATION can be used to
 * define the default setting for adding or not adding the thread
 * identification in a trace packet. This is done by defining
 * the macro to be your new default value.
 */
#ifdef EXECUTABLE_DEFAULT_HAS_THREAD_IDENTIFICATION
#define FW_DEFAULT_HAS_THREAD_IDENTIFICATION EXECUTABLE_DEFAULT_HAS_THREAD_IDENTIFICATION
#else
#define FW_DEFAULT_HAS_THREAD_IDENTIFICATION EAddThreadIdentification
#endif


/**
 * Add the program counter into the trace packet at run-time.
 * The program counter is used to indicate where the CPU is in the
 * instruction sequence. This can be used to locate the line of code
 * or routine the trace was sent from.
 */
enum THasProgramCounter
	{
	/**Do add the program counter to the trace packet.*/
	EAddProgramCounter = ETrue,
	/**Don't add the program counter*/
	ENoProgramCounter = EFalse
	};


/**
 * The EXECUTABLE_DEFAULT_HAS_PC can be used to
 * define the default setting for adding or not adding the
 * program counter in a trace packet. This is done by defining
 * the macro to be your new default value.
 */
#ifdef EXECUTABLE_DEFAULT_HAS_PC
#define FW_DEFAULT_HAS_PC EXECUTABLE_DEFAULT_HAS_PC
#else
#define FW_DEFAULT_HAS_PC ENoProgramCounter
#endif


/**
 * The Classifications in the All range should be used by the majority of
 * trace points. This range of Classifications are intended to identify which
 * of the most common trace use-cases a trace point is contributing to.
 * The Classifications in this series are defined solely by Symbian but are
 * intended for use by any software on a device.
 *
 * @see TClassification
 * @see TClassificationRange
 * @see EAllRangeFirst
 */
enum TClassificationAll
	{
	/**
	 * Used when a panic has occurred or when providing information on the execution
	 * state that lead to the decision to panic.
	 *
	 * A trace point with this Classification indicates a fatal condition is about to
     * occur which will halt the flow of program execution in the current thread.
	 *
	 * This Classification also provides information describing where a panic has been
	 * dealt with.
	 *
	 * EPanic = EAllRangeFirst
	 */
	EPanic = 192,

	/**
	 * Used when an error has occurred that means the current operation cannot continue
	 * but isnt sufficiently serious to cause a Panic. The trace points could contain
	 * not just the error code but any relevant information about the execution state
	 * when the error occurred.
	 *
	 * To be used for all types of error and includes situations where the errors are
	 * returned from a function or in a Leave.
	 *
	 * This Classification also provides information describing where an error has been
	 * dealt with.
	 */
	EError = 193,

	/**
	 * Used when something unexpected or unusual has occurred that does not stop the
	 * current operation from happening but may result in unintended side effects or
	 * actual errors later on.
	 */
	EWarning = 194,

	/**
	 * Used to detail normal activity at the edges of a module. Does not include errors
	 * or warnings as these are covered in other Classifications.
	 *
	 * Includes data about exported or published functions defined by module as well as
     * calls out of the module to get significant information. Exactly what is significant
     * is for the module owner to decide. For example, getting the contents of an .ini file to
     * determine which configuration to use might be significant but calling RArray::Count()
     * would not be.
	 *
	 * The information in this Classification should be enough to allow someone unfamiliar
	 * with the trace module to get a high level understanding of what functionality it has
	 * executed.
	 */
	EBorder = 195,

	/**
	 * Intended for tracing the state transitions of an application or service such as those
	 * performed by a machine.
	 *
	 * Trace packets using this Classification should contain the name of the
	 * changed state variable and the new value.
	 */
	EState = 196,

	/**
	 * Used to provide detailed information about the normal activity of a module
	 * to help a developer, who is familiar with the module, to understand what it is doing.
	 *
	 * Does not include errors or warnings as those are covered in other Classifications.
	 */
	EInternals = 197,

	/**
	 * Used when there is a need to output large amounts of data through individual trace
	 * points that would likely cause significant intrusion if included under one of the
	 * other Classifications.
	 *
	 * This Classification in intended to be used in conjunction with the Internals
	 * Classification to provide more details when debugging a specific module.
	 */
	EDump = 198,

	/**
	 * Used to provide comprehensive information on what paths the execution takes within
	 * functions.
	 *
	 * This Classification is intended only to be assigned by tools that add temporary
	 * instrumentation points specifically to output this data.
	 */
	EFlow = 199,

	/**
	 * Used to output data about the execution time, memory usage, disk usage, power
	 * utilisation and other system characteristics of the trace module.
	 *
	 * This data may need to be processed before it can provide affective metrics. E.g.
	 * the time between two timestamps might need to be computed.
	 *
	 * Intended only to be used to output system characteristic data that requires the
	 * smallest possible intrusion.
	 */
	ESystemCharacteristicMetrics = 200,

	/**
	 * Can be used when adding ad-hoc / temporary trace points if theres a need to
	 * distinguish it from existing trace.
	 */
	EAdhoc = 201,

	/**
	 * Provided to allow the following compile time assert:
	 * EClassificationAllHighWaterMark <= EAllRangeLast + 1
	 *
	 * @internalComponent
	 */
	EClassificationAllHighWaterMark,
	};


/**
 * The division of the Classifications into different ranges aims to manage the
 * contention for the namespace.
 * The stakeholders considered for the namespace are:
 * - Software vendors who own trace points in a base-port, whether a hardware
 *   manufacturer or their suppliers.
 * - Software vendors who own trace points in the UI layer, whether a phone
 *   manufacturer or their suppliers.
 * - ISVs who own trace points in source code installed on a device after it has been
 *   released.
 * - Symbian as the owner of all trace points in Symbian OS.
 *
 * It is recommended that all stakeholders use the All range as far as possible.
 * It is also strongly recommended that software provided by one stakeholder does
 * not use any Classifications from a range provided specifically for another
 * stakeholder.
 *
 * @see TClassification
 */
enum TClassificationRange
	{
	/**
	 * The Symbian One range is only for use on the device by Symbian.
	 *
	 * @see epoc32\include\e32btrace.h
	 *
	 * This range is defined and used by Symbian only.
	 */
	ESymbianOneRangeFirst = 0,

	/**
	 * @see ESymbianOneRangeFirst
	 */
	ESymbianOneRangeLast = 127,

	/**
	 * The Base Port range is only for use on the device by code provided solely as
	 * part of a base-port.
	 *
	 * It is expected that the owner of each base-port will manage the use of trace
	 * points by all the software vendors contributing to their base-port to prevent
	 * conflicts.
	 *
	 * This range may only be defined and used by device manufacturers.
	 */
	EBasePortRangeFirst = 128,

	/**
	 * @see EBasePortRangeFirst
	 */
	EBasePortRangeLast = 159,

	/**
	 * The UI Layer range is only for use on the device by code provided as part of
	 * the UI Layer. This includes all software needed to create a device
	 * that is not part of Symbian OS or the base-port.
	 *
	 * It is expected that the owner of each UI layer will prevent collisions between
	 * trace points provided by all the software vendors contributing software to the
     * UI layer.
	 *
	 * This range may only be defined and used by device manufacturers.
	 */
	EUiLayerRangeFirst = 160,

	/**
	 * @see EUiLayerRangeFirst
	 */
	EUiLayerRangeLast = 191,

	/**
	 * The Classifications in the All range should be used by the majority of trace
	 * points. This range of Classifications are intended to identify which of the
	 * most common trace use-cases a trace point is contributing to.
	 *
	 * The Classifications in this series are defined solely by Symbian but are
	 * intended for use by any software on a device. No software vendor other than
	 * Symbian should define Classifications in this range.
	 *
	 * When assigning one of these Classifications to a trace point you should
	 * also assign a ModuleUid to allow your trace points to be differentiated from
	 * other uses of the Classification in the system.
	 *
	 * These Classifications should only be enabled at run-time if the filtering on
	 * ModuleUids functionality is also enabled. This is to avoid accidentally causing
     * trace live-locks from occurring when the Classification is enabled. This could
     * happen because trace points in components involved in the current trace output
     * path might also be assigned these Classifications. Filtering on ModuleUids means
     * that those trace points can be activated only when its known to be safe to do
     * so and not accidentally enabled with a Classification.
	 *
	 * This range is definded by Symbian and may be used by anyone.
	 *
	 * @see TAllClassification
	 */
	EAllRangeFirst = 192,

	/**
	 * @see EAllRangeFirst
	 */
	EAllRangeLast = 221,

	/**
	 * This range may only be used and defined by Symbian.
	 */
	ESymbianTwoRangeFirst = 222,

	/**
	 * @see ESymbianTwoRangeFirst
	 */
	ESymbianTwoRangeLast = 245,

	/**
	 * The ISV range is only for use on the device by ISVs providing software that is
	 * installed on a device after it has been released and is not part of the base-port,
     * Symbian OS or the UI Layer.
	 *
	 * This range is only intended to provide enough Classifications for a single ISV
	 * application. Clashes that occur between different applications should be
	 * resolved by uninstalling one or other of them.
	 *
	 * This range is defined and used by 3rd party developers.
	 */
	EIsvRangeFirst = 246,

	/**
	 * @see EIsvRangeFirst
	 */
	EIsvRangeLast = 253,

	/**
	 * Only for use on the device by test code.
	 *
	 * Trace points with these Classifications should not be released as part of a
	 * production device.
	 */
	ETestingRangeFirst = 254,

	/**
	 * @see ETestingRangeFirst
	 */
	ETestingRangeLast = KMaxClassification,
	};

// Check high water mark for classification ranges for TClassificationAll
__ASSERT_COMPILE(EClassificationAllHighWaterMark <= EAllRangeLast + 1);



/**
 * @see TClassification
 * @see ESymbianTwoRangeFirst
 * @internalComponent
 */
enum TClassificationSymbianTwo
	{
	/**
	 * This classification is reserved for future use to allow the classification range
	 * to be expanded to cover more than current 256 different values.
	 *
	 * If trace is output on this Classification at some point in the future then this
	 * indicates another mechanism (to be decided) will be used to indicate the actual
	 * Classification for the trace.
	 *
	 * EClassificationExtension = 222
	 */
	EClassificationExtension = ESymbianTwoRangeFirst,

	/**
	 * Provided to allow the following compile time assert:
	 * EClassificationSymbianTwoHighWaterMark <= ESymbianTwoRangeLast + 1
	 *
	 * @internalComponent
	 */
	EClassificationSymbianTwoHighWaterMark,
	};
// Check high water marks for classification ranges
__ASSERT_COMPILE(EClassificationSymbianTwoHighWaterMark <= ESymbianTwoRangeLast + 1);

/**
 * @see TClassification
 * @see ETestingRangeFirst
 * @test
 */
enum TClassificationTesting
	{
	/**
	 * Only for use on the device by test code.
	 *
	 * Trace points with this Classification should not be released as part of a
	 * production device.
	 *
	 * ETesting1 = 254
	 */
	ETesting1 = ETestingRangeFirst,

	/**
	 * @see ETesting1
	 */
	ETesting2 = 255,

	/**
	 * Provided to allow the following compile time assert:
	 * EClassificationTestingHighWaterMark <= ETestingRangeLast + 1
	 *
	 * @internalComponent
	 */
	EClassificationTestingHighWaterMark,
	};

// Check high water marks for classification ranges
__ASSERT_COMPILE(EClassificationTestingHighWaterMark <= ETestingRangeLast + 1);


} //end of UTF namespace

#endif //E32UTRACE_BASIC_TYPES_H

