// Copyright (c) 2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
//

/**
 @file
 @internalTechnology
 @released
*/

#ifndef __SUPL_TI_API_H__
#define __SUPL_TI_API_H__

#include <e32base.h>

#include <lbs/lbsnetcommon.h>


class MLbsSuplTiObserver;

/**
SUPL Terminal Initiation Module API 

An API to enable a terminal client to determine terminal's position information using SUPL protocol and retrieve it.

The purpose of the API is to provide a dividing line between the SUPL OS and middleware layers, where typically the client
will be the OS side LBS SUPL proxy protocol module. Terminal initiation modules are written to support the API to provide
a wrapper around the middleware components.
*/
class CLbsSuplTiApi : public CBase
	{

public:
	/**
	Creates an instance of the CLbsSuplTiApi implementation with the given Uid.

	@param aObserver is a pointer to the initiation module's observer. The module
			MUST call MLbsSuplTiObserver::RequestComplete() once the location request
			has been completed.
	@param aEcomModuleId is the UID of the ECOM module to load.
	@return a pointer to an instance of the interface class implemented by the
			initation module. The module should return NULL if the specified ECOM module
			could not be loaded.
	
	@see MLbsSuplTiObserver
	*/
	IMPORT_C static CLbsSuplTiApi* NewL(MLbsSuplTiObserver& aObserver, TUid aEcomModuleId);


	/**
	Standard destructor that handles the ECom destruction.

	The client invokes the destructor on the initation module
	when it wishes to unload the module.
	
	The module should terminate all current requests and close
	any connections it may have.

	*/  
	IMPORT_C ~CLbsSuplTiApi();

	 
	/** 
	RequestLocation is an asynchronous call to determine the terminal's current
	position information or to determine additional positioning information such as
	assistance data.
	
	When the initiation module has completed the request it MUST signal this by calling
	MLbsSuplTiObserver::RequestComplete().

	The module may support one or more concurrent location requests, where by each request
	is identified by the aSessionId parameter. This session-id MUST be used when exchanging
	information with the SUPL server framework and returned by MLbsSuplTiObserver::RequestComplete().
	
	The client can cancel this request using CLbsSuplTiApi::CancelRequest().

	@param aSessionId is the Id of the location request. This is generated by the client and
			must be used by the initiation module when it performs actions releated to that
			request.
	@param aOptions should be of type TLbsNetPosRequestOptions.
		   It provides information about the location request, including quality of the location
		   required, and a flag to indicate this is the first request made by the client.
	@param aMethod This parameter contains a list of the positioning methods that should be used
	       to obtain the device's position.
	
	@see TLbsNetSessionId
	@see TLbsNetPosRequestOptions
	@see TLbsNetPosRequestMethod
	*/
	virtual void RequestLocation(const TLbsNetSessionId& aSessionId, const TLbsNetPosRequestOptionsBase& aOptions, const TLbsNetPosRequestMethod& aMethod) = 0;

	
	/**
	Cancels an outstanding asynchronous location request. Cancelling
	requests is typically attempted when an client is closing down.

	The initiation module MUST call the MLbsSuplTiObserver::RequestComplete()
	with either KErrCancel or KErrNone to indicate if the cancel
	was successful or not. However the call to MLbsSuplTiObserver::RequestComplete()
	is NOT required if this function returns an error.
		
	@param aSessionId The Id of the location request to be cancelled. This must be the same id used
	       during the location request.
	@return an error code. A initiation module should return KErrNotFound for invalid session ids.

	@see TLbsNetSessionId
	*/
	virtual TInt CancelRequest(const TLbsNetSessionId& aSessionId) = 0;

	
	/**
	This should be called when the request session is no longer required and prior to any call to ~CLbsSuplTiApi().
	All the outstanding requests must be cancelled before closing the session.
	
	The initiation module should use this opportunity to close any session and server connections it
	may have with any SUPL server framework.
	*/
	virtual void CloseSession() = 0;


	/**
	This is an synchronous method for retrieving position information  
	determined during RSuplTiApi::RequestLocation().

	This should be called by the client once the location request has completed, because
	the location information will not be available or current.
	
	The initiation module should return location information associated with the correct
	session given by the session ID.
	
	@param aSessionId the ID of session from which to obtain the location information.
	@param aPositionInfo the returned location information.
	@return an error code for the location information.

	@see TLbsNetSessionId
	@see TPositionInfoBase
	*/
	virtual TInt GetPosition(const TLbsNetSessionId& aSessionId, TPositionInfoBase& aPositionInfo) = 0;

protected:
    
	/**
	Constructor for the CLbsSuplTiApi
    
	@param aObserver is a pointer to the initiation module's observer, which is
			used to respond to requests.
    
	@see MLbsSuplTiObserver    
	 */
	IMPORT_C CLbsSuplTiApi(MLbsSuplTiObserver& aObserver);

	/**
	Leaving constructor for CLbsSuplTiApi. If required.
	*/
	IMPORT_C void ConstructL();

protected:
	/** Observer. */
	MLbsSuplTiObserver& iSuplTiObserver;
	
private:
	/** Instance identifier key. */
	TUid iDtor_ID_Key;
	};

/**
The MlbsSuplTiObserver class is used in partnership with CLbsSuplTiApi class. The
observer interface is used by a initiation module to inform the client of the location
request completion.
*/
class MLbsSuplTiObserver
	{
public:

	/**
	RequestComplete is used by the initiation module to inform the client
	of the completion of the location request.

	@param aReason is the return/error code the completed location request.
	@param aSessionId is the id of the completed request.
	*/
	virtual void RequestComplete(TInt aReason, const TLbsNetSessionId& aSessionId) = 0;
	};


#endif // __SUPL_TI_API_H__
