// Copyright (c) 2006-2009 Nokia Corporation and/or its subsidiary(-ies).
// All rights reserved.
// This component and the accompanying materials are made available
// under the terms of "Eclipse Public License v1.0"
// which accompanies this distribution, and is available
// at the URL "http://www.eclipse.org/legal/epl-v10.html".
//
// Initial Contributors:
// Nokia Corporation - initial contribution.
//
// Contributors:
//
// Description:
// This file defines the interface to the LBS protocol module. 
// The interface is implemented by one or more implementation modules.
// 
//

/**
 @file
 @publishedPartner
 @released
*/

#ifndef LBSNETPROTOCOLBASE_H_
#define LBSNETPROTOCOLBASE_H_

#include <lbs/lbsassistancedatabase.h>
#include <lbs/lbsnetcommon.h>

class TPositionInfoBase;
class RLbsAssistanceDataBuilderSet;
class TLbsExternalRequestInfo;
class TLbsNetProtocolModuleParams;

class TLbsNetSessionId;
class TLbsNetPosRequestPrivacy;
class TLbsNetPosRequestQuality;
class TLbsNetPosRequestMethod;

/**
The MLbsNetworkProtocolObserver class is used in partnership with
CLbsNetworkProtocolBase. The observer interface is used by a protocol
module to forward network requests and information to the LBS subsystem.
It is also the interface that a protocol module uses to provide responses
to requests made by the LBS subsystem via CLbsNetworkProtocolBase. 

When a network protocol module makes a request via the observer API the
LBS subsystem provides a response via the corresponding notification
method in the CLbsNetworkProtocolBase class. Although all the interface
methods in the MLbsNetworkProtocolObserver are synchronous they
will return quickly. The LBS subsystem will internally queue requests and
process them asynchronously. 

For example, when the network sends a message requesting the current
location of the handset, the protocol module should forwarded this request
via the method ProcessLocationRequest(). When the current position is
eventually calculated, the LBS subsystem informs the protocol
module by calling MLbsNetworkProtocolObserver::RespondLocationRequest(). 

Calls from the protocol module to the LBS subsystem may be unsolicited
and/or in response to requests made by the observer. For example, if
the subsystem requests GPS assistance data via
CLbsNetworkProtocolBase::RequestAssistanceData() the protocol module
will respond some time later via ProcessAssistanceData(). However, the
protocol module is also allowed to provide assistance data updates at
any time and not just in response to the requests from the observer. 

@see CLbsNetworkProtocolBase 

@publishedPartner
@released
*/
class MLbsNetworkProtocolObserver
	{
public:
	/**
	Values of type TLbsNetProtocolService describe the processes that
	can be active or queued in network protocol module.

	A single value is supplied as a parameter in ProcessLocationRequest() to indicate
	why the location request was generated.
	
	ProcessStatusUpdate() provides a mask of TLbsNetProtocolService values to
	indicate all the various tasks being performed or queued by the protocol module.
	
	@see ProcessLocationRequest()
	@see TLbsNetProtocolServiceMask
	@see ProcessStatusUpdate()
	*/
	enum TLbsNetProtocolService
		{
			/** No service */
		EServiceNone                = 0x00,
	
			/** Indicates that the protocol module has currently a request from
			the handset to perform a network calculation of the devices position. */
		EServiceNetworkLocation		 = 0x01,
		
			/** Indicates that the protocol module has currently a request to
			perform a self location. */
		EServiceSelfLocation   		 = 0x02,
		
			/** Indicates that the protocol module has currently a MT-LR request
			by an	external party.
			This initially results in the protocol module issuing a privacy request, that
			contains the details of	the request originator, followed by a location request. */
		EServiceMobileTerminated     = 0x04,
		
			/** Indicates that the protocol module has currently a request to 
			to send the device's current position to a remote third party. */
		EServiceTransmitThirdParty   = 0x08,
		
			/** Indicates that the protocol module has currently a location request
			that was generated by the network (NI-LR). In this situation, the protocol module
			will only generate a location request. There is no preceding privacy request. */
		EServiceNetworkInduced       = 0x10,
			/** Indicates that the protocol module has currently a location request
			that was generated by the network (Triggered MOLR, SUPL 2.0).
			 */
		EServiceTriggeredMolr      = 0x20		
		};
	
	/**
	TLbsNetProtocolServiceMask is used to hold a bit mask of values of TLbsNetProtocolService.
	This is used by ProcessStatusUpdate() to indicate the tasks currently being processed
	or queued by the network protocol module

	@see ProcessStatusUpdate()
	@see TLbsNetProtocolService
	*/
	typedef TUint32 TLbsNetProtocolServiceMask;
	
public:
	/**
	Get the current version of the observer interface

	*/
	IMPORT_C virtual TVersion Version() const;

	/**
	Retrieves from the LBS subsystem the current capabilities of the device.

	GetCurrentCapabilities() is used by the protocol module to obtain details of
	what types of network request the device is able to respond to. For example,
	this might include terminal based GPS but not terminal assisted.

	The protocol module should call GetCurrentCapabilities() each time it needs
	to obtain the device's capabilities. The capabilities may vary as the user
	or network configures the device.
	
	The device may support more than one positioning method, in which case the
	first one in the list is the preferred positioning method for the device. 
	The preferred positioning method is the method returned by calling 
	aCapabilities.GetPosMethod(0, method).

	@param aCapabilities Returns the current capabilities of the device.
	*/
	IMPORT_C virtual void GetCurrentCapabilities(
										TLbsNetPosCapabilities& aCapabilities) const = 0;
	/**
	Provides the LBS subsystem with details of the current status of the network protocol
	module

	ProcessStatusUpdate() provides a bit mask of values from TLbsNetProtocolService. The
	bit-mask indicates which services are currently being handled by the protocol module.

	For example, if the protocol module is currently processing a Mobile Terminated
	Location Request and a local request for assistance data, aActiveServiceMask will be
	set to EServiceMobileTerminated + EServiceAssistanceData.

	The method is called each time the services that are active in the protocol module
	change.


	@param aActiveServiceMask This parameter indicates which services are currently
	active in the the protocol module. The parameter contains a bit-mask of values
	from TLbsNetProtocolService. If the protocol module is idle, the parameter is set
	to zero.
	*/
	IMPORT_C virtual void ProcessStatusUpdate(
										TLbsNetProtocolServiceMask aActiveServiceMask) = 0;

	/**
	Ask the LBS subsystem to process a privacy request from the network.

	Privacy requests (for example an LCS-Notification) are typically the first stage of a Mobile
	Terminated Location Request (MT-LR). They are received from the network to determine if the
	handset is willing to reveal its location to an external third-party. 

	ProcessPrivacyRequest() supplies information about the external requestor but does not
	actually ask the device for its position. If the LBS subsystem responds indicating it is
	willing to reveal its location, the network will normally send the second stage of the MT-LR.
	This follow-on message is forward by the protocol module via a separate method
	(ProcessLocationRequest) and is the trigger for the LBS subsystem to calculate the current
	position. 

	The ProcessPrivacyRequest() method will return quickly. However, it does not indicate if
	request should be accepted or rejected. After the LBS subsystem has determined the user's
	preferences, it responds to the protocol module via a call to
	CLbsNetworkProtocolBase::RespondPrivacyRequest(). The response indicates if handset is
	willing to accept a location request. 

	If the terminal does indicate it is willing to accept a location request, the network will
	typically send a follow-on request (measurement control) that asks for the devivce's
	location. The protocol module forwards this second request via the method
	ProcessLocationRequest(). 

	In all subsequent requests and responses, the same unique session ID (aSessionId) must be used
	in order to connect the various requests and responses. 


	@param aSessionId The Id of the privacy request. This is generated by the protocol module and is used by
	       the LBS subsystem when it responses to the privacy request via
	       CLbsNetworkProtocolBase::RespondPrivacyRequest(). The same session identifier must be
	       also be used by the protocol module if and when it receives the corresponding location request
	       from the network. 

	@param aEmergency ETrue if this is an emergency privacy request, EFalse otherwise. 

	@param aPrivacy Privacy procedure requested by the network on how to advise the user,

	@param aRequestInfo The identity of the external third party that is requesting the handsets
	       position. 

	@see CLbsNetworkProtocolBase::RespondPrivacyRequest() 
	@see ProcessLocationRequest() 
	@see CLbsNetworkProtocolBase::RespondLocationRequest() 
	@see ProcessSessionComplete() 
	@see TLbsExternalRequestInfo 
	@see TLbsLocationNotification 
	*/
	IMPORT_C virtual void ProcessPrivacyRequest(
										const TLbsNetSessionId& aSessionId,
										TBool aEmergency, 
										const TLbsNetPosRequestPrivacy& aPrivacy,
										const TLbsExternalRequestInfo& aRequestInfo) = 0;
										
	/**
	Location (e.g. measurement control) requests are received from the network, instructing the
	terminal to calculate and return its current position. 

	When the network protocol module calls ProcessLocationRequest() the method will return
	quickly. This initiates a position calculation by the LBS subsystem. When the current
	position is available, the subsystem return the location to the module via a call to
	CLbsNetworkProtocolBase::RespondLocationRequest(). 

	The parameter aType indicates why the network has generated the location request and can take
	the following values: 

	(a) It is the second stage of a MT-LR. This is normally generated by the network if device has
	responded positively to the preceding privacy request. 

	(b) Location requests may also generated by the network when the terminal wishes to obtain
	assistance data - for example via a Mobile Originated Location Request (MO-LR). 

	(c) A request from the terminal to transmit its current location to a remote third-party (X3P)
	can also result in the network generating a location request. 

	(d) Under some location protocols, a request from the terminal for a network based
	location calculation (e.g. cell-id) may also result in the network asking the
	terminal for its position. 

	(e) It is also possible for the network to send location requests without any preamble. These
	Network Induced Location Requests (NI-LR) are generated by the network and instruct the
	terminal to calculate and return its current position. 

	The session ID parameter is used to connect a series of requests and responses. If this location
	request is the result of a MT-LR the session ID will be the same as that passed in the privacy
	request. 

	Similarly, when performing a transmit to third-party (X3P), the session ID of the location
	request will be the same as that used to initiate the transfer. 

	However, in the other cases the location request starts a new session. In this situation, the
	session ID will be unrelated to any previous request. 


	@param aSessionId The Id of the location request within the LBS sub-system. 

	@param aEmergency ETrue if this is an emergency request, EFalse otherwise. 

	@param aService Specifies the type of the service/session the request is related to.

	@param aQuality Stipulates both the desired accuracy of the location information and the maximum
	       time the network is prepared to wait. Zero values for these fields indicates there are no
	       accuracy or time constraints. 

	@param aMethod contains a list of the positioning methods that should be used to obtain the 
		   device's position in the network's order of preference

	Note:
	The aMethod parameter contains a network preferred list of positioning modes, however in order 
	to determine the final positioning mode the LBS subsystem is using more information, like 
	the default positioning mode ( @see CLbsAdmin); the number and the type of other positioning sessions
	and the capabilities of the AGPS positioning module (see the module configuration file).
 
	@see CLbsNetworkProtocolBase::RespondLocationRequest() 
	@see ProcessPrivacyRequest() 
	@see CLbsNetworkProtocolBase::RequestTransmitLocation() 
	@see CLbsNetworkProtocolBase::RequestSelfLocation() 
	@see CLbsNetworkProtocolBase::RequestNetworkLocation() 
	@see ProcessSessionComplete() 
	@see TLbsExternalRequestInfo 
	@see TLbsNetRequestQuality
	@see TLbsNetRequestMode 
	*/
	IMPORT_C virtual void ProcessLocationRequest(
										const TLbsNetSessionId& aSessionId,
										TBool aEmergency,
										TLbsNetProtocolService aService, 
										const TLbsNetPosRequestQuality& aQuality,
										const TLbsNetPosRequestMethod& aMethod)  = 0;

	/**
	The protocol module issues a single call to ProcessSessionComplete() to close a session and to
	return the final result code to the LBS subsystem. 

	The aReason parameter indicates the overall success or failure for the request. 

	The session ID parameter identifies the session that is being closed. This ID is allocated when
	a new request is created: 

	In the case of MT-LR, the session ID is created by the protocol module when it issues the initial
	privacy request (ProcessPrivacyRequest()). That session extends through the various
	responses and, if appropriate, the subsequent location request (ProcessLocationRequest()).
	The protocol modules issues a single call to ProcessSessionComplete() to indicate the overall
	success or failure of the MT-LR. 

	In normal situations, ProcessSessionComplete() will be invoked when the overall request has
	finished. However, it may occur at any stage - for example, in the case of a MT-LR the protocol
	module will complete the session immediately if there was an error transmitting the initial
	privacy response back to the network.

	Similarly, ProcessSessionComplete() is also used by the protocol module to terminate
	a request if a protocol conflict has arisen. For example, the location protocol may
	be unable to simulateously perform two requests.

	For a MT-LRs and NI-LRs the session ID are assigned by the network protocol module. For
	other requests, the session ID is generated by the LBS subsystem. In either situation, the
	assigned session ID is used by both the protocol module and the LBS subsystem to connect
	requests and responses arising from a particular request.



	@param aSessionId The Id of the request being closed. 

	@param aReason Reason for the completion of the request. KErrNone if the request is succecsfully
	       completed, or one of a range of error codes otherwise. 
	*/
	IMPORT_C virtual void ProcessSessionComplete(
										const TLbsNetSessionId& aSessionId,
										TInt  aReason) = 0;
	/**
	Assistance data may arrive as a result of the LBS subsystem requesting it via a call to
	the CLbsNetworkProtocolBase methods RequestSelfLocation() or RequestAssistanceData().
	However, assistance data can also be delivered unsolicited. For example, a MT-LR location
	request (e.g. measurement control) from the network will normally include a default set
	of assistance data. 
	
	This method can also be used by the network to request that the LBS subsystem deletes  
	all of its cached assistance data, or a subset of it. This is done by setting aReason to 
	KPositionAssistanceDataReset and aDataMask to specify which subset of assistance data must 
	be reset. The parameter aDataMask must not be 0, and it must be a valid bitmask of
	values specified by TLbsAssistanceDataItem. When using this method to reset assistance data, 
	the parameter aData can be left empty.


	@param aDataMask  Is a bitmask that specifies which subsets of GPS assistance data
	       are included in the aData parameter, or which subsets of GPS assistance data should 
	       be reset.

	@param aData A container that holds all of the GPS assistance data to be delivered. 

	@param aReason Delivery status for the GPS assistance data being supplied. KErrNone is passed if the
	       specified assistance data has been successfully retrieved from the network and delivered.
	       KPositionAssistanceDataReset can be passed to ask the LBS subsystem to reset its cached 
	       assistance data, or a subset of it. Otherwise, an error code is passed indicating the 
	       reason why the assistance data referenced in aData could not be obtained.  

	@see CLbsNetworkProtocolBase::RequestSelfLocation() 
	@see CLbsNetworkProtocolBase::RequestAssistanceData() 
	*/
	IMPORT_C virtual void ProcessAssistanceData(
										TLbsAsistanceDataGroup aDataMask,
										const RLbsAssistanceDataBuilderSet& aData,
										TInt aReason) = 0;
	
	/**
	Location Information may arrive as a result of the LBS subsystem requesting it
	via a call to CLbsNetworkProtocolBase::RequestNetworkLocation(). However,
	location information can be delivered unsolicited. For example, a location
	request (e.g. measurement control) from the network will typically include
	a reference location that is derived from the terminal's serving cell. The reference 
	location is injected at most once a session. This is valid even if assistance data 
	is reset during the positioning request in order to avoid impacting the standard 
	behaviour of the LBS sub-system when reset message is being received.

	The Session ID parameter indicates which request led to the location information being
	delivered. For a MT-LR, the session ID will be the ID assigned by the protocol module.
	This is the ID passed by the protocol module when it calls ProcessPrivacyRequest().

	If the location information was delivered as a result of a request generated
	by the terminal, the session ID should be the identifier used by the LBS subsystem
	when it initiated that request. This will be the ID passed to RequestSelfLocation(),
	RequestNetworkLocation() or RequestTransmitLocation() by the LBS subsystem.


	@param aSessionId The Id of the request under which the assistance data
	       has arrived.

	@param aPosInfo  Location details of terminal as provided by the network.
	       The parameter aPosInfo is normally of type TPositionInfo.       

	*/
	IMPORT_C virtual void ProcessLocationUpdate(
										const TLbsNetSessionId& aSessionId,
										const TPositionInfoBase& aPosInfo) = 0;

private:
	/**
	 * This methods is reserved for future expansion and should not be used.
	 * Currently not implemented.
	 */								
	IMPORT_C virtual TAny* ExtendedInterface(TInt aFunctionNumber, TAny* aPtr1, TAny* aPtr2);
	};



/**
The class is an extended version of MLbsNetworkProtocolObserver and should be used
together with @see CLbsNetworkProtocolBase2.

@publishedPartner
@released
*/
class MLbsNetworkProtocolObserver2 : public MLbsNetworkProtocolObserver
	{
public:
	IMPORT_C virtual TVersion Version() const;

	IMPORT_C virtual void ProcessAssistanceData(
										TLbsAsistanceDataGroup aDataMask,
										const RLbsAssistanceDataBuilderSet& aData,
										TInt aReason,
										const TLbsNetSessionIdArray& aSessionIdArray) = 0;
										
protected:
	// Note: The following methods are made protected to reinforce the fact that 
	//       they should no longer be used by protocol modules implementing the
	//       extended CLbsNetworkProtocolBase2 version of the interface. Please 
	//       use the extended versions as declared in the public section above.
	IMPORT_C virtual void ProcessAssistanceData(
										TLbsAsistanceDataGroup aDataMask,
										const RLbsAssistanceDataBuilderSet& aData,
										TInt aReason);

private:
	IMPORT_C virtual TAny* ExtendedInterface(TInt aFunctionNumber, TAny* aPtr1, TAny* aPtr2);

	};

	
/**
The CLbsNetworkProtocolBase is the abstract interface to a network protocol module. It is
used by the LBS susbstem to pass requests and responses to the module. 

The CLbsNetworkProtocolBase class is used in partnership with
MLbsNetworkProtocolObserver. The network protocol module responds to requests from the LBS
subsystem via that observer class. 

An actual implementation of a network protocol module will derive from
CLbsNetworkProtocolBase or one of its derived classes (see below) to provide the 
specific implementation of the interface methods. 

Although all the interface methods in the class are synchronous they must return immediately.
When the LBS subsystem makes a request via CLbsNetworkProtocolBase the protocol module must
queue the request internally and return control to the caller immediately. Afterwards, when
the protocol module provides a response it uses the corresponding notification method in the
MLbsNetworkProtocolObserver class. 

@see MLbsNetworkProtocolObserver 
@see MLbsNetworkProtocolBase2 

@publishedPartner
@released
*/
class CLbsNetworkProtocolBase : public CBase
	{
public:
	/**
	Creates an instance of the CLbsNetworkProtocolBase implementation with the given Uid.

	*/
	IMPORT_C static CLbsNetworkProtocolBase* NewL
			(
			TUid aImplementationUid,
			TLbsNetProtocolModuleParams& aParams
			);

	// wraps ECom object destruction
	/**
	Standard destructor that handles the ECom destruction.

	The LBS subsystem invokes the destructor on the protocol module
	when it wishes to unload the module. The protocol module should
	terminate all current requests close all connections.

	*/  
	IMPORT_C ~CLbsNetworkProtocolBase();

public:
	/** The response to a privacy request.
	
	@see MLbsNetworkProtocolObserver::ProcessPrivacyRequest()
	@see RespondPrivacyRequest()
	*/
	enum TLbsPrivacyResponse
		{
			/** Not Used */
		EPrivacyResponseUnknown = 0,
			/** Inform the network the terminal will accept its location request */
		EPrivacyResponseAccepted,
			/** Inform the network the terminal will reject its location request */
		EPrivacyResponseRejected,
			/** Inform the protocol module it should not respond to the location request.
			This option is currently not supported */
		EPrivacyResponseIgnored
		};
		
	/** The status of the Lbs positioning framework.
		It is a combined bitmask of information reported by all the Lbs framework components. */
	enum TLbsSystemStatus
		{
			/** No specific status information available */
		ESystemStatusNone				= 0x00,
			/** Inform the network at least one client of the Location Server is tracking
			    (the UpdateOptions interval is nonzero)
			@see RPositioner::SetUpdateOptions
			@see UpdateSelfLocationMode */
		ESystemStatusClientTracking	= 0x01

			// As it is a bitmask the following values should be 0x02, 0x04, 0x08 ...
 		};

public:

	/**
	Send a response to a privacy request from the network.

	RespondPrivacyRequest() is called by the LBS subsystem in response to the protocol module
	issuing a privacy request (MLbsNetworkProtocolObserver::ProcessPrivacyRequest()) on
	behalf of the network. 

	The method indicates to the protocol module whether the terminal is willing to accept or has
	rejected the network's request. 

	If the terminal has indicated it will accept a location request, the network will normally then
	issue a measurement control request. When the protocol module receives this follow-on
	request from the network it passed to the LBS subsystem via the
	MLbsNetworkProtocolObserver::ProcessLocationRequest() method. 

	The session ID parameter indicates which request is being responded to by the LBS subsystem.
	This same session ID must then be supplied by the protocol module if and when it issues a
	location request. 


	@param aSessionId Is the same session identifier was passed by the protocol module when it called
	       MLbsNetworkProtocolObserver::ProcessPrivacyRequest() to start the privacy request. 

	@param aResponse The response to the privacy request to send to the network. This indicates whether
	       the terminal has accepted or rejected the request for its current location. 

	@see MLbsNetworkProtocolObserver::ProcessPrivacyRequest() 
	@see TLbsPrivacyResponse 
	@see MLbsNetworkProtocolObserver::ProcessLocationRequest() 
	*/
	virtual void RespondPrivacyRequest(
									const TLbsNetSessionId& aSessionId,
									const TLbsPrivacyResponse& aResponse) = 0;
	
	/**
	RespondLocationRequest() is called by the LBS subsystem in response to the protocol module
	issuing a location request (MLbsNetworkProtocolObserver::ProcessLocationRequest()) on
	behalf of the network. 

	The method is used to return the terminal's current location to network via the protocol
	module. 

	If the terminal has indicated it will accept a location request the network will normally then
	issue a measurement control request. When the protocol module receives this follow-on
	request from the network it passed to the LBS subsystem via the
	MLbsNetworkProtocolObserver::ProcessLocationRequest() method. 

	The session ID parameter indicates which request is being responded to by the LBS subsystem.
	This same session ID must then be supplied by the protocol module if and when it issues
	a location request. 


	@param aSessionId [In] Contains the session identifier - indicating which location request is being
	       responded too. This is the same session ID passed by the network module to the LBS
	       subsystem when the module initiated the position calculation via a call to
	       MLbsNetworkProtocolObserver::ProcessLocationRequest(). 

	@param aReason [In] KErrNone if aPosition holds best location information available; 
           KErrCancel if the session has been cancelled explicitly by a user or by an application 
           acting on behalf of a user. 
	       KErrAccessDenied if the terminal has decided to reject the location request, 
	       or the  corresponding (and previous) privacy request, or it is unwilling to accept 
	       network induced location requests (NI-LRs); 
	       KErrInUse if LBS is currently busy processing another request(s); 
	       KErrNotSupported if the positioning mode(s) requested in the corresponding location 
	       request is not supported by the LBS subsystem or the AGPS Module.
	       
	@param aPosInfo [In] Provides information about the current location of terminal - provided the
	       parameter aReason is set to KErrNone. Otherwise, aPosInfo is unassigned. Normally,
	       aPosInfo will be of type TPositionInfo.

	@see MLbsNetworkProtocolObserver::ProcessLocationRequest() 
	@see MLbsNetworkProtocolObserver::ProcessPrivacyRequest() 
	@see MLbsNetworkProtocolObserver::ProcessSessionComplete() 
	@see TPositionInfoBase 
	@see TPositionInfo 
	*/
	virtual void RespondLocationRequest(
									const TLbsNetSessionId& aSessionId,
									TInt  aReason,
									const TPositionInfoBase& aPosInfo) = 0;

	/**
	RequestTransmitLocation() is called by the LBS subsystem to instruct the protocol module to
	send the device's position to a remote third-party. 

	This method automatically cancels any outstanding transmit request and replaces it with the
	new request. 

	The parameter aSessionId is generated by the LBS subsystem and is used connect all
	corresponding responses and further requests. The same session ID will be used the LBS
	subsystem itself in subsequent calls to the protocol module via CLbsNetworkProtocolBase. In
	return, it the same session ID must also be supplied by the protocol module when it
	responds to the LBS subsystem or issues related requests via MLbsNetworkProtocolObserver. 

	For example, a request to transmit the location will normally result in the network
	generating a location request to determine the terminal's current position. This location
	request is passed to the LBS subsystem via
	MLbsNetworkProtocolObserver::ProcessLocationRequest() using the session-id as the
	initiating RequestTransmitLocation(). 

	Note: If the LBS subsystem does wish to cancel a currently active transmit request and
	immediately replace it with a new request it should only use RequestTransmitLocation(). It
	should not attempt to explicitly cancel the outstanding request using
	CancelTransmitLocation(). Issuing a cancel and then submitting a subsequent transmit
	request can lead to inefficiencies at the network protocol level. 


	@param aSessionId Contains the session identifier. The session identifier is generated by the LBS
	       susbsystem and is used in all corresponding and subsequent responses and requests. 

	@param aDestination Specifies the remote third-party that is the target for the location
	       information transfer. 

	@param aPriority A protocol specific value that determines the precedence of the request. A high
	       priority may result in the currently active request being stopped. 

	@see CancelTransmitRequest() 
	@see MLbsNetworkProtocolObserver::ProcessLocationRequest() 
	@see MLbsNetworkProtocolObserver::ProcessSessionComplete() 
	*/
	virtual void RequestTransmitLocation(
									const TLbsNetSessionId& aSessionId,
									const TDesC& aDestination,
									TInt aPriority) = 0;

	/**
	CancelTransmitLocation() is called by the LBS subsystem to instruct the protocol module to
	stop the transfer of the device's position to a remote third-party. 

	Note: If the LBS subsystem wishes to stop the current transfer request and immediately submit
	an alternative request it should not use CancelTransmitLocation(). Instead,
	RequestTransmitLocation() should be used. Submitting a new transmit request,
	automatically cancels any outstanding transfer. 

	If the protocol module has sucessfully terminated a transfer request it will call
	MLbsNetworkProtocolObserver::ProcessSessionComplete() with a completion code of
	KErrNone. 

	If a cancel request was too late, MLbsNetworkProtocolObserver::ProcessSessionComplete()
	will be called as normal indicating the success or failure of the transmission. 


	@param aSessionId Identifies which transmit location request is to be stopped. 

	@param aReason Indicates why the LBS subsystem wants the transfer terminated. 

	@see RequestTransmitLocation() 
	@see MLbsNetworkProtocolObserver::ProcessSessionComplete() 
	*/									
	virtual void CancelTransmitLocation(
									const TLbsNetSessionId& aSessionId,
									TInt  aReason) = 0;
	
	/**
	RequestAssistanceData() is used by the LBS subsystem to request that the specified subsets of
	assistance data are downloaded from the network. 

	When assistance data is received from the network the protocol module forwards the
	information to the LBS subsystem via the method MLbsNetworkProtocolObserver::ProcessAssistanceData().

	Each call to RequestAssistanceData() supersedes any previous request. The protocol module
	has only to attempt to satisfy the current set of requested assistance data. 

	If the method is called with a data of zero, this indicates that no assistance data is
	currently required. If the protocol module is currently retrieving assistance data a mask
	of zero may allow it to cancel that request - provided the purpose of that active request
	is only to retrieve assistance data.

	If the network does not delivered the full set of requested assistance data, the protocol
	module is required to re-issue a request for the missing data. However, the protocol module
	should forward assistance data as and when it arrives from the network. This may result in the
	protocol module calling MLbsNetworkProtocolObserver::ProcessAssistanceData()
	multiple times with the assistance data being delivered to the LBS subsystem in batches. 

	If it has not been possible to retrieve any part of the requested assistance data, the protocol
	module indicates which subsets are missing by also calling
	MLbsNetworkProtocolObserver::ProcessAssistanceData() with the appropriate mask and a suitable
	error code.


	@param aDataRequestMask Is a bitmask that specifies which subsets of GPS assistance data are
	       required. If aDataRequestMask is set to zero, this indicates that no further assistance
	       data is needed. 

	@see MLbsNetworkProtocolObserver::ProcessAssistanceData() 
	*/
	virtual void RequestAssistanceData(
									TLbsAsistanceDataGroup aDataRequestMask) = 0;
	
	/**
	RequestNetworkLocation() is used by the LBS subsystem to request that the network
	calculates the current location of the handset and returns this to the terminal. 

	When the location information is received from the network the protocol module
	forwards the information to the LBS subsystem via the method
	MLbsNetworkProtocolObserver::ProcessLocationUpdate(). 

	The position information returned by is entirely network calculated and is normally
	expected to be an approximate location only. For example, related to the position
	of the current serving cell. 

	This method automatically cancels any outstanding network location request and
	replaces it with the new request. The CancelNetworkLocation() method is used when the
	LBS subsystem wants to a cancel request but does not immmediately wish to replace with
	another.

	The aSessionId parameter is generated by the LBS subsystem and is used to connect all
	corresponding responses and further requests. The same session ID will be used the LBS
	subsystem if it wishes to cancel the request. Similarly, the same session ID must also
	be supplied by the protocol module when it responds to the LBS subsystem
	via the MLbsNetworkProtocolObserver::ProcessLocationUpdate() method.

	The session ID is also passed by the protocol module when closes a session via the
	method MLbsNetworkProtocolObserver::ProcessSessionComplete().

	In addition to performing the normal session complete procedures, the
	ProcessSessionComplete() message is used by the LBS subsystem to determine if it is
	allowed to provide location information to the local applications. See the discussion
	on protocol conflicts under RequestSelfLocation for more information.



	@param aSessionId The Id of the network location request. This is generated by the LBS
	       and must be used by the protocol module when it returns the obtained position
	       information. 

	@param aOptions This must be of type TLbsNetPosRequestOptions.
	       

	@see RequestSelfLocation
	@see TLbsNetPosRequestOptions
	@see MLbsNetworkProtocolObserver::ProcessLocationUpdate()
	@see CancelNetworkLocation 
	*/	
	virtual void RequestNetworkLocation(
									const TLbsNetSessionId& aSessionId,
									const TLbsNetPosRequestOptionsBase& aOptions) = 0;	
	

	/**
	This method attempts to cancel a request to obtain the network based location
	of the terminal. The previous request will have been initiated by calling
	RequestNetworkLocation()

	Note: If the LBS subsystem wishes to stop the network location request and
	immediately issue a new network location request it should issue a subsequent
	call to RequestNetworkLocation(). Submitting a new network location request,
	automatically cancels any outstanding transfer.

	In some circumstances, the protocol module may still return the associated
	network based position calculation to the LBS subsystem. This situation can
	occur when the cancel request is performed after a point where it is no longer
	possible to stop the request being sent.


	@param aSessionId The Id of the network location request to be cancelled.
	@param aReason Indicates why the LBS subsystem wants the terminate the request. 

	@see RequestNetworkLocation() 
	@see MLbsNetworkProtocolObserver::ProcessLocationUpdate() 
	*/
	virtual void CancelNetworkLocation(
									const TLbsNetSessionId& aSessionId,
									TInt  aReason) = 0;

	/**
	RequestSelfLocation() is used by the LBS subsystem to inform the network protocol
	module that local applications wish to obtain the terminal's current location. This
	method is called every time one or more LBS client applications make location request.

	The parameters indicate if GPS asistance data is required and if one of the clients
	is requesting location for the first time. If GPS assistance data is required the
	protocol module will normally initiate a MO-LR. However, the protocol may decide
	to perform a network request for other reasons.

	When the protocol module has completed the request it signals this by calling
	MLbsNetworkProtocolObserver::ProcessSessionComplete(). If the protocol module had
	decided not to initiate a network MO-LR request it must still call
	MLbsNetworkProtocolObserver::ProcessSessionComplete().

	In addition to performing the normal session complete procedures, this session close
	message is used by the LBS subsystem to determine if it is allowed to provide location
	information to the local applications.

	If the protocol module completes the session with KErrPositionHighPriorityReceive or 
	KErrServerBusy error this indicates there has been a protocol conflict error. 
	Under this situation, the LBS subsystem reports this same error to all the self locate
	client applications that want location information at that particular time.

	Note, client applications that have specified a desired time interval between their
	successive updates may not be affected by protocol conflicts. Their current location
	requests will only be returned with this error if their requests coincides with the
	conflict.

	If the protocol module reports other type of errors, the LBS subsystem can still be
	attempt to provide location information to its client applications.

	When the protocol module does initiate a self locate MO-LR request with the network,
	this  will result in a number of messages and requests being exchanged. Throughout
	these exchanges, the protocol module must ensure it uses and tracks the session-id
	that was passed to RequestSelfLocation().

	For example, the terminal will generally receive a reference location from the
	network. The protocol module forwards that information to the LBS subsystem using  
	MLbsNetworkProtocolObserver::ProcessLocationUpdate(). The protocol module
	should set the session-id parameter to that original passed to RequestSelfLocation().

	Similarly, if the protocol module generates ProcessLocationRequest() it must
	use the same session-id. It should also expect the LBS subsystem to respond
	to that method using the same session-id.

	Submitting a new network location request, automatically cancels any outstanding transfer.
	
	Note: if the protocol module receives a new RequestSelfLocation before the previous one has completed he may or may not initiate a new network request. 
	In either case, the sessionid between LBS and the protocol module has already changed and the protocol module must behave accordingly. 
	If the protocol module does not issue a new network request, he is still required to generate a ProcessLocationRequest() with the new session id to synchronise with lbs.
	It is also possible that LBS will call RespondLocationRequest() with the old session id before receiving the ProcessLocationRequest() with the new session id;
	in this case the protocol module should not ignore that 'old' update.


	@param aSessionId The Id of the self location request. This is generated by the LBS
	       and must be used by the protocol module when it performs actions related to
	       that request. 

	@param aOptions This parameter should be of type TLbsNetPosRequestOptionsAssistance.
	       It provides information about the self location request including the required
	       assistance data.
	       
	@see TLbsNetPosRequestOptionsAssistance
	@see MLbsNetworkProtocolObserver::ProcessLocationUpdate() 
	@see MLbsNetworkProtocolObserver::ProcessLocationRequest() 
	@see MLbsNetworkProtocolObserver::ProcessSessionComplete() 
	*/
	virtual void RequestSelfLocation(
									const TLbsNetSessionId& aSessionId,
									const TLbsNetPosRequestOptionsBase& aOptions) = 0;	
	
	/**
	This method attempts to cancel a request to perform the network side of a
	self location request. The previous request will have been initiated by calling
	RequestSelfLocation()

	Note: If the LBS subsystem wishes to stop the self location request and
	immediately issue a new network location request it should issue a subsequent
	call to RequestSelfLocation(). Submitting a new network location request,
	automatically cancels any outstanding transfer.

	In some circumstances, the protocol module may still return information such
	as the reference location and assistance data. This situation can
	occur when the cancel request is performed after a point these are injected into
	the system.


	@param aSessionId The Id of the self location request being cancelled.

	@param aReason Indicates why the LBS subsystem wants the terminate the request. 
	
	@see RequestNetworkLocation() 
	@see MLbsNetworkProtocolObserver::ProcessLocationUpdate() 
	*/
	virtual void CancelSelfLocation(
									const TLbsNetSessionId& aSessionId,
									TInt  aReason) = 0;

	/** 
	The Network Gateway uses this to provide the protocol module with LBS system status advice - 
	this can be used in making conflict control decisions and also deciding when to 
	interact with the network.	
					
	
	@param aStatus The value of the status of the Lbs positioning framework.
	
	@see TLbsSystemStatus
	
	*/
	virtual void AdviceSystemStatus(
									TLbsSystemStatus aStatus) = 0;

public: //InternalTechnology
	/**
	 * This method is used to retrieve pointers to extended interfaces, if
	 * supported by derived classes.
	 */
	IMPORT_C virtual TAny* ExtendedInterface(TInt aFunctionNumber, TAny* aPtr1, TAny* aPtr2);


#ifndef SYMBIAN_ENABLE_SPLIT_HEADERS
	/**
	ExtendedInterface aFunctionNumber parameter.
	@internalTechnology
	*/
	enum TExtInterface
		{
		EExtInterfaceCount = 0,
		EExtInterface2,
		EExtInterfaceMax // Always keep at the end
		};

#endif

private:
	// Instance identifier key
    TUid iDtor_ID_Key;
	};


/**
CLbsNetworkProtocolBase2 is an extended version of CLbsNetworkProtocolBase, 
adding extended versions of the methods RequestTransmitLocation(), RespondPrivacyRequest() and RequestAssistanceData().

To determine whether a CLbsNetworkProtocolBase object supports this interface,
call the ExtendedInterface method specifying ETransmitLocationOptionsExtendedInterface 
as the function number. 

@see MLbsNetworkProtocolBase2

@publishedPartner
@released
*/
class CLbsNetworkProtocolBase2 : public CLbsNetworkProtocolBase
	{
public:
    IMPORT_C virtual void RequestTransmitLocation(
									const TLbsNetSessionId& aSessionId,
									const TDesC& aDestination,
									TInt aPriority,
									const TLbsNetPosRequestOptionsBase& aOptions) = 0;

	IMPORT_C virtual void RespondPrivacyRequest(
									const TLbsNetSessionId& aSessionId,
									const TLbsPrivacyResponse& aResponse,
									TInt aReason) = 0;
									
	IMPORT_C virtual void CancelExternalLocation(
									const TLbsNetSessionId& aSessionId,
									TInt aReason);

	IMPORT_C virtual void RequestAssistanceData(
									TLbsAsistanceDataGroup aDataRequestMask,
									const TLbsNetSessionIdArray& aSessionIdArray) = 0;
									
public: 
	/**
	 * This method is used to retrieve pointers to extended interfaces, if
	 * supported by derived classes.
	 */

	IMPORT_C virtual TAny* ExtendedInterface(TInt aFunctionNumber, TAny* aPtr1, TAny* aPtr2);


private:
	// Note: The following methods are made private to reinforce the fact that 
	//       they should no longer be implemented in derived classes. Please 
	//       implement the extended versions as declared in the public section above.
	IMPORT_C virtual void RequestTransmitLocation(
									const TLbsNetSessionId& aSessionId,
									const TDesC& aDestination,
									TInt aPriority);
									
	IMPORT_C virtual void RespondPrivacyRequest(
									const TLbsNetSessionId& aSessionId,
									const TLbsPrivacyResponse& aResponse);

	IMPORT_C virtual void RequestAssistanceData(
									TLbsAsistanceDataGroup aDataRequestMask);
	};


/**
This class encapsulated the construction parameters for a Network
Protocol Module implementation.

These parameters are passed by the LBS subsystem to the implementation
of the protocol module via ECom. The protocol module's ECom constructor
should take a single TAny* parameter. That parameter should be cast to
a variable of type TLbsNetProtocolModuleParams.

@see CLbsLocationSourceBase::NewL

@publishedPartner
@released
*/
class TLbsNetProtocolModuleParams : public TLbsNetClassTypeBase
	{
public:

	/**
	This class encapsulated the construction parameters for a Network
	Protocol Module implementation.

	When created, the LBS subsystem must provide a reference to the 
	module's observer.  The observer is used by the protocol module to
	forward requests and information recieved from the network.

	@see MLbsNetworkProtocolObserver
	@see CLbsNetworkProtocolBase::NewL
	*/
	IMPORT_C TLbsNetProtocolModuleParams
		(
		MLbsNetworkProtocolObserver& aObserver
		);

	/**
	Returns a reference to the network protocol observer. The observer is used
	by the protocol module to forward requests and information recieved from
	the network.

	@see MLbsNetworkProtocolObserver
	@see CLbsNetworkProtocolBase::NewL
	*/
	IMPORT_C MLbsNetworkProtocolObserver& Observer() const;
	
protected:
	MLbsNetworkProtocolObserver& iObserver;
	TUint8 iReserved[20];
	};





#endif // LBSNETPROTOCOLBASE_H_


