/*
* Copyright (c) 2743 Nokia Corporation and/or its subsidiary(-ies). 
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description:  Security interface for HTI framework will provide
*        implementation of security functionality:
*        - authenication (establishing security context),
*        - integrity (generating Message Integrity Code),
*        - confidentiality (message encryption/decryption).
*        The interface and terms are based on the GSS API [RFC 2743].
*
*/


#ifndef SECURITYMANAGER_H__
#define SECURITYMANAGER_H__

#include <e32base.h>


class CHtiSecurityManager : public CBase
    {
public:
    static CHtiSecurityManager* NewL();
    static CHtiSecurityManager* NewLC();

    virtual ~CHtiSecurityManager();

    /**
    * Processes the token in question and generates reply tokens
    * for security context establishment.
    * The function is called as many times as needed depending on
    * underlying authentication algorithm
    *
    * @param aToken token received from client that wish to establish
    *       communcation with HTI framework
    *
    * @return reply token that should be sent back to the client,
    *         transfer ownership
    */
    TDesC8* SetSecurityContext(const TDesC8& aToken);

    /**
    * Checks either security context has been established
    */
    TBool IsContextEstablashed() const;

    /**
    * Reset security context
    */
    void ResetSecurityContext();

    /**
    * Implementeion of integrity and confedentiality services.
    * It can generates message MIC and combine it with message.
    * Optionaly, the message also can be encrypted.
    *
    * @param aMessage message to be wrapped
    * @param aEncrypt flag indicates that message should be encrypted
    *
    * @return wrapped message ready to send, transfer ownership
    *
    */
    TDesC8* WrapL(const TDesC8& aMessage, TBool aEncrypt = EFalse);

    /**
    * Implementeion of integrity and confedentiality services.
    * It unwrape message that was previosly wrapped.
    *
    * @param aMessage wrapped message
    *
    * @return plain message ready for processing, transfer ownership
    *
    */
    TDesC8* UnwrapL(const TDesC8& aMessage);

    /**
    * Generates MIC for a message in question.
    *
    * @param aMessage plain message
    *
    * @return MIC generated by an underlying algorithm (e.g. CRC16)
    */
    TPtrC8 MIC(const TDesC8& aMessage) const;

protected:
    CHtiSecurityManager();
    void ConstructL();

protected:
    TBool iSecurityContext;
    };

#endif
