/*
* Copyright (c) 2006-2009 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description: 
* crypto asymmetric cipher application interface
*
*/


/**
 @file
 @publishedPartner
 @released
*/

#ifndef __CRYPTOAPI_ASYMMETRICCIPHERAPI_H__
#define __CRYPTOAPI_ASYMMETRICCIPHERAPI_H__

#include <e32base.h>
#include <cryptospi/cryptobaseapi.h>

namespace CryptoSpi
	{
	class MAsymmetricCipherBase;
	class MAsymmetricCipher;
	class MAsyncAsymmetricCipher;
	class CKey;
	class CCryptoParam;
	class CCryptoParams;
	
	/**
	Asymmetric Cipher API base class
	*/
	NONSHARABLE_CLASS(CAsymmetricCipherBase) : public CCryptoBase
		{
	public:
		/**
		Destructor
		*/
		virtual ~CAsymmetricCipherBase();
	
		/**
		Set the public key of this cipher. Reset() is called to reinitialise the cipher.
		@param aKey  the public key.
        @leave KErrArgument if aKey is not of the expected type.
        @leave KErrNotSupported if the key is not of valid length.
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/
		IMPORT_C void SetKeyL(const CKey& aKey);

		/**
		Set the crypto mode of this cipher. Reset() is called to reinitialise the cipher.
		@param aCryptoMode crypto mode.
		@leave KErrNotSupported if the specified mode is not supported.
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/
		IMPORT_C void SetCryptoModeL(TUid aCryptoMode);

		/**
		Set padding Mode of this cipher. Reset() is called to reinitialise the cipher.
		@param aPaddingMode padding mode.
		@leave KErrNotSupported if the specified mode is not supported.
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/
		IMPORT_C void SetPaddingModeL(TUid aPaddingMode);

		/**
		Gets the maximum size of input accepted by this object.	
		@return	The maximum input length allowed in bytes.
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/	 
		IMPORT_C TInt GetMaximumInputLengthL();

		/**
		Gets the maximum size of output that can be generated by this object.
		@return	The maximum output length in bytes.
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		 */	 
		IMPORT_C TInt GetMaximumOutputLengthL();
		
	
	protected:
		/**
		Constructor
		*/
		CAsymmetricCipherBase(MAsymmetricCipherBase* aAsymmetricCipher, TInt aHandle);		
		};

	/**
	Synchronous asymmetric cipher API, which wraps a synchronous asymmetric plugin implementation
	*/
	NONSHARABLE_CLASS(CAsymmetricCipher) : public CAsymmetricCipherBase
		{
	public:

		/**
		 * @internalComponent
		 *
		 * Create a CSyncAsymmetricCipher instance from the given MSyncAsymmetricCipher instance
		 * @param aAsymmetricCipher a  Async Asymmetric Cipher plugin instance
		 * @return Pointer to CSyncAsymmetricCipher
		 */
		static CAsymmetricCipher* NewL(MAsymmetricCipher* aAsymmetricCipher, TInt aHandle);

		/**
		Destructor
		*/
		IMPORT_C ~CAsymmetricCipher();


		/**
		Encrypts or decrypts aInput and appends the result to aOutput.
		@param aInput	The input data to be processed.
		@param aOutput	The resulting processed data appended to aOutput.
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/
		IMPORT_C void ProcessL(const TDesC8& aInput, TDes8& aOutput);

	private:
		
		/**
		Constructor
		*/
		CAsymmetricCipher(MAsymmetricCipher* aAsymmetricCipher, TInt aHandle);
		};

	/**
	Asynchronous asymmetric cipher API, which wraps a Asynchronous asymmetric plugin implementation
	*/
	NONSHARABLE_CLASS(CAsyncAsymmetricCipher) : public CAsymmetricCipherBase
		{
	public:

		/**
		 * @internalComponent
		 * 
		 * Create a CAsyncAsymmetricCipher instance from the given MAsyncAsymmetricCipher instance
		 * @param aAsyncAsymmetricCipher an async asymmetric Cipher plugin instance
		 * @return Pointer to CAsyncAsymmetricCipher
		 */
		static CAsyncAsymmetricCipher* NewL(MAsyncAsymmetricCipher* aAsyncAsymmetricCipher, TInt aHandle);

		/**
		Destructor
		*/
		IMPORT_C ~CAsyncAsymmetricCipher();

		/**
		Encrypts or decrypts aInput and appends the result to aOutput asynchronously
		@param aInput		The input data to be processed.
		@param aOutput	The resulting processed data appended to aOutput.
		@param aRequestStatus
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/
		IMPORT_C void ProcessL(const TDesC8& aInput, TDes8& aOutput, TRequestStatus& aRequestStatus);

		/**
		Cancel the outstanding request
		*/
		IMPORT_C void Cancel();

	private:
		/**
		Constructor
		*/
		CAsyncAsymmetricCipher(MAsyncAsymmetricCipher* aAsyncAsymmetricCipher, TInt aHandle);
		};


	/**
	the Factory to create synchronous and asynchronous asymmetric cipher instance 
	*/
	class CAsymmetricCipherFactory
		{
	public:

		/**
		Creates a new instance of an asymmetric cipher
		
		@param aCipher A reference to a pointer that should be set to point to the new asymmetric cipher object.
		@param aAlgorithmUid The asymmetric cipher algorithm to use (e.g. KRsaCipherUid)
		@param aKey The encryption/decryption key.
		@param aCryptoMode whether to encrypt or decrypt
		@param aPaddingMode The padding mode to use
		@param aAlgorithmParams Parameters that are specific to a particular algorithm. This is for extendibility and will normally be null.
		@return KErrNone if successful; otherwise, a system wide error code.
		*/
		IMPORT_C static void CreateAsymmetricCipherL(
												CAsymmetricCipher*& aCipher,
												TUid aAlgorithmUid,
												const CKey& aKey,
												TUid aCryptoMode,
												TUid aPaddingMode,
												const CCryptoParams* aAlgorithmParams);

		/**
		Creates a new instance of an asymmetric cipher.
		
		@param aCipher A reference to a pointer that should be set to point to the new asymmetric cipher object.
		@param aAlgorithmUid The asymmetric cipher algorithm to use (e.g. KRsaCipherUid)
		@param aKey The encryption/decryption key.
		@param aCryptoMode whether to encrypt or decrypt
		@param aPaddingMode The padding mode to use
		@param aAlgorithmParams Parameters that are specific to a particular algorithm. This is for extendibility and will normally be null.
		@return KErrNone if successful; otherwise, a system wide error code.
		*/
		IMPORT_C static void CreateAsyncAsymmetricCipherL(
												CAsyncAsymmetricCipher*& aCipher,
												TUid aAlgorithmUid,
												const CKey& aKey,
												TUid aCryptoMode,
												TUid aPaddingMode,
												const CCryptoParams* aAlgorithmParams);
		};
		
	}

#endif //__CRYPTOAPI_ASYMMETRICCIPHERAPI_H__

