/*
* Copyright (c) 2006-2009 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description: 
* crypto signature application interface
*
*/


/**
 @file
 @publishedPartner
 @released
*/

#ifndef __CRYPTOAPI_SIGNATUREAPI_H__
#define __CRYPTOAPI_SIGNATUREAPI_H__

#include <cryptospi/cryptobaseapi.h>

namespace CryptoSpi
	{
	class CKey;
	class MSignatureBase;
	class MSigner;
	class CCryptoParams;
	class MAsyncSigner;
	class MVerifier;
	class MAsyncVerifier;
	
	/**
	Base class for signer and verifier
	*/
	NONSHARABLE_CLASS(CSignatureBase) : public CCryptoBase
		{
	public:

		/**
		Set the padding mode for the signer or verifier. Reset() is called to reinitialise the cipher.
		@param aPaddingMode	The padding mode of the signer

		@leave KErrNotSupported if the padding mode is not supported.
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.

		*/
		IMPORT_C void SetPaddingModeL(TUid aPaddingMode);

		/**
		Set the private key for the signer or verifier. Reset() is called to reinitialise the cipher.
		@param aPrivateKey	The privatekey that used to sign
		@leave KErrArgument if key is not of the expected type.
		@leave KErrNotSupported if the key is not of valid length.
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/
		IMPORT_C void SetKeyL(const CKey& aPrivateKey);

		/**
		Gets the maximum size of input accepted by this object.
		@return The maximum length allowed in bytes
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/
		IMPORT_C TInt GetMaximumInputLengthL() const;

		/**
		Gets the maximum size of output that can be generated by this object.
		@return The maximum output length in bytes
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/	 
		IMPORT_C TInt GetMaximumOutputLengthL() const;

		/**
		Destructor
		*/
		virtual ~CSignatureBase();
	
	protected:
		/**
		 * @internalComponent
		 *
		 * Constructor
		 **/
		CSignatureBase(MSignatureBase* aSignatureProcessor, TInt aHandle);
		};


	/**
	Synchronous signer API, which wraps a synchronous signer plugin implementation
	*/
	NONSHARABLE_CLASS(CSigner) : public CSignatureBase
		{
	public:
		/**
		 * @internalComponent
		 *
		 * Create a CSigner instance from the given MSigner instance
		 * @param aSigner	The signer plugin instance
		 * @return A pointer to a CSigner instance
		 **/
		static CSigner* NewL(MSigner* aSigner, TInt aHandle);

		/**
		Destructor
		*/
		IMPORT_C ~CSigner();

		/**
		Signs the input hash
		@param aInput	The hash of the message to sign
		@param aSignature	The signature of the hash 
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/
		IMPORT_C void SignL(const TDesC8& aInput, CCryptoParams& aSignature);

	private:
		CSigner(MSigner* aSigner, TInt aHandle);
		};

	/**
	Asynchronous signer API, which wraps an asynchronous signer plugin implementation
	*/
	NONSHARABLE_CLASS(CAsyncSigner) : public CSignatureBase
		{
	public:

		/**
		 * @internalComponent
		 *
		 * Create a CAsyncSigner instance from the given MAsyncSigner instance
		 * @param aAsyncSigner The async signer plugin instance
		 * @return A pointer to a CAsyncSigner instance
		 **/
		static CAsyncSigner* NewL(MAsyncSigner* aAsyncSigner, TInt aHandle);

		/**
		Destructor
		*/
		IMPORT_C ~CAsyncSigner();

		/**
		Set the public key for the signer
		@param aInput	The hash of the message to sign
		@param aSignature	The signature of the hash 
		@param aRequestStatus
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/
		IMPORT_C void SignL(const TDesC8& aInput, CCryptoParams& aSignature, TRequestStatus& aRequestStatus);

		/**
		Cancel the outstanding request
		*/
		IMPORT_C void Cancel();

	private:

		CAsyncSigner(MAsyncSigner* aAsyncSigner, TInt aHandle);
		};
	
	/**
	Synchronous verifier API, which wraps a synchronous verifier plugin implementation
	*/
	NONSHARABLE_CLASS(CVerifier) : public CSignatureBase
		{
	public:

		/**
		 * @internalComponent
		 *
		 * Create a CVerifier instance from the given MVerifier instance
		 * @param aVerifier	The verifier plugin instance
		 * @return A pointer to a CVerifier instance
		 **/
		static CVerifier* NewL(MVerifier* aVerifier, TInt aHandle);

		/**
		Destructor
		*/
		IMPORT_C ~CVerifier();

		/**
		Verify the signature
		@param aInput	The hash of the message to be verified
		@param aSignature	The signature of the hash
		@param aVerificationResult indicates the success or failure of the verification
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/
		IMPORT_C void VerifyL(const TDesC8& aInput, const CCryptoParams& aSignature, TBool& aVerificationResult);

		/**
		Unsign the signature
		@param aOutput	The unsigned hash
		@param aSignature	The signature of the hash
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/
		IMPORT_C void InverseSignL(HBufC8*& aOutput, const CCryptoParams& aSignature);

	private:
		/**
		Constructor
		*/
		CVerifier(MVerifier* aVerifier, TInt aHandle);
		};

	/**
	Asynchronous verfier API, which wraps a asynchronous verifier plugin implementation
	*/
	NONSHARABLE_CLASS(CAsyncVerifier) : public CSignatureBase
		{
	public:

		/**
		 * @internalComponent
		 *
		 * Create a CAsyncVerifier instance from the given MAsyncVerifier instance
		 * @param aAsyncVerifier	The async verifier plugin instance
		 * @return A pointer to a CAsyncVerifier instance
		 */
		static CAsyncVerifier* NewL(MAsyncVerifier* aAsyncVerifier, TInt aHandle);

		/**
		Destructor
		*/
		IMPORT_C ~CAsyncVerifier();

		/**
		Verify the signature
		@param aInput	The hash of the message to be verified
		@param aSignature	The signature of the hash.
		@param aVerificationResult	Indicates the success or failure of the verification
		@param aRequestStatus the request status.
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/
		IMPORT_C void VerifyL(const TDesC8& aInput, const CCryptoParams& aSignature, TBool& aVerificationResult, TRequestStatus& aRequestStatus);

		/**
		Unsign the signature
		@param aOutput	The unsigned hash
		@param aSignature	The signature of the hash
		@param aRequestStatus the request status.
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/
		IMPORT_C void InverseSignL(HBufC8*& aOutput, const CCryptoParams& aSignature, TRequestStatus& aRequestStatus);

		/**
		Cancel the outstanding request
		*/
		IMPORT_C void Cancel();

	private:
		/**
		Constructor
		*/
		CAsyncVerifier(MAsyncVerifier* aAsyncVerifier, TInt aHandle);
		};


	/**
	the Factory to create synchronous and asynchronous signer and verifier instance 
	*/
	class CSignatureFactory
		{
	public:

		/**
		Creates a new instance of a synchronous signer.

		@param aSigner A reference to a pointer that should be set to point to the new signer object.
		@param aAlgorithmUid The algorithm to use.
		@param aKey The signing key.
		@param aPaddingMode The padding mode of the signer.
		@param aAlgorithmParams Parameters that are specific to a particular algorithm. This is for extendibility and will normally be null.
		@return KErrNone if successful; otherwise, a system wide error code.
		*/
		IMPORT_C static void CreateSignerL(CSigner*& aSigner,
										TUid aAlgorithmUid,
										const CKey& aKey,
										TUid aPaddingMode,
										const CCryptoParams* aAlgorithmParams);


		/**
		Creates a new instance of a asynchronous signer.

		@param aAsyncSigner A reference to a pointer that should be set to point to the new signer object.
		@param aAlgorithmUid The algorithm to use
		@param aKey The signing key.
		@param aPaddingMode The padding mode of the signer.
		@param aAlgorithmParams Parameters that are specific to a particular algorithm. This is for extendibility and will normally be null.
		@return KErrNone if successful; otherwise, a system wide error code.
		*/
		IMPORT_C static void CreateAsyncSignerL(CAsyncSigner*& aAsyncSigner,
										TUid aAlgorithmUid,
										const CKey& aKey,
										TUid aPaddingMode,
										const CCryptoParams* aAlgorithmParams);
		
		/**
		Creates a new instance of a verifier.
		
		@param aVerifier A reference to a pointer that should be set to point to the new verifier object.
		@param aAlgorithmUid The algorithm to use
		@param aKey The key to verify the signature with.
		@param aPaddingMode The padding mode of the signer.
		@param aAlgorithmParams Parameters that are specific to a particular algorithm. This is for extendibility and will normally be null.
		@return KErrNone if successful; otherwise, a system wide error code.
		*/
		IMPORT_C static void CreateVerifierL(CVerifier*& aVerifier,
										TUid aAlgorithmUid,
										const CKey& aKey,
										TUid aPaddingMode,
										const CCryptoParams* aAlgorithmParams);


		/**
		Creates a new instance of an asynchronous verifier.
		
		@param aAsyncVerifier A reference to a pointer that should be set to point to the new verifier object.
		@param aAlgorithmUid The algorithm to use
		@param aKey The key to verify the signature with.
		@param aPaddingMode The padding mode of the signer.
		@param aAlgorithmParams Parameters that are specific to a particular algorithm. This is for extendibility and will normally be null.
		@return KErrNone if successful; otherwise, a system wide error code.
		*/
		IMPORT_C static void CreateAsyncVerifierL(CAsyncVerifier*& aAsyncVerifier,
										TUid aAlgorithmUid,
										const CKey& aKey,
										TUid aPaddingMode,
										const CCryptoParams* aAlgorithmParams);

		};
	}

#endif //__CRYPTOAPI_SIGNATUREAPI_H__
