/*
* Copyright (c) 2006-2010 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description: 
* Signer Abstract interface
*
*/


/**
 @file
 @publishedPartner
 @released
*/

#ifndef __CRYPTOAPI_SIGNER_H__
#define __CRYPTOAPI_SIGNER_H__

#include <cryptospi/cryptoplugin.h>
#include <cryptospi/keys.h>

namespace CryptoSpi
	{
	/**
	The Signer definition. Intended to allow plug-ins to implement 
	extensible signature signer functionality, and to work with all
	known existing  signature algorithms, e.g. DSA, RSA etc
	*/
	class MSignatureBase : public MPlugin
		{
	public:
		/**
		Set the padding mode for the signer or verifier. Reset() is called to reinitialise the cipher.
		@param aPaddingMode The padding mode of the signer
		@leave KErrNotSupported if the padding mode is not supported.
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/
		virtual void SetPaddingModeL(TUid aPaddingMode) = 0;

		/**
		Set the private key for the signer or verifier. Reset() is called to reinitialise the cipher.
		@param aPrivateKey The privatekey that used to sign
		@leave KErrArgument if key is not of the expected type.
		@leave KErrNotSupported if the key is not of valid length.
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/
		virtual void SetKeyL(const CKey& aPrivateKey) = 0;

		/**
		Gets the maximum size of input accepted by this object.
		@return The maximum length allowed in bytes
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/	 
		virtual TInt GetMaximumInputLengthL() const = 0;

		/**
		Gets the maximum size of output that can be generated by this object.
		@return The maximum output length in bytes
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/	 
		virtual TInt GetMaximumOutputLengthL() const = 0;
		};

	class MSigner : public MSignatureBase
		{
	public:

		/**
		Signs the input hash
		@param aInput	The hash of the message to sign
		@param aSignature The signature of the hash 
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/
		virtual void SignL(const TDesC8& aInput, CCryptoParams& aSignature) = 0;
		};


	class MAsyncSigner : public MSignatureBase
		{
	public:

		/**
		Set the public key for the signer
		@param aInput	The hash of the message to sign
		@param aSignature The signature of the hash 
		@param aRequestStatus
		@leave ...	Any of the crypto error codes defined in 
  					cryptospi_errs.h or any of the system-wide error codes.
		*/
		virtual void SignL(const TDesC8& aInput, CCryptoParams& aSignature, TRequestStatus& aRequestStatus) = 0;

		/**
		Cancel the outstanding request
		*/
		virtual void Cancel() = 0;

		};

	} // namespace CryptoSpi

#endif //__CRYPTOAPI_SIGNER_H__
