/*
* Copyright (c) 2003-2009 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of the License "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description: 
*
*/


/**
 @file
 @publishedPartner
 @released
*/


#ifndef __SUPPLIER_H__
#define __SUPPLIER_H__

#include <e32base.h>
#include <e32std.h>
#include <caf/caftypes.h>

class RHTTPHeaders;
class RStringPool;

namespace ContentAccess 
	{
	class CImportFile;
	class CMetaDataArray;
	class CAgentResolver;
	class MFileHandleProvider;

#ifndef REMOVE_CAF1
	class CCafMimeHeader;
#endif

	/** 
	Used to import files using a content access agent. In the case of some
	DRM agents the import will be a transformation from an unprotected content type
	to a protected content type.
	 
	It creates CImportFile objects to import files into a Content Access
	Agent. CSupplier uses it's CAgentResolver to determine which agent should receive
	the file based upon the mime type of the file.
	
	Clients can check if a mime type is supported by by calling the
	CSupplier::IsImportSupported() function.
	
	The CSupplier does not dynamically update it's list of agents so applications
	should not use a long term persistent instance of CSupplier.
	
	@publishedPartner
	@released
	*/
	class CSupplier : public CBase
		{ 
	public:
		/** Create a new CSupplier 
		@return a CSupplier object
		*/
		IMPORT_C static CSupplier* NewL();

		/** Create a new CSupplier 
		@return a CSupplier object
		*/
		IMPORT_C static CSupplier* NewLC();

		/** destructor */
		virtual ~CSupplier();

		/** Sets the output directory for any files generated by the CAF.
		
		@param aOutputDirectory The preferred location to store any output files
		*/
		IMPORT_C void SetOutputDirectoryL(const TDesC& aOutputDirectory);

		/** Determines whether one of the CAF agents wants to import
		a file with the given mimetype

		If this function returns ETrue the application should use the CSupplier to 
		create a ContentAccess::CImportFile session and import the file into CAF.
		
		@param aMimeType	The mime type of the file determined by an Apparc recognition or transmission header information.
		@return				ETrue if the mimetype is recognized by one of the CAF agents, EFalse otherwise.
		*/
		IMPORT_C TBool IsImportSupported(const TDesC8& aMimeType);

		
		/** Get HTTP download headers from the agents

		Some DRM schemes require specific headers to be present in the HTTP request
		header. This function allows a client to retrieve these headers from the
		agents so any download request will satisfy the criteria required by 
		the HTTP server.

		@param aStringPool The string pool used by the HTTP session
		@param aRequestHeaders The agents will add any required fields to the headers
		*/
		IMPORT_C void PrepareHTTPRequestHeaders(RStringPool& aStringPool, RHTTPHeaders& aRequestHeaders) const;
		
		
		/** Creates a new CImportFile object and allow the agent to generate the output files 
		
		@param aMimeType			This should be one of the mime types supported by CAF, IsImportSupported() can be used to check this.
		@param aImportMetaData		Any additional information that may be useful for the agent performing the import. 
									Ideally this will include the mime headers for the file about to be passed into CAF.
		@param aSuggestedFileName	Suggested filename for the CAF agent to use when 
		 							creating the output files, Details of the output files produced can be obtained using
									CImportFile::OutputFileL().
		@return					    A CImportFile object to import the file.
		
		@leave KErrCANoAgent		No agent supports the mime type supplied in aMimeType
		*/
		IMPORT_C CImportFile* ImportFileL(const TDesC8& aMimeType, const CMetaDataArray& aImportMetaData, const TDesC& aSuggestedFileName);

		/** Creates a new CImportFile object where the caller must provide file handles to store output files. 
		
		This allows a server to store the output files within its own private directory. The CAF agent will write to the private directory 
		using the file handle provided by the server.

		If the agent requires a new output file it will return KErrCANewFileHandleRequired from CImportFile::Write() or CImportFile::WriteComplete() 
		The application performing the supply should provide the agent with a new file handle using CImportFile::ContinueWithNewOutputFile();
		
		@param aMimeType			This should be one of the mime types supported a CAF agent, IsImportSupported() can be used to check this.
		@param aImportMetaData		Any additional information that may be useful for the agent performing the import. 
									Ideally this will include the mime headers for the file about to be passed into CAF.
		@return					    A CImportFile object to import the file.
		
		@leave KErrCANoAgent		No agent supports the mime type supplied in aMimeType
		*/
		IMPORT_C CImportFile* ImportFileL(const TDesC8& aMimeType, const CMetaDataArray& aImportMetaData);

#ifndef REMOVE_CAF1
		/** Import a file
		@param aMimeHeader Mime headers associated with the file to import
		@param aSuggestedFileName The application supplied name for any output file created by the agent
		@deprecated 
		*/
		IMPORT_C CImportFile* ImportFileL(CCafMimeHeader &aMimeHeader, const TDesC &aSuggestedFileName);
#endif

	private:	
		CSupplier();
		void ConstructL();

		// Resolver used to figure out which CA agent is responsible for importing a particular file
		CAgentResolver* iAgentResolver;
		HBufC* iOutputDirectory;
		};
	}

#endif
