/*
* Copyright (c) 2008-2009 Nokia Corporation and/or its subsidiary(-ies).
* All rights reserved.
* This component and the accompanying materials are made available
* under the terms of "Eclipse Public License v1.0"
* which accompanies this distribution, and is available
* at the URL "http://www.eclipse.org/legal/epl-v10.html".
*
* Initial Contributors:
* Nokia Corporation - initial contribution.
*
* Contributors:
*
* Description: 
* Use this file to provide input to the utility that creates test data files for the shaper tests.
*
*/


/**
 @file
 @internalComponent 
*/

#include <stdlib.h>
#include <stdarg.h>
#include <e32svr.h>

//class used for holding input and output data for the shaper, for testing purposes.
struct TInputData
	{
	const TText16* iTextInput;
	/** start of text to shape, negative number means count from the end of
	the string */
	TInt iStart;
	/** end of text to shape, nonpositive number means count from the end of
	the string */
	TInt iEnd;
	const TText16* iTypeFaceName;
	const TText16* iFontFilename;
	const TText16* iOutputFilename;

	TPtrC TextInput() const {return TPtrC(iTextInput); }
	TPtrC TypeFaceName() const {return TPtrC(iTypeFaceName); }
	TPtrC FontFilename() const {return TPtrC(iFontFilename); }
	TPtrC OutputFilename() const {return TPtrC(iOutputFilename); }
	};

//************************************************************************************************
// TO CREATE FILES FOR USE WITH SHAPER TESTS, YOU WILL NEED TO:
// create an input data variable below
// any data added to this file should be left in place in case it is required again.
// the .cpp file can be edited to use a particular inputData variable as required.
//************************************************************************************************

// filename for TNR_Dev_OT font
const TText16* KTNR_Dev_OTTypeFaceName = (const TText16*) L"Devanagari OT Eval";
const TText16* KTNR_Dev_OTFont = (const TText16*) L"Z:\\Resource\\Fonts\\TNR_Dev_OT.ttf";

const TText16* KS60IndicTypeFaceName = (const TText16*) L"Series 60 Sans";
const TText16* KS60IndicFont = (const TText16*) L"Z:\\Resource\\Fonts\\s60ssb_v500.ttf";


//hindi text to be shaped by the shaper
const TInputData TNR_Dev_OTInputData[] =
	{
		{
		//this input data taken from the original iculayoutengine test data - letest
		(const TText16*)
		L"\x0936\x094D\x0930\x0940\x092E\x0926\x094D\x0020"
		L"\x092D\x0917\x0935\x0926\x094D\x0917\x0940\x0924"
		L"\x093E\x0020\x0905\x0927\x094D\x092F\x093E\x092F"
		L"\x0020\x0905\x0930\x094D\x091C\x0941\x0928\x0020"
		L"\x0935\x093F\x0937\x093E\x0926\x0020\x092F\x094B"
		L"\x0917\x0020\x0927\x0943\x0924\x0930\x093E\x0937"
		L"\x094D\x091F\x094D\x0930\x0020\x0909\x0935\x093E"
		L"\x091A\x0943\x0020\x0927\x0930\x094D\x092E\x0915"
		L"\x094D\x0937\x0947\x0924\x094D\x0930\x0947\x0020"
		L"\x0915\x0941\x0930\x0941\x0915\x094D\x0937\x0947"
		L"\x0924\x094D\x0930\x0947\x0020\x0938\x092E\x0935"
		L"\x0947\x0924\x093E\x0020\x092F\x0941\x092F\x0941"
		L"\x0924\x094D\x0938\x0935\x0903\x0020\x092E\x093E"
		L"\x092E\x0915\x093E\x0903\x0020\x092A\x093E\x0923"
		L"\x094D\x0921\x0935\x093E\x0936\x094D\x091A\x0948"
		L"\x0935\x0020\x0915\x093F\x092E\x0915\x0941\x0930"
		L"\x094D\x0935\x0924\x0020\x0938\x0902\x091C\x0935",
		0,0,
		KTNR_Dev_OTTypeFaceName,
		KTNR_Dev_OTFont,
		(const TText16*) L"c:\\tnr_dev_otshaperdata1.dat",
		},
		{
		//this input data as provided by licensee
		(const TText16*)
		L"\x0915\x094D\x0937\x0924\x094D\x0930\x093F\x092F"
		L"\x0020\x0909\x0926\x094D\x0926\x0947\x0936\x094D"
		L"\x0020\x0915\x094D\x0937\x093F\x092A\x094D\x0930"
		L"\x0020\x092d\x0942\x0930\x094d\x092d\x0941\x0935"
		L"\x0903\x0020\x0938\x094d\x0935\x0903\x0020\x0924"
		L"\x0924\x094d\x0938\x0935\x093f\x0924\x0941\x0930"
		L"\x094d\x0935\x0930\x0947\x0923\x094d\x092f\x0902"
		L"\x0020\x092d\x0930\x094d\x0917\x094b\x0020\x0926"
		L"\x0947\x0935\x0938\x094d\x092f\x0020\x0927\x0940"
		L"\x092e\x0939\x093f\x0020\x0927\x093f\x092f\x094b"
		L"\x0020\x092f\x094b\x0928\x0903\x0020\x092a\x094d"
		L"\x0930\x091a\x094b\x0926\x092f\x093e\x0924\x094d\x0020",
		0,0,
		KTNR_Dev_OTTypeFaceName,
		KTNR_Dev_OTFont,
		(const TText16*) L"c:\\tnr_dev_otshaperdata2.dat"
		},
		{
		//this input data as provided by licensee
		(const TText16*)
		L"\x092f\x0924\x094d\x0930"
		L"\x0020\x092f\x094b\x0917\x0947\x0936\x094d\x0935\x0930"
		L"\x0020\x0915\x0943\x0937\x094d\x0923\x094b\x0020\x092f"
		L"\x0924\x094d\x0930\x0020\x092a\x093e\x0930\x094d\x0925"
		L"\x094b\x0020\x0927\x0930\x094d\x0928\x0941\x0927\x0930\x0903\x0020"
		L"\x0924\x0924\x094d\x0930\x0020\x0936\x094d\x0930\x0940"
		L"\x0930\x094d\x0935\x093f\x091c\x092f\x094b\x0020\x092d"
		L"\x0942\x0924\x093f\x0930\x094d\x0927\x0941\x0935\x093e"
		L"\x0020\x0928\x0940\x0924\x093f\x0930\x094d\x092e\x0924"
		L"\x093f\x0930\x094d\x092e\x092e\x0020"
		L"\x0915\x0941\x0924\x0938\x094d\x0924\x094d\x0935\x093e"
		L"\x0020\x0915\x0936\x094d\x092e\x0932\x092e\x093f\x0926\x0902\x0020"
		L"\x0935\x093f\x0936\x092e\x0947\x0020\x0938\x092e\x0941"
		L"\x092a\x0938\x094d\x0925\x093f\x0924\x092e\x094d\x0020",
		0,0,
		KTNR_Dev_OTTypeFaceName,
		KTNR_Dev_OTFont,
		(const TText16*) L"c:\\tnr_dev_otshaperdata3.dat"
		},
		{
		//this input data as provided by licensee
		(const TText16*)
		L"\x0905\x0928\x093e\x0930\x094d\x092f\x091c\x0941\x0937"
		L"\x094d\x091f\x092e\x0938\x094d\x0935\x0930\x094d\x0917"
		L"\x094d\x092f\x092e\x0915\x0940\x0930\x094d\x0924\x093f"
		L"\x0915\x0930\x092e\x0930\x094d\x091c\x0941\x0928\x0020"
		L"\x0915\x094d\x0932\x0948\x092c\x094d\x092f\x0902\x0020"
		L"\x092e\x0020\x0938\x094d\x092e\x0020\x0917\x092e\x0903\x0020"
		L"\x092a\x093e\x0930\x094d\x0925\x0928\x0948\x0924\x0924\x094d"
		L"\x0924\x094d\x0935\x092f\x094d\x092f\x0941\x092a"
		L"\x092a\x0926\x094d\x092f\x0924\x0947\x0020"
		L"\x0915\x094d\x0937\x0941\x0926\x094d\x0930\x0902\x0020"
		L"\x0939\x0943\x0926\x092f\x0926\x094c\x0930\x094d\x092c\x0932"
		L"\x094d\x092f\x0902\x0020\x0924\x094d\x092f\x0915\x094d\x0924"
		L"\x094d\x0935\x094b\x0924\x094d\x0924\x093f\x0937\x094d\x091f"
		L"\x0020\x092a\x0930\x0902\x0924\x092a\x0020",
		0,0,
		KTNR_Dev_OTTypeFaceName,
		KTNR_Dev_OTFont,
		(const TText16*) L"c:\\tnr_dev_otshaperdata4.dat"
		},
		{
		// Test for defect DEF078032
		// Test that ansusvara+udatta u_matra+anusvara produce two dotted circles each
		// and don't crash.
		(const TText16*) L"\x902\x951 \x941\x902",
		0,0,
		KTNR_Dev_OTTypeFaceName,
		KTNR_Dev_OTFont,
		(const TText16*) L"c:\\tnr_dev_otshaperdata5.dat"
		}
	};


const TInputData Kannada_InputData[] =
	{
			{
			//this input data taken from licensee test sample data
			(const TText16*)
			L"\x0CA8\x0CAE\x0CCD\x0CAE\x0020\x0CB0\x0CBE\x0CB7\x0CCD\x0C9F\x0CCD\x0CB0\x0020\x0CAD\x0CBE\x0CB0\x0CA4\x002E\x0020"
			L"\x0CB2\x0C95\x0CCD\x0CB7\x0CCD\x0CAE\x0CCD\x0CAF\x0CBE"
			L"\x0C95\x0CCD\x0C95\x0CCD\x0C95\x0CCD\x0C95\x0CCD\x0C95\x0CCB\x0C82"
			L"\x0CB6\x0CCD\x0CB0\x0CC0\x0020\x0CAD\x0C97\x0CB5\x0CBE\x0CA8\x0CC1\x0CB5\x0CBE\x0C9A"
			L"\x0C95\x0CC1\x0CA4\x0CB8\x0CCD\x0CA4\x0CCD\x0CB5\x0CBE\x0020\x0C95\x0CB6\x0CCD\x0CAE\x0CB2\x0CAE\x0CBF\x0CA6\x0C82\x0020\x0CB5\x0CBF\x0CB7\x0CAE\x0020\x0CB8\x0CAE\x0CC1\x0CAA\x0CB8\x0CCD\x0CA5\x0CBF\x0CA4\x0CAE\x0CCD"
			L"\x0C85\x0CA8\x0CBE\x0CB0\x0CCD\x0CAF\x0C9C\x0CC1\x0CB7\x0CCD\x0CA0\x0CAE\x0CB8\x0CCD\x0CB5\x0CB0\x0CCD\x0C97\x0CCD\x0CAF\x0CAE\x0C95\x0CC0\x0CB0\x0CCD\x0CA4\x0CBF\x0C95\x0CB0\x0CAE\x0CB0\x0CCD\x0C9C\x0CC1\x0CA8"
			L"\x0C95\x0CCD\x0CB2\x0CC8\x0CAC\x0CCD\x0CAF\x0C82\x0020\x0CAE\x0CBE\x0020\x0CB8\x0CCD\x0CAE\x0020\x0C97\x0CAE\x0C83\x0020\x0CAA\x0CBE\x0CB0\x0CCD\x0CA5\x0020\x0CA8\x0CC8\x0CA4\x0CA4\x0CCD\x0CA4\x0CCD\x0CB5\x0CAF\x0CCD\x0CAF\x0CC1\x0CAA\x0CAA\x0CA7\x0CCD\x0CAF\x0CA4\x0CC7"
			L"\x0C95\x0CCD\x0CB7\x0CC1\x0CA6\x0CCD\x0CB0\x0C82\x0020\x0CB9\x0CC3\x0CA6\x0CAF\x0CA6\x0CCC\x0CB0\x0CCD\x0CAC\x0CB2\x0CCD\x0CAF\x0C82\x0020\x0CA4\x0CCD\x0CAF\x0C95\x0CCD\x0CA4\x0CCD\x0CB5\x0CCB\x0CA4\x0CCD\x0CA4\x0CBF\x0CB7\x0CCD\x0CA0\x0020\x0CAA\x0CB0\x0CA8\x0CCD\x0CA4\x0CAA"
			L"\x0CB8\x0CC1\x0CAA\x0CCD\x0CB0\x0CB8\x0CBE\x0CA6\x0C83\x0020\x0CAA\x0CCD\x0CB0\x0CB8\x0CA8\x0CCD\x0CA8\x0CBE\x0CA4\x0CCD\x0CAE\x0CBE"
			L"\x0CB5\x0CBF\x0CB6\x0CCD\x0CB5\x0CA7\x0CC3\x0C97\x0CCD\x0CB5\x0CBF\x0CB6\x0CCD\x0CB5\x0CAD\x0CC1\x0C97\x0CCD\x0CB5\x0CBF\x0CAD\x0CC1\x0C83"
			L"\x0CB8\x0CA4\x0CCD\x0C95\x0CB0\x0CBE\x0CA4\x0CBE\x0020\x0CB8\x0CA4\x0CCD\x0C95\x0CC3\x0C83\x0020\x0CB8\x0CBE\x0CA7\x0CC1\x0CB0\x0CCD\x0C9C\x0CB9\x0CCD\x0CA8\x0CC1\x0CB0\x0CCD\x0CA8\x0CBE\x0CB0\x0CBE\x0CAF\x0CA3\x0CCB\x0020\x0CA8\x0CB0\x0C83"
			L"\x0C85\x0CB8\x0C82\x0C96\x0CCD\x0CAF\x0CC7\x0CAF\x0CCB\x0C85\x0CAA\x0CCD\x0CB0\x0CAE\x0CC7\x0CAF\x0CBE\x0CA4\x0CCD\x0CAE\x0CBE\x0020\x0CB5\x0CBF\x0CB6\x0CBF\x0CB7\x0CCD\x0C9F\x0C83"
			L"\x0CB6\x0CBF\x0CB7\x0CCD\x0C9F\x0C95\x0CC3\x0C9A\x0CCD\x0C9B\x0CC1\x0C9A\x0CBF\x0C83"
			L"\x0CB8\x0CBF\x0CA6\x0CCD\x0CA7\x0CBE\x0CB0\x0CCD\x0CA5\x0C83\x0020\x0CB8\x0CBF\x0CA6\x0CCD\x0CA7\x0CB8\x0C82\x0C95\x0CB2\x0CCD\x0CAA\x0C83\x0020\x0CB8\x0CBF\x0CA6\x0CCD\x0CA7\x0CBF\x0CA6\x0C83"
			L"\x0CB8\x0CBF\x0CA6\x0CCD\x0CA7\x0CBF\x0CB8\x0CBE\x0CA7\x0CA8\x0C83",
			0,0,
			KS60IndicTypeFaceName,
			KS60IndicFont,
			(const TText16*) L"c:\\shaperkannadadata1.dat"
			},
			{
			//this input data taken from the web
			(const TText16*)
			L"\x0C95\x0020\x0C96\x0020\x0C97\x0020\x0C98\x0020\x0C99\x0020\x0C9A\x0020\x0C9B\x0020\x0C9C\x0CCD\x0CB2\x0020\x0C9D\x0020\x0C9E\x0020\x0C9F\x0020\x0CA0\x0020\x0CA1\x0020\x0CA2\x0020\x0CA3\x0020\x0CA4\x0020\x0CA5\x0020\x0CA6\x0020\x0CA7\x0020\x0CA8\x0020\x0CAA\x0020\x0CAB\x0020\x0CAC\x0020\x0CAD\x0020\x0CAE\x0020\x0CAF\x0020\x0CB0\x0020\x0CB1\x0020\x0CB2\x0020\x0CB3\x0020\x0CB5\x0020\x0CB6\x0020\x0CB7\x0020\x0CB8\x0020\x0CB9\x0020"
			L"\x0C95\x0CB0\x0C96\x0CB0\x0C97\x0CB0\x0C98\x0CB0\x0C99\x0CB0\x0C9A\x0CB0\x0C9B\x0CB0\x0C9C\x0CCD\x0CB2\x0CB0\x0C9D\x0CB0\x0C9E\x0CB0\x0C9F\x0CB0\x0CA0\x0CB0\x0CA1\x0CB0\x0CA2\x0CB0\x0CA3\x0CB0\x0CA4\x0CB0"
			L"\x0CB0\x0CA5\x0CB0\x0CA6\x0CB0\x0CA7\x0CB0\x0CA8\x0CB0\x0CAA\x0CB0\x0CAB\x0CB0\x0CAC\x0CB0\x0CAD\x0CB0\x0CAE\x0CB0\x0CAF\x0CB0\x0CB0\x0020\x0CB0\x0CB1\x0CB0\x0CB2\x0CB0\x0CB3\x0CB0\x0CB5\x0CB0\x0CB6\x0CB0\x0CB7\x0CB0\x0CB8\x0CB0\x0CB9\x0020"
			L"\x0C95\x0CCD\x0CB7\x0CB0\x0C9C\x0CCD\x0C9E\x0CB0\x0C95\x0CCD\x0CB7\x0CBF\x0CB0\x0C9C\x0CCD\x0C9E\x0CBF\x0CB0\x0C95\x0CCD\x0CB7\x0CB0\x0C9C\x0CCD\x0C9E\x0CC1\x0CB0\x0C95\x0CCD\x0CB7\x0CCD\x0CB2\x0CCD\x0CB2\x0CB0\x0C9C\x0CCD\x0C9E\x0CCD\x0CB2\x0CB0\x0CA1\x0CC8\x0CA0\x0CCD\x0CA7\x0CC8\x0CB0\x0CA1\x0CCD\x0CA3\x0CCB\x0CB0\x0C9F\x0CCD\x0CB7\x0CC4\x0CB0"
			L"\x0C95\x0CCD\x0020\x0C96\x0CCD\x0020\x0C97\x0CCD\x0020\x0C98\x0CCD\x0020\x0C99\x0CCD\x0020\x0C9A\x0CCD\x0020\x0C9B\x0CCD\x0020\x0C9C\x0CCD\x0020\x0C9D\x0CCD\x0020\x0C9E\x0CCD\x0020\x0C9F\x0CCD\x0020\x0CA0\x0CCD\x0020\x0CA1\x0CCD\x0020\x0CA2\x0CCD\x0C9F\x0CCD\x0020\x0CA3\x0CCD\x0C9F\x0CCD\x0020\x0CA4\x0CCD\x0020\x0CA5\x0CCD\x0020\x0CA6\x0CCD\x0020\x0CA7\x0CCD\x0020\x0CA8\x0CCD\x0020\x0CAA\x0CCD"
			L"\x0CAB\x0CCD\x0020\x0CAC\x0CCD\x0020\x0CAD\x0CCD\x0020\x0CAE\x0CCD\x0020\x0CAF\x0CCD\x0020\x0CB0\x0CCD\x0020\x0CB1\x0CCD\x0020\x0CB2\x0CCD\x0020\x0CB3\x0CCD\x0020\x0CB5\x0CCD\x0020\x0CB6\x0CCD\x0020\x0CB7\x0CCD\x0020\x0CB8\x0CCD\x0020\x0CB9\x0CCD\x0020\x0C95\x0CCD\x0CB7\x0CCD\x0020\x0C9C\x0CCD\x0C9E\x0CCD",
			0,0,
			KS60IndicTypeFaceName,
			KS60IndicFont,
			(const TText16*) L"c:\\shaperkannadadata2.dat"
			},
			{
			//this input data taken from the web
			(const TText16*)
			L"\x0C95\x0CCD\x200D\x0C95\x0020\x0C96\x0CCD\x200D\x0C95\x0020\x0C97\x0CCD\x200D\x0C95\x0020\x0C98\x0CCD\x200D\x0C95\x0020\x0C99\x0CCD\x200D\x0C95\x0020\x0C9A\x0CCD\x200D\x0C95\x0020\x0C9B\x0CCD\x200D\x0C95\x0020\x0C9C\x0CCD\x200D\x0C95\x0020\x0C9D\x0CCD\x200D\x0C95\x0020\x0C9E\x0CCD\x200D\x0C95\x0020\x0C9F\x0CCD\x200D\x0C95\x0020\x0CA0\x0CCD\x200D\x0C95\x0020\x0CA1\x0CCD\x200D\x0C95\x0020\x0CA2\x0CCD\x200D\x0C95\x0020\x0CA3\x0CCD\x200D\x0C95\x0020\x0CA4\x0CCD\x200D\x0C95\x0020\x0CA5\x0CCD\x200D\x0C95\x0020\x0CA6\x0CCD\x200D\x0C95\x0020\x0CA7\x0CCD\x200D\x0C95\x0020\x0CA8\x0CCD\x200D\x0C95\x0020\x0CAA\x0CCD\x200D\x0C95"
			L"\x0CAB\x0CCD\x200D\x0C95\x0020\x0CAC\x0CCD\x200D\x0C95\x0020\x0CAD\x0CCD\x200D\x0C95\x0020\x0CAE\x0CCD\x200D\x0C95\x0020\x0CAF\x0CCD\x200D\x0C95\x0020\x0CB0\x0CCD\x200D\x0C95\x0020\x0CB1\x0CCD\x200D\x0C95\x0020\x0CB2\x0CCD\x200D\x0C95\x0020\x0CB3\x0CCD\x200D\x0C95\x0020\x0CB5\x0CCD\x200D\x0C95\x0020\x0CB6\x0CCD\x200D\x0C95\x0020\x0CB7\x0CCD\x200D\x0C95\x0020\x0CB8\x0CCD\x200D\x0C95\x0020\x0CB9\x0CCD\x200D\x0C95\x0020\x0C95\x0CCD\x0CB7\x0CCD\x200D\x0C95\x0020\x0C9C\x0CCD\x0C9E\x0CCD\x200D\x0C95"
			L"\x0C95\x0CCD\x0C95\x0CB0\x0C96\x0CCD\x0C96\x0CB0\x0C97\x0CCD\x0C97\x0020\x0C98\x0CCD\x0C98\x0020\x0C99\x0CCD\x0C99\x0020\x0C9A\x0CCD\x0C9A\x0020\x0C9B\x0CCD\x0C9B\x0020\x0C9C\x0CCD\x0C9C\x0020\x0C9D\x0CCD\x0C9D\x0020\x0C9E\x0CCD\x0C9E\x0020\x0C9F\x0CCD\x0C9F\x0020\x0CA0\x0CCD\x0CA0\x0020\x0CA1\x0CCD\x0CA1\x0020\x0CA2\x0CCD\x0CA2\x0020\x0CA3\x0CCD\x0CA3\x0020\x0CA4\x0CCD\x0CA4\x0020\x0CA5\x0CCD\x0CA5\x0020\x0CA6\x0CCD\x0CA6\x0020\x0CA7\x0CCD\x0CA7\x0020\x0CA8\x0CCD\x0CA8\x0020\x0CAA\x0CCD\x0CAA"
			L"\x0CAB\x0CCD\x0CAB\x0020\x0CAC\x0CCD\x0CAC\x0020\x0CAD\x0CCD\x0CAD\x0020\x0CAE\x0CCD\x0CAE\x0020\x0CAF\x0CCD\x0CAF\x0020\x0CB0\x0CCD\x0CB0\x0CCD\x0CB0\x0CB0\x0020\x0CB1\x0CCD\x0CB1\x0020\x0CB2\x0CCD\x0CB2\x0020\x0CB3\x0CCD\x0CB3\x0020\x0CB5\x0CCD\x0CB6\x0020\x0CB6\x0CCD\x0CB6\x0020\x0CB7\x0CCD\x0CB7\x0020\x0CB8\x0CCD\x0CB8\x0020\x0CB9\x0CCD\x0CB9\x0020\x0CB0\x0CCD\x0CDE\x0CCD\x0CDE\x0020\x0CB0\x0CCD\x0C95\x0CCD\x0C95\x0CCD\x0C95\x0020\x0CB0\x0CCD\x0C95\x0CCD\x0C95\x0CCD\x0C95\x0020\x0020\x0C95\x0CCD\x0CB7\x0CCD\x0C95\x0020\x0C9C\x0CCD\x0C9E\x0CCD\x0C95\x0020\x0CA4\x0CCD\x0CA4\x0CCD\x0CB0"
			L"\x0C95\x0CCD\x0C95\x0CCD\x0020\x0C96\x0CCD\x0C96\x0CCD\x0020\x0C97\x0CCD\x0C97\x0CCD\x0020\x0C98\x0CCD\x0C98\x0CCD\x0020\x0C99\x0CCD\x0C99\x0CCD\x0020\x0C9A\x0CCD\x0C9A\x0CCD\x0020\x0C9B\x0CCD\x0C9B\x0CCD\x0020\x0C9C\x0CCD\x0C9C\x0CCD\x0020\x0C9D\x0CCD\x0C9D\x0CCD\x0020\x0C9E\x0CCD\x0C9E\x0CCD\x0020\x0C9F\x0CCD\x0C9F\x0CCD\x0020\x0CA0\x0CCD\x0CA0\x0CCD\x0020\x0CA1\x0CCD\x0CA1\x0CCD\x0020\x0CA2\x0CCD\x0CA2\x0CCD\x0020\x0CA3\x0CCD\x0CA3\x0CCD\x0020\x0CA4\x0CCD\x0CA4\x0CCD\x0020\x0CA5\x0CCD\x0CA5\x0CCD\x0020\x0CA6\x0CCD\x0CA6\x0CCD\x0020\x0CA7\x0CCD\x0CA7\x0CCD\x0020\x0CA8\x0CCD\x0CA8\x0CCD\x0020\x0CAA\x0CCD\x0CAA\x0CCD"
			L"\x0CAB\x0CCD\x0CAB\x0CCD\x0020\x0CAC\x0CCD\x0CAC\x0CCD\x0020\x0CAD\x0CCD\x0CAD\x0CCD\x0020\x0CAE\x0CCD\x0CAE\x0CCD\x0020\x0CAF\x0CCD\x0CAF\x0CCD\x0020\x0CB0\x0CCD\x0CB0\x0CCD\x0CB0\x0CCD\x0020\x0CB1\x0CCD\x0CB1\x0CCD\x0020\x0CB2\x0CCD\x0CB2\x0CCD\x0020\x0CB3\x0CCD\x0CB3\x0CCD\x0020\x0CB5\x0CCD\x0CB6\x0CCD\x0020\x0CB6\x0CCD\x0CB6\x0CCD\x0020\x0CB7\x0CCD\x0CB7\x0CCD\x0020\x0CB8\x0CCD\x0CB8\x0CCD\x0020\x0CB9\x0CCD\x0CB9\x0CCD\x0020\x0CB0\x0CCD\x0CDE\x0CCD\x0CDE\x0CCD\x0020\x0CB0\x0CCD\x0C95\x0CCD\x0C95\x0CCD\x0C95\x0CCD\x0020\x0CB0\x0CCD\x0C95\x0CCD\x0C95\x0CCD\x0C95\x0CCD\x0020\x0C95\x0CCD\x0CB7\x0CCD\x0C95\x0CCD\x0020\x0C9C\x0CCD\x0C9E\x0CCD\x0C95\x0CCD\x0020\x0CA4\x0CCD\x0CA4\x0CCD\x0CB0\x0CCD"
			L"\x0C95\x0CCD\x0C95\x0CCD\x0CB0\x0020\x0C96\x0CCD\x0C96\x0CCD\x0CB0\x0020\x0C97\x0CCD\x0C97\x0CCD\x0CB0\x0020\x0C98\x0CCD\x0C98\x0CCD\x0CB0\x0020\x0C99\x0CCD\x0C99\x0CCD\x0CB0\x0020\x0C9A\x0CCD\x0C9A\x0CCD\x0CB0\x0CB0\x0C9B\x0CCD\x0C9B\x0CCD\x0CB0\x0CCD\x0CB0\x0020\x0C9C\x0CCD\x0C9C\x0CCD\x0CB0\x0020\x0C9D\x0CCD\x0C9D\x0CCD\x0CB0\x0020\x0C9E\x0CCD\x0C9E\x0CCD\x0CB0\x0020\x0C9F\x0CCD\x0C9F\x0CCD\x0CB0\x0020\x0CA0\x0CCD\x0CA0\x0CCD\x0CB0\x0020\x0CA1\x0CCD\x0CA1\x0CCD\x0CB0\x0020\x0CA2\x0CCD\x0CA2\x0CCD\x0CB0\x0020\x0CA3\x0CCD\x0CA3\x0CCD\x0CB0\x0020\x0CA4\x0CCD\x0CA4\x0CCD\x0CB0\x0020\x0CA5\x0CCD\x0CA5\x0CCD\x0CB0\x0020\x0CA6\x0CCD\x0CA6\x0CCD\x0CA4\x0020\x0CA7\x0CCD\x0CA7\x0CCD\x0CA4\x0020\x0CA8\x0CCD\x0CA8\x0CCD\x0CA4\x0020\x0CAA\x0CCD\x0CAA\x0CCD\x0CB0"
			L"\x0CAB\x0CCD\x0CAB\x0CCD\x0CA4\x0020\x0CAC\x0CCD\x0CAC\x0CCD\x0CA4\x0020\x0CAD\x0CCD\x0CAD\x0CCD\x0CA4\x0CCD\x0CA4\x0020\x0CAE\x0CCD\x0CAE\x0CCD\x0CA4\x0020\x0CAF\x0CCD\x0CAF\x0CCD\x0CDE\x0020\x0CB0\x0CCD\x0CB0\x0CCD\x0CB0\x0CCD\x0CA4\x0020\x0CB1\x0CCD\x0CB1\x0CCD\x0CDE\x0020\x0CB2\x0CCD\x0CB2\x0CCD\x0CDE\x0020\x0CB3\x0CCD\x0CB3\x0CCD\x0CDE\x0020\x0CB5\x0CCD\x0CB6\x0CCD\x0CDE\x0020\x0CB6\x0CCD\x0CB6\x0CCD\x0CDE\x0020\x0CB7\x0CCD\x0CB7\x0CCD\x0CDE\x0020\x0CB8\x0CCD\x0CB8\x0CCD\x0CB0\x0020\x0CB9\x0CCD\x0CB9\x0CCD\x0CDE\x0020\x0C95\x0CCD\x0CB7\x0CCD\x0CB0\x0020\x0C9C\x0CCD\x0C9E\x0CCD\x0CB0\x0020\x0C95\x0CCD\x0CB7\x0CCD\x0CA4\x0020\x0C9C\x0CCD\x0C9E\x0CCD\x0CA4"
			L"\x0C95\x0CBE\x0020\x0C96\x0CBE\x0020\x0C97\x0CBE\x0020\x0C98\x0CBE\x0020\x0C99\x0CBE\x0020\x0C9A\x0CBE\x0020\x0C9B\x0CBE\x0020\x0C9C\x0CBE\x0020\x0C9D\x0CBE\x0020\x0C9E\x0CBE\x0020\x0C9F\x0CBE\x0020\x0CA0\x0CBE\x0020\x0CA1\x0CBE\x0020\x0CA2\x0CBE\x0020\x0CA3\x0CBE\x0020\x0CA4\x0CBE\x0020\x0CA5\x0CBE\x0020\x0CA6\x0CBE\x0020\x0CA7\x0CBE\x0020\x0CA8\x0CBE\x0020\x0CAA\x0CBE"
			L"\x0CAB\x0CBE\x0020\x0CAC\x0CBE\x0020\x0CAD\x0CBE\x0020\x0CAE\x0CBE\x0020\x0CAF\x0CBE\x0020\x0CB0\x0CBE\x0020\x0CB1\x0CBE\x0020\x0CB2\x0CBE\x0020\x0CB3\x0CBE\x0020\x0CB5\x0CBE\x0020\x0CB6\x0CBE\x0020\x0CB7\x0CBE\x0020\x0CB8\x0CBE\x0020\x0CB9\x0CBE\x0020\x0C95\x0CCD\x0CB7\x0CBE\x0020\x0C9C\x0CCD\x0C9E\x0CBE\x0020\x0020\x0C95\x0CCD\x0CB7\x0CCD\x0CB0\x0CBE\x0020\x0C9C\x0CCD\x0C9E\x0CCD\x0CB0\x0CBE",
			0,0,
			KS60IndicTypeFaceName,
			KS60IndicFont,
			(const TText16*) L"c:\\shaperkannadadata3.dat"
			}
		};
		
const TInputData Gujarati_InputData[] =
	{
			{
				//this input data taken from the web
				(const TText16*)
				L"\x0aad\x0abe\x0ab0\x0aa4\x0020\x0ab6\x0acd\x0ab0\x0ac0\x0ab2\x0a82\x0a95\x0abe\x0020\x0ab5\x0abf\x0ab0\x0ac1\x0aa6\x0acd\x0aa7\x0020\x0aa4\x0acd\x0ab0\x0ac0"
				L"\x0a9c\x0abe\x0020\x0ab5\x0aa8\x002d\x0aa1\x0ac7\x0020\x0aae\x0ac7\x0a9a\x0aae\x0abe\x0a82\x0020\x0aaf\x0ac1\x0ab5\x0ab0\x0abe\x0a9c\x0020\x0028\x0031\x0031"
				L"\x0037\x0029\x0020\x0a85\x0aa8\x0ac7\x0020\x0ab5\x0ac0\x0ab0\x0ac7\x0aa8\x0acd\x0aa6\x0acd\x0ab0\x0020\x0ab8\x0ab9\x0ac7\x0ab5\x0abe\x0a97\x0020\x0028\x0031"
				L"\x0031\x0036\x0029\x0020\x0aa8\x0abe\x0020\x0a86\x0a95\x0acd\x0ab0\x0aae\x0a95\x0020\x0ab6\x0aa4\x0a95\x0acb\x0020\x0aac\x0abe\x0aa6\x0020\x0aaf\x0ac2\x0ab8"
				L"\x0ac1\x0aab\x0020\x0aaa\x0aa0\x0abe\x0aa3\x0020\x0028\x0033\x0038\x0020\x0aa6\x0aa1\x0abe\x0aae\x0abe\x0a82\x0020\x0a85\x0aa3\x0aa8\x0aae\x0020\x0035\x0039"
				L"\x0029\x0020\x0a85\x0aa8\x0ac7\x0020\x0a95\x0aaa\x0acd\x0aa4\x0abe\x0aa8\x0020\x0aa7\x0acb\x0aa8\x0ac0\x0aa8\x0abe\x0020\x0a85\x0aa3\x0aa8\x0aae\x0020\x0033"
				L"\x0035\x0020\x0ab0\x0aa8\x0acb\x0aa8\x0ac0\x0020\x0aae\x0aa6\x0aa6\x0aa5\x0ac0\x0020\x0035\x0030\x0020\x0a93\x0ab5\x0ab0\x0acb\x0aae\x0abe\x0a82\x0020\x0035"
				L"\x0020\x0ab5\x0abf\x0a95\x0ac7\x0a9f\x0020\x0a97\x0ac1\x0aae\x0abe\x0ab5\x0ac0\x0aa8\x0ac7\x0020\x0033\x0036\x0033\x0020\x0ab0\x0aa8\x0acb\x0aa8\x0acb\x0020"
				L"\x0aaa\x0ab9\x0abe\x0aa1\x0020\x0a89\x0aad\x0ac1\x0a82\x0020\x0a95\x0ab0\x0ab5\x0abe\x0aae\x0abe\x0a82\x0020\x0ab8\x0aab\x0ab3\x0020\x0ab0\x0ab9"
				L"\x0acd\x0aaf\x0ac1\x0a82\x0020\x0a9b\x0ac7\x002e\x0020\x0ab6\x0acd\x0ab0\x0ac0\x0ab2\x0a82\x0a95\x0abe\x0aae\x0abe\x0a82\x0020\x0aad\x0abe\x0ab0\x0aa4\x0aa8"
				L"\x0acb\x0020\x0a86\x0020\x0a89\x0a9a\x0acd\x0a9a\x0aa4\x0aae\x0020\x0ab8\x0acd\x0a95\x0acb\x0ab0\x0020\x0a9b\x0ac7",
				0,0,
				KS60IndicTypeFaceName,
				KS60IndicFont,
				(const TText16*) L"c:\\shapergujaratidata1.dat"
			},
			{
				//this input data taken from the web
				(const TText16*)
				L"\x0a86\x0020\x0a85\x0a97\x0abe\x0a89\x0020\x0aaa\x0acd\x0ab0\x0ac7\x0aae\x0aa6\x0abe\x0ab8\x0abe\x0020\x0ab8\x0acd\x0a9f\x0ac7\x0aa1"
				L"\x0abf\x0aaf\x0aae\x0aae\x0abe\x0a82\x0020\x0aad\x0abe\x0ab0\x0aa4\x0ac0\x0aaf\x0020\x0a9f\x0ac0\x0aae\x0aa8\x0abe\x0020\x0a95\x0aaa\x0acd\x0aa4\x0abe\x0aa8"
				L"\x0020\x0aae\x0ab9\x0ac7\x0aa8\x0acd\x0aa6\x0acd\x0ab0\x0020\x0ab8\x0abf\x0a82\x0ab9\x0020\x0aa7\x0acb\x0aa8\x0ac0\x0a8f\x0020\x0a9f\x0acb\x0ab8\x0020"
				L"\x0a9c\x0ac0\x0aa4\x0ac0\x0aa8\x0ac7\x0020\x0aaa\x0acd\x0ab0\x0aa5\x0aae\x0020\x0aac\x0ac7\x0a9f\x0ac0\x0a82\x0a97\x0020\x0a95\x0ab0\x0ab5\x0abe\x0aa8"
				L"\x0acb\x0020\x0aa8\x0abf\x0ab0\x0acd\x0aa3\x0aaf\x0020\x0a95\x0ab0\x0acd\x0aaf\x0acb\x002e\x0020\x0ab8\x0a9a\x0abf\x0aa8\x0020\x0aa4\x0ac7\x0a82\x0aa1\x0ac1"
				L"\x0ab2\x0a95\x0ab0\x0aa8\x0ac7\x0020\x0aab\x0ab0\x0acd\x0aa8\x0abe\x0aa8\x0acd\x0aa1\x0acb\x0a8f\x0020\x0aac\x0ac0\x0a9c\x0ac0\x0020\x0a9c\x0020\x0a93\x0ab5"
				L"\x0ab0\x0aae\x0abe\x0a82\x0020\x0a8f\x0ab2\x0aac\x0ac0\x0aa1\x0aac\x0acd\x0ab2\x0acd\x0aaf\x0ac2\x0020\x0a86\x0a89\x0a9f\x0020\x0a95\x0ab0\x0ac0\x0aa8\x0ac7"
				L"\x0020\x0aae\x0acb\x0a9f\x0acb\x0020\x0a86\x0a82\x0a9a\x0a95\x0acb\x0020\x0a86\x0aaa\x0acd\x0aaf\x0acb\x002e\x0020\x0ab8\x0a9a\x0abf\x0aa8\x0020"
				L"\x0aac\x0abe\x0aa6\x0020\x0a97\x0acc\x0aa4\x0aae\x0020\x0a97\x0a82\x0aad\x0ac0\x0ab0\x0020\x0aaa\x0aa3\x0020\x0031\x0030\x0020\x0ab0\x0aa8\x0aa8\x0abe\x0020"
				L"\x0aa8\x0a9c\x0ac0\x0ab5\x0abe\x0020\x0ab8\x0acd\x0a95\x0acb\x0ab0\x0ac7\x0020\x0ab0\x0aa8\x0020\x0a86\x0a89\x0a9f\x0020\x0aa5\x0a88\x0020\x0a97"
				L"\x0aaf\x0abe\x0a82\x002e\x000d\x000a\x0a86\x0020\x0ab8\x0aa4\x0aa4\x0020\x0aa4\x0acd\x0ab0\x0ac0\x0a9c\x0ac0\x0020\x0ab5\x0abe\x0ab0\x0020\x0a9b\x0ac7\x0020"
				L"\x0a9c\x0acd\x0aaf\x0abe\x0ab0\x0ac7\x0020\x0ab8\x0a9a\x0abf\x0aa8\x0020\x0aa4\x0ac7\x0a82\x0aa1\x0ac1\x0ab2\x0a95\x0ab0\x0aa8\x0ac7\x0020\x0a85\x0a82"
				L"\x0aaa\x0abe\x0aaf\x0ab0\x0acb\x0a8f\x0020\x0a8f\x0ab2\x0aac\x0ac0\x0aa1\x0aac\x0acd\x0ab2\x0acd\x0aaf\x0ac1\x0020\x0a86\x0a89\x0a9f\x0020\x0a86"
				L"\x0aaa\x0acd\x0aaf\x0abe\x0a82\x0020\x0a9b\x0ac7\x002e\x0020\x0a86\x0ab6\x0acd\x0ab5\x0ab0\x0acd\x0aaf\x0aa8\x0ac0\x0020\x0ab5\x0abe\x0aa4\x0020\x0aa4"
				L"\x0acb\x0020\x0a8f\x0020\x0aaa\x0aa3\x0020\x0a9b\x0ac7\x0020\x0a95\x0ac7\x002c\x0020\x0a86\x0020\x0aa4\x0acd\x0ab0\x0aa3\x0ac7\x0aaf\x0020\x0aaa\x0acd\x0ab0"
				L"\x0ab8\x0a82\x0a97\x0acb\x0aae\x0abe\x0a82\x0020\x0ab8\x0a9a\x0abf\x0aa8\x0020\x0a85\x0a82\x0aaa\x0abe\x0aaf\x0ab0\x0acb\x0aa8\x0abe\x0020\x0a96"
				L"\x0acb\x0a9f\x0abe\x0020\x0aa8\x0abf\x0ab0\x0acd\x0aa3\x0aaf\x0aa8\x0acb\x0020\x0aad\x0acb\x0a97\x0020\x0aac\x0aa8\x0acd\x0aaf\x0abe\x0a82\x0020\x0a9b\x0ac7"
				L"\x002e\x0020\x0a86\x0a9c\x0ac7\x0020\x0aaa\x0aa3\x0020\x0aab\x0ab0\x0acd\x0aa8\x0abe\x0aa8\x0acd\x0aa1\x0acb\x0aa8\x0abe\x0020\x0ab2\x0ac7\x0a97\x0020\x0ab8"
				L"\x0acd\x0a9f\x0aae\x0acd\x0aaa\x0acd\x0ab8\x0aa5\x0ac0\x0020\x0aac\x0ab9\x0abe\x0ab0\x0020\x0a9c\x0aa4\x0abe\x0020\x0aa6\x0aa1\x0abe\x0020\x0aaa\x0ab0\x0020"
				L"\x0aa4\x0ac7\x0aae\x0aa8\x0ac7\x0020\x0a8f\x0ab2\x0aac\x0ac0\x0aa1\x0aac\x0ab2\x0acd\x0aaf\x0ac2\x0020\x0a86\x0a89\x0a9f\x0020\x0a86\x0aaa\x0ac0\x0020\x0aa6"
				L"\x0ac7\x0ab5\x0abe\x0aae\x0abe\x0a82\x0020\x0a86\x0ab5\x0acd\x0aaf\x0abe\x002e\x0020",
				0,0,
				KS60IndicTypeFaceName,
				KS60IndicFont,
				(const TText16*) L"c:\\shapergujaratidata2.dat"
			},
			{
				//this input data has been given by the licensee
				(const TText16*)
				L"\x0AAA\x0ACD\x0AB0\x0AA4\x0ABF\x0AB7\x0ACD\x0AA0\x0ABE\x0020\x0A85\x0AA8\x0AC7\x0020\x0A85\x0AA7\x0ABF\x0A95\x0ABE\x0AB0\x0ACB\x0AA8\x0AC0\x0020\x0AA6\x0AC3\x0AB7\x0ACD\x0A9F\x0ABF\x0A8F\x0020\x0AB8\x0AB0\x0ACD\x0AB5\x0020\x0AAE\x0ABE\x0AA8\x0AB5\x0ACB\x0020\x0A9C\x0AA8\x0ACD\x0AAE\x0AA5\x0AC0\x0020\x0AB8\x0ACD\x0AB5\x0AA4\x0A82\x0AA4\x0ACD\x0AB0\x0020\x0A85\x0AA8\x0AC7\x0020\x0AB8\x0AAE\x0ABE\x0AA8\x0020\x0AB9\x0ACB\x0AAF\x0020\x0A9B\x0AC7"
				L"\x0AA4\x0AC7\x0AAE\x0AA8\x0ABE\x0AAE\x0ABE\x0A82\x0020\x0AB5\x0ABF\x0A9A\x0ABE\x0AB0\x0AB6\x0A95\x0ACD\x0AA4\x0ABF\x0020\x0A85\x0AA8\x0AC7\x0020\x0A85\x0A82\x0AA4\x0A83\x0A95\x0AB0\x0AA3\x0020\x0AB9\x0ACB\x0AAF\x0020\x0A9B\x0AC7\x0020\x0A85\x0AA8\x0AC7\x0020\x0AA4\x0AC7\x0AAE\x0AA3\x0AC7\x0020\x0AAA\x0AB0\x0AB8\x0ACD\x0AAA\x0AB0\x0020\x0AAC\x0A82\x0AA7\x0AC1\x0AA4\x0ACD\x0AB5\x0AA8\x0AC0\x0020\x0AAD\x0ABE\x0AB5\x0AA8\x0ABE\x0AAF\x0AC0\x0020\x0AB5\x0AB0\x0ACD\x0AA4\x0AB5\x0AC1\x0A82"
				L"\x0A9C\x0ACB\x0A88\x0A8F"
				L"\x0AB0\x0ABE\x0AB7\x0ACD\x0A9F\x0ACD\x0AB0\x0AC0\x0AAF\x0020\x0AB6\x0ABF\x0A95\x0ACD\x0AB7\x0AA3\x0020\x0AA8\x0AC0\x0AA4\x0AC0\x0AA8\x0ABE\x0020\x0A85\x0AA8\x0AC1\x0AB8\x0A82\x0AA7\x0ABE\x0AA8\x0AAE\x0ABE\x0A82\x0020\x0AA4\x0AC8\x0AAF\x0ABE\x0AB0\x0020\x0AA5\x0AAF\x0AC7\x0AB2\x0ABE\x0020\x0AA8\x0AB5\x0ABE\x0020\x0A85\x0AAD\x0ACD\x0AAF\x0ABE\x0AB8\x0A95\x0ACD\x0AB0\x0AAE\x0AAE\x0ABE\x0A82"
				L"\x0AA6\x0AB0\x0AC7\x0A95\x0020\x0AB5\x0ABF\x0AB7\x0AAF\x0AA8\x0ABE\x0020\x0AAA\x0ABE\x0AA0\x0ACD\x0AAF\x0A95\x0ACD\x0AB0\x0AAE\x0AA8\x0AC7\x0020\x0AB5\x0AA7\x0AC1\x0020\x0AB8\x0AA7\x0AA8\x0020\x0A85\x0AA8\x0AC7\x0020\x0A85\x0AA7\x0ACD\x0AAF\x0AA4\x0AA8\x0020\x0A95\x0AB0\x0AB5\x0ABE\x0AA8\x0ACB\x0020\x0AAA\x0ACD\x0AB0\x0AAF\x0AA4\x0ACD\x0AA8\x0020\x0A95\x0AB0\x0AB5\x0ABE\x0AAE\x0ABE\x0A82"
				L"\x0A86\x0AB5\x0ACD\x0AAF\x0ACB\x0020\x0A9B\x0AC7\x002E\x0020\x0020\x0AB0\x0ABE\x0AB7\x0ACD\x0A9F\x0ACD\x0AB0\x0AC0\x0AAF\x0020\x0AB6\x0ABF\x0A95\x0ACD\x0AB7\x0AA3\x0020\x0AA8\x0AC0\x0AA4\x0ABF\x0AA8\x0ABE\x0020\x0AAE\x0AC1\x0AB8\x0AA6\x0ABE\x0AAE\x0ABE\x0A82\x0020\x0AB8\x0AAE\x0ABE\x0AA8\x0020\x0A85\x0AAD\x0ACD\x0AAF\x0ABE\x0AB8\x0A95\x0ACD\x0AB0\x0AAE\x0ACB\x0AA8\x0AC0\x0020\x0A85\x0AAE\x0AB2"
				L"\x0A95\x0AB0\x0AB5\x0ABE\x0020\x0AB8\x0ABE\x0AA5\x0AC7\x0020\x0A95\x0AC7\x0A9F\x0AB2\x0ABE\x0A82\x0A95\x0020\x0AB9\x0ABE\x0AB0\x0ACD\x0AA6\x0AB6\x0ACD\x0AAA\x0020\x0AA4\x0AA4\x0ACD\x0AA4\x0ACD\x0AB5\x0ACB\x0020\x0AA6\x0ABE\x0A96\x0AB2\x0020\x0A95\x0AB0\x0AB5\x0ABE\x0AA8\x0AC1\x0A82\x0020\x0AA8\x0A95\x0ACD\x0A95\x0AC0\x0020\x0A95\x0AB0\x0AB5\x0ABE\x0AAE\x0ABE\x0A82\x0020\x0A85\x0AB5\x0ACD\x0AAF\x0AC1\x0A82\x0020\x0A9B\x0AC7\x002E"
				L"\x0A86\x0020\x0AB9\x0ABE\x0AB0\x0ACD\x0AA6\x0AB6\x0ACD\x0AAA\x0020\x0AA4\x0AA4\x0ACD\x0AA4\x0ACD\x0AB5\x0ACB\x0A82\x0AAE\x0ABE\x0A82\x0020\x0AAD\x0ABE\x0AB0\x0AA4\x0AA8\x0AC0\x0020\x0AB8\x0ACD\x0AB5\x0AA4\x0A82\x0AA4\x0ACD\x0AB0\x0ACD\x0AAF\x0020\x0A9A\x0AB3\x0AB5\x0AB3\x002C\x0020\x0A86\x0AB5\x0AA3\x0AC0\x0020\x0AAC\x0A82\x0AA7\x0ABE\x0AB0\x0AA3\x0AC0\x0AAF\x0020\x0A9C\x0AB5\x0ABE\x0AAC\x0AA6\x0ABE\x0AB0\x0AC0\x0A93\x002C"
				L"\x0AB0\x0ABE\x0AB7\x0ACD\x0A9F\x0ACD\x0AB0\x0AC0\x0AAF\x0020\x0A90\x0A95\x0ACD\x0AAF\x002C\x0020\x0A86\x0AB5\x0AA3\x0ACB\x0020\x0AB8\x0ABE\x0A82\x0AB8\x0ACD\x0A95\x0AC3\x0AA4\x0ABF\x0A95\x0020\x0AB5\x0ABE\x0AB0\x0AB8\x0ACB\x002C\x0020\x0AB8\x0AB0\x0ACD\x0AB5\x0A9C\x0AA8\x0AB8\x0AAE\x0ABE\x0AA8\x0AA4\x0ABE\x002C\x0020\x0AB2\x0ACB\x0A95\x0AB6\x0ABE\x0AB9\x0AC0\x0020\x0A85\x0AA8\x0AC7"
				L"\x0AAC\x0ABF\x0AA8\x0AB8\x0ABE\x0A82\x0AAA\x0ACD\x0AB0\x0AA6\x0ABE\x0AAF\x0ABF\x0A95\x0AA4\x0ABE\x002C\x0020\x0A9C\x0ABE\x0AA4\x0ABE\x0AAF\x0020\x0AB8\x0AAE\x0ABE\x0AA8\x0AA4\x0ABE\x002C\x0020\x0AAA\x0AB0\x0ACD\x0AAF\x0ABE\x0AB5\x0AB0\x0AA3\x0020\x0AB8\x0AC1\x0AB0\x0A95\x0ACD\x0AB7\x0ABE\x002C\x0020\x0AB8\x0ABE\x0AAE\x0ABE\x0A9C\x0ABF\x0A95\x0020\x0A85\x0AB5\x0AB0\x0ACB\x0AA7\x0ACB\x0AA8\x0AC1\x0A82"
				L"\x0AA8\x0ABF\x0AB5\x0ABE\x0AB0\x0AA3\x0020\x0AA4\x0AA5\x0ABE\x0020\x0AB5\x0AC8\x0A9C\x0ACD\x0A9E\x0ABE\x0AA8\x0ABF\x0A95\x0020\x0A95\x0AC7\x0AB3\x0AB5\x0AA3\x0AC0\x0020\x0A9C\x0AC7\x0AB5\x0AC0\x0020\x0AAC\x0ABE\x0AAC\x0AA4\x0ACB\x0AA8\x0AC0\x0020\x0AB8\x0AAE\x0ABE\x0AB5\x0AC7\x0AB6\x0020\x0AA5\x0ABE\x0AAF\x0020\x0A9B\x0AC7\x002E\x0020\x0AB5\x0AB3\x0AC0"
				L"\x0AAE\x0ABE\x0AA7\x0ACD\x0AAF\x0AAE\x0ABF\x0A95\x0020\x0A95\x0AC7\x0020\x0A89\x0AAF\x0ACD\x0AAF\x0AA4\x0AB0\x0020\x0AAE\x0ABE\x0AA7\x0ACD\x0AAF\x0AAE\x0ABF\x0A95\x0020\x0A95\x0A95\x0ACD\x0AB7\x0ABE\x0A8F\x0020\x0A86\x0020\x0AAC\x0AA7\x0AC0\x0020\x0A9C\x0020\x0AAC\x0ABE\x0AAC\x0AA4\x0ACB\x0AA8\x0AC1\x0A82\x0020\x0AB5\x0AA7\x0AC1\x0020\x0AB8\x0AA7\x0AA8"
				L"\x0AB6\x0ABF\x0A95\x0ACD\x0AB7\x0AA3\x0020\x0A86\x0AAA\x0AB5\x0ABE\x0AA8\x0ACB\x0020\x0A89\x0AA6\x0ACD\x0AA6\x0AC7\x0AB6\x0020\x0AAA\x0AA3\x0020\x0AB8\x0ACD\x0AB5\x0AC0\x0A95\x0ABE\x0AB0\x0ABE\x0AAF\x0ACB\x0020\x0A9B\x0AC7",
				0,0,
				KS60IndicTypeFaceName,
				KS60IndicFont,
				(const TText16*) L"c:\\shapergujaratidata3.dat"
			}
		};

const TInputData Bengali_InputData[] =
	{
			{
				(const TText16*)
				L"\x098F\x0995\x0020\x09AC\x09CB\x0995\x09BE\x0020\x099C\x09CB\x09B2\x09BE\x0020\x099B\x09BF\x09B2\x0964\x0020\x09B8\x09C7\x0020\x0020\x098F\x0995\x09A6\x09BF\x09A8\x0020\x0995\x09BE\x09B8\x09CD\x09A4\x09C7\x0020\x09A8\x09BF\x09DF\x09C7\x0020\x09A7\x09BE\x09A8\x0020\x0995\x09BE\x099F\x09A4\x09C7\x0020\x0997\x09BF\x09DF\x09C7\x0020\x0996\x09C7\x09A4\x09C7\x09B0\x0020\x09AE\x09BE\x099D\x0996\x09BE\x09A8\x09C7\x0987\x0020\x0998\x09C1\x09AE\x09BF\x09DF\x09C7\x0020"
				L"\x09AA\x09DC\x09B2\x0964\x0020\x0998\x09C1\x09AE\x0020\x09A5\x09C7\x0995\x09C7\x0020\x0989\x09A0\x09C7\x0020\x0986\x09AC\x09BE\x09B0\x0020\x0995\x09BE\x09B8\x09CD\x09A4\x09C7\x0020\x09B9\x09BE\x09A4\x09C7\x0020\x09A8\x09BF\x09DF\x09C7\x0020\x09A6\x09C7\x0996\x09B2\x002C\x0020\x09B8\x09C7\x099F\x09BE\x0020\x09AC\x09A1\x09CD\x09A1\x0020\x0997\x09B0\x09AE\x0020\x09B9\x09DF\x09C7\x099B\x09C7\x0964\x0020\x0995\x09BE\x09B8\x09CD\x09A4\x09C7\x0996\x09BE\x09A8\x09BE\x0020"
				L"\x09B0\x09CB\x09A6\x0020\x09B2\x09C7\x0997\x09C7\x0020\x0997\x09B0\x09AE\x0020\x09B9\x09DF\x09C7\x099B\x09BF\x09B2\x002C\x0020\x0995\x09BF\x09A8\x09CD\x09A4\x09C1\x0020\x099C\x09CB\x09B2\x09BE\x0020\x09AD\x09BE\x09AC\x09B2\x09C7\x0020\x09A4\x09BE\x09B0\x0020\x099C\x09CD\x09AC\x09B0\x0020\x09B9\x09DF\x09C7\x099B\x09C7\x0964\x0020\x09A4\x0996\x09A8\x0020\x09B8\x09C7\x0020\x0060\x0986\x09AE\x09BE\x09B0\x0020\x0995\x09BE\x09B8\x09CD\x09A4\x09C7\x0020\x09A4\x09CB\x0020"
				L"\x09AE\x09B0\x09C7\x0020\x09AF\x09BE\x09AC\x09C7\x0020\x09B0\x09C7\x0021\x0027\x0020\x09AC\x09B2\x09C7\x0020\x09B9\x09BE\x0989\x0020\x09B9\x09BE\x0989\x0020\x0995\x09B0\x09C7\x0020\x0995\x09BE\x0981\x09A6\x09A4\x09C7\x0020\x09B2\x09BE\x0997\x09B2\x0964\x0020\x09AA\x09BE\x09B6\x09C7\x09B0\x0020\x0996\x09C7\x09A4\x09C7\x0020\x098F\x0995\x0020\x099A\x09BE\x09B7\x09BE\x0020\x0995\x09BE\x099C\x0020\x0995\x09B0\x099B\x09BF\x09B2\x0964\x0020\x099C\x09CB\x09B2\x09BE\x09B0\x0020"
				L"\x0995\x09BE\x09A8\x09CD\x09A8\x09BE\x0020\x09B6\x09C1\x09A8\x09C7\x0020\x09B8\x09C7\x0020\x09AC\x09B2\x09CD\x09B2\x002C\x0020\x0060\x0995\x09BF\x0020\x09B9\x09DF\x09C7\x099B\x09C7\x003F\x0027\x0020\x099C\x09CB\x09B2\x09BE\x0020\x09AC\x09B2\x09CD\x09B2\x09C7\x002C\x0020\x0060\x0986\x09AE\x09BE\x09B0\x0020\x0995\x09BE\x09B8\x09CD\x09A4\x09C7\x09B0\x0020\x099C\x09CD\x09AC\x09B0\x0020\x09B9\x09DF\x09C7\x099B\x09C7\x0964\x0027\x0020\x09A4\x09BE\x0020\x09B6\x09C1\x09A8\x09C7\x0020",
				0,0,
				KS60IndicTypeFaceName,
				KS60IndicFont,
				(const TText16*) L"c:\\shaperbengalidata1.dat"
			},
			{
				(const TText16*)
				L"\x099A\x09BE\x09B7\x09BE\x0020\x09B9\x09BE\x09B8\x09A4\x09C7\x0020\x09B9\x09BE\x09B8\x09A4\x09C7\x0020\x09AC\x09B2\x09CD\x09B2\x09C7\x002C\x0020\x0060\x0993\x0995\x09C7\x0020\x099C\x09B2\x09C7\x0020\x09A1\x09C1\x09AC\x09BF\x09DF\x09C7\x0020\x09B0\x09BE\x0996\x002C\x0020\x099C\x09CD\x09AC\x09B0\x0020\x09B8\x09C7\x09B0\x09C7\x0020\x09AF\x09BE\x09AC\x09C7\x0964\x0027\x0020\x099C\x09B2\x09C7\x0020\x09A1\x09C1\x09AC\x09BF\x09DF\x09C7\x0020\x0995\x09BE\x09B8\x09CD\x09A4\x09C7\x0020"
				L"\x09A0\x09BE\x09A3\x09CD\x09A1\x09BE\x0020\x09B9\x09B2\x002C\x0020\x099C\x09CB\x09B2\x09BE\x0993\x0020\x0996\x09C1\x09AC\x0020\x09B8\x09C1\x0996\x09C0\x0020\x09B9\x09B2\x0964\x0020\x09A4\x09BE\x09B0\x09AA\x09B0\x0020\x098F\x0995\x09A6\x09BF\x09A8\x0020\x099C\x09CB\x09B2\x09BE\x09B0\x0020\x09AE\x09BE\x09DF\x09C7\x09B0\x0020\x099C\x09CD\x09AC\x09B0\x0020\x09B9\x09DF\x09C7\x099B\x09C7\x0964\x0020\x09B8\x0995\x09B2\x09C7\x0020\x09AC\x09B2\x09CD\x09B2\x09C7\x002C\x0020"
				L"\x0060\x09AC\x09A6\x09CD\x09A6\x09BF\x0020\x09A1\x09BE\x0995\x0964\x0027\x0020\x099C\x09CB\x09B2\x09BE\x0020\x09AC\x09B2\x09CD\x09B2\x09C7\x002C\x0020\x0060\x0986\x09AE\x09BF\x0020\x0993\x09B7\x09C1\x09A7\x0020\x099C\x09BE\x09A8\x09BF\x0964\x0027\x0020\x09AC\x09B2\x09C7\x002C\x0020\x09B8\x09C7\x0020\x09A4\x09BE\x09B0\x0020\x09AE\x09BE\x0995\x09C7\x0020\x09AA\x09C1\x0995\x09C1\x09B0\x09C7\x0020\x09A8\x09BF\x09DF\x09C7\x0020\x0997\x09BF\x09DF\x09C7\x0020"
				L"\x099C\x09B2\x09C7\x09B0\x0020\x09AD\x09BF\x09A4\x09B0\x09C7\x0020\x099A\x09C7\x09AA\x09C7\x0020\x09A7\x09B0\x09B2\x0964\x0020\x09B8\x09C7\x0020\x09AC\x09C7\x099A\x09BE\x09B0\x09C0\x0020\x09AF\x09A4\x0987\x0020\x099B\x099F\x09AA\x099F\x0020\x0995\x09B0\x09C7\x002C\x0020\x099C\x09CB\x09B2\x09BE\x0020\x09A4\x09A4\x0987\x0020\x0986\x09B0\x09CB\x0020\x099A\x09C7\x09AA\x09C7\x0020\x09A7\x09B0\x09C7\x002C\x0020\x0986\x09B0\x0020\x09AC\x09B2\x09C7\x002C\x0020",
				0,0,
				KS60IndicTypeFaceName,
				KS60IndicFont,
				(const TText16*) L"c:\\shaperbengalidata2.dat"
			},
			{
				(const TText16*)
				L"\x0060\x09B0\x09CB\x09B8\x002C\x0020\x098F\x0987\x0020\x09A4\x09CB\x0020\x099C\x09CD\x09AC\x09B0\x0020\x09B8\x09BE\x09B0\x099B\x09C7\x0964\x0027\x0020\x09A4\x09BE\x09B0\x09AA\x09B0\x0020\x09AF\x0996\x09A8\x0020\x09AC\x09C1\x09DC\x09BF\x0020\x0986\x09B0\x0020\x09A8\x09DC\x099B\x09C7\x002D\x099A\x09DC\x099B\x09C7\x0020\x09A8\x09BE\x002C\x0020\x09A4\x0996\x09A8\x0020\x09A4\x09BE\x0995\x09C7\x0020\x09A4\x09C1\x09B2\x09C7\x0020\x09A6\x09C7\x0996\x09C7\x0020"
				L"\x09B8\x09C7\x0020\x09AE\x09B0\x09C7\x0020\x0997\x09C7\x099B\x09C7\x0964\x0020\x09A4\x0996\x09A8\x0020\x099C\x09CB\x09B2\x09BE\x0020\x099A\x09C7\x0981\x099A\x09BF\x09DF\x09C7\x0020\x0995\x09BE\x0981\x09A6\x09A4\x09C7\x0020\x09B2\x09BE\x0997\x09B2\x0020\x09A4\x09BF\x09A8\x09A6\x09BF\x09A8\x0020\x0995\x09BF\x099B\x09C1\x0020\x0996\x09C7\x09B2\x0020\x09A8\x09BE\x002C\x0020\x09AA\x09C1\x0995\x09C1\x09B0\x002D\x09AA\x09BE\x09DC\x0020\x09A5\x09C7\x0995\x09C7\x0020"
				L"\x0998\x09B0\x09C7\x0993\x0020\x0997\x09C7\x09B2\x0020\x09A8\x09BE\x0964\x0020\x098F\x0995\x0020\x09B6\x09BF\x09DF\x09BE\x09B2\x0020\x09B8\x09C7\x0987\x0020\x099C\x09CB\x09B2\x09BE\x09B0\x0020\x09AC\x09A8\x09CD\x09A7\x09C1\x0020\x099B\x09BF\x09B2\x0964\x0020\x09B8\x09C7\x0020\x099C\x09CB\x09B2\x09BE\x0995\x09C7\x0020\x0995\x09BE\x0981\x09A6\x09A4\x09C7\x0020\x09A6\x09C7\x0996\x09C7\x0020\x098F\x09B8\x09C7\x0020\x09AC\x09B2\x09CD\x09B2\x09C7\x002C\x0020"
				L"\x0060\x09AC\x09A8\x09CD\x09A7\x09C1\x002C\x0020\x09A4\x09C1\x09AE\x09BF\x0020\x0995\x09C7\x0981\x09A6\x0020\x09A8\x09BE\x002C\x0020\x09A4\x09CB\x09AE\x09BE\x0995\x09C7\x0020\x09B0\x09BE\x099C\x09BE\x09B0\x0020\x09AE\x09C7\x09DF\x09C7\x0020\x09AC\x09BF\x09DF\x09C7\x0020\x0995\x09B0\x09BE\x09AC\x0964\x0027\x0020\x09B6\x09C1\x09A8\x09C7\x0020\x099C\x09CB\x09B2\x09BE\x0020\x099A\x09CB\x0996\x0020\x09AE\x09C1\x099B\x09C7\x0020\x0998\x09B0\x09C7\x0020"
				L"\x0997\x09C7\x09B2\x0964\x0020\x09A4\x09BE\x09B0\x09AA\x09B0\x0020\x09A5\x09C7\x0995\x09C7\x0020\x09B8\x09C7\x0020\x09B0\x09CB\x099C\x0020\x09B6\x09BF\x09DF\x09BE\x09B2\x0995\x09C7\x0020\x09AC\x09B2\x09C7\x002C\x0020\x0060\x0995\x0987\x0020\x09AC\x09A8\x09CD\x09A7\x09C1\x002C\x0020\x09B8\x09C7\x0987\x0020\x09AF\x09C7\x0020\x09AC\x09B2\x09C7\x099B\x09BF\x09B2\x09C7\x003F\x0027",
				0,0,
				KS60IndicTypeFaceName,
				KS60IndicFont,
				(const TText16*) L"c:\\shaperbengalidata3.dat"
			}
		};

const TInputData Tamil_InputData[] =
	{
			{
			(const TText16*)
			L"\x0B95\x0B9F\x0BB5\x0BC1\x0BB3\x0BCD\x0020\x0BB5\x0BBE\x0BB4\x0BCD\x0BA4\x0BCD\x0BA4\x0BC1\x0020"
			L"\x0BAA\x0BC2\x0BB5\x0BBF\x0BA9\x0BCD\x0BAE\x0BC7\x0BB2\x0BCD\x0020\x0BB5\x0BA8\x0BCD\x0BA4\x0BB0\x0BC1\x0BB3\x0BC1\x0BAE\x0BCD\x0020\x0BAA\x0BC1\x0B99\x0BCD\x0B95\x0BB5\x0BA9\x0BCD\x0BA4\x0BA9\x0BCD\x0020\x0BAA\x0BCA\x0BB1\x0BCD\x0BAA\x0BBE\x0BA4\x0BAE\x0BCD\x0020"
			L"\x0BA8\x0BBE\x0BB5\x0BBF\x0BA9\x0BBE\x0BB2\x0BCD\x0020\x0BA8\x0BBE\x0BB3\x0BC1\x0BAE\x0BCD\x0020\x0BA8\x0BB5\x0BBF\x0BA9\x0BCD\x0BB1\x0BC7\x0BA4\x0BCD\x0BA4\x0BBF\x0020\x0BAE\x0BC7\x0BB5\x0BC1\x0BAE\x0BC1\x0B9F\x0BBF\x0020"
			L"\x0BAA\x0BC6\x0BB2\x0BCD\x0BB2\x0BBE\x0BAE\x0BCD\x0020\x0B89\x0BA3\x0BB0\x0020\x0B8E\x0BB4\x0BC1\x0BA4\x0BCD\x0BA4\x0BBF\x0BA9\x0BCD\x0020\x0B87\x0BB2\x0B95\x0BCD\x0B95\x0BA3\x0BA4\x0BCD\x0BA4\x0BC8\x0B9A\x0BCD\x0020"
			L"\x0B9A\x0BCA\x0BB2\x0BCD\x0BB2\x0BBE\x0BB2\x0BCD\x0020\x0B89\x0BB0\x0BC8\x0BAA\x0BCD\x0BAA\x0BA9\x0BCD\x0020\x0BA4\x0BCA\x0B95\x0BC1\x0BA4\x0BCD\x0BA4\x0BC1\x0020"
			L"\x0B85\x0BB5\x0BC8\x0BAF\x0B9F\x0B95\x0BCD\x0B95\x0BAE\x0BCD\x0020"
			L"\x0B89\x0BA3\x0BCD\x0BA3\x0020\x0BAE\x0BC1\x0B9F\x0BBF\x0BAF\x0BBE\x0BA4\x0020\x0BB5\x0BCB\x0BA4\x0BA8\x0BC0\x0BB0\x0BCD\x0020\x0BB5\x0BBE\x0BA9\x0BCD\x0BB5\x0BBE\x0BAF\x0BCD\x0BAA\x0BCD\x0BAA\x0B9F\x0BCD\x0020"
			L"\x0B9F\x0BC6\x0BA3\x0BCD\x0BA3\x0020\x0B85\x0BAE\x0BC1\x0BA4\x0BBE\x0BA9\x0020\x0BA4\x0BBF\x0BB2\x0BCD\x0BB2\x0BC8\x0BAF\x0BCB\x0020\x0020\x0BAE\x0BA3\x0BCD\x0BA3\x0BBF\x0BA9\x0BCD\x0BAE\x0BC7\x0BB2\x0BCD\x0020"
			L"\x0BA8\x0BB2\x0BCD\x0BB2\x0BBE\x0BB0\x0BC8\x0B9A\x0BCD\x0020\x0B9A\x0BC7\x0BB0\x0BCD\x0BA8\x0BCD\x0BA4\x0BB2\x0BBE\x0BB2\x0BCD\x0020\x0BA8\x0BBE\x0BA9\x0BCD\x0B9A\x0BCA\x0BA9\x0BCD\x0BA9\x0020\x0BAA\x0BC1\x0BA9\x0BCD\x0B9A\x0BCA\x0BB2\x0BCD\x0BB2\x0BC1\x0BAE\x0BCD\x0020"
			L"\x0B8E\x0BB2\x0BCD\x0BB2\x0BCB\x0BB0\x0BC1\x0BAE\x0BCD\x0020\x0B95\x0BC8\x0B95\x0BCA\x0BB3\x0BCD\x0BB5\x0BB0\x0BCD\x0020\x0B88\x0B99\x0BCD\x0B95\x0BC1\x0020"
			L"\x0B86\x0BB5\x0BBF\x0020\x0B85\x0B95\x0BB0\x0BAE\x0BC1\x0BA4\x0BB2\x0BCD\x0020\x0B86\x0BAF\x0BBF\x0BB0\x0BA3\x0BCD\x0B9F\x0BBE\x0BAF\x0BCD\x0020\x0B86\x0BAF\x0BCD\x0BA4\x0BAE\x0BBF\x0B9F\x0BC8\x0020",
			0,0,
			KS60IndicTypeFaceName,
			KS60IndicFont,
			(const TText16*) L"c:\\shapertamildata1.dat"
			},
			{
			(const TText16*)
			L"\x0BAE\x0BC6\x0BB5\x0BC1\x0B99\x0BCD\x0020\x0B95\x0B95\x0BB0\x0BAE\x0BC1\x0BA4\x0BA9\x0BCD\x0020\x0BAE\x0BC6\x0BAF\x0BCD\x0B95\x0BB3\x0BBE\x0BAE\x0BCD\x0020\x0BAE\x0BC2\x0BB5\x0BBE\x0BB1\x0BC1\x0B99\x0BCD\x0020"
			L"\x0B95\x0BA3\x0BCD\x0BA3\x0BC1\x0020\x0BAE\x0BC1\x0BB1\x0BC8\x0BAE\x0BC8\x0BAF\x0BBE\x0BB1\x0BCD\x0020\x0B95\x0BBE\x0B9F\x0BCD\x0B9F\x0BBF\x0BAF\x0BAE\x0BC1\x0BAA\x0BCD\x0020\x0BAA\x0BA4\x0BCD\x0BA4\x0BCA\x0BA9\x0BCD\x0BB1\x0BC1\x0020"
			L"\x0BA8\x0BA3\x0BCD\x0BA3\x0BC1\x0BAE\x0BC1\x0BA4\x0BB2\x0BCD\x0020\x0BB5\x0BC8\x0BAA\x0BCD\x0BAA\x0BBE\x0B95\x0BC1\x0020\x0BA8\x0BA9\x0BCD\x0B95\x0BC1\x0020"
			L"\x0B86\x0BA9\x0BCD\x0BB1\x0BB5\x0BC1\x0BAF\x0BBF\x0BB0\x0BCD\x0020\x0B88\x0BB0\x0BBE\x0BB1\x0BC1\x0BAE\x0BCD\x0020\x0B90\x0B99\x0BCD\x0B95\x0BC1\x0BB1\x0BBF\x0BB2\x0BCD\x0020\x0B8F\x0BB4\x0BCD\x0BA8\x0BC6\x0B9F\x0BBF\x0BB2\x0BBE\x0BAE\x0BCD\x0020"
			L"\x0BCD\x0020\x0B8A\x0BA9\x0BCD\x0BB1\x0BBF\x0BAF\x0020"
			L"\x0BB5\x0BA9\x0BCD\x0BAE\x0BC8\x0BAF\x0BC7\x0020\x0BAE\x0BC6\x0BA9\x0BCD\x0BAE\x0BC8\x0020\x0BAF\x0BBF\x0B9F\x0BC8\x0BAE\x0BC8\x0BAF\x0BBE\x0BAE\x0BCD\x0020\x0BB5\x0BBE\x0B9F\x0BCD\x0B95\x0BA3\x0BCD\x0BA3\x0BBE\x0BAF\x0BCD\x0020"
			L"\x0BA4\x0BCA\x0BA9\x0BCD\x0BAE\x0BC8\x0020\x0BAE\x0BC1\x0BAF\x0BB1\x0BCD\x0B9A\x0BBF\x0BAF\x0BBE\x0BB2\x0BCD\x0020\x0BA4\x0BCA\x0B95\x0BCD\x0B95\x0BC1\x0020"
			L"\x0B93\x0B99\x0BCD\x0B95\x0BC1\x0BAF\x0BBF\x0BB0\x0BCD\x0B95\x0BB3\x0BCD\x0020\x0B92\x0BB1\x0BCD\x0BB1\x0BBF\x0BB2\x0BCD\x0BAE\x0BC7\x0BB2\x0BCD\x0020\x0B8F\x0BB1\x0BBF\x0020\x0B89\x0BAF\x0BBF\x0BB0\x0BCD\x0BAE\x0BC6\x0BAF\x0BCD\x0BAF\x0BBE\x0BAF\x0BCD\x0020"
			L"\x0B86\x0B99\x0BCD\x0B95\x0BBF\x0BB0\x0BC1\x0020\x0BA8\x0BC2\x0BB1\x0BCD\x0BB1\x0BCA\x0BB0\x0BC1\x0BAA\x0BA4\x0BCD\x0020\x0BA4\x0BBE\x0BB1\x0BBE\x0B95\x0BC1\x0BAE\x0BCD\x0020\x0BAA\x0BBE\x0B99\x0BCD\x0B95\x0BC1\x0B9F\x0BC8\x0BAF\x0020"
			L"\x0BB5\x0BB2\x0BCD\x0BB2\x0BCA\x0BB1\x0BCD\x0BB1\x0BC1\x0020\x0BAE\x0BC6\x0BB2\x0BCD\x0BB2\x0BCA\x0BB1\x0BCD\x0BB1\x0BC1\x0020\x0BB5\x0BB0\x0BCD\x0B95\x0BCD\x0B95\x0BAE\x0BCD\x0020\x0B85\x0BB3\x0BAA\x0BC6\x0B9F\x0BC8\x0B95\x0BB3\x0BCD\x0020",
			0,0,
			KS60IndicTypeFaceName,
			KS60IndicFont,
			(const TText16*) L"c:\\shapertamildata2.dat"
			},
			{
			(const TText16*)
			L"\x0B9A\x0BCA\x0BB2\x0BCD\x0BB2\x0BCA\x0BB1\x0BCD\x0BB1\x0BBF\x0020\x0BA8\x0BC0\x0B9F\x0BCD\x0B9F\x0BA4\x0BCD\x0020\x0BA4\x0BCA\x0B95\x0BC1\x0BAE\x0BCD\x0020"
			L"\x0BA4\x0BCA\x0B9F\x0BB0\x0BCD\x0BA8\x0BCA\x0B9F\x0BBF\x0BB1\x0BCD\x0020\x0B95\x0BC0\x0BB4\x0BCD\x0BB5\x0BA9\x0BCD\x0BAE\x0BC8\x0020\x0BAE\x0BC7\x0BB2\x0BC1\x0B95\x0BB0\x0BAE\x0BCD\x0020\x0BAF\x0BAA\x0BCD\x0BAA\x0BBF\x0BA9\x0BCD\x0BAA\x0BC1\x0020"
			L"\x0B85\x0B9F\x0BC8\x0BAF\x0020\x0BB5\x0BB0\x0BC1\x0BAE\x0BBF\x0B95\x0BB0\x0BAE\x0BCD\x0020\x0B85\x0BA9\x0BCD\x0BB1\x0BBF\x0020\x0BAE\x0B9F\x0BA8\x0BB2\x0BCD\x0BB2\x0BBE\x0BAF\x0BCD\x0020"
			L"\x0BAE\x0BC1\x0BAE\x0BCD\x0BAE\x0BC8\x0BAF\x0BBF\x0B9F\x0BA4\x0BCD\x0020\x0BA4\x0BC8\x0BAF\x0BCC\x0BB5\x0BC1\x0B99\x0BCD\x0020\x0B95\x0BC1\x0BA9\x0BCD\x0BB1\x0BC1\x0BAE\x0BC1\x0BA9\x0BCD\x0020\x0BA9\x0BCA\x0BB1\x0BCD\x0BB1\x0BC1\x0BA3\x0BCD\x0B9F\x0BC7\x0BB1\x0BCD\x0020"
			L"\x0B9A\x0BC6\x0BAE\x0BCD\x0BAE\x0BC8\x0BAF\x0BC1\x0BAF\x0BBF\x0BB0\x0BCD\x0020\x0B8F\x0BB1\x0BC1\x0B9E\x0BCD\x0020\x0B9A\x0BC6\x0BB1\x0BBF\x0BA8\x0BCD\x0BA4\x0BC1\x0020"
			L"\x0B95\x0BC1\x0BB1\x0BBF\x0BB2\x0BCD\x0BA8\x0BC6\x0B9F\x0BBF\x0BB2\x0BCD\x0B95\x0BB3\x0BCD\x0020\x0B92\x0BA9\x0BCD\x0BB1\x0BBF\x0BB0\x0BA3\x0BCD\x0B9F\x0BC1\x0020\x0BAE\x0BC2\x0BA9\x0BCD\x0BB1\x0BB3\x0BB5\x0BC1\x0020\x0B95\x0BBE\x0BB2\x0BBE\x0B99\x0BCD\x0020"
			L"\x0B95\x0BC1\x0BB1\x0BC1\x0B95\x0BC1\x0BAE\x0BB5\x0BCD\x0020\x0BB5\x0BBE\x0BAF\x0BCD\x0BA4\x0BAE\x0BCD\x0020\x0B89\x0BAF\x0BBF\x0BB0\x0BCD\x0BAE\x0BC6\x0BAF\x0BCD\x0020\x0BAA\x0BC6\x0BB1\x0BC1\x0BAE\x0BC1\x0BAF\x0BBF\x0BB0\x0BC7\x0020"
			L"\x0BAE\x0BC6\x0BAF\x0BCD\x0BAF\x0BBE\x0BAF\x0BCD\x0BA4\x0BAE\x0BCD\x0020\x0B87\x0B89\x0B95\x0BCD\x0020\x0B95\x0BC1\x0BB1\x0BC1\x0B95\x0BCD\x0B95\x0BAE\x0BB0\x0BC8\x0020\x0BAE\x0BC6\x0BA9\x0BCD\x0BAE\x0BCA\x0BB4\x0BBF\x0BAF\x0BBE\x0BAF\x0BCD\x0020"
			L"\x0B90\x0B94\x0020\x0BB5\x0BB3\x0BB5\x0BCA\x0BA9\x0BCD\x0020\x0BB1\x0BB0\x0BC8\x0020",
			0,0,
			KS60IndicTypeFaceName,
			KS60IndicFont,
			(const TText16*) L"c:\\shapertamildata3.dat"
			}
		};

const TInputData Telugu_InputData[] =
	{
			{
			(const TText16*)
			L"\x0C26\x0C40\x0C28\x0C4D\x0C32\x0C4B\x0020\x0C2A\x0C4D\x0C30\x0C15\x0C43\x0C24\x0C3F\x0020\x0C09\x0C24\x0C4D\x0C15\x0C43\x0C37\x0C4D\x0C1F\x0C2E\x0C46\x0C56\x0C02\x0C26\x0C3F\x0020"
			L"\x0C38\x0C4D\x0C24\x0C4D\x0C30\x0C40\x0C32\x0C41\x0020\x0C07\x0C02\x0C26\x0C4D\x0C30\x0C41\x0C23\x0C4D\x0C23\x0C46\x0C56\x0C28\x0020\x0020"
			L"\x0C35\x0C4D\x0C2F\x0C3E\x0C2E\x0C3F\x0C36\x0C4D\x0C30\x0C47\x0C23\x0C47\x0C35\x0020\x0C32\x0C4B\x0C15\x0C47\x093D\x0C38\x0C4D\x0C2E\x0C3F\x0C28\x0C4D\x0C26\x0C4D\x0C35\x0C3F\x0C35\x0C3F\x0C27\x0C3E\x0020"
			L"\x0C15\x0C30\x0C4D\x0C2E\x0C23\x0C3E\x0C2E\x0C28\x0C3E\x0C30\x0C2E\x0C4D\x0C2D\x0C3E\x0C28\x0C4D\x0C28\x0C48\x0C37\x0C4D\x0C15\x0C30\x0C4D\x0C2E\x0C4D\x0C2F\x0C02\x0020"
			L"\x0C2F\x0C38\x0C4D\x0C24\x0C4D\x0C35\x0C3F\x0C28\x0C4D\x0C26\x0C4D\x0C30\x0C3F\x0C2F\x0C3E\x0C23\x0C3F\x0020\x0C2F\x0C1C\x0C4D\x0C1E\x0C3E\x0C30\x0C4D\x0C25\x0C3E\x0C24\x0C4D\x0C15\x0C30\x0C4D\x0C2E\x0C23\x0C4B\x093D\x0C28\x0C4D\x0C2F\x0C24\x0C4D\x0C30\x0020"
			L"\x0C38\x0C43\x0C37\x0C4D\x0C1F\x0C4D\x0C35\x0C3E\x0020\x0C07\x0C37\x0C4D\x0C1F\x0C3E\x0C28\x0C4D\x0C2D\x0C4B\x0C17\x0C3E\x0C28\x0C4D\x0C39\x0C3F\x0020\x0C2C\x0C4D\x0C30\x0C39\x0C4D\x0C2E\x0C4B\x0C26\x0C4D\x0C2D\x0C35\x0C02\x0020"
			L"\x0C2F\x0C38\x0C4D\x0C24\x0C4D\x0C35\x0C3E\x0C24\x0C4D\x0C2E\x0C30\x0C24\x0C3F\x0C30\x0C47\x0C35\x0020\x0C36\x0C4D\x0C30\x0C47\x0C2F\x0C3E\x0C28\x0C4D\x0C38\x0C4D\x0C35\x0C27\x0C30\x0C4D\x0C2E\x0C4B\x0020"
			L"\x0C2A\x0C4D\x0C30\x0C2F\x0C41\x0C15\x0C4D\x0C24\x0C4B\x093D\x0C2F\x0C02\x0020\x0C24\x0C38\x0C4D\x0C2E\x0C3E\x0C24\x0C4D\x0C24\x0C4D\x0C35\x0C2E\x0C3F\x0C28\x0C4D\x0C26\x0C4D\x0C30\x0C3F\x0C2F\x0C3E\x0C23\x0C4D\x0C2F\x0C3E\x0C26\x0C4C\x0020"
			L"\x0C07\x0C28\x0C4D\x0C26\x0C4D\x0C30\x0C3F\x0C2F\x0C3E\x0C23\x0C3F\x0020\x0C2A\x0C30\x0C3E\x0C23\x0C4D\x0C2F\x0C3E\x0C39\x0C41\x0C30\x0C3F\x0C28\x0C4D\x0C26\x0C4D\x0C30\x0C3F\x0C2F\x0C47\x0C2D\x0C4D\x0C2F\x0C03",
			0,0,
			KS60IndicTypeFaceName,
			KS60IndicFont,
			(const TText16*) L"c:\\shapertelugudata1.dat"
			},
			{
			(const TText16*)
			L"\x0C15\x0C4D\x0C37\x0C47\x0C24\x0C4D\x0C30\x0C15\x0C4D\x0C37\x0C47\x0C24\x0C4D\x0C30\x0C1C\x0C4D\x0C1E\x0C35\x0C3F\x0C2D\x0C3E\x0C17\x0020\x0C08\x0C30\x0C4D\x0C37\x0C4D\x0C2F"
			L"\x0020\x0C15\x0C15\x0C4D\x0C37\x0C4D\x0C2F\x0020\x0020\x0C38\x0C4D\x0C35\x0C3E\x0C24\x0C02\x0C24\x0C4D\x0C30\x0C4D\x0C2F\x0C2E\x0C41\x0020"
			L"\x0C05\x0C30\x0C4D\x0C18\x0C4D\x0C2F\x0C2E\x0C41\x0020\x0020\x0C09\x0C1B\x0C4D\x0C1A\x0C4D\x0C35\x0C3E\x0C38\x0020\x0020\x0C1C\x0C4D\x0C2F\x0C4B\x0C24\x0C4D\x0C38\x0C4D\x0C28\x0020"
			L"\x0020\x0C35\x0C38\x0C4D\x0C24\x0C4D\x0C30\x0C2E\x0C41\x0020\x0020\x0C27\x0C43\x0C24\x0C30\x0C3E\x0C37\x0C4D\x0C1F\x0C4D\x0C30\x0C41\x0C21\x0C41\x0020"
			L"\x0020\x0C07\x0C28\x0C4D\x0C38\x0C4D\x0C2A\x0C46\x0C15\x0C4D\x0C1F\x0C30\x0C4D\x0020\x0020\x0C35\x0C48\x0C36\x0C3F\x0C37\x0C4D\x0C1F\x0C4D\x0C2F\x0C2E\x0C41\x0020"
			L"\x0020\x0C38\x0C3E\x0C2E\x0C30\x0C4D\x0C27\x0C4D\x0C2F\x0C2E\x0C41\x0020\x0020\x0C26\x0C3E\x0C30\x0C3F\x0C26\x0C4D\x0C30\x0C4D\x0C2F\x0C2E\x0C41\x0020"
			L"\x0020\x0C32\x0C15\x0C4D\x0C37\x0C4D\x0C2E\x0C2F\x0C4D\x0C2F\x0020\x0020\x0C15\x0C30\x0C4D\x0C2B\x0C4D\x0C2F\x0C42\x0020\x0020\x0C30\x0C3E\x0C37\x0C4D\x0C1F\x0C4D\x0C30\x0C2E\x0C41\x0020"
			L"\x0020\x0C09\x0C24\x0C4D\x0C2A\x0C4D\x0C30\x0C47\x0C15\x0C4D\x0C37\x0020\x0020\x0C38\x0C02\x0C38\x0C4D\x0C15\x0C43\x0C24\x0C3F\x0020"
			L"\x0020\x0C24\x0C24\x0C4D\x0C15\x0C4D\x0C37\x0C47\x0C24\x0C4D\x0C30\x0C02\x0020\x0C07\x0C28\x0C4D\x0C26\x0C4D\x0C30\x0C3F\x0C2F\x0C3E\x0C30\x0C4D\x0C25\x0C47\x0C37\x0C41\x0020"
			L"\x0C05\x0C38\x0C15\x0C4D\x0C24\x0C3F\x0C30\x0C28\x0C2D\x0C3F\x0C37\x0C4D\x0C35\x0C19\x0C4D\x0C17\x0C03\x0020\x0C2F\x0C24\x0C4D\x0C24\x0C24\x0C4D\x0C2A\x0C4D\x0C30\x0C35\x0C15\x0C4D\x0C37\x0C4D\x0C2F\x0C3E\x0C2E\x0C3F\x0020"
			L"\x0C24\x0C1C\x0C4D\x0C1C\x0C4D\x0C2F\x0C4B\x0C24\x0C3F\x0C38\x0C4D\x0C24\x0C2E\x0C38\x0C03\x0020\x0C2A\x0C4D\x0C30\x0C15\x0C43\x0C24\x0C4D\x0C2F\x0C48\x0C35\x0020\x0C15\x0C43\x0C24\x0C4D\x0C38\x0C4D\x0C28\x0C02"
			L"\x0020\x0C38\x0C4C\x0C15\x0C4D\x0C37\x0C4D\x0C2E\x0C4D\x0C2F\x0C3E\x0C26\x0C3E\x0C15\x0C3E\x0C36\x0C02\x0020\x0C26\x0C4D\x0C35\x0C3E\x0C26\x0C36\x0C4B\x093D\x0C27\x0C4D\x0C2F\x0C3E\x0C2F\x0C03\x0020"
			L"\x0C38\x0C30\x0C4D\x0C35\x0C24\x0C4D\x0C30\x0C17\x0C2E\x0C1A\x0C3F\x0C28\x0C4D\x0C24\x0C4D\x0C2F\x0C1E\x0C4D\x0C1A\x0020",
			0,0,
			KS60IndicTypeFaceName,
			KS60IndicFont,
			(const TText16*) L"c:\\shapertelugudata2.dat"
			},
			{
			(const TText16*)
			L"\x0C38\x0C24\x0C4D\x0C2F\x0C26\x0C40\x0C15\x0C4D\x0C37\x0020\x0C24\x0C4D\x0C2F\x0C41\x0C26\x0C3F\x0C24\x0C47\x0C2D\x0C47\x0C36\x0C4D\x0C35\x0C30\x0020\x0C38\x0C28\x0C4D\x0C28\x0C3E\x0C39\x0C3E\x0C30\x0C4D\x0C2D\x0C1F\x0020"
			L"\x0C28\x0C4D\x0C2F\x0C41\x0C24\x0C4D\x0C2A\x0C3E\x0C26\x0C3F\x0C24\x0C3E\x0C2E\x0C41\x0C15\x0C4D\x0C24\x0C2E\x0020\x0C39\x0C43\x0C26\x0C4D\x0C2F\x0C02\x0C2C\x0C46\x0C56\x0C28\x0020"
			L"\x0C2A\x0C4D\x0C30\x0C47\x0C37\x0C3F\x0C24\x0C4B\x0C26\x0C4D\x0C2F\x0C1A\x0C4D\x0C1B\x0C26\x0020\x0C35\x0C1F\x0C15\x0C4D\x0C37\x0C4D\x0C2E\x0C3E\x0C1C\x0C2E\x0C41\x0C28\x0C4D\x200C\x0020"
			L"\x0C26\x0C02\x0C37\x0C4D\x0C1F\x0C4D\x0C30\x0C3F\x0C15\x0C32\x0020\x0C38\x0C30\x0C4D\x0C35\x0C24\x0C4D\x0C30\x0C17\x0C2E\x0C1A\x0C3F\x0C28\x0C4D\x0C24\x0C4D\x0C2F\x0C1E\x0C4D\x0020"
			L"\x0C15\x0C42\x0C1F\x0C38\x0C4D\x0C25\x0C2E\x0C1A\x0C32\x0C28\x0C4D\x0C27\x0C4D\x0C30\x0C41\x0C35\x0C2E\x0020\x0C2F\x0C47\x0020"
			L"\x0C24\x0C4D\x0C35\x0C15\x0C4D\x0C37\x0C30\x0C2E\x0C28\x0C3F\x0C30\x0C4D\x0C26\x0C47\x0C36\x0C4D\x0C2F\x0C2E\x0C35\x0C4D\x0C2F\x0C15\x0C4D\x0C24\x0020"
			L"\x0C05\x0C25\x0C48\x0C24\x0C26\x0C2A\x0C4D\x0C2F\x0C36\x0C15\x0C4D\x0C24\x0C4B\x093D\x0C38\x0020\x0C1C\x0C4D\x0C1E\x0C3E\x0C28\x0C2E\x0C2D\x0C4D\x0C2F\x0C3E\x0C38\x0C3E\x0C1C\x0C4D\x0C1C\x0C4D\x0C1E\x0C3E\x0C28\x0C3E\x0C26\x0C4D\x0C27\x0C4D\x0C2F\x0C3E\x0C28\x0C02\x0020"
			L"\x0C39\x0C30\x0C4D\x0C37\x0C3E\x0C2E\x0C30\x0C4D\x0C37\x0C2D\x0C2F\x0C4B\x0C26\x0C4D\x0C35\x0C47\x0C17\x0C48\x0C30\x0C4D\x0C2E\x0C41\x0C15\x0C4D\x0C24\x0020\x0C36\x0C24\x0C4D\x0C30\x0C4C\x0020\x0C1A\x0020"
			L"\x0C2E\x0C3F\x0C24\x0C4D\x0C30\x0C47\x0020\x0C2F\x0C4B\x0C28\x0C3F\x0C30\x0C4D\x0C2E\x0C39\x0C26\x0C4D\x0C2C\x0C4D\x0C30\x0C39\x0C4D\x0C2E\x0020\x0C38\x0C24\x0C4D\x0C24\x0C4D\x0C35\x0C02\x0020"
			L"\x0C28\x0C3F\x0C30\x0C4D\x0C2E\x0C32\x0C24\x0C4D\x0C35\x0C3E\x0C24\x0C4D\x0C2A\x0C4D\x0C30\x0C15\x0C3E\x0C36\x0C15\x0C2E\x0C28\x0C3E\x0C2E\x0C2F\x0C2E\x0020"
			L"\x0C26\x0C47\x0C39\x0C47\x093D\x0C38\x0C4D\x0C2E\x0C3F\x0C28\x0C4D\x0C2A\x0C4D\x0C30\x0C15\x0C3E\x0C36\x0020\x0C2A\x0C4D\x0C30\x0C35\x0C43\x0C24\x0C4D\x0C24\x0C3F\x0C30\x0C3E\x0C30\x0C2E\x0C4D\x0C2D\x0C03",
			0,0,
			KS60IndicTypeFaceName,
			KS60IndicFont,
			(const TText16*) L"c:\\shapertelugudata3.dat"
			}
		};

//	Test data for Gurmukhi scripts
const TInputData Gurmukhi_InputData[] =
	{
		{
		// Test for Gurmukhi digit
		(const TText16*) 
		L"\xa66\xa67\xa68\xa69\xa6a\xa6b\xa6c\xa6d\xa6e\xa6f"
		L"\xa15\xa66\xa17\xa67\xa19\xa68\xa25\xa69\xa27\xa6a\xa29\xa6b\xa3a\xa6c\xa3b\xa6d\xa3c\xa6e\xa40\xa6f"
		L"\xa3e\xa66\xa3e\xa67\xa3e\xa68\xa3e\xa69\xa3e\xa6a\xa3e\xa6b\xa3e\xa6c\xa3e\xa6d\xa3e\xa6e\xa3e\xa6f"
		L"\xa73\xa66\xa73\xa67\xa73\xa68\xa73\xa69\xa73\xa6a\xa73\xa6b\xa73\xa6c\xa73\xa6d\xa73\xa6e\xa73\xa6f"
		L"\xa4d\xa66\xa4d\xa67\xa4d\xa68\xa4d\xa69\xa4d\xa6a\xa4d\xa6b\xa4d\xa6c\xa4d\xa6d\xa4d\xa6e\xa4d\xa6f"
		L"\xa15\xa66\xa3e\xa17\xa67\xa3e\xa19\xa68\xa3e\xa25\xa69\xa3e\xa27\xa6a\xa3e\xa29\xa6b\xa3e\xa3a\xa6c\xa3e\xa3b\xa6d\xa3e\xa3c\xa6e\xa3e\xa40\xa6f\xa3e",
		0,0,
		KS60IndicTypeFaceName,
		KS60IndicFont,
		(const TText16*) L"c:\\shapergurmukhi1.dat"
		},
		{
		// Test for Gurmukhi and Latin
		(const TText16*) 
		L"\xa05\xa70\xa68\xa77"
		L"\x63\xa19\xa48"
		L"\xa1a\xa3f\x64\xa71"
		L"\xa15\xa69\xa3e"
		L"\xa17\xa6a\xa3e"
		L"\xa25\xa6b\xa3e"
		L"\xa35\xa6c\xa3e"
		L"\xa37\xa6d\xa3e"
		L"\xa40\xa6e\xa3e"
		L"\xa41\xa6f\xa3e",
		0,0,
		KS60IndicTypeFaceName,
		KS60IndicFont,
		(const TText16*) L"c:\\shapergurmukhi2.dat"
		},
		{
		//	Test for Gurmukhi ModifierAfterConsonantViramaZWJ
		//	Consonant + Virama + ZWJ + Modifier
		(const TText16*) 
		L"\xa15\xa4d\x200d\xa01"
		L"\xa15\xa4d\x200d\xa02"
		L"\xa15\xa4d\x200d\xa03"
		L"\xa15\xa4d\x200d\xa70",
		0,0,
		KS60IndicTypeFaceName,
		KS60IndicFont,
		(const TText16*) L"c:\\shapergurmukhi3.dat"
		},
		{
		// Test for Gurmukhi consonant leading sequence
		(const TText16*) 
		//	Consonant + Nukta
		L"\xa15\xa3c"
		//	Consonant + dependent vowel
		L"\xa16\xa4c"
		L"\xa17\xa4c"
		//	Consonant + Virama + HA(RA/VA/YA)
		L"\xa15\xa4d\xa39"
		L"\xa15\xa4d\xa30"
		L"\xa15\xa4d\xa35"
		L"\xa15\xa4d\xa3f"
		//	Consonant + Addak
		L"\xa15\xa71"
		//	Consonant + bindi
		L"\xa16\xa02"
		L"\xa16\xa40\xa02"
		L"\xa16\xa4b\xa02"
		//	Consonant + Tippi
		L"\xa25\xa70"
		L"\xa25\xa3f\xa70"
		L"\xa25\xa41\xa70"
		//	Nukta, conrresponding consonant, Nukta pair
		L"\xa33"
		L"\xa32\xa3c"
		L"\xa36"
		L"\xa38\xa3c"
		L"\xa59"
		L"\xaa1\xa3c"
		L"\xa5a"
		L"\xa17\xa3c"
		L"\xa5b"
		L"\xa1c\xa3c"
		L"\xa5e"
		L"\xa2b\xa3c",
		0,0,
		KS60IndicTypeFaceName,
		KS60IndicFont,
		(const TText16*) L"c:\\shapergurmukhi4.dat"
		},
		{
		// Test for Gurmukhi vowel leading sequence
		(const TText16*) 
		//	ARA, IRI, URA
		L"\xa05"
		L"\xa72"
		L"\xa73",
		0,0,
		KS60IndicTypeFaceName,
		KS60IndicFont,
		(const TText16*) L"c:\\shapergurmukhi5.dat"
		}
	};


//	Test data for Malayalam scripts
const TInputData Malayalam_InputData[] =
	{
		{
		// Test for Malayalam digit
		(const TText16*) 
		L"\xd66\xd67\xd68\xd69\xd6a\xd6b\xd6c\xd6d\xd6e\xd6f"
		L"\xd15\xd66\xd17\xd67\xd19\xd68\xd25\xd69\xd27\xd6a\xd29\xd6b\xd3a\xd6c\xd3b\xd6d\xd3c\xd6e\xd40\xd6f"
		L"\xd03\xd66\xd03\xd67\xd03\xd68\xd03\xd69\xd03\xd6a\xd02\xd6b\xd02\xd6c\xd02\xd6d\xd02\xd6e\xd02\xd6f"
		L"\xd3e\xd66\xd3f\xd67\xd40\xd68\xd41\xd69\xd42\xd6a\xd42\xd6b\xd43\xd6c\xd46\xd6d\xd47\xd6e\xd48\xd6f"
		L"\xd4d\xd66\xd4d\xd67\xd4d\xd68\xd4d\xd69\xd4d\xd6a\xd4d\xd6b\xd4d\xd6c\xd4d\xd6d\xd4d\xd6e\xd4d\xd6f"
		L"\xd15\xd66\xd3e\xd17\xd67\xd3e\xd19\xd68\xd3e\xd25\xd69\xd3e\xd27\xd6a\xd3e\xd29\xd6b\xd3e\xd3a\xd6c\xd3e\xd3b\xd6d\xd3e\xd3c\xd6e\xd3e\xd40\xd6f\xd3e",
		0,0,
		KS60IndicTypeFaceName,
		KS60IndicFont,
		(const TText16*) L"c:\\shapermalayalam1.dat"
		},
		{
		// Test for Malayalam and Latin
		(const TText16*) 
		L"\xd05\xd70\xd68\xd77"
		L"\x63\xd19\xd48"
		L"\xd1a\xd3f\x64\xd71"
		L"\xd15\xd69\xd3e"
		L"\xd17\xd6a\xd3e"
		L"\xd25\xd6b\xd3e"
		L"\xd35\xd6c\xd3e"
		L"\xd37\xd6d\xd3e"
		L"\xd40\xd6e\xd3e"
		L"\xd41\xd6f\xd3e",
		0,0,
		KS60IndicTypeFaceName,
		KS60IndicFont,
		(const TText16*) L"c:\\shapermalayalam2.dat"
		},
		{
		//	Test for Malayalam ModifierAfterConsonantViramaZWJ
		//	Consonant + Virama + ZWJ + Modifier
		(const TText16*) 
		L"\xd15\xd4d\x200d\xd02"
		L"\xd25\xd4d\x200d\xd02"
		L"\xd33\xd4d\x200d\xd03"
		L"\xd34\xd4d\x200d\xd03",
		0,0,
		KS60IndicTypeFaceName,
		KS60IndicFont,
		(const TText16*) L"c:\\shapermalayalam3.dat"
		},
		{
		// Test for Malayalam consonant leading sequence
		(const TText16*) 
		//	Consonant, and its split representation
		L"\xd14"
		L"\xd12\xd57"
		L"\xd4a"
		L"\xd46\xd3e"
		L"\xd4b"
		L"\xd47\xd3e"
		L"\xd4c"
		L"\xd46\xd57"
		//	Consonant + dependent vowel, two sequences to check context shaping
		L"\xd15\xd3e"
		L"\xd16\xd3f"
		L"\xd15\xd40"
		L"\xd15\xd41"
		L"\xd16\xd42"
		L"\xd15\xd43"
		L"\xd17\xd44"
		L"\xd15\xd45"
		L"\xd18\xd46"
		L"\xd2a\xd47"
		L"\xd25\xd48"
		//	Consonant + Matra /-e/, /-ee/, /-ai/ 
		L"\xd16\xd46"
		L"\xd17\xd47"
		L"\xd18\xd48"
		//	Consonant and Matras /-o/, /-oo/, /-au/ to shaper
		//	Consonant and 0x0d4c (first form of Matra /-au/ ) to shaper,
		//	Consonant, 0x0d46 and 0x0d57 (second form of Matra /-au/) to shaper002E
		L"\xd16\xd4a"
		L"\xd17\xd4b"
		L"\xd18\xd4c"
		L"\xd15\xd46\xd57"
		//	Consonant + 0x0d4d + Consonant
		L"\xd15\xd4d\xd15"
		L"\xd31\xd4d\xd31\xd4d"		
		//	Consonant (NNA, NA, RA, LA, LLA) + Virama + ZWJ to shaper
		//	KA + Virama + ZWJ to shaper.
		// 	other Consonant + Virama + ZWJ to shaper.
		L"\xd23\xd4d\x200d"
		L"\xd28\xd4d\x200d"
		L"\xd30\xd4d\x200d"
		L"\xd32\xd4d\x200d"
		L"\xd33\xd4d\x200d"
		L"\xd15\xd4d\x200d"
		L"\xd16\xd4d\x200d"
		//	RA + Virama + ZWJ to shaper,
		//	RRA + Viram + ZWJ to shaper.
		L"\xd30\xd4d\x200d"
		L"\xd31\xd4d\x200d"
		//	Chillu and Modifier to shaper
		//	Chillu and Virama to shaper
		L"\xd31\xd4d\x200d\xd05"
		L"\xd15\xd4d\x200d\xd14"
		L"\xd31\xd4d\x200d\xd4d"
		L"\xd15\xd4d\x200d\xd4d"
		//	One of (KA, GA, PA, BA, MA, LA, SHA, SA, HA), Virama and LA to shaper
		L"\xd15\xd4d\xd32"
		L"\xd17\xd4d\xd32"
		L"\xd2a\xd4d\xd32"
		L"\xd2c\xd4d\xd32"
		L"\xd2e\xd4d\xd32"
		L"\xd32\xd4d\xd32"
		L"\xd36\xd4d\xd32"
		L"\xd38\xd4d\xd32"
		L"\xd39\xd4d\xd32"
		//	CA, Virama and CA to shaper
		//	BA, Virama and BA to shaper
		//	YA, Virama and YA to shaper
		//	VA, Virama and VA to shaper
		//	other Consonant, Virama and the same Consonant to shaper.
		L"\xd1a\xd4d\xd1a"
		L"\xd2c\xd4d\xd2c"
		L"\xd2f\xd4d\xd2f"
		L"\xd35\xd4d\xd35"
		L"\xd15\xd4d\xd15"
		//	G.GA, C.CHA, K.TTA, SS.TTA, DD.DDA, NN.NNA, B.DA, B.DHA, P.PA, RR.RRA, SH.SHA, S.SA clusters to shaper
		L"\xd17\xd4d\xd17"
		L"\xd2c\xd4d\xd26"
		L"\xd1a\xd4d\xd1b"
		L"\xd2c\xd4d\xd27"
		L"\xd15\xd4d\xd1f"
		L"\xd2a\xd4d\xd2a"
		L"\xd37\xd4d\xd1f"
		L"\xd31\xd4d\xd31"
		L"\xd22\xd4d\xd22"
		L"\xd36\xd4d\xd36"
		L"\xd23\xd4d\xd23"
		L"\xd38\xd4d\xd38"
		//	Consonant clusters from REQ1.3.11 to shaper.
		//	Already covered in other sequence
		//	S.RR.RRA to shaper,
		//	N.RR.RRA to shaper
		L"\xd38\xd4d\xd31\xd31"
		L"\xd28\xd4d\xd31\xd31"
		//	NA, Virama, ZWJ, RRA to shaper
		//	NA, Virama, ZWJ, ZWNJ, RRA to shaper
		L"\xd28\xd4d\x200d\xd31"
		L"\xd28\xd4d\x200d\x200c\xd31"
		//	Consonant cluster and YA to shaper
		//	Consonant cluster and VA to shaper
		L"\xd15\xd4d\xd30\xd3f"
		L"\xd15\xd4d\xd30\xd35"
		//	YA, YAKAR to shaper
		//	VA, VAKAR to shaper
		L"\xd15\xd4d\xd2f"
		L"\xd15\xd4d\xd35"
		//	one of (LLA, LLLA, RRA) and YA to shaper
		//	one of (LLA, LLLA, RRA) and VA to shaper
		L"\xd33\xd2f"
		L"\xd34\xd2f"
		L"\xd31\xd2f"
		L"\xd33\xd35"
		L"\xd34\xd35"
		L"\xd31\xd35"
		//	Consonant cluster and RA to shaper
		L"\xd15\xd4d\xd30\xd30"
		//	one of (LLA, LLLA, RRA) and RA to shaper
		L"\xd33\xd30"
		L"\xd34\xd30"
		L"\xd31\xd30"
		//	YAKAR/VAKAR and RA to shaper.
		L"\xd15\xd4d\xd2f\xd30"
		L"\xd15\xd4d\xd35\xd30"
		//	Chillu
		//	Chillu and Vowel to shaper
		//	Chillu and Consonant to shaper
		//	Chillu and Modifier to shaper
		//	Chillu and Virama to shaper
		L"\xd06\xd23\xd4d\x200d"
		L"\xd06\xd28\xd4d\x200d"
		L"\xd06\xd30\xd4d\x200d"
		L"\xd06\xd32\xd4d\x200d"
		L"\xd06\xd33\xd4d\x200d"
		L"\xd23\xd4d\x200d\xd06"
		L"\xd28\xd4d\x200d\xd09"
		L"\xd30\xd4d\x200d\xd0a"
		L"\xd32\xd4d\x200d\xd0f"
		L"\xd33\xd4d\x200d\xd13"
		L"\xd23\xd4d\x200d\xd16"
		L"\xd28\xd4d\x200d\xd19"
		L"\xd30\xd4d\x200d\xd1a"
		L"\xd32\xd4d\x200d\xd1f"
		L"\xd33\xd4d\x200d\xd31"
		L"\xd23\xd4d\x200d\xd02"
		L"\xd28\xd4d\x200d\xd02"
		L"\xd30\xd4d\x200d\xd03"
		L"\xd32\xd4d\x200d\xd03"
		L"\xd33\xd4d\x200d\xd03"
		L"\xd23\xd4d\x200d\xd4d"
		L"\xd28\xd4d\x200d\xd4d"
		L"\xd30\xd4d\x200d\xd4d"
		L"\xd32\xd4d\x200d\xd4d"
		L"\xd33\xd4d\x200d\xd4d",
		0,0,
		KS60IndicTypeFaceName,
		KS60IndicFont,
		(const TText16*) L"c:\\shapermalayalam4.dat"
		},
		{
		// Test for Malayalam vowel leading sequence
		(const TText16*) 
		L"\xd05\xd06\xd07\xd08\xd09\xd0a\xd0b\xd60\xd0c\xd61\xd0e\xd0f\xd10\xd12\xd13\xd14"
		//	Vowel + modifer 
		L"\xd05\xd02\xd06\xd02\xd07\xd02\xd08\xd02\xd09\xd02\xd0a\xd03\xd0b\xd03\xd60\xd03\xd0c\xd03\xd61\xd03\xd0e\xd03\xd0f\xd03\xd10\xd03\xd12\xd03\xd13\xd03\xd14\xd03"
		//	Vowel, Modifier, Matra
		L"\xd05\xd02\xd3e\xd06\xd02\xd3f\xd07\xd02\xd40\xd08\xd02\xd41\xd09\xd02\xd42\xd0a\xd02\xd43\xd0b\xd02\xd46\xd60\xd02\xd47\xd0c\xd02\xd48\xd61\xd02\xd4a\xd0e\xd02\xd4b\xd0f\xd02\xd4c\xd10\xd02\xd3e\xd12\xd02\xd3f\xd13\xd02\xd40\xd14\xd02\xd4c"
		//	Vowel, Virama,
		L"\xd05\xd4d\xd06\xd4d\xd07\xd4d\xd08\xd4d\xd09\xd4d\xd0a\xd4d\xd0b\xd4d\xd60\xd4d\xd0c\xd4d\xd61\xd4d\xd0e\xd4d\xd0f\xd4d\xd10\xd4d\xd12\xd4d\xd13\xd4d\xd14\xd4d"
		//	Vowel, Matra,
		L"\xd05\xd3e\xd06\xd3f\xd07\xd40\xd08\xd41\xd09\xd42\xd0a\xd43\xd0b\xd46\xd60\xd47\xd0c\xd48\xd61\xd4a\xd0e\xd4b\xd0f\xd4c\xd10\xd3e\xd12\xd3f\xd40\xd13\xd4c\xd14\xd4c"
		//	Vowel Virma, Matra
		L"\xd05\xd4d\xd3e\xd06\xd4d\xd3f\xd07\xd4d\xd40\xd08\xd4d\xd41\xd09\xd4d\xd42\xd0a\xd4d\xd43\xd0b\xd4d\xd46\xd60\xd4d\xd47\xd0c\xd4d\xd48\xd61\xd4d\xd4a\xd0e\xd4d\xd4b\xd0f\xd4d\xd4c\xd10\xd4d\xd3e\xd12\xd4d\xd3f\xd13\xd4d\xd4c\xd14\xd4d\xd40",
		0,0,
		KS60IndicTypeFaceName,
		KS60IndicFont,
		(const TText16*) L"c:\\shapermalayalam5.dat"
		}
	};
